static char rcsid[] = "@(#)$Id: newalias.c,v 2.12 2021/07/13 07:58:37 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.12 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 utils/newalias.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *
 *****************************************************************************/

/** Install a new set of aliases for the 'Elm' mailer. 

	If invoked without the -g argument, it assumes that
  it is working with an individual users alias tables, and
  convert it to new format.
	If, however, it is invoked with the -g flag,
  it assumes that the user is updating the system alias
  file and uses the defaults for everything.

  The format for the input file is;
    alias1, alias2, ... = username = address
or  alias1, alias2, ... = groupname= member, member, member, ...
                                     member, member, member, ...

**/

#include "def_utils.h"
#include "s_newalias.h"

#include "addrlib.h"
#include "aliaslib.h"
#include "aliases_helper.h"

#include "reghelper.h"

DEBUG_VAR(Debug,__FILE__,"util");

const char * program_name = "newalias";
int    register_fd  = -1;
char * register_module = NULL;

static char version_buff[NLEN];

int main P_((int argc, char *argv[]));
int main(argc, argv)
     int argc;
     char *argv[];
{
    int  is_system = 0;		/* system file updating?     */

    char *oldinput= NULL;
    char *olddata = NULL;
    
    char *newfile = NULL;

    struct aliases_map  * map = NULL;
    struct aliases_map  * oldmap = NULL;
    char * tmp = NULL;
    charset_t                cs = NULL;
    struct editor_propline * pl = NULL;

    int err = 0;
    int a;
    int errcode = 0;
    int quiet = 0;

    FILE * F;

    int read_flags = 0;
    int r_dump = 0;

#if DEBUG
    init_debugfile("NEWALIAS");
#endif
    locale_init();

    REGHELPER_INIT(argv[0]);

    user_init();
    
    for (a = 1; a < argc; ++a) {
	if (strcmp(argv[a], "-g") == 0)
	    is_system = 1;

	else if (strcmp(argv[a], "-q") == 0)
	    quiet = 1;

#ifdef DEBUG
	else if (strcmp(argv[a], "-d") == 0) {	       
	    if (++a < argc)
		set_debugging(argv[a]);	 
	}
#endif
	else {
	    lib_error(CATGETS(elm_msg_cat,
			      NewaliasSet, NewaliasUsage, 
			      "Usage: %s [-g] [-q]\n"), argv[0]);
	    
	    err = 1;
	    goto fail;
	}
    }

    read_flags = is_system ? READ_FLAG_GLOBAL_ONLY : 0;
    
    init_addrlib(read_flags);
    init_aliaslib(read_flags);    
    init_defaults(read_flags);

    if (!is_system)
	read_rc_file(read_flags);
    else
	post_init_check(read_flags);
        
    elm_sfprintf(version_buff, sizeof version_buff,
		 FRM("%s PL%s"), VERSION, PATCHLEVEL);

#ifdef DEBUG
    { 
	int d = panic_dprint("\n\
======================================================\n\
Debug output of the NEWALIAS program (version %s).\n",
			     version_buff);

	if (d >= 50) {
#if 0	
	    panic_dprint("WARNING: Edit manually out sensitive information from that file!\n");
    
	    lower_prompt("WARNING: Debug file may include passwords -- edit it!");
	    sleep(5+sleepmsg);	    
#endif
	}
    }
#endif

	if (is_system) {   /* update system aliases */
	    oldinput  = old_system_text_file;
	    olddata   = old_system_data_file;
	    newfile   = system_aliases_file;
	    map       = system_aliases_map;
	    cs        = system_aliases_cs;
	    pl        = system_aliases_pl;

	} else {
	    oldinput  = old_user_text_file;
	    olddata   = old_user_data_file;
	    newfile   = user_aliases_file;
	    map       = user_aliases_map;
	    cs        = user_aliases_cs;
	    pl        = user_aliases_pl;
	}

	
	if (quiet && access(oldinput,READ_ACCESS) < 0) {
	    err = 0;
	    goto fail;
	}

	if (is_system)
	    elm_fprintf(stdout,
			CATGETS(elm_msg_cat, NewaliasSet, NewaliasConvertSystem,
				"Converting the system alias file..."));
	else
	    elm_fprintf(stdout,
			CATGETS(elm_msg_cat, NewaliasSet, 
				NewaliasConvertPersonal,
				"Converting your personal alias file..."));
	fflush(stdout);

	oldmap = load_old_aliases_map(oldinput);
	if (!oldmap) {
	    err = 1;
	    goto fail;
	}

	change_aliases_map(&map,oldmap);

	
	elm_fprintf(stdout,
		    CATGETS(elm_msg_cat, NewaliasSet, NewaliasProcessed,
			    "Processed %d aliases.\n"), 
		    aliases_map_item_count(oldmap));


	tmp = elm_message(FRM("%s.N"),newfile);
	errcode = can_open(tmp,"w");

	if (errcode) {
	    lib_error(CATGETS(elm_msg_cat, NewaliasSet, NewaliasFileNotWriteable,
			      "File %s is not writeable: %s"),
		      tmp, strerror(errcode));
	    
	    err++;
	    free(tmp);
	    goto fail;
	}
	
	F = fopen(tmp,"w");
	if (!F) {
	    errcode = errno;

	    lib_error(CATGETS(elm_msg_cat, NewaliasSet, NewaliasFileNotWriteable,
			      "File %s is not writeable: %s"),
		      tmp, strerror(errcode));
	    
	    err++;
	    free(tmp);
	    goto fail;
	}

	r_dump = dump_aliases_map(F,map,NULL,"NEWALIAS",version_buff,
			 cs,pl,NULL,&errcode);

	if (!r_dump) {
	    lib_error(CATGETS(elm_msg_cat, NewaliasSet, NewaliasFileNotWriteable,
			      "File %s is not writeable: %s"),
		      tmp, strerror(errcode));

	    fclose(F); /* ignore error */
	    
	    err++;
	    free(tmp);
	    goto fail;
	
	} else if (EOF == fclose(F)) {
	    errcode = errno;

	    lib_error(CATGETS(elm_msg_cat, NewaliasSet, NewaliasFileNotWriteable,
			      "File %s is not writeable: %s"),
		      tmp, strerror(errcode));
	    
	    err++;
	    free(tmp);
	    goto fail;
	}

	if (0 != rename(tmp,newfile)) {
	    int errcode = errno;
	    lib_error(CATGETS(elm_msg_cat, NewaliasSet, NewaliasFileNotRenamed,
			      "Failed to rename temporary file to %s: %s"),
		      newfile, strerror(errcode));
	    
	    err++;
	    free(tmp);
	    goto fail;
	}

	log_config(newfile);
	free(tmp);
	
	tmp = elm_message(FRM("%s.OLD"),oldinput);
	if (0 == rename(oldinput,tmp)) {
	    DPRINT(Debug,1,(&Debug,"%s renamed to %s\n",
			    oldinput,tmp));
	}
	free(tmp);
	delete_old_alias_files(olddata);

	elm_fprintf(stdout,
		    CATGETS(elm_msg_cat, NewaliasSet, NewaliasWrittenTo,
			    "Written %d aliases to %s\n"), 
		    aliases_map_item_count(map),
		    newfile);
 fail:
    if (err)
	lib_error(CATGETS(elm_msg_cat, NewaliasSet, NewaliasProgFailed,
			  "%s failed; exit code=%d"),
		  argv[0],err);

    if (map)
	free_aliases_map( &map);
    if (oldmap)
	free_aliases_map( &oldmap);

    return err;

}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
