static char rcsid[] = "@(#)$Id: elmconfwriter.c,v 2.11 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.11 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *       or Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_utils.h"
#include "s_me.h"
#include "s_elm.h"
#include "s_elmrc.h"

#include "rc_imp.h"
#include "save_opts.h"

#include "misclib.h"
#include "addrlib.h"
#include "aliaslib.h"
#include "mboxlib.h"
#include "melib.h"
#include "reghelper.h"

DEBUG_VAR(Debug,__FILE__,"util");

extern char *optarg;		
extern int   optind;		

int    register_fd     = -1; 
const char * program_name    = "elmconfwriter"; 
char * register_module = NULL;

static char version_buff[NLEN];

static const char OPTION_LETTERS[] = "d:c:lIb:";

int main P_((int argc, char *argv[]));
int main(argc, argv)
     int argc;
     char *argv[];
{
    int err = 0;
    int c;
    char * elmrc_changes = NULL;
    struct elmrc_recorder * recorder = NULL;
    FILE * elmrc_changes_F = NULL;
    int list_files = 0;
    int ignore = 0;
    char * backup_suffix = NULL;
    
    int read_flags = READ_FLAG_GLOBAL_ONLY;
    
    enum {
	conf_elmrc,
	conf_charset_map,
	conf_mime_types_map,
	conf_terminal_map,
	conf_iso2022_map,
#ifdef REMOTE_MBX
	conf_mail_services,
#endif
	conf_aliases_map,
	conf_mlist_map,
	conf_hash_marks,
	conf_tagfilter_entities,
	conf_LAST
    } config_file;

    enum {
	pass_check_backup,
	pass_action,
	pass_LAST
    } process_pass, start_pass = pass_action;

    int check_backup_ok =  1;

    char *backup_file[conf_LAST];

    for (config_file = conf_elmrc; config_file < conf_LAST; config_file++) {
	backup_file[config_file] = NULL;
    }
    
    
#if DEBUG
    init_debugfile("ELMCONFWRITER");
#endif
    
    while ((c = ELM_GETOPT(argc, argv, OPTION_LETTERS)) != EOF) {
	switch(c) {
	case 'd' : 	
#if DEBUG
	    set_debugging(optarg);	  
#endif
	    /* Error is printed later */
	    break;
	}
    }
    optind = 1;     /* Reset scanning */

    locale_init();

    REGHELPER_INIT(argv[0]);

while ((c =  ELM_GETOPT(argc, argv, OPTION_LETTERS)) != EOF) {
	
	switch(c) {
	    
	case 'd':
#if DEBUG
	    set_debugging(optarg);	 
#else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmArgsIngoringDebug,
			      "Warning: system created without debugging enabled - request ignored\n"));
#endif
	    break;
	    

	case 'c':
	    elmrc_changes = optarg;
	    if (0 != access(elmrc_changes,WRITE_ACCESS)) {
		int errcode = errno;
		
		if (errcode != ENOENT) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
				      "File %.50s is not writeable: %s"),
			      elmrc_changes, strerror(errcode));
		    err=1;
		    goto fail;
		}
	    }


	    elmrc_changes_F = open_or_create(elmrc_changes);
	    if (!elmrc_changes_F) {
		int errcode = errno;
		
		if (errcode != ENOENT) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
				      "File %.50s is not writeable: %s"),
			      elmrc_changes, strerror(errcode));
		    err=1;
		    goto fail;
		}
	    }

	    break;

        case 'l':
	    list_files = 1;
	    break;
	case 'b':
	    if (backup_suffix) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeDupBackupOpt,
				  "Duplicate backup option: Both -b%s and -b%s given."),
			  backup_suffix,optarg);
		err = 1;
	    }
	    backup_suffix = optarg;
	    start_pass    = pass_check_backup;
	    break;
	    
	case 'I':
	    ignore = READ_FLAG_IGNORE_MISSING;
	    break;
	    
	case '?':
	    err = 1;
	    goto fail;
	}

    }

    user_init();

    read_flags = READ_FLAG_GLOBAL_ONLY|ignore;
    
    init_misclib(read_flags);

    if (elmrc_changes_F) {
	recorder = enable_recorder(SYSTEM_RC);
	
	seed_history(elmrc_changes_F,elmrc_changes,recorder);
    }

    init_addrlib(read_flags);
    init_aliaslib(read_flags);
    init_mboxlib(read_flags);
    init_melib(read_flags);   
    init_defaults(read_flags);

    elm_sfprintf(version_buff, sizeof version_buff,
		 FRM("%s PL%s"), VERSION, PATCHLEVEL);
    
#ifdef DEBUG
    { 
	int d = panic_dprint("\n\
======================================================\n\
Debug output of the ELMCONFWRITER program (version %s).\n",
			     version_buff);

	if (d >= 50) {
#if 0	
	    panic_dprint("WARNING: Edit manually out sensitive information from that file!\n");
	    
	    lower_prompt("WARNING: Debug file may include passwords -- edit it!");
	    sleep(5+sleepmsg);	    
#endif
	}
    }
#endif
    
    post_init_check(read_flags);


    for (process_pass = start_pass; process_pass < pass_LAST; process_pass++) {
    
	for (config_file = conf_elmrc; config_file < conf_LAST; config_file++) {
	    char * targetfile = NULL;
	    
	    switch (config_file) {
	    case conf_elmrc: targetfile = system_rc_file;
		break;
	    case conf_charset_map: targetfile = system_mime_charsets;
		break;
	    case conf_mime_types_map: targetfile = system_mime_types;
		break;
	    case conf_terminal_map: targetfile = system_terminal_info;
		break;
	    case conf_iso2022_map: targetfile = system_iso2022_sets;
		break;
#ifdef REMOTE_MBX
	    case conf_mail_services: targetfile = system_mail_services;
		break;
#endif
	    case conf_aliases_map: targetfile = system_aliases_file;
		break;
	    case conf_mlist_map: targetfile = system_mlist_file;
		break;
	    case conf_hash_marks: targetfile = system_hash_marks;
		break;

	    case conf_tagfilter_entities:
		targetfile = system_tagfilter_entfile;
		break;
		
	    case conf_LAST:
		break;
	    }

	    switch (process_pass) {
	    case pass_check_backup:

		if (targetfile) {
		    int r5;
		    
		    backup_file[config_file] = safe_strdup(targetfile);
		    backup_file[config_file] = strmcat(backup_file[config_file],
						       backup_suffix);

		    r5 = access(backup_file[config_file],ACCESS_EXISTS);
		    switch (r5) {
			int errcode;
		    case -1:
			errcode = errno;

			if (ENOENT != errcode) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeBackupFileError,
					      "Backup file %s: %s -- Check -b%s option."),
				      backup_file[config_file],strerror(errcode),backup_suffix);
			    check_backup_ok = 0;
			    err = 1;
			}

			break;
		    case 0:
			lib_error(CATGETS(elm_msg_cat, MeSet, MeBackupFileExist,
					  "Backup file %s exists. Change -b%s option."),
				  backup_file[config_file],backup_suffix);
			check_backup_ok = 0;
			err = 1;
			
			break;
		    }

		}
		break;
		
	    case pass_action:
		if (!check_backup_ok)
		    break;  /* No action */
		
		if (targetfile) {
		    int r = 0;
		    FILE * f = NULL;
		    char * tmp = NULL;
		    struct stat target_stat;
		    int mode = 0644;
		    
		    enum syscall_status r1 = access(targetfile,WRITE_ACCESS);
		    enum syscall_status r2;
		    enum syscall_status r4;
		    
		    switch (r1) {
			
		    case syscall_error /* -1 */: {			
			int errcode = errno;
			
			if (ENOENT == errcode) {
			    
			    if (list_files)
				printf ("%s (does not exists)\n",
					targetfile);
			    
			    continue;
			}
		    
			if (list_files)
			    printf ("%s (error: %s)\n",
				    targetfile,strerror(errcode));
			else
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
					      "File %.50s is not writeable: %s"),
				      targetfile, strerror(errcode));
			err=1;
			continue;
		    }
			break;

		    case syscall_success /* 0 */:
			if (list_files) {
			    if (backup_file[config_file]) {
				FILE * orig_file_F   = NULL;
				FILE * backup_file_F = NULL;
				int fd;
				int r9a,r9b;
				int errx = 0;
				
				r2 = stat(targetfile,&target_stat);
				
				switch (r2) {
				case syscall_error /* -1 */: {
				    int errcode = errno;
				    
				    printf ("%s (error: %s)\n",
					    targetfile,strerror(errcode));

				    
				    err=1;
				    continue;			  
				}
				    break;
				    
				case syscall_success /* 0 */:
				    mode = target_stat.st_mode & 07777;
				    				
				    orig_file_F   = fopen(targetfile,"r");
				    
				    if (! orig_file_F) {
					int errcode = errno;
					
					printf ("%s (error: %s)\n",
						targetfile,strerror(errcode));
					err = 1;
					continue;
				    }

				    fd = open(backup_file[config_file],
					      O_WRONLY | O_CREAT | O_EXCL, mode);
				    if (-1 == fd) {
					int errcode = errno;
					
					printf ("%s -> %s (error: %s)\n",
						targetfile,backup_file[config_file],
						strerror(errcode));
					
					fclose(orig_file_F);
					err = 1;
					continue;					
				    }
				    
				    backup_file_F = fdopen(fd,"w");
				    if (! backup_file_F) {
					int errcode = errno;
					
					printf ("%s -> %s (error: %s)\n",
						targetfile,backup_file[config_file],
						strerror(errcode));
					
					fclose(orig_file_F);
					close(fd);
					unlink(backup_file[config_file]);  /* Ignore error */
					
					err = 1;
					continue;					

				    }
				    
				    if (0 != copy_to_fh(orig_file_F,backup_file_F)) {
					fclose(orig_file_F);
					fclose(backup_file_F);
				    } 

				    r9a = fclose(orig_file_F);
				    switch(r9a) {
				    case EOF: errx = errno; err = 1; break;
				    case 0: break;
				    }

				    r9b = fclose(backup_file_F);
				    switch(r9b) {
				    case EOF: errx = errno; err = 1; break;
				    case 0:                 break;
				    }
				    
				    if (0 == r9a && 0 == r9b) {
					int errxa = 0;
					if (change_file_utime(backup_file[config_file],
							      &target_stat,&errxa)) {
					    
					    enum syscall_status r9c =
						elm_chown(backup_file[config_file],
							  target_stat.st_uid,
							  target_stat.st_gid,
							  &errxa);

					    switch (r9c) {
					    case syscall_error:
						DPRINT(Debug,27,(&Debug,"elm_chown of %s failed\n",
								 backup_file[config_file]));
						
						if (0 == geteuid())
						    goto error1;
						break;
					    case syscall_success:
						DPRINT(Debug,27,(&Debug,"elm_chown of %s succeed\n",
								 backup_file[config_file]));
						break;
					    }
					    
					    printf ("%s -> %s\n",
						    targetfile,backup_file[config_file]);
					} else {
					error1:
						    
					    printf ("%s -> %s (error: %s)\n",
						    targetfile,backup_file[config_file],
						    errxa ? strerror(errxa) : "n/a");
					    
					    /* Ignore error */
					}
				    } else {
					printf ("%s -> %s (error: %s)\n",
						targetfile,backup_file[config_file],strerror(errx));
					unlink(backup_file[config_file]);  /* Ignore error */
				    }
				    continue;
				}

			    }
				
			    printf ("%s\n", targetfile);
			    continue;
			}
			break;
		    }

		    r2 = stat(targetfile,&target_stat);

		    switch (r2) {
		    case syscall_error /* -1 */: {
			int errcode = errno;
			
			lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNoStat,
					  "File %.30s: %.40s"),
				  targetfile, strerror(errcode));
			err=1;
			continue;			  
		    }
			break;

		    case syscall_success /* 0 */:
			mode = target_stat.st_mode & 07777;
		    
			if (0 != (mode & 00002)) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileGlobWritable,
					      "Config file %s is globally writable, access mode %05o"), 
				      targetfile,mode);
			    err=1;	  
			}
		    
			if (0 == (mode & 00004)) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileGlobUnreadable,
					      "Config file %s is not globally readable, access mode %05o"),
				      targetfile,mode);
			    err=1;
			} else
			    mode |= 0444;
			break;
		    }

		    
		    
		    switch (config_file) {
		    case conf_charset_map:
		    case conf_mime_types_map:
		    case conf_terminal_map: 
		    case conf_iso2022_map:
#ifdef REMOTE_MBX
		    case conf_mail_services:
#endif
		    case conf_aliases_map: 
		    case conf_mlist_map:
		    case conf_hash_marks:
		    case conf_tagfilter_entities: 
			
			tmp = elm_message(FRM("%s.N"),targetfile);
			f = open_or_create(tmp);
			
			if (!f) {
			    int errcode = errno;
			    
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
					      "File %.50s is not writeable: %s"),
				      tmp, strerror(errcode));
			    
			    err=1;
			    free(tmp);
			    continue;
			}
			
			break;
			
		    case conf_elmrc:
		    case conf_LAST:
			break;
		    }
		    
		    
		    switch (config_file) {
			int errcode;
		    case conf_elmrc:
			r = write_rc(targetfile,NULL,1,
				     "ELMCONFWRITER", NULL,
				     version_buff);
			break;
			
		    case conf_charset_map:
			dump_charset_map(f,system_charset_map,NULL,
					 "ELMCONFWRITER",version_buff,0);
			r = 1;
			break;
			
		    case conf_mime_types_map:
			dump_mime_types_map(f,system_mimetypes_map,NULL,
					    "ELMCONFWRITER",version_buff);
			r = 1;
			break;
			
		    case conf_terminal_map: 
			dump_terminal_map(f,system_terminal_map,NULL,
					  "ELMCONFWRITER",version_buff);
			r = 1;
			break;
			
		    case conf_iso2022_map:
			dump_iso2022_map(f,system_iso2022_map_conf,NULL,
					 "ELMCONFWRITER",version_buff);
			r = 1;
			break;
			
#ifdef REMOTE_MBX
		    case conf_mail_services:
			r = dump_service_entries(f,system_mail_services_conf,NULL,
						 "ELMCONFWRITER",version_buff,
						 system_mail_services_cs,
						 system_mail_services_pl);

			/* XXX error message ? */
			break;
#endif
			
		    case conf_aliases_map: 
			r = dump_aliases_map(f,system_aliases_map,NULL,
					     "ELMCONFWRITER",version_buff,
					     system_aliases_cs,
					     system_aliases_pl,
					     NULL,
					     &errcode);

			if (!r) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
					      "File %.50s is not writeable: %s"),
				      tmp, strerror(errcode));
			    
			    err=1;
			}
			
			break;
		    
		    case conf_mlist_map:
			dump_mlist_conf(f,system_mlist_conf,NULL,
					"ELMCONFWRITER",version_buff,
					system_mlist_cs,system_mlist_pl);
			r = 1;
			break;
			
		    case conf_hash_marks: 
			r = dump_hash_mark_entries(f,system_hash_marks_conf,NULL,
						   "ELMCONFWRITER",version_buff,
						   system_hash_marks_cs,
						   system_hash_marks_pl,
						   SYSTEM_RC);

			/* XXX error message ? */
			
			break;

		    case conf_tagfilter_entities:
			r = dump_tagfilter_entities(f,system_tagfilter_entities_conf,NULL,
						    "ELMCONFWRITER",version_buff,
						    system_tagfilter_entities_cs,
						    system_tagfilter_entities_pl,
						    SYSTEM_RC,&errcode);

			if (!r && errcode) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
					      "File %.50s is not writeable: %s"),
				      tmp, strerror(errcode));
			    
			    err=1;
			}

			break;
		    case conf_LAST:
			break;
		    }
		
		    switch (config_file) {
			enum syscall_status r3;
			
		    case conf_charset_map:
		    case conf_mime_types_map:
		    case conf_terminal_map:
		    case conf_iso2022_map: 
#ifdef REMOTE_MBX
		    case conf_mail_services:
#endif
		    case conf_aliases_map: 
		    case conf_mlist_map:
		    case conf_hash_marks:
		    case conf_tagfilter_entities:
			
			if (EOF == fclose(f)) {
			    int errcode = errno;
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
					      "File %.50s is not writeable: %s"),
				      tmp, strerror(errcode));
			    
			    err=1;
			    unlink(tmp);   /* Ignore error */
			    free(tmp);
			    continue;
			}

			if (backup_file[config_file]) {
			    enum syscall_status r5 = link(targetfile,backup_file[config_file]);
			    
			    switch (r5) {
				int errcode;
			    case syscall_error /* -1 */:
				errcode = errno;
				lib_error(CATGETS(elm_msg_cat, MeSet, MeBackupFailed,
						  "Backup %s => %s failed: %s"),
					  targetfile,backup_file[config_file],strerror(errcode));
				err=1;
				unlink(tmp);   /* Ignore error */
				free(tmp);
				continue;
			    case syscall_success /* 0 */:  /* List backup */
				printf ("%s -> %s\n",
					targetfile,backup_file[config_file]);
				break;
			    }
			}
			
			r3 = rename(tmp,targetfile);

			switch (r3) {
			case syscall_error /* -1 */: {
			    int errcode = errno;
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotRenamed,
					      "Failed to rename temporary file to %.50s: %.30s"),
				      targetfile, strerror(errcode));
			    
			    err=1;
			    unlink(tmp);   /* Ignore error */						    
			    free(tmp);
			    continue;
			}
			    break;
			case syscall_success /* 0 */:
			    break;
			}
			
			free(tmp);
			
			break;
			
		    case conf_elmrc:
		    case conf_LAST:
			break;
		    }
		    
		    if (!r) {
			err=1;
			continue;
		    }

		    r4 = chmod(targetfile,mode);
		    switch (r4) {
			
		    case syscall_error /* -1 */: {
			int errcode = errno;
			lib_error(CATGETS(elm_msg_cat, MeSet, MeFileModeNotChanged,
					  "Failed to change file %s mode (new mode %05o):  %s"),
				  targetfile,mode, strerror(errcode));
			err=1;
		    }
			break;

		    case syscall_success /* 0 */:		    
			log_config(targetfile);
			break;
		    }
		}
		break;
	    case pass_LAST:
		break;
	    }
	}
    }

    if (elmrc_changes_F) {
	rewind(elmrc_changes_F);

#ifdef FTRUNCATE
	/* Not really necessary */
	if (-1 == ftruncate(fileno(elmrc_changes_F),0)) {
	    DPRINT(Debug,27,(&Debug,"ftruncate of %s failed\n",elmrc_changes));
	}
#endif

	print_history_changed(elmrc_changes_F,recorder);

    }

 fail:
    for (config_file = conf_elmrc; config_file < conf_LAST; config_file++) {
	if (backup_file[config_file]) {
	    free(backup_file[config_file]);
	    backup_file[config_file] = NULL;
	}
    }
    
    if (elmrc_changes_F) 
	fclose(elmrc_changes_F);

    if (err)
	lib_error(CATGETS(elm_msg_cat, MeSet, MeProgFailed,
			  "%s failed; exit code=%d"),
		  argv[0],err);

    return err;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
