static char rcsid[] = "@(#)$Id: showmsg_c.c,v 2.24 2022/06/26 16:55:21 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.24 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/showmsg_c.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** This is an interface for the showmsg command line.  The possible
    functions that could be invoked from the showmsg command line are
    almost as numerous as those from the main command line and include
    the following;

	   |    = pipe this message to command...
	   !    = call Unix command
	   <    = scan message for calendar info
	   b    = bounce (remail) message
	   d    = mark message for deletion
	   f    = forward message
	   g    = group reply
	   h    = redisplay this message from line #1, showing headers
	   <CR> = redisplay this message from line #1, weeding out headers
	   i,q  = move back to the index page (simply returns from function)
	   J    = move to body of next message
	   j,n  = move to body of next undeleted message
	   K    = move to body of previous message
	   k    = move to body of previous undeleted message
	   m    = mail a message out to someone
	   N    = move to body of next unread message
	   p    = print this (all tagged) message
	   r    = reply to this message
	   s    = save this message to a maibox/folder 
	   t    = tag this message
	   u    = undelete message
           v    = view attachments
	   x    = Exit Elm NOW 

    all commands not explicitly listed here are beeped at.  Use i)ndex
    to get back to the main index page, please.

    This function returns when it is ready to go back to the index
    page.
**/

#include "def_elm.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"ELM");

#if ANSI_C
showmsg_cmd showmsg_main_cmd;
#endif

int showmsg_main_cmd(mailbox,cmd, pager_page)
     struct MailboxView *mailbox;
     int cmd;
     struct pager_page *pager_page;
{
    struct menu_common MENU;
    
    set_mcommon_from_mbxview(&MENU,mailbox);
			     
    switch (cmd) {

    case '|' : {
	int current = get_current(mailbox);
	
	put_cmd_name(pager_page,TRUE,
		     CATGETS(elm_msg_cat, ElmSet, ElmPipe, "Pipe"));

	
	if (give_message_data(mailbox,current-1,
			      NULL,NULL,NULL,
			      NO_mime_parse))
	    do_pipe(mailbox, pager_page->root,pager_page->prompt_area);    
	else {
	    DPRINT(Debug,3,(&Debug, 
			    "give_message_data [%d] fails",current-1));
	}
	
    }
	return 256;

    }

    return cmd;

}

#if ANSI_C
subpage_simple_redraw sb_showmsg_update_border;
#endif
int sb_showmsg_update_border(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{
    int li,co,i;

    DPRINT(Debug,9, (&Debug, "sb_showmsg_update_border\n"));
    menu_get_sizes(ptr,&li, &co);

    menu_ClearScreen(ptr);
    menu_MoveCursor(ptr,0,0);

    menu_StartXX(ptr,pg_BOLD);
    for (i = 0; i < co; i++)
	menu_Writechar(ptr,'-');
    menu_EndXX(ptr,pg_BOLD);

    return 1;
}

void showmsg_set_border(pager_page)
     struct pager_page *pager_page;
{
    int   LINES, COLUMNS;	

    menu_get_sizes(pager_page->root,&LINES, &COLUMNS);

    if (! pager_page->border_line)
	pager_page->border_line = new_menu_subpage(pager_page->root,
						   LINES-5,1,sb_showmsg_update_border,
						   pager_page->PARAM
						   );
    else
	menu_subpage_relocate(pager_page->border_line,pager_page->root,
			      LINES-5,1);
}

#if ANSI_C
subpage_simple_redraw sb_showmsg_update_prompt;
#endif
int sb_showmsg_update_prompt(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{
    struct menu_common *mptr = mp_lookup_mcommon(list,elm_mp_menu);

    DPRINT(Debug,9, (&Debug, "sb_showmsg_update_prompt\n"));

    menu_ClearScreen(ptr);
    
    menu_PutLineX(ptr,0,0,
		  FRM("%S"),mcommon_give_item(mptr, m_Prompt));

    menu_PutLineX(ptr,0, 45, 
		  CATGETS(elm_msg_cat, ElmSet, ElmUseIToReturnIndex,
			  "(Use 'i' to return to index.)"));
    return 1;
}




static void set_prompt P_((struct pager_page *pager_page));
static void set_prompt(pager_page)
     struct pager_page *pager_page;
{
    int   LINES, COLUMNS;	

    menu_get_sizes(pager_page->root,&LINES, &COLUMNS);

    if (! pager_page->prompt_area)
	pager_page->prompt_area = new_menu_subpage(pager_page->root,
						   LINES-4,4,sb_showmsg_update_prompt,
						   pager_page->PARAM);
    else
	menu_subpage_relocate(pager_page->prompt_area,pager_page->root,
			      LINES-4,4);
}

void put_cmd_name (
#if ANSI_C
		   struct pager_page *pager_page,
		   int will_mangle,
		   const char * format, const char *msg, ...
#else
		   pager_page,will_mangle,format,msg, va_alist
#endif
		   )
#if !ANSI_C
     struct pager_page *pager_page;
     int will_mangle;
     const char * format;
     const char *msg;
     va_dcl
#endif
{
    struct string *cmd;
    int r;
    int r1;
    struct menu_common *mptr = mp_lookup_mcommon(pager_page->PARAM,
						 elm_mp_menu);

    va_list vl;
    
    Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
    cmd = elm_smessage(0,format,msg,vl);
    va_end(vl);

    r  = menu_resized(pager_page->root);
    r1 = menu_need_redraw(pager_page->root);

    if (r1) {
	menu_ClearScreen(pager_page->root);
    }

    if (r || r1 || will_mangle) {
	showmsg_set_border(pager_page);
	menu_trigger_redraw(pager_page->border_line);
    }

    if (pager_page->border_line &&
	menu_need_redraw(pager_page->border_line)) {
	DPRINT(Debug,1, (&Debug, "border line redraw???\n"));
	sb_showmsg_update_border(pager_page->border_line, 
			 pager_page->PARAM);
    }
    
    if (r || r1 || will_mangle) {
	set_prompt(pager_page);
	menu_trigger_redraw(pager_page->prompt_area);
    }

    if (pager_page->border_line &&
	menu_need_redraw(pager_page->border_line)) {
	DPRINT(Debug,1, (&Debug, "prompt area redraw???\n"));
	sb_showmsg_update_border(pager_page->border_line, 
			 pager_page->PARAM);
    }

    if (pager_page->prompt_area &&
	menu_need_redraw(pager_page->prompt_area)) {
	DPRINT(Debug,1, (&Debug, "prompt area redraw???\n"));
	sb_showmsg_update_prompt(pager_page->prompt_area, 
			 pager_page->PARAM);
    }

    if (pager_page->prompt_area) {
	menu_PutLineX(pager_page->prompt_area,
		      0,0,
		      FRM("%S"),mcommon_give_item(mptr, m_Prompt));
	
	menu_Write_to_screen(pager_page->prompt_area,
			     FRM("%S"),cmd);
    }

    free_string(&cmd);
}

int process_showmsg_cmd(command, mailbox, XX, aview, pager_page, flags, pager_cmds)
     int command;
     struct MailboxView *mailbox;
     showmsg_cmd   XX;
     struct AliasView *aview;
     struct pager_page *pager_page;
     int flags;
     struct elm_commands *pager_cmds;  
{
    int     i, intbuf;		/* for dummy parameters...etc */

    struct string * msg_line = NULL;
    
    int LINES, COLUMNS;

    menu_get_sizes(pager_page->root, &LINES, &COLUMNS);   
    
    while (TRUE) {
	struct menu_common MENU;
    
	set_mcommon_from_mbxview(&MENU,mailbox);

	clear_error();

	command = XX(mailbox,command,pager_page);  

	switch (command) {

	case 0:
	    return 0;       /* QUIT loop */
	case 256:
	    break;         /* DONE */

	case '?':
	    if (pager_cmds) {
		int ch;
		
		 put_cmd_name(pager_page,TRUE,FRM("?"));

		 ch = help_generic(pager_cmds,TRUE,
				   pager_page->root,
				   pager_page->prompt_area);
		 
		 if (EOF == ch)
		     return ch;
	    }
	    break;

	case '!' : 
		put_cmd_name(pager_page,TRUE,
			     CATGETS(elm_msg_cat, ElmSet, ElmSystemCall,
				     "System call"));

		if (ALLOW_subshell)
		    subshell(mailbox,pager_page->root,pager_page->prompt_area);
		else
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmNoSubshellCommand,
				      "Subshell command isn't configured in this version of ELM."));
		
		break;


	case '<' : { 
	      int current = get_current(mailbox);

	      if (ENABLE_calendar) {
		  put_cmd_name(pager_page,FALSE,
			       CATGETS(elm_msg_cat, ElmSet, ElmScanForCalendarCmd,
				       "Scan messages for calendar entries"));
		  if (give_message_data(mailbox,current-1,
					NULL,NULL,NULL,
					/* !!! */
					NO_mime_parse))
		      /* FIX: mime structure is not currently used */
		      scan_calendar(mailbox);     
		  else {
		      DPRINT(Debug,3,(&Debug, 
				      "give_message_data [%d] fails",current-1));
		  }

	      } else {
		  
		  if (msg_line)
		      free_string(&msg_line);
		  msg_line =
		    format_string(CATGETS(elm_msg_cat, ElmSet, ElmCantScanForCalendar,
					  "Can't scan for calendar entries!"));
	      }
	}
	    break;

	case 'b' : {
	      int current = get_current(mailbox);
	      struct header_rec *hdr;
	      FILE              *F;

	      if (flags & SHOWMSG_NOMAIL)
		  goto bad_cmd;

	      put_cmd_name(pager_page,TRUE,
			   CATGETS(elm_msg_cat, ElmSet, ElmBounceMessage,
				   "Bounce message"));

	      if (give_message_data(mailbox,current-1,
				    &hdr,&F,NULL,
				    /* !!! */
				    NO_mime_parse))
		  remail(hdr,F, aview, 
			 pager_page->root,
			 pager_page->prompt_area);
	      else {
		  DPRINT(Debug,3,(&Debug, 
				  "give_message_data [%d] fails",current-1));
	      }

	  }
	      break;
	      
	  case ctrl('E'): {

	      put_cmd_name(pager_page,TRUE,
			   CATGETS(elm_msg_cat, ElmSet, ElmExtractPGPKey,
				   "Extract PGP public key"));
	      
	      if (use_PGP) {	     
		  int current = get_current(mailbox);
		  struct header_rec *hdr;
		  FILE *F;
	      
	      
		  if (give_message_data(mailbox,current-1,
					&hdr,&F,NULL,
					/* !!! */
					NO_mime_parse)) { 
		      if (hdr->pgp & PGP_PUBLIC_KEY) {
			  /* FIX: mime structure is not currently used */
			  
			  pgp_extract_public_key (hdr,F, 
						  pager_page->root);  
			  
		      } else {
			  
			  if (msg_line)
			      free_string(&msg_line);
			  msg_line =
			      format_string(CATGETS(elm_msg_cat, ElmSet,
						    ElmNoKeysOnMessage,
						    "This message does not contain any keys!"));
			  
		      }
		  } else {
		      DPRINT(Debug,3,(&Debug, 
				      "give_message_data [%d] fails",current-1));
		      
		  }
	      } else {
		  lib_error(CATGETS(elm_msg_cat, ElmSet, 
				    ElmNoPgpConfigured,
				    "PGP isn't configured in this version of ELM."));
	      }
	  }
	      break;

	  case ctrl('F'):
	      put_cmd_name(pager_page,TRUE,
			   CATGETS(elm_msg_cat, ElmSet,
				   ElmPassphraseForget,
				   "Forget passphrase"));
	      
	      forget_passphrase();
	    break;

	    case 'f' : {
		int current = get_current(mailbox);

		if (flags & SHOWMSG_NOMAIL)
		    goto bad_cmd;
		
		put_cmd_name(pager_page,TRUE,
			     CATGETS(elm_msg_cat, ElmSet, ElmForwardMessage,
				     "Forward message"));
		
		forward(current-1,mailbox, aview, 
			pager_page->root,
			pager_page->prompt_area,
			NULL);		

	    }
		break;

	    case 'g' : {
		int current = get_current(mailbox);
		
		if (flags & SHOWMSG_NOMAIL)
		    goto bad_cmd;

		put_cmd_name(pager_page,TRUE,
			     CATGETS(elm_msg_cat, ElmSet, ElmGroupReply,
				     "Group reply"));

		reply_to_everyone(current-1,mailbox, aview, 
				  pager_page->root,
				  pager_page->prompt_area,
				  NULL);
		
	    }
	    break;

	  case 'h' : 
	      {
		  int current = get_current(mailbox);

		  struct header_rec *hdr;
		  FILE *F;
		  
		  if (give_message_data(mailbox,current-1,
					  &hdr,&F,NULL,
					  mime_parse_routine)) {

		      /* it's been read now! */
		      if (ison_status_message(mailbox,current-1,status_basic,NEW))
			  clearf_status_message(mailbox,current-1,status_basic,NEW); 
		      
		      /* it's been read now! */
		      if (ison_status_message(mailbox,current-1,status_basic,UNREAD))
			  clearf_status_message(mailbox,current-1,status_basic,UNREAD); 
		      		      
		      if (elm_filter) { 
			  elm_filter = 0; 
			  intbuf = show_msg(hdr,F,current,
					    get_message_count(mailbox),
					    pager_page,pager_cmds);
			  elm_filter = 1;
			  return(intbuf);
		      } else
			  return(show_msg(hdr,F,current,
					  get_message_count(mailbox),
					  pager_page,
					  pager_cmds)); 
		  }  else {
		      DPRINT(Debug,3,(&Debug, 
				      "give_message_data [%d] fails",current-1));
		  }
	      }
	      break;

	    case 'q' :
	    case 'i' : {
		clear_error();		/* zero out pending msg   */
		return(0);		/* avoid <return> looping */
	    }

	case 'I': {
	    int current = get_current(mailbox);
	    int ch;

	    if (flags & SHOWMSG_NOMAIL)
		goto bad_cmd;

	    put_cmd_name(pager_page,TRUE,
			 CATGETS(elm_msg_cat, ElmSet,
				 ElmMailingListInfo,
				 "Mailing List Info"));

	    ch = list_info(current-1,mailbox, aview, pager_page->root,
			   pager_page->prompt_area);	
	    
	    if (EOF == ch)
		return ch;
	    if ('i' == ch) {
		clear_error();		/* zero out pending msg   */
		return(0);		
	    }

	}
	    break;
		
	  next_undel_msg :	/* a target for resolve mode actions */

	  case ' ' :
	  case 'n' : {
		    
	      int current = get_current(mailbox);

	      if((i=next_message(current-1, TRUE, &MENU)) != -1) {
		  
		  current = i+1;
		  set_current(mailbox,current);
		  
		  return(mbx_show_msg(mailbox,current-1,pager_page,
				      pager_cmds));
	      }
	  }
	  return(0);


	case ctrl('K'):

	    if (flags & SHOWMSG_NOMAIL)
		goto bad_cmd;

	    put_cmd_name(pager_page,TRUE,
			 CATGETS(elm_msg_cat, ElmSet,
				 ElmMailPGPPublicKey,
				 "Mail PGP public key"));

	    if (use_PGP) {
		pgp_mail_public_key(mailbox, aview, 
				    pager_page->root, 
				    pager_page->prompt_area);
	    } else {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNoPgpConfigured,
				  "PGP isn't configured in this version of ELM."));
	    }
	    break;

	case ctrl('N'):  {
	    int current = get_current(mailbox);

		  for (i = next_message(current-1, TRUE, &MENU);
			    i != -1;
		       i = next_message(i, TRUE, &MENU)) {
		      if(ison_status_message(mailbox,i,status_basic,UNREAD))
			  break;
		  }
		  if (i != -1) {
		      int current = i+1;

		      set_current(mailbox,current);

		      return(mbx_show_msg(mailbox,current-1,pager_page,
					  pager_cmds));

		  }
		  else 
		      return (0);
	  }

	case 'm' : {
	    int current = get_current(mailbox);
		
 		if (flags & SHOWMSG_NOMAIL)
		    goto bad_cmd;
		
		put_cmd_name(pager_page,TRUE,
			     CATGETS(elm_msg_cat, ElmSet, ElmMailMessage,
				     "Mail message"));
		
		send_msg_l(current-1,
			   NULL,NULL,NULL, MAIL_EDIT_MSG, 
			   mailbox, aview,
			   pager_page->root, 
			   pager_page->prompt_area,
			   NULL);		
	    }
		break;

	case 'O': 
	    put_cmd_name(pager_page,TRUE,
			 CATGETS(elm_msg_cat, ElmSet,	  
				 ElmOverrideCharset,
				 "Override charset"));

	    OverrideCharset(mailbox, 
			    pager_page->root, 
			    pager_page->prompt_area,
			    NULL);
	    break;

	case 'p' : {
	    int current = get_current(mailbox);
	    
	    put_cmd_name(pager_page,confirm_print,
			 CATGETS(elm_msg_cat, ElmSet, ElmPrintMessage,
				 "Print message"));
	    if (give_message_data(mailbox,current-1,
				  NULL,NULL,NULL,
				  /* !!! */
				  NO_mime_parse))
		/* FIX: mime structure is not used */
		(void) print_msg(FALSE, mailbox, 
				 pager_page->root,
				 pager_page->prompt_area);    
	    else {
		DPRINT(Debug,3,(&Debug, 
				"give_message_data [%d] fails",current-1));
		
	    }
	}
	    break;

	case 'P' : {
		int current = get_current(mailbox);

		put_cmd_name(pager_page,confirm_print,
			     CATGETS(elm_msg_cat, ElmSet, 
					    ElmPrintTextMessage,
				     "Print text of message"));


		if (give_message_data(mailbox,current-1,
					NULL,NULL,NULL,
					mime_parse_routine))
		    print_text(FALSE, mailbox, pager_page->root,
			       NULL, pager_page->prompt_area);  
		else {
		    DPRINT(Debug,3,(&Debug, 
				    "give_message_data [%d] fails",current-1));
		    
		}

	}
	    break;

	case 'r' : {
	    int current = get_current(mailbox);

	    if (flags & SHOWMSG_NOMAIL)
		goto bad_cmd;
		
	    put_cmd_name(pager_page,TRUE,
			 CATGETS(elm_msg_cat, ElmSet, ElmReplyToMessage,
				 "Reply to message"));

	    reply(current-1,mailbox, aview, 
		  pager_page->root,
		  pager_page->prompt_area,
		  NULL);

	}
	    break;

	case 'R' : {
		int current = get_current(mailbox);
		int ch;

		if (flags & SHOWMSG_NOMAIL)
		    goto bad_cmd;
		
		put_cmd_name(pager_page,TRUE,
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmGenericReply,
				     "Generic Reply"));

		ch = generic_reply(current-1,mailbox, aview, 
				   pager_page->root,
				   pager_page->prompt_area);	
		
		if (EOF == ch)
		    return ch;
		if ('i' == ch) {
		    clear_error();		/* zero out pending msg   */
		    return(0);		
		}		
	}
	    break;

	case '>' :
	case 'C' :
	case 's' : {
	    int current = get_current(mailbox);
		
		if (command != 'C')
		    put_cmd_name(pager_page,TRUE,
				 CATGETS(elm_msg_cat, ElmSet, ElmCapSave, "Save"));
		else
		    put_cmd_name(pager_page,TRUE,
				 CATGETS(elm_msg_cat, ElmSet, ElmCapCopy, "Copy"));

		if (give_message_data(mailbox,current-1,
					NULL,NULL,NULL,
				      NO_mime_parse)) {

		    save(TRUE, (command != 'C'),0,
			 mailbox, aview, 
			 pager_page->root, 
			 pager_page->prompt_area,
			 NULL);

		    
		} else {
		    DPRINT(Debug,3,(&Debug, 
				    "give_message_data [%d] fails",current-1));
		    
		}

		if (resolve_mode && command != 'C')
		    goto next_undel_msg;
	}
	    break;

	case 'S' : {
	    int current = get_current(mailbox);
		
		put_cmd_name(pager_page,TRUE,
			     CATGETS(elm_msg_cat, ElmSet, ElmSaveText,
				     "Save text"));
		
		if (give_message_data(mailbox,current-1,
					NULL,NULL,NULL,
				      mime_parse_routine)) {

		    save(TRUE, 0,1, mailbox, aview,
			 pager_page->root, 
			 pager_page->prompt_area,
			 NULL);
		    		    
		} else {
		    DPRINT(Debug,3,(&Debug, 
				    "give_message_data [%d] fails",current-1));
		    
		}
	}
	    break;

	  case 'v' : {
	      struct header_rec *hdr;
	      FILE *F;
	      int current = get_current(mailbox);

	      if (give_message_data(mailbox,current-1,
				    &hdr,&F,NULL,mime_parse_routine)) {
		  if ((hdr->status & MIME_MESSAGE)) {
		      
		      const struct remote_server * remote_server =
			  give_message_remote_server(mailbox,current-1);

		      mime_warnings(hdr);
		      
		      attach_menu(F, & (hdr->mime_rec), NULL, 
				  hdr->header_charset, NULL,hdr, aview,mailbox,
				  & (hdr->header_error),
				  remote_server);
		      
		  }
		  else
		      error ("This is not a MIME message!");
	      } else {
		  DPRINT(Debug,3,(&Debug, 
				  "give_message_data [%d] fails",current-1));
	      }
	      return(0);
	  }

	case 'V' : {

	    int current = get_current(mailbox);

	    if (flags & SHOWMSG_NOMAIL)
		goto bad_cmd;

	    put_cmd_name(pager_page,TRUE,
			 CATGETS(elm_msg_cat, ElmSet,
				 ElmViewDigest,
				 "View digest as mailbox"));
	    	    
	    FlushBuffer();
	    
	    if (give_message_data(mailbox,current-1,
				  NULL,NULL,NULL,mime_parse_routine)) {
		
		int ch = ViewDigest(mailbox,aview,pager_page->root);
		
		current = get_current(mailbox);

		if (EOF == ch)
		    return ch;
		if ('i' == ch) {
		    clear_error();		/* zero out pending msg   */
		    return(0);		
		}		
		
	    } else {
		DPRINT(Debug,3,(&Debug, 
				"give_message_data [%d] fails",current-1));
		
	    }	    
	}
	    break;

	case 'X' :
	    put_cmd_name(pager_page,TRUE,
			 CATGETS(elm_msg_cat, ElmSet, ElmQuickExit,
				 "Quick Exit"));

	    /* Command name will be reprinted */
	    {
		struct menu_common *mptr = mp_lookup_mcommon(pager_page->PARAM,
							     elm_mp_menu);
		menu_PutLineX(pager_page->prompt_area,0,0,
			      FRM("%S"),mcommon_give_item(mptr, m_Prompt));	    
	    }
	    return 'X';   /* Call caller */


	case 'x' : 
		put_cmd_name(pager_page,TRUE,
			     CATGETS(elm_msg_cat, ElmSet, ElmExit, "Exit"));
		
		/* Command name will be reprinted */
		{
		    struct menu_common *mptr = mp_lookup_mcommon(
								 pager_page->PARAM,
								 elm_mp_menu);
		    menu_PutLineX(pager_page->prompt_area,0,0,
				  FRM("%S"),mcommon_give_item(mptr, m_Prompt));
		}
		return 'x';    /* Call caller */
		
	    
	case ctrl('J'):
	case ctrl('M'):  {
	    int current = get_current(mailbox);

	    return(mbx_show_msg(mailbox,current-1,pager_page,
				pager_cmds));			

	}

	case EOF    : 
	    return EOF;   /* FAILURE */
	    
	default  : {
	    enum process_motion_actions C
		= process_motion_command(command,&MENU,&msg_line);
	    
	    if (motion_done == C)
		break;

	    switch(C) {

	    case motion_view: {
		int current = get_current(mailbox);

		return(mbx_show_msg(mailbox,current-1,pager_page,
				    pager_cmds));			
	    }
		
	    case  motion_exit_loop:
		return 0;

	    case motion_EOF:
		return EOF;

	    case motion_done:
		/* Not used */
		break;
	    }
	}


	bad_cmd:
	    putchar((char) 007);	/* BEEP! */
	    break;
	}

	/* whole screen */
	if (menu_resized(pager_page->root)) {
	    menu_get_sizes(pager_page->root, &LINES, &COLUMNS);   
	    menu_trigger_redraw(pager_page->root);
	}
	if (menu_need_redraw(pager_page->root)) {
	    menu_ClearScreen(pager_page->root);
	    menu_redraw_children(pager_page->root);
	}

	/* display prompt */
	if (pager_page->prompt_area) {
	    /* clear what was left over from previous command 
	     * and display last generated message.
	     */
	    
	    /* create areas if missing */
	    if (!pager_page->border_line)
		sb_showmsg_update_border(pager_page->border_line,
				 pager_page->PARAM);


	    if (menu_need_redraw(pager_page->border_line)) {
		DPRINT(Debug,1, (&Debug, "border line redraw???\n"));
		sb_showmsg_update_border(pager_page->border_line, 
				 pager_page->PARAM);
	    }

	    if (menu_need_redraw(pager_page->prompt_area)) {
		DPRINT(Debug,1, (&Debug, "prompt area redraw???\n"));
	    }
	    /* Need clear previous command from prompt */
	    sb_showmsg_update_prompt(pager_page->prompt_area, 
				 pager_page->PARAM);
	    

	    if (msg_line)
		menu_print_center(pager_page->root,LINES-1,msg_line);
	    
	    {
		struct menu_common *mptr = mp_lookup_mcommon(
							     pager_page->PARAM,
							     elm_mp_menu);

		menu_PutLineX(pager_page->prompt_area,0,0,
			      FRM("%S"),mcommon_give_item(mptr, m_Prompt));
	    }

	    command = menu_ReadCh(pager_page->prompt_area,
				  'i'|READCH_CURSOR);   /* get next command from user */


	  } else {
	      /* display bottom line prompt with last generated message */
	      menu_MoveCursor(pager_page->root,LINES-1, 0);
	      menu_CleartoEOS(pager_page->root);

	      menu_StartXX(pager_page->root,pg_STANDOUT);
	      if (msg_line)
		  menu_Write_to_screen(pager_page->root,
				       CATGETS(elm_msg_cat, ElmSet, ElmCommandLine,
					       "%S Command ('i' to return to index): "),
				       msg_line);
	      else
		  menu_Write_to_screen(pager_page->root,
				       CATGETS(elm_msg_cat, ElmSet, ElmCommandLineNoErr,
					       "Command ('i' to return to index): "));

	      menu_EndXX(pager_page->root,pg_STANDOUT);

	      command = menu_ReadCh(pager_page->root,
				    'i'|READCH_CURSOR);   /* get next command from user */

	  }

	if (msg_line)
	    free_string(& msg_line); /* null last generated message */
    }

}




/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
