static char rcsid[] = "@(#)$Id: showmsg.c,v 2.33 2022/08/19 16:13:05 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.33 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/showmsg.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** This file contains all the routines needed to display the specified
    message.
**/

#include "def_elm.h"
#include "s_elm.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"ui");
DEBUG_VAR(DebugMime,__FILE__,"mime");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif       

void OverrideCharset(mailbox, page, prompt_area, header_area)
     struct MailboxView *mailbox;
     struct menu_context  *page;
     struct menu_context  *prompt_area;
     struct menu_context  *header_area;
{
    int tagged = 0;
    int current              = get_current(mailbox);
    struct header_rec * chdr = give_header(mailbox,current-1);

    char override_charset[40];
    int code;
    charset_t res = NULL;
    int mc,i;
    int delay_redraw = 0;
    int line;
    const char * MIME_name_o = NULL;
    const char * MIME_name_d = get_charset_MIME_name(display_charset);

    mc = get_message_count(mailbox);
    for (i=0; i < mc; i++) {
	if (ison_status_message(mailbox,i,status_basic,TAGGED)) {
	    tagged++;
	}
    }

    if (!tagged && chdr && chdr->override_charset &&
	(MIME_name_o = get_charset_MIME_name(chdr->override_charset)))
	strfcpy(override_charset,MIME_name_o, sizeof override_charset);
    else if (MIME_name_d)
	strfcpy(override_charset,MIME_name_d, sizeof override_charset);
    else
	strfcpy(override_charset,"none", sizeof override_charset);

 redraw:	    
    if (tagged > 1) 
	menu_print_format_center(prompt_area,1,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmOverrideTagged,
					 "Override charset of %d tagged messages. Use \"none\" to cancel override."),
				 tagged);
    else if (1 == tagged) 
	menu_print_format_center(prompt_area,1,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmOverrideTaggedOne,
					 "Override charset of tagged message. Use \"none\" to cancel override."));
    else if (chdr && chdr->override_charset)
	menu_print_format_center(prompt_area,1,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmOverrideCurrentOver,
					 "Override charset of current message. Use \"none\" to cancel override."));
    else
	menu_print_format_center(prompt_area,1,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmOverrideCurrent,
					 "Override charset of current message."));


    /* FIXME --optionally_enter*  should use prompt_area */
    line = menu_GetAbsLine(prompt_area,2);

    menu_PutLineX(page,line,0,CATGETS(elm_msg_cat, ElmSet, ElmOverrideCSprompt,
				      "Override charset: "));
    code = optionally_enter(override_charset,-1,-1,OE_REDRAW_MARK|
			    OE_SIG_CHAR /* Ctrl-C */,
			    sizeof override_charset, page);

    switch(code) {
    case REDRAW_MARK:
	menu_ClearScreen(page);   /* Reset possible redraw flag */
	delay_redraw++;   /* Can't trigger redraw yet... */

	goto redraw;
    case -1:    /* Interrupt */
	if (delay_redraw)
	    menu_trigger_redraw(page);
	menu_trigger_redraw(prompt_area);

	return;
    case 1:     /* EOF */
	if (delay_redraw)
	    menu_trigger_redraw(page);
	menu_trigger_redraw(prompt_area);

	return;
    case 0:
	/* OK */
	break;
    default:
	if (delay_redraw)
	    menu_trigger_redraw(page);
	menu_trigger_redraw(prompt_area);

	DPRINT(Debug,9,(&Debug, "optionally enter returned %d (unexpected)n",
			code));
	return;
    }

    if (delay_redraw)
	menu_trigger_redraw(page);
    menu_trigger_redraw(prompt_area);

    if (! override_charset[0])
	return;

    if (0 == strcmp(override_charset,"none"))
	res = NULL;
    else {
	int j;

	res = MIME_name_to_charset(override_charset,0);

	if (!res) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnknowCharset,
			      "Charset %s is unknown."),
		      override_charset);
	    
	    goto redraw;
	}

	j = charset_properties(res);
	
	if (! (j & CS_printable) ||
	    ! (j & CS_mapping)) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCharsetBadOverride,
			      "Can't use charset %s for override."),
		      override_charset);
	    goto redraw;
	}


    }


    if (tagged == 0) {
	tagged = 1;
	setf_status_message(mailbox,current-1,status_basic,TAGGED);
    }

    mc = get_message_count(mailbox);
    for (i=0; i < mc; i++) {	/* get each tagged msg */
	if (ison_status_message(mailbox,i,status_basic,TAGGED)) {
	    struct header_rec * hdr = give_header(mailbox,i);

	    hdr->override_charset = res;

	    /* If charset is overrided, it effects classification */
	    if (hdr->mime_parsed)
		mime_classify_media(&(hdr->mime_rec),hdr,
				    DISP_INLINE == hdr->mime_rec.disposition ? 
				    mime_classify_all :
				    /* Do not ask program on "Mailcap program selection"
				       if later part is skipped as attachment 
				    */
				    mime_classify_skip_mailcap);
	    
	    clearf_status_message(mailbox,i,status_basic,TAGGED);

	    if (header_area) {
		struct menu_common MENU;
		int vis;

		set_mcommon_from_mbxview(&MENU,mailbox);

		vis = compute_visible(i+1, &MENU);
		menu_header_status_update(header_area,vis-1);
	    }
	}
    }
}


/* Set *cancel = 1 if no showing message */
static void print_errors_1 P_((struct header_rec *hdr, 
			       int current, int message_count,
			       int *cancel, 
			       struct string *prompt,
			       struct string *view_text,
			       struct menu_context  *page,
			       int start_line,
			       struct header_errors *header_error,
			       int *cancel_ret
			       ));
static void print_errors_1(hdr,current,message_count,cancel,prompt,
			   view_text,page,start_line,header_error,
			   cancel_ret)
     struct header_rec *hdr;
     int current; 
     int message_count;
     int *cancel;
     struct string *prompt;
     struct string *view_text;
     struct menu_context  *page;     
     int start_line;
     struct header_errors *header_error;
     int *cancel_ret;
{

    int li,co;
    int redraw = 1;
    int last_line = 0;

    if (cancel)
	*cancel = 0;
    if (!header_error)
	return;

    if (cancel_ret)
	*cancel_ret = 0;

    menu_get_sizes(page,&li,&co);

    do {
	int defans = 'v';
	int ch = '\0';

    resize_mark:
	if (menu_resized(page)) {
	    menu_get_sizes(page,&li,&co);
	    redraw = 1;
	}

	if (menu_need_redraw(page) ||    
	    redraw) {
	    
	    int X = 1;
	    int mlen =  get_header_errors_count(header_error);
	    int prompt_line = li - 3;
	    int m;

	    redraw = 0;

	    menu_ClearScreen(page);
	    
	    if (hdr && title_messages && current > 0) {
		struct string * T = title_text(hdr,current,message_count,co,
					       display_charset);
		
		menu_StartXX(page,pg_UNDERLINE);
		menu_PutLineX(page,0,0,FRM("%S"),T);
		menu_EndXX(page,pg_UNDERLINE);

		free_string(&T);
		
		if (hdr->subject)
		    menu_print_format_center(page,1,FRM("%S"),
					     hdr->subject);
		X = 3;
	    }
	    
	    menu_StartXX(page,pg_BOLD);
	    menu_print_format_center(page,X, 
				     CATGETS(elm_msg_cat,
					     MeSet, MeHaveParsingErrors,
					     "There is parsing errors on message"));
	    menu_EndXX(page,pg_BOLD);

	    X += 2;
	    
	    for (m = start_line; m < mlen; m++) {
		const struct string *S = 
		    get_header_errors_msg(header_error,m);
		int Slen = string_len(S);
		int ptr = 0;
				
		if (X+1 >= prompt_line)
		    break;
		
		do {
		    int visible_len;
		    struct string * S1 = curses_printable_clip(S,&ptr,Slen,
							       &visible_len,
							       co);
		    
		    if (S1) {
			menu_PutLineX(page,X,0,FRM("%S"),S1);
			X++;
			
			free_string(&S1);
		    } else 
			break;
		    
		    if (X+1 >= prompt_line)
			break;
		    
		} while (ptr < Slen);
		last_line = m;
		
		X++;
		
	    }

	    menu_PutLineX(page,prompt_line,0,
			  FRM("%S"),prompt);
	}

	menu_CleartoEOLN(page);   /* Clear current answer */
	
	menu_Write_to_screen(page,
			     FRM("%c%c"),defans,BACKSPACE);
	FlushBuffer();


	ch = menu_ReadCh(page,REDRAW_MARK|READCH_CURSOR|
			 READCH_resize|READCH_sig_char);
	
	if (ch == '\n') 
	    ch = defans;
	
	switch (ch) {
	case TERMCH_interrupt_char:	
	    if (cancel)
		*cancel = 1;
	    goto quit;

	case 'i':
	case 'q':
	case 'c':
	    if (cancel) {
		menu_Write_to_screen(page,
				     CATGETS(elm_msg_cat, MeSet,
					     MeErrorCancel,
					     "Cancel"));
		FlushBuffer();
		*cancel = 1;
		goto quit;
	    }
	    break;

	case RESIZE_MARK:
	    
	    DPRINT(Debug,4, (&Debug,"    .... resizing\n"));
	    goto resize_mark;

	case REDRAW_MARK:
	    redraw = 1;
	    menu_ClearScreen(page);
	    break;

	case '-':
	case LEFT_MARK:
	case PAGEUP_MARK: {
	    int L = last_line - start_line +1;

	    start_line -= L;
	    if (start_line < 0)
		start_line = 0;
	    
	    redraw = 1;
	}
	    break;
	
	case '+':
	case RIGHT_MARK:
	case PAGEDOWN_MARK: {
	    start_line = last_line +1;
	    
	    redraw = 1;
	}
	    break;

	case DOWN_MARK:
	    if (cancel && cancel_ret) {
		/* Go to next message */
		*cancel_ret = DOWN_MARK;
		*cancel = 1;
		goto quit;
	    }
	    break;

	case UP_MARK:
	    if (cancel && cancel_ret) {
		/* Go to previous message */
		*cancel_ret = UP_MARK;
		*cancel = 1;
		goto quit;
	    }
	    break;
	    
	case 'v':
	    menu_Write_to_screen(page,FRM("%S"),view_text);

	    FlushBuffer();
	    
	    if (cancel)
		*cancel = 0;
	    goto quit;
	}
    } while(1);

 quit:
    return;
}

/* Set *cancel = 1 if no showing message */
static void print_errors P_((struct header_rec *hdr, 
			     int current, int message_count,
			     int *cancel,
			     int *cancel_ret));
static void print_errors(hdr,current,message_count,cancel,
			 cancel_ret)
     struct header_rec *hdr;
     int current; 
     int message_count;
     int *cancel;
     int *cancel_ret;
{
    struct menu_context  *page = new_menu_context();

    struct string *prompt     = NULL;
    struct string *view_text  = NULL;

   
    if (cancel)
	 prompt = format_string(CATGETS(elm_msg_cat, MeSet, MeErrorPrompt1,
					"Press enter for default, v)iew message or c)ancel: "));
    else
	prompt = format_string(CATGETS(elm_msg_cat, MeSet, MeErrorPrompt2,
				       "Press enter to view message: "));

    view_text = format_string(CATGETS(elm_msg_cat, MeSet,
				      MeErrorView,
				      "View message"));

    print_errors_1(hdr,current,message_count,cancel,prompt,
		   view_text,page,0,hdr->header_error,
		   cancel_ret);

    free_string(&view_text);
    free_string(&prompt);
    erase_menu_context(&page);
}

void print_errors_att(header_error,start_line,cancel)
     struct header_errors *header_error;
     int start_line;
     int *cancel;
{
    struct menu_context  *page = new_menu_context();

    struct string *prompt     = NULL;
    struct string *view_text  = NULL;

   
    if (cancel)
	 prompt = format_string(CATGETS(elm_msg_cat, MeSet, MeErrorPrompt3,
					"Press enter for default, v)iew message structure or c)ancel: "));
    else
	prompt = format_string(CATGETS(elm_msg_cat, MeSet, MeErrorPrompt4,
				       "Press enter to view message structure: "));

    view_text = format_string(CATGETS(elm_msg_cat, MeSet,
				      MeErrorViewStructure,
				      "View message structure"));

    print_errors_1(NULL,0,0,cancel,prompt,
		   view_text,page,start_line,header_error,
		   NULL);

    free_string(&prompt);
    erase_menu_context(&page);
}


struct screen_parts2 {
    struct menu_context  * title_area;
    struct menu_context  * prompt_area;
};

static void set_prompt_screen P_((struct menu_context  *page, struct screen_parts2 *LOC,
				  int current));
static void set_prompt_screen(page,LOC,current)
     struct menu_context  *page; 
     struct screen_parts2 *LOC;
     int current;
{
    int   LINES, COLUMNS;	
    int C = 5;

    menu_get_sizes(page,&LINES, &COLUMNS);

    if (title_messages && current > 0) {
	C = 7;
    }

    /* 1)  Title part of screen */

    if (! LOC->title_area)
	LOC->title_area = new_menu_subpage(page,0,C,
					   subpage_simple_noredraw,NULL);
    else
	menu_subpage_relocate(LOC->title_area,page,0,C);


    /* 2)  Prompt part of screen */

    if (! LOC->prompt_area)
	LOC->prompt_area = new_menu_subpage(page,C,LINES-C,
					    subpage_simple_noredraw,NULL);
    else
	menu_subpage_relocate(LOC->title_area,page,C,LINES-C);
}

static void free_prompt_screen P_((struct screen_parts2 *LOC));
static void free_prompt_screen(LOC)
     struct screen_parts2 *LOC;
{
    erase_menu_context (&(LOC->title_area));
    erase_menu_context (&(LOC->prompt_area));    
}

static void redraw_prompt_screen P_((struct menu_context  *page));
static void redraw_prompt_screen(page)
     struct menu_context  *page; 
{
    menu_ClearScreen(page);

    /* Call refresh routines of children */
    menu_redraw_children(page);
    
    show_last_error();    
}


/* Returns 0 on quit */


struct update_params {
    struct prompt_programs_list  ** list; 
    int  *list_len;
    int *current_offset;
};

static void update_prompt P_((struct update_params *P,	   
			      int idx,
			      struct menu_context  *page));

enum use_metamail {
    no_metamail_available_ask = -3,
    no_metamail_available = -2,
    metamail_disabled = -1,
    no_metamail_ask   = 0,
    metamail_ask
};

enum prompt_have_other {
    prompt_space = -1,
    prompt_no_yn,
    prompt_yn,
    prompt_yn_other
};


static void prompt_Z P_((struct menu_context  *page,
			 enum use_metamail ask_metamail,
			 int view_parts,
			 enum prompt_have_other have_other));
static void prompt_Z(page,ask_metamail, view_parts,have_other)
     struct menu_context  *page;
     enum use_metamail ask_metamail; /* -1 == metamail disabled */
     int view_parts;
     enum prompt_have_other have_other; 
{
    int lin,col;

    menu_get_sizes(page, &lin,&col);   

    DPRINT(Debug,15,(&Debug,"prompt_Z: ask_metamail %d",
		     ask_metamail));
    switch (ask_metamail) {
    case no_metamail_available: DPRINT(Debug,15,(&Debug," no_metamail_available")); break;
    case metamail_disabled:  DPRINT(Debug,15,(&Debug," metamail_disabled")); break;
    case no_metamail_ask:    DPRINT(Debug,15,(&Debug," no_metamail_ask")); break;
    case no_metamail_available_ask: DPRINT(Debug,15,(&Debug," no_metamail_available_ask")); break;
    case metamail_ask:       DPRINT(Debug,15,(&Debug," metamail_ask")); break;
    }
    DPRINT(Debug,15,(&Debug," view_parts %d have_other %d",view_parts,have_other));    
    switch (have_other) {
    case prompt_space:     DPRINT(Debug,15,(&Debug," prompt_space"));    break;
    case prompt_no_yn:     DPRINT(Debug,15,(&Debug," prompt_no_yn"));    break;
    case prompt_yn:        DPRINT(Debug,15,(&Debug," prompt_yn"));       break;
    case prompt_yn_other:  DPRINT(Debug,15,(&Debug," prompt_yn_other")); break;
    }
    DPRINT(Debug,15,(&Debug,"\n"));
    
    switch (ask_metamail) {
    case metamail_disabled:
    case no_metamail_available:
	if (view_parts) 
	    menu_PutLineX(page,		      
			  lin-2,0,
			  CATGETS(elm_msg_cat, ElmSet, ElmMailcapPageB,	
				  "Press enter for default, v)iew parts or p)age mail: "));
	
	
	else
	    menu_PutLineX(page,		      
			  lin-2,0,
			  CATGETS(elm_msg_cat, ElmSet, ElmMailcapPageA,	   
				  "Press enter for default or p)age mail: "));    
	break;
	
    case metamail_ask:
	if (view_parts) 
	    menu_PutLineX(page,		      
			  lin-2,0,
			  CATGETS(elm_msg_cat, ElmSet, ElmMailcapPageC,
				  "Press enter for default, use m)etamail, v)iew parts or p)age mail: "));
	
	else
	    menu_PutLineX(page,		      
			  lin-2,0,
			  CATGETS(elm_msg_cat, ElmSet, ElmMailcapPage,
				  "Press enter for default, use m)etamail or p)age mail: "));    
	break;
	
    case no_metamail_ask:
    case no_metamail_available_ask:
	{
	struct string * message =
	    format_string(CATGETS(elm_msg_cat, ElmSet, ElmMailcapDefault,
				  "Press enter for default"));

	switch (have_other) {
	case prompt_space:
	    if (string_len(message) + 45 < col)
		elm_append_message(&message,
				   CATGETS(elm_msg_cat, ElmSet, ElmMailcapUseSpace,
					   " or use space"));
	    break;

	case prompt_no_yn:
	    break;
	    
	case prompt_yn_other:
	    if (string_len(message) + 45 < col) {
		elm_append_message(&message,
				   CATGETS(elm_msg_cat, ElmSet, ElmMailcapUseYNOther,
					   " or use %c/%c/o)ther"),
				   *def_ans_yes, *def_ans_no);
		break;
	    }
	    /* FALLTHRU */
	case  prompt_yn:
	    if (string_len(message) + 35 < col)	
		elm_append_message(&message,
				   CATGETS(elm_msg_cat, ElmSet, ElmMailcapUseYN,
					   " or use %c/%c"),
				   *def_ans_yes, *def_ans_no);
	    break;
	}

	/* metamail still possible */
	if (no_metamail_ask == ask_metamail && string_len(message) + 35 < col) 
	    elm_append_message(&message,
			       CATGETS(elm_msg_cat, ElmSet, ElmMailcapUseMetamail,
				       ", use m)etamail"));

		    
	if (view_parts && string_len(message) + 25 < col)
	    elm_append_message(&message,
			       CATGETS(elm_msg_cat, ElmSet, ElmMailcapViewParts,
				       ", v)iew parts"));

	if (string_len(message) + 12 < col)
	    elm_append_message(&message,
			       CATGETS(elm_msg_cat, ElmSet, ElmMailcapViewDoneTail,
				       " or d)one: "));
	else
	    elm_append_message(&message,FRM(": "));
				       
	menu_PutLineX(page,
		      lin-2,0,FRM("%S"),message);
	
	free_string(&message);
	
    }
	
	break;
    }	
}

static void check_prompt_screen P_((struct screen_parts2 *LOC,
				    struct update_params *P,
				    enum use_metamail ask_metamail,
				    int current,
				    int message_count,
				    struct header_rec *hdr,
				    int view_parts,
				    enum prompt_have_other have_other
				    ));
static void check_prompt_screen(LOC,P,ask_metamail,current,message_count,
				hdr,view_parts,have_other)
     struct screen_parts2 *LOC;
     struct update_params *P;
     enum use_metamail ask_metamail;          /* -1 == metamail disabled */
     int current;
     int message_count;
     struct header_rec *hdr;
     int view_parts;
     enum prompt_have_other have_other;
{

    int f = hdr->mime_rec.mime_flags;  /* NOTPLAIN_need_metamail           0x01
					  NOTPLAIN_need_mailcap            0x02
					  NOTPLAIN_canuse_mailcap          0x04
					  
					  NOTPLAIN_metamail_blacklisted    0x10
				       */

    
    DPRINT(Debug,11,(&Debug,
		     "check_prompt_screen: %s/%s mime_flags %d (%s) ask_metamail %d",
		     get_major_type_name(hdr->mime_rec.TYPE), 
		     get_subtype_name(hdr->mime_rec.TYPE),
		     f,mime_debug_classify_f(f),ask_metamail));

    switch (ask_metamail) {
    case no_metamail_available: DPRINT(Debug,15,(&Debug," no_metamail_available")); break;
    case metamail_disabled:  DPRINT(Debug,11,(&Debug," metamail_disabled")); break;
    case no_metamail_ask:    DPRINT(Debug,11,(&Debug," no_metamail_ask")); break;
    case no_metamail_available_ask: DPRINT(Debug,15,(&Debug," no_metamail_available_ask")); break;
    case metamail_ask:       DPRINT(Debug,11,(&Debug," metamail_ask")); break;
    }
    DPRINT(Debug,11,(&Debug," view_parts %d have_other %d",view_parts));
    switch (have_other) {
    case prompt_space:     DPRINT(Debug,11,(&Debug," prompt_space"));    break;
    case prompt_no_yn:     DPRINT(Debug,11,(&Debug," prompt_no_yn"));    break;
    case prompt_yn:        DPRINT(Debug,11,(&Debug," prompt_yn"));       break;
    case prompt_yn_other:  DPRINT(Debug,11,(&Debug," prompt_yn_other")); break;
    }
    DPRINT(Debug,11,(&Debug,"\n"));
    
    /* 1)  Title part of screen */
    if (menu_resized(LOC->title_area)) {
	DPRINT(Debug,1, (&Debug, "title area resized\n"));

    }

    if (menu_need_redraw(LOC->title_area)) {
	int X = 1;

	DPRINT(Debug,7, (&Debug, "title area redraw\n"));

	menu_ClearScreen(LOC->title_area);

	if (title_messages && current > 0) {
	    int li,co;
	    struct string *T;

	    menu_get_sizes(LOC->title_area,&li,&co);

	    X = 3;

	    T = title_text(hdr,current,message_count,
			   co,
			   display_charset);

	    menu_StartXX(LOC->title_area,pg_UNDERLINE);
	    menu_PutLineX(LOC->title_area,0,0,FRM("%S"),T);
	    menu_EndXX(LOC->title_area,pg_UNDERLINE);

	    free_string(&T);


	    if (hdr->subject)
		menu_print_format_center(LOC->title_area,1,
					 FRM("%S"),
					 hdr->subject);

	}

	switch (ask_metamail) {
	case no_metamail_available:     
	    menu_StartXX(LOC->title_area,pg_BOLD);
	    menu_print_format_center(LOC->title_area,X,
				     CATGETS(elm_msg_cat, ElmSet, ElmMetamailUnavailMenu,
					     "No metamail available"));
	    menu_EndXX(LOC->title_area,pg_BOLD);
	    
	    menu_PutLineX(LOC->title_area,X+2,0,
			  CATGETS(elm_msg_cat, ElmSet, ElmMetamailMenu1a,
				  "Viewing of this message requires metamail, but it can not be used."));
	    
	    menu_PutLineX(LOC->title_area,X+3,0,
			  CATGETS(elm_msg_cat, ElmSet, ElmMetamailMenu2a,
				  "Press 'p' to page message without metamail."));
	    break;
	    
	case metamail_disabled: 
	    menu_StartXX(LOC->title_area,pg_BOLD);

	    menu_print_format_center(LOC->title_area,X,
				     CATGETS(elm_msg_cat, ElmSet, ElmMetamailDisMenu,
					     "Metamail disabled"));
	    menu_EndXX(LOC->title_area,pg_BOLD);
	    
	    if (0 != (f & NOTPLAIN_canuse_mailcap) ||
		0 != (f & NOTPLAIN_need_mailcap)) {

		menu_PutLineX(LOC->title_area,X+2,0,
			      CATGETS(elm_msg_cat, ElmSet, ElmMailcapMenu1,
				      "Viewing of this message requires external programs (press 'y' to confirm.)"));
	    
		menu_PutLineX(LOC->title_area,X+3,0,
			      CATGETS(elm_msg_cat, ElmSet, ElmMailcapMenu2b,
				      "To refuse external program press 'n'."));
 
	    } else if (0 != (f & NOTPLAIN_need_metamail)) {

		menu_PutLineX(LOC->title_area,X+2,0,
			      CATGETS(elm_msg_cat, ElmSet, ElmMetamailMenu1a,
				      "Viewing of this message requires metamail, but it can not be used."));

		menu_PutLineX(LOC->title_area,X+3,0,
			  CATGETS(elm_msg_cat, ElmSet, ElmMetamailMenu2a,
				  "Press 'p' to page message without metamail."));
	    }	
	    break;

	case metamail_ask: 
	    menu_StartXX(LOC->title_area,pg_BOLD);

	    menu_print_format_center(LOC->title_area,X,
				     CATGETS(elm_msg_cat, ElmSet, ElmMetamailMenu,
					     "Metamail selection"));
	    menu_EndXX(LOC->title_area,pg_BOLD);

	    menu_PutLineX(LOC->title_area,X+2,0,
			   CATGETS(elm_msg_cat, ElmSet, ElmMetamailMenu1,
				   "Viewing of this message requires metamail. Press 'm' to select metamail."));
	    menu_PutLineX(LOC->title_area,X+3,0,
			  CATGETS(elm_msg_cat, ElmSet, ElmMetamailMenu2,
				  "Or you can p)age message without metamail."));	    	
	    break;
	    
	case no_metamail_ask:
	case no_metamail_available_ask: 

	    menu_StartXX(LOC->title_area,pg_BOLD);

	    menu_print_format_center(LOC->title_area,X,
				     CATGETS(elm_msg_cat, ElmSet, ElmMailcapMenu,
					     "Mailcap program selection"));
	    menu_EndXX(LOC->title_area,pg_BOLD);


	    menu_PutLineX(LOC->title_area,X+2,0,
			  CATGETS(elm_msg_cat, ElmSet, ElmMailcapMenu1,
				  "Viewing of this message requires external programs (press 'y' to confirm.)"));

	    if (no_metamail_ask == ask_metamail)	    
		menu_PutLineX(LOC->title_area,X+3,0,
			      CATGETS(elm_msg_cat, ElmSet, ElmMailcapMenu2,
				      "To refuse external program press 'n'. Also you can page message with m)etamail."));
	    else
		menu_PutLineX(LOC->title_area,X+3,0,
			      CATGETS(elm_msg_cat, ElmSet, ElmMailcapMenu2e,
				      "To refuse external program press 'n'. "));
		    	
	    break;
	    
	}
    }

    /* 2)  Prompt part of screen */

    if (menu_resized(LOC->prompt_area)) {
	DPRINT(Debug,1, (&Debug, "prompt area resized\n"));

    }

    if (menu_need_redraw(LOC->prompt_area)) {

	DPRINT(Debug,7, (&Debug, "prompt area redraw\n"));

	update_prompt(P,-1, LOC->prompt_area);	

	prompt_Z(LOC->prompt_area,ask_metamail,view_parts,have_other);
    }
}

static int give_line_position P_((struct prompt_programs_list  * list,
				  int                   list_len,
				  int                   idx,
				  int                   start_list_position));
static int give_line_position(list,list_len,idx, start_list_position)
    struct prompt_programs_list  * list;
    int                   list_len;
    int                   idx;
    int                   start_list_position;
{
    int r = -1;
    int X1;
    
    if (idx < 0 || idx >= list_len || !list)
	goto failure;
    
    X1          = list[idx].list_position;
    r           = X1 - start_list_position +3;

 failure:

    DPRINT(Debug,16, (&Debug,
		      "give_line_position(%p,%d,%d,%d)=%d\n",
		      list,list_len,idx,start_list_position,r));
    
    return r;
}

static void update_prompt (P,idx,page)
     struct update_params *P;
     int idx;
     struct menu_context  *page; 
{
    int LINES, COLUMNS;
    struct prompt_programs_list  ** list = P->list;
    int  *list_len              = P->list_len;
    int *current_offset         = P->current_offset;
    int start_list_position = 0;
       
    menu_get_sizes(page,&LINES, &COLUMNS);

    if ( *current_offset >= 0 &&
	 *current_offset < *list_len)
	start_list_position = (*list)[*current_offset].list_position;
    else {
	DPRINT(Debug,15, (&Debug, 
			  "update_prompt: *current_offset=%d out of range\n",
			  *current_offset));
	
	return;
    }


    if (idx < 0) {
	int i;

	menu_ClearScreen(page);



	for (i = *current_offset; 
	     i >= 0 && 
		 i < *list_len;
	     i++) {

	    int X = give_line_position(*list,*list_len,i,start_list_position);

	    if (X < 0)
		continue;
	    
	    if (X >= LINES-3)
		break;
	    
	    update_prompt(P,i, page);
		 
	}

    } else if (idx >= 0 &&
	       idx >= *current_offset &&
	       idx < *list_len
	       ) {
	
	int X = give_line_position(*list,*list_len,idx,start_list_position);
	
	media_type_t            T;
	int print_idx = 1;

	if (! (*list)[idx].structure)
	    return;

	if (MIME_magic != (*list)[idx].structure->magic)
	    mime_panic(__FILE__,__LINE__,"update_prompt",
		       "Bad magic number (mimeinfo)");
	   	
	T =  (*list)[idx].structure->TYPE;
	
	if (X < 0)
	    return;
	
	if (X >= LINES-3)
	    return;

	
	if ((*list)[idx].printable_command) {

	    struct string * S  = NULL;
	    struct string * S1 = NULL;
	    int pos = 0;
	    int visible_len = 0;
	    
	    print_idx = 0;
	    
	    S =  format_string(CATGETS(elm_msg_cat, ElmSet, ElmUseMailcap,
				       "%2d Use '%s' to show %s/%s"),
			       idx,
			       (*list)[idx].printable_command,
			       get_major_type_name(T), get_subtype_name(T));
	    S1 = curses_printable_clip(S,&pos,string_len(S),
				       &visible_len,
				       COLUMNS-10);
	    
	    menu_PutLineX(page,X,0,FRM("%S ? "),S1);
	    menu_CleartoEOLN(page);
	    
	    (*list)[idx].col = visible_len+3;

	    if ((*list)[idx].structure->handler_data) {
		
		if (MIME_selector_magic != (*list)[idx].structure->handler_data->magic)
		    mime_panic(__FILE__,__LINE__,"update_prompt",
			       "Bad magic number (handler_data)");
		
		if ((*list)[idx].structure->handler_data->use_entry ||
		    (*list)[idx].selected) {
		    
		    if ((*list)[idx].structure->handler_data->use_entry)
			menu_PutLineX(page,X,(*list)[idx].col,
				      CATGETS(elm_msg_cat, ElmSet, ElmYesWord, 
					      "Yes."));
		    else
			menu_PutLineX(page,X,(*list)[idx].col,
				      CATGETS(elm_msg_cat, ElmSet, ElmNoWord, 
					      "No."));
		}
	    }
	    
	    free_string(&S1);
	    free_string(&S);

	    X += 2;
	}

	if (X >= LINES-3)
	    return;

	if ((*list)[idx].blacklisted_command && 
	    (*list)[idx].structure->metamail_blacklisted) {

	    struct string * S  = NULL;
	    struct string * S1 = NULL;
	    int pos = 0;
	    int visible_len = 0;
	    
	    if (print_idx)
		S = format_string(CATGETS(elm_msg_cat, ElmSet, ElmMetamailMailcapBL,
					  "%2d '%s' blacklisted for %s/%s (metamail)"),
				  idx,
				  (*list)[idx].blacklisted_command,
				  get_major_type_name(T), get_subtype_name(T));
	    else
		S = format_string(CATGETS(elm_msg_cat, ElmSet, ElmMetamailMailcapBL1,
					   "  '%s' blacklisted for %s/%s (metamail)"),
				  (*list)[idx].blacklisted_command,
				  get_major_type_name(T), get_subtype_name(T));

	    S1 = curses_printable_clip(S,&pos,string_len(S),
				       &visible_len,
				       COLUMNS-10);
	    
	    menu_PutLineX(page,X,0,FRM("%S "),S1);
	    menu_CleartoEOLN(page);

	    if (print_idx)
		(*list)[idx].col = visible_len+1;
	    
	    free_string(&S1);
	    free_string(&S);
	    
	    print_idx = 0;	    
	}

	if (X >= LINES-3)
	    return;
	
	if ((*list)[idx].need_metamail) {

	    struct string * S  = NULL;
	    struct string * S1 = NULL;
	    int pos = 0;
	    int visible_len = 0;

	    if ((*list)[idx].type_not_found) {
		if (print_idx)
		    S = format_string(CATGETS(elm_msg_cat, ElmSet, ElmMetamailNeed,
					      "%2d Unsupported %s/%s needs metamail"),
				      idx, get_major_type_name(T), get_subtype_name(T));
		else
		    S = format_string(CATGETS(elm_msg_cat, ElmSet, ElmMetamailNeed1,
					      "  Unsupported %s/%s needs metamail"),
				      get_major_type_name(T), get_subtype_name(T));
	    } else {
		if (print_idx)
		    S = format_string(CATGETS(elm_msg_cat, ElmSet, ElmMetamailNeed2,
					      "%2d Part %s/%s needs metamail"),
				      idx, get_major_type_name(T), get_subtype_name(T));
		else
		    S = format_string(CATGETS(elm_msg_cat, ElmSet, ElmMetamailNeed3,
					      "  Part %s/%s needs metamail"),
				      get_major_type_name(T), get_subtype_name(T));

	    }
		
	    S1 = curses_printable_clip(S,&pos,string_len(S),
				       &visible_len,
				       COLUMNS-10);
	    
	    menu_PutLineX(page,X,0,FRM("%S "),S1);
	    menu_CleartoEOLN(page);

	    if (print_idx)
		(*list)[idx].col = visible_len+1;
	    
	    free_string(&S1);
	    free_string(&S);
	}
	
    }

}

static char * truncated_cmd P_((char *cmd));
static char * truncated_cmd(cmd)
     char *cmd;
{
    char *x;

    if (strlen(cmd) > 15 && (x = strrchr(cmd,'/')) && x != cmd) {
	
	char * s = elm_message(FRM("...%s"),x);
	
	free(cmd);
	cmd = s;
    }
    
    return cmd;
}

char * check_mailcap_view_cmd(structure,is_ok)
     struct mimeinfo *structure;
     int *is_ok;
{
    char *cmd = NULL;
    int IS_OK = 0;
    
    if (MIME_magic != structure->magic)
	mime_panic(__FILE__,__LINE__,"check_mailcap_view_cmd",
		   "Bad magic number (mimeinfo)");

    if (structure->handler_data) {
	if (MIME_selector_magic != structure->handler_data->magic)
	    mime_panic(__FILE__,__LINE__,"check_mailcap_view_cmd",
		       "Bad magic number (handler_data)");
	
	cmd = mailcap_view_command(structure->handler_data->entry,
				   structure);
	
	if (cmd) {
	    const char **  trusted_programs = NULL;
	    int i;
	    
	    
	    trusted_programs = 
		give_dt_path_as_elems(&internal_mailcap_t_programs,
				      "internal-mailcap-trusted-programs");
	    
	    	   	    
	    if (trusted_programs) {
		for (i = 0; trusted_programs[i]; i++) {
		    if (0 == strcmp(cmd,
				    trusted_programs[i]))
			IS_OK++;
		}
	    }
	    
	    DPRINT(DebugMime,11,(&DebugMime,
				 "check_mailcap_view_cmd: view command %s%s\n",
				 cmd,IS_OK ? " (OK)" :""));
	    
	    cmd = truncated_cmd(cmd);
	}
    }

    if (is_ok)
	*is_ok = IS_OK;
    
    return cmd;
}



static void prompt_programs_1 P_((struct mimeinfo *structure,
				 struct prompt_programs_list  ** list, 
				  int  *list_len,
				  int *current_offset,
				  struct menu_context  *page,
				  int blacklist_only));

static void prompt_programs_1(structure, list, list_len, current_offset,
			      page,blacklist_only)
     struct mimeinfo *structure;
     struct prompt_programs_list  ** list; 
     int  *list_len;
     int *current_offset;
     struct menu_context  *page;
     int blacklist_only;
{
    int LINES, COLUMNS;

    menu_get_sizes(page, &LINES, &COLUMNS);   

    if (MIME_magic != structure->magic)
	mime_panic(__FILE__,__LINE__,"prompt_programs_1",
		   "Bad magic number (mimeinfo)");
    
    DPRINT(DebugMime,11,(&DebugMime,
			 "prompt_programs_1: type=%s/%s, mime_flags=%d (%s)%s\n",
			 get_major_type_name(structure->TYPE), 
			 get_subtype_name(structure->TYPE),
			 structure->mime_flags,
			 mime_debug_classify_f(structure->mime_flags),
			 blacklist_only ? " -- blacklist only" : ""));
			     
    if (structure->mime_flags) {
	char * cmd = NULL;
	char * blacklisted = NULL;
	int IS_OK = 0;
	int need_metamail  = 0;
        int type_not_found = 0;

	mime_t * skip_it = NULL;
		
	if (structure->handler_data) {

	    if (MIME_selector_magic != structure->handler_data->magic)
		mime_panic(__FILE__,__LINE__,"prompt_programs_1",
			   "Bad magic number (handler_data)");

	    if (structure->handler_data->entry &&
		!blacklist_only) {

		cmd = check_mailcap_view_cmd(structure,&IS_OK);
			    
	    }
	}
	
	if (0 != (structure->mime_flags & NOTPLAIN_need_metamail) &&
	    !cmd  &&
	    !blacklist_only) {
	    
	    need_metamail = 1;

	    DPRINT(DebugMime,11,(&DebugMime,
				 "prompt_programs_1: type=%s/%s, mime_flags=%d (%s) -- need metamail",
				 get_major_type_name(structure->TYPE), 
				 get_subtype_name(structure->TYPE),
				 structure->mime_flags,
				 mime_debug_classify_f(structure->mime_flags)));

	    if (0 != (structure->mime_flags & NOTPLAIN_type_not_found)) {
		DPRINT(DebugMime,11,(&DebugMime,", type not found"));
		type_not_found = 1;
	    }
	    DPRINT(DebugMime,11,(&DebugMime,"\n"));
	}

	if (structure->metamail_blacklisted) {

	    blacklisted = 
		metamail_mailcap_view_command(structure->metamail_blacklisted,
					      structure);	    

	    DPRINT(DebugMime,11,(&DebugMime,
				 "prompt_programs_1: metamail view command blacklisted %s\n",
				 blacklisted));
	    
	    blacklisted = truncated_cmd(blacklisted);
	}


	if (cmd || blacklisted || need_metamail) {
	    int i;

	    *list = safe_array_realloc(*list,
				       (*list_len +1),
				       sizeof ((*list)[0]));
	    
	    bzero((void *) &( (*list)[*list_len] ),
		      sizeof    ( (*list)[*list_len] ));
	    
	    
	    (*list)[*list_len].structure          = structure;
	    (*list)[*list_len].printable_command  = cmd;
	    (*list)[*list_len].blacklisted_command = blacklisted;
	    (*list)[*list_len].selected           = 0;
	    (*list)[*list_len].need_metamail      = need_metamail;
	    (*list)[*list_len].type_not_found     = type_not_found;
	    (*list)[*list_len].col                = COLUMNS-7;
	    
	    if (*list_len > 0) {
		(*list)[*list_len].list_position =
		    (*list)[(*list_len)-1].list_position + 3;

		if ((*list)[(*list_len)-1].printable_command &&
		    (*list)[(*list_len)-1].blacklisted_command)
		    (*list)[*list_len].list_position += 2;

		if (((*list)[(*list_len)-1].printable_command ||
		     (*list)[(*list_len)-1].blacklisted_command) &&
		    (*list)[(*list_len)-1].need_metamail)
		    (*list)[*list_len].list_position += 2;
		
	    } else
		(*list)[*list_len].list_position = 0;
	    
	    
	    if (IS_OK) {
		(*list)[*list_len].selected           = 1;

		if (MIME_magic != (*list)[*list_len].structure->magic)
		    mime_panic(__FILE__,__LINE__,"prompt_programs_1",
			       "Bad magic number (mimeinfo)");
		
		if (MIME_selector_magic != (*list)[*list_len].structure->handler_data->magic)
		    mime_panic(__FILE__,__LINE__,"prompt_programs_1",
			       "Bad magic number (handler_data)");
	    		
		(*list)[*list_len].structure->handler_data->use_entry = 1;
	    }
	    
	    i = (*list_len)++;
	    
	    {
		struct update_params P;
		
		P.list           = list;
		P.list_len       = list_len;
		P.current_offset = current_offset;
		
		update_prompt(&P,i,page);
	    }
	    
	} else {
	    DPRINT(DebugMime,11,(&DebugMime,
				 "prompt_programs_1: type=%s/%s, mime_flags=%d (%s) -- no prompt\n",
				 get_major_type_name(structure->TYPE), 
				 get_subtype_name(structure->TYPE),
				 structure->mime_flags,
				 mime_debug_classify_f(structure->mime_flags)));
	}



	/* Is alternative -- just process it */
	if (structure->handler_data) {

	    if (MIME_selector_magic != structure->handler_data->magic)
		mime_panic(__FILE__,__LINE__,"prompt_programs_1",
			   "Bad magic number (handler_data)");
	    
	    if (structure->handler_data->selected_alternative &&
		!blacklist_only) {

		if (structure->handler_data->selected_alternative->magic != MIME_magic)
		    mime_panic(__FILE__,__LINE__,"prompt_programs_1",
			       "Bad magic number (alternatice)");

		DPRINT(DebugMime,11,(&DebugMime,
				     "prompt_programs_1: type=%s/%s, mime_flags=%d (%s) -- processing alternative\n",
				     get_major_type_name(structure->TYPE), 
				     get_subtype_name(structure->TYPE),
				     structure->mime_flags,
				     mime_debug_classify_f(structure->mime_flags)));
		
		prompt_programs_1(structure->handler_data->selected_alternative,
				  list,list_len,current_offset, page,0);
		
		blacklist_only = 1;
		skip_it = structure->handler_data->selected_alternative;
	    } 
	}
	

	/* Process subparts recursively */
	if (structure->parser_data) {
	    int count = mime_parser_subparts(structure->parser_data);
	    int i;

	    if (count > 0) {
		DPRINT(DebugMime,11,(&DebugMime,
				     "prompt_programs_1: type=%s/%s, mime_flags=%d (%s) -- processing %d subparts\n",
				     get_major_type_name(structure->TYPE), 
				     get_subtype_name(structure->TYPE),
				     structure->mime_flags,
				     mime_debug_classify_f(structure->mime_flags),
				     count));
	    }
	    
	    for (i = 0; i < count; i++) {
		mime_t *att = 	mime_parser_index(structure->parser_data,i);

		if (att->magic != MIME_magic)
		    mime_panic(__FILE__,__LINE__,"prompt_programs_1",
			       "Bad magic number (subpart)");

		if (skip_it == att)
		    continue;

		DPRINT(DebugMime,11,(&DebugMime,
				     "prompt_programs_1: part %d/%d%s\n",
				     i,count,
				     blacklist_only ? " (blacklist only)" : ""));
		
		prompt_programs_1(att,
				  list,list_len,current_offset, page,
				  blacklist_only);
	    
	    }

	} else {
	    DPRINT(DebugMime,11,(&DebugMime,
				 "prompt_programs_1: type=%s/%s, mime_flags=%d (%s) -- no parser data\n",
				 get_major_type_name(structure->TYPE), 
				 get_subtype_name(structure->TYPE),
				 structure->mime_flags,
				 mime_debug_classify_f(structure->mime_flags)));
	}
	
    } else {
	DPRINT(DebugMime,11,(&DebugMime," -- no mime flags\n"));
    }
}


/* return TRUE if metamail is used instead,
   -1 to cancel
 */



static enum need_meta_state  prompt_programs P_((struct header_rec *hdr,
						 enum use_metamail ask_metamail,
						 int current, int message_count,
						 int *cancel_ret));
static enum need_meta_state  prompt_programs(hdr, ask_metamail, current, message_count,
					     cancel_ret)
     struct header_rec *hdr;
     enum use_metamail ask_metamail;                 /* -1 == metamail disabled */
     int current;
     int message_count;
     int *cancel_ret;
{
    enum need_meta_state  r = metamail_needed;
    struct mimeinfo *structure = &(hdr->mime_rec);

    struct prompt_programs_list  * list     = NULL;
    int                   list_len = 0;
    int                   offset   = 0;

    struct menu_context  *page = new_menu_context();

    struct screen_parts2 LOC = { NULL, NULL };
    struct update_params P;
    int q = 0;
    int ch = '\0';

    int mailcap_answered = 0;
    int need_redraw = 0;

    enum prompt_have_other have_other = prompt_no_yn;
    
    DPRINT(Debug,11,(&Debug,
		     "prompt_programs: ask_metamail=%d",
		     ask_metamail));
    switch (ask_metamail) {
    case no_metamail_available:     DPRINT(Debug,11,(&Debug," no_metamail_available")); break;
    case metamail_disabled:  DPRINT(Debug,11,(&Debug," metamail_disabled")); break;
    case no_metamail_ask:    DPRINT(Debug,11,(&Debug," no_metamail_ask")); break;
    case no_metamail_available_ask: DPRINT(Debug,15,(&Debug," no_metamail_available_ask")); break;

    case metamail_ask:       DPRINT(Debug,11,(&Debug," metamail_ask"));

	break;
    }
    DPRINT(Debug,11,(&Debug,", current/message_count=%d/%d",
		     current,message_count));    
    if (cancel_ret) {
	DPRINT(Debug,11,(&Debug,", have cancel_ret"));
	
	*cancel_ret              = 0;
    }    
    DPRINT(Debug,11,(&Debug,"\n"));
    
    if (MIME_magic != structure->magic)
	mime_panic(__FILE__,__LINE__,"prompt_programs",
		   "Bad magic number (mimeinfo)");
            
    if (ask_metamail > no_metamail_ask /* 0 */) {
	if (!have_metamail()) {
	    DPRINT(Debug,11,(&Debug,
			     "prompt_programs: Disabling metamail prompt\n"));
	    ask_metamail = no_metamail_available;
	}
    }
		
    P.list           = &list;
    P.list_len       = &list_len;
    P.current_offset = &offset;

    set_prompt_screen(page,&LOC,current);

    redraw_prompt_screen(page);
    
    prompt_programs_1(structure,&list,&list_len,&offset, LOC.prompt_area,0);

    check_prompt_screen(&LOC,&P,ask_metamail, current,message_count,hdr,
			cancel_ret != NULL,have_other);

    /* Check if all programs was trusted ... */
    
    while (q < list_len &&
	   list[q].selected &&
	   list[q].structure) {

	if (MIME_magic != list[q].structure->magic)
	    mime_panic(__FILE__,__LINE__,"prompt_programs",
		       "Bad magic number (mimeinfo)");
	
	if (list[q].structure->handler_data) {
	    
	    if (MIME_selector_magic != list[q].structure->handler_data->magic)
		mime_panic(__FILE__,__LINE__,"prompt_programs",
			   "Bad magic number (handler_data)");
	    
	    if (list[q].structure->handler_data->use_entry)
		q++;
	    else
		break;
	} else
	    break;
    }
    
    r = metamail_none;
    if (q >= list_len && !internal_mailcap_t_prompt && list_len > 0) 
	goto out;
    
    do {
	int defans;
	int sel;
	int start_list_position;
	
    resize_mark:
	sel = q;
	start_list_position = 0;
	
	if (offset >= 0 &&
	    offset < list_len)
	    start_list_position =  list[offset].list_position;

	defans = *def_ans_no;
	
	if (menu_resized(page)) {
	    DPRINT(Debug,4, (&Debug,"    .... resizing\n"));
	    
	    set_prompt_screen(page,&LOC,current);
	    need_redraw = 1;
	}
	
	if (menu_need_redraw(page) || need_redraw) {
	    DPRINT(Debug,4, (&Debug,"    .... redrawing\n"));

	    need_redraw = 0;	    
	    redraw_prompt_screen(page);		    	      
	}
	
	check_prompt_screen(&LOC,&P,ask_metamail,current, message_count,
			    hdr,cancel_ret != NULL,have_other);

	
	/* Search first not selected entry */
	while (sel < list_len &&
	       list[sel].selected)
	    sel++;

	if (q >= list_len) {

	    have_other = prompt_no_yn;
	    
	    prompt_Z(LOC.prompt_area,ask_metamail,
		     cancel_ret != NULL,have_other);

	    if (no_metamail_available_ask == ask_metamail) {

		defans = 'd';
		
	    } else if (ask_metamail < 0) {

				
		defans = 'p';

	    }  else if (ask_metamail > 0) {

		defans = mailcap_answered ? 'p' : 'm';

	    } else
		defans = 'd';
	    	    
	} else {

	    int X = give_line_position(list,list_len,q,start_list_position);
	    
	    enum prompt_have_other new_have_other = prompt_no_yn;
	    
	    int lin,col;
	    menu_get_sizes(LOC.prompt_area, &lin,&col);   
	    
	    if (X <= 1) {
		offset = q-1;
		if (offset < 0)
		    offset = 0;
		
		ch = REDRAW_MARK;
		menu_trigger_redraw(LOC.prompt_area);
		goto redraw1;
	    }
	    
	    if (X >= lin-3) {
		offset = q;
		
		ch = REDRAW_MARK;
		menu_trigger_redraw(LOC.prompt_area);
		goto redraw1;		 
	    }


	    if (! list[q].printable_command) {
		new_have_other =  prompt_space;
		
		defans = ' ';  /* Dummy if no question */
	    } else if (list[q].structure) {

		if (MIME_magic != list[q].structure->magic)
		    mime_panic(__FILE__,__LINE__,"prompt_programs",
			       "Bad magic number (mimeinfo)");

		if (list[q].structure->handler_data)  {
		    
		    if (MIME_selector_magic != list[q].structure->handler_data->magic)
			mime_panic(__FILE__,__LINE__,"prompt_programs",
				   "Bad magic number (handler_data)");
		    
		    
		    if (list[q].structure->handler_data->other_entries)
			new_have_other = prompt_yn_other;
		    else
			new_have_other = prompt_yn;
		}
	    }

	    if (have_other != new_have_other) {
		have_other = new_have_other;

		prompt_Z(LOC.prompt_area,ask_metamail,
			 cancel_ret != NULL,have_other);
	    }
		
	    menu_MoveCursor(LOC.prompt_area,X,list[q].col);
	}

	
	menu_CleartoEOLN(LOC.prompt_area);   /* Clear current answer */
	
	menu_Write_to_screen(LOC.prompt_area,
			     FRM("%c%c"),defans,BACKSPACE);
	FlushBuffer();
	
	
	ch = menu_ReadCh(LOC.prompt_area,REDRAW_MARK|READCH_CURSOR|
			 READCH_resize|READCH_sig_char);
	
	if (ch == '\n') 
	    ch = defans;


	if (q < list_len) {
	    
	    if (ch == *def_ans_yes) {

		if (list[q].printable_command) {
		    list[q].selected = 1;

		    if (list[q].structure) {

			if (MIME_magic != list[q].structure->magic)
			    mime_panic(__FILE__,__LINE__,"prompt_programs",
				       "Bad magic number (mimeinfo)");
								    
			if (list[q].structure->handler_data)  {
			    
			    if (MIME_selector_magic != list[q].structure->handler_data->magic)
				mime_panic(__FILE__,__LINE__,"prompt_programs",
					   "Bad magic number (handler_data)");
		    
			    list[q].structure->handler_data->use_entry = 1;
			}
		    }
		    
		    update_prompt(&P,q, LOC.prompt_area);
		    mailcap_answered = 1;
		}

		q++;

		goto redraw1;
		
	    } else if (ch == *def_ans_no) {
		
		if (list[q].printable_command) {
		    list[q].selected = 1;

		    if (list[q].structure) {
			
			if (MIME_magic != list[q].structure->magic)
			    mime_panic(__FILE__,__LINE__,"prompt_programs",
				       "Bad magic number (mimeinfo)");
					    
			if (list[q].structure->handler_data)  {
			    
			    if (MIME_selector_magic != list[q].structure->handler_data->magic)
				mime_panic(__FILE__,__LINE__,"prompt_programs",
					   "Bad magic number (handler_data)");
			    
			    list[q].structure->handler_data->use_entry = 0;
			}
		    }
		    
		    update_prompt(&P,q, LOC.prompt_area);
		    mailcap_answered = 1;
		}

		q++;

		goto redraw1;
		
	    } else if (ch == ' ' &&
		       (!list[q].printable_command ||
			list[q].selected)) {

		list[q].selected = 1;
		
		/* Space is OK if no question */

		/* When there is answered question just
		   use current value of 'use_entry'
		*/
		
		if (list[q].printable_command)
		    update_prompt(&P,q, LOC.prompt_area);
		
		q++;

		goto redraw1;
		
	    }
	}


    navigate:
	switch (ch) {
	case TERMCH_interrupt_char:	
	    r = metamail_cancel /* -1 */;
	    goto failure;
	    
	case EOF:
	    r = metamail_EOF  /* != EOF */;
	    goto failure;
	    
	case 'i':
	case 'q':
	    menu_Write_to_screen(LOC.prompt_area,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmQuit,
					 "Quit"));
	    FlushBuffer();
	    r = metamail_cancel /* -1 */;
	    goto failure;

	case RESIZE_MARK:
	    
	    DPRINT(Debug,4, (&Debug,"    .... resizing\n"));
	    goto resize_mark;
	    
	case REDRAW_MARK:
	    DPRINT(Debug,4, (&Debug,"    .... redraw requested\n"));
	    
	    need_redraw = 1;	    
	    
	    break;
	    
	case '-':
	case LEFT_MARK:
	case PAGEUP_MARK: {
	    int lin,col;
	    int pos = start_list_position;

	    menu_get_sizes(LOC.prompt_area, &lin,&col);   

	    pos -= lin -3;

	    while (offset > 0 &&
		   list[offset-1].list_position <= pos) {
		offset --;
		start_list_position =  list[offset].list_position;
	    }
	    
	    if (offset < 0)
		offset = 0;
	    q = offset;
	    
	    menu_trigger_redraw(LOC.prompt_area);
	}
	    goto redraw1;
	    
	    
	case '+':
	case RIGHT_MARK:
	case PAGEDOWN_MARK: {
	    int lin,col;
	    int pos = start_list_position;

	    menu_get_sizes(LOC.prompt_area, &lin,&col);   

	    pos += lin -3;

	    while (offset < list_len-1 &&
		   list[offset+1].list_position <= pos) {
		offset ++;
		start_list_position =  list[offset].list_position;

	    }

	    if (offset >= list_len) 
		offset = list_len-1;
	    if (offset < 0)
		offset = 0;
	    q = offset;
	    
	    menu_trigger_redraw(LOC.prompt_area);
	}
	    goto redraw1;
	    
	case DOWN_MARK:
	    if (q < list_len) {
		update_prompt(&P,q, LOC.prompt_area);
		q++;
	    } else if (cancel_ret) {
		/* Go to next message */
		*cancel_ret = DOWN_MARK;
		r = metamail_cancel /* -1 */;
		
		goto failure;
	    }
	    break;
	    
	case UP_MARK:
	    if (q < list_len) 
		update_prompt(&P,q, LOC.prompt_area);
	    if (q > 0)
		q--;
	    else if (cancel_ret) {
		/* Go to previous message */
		*cancel_ret = UP_MARK;
		r = metamail_cancel /* -1 */;
		
		goto failure;
	    }
	    break;
	    
	case 'd':
	done:
	    ch = 0;
	    
	    if (q < list_len) 
		update_prompt(&P,q, LOC.prompt_area);
	    
	    prompt_Z(LOC.prompt_area,ask_metamail,
		     cancel_ret != NULL,have_other);
	    
	    menu_Write_to_screen(LOC.prompt_area,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmDone,
					 "Done"));
	    FlushBuffer();
	    break;
	    
	case 'm':
	    if (q < list_len) 
		update_prompt(&P,q, LOC.prompt_area);
	    
	    prompt_Z(LOC.prompt_area,ask_metamail,
		     cancel_ret != NULL,have_other);
	    
	    menu_Write_to_screen(LOC.prompt_area,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmMetamail,
					 "Metamail"));		
	    FlushBuffer();

	    
	    if (have_metamail() && 
		ask_metamail >= no_metamail_ask /* 0 */) {
		ch = 0;
		r = metamail_needed;
	    } else
		sleep_message();
	    
	    break;

	case 'o':
	    if (q < list_len) {

		if (list[q].structure) {

		    if (MIME_magic != list[q].structure->magic)
			mime_panic(__FILE__,__LINE__,"prompt_programs",
				   "Bad magic number (mimeinfo)");
		    		
		    if (list[q].structure->handler_data)  {
			
			if (MIME_selector_magic != list[q].structure->handler_data->magic)
			    mime_panic(__FILE__,__LINE__,"prompt_programs",
				   "Bad magic number (handler_data)");
		    
			
			if (list[q].structure->handler_data->other_entries) {
			    
			    int X = give_line_position(list,list_len,q,start_list_position);
			    			    
			    int lin,col;

			    enum prompt_mailcap_other_stat other_ret;
			    
			    menu_get_sizes(LOC.prompt_area, &lin,&col);  
			    
			    if (X < 0 || X >= lin)
				break;
			    
			    menu_PutLineX(LOC.prompt_area,X,list[q].col,
					  CATGETS(elm_msg_cat, ElmSet, ElmMailcapOtherT,
						  "Other"));
			    
			    FlushBuffer();

			    other_ret = prompt_mailcap_other(& (list[q]), page,&ch);

			    /*  prompt_mailcap_other causes redraw of page */

			    need_redraw = 1;	    

			    switch (other_ret) {
			    case prompt_mailcap_other_EOF:
				r = metamail_EOF  /* != EOF */;
				goto failure;

			    case prompt_mailcap_other_none:
				
				while (sel < list_len &&
				       list[sel].selected)
				    sel++;
				q = sel;
				
				break;
			    case prompt_mailcap_other_navigate:

				/* ch chould include new command */
				
				goto navigate;
				
			    }
			}
		    }
		}
	    }
	    break;
	    
	case 'p':
	    if (q < list_len) 
		update_prompt(&P,q, LOC.prompt_area);
	    
	    while (sel < list_len &&
		   !list[sel].printable_command) 
		sel++;

	    if (sel >= list_len)
		goto done;


	    q = sel;

	    if (ask_metamail > 0)
		ask_metamail = 0;
	    
	    menu_trigger_redraw(LOC.prompt_area);
	    menu_trigger_redraw(LOC.title_area);
	    
	    break;
	    
	case 'v':
	    if (cancel_ret) {
		if (q < list_len) 
		    update_prompt(&P,q, LOC.prompt_area);

		prompt_Z(LOC.prompt_area,ask_metamail,1,have_other);
	    
		menu_Write_to_screen(LOC.prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, 
					 ElmViewParts,
					     "View parts of message"));
		FlushBuffer();
		
		*cancel_ret = 'v';
		r = metamail_cancel /* -1 */;
		goto failure;
	    }
	    break;

	default: {

	    int pos_lin, pos_col;
	     
	    prompt_Z(LOC.prompt_area,ask_metamail,
		     cancel_ret != NULL,have_other);

	    
	    menu_GetXYLocation(LOC.prompt_area,&pos_lin,&pos_col);
		
	    menu_Write_to_screen(LOC.prompt_area,
				 FRM("%c??"), 07);
	    FlushBuffer();
	    if (sleepmsg > 0)
		error_sleep((sleepmsg + 1) / 2);

	    menu_MoveCursor(LOC.prompt_area,pos_lin,pos_col);

	    menu_CleartoEOLN(LOC.prompt_area);   

	}
	    
	    break;
	}

    redraw1:
	
	FlushBuffer();
	
    } while(ch);
    
    
 out:
 failure:
    
    if (list) {
	int i;
	
	for (i = 0; i < list_len; i++) {
	    if (list[i].printable_command) {
		free(list[i].printable_command);
		list[i].printable_command = NULL;
	    }
	
	
	    if (list[i]. blacklisted_command) {
		free(list[i].blacklisted_command);
		list[i].blacklisted_command = NULL;
	    }
	}

	free(list);
	list = NULL;

    }
    list_len = 0;
    
    free_prompt_screen(&LOC);
    erase_menu_context(&page);

    DPRINT(Debug,11,(&Debug,
		     "prompt_programs=%d",r));
    switch (r) {
    case metamail_EOF:  /* != EOF */  DPRINT(Debug,11,(&Debug, " metamail_EOF"));    break;
    case metamail_cancel:             DPRINT(Debug,11,(&Debug, " metamail_cancel")); break;
    case metamail_none:               DPRINT(Debug,11,(&Debug, " metamail_none"));   break;
    case metamail_needed:             DPRINT(Debug,11,(&Debug, " metamail_needed")); break;
    }

    if (cancel_ret) {
	DPRINT(Debug,11,(&Debug,"; *cancel_ret=%d", *cancel_ret));
	
	switch (*cancel_ret) {
	case DOWN_MARK:  DPRINT(Debug,11,(&Debug," DOWN_MARK")); break;
	case UP_MARK:    DPRINT(Debug,11,(&Debug," UP_MARK"));   break;
	case EOF:        DPRINT(Debug,11,(&Debug," EOF"));
	    /* Not used */
	    break;
	default:
	    if (isascii(*cancel_ret) && isprint(*cancel_ret)) {
		DPRINT(Debug,11,(&Debug, " '%c'",*cancel_ret));
	    }
	    break;
	}       
    }    
    DPRINT(Debug,11,(&Debug,"\n"));
    
    return r;
}


/* -1 == cancel */
enum need_meta_state need_meta (hdr,current,message_count, cancel_ret)
     struct header_rec *hdr;
     int current, message_count;
     int *cancel_ret;
{
    /* Determine whether or not we need to call metamail to display the
     * message contents.
     */
    enum need_meta_state result = metamail_none;

    int X = hdr->mime_rec.mime_flags;  /* NOTPLAIN_need_metamail           0x01
					  NOTPLAIN_need_mailcap            0x02
					  NOTPLAIN_canuse_mailcap          0x04
					  
					  NOTPLAIN_metamail_blacklisted    0x10
				       */

    DPRINT(Debug,11,(&Debug,
		     "need_meta: %s/%s mime_flags %d (%s)\n",
		     get_major_type_name(hdr->mime_rec.TYPE), 
		     get_subtype_name(hdr->mime_rec.TYPE),
		     X,mime_debug_classify_f(X)));

    DPRINT(Debug,11,(&Debug,
		     "need_meta: status %d%s%s%s\n",
		     hdr->status,
		     (hdr->status & MIME_MESSAGE)     ? " MIME_MESSAGE" : "",		     
		     (hdr->status & MIME_UNSUPPORTED) ? " MIME_UNSUPPORTED" : "",
		     (hdr->status & PRE_MIME_CONTENT) ? " PRE_MIME_CONTENT" : ""));
    
    if ((hdr->status & MIME_MESSAGE) && (hdr->status & MIME_UNSUPPORTED )) {
	DPRINT(Debug,11,(&Debug,
			 "need_meta: Usupported mime version\n"));
	
	if (0 != (X & NOTPLAIN_metamail_blacklisted))
	    
	    result = prompt_programs(hdr,metamail_disabled,
				     current,message_count,
				     cancel_ret);

	else if (have_metamail()) {
	    if (prompt_metamail)
		result = prompt_programs(hdr,metamail_ask,
					 current,message_count,
					 cancel_ret);
	    else 
		result = metamail_needed;
	} else 
	    result = metamail_none;

    /* Do not call metamail if Disposition is not inline */
    } else if ((hdr->status & MIME_MESSAGE) && 
	       hdr->mime_rec.disposition != DISP_INLINE) {

	DPRINT(Debug,11,(&Debug,
			 "need_meta: Disposition is not inline\n"));
	
	result = metamail_none;

    } else if (hdr->status & (MIME_MESSAGE | PRE_MIME_CONTENT)) {

	DPRINT(Debug,11,(&Debug,
			 "need_meta: %s%s\n",
			 (hdr->status & MIME_MESSAGE) ? "Mime message" : "",
			 (hdr->status & PRE_MIME_CONTENT) ? " Pre-Mime content" : ""
			 ));

			
	if (0 != (X & NOTPLAIN_metamail_blacklisted))

	    result = prompt_programs(hdr,metamail_disabled,
				     current,message_count,
				     cancel_ret);

	else if (0 != (X & NOTPLAIN_need_metamail) && have_metamail()) {
	    if (prompt_metamail)
		result = prompt_programs(hdr,metamail_ask,
					 current,message_count,
					 cancel_ret);
	    else 
		result = metamail_needed;
	} else if (0 != (X & NOTPLAIN_canuse_mailcap) ||
		   0 != (X & NOTPLAIN_need_mailcap)) {

	    if (have_metamail())
		result = prompt_programs(hdr,no_metamail_ask,
					 current,message_count,
					 cancel_ret);
	    else
		result = prompt_programs(hdr,no_metamail_available_ask,
					 current,message_count,
					 cancel_ret);
	    
	} else if (0 != (X & NOTPLAIN_need_metamail)) {
	    result = prompt_programs(hdr,no_metamail_available,
				     current,message_count,
				     cancel_ret);

	} else {
	    DPRINT(Debug,11,(&Debug,
			     "need_meta: Unsupported?\n"));
	}

    } else {
	DPRINT(Debug,11,(&Debug,
			 "need_meta: Plain message?\n"));
    }

    
    if (result > 0 && 0 != (X & NOTPLAIN_metamail_blacklisted)) {

	DPRINT(Debug,1,(&Debug, "need_meta: Metamail disabled !!!\n"));

	result = -1;
    }


    DPRINT(Debug,9,(&Debug, "need_meta=%d",result));
    switch (result) {
    case metamail_EOF:  /* != EOF */  DPRINT(Debug,9,(&Debug, " metamail_EOF"));    break;
    case metamail_cancel:             DPRINT(Debug,9,(&Debug, " metamail_cancel")); break;
    case metamail_none:               DPRINT(Debug,9,(&Debug, " metamail_none"));   break;
    case metamail_needed:             DPRINT(Debug,9,(&Debug, " metamail_needed")); break;
    }
    DPRINT(Debug,9,(&Debug, "\n"));
    
    return result;
}

int show_msg(current_header, infile, current,message_count, pager_page,
	     pager_cmds)
     struct header_rec *current_header;
     FILE *infile;
     int current, message_count;
     struct pager_page *pager_page;
     struct elm_commands *pager_cmds;
{
	/*** Display number'th message.  Get starting and ending lines
	     of message from headers data structure, then fly through
	     the file, displaying only those lines that are between the
	     two!

	     Return 0 to return to the index screen or a character entered
	     by the user to initiate a command without returning to
	     the index screen (to be processed via process_showmsg_cmd()).
	***/



    enum need_meta_state is_need_meta =  metamail_none;
    int LINES, COLUMNS;
    const char * metamail_value = NULL;
    int cancel_it = 0;
    int cancel_ret = 0;

    int ret_val = 0;

    menu_get_sizes(pager_page->root,&LINES, &COLUMNS);
    
    /* erase showmsg_c border line and prompt area */

    if (pager_page->border_line)
	erase_menu_context( &(pager_page->border_line));
    if (pager_page->prompt_area)
	erase_menu_context( &(pager_page->prompt_area));

#if 0
    /* Do not erase screen on here, copying mail may take long time
       lefting screen empty before it can be displayed
    */

    menu_ClearScreen(pager_page->root);
#endif


    DPRINT(Debug,4,(&Debug, 
		    "displaying %d lines from message #%d\n", 
		    current_header->lines, 
		    current));
    
    print_errors(current_header,current,message_count,&cancel_it, &cancel_ret);
    if (cancel_it) {	
	DPRINT(Debug,4,(&Debug, 
			"view canceled because of errors - show_msg returning %d\n",
			cancel_ret));

	return cancel_ret;
    }
    
    is_need_meta = need_meta(current_header,current,message_count,&cancel_ret);


    switch (is_need_meta) {
    case metamail_cancel:

	DPRINT(Debug,4,(&Debug, 
			"view canceled because of errors - show_msg returning %d\n",
			cancel_ret));

	ret_val = cancel_ret;
	break;

    case metamail_needed:
	if ((metamail_value = have_metamail())) {
	    char fname[STRING], Cmd[SLEN];
	    int err;
	    FILE *fpout;
	    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",
						   NULL,NULL);
	    
	    if (!tmp)
		return 0;
	    
	    
	    elm_sfprintf(fname, sizeof fname,
			 FRM("%semm.%d.%d"), tmp, getpid(), getuid());
	    
	    if ((fpout = safeopen_rdwr(fname,&err)) == NULL) {
		DPRINT(Debug,1,(&Debug, 
				"Error: open of temporary file %s, errno %s (show_message)\n",
				fname, strerror(err)));
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmCantOpenAsOutputFile,
				  "Can't open \"%s\" as output file! (%s)."),
			  fname, strerror(err));
		
		ret_val = 0;
		goto done;
	    }
	    /* Let metamail decode it! 
	     * (Now CM_DECODE also decodes MIME and PGP) -KEH
	     *
	     * CM_LF: Metamail can not cope with CRLF
	     */
	    if (!copy_message_f(infile,current_header,
				fpout, CM_LF, NULL,NULL)) {
		/* FAIL */
		fclose (fpout);
		unlink (fname);
		ret_val = 0;
		goto done;
	    }
	    (void) fclose (fpout);
	    elm_sfprintf(Cmd, sizeof Cmd,
			 FRM("%s -p -z -m Elm %s"), 
			 metamail_value, fname);
	    
	    {
		struct menu_context *xpage;
		
		xpage = Raw(OFF);
		
		menu_ClearScreen(xpage);
		menu_Write_to_screen(xpage,
				     CATGETS(elm_msg_cat, MeSet, 
					     MeExecuteMetamail,
					     "Executing metamail...\n"));
		FlushBuffer();
	    }
	    
	    system_call(Cmd, SY_ENAB_SIGINT|SY_ENV_METAMAIL, NULL);
	    (void) unlink (fname);
	    
	    if (prompt_after_metamail) {
		struct menu_context *cpage;
		int ch;
		
		cpage = Raw(ON | NO_TITE);	/* Raw on but don't switch screen */
	    redraw:
		menu_get_sizes(cpage,&LINES, &COLUMNS);
		
		menu_PutLineX(cpage,
			      LINES-1,0, 
			      CATGETS(elm_msg_cat, ElmSet, 
				      ElmPressAnyKeyIndex,
				      "Press any key to return to index."));
		
		ch = menu_ReadCh(cpage, REDRAW_MARK);
		if (REDRAW_MARK == ch)
		    goto redraw;
		
		menu_Write_to_screen(cpage,FRM("\r\n"));
		Raw(OFF | NO_TITE); /* Raw off so raw on takes effect */
		
		ret_val = (EOF == ch) ? EOF : 0;
	    } else
		ret_val = 0		    ;
		    
	    Raw(ON); /* Finally raw on and switch screen */
	    goto done;
	}

	/* FALLTRU */
		
    case metamail_none: {
	int r = metapager (infile, current_header,  METAPAGER_do_headers,
			   current, message_count, pager_page,
			   pager_cmds);
	
	DPRINT(Debug,4,(&Debug, 
			"view done - show_msg returning %d",
			r));
	if (isascii(r) && isprint(r)) {
	    DPRINT(Debug,9,(&Debug, " (%c)",r));
	}
	DPRINT(Debug,4,(&Debug,"\n"));
	
	ret_val = r;
    }
	break;

    case metamail_EOF:
	ret_val = EOF;
	break;
    }

 done:
    
    DPRINT(Debug,10,(&Debug,"show_msg=%d",
		     ret_val));
    if (EOF == ret_val) {
	DPRINT(Debug,10,(&Debug, " EOF"));
    } else if (isascii(ret_val) && isprint(ret_val)) {
	DPRINT(Debug,10,(&Debug, " '%c'",ret_val));
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return ret_val;
}

int mbx_show_msg (mailbox,msg,pager_page,pager_cmds)
     struct MailboxView *mailbox;
     int msg;
     struct pager_page *pager_page;
     struct elm_commands *pager_cmds;
{
    struct header_rec *hdr;
    FILE *F;
    int ret_val = 0;
    
    if (give_message_data(mailbox,msg,
			  &hdr,&F,NULL,
			  mime_parse_routine)) {

	/* it's been read now! */
	if (ison_status_message(mailbox,msg,status_basic,NEW))
	    clearf_status_message(mailbox,msg,status_basic,NEW); 
		
	/* it's been read now! */
	if (ison_status_message(mailbox,msg,status_basic,UNREAD))
	    clearf_status_message(mailbox,msg,status_basic,UNREAD); 

	ret_val = show_msg(hdr,F,msg+1,
			   get_message_count(mailbox),
			   pager_page,pager_cmds);
    } else {
	DPRINT(Debug,3,(&Debug, 
			"mbx_show_msg: give_message_data [%d] fails",msg));
    }
    
    DPRINT(Debug,10,(&Debug,"mbx_show_msg=%d",
		     ret_val));
    if (EOF == ret_val) {
	DPRINT(Debug,10,(&Debug, " EOF"));
    } else if (isascii(ret_val) && isprint(ret_val)) {
	DPRINT(Debug,10,(&Debug, " '%c'",ret_val));
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return ret_val;    
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
