static char rcsid[] = "@(#)$Id: header.c,v 2.8 2022/07/14 14:16:01 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.8 $   $State: Exp $
 *
 *  Author: Kari Hurtta <elm@elmme-mailer.org>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_screen.h"
#include "def_scommon.h"

/* For INT_MAX and LONG_MAX */
#include <limits.h>

DEBUG_VAR(Debug,__FILE__,"screen");

#define MENU_HEADER_magic       0xF804
struct menu_header {
    struct subpage_common  c;
    unsigned short         magic;      /* MENU_HEADER_magic */

    int       params[HEADER_PARAMS_COUNT];     
    header_line_redraw     * line_redraw;
    header_line_redraw     * current_redraw;
    header_param_changed   * param_change;
    header_line_redraw     * status_redraw;
    header_line_separator_index * separator_index;
    header_separator_redraw * separator_redraw;
    header_separator_start  * separator_start;
    header_setup_init       * setup_init;
    header_setup_line       * setup_line;
    
    struct page_mode {
	enum {
	    page_mode_none,
	    page_mode_header,
	    page_mode_separator
	}   mode;
	int index;
	
    } * page_index;
    int page_index_len;
    
};


#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

S_(w_menu_init mheader_init)
static void mheader_init P_((struct menu_context *ctx));
static void mheader_init(ctx)
     struct menu_context *ctx;
{
    int i;

    DPRINT(Debug,10, (&Debug, "mheader_init: %p\n",
		      ctx));

    ctx->u.header = safe_malloc (sizeof (* ctx->u.header));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)ctx->u.header,sizeof (* ctx->u.header));

    scommon_zero(&(ctx->u.header->c)); 
    ctx->u.header->magic          = MENU_HEADER_magic;

    for (i = 0; i < HEADER_PARAMS_COUNT; i++)
	ctx->u.header->params[i] = 0;

    ctx->u.header->line_redraw     = header_line_noredraw;
    ctx->u.header->current_redraw  = header_line_noredraw;
    ctx->u.header->param_change    = null_header_param_changed;
    ctx->u.header->status_redraw    = header_line_noredraw;
    ctx->u.header->separator_index  = null_header_line_separator_index;
    ctx->u.header->separator_redraw = header_separator_noredraw;
    ctx->u.header->separator_start  = null_header_separator_start;
    ctx->u.header->setup_init       = header_setup_noinit;
    ctx->u.header->setup_line       = header_setup_noline;
    
    ctx->u.header->page_index     = NULL;
    ctx->u.header->page_index_len = 0;
}

S_(w_menu_free mheader_free)
static void mheader_free P_((struct menu_context *ctx));
static void mheader_free(ctx)
     struct menu_context *ctx;
{

    DPRINT(Debug,10, (&Debug, "mheader_free: %p\n",
		      ctx));

    if (MENU_HEADER_magic != ctx->u.header->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"subpage_free",
	      "Bad magic number",0);

    scommon_clear( & (ctx->u.header->c) );

    if (ctx->u.header->page_index) {
	free(ctx->u.header->page_index);
	ctx->u.header->page_index = NULL;
    }
    ctx->u.header->page_index_len = 0;
    
    ctx->u.header->magic = 0;    /* Invalidate */
    free(ctx->u.header);
    ctx->u.header = NULL;
}

static int header_index_to_page_index P_((struct menu_context *ctx,
					  int value));
static int header_index_to_page_index(ctx,value)
     struct menu_context *ctx;
     int value;
{
    int top     = ctx->u.header->params[header_top_line];
    int bottom  = ctx->u.header->params[header_bottom_line];
    int i;
	
    if (value < top || value > bottom) {
	DPRINT(Debug,12,(&Debug,"header_index_to_page_index=-1 value=%d   top=%d bottom=%d\n",
			 value,top,bottom));
	return -1;
    }
    
    if (ctx->u.header->page_index) {
	
	for (i = 0; i < ctx->u.header->page_index_len; i++) {
	    
	    if (page_mode_header == ctx->u.header->page_index[i].mode &&
		value            == ctx->u.header->page_index[i].index) {

		DPRINT(Debug,12,(&Debug,"header_index_to_page_index=%d value=%d\n",
				 i,value));
		return i;
	    }
	}

	DPRINT(Debug,12,(&Debug,"header_index_to_page_index=-1 value=%d page_index_len=%d\n",
			 value,ctx->u.header->page_index_len));
	return -1;
    }

    i = value - top;

    DPRINT(Debug,12,(&Debug,"header_index_to_page_index=%d value=%d   NO page_index\n",
		     i,value));
    return i;
}


/* -1 == no current */
static void header_change_current P_((struct menu_context *ctx,
				      int old_value));
static void header_change_current(ctx,old_value)
     struct menu_context *ctx;
     int old_value;
{
    int current = ctx->u.header->params[header_current];


    if (HEADER_NO_CURRENT != old_value) {
	int i = header_index_to_page_index(ctx,
					   old_value);
	
	if (i >= 0 && i < ctx->lines)
	    ctx->u.header->current_redraw(ctx, 
					  ctx->u.header->c.param_ptr,
					  i,
					  old_value,
					  0);
    }

    if (HEADER_NO_CURRENT != current) {
	int i = header_index_to_page_index(ctx,
					   current);
	if (i >= 0 && i < ctx->lines)
	    ctx->u.header->current_redraw(ctx, 
					  ctx->u.header->c.param_ptr,
					  i,
					  current,
					  1);
    }
}

static char * param_names[HEADER_PARAMS_COUNT] = {
    "header_top_line",
    "header_current",
    "header_bottom_line"
};


/* Return 1 if page_index available */
static int calculate_page_separators P_(( struct menu_context  * subpage));

S_(w_menu_do_redraw mheader_do_redraw)
static int mheader_do_redraw P_((struct menu_context *ctx));
static int mheader_do_redraw(ctx)
     struct menu_context *ctx;
{
    int current, line,i;
    int oldbottom;
    int setup_init_result;
    
    if (MENU_HEADER_magic != ctx->u.header->magic) 
	panic("SCREEN PANIC",__FILE__,__LINE__,"header_do_redraw",
	      "Bad magic number",0);

    current    = ctx->u.header->params[header_current];
    line       = ctx->u.header->params[header_top_line];
    oldbottom  = ctx->u.header->params[header_bottom_line];
        
    DPRINT(Debug,12,(&Debug,"mheader_do_redraw: current=%d line=%d botton=%d %d lines\n",
		     current,line,oldbottom, ctx->lines));

    if (ctx->lines < 1) {
	if (ctx->u.header->page_index) {
	    free(ctx->u.header->page_index);
	    ctx->u.header->page_index = NULL;
	}
	ctx->u.header->page_index_len = 0;

	DPRINT(Debug,12,(&Debug,"mheader_do_redraw=0 (no lines)\n"));
	return 0;	
    }

    /* calculate_page_separators() needs information which is established
       with ..->setup_init()

       Specially it initializes used current date 
    */
    
    setup_init_result = ctx->u.header->setup_init(ctx, 
						  ctx->u.header->c.param_ptr);
    if (setup_init_result) {
	DPRINT(Debug,12,(&Debug,
			 "mheader_do_redraw: Will need do redraw setup; setup_init gives %d\n",
			 setup_init_result));
    }
    
    if (calculate_page_separators(ctx)) {

	DPRINT(Debug,12,(&Debug,"mheader_do_redraw: Have separator lines\n"));

	if (setup_init_result) {
	    DPRINT(Debug,12,(&Debug,"mheader_do_redraw: Doing redraw setup (with separators)\n"));

	    for (i = 0;
		 i < ctx->u.header->page_index_len && i < ctx->lines; i++) {
		int idx = ctx->u.header->page_index[i].index;
		
		switch (ctx->u.header->page_index[i].mode) {
		case page_mode_none:
		case page_mode_separator:
		    break;
		    
		case  page_mode_header:
		    ctx->u.header->setup_line(ctx, 
					      ctx->u.header->c.param_ptr,
					      idx);
		    
		}
	    }
	}
	
	DPRINT(Debug,12,(&Debug,"mheader_do_redraw: Doing redraw (with separators)\n"));
	
	for (i = 0;
	     i < ctx->u.header->page_index_len && i < ctx->lines; i++) {
	    int idx = ctx->u.header->page_index[i].index;
	    
	    switch (ctx->u.header->page_index[i].mode) {

	    case page_mode_none:
		ctx->u.header->separator_redraw(ctx, 
						ctx->u.header->c.param_ptr,
						i,
						HEADER_NO_SEPARATOR);
		break;

	    case page_mode_separator:
		 ctx->u.header->separator_redraw(ctx, 
						 ctx->u.header->c.param_ptr,
						 i,
						 idx);
		 break;

	    case  page_mode_header:
		ctx->u.header->line_redraw(ctx, 
					   ctx->u.header->c.param_ptr,
					   i,
					   idx,
					   idx == current);
		break;
	    }	    
	}

	/* Does not happen */
	for (; i < ctx->lines; i++) {
	    ctx->u.header->separator_redraw(ctx, 
					    ctx->u.header->c.param_ptr,
					    i,
					    HEADER_NO_SEPARATOR);
	}
	
    } else {
	
	DPRINT(Debug,12,(&Debug,"mheader_do_redraw: No separator lines\n"));

	if (setup_init_result) {
	    DPRINT(Debug,12,(&Debug,"mheader_do_redraw: Doing redraw setup\n"));
	    
	    for (i = 0; i < ctx->lines; i++) {
		ctx->u.header->setup_line(ctx, 
					  ctx->u.header->c.param_ptr,
					  line);	       	    
		line++;
	    }
	}

	line       = ctx->u.header->params[header_top_line];

	DPRINT(Debug,12,(&Debug,"mheader_do_redraw: Doing redraw\n"));
	
	for (i = 0; i < ctx->lines; i++) {
	    ctx->u.header->line_redraw(ctx, 
				       ctx->u.header->c.param_ptr,
				       i,
				       line,
				       line == current);
	    line++;
	}
    }
	
    DPRINT(Debug,12,(&Debug,"mheader_do_redraw=1 (redraw done)\n"));
    return 1;
}

static struct  menu_draw_routine HEADER_ROUTINES = {
    MENU_DRAW_magic,

    mheader_init,
    mheader_free,
	
    scommon_Writechar,
    scommon_PutLineS,
    scommon_PutLine0,
    scommon_CleartoEOLN,
    scommon_CleartoEOS,
    scommon_ReadCh2,
    scommon_ScreenSize,
    scommon_GetXYLocation,
    scommon_ClearScreen,
    scommon_MoveCursor,
    scommon_changemode,
    scommon_SyncEnter,
    scommon_SyncLeave,
    scommon_resized,
    scommon_calculate_line,
    scommon_get_linelimit,
    scommon_reset_redraw,
    mheader_do_redraw,
    scommon_WriteUnicode,
    scommon_calculate_rline
};


void header_line_noredraw(ptr,list,line_number,index,is_current)
     struct menu_context  *ptr;
     struct menu_param *list;
     int line_number;
     int index;
     int is_current;
{
    return;
}

void null_header_param_changed(ptr,list,param_index)
     struct menu_context  *ptr;
     struct menu_param *list;	       
     enum header_params param_index;
{
    return;
}

int null_header_line_separator_index(ptr,list,index)
     struct menu_context  * ptr;
     struct menu_param    * list;
     int                    index;
{
    return  HEADER_NO_SEPARATOR;
}

void header_separator_noredraw(ptr,list,line_number,separator_index)
     struct menu_context * ptr;
     struct menu_param   * list;
     int                   line_number;
     int                   separator_index;
{
    return;
}

int null_header_separator_start(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{
    return 0;  
}

int header_setup_noinit(ptr,list)
     struct menu_context  * ptr;
     struct menu_param    * list;
{
    return 0;
}

void header_setup_noline(ptr,list,index)
     struct menu_context  * ptr;
     struct menu_param    * list;
     int index;
{
    return;
}

struct menu_context * new_menu_header(parent,rel_start_line,lines,
				      line_redraw,current_redraw,
				      param_change,status_redraw,
				      separator_index,separator_redraw,
				      separator_start,
				      setup_init, setup_line,
				      list)
     struct menu_context  *parent;
     int rel_start_line;
     int lines;
     header_line_redraw          * line_redraw;
     header_line_redraw          * current_redraw;
     header_param_changed        * param_change;
     header_line_redraw          * status_redraw;
     header_line_separator_index * separator_index;
     header_separator_redraw     * separator_redraw;
     header_separator_start      * separator_start;
     header_setup_init           * setup_init;
     header_setup_line           * setup_line;
     /* only rereference is saved */
     struct menu_param *list;
{
    int rline = 0;
    int lline = 0;

    struct menu_context  *ret =  new_menu_type(&HEADER_ROUTINES);

    if (MENU_CONTEXT_magic != parent->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"new_menu_header",
	      "Bad parent magic number",0);

    ret->lines                = lines;
    ret->columns              = parent->columns;
    ret->u.header->c.rel_start_line = rel_start_line;
    ret->u.header->c.param_ptr     = list;

    ret->u.header->line_redraw     = line_redraw; 
    ret->u.header->current_redraw  = current_redraw; 
    ret->u.header->param_change    = param_change;
    ret->u.header->status_redraw   = status_redraw;
    ret->u.header->separator_index = separator_index;
    ret->u.header->separator_redraw = separator_redraw;
    ret->u.header->separator_start  = separator_start;
    ret->u.header->setup_init       = setup_init;
    ret->u.header->setup_line       = setup_line;
    
    if (MENU_DRAW_magic  != parent->routine->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"new_menu_header",
	      "Bad parent draw magic number",0);

    parent->routine->wra_calculate_line(parent,rel_start_line,&rline);

    lline = rline + lines -1;    /* calculate_printing_area() fixes */

    attach_subpage(parent,ret,rline,lline);

    calculate_printing_area(ret);
    
    return ret;
}

/* Return 1 if page_index available */
static int calculate_page_separators(subpage)
     struct menu_context  * subpage;
{
    int top            = subpage->u.header->params[header_top_line];
    int last_separator = HEADER_NO_SEPARATOR;
    int oldbottom      = subpage->u.header->params[header_bottom_line];
    int r = 0;
    
    if (subpage->lines < 1) {
	DPRINT(Debug,8,(&Debug,"calculate_page_separators: No lines\n"));
	
	if (subpage->u.header->page_index) {
	    free(subpage->u.header->page_index);
	    subpage->u.header->page_index = NULL;
	}
	subpage->u.header->page_index_len = 0;
    
	subpage->u.header->params[header_bottom_line]
	    = top + subpage->lines -1;

	/* Initialize */
    } else if (subpage->u.header->separator_start(subpage, 
						  subpage->u.header->c.param_ptr)) {
	       	       
	int i;
	int line = top;
	
	DPRINT(Debug,8,(&Debug,"calculate_page_separators: Scanning separators\n"));
	
	if (subpage->u.header->page_index_len  !=  subpage->lines) {
	    DPRINT(Debug,12,(&Debug,"calculate_page_separators: reallocing page_index, page_index_len=%d => %d\n",
			     subpage->u.header->page_index_len,
			     subpage->lines));
	    subpage->u.header->page_index =
		safe_array_realloc(subpage->u.header->page_index,
				   subpage->lines,
				   sizeof (subpage->u.header->page_index[0]));
	
	    subpage->u.header->page_index_len = subpage->lines;
	    
	    for (i = 0; i < subpage->u.header->page_index_len; i++) {
		subpage->u.header->page_index[i].mode  = page_mode_none;
		subpage->u.header->page_index[i].index = 0;
	    }
	}
	
	for (i = 0; i < subpage->lines; i++) {
	    int sep = HEADER_NO_SEPARATOR;
	    	    
	    if (subpage->lines > 1)
		sep = 
		    subpage->u.header->separator_index(subpage, 
						       subpage->u.header->c.param_ptr,
						       line);
	    
	    if (sep != HEADER_NO_SEPARATOR &&
		sep != last_separator) {
		
		if (i == subpage->lines-1) {   /* No separator to last line !! */
		    subpage->u.header->page_index[i].mode  = page_mode_none;
		    subpage->u.header->page_index[i].index = 0;
		} else {
		    last_separator = sep;
		    
		    subpage->u.header->page_index[i].mode   = page_mode_separator;
		    subpage->u.header->page_index[i].index  = sep;
		}
		
	    } else {
		subpage->u.header->page_index[i].mode  = page_mode_header;
		subpage->u.header->page_index[i].index = line;
		
		subpage->u.header->params[header_bottom_line] = line;
		line++;
	    }
	}

	r = 1;
    } else {
	DPRINT(Debug,8,(&Debug,"calculate_page_separators: No separators\n"));
	
	if (subpage->u.header->page_index) {
	    free(subpage->u.header->page_index);
	    subpage->u.header->page_index = NULL;
	}
	subpage->u.header->page_index_len = 0;
	
	subpage->u.header->params[header_bottom_line]
	    = top + subpage->lines -1;
    }

    if (oldbottom != subpage->u.header->params[header_bottom_line])  {
	DPRINT(Debug,8,(&Debug,"calculate_page_separators: %s changed %d => %d\n",
			param_names[header_bottom_line],oldbottom,
			subpage->u.header->params[header_bottom_line]));

	subpage->u.header->param_change(subpage,subpage->u.header->c.param_ptr,
					header_bottom_line);    	
    }
    
    DPRINT(Debug,8,(&Debug,"calculate_page_separators=%d%s\n",
		    r,
		    r ? ", page_index available" : ", no page_index"));
    return r;
}

/* Returns new  header_top_line value */
int  menu_header_change_page(subpage,delta_line)
     struct menu_context  * subpage;
     int delta_line /* +lines or -lines */;
{
    int old_value;
    int value;
    
    if (&HEADER_ROUTINES != subpage->routine)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_change_page",
	      "Not a header subpage",0);

    if (MENU_HEADER_magic != subpage->u.header->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_change_page",
	      "Bad magic number",0);

    old_value = subpage->u.header->params[header_top_line];

    DPRINT(Debug,8,(&Debug,"menu_header_change_page: delta_line=%d\n",
		    delta_line));

    /* Initialize */
    if (subpage->u.header->separator_start(subpage, 
					   subpage->u.header->c.param_ptr)) {
	int direction = 1;
	int counter   = 0;
	int last_separator = HEADER_NO_SEPARATOR;
	
	value = old_value;

	
	DPRINT(Debug,8,(&Debug,"menu_header_change_page: Scanning separators\n"));
	
	if (delta_line > 0) {
	    direction = 1;
	} else if (delta_line < 0) {
	    direction = -1;
	} else
	    goto no_change;

	while (counter < INT_MAX &&
	       (counter < delta_line * direction ||
		value == old_value)
	       ) {
	    int sep = 
		subpage->u.header->separator_index(subpage, 
						   subpage->u.header->c.param_ptr,
						   value);
	    if (sep != last_separator &&
		sep != HEADER_NO_SEPARATOR) {
		last_separator = sep;
		counter ++;		
	    } else {

		if (direction < 0 && old_value == INT_MIN)
		    break;
		else if (direction > 0 && value == INT_MAX)
		    break;
		else {
		    value += direction;
		
		    counter ++;
		}
	    }
	}
	
	DPRINT(Debug,8,(&Debug,"menu_header_change_page: counter=%d direction=%d\n",
			counter,direction));
	
    } else {
	DPRINT(Debug,8,(&Debug,"menu_header_change_page: No separators\n"));
	
	if (delta_line > 0) {
	    if (INT_MAX - delta_line > old_value)
		value = old_value + delta_line;
	    else
		value = old_value;
	} else if (delta_line < 0) {
	    if (INT_MIN - delta_line < old_value)
		value = old_value + delta_line;
	    else
		value = old_value;
	    
	    if (value < 0)
		value = 0;
	} else {
	no_change:
	    value = old_value;
	}

    }
	
    subpage->u.header->params[header_top_line] = value;
    
    DPRINT(Debug,8,(&Debug,"menu_header_change_page: %s changed %d => %d\n",
		    param_names[header_top_line],old_value,value));

    subpage->u.header->param_change(subpage,subpage->u.header->c.param_ptr,
				    header_top_line);
    
    if (value != old_value) 
	menu_trigger_redraw(subpage);
    else
	calculate_page_separators(subpage);	


     DPRINT(Debug,8,(&Debug,"menu_header_change_page=%d\n",
		     value));

     
     return value;
}

void menu_header_change(subpage,param,value)
     struct menu_context  *subpage;
     enum header_params  param;
     int value;
{
    int old_value;

    if (&HEADER_ROUTINES != subpage->routine)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_change",
	      "Not a header subpage",0);

    if (MENU_HEADER_magic != subpage->u.header->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_change",
	      "Bad magic number",0);

    if (param < 0 || param >= HEADER_PARAMS_COUNT)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_change",
	      "Bad param number",0);
	
    old_value = subpage->u.header->params[param];

    subpage->u.header->params[param] = value;

    DPRINT(Debug,8,(&Debug,"menu_header_change: %s changed %d => %d\n",
		    param_names[param],old_value,value));

    subpage->u.header->param_change(subpage,subpage->u.header->c.param_ptr,
				    param);
    
    switch (param) {
	
    case header_top_line:
		
	if (value != old_value)
	    menu_trigger_redraw(subpage);
	else
	    calculate_page_separators(subpage);	

	subpage->u.header->param_change(subpage,subpage->u.header->c.param_ptr,
					header_bottom_line);
	
	break;
    case header_current:
	if (value != old_value)
	    header_change_current(subpage,old_value);
	break;
    default:
	break;
    }        
}

int menu_header_get(subpage,param)
     struct menu_context  *subpage;
     enum header_params  param;
{
    if (&HEADER_ROUTINES != subpage->routine)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_get",
	      "Not a header subpage",0);

    if (MENU_HEADER_magic != subpage->u.header->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_get",
	      "Bad magic number",0);

    if (param < 0 || param >= HEADER_PARAMS_COUNT)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_get",
	      "Bad param number",0);
	
    return subpage->u.header->params[param];
}

void menu_header_relocate(subpage,parent,rel_start_line,lines)
     struct menu_context  *subpage;
     struct menu_context  *parent;
     int rel_start_line;
     int lines;
{
    if (&HEADER_ROUTINES != subpage->routine)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_relocate",
	      "Not a header subpage",0);
    
    scommon_relocate(subpage,parent,rel_start_line,lines);

    calculate_page_separators(subpage);	
}

void menu_header_status_update(subpage,item)
     struct menu_context  *subpage;
     int item;
{
    int i, current, top;

    if (&HEADER_ROUTINES != subpage->routine)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_status_update",
	      "Not a header subpage",0);

    if (MENU_HEADER_magic != subpage->u.header->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_status_update",
	      "Bad magic number",0);

    current = subpage->u.header->params[header_current];
    top     = subpage->u.header->params[header_top_line];

    i = header_index_to_page_index(subpage,item);
    if (i >= 0 && i < subpage->lines) {
	DPRINT(Debug,11, (&Debug, 
			  "menu_header_status_update(%p,%d): current=%d, top=%d => i=%d\n",
			  subpage,item,current,top,i));

	subpage->u.header->status_redraw(subpage, 
					 subpage->u.header->c.param_ptr,
					 i,
					 item,
					 current == item);
    } else {
	DPRINT(Debug,11, (&Debug, 
			  "menu_header_status_update(%p,%d): current=%d, top=%d => i=%d (OUT OF RANGE), lines=%d\n",
			  subpage,item,current,top,i,subpage->lines));	
    }

}

void menu_header_change_item(subpage,item)
     struct menu_context  *subpage;
     int item;
{
    int i, current, top;

    if (&HEADER_ROUTINES != subpage->routine)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_change_item",
	      "Not a header subpage",0);
    
    if (MENU_HEADER_magic != subpage->u.header->magic)
	panic("SCREEN PANIC",__FILE__,__LINE__,"menu_header_change_item",
	      "Bad magic number",0);
    
    current = subpage->u.header->params[header_current];
    top     = subpage->u.header->params[header_top_line];

    i = header_index_to_page_index(subpage,item);
    if (i >= 0 && i < subpage->lines) {
	DPRINT(Debug,11, (&Debug, 
			  "menu_header_change_item(%p,%d): current=%d, top=%d => i=%d\n",
			  subpage,item,current,top,i));
	
	subpage->u.header->line_redraw(subpage, 
				       subpage->u.header->c.param_ptr,
				       i,
				       item,
				       current == item);
    } else {
	DPRINT(Debug,11, (&Debug, 
			  "menu_header_change_item(%p,%d): current=%d, top=%d => i=%d (OUT OF RANGE), lines=%d\n",
			  subpage,item,current,top,i,subpage->lines));	
    }      
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
