static char rcsid[] = "@(#)$Id: screen.c,v 2.15 2022/08/19 16:13:05 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.15 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************
 *  Based on Elm 2.4 src/screen.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/**  screen display routines for ELM program 

**/

#include "def_elm.h"
#include "s_elm.h"
#include "s_aliases.h"

DEBUG_VAR(Debug,__FILE__,"ui");


#define  minimum(a,b)	((a) < (b) ? (a) : (b))

static char *us2s P_((unsigned char *str));
static char *us2s(str)
     unsigned char *str;
{
    return (char *)str;
}


void showscreen(page)
     struct menu_context *page;   /* new page assumed */
{
    menu_ClearScreen(page);

    /* Call refresh routines of children */
    menu_redraw_children(page);
    
    show_last_error();
	
    /* define_softkeys(MAIN); */
}

void set_folder_window_title(folder, title_s, icon_s)
     struct folder_info *folder;
     char *title_s, *icon_s;
{
    /* For a moment we assume that widow titles are ascii only */

    charset_t ascii_ptr = MIME_name_to_charset("US-ASCII",0);

    struct string *title_1 = 
	format_string(CATGETS(elm_msg_cat, ElmSet, 
			      ElmWindowTitle,
			      "Elm: %s %S"),
		      folder_type(folder),
		      folder->cur_folder_disp);

    struct string *title_2;    
    char *x;
    char *icon, *title_3;

    if (!ascii_ptr)
        panic("ELM PANIC",__FILE__,__LINE__,"set_folder_window_title",
              "US-ASCII not found",0);
   
    title_2 = convert_string(ascii_ptr,title_1,0);

    /* Keep it simple ... */
    x = strrchr(folder->cur_folder_sys,'/');
    if (x)
	x++;
    else 
	x = folder->cur_folder_sys;

    icon = elm_message(CATGETS(elm_msg_cat, ElmSet, 
			       ElmWindowIcon,
			       "Elm: %s"),
		       x);

    title_3 = us2s(stream_from_string(title_2,1,NULL));

    set_terminal_titles(title_s ? title_s : title_3,
			icon_s ? icon_s : icon);
    
    free(title_3);
    free(icon);
    free_string(&title_2);
    free_string(&title_1);
}

E_(subpage_simple_redraw sb_update_title)
int sb_update_title P_((struct menu_context  *ptr,
			struct menu_param *list));
int sb_update_title(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{
    struct menu_common *mptr = mp_lookup_mcommon(list,elm_mp_menu);
    int modified             = mp_lookup_integer(list,elm_mp_modified);

    int message_count  = mcommon_get_count(mptr);
    int selected       = mcommon_get_selected(mptr);
    struct string * f1 = mcommon_title(mptr);

    struct string * buffer = NULL;
    struct string * buffer2 = NULL;
    struct string * modmsg = NULL;

    int l1,l2,l;
    int LINES, COLUMNS;
    
    DPRINT(Debug,9, (&Debug, "sb_update_title\n"));

    menu_ClearScreen(ptr);
    menu_get_sizes(ptr, &LINES, &COLUMNS);   

    if (modified) {
	modmsg = format_string(CATGETS(elm_msg_cat, AliasesSet, 
				       AliasesModified,
				       "(modified, resync needed) "));
    }
    else {
	modmsg = new_string(system_charset);
    }

   
    if (selected)
	buffer = format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmShownXWithSelect,
				       "%S with %d shown out of %d"),
			       f1,selected, 
			       message_count);
    else 
       	buffer = format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmShownXNoSelectItem,
				       "%S with %d %S"),
			       f1, message_count,
			       (message_count == 1) ? 
			       mcommon_give_item(mptr,m_item) :
			       mcommon_give_item(mptr,m_items));

    

    free_string(&f1);
    
    l1 = string_len(buffer);
    
    buffer2 = format_string(FRM("%S[ELM %s]"),
			    modmsg,version_buff);
    l2 = string_len(buffer2);
    l = l1 + l2 + 1; /* Assumed */

    menu_StartXX(ptr,pg_BOLD);
    
    if (l > COLUMNS) {
	if (l2 < COLUMNS)
	    menu_PutLineX(ptr,2,(COLUMNS - l2)/2,FRM("%S"),buffer2);
	if (l1 > COLUMNS) 
	    menu_PutLineX(ptr,1,1,FRM("%S"),buffer);
	else
	    menu_PutLineX(ptr,1,(COLUMNS - l1)/2,FRM("%S"),buffer);
    } else {
	menu_PutLineX(ptr,1,(COLUMNS - l)/2,FRM("%S %S"),buffer,buffer2);
    }

    menu_EndXX(ptr,pg_BOLD);

    free_string(&modmsg);
    free_string(&buffer);
    free_string(&buffer2);
    
    return 1;   /* subpage area updated completely */
}

E_(subpage_simple_redraw sb_show_menu);
int sb_show_menu(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list; 
{
    enum user_level_v ul = 
	give_dt_enumerate_as_int(&user_level);

    DPRINT(Debug,9, (&Debug, "sb_show_menu\n"));

    menu_ClearScreen(ptr);

    if (ul == user_level_beginner) {	/* a rank beginner.  Give less options  */
	menu_print_format_center(ptr,
				 0, CATGETS(elm_msg_cat, ElmSet, 
					    ElmLevel0MenuLine1,
					    "You can use any of the following commands by pressing the first character;"));
	menu_print_format_center(ptr,
				 1, CATGETS(elm_msg_cat, ElmSet, 
					    ElmLevel0MenuLine2,
					    "d)elete or u)ndelete mail,  m)ail a message,  r)eply or f)orward mail,  q)uit"));
	menu_print_format_center(ptr,
				 2, CATGETS(elm_msg_cat, ElmSet, 
					    ElmLevel0MenuLine3,
					    "To read a message, press <return>.  j = move down, k = move up, ? = help"));
    } else {
	menu_print_format_center(ptr,
				 0, CATGETS(elm_msg_cat, ElmSet, 
					    ElmLevel1MenuLine1,
					    "|=pipe, !=shell, ?=help, <n>=set current to n, /=search pattern"));
        menu_print_format_center(ptr,
				 1, CATGETS(elm_msg_cat, ElmSet, 
					    ElmLevel1MenuLine2,
					    "a)lias, C)opy, c)hange folder, d)elete, e)dit, f)orward, g)roup reply, m)ail,"));
	menu_print_format_center(ptr,
				 2, CATGETS(elm_msg_cat, ElmSet, 
					    ElmLevel1MenuLine3,
					    "n)ext, o)ptions, p)rint, q)uit, r)eply, s)ave, t)ag, u)ndelete, or e(x)it"));
    }
    
    return 1;
}

void draw_dummy_hl(ptr,line_number,is_current)
     struct menu_context  *ptr; 
     int line_number;
     int is_current;
{
    int LINES, COLUMNS;

    menu_get_sizes(ptr, &LINES, &COLUMNS);

    menu_MoveCursor(ptr,line_number,0);

    if (is_current) {
	int l = 0;

	menu_StartXX(ptr,pg_STANDOUT);

	while (l < COLUMNS) {
	    menu_Writechar(ptr,' ');
	    l++;
	}
	menu_EndXX(ptr,pg_STANDOUT);
    }
    
    menu_Writechar(ptr,'\r');
    menu_Writechar(ptr,'\n');
}


E_(header_line_redraw hdr_show_header)
void hdr_show_header(ptr,list,line_number,index,is_current)
     struct menu_context  *ptr;
     struct menu_param *list;	   
     int line_number;
     int index;
     int is_current;
{
    struct menu_common *menu = mp_lookup_mcommon(list,elm_mp_menu);
    int count                = mcommon_get_count(menu);
    int selected             = mcommon_get_selected(menu);
    int rindex               = index;

    DPRINT(Debug,9, (&Debug,
		     "hdr_show_header: line_number=%d index=%d is_current=%d\n",
		     line_number,index,is_current));

    /* mcommon_screen_print_menu_line() looks current pointer
       from 'menu', so is_current is not needed
    */
    
    menu_ClearLine(ptr,line_number);	    
    menu_MoveCursor(ptr,line_number,0);

    if (selected) {
	if (index < 0 || index >= selected)
	    goto out;
	rindex = visible_to_index(index+1,menu);
    }

    if (rindex >= 0 && rindex < count)
	mcommon_screen_print_menu_line(menu,rindex,ptr,list);
    else {
    out:
	if (has_highlighting && ! arrow_cursor) {
	    draw_dummy_hl(ptr,line_number,is_current);
	} else {
	    if (is_current)
		menu_PutLine0(ptr,line_number,0,"->");
	}
    }
}

E_(header_line_redraw hdr_show_current)
void hdr_show_current(ptr,list,line_number,index,is_current)
     struct menu_context  *ptr;
     struct menu_param *list;	   
     int line_number;
     int index;
     int is_current;
{
    struct menu_common *menu = mp_lookup_mcommon(list,elm_mp_menu);
    int count                = mcommon_get_count(menu);
    
    DPRINT(Debug,9, (&Debug,
		     "hdr_show_current: line_number=%d index=%d is_current=%d\n",
		     line_number,index,is_current));

    if (has_highlighting && ! arrow_cursor) {
	int selected             = mcommon_get_selected(menu);
	int rindex               = index;

	/* mcommon_screen_print_menu_line() looks current pointer
	   from 'menu', so is_current is not needed
	*/
	menu_ClearLine(ptr,line_number);	    
	menu_MoveCursor(ptr,line_number,0);

	if (selected) {
	    if (index < 0 || index >= selected)
		goto out;
	    rindex = visible_to_index(index+1,menu);
	}

		
	if (rindex >= 0 && rindex < count)
	    mcommon_screen_print_menu_line(menu,rindex,ptr,list);
	else {
	out:
	    draw_dummy_hl(ptr,line_number,is_current);
	}
    } else {
	if (!is_current)
	    menu_PutLine0(ptr,line_number,0,"->");
	else
	    menu_PutLine0(ptr,line_number,0,"  ");  /* remove old pointer... */      
    }
}


E_(header_line_separator_index hdr_line_separator_index)
int hdr_line_separator_index(ptr,list,index)
     struct menu_context  *ptr;
     struct menu_param *list;	    
     int index;
{
    struct menu_common *menu = mp_lookup_mcommon(list,elm_mp_menu);
    int count                = mcommon_get_count(menu);
    int selected             = mcommon_get_selected(menu);
    int rindex               = index;
    int ret = HEADER_NO_SEPARATOR;
    
    DPRINT(Debug,9, (&Debug,
		     "hdr_line_separator_index: index=%d\n",
		     index));

    if (selected) {
	if (index < 0 || index >= selected)
	    goto out;
	rindex = visible_to_index(index+1,menu);
    }

    if (rindex >= 0 && rindex < count) {
	ret = mcommon_get_separator_index(menu,rindex);
    } else {
    out:
	ret = HEADER_NO_SEPARATOR;
    }

    DPRINT(Debug,9, (&Debug,
		     "hdr_line_separator_index=%d%s\n",
		     ret,
		     HEADER_NO_SEPARATOR == ret ?
		     " HEADER_NO_SEPARATOR" : ""));
		        
    return ret;
}

E_(header_separator_redraw hdr_show_separator);
void hdr_show_separator(ptr,list,line_number,separator_index)
     struct menu_context  *ptr;
     struct menu_param *list;
     int line_number;
     int separator_index /* may be HEADER_NO_SEPARATOR if last line */;
{
    struct menu_common *menu = mp_lookup_mcommon(list,elm_mp_menu);
    int                count = mcommon_separator_count(menu);

    DPRINT(Debug,9, (&Debug,
		     "hdr_show_separator: line_number=%d separator_index=%d%s\n",
		     line_number,
		     separator_index,
		     HEADER_NO_SEPARATOR == separator_index ?
		     " HEADER_NO_SEPARATOR" : ""));

    menu_ClearLine(ptr,line_number);	    
    menu_MoveCursor(ptr,line_number,0);
    
    if (separator_index >= 0 && separator_index < count) {

	mcommon_screen_print_sep_line(menu,separator_index,ptr,list);

    } else {
	menu_Writechar(ptr,'\r');
	menu_Writechar(ptr,'\n');
    }

}

E_(header_separator_start hdr_init_separators);
int  hdr_init_separators(ptr,list)
     struct menu_context * ptr;
     struct menu_param   * list;
{
    int r;
    struct menu_common *menu = mp_lookup_mcommon(list,elm_mp_menu);

    DPRINT(Debug,9, (&Debug, "hdr_init_separators\n"));

    r = mcommon_init_separators(menu,ptr);

    DPRINT(Debug,9, (&Debug, "hdr_init_separators=%d\n",
		     r));
    return r;
}

E_(header_setup_init hdr_setup_init);
int hdr_setup_init(ptr,list)
     struct menu_context * ptr;
     struct menu_param   * list;
{
    struct menu_common *menu = mp_lookup_mcommon(list,elm_mp_menu);
    
    int r = 1;
    
    DPRINT(Debug,9, (&Debug, "hdr_setup_init\n"));

    r = mcommon_screen_setup_menu_init(menu,ptr,list);

    DPRINT(Debug,9, (&Debug, "hdr_setup_init=%d\n",
		     r));
    return r;

}

E_(header_setup_line hdr_setup_line)
void hdr_setup_line(ptr,list,index)
     struct menu_context  *ptr;
     struct menu_param *list;
     int index;
{
    struct menu_common *menu = mp_lookup_mcommon(list,elm_mp_menu);
    int count                = mcommon_get_count(menu);
    int selected             = mcommon_get_selected(menu);
    int rindex               = index;

    DPRINT(Debug,9, (&Debug,
		     "hdr_setup_line: index=%d\n",
		     index));

    if (selected) {
	if (index < 0 || index >= selected)
	    goto out;
	rindex = visible_to_index(index+1,menu);
    }

    if (rindex >= 0 && rindex < count)
	mcommon_screen_setup_menu_line(menu,rindex,ptr,list);
    else {
    out:
	/* Out of range */;
	
    }
}

void copy_current(menu,ptr)
     struct menu_common *menu; 
     struct menu_context *ptr;
{
    int current = mcommon_get_current(menu);
    int j UNUSED_VAROK = menu_header_get(ptr,header_current);
    /* Compute visible takes account is there selected messages or not.
       Note that it is 1 based ... */
    int n = compute_visible(current,menu)-1;

    DPRINT(Debug,5,(&Debug,  
		    "copy_current: current %d: header_page current %d => %d\n",
		    current,j,n));

    menu_header_change(ptr, header_current,n);
}

void hdr_show_status(ptr,list,line_number,index,is_current)
     struct menu_context  *ptr;
     struct menu_param *list;	   
     int line_number;
     int index;
     int is_current;
{
    struct menu_common *menu = mp_lookup_mcommon(list,elm_mp_menu);
    int count                = mcommon_get_count(menu);
    int selected             = mcommon_get_selected(menu);
    int rindex               = index;
    char * s = "";
    char tagged = ' ';

    DPRINT(Debug,9, (&Debug, "hdr_show_status\n"));
    
    if (selected) {
	if (index < 0 || index >= selected)
	    goto out;
	rindex = visible_to_index(index+1,menu);
    }
    
    if (rindex >= 0 && rindex < count) {
	s = mcommon_show_status(menu,rindex);
	if (mcommon_ison_status(menu,rindex,status_basic,TAGGED))
	    tagged = '+';
	else if (mcommon_ison_status(menu,rindex,status_1,S1_FLAGGED))
	    tagged = 'F';
	else
	    tagged = ' ';
    } else {
    out:
	s = "  ";
	tagged = ' ';
    }

    menu_MoveCursor(ptr,line_number,0);

    if (has_highlighting && ! arrow_cursor && is_current)
       	menu_StartXX(ptr,pg_STANDOUT);

    menu_Write_to_screen(ptr,
			 FRM("%.2s%.2s%c"),
			 (is_current && arrow_cursor)? "->" : "  ",
			 s,tagged);

    if (has_highlighting && ! arrow_cursor && is_current)
       	menu_EndXX(ptr,pg_STANDOUT);

    return;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */




