static char rcsid[] = "@(#)$Id: remail.c,v 2.29 2022/08/28 07:58:12 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.29 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>  
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/remail.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** For those cases when you want to have a message continue along
    to another person in such a way as they end up receiving it with
    the reply address the person YOU received the mail from.

**/

#include "def_sndmail.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"mail");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif       

static void remail_sent P_((union any_fd fd, char * title, 
			    struct run_state *rs,
			    int ret, int exit_stat));


static void remailing_message P_((int background));
static void remailing_message(background)
     int background;
{
    if (!background)
	lib_transient(CATGETS(elm_msg_cat, ElmSet, ElmResendingMail,
			      "Resending mail..."));
    else
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingMailBackground,
			  "Resending mail... in background"));
    
    FlushBuffer();
}

static int get_current_mime_hdr P_((FILE *infile,
				    struct mimeinfo *mimehdr,
				    long offset,
				    header_list_ptr *headers));
static int get_current_mime_hdr(infile,mimehdr,offset,headers)
     FILE *infile;
     struct mimeinfo *mimehdr;
     long offset;
     header_list_ptr *headers;
{
    long new_offset;

    *headers = NULL;

    if (mimehdr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"get_current_mime_hdr",
		   "Bad magic number");

    new_offset = mimehdr->begin_offset - offset;

    DPRINT(Debug,8,(&Debug,
		    "get_current_mime_hdr:  base=%ld begin_offset=%ld => %ld\n",
		    offset,mimehdr->begin_offset,new_offset));

    if (-1 == fseek(infile,new_offset,SEEK_SET)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingSeekMimeHeaderFailed,
			  "Failed to seek beginning of mime headers when resending mail"));
	return 0;
    }

    *headers = file_read_headers(infile,RHL_CHECK_HEADER|RHL_MARK_FOLDING);

    return 1;
}


static FILE * give_output_file P_((void));
static FILE * give_output_file() 
{
    static int count = 0;

    char *fname;
    FILE * F;

    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    if (!tmp)
        return NULL;
  
    fname = elm_message(FRM("%selmmimebodyres.%d.%d"), 
			tmp, getpid (),count++);
    F     = safeopen_rdwr(fname,NULL);

    if (!F) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingCreateMimeBodyTemp,
			  "Failed to create file %s when resending mail"),
		  fname);

	free(fname);  fname= NULL;
	return NULL;
    }

    unlink(fname);  /* We can unlink it now ... */
    free(fname);  fname= NULL;

    return F;
}


static int copy_to_output P_((FILE *old_body, FILE *new_body, int is_binary));
static int copy_to_output(old_body,new_body,is_binary)
     FILE *old_body; 
     FILE *new_body; 
     int is_binary;
{
    while(1) {
	int ch = getc(old_body);
	
	if (EOF == ch)
	    break;
	
	if (! is_binary) {
	    if ('\r' == ch) {
		int ch1 = getc(old_body);
		
		putc(ch,new_body);
		if (EOF == ch1)
		    break;

		if ('\n' == ch1) 
		    putc(ch1,new_body);
		else
		    ungetc(ch1,old_body);
       
		continue;

	    } else if ('\n' == ch) {
		/* Temporary convert \n => \r\n
		   in that point it is not known if caller 
		   will be using ENCODING_BINARY on parent type
		*/
		
		putc('\r',new_body);
	    }
	}
	
	putc(ch,new_body);			    
    }
    
    if (ferror(old_body) || EOF == fflush(new_body)) {
	
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingCopyMimeBodyTemp,
			  "Failed to copy mime body when resending mail"));
	
	
	return 0;
    }

    return 1;
}

static int encode_to_output P_((FILE *old_body, FILE *new_body, int is_binary,
				enum encoding encoding,
				struct media_type *TYPE));
static int encode_to_output(old_body,new_body,is_binary,encoding, TYPE)
     FILE *old_body; 
     FILE *new_body; 
     int is_binary;
     enum encoding encoding;
     struct media_type *TYPE;
{
    int is_text = give_text_type_code(TYPE);
    struct out_state * XXX = new_out_state(STATE_out_file);

    
    DPRINT(Debug,10,(&Debug,
		     "encode_to_output: is_binary=%d, encoding=%d (%s), type=%s/%s => is_text=%d\n",
		     is_binary,encoding,ENCODING(encoding),
		     get_major_type_name(TYPE), get_subtype_name(TYPE),
		     is_text));

    set_out_state_file(new_body,XXX);

    /* Temporary convert \n => \r\n
       in that point it is not known if caller 
       will be using ENCODING_BINARY on parent type
    */
    set_out_state_EOLN(XXX,1);
    
    write_encoded(old_body,XXX,encoding,is_text,NULL,is_binary);

    /* Does not close file */

    free_out_state(&XXX);

    if (ferror(old_body) || EOF == fflush(new_body)) {
	
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingCopyMimeBodyTemp,
			  "Failed to copy mime body when resending mail"));
	
	
	return 0;
    }

    return 1;
}

/* -------------------------------------------------------------------- */

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif


typedef int mime_field_writer_f P_((struct out_state *mailer,
				    mime_t *mime_part,
				    enum encoding top_encoding,
				    charset_t header_charset));

S_(mime_field_writer_f write_content_type)
static int write_content_type P_((struct out_state *mailer,
				  mime_t *mime_part,
				  enum encoding top_encoding,
				  charset_t header_charset));
static int write_content_type(mailer,mime_part,top_encoding,header_charset)
     struct out_state *mailer;
     mime_t *mime_part;
     enum encoding top_encoding;
     charset_t header_charset;
{
    char              * s = NULL;
    struct mime_param * e = NULL;

    if (!mime_part->TYPE)
	return 0;

    s = elm_message(FRM("%s/%s"),
		    get_major_type_name(mime_part->TYPE),
		    get_subtype_name(mime_part->TYPE));

    if (mime_part->TYPE_opts) {
	enum mime_parameters_v mp = 
	    give_dt_enumerate_as_int(&mime_parameters);
    
	if (mp == mime_parameters_plain)
	    return 0;

	e = mime_param_encode(mime_part->TYPE_opts);
    }

    mime_write_header(mailer,"Content-Type",s,e,top_encoding);
    
    free(s);			  

    if (e)
	free_mime_param(&e);

    return 1;
}

S_(mime_field_writer_f write_content_disposition)
static int write_content_disposition P_((struct out_state *mailer,
					 mime_t *mime_part,
					 enum encoding top_encoding,
					 charset_t header_charset));
static int write_content_disposition(mailer,mime_part,top_encoding,
				     header_charset)
     struct out_state *mailer;
     mime_t *mime_part;
     enum encoding top_encoding;
     charset_t header_charset;
{
    struct mime_param * e = NULL;

    if (mime_part->DISPOSITION_opts) {
	enum mime_parameters_v mp = 
	    give_dt_enumerate_as_int(&mime_parameters);

	if (mp == mime_parameters_plain)
	    return 0;

	e = mime_param_encode(mime_part->DISPOSITION_opts);
    }

    mime_write_header(mailer,"Content-Disposition",
		      DISPOSITION(mime_part->disposition),
		      e,top_encoding);

    
    if (e)
	free_mime_param(&e);
    
    return 1;
}

S_(mime_field_writer_f write_content_description)
static int write_content_description P_((struct out_state *mailer,
					 mime_t *mime_part,
					 enum encoding top_encoding,
					 charset_t header_charset));
static int write_content_description(mailer,mime_part,top_encoding,
				     header_charset)
     struct out_state *mailer;
     mime_t *mime_part;
     enum encoding top_encoding;
     charset_t header_charset;
{
    if (! mime_part->description)
	return 0;

    write_string_header(mailer,"Content-Description",
			mime_part->description,
			top_encoding,1,
			header_charset);
    return 1;
}


static struct mime_field_handler {
    header_ptr              hdr_name;
    mime_field_writer_f   * func;
}        * mime_field_handlers      = NULL;
static int mime_field_handler_count = 0;

static void add_field_handler P_((const char *name, mime_field_writer_f  *f));
static void add_field_handler(name,f)
     const char *name; 
     mime_field_writer_f *f;
{
    mime_field_handlers = safe_array_realloc(mime_field_handlers,
					     (mime_field_handler_count+1),
					     sizeof (mime_field_handlers[0]));
    
    mime_field_handlers[mime_field_handler_count].hdr_name = 
	find_header(name,1);
    mime_field_handlers[mime_field_handler_count].func = f;

    DPRINT(Debug,12,(&Debug,
		     "add_handler: %d: hdr_name=%p func=%p name=%s\n",
		     mime_field_handler_count,
		     mime_field_handlers[mime_field_handler_count].hdr_name,
		     mime_field_handlers[mime_field_handler_count].func,
		     name));
		        
    mime_field_handler_count++;
}


static int is_handled_field P_((header_ptr hdr_name,
				 struct mime_field_handler *handler_fields,
				 int                    handler_field_count));
static int is_handled_field(hdr_name,handler_fields,handler_field_count)
     header_ptr hdr_name;
     struct mime_field_handler *handler_fields;
     int                    handler_field_count;
{
    int i;

    for (i = 0; i < handler_field_count; i++) 
	if (hdr_name == handler_fields[i].hdr_name) 
	    return 1;

    return 0;
}

static int can_handle_field P_((header_ptr hdr_name,
				 struct mime_field_handler  **handled_fields,
				 int                    *handled_field_count));

static int can_handle_field(hdr_name,handled_fields,handled_field_count)
     header_ptr hdr_name;
     struct mime_field_handler **handled_fields;
     int                    *handled_field_count;
{
    const char * name UNUSED_VAROK = give_header_name(hdr_name);
    int i;

    if (is_handled_field(hdr_name,*handled_fields,*handled_field_count)) {
	DPRINT(Debug,9,(&Debug,
			"can_handle_field=1: name=%s  hdr_name=%p duplicate header (OK)\n",
			name,hdr_name));
	return 1;
    }

    for (i = 0; i < mime_field_handler_count; i++)
	if (hdr_name == mime_field_handlers[i].hdr_name) 
	    goto found;

    DPRINT(Debug,9,(&Debug,"can_handle_field=0: name=%s hdr_name=%p\n",
		    name,hdr_name));
    return 0;
	
 found:

    (*handled_fields) = safe_array_realloc((*handled_fields),
					   ((*handled_field_count)+1),
					   sizeof ((*handled_fields)[0]));
        
    (*handled_fields)[(*handled_field_count)] = mime_field_handlers[i];
    (*handled_field_count)++;

    DPRINT(Debug,9,(&Debug,"can_handle_header=1: name=%s\n",
		    name));
    return 1;
}


static int handle_fields P_((struct mime_field_handler  *handled_fields,
			     int                        handled_field_count,
			     struct out_state *mailer,
			     mime_t *mime_part,
			     enum encoding top_encoding,
			     charset_t header_charset));
static int handle_fields(handled_fields,handled_field_count,
			 mailer,mime_part,top_encoding,header_charset)
     struct mime_field_handler  *handled_fields;
     int                         handled_field_count;
     struct out_state *mailer;
     mime_t *mime_part;
     enum encoding top_encoding;
     charset_t header_charset;
{
    int i;
    
    /* Temporray assume ENCODING_BINARY in that point 
       it is not known if caller 
       will be using ENCODING_BINARY on parent type
    */
    
    for (i = 0; i < handled_field_count; i++) {
	if (! handled_fields[i].func(mailer,
				     mime_part,  top_encoding,
				     header_charset)) {
	    
	    return 0;

	}
    }
    
    if (state_puts (i > 1 ?
		    "X-MIME-Autoconverted: fields " :
		    "X-MIME-Autoconverted: field ",
		    mailer) == EOF) {

	return 0;
    }
	    
    for (i = 0; i < handled_field_count; i++) {
	const char * hdr_name = give_header_name(handled_fields[i].hdr_name);
	
	if (i > 0) 
	    state_puts (", ",mailer);   
	state_puts (hdr_name,mailer);	    
    }
	    
    if (state_printf (mailer, FRM(" by %s\n"),
		      hostfullname)  == EOF) {
	return 0;
    }	

    return 1;
}


/* -------------------------------------------------------------------- */



typedef int convert_mime_type_f P_((FILE *old_body, mime_t *mime_part,
				    long offset,
				    FILE **new_body, enum encoding *new_encoding,
				    int is_binary, int need_enc,
				    enum mailer_capab  mailer_level,
				    charset_t header_charset));
struct mt_handle_remail {
    convert_mime_type_f * converter;

};

/* --------------------- leaf types ------------------------------------ */

S_(convert_mime_type_f remail_leaf)
static int remail_leaf P_((FILE *old_body, mime_t *mime_part, long offset,
			   FILE **new_body, enum encoding *new_encoding,
			   int is_binary, int need_enc,
			   enum mailer_capab  mailer_level,
			   charset_t header_charset));
static int remail_leaf(old_body,mime_part,offset,
		       new_body,new_encoding,is_binary,need_enc,
		       mailer_level,header_charset)
     FILE *old_body; 
     mime_t *mime_part;
     FILE **new_body; 
     long offset;
     enum encoding *new_encoding;
     int is_binary;
     int need_enc;
     enum mailer_capab  mailer_level;
     charset_t header_charset;
{
    FILE *result = NULL;

    if (mime_part->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"remail_leaf",
		   "Bad magic number");

    DPRINT(Debug,12,(&Debug,
		    "remail_leaf: need_enc=%d%s%s, offset=%ld, encoding=%d (%s)\n",
		    need_enc,
		    (need_enc & HAVE_BINARY) ? " BINARY" : "",
		    (need_enc & HAVE_8BIT)   ? " 8BIT"   : "",
		     offset,
		     mime_part->encoding,ENCODING(mime_part->encoding)));

    /* This resets possible wrong Content-Transfer-Encoding */

    if (0 != (need_enc & HAVE_BINARY)) {

	/* Caller is already checked this? */
	if (mime_part->encoding < ENCODING_NONE ||
	    mime_part->encoding > ENCODING_BINARY) {
	    
	    DPRINT(Debug,1,(&Debug,
			    "remail_leaf: binary data, bad encoding %d (%s)\n",
			    mime_part->encoding,ENCODING(mime_part->encoding)));
	    
	    return 0;
	}
	
	if (mailer_level >= mailer_binary) {
	    DPRINT(Debug,9,(&Debug,
			    "remail_leaf: binary mime, mailer supports binarymime\n"));
	    
	    result = give_output_file();
	    if (!result) 
		return 0;
	    
	    if (! copy_to_output(old_body,result, is_binary)) {
		fclose(result);
		return 0;
	    }
	    
	    *new_encoding = ENCODING_BINARY;
	    *new_body     = result;
	    return 1;	    
	}

	DPRINT(Debug,9,(&Debug,
			"remail_leaf: binary mime, encoding needed\n"));

    } else if (0 != (need_enc & HAVE_8BIT)) {

	/* Caller is already checked this? */
	if (mime_part->encoding < ENCODING_NONE ||
	    mime_part->encoding > ENCODING_BINARY) {
	    
	    DPRINT(Debug,1,(&Debug,
			    "remail_leaf: 8 bit data, bad encoding %d (%s)\n",
			    mime_part->encoding,ENCODING(mime_part->encoding)));
	    
	    return 0;
	}

	if (mailer_level >= mailer_8bit) {
	    DPRINT(Debug,9,(&Debug,
			    "remail_leaf: 8-bit mime, mailer supports 8bitmime\n"));
	    
	    result = give_output_file();
	    if (!result) 
		return 0;
	    
	    if (! copy_to_output(old_body,result, is_binary)) {
		fclose(result);
		return 0;
	    }
	    
	    *new_encoding = ENCODING_8BIT;
	    *new_body     = result;
	    return 1;	    
	}


	DPRINT(Debug,9,(&Debug,
			"remail_leaf: 8 bit mime, encoding needed\n"));
    
    } else {
	DPRINT(Debug,9,(&Debug,
			"remail_leaf: 7-bit or encoded MIME, no changes\n"));

	
	result = give_output_file();
	if (!result) 
	    return 0;
	
	if (! copy_to_output(old_body,result, is_binary)) {
	    fclose(result);
	    return 0;
	}
	
	*new_encoding = mime_part->encoding; /* preserver old encoding */
	*new_body     = result;
	return 1;	    
    }

    result = give_output_file();
    if (!result) 
	return 0;

    *new_encoding = ENCODING_QUOTED;

    if (! encode_to_output(old_body, result, is_binary, *new_encoding,  
			   mime_part->TYPE)) {
	
	fclose(result);
	return 0;
    }
    
    *new_body     = result;

    return 1;
}

static struct mt_handle_remail leaf_REMAIL = { remail_leaf };


/* --------------------- message subtypes ------------------------------- */

S_(convert_mime_type_f remail_message)
static int remail_message P_((FILE *old_body, mime_t *mime_part,
			      long offset,
			      FILE **new_body, enum encoding *new_encoding,
			      int is_binary,
			      int need_enc,enum mailer_capab  mailer_level,
			      charset_t header_charset));
static int remail_message(old_body,mime_part,offset,
			  new_body,new_encoding,is_binary,need_enc,
			  mailer_level,header_charset)
     FILE *old_body; 
     mime_t *mime_part;
     long offset;
     FILE **new_body; 
     enum encoding *new_encoding;
     int is_binary;
     int need_enc;
     enum mailer_capab  mailer_level;
     charset_t header_charset;
{
    FILE *result = NULL;

    if (mime_part->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"remail_message",
		   "Bad magic number");


    DPRINT(Debug,12,(&Debug,
		     "remail_message: need_enc=%d%s%s, offset=%ld, encoding=%d (%s)\n",
		     need_enc,
		     (need_enc & HAVE_BINARY) ? " BINARY" : "",
		     (need_enc & HAVE_8BIT)   ? " 8BIT"   : "",
		     offset,
		     mime_part->encoding,ENCODING(mime_part->encoding)));


    if (mime_part->encoding < ENCODING_NONE ||
	mime_part->encoding > ENCODING_BINARY) {

	DPRINT(Debug,8,(&Debug,
			"remail_message: Part is encoded, which is not allowed for subtypes of message.\n"));

	if (ENCODING_ILLEGAL       == mime_part->encoding ||
	    ENCODING_EXPERIMENTAL  == mime_part->encoding)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmResendingFailedBadEncoding0,
			      "Found bad encoding for mime type %s/%s when resending mail"),
		      get_major_type_name(mime_part->TYPE), 
		      get_subtype_name(mime_part->TYPE));
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmResendingFailedBadEncoding,
			      "Found bad encoding %s for mime type %s/%s when resending mail"),
		      ENCODING(mime_part->encoding),
		      get_major_type_name(mime_part->TYPE), 
		      get_subtype_name(mime_part->TYPE));
	
	return 0;
    }

    /* This resets possible wrong Content-Transfer-Encoding */

    if (0 != (need_enc & HAVE_BINARY)) {

	if (mailer_level >= mailer_binary) {
	    DPRINT(Debug,9,(&Debug,
			    "remail_message: binary mime, mailer supports binarymime\n"));
	    
	    result = give_output_file();
	    if (!result) 
		return 0;
	    
	    if (! copy_to_output(old_body,result, is_binary)) {
		fclose(result);
		return 0;
	    }
	    
	    *new_encoding = ENCODING_BINARY;
	    *new_body     = result;
	    return 1;	    
	}

	DPRINT(Debug,9,(&Debug,
			"remail_message: binary mime, we lost\n"));


    }  else if (0 != (need_enc & HAVE_8BIT)) {

	if (mailer_level >= mailer_8bit) {
	    DPRINT(Debug,9,(&Debug,
			    "remail_message: 8-bit mime, mailer supports 8bitmime\n"));
	    
	    result = give_output_file();
	    if (!result) 
		return 0;
	    
	    if (! copy_to_output(old_body,result, is_binary)) {
		fclose(result);
		return 0;
	    }
	    
	    *new_encoding = ENCODING_8BIT;
	    *new_body     = result;
	    return 1;	    
	}

	DPRINT(Debug,9,(&Debug,
			"remail_message: 8 bit mime, we lost\n"));

    } else {
	DPRINT(Debug,9,(&Debug,
			"remail_message: 7-bit mime, no changes\n"));

	
	result = give_output_file();
	if (!result) 
	    return 0;
	
	if (! copy_to_output(old_body,result, is_binary)) {
	    fclose(result);
	    return 0;
	}
	
	*new_encoding = ENCODING_7BIT;
	*new_body     = result;
	return 1;	    
    }

    lib_error(CATGETS(elm_msg_cat, ElmSet, 
		      ElmResendingEncodingNotAllowedMessage,
		      "Can't encode subtypes of message (type %s/%s) when resending mail"),
	      get_major_type_name(mime_part->TYPE), 
	      get_subtype_name(mime_part->TYPE));
    
    
    return 0;
}

static struct mt_handle_remail message_REMAIL = { remail_message };

/* --------------------- message/rfc822 type ---------------------------- */


/* Creates temporary file */
static FILE * read_mime_body P_(( FILE *infile, mime_t *mime_part ));


/*  ENCODING_ILLEGAL      -- encoding is not changed ... 
    ENCODING_EXPERIMENTAL -- encoding is not changed ...
*/
static int convert_body P_((FILE *infile, mime_t *mime_part,
			    long offset,
			    FILE **new_body, enum encoding *new_encoding,
			    int is_binary,enum mailer_capab  mailer_level,
			    charset_t header_charset));

static int copy_nonmime_body P_((FILE *infile, mime_t *mime_part,
				 long offset,
				 FILE **new_body));
static int copy_nonmime_body(infile,mime_part,offset,new_body)
     FILE *infile; 
     mime_t *mime_part;
     long offset;
     FILE **new_body;
{
    FILE * old_body = NULL;
    FILE * result;
    long new_offset;

    /* This routine assumes 7-bit content */

    *new_body     = NULL;

    new_offset = mime_part->offset - offset;

    DPRINT(Debug,8,(&Debug,
		    "copy_nonmime_body:  base=%ld offset=%ld => %ld\n",
		    offset,mime_part->offset,new_offset));

    if (-1 == fseek(infile,new_offset,SEEK_SET)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingSeekEmbBodyFailed,
			  "Failed to seek beginning of embedded mail body when resending mail"));
	return 0;
    }

    old_body = read_mime_body(infile,mime_part);

    if (! old_body)
	return 0;

    result = give_output_file();
    if (!result) {
	fclose(old_body);
	return 0;
    }
    
    while(1) {
	int ch = getc(old_body);
	
	if (EOF == ch)
	    break;
	
	if ('\r' == ch) {
	    int ch1 = getc(old_body);
	    
	    putc(ch,result);
	    if (EOF == ch1)
		break;
	    
	    putc(ch1,result);
	    
	} else if ('\n' == ch) {
	    /* Temporary convert \n => \r\n
	       in that point it is not known if caller 
	       will be using ENCODING_BINARY on parent type
	    */
	    
	    putc('\r',result);
	}
	
	putc(ch,result);			    
    }
    
    if (ferror(old_body) || EOF == fflush(result)) {
	
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingCopyEmbBodyTemp,
			  "Failed to embedded mail body when resending mail"));
	
	fclose(old_body);
	fclose(result);
	return 0;
    }

    fclose(old_body);
    rewind(result);
    
    DPRINT(Debug,8,(&Debug,
		    "copy_nonmime_body: Done LR -> CRLF conversion\n"));
    
    *new_body = result;
    return 1;
}

S_(convert_mime_type_f remail_rfc822)
static int remail_rfc822 P_((FILE *old_body, mime_t *mime_part,long offset,
			     FILE **new_body, enum encoding *new_encoding,
			     int is_binary, int need_enc,
			     enum mailer_capab  mailer_level,
			     charset_t header_charset));
static int remail_rfc822(old_body,mime_part,offset,new_body,new_encoding,
			 is_binary,need_enc,
			 mailer_level,header_charset)
     FILE *old_body; 
     mime_t *mime_part;
     long offset;
     FILE **new_body; 
     enum encoding *new_encoding;
     int is_binary;
     int need_enc;
     enum mailer_capab  mailer_level;
     charset_t header_charset;
{

    if (mime_part->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"remail_rfc822",
		   "Bad magic number");

    DPRINT(Debug,12,(&Debug,
		     "remail_rfc822: need_enc=%d%s%s, offset=%ld, encoding=%d (%s)\n",
		     need_enc,
		     (need_enc & HAVE_BINARY) ? " BINARY" : "",
		     (need_enc & HAVE_8BIT)   ? " 8BIT"   : "",
		     offset,
		     mime_part->encoding,ENCODING(mime_part->encoding)));


    if (!mime_part->parser_data || 
	mime_parser_subparts(mime_part->parser_data) != 1) {

	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingFailedBodyLocEmbedded,
			  "Failed to locate body of embedded mail when resending mail"));

	return 0;

    } else {
	struct out_state * XXX = new_out_state(STATE_out_file);

	int have_8bit_header   = 0;
	int mime_version_seen  = 0;
	int is_mime            = 0;
	int mime_version_added = 0;
	int r = 0;

	struct mime_field_handler *handled_fields        = NULL;
	int                        handled_field_count   = 0;

	mime_t * nested_part = mime_parser_index(mime_part->parser_data,0);

	header_list_ptr headers         = NULL;
	header_list_ptr next_hdr;
	       
	FILE           * nested_result = NULL;
	enum encoding    nested_result_encoding = ENCODING_NONE;

	FILE * target = NULL;

	if (nested_part->magic != MIME_magic)
	    mime_panic(__FILE__,__LINE__,"remail_rfc822",
		       "Bad magic number (nested_part)");
	
	r = get_current_mime_hdr(old_body,nested_part, offset, &headers);
	if (! r) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmResendingFailedHeaderEmbedded,
			      "Failed to read mail headers of embedded mail when resending mail"));
	    goto fail;
	}
   
  
	for (next_hdr = headers; 
	     next_hdr; 
	     next_hdr = next_hdr -> next_header) {
	    const char * hdr_name = give_header_name(next_hdr->header_name);
	    
	    if (check_8bit_str(next_hdr->body)) {
		DPRINT(Debug,8,(&Debug, 
				"remail_rfc822: Found 8-bit data from %s header field\n",
				hdr_name));			    

		if (!can_handle_field(next_hdr->header_name,
				      &handled_fields,
				      &handled_field_count)) {		   		    
		    have_8bit_header = 1;
		} 
	    }
	    
	    if (0 == istrcmp("MIME-Version", hdr_name))
		mime_version_seen = 1;

	}
	
	if (have_8bit_header && 
	    (!allow_no_hdrencoding || mailer_level < mailer_8bit)) {
	    
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmResending8BitEmbdedHeaderUnsupported,
				  "Can't handle 8-bit data on embedded mail headers when resending mail"));
		
		r = 0;
		goto fail;
	    
	}
		
	/* TODO: Handle Pre-MIME Content-types ... */

	if (mime_version_seen || !req_mime_bodyencoding 
	    ||
	    (ENCODING_NONE != nested_part->encoding && 
	     ENCODING_7BIT != nested_part->encoding) 
	    ||
	    0 == (need_enc & HAVE_BINARY) ||
	    0 == (need_enc & HAVE_8BIT)) {

	    r = convert_body(old_body,nested_part,offset,
			     &nested_result,&nested_result_encoding,
			     is_binary, mailer_level,header_charset);
	    is_mime = 1;

	} else {
	    r = copy_nonmime_body(old_body,nested_part,offset,
				  &nested_result);
	    nested_result_encoding = ENCODING_NONE;   /* No header to non-mime messages */
	}
				  
	if (!r) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingFailedEmbBody,
			  "Failed to process embedded mail when resending mail"));

	    goto fail;
	    
	}		     


	target = give_output_file();
	if (!target) {
      	    r = 0;
	    goto fail;
	}

	set_out_state_file(target,XXX);
	
	/* Temporary convert \n => \r\n
	   in that point it is not known if caller 
	   will be using ENCODING_BINARY on parent type
	*/
	set_out_state_EOLN(XXX,1);

	if ((!mime_version_seen && is_mime) 
	    ||
	    handled_field_count > 0 
	    ||
	    (nested_result_encoding != ENCODING_ILLEGAL &&
	     nested_result_encoding != ENCODING_NONE    &&
	     nested_result_encoding != ENCODING_EXPERIMENTAL &&
	     nested_result_encoding != nested_part->encoding)) {	
	    if (state_printf(XXX, FRM("MIME-Version: 1.0\n")) == EOF) {
		r = 0;
		goto fail;
	    }

	    mime_version_added = 1;
	}
	
	if (nested_result_encoding != ENCODING_ILLEGAL &&
	    nested_result_encoding != ENCODING_NONE    &&
	    nested_result_encoding != ENCODING_EXPERIMENTAL &&
	    nested_result_encoding != nested_part->encoding) {

	    if (state_printf(XXX, FRM("Content-Transfer-Encoding: %s\n"),
			      ENCODING(nested_result_encoding)) == EOF) {
		r = 0;
		goto fail;
	    }

	    if (state_printf(XXX, 
			      FRM("X-MIME-Autoconverted: from %s to %s by %s\n"),
			      ENCODING(nested_part->encoding),
			      ENCODING(nested_result_encoding),
			      hostfullname)  == EOF) {
		r = 0;
		goto fail;
	    }
	}

	if (handled_fields) {
	    if (! handle_fields(handled_fields,handled_field_count,
			  XXX,
			  nested_part, ENCODING_BINARY,
				header_charset)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmResending8BitEmbdedHeaderUnsupported,
				  "Can't handle 8-bit data on embedded mail headers when resending mail"));
		
		r = 0;
		goto fail;		    
	    }
	    
	}

	for (next_hdr = headers; 
	     next_hdr; 
	     next_hdr = next_hdr -> next_header) {
	    const char * hdr_name = give_header_name(next_hdr->header_name);
	    	   	    
	    if (0 == istrcmp("Content-Transfer-Encoding",hdr_name) &&
		nested_result_encoding != ENCODING_ILLEGAL  &&
		nested_result_encoding != ENCODING_EXPERIMENTAL &&
		nested_result_encoding != nested_part->encoding)
		continue;
	    
	    if (0 == istrcmp("MIME-Version",hdr_name) &&
		mime_version_added)
		continue;

	    if (is_handled_field(next_hdr->header_name,
				 handled_fields,
				 handled_field_count))
		continue;

	    state_write_raw_header(XXX,&NULL_decode_opt,
				   next_hdr);
	}

	/* Must use state_printf so EOLN_is_CRLF conversion occurs! */
	if (state_printf(XXX, 
			 FRM("\n")) == EOF) {
	    r = 0;
	    goto fail;
	}      
	

	rewind(nested_result);
	
	DPRINT(Debug,5,(&Debug,
			"remail_rfc822: Doing copy\n"));
    
	if (! copy_to_output(nested_result,target,is_binary)) {
	    r = 0;
	    goto fail;
	}
	    
	*new_body = target; target = NULL;
	
	if (nested_result_encoding >= ENCODING_NONE &&
	    nested_result_encoding <= ENCODING_BINARY) {
	    *new_encoding = nested_result_encoding;
	} else
	    *new_encoding = ENCODING_7BIT;          /* Encoded result produces 7-bit output */

	if (have_8bit_header && *new_encoding < ENCODING_8BIT) {
	    DPRINT(Debug,8,(&Debug, 
			    "remail_rfc822: 8-bit header fields => raising encoding to 8-bit\n"));
	    *new_encoding = ENCODING_8BIT;
	}

	r = 1;

    fail:
	free_out_state(&XXX);

      if (headers)
	  delete_headers(&headers);

      if (nested_result)
	  fclose(nested_result);

      if (target)
	  fclose(target);

      if (handled_fields)
	  free(handled_fields);


      return r;

    }

    /* NOT REACHED */
}

static struct mt_handle_remail rfc822_REMAIL = { remail_rfc822 };


/* --------------------- multipart subtypes ---------------------------- */

S_(convert_mime_type_f remail_multipart)
static int remail_multipart P_((FILE *old_body, mime_t *mime_part,long offset,
				FILE **new_body, enum encoding *new_encoding,
				int is_binary, int need_enc,
				enum mailer_capab  mailer_level,
				charset_t header_charset));
static int remail_multipart(old_body,mime_part,offset,
			    new_body,new_encoding,is_binary,need_enc,
			    mailer_level,header_charset)
     FILE *old_body; 
     mime_t *mime_part;
     long offset;
     FILE **new_body; 
     enum encoding *new_encoding;
     int is_binary;
     int need_enc;
     enum mailer_capab  mailer_level;
     charset_t header_charset;
{
    int idx;
    int part_count;
    enum encoding  result_encoding = ENCODING_NONE;
    FILE *         result = NULL;
    const char * boundary = NULL;
    int r = 0;

    struct out_state * XXX = new_out_state(STATE_out_file);

    if (mime_part->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"remail_multipart",
		   "Bad magic number");

    DPRINT(Debug,12,(&Debug,
		     "remail_multipart: need_enc=%d%s%s, offset=%ld, encoding=%d (%s)\n",
		     need_enc,
		     (need_enc & HAVE_BINARY) ? " BINARY" : "",
		     (need_enc & HAVE_8BIT)   ? " 8BIT"   : "",
		     offset,
		     mime_part->encoding,ENCODING(mime_part->encoding)));


    if (! mime_part->parser_data) {

	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingFailedBodyLocMultipart,
			  "Failed to locate body of multipart mime type when resending mail"));

	return 0;
    } 

    part_count = mime_parser_subparts(mime_part->parser_data);

    boundary = mime_get_boundary(mime_part->TYPE_opts);
    if (!boundary) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingFailedBoundary,
			  "Failed to locate boundary of multipart mime type when resending mail"));	
	r = 0;
	goto fail;
    }

    if (check_8bit_str(boundary)) {
	DPRINT(Debug,8,(&Debug, 
			"remail_multipart: Seen 8-bit data on boundary\n"));
	
	/* Well, 8BITMIME mailer may have some difficulties on downgrading
	         if boundary includes 8-bit content
	*/
	if (mailer_level < mailer_8bit) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmResending8BitBoundaryUnsupported,
			      "Can't handle 8-bit data on multipart boundary when resending mail"));
	    r = 0;
	    goto fail;
	}
	    
	if (result_encoding < ENCODING_8BIT)
	    result_encoding = ENCODING_8BIT;

    }

    /* XXXX   this does not preserve multpart preamble
              or postamble
    */
    

    result = give_output_file();
    if (!result) 
	return 0;
    

    set_out_state_file(result,XXX);
    /* Temporary convert \n => \r\n
       in that point it is not known if caller 
       will be using ENCODING_BINARY on parent type
    */
    set_out_state_EOLN(XXX,1);

    for (idx = 0; idx < part_count; idx++) {
	mime_t * nested_part = mime_parser_index(mime_part->parser_data,idx);

	int have_8bit_header  = 0;
	header_list_ptr headers         = NULL;
	header_list_ptr next_hdr;

	struct mime_field_handler *handled_fields        = NULL;
	int                        handled_field_count   = 0;
	       
	FILE           * nested_result = NULL;
	enum encoding    nested_result_encoding = ENCODING_NONE;
	
	if (nested_part->magic != MIME_magic)
	    mime_panic(__FILE__,__LINE__,"remail_multipart",
		       "Bad magic number (nested_part)");

	r = get_current_mime_hdr(old_body,nested_part,offset,&headers);
	if (! r) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmResendingFailedHeaderMultpart,
			      "Failed to read mime headers on multipart when resending mail"));
	    goto fail1;
	}

	for (next_hdr = headers; 
	     next_hdr; 
	     next_hdr = next_hdr -> next_header) {
	    const char * hdr_name UNUSED_VAROK = 
		give_header_name(next_hdr->header_name);
	    
	    if (check_8bit_str(next_hdr->body)) {
		DPRINT(Debug,8,(&Debug, 
				"remail_rfc822: Found 8-bit data from %s header field\n",
				hdr_name));	

		if (!can_handle_field(next_hdr->header_name,
				      &handled_fields,
				      &handled_field_count)) {			    
		    have_8bit_header = 1;
		}
	    }
	}

	if (have_8bit_header) {
 
	    if (!allow_no_hdrencoding || mailer_level < mailer_8bit) {
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmResending8BitMultipartHeaderUnsupported,
			      "Can't handle 8-bit data on multipart headers when resending mail"));
	    

		r = 0;
		goto fail1;
	    }
	    
	    if (result_encoding < ENCODING_8BIT)
		result_encoding = ENCODING_8BIT;	    
	}

	/* Separating boundary */
	state_nlputs("\n--", XXX);
	state_puts(boundary,XXX);
	state_nlputs("\n", XXX);

	r = convert_body(old_body,nested_part,offset,
			 &nested_result,&nested_result_encoding,
			 is_binary, mailer_level,
			 header_charset);

	if (!r) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmResendingFailedMultiBody,
			      "Failed to process multipart body of mail when resending mail"));
	    
	    goto fail1;

	}

	if (nested_result_encoding != ENCODING_ILLEGAL &&
	    nested_result_encoding != ENCODING_NONE    &&
	    nested_result_encoding != ENCODING_EXPERIMENTAL &&
	    nested_result_encoding != nested_part->encoding) {
	    
	    if (state_printf(XXX, FRM("Content-Transfer-Encoding: %s\n"),
			      ENCODING(nested_result_encoding)) == EOF) {
		r = 0;
		goto fail1;
	    }
	    
	    if (state_printf(XXX, 
			      FRM("X-MIME-Autoconverted: from %s to %s by %s\n"),
			      ENCODING(nested_part->encoding),
			      ENCODING(nested_result_encoding),
			      hostfullname)  == EOF) {
		r = 0;
		goto fail1;
	    }
	}
	
	if (handled_fields) {
	    if (! handle_fields(handled_fields,handled_field_count,
				XXX,
				nested_part, ENCODING_BINARY,
				header_charset)) {
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmResending8BitMultipartHeaderUnsupported,
				  "Can't handle 8-bit data on multipart headers when resending mail"));
		
		r = 0;
		goto fail1;		    
	    }	    
	}


	for (next_hdr = headers; 
	     next_hdr; 
	     next_hdr = next_hdr -> next_header) {
	    const char * hdr_name = give_header_name(next_hdr->header_name);
	    	   	    
	    if (0 == istrcmp("Content-Transfer-Encoding",hdr_name) &&
		nested_result_encoding != ENCODING_ILLEGAL  &&
		nested_result_encoding != ENCODING_EXPERIMENTAL &&
		nested_result_encoding != nested_part->encoding)
		continue;

	    if (is_handled_field(next_hdr->header_name,
				 handled_fields,
				 handled_field_count))
		continue;
	    
	    state_write_raw_header(XXX,&NULL_decode_opt,
				   next_hdr);
	}

	/* Must use state_printf so EOLN_is_CRLF conversion occurs! */
	if (state_printf(XXX, 
			 FRM("\n")) == EOF) {
	    r = 0;
	    goto fail1;
	}      

	rewind(nested_result);
	
	DPRINT(Debug,5,(&Debug,
			"remail_multipart: Doing copy, part %d/%d\n",idx,part_count));

	if (! copy_to_output(nested_result,result,is_binary)) {
	    r = 0;
	    goto fail1;
	}

	if (nested_result_encoding >= ENCODING_NONE &&
	    nested_result_encoding <= ENCODING_BINARY) {
	    if (result_encoding < nested_result_encoding)
		result_encoding = nested_result_encoding;
	}


	r = 1;
    fail1:
	
	if (handled_fields)
	    free(handled_fields);

	if (nested_result)
	    fclose(nested_result);

	if (!r)
	    goto fail;
    }

    /* Ending boundary */
    state_nlputs("\n--", XXX);
    state_puts(boundary,XXX);
    state_nlputs("--\n", XXX);


    *new_body     = result; result = NULL;
    *new_encoding = result_encoding;
    r = 1;

 fail:
    free_out_state(&XXX);
   
    if (result)
	fclose(result);

    return r;
}

static struct mt_handle_remail multipart_REMAIL = { remail_multipart };


/* ----- multipart/signed special casing ------------------------------- */

S_(convert_mime_type_f remail_signed)
static int remail_signed P_((FILE *old_body, mime_t *mime_part,long offset,
			     FILE **new_body, enum encoding *new_encoding,
			     int is_binary, int need_enc,
			     enum mailer_capab  mailer_level,
			     charset_t header_charset));
static int remail_signed(old_body,mime_part,offset,
			    new_body,new_encoding,is_binary,need_enc,
			    mailer_level,header_charset)
     FILE *old_body; 
     mime_t *mime_part;
     long offset;
     FILE **new_body; 
     enum encoding *new_encoding;
     int is_binary;
     int need_enc;
     enum mailer_capab  mailer_level;
     charset_t header_charset;
{
    FILE *         result = NULL;

    if (mime_part->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"remail_signed",
		   "Bad magic number");

    DPRINT(Debug,12,(&Debug,
		     "remail_signed: need_enc=%d%s%s, offset=%ld, encoding=%d (%s)\n",
		     need_enc,
		     (need_enc & HAVE_BINARY) ? " BINARY" : "",
		     (need_enc & HAVE_8BIT)   ? " 8BIT"   : "",
		     offset,
		     mime_part->encoding,ENCODING(mime_part->encoding)));

    /* multipart/signed can not converted without invalidating signature.
     *
     * It is assumed that EOLNs are nornalized to CRLF (or to LF) before
     * calculation signature, so that changes on EOLN format does not
     * invalidate signature.
     *
     *  multipart/signed allows only 7-bit data
     */

    if (0 != (need_enc & HAVE_BINARY) ||
	0 != (need_enc & HAVE_8BIT) 
	||
	(mime_part->encoding != ENCODING_NONE &&
	 mime_part->encoding != ENCODING_7BIT)
	) {
					 
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingNot7BadEncoding,
			  "Found 8-bit or binary data or bad encoding on multipart/signed when resending mail"));

	return 0;
    }

    /* Do not process parts recursively so that signature is
     * preserved.
     */
	
    DPRINT(Debug,9,(&Debug,
		    "remail_signed: 7-bit mime, no changes\n"));					 

    result = give_output_file();
    if (!result) 
	return 0;
    
    if (! copy_to_output(old_body,result, is_binary)) {
	fclose(result);
	return 0;
    }
    
    *new_encoding = ENCODING_7BIT;
    *new_body     = result;
    return 1;	    
}

static struct mt_handle_remail signed_REMAIL = { remail_signed };

/* --------------------------------------------------------------------- */


#if __GNUC__ 
#define MTH        struct media_type_handle          
#define REMAIL(A)   handle_remail, { remail: & A }

#define REGISTER_MT_HANDLER(A,B,R) \
   register_mt_handler(give_media_type2(A,B,1),& R)
#define REGISTER_MT_DEFHANDLER(A,R) \
   register_mt_defhandler(A,& R)

#else
#define MTH         struct COMPAT_media_type_handle
#define REMAIL(A)   handle_remail, (void *) & A

#define REGISTER_MT_HANDLER(A,B,R) \
   register_mt_handler(give_media_type2(A,B,1),(struct media_type_handle *)& R)
#define REGISTER_MT_DEFHANDLER(A,R) \
   register_mt_defhandler(A,(struct media_type_handle *)& R)

#endif

static MTH leaf_remail_1                   = { REMAIL(leaf_REMAIL) };
static MTH message_remail_1                = { REMAIL(message_REMAIL) };
static MTH rfc822_remail_1                 = { REMAIL(rfc822_REMAIL) };
static MTH multipart_remail_1              = { REMAIL(multipart_REMAIL) };
static MTH signed_remail_1                 = { REMAIL(signed_REMAIL) };

static int remail_register_converters_called = 0; 
static void remail_register_converters P_((void));
static void remail_register_converters()
{
    if (remail_register_converters_called)
	return;

    REGISTER_MT_DEFHANDLER(MIME_TYPE_LEAF,leaf_remail_1);

    /* for message/rfc822 there is own parser, it should apply
       before default parser...
     */
    REGISTER_MT_DEFHANDLER(MIME_TYPE_MESSAGE,message_remail_1);

    REGISTER_MT_HANDLER(MIME_TYPE_MESSAGE,"rfc822",rfc822_remail_1);

    REGISTER_MT_DEFHANDLER(MIME_TYPE_MULTIPART,multipart_remail_1);

    REGISTER_MT_HANDLER(MIME_TYPE_MULTIPART,"signed",signed_remail_1);

    remail_register_converters_called = 1;
}

/* Creates temporary file */
static FILE * read_mime_body(infile,mime_part) 
     FILE *infile; 
     mime_t *mime_part;
{
    static int count = 0;
    int debug = 0;

    int	bytes = 0; /* the total number of bytes read so far */
    char *fname;
    FILE * F;

    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    if (!tmp)
        return NULL;
 
    if (mime_part->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"read_mime_body",
		   "Bad magic number");


    fname = elm_message(FRM("%selmmimebody.%d.%d"), 
			      tmp, getpid (),count++);
    F     = safeopen_rdwr(fname,NULL);
    
    if (!F) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingCreateMimeBodyTemp,
			  "Failed to create file %s when resending mail"),
		  fname);

	free(fname);  fname= NULL;
	return NULL;
    }

    DPRINT(Debug,10,(&Debug, 
		     "read_mime_body: length=%d, tmpfile=%s, offset=%ld\n",
		     mime_part->length,fname,
		     ftell(infile)));

    unlink(fname);  /* We can unlink it now ... */
    free(fname);  fname= NULL;
    
    while (1) {
	int ch;
    
	if (bytes >= mime_part->length)
	    break;

	if (!debug) {
	    DPRINT(Debug,15,(&Debug, "read_mime_body: ->"));
	    debug = 1;
	}


	ch =  getc(infile);
	if (EOF == ch) {
	    if (debug) {
		DPRINT(Debug,15,(&Debug, "\n"));
	    }
	    DPRINT(Debug,10,(&Debug, 
			     "read_mime_body: EOF  bytes=%d\n",
			     bytes));
	    break;
	}

	if ('\n' == ch) {
	    if (debug) {
		DPRINT(Debug,15,(&Debug, "\n"));
		debug = 0;
	    }
	} else {
	    DPRINT(Debug,15,(&Debug, "%c",ch));
	}


	bytes++;

	putc(ch,F);
    }

    if (debug) {
	DPRINT(Debug,15,(&Debug, "\nread_mime_body: ... NO NL\n"));
    }


    if (ferror(infile) || EOF == fflush(F)) {

	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingCopyMimeBodyTemp,
			  "Failed to copy mime body when resending mail"));

	fclose(F);
		  
	return NULL;
    }

    DPRINT(Debug,10,(&Debug, 
		     "read_mime_body: %d bytes readed\n",
		     bytes));


    return F;
}

/*  ENCODING_ILLEGAL      -- encoding is not changed ... 
    ENCODING_EXPERIMENTAL -- encoding is not changed ...
*/
static int convert_body(infile,mime_part,offset,
			new_body,new_encoding,is_binary,mailer_level,
			header_charset)
     FILE *infile; 
     mime_t *mime_part;
     long offset;
     FILE **new_body; 
     enum encoding *new_encoding;
     int is_binary;
     enum mailer_capab  mailer_level;
     charset_t header_charset;

{
    struct media_type_handle * H = NULL;
    int is_default = 0;
    int need_enc = 0;
    long new_offset;


    FILE * old_body = NULL;
    int idx = 0;

    *new_body     = NULL;
    *new_encoding = ENCODING_NONE;

    if (mime_part->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"convert_body",
		   "Bad magic number");

    new_offset = mime_part->offset - offset;

    DPRINT(Debug,8,(&Debug,
		    "convert_body:  base=%ld offset=%ld => %ld\n",
		    offset,mime_part->offset,new_offset));


    if (-1 == fseek(infile,new_offset,SEEK_SET)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingSeekMimeBodyFailed,
			  "Failed to seek beginning of mime body when resending mail"));
	return 0;
    }

    old_body = read_mime_body(infile,mime_part);

    if (! old_body)
	return 0;

    need_enc        = needs_encoding(old_body,NULL,0);

    DPRINT(Debug,8,(&Debug,
		    "convert_body: need_enc=%d%s%s\n",
		    need_enc,
		    (need_enc & HAVE_BINARY) ? " BINARY" : "",
		    (need_enc & HAVE_8BIT)   ? " 8BIT"   : ""));

    if ((mime_part->encoding < ENCODING_NONE || 
	 mime_part->encoding > ENCODING_BINARY) &&
	( 0 != (need_enc & HAVE_BINARY) ||
	  0 != (need_enc & HAVE_8BIT))) {

	/* If data is already encoded it must not have BINARY or 8BIT
	   after encoding ....
	*/

	DPRINT(Debug,8,(&Debug,
			"convert_body: Part is encoded, but result is not 7-bit\n"));

	if (ENCODING_ILLEGAL       == mime_part->encoding ||
	    ENCODING_EXPERIMENTAL  == mime_part->encoding)
	  lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingFailedBadEncoding0,
			  "Found bad encoding for mime type %s/%s when resending mail"),
		  get_major_type_name(mime_part->TYPE), 
		  get_subtype_name(mime_part->TYPE));
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmResendingFailedBadEncoding,
			      "Found bad encoding %s for mime type %s/%s when resending mail"),
		      ENCODING(mime_part->encoding),
		      get_major_type_name(mime_part->TYPE), 
		      get_subtype_name(mime_part->TYPE));

	fclose(old_body);

	return 0;
    }


    /* Do recursive checking first, so that errors are reported */

    DPRINT(Debug,8,(&Debug,
		    "convert_body: Searching converter for %s/%s\n",
		    get_major_type_name(mime_part->TYPE), 
		    get_subtype_name(mime_part->TYPE)));

    while(walk_mt_handler(mime_part->TYPE,&H,&is_default,handle_remail)) {

	enum encoding   encoding = ENCODING_NONE;
	FILE                 * F = NULL;                

	convert_mime_type_f * f;

	if (H->type != handle_remail) {
            mime_panic(__FILE__,__LINE__,"convert_body",
                       "Unexpected handler type");
            continue; /* not reached */
        }

	rewind(old_body);

	f = H->p.remail->converter;

	DPRINT(Debug,8,(&Debug,
			"convert_body: Searching converter for %s/%s -> try #%d trying %p\n",
			get_major_type_name(mime_part->TYPE), 
			get_subtype_name(mime_part->TYPE),
			idx,f));

	if (f(old_body,mime_part,mime_part->offset  /* ? ? ? */,
	      &F,&encoding,is_binary,need_enc,
	      mailer_level,header_charset)) {

	    *new_body        = F;
	    *new_encoding    = encoding;

	    DPRINT(Debug,10,(&Debug,
                             "convert_body: converter succeed for %s/%s, try %d\n",
			     get_major_type_name(mime_part->TYPE), 
			     get_subtype_name(mime_part->TYPE),
			     idx));

	    fclose(old_body); old_body = NULL;

	    return 1;

	}

	idx++;

	if (F)
	    fclose(F);
    }
    
    DPRINT(Debug,8,(&Debug,
                    "convert_body: No one converter succeed for type %p   %s/%s (%d tried)\n",
                    mime_part->TYPE,
		    get_major_type_name(mime_part->TYPE), 
		    get_subtype_name(mime_part->TYPE),
		    idx));


    /* Use failback, if converters failed because of erros.
       Errors are now reported 
    */

    if (ENCODING_BINARY == mime_part->encoding) {
	if (0 == (need_enc & HAVE_8BIT) &&
	    0 == (need_enc & HAVE_BINARY)) {
	    DPRINT(Debug,8,(&Debug,
			    "convert_body: 7 bit body -- resetting binary encoding\n"));
	    *new_encoding = ENCODING_7BIT;
	    goto process_as_is;
	}

	if (0 == (need_enc & HAVE_BINARY)) {
	    DPRINT(Debug,8,(&Debug,
			    "convert_body: 8 bit body -- resetting binary encoding\n"));
	    *new_encoding = ENCODING_8BIT;
	    goto process_as_is;
	}

	if (mailer_level >= mailer_binary) {

	    DPRINT(Debug,8,(&Debug,
			    "convert_body: Mailer supports BINARYMIME -- data is handled as is\n"));
	    
	    *new_encoding = ENCODING_ILLEGAL; /* Do not rewrite Content-Transfer-Encoding */
	    goto process_as_is;
	}
    }

    if (ENCODING_8BIT == mime_part->encoding &&
	0 == (need_enc & HAVE_BINARY)) {

	if (0 == (need_enc & HAVE_8BIT)) {
	    DPRINT(Debug,8,(&Debug,
			    "convert_body: 7 bit body -- resetting 8 bit encoding\n"));
	    *new_encoding = ENCODING_7BIT;
	    goto process_as_is;
	}

	if (mailer_level >= mailer_8bit) {
	    DPRINT(Debug,8,(&Debug,
			    "convert_body: Mailer supports 8BITMIME -- 8 bit data is handled as is\n"));
	    
	    *new_encoding = ENCODING_ILLEGAL; /* Do not rewrite Content-Transfer-Encoding */
	    goto process_as_is;
	}
    }

    if ((ENCODING_NONE == mime_part->encoding ||
	 ENCODING_7BIT == mime_part->encoding) &&
	0 == (need_enc & HAVE_BINARY) &&
	0 == (need_enc & HAVE_8BIT)) {

	FILE * result;

	DPRINT(Debug,8,(&Debug,
			"convert_body: 7 bit data is handled as is\n"));

	*new_encoding = ENCODING_ILLEGAL; /* Do not rewrite Content-Transfer-Encoding */

    process_as_is:
	result = NULL;

	if (is_binary) {
	    DPRINT(Debug,8,(&Debug,
			    "convert_body: binary: No conversions\n"));

	    rewind(old_body);
	    *new_body = old_body;
	    return 1;
	}

	result = give_output_file();
	if (!result) {
	    fclose(old_body);
	    return 0;
	}

	rewind(old_body);

	while(1) {
	    int ch = getc(old_body);

	    if (EOF == ch)
		break;

	    if ('\r' == ch) {
		int ch1 = getc(old_body);

		putc(ch,result);
		if (EOF == ch1)
		    break;

		putc(ch1,result);

	    } else if ('\n' == ch) {
		/* Temporary convert \n => \r\n
		   in that point it is not known if caller 
		   will be using ENCODING_BINARY on parent type
		*/
		  
		putc('\r',result);
	    }

	    putc(ch,result);			    
	}

	if (ferror(old_body) || EOF == fflush(result)) {

	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmResendingCopyMimeBodyTemp,
			      "Failed to copy mime body when resending mail"));

	    fclose(old_body);
	    fclose(result);
	    return 0;
	}

	fclose(old_body);
	rewind(result);

	DPRINT(Debug,8,(&Debug,
			"convert_body: Done LR -> CRLF conversion\n"));
       
	*new_body = result;
	return 1;
    }


    fclose(old_body); old_body = NULL;

    lib_error(CATGETS(elm_msg_cat, ElmSet, 
		      ElmResendingFailedToHandle,
		      "Failed to handle mime type %s/%s when resending mail"),
	      get_major_type_name(mime_part->TYPE), 
	      get_subtype_name(mime_part->TYPE));

    return 0;
}

/* ------------------------------------------------------------------------------  */

typedef int header_field_writer_f P_((struct out_state *mailer,
				      struct header_rec *current_header,
				      enum encoding top_encoding));

S_(header_field_writer_f write_from)
static int write_from P_((struct out_state *mailer,
			   struct header_rec *current_header,
			   enum encoding top_encoding));
static int write_from(mailer,current_header,top_encoding)
     struct out_state *mailer;
     struct header_rec *current_header;
     enum encoding top_encoding;
{
    if (!current_header->from)
	return 0;
    write_addr_header(mailer,"From",
		      current_header->from,
		      top_encoding,1,
		      current_header->header_charset,0);

    return 1;
}

S_(header_field_writer_f write_to)
static int write_to P_((struct out_state *mailer,
			struct header_rec *current_header,
			enum encoding top_encoding));
static int write_to(mailer,current_header,top_encoding)
     struct out_state *mailer;
     struct header_rec *current_header;
     enum encoding top_encoding;
{
    if (!current_header->to)
	return 0;
    write_addr_header(mailer,"To",current_header->to,
		      top_encoding,1,
		      current_header->header_charset,1);

    return 1;
}

S_(header_field_writer_f write_cc)
static int write_cc P_((struct out_state *mailer,
			struct header_rec *current_header,
			enum encoding top_encoding));
static int write_cc(mailer,current_header,top_encoding)
     struct out_state *mailer;
     struct header_rec *current_header;
     enum encoding top_encoding;
{
    if (!current_header->cc)
	return 0;
    write_addr_header(mailer,"Cc",current_header->cc,
		      top_encoding,1,
		      current_header->header_charset,1);

    return 1;
}


S_(header_field_writer_f write_reply_to)
static int write_reply_to P_((struct out_state *mailer,
			struct header_rec *current_header,
			enum encoding top_encoding));
static int write_reply_to(mailer,current_header,top_encoding)
     struct out_state *mailer;
     struct header_rec *current_header;
     enum encoding top_encoding;
{
    if (!current_header->reply_to)
	return 0;
    write_addr_header(mailer,"Reply-To",current_header->reply_to,
		      top_encoding,1,
		      current_header->header_charset,1);

    return 1;
}



S_(header_field_writer_f write_in_reply_to)
static int write_in_reply_to P_((struct out_state *mailer,
				 struct header_rec *current_header,
				 enum encoding top_encoding));
static int write_in_reply_to(mailer,current_header,top_encoding)
     struct out_state *mailer;
     struct header_rec *current_header;
     enum encoding top_encoding;
{
    if (!current_header->in_reply_to)
	return 0;
    write_references_header(mailer,"In-Reply-To",current_header->in_reply_to,
			    top_encoding,1,
			    current_header->header_charset);
    
    return 1;
}

S_(header_field_writer_f write_references)
static int write_references P_((struct out_state *mailer,
			   struct header_rec *current_header,
			   enum encoding top_encoding));
static int write_references(mailer,current_header,top_encoding)
     struct out_state *mailer;
     struct header_rec *current_header;
     enum encoding top_encoding;
{
    if (!current_header->references)
	return 0;
    write_references_header(mailer,"References",current_header->references,
			    top_encoding,1,
			    current_header->header_charset);
    
    return 1;
}

S_(header_field_writer_f write_subject)
static int write_subject P_((struct out_state *mailer,
			   struct header_rec *current_header,
			   enum encoding top_encoding));
static int write_subject(mailer,current_header,top_encoding)
     struct out_state *mailer;
     struct header_rec *current_header;
     enum encoding top_encoding;
{
    if (!current_header->subject)
	return 0;
    write_string_header(mailer,"Subject",current_header->subject,
			top_encoding,1,
			current_header->header_charset);

    return 1;
}


S_(header_field_writer_f write_msgid)
static int write_msgid P_((struct out_state *mailer,
			   struct header_rec *current_header,
			   enum encoding top_encoding));
static int write_msgid(mailer,current_header,top_encoding)
     struct out_state *mailer;
     struct header_rec *current_header;
     enum encoding top_encoding;
{
    if (!current_header->message_id)
	return 0;
    write_msgid_header(mailer,"Message-ID",
		      current_header->message_id,
		      top_encoding,1,
		      current_header->header_charset);

    return 1;
}


static struct header_handler {
    header_ptr              hdr_name;
    header_field_writer_f * func;
}        * header_handlers      = NULL;
static int header_handler_count = 0;

static void add_handler P_((const char *name, header_field_writer_f *f));
static void add_handler(name,f)
     const char *name; 
     header_field_writer_f *f;
{
    header_handlers = safe_array_realloc(header_handlers,
					 (header_handler_count+1),
					 sizeof (header_handlers[0]));
    
    header_handlers[header_handler_count].hdr_name = find_header(name,1);
    header_handlers[header_handler_count].func = f;

    DPRINT(Debug,12,(&Debug,
		     "add_handler: %d: hdr_name=%p func=%p name=%s\n",
		     header_handler_count,
		     header_handlers[header_handler_count].hdr_name,
		     header_handlers[header_handler_count].func,
		     name));
		        
    header_handler_count++;
}

static int is_handled_header P_((header_ptr hdr_name,
				 struct header_handler *handled_headers,
				 int                    handled_header_count));
static int is_handled_header(hdr_name,handled_headers,handled_header_count)
     header_ptr hdr_name;
     struct header_handler *handled_headers;
     int                    handled_header_count;
{
    int i;

    for (i = 0; i < handled_header_count; i++) 
	if (hdr_name == handled_headers[i].hdr_name) 
	    return 1;

    return 0;
}

static int can_handle_header P_((header_ptr hdr_name,
				 struct header_handler **handled_headers,
				 int                    *handled_header_count));

static int can_handle_header(hdr_name,handled_headers,handled_header_count)
     header_ptr hdr_name;
     struct header_handler **handled_headers;
     int                    *handled_header_count;
{
    const char * name UNUSED_VAROK = give_header_name(hdr_name);
    int i;

    if (is_handled_header(hdr_name,*handled_headers,*handled_header_count)) {
	DPRINT(Debug,9,(&Debug,
			"can_handle_header=1: name=%s  hdr_name=%p duplicate header (OK)\n",
			name,hdr_name));
	return 1;
    }

    for (i = 0; i < header_handler_count; i++)
	if (hdr_name == header_handlers[i].hdr_name) 
	    goto found;

    DPRINT(Debug,9,(&Debug,"can_handle_header=0: name=%s hdr_name=%p\n",
		    name,hdr_name));
    return 0;
	
 found:

    (*handled_headers) = safe_array_realloc((*handled_headers),
					    ((*handled_header_count)+1),
					    sizeof ((*handled_headers)[0]));
       
    (*handled_headers)[(*handled_header_count)] = header_handlers[i];
    (*handled_header_count)++;

    DPRINT(Debug,9,(&Debug,"can_handle_header=1: name=%s\n",
		    name));
    return 1;
}


static void remail_register_handlers P_((void));
static void remail_register_handlers()
{
    
    if (header_handlers)
	return;
    
    add_handler("From",        write_from);
    add_handler("To",          write_to);
    add_handler("CC",          write_cc);
    add_handler("Reply-To",    write_reply_to);
    add_handler("In-Reply-To", write_in_reply_to);
    add_handler("References",  write_references);
    add_handler("Message-ID",  write_msgid);
    add_handler("Subject",     write_subject);

    DPRINT(Debug,9,(&Debug,"remail_register_handlers: %d mail handlers\n",
		    header_handler_count));


    add_field_handler("Content-Type",        write_content_type);
    add_field_handler("Content-Disposition", write_content_disposition);
    add_field_handler("Content-Description", write_content_description);
    
    DPRINT(Debug,9,(&Debug,"remail_register_handlers: %d mime handlers\n",
		    mime_field_handler_count));
}

static int handle_headers P_((struct header_handler * handled_headers,
			      int                     handled_header_count,
			      struct out_state *mailer,
			      struct header_rec *current_header,
			      enum encoding top_encoding));
static int handle_headers(handled_headers,handled_header_count,mailer,
			  current_header,top_encoding)
     struct header_handler * handled_headers;
     int                     handled_header_count;
     struct out_state *mailer;
     struct header_rec *current_header;
     enum encoding top_encoding;
{
    int i;
    
    for (i = 0; i < handled_header_count; i++) {
	if (! handled_headers[i].func(mailer,current_header,top_encoding)) {
	    
	    return 0;
	}
    }
    
    if (state_puts (i > 1 ?
		    "X-MIME-Autoconverted: fields " :
		    "X-MIME-Autoconverted: field ",
		    mailer) == EOF) {

	return 0;
    }
    
    for (i = 0; i < handled_header_count; i++) {
	const char * hdr_name = give_header_name(handled_headers[i].hdr_name);
	
	if (i > 0) 
	    state_puts (", ",mailer);   
	state_puts (hdr_name,mailer);	    
    }
    
    if (state_printf (mailer, FRM(" by %s\n"),
		      hostfullname)  == EOF) {

	return 0;
    }	

    return 1;
}

int  convert_remail_message_1(infile,current_header,mailer_level,
			      target,encoding, sender)
     FILE *infile;
     struct header_rec *current_header;
     enum mailer_capab  mailer_level;
     struct out_state      * target;
     enum encoding          * encoding;
     const char *sender;
{
    FILE           * new_body       = NULL;
    enum encoding  new_encoding     = ENCODING_NONE;
    header_list_ptr headers         = NULL;
    header_list_ptr next_hdr;

    int body_bytes = 0;
    int buf_len;
    char buffer[VERY_LONG_STRING];

    int is_binary                   = 0;
    int r = 0;
    int mime_version_seen = 0;
    int have_8bit_header = 0;
    int mime_version_added = 0;

   
    struct header_handler * handled_headers = NULL;
    int                     handled_header_count = 0;

    struct mime_field_handler *handled_fields        = NULL;
    int                        handled_field_count   = 0;
	       
    remail_register_converters();
    remail_register_handlers();

    if (current_header->mime_rec.magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"convert_remail_message_1",
		   "Bad magic number");

    if ( current_header->mime_rec.encoding == ENCODING_BINARY && 
	 !(current_header->binary)) {
	DPRINT(Debug,2,(&Debug, 
			"convert_remail_message_1: Original is binary message"));
       
	is_binary = 1;
    }

    r = get_current_mime_hdr(infile,& (current_header->mime_rec), (long) 0,
			     &headers);
    if (! r) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingFailedHeader,
			  "Failed to read mail headers when resending mail"));
	goto fail;
    }

    for (next_hdr = headers; 
	 next_hdr; 
	 next_hdr = next_hdr -> next_header) {
	const char * hdr_name = give_header_name(next_hdr->header_name);

	if (check_8bit_str(next_hdr->body)) {
	    DPRINT(Debug,8,(&Debug, 
			    "convert_remail_message_1: Found 8-bit data from %s header field\n",
			    hdr_name));		

	    if (!allow_no_hdrencoding) {
		if (!can_handle_header(next_hdr->header_name,
				       &handled_headers,
				       &handled_header_count) 
		    
		    &&
		    
		    !can_handle_field(next_hdr->header_name,
				      &handled_fields,
				      &handled_field_count))	    
		    
		    have_8bit_header = 1;
	    }
	}

	if (0 == istrcmp("MIME-Version", hdr_name))
	    mime_version_seen = 1;

    }

    if (have_8bit_header) {

	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResending8BitHeaderUnsupported,
			  "Can't handle 8-bit data on mail headers when resending mail"));

	r = 0;
	goto fail;
    }
    
    r = convert_body(infile, & (current_header->mime_rec),
		     (long) 0,
		     &new_body,&new_encoding, is_binary,
		     mailer_level,
		     current_header->header_charset);
    if (!r) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmResendingFailedBody,
			  "Failed to process mail when resending it"));

	goto fail;

    }

    /* Let state routines be aware of encoding */
    if (new_encoding == ENCODING_BINARY)
	set_out_state_EOLN(target,1);

    if (!mime_version_seen || 
	handled_field_count > 0 
	||
	(new_encoding != ENCODING_ILLEGAL &&
	 new_encoding != ENCODING_NONE    &&
	 new_encoding != ENCODING_EXPERIMENTAL &&
	 new_encoding != current_header->mime_rec.encoding)
	) {	
	if (state_printf (target, FRM("MIME-Version: 1.0\n")) == EOF) {
	    r = 0;
	    goto fail;
	}

	mime_version_added = 1;
    }


    if (sender &&
	state_printf (target, FRM("Sender: %s\n"), sender) == EOF) {
	r = 0;
	goto fail;
    }

    if (new_encoding != ENCODING_ILLEGAL &&
	new_encoding != ENCODING_NONE    &&
	new_encoding != ENCODING_EXPERIMENTAL &&
	new_encoding != current_header->mime_rec.encoding) {

	if (state_printf (target, FRM("Content-Transfer-Encoding: %s\n"),
			  ENCODING(new_encoding)) == EOF) {
	    r = 0;
	    goto fail;
	}

	if (state_printf (target, 
			  FRM("X-MIME-Autoconverted: from %s to %s by %s\n"),
			  ENCODING(current_header->mime_rec.encoding),
			  ENCODING(new_encoding),
			  hostfullname)  == EOF) {
	    r = 0;
	    goto fail;
	}
    }

    if (handled_headers) {
	if (! handle_headers(handled_headers,handled_header_count,
			     target,current_header,new_encoding)) {

	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmResending8BitHeaderUnsupported,
			      "Can't handle 8-bit data on mail headers when resending mail"));
	    
	    r = 0;
	    goto fail;	    
	}

    }

    if (handled_fields) {
	if (! handle_fields(handled_fields,handled_field_count,
			    target,
			    & current_header->mime_rec,
			    new_encoding,
			    current_header->header_charset)) {
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmResending8BitHeaderUnsupported,
			      "Can't handle 8-bit data on mail headers when resending mail"));
	    
	    r = 0;
	    goto fail;	    
	}	
    }

    for (next_hdr = headers; 
	 next_hdr; 
	 next_hdr = next_hdr -> next_header) {
	const char * hdr_name = give_header_name(next_hdr->header_name);
	
	if (0 == istrcmp("Sender",hdr_name) ||
	    0 == istrcmp("Bcc",hdr_name) ||
	    0 == istrcmp("Content-Length",hdr_name) ||
	    0 == istrcmp("Status",hdr_name)) 
	    continue;
	
	
	if (0 == istrcmp("Content-Transfer-Encoding",hdr_name) &&
	    new_encoding != ENCODING_ILLEGAL  &&
	    new_encoding != ENCODING_EXPERIMENTAL &&
	    new_encoding != current_header->mime_rec.encoding)
	    continue;
	
	if (0 == istrcmp("MIME-Version",hdr_name) &&
	    mime_version_added)
	    continue;

	if (is_handled_header(next_hdr->header_name,
			      handled_headers,
			      handled_header_count))
	    continue;

	if (is_handled_field(next_hdr->header_name,
			     handled_fields,
			     handled_field_count))
	    continue;
	
	state_write_raw_header(target,&NULL_decode_opt,next_hdr);
    }
    	
    /* Must use state_printf so EOLN_is_CRLF conversion occurs! */
    if (state_printf(target, 
		     FRM("\n")) == EOF) {
        r = 0;
	goto fail;
    }      
    
    *encoding = new_encoding;

    rewind(new_body);

    DPRINT(Debug,5,(&Debug,
		    "convert_remail_message_1: Doing copy\n"));
     
    while ((buf_len = mail_gets(buffer, sizeof buffer, new_body)) > 0) {
      	
	int err;
	
	body_bytes += buf_len;
	
	DPRINT(Debug,15,(&Debug,
			 "convert_remail_message_1: -> %.*s",
			 buf_len,buffer));
	if (buf_len < 1 || buffer[buf_len-1] != '\n') {
	    DPRINT(Debug,15,(&Debug,
			     "\nconvert_remail_message_1 <-- NO NEWLINE\n"));
	    
	} else if (!is_binary || new_encoding != ENCODING_BINARY) {
	    
	    /* Take care of CRLF => LF conversion or
	       LF -> CRLF conversion 
	    */
	    state_convert_EOLN(buffer,&buf_len,sizeof buffer,target);
	}
	    
	err = state_put(buffer, buf_len, target);
	if (err != buf_len) {
	    DPRINT(Debug,1,(&Debug, 
			    "convert_remail_message_1 fails (err=%d != %d=buf_len)\n",
			    err,buf_len));
	    r = 0;
	    goto fail;
	}
    }
        
    DPRINT(Debug,5,(&Debug,
		    "convert_remail_message_1: Read %d bytes from body\n",
		    body_bytes));
    
    if (ferror(new_body)) {
	DPRINT(Debug,1,(&Debug, 
			"convert_remail_message_1 fails to read new body\n"));
	r = 0;
    }
    
 fail:

    if (headers)
	delete_headers(&headers);
    if (new_body) {
	fclose(new_body);
	new_body = NULL;
    }

    if (handled_headers)
        free(handled_headers);

    if (handled_fields)
	free(handled_fields);

    DPRINT(Debug,8,(&Debug, 
		    "convert_remail_message_1=%d\n",
		    r));

    return r;
}

static int convert_remail_message P_((FILE *infile,
				      struct header_rec *current_header,
				      struct mailer_info *mailer_info,
				      struct mail_send_state * mailer,
				      enum encoding          * encoding,
				      const char             * sender
				      ));
static int convert_remail_message(infile,current_header,mailer_info,mailer,
				  encoding, sender)
     FILE *infile;
     struct header_rec *current_header;
     struct mailer_info *mailer_info;
     struct mail_send_state * mailer;
     enum encoding          * encoding;
     const char             * sender;
{
    enum mailer_capab  mailer_level   = get_mailer_level(mailer_info);
    struct out_state              * target = get_mail_outfd(mailer);

    int r = convert_remail_message_1(infile,current_header,mailer_level,
				     target,encoding,
				     sender);
    
    DPRINT(Debug,8,(&Debug, 
		    "convert_remail_message=%d\n",
		    r)); 

    return r;
}

/* Return 1 if remailing is wanted,  
   also EOF is possible 
*/
static enum remail_confirm_res {
    remail_conf_EOF   = EOF,
    remail_conf_FALSE = FALSE,
    remail_conf_TRUE  = TRUE
    
} query_remail_confirm P_(( struct mailing_headers        * headers,
			    struct menu_context           * parent_page,
			    enum rp_classify_domain       * env_from_domain_class_p,
			    struct classify_domain_result * env_from_cdr,
			    struct mailer_info            * mailer_info,
			    struct AliasView              * aview));
static enum remail_confirm_res query_remail_confirm(headers,parent_page,
						    env_from_domain_class_p,
						    env_from_cdr,mailer_info,
						    aview)
     struct mailing_headers        * headers;
     struct menu_context           * parent_page;
     enum rp_classify_domain       * env_from_domain_class_p;
     struct classify_domain_result * env_from_cdr;
     struct mailer_info            * mailer_info;
     struct AliasView              * aview;
{
    struct menu_context *page        = new_menu_context();
                                       /* Clear page and make 4 line prompt area */
    struct menu_context *prompt_area = prompt_mailmsg_page_interactive(page);

    struct string *prompt_mssg_s =
	format_string(CATGETS(elm_msg_cat, ElmSet, ElmSureYouWantToRemail,
			      "Are you sure you want to remail this message (%c/%c)? "),
		      *def_ans_yes, *def_ans_no);
    
    enum remail_confirm_res ret = remail_conf_FALSE;
    int do_redraw = 1;
    int LINES, COLUMNS;
    int env_from_message = 1;
    int bad_cmd = 0;
    int cmd;			/* command to perform			*/
    int x_coord = 0, y_coord = 0;

    struct hdrmenu_context redraw_context = NULL_hdrmenu_context;
    int force_cmd = '\0';	    /* command to do, '\0' to prompt user for cmd    */
    
    
    menu_get_sizes(page,&LINES, &COLUMNS);
    
    for (;;) {
	int def_cmd = *def_ans_yes;   
	menu_set_default(page);
	
	if (0) {
	redraw: 
	     do_redraw = 1;
	}

    resize_mark:
	if (menu_resized(page)) {
	    
	    menu_get_sizes(page,&LINES, &COLUMNS);

	    menu_subpage_relocate(prompt_area,page,LINES-4,4);

	    do_redraw = 1;

	} else if (menu_need_redraw(page))
	    do_redraw = 1;
	
	if (do_redraw) {

	    do_redraw = 0;
	    
	    menu_ClearScreen(page);
	    menu_StartXX(page,pg_BOLD);

	    menu_print_format_center(page,0, 
				     CATGETS(elm_msg_cat, ElmSet,
					     ElmBounceScreenTitle, 
					     "Bounce Pre-Send Screen"));

	    menu_EndXX(page,pg_BOLD);

	    show_last_error();

	    show_presend_headers(headers,
				 /* hdr_charset */ ASCII_SET,
				 page,
				 show_prehdr_bounce);	    

	    
	    if (headers->env_from)
		mail_show_env_from(headers,*env_from_domain_class_p,page);
	    
	    menu_redraw_children(page);
	    
	}

	if (env_from_message && headers->env_from) {
	    env_from_message = 0;
	    
	    if (mail_env_from_def_edit(*env_from_domain_class_p,
				       headers,env_from_cdr))
		def_cmd = 'R';
	}

	/* complain if last entry was bad */
	if (bad_cmd) {
	    menu_Write_to_screen(prompt_area,
				 FRM("%c??"), 07);
	    FlushBuffer();
	    
	    if (sleepmsg > 0)
		error_sleep((sleepmsg + 1) / 2);
	    bad_cmd = 0;
	}

#ifdef BACKGROUD_PROCESSES      
	if (handle_sigchld)
	    sigchld_handler();
#endif

	if (menu_resized(prompt_area) ||
	    menu_need_redraw(prompt_area)) {
	
	    menu_ClearScreen(prompt_area);
	    
	    menu_PutLineX(prompt_area,
			  0, 0, FRM("%S"),prompt_mssg_s);
	    
	    show_last_error();
	}

	/* if we don't have a cmd, display prompt and get response from user */
	if (force_cmd != '\0' &&
	    isascii(force_cmd)) {

	    cmd = force_cmd;     /* Removed tolower() from here */
		
	    force_cmd = '\0';

	} else {
	    clear_hdrmenu_context(&redraw_context);
	    
	    menu_PutLineX(prompt_area,
			  0, 0, FRM("%S"),prompt_mssg_s);
	
	    menu_GetXYLocation(prompt_area,
			       &x_coord, &y_coord);
	    
	    menu_CleartoEOLN(prompt_area);
	    
	    menu_PutLineX(prompt_area,
			  0, y_coord, FRM("%c"),def_cmd);
	    FlushBuffer();
	
	    menu_MoveCursor(prompt_area,
			    x_coord, y_coord);
	    cmd = menu_ReadCh(prompt_area,REDRAW_MARK|READCH_resize|READCH_sig_char);
	    
	    if (isascii(cmd) && 
		isprint(cmd)) {
		DPRINT(Debug,8,(&Debug,"... command: '%c'\n",
				cmd));
	    } else {
		DPRINT(Debug,8,(&Debug,"... command: %d\n",
				cmd));
	    }

	    if (cmd == REDRAW_MARK) {
		DPRINT(Debug,4, (&Debug,"       .... redrawing\n"));
		menu_ClearScreen(page);   /* Clear possible redraw mark */
		
		/* Call refresh routines of children */
		menu_redraw_children(page);
		
		if (menu_need_redraw(prompt_area))		    
		    menu_ClearScreen(prompt_area);   /* Clear redraw mark from prompt_area*/
		
		goto redraw;
	    }

	    if (cmd == RESIZE_MARK) {
		
		DPRINT(Debug,4, (&Debug,"       .... resizing\n"));
		goto resize_mark;
	    }

	    if (cmd == EOF)
		goto handle_EOF;
	    
	    clear_error();
	    
	    /* Put cursor pack to correct place */
	    menu_MoveCursor(prompt_area,
			    x_coord, y_coord);

	}
	    
	switch (cmd) {
	case '\n':
	case '\r':
	    cmd = def_cmd;

	    if (isascii(cmd) && 
		isprint(cmd)) {
		DPRINT(Debug,8,(&Debug,"... default command: '%c'\n",
				cmd));
	    } else {
		DPRINT(Debug,8,(&Debug,"... default command: %d\n",
				cmd));
	    }
	}
	
	if (cmd == *def_ans_yes) {
	    menu_Write_to_screen(prompt_area,
				 CATGETS(elm_msg_cat, ElmSet, ElmVfyMenuSend,
					 "Send"));
	    FlushBuffer();
	    
	    ret =  remail_conf_TRUE;
	    break;
	    
	} else if (cmd == *def_ans_no ) {

	    menu_Write_to_screen(page,
                                 CATGETS(elm_msg_cat, ElmSet, ElmNoWord, 
                                         "No."));
	    FlushBuffer();

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBounceCancelled,
			      "Bounce of message canceled."));
	    
	    ret =  remail_conf_FALSE;
	    break;
	} else if (TERMCH_interrupt_char == cmd) {

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBounceCancelled,
			      "Bounce of message canceled."));
	    ret =  remail_conf_FALSE;
	    break;
	}

	/* handle command */
	switch (cmd) {
	case ctrl('L'):
	    goto redraw;

		
	case 'R':
	    {
		int c = mail_edit_return_path(headers, page);

		DPRINT(Debug,8,(&Debug,"mail_edit_return_path returned %d\n",c));

		if (headers->env_from)  {
		    *env_from_domain_class_p =
			env_from_classify_domain(headers->env_from,
						 env_from_cdr,
						 mailer_info);
		    env_from_message = 1;
		    if (env_from_cdr->domain_rewritten) {
			mail_show_env_from(headers,*env_from_domain_class_p,page);
			env_from_cdr->domain_rewritten = 0;
		    }
		} else
		    *env_from_domain_class_p = env_from_no_domain;
		
		if (0 == c)
		    bad_cmd = TRUE;
		else if (REDRAW_MARK == c)
		    goto redraw;		    

		menu_trigger_redraw(prompt_area);
	    }
	    break;

	default:
	    {
		int c = presend_action(headers,mailer_info,cmd,
				       &redraw_context,
				       /* hdr_charset */ ASCII_SET,
				       aview, page, prompt_area,
				       /* parent_message */ NULL,
				       show_prehdr_bounce);

				if (0 == c) {   /* Accept also uppercase letters */
		    int old = cmd;
#ifdef ASCII_CTYPE
		    if (isascii(cmd))
#endif
			cmd = tolower((unsigned char)cmd);
		    
		    if (old != cmd)
			c = presend_action(headers,mailer_info,cmd,
					   &redraw_context,
					   /* hdr_charset */ ASCII_SET,
					   aview, page, prompt_area,
					   /* parent_message */ NULL,
					   show_prehdr_bounce);

		    if (0 == c)
			bad_cmd = TRUE;
		    else if (REDRAW_MARK == c) {
			force_cmd = cmd;  /* repreat command after redraw */
			goto redraw;		    
		    }
		    clear_hdrmenu_context(&redraw_context);
		    
		} else if (REDRAW_MARK == c)
		    goto redraw;		 

		menu_trigger_redraw(prompt_area);   

	       
	    }
	    break;
	}
    }


    if (0) {	
    handle_EOF:
	DPRINT(Debug,2,(&Debug,"EOF on query_remail_confirm\n"));
	ret = remail_conf_EOF;
    }
    
    clear_hdrmenu_context(&redraw_context);
   
    if (prompt_area)
	erase_menu_context(&prompt_area);


    if (page)
	erase_menu_context(&page);

    if (parent_page)
	menu_trigger_redraw(parent_page);

    free_string(& prompt_mssg_s);
    
    
    DPRINT(Debug,8,(&Debug,"query_remail_confirm=%d",
		    ret));
    switch(ret) {
    case remail_conf_EOF:   DPRINT(Debug,8,(&Debug," remail_conf_EOF"));   break;
    case remail_conf_FALSE: DPRINT(Debug,8,(&Debug," remail_conf_FALSE")); break;
    case remail_conf_TRUE:  DPRINT(Debug,8,(&Debug," remail_conf_TRUE"));  break;
    }
    DPRINT(Debug,8,(&Debug,"\n"));
    
    return ret;
}

void remail(current_header,F1, aview, page, prompt_area) 
     struct header_rec *current_header;
     FILE *F1;
     struct AliasView     *aview;
     struct menu_context  *page;
     struct menu_context  *prompt_area;
{
  
    struct mail_send_state * mailer = NULL;

    enum encoding encoding = ENCODING_NONE;
    char title[80];
    struct mailing_headers mailing_headers;

    struct mailer_info *mailer_info   = get_mailer_info();
    int li,co;
    int delay_redraw = 0;
    int cnt;
    int r UNUSED_VAROK;
    enum rp_classify_domain env_from_domain_class   = env_from_no_domain;
    struct classify_domain_result    env_from_cdr   = NULL_classify_domain_result;
    enum remail_confirm_res do_remail = remail_conf_FALSE;
    int Exstat = 0;   /* OK */
    int have_eof = 0;
    char * sender = NULL;
    char * sender_domain = NULL;
    
    menu_get_sizes(prompt_area, &li, &co);   

    if (!mailer_info)
	return;

    /* Actually headers are not written. headers structure is just
     * used to pass mailing addresses to backend .
     */
    zero_mailing_headers(&mailing_headers);
    
    if (!current_header || !F1)
	return;

    sender =  from_addr_literal(mailer_info, &sender_domain);
    
    DPRINT(Debug,8,(&Debug,"remail: sender %s",
		    sender));

    if (sender_domain) {
	int sender_ok = 0;
	int          is_literal;
	const char * whitelisted_name = NULL;
	const char * reserved_name    = NULL;
	
	DPRINT(Debug,8,(&Debug,"; sender domain %s\n",
			sender_domain));
	
	sender_ok = build_address_classify_domain(sender_domain,
						  &is_literal,
						  &whitelisted_name,
						  &reserved_name);

	if (!sender_ok) {
	    build_address_classify_message(sender,sender_domain,
					   is_literal,
					   whitelisted_name,
					   reserved_name);

	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmResendDomainFail,
			      "Domain %s for Sender address %s not valid, resending canceled."),
		      sender_domain,
		      sender);

	    DPRINT(Debug,8,(&Debug,"remail: Bad domain %s, resending canceled\n",
			    sender_domain));
	    goto free_it;
	}
						  
    } else {
	DPRINT(Debug,8,(&Debug,"\n"));
    }
    
    
    /* free_mailing_headers will free this */
    if (!mailing_headers.env_from)
	mailing_headers.env_from = mailer_get_env_from(mailer_info);
    
    /* Read .mailheaders for Return-Path (for envelope sender actually) */
    import_mailheaders(&mailing_headers,
		       import_mh_returnpath);

    if (mailing_headers.env_from) {
	env_from_domain_class =
	    env_from_classify_domain(mailing_headers.env_from,
				     &env_from_cdr,
				     mailer_info);
    } else {
	DPRINT(Debug,8,(&Debug,"remail: ... no envelope sender (Return-Path)\n"));
    }
    
    if (! get_to(&mailing_headers.to,mailer_info, aview, page,
		 NULL, NULL,prompt_area))
	goto free_it;
  

    /** Glet's ensure the person really wants to 
	remail it... **/

    if (mailing_headers.env_from) {

	do_remail = query_remail_confirm(&mailing_headers,
					 page,
					 &env_from_domain_class,
					 &env_from_cdr,
					 mailer_info,
					 aview);

	if (remail_conf_EOF == do_remail) {
	     DPRINT(Debug,4,(&Debug, 
			    "remail: Got EOF ... \n"));
	}
					 
    } else {
	int ch;
    again:  
	display_to(mailing_headers.to, page,prompt_area);
	
	menu_ClearLine(prompt_area,2);
	menu_ClearLine(prompt_area,3);
	
	/* Show possible error seen on conversion */
	show_last_error();
	
	ch = prompt_letter(2,"",*def_ans_yes,
			   PROMPT_yesno|PROMPT_redraw_mark|PROMPT_ctrlL|
			   PROMPT_cancel,
			   prompt_area,
			   
			   CATGETS(elm_msg_cat, ElmSet, ElmSureYouWantToRemail,
				   "Are you sure you want to remail this message (%c/%c)? "),
			   *def_ans_yes, *def_ans_no);
	if (('L'&31)    == ch ||
	    REDRAW_MARK == ch) {
	    menu_ClearScreen(page);   /* Reset possible redraw flag */
	    
	    /* Call refresh routines of children */
	    menu_redraw_children(page);
	    
	    if (menu_need_redraw(prompt_area))		    
		menu_ClearScreen(prompt_area); /* clear redraw flag */
	    
	    delay_redraw++;   /* Can't trigger redraw yet... */
	    goto again;
	}
	if (delay_redraw)
	    menu_trigger_redraw(page);

	if (ch == *def_ans_no ||
	    TERMCH_interrupt_char == ch) { /* another day, another No... */
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBounceCancelled,
			      "Bounce of message canceled."));

	    do_remail = remail_conf_FALSE;
	    
	} else if (EOF == ch) {
	    DPRINT(Debug,4,(&Debug, 
			    "remail: Got EOF ... \n"));
	    do_remail = remail_conf_EOF;

	} else if (ch == *def_ans_yes) {
	    do_remail = remail_conf_TRUE;
	} else {
	    DPRINT(Debug,4,(&Debug, 
			    "remail: Unexpected answer ch=x%02x\n",
			    ch));
	}
    }

    switch (do_remail) {	
    case remail_conf_TRUE:
	DPRINT(Debug,8,(&Debug, 
			"remail: do_remail=remail_conf_TRUE\n"));
	
	
	{
	    char **argvt = argv_from_headers(&mailing_headers);
	    
	    if (! argvt) {
		DPRINT(Debug,4,(&Debug,"remail: No mail addresses\n"));
		goto free_it; 
	    }
	    
	    mailer = mailer_init(argvt,0,sendmail_verbose,mailer_info,
				 mailing_headers.env_from);
	    free_argv(&argvt);
	}

	if (!mailer) 
	    goto free_it;
	
	/** now let's copy the message into the newly opened
	    buffer... **/
	
	if (current_header->status & MIME_MESSAGE) {
	    
	    if (0 != mime_warnings(current_header)) {
		
		DPRINT(Debug,4,(&Debug, 
				"remail: Bad-MIME message .. remailing as is\n"));
		goto as_is;
	    }
	    
	    DPRINT(Debug,4,(&Debug, 
			    "remail: Doing possible mime conversions\n"));
	    
	    /* Do possible conversions   */
	    
	    if (!convert_remail_message(F1,current_header,
					mailer_info,mailer,
					&encoding,
					sender
					))
		goto free_it;
	    
	}  else {
	    
	    DPRINT(Debug,4,(&Debug, 
			"remail: Non-MIME message .. remailing as is\n"));
	    
	as_is:
	    
	    if (!copy_message_2(F1,
				current_header,		     
				get_mail_outfd(mailer), 
				CM_REMOVE_ENVELOPE | CM_REMAIL,
				0,
				sender)) {
		goto free_it;
	    }
	}    
	   
	if (mailing_headers.env_from) 
	    mail_env_from_presend(&env_from_domain_class,
				  &mailing_headers,&env_from_cdr);	   	  	
	
	if (! mailing_headers.to.addrs)
	    elm_sfprintf(title, sizeof title,
			 CATGETS(elm_msg_cat, ElmSet, ElmRemailToSomeone,
				 "Remail to someone..."));
	else if ((cnt = addr_list_item_count(mailing_headers.to.addrs)) == 1) {
	    int group;
	    const struct address * X = 
		addr_list_get_item(mailing_headers.to.addrs,0,&group);
	    const char * addr = address_get_ascii_addr(X);
	    
	    elm_sfprintf(title, sizeof title,
			 CATGETS(elm_msg_cat, ElmSet, ElmRemailTo,
				 "Remail to %.50s..."),
			 addr ? addr : "(someone)");
	    
	} else
	    elm_sfprintf(title, sizeof title,
			 CATGETS(elm_msg_cat, ElmSet, ElmRemailToXRecipients,
				 "Remail to %d recipients (To:)"),
			 cnt);
	
	
	r = mail_backend3(&mailer,remail_sent,encoding,title,remailing_message,
			  &Exstat,NULL,
			  out_state_ftell(get_mail_outfd(mailer)));
	
	DPRINT(Debug,4,(&Debug,"mail_backend2 returned %d; Exstat=%d\n",
			r,Exstat));
	
	/* 0   may be:    start failed
	 *                mailing contined on backend
	 *
	 * 1              command completed, perhaps with failure 
	 */
	break;
    case remail_conf_FALSE:
	DPRINT(Debug,8,(&Debug, 
			"remail: do_remail=remail_conf_FALSE, no remail\n"));
	break;
    case remail_conf_EOF:
	DPRINT(Debug,8,(&Debug, 
			"remail: do_remail=remail_conf_EOF\n"));
	have_eof = 1;
	break;
    }
	
 free_it:

    if (sender) {
	free(sender);
	sender = NULL;
    }

    if (sender_domain) {
	free(sender_domain);
	sender_domain = NULL;
    }
	
    if(mailer)
	free_mail_send_state(&mailer);

    free_mailing_headers(&mailing_headers);

    free_mailer_info(&mailer_info);

    menu_trigger_redraw(prompt_area);

    if (have_eof) {
	if (0 == Exstat) {
	    Exstat = 
#ifdef  EX_UNAVAILABLE
		EX_UNAVAILABLE  /* Give better error status */
#else
		1
#endif
		;	    
	}
	
	DPRINT(Debug,8,(&Debug,
			"remail: Exiting with exit status %d\n",
			Exstat));
	
	leave(0,Exstat);	
    }
    
    return;
}

static void remail_sent (fd,title,rs,ret,exit_code)
     union any_fd fd; 
     char * title;
     struct run_state *rs;
     int ret; 
     int exit_code;
{
    lower_prompt(title);
    
    if (ret < 0) 
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmFailSignal,
			  "%.30s fail: Signal?"),
		  get_mailer_path(fd.mail_fd));
    else if (ret > 0) {
	if (rs->save_errno)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmFailErrno,
			      "Failed: %.30s: %.40s"),
		      get_mailer_path(fd.mail_fd),
		      strerror(rs->save_errno));
	else if (exit_code) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmMailerReturnedError,
			      "mailer returned error status %d"), 
		      exit_code);
	} else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailResent, 
			      "Mail resent."));
    } else {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmLostErrno,"%.30s lost: %.40s"),
		  get_mailer_path(fd.mail_fd),
		  strerror(rs->save_errno));
    }
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
