static char rcsid[] = "@(#)$Id: span_range.c,v 1.13 2024/10/13 10:57:10 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.13 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_pager.h"

DEBUG_VAR(Debug,__FILE__,"pager");

static enum span_result span_range_prefix P_((struct menu_context *ctx,
					      int cur_line, int *cur_col,
					      uint16 first_char,
					      struct pager_range  * pager_range,
					      int *this_flags,
					      int *mayclear,
					      int wrap_indicator));

static enum span_result span_range_prefix(ctx,cur_line,cur_col,
					  first_char, pager_range, this_flags,
					  mayclear,wrap_indicator)
     struct menu_context *ctx;
     int cur_line; 
     int *cur_col;
     uint16 first_char;
     struct pager_range  * pager_range;
     int *this_flags;
     int *mayclear;     
     int wrap_indicator;
{
    enum span_result res = span_ok;

    int CUR_col    = *cur_col;
    int THIS_flags = *this_flags;
    int MAY_clear = *mayclear;

    int START_col UNUSED_VAROK = CUR_col;

    int range_flags = get_pager_range_flags(pager_range);
    int quote_level = get_pager_range_quote_level(pager_range);  /* Original flowed quote */
    int indent      = get_pager_range_indent(pager_range);
    
    int WP =  /* pager_indicate_wrapping */ wrap_indicator;

    int LINES, COLUMNS;

    menu_get_sizes(ctx, &LINES, &COLUMNS);

    DPRINT(Debug,12,(&Debug,
		     "span_range_prefix: line = %d, col = %d, first_char=%04x, LINES=%d, COLUMNS=%d\n",
		     cur_line,CUR_col,first_char, LINES, COLUMNS));

    DPRINT(Debug,12,(&Debug,
		     "span_range_prefix: range_flags = %d, quote_level = %d, indent = %d\n",
		     range_flags,quote_level,indent));

    if (ison(range_flags,PR_REPLY_QUOTE))                     /* Should not happen */
	quote_level++;

    if (quote_level && quote_prefix) {
	int lastX = -1, i;
	uint16 repchar = 0x0;
	int POS = 0;

	int len = string_len(quote_prefix);

	DPRINT(Debug,15,(&Debug,
			 "span_range_prefix: (len=%d) prefix=%S\n",
			 len,quote_prefix));

	for (i = 0; i < len; i++) {
	    uint16 ch  = give_unicode_from_string(quote_prefix,i);

	    if (0 != unicode_ch(ch,UOP_noctrl) &&
		0 == unicode_ch(ch,UOP_space)) {
		repchar = ch;
		lastX = i;
	    }
	}
	
	if (len > 0)
	    first_char = give_unicode_from_string(quote_prefix,0);

	if (ison(range_flags,PR_QUOTE_L)) {                  /* Should not happen */
	    int need_quote = 0x005B /* [ */ == first_char;
	    
	    DPRINT(Debug,12,(&Debug,
			     "span_range_prefix: need_quote=%d first_char=%04x\n",
			     need_quote,first_char));

	    if (need_quote) {
		menu_StartXX(ctx,pg_BOLD);
		menu_Writechar(ctx,'[');
		menu_EndXX(ctx,pg_BOLD);
	    }
	}

	if (lastX >= 0) {
	    DPRINT(Debug,12,(&Debug,
			     "span_range_prefix: lastX=%d, repchar=%04x\n",
			     lastX,repchar));

	    res = span_line(ctx,cur_line,&CUR_col,quote_prefix,&POS,&MAY_clear,
			    THIS_flags,COLUMNS/3,lastX+1,WP,
			    NULL /* No collapse of space */);

	    if (res != span_ok || POS <= lastX) {
		DPRINT(Debug,12,(&Debug,
				 "span_range_prefix: failure, pos=%d, col=%d, res=%d\n",
				 POS,CUR_col,res));
		goto failure;
	    }
	} else {
	    DPRINT(Debug,12,(&Debug,
			     "span_range_prefix: No printable characters on prefix\n"));
	}

	if (quote_level > 1 && repchar) {
	    int x;

	    if (THIS_flags)
		menu_StartXX(ctx,THIS_flags);


	    for (x = 0; x < quote_level - 1 && CUR_col < COLUMNS/2; x++) {
	
		menu_WriteUnicode(ctx,repchar);
		CUR_col++;       
	    }
	
	    if (THIS_flags)
		menu_EndXX(ctx,THIS_flags);

	}


	if (POS < len) {
	    res = span_line(ctx,cur_line,&CUR_col,quote_prefix,&POS,&MAY_clear,
			    THIS_flags,COLUMNS/3,len-lastX,WP,
			    NULL /* No collapse of space */);
	    
	    if (res != span_ok || POS < len) {
		DPRINT(Debug,12,(&Debug,
				 "span_range_prefix: failure, pos=%d, col=%d, res=%d\n",
				 POS,CUR_col,res));
		goto failure;
	    }	    
	}

    failure:
	;

    } else if (ison(range_flags,PR_QUOTE_L)) {            /* Should not happen */
	int need_quote = 0x005B /* [ */ == first_char;
	
	DPRINT(Debug,12,(&Debug,
			 "span_range_prefix: need_quote=%d first_char=%04x\n",
			 need_quote,first_char));
	
	menu_StartXX(ctx,pg_BOLD);
	if (need_quote)
	    menu_Writechar(ctx,'[');
	menu_EndXX(ctx,pg_BOLD);
	
    }

    while (CUR_col < COLUMNS - COLUMNS/3 && 
	   CUR_col < indent) {	
	menu_WriteUnicode(ctx,0x0020 /* SPACE */ );
	CUR_col++;       
    }

    *cur_col    = CUR_col;
    *this_flags = THIS_flags;
    *mayclear   = MAY_clear;

    DPRINT(Debug,12,(&Debug,
		     "span_range_prefix=%d",res));
    switch (res) {
    case span_failure:  DPRINT(Debug,12,(&Debug," span_failure")); break;
    case span_ok:       DPRINT(Debug,12,(&Debug," span_ok"));      break;
    case span_FF_seen:  DPRINT(Debug,12,(&Debug," span_FF_seen")); break;
    case span_EOS:      DPRINT(Debug,12,(&Debug," span_EOS"));     break;
    case span_NL_seen:  DPRINT(Debug,12,(&Debug," span_NL_seen")); break;
    case span_wrapping:        DPRINT(Debug,12,(&Debug," span_wrapping")); break;
    case span_wordwrapped:     DPRINT(Debug,12,(&Debug," span_wordwrapped")); break;
    case span_clear_alt_data:  DPRINT(Debug,12,(&Debug," span_clear_alt_data")); break;
    case span_space_collapsed: DPRINT(Debug,12,(&Debug," span_space_collapsed")); break;
    }    
    DPRINT(Debug,12,(&Debug," (col = %d -> %d, line = %d)\n",
		     START_col, CUR_col, cur_line));

    return res;
}

static enum span_result span_range_middle P_((struct menu_context *ctx,
					      int cur_line, int *cur_col,
					      uint16 first_char,
					      struct pager_range  * pager_range,
					      int *this_flags,
					      int *mayclear));

static enum span_result span_range_middle(ctx,cur_line,cur_col,
				   first_char, pager_range, this_flags,
				   mayclear)
     struct menu_context *ctx;
     int cur_line; 
     int *cur_col;
     uint16 first_char UNUSED_VAROK;
     struct pager_range  * pager_range;
     int *this_flags;
     int *mayclear;     
{
    enum span_result res = span_ok;

    int CUR_col    = *cur_col;
    int THIS_flags = *this_flags;
    int MAY_clear = *mayclear;

    int START_col UNUSED_VAROK = CUR_col;

    int range_flags UNUSED_VAROK = get_pager_range_flags(pager_range);
    int quote_level UNUSED_VAROK =
	get_pager_range_quote_level(pager_range);  /* Original flowed quote */
    int indent      = get_pager_range_indent(pager_range);

    int LINES, COLUMNS;

    menu_get_sizes(ctx, &LINES, &COLUMNS);

    DPRINT(Debug,12,(&Debug,
		     "span_range_middle: line = %d, col = %d, first_char=%04x, LINES=%d, COLUMNS=%d\n",
		     cur_line,CUR_col,first_char, LINES, COLUMNS));

    DPRINT(Debug,12,(&Debug,
		     "span_range_middle: range_flags = %d, quote_level = %d, indent = %d\n",
		     range_flags,quote_level,indent));

    while (CUR_col < COLUMNS - COLUMNS/3 && 
	   CUR_col < indent) {	
	menu_WriteUnicode(ctx,0x0020 /* SPACE */ );
	CUR_col++;       
    }

    *cur_col    = CUR_col;
    *this_flags = THIS_flags;
    *mayclear   = MAY_clear;

    DPRINT(Debug,12,(&Debug,
		     "span_range_middle=%d",res));
    switch (res) {
    case span_failure:  DPRINT(Debug,12,(&Debug," span_failure")); break;
    case span_ok:       DPRINT(Debug,12,(&Debug," span_ok"));      break;
    case span_FF_seen:  DPRINT(Debug,12,(&Debug," span_FF_seen")); break;
    case span_EOS:      DPRINT(Debug,12,(&Debug," span_EOS"));     break;
    case span_NL_seen:  DPRINT(Debug,12,(&Debug," span_NL_seen")); break;
    case span_wrapping: DPRINT(Debug,12,(&Debug," span_wrapping")); break;
    case span_wordwrapped:     DPRINT(Debug,12,(&Debug," span_wordwrapped"));     break;
    case span_clear_alt_data:  DPRINT(Debug,12,(&Debug," span_clear_alt_data"));  break;
    case span_space_collapsed: DPRINT(Debug,12,(&Debug," span_space_collapsed")); break;
    }    
    DPRINT(Debug,12,(&Debug," (col = %d -> %d, line = %d)\n",
		     START_col, CUR_col, cur_line));

    return res;
}

const struct span_helper_data NULL_span_helper_data = {
    SPAN_HELPER_DATA_magic,

    0,0,0,NULL,0,0,0,0,0,0,0,0,0,0,0,0
};

static enum spanh_prefix_res {
    spanh_exit = 0,
    spanh_main,
} span_helper_prefix P_(( struct menu_context * ctx,
			  struct pager_range      * pager_range,
			  struct pg_wrap_info     * info,
			  struct span_helper_data * data,
			  uint16                    ch,
			  enum span_result        * span_line_result_p
			  ));

enum spanh_prefix_res span_helper_prefix(ctx,pager_range,info,data,ch,
					 span_line_result_p)
     struct menu_context * ctx;
     struct pager_range      * pager_range;
     struct pg_wrap_info     * info;
     struct span_helper_data * data;
     uint16                    ch;
     enum span_result        * span_line_result_p;
{
    int L_mayclear      = 0;
    int WP;

    enum spanh_prefix_res res = spanh_exit;

    if (SPAN_HELPER_DATA_magic != data->magic)
	panic("PAGER PANIC",__FILE__,__LINE__,
	      "span_helper_prefix",
	      "Bad magic number",0);

    L_mayclear          = data->mayclear;
    WP  =  /* pager_indicate_wrapping */ data->wrap_indicator;
    
    if (pager_range) {
	data->range_flags = get_pager_range_flags(pager_range);
	
	DPRINT(Debug,9,(&Debug, 
			"span_helper_prefix: pager range: quote_level=%d range_flags=%d\n",
			get_pager_range_quote_level(pager_range),
			data->range_flags));
    }

    
    
    if (data->start_of_line) {
	DPRINT(Debug,10,(&Debug, 
			 "span_helper_prefix: virtual_line=%d, start of line\n",
			 data->virtual_line));
	
	record_pg_wrap_info(info,&(data->virtual_line),data->idx,
			    data->buffer_x,data->lineext);
    }

    
    if (data->start_of_line && pager_range) {
	
	/* 1) Writes quote prefix */
	
	* span_line_result_p =
	    span_range_prefix(ctx,
			      data->cur_line,&(data->cur_col),
			      ch,pager_range,
			      &(data->this_flags),
			      &L_mayclear,
			      WP
			      );
	
	DPRINT(Debug,9,(&Debug, 
			"span_helper_prefix: span_range_prefix result = %d, ",
			* span_line_result_p));
	
	data->start_of_line = 0;
	data->p_start_COL   = data->cur_col;
	
	if (data->start_of_range) {
	    DPRINT(Debug,9,(&Debug,"start of range (range changed) ignored, "));
	    data->start_of_range = 0;
	}

	res = spanh_exit;
	
    } else if (data->start_of_range && pager_range) {

	/* 2) Writes indentation */
	
	* span_line_result_p =
	    span_range_middle(ctx,
			      data->cur_line,&(data->cur_col),
			      ch,pager_range,
			      &(data->this_flags),
			      &L_mayclear);
	
	DPRINT(Debug,9,(&Debug, 
			"span_helper_prefix: span_range_middle result = %d, ",
			* span_line_result_p));

	data->start_of_range = 0;

	res = spanh_exit;
	
    } else {
	
	res = spanh_main;
	
	DPRINT(Debug,9,(&Debug, 
			"span_helper_prefix: "));
    }

    data->mayclear      = L_mayclear ? 1 : 0;

    DPRINT(Debug,9,(&Debug, "return %d",res));
    switch (res) {
    case spanh_exit: DPRINT(Debug,9,(&Debug, " spanh_exit")); break;
    case spanh_main: DPRINT(Debug,9,(&Debug, " spanh_main")); break;
    }
    DPRINT(Debug,9,(&Debug, "\n"));

    return res;
}

enum span_result span_alt_data_helper (ctx,pager_range,info,
				       alt_buffer,
				       res_buffer,
				       res_pg_flags,
				       data)
     struct menu_context * ctx;
     struct pager_range  * pager_range;
     struct pg_wrap_info * info;
     struct string_alt   * alt_buffer;
     struct string      ** res_buffer;
     int                 * res_pg_flags;
     struct span_helper_data * data;
{
    enum span_result ret = span_clear_alt_data;
    uint16 ch = 0;   /* First character may be needed */

    enum string_alt_type alt_data_type = string_alt_none;
    const union string_alt_value alt_data_value =
	get_string_alt_value(alt_buffer,&alt_data_type);

    int LINES, COLUMNS;
    

    int character_ok = 0; 
    
    	
    if (SPAN_HELPER_DATA_magic != data->magic)
	panic("PAGER PANIC",__FILE__,__LINE__,
	      "span_alt_data_helper",
	      "Bad magic number",0);

    menu_get_sizes(ctx, &LINES, &COLUMNS);
    
    switch (alt_data_type) {
    case string_alt_none:
	
	DPRINT(Debug,10,(&Debug, "span_alt_data_helper: empty alt data\n"));
       	
	goto out;
	    
    case string_alt_text:
	    
	if (*res_buffer)
	    free_string(res_buffer);
	*res_buffer = dup_string(alt_data_value.text);
	data->buffer_x = 0;
	
	DPRINT(Debug,10,(&Debug,
			 "span_alt_data_helper: Set string\n"));
	DEBUG_PRINT_STRING(Debug,12,
			   "span_alt_data_helper: buffer = ",
			   "                    : buffer > ",
			   *res_buffer);

	goto out;
	    
    case string_alt_entity: {
	int                    ref_key_pg_flags = 0;
	const struct string  * ref_key =
	    out_entity_reference_key(alt_data_value.entity,
				     & ref_key_pg_flags);
	const struct string  * text_value = 
	    out_entity_text_value(alt_data_value.entity);
	
	struct string  * res1 = NULL;

	DPRINT(Debug,11,(&Debug,
			 "span_alt_data_helper: entity %S\n",
			 ref_key));
	
	if (text_value && (res1 = curses_available_string(text_value))) {
	    if (*res_buffer)
		free_string(res_buffer);
	    *res_buffer = res1;
	    res1 = NULL;
	    data->buffer_x = 0;
	    
	    DPRINT(Debug,10,(&Debug,
			     "span_alt_data_helper: Set string\n"));

	    DEBUG_PRINT_STRING(Debug,12,
			       "span_alt_data_helper: buffer = ",
			       "                    : buffer > ",
			       *res_buffer);
	    
	    goto out;
	    
	}
	
	ch = out_entity_unicode_value(alt_data_value.entity);

	switch (ch) {
	case UNICODE_BAD_CHAR:
	    character_ok = 0;
	    break;
	case UNICODE_SOFT_HYPHEN:   /* What to do ? */
	    character_ok = 1;
	    break;
	    
	default: {
	    charset_t utf8 = MIME_name_to_charset("UTF-8",0);

	    enum charset_unicode_stat x;

	    DPRINT(Debug,11,(&Debug, "span_alt_data_helper: entity unicode value x%04x\n",
			     ch));
	    
	    if (utf8 &&
		(0 != (charset_properties(utf8) & CS_printable)) &
		curses_available_charset(utf8)) {
		x = string_type_have_unicode(utf8,ch);
	    } else {
		x = string_type_have_unicode(display_charset,ch);
	    }

	    switch (x) {
	    case charset_unicode_bad:
	    case charset_unicode_unknown:
	    case charset_missing_unicode:
		character_ok = 0;
		break;
	    case charset_have_unicode:
		character_ok = 1;
	    }
	}
	    break;
	}

	if (! character_ok) {
	    int len         = string_len(ref_key);
	    
	    /* This should be  0x0026 '&' */
	    if (len > 0)
		ch = give_unicode_from_string(ref_key,0);

	}
		    
    }
	    break;
    }

    switch(span_helper_prefix(ctx,pager_range,info,data,ch,&ret)) {
    case spanh_exit:
	
	break;

    case spanh_main: {
		
	if (data->start_of_line) {
	    DPRINT(Debug,10,(&Debug, 
			     "span_alt_data_helper: No pager range on beginning of line\n"));
	    
	    data->start_of_line = 0;
	    data->p_start_COL   = 0;
	}

	if (data->start_of_range) {
	    DPRINT(Debug,10,(&Debug, 
			     "span_alt_data_helper: start of range (range changed) without range\n"));
	    
	    data->start_of_range = 0;
	}


	if (data->joined_line) {
	    DPRINT(Debug,10,(&Debug, 
			     "span_alt_data_helper: this is joined line\n"));
	}

	
	 switch (alt_data_type) {
	 case string_alt_none:
	 case string_alt_text:
	     goto out;

	 case string_alt_entity: {

	     /* Not very good ---- */
	     
	     if ( character_ok) {
		 charset_t utf8 = MIME_name_to_charset("UTF-8",0);
		 		 
		 if (*res_buffer)
		     free_string(res_buffer);
		 *res_buffer = new_string(utf8 ? utf8 : display_charset);

		 /* This can fail silently */
		 add_unicode_to_string(*res_buffer,1,&ch);

		 DPRINT(Debug,10,(&Debug,
				  "span_alt_data_helper: Set string\n"));

		 DEBUG_PRINT_STRING(Debug,12,
				    "span_alt_data_helper: buffer = ",
				    "                    : buffer > ",
				    *res_buffer);

		 
		 
	     } else {

		 int                    ref_key_pg_flags = 0;
		 const struct string  * ref_key =
		     out_entity_reference_key(alt_data_value.entity,
					      & ref_key_pg_flags);

		 if (*res_buffer)
		     free_string(res_buffer);
		 *res_buffer    = dup_string(ref_key);
		 *res_pg_flags |= ref_key_pg_flags;

		 data->buffer_x = 0;
		 
		 DPRINT(Debug,10,(&Debug,
				  "span_alt_data_helper: Set string\n"));
		 
		 DEBUG_PRINT_STRING(Debug,12,
				    "span_alt_data_helper: buffer = ",
				    "                    : buffer > ",
				    *res_buffer);		 
	     }

	 }
	     break;
	 }
    }
	break;
    }
	
out:

    DPRINT(Debug,9,(&Debug,"span_alt_data_helper = %d",
		    ret));
    switch (ret) {
    case span_failure:  DPRINT(Debug,9,(&Debug," span_failure")); break;
    case span_ok:       DPRINT(Debug,9,(&Debug," span_ok"));      break;
    case span_FF_seen:  DPRINT(Debug,9,(&Debug," span_FF_seen")); break;
    case span_EOS:      DPRINT(Debug,9,(&Debug," span_EOS"));     break;
    case span_NL_seen:  DPRINT(Debug,9,(&Debug," span_NL_seen")); break;
    case span_wrapping: DPRINT(Debug,9,(&Debug," span_wrapping")); break;
    case span_wordwrapped:     DPRINT(Debug,9,(&Debug," span_wordwrapped"));      break;
    case span_clear_alt_data:  DPRINT(Debug,9,(&Debug," span_clear_alt_data"));   break;
    case span_space_collapsed: DPRINT(Debug,12,(&Debug," span_space_collapsed")); break;
    }    
    DPRINT(Debug,9,(&Debug,"\n"));
    
    return ret;
}

enum span_result span_helper (ctx,pager_range,info,buffer,data)
     struct menu_context * ctx;
     struct pager_range  * pager_range;
     struct pg_wrap_info * info;
     struct string       * buffer;
     struct span_helper_data * data;
{
    
    enum span_result span_line_result = span_failure;

    int buffer_len      = string_len(buffer);
    int LINES, COLUMNS;
    uint16 ch = 0;   /* First character may be needed */
    
    int L_mayclear;
    int L_on_space;
    
    if (SPAN_HELPER_DATA_magic != data->magic)
	panic("PAGER PANIC",__FILE__,__LINE__,
	      "span_helper",
	      "Bad magic number",0);
    
    L_mayclear          = data->mayclear;
    L_on_space          = data->on_space && ison(data->range_flags,
						 PR_COLLAPSE_NOINHERIT);
    
    menu_get_sizes(ctx, &LINES, &COLUMNS);

    if (buffer_len > 0)
	ch = give_unicode_from_string(buffer,0);


    switch(span_helper_prefix(ctx,pager_range,info,data,ch,&span_line_result)) {
    case spanh_exit:

	DPRINT(Debug,9,(&Debug,"span_helper: "));

	if (data->buffer_x >= buffer_len) {
	    DPRINT(Debug,9,(&Debug, 
			    "no buffer left (buffer_x=%d buffer_len=%d), ",
			    data->buffer_x,buffer_len));
	    goto buffer_handled;
	}
	
	break;

    case spanh_main: {
	int center_cur_col  = data->cur_col;
	int center_buffer_x = data->buffer_x;
	
	int WP  =  /* pager_indicate_wrapping */ data->wrap_indicator;
	
	int  max_width = COLUMNS;

	/* 3) Writes actual text */
	
	if (data->start_of_line) {
	    DPRINT(Debug,10,(&Debug, 
			     "span_helper: No pager range on beginning of line\n"));
	    
	    data->start_of_line = 0;
	    data->p_start_COL   = 0;
	}
	
	if (data->start_of_range) {
	    DPRINT(Debug,10,(&Debug, 
			     "span_helper: start of range (range changed) without range\n"));
	    
	    data->start_of_range = 0;
	}


	if (data->joined_line) {
	    DPRINT(Debug,10,(&Debug, 
			     "span_helper: this is joined line\n"));
	}
	
	if (ison(data->range_flags,PR_WORD_WRAP) 
	    || 
	    (data->force_word_wrap && isoff(data->range_flags,PR_PREFORMAT))
	    ) {
	    
	    span_line_result = span_words(ctx,
					  data->cur_line,&(data->cur_col),
					  buffer,&(data->buffer_x),
					  &L_mayclear,
					  data->this_flags,
					  max_width  /* max_width*/,
					  buffer_len /* max_len */,
					  data->joined_line,
					  data->p_start_COL,
					  isoff(data->range_flags, 
						PR_MAX_WIDTH) ? 
					  data->p_width : 0,
					  WP,
					  ison(data->range_flags,
					       PR_COLLAPSE_NOINHERIT) ?
					  &L_on_space : NULL);
	    
	    DPRINT(Debug,9,(&Debug, 
			    "span_helper: span_words result = %d, ",
			    span_line_result));
	    
	    
	} else {
	    
	    /* This is the part of the code that actually displays on 
	     * the screen 
	     */
	    
	    span_line_result = span_line(ctx,
					 data->cur_line,&(data->cur_col),
					 buffer,&(data->buffer_x),
					 &L_mayclear,
					 data->this_flags,
					 max_width  /* max_width*/,
					 buffer_len /* max_len */,
					 WP,
					 ison(data->range_flags,
					      PR_COLLAPSE_NOINHERIT) ?
					 &L_on_space : NULL);
	    DPRINT(Debug,9,(&Debug, 
			    "span_helper: span_line result = %d, ",
			    span_line_result));
	    
	}

	/* Redraw on centered */
	
	if (! data->joined_line &&
	    ! data->lineext &&
	    ison(data->range_flags,PR_CENTER_THIS) &&
	    span_line_result >= span_ok &&
	    data->buffer_x > center_buffer_x &&
	    data->cur_col  > center_cur_col &&
	    center_cur_col < max_width) {

	    
	    int end_col = data->cur_col;

	    int used_len = end_col - center_cur_col;
	    int avail_space = max_width - center_cur_col;

	    if (avail_space > used_len) {

		int new_start_col = data->p_start_COL;
		
		int new_start_pos =
		    center_cur_col +
		    (avail_space - used_len) /2;

		if (isoff(data->range_flags, PR_MAX_WIDTH) &&
		    data->p_width > 0 && data->p_width < max_width) {

		    /* This does not depend printed line length 
		       -- also move reference length for wrapping

		     */
		    
		    new_start_col
			+= ( max_width - data->p_width) / 2;

		    if (new_start_col > new_start_pos)
			new_start_col = new_start_pos;			
		}
		
		if (new_start_pos > center_cur_col) {
		    DPRINT(Debug,10,(&Debug, 
				     "span_helper: centering this line (redraw), start col = %d => %d -- p_start_COL = %d => %d\n",
				     center_cur_col,new_start_pos,
				     data->p_start_COL,new_start_col
				     ));
		    
		    /* Need reddraw line */
		    
		    menu_MoveCursor(ctx,data->cur_line,center_cur_col);
		    menu_CleartoEOLN(ctx);
		    
		    data->cur_col  = new_start_pos;
		    data->buffer_x = center_buffer_x;
		    
		    menu_MoveCursor(ctx,data->cur_line,data->cur_col);

		    /* Because this is centered, assume that space not need to be printed */
		    L_on_space          = 0;
		    
		    if (ison(data->range_flags,PR_WORD_WRAP) 
			|| 
			(data->force_word_wrap && isoff(data->range_flags,PR_PREFORMAT))
			) {
			
			span_line_result = span_words(ctx,
						      data->cur_line,&(data->cur_col),
						      buffer,&(data->buffer_x),
						      &L_mayclear,
						      data->this_flags,
						      max_width  /* max_width*/,
						      buffer_len /* max_len */,
						      data->joined_line,
						      new_start_col /* p_start_COL */,
						      isoff(data->range_flags, 
							    PR_MAX_WIDTH) ? 
						      data->p_width : 0,
						      WP,
						      ison(data->range_flags,
							   PR_COLLAPSE_NOINHERIT) ?
						      &L_on_space : NULL);
			
			DPRINT(Debug,9,(&Debug, 
					"span_helper: span_words redraw result = %d, ",
					span_line_result));
			
		    } else {
			
			/* This is the part of the code that actually displays on 
			 * the screen 
			 */
			
			span_line_result = span_line(ctx,
						     data->cur_line,&(data->cur_col),
						     buffer,&(data->buffer_x),
						     &L_mayclear,
						     data->this_flags,
						     max_width  /* max_width*/,
						     buffer_len /* max_len */,
						     WP,
						     ison(data->range_flags,
							  PR_COLLAPSE_NOINHERIT) ?
						     &L_on_space : NULL
						     );
			DPRINT(Debug,9,(&Debug, 
					"span_helper: span_line redaw result = %d, ",
					span_line_result));
	    
		    }		    		    		    
		}
	    }	    	   
	}
       	
	buffer_handled:
	if (data->lineext) {
	    DPRINT(Debug,9,(&Debug," line extension (joining), "));
	    
	    data->joined_line = 1;
	} else if (isoff(data->range_flags,PR_JOIN_LINES)) {
	    DPRINT(Debug,9,(&Debug," line wrapping, "));
	    
	    data->start_of_line = 1;  /* Need new line */
	    L_on_space          = 0;
	    
	} else if (ison(data->range_flags,PR_ADD_SPACE) ||
		   L_on_space) {

	    if (data->p_start_COL >= data->cur_col) {
		DPRINT(Debug,9,(&Debug," joining lines (no space on start), "));
	    } else if (COLUMNS > data->cur_col+1) {
		menu_MoveCursor(ctx,data->cur_line,data->cur_col);
		menu_CleartoEOLN(ctx);
		menu_Writechar(ctx,' ');
		data->cur_col++;
		
		DPRINT(Debug,9,(&Debug," joining lines with space, "));

	    } else {
		DPRINT(Debug,9,(&Debug," joining lines (no space), "));
	    }
	    data->joined_line = 1;
	    L_on_space        = 0;

	} else {
	    DPRINT(Debug,9,(&Debug," joining lines, "));
	    data->joined_line = 1;
	}

	data->mayclear      = L_mayclear ? 1 : 0;
	data->on_space      = L_on_space ? 1 : 0;

	if (data->mayclear) {
	    DPRINT(Debug,9,(&Debug," may clear, "));
	}
	if (data->on_space) {
	    DPRINT(Debug,9,(&Debug," on space, "));
	}
	
    }
	break;
    }
	   
    DPRINT(Debug,9,(&Debug," span_helper=%d",span_line_result));
    switch (span_line_result) {
    case span_failure:  DPRINT(Debug,9,(&Debug," span_failure")); break;
    case span_ok:       DPRINT(Debug,9,(&Debug," span_ok"));      break;
    case span_FF_seen:  DPRINT(Debug,9,(&Debug," span_FF_seen")); break;
    case span_EOS:      DPRINT(Debug,9,(&Debug," span_EOS"));     break;
    case span_NL_seen:  DPRINT(Debug,9,(&Debug," span_NL_seen")); break;
    case span_wrapping: DPRINT(Debug,9,(&Debug," span_wrapping")); break;
    case span_wordwrapped:     DPRINT(Debug,9,(&Debug, " span_wordwrapped")); break;
    case span_clear_alt_data:  DPRINT(Debug,9,(&Debug, " span_clear_alt_data")); break;
    case span_space_collapsed: DPRINT(Debug,12,(&Debug," span_space_collapsed")); break;
    }    
    DPRINT(Debug,9,(&Debug,"\n"));
	
    return span_line_result;
}

void clear_span_helper_data(data)
     struct span_helper_data *data;
{
    if (SPAN_HELPER_DATA_magic != data->magic)
	panic("PAGER PANIC",__FILE__,__LINE__,
	      "clear_span_helper_data",
	      "Bad magic number",0);

    if (data->lineext)
	free_pager_lineext(&(data->lineext));
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
