static char rcsid[] = "@(#)$Id: span_line.c,v 1.20 2024/10/13 10:57:10 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.20 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *
 *  Partially based on src/builtin++.c, which is initially 
 *     written by: Michael Elkins <elkins@aero.org>, 1995
 *
 *****************************************************************************/

#include "def_pager.h"

DEBUG_VAR(Debug,__FILE__,"pager");

/* Return 0 if wrapped */
static int span_tab P_((struct menu_context *ctx,int *cur_col, int *pos,
			int this_flags, int max_width, int START_pos, int START_col,
			int cur_line));
static int span_tab(ctx,cur_col,pos,this_flags,max_width, START_pos,START_col,cur_line)
     struct menu_context *ctx;
     int *cur_col; 
     int *pos;
     int this_flags; 
     int max_width; 
     int START_pos; 
     int START_col;
     int cur_line;
{
    int r = 0;
    int CUR_col               = *cur_col;
    int  POS                  = *pos;

    int new_val = ((CUR_col / get_tabspacing() ) +1) * get_tabspacing(); 

    int LINES, COLUMNS;
    
    menu_get_sizes(ctx, &LINES, &COLUMNS);

    if (new_val >= COLUMNS -1) {
	
	if (COLUMNS - START_col <= max_width) {

	    DPRINT(Debug,12,(&Debug,
			     "span_tab: POS = %d, -- end of line = %d -> %d (consuming TAB)\n",
			     POS,CUR_col,new_val)); 
	    
	    if (this_flags)
		menu_StartXX(ctx,this_flags);
	    
	    menu_Writechar(ctx,'\t');
	    
	    if (this_flags)
		menu_EndXX(ctx,this_flags);
	    
	    CUR_col = COLUMNS -1;
	    menu_MoveCursor(ctx,cur_line,CUR_col);
	    
	    POS++;
	    r = 0;
	    goto no_space;
	} else
	    goto too_width;
    }
 

    /* XXX   -- is this correct? */
    if (new_val - START_col >= max_width) {
    too_width:
	DPRINT(Debug,12,(&Debug,
			 "span_tab: POS = %d   -- new line len = %d, max_width = %d reached\n",
			 POS,new_val, max_width)); 
			
	r = 0;
	goto no_space;
    }

    if (this_flags)
	menu_StartXX(ctx,this_flags);
    
    menu_Writechar(ctx,'\t');
    CUR_col = new_val;

    if (this_flags)
	menu_EndXX(ctx,this_flags);
    
    r = 1;

 no_space:
    *cur_col = CUR_col;
    *pos     = POS;

    DPRINT(Debug,12,(&Debug,"span_tab=%d\n",r));

    return r;
}

static int span_control P_((struct menu_context *ctx, int cur_line,
			    int *cur_col, uint16 ch,
			    int *pos,int *mayclear,int this_flags,
			    int max_width, enum span_result *ret,
			    int START_pos, int START_col, 
			    int wrap_indicator,
			    int word_wrap,
			    int gchar_flags,
			    unsigned char bytech,
			    int * on_space_p));
static int span_control(ctx,cur_line,cur_col,ch,pos,mayclear,this_flags,
			max_width,ret,START_pos,START_col,wrap_indicator,
			word_wrap,gchar_flags,bytech,on_space_p)
     struct menu_context *ctx; 
     int cur_line;
     int *cur_col;
     uint16 ch;
     int *pos;
     int *mayclear;
     int this_flags;
     int max_width; 
     enum span_result *ret;
     int START_pos;
     int START_col;
     int wrap_indicator;
     int word_wrap;   
     int gchar_flags;
     unsigned char bytech;
     int * on_space_p;
{
    int ok = 1;

    int CUR_col               = *cur_col;
    int  POS                  = *pos;
    int MAY_clear             = *mayclear;
    enum span_result RET_code = *ret;
    int LINES, COLUMNS;

    int temp_line, temp_col;

    menu_get_sizes(ctx, &LINES, &COLUMNS);

    /* If no space for next character */
    if (CUR_col >= COLUMNS -wrap_indicator) {
	DPRINT(Debug,12,(&Debug,
			 "span_control: POS = %d, ch=%04x -- end of line = %d (wrap indicator %d)\n",
			 POS,ch, CUR_col,wrap_indicator)); 
	
	
	goto last_char;
    }

    if (CUR_col - START_col >= max_width) {
	DPRINT(Debug,12,(&Debug,"span_control: POS = %d,ch=%04x  -- line len = %d, max_width = %d reached\n",
			 POS,ch,CUR_col, max_width)); 
	
    last_char:
	if (0 != (gchar_flags & GCHAR_unicode)) {

	    switch(ch) {
	    case 0x000C:   /* FF */
	    case 0x000B:   /* VT */
		RET_code = span_FF_seen;
		DPRINT(Debug,12,(&Debug,
				 "span_control: Consuming last character ch=%04x\n",
				 ch));
		POS++;
		break;
		
	    case 0x000A:   /* LF */
		RET_code = span_NL_seen;
		DPRINT(Debug,12,(&Debug,
				 "span_control: Consuming last character ch=%04x\n",
				 ch));
		POS++;
		break;
		
	    default:
		if (word_wrap && unicode_ch(ch,UOP_space)) {
		    DPRINT(Debug,12,(&Debug,
				     "span_control: Consuming last character ch=%04x (space)\n",
				     ch));
		    
		    POS++;
		    RET_code = span_wordwrapped;
		    
		} else if (START_pos < POS)
		    RET_code = span_wrapping;

		break;
	    }
	} else if (START_pos < POS)
	    RET_code = span_wrapping;
	
	ok = 0;
	goto no_space;
    }

    menu_GetXYLocation (ctx,&temp_line, &temp_col);
    
    if (temp_line != cur_line || temp_col != CUR_col) {
	DPRINT(Debug,5,(&Debug,
			"span_control: location does not match (line,col) = (%d,%d) != (%d,%d) expected\n",
			temp_line,temp_col,cur_line,CUR_col));
		   
			    
	RET_code = span_failure;
	ok = 0;
	goto no_space;
    }
 
    if (ison(gchar_flags,GCHAR_unicode) &&
	(ch <= 31 || ch == 0x007F /* ^? */ )) {
	
	/* Make sure that carbage from previous
	   output is not left */
	if (MAY_clear) {
	    menu_CleartoEOLN(ctx);
	    MAY_clear = 0;
	}

	switch(ch) {
	    int add_space;
	    
	case 0x0009:  /* HT */

	    if (on_space_p) {
		DPRINT(Debug,12,(&Debug,
				 "span_control: POS = %d, ch=%04x eating space (TAB) \n",
				 POS,ch));
		* on_space_p = 1;
		POS++;
	    } else if (! span_tab(ctx,&CUR_col,&POS,this_flags,max_width,
			   START_pos,START_col,cur_line)) {

		DPRINT(Debug,12,(&Debug,
				 "span_control: ch=%04x  wrapped\n",
				 ch));

		if (word_wrap)
		    RET_code = span_wordwrapped;
		else
		    RET_code = span_wrapping;
		ok = 0;
		goto no_space;

	    }

	    break;
	    
	case 0x000C:   /* FF */
	case 0x000B:   /* VT */
	case 0x000A:   /* LF */
	    goto last_char;

	case 0x007F:
	default:

	    add_space = on_space_p && *on_space_p;
	    
	    if (CUR_col >= COLUMNS -1 -wrap_indicator ||
		CUR_col - START_col >= max_width -1 - add_space) {
		
		DPRINT(Debug,12,(&Debug,
				 "span_control: POS = %d, ch=%04x -- no space for ctrl, col = %d, max_width = %d (wrap indicator %d)\n",
				 POS,ch, CUR_col,max_width,wrap_indicator));

		
		if (START_pos < POS)
		    RET_code = span_wrapping;
		
		ok = 0;
		goto no_space;
	    }

	    if (on_space_p) {
		menu_Writechar(ctx,' ');
		CUR_col ++;
		*on_space_p = 0;
	    }
	    
	    menu_StartXX(ctx,pg_REVERSE);
	    
	    if (0x007F == ch) {
		menu_Writechar(ctx,'^');			
		menu_Writechar(ctx,'?');
	    } else {
		/* Output character set is assumed to be ASCII
		   compatible on here: */
		menu_Writechar(ctx,'^');			
		menu_Writechar(ctx,ch + 64);
		
	    }			
	    menu_EndXX(ctx,pg_REVERSE);
	    
	    CUR_col += 2;
	    break;
	    
	}

	POS++;
	goto done;
    }

    if (0 != (gchar_flags & GCHAR_unicode)) {
	switch(ch) {
	    
	case UNICODE_BAD_CHAR:
	    
	    menu_StartXX(ctx,pg_REVERSE);
	    menu_Writechar(ctx,'?');
	    menu_EndXX(ctx,pg_REVERSE);
	    CUR_col ++;
	    POS++;
	    break;
	    
	case UNICODE_NO_BREAK_SPACE:  /* Not really work */
	    
	    if (this_flags)
		menu_StartXX(ctx,this_flags);
	    menu_Writechar(ctx,' ');
	    if (this_flags)
		menu_EndXX(ctx,this_flags);
	    
	    CUR_col++;
	    POS++;
	    break;
	    
	case UNICODE_SOFT_HYPHEN:
	    
	    /* If we are on last position then print it and wrapping 
	       otherwise ignore this character
	    */
	    if (CUR_col >= COLUMNS -1 -wrap_indicator ||
		CUR_col - START_col >= max_width -1) {
		
		DPRINT(Debug,12,(&Debug,
				 "span_control: POS = %d, ch=%04x --  wrap on soft hyphen, col = %d, max_width = %d (wrap indicator %d)\n",
				 POS,ch, CUR_col,max_width,wrap_indicator));
		
		if (this_flags)
		    menu_StartXX(ctx,this_flags);
		menu_Writechar(ctx,'-');
		if (this_flags)
		    menu_EndXX(ctx,this_flags);
		
		CUR_col++;
		POS++;
		
		if (word_wrap)
		    RET_code = span_wordwrapped;
		else
		    RET_code = span_wrapping;
		ok = 0;
		goto no_space;
	    }
	    
	    POS++;  /* Skip character */
	    break;

	default:
	    if (0 == unicode_ch(ch,UOP_noctrl)) {
		
		if (0 != (gchar_flags & GCHAR_bytecode))
		    goto show_bytecode;

		if (CUR_col >= COLUMNS -6 -wrap_indicator ||
		    CUR_col - START_col >= max_width -6) {
		    
		    DPRINT(Debug,12,(&Debug,
				     "span_control: POS = %d, ch=%04x -- no space for ctrl, col = %d, max_width = %d (wrap indicator %d)\n",
				     POS,ch, CUR_col,max_width,
				     wrap_indicator));
		    
		    if (START_pos < POS)
			RET_code = span_wrapping;
		    
		    ok = 0;
		    goto no_space;
		}
		
		menu_StartXX(ctx,pg_REVERSE);
		menu_Write_to_screen(ctx,
				     FRM("^u%04X"),ch);			
		menu_EndXX(ctx,pg_REVERSE);
		CUR_col += 6;
		POS++;
		
	    } else if (0 != unicode_ch(ch,UOP_space)) {

		if (on_space_p) {
		    DPRINT(Debug,12,(&Debug,
				     "span_control: POS = %d, ch=%04x eating space\n",
				     POS,ch));

		    * on_space_p = 1;
		    POS++;
		} else {		    
		    DPRINT(Debug,12,(&Debug,
				     "span_control: POS = %d, ch=%04x, advance space\n",
				     POS,ch));
		    if (this_flags)
			menu_StartXX(ctx,this_flags);
		    
		    menu_WriteUnicode(ctx,ch);
		    
		    if (this_flags)
			menu_EndXX(ctx,this_flags);
		    
		    CUR_col++;
		    POS++;
		}
	    }
	    break;
	}

    } else {
	if (0 != (gchar_flags & GCHAR_bytecode)) {
	show_bytecode:

	    if (CUR_col >= COLUMNS -4 -wrap_indicator ||
		CUR_col - START_col >= max_width -4) {
		
		DPRINT(Debug,12,(&Debug,
				 "span_control: POS = %d, bytech=%02x -- no space for ctrl, col = %d, max_width = %d (wrap indicator %d)\n",
				 POS,bytech, CUR_col,max_width,
				 wrap_indicator));
		
		if (START_pos < POS)
		    RET_code = span_wrapping;
		
		ok = 0;
		goto no_space;
	    }

	    menu_StartXX(ctx,pg_REVERSE);
	    menu_Write_to_screen(ctx,
				 FRM("^x%02X"),bytech);			
	    menu_EndXX(ctx,pg_REVERSE);
	    CUR_col += 4;
	    POS++;
	
	} else {

	    menu_StartXX(ctx,pg_REVERSE);
	    menu_Writechar(ctx,'?');		    
	    menu_EndXX(ctx,pg_REVERSE);
	    
	    CUR_col += 1;
	    POS++;
	}

    }
    
    /* OK */
    
 done:
 no_space:

    *cur_col  = CUR_col;
    *pos      = POS;
    *ret      = RET_code;
    *mayclear = MAY_clear;

    DPRINT(Debug,12,(&Debug,"span_control=%d (ret code = %d)\n",
		     ok,RET_code));
    return ok;
}

enum span_result span_line(ctx,cur_line,cur_col,buffer,pos,mayclear,
			   this_flags,max_width,max_len,wrap_indicator,
			   on_space_p)
     struct menu_context *ctx;
     int cur_line;  
     int *cur_col;
     const struct string * buffer;
     int *pos;
     int *mayclear;
     int this_flags;
     int max_width;
     int max_len;
     int wrap_indicator;
     int * on_space_p;
{
    enum span_result ret = span_failure;
    
    int CUR_col  = *cur_col;
    int POS      = *pos;
    int MAY_clear = *mayclear;

    int buffer_len = string_len(buffer);
    int LINES, COLUMNS;

    int START_col = CUR_col;
    int START_pos = POS;
    int WP =  /* pager_indicate_wrapping */ wrap_indicator;
    int on_space = 0;

    if (on_space_p)
	on_space = *on_space_p;
    
    DPRINT(Debug,12,(&Debug,
		     "span_line: pos=%d, buffer len = %d, max_len=%d, line = %d, col = %d,\n",
		     POS,buffer_len,max_len,cur_line,CUR_col));

    menu_get_sizes(ctx, &LINES, &COLUMNS);

    DPRINT(Debug,12,(&Debug,
		     "span_line: LINES=%d, COLUMNS=%d, max_width=%d\n",
		     LINES,COLUMNS,max_width));

    if (POS >= buffer_len) {
	DPRINT(Debug,5,(&Debug,"span_line -- end of string\n"));
	ret = span_EOS;
	goto no_space;
    }
    
    if (cur_line < 0 || cur_line > LINES-1 ||
	CUR_col < 0 || CUR_col >= COLUMNS-1 -WP) {

	DPRINT(Debug,5,(&Debug,"span_line -- no space (wrap indicator %d)",
			WP));

	if (WP && cur_line >= 0 && cur_line <= LINES-1 &&
	    CUR_col > 0 && CUR_col <= COLUMNS-1 && POS < buffer_len) {
	    ret = span_wrapping;

	    DPRINT(Debug,5,(&Debug,", wrapping"));
	}
	DPRINT(Debug,5,(&Debug,"\n"));
	
	goto no_space;
    }

    menu_MoveCursor(ctx,cur_line,CUR_col);
    
    ret = span_ok;
    while (POS < buffer_len) {
	uint16         ch     = UNICODE_BAD_CHAR;
	unsigned char  bytech = 0;
	int            gchar_flags;
	int len = 0;   /* Length for next clip ... */
	int width;
	struct string * data = NULL;
	int visible_len;
	
	if (POS == buffer_len-1)  /* do NOT wrap with \ on last character  */
	    WP = 0;	   	  /* of buffer                             */
	    
	if (POS - START_pos >= max_len) {
	    DPRINT(Debug,12,(&Debug,"span_line: POS = %d -- line len = %d, max_len = %d reached\n",
			     POS,CUR_col, max_len)); 
	    
	    break;
	}

	gchar_flags = give_character_from_string(buffer,POS,
						 &ch,&bytech);

	if (! span_control(ctx,cur_line,&CUR_col,ch,&POS,&MAY_clear,
			   this_flags,max_width,
			   &ret,START_pos,START_col,WP,0,
			   gchar_flags,bytech,
			   on_space_p ? &on_space : NULL))
	    goto no_space;

	DPRINT(Debug,9,(&Debug, 
			"span_line after ctrl pos=%d (col = %d)",
			POS,CUR_col));
	if (POS >= buffer_len) {
	    DPRINT(Debug,9,(&Debug, ", end of buffer"));

	    if (on_space) {
		DPRINT(Debug,9,(&Debug, ", space collapsed"));
		ret = span_space_collapsed;
	    }
	    DPRINT(Debug,9,(&Debug,"\n"));
	    
	    break;
	} else if (POS == buffer_len-1) { /* do NOT wrap with \ on last character  */
	    WP = 0;	   	  /* of buffer                             */
	    DPRINT(Debug,9,(&Debug, ", last character"));
	}
	DPRINT(Debug,9,(&Debug,"\n"));

	if (on_space_p) {

	    int old_POS = POS;
	    
	    while (POS  < buffer_len) {
		
		gchar_flags = give_character_from_string(buffer,POS,
							 &ch,NULL);
		
		if (isoff(gchar_flags,GCHAR_unicode))
		    break;

	        if (ch == UNICODE_NO_BREAK_SPACE)
		    break;
		
		if (0 == unicode_ch(ch,UOP_space))
		    break;
		
		on_space = 1;
		POS++;
	    }

	    DPRINT(Debug,9,(&Debug, 
			    "span_line collapse space pos=%d => %d (col = %d)%s",
			    old_POS,POS,CUR_col,
			    on_space ? ", have space seen" : ""));
	    
	    if (POS >= buffer_len) {
		DPRINT(Debug,9,(&Debug, ", end of buffer"));

		if (on_space) {
		    DPRINT(Debug,9,(&Debug, ", space collapsed"));
		    ret = span_space_collapsed;
		}
		DPRINT(Debug,9,(&Debug,"\n"));
		
		break;
	    } else if (POS == buffer_len-1) { /* do NOT wrap with \ on last character  */
		WP = 0;	   	  /* of buffer                             */
		DPRINT(Debug,9,(&Debug, ", last character"));
	    }
	    DPRINT(Debug,9,(&Debug,"\n"));	    
	}

	if (POS < buffer_len && on_space) {
	    if (POS - START_pos >= max_len) {
		DPRINT(Debug,12,(&Debug,"span_line: POS = %d -- line len = %d, max_len = %d reached\n",
				 POS,CUR_col, max_len)); 
		
		break;
	    }

	    if (CUR_col >= COLUMNS-WP) {
		DPRINT(Debug,12,(&Debug,
				 "span_line: POS = %d -- no space for space, col=%d\n",
				 POS,CUR_col));
		break;				 
	    }

	    gchar_flags = give_character_from_string(buffer,POS,
						     &ch,NULL);

	    if (isoff(gchar_flags,GCHAR_unicode) ||
		ch == UNICODE_NO_BREAK_SPACE) {

		DPRINT(Debug,12,(&Debug,
				 "span_line: POS = %d, adding space\n",
				 POS));
		
		if (this_flags)
		    menu_StartXX(ctx,this_flags);
		
		menu_Writechar(ctx,' ');

		if (this_flags)
		    menu_EndXX(ctx,this_flags);
		
		CUR_col ++;
	    }
	}
	on_space = 0;
	
	/* calculate clip len ... */
	while (POS + len     < buffer_len &&
	       POS + len - START_pos <  max_len &&	   
	       CUR_col + len < COLUMNS &&
	       CUR_col + len - START_col < max_width) {
	    
	    if (POS + len < buffer_len -1 &&
		CUR_col + len >= COLUMNS-WP)
		break;

	    gchar_flags = give_character_from_string(buffer,POS+len,
						     &ch,NULL);

	    if (isoff(gchar_flags,GCHAR_unicode))
		break;
	    
	    if (ch <= 31 || ch == 0x007F /* ^? */)
		break;

	    if (on_space_p) {  /* stop for space */
		if (0 != unicode_ch(ch,UOP_space))
		    break;
	    }
	    
	    switch(ch) {
	    case UNICODE_BAD_CHAR:
	    case UNICODE_NO_BREAK_SPACE:
	    case UNICODE_SOFT_HYPHEN:
		goto text_found;
	    default:
		if (0 == unicode_ch(ch,UOP_noctrl))
		    goto text_found;
		break;
	    }
	    
	    len++;
	}

    text_found:	
	DPRINT(Debug,9,(&Debug, 
			"span_line after text  pos=%d ( len = %d)\n",
			POS,len));
	
	if (len < 1)
	    continue;    /* next character was also control char */

	if (WP && POS + len < buffer_len-1)
	    width = COLUMNS - CUR_col -WP;
	else
	    width = COLUMNS - CUR_col;
	if (CUR_col + width - START_col > max_width) {
	    width = max_width - CUR_col + START_col;

	    DPRINT(Debug,12,(&Debug, 
			     "span_line: limiting width to %d, max_width=%d (wrap indicator %d)\n",
			     width,max_width,WP));
	}

	/* curses_printable_clip updates POS */
	data = curses_printable_clip(buffer,&POS,len,&visible_len,width);

	if (data) {
	    DPRINT(Debug,15,(&Debug, 
			     "span_line: (len=%d) data=%S\n",string_len(data),data));

	    CUR_col += visible_len;   
		    
	    DPRINT(Debug,9,(&Debug, 
			    "span_line after clip pos=%d (len = %d, col=%d  visible_len=%d)\n",
			    POS,len,CUR_col, visible_len));

	    if (this_flags)
		menu_StartXX(ctx,this_flags);

	    menu_Write_to_screen(ctx,FRM("%S"),data);

	    if (this_flags)
		menu_EndXX(ctx,this_flags);
	    
	    free_string(&data);

	    if (visible_len < 1) {
		DPRINT(Debug,9,(&Debug, 
				"span_line: No space for next character on line"));
		if (START_pos < POS) {

		    DPRINT(Debug,9,(&Debug, ", wrapping"));
		    ret = span_wrapping;
		}

		DPRINT(Debug,9,(&Debug,"\n"));
		goto no_space;
	    }

	} else {
	    /* Treat next character as control character */

	    menu_StartXX(ctx,pg_REVERSE);
	    menu_Writechar(ctx,'?');		    
	    menu_EndXX(ctx,pg_REVERSE);

	    CUR_col += 1;
	    POS++;

	}
	
    }
    
 no_space:
    if (/* pager_indicate_wrapping */ wrap_indicator &&
	ret == span_wrapping && CUR_col < COLUMNS) {
	DPRINT(Debug,12,(&Debug,"span_line: wrapping, adding \\ to col = %d\n",
			 CUR_col));
	menu_StartXX(ctx,pg_BOLD);
	menu_Writechar(ctx,'\\');
	menu_EndXX(ctx,pg_BOLD);
	CUR_col++;
    }


    if (POS == START_pos && ret == span_ok) {
	
	DPRINT(Debug,12,(&Debug,"span_line: -- failure -- no advance\n"));
	ret = span_failure;
    }
    
    *cur_col  = CUR_col;
    *pos      = POS;
    *mayclear = MAY_clear;

    if (on_space_p)
	*on_space_p = on_space;
    
    DPRINT(Debug,12,(&Debug,
		     "span_line=%d",ret));
    switch (ret) {
    case span_failure:  DPRINT(Debug,12,(&Debug," span_failure")); break;
    case span_ok:       DPRINT(Debug,12,(&Debug," span_ok"));      break;
    case span_FF_seen:  DPRINT(Debug,12,(&Debug," span_FF_seen")); break;
    case span_EOS:      DPRINT(Debug,12,(&Debug," span_EOS"));     break;
    case span_NL_seen:  DPRINT(Debug,12,(&Debug," span_NL_seen")); break;
    case span_wrapping: DPRINT(Debug,12,(&Debug," span_wrapping")); break;
    case span_wordwrapped: DPRINT(Debug,12,(&Debug," span_wordwrapped")); break;
    case span_clear_alt_data: DPRINT(Debug,12,(&Debug," span_clear_alt_data")); break;
    case span_space_collapsed: DPRINT(Debug,12,(&Debug," span_space_collapsed")); break;
    }    
    DPRINT(Debug,12,(&Debug," (pos=%d -> %d, col = %d -> %d, line = %d)\n",
			 START_pos, POS, START_col, CUR_col, cur_line));
    return ret;
}

/* Word wrapping */
enum span_result span_words(ctx,cur_line,cur_col,buffer,pos,
			    mayclear,this_flags,max_width,max_len,
			    joined_line, p_start_COL,p_width,wrap_indicator,
			    on_space_p)
     struct menu_context *ctx;
     int cur_line; 
     int *cur_col;
     const struct string * buffer;
     int *pos; 
     int *mayclear;
     int this_flags;
     int max_width;
     int max_len;
     int joined_line;  
     int p_start_COL;
     int p_width;
     int wrap_indicator;
     int * on_space_p;
{
    enum span_result ret = span_failure;
    
    int CUR_col  = *cur_col;
    int POS      = *pos;
    int MAY_clear = *mayclear;

    int buffer_len = string_len(buffer);
    int LINES, COLUMNS;

    int START_col = CUR_col;
    int START_pos = POS;
    int WP = /* pager_indicate_wrapping */ wrap_indicator;
    int on_space = 0;

    if (on_space_p)
	on_space = * on_space_p;
    
    DPRINT(Debug,12,(&Debug,
		     "span_words: pos=%d, buffer len = %d, max_len=%d, line = %d, col = %d,\n",
		     POS,buffer_len,max_len,cur_line,CUR_col));

    menu_get_sizes(ctx, &LINES, &COLUMNS);

    DPRINT(Debug,12,(&Debug,
		     "span_words: LINES=%d, COLUMNS=%d, max_width=%d, joined_line=%d, p_start_COL=%d\n",
		     LINES,COLUMNS,max_width,joined_line,p_start_COL));

    if (p_width) {
	DPRINT(Debug,12,(&Debug,
			 "span_words: paragraph width = %d\n",p_width));
    }

    if (POS >= buffer_len) {
	DPRINT(Debug,5,(&Debug,"span_words -- end of string\n"));
	ret = span_EOS;
	goto no_space;
    }
    
    if (joined_line && (CUR_col < 0 || CUR_col >= COLUMNS-1 -WP)) {
	DPRINT(Debug,5,(&Debug,
			"span_words -- no space -- joined line (wrap indicator %d)\n",
			WP));

	ret = span_wordwrapped;	
	goto no_space;
    }

    if (joined_line && ((p_width > 0 && CUR_col > p_start_COL + p_width) 
			||
			(p_width < 0 && CUR_col > COLUMNS + p_width))
	) {

	DPRINT(Debug,5,(&Debug,
			"span_words -- joined line, paragraph width = %d, word wrapping at beginning\n",
			p_width));
	ret = span_wordwrapped;	
	goto no_space;
    }

    if (cur_line < 0 || cur_line > LINES-1 ||
	CUR_col < 0 || CUR_col >= COLUMNS-1 -WP) {

	DPRINT(Debug,5,(&Debug,"span_words -- no space (wrap indicator %d)\n",
			WP));

	goto no_space;
    }

    menu_MoveCursor(ctx,cur_line,CUR_col);

    ret = span_ok;
    while (POS < buffer_len) {
	uint16         ch = UNICODE_BAD_CHAR;
	unsigned char  bytech = 0;
	int            gchar_flags;
	int CLIP_start_pos = POS;
	int CLIP_start_col = CUR_col;
	int   overflow = 0;
	int   quit     = 0;

	int as_control = 0;
	int wordwrap_after = 0;

	struct clip_data {
	    int start_pos;
	    int len;

	    struct string *data;
	    
	    int start_col;
	    int visible_len;

	    uint16 word_break;   /* Word break character */
	    uint16 no_break;   /* No break character */

	} * clips = NULL;
	int clip;

	if (POS == buffer_len-1)  /* do NOT wrap with \ on last character  */
	    WP = 0;	   	  /* of buffer                             */
	    
	if (POS - START_pos >= max_len) {
	    DPRINT(Debug,12,(&Debug,"span_words: POS = %d -- line len = %d, max_len = %d reached\n",
			     POS,CUR_col, max_len)); 
	    
	    break;
	}

	gchar_flags = give_character_from_string(buffer,POS,
						 &ch,&bytech);

	if (! span_control(ctx,cur_line,&CUR_col,ch,&POS,&MAY_clear,
			   this_flags,max_width,
			   &ret,START_pos,CLIP_start_col,WP,1,
			   gchar_flags,bytech,
			   on_space_p ? &on_space : NULL

			   ))
	    goto no_space;
	
	if (joined_line && POS > START_pos) {
	    DPRINT(Debug,9,(&Debug, 
			    "span_words: Resetting joined_line flag\n"));
	    joined_line = 0;
	}

	DPRINT(Debug,9,(&Debug, 
			"span_words after ctrl pos=%d (col = %d)",
			POS,CUR_col));
	if (POS >= buffer_len) {
	    DPRINT(Debug,9,(&Debug, ", end of buffer"));
	    
	    if (on_space) {
		DPRINT(Debug,9,(&Debug, ", space collapsed"));
		ret = span_space_collapsed;
	    }
	    DPRINT(Debug,9,(&Debug,"\n"));
	    
	    break;
	} else if (POS == buffer_len-1) { /* do NOT wrap with \ on last character  */
	    WP = 0;	   	  /* of buffer                             */
	    DPRINT(Debug,9,(&Debug, ", last character"));
	}
	DPRINT(Debug,9,(&Debug,"\n"));

	if (on_space_p) {
	    
	    int old_POS = POS;
	    
	    while (POS  < buffer_len) {
		
		gchar_flags = give_character_from_string(buffer,POS,
							 &ch,NULL);
		
		if (isoff(gchar_flags,GCHAR_unicode))
		    break;

	        if (ch == UNICODE_NO_BREAK_SPACE)
		    break;
		
		if (0 == unicode_ch(ch,UOP_space))
		    break;
		
		on_space = 1;
		POS++;
	    }

	    DPRINT(Debug,9,(&Debug, 
			    "span_words collapse space pos=%d => %d (col = %d)%s",
			    old_POS,POS,CUR_col,
			    on_space ? ", have space seen" : ""));
	    
	    if (POS >= buffer_len) {
		DPRINT(Debug,9,(&Debug, ", end of buffer"));

		if (on_space) {
		    DPRINT(Debug,9,(&Debug, ", space collapsed"));
		    ret = span_space_collapsed;
		}
		DPRINT(Debug,9,(&Debug,"\n"));
		
		break;
	    } else if (POS == buffer_len-1) { /* do NOT wrap with \ on last character  */
		WP = 0;	   	  /* of buffer                             */
		DPRINT(Debug,9,(&Debug, ", last character"));
	    }
	    DPRINT(Debug,9,(&Debug,"\n"));	    
	}

	if (POS < buffer_len && on_space) {
	    if (POS - START_pos >= max_len) {
		DPRINT(Debug,12,(&Debug,"span_words: POS = %d -- line len = %d, max_len = %d reached\n",
				 POS,CUR_col, max_len)); 
		
		break;
	    }

	    if (CUR_col >= COLUMNS-WP) {
		DPRINT(Debug,12,(&Debug,
				 "span_words: POS = %d -- no space for space, col=%d\n",
				 POS,CUR_col));
		break;				 
	    }

	    gchar_flags = give_character_from_string(buffer,POS,
						     &ch,NULL);

	    if (isoff(gchar_flags,GCHAR_unicode) ||
		ch == UNICODE_NO_BREAK_SPACE) {

		DPRINT(Debug,12,(&Debug,
				 "span_words: POS = %d, adding space\n",
				 POS));
		
		if (this_flags)
		    menu_StartXX(ctx,this_flags);
		
		menu_Writechar(ctx,' ');

		if (this_flags)
		    menu_EndXX(ctx,this_flags);
		
		CUR_col ++;
	    }
	}
	on_space = 0;
	
	CLIP_start_pos = POS;
	CLIP_start_col = CUR_col;

	for (clip = 0; !overflow  && !quit &&
		 CLIP_start_pos < buffer_len; clip++) {
	    int len = 0;   /* Length for next clip ... */
	    int width;
	    struct string * data = NULL;

	    int visible_len;

	    uint16 word_break = 0;   /* Word break character */
	    uint16 no_break   = 0;   /* No break character */

	    int pos1;
	    
	    /* calculate clip len ... */
	    while (CLIP_start_pos + len     < buffer_len) {
	    
		gchar_flags = give_character_from_string(buffer,
							 CLIP_start_pos+len,
							 &ch,NULL);
	    	       		
		if (0 == (gchar_flags & GCHAR_unicode)) {
		    quit = 1;
		    goto text_found;
		}

		switch(ch) {
		case 0x0009:  /* HT */
		    word_break = ch;       		    
		    goto text_found;

		case UNICODE_BAD_CHAR:
		case UNICODE_NO_BREAK_SPACE:
		    no_break = ch;
		    
		    goto text_found;
		    
		case UNICODE_SOFT_HYPHEN:
		    word_break = ch;
		    goto text_found;
		    
		default:
		    if (ch <= 31 || ch == 0x007F /* ^? */) {
			quit = 1;
			goto text_found;
		    }

		    if (0 == unicode_ch(ch,UOP_noctrl)) {
			quit = 1;
			goto text_found;
		    }
		    
		    word_break = unicode_ch(ch,UOP_space);
		    if (word_break) 		
			goto text_found; 		    
		    break;
		}
		
		if (CLIP_start_pos + len - START_pos <  max_len &&	   
		    CLIP_start_col + len < COLUMNS &&
		    CLIP_start_col + len - START_col < max_width) {
		    
		    if (CLIP_start_pos + len < buffer_len-1 &&
			CLIP_start_col + len >= COLUMNS-WP) {

			overflow = 1;
			break;
		    }

		    len++;
		} else {
		    overflow = 1;
		    break;
		}
	    }
	    
	text_found:

	    if (word_break) {
		if (on_space_p) {
		    DPRINT(Debug,9,(&Debug, 
				    "span_words [%d] converting word_break=%04x",
				    clips,word_break));
		    
		    word_break = 0x0020 /* SPACE */;
		    
		    DPRINT(Debug,9,(&Debug, " => %04x collapse space\n",
				    word_break));			    
		}
	    }
	    
	    DPRINT(Debug,9,(&Debug, 
			    "span_words after text [%d] pos=%d (len = %d), word_break=%04x no_break=%04x%s%s\n",
			    clip,CLIP_start_pos,len,
			    word_break,no_break,
			    overflow ? ", overflow" : "",
			    quit ? ", quit" : ""));

	    if (len < 1) {

		if (overflow) {
		    DPRINT(Debug,12,(&Debug, 
				     "span_words: [%d] at end of line\n",clip));
		    wordwrap_after = 1;
		} else {
		    DPRINT(Debug,12,(&Debug, 
				     "span_words: [%d] at control\n",clip));
		}

		if (clip < 1)
		    goto no_clip;

		break;    /* next character was also control char */
	    }

	    if (WP && CLIP_start_pos + len < buffer_len-1)
		width = COLUMNS - CLIP_start_col -WP;
	    else
		width = COLUMNS - CLIP_start_col;

	    if (CLIP_start_col + width - START_col > max_width) {
		width = max_width - CLIP_start_col + START_col;
		
		DPRINT(Debug,12,(&Debug, 
				 "span_words: [%d] limiting width to %d, max_width=%d (wrap indicator %d)\n",
				 clip,width,max_width,WP));
	    }


	    pos1 = CLIP_start_pos;

	    /* curses_printable_clip updates POS */
	    data = curses_printable_clip(buffer,&CLIP_start_pos,len,&visible_len,
					 width);

	    if (CLIP_start_pos < pos1 + len) {
		DPRINT(Debug,12,(&Debug, 
				 "span_words: [%d] pos=%d, last pos=%d, overflow\n",
				 clip,CLIP_start_pos,pos1 + len));
		overflow = 1;
	    }

	    if (visible_len < 1) {
		DPRINT(Debug,9,(&Debug, 
				"span_words: [%d] No space for next character on line\n",
				clip));

		if (data)
		    free_string(&data);
		break;
	    }

	    if (data) {
		DPRINT(Debug,15,(&Debug, 
				 "span_words: [%d] (len=%d) data=%S\n",clip,string_len(data),data));

		DPRINT(Debug,9,(&Debug, 
				"span_words [%d] after clip pos=%d (len = %d, col=%d  visible_len=%d)\n",
				clip,
				CLIP_start_pos,len,
				CLIP_start_col + visible_len, visible_len));

		DPRINT(Debug,20,(&Debug, 
				 "span_words [%d] = %S\n",clip,data));

		clips = safe_array_realloc(clips,
					   (clip+1), sizeof(clips[0]));
		
		
		clips[clip].start_pos  = pos1;
		clips[clip].len  = CLIP_start_pos - pos1;
		clips[clip].data = data;
		clips[clip].start_col   = CLIP_start_col;
		clips[clip].visible_len = visible_len; 
		clips[clip].word_break = word_break;
		clips[clip].no_break = no_break;
		
		CLIP_start_col += visible_len;
		
		if (word_break || no_break) {  /* Assumed one character wide */
		    CLIP_start_pos++;

		    if (0x0009 /* HT */ == word_break)
			CLIP_start_col = (( CLIP_start_col / 
					    get_tabspacing() ) +1) * 
			    get_tabspacing(); 
		    else
			CLIP_start_col++;
		}
	    } else {
		if (clip < 1)
		    as_control = 1;

		DPRINT(Debug,9,(&Debug, 
				"span_words: [%d] not printable%s\n",
				clip, as_control ? ", as control" : ""));


		break;
	    }
	}

	if (clip < 1) {
	    
	    if (as_control) {
		/* Treat next character as control character */

		menu_StartXX(ctx,pg_REVERSE);
		menu_Writechar(ctx,'?');		    
		menu_EndXX(ctx,pg_REVERSE);
		
		CUR_col += 1;
		POS++;

	    } else {
		DPRINT(Debug,9,(&Debug, 
				"span_words: No space for next character on line\n"));
		if (START_pos < POS)
		    ret = span_wrapping;
		
		goto no_space;
	    }

	no_clip:
	    
	    DPRINT(Debug,12,(&Debug, 
			     "span_words: No clips\n"));


	} else {
	    int done = 0;

	    if (1 == clip && overflow && joined_line) {
		DPRINT(Debug,9,(&Debug, 
				"span_words: No space for first word, joined line\n"));
		ret = span_wordwrapped;
	    } else  if (1 == clip && overflow) {
		/* Word wrapping not possible */

		DPRINT(Debug,9,(&Debug, 
				"span_words: First word truncated, wrapped\n"));

		if (this_flags)
		    menu_StartXX(ctx,this_flags);

		menu_Write_to_screen(ctx,FRM("%S"),clips[0].data);

		if (this_flags)
		    menu_EndXX(ctx,this_flags);

		POS = clips[0].start_pos + clips[0].len;

		CUR_col += clips[0].visible_len;

		ret = span_wrapping;			       

	    } else {
		int last_clip = -1;
		int i;

		if (overflow) {
		    
		    DPRINT(Debug,9,(&Debug, 
				    "span_words: all does not fit to line, %d clips\n",
				    clip));
		    
		    for ( i = 0; i < clip; i++) {
			if (clips[i].word_break) 
			    last_clip = i;
			else if (! clips[i].no_break) {
			    DPRINT(Debug,9,(&Debug, 
					    "span_words: [%d] no separator%s%s\n",
					    i,
					    i < clip-1 ? ", not last" : "",
					    i == clip-1 && overflow ? 
					    ", overflow" : ""));
			    
			    if (i < clip-1)   /* Should not happend */
				break;			
			}
		    }

		    if (last_clip >= 0) {
			DPRINT(Debug,9,(&Debug, 
					"span_words: last clip = %d (pos = %d..%d)%s\n",
					last_clip,
					clips[last_clip].start_pos,
					clips[last_clip].start_pos + clips[last_clip].len,
					last_clip < clip-1 ? ", not all" : ""));
		    }

		    
		} else {
		    /* Fixed certain case where builtin pager refused
		       display rest of mail when word-wrapping line 
		       line which include non-break space!
		    */
		    
		    last_clip = clip -1;
		    
		    DPRINT(Debug,9,(&Debug, 
				    "span_words: using all %d clips, next pos %d%s\n",
				    clip, CLIP_start_pos,
				    CLIP_start_pos >= buffer_len ? " (end of buffer)" : ""));
		} 
				
		if (last_clip < 0) {
		    DPRINT(Debug,9,(&Debug, 
				    "span_words: Can't word wrap, %d clips, %s\n",
				    clip,
				    overflow ? "overflow" : "no overflow"));

		    if (overflow && joined_line) {
			DPRINT(Debug,9,(&Debug, 
					"span_words: joined line, word wrapping before first clip\n"));
			ret = span_wordwrapped;
		    } else if (overflow && clip > 0) {
			last_clip = clip -1;
			DPRINT(Debug,9,(&Debug, 
					"span_words: truncating on clip %d, wrapped\n",
					last_clip));
			ret = span_wrapping;
		    } else
			done = 1;
		} else {    /* 'Got' word */

		    if (last_clip < clip-1) {
			ret = span_wordwrapped;
			
			DPRINT(Debug,9,(&Debug, 
					"span_words: word wrapping after clip %d\n",
					last_clip));
		    }
		}

		for (i = 0; i <= last_clip && !done; i++) {

		    if (this_flags)
			menu_StartXX(ctx,this_flags);
		    
		    menu_Write_to_screen(ctx,FRM("%S"),clips[i].data);
		    
		    if (this_flags)
			menu_EndXX(ctx,this_flags);
		    
		    
		    POS = clips[i].start_pos + clips[i].len;
		    
		    CUR_col += clips[i].visible_len;
		    
		    if (CUR_col >= COLUMNS) {
			DPRINT(Debug,9,(&Debug, 
					"span_words: [%d] at end of line, col=%d%s\n",
					i,CUR_col,
					clips[i].word_break ? "" : ", no word break"));
			
			if (! clips[i].word_break)
			    ret = span_wrapping;
			done=1;
			break;
		    }
		    
		    if (clips[i].word_break && 
			((p_width > 0 &&  CUR_col > p_start_COL + p_width) 
			 ||
			 (p_width < 0 && CUR_col > COLUMNS + p_width))
			) {
			DPRINT(Debug,9,(&Debug, 
					"span_words: [%d] word wrapping, paragraph width = %d, col=%d\n",
					
					i,p_width,CUR_col));
			ret = span_wordwrapped;			
			done = 1;
		    }
		    
		    switch(clips[i].word_break) {
		    case 0x0009:  /* HT */
			
			/* Make sure that carbage from previous
			   output is not left */
			if (MAY_clear) {
			    menu_CleartoEOLN(ctx);
			    MAY_clear = 0;
			}
			
			if (! span_tab(ctx,&CUR_col,&POS,this_flags,max_width,
				       START_pos,START_col,cur_line)) {
			    
			    DPRINT(Debug,12,(&Debug,
					     "span_words: [%d] ch=%04x wrapped\n",
					     i,ch)); 
			    
			    done=1;
			    
			    
			    goto quit_clip;
			}
			
			break;
			
		    case UNICODE_SOFT_HYPHEN:   /* Don't write (soft) hyphen
						   if inside of word */
			if (i == last_clip || done) {
			    if (this_flags)
				menu_StartXX(ctx,this_flags);
			    
			    menu_Writechar(ctx,'-');
			    
			    if (this_flags)
				menu_EndXX(ctx,this_flags);
			    
			    CUR_col++;
			    POS++;
			}
			break;
			
		    case 0x0000:  /* None */
			break;
			
		    default:   /* Any other space */
			POS++;
			CUR_col++;
			
			if (this_flags)
			    menu_StartXX(ctx,this_flags);
			
			menu_WriteUnicode(ctx,clips[i].word_break);
			
			if (this_flags)
			    menu_EndXX(ctx,this_flags);
			break;
			
		    }
		    
		    switch(clips[i].no_break) {			
		    case UNICODE_BAD_CHAR:
			
			menu_StartXX(ctx,pg_REVERSE);
			menu_Writechar(ctx,'?');
			menu_EndXX(ctx,pg_REVERSE);
			CUR_col ++;
			POS++;
			
			break;
		    case UNICODE_NO_BREAK_SPACE:
			if (this_flags)
			    menu_StartXX(ctx,this_flags);
			
			menu_Writechar(ctx,' ');
			
			if (this_flags)
			    menu_EndXX(ctx,this_flags);
			
			CUR_col++;
			POS++;
			break;
		    }		
		}
	    }

	quit_clip:
	    if (clips) {
		int i;

		for ( i = 0; i < clip; i++) {
		    
		    if (clips[i].data)
			free_string(&(clips[i].data));
		}
		free(clips);
		clips = NULL;
	    }

	    if (overflow && POS < buffer_len && ret == span_ok) {

		gchar_flags = give_character_from_string(buffer,POS,
							 &ch,NULL);
		
		if (0 != (gchar_flags & GCHAR_unicode) &&
		    unicode_ch(ch,UOP_space)) {
		    DPRINT(Debug,9,(&Debug, 
				    "span_words: ignoring space %04x at pos=%d col=%d, will word wrap\n",
				    ch,POS,CUR_col));
		    ret = span_wordwrapped;
		    POS++;
		}		    
	    }

	    if (wordwrap_after && ret == span_ok) {
		DPRINT(Debug,9,(&Debug, 
				"span_words: word wrapping after words, pos=%d col=%d\n",
				POS,CUR_col));
		ret = span_wordwrapped;		
	    }

	    if (ret == span_wrapping) {
		DPRINT(Debug,9,(&Debug, 
				"span_words: is wrapping, pos=%d, col=%d\n",
				POS,CUR_col));
		goto no_space;
	    }

	    if (done) {
		DPRINT(Debug,9,(&Debug, 
				"span_words: done, pos=%d, col=%d\n",
				POS,CUR_col));
		goto no_space;
	    }

	    if (ret == span_wordwrapped) {
		DPRINT(Debug,9,(&Debug, 
				"span_words: word wrapped, pos=%d, col=%d\n",
				POS,CUR_col));
		goto no_space;
	    }

	}
    }

 no_space:
    if (/* pager_indicate_wrapping */ wrap_indicator &&
	ret == span_wrapping && CUR_col < COLUMNS) {
	DPRINT(Debug,12,(&Debug,"span_words: wrapping, adding \\ to col = %d\n",
			 CUR_col));
	menu_StartXX(ctx,pg_BOLD);
	menu_Writechar(ctx,'\\');
	menu_EndXX(ctx,pg_BOLD);
	CUR_col++;
    }


    if (POS == START_pos && ret == span_ok) {
	
	DPRINT(Debug,12,(&Debug,"span_words: -- failure -- no advance\n"));
	ret = span_failure;
    }
    
    *cur_col  = CUR_col;
    *pos      = POS;
    *mayclear = MAY_clear;

    if (on_space_p)
	*on_space_p = on_space;
    
    DPRINT(Debug,12,(&Debug,
		     "span_words=%d",ret));
    switch (ret) {
    case span_failure:  DPRINT(Debug,12,(&Debug," span_failure")); break;
    case span_ok:       DPRINT(Debug,12,(&Debug," span_ok"));      break;
    case span_FF_seen:  DPRINT(Debug,12,(&Debug," span_FF_seen")); break;
    case span_EOS:      DPRINT(Debug,12,(&Debug," span_EOS"));     break;
    case span_NL_seen:  DPRINT(Debug,12,(&Debug," span_NL_seen")); break;
    case span_wrapping: DPRINT(Debug,12,(&Debug," span_wrapping")); break;
    case span_wordwrapped: DPRINT(Debug,12,(&Debug," span_wordwrapped")); break;
    case span_clear_alt_data: DPRINT(Debug,12,(&Debug," span_clear_alt_data")); break;
    case span_space_collapsed: DPRINT(Debug,12,(&Debug," span_space_collapsed")); break;
    }    
    DPRINT(Debug,12,(&Debug," (pos=%d -> %d, col = %d -> %d, line = %d)\n",
		     START_pos, POS, START_col, CUR_col, cur_line));
    return ret;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */



