static char rcsid[] = "@(#)$Id: newmbox.c,v 2.22 2024/06/23 07:38:36 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.22 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/newmbox.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/**  read new folder **/

#include "def_elm.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"mbox");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif       

/* For INT_MAX */
#include <limits.h>


#if  !defined(ANSI_C) && !defined(atol) /* avoid problems with systems that declare atol as a macro */
extern void rewind();
extern long atol();
#endif

/* -1  ... failure */
enum read_hdr_mode {
    read_hdr_normal     = 0,
    read_hdr_new_only   = 1,
    read_hdr_reconnect  = 5
};

static int read_headers P_((enum read_hdr_mode       read_mode,
			    struct current_storage * storage,
			    struct menu_context    * page,
			    RECONNECT_MODE         * reconnect_mode_ptr /* Hack for SESSIONLOCK_TRUNCATE */
			    ));   /* Prototype */

enum sessionlock_status  open_folder_lock(direction,folder) 
     enum lock_direction direction;
     struct folder_info *folder;
{
    enum sessionlock_status ret;
        
    DPRINT(Debug,10,(&Debug, 
		     "open_folder_lock: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    ret = sessionlock_folder(folder,SESSIONLOCK_CHECK,NULL,
			     NULL);
    switch (ret) {
    case sessionlock_fail:
	goto cleanup;

    default:
	lock(direction, folder);
    }


 cleanup:    
    DPRINT(Debug,10,(&Debug,
		     "open_folder_lock=%d",
		     ret));
    switch (ret) {
    case sessionlock_fail: DPRINT(Debug,10,(&Debug, " sessionlock_fail"));
	break;
    case sessionlock_open: DPRINT(Debug,10,(&Debug, " sessionlock_open"));
	break;
    case sessionlock_reconnect:
	DPRINT(Debug,10,(&Debug, " sessionlock_reconnect")); break;
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return ret;
}

int read_new_mails(storage, page) 
     struct current_storage *storage;
     struct menu_context *page;
{
    int result = 0;
    
    enum sessionlock_status ret;
    int r = 0;
    RECONNECT_MODE    reconnect_mode  = NULL;
    
    ret = sessionlock_folder(storage->current_folder,
			     SESSIONLOCK_NONE,NULL,
			     &reconnect_mode
			     );

    if (reconnect_mode) {
	DPRINT(Debug,12,(&Debug, 
			 "read_new_mails: reconnect_mode set %p\n",
			 reconnect_mode));
    }
    
    switch (ret) {
    case sessionlock_fail:
	
	result = 0;
	goto cleanup;
	
    case sessionlock_open:
	r = read_headers(read_hdr_new_only,storage,page,
			 &reconnect_mode);  /* Returns number of mails readed */
	break;
    case sessionlock_reconnect:
	r = read_headers(read_hdr_reconnect,storage,page,
			 &reconnect_mode);  /* Returns number of mails readed */
	break;
    }
    
    if (r < 0 ||
	ferror_folder(storage->current_folder,1)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorWhenReading,
			  "Error when reading: %S"),
		  storage->current_folder->cur_folder_disp);
	
	result = 0;
	goto cleanup;

    }

    DPRINT(Debug,12,(&Debug, "read_new_mails: Got %d mails\n",r));    
    result = 1;
    
 cleanup:
    if (reconnect_mode)
	free_reconnect_mode(& reconnect_mode);
    
    DPRINT(Debug,12,(&Debug, "read_new_mails=%d\n",
		     result));
		     
    return result;
}

int reconnect_mails(storage,page)
     struct current_storage *storage;
     struct menu_context *page;
{
    int result = 0;
    
    enum sessionlock_status ret;
    int r = 0;
    RECONNECT_MODE    reconnect_mode  = NULL;
    
    ret = sessionlock_folder(storage->current_folder,
			     SESSIONLOCK_RECONNECT,  /* SESSIONLOCK_NONE may use EXAMINE on IMAP */
			     NULL,
			     &reconnect_mode
			     );

    if (reconnect_mode) {
	DPRINT(Debug,12,(&Debug, 
			 "reconnect_mails: reconnect_mode set %p\n",
			 reconnect_mode));
    }

    switch (ret) {
    case sessionlock_fail:

	result = 0;
	goto cleanup;
	
	break;
    case sessionlock_open:
    case sessionlock_reconnect:
	r = read_headers(read_hdr_reconnect,storage,page,
			 &reconnect_mode);
	/* Returns number of mails readed */
	break;
    }

    if (r < 0 ||
	ferror_folder(storage->current_folder,1)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorWhenReading,
			  "Error when reading: %S"),
		  storage->current_folder->cur_folder_disp);

	result = 0;
	goto cleanup;

    }

    DPRINT(Debug,12,(&Debug, "reconnect_mails: Got %d mails\n",r));
    result = 1;
    
 cleanup:
     if (reconnect_mode)
	free_reconnect_mode(& reconnect_mode);
     
     DPRINT(Debug,12,(&Debug, "reconnect_mails=%d\n",
		      result));		     
     return result;
 
}


void add_new_mailbox (new_folder,mailbox,parent_page,quotad)
     struct folder_info   * new_folder;
     struct MailboxView   * mailbox;
     struct menu_context  * parent_page;
     struct quota_display * quotad; 
{
    struct current_storage *storage = NULL;

    int mbc = get_storage_count(mailbox);
    int i;
    enum sessionlock_status ret = sessionlock_fail;
    int same_file = 0;
    int r = 0;
    int was_reconnected;
    RECONNECT_MODE    reconnect_mode  = NULL;
    struct cancel_data *cd            = NULL;

    struct menu_context  * current_page = parent_page;

    if (quotad) {
	current_page =  quota_display_current_page(quotad);
    }
    
    /* Shold not be same file, but check anyway ... */

    for (i = 0; i < mbc; i++) {
	struct current_storage *s = get_storage(mailbox,i);

	if (!s)
	    continue;

	if (s->current_folder &&
	    new_folder == s->current_folder) {
	    storage   = s;
	    same_file++;
	    break;
	}
    }    

    if (new_folder->was_reconnected) {
	DPRINT(Debug,10,(&Debug, 
			 "add_new_mailbox: Mailfile was reconnected: %s\n",
			 new_folder->cur_folder_sys));
    }
    was_reconnected = new_folder->was_reconnected;

    if (quotad) {
	struct mail_quotaroot_list * quota_list = NULL;

	setup_mbx_cancel_message(&cd,mbx_checking_mailbox);

	quota_list = give_folder_quotaroot_l(new_folder,cd,quotam_normal);

	if (quota_list) {
	    quota_display_add_qlist(quotad,quota_list,
				    new_folder->cur_folder_disp,cd);

	    free_mail_quotaroot_list(&quota_list);	   				    
	}

	if (is_canceled(cd)) {
	    DPRINT(Debug,10,(&Debug, 
			     "add_new_mailbox: give_folder_quotaroot_l() canceled\n"));
	    goto cleanup;
	}

	free_cancel(&cd);
    }
    
    if (! same_file) {

	storage = new_storage(new_folder);
	
	ret = sessionlock_folder(storage->current_folder,
				 SESSIONLOCK_NORMAL,NULL,
				 &reconnect_mode);
      
    } else {
	

	DPRINT(Debug,1,(&Debug, 
			"WARNING -- add_new_mailbox is reopening file which is already open: %s\n",
			storage->current_folder->cur_folder_sys));
	
	
	/* folder (as opposite to mailbox) can be session locked
	 * only when it is opened. That call can do session locking
	 * for both (folders and mailboxes)
	 */
	
	ret = sessionlock_folder(storage->current_folder,
				 SESSIONLOCK_TRUNCATE,
				 NULL,
				 &reconnect_mode);
    }

    if (reconnect_mode) {
	DPRINT(Debug,10,(&Debug, 
			 "add_new_mailbox: reconnect_mode set %p\n",
			 reconnect_mode));
    }

    switch (ret) {
    case sessionlock_fail:
	
	if (!same_file) {
	    int r;

	    setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);
	    
	    DPRINT(Debug,9, (&Debug, 
			     "add_new_mailbox: freeing storage\n"));
	    r = free_storage(&storage,cd);
	    if (!r) {
		DPRINT(Debug,9, (&Debug, 
				 "add_new_mailbox: free_storage failed\n"));
	    }	
	    if (is_canceled(cd)) {
		DPRINT(Debug,9, (&Debug, 
				 "add_new_mailbox: Cancel on free_storage\n"));
	    }	   
	}
	    
	DPRINT(Debug,1,(&Debug, 
			"add_new_mailbox: WARNING -- add_new_mailbox failed: %s\n",
			storage->current_folder->cur_folder_sys));
	
	goto cleanup;
   
    case sessionlock_open:
	DPRINT(Debug,10,(&Debug, 
			 "add_new_mailbox Mailfile opened: %s\n",
			 storage->current_folder->cur_folder_sys));

	clear_error();

	if (was_reconnected) {
	    r = read_headers(read_hdr_reconnect,storage,current_page,
			     &reconnect_mode);  /* Returns number of mails readed */
	} else {
	    r= read_headers(read_hdr_normal,storage, current_page,
			    &reconnect_mode); /* Returns number of mails readed */
	}

	break;

    case sessionlock_reconnect:
	DPRINT(Debug,10,(&Debug, 
			 "Mailfile opened: %s (reconnected?)\n",
			 storage->current_folder->cur_folder_sys));

	clear_error();

	r = read_headers(read_hdr_reconnect,storage,current_page,
			 &reconnect_mode);  /* Returns number of mails readed */
	break;
    }
    
    if (r < 0 || 
	ferror_folder(storage->current_folder,1)) {

	DPRINT(Debug,1,(&Debug, 
			"WARNING -- add_new_mailbox failed: %s\n",
			storage->current_folder->cur_folder_sys));

	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorWhenReading,
			  "Error when reading: %S"),
		  storage->current_folder->cur_folder_disp);

	if (!same_file) {
	    int r;
	    struct cancel_data *cd = NULL;

	    setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);

	    DPRINT(Debug,9, (&Debug, 
			     "add_new_mailbox: freeing storage\n"));
	    r = free_storage(&storage,cd);
	    if (!r) {
		DPRINT(Debug,9, (&Debug, 
				 "add_new_mailbox: free_storage failed\n"));
	    }	
	    if (is_canceled(cd)) {
		DPRINT(Debug,9, (&Debug, 
				 "add_new_mailbox: Cancel on free_storage\n"));
	    }

	}
	
	goto cleanup;
    }

    if(!same_file) 		
	add_storage(mailbox,storage);

    update_view(mailbox);

    resort_mailbox(mailbox, !same_file);

 cleanup:
    if (cd)
	free_cancel(&cd);	    

    if (reconnect_mode)
	free_reconnect_mode(& reconnect_mode);
   
    return;
}

/* If new folder is freed / moved to new owner,
   then *new_folder == NULL is NULL
   if *new_folder is not modified (set to NULL)
   caller must free *new_folder 

   reads mailbox
*/
struct MailboxView * enter_mailbox(new_folder,parent_page,quotad)
     struct folder_info **new_folder;
     struct menu_context *parent_page;
     struct quota_display *quotad; 
{
    struct MailboxView * mailbox = NULL;
    struct current_storage *storage = NULL;
    enum sessionlock_status  ret;
    int r = 0;
    int was_reconnected;
    RECONNECT_MODE reconnect_mode  = NULL;
    struct cancel_data *cd         = NULL;
    
    struct menu_context *current_page = parent_page;

    if (quotad)
	current_page = quota_display_current_page(quotad);
    
    if ((*new_folder)->was_reconnected) {
	DPRINT(Debug,10,(&Debug, 
			 "enter_mailbox: Mailfile was reconnected: %s\n",
			 (*new_folder)->cur_folder_sys));
    }
    was_reconnected = (*new_folder)->was_reconnected;


    if (quotad) {
	struct mail_quotaroot_list * quota_list = NULL;

	setup_mbx_cancel_message(&cd,mbx_checking_mailbox);

	quota_list = give_folder_quotaroot_l(*new_folder,cd,quotam_normal);

	if (quota_list) {
	    quota_display_add_qlist(quotad,quota_list,
				    (*new_folder)->cur_folder_disp,
				    cd);

	    free_mail_quotaroot_list(&quota_list);	   				    
	}

	if (is_canceled(cd)) {
	    DPRINT(Debug,10,(&Debug, 
			     "enter_mailbox: give_folder_quotaroot_l() canceled\n"));
	    goto cleanup;
	}

	free_cancel(&cd);
    }
    
    ret = sessionlock_folder(*new_folder,SESSIONLOCK_NORMAL,
			     NULL,&reconnect_mode);

    if (reconnect_mode) {
	DPRINT(Debug,10,(&Debug, 
			 "enter_mailbox: reconnect_mode set %p\n",
			 reconnect_mode));
    }

    switch (ret) {
	
    case sessionlock_fail:
	mailbox = NULL;
	goto cleanup;

    case sessionlock_open:
	storage = new_storage(*new_folder);
	*new_folder = NULL;     /* Storage takes care of freeing */

	if (was_reconnected) {
	    r = read_headers(read_hdr_reconnect,storage,current_page,
			     &reconnect_mode
			     );  /* Returns number of mails readed */
	} else {	
	    r= read_headers(read_hdr_normal,storage,
			    current_page,
			    &reconnect_mode
			    ); /* Returns number of mails readed */
	}
	break;
	
    case sessionlock_reconnect:   /* reconnected ??? */
	storage = new_storage(*new_folder);
	*new_folder = NULL;     /* Storage takes care of freeing */

	r = read_headers(read_hdr_reconnect,storage,current_page,
			 &reconnect_mode
			 );  /* Returns number of mails readed */
	break;
    }

    if (!storage) {
	mailbox = NULL;
	goto cleanup;
    }
    
    if (r < 0 || 
	ferror_folder(storage->current_folder,1)) {
	int r;

	
	DPRINT(Debug,1,(&Debug, 
			"error when reading mailfile: %s\n",
			storage->current_folder->cur_folder_sys));
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorWhenReading,
			  "Error when reading: %S"),
		  storage->current_folder->cur_folder_disp);

	setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);
	
	DPRINT(Debug,9, (&Debug, 
			 "enter_mailbox: freeing storage\n"));
	
	 r = free_storage(&storage,cd);   /* Frees folder */

	if (!r) {
		DPRINT(Debug,9, (&Debug, 
				 "enter_mailbox: free_storage failed\n"));
	}	
	if (is_canceled(cd)) {
	    DPRINT(Debug,9, (&Debug, 
			     "enter_mailbox: Cancel on free_storage\n"));
	}
	
	mailbox = NULL;
	goto cleanup;
    }

    mailbox = malloc_mailbox_view(storage);

    update_view(mailbox);
    resort_mailbox(mailbox,0);

 cleanup:
    if (reconnect_mode)
	free_reconnect_mode(& reconnect_mode);
    if (cd)
	free_cancel(&cd);
    
    return mailbox;
}


void newmbox_1(new_folder, mailbox, append, parent_page,was_reconnected,
	       reconnect_ptr, quotad)
     struct folder_info   * new_folder;
     struct MailboxView  ** mailbox;
     int                    append;
     struct menu_context  * parent_page;
     int                    was_reconnected;
     RECONNECT_MODE       * reconnect_ptr;
     struct quota_display * quotad; /* may be NULL */
{
    enum sessionlock_status ret;
    struct current_storage *storage = NULL;

    enum read_hdr_mode read_mode = read_hdr_normal;

    
    /* If we were reading a spool file, we need to
     * truncate the corresponding tempfile.
     */

    int same_file  = 0;
    RECONNECT_MODE reconnect_mode = NULL;
    struct cancel_data *cd        = NULL;
    int r;

    struct menu_context *current_page = parent_page;

    if (quotad)
	current_page = quota_display_current_page(quotad);

    
    if (reconnect_ptr) {
	reconnect_mode = *reconnect_ptr;
	*reconnect_ptr = NULL;
    }

    if (reconnect_mode) {
	DPRINT(Debug,10,(&Debug, 
			 "newmbox_1: reconnect_mode set %p\n",
			 reconnect_mode));
    }
    
    if (*mailbox) {
	int mbc = get_storage_count(*mailbox);
	int i;

	for (i = 0; i < mbc; i++) {
	    struct current_storage *s = get_storage(*mailbox,i);

	    if (!s)
		continue;

	    if (s->current_folder &&
		new_folder == s->current_folder) {
		storage   = s;
		same_file++;
		break;
	    }
	}    
    }

    if (new_folder->was_reconnected) {
	DPRINT(Debug,10,(&Debug, 
			 "newmbox_1: Mailfile was reconnected: %s\n",
			 new_folder->cur_folder_sys));
    }
    if (was_reconnected) {
	DPRINT(Debug,10,(&Debug, 
			 "newmbox_1: Caller set was reconnected\n"));
    } else {
	if (new_folder->was_reconnected) {
	    DPRINT(Debug,10,(&Debug, 
			     "newmbox_1: Mailfile was reconnected: %s\n",
			 new_folder->cur_folder_sys));
	}
	
	was_reconnected = new_folder->was_reconnected;
    }

    if (quotad) {
	struct mail_quotaroot_list * quota_list = NULL;

	setup_mbx_cancel_message(&cd,mbx_checking_mailbox);

	quota_list = give_folder_quotaroot_l(new_folder,cd,quotam_normal);

	if (quota_list) {
	    quota_display_add_qlist(quotad,quota_list,
				    new_folder->cur_folder_disp,cd);

	    free_mail_quotaroot_list(&quota_list);	   				    
	}

	if (is_canceled(cd)) {
	    DPRINT(Debug,10,(&Debug, 
			     "newmbox_1: give_folder_quotaroot_l() canceled\n"));
	    goto cleanup;
	}

	free_cancel(&cd);
    }


    
    if (! same_file) {

	DPRINT(Debug,10,(&Debug, 
			 "newmbox_1: Mailfile is new, opening normally\n"));
	
	storage = new_storage(new_folder);
      
	ret = sessionlock_folder(storage->current_folder,
				 SESSIONLOCK_NORMAL,NULL,
				 &reconnect_mode);
      
    } else {
	/* folder (as opposite to mailbox) can be session locked
	 * only when it is opened. That call can do session locking
	 * for both (folders and mailboxes)
	 */

	ret = sessionlock_folder(storage->current_folder,
				 SESSIONLOCK_TRUNCATE,NULL,
				 &reconnect_mode);
    }

    switch (ret) {
    case  sessionlock_fail:
	goto fail_it;
	break;
    case sessionlock_open:
	if (was_reconnected)
	    read_mode = read_hdr_reconnect;
	else
	    read_mode = read_hdr_normal;
	break;
    case sessionlock_reconnect:
	read_mode = read_hdr_reconnect;
	break;
    }

    DPRINT(Debug,10,(&Debug, 
		     "newmbox_1: Mailfile opened: %s\n",
		     storage->current_folder->cur_folder_sys));

    clear_error();

 reread:
    if (reconnect_mode) {
	DPRINT(Debug,10,(&Debug, 
			 "newmbox_1: reconnect_mode set %p\n",
			 reconnect_mode));
    }
    
    r= read_headers(read_mode,storage, current_page,&reconnect_mode); /* Returns number of mails readed */

    if (ferror_folder(storage->current_folder,1)) {
	int LINES, COLUMNS;

	menu_get_sizes(current_page,&LINES, &COLUMNS);

	DPRINT(Debug,1,(&Debug, 
			"newmbox_1: error when reading mailfile: %s\n",
			storage->current_folder->cur_folder_sys));
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorWhenReading,
			  "Error when reading: %S"),
		  storage->current_folder->cur_folder_disp);
	sleep_message();
	if (*def_ans_yes == want_to ("Error when reading! Reread folder?",
				     *def_ans_yes,LINES-1,1,
				     current_page)) {
	    
	    DPRINT(Debug,1,(&Debug, 
			    "rereading mailfile: %s\n",
			    storage->current_folder->cur_folder_sys));
	    
	    ret = sessionlock_folder(storage->current_folder,
				     SESSIONLOCK_TRUNCATE,NULL,
				     &reconnect_mode);
	    switch(ret) {
	    case  sessionlock_fail:
		goto fail_it;
	    case sessionlock_open:
		if (was_reconnected) {
		    DPRINT(Debug,10,(&Debug, "Was reconnected when truncating?\n"));
		    read_mode = read_hdr_reconnect;
		} else 
		    read_mode = read_hdr_normal;
		goto reread;
	    case sessionlock_reconnect:
		read_mode = read_hdr_reconnect;
		
		goto reread;
	    }
	}
	goto fail_it;
    }
    
   
    if (r < 0) {
	DPRINT(Debug,1,(&Debug, 
			"read_headers() failed when reading mailfile: %s\n",
			storage->current_folder->cur_folder_sys));
	
    fail_it:

	if (!same_file) {
	    int r;

	    setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);

	    DPRINT(Debug,9, (&Debug, 
			 "newmbox_1: freeing storage\n"));
	    r = free_storage(&storage,cd);
	    if (!r) {
		DPRINT(Debug,9, (&Debug, 
				 "newmbox_1: free_storage failed\n"));
	    }	
	    if (cd && is_canceled(cd)) {
		DPRINT(Debug,9, (&Debug, 
				 "newmbox_1: Cancel on free_storage\n"));
	    }

	    DPRINT(Debug,9, (&Debug, 
			     "newmbox_1: failure -- cleanup\n"));
	    goto cleanup;
	}

	DPRINT(Debug,1, (&Debug, 
			 "newmbox_1: failure -- crashing\n"));
	if (cd)
	    free_cancel(&cd);
	if (reconnect_mode)
	    free_reconnect_mode(& reconnect_mode);
    
	
	rm_temps_exit();        /* ????  TODO -- is needed? */
	goto cleanup;    /* not used */
    }

    if(!same_file) {		

	if (!append) {
	    if (*mailbox) {
		int r;

		setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);

		DPRINT(Debug,9, (&Debug, 
				 "newmbox_1: freeing mailbox\n"));
		r = free_mailbox(mailbox,cd);
		if (!r) {
		    DPRINT(Debug,9, (&Debug, 
				     "newmbox_1: free_mailbox failed\n"));
		}	
		if (is_canceled(cd)) {
		    DPRINT(Debug,9, (&Debug, 
				     "newmbox_1: Cancel on free_mailbox\n"));
		}
	    }
	    *mailbox = malloc_mailbox_view(storage);
	} else
	    add_storage(*mailbox,storage);

    }
    
    /* limit mode off -- we are recreating headers 
       We loose the 'Visible' flag 
    */
    set_selected(*mailbox, 0);    
        
    update_view(*mailbox);

    resort_mailbox(*mailbox, append && !same_file);

 cleanup:
    if (cd)
	free_cancel(&cd);
    
    if (reconnect_ptr) {
	*reconnect_ptr = reconnect_mode;
	reconnect_mode = NULL;
    } else if (reconnect_mode)
	 free_reconnect_mode(& reconnect_mode);
    
    return;
}

#if ANSI_C
parse_header_callback parse_header_routine;
#endif
int parse_header_routine(folder,read_state_ptr,current_header,
				parsed_headers)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     struct header_rec *current_header;
     header_list_ptr parsed_headers;
{
    DPRINT(Debug,12,(&Debug,
		     "parse_header_routine: Handling hdr index #%d",
		     current_header->index_number_X));
    if (current_header->env_from[0]) {
	DPRINT(Debug,12,(&Debug,", env from %s",
			 current_header->env_from));
    }
    if (! parsed_headers) {
	DPRINT(Debug,12,(&Debug,", no parsed headers"));
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    current_header->exit_disposition = UNSET;
    /* header_parse_helper may set 
       current_header->exit_disposition = HIDE   */

    /* copy_envelope_folder() may have set current_header->content_length */

    current_header->have_from        = FALSE;

    /* copy_envelope_folder() have set current_header->header_charset */

    header_parse_helper(current_header,parsed_headers);

    return 1;
}

struct counter_data {
    struct menu_context    * page;
    
    int count_x, count_y, lastpercent;
    struct string * truncated_name;
    
    long   previous_bytes;
    int    previous_messages;

    int prompt_factor;
    int skipped_factor;

    
    int tail_x, tail_y;
    time_t   tail_time;
};

enum counter_mode { COUNT_initialize,
		    COUNT_update,
		    COUNT_finalize,
		    COUNT_refresh,
		    COUNT_fail_cleanup
};

const int counter_tail_showtime = 5;   /* Show tail (new mails arrived) message 5 seconds */

static int digits P_((int val));
static int digits(val)
     int val;
{
    int ret = 1;

    while (val >= 10 && ret < 9) {

	val /= 10;
	ret++;
	
    }
    
    return ret;
}

static void reading_updater P_((enum counter_mode mode,
				struct string *name,
				struct counter_data *counter,
				int count, int percent,
				int skipcount, int percent2,
				struct menu_context    * page  /* May be NULL */,
				int num_messages /* maybe -1 */
				));
static void reading_updater(mode,
			    name, counter,count,percent,skipcount,
			    percent2,page,num_messages)
     enum counter_mode mode;
     struct string *name;
     struct counter_data *counter;
     int count; 
     int percent;
     int skipcount; 
     int percent2;
     struct menu_context    * page; /* May be NULL *, if mode != COUNT_initialize */
     int num_messages /* maybe -1 */;   
{
    int initialize = mode == COUNT_initialize;

    int width;
    int LINES, COLUMNS;
    int have_skipped = 0;

    int d1, d2, d3 = 0, d4 = 0,d5 = 0;
    int w;
    
    int text_x, text_y;
    int total = 0;

    int len;

    int recalculate = 0;
    
    if (initialize) {
	/* bzero is defined hdrs/elm_defs.h */
	bzero((void *)counter,sizeof (*counter));
	
	counter->truncated_name = NULL;
	counter->previous_bytes = 0;
	counter->previous_messages = 0;
	counter->page           = page;

	counter->prompt_factor  = 32;
	counter->skipped_factor = 19;
	
    } else if (page) {
	if (counter->page != page) {
	    DPRINT(Debug,10,(&Debug, "reading_updater: page mismatch %p != %p (counter page)\n",
			     page,counter->page));
	}
    } else
	page = counter->page;

    menu_get_sizes(page,&LINES, &COLUMNS);

    width = COLUMNS - 53
	        - ((skipcount >= 10000) ? 1 : 0)
	        - ((count >= 10000) ? 1 : 0);


    

    if (count > 0)
	total += count;
    if (counter->previous_messages > 0) {
	total += counter->previous_messages;
	d4 = digits(counter->previous_messages) +1;
    }
	
    /* Another way */
    
    d1 = digits(count);
    d2 = digits(skipcount);
    d5 = digits(total);

    have_skipped = (skipcount > 0 || percent2 >= 0);

    do {
	if (recalculate) {
	    DPRINT(Debug,10,(&Debug, "reading_updater: recalculate loop #%d\n",
			     recalculate));   
	}
    
	if (num_messages > 0) {
	    d3 = digits(num_messages);

	    /* num_messages do not truncate filename,
	       but it is printed if there is enough
	       space
	    */
	    
	    w = COLUMNS - counter->prompt_factor - 2 * d3 - d4;
	    
	    
	} else 
	    w = COLUMNS - counter->prompt_factor - d1 - d2;
	
	if (have_skipped) {
	    
	    if (w > counter->skipped_factor + d2)
		w -= counter->skipped_factor - d2;
	    else  {
		DPRINT(Debug,10,(&Debug, "reading_updater: skipped text disabled, width=%d d1=%d d2=%d d3=%d d4=%d d5=%d skipped_factor=%d\n",
				 width,d1,d2,d3,d4,d5,counter->skipped_factor));
		
		have_skipped = 0;
	    }
	}
        
	if (w < width) {
	    DPRINT(Debug,15,(&Debug, "reading_updater: name width=%d => %d\n",
			     width,w));
	    width = w;
	}
	
	if (width <= 6) {
	    DPRINT(Debug,10,(&Debug, "reading_updater: width=%d d1=%d d2=%d d3=%d d4=%d d5=%d prompt_factor=%d\n",
			     width,d1,d2,d3,d4,d5,counter->prompt_factor));
	    
	    if (initialize && !recalculate)
		return;
	    
	    menu_MoveCursor(page,counter->count_x,0);
	    menu_CleartoEOLN(page);
	    
	    goto out;
	}
	
	if (COUNT_fail_cleanup == mode) {
	    menu_MoveCursor(page,counter->count_x,0);
	    menu_CleartoEOLN(page);
	    goto out;
	}
	
	if ((len = string_len( counter->truncated_name ? 
			       counter->truncated_name : name ))
	    > width) {
	    int A = width/2 -2;
	    int B = string_len(name);
	    int C = B-A;
	    struct string *D = clip_from_string(name,&C,A);
	    
	    DPRINT(Debug,10,(&Debug,
			     "reading_updater: truncating name width=%d => %d name=%S",
			     len,width,
			     counter->truncated_name ? 
			     counter->truncated_name : name));
	    
	    if (counter->truncated_name)
		free_string(&(counter->truncated_name));
	    
	    counter->truncated_name = 
		format_string(FRM("%.*S...%S"),
			      A,name,D);
	    
	    DPRINT(Debug,10,(&Debug," => %S\n",
			     counter->truncated_name));
	    
	    free_string(&D);
	    initialize++;	    

	}
	    
	if (need_refresh_low_line() || 
	    mode == COUNT_refresh)
	    initialize++;
	
	if (initialize) {
	    int A;
	    
	    menu_PutLineX(page,
			  LINES-1, 0, CATGETS(elm_msg_cat, ElmSet, 
					      ElmReadingInMessage,
					      "Reading in %S, message: ."),
			  counter->truncated_name ? counter->truncated_name : name);
	    
	    menu_GetXYLocation(page,
			       &(counter->count_x),&(counter->count_y)); 
	    counter->count_y--;
	    counter->lastpercent = -1;
	    menu_CleartoEOLN(page);
	    
	    A = counter->count_y -
		string_len(counter->truncated_name ? counter->truncated_name : name)
		-9;
	    
	    if (A > counter->prompt_factor) {
		
		DPRINT(Debug,10,(&Debug,
				 "reading_updater: prompt_factor=%d => %d\n",
				 counter->prompt_factor,A));
		
		counter->prompt_factor = A;
		recalculate++;
		
	    } else
		recalculate = 0;
	    
	} else
	    recalculate = 0;
		    
    } while (recalculate && recalculate < 5);
	    
    if (num_messages > 0 &&
       	COLUMNS >
	counter->count_y +
	d5 + d3 + d4 + 22 + (have_skipped ? d3 : 0 )
	&&
	count     <= total &&
	skipcount <= num_messages) {

	if (counter->previous_messages > 0) {
	    menu_PutLineX(page,
			  counter->count_x, counter->count_y, 
			  FRM("%*d/%d+%d (%02d%%)%s "),
			  d5,count,
			  num_messages,counter->previous_messages,
			  percent,
			  have_skipped ? "," : ""
			  );
	} else {	
	    menu_PutLineX(page,
			  counter->count_x, counter->count_y, 
			  FRM("%*d/%d (%02d%%)%s "),
			  d3,count,num_messages,percent,
			  have_skipped ? "," : ""
			  );
	}

    } else {
	menu_PutLineX(page,
		      counter->count_x, counter->count_y, 
		      FRM("%d (%02d%%)%s "), count,percent,
		      have_skipped ? "," : ""
		      );
	
    }

    menu_GetXYLocation(page,
		       &text_x,&text_y); 

    if (counter->count_x == text_x &&
	num_messages > 0 &&
	(count     > total ||
	 skipcount > num_messages)) {
	menu_CleartoEOLN(page);
    }
    
    if (have_skipped && counter->count_x == text_x) {
	int test_x,test_y;
	
	menu_PutLineX(page,
		      text_x, text_y -1,
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmSkippingInMessage,
			      " %d (%02d%%) skipped "),
		      skipcount,percent2);

	menu_GetXYLocation(page,
			   &test_x,&test_y); 

	if (text_x == test_x && test_y >  text_y) {
	    
	    int A =  test_y - text_y - d2;

	    if (A > counter->skipped_factor) {
		DPRINT(Debug,10,(&Debug,
				 "reading_updater: skipped_factor=%d => %d\n",
				 counter->skipped_factor,A));
		
		counter->skipped_factor = A;
	    }
	}
    }

    if (initialize) {
	menu_GetXYLocation(page,
			   &(counter->tail_x),&(counter->tail_y)); 
	counter->tail_time = 0;
	
    } else if (counter->tail_time) {
	int tail_x, tail_y;
	time_t now = 0;

	menu_GetXYLocation(page,&tail_x,&tail_y);

	if (tail_x == counter->count_x) {
	    if (tail_y > counter->tail_y &&
		counter->tail_x == tail_x) {
		
		menu_CleartoEOLN(page);
		counter->tail_y = tail_y;
		
	    } else if (time(&now) >
		       counter->tail_time + counter_tail_showtime) {
		
		menu_CleartoEOLN(page);
		counter->tail_time = 0;
		counter->tail_y = tail_y;
		counter->tail_x = tail_x;
		
	    } else if (COUNT_finalize == mode) {
		
		if (now < counter->tail_time + counter_tail_showtime) {

		    int w = counter->tail_time + counter_tail_showtime - now;
		    
		    error_sleep(w);
		}

		menu_CleartoEOLN(page);
		counter->tail_time = 0;
		counter->tail_y = tail_y;
		counter->tail_x = tail_x;		
	    }
	    
	} else {
	    counter->tail_time = 0;
	    counter->tail_y = tail_y;
	    counter->tail_x = tail_x;
	}
	    
    } else
	menu_GetXYLocation(page,&(counter->tail_x),&(counter->tail_y)); 

 out:
    if (COUNT_finalize == mode ||
	COUNT_fail_cleanup == mode) {
	/** move the cursor to the beginning of the current line **/
	menu_Writechar(page,'\r');

	if (counter->truncated_name)
	    free_string(&(counter->truncated_name));
    }
    FlushBuffer();
}


void parse_body_line_helper(current_header,buffer,len)
     struct header_rec * current_header;
     const char        * buffer;
     const int           len;
{
    DPRINT(Debug,45,(&Debug, "parse_body_line_helper: len=%d\n",len));
    
    if (buffer[len]) {
	mime_panic(__FILE__,__LINE__,"parse_body_line_helper",
		   "Terminating \0 is missing");

    }

    switch (buffer[0]) {
	
    case 'F':
	
	if (first_word(buffer,"From ")) {
	    if (!current_header->have_from) {
		current_header->have_from = 1;
		DPRINT(Debug,10,(&Debug, 
				 "-- Message have unescaped From \n"));
	    }
	}
	
	break;
	
    case '[':
	
	if (mime_body_keywords && first_word(buffer, START_ENCODE)) {
	    current_header->encrypted = 1;
	    DPRINT(Debug,10,(&Debug, "-- encrypted: %s\n",buffer));
	    
	    /* Text/plain can be converted to application/X-ELM-encode
	     */
	    if (get_major_type_code(current_header->mime_rec.TYPE) == 
		MIME_TYPE_TEXT && 
		(istrcmp(get_subtype_name(current_header->
					  mime_rec.TYPE),"plain")==0)) {
		current_header->mime_rec.TYPE = 
		    give_media_type2(MIME_TYPE_APPLICATION,
				     "X-ELM-encode",0);
		if (!current_header->mime_rec.TYPE)
		    mime_panic(__FILE__,__LINE__,"parse_body_line_helper",
			       "application/X-ELM-encode is not known");
		
		/* Keep current_header->mime_rec.type_opts ! */
	    }
	    DPRINT(Debug,10,(&Debug, 
			     "-- encrypted, %d/%s\n",
			     get_major_type_code(current_header->
						 mime_rec.TYPE),
			     get_subtype_name(current_header->mime_rec.TYPE)));
	    
	}
	
	break;
	
    case '-':
	
	if (use_PGP) {
	    if (strncmp(buffer, "-----BEGIN PGP", 14) == 0) {
		if (strncmp(buffer + 15, "PUBLIC", 6) == 0) {
		    current_header->pgp |= PGP_PUBLIC_KEY;
		    DPRINT(Debug,10,(&Debug, "-- PGP PUBLIC KEY: %s\n",
				     buffer));
		} else {
		    
		    /* Text/plain can be converted to 
		     * application/pgp.  There will a possible
		     * loss of surrounding text, but there is no
		     * way to get around this without having the
		     * sender use the proper MIME type.  
		     */
		    
		    if (get_major_type_code(current_header->mime_rec.TYPE) 
			== MIME_TYPE_TEXT && 
			istrcmp(get_subtype_name(current_header->mime_rec.TYPE),
				"plain")==0) {
			current_header->mime_rec.TYPE = 
			    give_media_type2(MIME_TYPE_APPLICATION,"pgp",1);
			/* Keep current_header->mime_rec.type_opts ! */
		    }
			
		    if (strncmp(buffer + 15, "SIG", 3) == 0) {
			const char *mp;
			current_header->pgp |= PGP_SIGNED_MESSAGE;
			DPRINT(Debug,10,(&Debug, "-- PGP signed: %s\n",
					 buffer));
			
			mp = get_mime_param_compat(current_header->
						   mime_rec.TYPE_opts,
						   "x-action");
			if (!mp) {
			    mime_params_add_compat(& (current_header->
						      mime_rec.TYPE_opts),
						   "x-action", "sign");
			    
			}
			DPRINT(Debug,10,(&Debug, 
					 "-- PGP signed, %s/%s; \n",
					 get_major_type_name(current_header->
							     mime_rec.TYPE),
					 get_subtype_name(current_header->
							  mime_rec.TYPE)));
			
		    } else {
			const char *mp;
			
			current_header->pgp |= PGP_MESSAGE;
			
			DPRINT(Debug,10,(&Debug, 
					 "-- PGP message: %s\n",
					 buffer));
			
			mp = get_mime_param_compat(current_header->
						   mime_rec.TYPE_opts,
						   "x-action");
			if (!mp) {
			    mime_params_add_compat(& (current_header->
						      mime_rec.TYPE_opts),
						   "x-action", "encrypt");
			}
			DPRINT(Debug,10,(&Debug, 
					 "-- PGP message, %s/%s\n",
					 get_major_type_name(current_header->
							     mime_rec.TYPE),
					 get_subtype_name(current_header->
							  mime_rec.TYPE)));
			
		    } 
		}
	    }
	}
	
	break;
    }   
}


#if ANSI_C
parse_body_callback parse_body_routine;
#endif
int parse_body_routine(current_folder,read_state_ptr,current_header,
		       parsed_headers,counter
		       )
     struct folder_info *current_folder;
     READ_STATE read_state_ptr;
     struct header_rec *current_header;
     header_list_ptr parsed_headers;
     struct counter_data *counter;
{
    int ret = 0;
    int content_length_found = current_header->content_length > -1L;

    long content_start = -1L;
    long content_remaining = -1L;
    long tmp;

    char * buffer = NULL;
    int len = 0;

    int last_empty_line = 0;

    enum copy_env_end_status endstat = copy_env_end_failure;
    long newbytes = 0;
    int newmails  = 0;
    int num_messages = -1;   /* -1 if not available (local mail) */

    if (content_length_found)
	content_remaining = current_header->content_length;
    else
	content_remaining = -1;
    content_start     = copy_fbytes_folder(current_folder,
					   read_state_ptr);

    /* If IMAP/POP mailbox */   
    num_messages = copy_num_messages_folder(current_folder);
    
    DPRINT(Debug,10,(&Debug,
		     "parse_body_routine: reading body"));
    if (content_remaining >= 0L) {
	DPRINT(Debug,10,(&Debug,", content_remaining %ld",
			 content_remaining));
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    

    /* Notice that mark_keep_folder() also uses mime_rec.offset */
    start_body_helper(current_header,content_start,parsed_headers);

 reset_body:

    while (copy_body_folder(current_folder,read_state_ptr,
			    &buffer,&len,&content_remaining)) {
	int skipcount;
	long skipbytes;
	long f = copy_fbytes_folder(current_folder,
				    read_state_ptr);
	int percent;	
	long total_size =
	    current_folder->mailfile_size;

	if (counter)
	    total_size += counter->previous_bytes;

	last_empty_line =   len == 1 && '\n' == buffer[0];

	copy_skipcount_folder(current_folder,read_state_ptr,
			      &skipcount,&skipbytes);

	f += skipbytes;

	/* calculation with integers overflow on big folders! */
	percent = (int)(f * 100.0 / total_size);
	
	if (counter &&
	    counter->lastpercent / readdatapercentinc != 
	    percent / readdatapercentinc) {

	    if (skipbytes > 0 || skipcount > 0) {
		/* Show skip percent without looking previous messages */
		
		int percent2 = (int)(skipbytes * 100.0 /
				     current_folder->mailfile_size);
		
		reading_updater(COUNT_update,
				current_folder->cur_folder_disp,counter,
				current_header->index_number_X,
				percent,skipcount,percent2,NULL,
				num_messages);
		
	    } else
		reading_updater(COUNT_update,
				current_folder->cur_folder_disp,counter,
				current_header->index_number_X,
				percent,-1,-1,NULL,
				num_messages);
	    
	    counter->lastpercent = percent;
	    
	    DPRINT(Debug,13,(&Debug, "** fbytes=%ld,size=%ld (total_size=%ld) -> %d%%\n",
			     f,current_folder->mailfile_size,
			     total_size,percent));		
	}
	
	parse_body_line_helper(current_header,buffer,len);
		
	if (buffer)
	    free(buffer);
    }
    
    current_header->lines =  copy_lines_folder(current_folder,
					       read_state_ptr);
    tmp = current_header->content_length;
    if (content_start >= 0L)
	current_header->content_length = 
	    copy_fbytes_folder(current_folder,read_state_ptr) - 
	    content_start;
    else
	current_header->content_length = 0;
    DPRINT(Debug,12,(&Debug, "-- content-length=%d\n",
		     current_header->content_length));

    /* Work around message
          Message contains bare newlines
       from some IMAP servers. Do not include terminating
       empty line (only LF on line) from end of mail,
       which is part of mailbox format.

       When message is downloaded from IMAP or POP servers
       there are all lines terminated with CR LF. However
       ending line which is part of mailbox format is ended
       by LF  (added by real_end_we_local()).


       Possible real empty lines from IMAP and POP servers
       are ended with CR LF so that do not effect them.
    */

    if (!have_MMDF &&
	last_empty_line) {

	if (current_header->content_length < 1)
	    mime_panic(__FILE__,__LINE__,"parse_body_routine",
		       "content-length fixup error");

	current_header->content_length --;

	DPRINT(Debug,12,(&Debug, "-- FIXUP content-length=%d\n",
			 current_header->content_length));

    }

    if (tmp >= 0 && current_header->content_length != tmp) {
	DPRINT(Debug,10,(&Debug, 
			 "  DIFFERS from header value (%ld)\n",
			 tmp));
    }

    endstat = copy_envelope_end_folder(current_folder,read_state_ptr,
				       &newbytes,
				       &newmails);

    switch (endstat) {
    case copy_env_end_mismatch:
	DPRINT(Debug,10,(&Debug,
			 "parse_body_routine: copy ended not on end of folder"));
	if (content_remaining >= 0L) {
	    DPRINT(Debug,10,(&Debug,", content_remaining %ld",
			     content_remaining));
	}
	DPRINT(Debug,10,(&Debug,"\n"));
	
	if (content_length_found) {
	    content_length_found = 0;
	    
	    if (copy_envelope_reset_body(current_folder,read_state_ptr,
					 &content_remaining))
		goto reset_body;
	    
	}
	/* FALLTHRU */
    case copy_env_end_failure:

	
	DPRINT(Debug,10,(&Debug, "-- Message parsing FAILED.\n"));
	ret = 0;
	goto clean;
	
    case  copy_env_end_newmail:   /* IMAP mailbox got new mail when parsing */
	DPRINT(Debug,10,(&Debug,
			 "parse_body_routine: Got new mail"));
	if (newbytes  >= 0L) {
	    DPRINT(Debug,10,(&Debug,", %ld new bytes",
			     newbytes));
	}
	if (newmails >= 0) {
	    DPRINT(Debug,10,(&Debug,", %d new mails",
			     newmails));
	}	
	DPRINT(Debug,10,(&Debug,"\n"));

	counter->lastpercent = 0;   /* percent counter is now wrong */
	
	if (counter->count_x == counter->tail_x) {
	    int LINES, COLUMNS;
	    
	    menu_get_sizes(default_context,&LINES, &COLUMNS);

	    if (counter->count_x < LINES &&
		COLUMNS - counter->tail_y > 17) {
		int left = COLUMNS - counter->tail_y;

		struct string * msg = NULL;
		struct string * D   = NULL;
		int pos = 0;
		int visible_len = 0;

		if (newmails > 1) 
		    msg = format_string(CATGETS(elm_msg_cat, ElmSet, 
						ElmXMsgsArrived,
						", %d msgs arrived"),
					newmails);
		else
		    msg = format_string(CATGETS(elm_msg_cat, ElmSet, 
						ElmNewMsgArrived,
						", new msg arrived"));


		D = curses_printable_clip(msg,&pos,left,&visible_len,
					  left);

		if (D) {
		    PutLineX(counter->tail_x, counter->tail_y,
			     FRM("%S"),D);
		    CleartoEOLN();
		    time(& (counter->tail_time));
		    
		    free_string(&D);
		}
		free_string(&msg);
		
	    }
	}

	/* FALLTHRU */
    case copy_env_end_match:
	DPRINT(Debug,15,(&Debug, "-- Body parsed.\n"));
	current_header->body_parsed = 1;
	ret = 1;
	break;
    }

 clean:
    DPRINT(Debug,10,(&Debug,
		     "parse_body_routine=%d\n",ret));
    return ret;
}

void resort_mailbox(mailbox, add_new_only)
     struct MailboxView *mailbox;
     int add_new_only;
{
    int current;
    int count   = get_message_count(mailbox);

    /* Sort folder *before* we establish the current message, so that
     * the current message is based on the post-sort order.
     */
    
    sort_mailbox(count, 1, mailbox);         
    
    /* Now lets figure what the current message should be.  If we are
     * only reading in newly added messages from a mailfile that
     * already had some messages, current should remain the same.  If
     * we have a folder of no messages, current should be zero.
     * Otherwise, if we have point_to_new on then the current message
     * is the first message of status NEW if there is one.  Otherwise,
     * if we have point_to_flagged on then the current message is the
     * first flagged message if there is one.  If we don't have
     * point_to_new or point_to_flagged on or if there are no messages
     * of of status NEW OR flagged messages, then the current message
     * is the first message.
     */

    current = get_current(mailbox);
    if(!(add_new_only && current != 0)) {
	if(count == 0)
	    current = 0;
	else {
	    current = 1;

	    if (point_to_new) {
		int another_count;

		for(another_count = 0; another_count < count; 
		    another_count++) {

		    if (ison_status_message(mailbox,another_count,
					    status_basic,NEW)) {
			current = another_count+1;

			DPRINT(Debug,10,(&Debug, 
					 "resort_mailbox: point_to_new: found current = %d\n",
					 current));

			goto found_new;
		    }
		}

		for(another_count = 0; another_count < count; 
		    another_count++) {

		    if (ison_status_message(mailbox,another_count,
					    status_basic,UNREAD)) {
			current = another_count+1;

			DPRINT(Debug,10,(&Debug, 
					 "resort_mailbox: point_to_new: found unread current = %d\n",
					 current));


			goto found_new;
		    }
		}

		DPRINT(Debug,10,(&Debug, 
				 "resort_mailbox: point_to_new: not found\n"));

	    }

	    if (point_to_flagged) {
		int another_count;
		
		for(another_count = 0; another_count < count; 
		    another_count++) {
		    
		    if (ison_status_message(mailbox,another_count,status_1,
					    S1_FLAGGED)) {
			current = another_count+1;

			DPRINT(Debug,10,(&Debug, 
					 "resort_mailbox: point_to_flagged: found current = %d\n",
					 current));


			goto found_new;
		    }
		}

		DPRINT(Debug,10,(&Debug, 
				 "resort_mailbox: point_to_flagged: not found\n"));

	    }

	    switch (give_dt_sort_as_int(&sortby)) {
	    case SENT_DATE:
	    case RECEIVED_DATE:
	    case THREAD:
	    case MAILBOX_ORDER:
		current = count;
	    }

	    DPRINT(Debug,10,(&Debug, 
			     "resort_mailbox: current = %d\n",
			     current));

	    found_new: ;

	}

	set_current(mailbox,current);

    }
}

static void updater_helper P_((struct folder_info  * current_folder,
			       READ_STATE            read_state_ptr,
			       struct counter_data * counter,
			       int                   count,
			       int                   refresh
			       ));
static void updater_helper( current_folder,read_state_ptr,
			    counter,count, refresh)
     struct folder_info  * current_folder;
     READ_STATE            read_state_ptr;
     struct counter_data * counter;
     int                   count;
     int                   refresh;
{
    int skipcount;
    long skipbytes;
    int percent = 0;
    
    long f = copy_fbytes_folder(current_folder,read_state_ptr);

    long total_size =
	current_folder->mailfile_size +
	counter->previous_bytes;

    /* -1 if not available (local mail) */
    int num_messages =  copy_num_messages_folder(current_folder);
       
    copy_skipcount_folder(current_folder,read_state_ptr,&skipcount,&skipbytes);
		    
    f += skipbytes;

    /* calculation with integers overflow on big folders! */
    
    if (total_size > 0)
	percent = (int)(f * 100.0 / total_size);
		    
    if (skipbytes > 0 || skipcount > 0) {
	/* Show skip percent without looking previous messages */
	int percent2 = 0;

	if (current_folder-> mailfile_size > 0)
	    percent2 = (int)(skipbytes * 100.0 /
			     current_folder-> mailfile_size);
	
	reading_updater(refresh ? COUNT_refresh : COUNT_update,
			current_folder->cur_folder_disp,
			counter,
			count,percent,skipcount,percent2,NULL,
			num_messages);
			
    } else
	reading_updater(refresh ? COUNT_refresh : COUNT_update,
			current_folder->cur_folder_disp,
			counter,
			count,percent,-1,-1,NULL,
			num_messages);
		    
    counter->lastpercent = percent;
}

static int read_headers(read_mode, storage, page, reconnect_mode_ptr)
     enum read_hdr_mode       read_mode;
     struct current_storage * storage;    
     struct menu_context    * page;
     RECONNECT_MODE         * reconnect_mode_ptr /* Hack for SESSIONLOCK_TRUNCATE */;
{
    /** Reads the headers into the headers[] array and leaves the
	file rewound for further I/O requests.   If the file being
	read is a mail spool file (ie incoming) then it is copied to
	a temp file and closed, to allow more mail to arrive during 
	the elm session.  If 'add_new_only' is set, the program will copy
	the status flags from the previous data structure to the new 
	one if possible and only read in newly added messages.
    **/

    int count = 0;    
    int ret = -1;

    enum prepare_mode prepare_folder_mode = PREPARE_NORMAL;

    struct counter_data COUNTER;

    struct read_folder_state * read_state_ptr     = NULL;
    struct reconnect_mode    * reconnect_mode     = NULL;
    const char *msg;
    int LINES, COLUMNS;
    int delay_redraw = 0;

    struct previous_data * previous_headers = NULL;
    size_t                 previous_count = 0;
    long                   previous_bytes = 0;

    int                    previous_messages = 0;
    
    int num_messages = -1;
    
    menu_get_sizes(page,&LINES, &COLUMNS);

    DPRINT(Debug,10,(&Debug,
		     "read_headers: read_mode=%d",
		     read_mode));
    switch (read_mode) {
    case read_hdr_normal:    DPRINT(Debug,10,(&Debug," read_hdr_normal"));    prepare_folder_mode = PREPARE_NORMAL;    break;
    case read_hdr_new_only:  DPRINT(Debug,10,(&Debug," read_hdr_new_only"));  prepare_folder_mode = PREPARE_NEW_ONLY;  break;
    case read_hdr_reconnect: DPRINT(Debug,10,(&Debug," read_hdr_reconnect")); prepare_folder_mode = PREPARE_RECONNECT; break;
    }

    if (reconnect_mode_ptr && *reconnect_mode_ptr) {
	reconnect_mode = *reconnect_mode_ptr;
	*reconnect_mode_ptr = NULL;
	
	DPRINT(Debug,10,(&Debug, "; *reconnect_mode_ptr=%p",
			 *reconnect_mode_ptr));   
    }

    DPRINT(Debug,10,(&Debug,"\n"));
    
    
    if (!prepare_read_folder(storage->current_folder,
			     prepare_folder_mode,
			     &read_state_ptr,
			     &reconnect_mode)) {
	sleep_message();
	ret = -1;  /* FAILURE */
	goto finish;
    }

    if (reconnect_mode) {


	DPRINT(Debug,7,(&Debug, 
			"read_headers: Resetting storage message count %d\n",
			storage->message_count));

	if (storage->message_count > 0) {
	    size_t i;
	    size_t malloc_count = storage->message_count;
	    
	    previous_count = 0;
		
	    previous_headers =
		safe_calloc(malloc_count,
			    sizeof previous_headers[0]);

	    for (i = 0; i < malloc_count; i++) {
		previous_headers[previous_count].rec = NULL;
		previous_headers[previous_count].found_index = -1;  /* Not found */

		if (storage->headers[i]) {

		    if (ison(storage->headers[i]->status1,S1_NODATA)) {

			header_free(& (storage->headers[i]));
		    } else {		   		    
			if (storage->headers[i]->offset >= 0L &&
			    storage->headers[i]->mime_rec.begin_offset >
			    storage->headers[i]->offset)
			    previous_bytes +=
				(storage->headers[i]->mime_rec.begin_offset - storage->headers[i]->offset);
			
			if (storage->headers[i]->content_length > 0L)
			    previous_bytes +=  storage->headers[i]->content_length;
			
			/* Swap header */
			previous_headers[previous_count].rec = storage->headers[i];
			storage->headers[i] = NULL;
			previous_count++;

			
		    }
		}
	    }

	    DPRINT(Debug,10,(&Debug,
			     "read_headers: previous_count=%zu previous_bytes=%ld\n",
			     previous_count,previous_bytes));	    

	    if (previous_count < (size_t)INT_MAX)
		previous_messages = previous_count;
	    else {
		DPRINT(Debug,10,(&Debug,
				 "read_headers: Message count overflow\n"));
		previous_messages = INT_MAX;
	    }
	    
	}
	
	storage->message_count = 0;
	
	/* storage->headers vector is still allocated,
	   theefore storage->max_headers is not reset 
	*/
	
    } else {
	switch (read_mode) {
	case read_hdr_new_only:
	    count = storage->message_count;		/* next available  */
	    break;
	case read_hdr_normal:
	    break;
	case read_hdr_reconnect:   /* should not happen when reconnect_mode == NULL */
	    DPRINT(Debug,10,(&Debug,
			     "read_headers: reconnect mode not available\n"));
	    break;
	}
    }
    
    ClearLine(LINES-2);
    ClearLine(LINES-1);

    /* If IMAP/POP mailbox */   
    num_messages = copy_num_messages_folder(storage->current_folder);
    
    /* Initialize */
    reading_updater(COUNT_initialize,
		    storage->current_folder->cur_folder_disp,&COUNTER,
		    count,0,-1,-1,page,num_messages);

    
    
    if ((msg = is_forwarded_folder(storage->current_folder,
				   read_state_ptr)) != NULL) {
	lib_transient(CATGETS(elm_msg_cat, ElmSet, ElmMailBeingForwardTo,
			      "Mail being forwarded to %s"), 
		      msg);	
    } else {
	enum copy_env_status status = copy_env_eof;
	struct header_rec * work = NULL;
	int refresh = 0;
	int read_folder = 1;
	

	while (read_folder) {
	    
	    if (raw_off_called()) {
		DPRINT(Debug,10,(&Debug, "read_headers: Need  redraw .... \n"));

		/* Reset raw_off_called() */
		ClearScreen(0  /* No interrupt */);

		/* Call refresh routines of children */
		menu_redraw_children(page);
    
		delay_redraw++;
		refresh = 1;
	    }

	    if (!work)
		work = malloc_header_rec();
	    else
		header_clear(work);
	    work->index_number_X = count+1;
	    
	    status =
		copy_envelope_folder(storage->current_folder,read_state_ptr,
				     work,
				     parse_header_routine,
				     parse_body_routine,
				     &COUNTER
				     ) ;

	    switch (status) {
	    case copy_env_ok: 
	    case copy_env_no_data: 

		if (reconnect_mode) {
		    size_t                 previous_index  = 0;     /* Undetermined */
		    struct previous_data * previous_header = NULL;  /* Not found    */

		    previous_header = search_previous_header(storage->current_folder,
							     work,reconnect_mode,
							     previous_headers,
							     previous_count,
							     &previous_index);
		    
		    if (previous_header) {
			if (previous_index >= previous_count ||
			    previous_header != &(previous_headers[previous_index]))
			    panic("MBX PANIC",__FILE__,__LINE__,"read_headers",
				  "Bad previous_index or previous_header",0);
		       			
			DPRINT(Debug,10,(&Debug, "read_headers: message #%d: previous index %lu",
					 count,(unsigned long)previous_index));
			if (previous_header->found_index >= 0) {
			    DPRINT(Debug,10,(&Debug, " ... already found for message #%d\n",
					     previous_headers[previous_index].found_index));
			} else {
			    DPRINT(Debug,10,(&Debug, "\n"));
			    previous_header->found_index = count;

			    if (previous_header->rec) {
			    
				if (copy_env_ok == status) {
				    
				    if (previous_header->rec->offset >= 0L &&
					previous_header->rec->mime_rec.begin_offset >
					previous_header->rec->offset)
					previous_bytes -=
					    (previous_header->rec->mime_rec.begin_offset -
					     previous_header->rec->offset);
				    
				    if (previous_header->rec->content_length > 0) 
					previous_bytes -= previous_header->rec->content_length;

				    previous_messages--;
				    				    
				    if (COUNTER.previous_bytes != previous_bytes) {
					DPRINT(Debug,10,(&Debug,"read_headers:  previous_bytes=%ld",
							 previous_bytes));
					
					if (previous_bytes < 0) {
					    DPRINT(Debug,10,(&Debug," UNDERFLOW"));
					    previous_bytes = 0;
					}
					DPRINT(Debug,10,(&Debug,"\n"));
				    }

				    DPRINT(Debug,10,(&Debug,"read_headers: previous_messages=%d",
						     previous_messages));

				    if (previous_messages < 0) {
					DPRINT(Debug,10,(&Debug," UNDERFLOW"));
					previous_messages = 0;
				    }
				    DPRINT(Debug,10,(&Debug,"\n"));
				}

				if (previous_header->rec->status_chgd) {
				
				    DPRINT(Debug,10,(&Debug, "read_headers: message #%d: previous was changed status",
						     count));

#define X(field,F,T)                if (ison(previous_header->rec->field,F) && isoff(work->field,F)) { \
					setit(work->field,F);		\
					DPRINT(Debug,10,(&Debug,"; set %s",T)); \
					work->status_chgd = 1;            \
				    }					\
				    else if (isoff(previous_header->rec->status,F) && ison(work->status,F)) { \
					clearit(work->status,F);		\
					DPRINT(Debug,10,(&Debug,"; clear %s",T)); \
					work->status_chgd = 1;            \
				    }
				    
				    X(status,UNREAD,"UNREAD");
				    X(status,DELETED,"DELETED");
				    X(status,NEW,"NEW");
				    X(status,TAGGED,"TAGGED");
				    X(status,VISIBLE,"VISIBLE");
				    X(status,REPLIED_TO, "REPLIED_TO");
				    X(status1,S1_FLAGGED,"S1_FLAGGED");

				    if (! work->status_chgd) {
					DPRINT(Debug,10,(&Debug, ", no changes to current"));
				    }
				    
				    DPRINT(Debug,10,(&Debug, "\n"));
#undef X
				}
			    }			
			}
		    }
		    
		    COUNTER.previous_bytes     = previous_bytes;
		    COUNTER.previous_messages  = previous_messages;
		}
		
		if (count % readmsginc == 0 || refresh) {

		    updater_helper(storage->current_folder,
				   read_state_ptr,
				   &COUNTER,
				   count,refresh);
		    refresh = 0;
		}

		
		if (copy_env_ok == status) {
		    DPRINT(Debug,10,(&Debug, "-- Message %d parsed.\n",count));
		} else {
		    work->status1 |= S1_NODATA;
		    DPRINT(Debug,10,(&Debug, "-- Message %d not handled.\n",count));
		}
				
		/** allocate new header pointers, if needed... **/
		realloc_headers(storage,count);
		
		if (storage->max_headers <= count)
		    panic("MBX PANIC",__FILE__,__LINE__,"read_headers",
			  "Bad storage->max_headers",0);

		if (storage->headers[count])
		    free_header_rec(& (storage->headers[count]));
		storage->headers[count] = work;
		work = NULL;		
		storage->headers[count]->index_number_X = count+1;
		
		count++;		
	    
		break;

	    case copy_env_format:
		DPRINT(Debug,10,(&Debug,
				 "read_headers: message %d -- copy_envelope_folder returns copy_env_format\n",
				 count));
		read_folder = 0;
		break;
		
	    case copy_env_eof:
		DPRINT(Debug,10,(&Debug,
				 "read_headers: message %d -- copy_envelope_folder returns copy_env_eof\n",
				 count));

		read_folder = 0;
		break;
	    }
	}
	if (work)
	    free_header_rec( & work);

	if (copy_env_format == status) {
	    sleep_message();
	    end_read_folder(storage->current_folder,&read_state_ptr,
			    &reconnect_mode,
			    1);


	    goto failure;
	}
    }

    if (previous_headers) {
	int i;

	enum copy_env_status status = copy_env_eof;  
	int refresh = 0;
	int read_folder = 1;
	int y = 0;  /* for counter refresh */
	struct header_rec * work = NULL;

	num_messages =  copy_num_messages_folder(storage->current_folder);	
		
	DPRINT(Debug,10,(&Debug,
			 "read_headers: Have %d previous headers, starting count %d\n",
			 previous_count,count));
			
	for (i = 0; i < previous_count && read_folder; i++) {
	    int current_found = 0;
	    int idx = previous_headers[i].found_index;
	    struct header_rec *replaced_entry = NULL;

	    if (raw_off_called()) {
		DPRINT(Debug,10,(&Debug,
				 "read_headers: Need  redraw .... \n"));

		/* Reset raw_off_called() */
		ClearScreen(0  /* No interrupt */);

		/* Call refresh routines of children */
		menu_redraw_children(page);
    
		delay_redraw++;
		refresh = 1;
	    }
	    
	    if (idx >= 0) {
		if (idx >= count ||
		    ! storage->headers[idx])
		    panic("MBX PANIC",__FILE__,__LINE__,"read_headers",
			  "Bad found_index",0);
		
		DPRINT(Debug,10,(&Debug,
				 "read_headers: previous message #%d: found from current #%d, hdr index #%d",
				 i,idx,
				 storage->headers[idx]->index_number_X));
		
		if (0 != (storage->headers[idx]->status1 & S1_NODATA)) {
		    DPRINT(Debug,10,(&Debug, " ... no data\n"));
		    replaced_entry = storage->headers[idx];
		} else if ( ! storage->headers[idx]->body_parsed) {
		    DPRINT(Debug,10,(&Debug, " ... body not parsed\n"));
		    replaced_entry = storage->headers[idx];
		} else {
		    DPRINT(Debug,10,(&Debug, "\n"));
		    current_found = 1;		    
		}
	    }

	    if (!current_found) {

		if (replaced_entry) {
		    if (work)
			free_header_rec(& work);
		    work = replaced_entry;

		} else if (previous_headers[i].rec &&
			   ison(previous_headers[i].rec->status,DELETED)) {

		    /* Allow re$yncronize to delete messages !! */
		    
		    DPRINT(Debug,10,(&Debug,
				     "read_headers: previous message #%d: skipping, was hdr index #%d: marked as DELETED\n",
				     i,previous_headers[i].rec->index_number_X));
		    
		    continue;
		} else {
		    if (!work)
			work = malloc_header_rec();
		    else
			header_clear(work);
		    work->index_number_X = count+1;
		    work->status1 = S1_RECONNECT_LOST;
		}
		
		DPRINT(Debug,10,(&Debug,
				 "read_headers: previous message #%d: reading",
				 i));
		if (previous_headers[i].rec) {
		    DPRINT(Debug,10,(&Debug, ", was hdr index #%d",
				     previous_headers[i].rec->index_number_X));	  
		}
		DPRINT(Debug,10,(&Debug, ", current hdr index #%d\n",
				 work->index_number_X));
		
		/* Copy old mail */
		
		status =
		    copy_previous_mail(storage->current_folder,read_state_ptr,
				       work,
				       parse_header_routine,
				       parse_body_routine,
				       &COUNTER,
				       reconnect_mode,
				       &previous_headers[i]);

		 if (y % readmsginc == 0 || refresh) {
		     
		     /* Not optimal */
		     
		     updater_helper(storage->current_folder,
				    read_state_ptr,
				    &COUNTER,
				    count,refresh);
		     refresh = 0;
		 }

		 switch (status) {
		 case copy_env_ok: 
		     

		     if (replaced_entry) {

			 replaced_entry->index_number_X = idx+1;

		     } else {
			 /** allocate new header pointers, if needed... **/
			 realloc_headers(storage,count);
			 
			 if (storage->max_headers <= count)
			     panic("MBX PANIC",__FILE__,__LINE__,"read_headers",
				   "Bad storage->max_headers",0);
			 
			 if (storage->headers[count])
			     free_header_rec(& (storage->headers[count]));
			 storage->headers[count] = work;
			 work = NULL;		
			 storage->headers[count]->index_number_X = count+1;
			 
			 count++;					 
		     }
		     
		     		     
		     DPRINT(Debug,10,(&Debug, "read_headers: previous message #%d parsed",
				      i));

		     if (0) {
		     case copy_env_no_data: 
			 DPRINT(Debug,10,(&Debug, "read_headers: previous message #%d: not handled",
					  i));
		     }

		     if (0) {
		     case copy_env_format:
			 DPRINT(Debug,10,(&Debug, "read_headers: previous message #%d: copy_previous_mail returns copy_env_format",
					  i));
			 read_folder = 0;
		     }

		     if (0) {
		     case copy_env_eof:
			 DPRINT(Debug,10,(&Debug, "read_headers: previous message #%d: copy_previous_mail returns copy_env_eof",
					  i));
			 read_folder = 0;
		     }
		     
		     
		     if (previous_headers[i].rec) {
			 DPRINT(Debug,10,(&Debug, ", was hdr index #%d",
					  previous_headers[i].rec->index_number_X));	  
		     }
		     if (replaced_entry) {
			 DPRINT(Debug,10,(&Debug, ", current hdr index #%d",
					  replaced_entry->index_number_X));
		     }
		     DPRINT(Debug,10,(&Debug, "\n"));

		     y++;
		     
		     
		     break;		     
		 }

		 if (work == replaced_entry)
		     work = NULL;

	    }
	}

	if (work)
	    free_header_rec( & work);

	if (copy_env_format == status) {
	    sleep_message();
	    end_read_folder(storage->current_folder,&read_state_ptr,
			    &reconnect_mode,
			    1);
	    
	    goto failure;
	}       
    }	

    
    {
	/* Final counter */
	int skipcount;
	long skipbytes;
	int percent;
	
	long f = copy_fbytes_folder(storage->current_folder,
				    read_state_ptr);
	
	long total_size =
	    storage->current_folder->mailfile_size +
	    COUNTER.previous_bytes;
	
	copy_skipcount_folder(storage->current_folder,read_state_ptr,
			      &skipcount,&skipbytes);
	
	f += skipbytes;
	
	if (total_size >0)
	    percent = (int)(f * 100.0 / total_size);
	else 
	    percent = 100;
	/* calculation with integers overflow on big folders! */
	
	/* Should be 100% now ... */

	if (skipbytes > 0 || skipcount > 0) {
	    int percent2 = 100;

	    if (storage->current_folder->mailfile_size > 0)
		percent2 = (int)(skipbytes * 100.0 /
				 storage->current_folder->mailfile_size);
	    
	    reading_updater(COUNT_finalize,
			    storage->current_folder->cur_folder_disp,&COUNTER,
			    count,percent,skipcount,percent2,page,
			    num_messages
			    );

	} else
	    reading_updater(COUNT_finalize,
			    storage->current_folder->cur_folder_disp,&COUNTER,
			    count,percent,-1,-1,page,
			    num_messages
			    );
	
	COUNTER.lastpercent = percent;
    }

    
    if (!end_read_folder(storage->current_folder,&read_state_ptr,
			 &reconnect_mode,0)) {
	sleep_message();

	goto failure;
    } 

    /* We need set per storage variable message_count */
    DPRINT(Debug,7,(&Debug, 
		    "read_headers: Updating storage message count %d -> %d\n",
		    storage->message_count,count));
    storage->message_count = count;

    clear_error();



    ret = count;
    goto finish;

 failure:

    reading_updater(COUNT_fail_cleanup,
		    storage->current_folder->cur_folder_disp,&COUNTER,
		    count,COUNTER.lastpercent,-1,-1,page,num_messages);
    show_last_error();
           
    DPRINT(Debug,7,(&Debug, 
		    "read_headers: Updating storage message count %d -> %d on FAILURE\n",
		    storage->message_count,count));
    storage->message_count = count;
    ret = -1;  /* FAILURE */
    
 finish:
    
    if (delay_redraw) {
	DPRINT(Debug,10,(&Debug, "read_headers: Triggering redraw\n"));
	
	menu_trigger_redraw(page);
	show_last_error();	    
    }

    if (previous_headers) {
	int i;

	for (i = 0; i < previous_count; i++) {
	    if (previous_headers[i].rec) 
		header_free(& (previous_headers[i].rec));	    
	}
	free(previous_headers);
	previous_headers = NULL;
    }
    previous_count = 0;
    
    if (reconnect_mode_ptr) {
	*reconnect_mode_ptr = reconnect_mode;
	reconnect_mode = NULL;
    } else if (reconnect_mode) {
	DPRINT(Debug,10,(&Debug,
			 "read_headers: Freeing reconnect_mode %p\n",
			 reconnect_mode));
	
	free_reconnect_mode(& reconnect_mode);
    }
    
    DPRINT(Debug,10,(&Debug, "read_headers=%d%s",
		     ret,
		     ret < 0 ? " (FAILURE)" : " messages read"));
    if (reconnect_mode_ptr && *reconnect_mode_ptr) {
	DPRINT(Debug,10,(&Debug, "; *reconnect_mode_ptr=%p",
			 *reconnect_mode_ptr));
    }    
    DPRINT(Debug,10,(&Debug, "\n"));
    
    return ret;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
