static char rcsid[] = "@(#)$Id: mime_encode.c,v 2.17 2020/05/21 02:58:29 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.17 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include <string.h>

#include "def_elm.h"
#include "s_elm.h"

#include <sys/time.h>

DEBUG_VAR(Debug,__FILE__,"mime");

extern short mime_count;

enum encoding update_encoding(top_encoding,encoding)
     enum encoding  *top_encoding;
     enum encoding encoding;
{
    if (encoding == ENCODING_8BIT &&
	(*top_encoding) != ENCODING_BINARY)
	(*top_encoding) = ENCODING_8BIT;
    if (encoding == ENCODING_BINARY)
	(*top_encoding) = ENCODING_BINARY;
    
    return (*top_encoding);
}

char * mime_generate_boundary (str, size)
     char *str;
     int size;
{
  time_t t = time (NULL);
	
  elm_sfprintf (str, size,
		FRM("ELM%d-%d-%d_"), t, getpid(), mime_count++);
  return str;
}


int attach_message(part,mailer,mime_info,X)
     mime_t *part; 
     struct out_state *mailer;
     mime_send_t *mime_info;
     struct mime_send_part * X;
{
    int err;
    FILE *srcfp;
    int is_text;


    if (part->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"attach_message",
		   "Bad magic number");

    if ((err = can_open(part->pathname0,"r")) != 0) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailErrnoS,
			  "Failed: %S: %.40s"),
		  part->dispname,strerror(err));
	return 0;
    }

    srcfp = fopen (part->pathname0, "r");
    if (!srcfp) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorOpeningNameS,
			  "Error opening %S!"), 
		  part->dispname);
    }
    
    /* 1 if is text type (true)
     * 0 if not text type
     * -1 if can't be encoded (ie structured) Message/ or Multpart/
     */

    is_text = decoder_is_textual(part);

    if (is_text < 0 && (part->encoding == ENCODING_QUOTED ||
			part->encoding == ENCODING_BASE64)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet,ElmDontEncoding,
			  "Content-Type don't allow encoding -- ignoring this part."));
	return 0;
    }

    update_encoding(&(mime_info->encoding_top),part->encoding);
	

    X -> TYPE    = part ->TYPE;

    if (part->TYPE_opts)
	X -> TYPE_opts_part = copy_mime_param(part->TYPE_opts);
    X ->disposition   = part -> disposition;
    if (part->DISPOSITION_opts)
	X -> DISPOSITION_opts = copy_mime_param(part->DISPOSITION_opts);
    if (part->description)
	X -> description      = dup_string(part->description);
    X -> encoding_part = part->encoding;

    X->start_loc = out_state_ftell(mailer);

    write_encoded (srcfp, mailer, part->encoding, is_text,
		   mime_info, 
		   0 /* Not known is text have already \r\n EOLNs */ );

    X->end_loc = out_state_ftell(mailer);

    fclose (srcfp);

    /* !! */
    if (ferror(out_state_FILE(mailer))) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmWriteFailedCopy,
			  "Write failed to temp file in copy"));
	return 0;   /* Just indicate failure */
    }

    DPRINT(Debug,4, 
	   (&Debug, 
	    "Preparing mail for sending: part %d (attach) is %d bytes (%s)\n",
	    X-mime_info->top_parts,
	    X->end_loc-X->start_loc,
	    ENCODING(X->encoding_part)));
    
    return 1; /* ok */
}

void write_encoded (srcfp, mailer, encoding, is_text, 
		    mime_info, source_is_binary)
     FILE *srcfp;
     struct out_state *mailer;
     enum encoding encoding;
     int is_text;
     mime_send_t *mime_info;
     int source_is_binary;
{
    char buffer[VERY_LONG_STRING];
    int line_len;
    enum encoding top_encoding = ENCODING_NONE;
    
    DPRINT(Debug,12,(&Debug, 
		     "write_encoded: encoding=%d (%s), is_text=%d",
		     encoding,ENCODING(encoding),is_text));

    if (mime_info) {
	DPRINT(Debug,12,(&Debug,
			 ", mime_info->encoding_top=%d (%s)",
			 mime_info->encoding_top,
			 ENCODING(mime_info->encoding_top)));
	top_encoding = mime_info->encoding_top;
    }

    DPRINT(Debug,12,(&Debug,", mailer->EOLN_is_CRLF=%d, source_is_binary=%d\n",
		     out_state_EOLN_is_CRLF(mailer),source_is_binary));

    if (encoding == ENCODING_BASE64)
	base64_encode (srcfp, mailer, is_text, top_encoding, source_is_binary);
    else if (encoding == ENCODING_QUOTED)
	quoted_printable_encode (srcfp, mailer, is_text, 
				 top_encoding,source_is_binary);
    else if ((is_text > 0 || encoding < ENCODING_BINARY) && 
	     !source_is_binary) {

	DPRINT(Debug,13,(&Debug, 
			 "write_encoded: Normalizing EOLN\n"));

	while ((line_len = mail_gets(buffer, sizeof(buffer)-1, srcfp)) > 0) {

	    /* Does \n => \r\n conversion if 
               mailer->EOLN_is_CRLF is set
	    */
	    state_convert_EOLN(buffer,&line_len,sizeof buffer,mailer);
	    
	    state_put(buffer, line_len, mailer);
	}

    } else {

	DPRINT(Debug,13,(&Debug, 
			 "write_encoded: Copying as is (binary?)\n"));
	
	/* This assumes that other encodings are already done ... */
	
	if (is_text < 0 && encoding == ENCODING_BINARY && !source_is_binary) {
	    
	    /* If part is structured, it is not know when EOLN needed to
	       to be normalized
	    */
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmBinaryEncodingNotSupported,
			      "Binary encoding is not supported for structured types."));
	}

	while (1) {
	    if ((line_len = fread(buffer, 1,sizeof(buffer)-1, srcfp)) <= 0)
		break;     

	    state_put(buffer, line_len, mailer);
	}
    }
}

#define LEN (out_state_ftell(mailer)-POS)
#define WRAP print_EOLN(mailer,top_encoding), \
 (POS = out_state_ftell(mailer)), state_putc(' ',mailer)

#define PUT(c)     {  int c1=(c); state_putc(c1,mailer); \
                      DPRINT(Debug,9, (&Debug,"%c",c1)); }
#define PUTS(s)    {  const char * s1=(s); state_puts(s1,mailer); \
                      DPRINT(Debug,9, (&Debug,"%s",s1)); }



void mime_write_header(mailer,hdr_name,first_part,opts,top_encoding)
     struct out_state *mailer;
     const char *hdr_name;
     const char *first_part;
     struct mime_param * opts;
     enum encoding top_encoding;
{
    long POS = out_state_ftell(mailer);

    state_puts(hdr_name,mailer);
    state_puts(": ",mailer);

    DPRINT(Debug,9,
	   (&Debug, "mime_write_header: %s:",
	    hdr_name));
    
    PUTS(first_part);

    if (opts) {
	char **V = encode_mime_params_v(opts);
	char **X;

	if (V) {
	    for (X = V; *X; X++) {
		char * c;
		
		
		PUT(';');
		
		if (LEN+strlen(*X) > 75) {
		    WRAP;
		    DPRINT(Debug,9,(&Debug, "[WRAP]\n          ")); 
		} else {
		    PUT(' ');
		}
		
		for (c = *X; *c != '\0'; c++) {  
		    if (*c == '\n' || (whitespace(*c) && (LEN > 120))) {    
			WRAP;                                               
			DPRINT(Debug,9,(&Debug, "[WRAP]\n          ")); 
		    } else {                                                
			PUT(*c);                                            
		    }              	       
		}

		free(*X); *X = NULL;
	    }

	    free(V); 
	}
    }

    print_EOLN(mailer,top_encoding);
    DPRINT(Debug,9,
	   (&Debug, "[EOLN]\n"));

}


void mime_write_part_headers(mailer,ptr,part)
     struct out_state *mailer;
     mime_send_t *ptr;
     struct mime_send_part *part;
{
    /* 1) Content-Transfer-Encoding */
    if (part->encoding_part < ENCODING_EXPERIMENTAL &&
	part->encoding_part > ENCODING_NONE) {
	state_printf(mailer,FRM("Content-Transfer-Encoding: %s"),
		     ENCODING(part->encoding_part));
	print_EOLN(mailer,ptr->encoding_top);
    } else if (part->encoding_part_text) {
	state_printf(mailer,FRM("Content-Transfer-Encoding: %s"),
		     part->encoding_part_text);
	print_EOLN(mailer,ptr->encoding_top);
    }
    
    /* 2) Content-Type */
    {
	char * s = elm_message(FRM("%s/%s"),
				     get_major_type_name(part->TYPE),
				     get_subtype_name(part->TYPE));

	mime_write_header(mailer,"Content-Type",s,
			  part->TYPE_opts_part,
			  ptr->encoding_top);
			  
	free(s);			  
    }

    /* 3) Content-Disposition */
    if (part->disposition != DISP_INLINE ||
	part->DISPOSITION_opts) {
	
	mime_write_header(mailer,"Content-Disposition",
			  DISPOSITION(part->disposition),
			  part->DISPOSITION_opts,
			  ptr->encoding_top);
    }

    /* 4) Content-Description */
    if (part->description) {
	write_string_header(mailer,"Content-Description",
			    part->description,ptr->encoding_top,
			    ptr->encode_hdr,ptr->hdr_charset);
    }
}

void mime_write_top_headers(mailer, ptr)
     struct out_state *mailer;
     mime_send_t *ptr;
{
    state_puts(MIME_HEADER, mailer);
    print_EOLN(mailer, ptr->encoding_top);  

    if (ptr->msg_is_multipart) {

	mime_write_header(mailer,"Content-Type","multipart/mixed",
			  ptr->TYPE_opts_top,ptr->encoding_top);

	state_printf(mailer,FRM("Content-Transfer-Encoding: %s"),
		     ENCODING(ptr->encoding_top));
	print_EOLN(mailer, ptr->encoding_top);  

    } else if (1 == ptr->top_parts_count) {
	const char *MIME_name_p = NULL; 
	    
	if (MIME_TYPE_TEXT == get_major_type_code(ptr->top_parts[0].TYPE) &&
	    0 == istrcmp("Plain",get_subtype_name(ptr->top_parts[0].TYPE)) &&
	    ptr->top_parts[0].result_charset &&
	    (MIME_name_p = 
	     get_charset_MIME_name(ptr->top_parts[0].result_charset)) &&
	    0 == istrcmp("US-ASCII", MIME_name_p) &&
	    (ENCODING_NONE == ptr->top_parts[0].encoding_part ||
	     ENCODING_7BIT == ptr->top_parts[0].encoding_part) &&
	    !send_mime_plain) {
	    DPRINT(Debug,2,(&Debug, 
			    "mime_write_header: send_mime_plain==%d: Omitting mime headers.\n",
			    send_mime_plain));
	    return;
	} else {
	    mime_write_part_headers(mailer,ptr,&(ptr->top_parts[0]));
	} 
    } else {
	DPRINT(Debug,2,(&Debug, 
			"mime_write_header: msg_is_multipart=%d, top_parts_count=%d: Odd values?\n",
		   ptr->msg_is_multipart,ptr->top_parts_count));
    }
} 

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
