static char rcsid[] = "@(#)$Id: mime.c,v 2.37 2021/07/07 16:28:00 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.37 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *            or Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/mime.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/


#include "def_elm.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"mime");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif       

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

void clear_mime_send_info(mime_info)
     mime_send_t *mime_info;
{
    /* bzero is defined on hdrs/defs.h */
    bzero((void *)mime_info, sizeof (*mime_info));

    mime_info->raw_level        = mailer_7bit;

    mime_info->encoding_top     = ENCODING_7BIT;
    mime_info->TYPE_opts_top    = NULL;
    mime_info->hdr_charset      = text_charset;   
    mime_info->mime_boundary[0] = '\0';

    /* FIXME: is this correct? 
              ... we also should avoid ISO2022 charsets on here
     */


    mime_info->encode_hdr       = 1;

    mime_info->top_parts_count  = 0;
    mime_info->top_parts        = NULL;
    
    /* These fields are not on correct place */
    mime_info->cl_offset = mime_info->cl_start = mime_info->cl_end = 0;
}

static struct mime_send_part * new_part P_((mime_send_t *mime_info));
static struct mime_send_part * new_part(mime_info)
     mime_send_t *mime_info;
{
    int ptr = mime_info->top_parts_count;

    mime_info->top_parts = safe_array_realloc(mime_info->top_parts,
					      (mime_info->top_parts_count +1),
					      sizeof (struct mime_send_part));
    mime_info->top_parts_count++;


    mime_info->top_parts[ptr].encoding_part      = ENCODING_7BIT;
    mime_info->top_parts[ptr].encoding_part_text = NULL;
    mime_info->top_parts[ptr].TYPE               = 0;
    mime_info->top_parts[ptr].TYPE_opts_part     = NULL;
	
    mime_info->top_parts[ptr].disposition        = 0;
    mime_info->top_parts[ptr].DISPOSITION_opts   = NULL;

    mime_info->top_parts[ptr].description        = NULL; 

    mime_info->top_parts[ptr].is_text            = 0;
    mime_info->top_parts[ptr].save_it_on_copy    = 0;
    mime_info->top_parts[ptr].start_loc          = 0;
    mime_info->top_parts[ptr].end_loc            = 0;
    mime_info->top_parts[ptr].result_charset     = NULL;

    DPRINT(Debug,4,
	   (&Debug,
	    "Preparing mail for sending: new_part %d\n",
	    ptr));
    /* Notice that returned pointer is valid only until
     * new_part() is called again
     */
    return &(mime_info->top_parts[ptr]);
}

static void reset_parts P_((mime_send_t *mime_info));
static void reset_parts(mime_info)
     mime_send_t *mime_info;
{
    int i;
    
    for (i = 0; i < mime_info->top_parts_count; i++) {
	mime_info->top_parts[i].encoding_part = 0;
	if (mime_info->top_parts[i].encoding_part_text) {
	    free(mime_info->top_parts[i].encoding_part_text);
	    mime_info->top_parts[i].encoding_part_text = NULL;
	}

	mime_info->top_parts[i].TYPE = NULL;
	if (mime_info->top_parts[i].TYPE_opts_part) 
	    free_mime_param(&(mime_info->top_parts[i].TYPE_opts_part));
	       	
	mime_info->top_parts[i].disposition = 0;
	if (mime_info->top_parts[i].DISPOSITION_opts) 
	    free_mime_param(&(mime_info->top_parts[i].DISPOSITION_opts));	   
	
	if (mime_info->top_parts[i].description) {
	    free_string(&(mime_info->top_parts[i].description));
	    mime_info->top_parts[i].description = NULL;
	}
	    

	mime_info->top_parts[i].is_text         = 0;
	mime_info->top_parts[i].save_it_on_copy = 0;
	mime_info->top_parts[i].result_charset  = NULL;
	mime_info->top_parts[i].start_loc       = 0;
	mime_info->top_parts[i].end_loc         = 0;
    }

    if (mime_info->top_parts) {
	free(mime_info->top_parts);
	mime_info->top_parts = NULL;
    }
    mime_info->top_parts_count = 0;
}

void free_mime_send_info(mime_info)
     mime_send_t *mime_info;
{
    reset_parts(mime_info);

    mime_info->encoding_top     = 0;
    mime_info->mime_boundary[0] = '\0';
    if (mime_info->TYPE_opts_top)
	free_mime_param(&(mime_info->TYPE_opts_top));
    mime_info->hdr_charset      = NULL;
    mime_info->encode_hdr       = 1;
    mime_info->msg_is_multipart = 0;
    mime_info->cl_offset           = 0;
    mime_info->cl_start            = 0;
    mime_info->cl_end              = 0;
}


static int write_one_part P_((mime_send_t *mime_info,
			      struct mime_send_part * X,
			      struct out_state *mailer,
			      FILE *tmp, 
			      struct menu_context *page,
			      struct menu_context *prompt_area));
		       
static int write_one_part(mime_info,X,mailer,tmp,page,prompt_area)
     mime_send_t *mime_info;
     struct mime_send_part * X;
     struct out_state *mailer;
     FILE *tmp;
     struct menu_context *page;     
     struct menu_context *prompt_area;
{
    int need_enc;
    charset_t ascii_ptr = MIME_name_to_charset("US-ASCII",0);

    if (!ascii_ptr)
	panic("MIME PANIC",__FILE__,__LINE__,"write_one_part",
	      "US-ASCII not found",0);

    rewind(tmp);

    /* Determine how this message should be MIME encoded:
     * 7BIT, 8BIT, BINARY or QUOTED-PRINTABLE.
     */
    need_enc = needs_encoding (tmp,NULL,0);
    rewind(tmp);

    if (need_enc & HAVE_BINARY) {
	if (mime_info->raw_level >= mailer_binary)
	    X->encoding_part = ENCODING_BINARY;
	else
	    /* no -BBINARYMIME option */
	    X->encoding_part = ENCODING_QUOTED;
    }
    else if (need_enc & HAVE_8BIT) {
	if (mime_info->raw_level >= mailer_8bit)
	    /* Just send 8BIT anyway */
	    X->encoding_part = ENCODING_8BIT;
	else
	    /* no -B8BITMIME option */
	    X->encoding_part = ENCODING_QUOTED;
    }

    DPRINT(Debug,12,(&Debug,
		     "write_one_part: need_enc=%d%s%s%s => encoding=%d (%s)\n",
		     need_enc,
		     (need_enc & HAVE_BINARY) ? " BINARY" : "",
		     (need_enc & HAVE_8BIT)   ? " 8BIT"   : "",
		     (need_enc & HAVE_CTRL)   ? " CTRL"   : "",
		     X->encoding_part,ENCODING(X->encoding_part)));

    /* Control characters can send with encoding 7BIT
     * HAVE_BINARY takes care really severe stuff */
    if ((need_enc & HAVE_CTRL) &&
	X->encoding_part != ENCODING_QUOTED) {
	if (! prompt_area) 		
	    X->encoding_part =  ENCODING_QUOTED;
	else { 
	    int  ch = prompt_letter(0,"",*def_ans_yes,
			       PROMPT_yesno|PROMPT_cancel,
			       prompt_area,
			       CATGETS(elm_msg_cat, ElmSet, 
				       ElmTextHaveControl,
				       "Text part %d have control characters. Encode text with quoted-printable? "),
			       (X - mime_info->top_parts) +1);

	    if (ch == *def_ans_yes) 
		X->encoding_part =  ENCODING_QUOTED;
	    else if (ch != *def_ans_no) {
		return 0;    /* Go again to verify_transmission loop */
	    }
	}
	
	DPRINT(Debug,12,(&Debug,
			 "write_one_part: encoding=%d (%s)\n",
			 X->encoding_part,ENCODING(X->encoding_part)));
    }

    /* This update may be little late for previous body parts
       but at least headers go correctly also for previous
       body parts ...
    */
    update_encoding(&(mime_info->encoding_top),X->encoding_part);

    if (X->result_charset) {
	int ascii_compat = charset_ok_p(X->result_charset);
	const char *MIME_name_X = 
	    get_charset_MIME_name(X->result_charset);


	/* Update Charset of part */

	DPRINT(Debug,8,(&Debug, 
			"charset=%s (%p)  8bit=%s  charset ascii compatible=%s\n",
			MIME_name_X ? MIME_name_X : "???",
			X->result_charset,
			(need_enc & HAVE_8BIT) ? "yes" : "no",
			ascii_compat ? "yes" : "no"));

	/* 1) If charset can be replaced with US-ASCII, do it */
	if (!(need_enc & HAVE_8BIT) &&
	    X->result_charset != ascii_ptr &&
	    ascii_compat) {

	    X->result_charset = ascii_ptr;
	    DPRINT(Debug,8,(&Debug, "Replacing charset with US-ASCII\n"));
	}

	/* 2) If charset claim to be US-ASCII, but there is 8-bit data
           use UNKNOWN-8BIT instead  (should not happen...)
	*/
	else if ((need_enc & HAVE_8BIT) &&
		 MIME_name_X && istrcmp(MIME_name_X,"US-ASCII") == 0) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUsingUNKNOWN8BIT,
			      "Text has 8BIT data and charset=US-ASCII, using charset=UNKNOWN-8BIT instead."));
	    
	    X->result_charset = MIME_name_to_charset("UNKNOWN-8BIT",1);

	    DPRINT(Debug,8,(&Debug, "Replacing charset with UNKNOWN-8BIT\n"));
	}

	/* Add charset to options */
	mime_params_add_compat(&(X->TYPE_opts_part), 
			       "charset", 
			       MIME_name_X ? MIME_name_X : "UNKNOWN-8BIT");
    }

    state_putc('\n',mailer);        /* filler */
    X->start_loc = out_state_ftell(mailer);

    write_encoded(tmp,mailer,X->encoding_part,
		  1 /* is text */,
		  mime_info, 
		  0 /* Editor buffer do not have \r\n end of lines */);
	
    X->end_loc = out_state_ftell(mailer);

    /* !! */
    if (ferror(out_state_FILE(mailer))) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmWriteFailedCopy,
			  "Write failed to temp file in copy"));
	return 0;   /* Just indicate failure */
    }

    DPRINT(Debug,4,
	   (&Debug,
	    "Preparing mail for sending: part %d is %d bytes (%s)\n",
	    X-mime_info->top_parts,
	    X->end_loc-X->start_loc,
	    ENCODING(X->encoding_part)));
    
    return 1;
}


static enum body_keyword {
    bk_error   = -1,
    bk_none    = 0,
    bk_START_ENCODE,
    bk_END_ENCODE,
    bk_DONT_SAVE,
    bk_MIME_INCLUDE,
    bk_MIME_ATTACH,
    bk_SEND_CHARSET,
    bk_DISABLE_KEYWORDS

} pick_keyword P_((char **buffer, int *len, char **keyword_data));

/* 'buffer' is assumed to be alloced 

   'buffer' is modified to include data after keyword (if any)
*/

/* 1 = succeed
   0 = failure */
static int include_part P_((FILE *dest,enum body_keyword  tag,
			    char *keyword_data,charset_t from_charset,
			    struct mime_send_part * X,
			    struct mailer_info  *mailer_info,
			    mime_send_t *mime_info,
			    const struct remote_server *remote_server));


static struct {
    const char      *  keyword;
    enum body_keyword  tag;
    int                has_params;

} bk_keywords[] = {
    
    { "encode",  bk_START_ENCODE, 0 }, 
    { "clear",   bk_END_ENCODE,   0 },
    { "no save", bk_DONT_SAVE,    0 },
    { "nosave",  bk_DONT_SAVE,    0 },
    { "include", bk_MIME_INCLUDE, 1 },
    { "attach",  bk_MIME_ATTACH,  1 },
    { "send-as-charset", bk_SEND_CHARSET,1 },
    { "send-with-charset", bk_SEND_CHARSET,1 },
    { "no keywords", bk_DISABLE_KEYWORDS, 0 },
    { "nokeywords",  bk_DISABLE_KEYWORDS, 0 },
    
    { NULL,      0,               0 }
};


static enum body_keyword pick_keyword(buffer,len,keyword_data)
     char **buffer; 
     int *len; 
     char **keyword_data;
{
    char *end   = NULL;
    char *endq  = NULL;
    char *spc   = NULL;

    enum body_keyword ret = bk_error;

    int q = 0;
    int i;
    int l1;


    if (*len < 1 || '[' != (*buffer)[0]) 
	goto no_keyword;
    

    /* "[[" */
    if (*len > 1 && '[' == (*buffer)[1]) {

	memmove(*buffer, (*buffer)+1, (*len)-1);

	(*len)--;
	goto no_keyword;
    }

    /* "[ " */
    if (*len > 1 && ' ' == (*buffer)[1])     
	goto no_keyword;

    for (i = 1; i < *len; i++) {
	
	if (']' == (*buffer)[i])
	    end = & ((*buffer)[i]);

	if (! (*buffer)[i]) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeywordNUL,
			      "NUL character on keyword line")); 
	    
	    goto have_error;
	}
	
	if ('\\' == (*buffer)[i]) {
	    i++;
	    continue;
	}

	if ('"' == (*buffer)[i]) {
	    q = !q;

	    if (!spc) {

		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeywordQuotation1,
			  "Quotation error on keyword line, no space before quotation."));
		goto have_error;
	    }

	    continue;
	}

	if (!q && !spc && ' ' == (*buffer)[i]) {
	    spc = & ((*buffer)[i]);
	    continue;
	}

	if (!q && ']' == (*buffer)[i]) {
	    endq = & ((*buffer)[i]);
	    break;
	}

    }

    if (!end) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeywordBrace,
			  "Missing ending keyword ] on line starting with [")); 

	goto have_error;
    }

    if (!endq) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeywordQuotation,
			  "Quotation error on keyword line, no ] outside of quotations"));
	goto have_error;
    }

    l1 = (endq - (*buffer)) -1;

    if (l1 < 1) {

	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeywordEmpty,
			  "Empty [] on keyword line"));
	goto have_error;       
    }


    /* 1) keyword without params */
    for (i = 0; bk_keywords[i].keyword; i++) {

	if (0 == strncmp(bk_keywords[i].keyword,
			 (*buffer)+1,l1)) {

	    if (bk_keywords[i].has_params) {

		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmBadKeywordReqPara,
				  "Keyword [%s] requires parameters."),
			  bk_keywords[i].keyword);

		goto have_error;       
	    }
	    
	    if (*keyword_data) {
		free(*keyword_data);
		*keyword_data = NULL;
	    }

	    ret = bk_keywords[i].tag;
	    goto have_keyword;

	}
    }

    if (!spc) {

	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeywordL,
			  "Bad keyword on text: [%.*s]"),
		  l1,(*buffer)+1);
	
	goto have_error;
    }

    /* 2) keyword with params */

    l1 = (spc - (*buffer)) -1;

    if (l1 < 1) 
	panic("MIME PANIC",__FILE__,__LINE__,"pick_keyword",
	      "Indexing error",0);

    for (i = 0; bk_keywords[i].keyword; i++) {
	
	if (0 == strncmp(bk_keywords[i].keyword,
			 (*buffer)+1,l1)) {

	    if (!bk_keywords[i].has_params) {

		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmBadKeywordNotPara,
				  "Keyword [%s] do not have parameters."),
			  bk_keywords[i].keyword);

		goto have_error;       
	    }
	    

	    l1 = endq - (spc+1);

	    if (l1 > 0) {
		
		*keyword_data = safe_realloc(*keyword_data,
					     l1+1);

		memcpy(*keyword_data,spc+1,l1);
		(*keyword_data)[l1] = '\0';

	    } else if (*keyword_data) {
		free(*keyword_data);
		*keyword_data = NULL;
	    }

	     if (l1 < 0) 
		panic("MIME PANIC",__FILE__,__LINE__,"pick_keyword",
		      "Indexing error",0);
	    

	     ret = bk_keywords[i].tag;
	     goto have_keyword;
	}
    }


    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeywordL,
		      "Bad keyword on text: [%.*s]"),
	      (endq-(*buffer))-1,(*buffer)+1);
    
 have_error:
    if (*keyword_data) {
	free(*keyword_data);
	*keyword_data = NULL;
    }
    return bk_error;
    
 have_keyword:

    DPRINT(Debug,4, 
	   (&Debug, "pick_keyword: keyword=%d buffer=%.*s\n",
	    ret,*len,*buffer));


    /* malloc_gets do not return \n */
    if (endq+1 < (*buffer) + *len &&
	*(endq+1) == '\n')
	endq++;

    l1 = ((*buffer) + *len) - (endq+1);
    
    if (l1 > 0) 
	memmove(*buffer, endq+1, l1);
    
    else {
	free(*buffer);
	*buffer = NULL;
    }
    
    if (l1 < 0) 
	panic("MIME PANIC",__FILE__,__LINE__,"pick_keyword",
	      "Indexing error",0);
    
    *len = l1;

    
    return ret;

 no_keyword:
    if (*keyword_data) {
	free(*keyword_data);
	*keyword_data = NULL;
    }
    
    return bk_none;
}

static FILE * gen_tempfile P_((void));
static FILE * gen_tempfile()
{
    int i;
    FILE * tmpfile;

    static int tmpcount = 0;
    int err = 0;
    
    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    if (!tmp)
	tmp = "/tmp/";


    for (i = 0; i < 10; i++) {
	char * filename = elm_message(FRM("%selmsd-%d-%d"),
				      tmp, getpid (),
				      tmpcount++);
	tmpfile = safeopen_rdwr(filename,&err);
	if (!tmpfile) {

	    DPRINT(Debug,10,(&Debug,   
			     "gen_tempfile: safeopen_rdwr: %s: %s (errno %d)\n",
			     filename,
			     strerror(err),err));
	} else {
	    unlink(filename);

	    DPRINT(Debug,10,(&Debug,   
			     "gen_tempfile: using temp file (%s)\n",
			     filename));
	}
	free(filename);
	if (tmpfile)
	    break;
    }

    if (!tmpfile) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmOpenFailedCopy,
			  "Temp file open failed to in copy"));
    }

    return tmpfile;
}


/* Returns 0 if '\n' is not seen */

static int copy_line_helper P_((FILE *source, char **buffer,
				int *len, int *alloced));
static int copy_line_helper(source,buffer,len,alloced)
     FILE *source; 
     char **buffer;
     int *len; 
     int *alloced;
{
    int ptr = 0;
    
    *len = 0;

    while (1) {
	int c;

	if (*alloced < ptr+80) {

	    *alloced = ptr+80;
	    *buffer = safe_realloc(*buffer,*alloced);
	}
	    
	c = getc(source);

	if (EOF == c) {
	    *len = ptr;
	    return 0;
	}

	(*buffer)[ptr] = c;
	ptr++;

	if ('\n' == c) {
	    *len = ptr;
	    break;
	}	 
    }

    return 1;
}


/* Copies buffer to block and then all lines until
   [ line is seen

   Line starting with [ is left to buffer
*/

static int copy_block_continue P_((FILE *target,FILE *source, char **buffer, 
				   int *len));
static int copy_block_continue(target,source,buffer,len)
     FILE *target;
     FILE *source; 
     char **buffer; 
     int *len;
{
    int alloced  = 0;


    if (*len > 0) {
	
	if (fwrite(*buffer,1,*len,target) != *len) 
	    goto write_failed;
	
	alloced = *len;
	*len = 0;
    }

    while (1) {

	/* Returns 0 if '\n' is not seen */
	if (!copy_line_helper(source,buffer,len,&alloced))
	    goto out;

	if (*len < 1)
	    panic("MIME PANIC",__FILE__,__LINE__,"copy_block_continue",
		  "Indexing error",0);

	if ('[' == (*buffer)[0])
	    break;

	if (fwrite(*buffer,1,*len,target) != *len) 
	    goto write_failed;
	
	*len = 0;
    }

 out:

    if (ferror(source)) {
	DPRINT(Debug,10,(&Debug,  "copy_block_continue: Read failed\n"));

	return 0;
    }

    if (0 != fflush(target)) {
	DPRINT(Debug,10,(&Debug,  
			 "copy_block_continue: fflush() failed on tmpfile\n"));

	return 0;
    }

    if (*len > 0) {

	if (alloced <= *len) {
	    alloced = (*len)+1;
	    *buffer = safe_realloc(*buffer,alloced);
	}
	(*buffer)[*len] = '\0';

    } else {

	if (*buffer) 
	    free(*buffer);
	*buffer = NULL;

    }

    return 1;

 write_failed:

    DPRINT(Debug,10,(&Debug, 
		     "copy_block_continue: write failed to tmpfile\n"));

    return 0;
}


static FILE * copy_block P_((FILE *source, char **buffer, int *len));
static FILE * copy_block(source,buffer,len)
     FILE *source; 
     char **buffer; 
     int *len;
{
    FILE * res = gen_tempfile();

    if (!res) {

	DPRINT(Debug,10,(&Debug,  "copy_block: No tmpfile\n"));

	return NULL;
    }

    if (copy_block_continue(res,source,buffer,len))
        return res;
    
    DPRINT(Debug,10,(&Debug,  "copy_block: failed\n"));

    if (*buffer) 
	free(*buffer);
    *buffer = NULL;
    *len    = 0;

    fclose(res);
    return NULL;
}

struct try_charset {
    charset_t             CS1;
    FILE                * result;
    int                   failed; 

};

static int add_try_charset P_((struct try_charset **list, int *count, 
			       charset_t cs));
static int add_try_charset(list,count,cs)
     struct try_charset **list; 
     int *count; 
     charset_t cs;
{
    struct try_charset  * LIST  = *list;
    int                   COUNT = *count;
    int ret;
    FILE * F = gen_tempfile();

    if (!F)
	return -1;


    LIST = safe_array_realloc(LIST,(COUNT+1), sizeof (LIST[0]));

    ret = COUNT++;

    LIST[ret].result = F;
    LIST[ret].CS1    = cs;
    LIST[ret].failed = 0;

    *list  = LIST;
    *count = COUNT; 

    return ret;
}


static int find_conversion P_((FILE * partfile, charset_t from_charset, 
			       charset_t to_charset,
			       FILE ** tmpfile, charset_t *tmpfile_charset,
			       int partnum,
			       struct menu_context *page,
			       struct menu_context *prompt_area));


static int find_conversion(partfile,from_charset,to_charset,
			   tmpfile, tmpfile_charset,
			   partnum,page,prompt_area)
     FILE                * partfile;
     charset_t             from_charset; 
     charset_t             to_charset;
     FILE               ** tmpfile; 
     charset_t           * tmpfile_charset;
     int                   partnum;
     struct menu_context * page;  
     struct menu_context * prompt_area;
{
    charset_t ascii_ptr = MIME_name_to_charset("US-ASCII",0);
    charset_t utf8cs = MIME_name_to_charset("UTF-8",0);
    
    struct try_charset  * trylist  = NULL;
    int                   trycount = 0; 

    char * buffer = NULL;
    int    alloced = 0;
    int    NL_seen = 0;
    int    from_errors = 0;
    int i;
	    
    if (!ascii_ptr)
	panic("MIME PANIC",__FILE__,__LINE__,"find_conversion",
	      "US-ASCII not found",0);

    if (!utf8cs)
	panic("CHARSET PANIC",__FILE__,__LINE__,"find_conversion",
	      "UTF-8 not found",0);


    *tmpfile         = NULL;
    *tmpfile_charset = NULL;

    add_try_charset(&trylist,&trycount,ascii_ptr);

    if (to_charset != ascii_ptr)    
	add_try_charset(&trylist,&trycount,to_charset);

    if (to_charset != utf8cs)    
	add_try_charset(&trylist,&trycount,utf8cs);


    do {
	int    len    = 0;

	/* Returns 0 if '\n' is not seen */
	NL_seen = copy_line_helper(partfile,&buffer,&len,&alloced);

	if (len > 0) {
	    int ERRORS  = 0;
	    struct string * in_str  = new_string(from_charset);

	    int parsed = 
		add_streambytes_to_string(in_str,len,s2us(buffer),&ERRORS);

	    if (parsed != len || ERRORS) {
		DPRINT(Debug,10,(&Debug,
				 "Failed to parse line: len = %d parsed = %d ERRORS = %d\n",
				 len,parsed,ERRORS));
		from_errors = 1;
	    }


	    for (i = 0; i < trycount; i++) {
		struct string * out_str = NULL; 
		int ERRORS2 = 0;

		char * result = NULL;
		int resultlen = 0;
		
		out_str = convert_string2(trylist[i].CS1,in_str,&ERRORS2);

		if (ERRORS2 > 0 && !trylist[i].failed) {
		    const char *MIME_name_c UNUSED_VAROK = 
			get_charset_MIME_name(trylist[i].CS1);

		    DPRINT(Debug,10,(&Debug,
				     "Failed to convert (%d) to charset %s\n",
				     i,
				     MIME_name_c ? 
				     MIME_name_c : 
				     "<no MIME name>"));
		    trylist[i].failed = 1;
		}

		bytestream_from_string(out_str,&result,&resultlen);

		if (resultlen != fwrite(result,1,resultlen,trylist[i].result)) {

		    DPRINT(Debug,10,(&Debug,
				     "Failed to write tempfile (%d)\n",
				     i));
		    
		    trylist[i].failed = 1;
		}
		    	       	    
		free(result);
		free_string(&out_str);
	    }

	    free_string(&in_str);
	}

    } while (NL_seen);

    if (buffer)
	free(buffer);
        
    if (ferror(partfile)) {
	DPRINT(Debug,10,(&Debug,  "find_conversion: Read failed\n"));

	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmFailedCopyBlock,
			  "Failed to copy block."));
		
	goto failure;
    }

    if (from_errors) {
	const char *MIME_name_f = 
	    get_charset_MIME_name(from_charset);

	int ch = 0;

	if (prompt_area)
	    ch = prompt_letter(0,"",*def_ans_no,
			       PROMPT_yesno|PROMPT_cancel,
			       prompt_area,
			       CATGETS(elm_msg_cat, ElmSet, 
				       ElmTextHaveBadCharacters,
				       "Text part %d have bad %s characters. Send anyway? "),
			       partnum+1,
			       MIME_name_f ? MIME_name_f : "<no MIME name>");
	
	if (ch != *def_ans_yes) 
	    goto failure;	    		
    }

    
    for (i = 0; i < trycount; i++) {
	
	const char *MIME_name_t UNUSED_VAROK = get_charset_MIME_name(trylist[i].CS1);

	DPRINT(Debug,10,(&Debug,
			 "Try charset (%d)=%s, failed=%d\n",
			 i,
			 MIME_name_t ? MIME_name_t : "<no MIME name>",
			 trylist[i].failed));

	if (! trylist[i].failed) {

	    *tmpfile         = trylist[i].result;
	    *tmpfile_charset = trylist[i].CS1;
	    
	    break;
	}

    }

    if (!*tmpfile) {
	
	const char *MIME_name_f = get_charset_MIME_name(from_charset);

	int ch = 0;

	if (prompt_area)
	    ch = prompt_letter(0,"",*def_ans_no,
			       PROMPT_yesno|PROMPT_cancel,
			       prompt_area,
			       CATGETS(elm_msg_cat, ElmSet, 
				       ElmTextHaveBadCharacters1,
				       "Failed convert %s characters on part %d. Send anyway? "),
			       MIME_name_f ? MIME_name_f : "<no MIME name>",
			       partnum+1);
	
	if (ch != *def_ans_yes) 
	    goto failure;
	
	if (trycount < 1)
	    panic("MIME PANIC",__FILE__,__LINE__,"find_conversion",
		  "Indexing error",0);


	/* Select last charset on list (normally UTF-8) */

	*tmpfile         = trylist[trycount-1].result;
	*tmpfile_charset = trylist[trycount-1].CS1;
	
    } else if (to_charset       != *tmpfile_charset &&
	       ascii_ptr        != *tmpfile_charset) {

	const char *MIME_name_tf = get_charset_MIME_name(*tmpfile_charset);
	const char *MIME_name_to = get_charset_MIME_name(to_charset);

	int ch = 0;

	if (prompt_area)
	    ch = prompt_letter(0,"",*def_ans_yes,
			       PROMPT_yesno|PROMPT_cancel,
			       prompt_area,
			       CATGETS(elm_msg_cat, ElmSet, 
				       ElmTextHaveBadCharacters2,
				       "Using charset %s instead of %s on part %d. Send anyway? "),
			       MIME_name_tf ? MIME_name_tf : "<no MIME name>",
			       MIME_name_to ? MIME_name_to : "<no MIME name>",
			       partnum+1);
	else
	    ch = *def_ans_yes;
	       	
	if (ch != *def_ans_yes) 
	    goto failure;

    }

    if (trylist) {

	for (i = 0; i < trycount; i++) {
	    if (trylist[i].result &&
		*tmpfile != trylist[i].result)
		fclose(trylist[i].result);
	}
	free(trylist);
    }

    {
	const char *MIME_name_fr UNUSED_VAROK = 
	    get_charset_MIME_name(from_charset);
	const char *MIME_name_tf UNUSED_VAROK = 
	    get_charset_MIME_name(*tmpfile_charset);
	
	DPRINT(Debug,4, 
	       (&Debug, 
		"Part %d prepare: conversion from charset %s to charset %s\n",
		partnum+1,
		MIME_name_fr ? MIME_name_fr : "<no MIME name>",
		MIME_name_tf ? MIME_name_tf : "<no MIME name>"));
	
    }
    
    return 1;
    
 failure:

    if (trylist) {

	for (i = 0; i < trycount; i++) {
	    if (trylist[i].result)
		fclose(trylist[i].result);
	}
	free(trylist);
    }

    *tmpfile         = NULL;
    *tmpfile_charset = NULL;

    return 0;


}

/* Returns 1 if OK */
int convert_text(source,dest,mime_info,from_charset,to_charset,do_pgp,
		 attachments,mailer_info,page,prompt_area,
		 remote_server)
     FILE *source;
     FILE *dest; 
     mime_send_t *mime_info;
     charset_t from_charset;
     charset_t to_charset;
     int do_pgp;
     struct Attachments  *attachments;
     struct mailer_info  *mailer_info;
     struct menu_context *page;     
     struct menu_context *prompt_area;
     const struct remote_server *remote_server;
{
    int ok = 1;
    int no_save = 0;
    int keyword_error = 0;
    int have_keywords = mime_body_keywords;

    reset_parts(mime_info);
	      
    if (use_PGP &&
	do_pgp) {   

	struct mime_send_part *X = new_part(mime_info);

	X->save_it_on_copy    = 1;
	X->is_text            = 0;
	/* write_one_part will fix encoding */
	X->encoding_part      = ENCODING_7BIT;
	X->encoding_part_text = NULL;

	if (do_pgp & PGP_MESSAGE) {
	    enum pgp_encrypt_type_v pe = 
		give_dt_enumerate_as_int(&pgp_encrypt_type);
		    
	    switch (pe) {
	    case pgp_encrypt_application_pgp:
	    default:
		X -> TYPE = give_media_type2(MIME_TYPE_APPLICATION,"pgp",1);
		break;
	    case pgp_encrypt_text_plain:
		X -> TYPE = give_media_type2(MIME_TYPE_TEXT,"plain",0);
		if (!X -> TYPE)
		    mime_panic(__FILE__,__LINE__,"convert_text",
			       "text/plain is not known");
		break;
	    case pgp_encrypt_text_x_pgp:
		X -> TYPE = give_media_type2(MIME_TYPE_TEXT,"x-pgp",1);
		break;
	    }
	} else {
	    enum pgp_sign_type_v ps = 
		give_dt_enumerate_as_int(&pgp_sign_type);

	    switch (ps) {
	    case pgp_sign_application_pgp:
	    default:
		X -> TYPE = give_media_type2(MIME_TYPE_APPLICATION,"pgp",1);
		break;
	    case pgp_sign_text_plain:
		X -> TYPE = give_media_type2(MIME_TYPE_TEXT,"plain",0);
		if (!X -> TYPE)
		    mime_panic(__FILE__,__LINE__,"convert_text",
			       "text/plain is not known");
		break;
	    case pgp_sign_text_x_pgp:
		X -> TYPE = give_media_type2(MIME_TYPE_TEXT,"x-pgp",1);
		break;
	    }
	}

	if (do_pgp & PGP_PUBLIC_KEY)
            mime_params_add_compat(&(X->TYPE_opts_part),"format","keys-only");
	else {
            mime_params_add_compat(&(X->TYPE_opts_part), "format", "text");
            /* This next bit is a non-non-standard, but exmh does this and it
             * can be very useful when parsing the message.
             */
            if (do_pgp & PGP_MESSAGE) {
		mime_params_add_compat(&(X->TYPE_opts_part), "x-action",
				       (do_pgp & PGP_SIGNED_MESSAGE) ? 
				       "encryptsign" : "encrypt");
            } else
		mime_params_add_compat(&(X->TYPE_opts_part), "x-action", "sign");
	}
        
	/* If message is signed, it can not be character set converted! */
	if (get_major_type_code(X -> TYPE)    == MIME_TYPE_TEXT)
	    X -> result_charset = from_charset;
	
	{
	    struct out_state * A = new_out_state(STATE_out_file);

	    set_out_state_file(dest,A);

	    /* Let state routines be aware of encoding */
	    if (mime_info->encoding_top == ENCODING_BINARY)
		set_out_state_EOLN(A,1);
	    
	    /* Whole source -- no splitting ... */
	    if (!write_one_part(mime_info,X,A,source, page,
				prompt_area))
		ok = 0;   /* Write failed or user canceled */

	    /* out_state_destroy does not fclose() dest */
	    free_out_state(&A);
	}

    } else { 

	int crypted         = FALSE;
	int gotten_key      = 0;
	char * buffer       = NULL;
	int    buffer_len   = 0;

	if (from_charset != to_charset && 
	    (!(CS_mapping & charset_properties(from_charset)) ||
	     !(CS_mapping & charset_properties(to_charset)))) {

	    const char *MIME_name_fr = get_charset_MIME_name(from_charset);
	    const char *MIME_name_to = get_charset_MIME_name(to_charset);

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantConvertCharset,
			      "Can't convert text from charset %s to charset %s"),
		      MIME_name_fr ? MIME_name_fr : "<no MIME name>",
		      MIME_name_to ? MIME_name_to : "<no MIME name>");
	    to_charset = from_charset;   /* No conversion */
	    sleep_message();   /* Needed ... */
	}
	
	if (from_charset != to_charset) {

	    const char *MIME_name_fr UNUSED_VAROK = 
		get_charset_MIME_name(from_charset);
	    const char *MIME_name_to UNUSED_VAROK = 
		get_charset_MIME_name(to_charset);

	    DPRINT(Debug,4, 
		   (&Debug, 
		    "Preparing mail for sending: conversion from charset %s to charset %s\n",
		    MIME_name_fr ? MIME_name_fr : "<no MIME name>",
		    MIME_name_to ? MIME_name_to : "<no MIME name>"));
	}
	
	mime_info->hdr_charset  = to_charset;
	
	while (!feof(source)) {
	    
	    enum body_keyword kw = bk_none;
	    FILE * partfile  = copy_block(source,&buffer,&buffer_len);
	    char * keyword_data = NULL;
	    int testc;

	    if (!partfile) 
		goto failure0;

	    
	    while(buffer_len > 0) {

		DPRINT(Debug,12,
		       (&Debug,
			"convert_text: keyword candinate (len=%d) buffer=%.*s\n",
			buffer_len,buffer_len,buffer));

		if (have_keywords)
		    kw = pick_keyword(&buffer,&buffer_len,&keyword_data);
		else {
		    DPRINT(Debug,12,
			   (&Debug,
			    "convert_text: keywords disabled\n"));
		}


		if (bk_none != kw) 
		    break;

		if (keyword_data)
		    panic("MIME PANIC",__FILE__,__LINE__,"convert_text",
			  "Unexpected keyword data",0);


		if (! copy_block_continue(partfile,source,&buffer,&buffer_len)) {

		    fclose(partfile);

		    goto failure0;
		}
	    }


#ifdef DEBUG
	    if (buffer_len > 0) {
		DPRINT(Debug,12,
		       (&Debug,
			"convert_text: kw=%d, left (len=%d) buffer=%.*s\n",
			kw,buffer_len,buffer_len,buffer));
	    } else {
		DPRINT(Debug,12,
		       (&Debug,
			"convert_text: kw=%d\n",kw));
	    }
#endif

	    rewind(partfile);

	    testc = getc(partfile);

	    /* 1) Handle  file block */
	    if (EOF != testc) {   /* Have text part */

		struct mime_send_part * X =  NULL;
		FILE *         tmpfile   = NULL;
		charset_t      tmpfile_charset  = NULL;


		ungetc(testc,partfile);

		if (!find_conversion(partfile,from_charset, to_charset,
				     &tmpfile,&tmpfile_charset,
				     mime_info->top_parts_count,page,
				     prompt_area)) {

		    DPRINT(Debug,4,(&Debug, "Failed to convert charset\n"));

		    if (keyword_data) {
			free(keyword_data);
			keyword_data = NULL;
		    }
		    
		    fclose(partfile);

		    goto failure1;
		}

		rewind(tmpfile);


		X =  new_part(mime_info);

		X -> save_it_on_copy     =  !no_save;
		X -> is_text             = 0;           /* strange ... */
		
		/* write_one_part will fix encoding */
		X -> encoding_part       = ENCODING_7BIT;
		X -> encoding_part_text  = NULL;

		X -> result_charset      = tmpfile_charset;

		if (crypted) {   /* Obsolete */

		    FILE * tmpfile2 = gen_tempfile();
		    char buffer[1024];
		    int line_len = 0;

		    struct out_state * A = NULL;

		    if (!gotten_key)
			panic("MIME PANIC",__FILE__,__LINE__,"convert_text",
			      "crypted without key",0);

		    X -> TYPE       = 
			give_media_type2(MIME_TYPE_APPLICATION,"X-ELM-encode",
					 0);
		    if (!X -> TYPE)
			mime_panic(__FILE__,__LINE__,
				   "convert_text",
				   "application/X-ELM-encode is not known");
		    
		    if (!tmpfile2) {
			fclose(tmpfile);

			DPRINT(Debug,4,(&Debug, 
					"Failed to generate extra tempfile\n"));
			
			fclose(partfile);
			
			goto failure0;

		    }

		    get_key_no_prompt();		/* reinitialize.. */
		    		    
		    fprintf(tmpfile2,"%s\n",START_ENCODE);

		    while (0 < (line_len = 
				mail_gets(buffer, sizeof buffer, tmpfile))) {

			encode(buffer);
			fputs(buffer,tmpfile2);
		    }

		    fprintf(tmpfile2,"%s\n",END_ENCODE);

		    rewind(tmpfile2);

		    X -> TYPE       = 
			give_media_type2(MIME_TYPE_APPLICATION,"X-ELM-encode",
					 0);
		    if (!X -> TYPE)
			mime_panic(__FILE__,__LINE__,
				   "convert_text",
				   "application/X-ELM-encode is not known");
		    

		    A = new_out_state(STATE_out_file);
		    set_out_state_file(dest,A);
		    
		    /* Let state routines be aware of encoding */
		    if (mime_info->encoding_top == ENCODING_BINARY)
			set_out_state_EOLN(A,1);
		    
		    if (!write_one_part(mime_info,X,A,tmpfile2,page,
					prompt_area))
			ok = 0;   /* Write failed or user canceled */
		    
		    /* out_state_destroy does not fclose() dest */
		    free_out_state(&A);
		    
		    fclose(tmpfile2);
		    
		} else {

		    struct out_state * A = NULL;
		 
		    X -> TYPE = give_media_type2(MIME_TYPE_TEXT,"plain",0);
		    if (!X -> TYPE)
			mime_panic(__FILE__,__LINE__,
				   "convert_text",
				   "text/plain is not known");
   
		    A = new_out_state(STATE_out_file);
		    set_out_state_file(dest,A);

		    /* Let state routines be aware of encoding */
		    if (mime_info->encoding_top == ENCODING_BINARY)
			set_out_state_EOLN(A,1);

		    if (!write_one_part(mime_info,X,A,tmpfile,page,
					prompt_area))
		    ok = 0;   /* Write failed or user canceled */

		    /* out_state_destroy does not fclose() dest */
		    free_out_state(&A);
		}


		fclose(tmpfile);
	    } 

	    if (ferror(partfile)) {

		DPRINT(Debug,4,(&Debug, "Failed read from temporary file\n"));

		fclose(partfile);

		goto failure0;
	    }
	    

	    /* 2) Handle keyword     */

	    switch (kw) {

	    case bk_error:
		if (keyword_error++ == 0) {
		    if (RawState())
			lower_prompt(catgets(elm_msg_cat, ElmSet, 
					     ElmBadKeywordHint,
					     "Use [[ to specify line starting with ["));
		}
		ok = 0;
		break;

	    case bk_none:
		break;

	    case bk_START_ENCODE:      /* Obsolete */

		if ( ! OPMODE_IS_INTERACTIVE(opmode)) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoEncryptInBatch,
				      "Sorry. Cannot send encrypted mail in \"batch mode\".\n"));
		    ok = 0;
		} else if (!gotten_key) {
		    if (1 == getkey(ON)) {
			gotten_key = 1;
			crypted    = 1;
		    } else
			ok = 0;		   			
		} else
		   crypted    = 1;
		break;

	    case bk_END_ENCODE:      /* Obsolete */
		crypted = 0;
		break;
		

	    case bk_DONT_SAVE:
		no_save = 1;
		break;


	    case bk_MIME_INCLUDE:
	    case bk_MIME_ATTACH: {

		struct mime_send_part *X  = new_part(mime_info);
		
		X->save_it_on_copy   =  !no_save;
		X->is_text           = 0;
		
		if (!include_part(dest,kw,
				  keyword_data,from_charset,
				  X,mailer_info,mime_info,
				  remote_server))
		    ok = 0;
		
	    }
		break;
	
	    case bk_SEND_CHARSET: {
		
		if (!keyword_data) 
		    to_charset = text_charset;
		else { 
		    charset_t newcs = to_charset;

		    const char * T = keyword_data;

		    switch (special_charset_keyword(keyword_data)) {
		    case cskw_SYSTEM:  newcs = system_charset;  break;
		    case cskw_DISPLAY: newcs = display_charset; break;
		    case cskw_TEXT:    newcs = text_charset;    break;
		    default:
			newcs = MIME_name_to_charset(keyword_data,0);
		    }

		    if (newcs && get_charset_MIME_name(newcs))
			T = get_charset_MIME_name(newcs);

		    if (!newcs || ! get_charset_MIME_name(newcs) 
			||
			((from_charset != newcs && 
			  0 == (CS_mapping & 
				charset_properties(from_charset))) 
			 ||
			 0 == (CS_mapping & 
			       charset_properties(newcs))
			 )
			) {

			const char *MIME_name_fr = 
			    get_charset_MIME_name(from_charset);

			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantConvertCharset,
					  "Can't convert text from charset %s to charset %s"),
				  MIME_name_fr ? MIME_name_fr : 
				  "<no MIME name>",
				  T ? T : "<no MIME name>");
			ok = 0;
		    } else {
			to_charset = newcs;

			DPRINT(Debug,12,
			       (&Debug,
				"convert_text: to charset %s\n",T));
		    }
		}

	    }
		break;

	    case bk_DISABLE_KEYWORDS:
		have_keywords = 0;

		DPRINT(Debug,12,
		       (&Debug,
			"convert_text: disabling keywords\n"));

		break;
		
	    }
	    
	    if (keyword_data) {
		free(keyword_data);
		keyword_data = NULL;
	    }

	    fclose(partfile);

	}

	
	if (0) {
	failure0:

	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmFailedCopyBlock,
			      "Failed to copy block."));
	failure1:
	    ok = 0;
	}

    }

    if (attachments->attachment_count > 0) {

	if (use_PGP &&
	    do_pgp) {

	    if (do_pgp & PGP_MESSAGE)
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoPgpAttachEncode,
				  "WARNING: I PGP encode only main message"));
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoPgpAttachSign,
				  "WARNING: I PGP sign only main message"));
	}

	{
	    int i;
	    struct out_state * A = new_out_state(STATE_out_file);

	    set_out_state_file(dest,A);
	    
	    /* Let state routines be aware of encoding */
	    if (mime_info->encoding_top == ENCODING_BINARY)
		set_out_state_EOLN(A,1);


	    /* now add attachments */
	    for (i = 0; i < attachments->attachment_count; i++) {
		struct mime_send_part * X  = new_part(mime_info);

		X->save_it_on_copy   = !no_save;
		X->is_text           = 0;
		if (!attach_message (& (attachments->attach_files[i]), 
				     A, mime_info, X))
		    ok = 0;
	    }

	    /* out_state_destroy does not fclose() dest */
	    free_out_state(&A);
	}	
    }

    if (keyword_error > 1 && RawState())
	error_sleep(sleepmsg *2);
    
    return ok;
}

int check_for_multipart(filedesc, mime_info, mailer_info, from_charset, 
			remote_server)
     FILE *filedesc;
     mime_send_t *mime_info;
     struct mailer_info  *mailer_info;
     charset_t from_charset;
     const struct remote_server *remote_server;

{
    char * buffer = NULL;
    int len = 0;
    int alloced = 0;

    int Multi_Part = FALSE;
    int    NL_seen = 0;    
    int have_keywords = mime_body_keywords;

    do {
	NL_seen = copy_line_helper(filedesc,&buffer,&len,&alloced);

	if (len > 0 && have_keywords && buffer[0] == '[') {

	    char * keyword_data = NULL;

	    enum body_keyword  kw = pick_keyword(&buffer,&len,&keyword_data);

	    alloced = 0;   /* pick_keywords modifies buffer */

	    switch(kw) {
	    
	    case bk_MIME_INCLUDE:
	    case bk_MIME_ATTACH:  {

		Multi_Part = TRUE;
		if (!include_part(filedesc,kw,
				  keyword_data,from_charset,
				  NULL,mailer_info,mime_info,
				  remote_server)) {

		    if (keyword_data)
			free(keyword_data);

		    return -1;
		}
	    }
		
		break;

	    case bk_DISABLE_KEYWORDS:
		have_keywords = 0;

		break;
	    case bk_error:

		if (keyword_data)
		    free(keyword_data);


		return -1;

	    case bk_none:
	    case bk_START_ENCODE:
	    case bk_END_ENCODE:
	    case bk_DONT_SAVE:
	    case bk_SEND_CHARSET:
		/* Not used */
		break;
	
	    }

	    if (keyword_data)
		free(keyword_data);

	}

    } while(NL_seen);

    if (buffer)
	free(buffer);

    rewind(filedesc);
    return(Multi_Part);

}

/* 1 = succeed
   0 = failure */
static int include_part(dest,tag,keyword_data,from_charset,
			X,mailer_info,mime_info,remote_server)

     FILE *dest;
     enum body_keyword  tag;
     char *keyword_data;
     charset_t from_charset;
     struct mime_send_part * X;
     struct mailer_info  *mailer_info;
     mime_send_t *mime_info;
     const struct remote_server *remote_server;   
{
    int check = (X == NULL);
    char *ptr = NULL;
    char *incptr;
    char Include_Filename[SLEN];
    struct string          * orig_filename = NULL;
    struct folder_browser  * expanded_file = new_browser(selection_file);
    char                   * expanded_2    = NULL;
    int                      is_copy = 0;
    int ret = 1;

    media_type_t       Type      = NULL;
    struct mime_param *Type_opts = NULL;

    char Encoding[SLEN];

    enum encoding      Enc_Type  = ENCODING_NONE;
    FILE *incfile;
    int is_text;

    int is_attachment = -1;

    if (remote_server)
        browser_set_remote_server(expanded_file,remote_server);

    if (!keyword_data) {
	DPRINT(Debug,9,
	       (&Debug, "include_part: tag=%d, missing parameters\n",
		tag));
	
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoIncludeFilename,
			  "No Filename given, include line ignored"));
	ret = 0;
	goto check_out;
    }

    DPRINT(Debug,9,
	   (&Debug, "include_part: tag=%d, parameters=%s\n",
	    tag, keyword_data)); 

    ptr = keyword_data;

    if (tag == bk_MIME_INCLUDE)
	is_attachment = 0;
    else if (tag == bk_MIME_ATTACH)
	is_attachment = 1;
    else 
	panic("MIME PANIC",__FILE__,__LINE__,"include_part",
	      "Bad keyword",0);

    while ((*ptr != '\0') && (*ptr == ' '))
	ptr++;
    incptr = Include_Filename;
    while ((*ptr != ' ') && (*ptr != '\0') &&
	   (incptr < Include_Filename + sizeof(Include_Filename) -1))
	*incptr++ = *ptr++;
    *incptr = '\0';

    if (strlen(Include_Filename) == 0) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoIncludeFilename,
			  "No Filename given, include line ignored"));
	ret = 0;
	goto check_out;
    }

    orig_filename = new_string2(from_charset,s2us(Include_Filename));

    if (! select_dir_item(expanded_file,&orig_filename, NULL)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedToExpandInclude,
			  "Failed to expand include filename"));
	ret = 0;
	goto check_out;
    }


    while ((*ptr != '\0') && (*ptr == ' '))
	ptr++;

    if ('\0' == *ptr) {
	FILE * fp;
	int tmp;
	char * ext = NULL;
	char *p = strrchr(Include_Filename,'.');
	struct scan_list * scanlist;
	int check_utf8 =  is_utf8_charset(display_charset);


	if (p)
	    ext = p+1;

	is_text = 0;    /* Assume non-textual type   */

	if (!dir_make_ref(expanded_file,&expanded_2,&is_copy,  is_text)) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIncludeCannotAccess,
			      "Include File can't be accessed"), 0);
	    ret = 0;
	    goto check_out;
	}

	fp = fopen (expanded_2, "r");
	if (!fp) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIncludeCannotAccess,
			      "Include File can't be accessed"), 0);
	    ret = 0;
	    goto check_out;
	}

	scanlist = init_scanlist(ext);

	tmp = needs_encoding (fp,scanlist,check_utf8);

	fclose(fp);
	
	/* Autodectect content-type */

	if (! detect_mime_type(Include_Filename,&Type,&Type_opts,ext,
			       scanlist)) {

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoContentTypeGenerated,
			      "Unable to guess Content-Type, include line ignored"));



	    if (scanlist)
		free_scan_list(&scanlist);

	    ret = 0;
	    goto check_out;
	}

	Encoding[0] = '\0';

	is_text = give_text_type_code(Type);   /* Update guess */

	if (tmp & HAVE_BINARY) {

	    if (mime_info->raw_level < mailer_binary)
		Enc_Type = ENCODING_BASE64;
	    else
		Enc_Type = ENCODING_BINARY;

	} else if (tmp & HAVE_8BIT) {
	    
	    if (is_text > 0) {
		if (mime_info->raw_level < mailer_8bit)
		    Enc_Type = ENCODING_QUOTED;
		else
		    Enc_Type = ENCODING_8BIT;
	    } else {
		if (mime_info->raw_level < mailer_8bit)
		    Enc_Type = ENCODING_BASE64;
		else
		    Enc_Type = ENCODING_8BIT;
	    }
	}

	if (get_major_type_code(Type) == MIME_TYPE_TEXT && 
	    (tmp & HAVE_8BIT)) {

	    const char *MIME_name_d = get_charset_MIME_name(display_charset);

	    const char *cs;
	    
	    cs = get_mime_param_compat(Type_opts,"charset");

	    if (!cs && check_utf8 && !(tmp & HAVE_UTF8)) {

		DPRINT(Debug,9,
		       (&Debug, 
			"include: utf8 check failed -- adding charset=UNKNOWN-8BIT\n"));
		
		mime_params_add_compat(& Type_opts,
				       "charset","UNKNOWN-8BIT");
		
		
		
		
	    } else if (!cs && MIME_name_d) {
		
		DPRINT(Debug,9,
		       (&Debug, 
			"include: Adding charset=%s\n",
			MIME_name_d));
		
		mime_params_add_compat(& Type_opts,
				       "charset", 
				       MIME_name_d);
	    }
	}

	if (scanlist)
	    free_scan_list(&scanlist);

    } else {
	char Primary_Type[SLEN];
	char SubType[SLEN];
	char Params[STRING];

	incptr = Primary_Type;
	while ((*ptr != ' ') && (*ptr != '\0') && (*ptr!='/')
	       && (*ptr != ';') 
	       && (incptr < Primary_Type + sizeof(Primary_Type) -1))
	    *incptr++ = *ptr++;
	*incptr = '\0';
	while ((*ptr != '\0') && (*ptr == ' '))
	    ptr++;
	incptr = SubType;
	if (*ptr == '/') {
	    ptr++;
	    while ((*ptr != '\0') && (*ptr == ' '))
		ptr++;
	    while ((*ptr != ' ') && (*ptr != '\0') && 
		   (*ptr!=';') && (incptr < SubType + sizeof(SubType) -1))
		*incptr++ = *ptr++;
	}
	*incptr = '\0';
	
	if (strlen(Primary_Type) == 0 || strlen(SubType) == 0 ) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoContentTypeGiven,
			      "No Content-type given, include line ignored"));
	    ret = 0;
	    goto check_out;
	}

	while ((*ptr != '\0') && (*ptr == ' '))
	    ptr++;
	incptr = Params;
	while (*ptr == ';') {
	    ptr++;
	    if (incptr > Params) {
		*incptr++ = ';';
	    } else if (*ptr == ' ')
		ptr++;
	    
	    while ((*ptr != '\0') && (*ptr == ' ')
		   && (incptr < Params + sizeof(Params) -1))
		*incptr++ = *ptr++;
	    
	    while ((*ptr != ' ')  && (*ptr != '\0') && (*ptr!=';')
		   && (incptr < Params + sizeof(Params) -1))
		*incptr++ = *ptr++;
	    while ((*ptr != '\0') && (*ptr == ' '))
		ptr++;
	}
	*incptr = '\0';
	
	
	while ((*ptr != '\0') && (*ptr == ' '))
	    ptr++;
	incptr = Encoding;
	while ((*ptr != ' ') && (*ptr != '\0')
	       && (incptr < Encoding + sizeof(Encoding) -1))
	    *incptr++ = *ptr++;
	*incptr = '\0';


	if (! Encoding[0]) {
	    Enc_Type = ENCODING_NONE;
	    
	    DPRINT(Debug,11,(&Debug,
			 "include_part: Encoding not given\n"));
	} else
	    Enc_Type = check_encoding(Encoding);
	
	if (Enc_Type == ENCODING_ILLEGAL) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmEncodingIsIllegal,
			      "Encoding is illegal"));
	    ret = 0;
	    goto check_out;
	}

	Type = give_media_type(Primary_Type,SubType,1);

	/* 1 if is text type (true)
	 * 0 if not text type
	 * -1 if can't be encoded (ie structured) Message/ or Multipart/ 
	 */


	if (Params[0]) {   
	    enum mime_parameters_v mp = 
		give_dt_enumerate_as_int(&mime_parameters);
	    /* 0 == plain
	       1 == encoded
	       2 == plain-and-encoded
	    */
	    
	    struct string * XX = new_string2(from_charset,s2us(Params));
	    
	    switch(mp) {
	    case mime_parameters_plain:
	        Type_opts = parse_mime_param_string(XX,0,1);
		break;
	    case mime_parameters_encoded:
		Type_opts = parse_mime_param_string(XX,0,0);
		break;
	    default:
		Type_opts = parse_mime_param_string(XX,
						    MAX_COMPAT_LEN,
						    0);
		break;
	    }
	    free_string(&XX);	    
	}

	/* 7bit and 8bit are only allowed for line orienteed types */
	if (Enc_Type == ENCODING_7BIT || 
	    Enc_Type == ENCODING_8BIT) {
	    DPRINT(Debug,11,(&Debug,
			     "include_part: textual encoding (%s)\n",
			     ENCODING(Enc_Type)));
	    is_text = 1;
	} else
	    is_text = give_text_type_code(Type);

	if (!dir_make_ref(expanded_file,&expanded_2,&is_copy,  is_text)) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIncludeCannotAccess,
			      "Include File can't be accessed"), 0);
	    ret = 0;
	    goto check_out;
	}

	/* Don't allow 7BIT if 8-bit charcters in any type,    
	 * don't allow 8BIT if 'binary' characters       - K E H */
	if (Enc_Type == ENCODING_7BIT 
	    || Enc_Type == ENCODING_NONE
	    || Enc_Type == ENCODING_8BIT) {
	    
	    FILE * fp = fopen (expanded_2, "r");
	    if (fp) {
		int check_utf8 =  is_utf8_charset(display_charset);

		int tmp = needs_encoding (fp,NULL,check_utf8);
		fclose(fp);
		
 		if (tmp & HAVE_BINARY) {
		    
		    if (Enc_Type != ENCODING_NONE)
			lib_error(CATGETS(elm_msg_cat, ElmSet,ElmIncludeBINARY,
					  "Include file has BINARY data."));  
		    
		    if (Enc_Type == ENCODING_7BIT || 
			Enc_Type == ENCODING_8BIT) {
			/* indicate failure */
			ret = 0;
			goto check_out;
		    }
		    
		    
		    if (Enc_Type == ENCODING_NONE &&
			mime_info->raw_level < mailer_binary)
			Enc_Type = ENCODING_BASE64;
		    else
			Enc_Type = ENCODING_BINARY;
		    
		    
		} else if ((tmp & HAVE_8BIT) && Enc_Type != ENCODING_8BIT) {
		    
		    if (Enc_Type != ENCODING_NONE)
			
			lib_error (CATGETS(elm_msg_cat, ElmSet,ElmInclude8BIT,
					   "Include file has 8BIT data."));   
		    if (Enc_Type == ENCODING_7BIT) {
			/* indicate failure */
			ret = 0;
			goto check_out;
		    }	      	
		    
		    if (is_text > 0) {
			if (Enc_Type == ENCODING_NONE &&
			    mime_info->raw_level < mailer_8bit)
			    Enc_Type = ENCODING_QUOTED;
			else
			    Enc_Type = ENCODING_8BIT;
		    } else {
			if (Enc_Type == ENCODING_NONE &&
			    mime_info->raw_level < mailer_8bit)
			    Enc_Type = ENCODING_BASE64;
			else
			    Enc_Type = ENCODING_8BIT;
		    }
		}
		
		if (get_major_type_code(Type) == MIME_TYPE_TEXT && 
		    (tmp & HAVE_8BIT)) {
		    const char *cs;
		    
		    const char *MIME_name_d = 
			get_charset_MIME_name(display_charset);

		    cs = get_mime_param_compat(Type_opts,"charset");
		    
		    if (!cs && check_utf8 && !(tmp & HAVE_UTF8)) {
			
			DPRINT(Debug,9,
			       (&Debug, 
				"include: utf8 check failed -- adding charset=UNKNOWN-8BIT\n"));
			
			mime_params_add_compat(& Type_opts,
					       "charset","UNKNOWN-8BIT");
			
			
			
			
		    } else if (!cs && MIME_name_d) {
			
			DPRINT(Debug,9,
			       (&Debug, 
				"include: Adding charset=%s\n",
				MIME_name_d));
			
			mime_params_add_compat(& Type_opts,
					       "charset", 
					       MIME_name_d);
		    }
		}
	    }
	}
    
    } 


    if (Enc_Type >= ENCODING_EXPERIMENTAL) {
	
	FILE * fp = fopen (expanded_2, "r");
	if (fp) {
	     
	    int tmp = needs_encoding (fp,NULL,0);
	    fclose(fp);

	    if ((tmp & HAVE_BINARY) || (tmp & HAVE_8BIT)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet,ElmMustBeAlreadyEncoded,
                                  "Files with eXperimental (X-...) encoding must have encoded first."));

		ret = 0;   /* indicate failure */
		goto check_out;
	    }

	 }
    }

    if (Enc_Type == ENCODING_8BIT) {
	if (mime_info->raw_level < mailer_8bit) {
	    lib_error (CATGETS(elm_msg_cat, ElmSet,ElmDoesnt8BIT,
			       "Mailer (MTA) doesn't support 8BIT encoding."));
	    ret = 0;   /* indicate failure */
	    goto check_out;		
	}
    }
    
    if (Enc_Type == ENCODING_BINARY) {
	if (mime_info->raw_level < mailer_binary) {
	    lib_error (CATGETS(elm_msg_cat, ElmSet,ElmDoesntBINARY,
			       "Mailer (MTA) doesn't support BINARY encoding!"));
	    ret = 0;
	    goto check_out;
	}
    }

    
    if (is_text < 0 && Enc_Type > ENCODING_BINARY) {
	lib_error(CATGETS(elm_msg_cat, ElmSet,ElmDontEncoding,
			  "Content-Type don't allow encoding -- ignoring this part."));
	
	ret = 0;
	goto check_out;
    }
    
    
    update_encoding(&(mime_info->encoding_top),Enc_Type);

    if (check) {
	check_out:

	if (orig_filename)
	    free_string(&orig_filename);
	if (expanded_file)
	    free_dir(&expanded_file);
	
	if (expanded_2) {
	    if (is_copy &&
		0 == unlink(expanded_2)) {
		DPRINT(Debug,4,(&Debug, 				
				"include_part: '%s' unlinked\n",
				expanded_2));
	    }
	    free(expanded_2);
	    
	}

	if (Type_opts)
	    free_mime_param(&Type_opts);
	
	return ret;
    }

      
    incfile = fopen (expanded_2, "r");
    if (incfile) {

	DPRINT(Debug,4,(&Debug, 				
			"include_part: '%S' C-T=%s/%s Enc=%d (%s) is_text=%d\n",
			orig_filename,
			get_major_type_name(Type),
			get_subtype_name(Type), 
			Enc_Type,
			ENCODING(Enc_Type),
			is_text));
	
	X->encoding_part = Enc_Type;
	if (0 == strincmp("x-",Encoding,2)) {
	    X->encoding_part_text = safe_strdup(Encoding);
	    
	    DPRINT(Debug,4,(&Debug, 				
			    "include_part: encoding=%s\n",
			    X->encoding_part_text));
	}
	X->TYPE = Type;

	if (Type_opts) {
	    if (X->TYPE_opts_part)
		free_mime_param(& X->TYPE_opts_part);
	    
	    X->TYPE_opts_part = Type_opts;  
	    Type_opts = NULL;
	    
	}


	/* When user uses [include ...] it is better use disposition
	 * 'inline' instead of 'attachment'
	 */
	{ 
		enum mime_parameters_v mp = 
		    give_dt_enumerate_as_int(&mime_parameters);
		/* 0 == plain
		   1 == encoded
		   2 == ascii-and-encoded
		*/

		int was_ok = 0;
		char * p2 = pick_name_ascii(Include_Filename,&was_ok);
		char * compat = NULL;
		int   have_encoded = 0;

		if (*p2) {
		    int OK = 0;
		    compat = compat_filename(p2,&OK);
		    if (!OK)
			was_ok = 0;
		}
		free(p2); p2 = NULL;
	       
		if (mp > mime_parameters_plain &&
		    (!was_ok || !can_ascii_string(orig_filename))) {
		    
		    
		    struct string *p1 = pick_name(orig_filename);
		    
		    if (p1) {
			mime_params_add(& (X->DISPOSITION_opts),
					"filename", p1);
			
			free_string(&p1);
			
			have_encoded ++;
		    }
		}
		
		DPRINT(Debug,9,
		       (&Debug,
			"include: mime_parameters=%d have_encoded=%d was_ok=%d\n",
			mp,have_encoded));
			

		if (compat) {
		    if (mime_parameters_plain == mp ||
			mime_parameters_plain_and_encoded == mp ||
			! have_encoded) {
		    
			mime_params_add_compat(&(X->DISPOSITION_opts),"filename",
					       compat);      
			
		    }
		    free(compat); compat = NULL;
		}
		
	}

	X->disposition = is_attachment ? DISP_ATTACH: DISP_INLINE;

	fputc('\n',dest);        /* filler */
	X->start_loc = ftell(dest);

	    /* Encoding rules are different for text and non-text. 
	     * For text we must do \n -> \r\n before base64 encoding */
    

	{
		struct out_state * A = new_out_state(STATE_out_file);

		set_out_state_file(dest,A);
		
		/* Let state routines be aware of encoding */
		if (mime_info->encoding_top == ENCODING_BINARY)
		    set_out_state_EOLN(A,1);

		/* For ENCODING_EXPERIMENTAL this is already supposed to be 
		   encoded */
		write_encoded(incfile,A,Enc_Type,is_text,mime_info,
			      0 /* Don't know if source is already \r\n EOLNs */);
    
		/* out_state_destroy does not fclose() dest */
		free_out_state(&A);
	}

	X->end_loc = ftell(dest);
	fclose(incfile);

	if (ferror(dest)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmWriteFailedCopy,
				  "Write failed to temp file in copy"));
		/* Just indicate failure */
		ret = 0;
		goto fail_out;
	}

	    DPRINT(Debug,4,(&Debug,   
			    "Preparing mail for sending: part %d (include) is %d bytes (%s)\n",
			    X-mime_info->top_parts,
			    X->end_loc-X->start_loc,
			    ENCODING(X->encoding_part)));

    } else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantOpenIncludedFile,
			      "Can't open included File"));
	    ret = 0;
	    goto fail_out;
    }

 fail_out:
	if (orig_filename)
	    free_string(&orig_filename);
	if (expanded_file)
	    free_dir(&expanded_file);

	if (expanded_2) {
	    if (is_copy && 
		0 == unlink(expanded_2)) {
		DPRINT(Debug,4,(&Debug, 				
				"include_part: '%s' unlinked\n",
				expanded_2));
	    }
	    free(expanded_2);
	}

	if (Type_opts)
	    free_mime_param(&Type_opts);


	return ret;
}



const char * have_metamail()
{
    const char * metamail_value = give_dt_estr_as_str(&metamail_path_e, 
						      "metamail",NULL,NULL);
    const char * return_value   = metamail_value;
		         
    if (!metamail_value)
	return NULL;
    
    if (strcmp(metamail_value,"none") == 0 || 
	metamail_value[0] == '\0') {
	return_value = NULL;
    } else if (metamail_value[0] == '/') {
	if (-1 == access(metamail_value,EXECUTE_ACCESS)) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, ElmSet,ElmCantMetamail,
			      "Can't execute metamail: %s: %s"),
		      metamail_value,
		      strerror(err));
	    DPRINT(Debug,6, (&Debug, 
			     "have_metamail: no access %s: %s\n",
			     metamail_value,
			     strerror(err)));
	    sleep_message();
	    return_value = NULL;
	}
    }
    
    DPRINT(Debug,5,(&Debug,   
		    "have_metamail=%s\n",
		    return_value ? return_value : "<NULL>"));
    return return_value;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
