static char rcsid[] = "@(#)$Id: messages.c,v 2.20 2021/01/10 15:47:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.20 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_messages.h"

DEBUG_VAR(Debug,__FILE__,"messages");

/* SHRT_MAX LONG_MAX */
#include <limits.h>

#if ANSI_C
#define E_(x) extern x;
#else
#define E_(x)
#endif

static struct MailboxView    ** mailbox_list = NULL;
static int                      mailbox_list_len = 0;

struct MailboxView    * give_next_open_mailbox(idx, signal)
     int *idx;
     int signal;
{
    int i UNUSED_VAROK = *idx;

    struct MailboxView    *ret = NULL;

    do {
	if (*idx >= mailbox_list_len || *idx < 0) {
	    SIGDPRINT(Debug,7,(&Debug,
			       "give_next_open_mailbox[%d->%d] = NULL\n",
			       i,*idx));
	    
	    return NULL;
	}
	    
	if (mailbox_list[*idx]) {
	    
	    if (mailbox_list[*idx]->magic !=  MAILBOXVIEW_magic)
		panic("MBX VIEW PANIC",__FILE__,__LINE__,"malloc_view",
		      "Bad type magic number",signal);


	    ret = mailbox_list[*idx];
	}

	(*idx)++;

    } while (!ret);

    SIGDPRINT(Debug,7,(&Debug,
		       "give_next_open_mailbox[%d->%d] = %p\n",
		       i,*idx,ret));

    return ret;
}

struct MailboxView * malloc_view(t)
     struct mailbox_type *t;
{
    int idx;

    struct MailboxView *ret, **X;

    ret = safe_zero_alloc(sizeof (*ret));

    ret->magic         = MAILBOXVIEW_magic;
    ret->mailbox_type  = t;
    ret->status_type   = &status_common;   /* Caller may override */
    ret->u.dummy       = NULL;

    ret->current       = 0;
    ret->selected      = 0;
    ret->mailbox_title = NULL;

    ret->view          = NULL;
    ret->view_len      = 0;

    ret->thread_view   = NULL;
    ret->today_info    = NULL;

    ret->separators     = NULL;
    ret->separators_len = 0;

    if (ret->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"malloc_view",
	      "Bad type magic number",0);

    ret->mailbox_type->mt_init_it(ret);

    for (idx = 0; idx < mailbox_list_len; idx++)
	if (!mailbox_list[idx]) {
	    DPRINT(Debug,7,(&Debug,
			       "malloc_view: Re-using mailbox index %d\n",
			       idx));
	    goto found;
	}

    X = safe_array_realloc(mailbox_list,(mailbox_list_len+1), sizeof (*X));
    X[mailbox_list_len] = NULL;
    mailbox_list = X;
    idx = mailbox_list_len++;

    DPRINT(Debug,7,(&Debug,
		       "malloc_view: Allocing new mailbox index %d\n",
		       idx));

 found:
    mailbox_list[idx] = ret;

    return ret;
}

E_(mt_add_mailbox_storage NO_add_mailbox_storage)
void NO_add_mailbox_storage(mailbox,storage)
     struct MailboxView *mailbox;
     struct current_storage *storage;
{
    panic("MBX VIEW PANIC",__FILE__,__LINE__,"NO_add_mailbox_storage",
	  "NO_add_mailbox_storage called",0);
}

int query_mailbox_flag(mailbox,q)
     struct MailboxView *mailbox;
     enum query_mailbox q;
{
    int r = 0;

    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"query_mailbox_flag",
	      "Bad magic number",0);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"query_mailbox_flag",
	      "Bad type magic number",0);

    switch (q) {
    case qf_can_add_storage:
	r = (NO_add_mailbox_storage != mailbox->mailbox_type->mt_add_it_storage);
	break;
    }

    return r;
}

void add_storage(mailbox,storage)
     struct MailboxView *mailbox;
     struct current_storage *storage;
{
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"add_storage",
	      "Bad magic number",0);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"add_storage",
	      "Bad type magic number",0);

    mailbox->mailbox_type->mt_add_it_storage(mailbox,storage);
}

void add_digest(mailbox,list,received_time,env_from,F,defcharset)
     struct MailboxView *mailbox;
     mime_t *list;
     time_t received_time;
     char *env_from;
     FILE *F;
     charset_t defcharset;
{
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"add_storage",
	      "Bad magic number",0);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"add_storage",
	      "Bad type magic number",0);

    if (list->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"add_digest",
		   "Bad magic number");

    mailbox->mailbox_type->mt_add_it_digest(mailbox,list,received_time,
					    env_from,F, defcharset);
}	

static void free_menu_separator P_((struct menu_separator **sep));
static void free_menu_separator(sep)
     struct menu_separator **sep;
{
    if (MENU_SEPARATOR_magic != (*sep)->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"free_menu_separator",
	      "Bad magic number (menu_separator)",0);

    (*sep)->magic = 0; /* Invalidate */
    free(*sep);
    *sep = NULL;
}

static void free_today_info P_((struct today_info ** today_info));
static void free_today_info(today_info)
     struct today_info ** today_info;
{
    if (TODAY_INFO_magic != (*today_info)->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"free_today_info",
	      "Bad magic number (menu_separator)",0);

    (*today_info)->magic = 0;  /* Invalidate */
    free(*today_info);
    *today_info = NULL;
}

static int free_mailbox1 P_((struct MailboxView **mailbox,
			      struct cancel_data *cd));

static int free_mailbox1(mailbox,cd)
     struct MailboxView **mailbox;
     struct cancel_data *cd;
{
    int ret = 0;
    
    if ((*mailbox)->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"free_mailbox1",
	      "Bad type magic number",0);
    
    ret = (*mailbox)->mailbox_type->mt_free_it(*mailbox,cd);
    if (!ret) {
	 DPRINT(Debug,10,(&Debug,
			  "free_mailbox1: ...->mt_free_it failed\n"));
    }
    
    if ((*mailbox)->mailbox_title) {
	DPRINT(Debug,10,(&Debug,
			 "free_mailbox1: mailbox_title=%S\n",
			 (*mailbox)->mailbox_title));
	free_string (& ((*mailbox)->mailbox_title));
    }
	
    if ((*mailbox)->view) {
	free((*mailbox)->view);
        (*mailbox)->view = NULL;
    }
    (*mailbox)->view_len = 0;

    if ((*mailbox)->thread_view)
	free_thread_view(& ((*mailbox)->thread_view));

    if ((*mailbox)->today_info)
	free_today_info(&  ((*mailbox)->today_info));
    
    if ((*mailbox)->separators) {
	int i;

	for (i = 0; i < (*mailbox)->separators_len; i++)
	    if ((*mailbox)->separators[i])
		free_menu_separator(& ((*mailbox)->separators[i]));
	
	free((*mailbox)->separators);
	(*mailbox)->separators = NULL;
    }
    (*mailbox)->separators_len = 0;
    
    (*mailbox)->magic = 0;    /* Invalidate */
    free(*mailbox);
    *mailbox = NULL;

    DPRINT(Debug,10,(&Debug,
		    "free_mailbox1=%d\n",
		    ret));

    
    return ret;
}

int  free_mailbox(mailbox,cd)
     struct MailboxView **mailbox;
     struct cancel_data *cd;
{
    int ret = 0;
    int i;

    if ((*mailbox)->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"free_mailbox",
	      "Bad magic number",0);

    for ( i = 0; i < mailbox_list_len; i++) {
	if (mailbox_list[i] == *mailbox) {

	    DPRINT(Debug,7,(&Debug,
			    "free_mailbox: Mailbox index %d goes free\n",
			    i));
	    
	    mailbox_list[i] = NULL;
	    goto okei;
	}
    }

    panic("MBX VIEW PANIC",__FILE__,__LINE__,"free_mailbox",
	  "mailbox not found from list",0);
	
 okei:
    ret = free_mailbox1(mailbox,cd);

    DPRINT(Debug,7,(&Debug,
		    "free_mailbox=%d\n",
		    ret));
    return ret;
}

void free_all_mailboxes(cd)
     struct cancel_data *cd;
{
    if (mailbox_list) {
	int i;
	
	for ( i = 0; i < mailbox_list_len; i++) {
	    if (mailbox_list[i]) {
		
		SIGDPRINT(Debug,7,(&Debug,"free_all_mailboxes: [%d]\n",
				   i));
		
		if (mailbox_list[i]->magic         != MAILBOXVIEW_magic)
		    panic("MBX VIEW PANIC",__FILE__,__LINE__,"free_all_mailboxes",
			  "Bad magic number",0);
		
		free_mailbox1(& (mailbox_list[i]),cd);
	    }
	}
	
	free(mailbox_list);
	mailbox_list = NULL;
    }
    mailbox_list_len = 0;
}

static time_t calculate_fallback P_((time_t gmttime,int tz_mins));
static time_t calculate_fallback(gmttime,tz_mins)
     time_t gmttime;
     int tz_mins;
{
    if (0 == tz_mins)
	return gmttime;
    else if ((tz_mins > 0 &&
	      tz_mins*60 < gmttime)
	     ||
	     (tz_mins < 0 &&
	      time_MAX +  tz_mins*60 > gmttime)
	      )
	return gmttime - tz_mins*60;
    else {
	DPRINT(Debug,1,(&Debug,
			"message_hdr_setup_init: overflow on time of day, startgmt=%ld, offset = %ld\n",
			(long)gmttime,
			(long)(tz_mins*60)
			));
	return 0;
    }
}

int message_hdr_setup_init(mailbox,list)
     struct MailboxView * mailbox;
     struct menu_param  * list;
{
    int r = 0;
    int use_today = 0;

    static int report_time_MAX = 1;
    
    if (report_time_MAX) {
	DPRINT(Debug,1,(&Debug,
			"Using time_MAX = %ld\n",
			(long)time_MAX));
	report_time_MAX = 0;
    }
        
    if (MAILBOXVIEW_magic != mailbox->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"message_hdr_setup_init",
	      "Bad magic number (MailboxView)",0);
    
    if (MAILBOXTYPE_magic != mailbox->mailbox_type->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"message_hdr_setup_init",
	      "Bad type magic number",0);
    
    if ((use_today = dt_flag_is_set(&mail_menu_time,mmtime_today_flag)) ||
	dt_flag_is_set(&mail_menu_time,(mmtime_todaysep_flag))
	) {
	time_t T      = time(NULL);
	struct tm *A = NULL;

	if (mailbox->today_info) {
	    if (TODAY_INFO_magic != mailbox->today_info->magic)
		panic("MBX VIEW PANIC",__FILE__,__LINE__,"message_hdr_setup_init",
		      "Bad magic number (today_info)",0);
	} else {
	    mailbox->today_info = safe_zero_alloc(sizeof (*(mailbox->today_info)));
	    mailbox->today_info->magic = TODAY_INFO_magic;
	}

	/* get_tz_mins()   gives gmtime - localtime,
	   but offset, in minutes, EAST of GMT is used
	*/
	
	mailbox->today_info->tz_mins = -get_tz_mins(T);

	DPRINT(Debug,16,(&Debug,
			 "message_hdr_setup_init: tz_mins=%d\n",
			 mailbox->today_info->tz_mins));
	
	A = localtime(&T);

	if (A) {
	    struct tm tmbuf UNUSED_VAROK = *A;
	    
	    char *c UNUSED_VAROK = asctime(A);

	    if (c) {
		/* asctime() includes \n */
		DPRINT(Debug,16,(&Debug,
				 "message_hdr_setup_init: now is %s",c));
	    }
	    
	    mailbox->today_info->tm_mday = A->tm_mday;
	    mailbox->today_info->tm_mon  = A->tm_mon;
	    mailbox->today_info->tm_year = A->tm_year;

	    DPRINT(Debug,16,(&Debug,
			     "message_hdr_setup_init: tm_mday=%d (1-31), tm_mon=%d (0-11), tm_year=%d (Year - 1900)\n",
			     mailbox->today_info->tm_mday,
			     mailbox->today_info->tm_mon,
			     mailbox->today_info->tm_year));

	    mailbox->today_info->today_start_time = 0;
	    mailbox->today_info->today_end_time   = 0;
	    
#ifdef MKTIME

	    /* It is unclear how this behaves if 
	        timezone offset changes during day 
		(daylight time versus normal time)
	    */
	    
	    tmbuf.tm_sec  = 0;
	    tmbuf.tm_min  = 0;
	    tmbuf.tm_hour = 0;
	    tmbuf.tm_isdst = -1;   /* daylight saving time information not available */
	    
	    c = asctime(&tmbuf);
	    if (c) {
		DPRINT(Debug,16,(&Debug,
				 "message_hdr_setup_init: start of day %s",c));
	    }
	    mailbox->today_info->today_start_time = mktime(&tmbuf);
	    if ((time_t)-1 == mailbox->today_info->today_start_time) {
		DPRINT(Debug,16,(&Debug,
				 "message_hdr_setup_init: mktime failed for start of day\n"));
	    } else {
		DPRINT(Debug,16,(&Debug,
				 "message_hdr_setup_init: mktime for start of day succeed\n"));

	    }

	    /* It is unclear how this behaves if 
	       timezone offset changes during day 
	       (daylight time versus normal time)
	    */

	    tmbuf.tm_sec  = 59;
	    tmbuf.tm_min  = 59;
	    tmbuf.tm_hour = 23;
	    tmbuf.tm_isdst = -1;  /* daylight saving time information not available */

	    c = asctime(&tmbuf);
	    if (c) {
		DPRINT(Debug,16,(&Debug,
				 "message_hdr_setup_init: end of day %s",c));
	    }
	    mailbox->today_info->today_end_time = mktime(&tmbuf);
	    if ((time_t)-1 == mailbox->today_info->today_end_time) {
		DPRINT(Debug,16,(&Debug,
				 "message_hdr_setup_init: mktime failed for end of day\n"));
	    } else {
		DPRINT(Debug,16,(&Debug,
				 "message_hdr_setup_init: mktime for end of day succeed\n"));
	    }
#endif

	    /* Fallback if MKTIME is not defined */
	    if (! mailbox->today_info->today_start_time ||
		(time_t)-1 == mailbox->today_info->today_start_time) {
		
		/* This does not take account if timezone offset changes during day 
		   (daylight time versus normal time)

		   make_gmttime() retuns long,
		   LONG_MAX or LONG_MIN on overflow
		*/
		long S = make_gmttime(1900+tmbuf.tm_year,
				      1+tmbuf.tm_mon,
				      tmbuf.tm_mday,
				      0,0,0);

		if (S > 0 && S < LONG_MAX) {
		    
		    DPRINT(Debug,16,(&Debug,
				     "message_hdr_setup_init: start of day on GMT is %ld",
				     S));

		    if (S < time_MAX) {
			time_t startgmt = S;
			
			c = ctime(&startgmt);
			if (c) {
			    DPRINT(Debug,16,(&Debug," = %s",
					     c));
			} else {
			    DPRINT(Debug,16,(&Debug,"\n"));
			}

			mailbox->today_info->today_start_time =
			    calculate_fallback(startgmt,
					       mailbox->today_info->tz_mins);

		    } else {
			DPRINT(Debug,16,(&Debug," .. overflow\n"));
		    }
		} else {
		    DPRINT(Debug,16,(&Debug,
				     "message_hdr_setup_init: start of day on GMT gives %ld -- failed\n",
				     S));
		}
	    }
	    
	    /* Fallback if MKTIME is not defined */
	    if (! mailbox->today_info->today_end_time ||
		(time_t)-1 == mailbox->today_info->today_end_time) {
		
		/* This does not take account if timezone offset changes during day 
		   (daylight time versus normal time)

		   make_gmttime() retuns long,
		   LONG_MAX or LONG_MIN on overflow
		*/

		long E = make_gmttime(1900+tmbuf.tm_year,
				      1+tmbuf.tm_mon,
				      tmbuf.tm_mday,
				      23,59,59);

		if (E > 0 && E < LONG_MAX) {
				   		    
		    DPRINT(Debug,16,(&Debug,
				     "message_hdr_setup_init: end of day on GMT is %ld",
				     E));
		    if (E < time_MAX) {
			time_t endgmt = E;
			
			c = ctime(&endgmt);
			if (c) {
			    DPRINT(Debug,16,(&Debug," = %s",
					     c));
			} else {
			    DPRINT(Debug,16,(&Debug,"\n"));
			}

			mailbox->today_info->today_end_time =
			    calculate_fallback(endgmt,
					       mailbox->today_info->tz_mins);

		    } else {
			DPRINT(Debug,16,(&Debug," .. overflow\n"));
		    }
		    
		} else {
		    DPRINT(Debug,16,(&Debug,
				     "message_hdr_setup_init: end of day on GMT gives %ld -- failed\n",
				     E));
		}			    		
	    }

	    if (mailbox->today_info->today_start_time &&
		(time_t)-1 != mailbox->today_info->today_start_time) {
		
		DPRINT(Debug,16,(&Debug,
				 "message_hdr_setup_init: local start of day is %ld",
				 (long)(mailbox->today_info->today_start_time)));
		c = ctime(& (mailbox->today_info->today_start_time));
		if (c) {
		    DPRINT(Debug,16,(&Debug," = %s",
				     c));
		} else {
		    DPRINT(Debug,16,(&Debug,"\n"));
		}
	    }

	    if (mailbox->today_info->today_end_time &&
		(time_t)-1 != mailbox->today_info->today_end_time) {
		DPRINT(Debug,16,(&Debug,
				 "message_hdr_setup_init: local end of day is %ld",
				 (long)(mailbox->today_info->today_end_time)));
		c = ctime(& (mailbox->today_info->today_end_time));
		if (c) {
		    DPRINT(Debug,16,(&Debug," = %s",
				     c));
		} else {
		    DPRINT(Debug,16,(&Debug,"\n"));
		}
	    }

	    mailbox->rebuild_menu_date = use_today;
	    
	} else {
	    DPRINT(Debug,1,(&Debug,
			    "message_hdr_setup_init T=%ld -- localtime failed\n",
			    (long)T));
	    goto cleanup;
	}

	r = use_today;
    } else {
    cleanup:
	DPRINT(Debug,16,(&Debug,
			 "message_hdr_setup_init: No today_info\n"));
	
	if (mailbox->today_info)
	    free_today_info(& (mailbox->today_info));
    }

    if (list) {
	int time_menu_width = 6;
	int size_lines_width = 4;
	
	mp_list_set_integer(list,elm_mp_time_menu_width,time_menu_width);
	mp_list_set_integer(list,elm_mp_size_with,size_lines_width);

	
	DPRINT(Debug,14,(&Debug,
			 "message_hdr_setup_init: time_menu_width %d size_lines_width %d\n",
			 time_menu_width,size_lines_width));
	
	r = 1;
    }

    
    DPRINT(Debug,16,(&Debug,
		     "message_hdr_setup_init=%d\n",
		     r));
    
    return r;
}

void message_hdr_setup_line(mailbox,idx,list)
     struct MailboxView *mailbox;
     int idx;
     struct menu_param   * list;
{
    int use_today = 0;
    short this_year = 0;
    int  use_year_or_time = dt_flag_is_set(&mail_menu_time,mmtime_year_time_flag);
    enum time_menu_mode mode =
	use_year_or_time ? time_menu_month_day_year : time_menu_month_day;
    struct header_rec * h = NULL;
    
    if (MAILBOXVIEW_magic != mailbox->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"message_hdr_setup_line",
	      "Bad magic number (MailboxView)",0);
    
    if (MAILBOXTYPE_magic != mailbox->mailbox_type->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"message_hdr_setup_line",
	      "Bad type magic number",0);

    if (idx < 0 || idx >= mailbox->view_len)
	return;

    if (! mailbox->rebuild_menu_date && !list) {
	DPRINT(Debug,16,(&Debug,
			 "message_hdr_setup_line: %d: make_menu_date() already recalled or message_hdr_setup_init() not done\n",
			 idx));
	return;
    }
    
    h = mailbox->mailbox_type->mt_give_header_it(mailbox,idx,
						 & (mailbox->view[idx]));
    if (!h)
	return;
    
    if (mailbox->today_info) {
	if (TODAY_INFO_magic != mailbox->today_info->magic)
	    panic("MBX VIEW PANIC",__FILE__,__LINE__,"message_hdr_setup_line",
		  "Bad magic number (today_info)",0);
	
	use_today = dt_flag_is_set(&mail_menu_time,mmtime_today_flag);
	this_year = mailbox->today_info->tm_year+1900;

	if ( !h->time_menu_year ||
	     this_year == h->time_menu_year) {
	
	    if (use_today)
		mode = use_year_or_time ? time_menu_hhmm_or_year : time_menu_hhmm_24hour;
	    else if (use_year_or_time)
		mode = time_menu_month_day_hhmm;
	}
    }

    make_menu_date(h,mode,mailbox->today_info);

    DPRINT(Debug,16,(&Debug,
		     "message_hdr_setup_line: %d #%d: %S %d : make_menu_date() done\n",
		     idx,h->index_number_X,
		     h->time_menu_s,h->time_menu_year));
        
    if (list) {
	int time_menu_width = mp_lookup_integer(list,elm_mp_time_menu_width);
	int L = string_len(h->time_menu_s);

	if (L > time_menu_width) {
	    time_menu_width = L;
	    mp_list_set_integer(list,elm_mp_time_menu_width,time_menu_width);
		
	    DPRINT(Debug,14,(&Debug,
			     " ... message_hdr_setup_line idx #%d '%S' time_menu_width %d\n",
			     idx,h->time_menu_s,time_menu_width));
	}	
    }

    make_size_lines(h,mailbox,idx);

    DPRINT(Debug,16,(&Debug,
		     "message_hdr_setup_line: %d #%d: %S : make_size_lines() done\n",
		     idx,h->index_number_X,
		     h->size_lines_s));

    if (list) {
	int  size_lines_width = mp_lookup_integer(list,elm_mp_size_with);
	int L = string_len(h->size_lines_s);

	if (L > size_lines_width) {
	    size_lines_width = L;
	    
	    mp_list_set_integer(list,elm_mp_size_with,size_lines_width);

	    DPRINT(Debug,14,(&Debug,
			     " ... message_hdr_setup_line idx #%d '%S' size_lines_width %d\n",
			     idx,h->size_lines_s,size_lines_width));
	}
    }

    
}


static struct menu_separator *new_menu_separator P_((enum separator_type sep_type,
						     int   sep_value));
static struct menu_separator *new_menu_separator(sep_type,sep_value)
     enum separator_type sep_type;
     int                 sep_value;
{
    struct menu_separator *ret = safe_zero_alloc(sizeof (*ret));

    ret->magic     = MENU_SEPARATOR_magic;
    ret->sep_type  = sep_type;
    ret->sep_value = sep_value;
    
    return ret;
}

static int generate_separator_number P_((struct MailboxView *mailbox,
					 enum separator_type sep_type,
					 int                 sep_value));
static int generate_separator_number(mailbox,sep_type,sep_value)
     struct MailboxView *mailbox;
     enum separator_type sep_type;
     int                 sep_value;
{
    if (mailbox->separators) {
	int i;
	
	for (i = 0; i < mailbox->separators_len; i++) {

	    if (MENU_SEPARATOR_magic != mailbox->separators[i]->magic)
		panic("MBX VIEW PANIC",__FILE__,__LINE__,"generate_separator_number",
		      "Bad magic number (menu_separator)",0);
		
	    if (sep_type  == mailbox->separators[i]->sep_type &&
		sep_value == mailbox->separators[i]->sep_value)
		return i;
	}
    }

    mailbox->separators =
	safe_array_realloc(mailbox->separators,
			   mailbox->separators_len+1,
			   sizeof (mailbox->separators[0]));

    mailbox->separators[mailbox->separators_len] =
	new_menu_separator(sep_type,sep_value);

    return mailbox->separators_len++;
}

static void update_separators P_((struct MailboxView *mailbox));
static void update_separators(mailbox)
     struct MailboxView *mailbox;
{
    int i;
    int need_separator = 0;
    short this_year = 0;
    int use_today = 0;
    int today_sep = 0;
    int  use_year_or_time = dt_flag_is_set(&mail_menu_time,mmtime_year_time_flag);
    
    if (MAILBOXVIEW_magic != mailbox->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_separators",
	      "Bad magic number (MailboxView)",0);
    
    if (MAILBOXTYPE_magic != mailbox->mailbox_type->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_separators",
	      "Bad type magic number",0);

    if (mailbox->today_info) {
	if (TODAY_INFO_magic != mailbox->today_info->magic)
	    panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_separators",
		  "Bad magic number (today_info)",0);
	
	use_today = dt_flag_is_set(&mail_menu_time,mmtime_today_flag);
	this_year = mailbox->today_info->tm_year+1900;
	today_sep = dt_flag_is_set(&mail_menu_time,mmtime_todaysep_flag);

	DPRINT(Debug,16,(&Debug,
			 "update_separators: this_year=%d%s%s\n",
			 this_year,
			 use_today ? ", uses today" : "",
			 today_sep ? ", uses today-separator" : ""));
			 
	
    } else {
	time_t T      = time(NULL);
	struct tm *A = localtime(&T);

	if (A) {
	    this_year  = A->tm_year+1900;

	    DPRINT(Debug,16,(&Debug,
			     "update_separators: this_year=%d (no today info)\n",
			     this_year));
	} else {
	    DPRINT(Debug,1,(&Debug,
			    "update_separators T=%ld -- localtime failed\n",
			    (long)T));
	    need_separator = 1;
	}
    }

    if (this_year) {	       
	if (mailbox->rebuild_menu_date) {
	    DPRINT(Debug,16,(&Debug,
			     "update_separators: will recall make_menu_date()\n"));
	}
	
	/* Check that there is at least one mail other
	   that currect year
	*/
	for (i = 0; i < mailbox->view_len && ! need_separator; i++) {
	    
	    if (HEADER_NO_SEPARATOR ==  mailbox->view[i].separator_number) {
		struct header_rec * h =
		    mailbox->mailbox_type->mt_give_header_it(mailbox,i,
							     & (mailbox->view[i]));
		
		if (h) {
		    enum time_menu_mode mode =
			use_year_or_time ? time_menu_month_day_year : time_menu_month_day;

		    if ( !h->time_menu_year ||
			 this_year == h->time_menu_year) {
			if (use_today)
			    mode = use_year_or_time ? time_menu_hhmm_or_year : time_menu_hhmm_24hour;
			else if (use_year_or_time)
			    mode = time_menu_month_day_hhmm;
		    }
		    
		    if (! h->time_menu_year ||
			mode != h->time_menu_type ||
			mailbox->rebuild_menu_date ||
			today_sep			
			) {
			DPRINT(Debug,16,(&Debug,
					 "update_separators: %d #%d: mode %d => %d\n",
					 i,h->index_number_X,
					 h->time_menu_type,mode));
			make_menu_date(h,mode,mailbox->today_info);
			if (mode != h->time_menu_type) {
			    DPRINT(Debug,16,(&Debug,
					     "update_separators: ... mode => %d\n",
					     h->time_menu_type));
			}
		    }
		    
		    if (this_year != h->time_menu_year
			||
			( h->time_menu_today && use_today)
			||
			( h->time_menu_today && today_sep)
			)
			need_separator = 1;
		}
		
	    } else {
		need_separator = 1;

	    }
	}
    } else {
	need_separator = 1;
	DPRINT(Debug,1,(&Debug,
			"update_separators -- this_year not available\n"));
    }

    if (need_separator) {
	DPRINT(Debug,16,(&Debug,
			 "update_separators: needs separators, %d messages\n",
			 mailbox->view_len));
	
	for (i = 0; i < mailbox->view_len; i++) {
	    struct header_rec * h = NULL;
	    enum separator_type sep_type = sep_none;
	    
	    if (mailbox->view[i].separator_number >= 0 &&
		mailbox->view[i].separator_number < mailbox->separators_len &&
		mailbox->separators[mailbox->view[i].separator_number]) {

		if (MENU_SEPARATOR_magic != 	mailbox->separators[mailbox->view[i].separator_number]->magic)
		    panic("MBX VIEW PANIC",__FILE__,__LINE__,"generate_separator_number",
			  "Bad magic number (menu_separator)",0);

		if (sep_year ==
		    mailbox->separators[mailbox->view[i].separator_number]->sep_type &&
		    ! use_today &&
		    ! today_sep &&
		    ! mailbox->rebuild_menu_date
		    )
		    continue;

		sep_type = mailbox->separators[mailbox->view[i].separator_number]->sep_type;
	    }

	    h =	mailbox->mailbox_type->mt_give_header_it(mailbox,i,
							 & (mailbox->view[i]));
	    if (h) {
		enum time_menu_mode mode =
		    use_year_or_time ? time_menu_month_day_year : time_menu_month_day;

		if (sep_year == sep_type &&
		    ! h->time_menu_today &&
		    ! today_sep &&
		    ! mailbox->rebuild_menu_date
		    )
		    continue;

		if ( !h->time_menu_year ||
		     this_year == h->time_menu_year) {		    
		    if (use_today)
			mode = 	use_year_or_time ? time_menu_hhmm_or_year : time_menu_hhmm_24hour;
		    else if (use_year_or_time)
			mode = time_menu_month_day_hhmm;
		}
		    
		if (! h->time_menu_year ||
		    mode != h->time_menu_type ||
		    mailbox->rebuild_menu_date ||
		    today_sep)	{
		    DPRINT(Debug,16,(&Debug,
				     "update_separators: %d #%d: mode %d => %d\n",
				     i,h->index_number_X,
				     h->time_menu_type,mode));
		    make_menu_date(h,mode,mailbox->today_info);
		    if (mode != h->time_menu_type) {
			DPRINT(Debug,16,(&Debug,
					 "update_separators: ... mode => %d\n",
					 h->time_menu_type));
		    }
		}
		
		if (h->time_menu_today &&
		    ( use_today || today_sep)) {		    

		    mailbox->view[i].separator_number =
			generate_separator_number(mailbox,sep_today,0);

		     DPRINT(Debug,16,(&Debug,
				      "update_separators: %d #%d: %S %d : today separator => %d\n",
				      i,h->index_number_X,
				      h->time_menu_s,h->time_menu_year, mailbox->view[i].separator_number));
		    
		} else if (h->time_menu_year > 1900 &&
		    (!this_year || h->time_menu_year <= this_year)) {
		    mailbox->view[i].separator_number =
			generate_separator_number(mailbox,sep_year,
						  h->time_menu_year);

		    DPRINT(Debug,16,(&Debug,
				    "update_separators: %d #%d: %S %d : year separator => %d\n",
				     i,h->index_number_X,
				     h->time_menu_s,h->time_menu_year, mailbox->view[i].separator_number));
		    
		} else {
		    mailbox->view[i].separator_number =
			generate_separator_number(mailbox,sep_other,0);
		    
		    DPRINT(Debug,16,(&Debug,
				     "update_separators: %d #%d: %S %d : other separator => %d\n",
				     i,h->index_number_X,
				     h->time_menu_s,h->time_menu_year, mailbox->view[i].separator_number));


		}
		
	    } else
		mailbox->view[i].separator_number = HEADER_NO_SEPARATOR;
	    
	}

	if (mailbox->rebuild_menu_date) {
	    mailbox->rebuild_menu_date = 0;
	    DPRINT(Debug,16,(&Debug,
			     "update_separators: make_menu_date() recalled\n"));
	}
    } else {
	DPRINT(Debug,16,(&Debug,"update_separators: separators not needed\n"));
    }
}

/* Return 1 if redraw required */
int update_view(mailbox)
     struct MailboxView *mailbox;
{
    int ret;
    struct folder_view cur;
    int current_set = 0;

    zero_folder_view(& cur);

    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_view",
	      "Bad magic number",0);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"update_view",
	      "Bad type magic number",0);

    if (mailbox->current > 0 && mailbox->current <= mailbox->view_len) {
	cur = mailbox->view[mailbox->current-1];
	current_set++;
    }

    ret = mailbox->mailbox_type->mt_update_view_it(mailbox);

    if (ret && current_set) {
	int i;

	for (i = 0; i < mailbox->view_len; i++) {
	    if (cur.mailbox_number == mailbox->view[i].mailbox_number &&
		cur.index == mailbox->view[i].index) {
		DPRINT(Debug,7,(&Debug,
				"update_view: Changing current %d -> %d\n",
				mailbox->current,i+1));
		mailbox->current = i+1;
		break;
	    }
	}
    }

    if (ret && mailbox->separators_len > 0)
	update_separators(mailbox);

    DPRINT(Debug,7,(&Debug,
		    "update_view=%d\n",ret));
    return ret;
}

struct folder_info * get_main_folder(mailbox)
     struct MailboxView *mailbox;
{

    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_main_folder",
	      "Bad magic number",0);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_main_folder",
	      "Bad type magic number",0);

    return mailbox->mailbox_type->mt_get_main_it_folder(mailbox);
}

/* Can be called from signal handler */
struct current_storage * get_storage(mailbox,i)
     struct MailboxView *mailbox;
     int i;
{
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_storage",
	      "Bad magic number",1);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_storage",
	      "Bad type magic number",1);

    return mailbox->mailbox_type->mt_get_it_storage(mailbox,i);
}

/* Can be called from signal handler */
int get_storage_count(mailbox)
     struct MailboxView *mailbox;
{
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_storage_count",
	      "Bad magic number",1);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_storage_count",
	      "Bad type magic number",1);

    return mailbox->mailbox_type->mt_get_it_storage_count(mailbox);
}


void give_index_number(mailbox,index,res) 
     struct MailboxView *mailbox;
     int index; 
     struct folder_view *res;
{   
    zero_folder_view(res);

    if (index < 0 || index >= mailbox->view_len)
	return;

    *res = mailbox->view[index];
}

struct header_rec * give_header(mailbox,index)
     struct MailboxView *mailbox;
     int index;
{
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_header",
	      "Bad magic number",0);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_header",
	      "Bad type magic number",0);

    if (index < 0 || index >= mailbox->view_len)
	return NULL;


    return mailbox->mailbox_type->mt_give_header_it(mailbox,index,
						    & mailbox->view[index]);
}


/*  current == index+1     value is used only by caller */
int get_current(mailbox) 
     struct MailboxView *mailbox;
{
    return mailbox->current;
}

/*  current == index+1     value is used only by caller */
void set_current(mailbox,cur)
     struct MailboxView *mailbox;
     int cur;
{
    mailbox->current = cur;
}

int get_selected(mailbox)
     struct MailboxView *mailbox;
{
    return mailbox->selected;
}

void set_selected(mailbox,sel)
     struct MailboxView *mailbox;
     int sel;
{
    mailbox->selected = sel;
}

int get_message_count(mailbox)
     struct MailboxView *mailbox;
{
    return mailbox->view_len;
}

struct header_rec * give_header_s(s)
     struct sort_data *s;
{

    if (s->sort_data_type->magic != SORTDATATYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_header_s",
	      "Bad magic number",0);
	   
    return s->sort_data_type->sdt_give_header(s,& s->w);
}

void give_index_number_s(s,res)
     struct sort_data *s;
     struct folder_view *res;
{
    *res = s->w;
}

void sort_mailbox_view(mailbox,func)
     struct MailboxView *mailbox;
     hdr_compare_func   *func;
{

    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"sort_mailbox_view",
	      "Bad magic number",0);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"sort_mailbox_view",
	      "Bad type magic number",0);

    if (mailbox->view_len < 2)
	return;

    mailbox->mailbox_type->mt_sort_it_view(mailbox,func);

}

/* Return 1 if succeed, 0 is fails --
   seeks to {header}->offset   (if ret_F set)
*/
int give_message_data(mailbox,index,ret_header,ret_F,
		      counter,parse_mime)
     struct MailboxView *mailbox;
     int index;
     struct header_rec **ret_header;
     FILE              **ret_F;
     struct counter_data *counter;
     parse_mime_callback *parse_mime;
{
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_message_data",
	      "Bad magic number",0);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_message_data",
	      "Bad type magic number",0);

    if (index < 0 || index >= mailbox->view_len)
	return 0;

    if (ret_header)
	*ret_header = NULL;
    if (ret_F)
	*ret_F      = NULL;

    return mailbox->mailbox_type->
	mt_give_message_data_it(mailbox,index,
				ret_header,ret_F,
				counter,parse_mime,
				& mailbox->view[index]);
}

/* Return 1 if succeed, 0 is fails 
   Give message size for menu
*/

int give_message_menu_size(mailbox,index,ret_size)
     struct MailboxView * mailbox;
     int                  index;
     unsigned long      * ret_size;
{
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_message_menu_size",
	      "Bad magic number",0);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_message_menu_size",
	      "Bad type magic number",0);

    if (index < 0 || index >= mailbox->view_len)
	return 0;

    if (ret_size)
	*ret_size = 0;

    return mailbox->mailbox_type->
	mt_give_message_msize_it(mailbox,& mailbox->view[index],
				 ret_size);
}

/* index == -1    first folder remote server */
const struct remote_server * give_message_remote_server(mailbox,index)
     struct MailboxView *mailbox; 
     int index;   
{
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__," give_message_remote_server",
	      "Bad magic number",0);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__," give_message_remote_server",
	      "Bad type magic number",0);

    if (index < 0 || index >= mailbox->view_len)
	return mailbox->mailbox_type->
	    mt_give_message_remote_server_it(mailbox,index,NULL);
    else
	return mailbox->mailbox_type->
	    mt_give_message_remote_server_it(mailbox,index,
					     & mailbox->view[index]);
}

void write_mailbox_info(fp,mailbox)
     FILE *fp; 
     struct MailboxView *mailbox;
{
    int i;
    int count;

    int cur_s   = -1;
    int cur_idx = -1;
    int s;

    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"write_mailbox_info",
	      "Bad magic number",0);

    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"write_mailbox_info",
	      "Bad type magic number",0);
    
    count = mailbox->mailbox_type->mt_get_it_storage_count(mailbox);

    fprintf(fp, "S%d\n", count);
	    
    
    /* HACK */
    if (mailbox->current > 0 && mailbox->current <= mailbox->view_len) {
	cur_s   = mailbox->view[mailbox->current -1].mailbox_number;
	cur_idx = mailbox->view[mailbox->current -1].index;
    }
    
    for (s = 0; s < count; s++) {
	fprintf(fp, "=%d\n",s);

	if (s == cur_s)
	    mailbox->mailbox_type->mt_write_it_info(fp,mailbox,s,cur_idx);
	else
	    mailbox->mailbox_type->mt_write_it_info(fp,mailbox,s,-1);

    }

    fprintf(fp, "V%d\n", mailbox->view_len);
    
    for (i = 0; i < mailbox->view_len; i++) {
	fprintf(fp, "i%d %d %d\n",
		mailbox->view[i].mailbox_number,
		mailbox->view[i].index,
		mailbox->view[i].thread_number);
    }

    /* count up the number of tagged messages */
    count = 0;
    for (i = 0 ; i < mailbox->view_len ; i++)  {
	struct header_rec * hdr = give_header(mailbox,i);
	
	if (hdr &&
	    (hdr->status & TAGGED))
	    ++count;
    }
    
    /* write out selected messages */
    if (count > 0) {
	/* we found tagged messages - write them out */
	fprintf(fp, "C%d\n", count);
	for (i = 0 ; i < mailbox->view_len ; i++) {
	    struct header_rec * hdr = give_header(mailbox,i);

	    if (hdr && (hdr->status & TAGGED))
		fprintf(fp, "S%d\n", i+1);
	}
    } else if (mailbox->current > 0) {
	/* no tagged messages - write out the selected message */
	fprintf(fp, "C1\nS%d\n", mailbox->current);
    } else {
	/* hmmm...must be an empty mailbox */
	fprintf(fp, "C0\n");
    }    
}

/* caller must string_free() result -- calculated if not set */
struct string * mailbox_title(mailbox)
     struct MailboxView *mailbox;
{

    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"mailbox_title",
	      "Bad magic number",0);

    if (mailbox->mailbox_title) {
	return dup_string(mailbox->mailbox_title);
    }

    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"mailbox_title",
	      "Bad type magic number",0);

    return mailbox->mailbox_type->mt_it_title(mailbox);
}

/* must not freed -- shared/stored to structure
   -- set_mailbox_title will free_string()
   it later */
void set_mailbox_title(mailbox,value)
     struct MailboxView *mailbox;
     struct string *value;
{
    if (mailbox->mailbox_title) 
	free_string( & (mailbox->mailbox_title) );
    
    mailbox->mailbox_title = value;
}

const struct menu_separator * give_menu_separator(mailbox,sepindex)
     struct MailboxView * mailbox;
     int                  sepindex;
{
    if (MAILBOXVIEW_magic != mailbox->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_menu_separator",
	      "Bad magic number (MailboxView)",0);

    if (! mailbox->separators)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_menu_separator",
	      "No separators",0);

    if (sepindex < 0 || sepindex >=  mailbox->separators_len ||
	! mailbox->separators[sepindex])
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_menu_separator",
	      "Bad sepindex",0);

    if (MENU_SEPARATOR_magic != mailbox->separators[sepindex]->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"give_menu_separator",
	      "Bad magic number (menu_separator)",0);

    return mailbox->separators[sepindex];
}



int menu_separator_count(mailbox)
     struct MailboxView *mailbox;
{
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"menu_separator_count",
	      "Bad magic number",0);

    return mailbox->separators_len;
}

int menu_separator_init(mailbox,subpage_lines,subpage_columns)
     struct MailboxView * mailbox;     
     int                  subpage_lines;
     int                  subpage_columns;
{
    int r = 0;
    
    int yseparator_flag = 0;
    int aseparator_flag = 0;
    int todaysep_flag  = 0;
    int badsort = 0;
    
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"menu_separator_init",
	      "Bad magic number",0);

    DPRINT(Debug,16,(&Debug,
		     "menu_separator_init: subpgate_lines=%d, subpage_columns=%d\n",
		     subpage_lines,subpage_columns));

    
    if ((yseparator_flag = dt_flag_is_set(&mail_menu_time,mmtime_yseparator_flag))) {
	update_separators(mailbox);
    } else if (((aseparator_flag = dt_flag_is_set(&mail_menu_time,mmtime_aseparator_flag)) &&
		subpage_lines > 20)
	       ||
	       ((todaysep_flag = dt_flag_is_set(&mail_menu_time,(mmtime_todaysep_flag))) &&
		subpage_lines > 5)
	       ) {

	switch (abs(give_dt_sort_as_int(&sortby))) {
	case MAILBOX_ORDER:
	case SENT_DATE:
	case RECEIVED_DATE:
	    update_separators(mailbox);
	    break;
	default:
	    badsort = 1;
	    goto out;
	}
    } else
	goto out;
        
    r = mailbox->separators_len > 0;

 out:

    DPRINT(Debug,16,(&Debug,
		     "menu_separator_init=%d"));
    if (mailbox->separators_len) {
	DPRINT(Debug,16,(&Debug,", separators_len=%d",
			 mailbox->separators_len));
    }
    if (yseparator_flag) {
	DPRINT(Debug,16,(&Debug,", used year-separator"));
    }
    if (aseparator_flag) {
	DPRINT(Debug,16,(&Debug,", used auto-separator"));
    }
    if (todaysep_flag) {
	DPRINT(Debug,16,(&Debug,", used today-separator"));
    }
    if (badsort) {
	DPRINT(Debug,16,(&Debug,", sorting order not good for separators"));
    }
    
    DPRINT(Debug,16,(&Debug,"\n"));
    
    return r;
}

int get_menu_separator_index(mailbox,index)
     struct MailboxView *mailbox;
     int index;
{
    if (mailbox->magic         != MAILBOXVIEW_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_menu_separator_index",
	      "Bad magic number",0);
    
    if (mailbox->mailbox_type->magic != MAILBOXTYPE_magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_menu_separator_index",
	      "Bad type magic number",0);

    if (index < 0 || index >= mailbox->view_len)
	return HEADER_NO_SEPARATOR;

    if (mailbox->view[index].separator_number >= 0 &&
	mailbox->view[index].separator_number < mailbox->separators_len &&
	mailbox->separators[mailbox->view[index].separator_number]) {

	if (MENU_SEPARATOR_magic != 	mailbox->separators[mailbox->view[index].separator_number]->magic)
	    panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_menu_separator_index",
			  "Bad magic number (menu_separator)",0);

	return mailbox->view[index].separator_number;
    }
    
    return HEADER_NO_SEPARATOR;
}

const struct today_info * get_today_info(mailbox) 
     struct MailboxView *mailbox;
{
    if (MAILBOXVIEW_magic != mailbox->magic)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_today_info",
	      "Bad magic number (MailboxView)",0);

    if (mailbox->today_info) {
	if (TODAY_INFO_magic != mailbox->today_info->magic)
	    panic("MBX VIEW PANIC",__FILE__,__LINE__,"get_today_info",
		  "Bad magic number (today_info)",0);

	return mailbox->today_info;
    }

    return NULL;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
