static char rcsid[] = "@(#)$Id: mailfile.c,v 2.8 2021/10/23 15:05:09 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.8 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_messages.h"

#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"messages");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

/* For INT_MAX */
#include <limits.h>

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}


/*
 * Files given with
 *     --mail-file=
 * option.
 *
 * Acts either as mailbox or attachments
 *
 */

#include <errno.h> 
#ifndef ANSI_C
extern int errno;
#endif

static struct mail_file_info {

    char * filename;

    FILE * f;

    header_list_ptr      header_list;
    struct header_rec  * header_rec;

    long                 content_start;

    unsigned             header_list_incomplete:1;
    unsigned             binary:1;
    
} * mail_file_list;
static int mail_file_list_len;

static enum mail_file_mode  mode = no_mail_files_given;

int add_mail_file(mailfile)
    const char *mailfile;
{
    if (access(mailfile,READ_ACCESS) < 0) {
	int err = errno;
	lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileFailedRead,
			  "Failed to read mail-file %s: %s"),
		  mailfile,strerror(err));
	return 0;
    }
    
    /* Actual checking is postponed to check_mail_file_list()
     */

    mail_file_list = safe_array_realloc(mail_file_list,
					mail_file_list_len+1,
					sizeof (mail_file_list[0]));
    
    mail_file_list[mail_file_list_len].filename = safe_strdup(mailfile);
    mail_file_list[mail_file_list_len].f = NULL;
    mail_file_list[mail_file_list_len].header_rec  = NULL;
    mail_file_list[mail_file_list_len].header_list = NULL;
    mail_file_list[mail_file_list_len].content_start = -1;
    mail_file_list[mail_file_list_len].header_list_incomplete = 1;
    mail_file_list[mail_file_list_len].binary        = 0;
    
    mail_file_list_len++;

    if (no_mail_files_given == mode)
	mode = mail_files_added;

    return 1;
}

void free_mail_file_list()
{
    if (mail_file_list) {
	int i;

	for (i = 0; i < mail_file_list_len; i++) {
	    if (mail_file_list[i].filename) {
		free(mail_file_list[i].filename);
		mail_file_list[i].filename = NULL;
	    }
	    if (mail_file_list[i].f) {
		fclose(mail_file_list[i].f);
		mail_file_list[i].f = NULL;
	    }
	    if (mail_file_list[i].header_rec)
		header_free(& (mail_file_list[i].header_rec));
	    if (mail_file_list[i].header_list)
		delete_headers(& (mail_file_list[i].header_list));	    
	}

	free(mail_file_list);
	mail_file_list = NULL;

	mode = no_mail_files_given;
    }

}

static const struct char_state {
    
    int            lineno;
    
    unsigned  begin_of_line    :1;
    unsigned  binary           :1;
} NULL_char_state = {
    0,   /* first character on file increments lineno to 1 */
    1,   /* on begin of line if nothing is read            */
    1,   /* assume binary message                          */ 
};

static enum char_status {
    char_error = -2,
    char_EOF   = -1,
    char_end_of_headers = 0,
    char_begin_of_header = 1,
    char_begin_of_continuation = 2,
    char_other = 3
} get_next_char P_((const char *mailfile,
		    FILE *f,
		    struct char_state *state,
		    char *result));

static enum char_status get_next_char(mailfile,f,state,result)
     const char        * mailfile;
     FILE              * f;
     struct char_state * state;
     char              * result;
{
    int CR_seen = 0;
    
    /* Look header line */
    int c = fgetc(f);

    *result = '\0';

    switch(c) {
    case EOF:
	if (ferror(f)) {
	    int err = errno;
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileFailedRdLineNo,
			      "Failed to read mail-file %s:%d: %s"),
		      mailfile,state->lineno,strerror(err));
	    
	    return char_error;
	}
	
	if (feof(f)) {
	    if (! state->begin_of_line) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileUnexpEOFLineNo,
				  "Unexpected end of file on mail-file %s:%d"),
			  mailfile,state->lineno);

		return char_error;
	    }

	    return char_EOF;
	}
	
	break;

    case '\0':
	if (state->begin_of_line) 
	    state->lineno++;
	
	lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileNUL,
			  "NUL char on mail-file %s:%d"),
		  mailfile,state->lineno);

	state->begin_of_line = 0;
	return char_error;
	
    case '\r':

	if (state->begin_of_line) 
	    state->lineno++;

	CR_seen = 1;
	c = fgetc(f);

	if ('\n' == c)
	    goto got_nl;
	if (EOF == c) {
	    if (ferror(f)) {
		int err = errno;
		
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileFailedRdLineNo,
				  "Failed to read mail-file %s:%d: %s"),
			  mailfile,state->lineno,strerror(err));
		
		return char_error;
	    }

	    if (feof(f)) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileUnexpEOFLineNo,
				  "Unexpected end of file on mail-file %s:%d"),
			  mailfile,state->lineno);

		return char_error;
	    }
	}
	
	lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileStaleCR,
			  "Stale CR on mail-file %s:%d"),
		  mailfile,state->lineno);
	
	state->begin_of_line = 0;
	*result = c;
	return char_error;
       	
    case '\n':

	if (state->begin_of_line) 
	    state->lineno++;
	
    got_nl:

	if (!CR_seen && state->binary) {
	    DPRINT(Debug,12,(&Debug,"get_next_char: Seen NL without CR, resetting binary flag\n"));
	    state->binary = 0;
	}
	
	if (state->begin_of_line) {
	    
	    *result = c;
	    return char_end_of_headers;
	}
	
	state->begin_of_line = 1;
	*result = c;
	return char_other;

    case ' ':
    case '\t':

	if (state->begin_of_line) {
	    state->lineno++;
	    
	    if (state->lineno < 1) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileStartSpace,
				  "Unexpected space on beginning of mail-file %s:%d"),
			  mailfile,state->lineno);

		state->begin_of_line = 0;
		*result = c;
		return char_error;
	    }
	    
	    state->begin_of_line = 0;
	    *result = c;
	    return char_begin_of_continuation;
	}

	state->begin_of_line = 0;
	*result = c;
	return char_other;

    case ':':
	
	if (state->begin_of_line) {
	    state->lineno++;
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileStartColon,
			      "Unexpected colon(:) on beginning of line at mail-file %s:%d"),
		      mailfile,state->lineno);
	    
	    state->begin_of_line = 0;
	    *result = c;
	    return char_error;
	}

	*result = c;
	return char_other;	
    	    
    default:

	if (state->begin_of_line) {
	    state->lineno++;
	    
	    if (isascii(c) && isprint(c)) {
		state->begin_of_line = 0;
		*result = c;
		return char_begin_of_header;
	    }

	    lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileStartUnexp,
			      "Unexpected character 0x%02x on beginning of line at mail-file %s:%d"),
		      (unsigned int)c,mailfile,state->lineno);

	    state->begin_of_line = 0;
	    *result = c;
	    return char_error;
	}
		
	*result = c;
	return char_other;	
    }

    return char_error;
}

#define MAX_HEADER_FIELD_NAME_LEN 80
#define MAX_FOLDED_HDR 1024

static int check_open_mailfile P_((int idx));
static int check_open_mailfile(idx)
     int idx;
{
    const char *mailfile = mail_file_list[idx].filename;
    FILE * f             = mail_file_list[idx].f;
    int err              = can_open(mailfile,"r");
    int ret = 0;
    int header_ok = 0;
    int header_list_incomplete = 0;
    
    struct char_state state = NULL_char_state;
    char name[MAX_HEADER_FIELD_NAME_LEN+1] = "<begin>";
    header_list_ptr header_item            = NULL;
    header_list_ptr last_header            = NULL;
    int x                                  = 7;
    
    char header_body[MAX_FOLDED_HDR+1]     = "";
    int  y = 0;
    
    if (0 == err) {
	if (f) {
	    DPRINT(Debug,12,(&Debug,"check_open_mailfile: Opened mail-file (%d) -- already open\n",
			     idx));
	    rewind(f);
	} else {		    
	    f = fopen(mailfile,"r");
	    
	    if (f) {
		DPRINT(Debug,12,(&Debug,"check_open_mailfile: Opened mail-file (%d): %s\n",
				 idx, mailfile));
		
		mail_file_list[idx].f = f;
	    } else {
		err = errno;
	    }
	} 
    }
    
    if (!f || err) {
	
	lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileFailedOpen,
			  "Failed to open mail-file %s: %s"),
		  mailfile,strerror(err));
	return 0;
    }
    
    while (!ret) {
	/* On beginning of line */
	
	char c = '\0';
	enum char_status s =  get_next_char(mailfile,f,&state,&c);
	int init_header = 0;
	
	switch (s) {
	    header_ptr h;
	    
	case char_error:       return 0;
	case char_EOF:
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileGotEOF,
			      "Got end of file before end of header (block) on mail-file %s:%d"),
		      mailfile,state.lineno);
	    return 0;
	case char_end_of_headers:
	    ret = 1;
	    mail_file_list[idx].content_start = ftell(f);
	    
	    DPRINT(Debug,12,(&Debug,"check_open_mailfile: Opened mail-file #%d: content start %ld\n",
			     idx,mail_file_list[idx].content_start));
	    break;
	case char_begin_of_header:
	    x = 0;
	    
	    do {
		name[x++] = c;
		
		s =  get_next_char(mailfile,f,&state,&c);
	    } while (x < (sizeof name) -1 &&
		     char_other == s &&
		     '\n' != c &&
		     ':' != c &&
		     !whitespace(c));
	    
	    name[x] = '\0';
	    
	    if (x > MAX_HEADER_FIELD_NAME_LEN) {		
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileTooLongeHdrFldName,
				  "Too long (max %d) header field name on mail-file %s:%d: %.*s"),
			  MAX_HEADER_FIELD_NAME_LEN,
			  mailfile,state.lineno,
			  x,name);
		return 0;
	    }
	    
	    while (char_other == s &&
		   whitespace(c)) {
		s =  get_next_char(mailfile,f,&state,&c);
	    }
	    
	    if (char_other != s ||
		':' != c) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileInvHeaderField,
				  "Invalid header field on mail-file %s:%d: %.*s"),
			  mailfile,state.lineno,
			  x,name);
		return 0;
	    }
	    
	    
	    /* Because check_mail_file_list() is called on 
	       init, only static header list is known. New
	       headers found from mailboxes are not
	       yet known for find_header()
	    */
		    
	    h = find_header(name,
			    0   /* do not create new header name */
			    );
	    if (h) {
		header_ok = 1;
		header_item =
		    update_header_list1(& (mail_file_list[idx].header_list),
					&  last_header,h);
		
	    } else {
		/* All instrested headers are on static list
		   anyway 
		*/

		DPRINT(Debug,12,(&Debug,"check_open_mailfile: Opened mail-file #%d: unknown header field name: %*.s\n",
				 idx,x,name));
		
		header_list_incomplete = 1;
		header_item = NULL;
	    }

	    /* skip space afer : -character */		

	    do {
		s =  get_next_char(mailfile,f,&state,&c);
	    } while (char_other == s && whitespace(c));
	    

	    if (char_other != s) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileInvHeaderField,
				  "Invalid header field on mail-file %s:%d: %.*s"),
			  mailfile,state.lineno,
			  x,name);
		return 0;
	    }

	    init_header = 1;

	    if ('\n' == c) {
		y = 0;
		header_body[y] = '\0';

		DPRINT(Debug,12,(&Debug,"check_open_mailfile: Opened mail-file #%d: empty body on header field: %*.s\n",
				 idx,x,name));
		goto no_header_body;
	    }

	    /* FALLTHRU */
	    
	case char_begin_of_continuation:
	    
	    y = 0;
	    do {
		if (0 == y && !init_header && whitespace(c)) {

		    /* Header folding represens one space even when TAB
		       is used
		    */
		    c = ' ';
	
		} 

		header_body[y++] = c;

		s =  get_next_char(mailfile,f,&state,&c);
		
	    } while (y < (sizeof header_body) -1 &&
		     char_other == s &&
		     '\n' != c);
	    
	    header_body[y] = '\0';
	    if (y > MAX_FOLDED_HDR) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileTooLongeHdrFldLine,
				  "Too long (max %d) header field line on mail-file %s:%d: %.*s"),
			  MAX_FOLDED_HDR,
			  mailfile,state.lineno,
			  x,name);
		return 0;
	    }

	    if (0 == y) {
		DPRINT(Debug,12,(&Debug,
				 "check_open_mailfile: Opened mail-file #%d: empty continuation line on header field: %*.s\n",
				 idx,x,name));
	    }
	    
	no_header_body:
	    if (header_item) {
		if (header_item -> magic != HEADER_magic)
		    panic("HEADERS PANIC",__FILE__,__LINE__,
			  "check_open_mailfile",
			  "Bad magic number (header list)",0);
		
		if (init_header)
		    header_item -> body = strmcpy(header_item -> body,header_body);
		else
		    header_item -> body = strmcat(header_item -> body,header_body);
		
	    }
	    
	    if (char_other != s ||
		'\n' != c) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileInvHeaderField,
				  "Invalid header field on mail-file %s:%d: %.*s\n"),
			  mailfile,state.lineno,
			  x,name);
		return 0;
	    }
	    
	    break;
	    
	case char_other:
	    /* should not happen */
	    return 0;	    
	}
    }
    
    if (!header_ok) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileNoKnowHeaderFieldfound,
			  "No known header field names found from mail-file %s"),
		  mailfile);
	return 0;
    }

    mail_file_list[idx].header_list_incomplete = header_list_incomplete;
    mail_file_list[idx].binary                 = state.binary;
    
    return 1;
}

int check_mail_file_list()    
{

    if (mail_file_list) {
	int idx;

	for (idx = 0; idx < mail_file_list_len; idx++) {

	    if (!  check_open_mailfile(idx))
		return 0;
	}
    }
	    
    return 1;
}



enum mail_file_mode  mail_files_used()
{
    if (mail_file_list) {

	return mode;
    }

    return no_mail_files_given;
}


/* -------------------------------------------------------- */

S_(mt_init_mailbox mt_init_mfiles)
static void mt_init_mfiles P_((struct MailboxView *mbx));

static void mt_init_mfiles(mbx)
     struct MailboxView *mbx;
{
    mbx->u.dummy = NULL;
}


S_(mt_free_mailbox mt_free_mfiles)
static int mt_free_mfiles P_((struct MailboxView *mbx,
			       struct cancel_data *cd));
static int mt_free_mfiles(mbx,cd)
     struct MailboxView *mbx;
     struct cancel_data *cd;
{
    int ret = 1;
    
    if (mbx->u.dummy) {
	free(mbx->u.dummy);
	mbx->u.dummy = NULL;
    }

    DPRINT(Debug,10,(&Debug,
		     "mt_free_mfiles=%d\n",
		     ret));
    return ret;  
}

static void mt_make_mfiles_view P_((struct MailboxView *mailbox));

/* Return 1 if redraw required */
S_(mt_update_view_mailbox mt_update_view_mfiles)
/* Return 1 if redraw required */
static int mt_update_view_mfiles P_((struct MailboxView *mailbox));
static int mt_update_view_mfiles(mailbox)
     struct MailboxView *mailbox;
{
    int ret = 0;

    if (mail_file_list_len != mailbox->view_len) {

	mt_make_mfiles_view(mailbox);
	    
	ret = 1;
    }

    DPRINT(Debug,7,(&Debug,
		    "mt_update_view_mfiles=%d\n",
		    ret));
    
    return ret;
}

S_(mt_get_main_mailbox_folder mt_get_main_mfiles_folder)
static struct folder_info * mt_get_main_mfiles_folder P_((struct MailboxView *mailbox));
static struct folder_info * mt_get_main_mfiles_folder(mailbox)
     struct MailboxView *mailbox;
{
    return NULL;
}

/* Can be called from signal handler */
S_(mt_get_mailbox_storage mt_get_mfiles_storage)
/* Can be called from signal handler */
static struct current_storage * mt_get_mfiles_storage P_((struct MailboxView *mailbox,
							  int i));
/* Can be called from signal handler */
static struct current_storage * mt_get_mfiles_storage(mailbox,i) 
     struct MailboxView *mailbox;
     int i;
{

    return NULL;
}

/* Can be called from signal handler */
S_(mt_get_mailbox_storage_count mt_get_mfiles_storage_count)
/* Can be called from signal handler */
static int mt_get_mfiles_storage_count P_((struct MailboxView *mailbox));

/* Can be called from signal handler */
static int mt_get_mfiles_storage_count(mailbox) 
     struct MailboxView *mailbox;
{
    return 1;
}

static int update_mail_file_header P_((int idx));
static int update_mail_file_header(idx)
     int idx;
{
    long mailfile_size = INT_MAX;

    
    if (!mail_file_list[idx].f) {
	if (! check_open_mailfile (idx))
	    return 0;
    }

    if (!mail_file_list[idx].header_rec) {
	struct stat buffer;
	enum syscall_status  r;

	int fd = fileno(mail_file_list[idx].f);
	
	mail_file_list[idx].header_rec =
	    safe_zero_alloc(sizeof(* (mail_file_list[idx].header_rec)));
	header_zero(mail_file_list[idx].header_rec);
	
	mail_file_list[idx].header_rec->mime_rec.begin_offset = 0L;
	mail_file_list[idx].header_rec->offset = 0L;

	/* read_folder_headers_helper may set header_charset */
	mail_file_list[idx].header_rec->header_charset         = display_charset;
	
	mail_file_list[idx].header_rec->binary = mail_file_list[idx].binary;
		
	/* read_folder_headers_helper may set content-length */

	/* Parses Return-Path, Message-Id, Content-Length,
	   Date, MIME-Version, X-ELM-OSV
	*/
	read_folder_headers_helper(mail_file_list[idx].header_rec,
				   mail_file_list[idx].header_list);

	r = fstat(fd,&buffer);

	switch (r) {
	    int err;
	    case syscall_success: /* 0 */
		
	    mail_file_list[idx].header_rec->received_time = buffer.st_mtime;
	    
	    if (mail_file_list[idx].content_start >= 0) {
		long old =
		    mail_file_list[idx].header_rec->content_length;

		mail_file_list[idx].header_rec->mime_rec.offset =
		    mail_file_list[idx].content_start;
		mail_file_list[idx].header_rec->content_length =
		    buffer.st_size - mail_file_list[idx].content_start;

		if (buffer.st_size >= LONG_MAX) {
		    mailfile_size = LONG_MAX;
		} else {
		    mailfile_size = buffer.st_size;
		}
		    
		if (old != -1L) {

		    if (old != mail_file_list[idx].header_rec->content_length) {
			DPRINT(Debug,12,(&Debug,
					 "update_mail_file_header: mail-file #%d: content start %ld, size %ld => fixing content length %ld => %ld\n",
					 idx,mail_file_list[idx].content_start,
					 (long)(buffer.st_size),
					 old,
					 mail_file_list[idx].header_rec->content_length));
			
		    }
		    
		} else {		
		    DPRINT(Debug,12,(&Debug,
				     "update_mail_file_header: mail-file #%d: content start %ld, size %ld => content length %ld\n",
				     idx,mail_file_list[idx].content_start,
				     (long)(buffer.st_size),
				     mail_file_list[idx].header_rec->content_length));
		}
		
	    }
	    break;
	    
	case syscall_error: /* -1 */
	    err = errno;
	    	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileFailedStat,
			      "Failed to stat mail-file %s (fileno %d): %s"),
		      mail_file_list[idx].filename,
		      fd,strerror(err));
	    break;
	}

	/* Parses Sensitivity, Importance, Priority,
	   Action, Content-Type, Status, 
	   Subject, From, To, Cc
	   
	   And X-IMAP is used on message-hide-hack
	*/
	
	header_parse_helper(mail_file_list[idx].header_rec,
			    mail_file_list[idx].header_list);

	if (mail_file_list[idx].header_rec->lines < 1) {


	    
	}	
    }

    if (mail_file_list[idx].content_start >= 0 &&
	! (mail_file_list[idx].header_rec->body_parsed)
	) {

	int lines = 0;
	int c;
	long pos;

	char *buffer = NULL;
	int   len = 0; 
	
	rewind(mail_file_list[idx].f);
	
	while (EOF != (c = getc(mail_file_list[idx].f))) {
	    if ('\n' == c) {
		lines++;

		if (ftell(mail_file_list[idx].f) >= mail_file_list[idx].content_start)
		    break;
	    }
	}
	pos = ftell(mail_file_list[idx].f);
	
	DPRINT(Debug,7,(&Debug,
			"update_mail_file_header: mail-file #%d: %d lines on header block, pos %ld %s\n",
			idx,lines,
			pos,
			pos == mail_file_list[idx].content_start ? "== content start" : "ERROR?"));

	start_body_helper(mail_file_list[idx].header_rec,
			  mail_file_list[idx].content_start,
			  mail_file_list[idx].header_list);

	while (mbx_read_line(mail_file_list[idx].f,&buffer,&len,
			     mbx_max_line_read(mailfile_size,pos))) {

	    if (len > 0) {
		if ('\n' == buffer[len-1])
		    lines++;
		
		parse_body_line_helper(mail_file_list[idx].header_rec,
				       buffer,len);

	    }
	}

	if (buffer) {
	    free(buffer);
	    buffer = NULL;
	}
	
	mail_file_list[idx].header_rec->body_parsed = 1;

	DPRINT(Debug,7,(&Debug,
			"update_mail_file_header: mail-file #%d: %d lines\n",
			idx,lines));
	
	mail_file_list[idx].header_rec->lines = lines;
    }

    return 1;
}

S_(mt_give_header_mailbox mt_give_header_mfiles)
static struct header_rec * mt_give_header_mfiles P_((struct MailboxView *mailbox,
						     int index,
						     struct folder_view *v));
static struct header_rec * mt_give_header_mfiles(mailbox,index,v)
     struct MailboxView *mailbox;
     int index;
     struct folder_view *v;
{
    int mbx,idx;
    
    mbx = v->mailbox_number;
    if (mbx != 0)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,
	      "mt_give_header_mfiles",
	      "Bad mailbox number",0);
    
    idx = v->index;
    if (idx < 0 || idx >= mail_file_list_len)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,
	      "mt_give_header_mfiles",
	      "Bad internal index",0);

    if (! mail_file_list[idx].header_rec) {

	if (! update_mail_file_header(idx))
	    return NULL;
    }
       
    return mail_file_list[idx].header_rec;
}

S_(sdt_give_header_s sdt_give_header_mf)
static struct header_rec * sdt_give_header_mf P_((struct sort_data *s,
						  struct folder_view *v));
static struct header_rec * sdt_give_header_mf(s,v)
     struct sort_data *s;  /* NOT USED */
     struct folder_view *v;
{
    if (v->index >= mail_file_list_len)
	return NULL;
    
    return mail_file_list[v->index].header_rec;
}

static struct sort_data_type mailfile_sort = {
    SORTDATATYPE_magic,
    sdt_give_header_mf
};


S_(mt_sort_mailbox_view mt_sort_mfiles_view)
static void mt_sort_mfiles_view P_((struct MailboxView *mailbox,
				     hdr_compare_func   *func));
static void mt_sort_mfiles_view(mailbox,func)
     struct MailboxView *mailbox;
     hdr_compare_func   *func;
{
    int i;
    struct sort_data * array;

    /* Little dirty ... */
    typedef int (*compar) P_((const void *, const void *));
    compar X = (compar) func;
    
    array = safe_calloc(mailbox->view_len, sizeof (array[0]));
    for (i = 0; i < mailbox->view_len; i++) {

	array[i].w              = mailbox->view[i];
	array[i].t              = mailbox->thread_view;   /* For thread sorting */
	array[i].sort_data_type = &mailfile_sort;
	array[i].u.dummy        = NULL;
	
    }
	
    qsort(array,mailbox->view_len,sizeof (array[0]), X);
   
    for (i = 0; i < mailbox->view_len; i++) {
	mailbox->view[i] = array[i].w;
    }

    free(array);
}

S_(mt_give_message_data_mailbox mt_give_message_data_mfiles)
static int mt_give_message_data_mfiles P_((struct MailboxView *mailbox,
					   int index,
					   struct header_rec **ret_header,
					   FILE              **ret_F,
					   struct counter_data *counter,
					   parse_mime_callback *parse_mime,
					   struct folder_view *v));
static int mt_give_message_data_mfiles(mailbox,index,ret_header,ret_F,
					  counter,parse_mime,v)
     struct MailboxView *mailbox;
     int index;
     struct header_rec **ret_header;
     FILE              **ret_F;
     struct counter_data *counter;
     parse_mime_callback *parse_mime;
     struct folder_view *v;
{
    int mbx,idx;
    
    mbx = v->mailbox_number;
    if (mbx != 0)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,
	      "mt_give_message_data_mfiles",
	      "Bad mailbox number",0);
    
    idx = v->index;
    if (idx < 0 || idx >= mail_file_list_len)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,
	      "mt_give_message_data_mfiles",
	      "Bad internal index",0);

    if (!mail_file_list[idx].header_rec ||
	!mail_file_list[idx].f) {
	
	if (!update_mail_file_header(idx))
	    return 0;
    }

    if (ret_header) {
	*ret_header = mail_file_list[idx].header_rec;
		    
	DPRINT(Debug,10,(&Debug,"mt_give_message_data_mfiles: %d -- giving header\n",
			 idx));
    }
    
    if (ret_F) {

	if (! (mail_file_list[idx].header_rec->mime_parsed) && 
	    parse_mime == NO_mime_parse) {
	    DPRINT(Debug,10,(&Debug,
			     "mt_give_message_data_mfiles: mime structure not needed\n"));
	} else if (! (mail_file_list[idx].header_rec->mime_parsed)) {
	    int status;
	    
	    DPRINT(Debug,10,(&Debug,
			     "mt_give_message_data_mfiles: Need parse mime structure\n"));
	    	    
	    rewind(mail_file_list[idx].f);

	    status = parse_mime(NULL,mail_file_list[idx].header_rec,
				mail_file_list[idx].f);
	    
	    if (status <= 0) {
		DPRINT(Debug,10,(&Debug,
				 "mt_give_message_data_mfiles: parse_mime callback failed (%d)\n",
				 status));
	    }
	}

	rewind(mail_file_list[idx].f);
	*ret_F = mail_file_list[idx].f;
    }

    DPRINT(Debug,10,(&Debug,"mt_give_message_data_mfiles: %d -- succeed\n",
		     idx));
    
    return 1;
}

S_(mt_give_message_menu_size mt_give_message_msize_mfiles)
static int mt_give_message_msize_mfiles P_((struct MailboxView * mailbox,
					    struct folder_view * v,
					    unsigned long      * ret_size));
static int mt_give_message_msize_mfiles(mailbox,v,ret_size)
     struct MailboxView * mailbox;
     struct folder_view * v;
     unsigned long      * ret_size;
{
    int mbx,idx;
    
    mbx = v->mailbox_number;
    if (mbx != 0)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,
	      "mt_give_message_msize_mfiles",
	      "Bad mailbox number",0);
    
    idx = v->index;
    if (idx < 0 || idx >= mail_file_list_len)
	panic("MBX VIEW PANIC",__FILE__,__LINE__,
	      "mt_give_message_msize_mfiles",
	      "Bad internal index",0);
    
    if (mail_file_list[idx].f &&
	ret_size) {
	int fd = fileno(mail_file_list[idx].f);

	struct stat buffer;
	enum syscall_status  r;

	r = fstat(fd,&buffer);
	switch(r) {
	    int err;
	case syscall_success: /* 0 */

	    /*  buffer.st_size have type off_t
		and can be 64 bit
	    */

	    if (buffer.st_size >= LONG_MAX) {
		if (ret_size)
		    *ret_size = LONG_MAX;
	    } else {
		if (ret_size)
		    *ret_size =  buffer.st_size;
	    }
	    return 1;
	case syscall_error: /* -1 */
	    err = errno;
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileFailedStat,
			      "Failed to stat mail-file %s (fileno %d): %s"),
		      mail_file_list[idx].filename,
		      fd,strerror(err));
	    break;
	}
    }
    
    return 0;
}

S_(mt_write_mailbox_info mt_write_mfiles_info)
static void mt_write_mfiles_info P_((FILE *fp, struct MailboxView *mailbox,
				     int s, int cur_idx));
static void mt_write_mfiles_info(fp,mailbox,s, cur_idx)
     FILE *fp; 
     struct MailboxView *mailbox;
     int s;
     int cur_idx;
{
    /* EMPTY */
}

S_(mt_mailbox_title mt_mfiles_title)
static struct string * mt_mfiles_title P_((struct MailboxView *mailbox));
static struct string * mt_mfiles_title(mailbox)
     struct MailboxView *mailbox;
{

    if (1 == mailbox->view_len &&
	1 == mail_file_list_len) {

	if (hostname[0] && menu_display_host)
	    return format_string(CATGETS(elm_msg_cat, MeSet, 
					 MeMailFileNOn,
					 "Mail file %s on %s"),
				 mail_file_list[0].filename,
				 hostname);

	return format_string(CATGETS(elm_msg_cat, MeSet, 
				     MeMailFileN,
				     "Mail file %s"),
			     mail_file_list[0].filename);

    } 
    

    if (hostname[0] && menu_display_host) 
	return format_string(CATGETS(elm_msg_cat, MeSet, 
				     MeMailFilesOn,
				     "Mail files collection on %s"),
			     hostname);
    
    return format_string(CATGETS(elm_msg_cat, MeSet, 
				 MeMailFiles,
				 "Mail files collection"));
}

S_(mt_make_mailbox_view mt_make_mfiles_view)
static void mt_make_mfiles_view (mailbox)
     struct MailboxView *mailbox;
{
    int x;
    
    if (mail_file_list_len < 1) {
	if (mailbox->view)
	    free(mailbox->view);
	mailbox->view     = NULL;
	mailbox->view_len = 0;

	return;
    }

    mailbox->view = 
	safe_array_realloc(mailbox->view,
			   mail_file_list_len,
			   sizeof ( mailbox->view[0]));

    for (x = 0; x < mail_file_list_len ; x++) {
	zero_folder_view(& (mailbox->view[x]));

	mailbox->view[x].mailbox_number = 0;
	mailbox->view[x].index          = x;
    }
    mailbox->view_len =  mail_file_list_len;
}

S_(mt_add_mailbox_digest mt_add_mfiles_digest)
static void mt_add_mfiles_digest P_((struct MailboxView *mailbox,
				      mime_t *list,
				      time_t received_time,
				      char *env_from,
				      FILE *F,
				      charset_t defcharset
				      ));
static void mt_add_mfiles_digest(mailbox, list, received_time, env_from, F,
				    defcharset
				    )
     struct MailboxView *mailbox;
     mime_t *list;
     time_t received_time;
     char *env_from;
     FILE *F;
     charset_t defcharset;
{
    panic("MBX VIEW PANIC",__FILE__,__LINE__,"mt_add_mfiles_digest",
	  "mt_add_canceldir_digest called",0);
}

S_(mt_give_message_remote_server_mailbox mt_give_message_remote_server_mfiles)
static const struct remote_server * mt_give_message_remote_server_mfiles P_((
           struct MailboxView *mailbox,
	   int index,
	   struct folder_view *v));
static const struct remote_server * mt_give_message_remote_server_mfiles(
		    mailbox,index,v)
     struct MailboxView *mailbox;
     int index;
     struct folder_view *v;
{
    return NULL;
}


static struct mailbox_type mt_mfiles = {
    MAILBOXTYPE_magic,
    mt_init_mfiles,
    mt_free_mfiles,
    NO_add_mailbox_storage,
    mt_update_view_mfiles,
    mt_get_main_mfiles_folder,
    mt_get_mfiles_storage,
    mt_get_mfiles_storage_count,
    mt_give_header_mfiles,
    mt_sort_mfiles_view,
    mt_give_message_data_mfiles,
    mt_write_mfiles_info,
    mt_mfiles_title,
    mt_make_mfiles_view,
    mt_add_mfiles_digest,
    mt_give_message_remote_server_mfiles,
    mt_give_message_msize_mfiles
};

struct MailboxView *  mailbox_view_mail_files(one_mail_file)
     char **one_mail_file /* result malloced */;
{
    struct MailboxView    *ret = NULL;
    
    if (mail_file_list &&
	( mail_files_as_mailbox == mode ||
	  mail_files_added      == mode )	
	) {

	ret = malloc_view(&mt_mfiles);
	ret->mailbox_type->mt_make_it_view(ret);
	mode =  mail_files_as_mailbox;

	if (one_mail_file && 1 == mail_file_list_len)
	    *one_mail_file =
		strmcpy(*one_mail_file,
			mail_file_list[0].filename);
    }

    return ret;
}

int inline_parts_from_mail_files(attach,mailer_info)
     struct Attachments *attach;     
     struct mailer_info  *mailer_info;
{
    int r = 0;
    
    if (mail_file_list &&
	(   send_mail_files == mode || 
	   mail_files_added      == mode )	
	) {
	int i;

	for (i = 0; i < mail_file_list_len; i++) {
	    int fd = fileno(mail_file_list[i].f);
	    mime_t tmp;
	    struct stat buffer;
	    int need_enc;
	    enum syscall_status  r;
	    
	    mime_t_zero(&tmp);
	    
	    tmp.length = 0;	    
	    r = fstat(fd,&buffer);
	    switch(r) {
		int err;
	    case syscall_success: /* 0 */

		/*  buffer.st_size have type off_t
		    and can be 64 bit
		*/

		if (buffer.st_size >= LONG_MAX) {
		    tmp.length = LONG_MAX;
		} else {
		    tmp.length = buffer.st_size;
		}
		break;
	    case syscall_error: /* -1 */
		err = errno;
		
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMailFileFailedStat,
				  "Failed to stat mail-file %s (fileno %d): %s"),
			  mail_file_list[i].filename,
			  fd,strerror(err));
		break;
	    }

	    tmp.pathname0 = safe_strdup(mail_file_list[i].filename);
	    tmp.dispname  = new_string2(system_charset,
					s2us(tmp.pathname0));
	    
	    lib_transient(CATGETS(elm_msg_cat, MeSet,
				  MeCheckingMailFileS,			
				  "Checking mail-file %S..."), 
			  tmp.dispname);

	    /* rewinds file */
	    need_enc = needs_encoding (mail_file_list[i].f,NULL,0);
	    
	    if (need_enc & HAVE_BINARY) {
		
		if(mailer_info &&
		   query_mailer_info(mailer_info,MI_HAVE_BINARYMIME)) {
		    DPRINT(Debug,8,(&Debug,"mailer supports BINARYMIME\n"));
		    tmp.encoding = ENCODING_BINARY;
		} else
		    tmp.encoding = ENCODING_QUOTED;

	    } else if (need_enc & HAVE_8BIT) {
	
		if(mailer_info &&
		   query_mailer_info(mailer_info,MI_HAVE_8BITMIME)) {
		    DPRINT(Debug,8,(&Debug,"mailer supports 8BITMIME\n"));
		    tmp.encoding = ENCODING_8BIT;
		    
		} else
		    tmp.encoding = ENCODING_QUOTED;	
	    }


	    tmp.disposition = DISP_INLINE;

	    if (tmp.encoding >= ENCODING_NONE &&
		tmp.encoding <= ENCODING_BINARY) {

		DPRINT(Debug,8,
		       (&Debug, 
			"inline_parts_from_mail_files: #%d: Using message/rfc822\n",i));
		
		tmp.TYPE = give_media_type2(MIME_TYPE_MESSAGE,"rfc822",0);
		if (!tmp.TYPE)
		    mime_panic(__FILE__,__LINE__,"inline_parts_from_mail_files",
			       	   "message/rfc822 is not known");

		clear_error();  /* Remove Checking ... -message */
		    
	    } else {

		DPRINT(Debug,8,
		       (&Debug, 
			"inline_parts_from_mail_files: #%d: Using application/octet-stream\n",i));
		
		tmp.TYPE = give_media_type2( MIME_TYPE_APPLICATION,"octet-stream",0);
		if (!tmp.TYPE)
		    mime_panic(__FILE__,__LINE__,"inline_parts_from_mail_files",
			       "application/octet-stream is not known");		

		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeCheckingMailFileSOctetStream,			
				  "Checking mail-file %S... will send as application/octet-stream"), 
			  tmp.dispname);

	    }
		
	    add_Attachments(&attach_files,&tmp);	
	}

	

	mode = send_mail_files;
	r = mail_file_list_len;
    }

    return r;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
