static char rcsid[] = "@(#)$Id: hdrconfg.c,v 2.39 2022/08/28 07:58:12 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.39 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/hdrconfg.c. That code was following copyright:
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/**   This file contains the routines necessary to be able to modify
      the mail headers of messages on the way off the machine.  The
      headers currently supported for modification are:

	Subject:
	To:
	Cc:
	Bcc:
	Reply-To:
	Expires:
	Date:
	Priority:
	Precedence:
	In-Reply-To:
	Action:

	<user defined>
**/

#include "def_elm.h"
#include "s_elm.h"
#include "s_me.h"

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif         

DEBUG_VAR(Debug,__FILE__,"header");

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}


static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}


/*
 * Option flags for the fields in a (struct hdr_menu_item).
 */
enum hf_disp_option {           /* enumerated display option, not mask	*/
    HF_DISP_1ROW =	0001,	/* field is displayed on one line	*/
    HF_DISP_2ROW =	0002,	/* field display spans two lines	*/
    HF_DISP_3ROW =	0003,	/* field display spans three lines	*/
    HF_DISP_LEFT =	0004,	/* field occupies left half of a line	*/
    HF_DISP_RIGHT =	0005	/* field occupies right half of a line	*/
};
#define HF_DISP_MASK	0007	/* -- mask to pull out display option	*/
enum hf_bit_option {            /* bit positions corresping of bitmask  */
    HF_bit_PROMPT_EXP = 3,
    HF_bit_PROMPT_USR,
    HF_bit_PROMPT_MASK,
    HF_bit_APPENDENTRY,
    HF_bit_ADD_COMMA,
    HF_bit_ONPRE_SENDMENU,
    HF_bit_ONPRE_BOUNCEMENU,
    HF_bit_USE_EXP_ADDRESS
};
#define HF_PROMPT_EXP	(1 <<  HF_bit_PROMPT_EXP)	/* prompt for expires data entry	*/
#define HF_PROMPT_USR	(1 <<  HF_bit_PROMPT_USR)	/* prompt for user defined hdr entry	*/
#define HF_PROMPT_MASK	(1 <<  HF_bit_PROMPT_MASK)	/* -- mask to pull out prompt option	*/
#define HF_APPENDENTRY	(1 << HF_bit_APPENDENTRY)	/* append user entry to existing value	*/
#define HF_ADD_COMMA    (1 << HF_bit_ADD_COMMA)		/* add comma if already data            */
#define HF_ONPRE_SENDMENU   (1 << HF_bit_ONPRE_SENDMENU)
#define HF_ONPRE_BOUNCEMENU (1 << HF_bit_ONPRE_BOUNCEMENU)
#define HF_USE_EXP_ADDRESS  (1 << HF_bit_USE_EXP_ADDRESS)  /* Use struct expanded_address and prompt_expanded_address */





struct hdrmenu_context NULL_hdrmenu_context = {
    HDRMENU_CONTEXT_magic,
    NULL,
    NULL,
    NULL,
    0
};

void clear_hdrmenu_context(cnt)
     struct hdrmenu_context *cnt;
{
    if (HDRMENU_CONTEXT_magic != cnt->magic)
	panic("INPUT PANIC",__FILE__,__LINE__,
              "clear_hdrmenu_context",
              "Bad magic number (hdrmenu_context)",0);

    if (cnt->buffer)
	free_string(& (cnt->buffer));
    if (cnt->exp_buffer) {
	free_expanded_address(cnt->exp_buffer);

	cnt->exp_buffer->magic = 0;
	free(cnt->exp_buffer);
	cnt->exp_buffer = NULL;	
    }

    if (cnt->title)
	free_string(& (cnt->title));
    
    cnt->redrawing = 0;
}

static void move_hdrmenu_context P_((struct hdrmenu_context *cnt,
				     struct hdrmenu_context *src
				     ));
static void move_hdrmenu_context(cnt,src)
     struct hdrmenu_context *cnt;
     struct hdrmenu_context *src;
{
    clear_hdrmenu_context(cnt);
    
    if (HDRMENU_CONTEXT_magic != src->magic)
	panic("INPUT PANIC",__FILE__,__LINE__,
              "clear_hdrmenu_context",
              "Bad magic number (hdrmenu_context, src)",0);

    cnt->buffer = src->buffer;
    src->buffer = NULL;

    cnt->exp_buffer = src->exp_buffer;
    src->exp_buffer = NULL;

    cnt->title = src->title;
    src->title = NULL;
    
    cnt->redrawing = src->redrawing;
    src->redrawing = 0;
}


struct hdr_menu_item;
typedef void inpval_proc_t P_((struct hdr_menu_item     * h,
			       struct mailing_headers   * headers,
			       struct hdrmenu_context   * cnt,
			       charset_t hdr_charset
			       ));

typedef struct string * expval_proc_t P_((struct hdr_menu_item *h,
					  struct mailing_headers *headers,
					  charset_t hdr_charset
					  ));

enum free_only_t { hdrproc_set_value = FALSE,
		   hdrproc_free_only = TRUE };

typedef int hdrproc_t  P_((struct hdr_menu_item     * h,
			   struct mailing_headers   * headers,
			   struct hdrmenu_context   * cnt,
			   enum free_only_t           free_only,
			   struct mailer_info       * mailer_info,
			   charset_t                  hdr_charset,
			   struct AliasView         * aview,
			   struct menu_context      * page,
			   struct menu_context      * prompt_area,
			   struct header_rec        * parent_message
			   ));



/*
 * Structure to describe a header which can be edited in this menu.
 */
struct hdr_menu_item {
    int menucmd;       /* The single keystroke (lower-case letter) the	*
			*   user strikes to edit this menu item.	*/

    char *hdrname;	/* Header name to display in the menu.  Parens	*
			 *   should be used to bracket the "menucmd"	*
			 *   char in the name, e.g. "S)ubject".  This	*
			 *   will be NULL for the user-defined header.	*/

    int lineno;		/* Screen line at which the field is displayed.	*/

    int flags;		/* Various flags which effect the display and	*
			 *   user entry of this item.			*/

    inpval_proc_t *inpval_proc;  
                        /* Returns pointer to the buffer to hold the    *
                         *  value entered by the user. Needs to be      *
                         * dealloced with hdrproc			*/

    expval_proc_t *expval_proc;  
                        /* Returns to tointer to the expanded header    *
			 * value to display.  Returned valued need to   *
			 * be free()ed.                                 */

    hdrproc_t *hdrproc;	/* Pointer to a procedure which verifies the	*
			 *   user data entry, and if required converts	*
			 *   the "inpval" value to "expval" value. Does *
                         * needed deallocation of buffer returned by    *
                         * inpval_proc                                  */
};

/*
 * Local procedures.
 */
static int hdrmenu_get P_((struct hdr_menu_item  * h,
			   struct mailing_headers * headers,
			   struct hdrmenu_context * redraw_context,
			   struct mailer_info  *mailer_info,
			   charset_t hdr_charset,
			   struct AliasView *aview,     
			   struct menu_context  *page,
			   struct menu_context  *prompt_area,
			   struct header_rec *parent_message));
static void hdrmenu_put P_((struct hdr_menu_item *h,
			    int already_clear,
			    struct mailing_headers *headers,
			    charset_t hdr_charset,
			    struct menu_context  *page));


/* From: -header -------------------------------------------------------- */

static void inpval_from P_((struct hdr_menu_item     * h,
			    struct mailing_headers   * headers,
			    struct hdrmenu_context   * cnt,
			    charset_t                  hdr_charset));
static struct string * expval_from P_((struct hdr_menu_item *h,
				       struct mailing_headers *headers,
				       charset_t hdr_charset));
static int hdrproc_from  P_((struct hdr_menu_item    * h,
			     struct mailing_headers   * headers,
			     struct hdrmenu_context   * cnt,
			     enum free_only_t           free_only,
			     struct mailer_info       * mailer_info,
			     charset_t                  hdr_charset,
			     struct AliasView         * aview,
			     struct menu_context      * page,
			     struct menu_context      * prompt_area,
			     struct header_rec        * parent_message
			     ));


/* To: -header -------------------------------------------------------- */

static void inpval_to P_((struct hdr_menu_item     * h,
			  struct mailing_headers   * headers,
			  struct hdrmenu_context   * cnt,
			  charset_t                  hdr_charset));
static struct string * expval_to P_((struct hdr_menu_item *h,
				     struct mailing_headers *headers,
				     charset_t hdr_charset));
static int hdrproc_to  P_((struct hdr_menu_item     * h,
			   struct mailing_headers   * headers,
			   struct hdrmenu_context   * cnt,
			   enum free_only_t           free_only,
			   struct mailer_info       * mailer_info,
			   charset_t                  hdr_charset,
			   struct AliasView         * aview,
			   struct menu_context      * page,
			   struct menu_context      * prompt_area,
			   struct header_rec        * parent_message));

/* CC: -header -------------------------------------------------------- */

static void inpval_cc P_((struct hdr_menu_item     * h,
			  struct mailing_headers   * headers,
			  struct hdrmenu_context   * cnt,
			  charset_t                  hdr_charset));
static struct string * expval_cc P_((struct hdr_menu_item *h,
				     struct mailing_headers *headers,
				     charset_t hdr_charset));
static int hdrproc_cc  P_((struct hdr_menu_item     * h,
			   struct mailing_headers   * headers,
			   struct hdrmenu_context   * cnt,
			   enum free_only_t           free_only,
			   struct mailer_info       * mailer_info,
			   charset_t                  hdr_charset,
			   struct AliasView         * aview,
			   struct menu_context      * page,
			   struct menu_context      * prompt_area,
			   struct header_rec        * parent_message));

/* BCC: -header -------------------------------------------------------- */

static void inpval_bcc P_((struct hdr_menu_item     * h,
			   struct mailing_headers   * headers,
			   struct hdrmenu_context   * cnt,
			   charset_t                  hdr_charset));
static struct string * expval_bcc P_((struct hdr_menu_item *h,
				      struct mailing_headers *headers,
				      charset_t hdr_charset));
static int hdrproc_bcc  P_((struct hdr_menu_item     * h,
			    struct mailing_headers   * headers,
			    struct hdrmenu_context   * cnt,
			    enum free_only_t           free_only,
			    struct mailer_info       * mailer_info,
			    charset_t                  hdr_charset,
			    struct AliasView         * aview,
			    struct menu_context      * page,
			    struct menu_context      * prompt_area,
			    struct header_rec        * parent_message));


/* Subject: -header -------------------------------------------------------- */

static void inpval_subject P_((struct hdr_menu_item     * h,
			       struct mailing_headers   * headers,
			       struct hdrmenu_context   * cnt,
			       charset_t hdr_charset));
static struct string * expval_subject P_((struct hdr_menu_item *h,
					  struct mailing_headers *headers,
					  charset_t hdr_charset));
static int hdrproc_subject  P_((struct hdr_menu_item     * h,
				struct mailing_headers   * headers,
				struct hdrmenu_context   * cnt,
				enum free_only_t           free_only,
				struct mailer_info       * mailer_info,
				charset_t                  hdr_charset,
				struct AliasView         * aview,
				struct menu_context      * page,
				struct menu_context      * prompt_area,
				struct header_rec        * parent_message));


/* Reply-To: -header -------------------------------------------------- */

static void inpval_reply_to P_((struct hdr_menu_item     * h,
				struct mailing_headers   * headers,
				struct hdrmenu_context   * cnt,
				charset_t                  hdr_charset)); 
static struct string * expval_reply_to P_((struct hdr_menu_item *h,
					   struct mailing_headers *headers,
					   charset_t hdr_charset));
static int hdrproc_reply_to  P_((struct hdr_menu_item     * h,
				 struct mailing_headers   * headers,
				 struct hdrmenu_context   * cnt,
				 enum free_only_t          free_only,
				 struct mailer_info       * mailer_info,
				 charset_t                  hdr_charset,
				 struct AliasView         * aview,
				 struct menu_context      * page,
				 struct menu_context      * prompt_area,
				 struct header_rec        * parent_message));


/* Action: -header ---------------------------------------------------- */

static void inpval_action P_((struct hdr_menu_item     * h,
			      struct mailing_headers   * headers,
			      struct hdrmenu_context   * cnt,
			      charset_t                  hdr_charset));
static struct string * expval_action P_((struct hdr_menu_item *h,
					 struct mailing_headers *headers,
					 charset_t hdr_charset));
static int hdrproc_action  P_((struct hdr_menu_item     * h,
			       struct mailing_headers   * headers,
			       struct hdrmenu_context   * cnt,
			       enum free_only_t           free_only,
			       struct mailer_info       * mailer_info,
			       charset_t                  hdr_charset,
			       struct AliasView         * aview,
			       struct menu_context      * page,
			       struct menu_context      * prompt_area,
			       struct header_rec        * parent_message));

/* Expires: -header ---------------------------------------------------- */

static void inpval_expires P_((struct hdr_menu_item     * h,
			       struct mailing_headers   * headers,
			       struct hdrmenu_context   * cnt,
			       charset_t                  hdr_charset));
static struct string * expval_expires P_((struct hdr_menu_item *h,
					  struct mailing_headers *headers,
					  charset_t hdr_charset));
static int hdrproc_expires  P_((struct hdr_menu_item     * h,
				struct mailing_headers   * headers,
				struct hdrmenu_context   * cnt,
				enum free_only_t           free_only,
				struct mailer_info       * mailer_info,
				charset_t                  hdr_charset,
				struct AliasView         * aview,
				struct menu_context      * page,
				struct menu_context      * prompt_area,
				struct header_rec        * parent_message));

/* Priority: -header ---------------------------------------------------- */

static void inpval_priority P_((struct hdr_menu_item     * h,
				struct mailing_headers   * headers,
				struct hdrmenu_context   * cnt,
				charset_t                  hdr_charset));
static struct string * expval_priority P_((struct hdr_menu_item *h,
					   struct mailing_headers *headers,
					   charset_t hdr_charset));
static int hdrproc_priority  P_((struct hdr_menu_item     * h,
				 struct mailing_headers   * headers,
				 struct hdrmenu_context   * cnt,
				 enum free_only_t           free_only,
				 struct mailer_info       * mailer_info,
				 charset_t                  hdr_charset,
				 struct AliasView         * aview,
				 struct menu_context      * page,
				 struct menu_context      * prompt_area,
				 struct header_rec        * parent_message));

/* Precedence: -header ---------------------------------------------------- */

static void inpval_precedence P_((struct hdr_menu_item     * h,
				  struct mailing_headers   * headers,
				  struct hdrmenu_context   * cnt,
				  charset_t hdr_charset));
static struct string * expval_precedence P_((struct hdr_menu_item *h,
					     struct mailing_headers *headers,
			     charset_t hdr_charset));
static int hdrproc_precedence  P_((struct hdr_menu_item     * h,
				   struct mailing_headers   * headers,
				   struct hdrmenu_context   * cnt,
				   enum free_only_t           free_only,
				   struct mailer_info       * mailer_info,
				   charset_t                  hdr_charset,
				   struct AliasView         * aview,
				   struct menu_context      * page,
				   struct menu_context      * prompt_area,
				   struct header_rec        * parent_message));

/* In-Reply-To: -header ---------------------------------------------------- */

static void inpval_in_reply_to P_((struct hdr_menu_item     * h,
				   struct mailing_headers   * headers,
				   struct hdrmenu_context   * cnt,
				   charset_t                  hdr_charset));
static struct string * expval_in_reply_to P_((struct hdr_menu_item *h,
					      struct mailing_headers *headers,
			     charset_t hdr_charset));
static int hdrproc_in_reply_to  P_((struct hdr_menu_item     * h,
				    struct mailing_headers   * headers,
				    struct hdrmenu_context   * cnt,
				    enum free_only_t           free_only,
				    struct mailer_info       * mailer_info,
				    charset_t                  hdr_charset,
				    struct AliasView         * aview,
				    struct menu_context      * page,
				    struct menu_context      * prompt_area,
				    struct header_rec        * parent_message));

/* User defined header ---------------------------------------------------- */

static void inpval_userhdr P_((struct hdr_menu_item     * h,
			       struct mailing_headers   * headers,
			       struct hdrmenu_context   * cnt,
			       charset_t                  hdr_charset));
static struct string * expval_userhdr P_((struct hdr_menu_item *h,
					  struct mailing_headers *headers,
			     charset_t hdr_charset));
static int hdrproc_userhdr  P_((struct hdr_menu_item     * h,
				struct mailing_headers   * headers,
				struct hdrmenu_context   * cnt,
				enum free_only_t           free_only,
				struct mailer_info       * mailer_info,
				charset_t                  hdr_charset,
				struct AliasView         * aview,
				struct menu_context      * page,
				struct menu_context      * prompt_area,
				struct header_rec        * parent_message));

/*
 * Definition of all the header editing menu fields.
 */
struct hdr_menu_item hmenu_item_list[] = {
    { 't', "T)o",		 2,
      HF_DISP_3ROW|HF_APPENDENTRY|HF_ADD_COMMA|
      HF_ONPRE_SENDMENU|HF_ONPRE_BOUNCEMENU|
      HF_USE_EXP_ADDRESS,
      inpval_to, expval_to, hdrproc_to },
    { 'c', "C)c",		 5,
      HF_DISP_3ROW|HF_APPENDENTRY|HF_ADD_COMMA|HF_ONPRE_SENDMENU|
      HF_USE_EXP_ADDRESS,
      inpval_cc, expval_cc, hdrproc_cc },
    { 'b', "B)cc",		 8, HF_DISP_2ROW|HF_APPENDENTRY|
      HF_ADD_COMMA|HF_ONPRE_SENDMENU|
      HF_USE_EXP_ADDRESS,
      inpval_bcc, expval_bcc, hdrproc_bcc },
    { 's', "S)ubject",		10, HF_DISP_2ROW|HF_ONPRE_SENDMENU, 
      inpval_subject, expval_subject, hdrproc_subject },
    { 'r', "R)eply-to",		12, HF_DISP_1ROW|HF_USE_EXP_ADDRESS, 
      inpval_reply_to, expval_reply_to, hdrproc_reply_to },
    { 'a', "A)ction",		13, HF_DISP_LEFT, 
      inpval_action, expval_action, hdrproc_action },
    { 'e', "E)xpires",		13, HF_DISP_RIGHT|HF_PROMPT_EXP, 
      inpval_expires, expval_expires, hdrproc_expires },
    { 'p', "P)riority",		14, HF_DISP_LEFT, 
      inpval_priority, expval_priority, hdrproc_priority },

    { 'n', "Precede(n)ce",	14, HF_DISP_RIGHT, 
      inpval_precedence, expval_precedence, hdrproc_precedence },

    { 'i', "I)n-reply-to",	15, HF_DISP_2ROW,
      inpval_in_reply_to, expval_in_reply_to, hdrproc_in_reply_to },

    { 'u', NULL,		17, HF_DISP_1ROW|HF_PROMPT_USR,
      inpval_userhdr, expval_userhdr, hdrproc_userhdr },
    { 'f', "F)rom",		-1, HF_USE_EXP_ADDRESS,
      inpval_from, expval_from, hdrproc_from },
    { -1, NULL, -1, -1, NULL, NULL, NULL },
};

/*
 * Selection of individual fields.  The indices *must* correspond
 * to the above "hmenu_item_list[]" list.
 */
#define hmenu_to		(hmenu_item_list[0])
#define hmenu_cc		(hmenu_item_list[1])
#define hmenu_bcc		(hmenu_item_list[2])
#define hmenu_subject		(hmenu_item_list[3])
#define hmenu_replyto		(hmenu_item_list[4])
#define hmenu_action		(hmenu_item_list[5])
#define hmenu_expires		(hmenu_item_list[6])
#define hmenu_priority		(hmenu_item_list[7])
#define hmenu_precedence	(hmenu_item_list[8])
#define hmenu_inreplyto		(hmenu_item_list[9])
#define hmenu_userdef		(hmenu_item_list[10])
#define hmenu_from		(hmenu_item_list[11])

void show_presend_headers(headers,hdr_charset,page,show_prehdr_mode)
     struct mailing_headers *headers;
     charset_t hdr_charset;
     struct menu_context  *page;
     enum show_presend_mode show_prehdr_mode;
{
    struct hdr_menu_item *h;

    int f = 0;

    switch (show_prehdr_mode) {
    case show_prehdr_normal: f = HF_ONPRE_SENDMENU;    break;
    case show_prehdr_bounce: f = HF_ONPRE_BOUNCEMENU;  break;

    }
    

    for (h = hmenu_item_list ; h->menucmd > 0 ; ++h) {
	if (h->lineno > 0 && 0 != (h->flags & f))
	    hdrmenu_put(h, TRUE, headers,hdr_charset,page);
    }

}

/* returns 0 if command not found
           1 if found 
	   REDRAW_MARK is redraw required 
*/
int presend_action(headers,mailer_info,c,redraw_context,
		   hdr_charset,aview,page,
		   prompt_area, parent_message,show_prehdr_mode)
     struct mailing_headers * headers;    
     struct mailer_info     * mailer_info;
     int                      c;
     struct hdrmenu_context * redraw_context;
     charset_t                hdr_charset;
     struct AliasView       * aview;
     struct menu_context    * page;
     struct menu_context    * prompt_area;
     struct header_rec      * parent_message;
     enum show_presend_mode   show_prehdr_mode;
{
   struct hdr_menu_item *h;
   int LINES, COLUMNS;

   int f = 0;

   switch (show_prehdr_mode) {
   case show_prehdr_normal: f = HF_ONPRE_SENDMENU;    break;
   case show_prehdr_bounce: f = HF_ONPRE_BOUNCEMENU;  break;
       
   }

   menu_get_sizes(page, &LINES, &COLUMNS);   

      
   for (h = hmenu_item_list ; h->menucmd > 0 ; ++h) {
       if (ison(h->flags,f)) {
	   if (h->menucmd == c) {
	       int status;

	       MoveCursor(LINES-4,0);
	       CleartoEOS();

	       status = hdrmenu_get(h,headers,
				    redraw_context,
				    mailer_info,hdr_charset,aview,
				    page,prompt_area, parent_message);
	       if (REDRAW_MARK == status) {
		   return REDRAW_MARK;
	       }

	       if (redraw_context)
		   clear_hdrmenu_context(redraw_context);
	       
	       if (status != 0) {
		   Writechar('\007');
		   return 1;
	       }
	       if (h->lineno > 0)
		   hdrmenu_put(h, FALSE,headers,hdr_charset,page);
	       return 1;		
	   }	   
       }
   }

   if (redraw_context)
       clear_hdrmenu_context(redraw_context);
   
   return 0;
}

void edit_headers(headers,mailer_info,hdr_charset, hdr_encoding_supported,
		  aview,parent_message)
     struct mailing_headers *headers;    
     struct mailer_info  *mailer_info;
     charset_t hdr_charset;
     int hdr_encoding_supported;
     struct AliasView *aview;
     struct header_rec *parent_message;  /* Replying ... */
{
    int c, do_redraw;
    struct hdr_menu_item *h;
    int precmd = 0;

    unsigned int editor_keyword = 0, editor_mask = 0;
    const char * editor_val = give_dt_estr_as_str(&editor_e,"editor",
						  &editor_keyword,
						  &editor_mask);
    int editor_available = 
	editor_val != NULL &&
	editor_kw_NO == editor_keyword;
    enum user_level_v ul = 
	give_dt_enumerate_as_int(&user_level);

    int LINES, COLUMNS;

    struct menu_context  *page = new_menu_context();

    struct menu_context  * prompt_area = NULL;
    struct elm_commands  * edithdr_cmds =  give_edithdr_commands();

    struct hdrmenu_context redraw_context = NULL_hdrmenu_context;

    menu_get_sizes(page, &LINES, &COLUMNS);   

    /*
     * Placement of prompts and messages at the bottom of the screen.
     */
    
    prompt_area = 
	new_menu_subpage(page,LINES-5,5,
			 subpage_simple_noredraw,NULL);

    /* expand out all of the header values */
    /* menu displays expanded values, user edits unexpended versions */

    do_redraw = TRUE;
    while (TRUE) {	/* forever */
	menu_set_default(page);

    resize_mark:
	if (menu_resized(page)) {
	    
	    menu_get_sizes(page,&LINES, &COLUMNS);
	    
	    menu_subpage_relocate(prompt_area,page,LINES-5,5);

	    do_redraw = 1;

	} else if (menu_need_redraw(page))
	    do_redraw = 1;


	if (headers->from.addrs &&
	    addr_list_item_count(headers->from.addrs) > 0 &&
	    hmenu_from.lineno == -1 && 
	    (hmenu_to.flags & HF_DISP_MASK) == HF_DISP_3ROW) {
	    /* Hack: 
	       From: -header take space from To: -header ... 
	    */

	    hmenu_from.lineno = hmenu_to.lineno++;
	    hmenu_to.flags = 
		( hmenu_to.flags & ~HF_DISP_MASK) |
		HF_DISP_2ROW;
	    hmenu_from.flags = /* preserve  HF_USE_EXP_ADDRESS */
		( hmenu_from.flags & ~HF_DISP_MASK) |
		HF_DISP_1ROW;
	    do_redraw = 1;
	}

	/* redraw the entire display if required */
	if (do_redraw) {
	    menu_ClearScreen(page);

	    menu_StartXX(page,pg_BOLD);	    
	    menu_print_format_center(page,0, 
				     CATGETS(elm_msg_cat, ElmSet,
					     ElmHdrmenuScreenTitle, 
					     "Message Header Edit Screen"));
	     menu_EndXX(page,pg_BOLD);

	    for (h = hmenu_item_list ; h->menucmd > 0 ; ++h) {
		if (h->lineno > 0)
		    hdrmenu_put(h, TRUE, headers, hdr_charset, page);
	    }
	    do_redraw = FALSE;

	    menu_redraw_children(page);
	}

	if (menu_resized(prompt_area) ||
	    menu_need_redraw(prompt_area)) {
	
	    menu_ClearScreen(prompt_area);
	    
	    /* display the instructions */
	    if (ul > user_level_beginner && editor_available) {
		
		if (ALLOW_subshell)
		    menu_print_format_center(prompt_area,0, 
					     CATGETS(elm_msg_cat, ElmSet,
						     ElmHdrmenuInstruct1,
						     "Choose header, u)ser defined header, !)shell, invoke e(d)itor, or <return>."));
		else
		    menu_print_format_center(prompt_area,0,
					     CATGETS(elm_msg_cat, ElmSet,
						     ElmHdrmenuInstruct1NoShell,
						     "Choose header, u)ser defined header, invoke e(d)itor, or <return>."));
		
	    } else {
		
		if (ALLOW_subshell)
		    menu_print_format_center(prompt_area,0,
					     CATGETS(elm_msg_cat, ElmSet,
						     ElmHdrmenuInstruct,
						     "Choose header, u)ser defined header, !)shell, or <return>."));
		else
		    menu_print_format_center(prompt_area,0,
					     CATGETS(elm_msg_cat, ElmSet,
						     ElmHdrmenuInstructNoShell,
						     "Choose header, u)ser defined header, or <return>."));
		
	    }	    	    	    
	}
	    
#ifdef BACKGROUD_PROCESSES      
	if (handle_sigchld)
	    sigchld_handler();
#endif
       
	/* prompt for command */
	menu_PutLineX(prompt_area,2, 0, 
		      CATGETS(elm_msg_cat, ElmSet,
			      ElmHdrmenuPrompt, "Choice: "));
	if (precmd) {
	    c = precmd;
	    precmd = 0;
	} else {
	    clear_hdrmenu_context(&redraw_context);
	    
	    c = menu_ReadCh(prompt_area,
			    REDRAW_MARK|READCH_resize|READCH_sig_char);
	    if (c == REDRAW_MARK) {
		menu_ClearScreen(page);   /* Clear possible redraw mark */
		
		/* Call refresh routines of children */
		menu_redraw_children(page);

		if (menu_need_redraw(prompt_area))		    
		    menu_ClearScreen(prompt_area);   /* Clear redraw mark from prompt_area*/

		do_redraw = TRUE;
		continue;
	    }

	    if (c == TERMCH_interrupt_char)
		goto OUT;

	    if (c == RESIZE_MARK) {
		DPRINT(Debug,4, (&Debug, "  ... resizing\n"));
		goto resize_mark;
	    }

	}

#ifdef ASCII_CTYPE
	if (isascii(c))
#endif
	  c = tolower(c);

	menu_ClearScreen(prompt_area);

	/* execute the command */
	switch (c) {

	case EOF:
	case RETURN:
	case LINE_FEED:
	case 'q':
	    goto OUT;

	case HELP_MARK:
	case '?': {			/* help */
	    int X UNUSED_VAROK =  help_generic(edithdr_cmds,FALSE, 
					       page, prompt_area);

# if 0	    
	    if (EOF == X) 
		ret = EOF;         
#endif

	    goto OUT;
	}
	    break;


	case '!':
	   
	    if (ALLOW_subshell)
		subshell(NULL, page, prompt_area);   /* !! */
	    else
		  lib_error(CATGETS(elm_msg_cat, ElmSet, 
				    ElmNoSubshellCommand,
				    "Subshell command isn't configured in this version of ELM."));

	    if (menu_need_redraw(page))
		do_redraw = TRUE;
	    break;

	case ctrl('L'):
	    do_redraw = TRUE;
	    break;

	case 'd':
	    if (editor_val) {
		editor_available &=  have_editor(editor_val);
		
		if (editor_available)
		    edit_headers_on_editor(headers,editor_val, 
					   hdr_encoding_supported,
					   page);
	    }

	    do_redraw = TRUE;
	    break;


	default:
	    for (h = hmenu_item_list ; h->menucmd > 0 ; ++h) {
		if (h->menucmd == c) {
		    int status = hdrmenu_get(h,headers,
					     &redraw_context,
					     mailer_info,
					     hdr_charset,aview,
					     page,prompt_area,
					     parent_message);
		    if (REDRAW_MARK == status) {
			precmd = c;
			do_redraw = TRUE;
			break;
		    }

		    clear_hdrmenu_context(&redraw_context);
		    
		    if (status != 0) {
			Writechar('\007');
			break;
		    }
		    if (h->lineno > 0)
			hdrmenu_put(h, FALSE,headers, hdr_charset, page);
		    break;
		}
	    }
	    if (h->menucmd <= 0) {
		menu_print_format_center(prompt_area,3, 
					 CATGETS(elm_msg_cat, ElmSet,
						 ElmHdrmenuBadChoice, 
						 "No such header!"));
		Writechar('\007');
	    }
	    break;
	}

    }

 OUT:
    clear_hdrmenu_context(&redraw_context);

    free_commands(&edithdr_cmds);

    erase_menu_context(&prompt_area);

    erase_menu_context(&page);
    return;
}




/*
 * Prompt the user for a header value, and do any required post-processing.
 */
static int hdrmenu_get(h,headers,redraw_context,
		       mailer_info,hdr_charset,aview,page,
		       prompt_area,parent_message)
     struct hdr_menu_item *h;
     struct mailing_headers *headers;
     struct hdrmenu_context * redraw_context;
     struct mailer_info  *mailer_info;     
     charset_t hdr_charset;
     struct AliasView *aview;
     struct menu_context  *page;
     struct menu_context  *prompt_area;
     struct header_rec *parent_message;        /* Replying ... */
{
    char *s;
    int plen, ret, do_append, do_comma;
    char header_name[20];
    int LINES, COLUMNS;

    const int line = 2;

    struct hdrmenu_context buffer_content = NULL_hdrmenu_context;   
    
    
    menu_get_sizes(page,&LINES, &COLUMNS);

    menu_ClearLine(prompt_area,0);

    /* display the instructions */
    switch (h->flags & HF_PROMPT_MASK) {
    case HF_PROMPT_EXP:
	menu_print_format_center(prompt_area,0,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmHdrmenuGetExpiresInstruct,
					 "In how many days should this message expire? "));
	break;
    case HF_PROMPT_USR:
	menu_print_format_center(prompt_area,0,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmHdrmenuGetUserdefInstruct,
					 "Enter in the format \"HeaderName: HeaderValue\"."));
	break;
    default:
	menu_print_format_center(prompt_area,0,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmHdrmenuGetInstruct, 
					 "Enter value for the header."));
	break;
    }

    /* display a prompt */
    plen = 0;

    if (h->hdrname != NULL) {
	
	for (s = h->hdrname ; *s != '\0' ; ++s) {
	    if (*s != '(' && *s != ')') {
		if (plen > sizeof header_name -4)
		    break;
		header_name[plen] = *s;
		++plen;
	    }
	}
	header_name[plen++] = ':';
	header_name[plen++] = ' ';
    }
    header_name[plen] = '\0';

    /* get input from the user */
    do_append = ison(h->flags,HF_APPENDENTRY);

    if (redraw_context) {
	if (HDRMENU_CONTEXT_magic != redraw_context->magic)
	    panic("INPUT PANIC",__FILE__,__LINE__,
		  "hdrmenu_get",
		  "Bad magic number (hdrmenu_context, redraw_context)",0);
	
	if (redraw_context->redrawing) {
	    move_hdrmenu_context(& buffer_content,redraw_context);
	    do_append = 1;  /* Append if this was redraw */
	} else
	    goto new_inpval;
    } else {
    new_inpval:	
	h->inpval_proc(h,headers,& buffer_content,
		       hdr_charset);
	
	if (plen > 0)
	    buffer_content.title = format_string(CATGETS(elm_msg_cat, ElmSet,
							 ElmHdrmenuHdrAddressSel,
							 "%saddress selection"),
						 header_name);
	else
	    buffer_content.title = format_string(CATGETS(elm_msg_cat, ElmSet,
							 ElmHdrmenuHdrAddressSel1,
							 "Address selection"));
	    

    }
    
    do_comma = (ison(h->flags,HF_ADD_COMMA) &&
		string_len(buffer_content.buffer));

    if (ison(h->flags,HF_USE_EXP_ADDRESS) && buffer_content.exp_buffer) {

	DPRINT(Debug,10, (&Debug,
			  "hdrmenu_get: %s - using prompt_expanded_address()\n",
			  header_name));
	
	ret = prompt_expanded_address(page,
				      buffer_content.exp_buffer,
				      mailer_info,
				      & (buffer_content.buffer),
				      aview,
				      buffer_content.title  /* Title on full page mode */,
				      prompt_area,line,0,
				      OE_TABACTION |
				      (do_append ? OE_APPEND_CURRENT : 0) | 
				      (do_comma ? OE_ADD_COMMA : 0) |
				      OE_REDRAW_MARK|OE_SIG_CHAR /* Ctrl-C */, 
				      FRM("%s"),header_name);   
	
    } else {
	/* FIXME --optionally_enter*  should use prompt_area */
	int line1 = menu_GetAbsLine(prompt_area,line);

	DPRINT(Debug,10, (&Debug,
			  "hdrmenu_get: %s - using optionally_enter2()\n",
			  header_name));

	
	ret = optionally_enter2(page,
				& (buffer_content.buffer),
				line1, 0, 
				(do_append ? OE_APPEND_CURRENT : 0) | 
				(do_comma ? OE_ADD_COMMA : 0) |
				OE_REDRAW_MARK|OE_SIG_CHAR /* Ctrl-C */, 
				FRM("%s"),header_name);      
    }
    if (ret == REDRAW_MARK) {

	if (redraw_context) {
	    if (HDRMENU_CONTEXT_magic != redraw_context->magic)
		panic("INPUT PANIC",__FILE__,__LINE__,
		      "hdrmenu_get",
		      "Bad magic number (hdrmenu_context, redraw_context)",0);
	    
	    move_hdrmenu_context(redraw_context,&buffer_content);
	    redraw_context->redrawing = 1;

	    clear_hdrmenu_context(&buffer_content);
	} else {
	    h->hdrproc(h,headers,&buffer_content,
		       hdrproc_free_only,mailer_info,hdr_charset,aview,page,
		       prompt_area,parent_message);
	}
	
	return REDRAW_MARK;
    }

    menu_ClearScreen(prompt_area);

    /* bail out on error */
    if (ret != 0) {
	if (buffer_content.buffer || buffer_content.exp_buffer)
	    h->hdrproc(h,headers,& buffer_content,
		       hdrproc_free_only,mailer_info,hdr_charset,aview,
		       page,prompt_area,parent_message);

	if (redraw_context)
	    clear_hdrmenu_context(redraw_context);
	
	return -1;
    }

    if (redraw_context)
	clear_hdrmenu_context(redraw_context);
    
    /* see if there is some processing required on this value */
    return h->hdrproc(h,headers,& buffer_content,
		      hdrproc_set_value,mailer_info,hdr_charset,
		      aview,page,prompt_area,parent_message);
}


/*
 * Dispay a header and its value in the appropriate field.
 */
static void hdrmenu_put(h, already_clear, headers, hdr_charset,
			page)
     struct hdr_menu_item *h;
     int already_clear;
     struct mailing_headers *headers; 
     charset_t hdr_charset;
     struct menu_context  *page;
{
    struct string * buffer = NULL;
    char    *p;
    int     start_row, max_row, start_col, max_col, row, col;
    int X;
    int Bl;
    int LINES, COLUMNS;

    menu_get_sizes(page,&LINES, &COLUMNS);

    /* figure out the dimensions of the field */
    switch (h->flags & HF_DISP_MASK) {
    case HF_DISP_LEFT:
	start_row = h->lineno;		max_row = h->lineno;
	start_col = 0;			max_col = COLUMNS/2 - 2;
	break;
    case HF_DISP_RIGHT:
	start_row = h->lineno;		max_row = h->lineno;
	start_col = COLUMNS/2 + 1;	max_col = COLUMNS-1;
	break;
    case HF_DISP_3ROW:
	start_row = h->lineno;		max_row = h->lineno+2;
	start_col = 0;			max_col = COLUMNS-1;
	break;
    case HF_DISP_2ROW:
	start_row = h->lineno;		max_row = h->lineno+1;
	start_col = 0;			max_col = COLUMNS-1;
	break;
    default:
	start_row = h->lineno;		max_row = h->lineno;
	start_col = 0;			max_col = COLUMNS-1;
	break;
    }

    /* display the header name */
    menu_MoveCursor(page,start_row, start_col);
    
    if (h->hdrname != NULL) {
	for (p = h->hdrname ; *p != '\0' ; ++p)
	    menu_Writechar(page,*p);
	menu_Writechar(page,':');
	menu_Writechar(page,' ');
    }

    /* display the header value */
    menu_GetXYLocation(page,&row, &col);

    buffer=h->expval_proc(h,headers,hdr_charset);          
    
    Bl = buffer ? string_len(buffer) : 0;
    for (X = 0; buffer && X < Bl && row <= max_row; row++) {
	struct string * str = NULL;
	int visible_len;
	
	int space_left;
	int cur_col = col;

    retry:
	if (row == max_row) /* neat hack alert */
	    space_left = max_col-cur_col-4;
	else 
	    space_left = max_col-cur_col;

	str = curses_printable_clip(buffer,&X,Bl,&visible_len,space_left);

	if (str) {
	    menu_MoveCursor(page,row, col);
	    menu_Write_to_screen(page,FRM("%S"),str);
	
	    if (row == max_row && X < string_len(buffer)) {
		menu_Write_to_screen(page,FRM(" ..."));
		cur_col += 4;
	    }

	    free_string(&str);
	    cur_col += visible_len;
	} else {
	    /* Treat next character as control character */
	    menu_StartXX(page,pg_BOLD);
	    menu_Writechar(page,'?');		    
	    menu_EndXX(page,pg_BOLD);
	    
	    cur_col++;
	    if (cur_col < max_col && X < Bl)
		goto retry;
	}

	/* If we are not end of string -- clear rest on case of double with characters */
	if (X < Bl && cur_col < max_col && !already_clear) {
	    while (cur_col++ <= max_col)
		menu_Writechar(page,' ');
	}

    }
    menu_GetXYLocation(page,&row, &col);

    /* save some drawing if we know the screen is already empty */
    if (!already_clear) {
	
	/* clear out remaining space in this line of the field */
	if (max_col == COLUMNS-1) {
	    /* people on slow terminals might appreciate doing it this way */
	    menu_CleartoEOLN(page);
	} else {
	    while (col++ <= max_col)
		menu_Writechar(page,' ');
	}
	
	/* for multi-line fields, clear out any unused lines */
	/* this assumes that multi-line fields span the entire screen width */
	while (++row <= max_row) {
	    /* grrrrrr -- this is a multi-statement macro */
	    menu_ClearLine(page,row);
	}

    }

    if (buffer)
	free_string(&buffer);
}


static void hdr_to_buffer P_((struct expanded_address    addrs,
			      struct hdrmenu_context   * cnt,
			      int                        hf_flags  /* HF_USE_EXP_ADDRESS */
			      ));




static void hdr_to_buffer(addrs,cnt,hf_flags)
     struct expanded_address    addrs;
     struct hdrmenu_context   * cnt;
     int                        hf_flags  /* HF_USE_EXP_ADDRESS */;
{
    if (HDRMENU_CONTEXT_magic != cnt->magic)
	panic("INPUT PANIC",__FILE__,__LINE__,
              "dr_to_buffer",
              "Bad magic number (hdrmenu_context)",0);
    
    if (cnt->exp_buffer) {
	free_expanded_address(cnt->exp_buffer);
    } else if (ison(hf_flags,HF_USE_EXP_ADDRESS)) {
	
	cnt->exp_buffer = safe_zero_alloc(sizeof (*(cnt->exp_buffer)));
	zero_expanded_address(cnt->exp_buffer);
    }

    
    if (cnt->exp_buffer && ison(hf_flags,HF_USE_EXP_ADDRESS)) {
	copy_expanded_address(cnt->exp_buffer,&addrs);
    }
		   
    expanded_to_edit_buffer(&(cnt->buffer),addrs);
}

void display_add_phrase(buffer,str)
     struct string ** buffer;
     const struct string *str;

{
    /* We do not quote displayed (only) address */
    enum phrase_display_mode_v pd = 
	give_dt_enumerate_as_int(&phrase_display_mode);

    switch (pd) {
    case phrase_display_plain: 
	append_string(buffer,str,1);
	break;

    default:
	/* We do not (backslash) quote displayed (only) address */
	
	if (! *buffer)
	    *buffer =  new_string(display_charset);
	
	add_ascii_to_string(*buffer,s2us("\""));
	append_string(buffer,str,1);
	
	add_ascii_to_string(*buffer,s2us("\""));
	break;
    }    
}


void display_add_one_address(buffer,address)
     struct string ** buffer;
     const struct address * address;
{
    
    const char          * addr     = address_get_ascii_addr(address);
    const struct string * fullname = address_get_phrase(address);
    const struct string * comment  = address_get_comment(address);

    if (!*buffer)
	*buffer = new_string(display_charset);

    if (addr || fullname || comment) {
	if ((fullname && string_len(fullname)) 
	    || !addr || !addr[0] || '@' == addr[0]) {
	    
	    if (fullname) {
		
		/* We do not quote displayed (only) address */
		
		display_add_phrase(buffer,fullname);
		
	    }		    		    
	    
	    add_ascii_to_string(*buffer,s2us(" <"));
	    if (addr)
		add_ascii_to_string(*buffer,cs2us(addr));
	    add_ascii_to_string(*buffer,s2us(">"));
	    
	} else
	    add_ascii_to_string(*buffer,cs2us(addr));
    
	if (comment && string_len(comment)) {
	    /* We do not quote displayed (only) address */
	    struct string * temp;
	    
	    add_ascii_to_string(*buffer,s2us(" ("));
	    temp = cat_strings(*buffer,comment,1);
	    add_ascii_to_string(temp,s2us(")"));
	    free_string(buffer);
	    
	    *buffer = temp;
	}
    }
}


struct string *hdr_to_expval(addrs)
     struct expanded_address addrs;
{
    struct string * buffer = NULL;

    if (addrs.addrs) {
	int idx, addr_item_count  = addr_list_item_count(addrs.addrs);
	int current_group = -1;   /* Handle only non-empty groups */

	for (idx = 0; idx < addr_item_count; idx++) {

	    int group = -1;
	    const struct address * address = 
		addr_list_get_item(addrs.addrs,idx,&group);
	    
	    
	    if (buffer) {
		if (current_group != group &&
		    current_group != -1) {
		    add_ascii_to_string(buffer,s2us(";"));
		    current_group = -1;
		}

		add_ascii_to_string(buffer,s2us(", "));
	    } else
		buffer = new_string(display_charset);
	
	    if (group >= 0 &&
		current_group == -1) {
		/* Handle only non-empty groups */

		const struct string * groupname1 =
		    addr_list_get_group(addrs.addrs,group);

		display_add_phrase(&buffer,groupname1);
				
		add_ascii_to_string(buffer,s2us(": "));
		current_group = group;
	    }

	    display_add_one_address(&buffer,address);


	}

	if (current_group != -1) {
	    add_ascii_to_string(buffer,s2us(";"));
	    current_group = -1;
	}
    }

    return buffer;
}

charset_t want_convert_to_utf7(text)
     const struct string * text;
{
    charset_t utf7;
    const char *MIME_name_t = get_string_MIME_name(text);

    if (convert_utf_header && 
	MIME_name_t && 0 == istrcmp(MIME_name_t,"UTF-8") &&
	0 != (CS_mapping & 
	      charset_properties(get_string_type(text))) &&
	(utf7 =  MIME_name_to_charset("UTF-7",0)) &&
	0 != (CS_mapping & charset_properties(utf7)))

	return utf7;

    return NULL;
}


/*
 * Process the to, cc, and bcc headers.  The value entered by the
 * user is expanded.  A successful status is always returned.
 */


static int buffer_to_header P_((struct expanded_address  * addrs,
				struct hdrmenu_context   * cnt,
				enum free_only_t           free_only,     
				struct mailer_info       * mailer_info,
				struct AliasView         * aview,
				int                        hf_flags  /* HF_USE_EXP_ADDRESS */
				));

static int buffer_to_header(addrs,cnt,free_only,mailer_info,aview,hf_flags)
     struct expanded_address  * addrs;
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;
     struct mailer_info       * mailer_info;
     struct AliasView         * aview;
     int                        hf_flags  /* HF_USE_EXP_ADDRESS */;
{
    if (HDRMENU_CONTEXT_magic != cnt->magic)
	panic("INPUT PANIC",__FILE__,__LINE__,
              "buffer_to_header",
              "Bad magic number (hdrmenu_context)",0);
    
    switch (free_only) {
    case hdrproc_set_value:
	if (ison(hf_flags,HF_USE_EXP_ADDRESS) && cnt->exp_buffer) {
	    copy_expanded_address(addrs,cnt->exp_buffer);
	} else {
	    update_expanded_from_edit_buffer(addrs,cnt->buffer,mailer_info, aview);
	}
	break;
    case hdrproc_free_only:
	break;
    }

    clear_hdrmenu_context(cnt);
    
    return 0;
}

static void text_to_buffer P_((char *text,struct string **ptr,
			       charset_t hdr_charset));
static void text_to_buffer(text,ptr,hdr_charset)
     char *text;
     struct string **ptr;
     charset_t hdr_charset;
{
    if (text)
	*ptr = new_string2(hdr_charset,s2us(text));
    else 
	*ptr = new_string(hdr_charset);
}

static struct string * text_to_expval P_((char *text,
					  charset_t hdr_charset));
static struct string * text_to_expval(text,hdr_charset)
     char *text;
     charset_t hdr_charset;
{
    if (!text)
	return new_string(hdr_charset);
    else
	return new_string2(hdr_charset,s2us(text));
}

static int buffer_to_text P_((char **text,struct string **ptr,int free_only,
			      charset_t hdr_charset));
static int buffer_to_text(text,ptr,free_only,hdr_charset)
     char **text;
     struct string **ptr;
     int free_only;
     charset_t hdr_charset;
{

    if (!free_only) {
	struct string * X;

	if (can_ascii_string(*ptr))
	    X = ascify_string(*ptr);
	else
	    X = convert_string(hdr_charset,
			       *ptr,0);

	if (*text)
	    free(*text);
	*text = us2s(stream_from_string(X,0,NULL));
	free_string(&X);
    }
    free_string(ptr);
    return 0;
}

/* From: -header -------------------------------------------------------- */

static void inpval_from (h,headers,cnt,hdr_charset)
     struct hdr_menu_item      * h;  
     struct mailing_headers    * headers;
     struct hdrmenu_context    * cnt;
     charset_t                   hdr_charset;    /* NOT USED */
{    
    hdr_to_buffer(headers->from,cnt,h->flags);
}

static struct string * expval_from (h,headers,hdr_charset)
     struct hdr_menu_item *h;    /* NOT USED */
     struct mailing_headers *headers;
     charset_t hdr_charset;      /* NOT USED */
{
    return hdr_to_expval(headers->from);
}

static int hdrproc_from  (h,headers,cnt,free_only,mailer_info,hdr_charset,
			  aview,page,prompt_area,parent_message)
     struct hdr_menu_item     * h;             
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;
     struct mailer_info       * mailer_info;
     charset_t                  hdr_charset;       /* NOT USED */
     struct AliasView         * aview;
     struct menu_context      * page;              /* NOT USED */
     struct menu_context      * prompt_area;       /* NOT USED */
     struct header_rec        * parent_message;    /* NOT USED */
{
    return buffer_to_header(&headers->from,cnt,
			    free_only,mailer_info,
			    aview,h->flags);
}

/* To: -header -------------------------------------------------------- */

static void inpval_to (h,headers,cnt,hdr_charset)
     struct hdr_menu_item     * h;                
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     charset_t                  hdr_charset;       /* NOT USED */
{
    hdr_to_buffer(headers->to,cnt,h->flags);
}


static struct string * expval_to (h,headers,hdr_charset)
     struct hdr_menu_item *h;                   /* NOT USED */
     struct mailing_headers *headers;
     charset_t hdr_charset;                     /* NOT USED */
{
    return hdr_to_expval(headers->to);
}

static int hdrproc_to  (h,headers,cnt,free_only,mailer_info,hdr_charset,
			aview,page,prompt_area,parent_message)
     struct hdr_menu_item     * h;                  
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;
     struct mailer_info       * mailer_info;
     charset_t                  hdr_charset;        /* NOT USED */
     struct AliasView         * aview;
     struct menu_context      * page;               /* NOT USED */
     struct menu_context      * prompt_area;        /* NOT USED */
     struct header_rec        * parent_message;     /* NOT USED */
{
    return buffer_to_header(&headers->to,cnt,
			    free_only,mailer_info,
			    aview,h->flags);
}

/* CC: -header -------------------------------------------------------- */

static void inpval_cc (h,headers,cnt,hdr_charset)
     struct hdr_menu_item     * h;                   
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     charset_t                  hdr_charset;            /* NOT USED */
{
    hdr_to_buffer(headers->cc,cnt,h->flags);
}

static struct string * expval_cc (h,headers,hdr_charset)
     struct hdr_menu_item *h;                      /* NOT USED */
     struct mailing_headers *headers;
     charset_t hdr_charset;                        /* NOT USED */
{
    return hdr_to_expval(headers->cc);
}

static int hdrproc_cc  (h,headers,cnt,free_only,mailer_info,hdr_charset,
			aview,page,prompt_area,parent_message)
     struct hdr_menu_item     * h;                       
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;
     struct mailer_info       * mailer_info;
     charset_t                  hdr_charset;        /* NOT USED */
     struct AliasView         * aview; 
     struct menu_context      * page;               /* NOT USED */
     struct menu_context      * prompt_area;        /* NOT USED */
     struct header_rec        * parent_message;     /* NOT USED */
{
    return buffer_to_header(&headers->cc,cnt,free_only,mailer_info,
			    aview,h->flags);
}

/* BCC: -header -------------------------------------------------------- */

static void inpval_bcc (h,headers,cnt,hdr_charset)
     struct hdr_menu_item     * h;                      
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     charset_t                  hdr_charset;      /* NOT USED */
{
    hdr_to_buffer(headers->bcc,cnt,h->flags);
}

static struct string * expval_bcc (h,headers,hdr_charset)
     struct hdr_menu_item *h;                      /* NOT USED */
     struct mailing_headers *headers;
     charset_t hdr_charset;                        /* NOT USED */
{
    return hdr_to_expval(headers->bcc);
}

static int hdrproc_bcc  (h,headers,cnt,free_only,mailer_info,hdr_charset,
			 aview,page,prompt_area,parent_message)
     struct hdr_menu_item     * h;                      
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;
     struct mailer_info       * mailer_info;
     charset_t                  hdr_charset;            /* NOT USED */
     struct AliasView         * aview;
     struct menu_context      * page;                   /* NOT USED */
     struct menu_context      * prompt_area;            /* NOT USED */
     struct header_rec        * parent_message;         /* NOT USED */
{
    return buffer_to_header(&headers->bcc,cnt,
			    free_only,mailer_info,
			    aview,h->flags);
}



/* Subject: -header -------------------------------------------------------- */

static void inpval_subject (h,headers,cnt,hdr_charset)
     struct hdr_menu_item     * h;            /* NOT USED */
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     charset_t                  hdr_charset;
{
    clear_hdrmenu_context(cnt);
    
    if (headers->subject) {
	cnt->buffer = dup_string(headers->subject);
    } else
	cnt->buffer = new_string(hdr_charset);
}

static struct string * expval_subject (h,headers,hdr_charset)
     struct hdr_menu_item *h;                            /* NOT USED */
     struct mailing_headers *headers;
     charset_t hdr_charset;
{
    struct string * r;

    if (headers->subject)
	r = dup_string(headers->subject);
    else
	r = new_string(hdr_charset);
    return r;
}


static int hdrproc_subject  (h,headers,cnt,free_only,mailer_info,hdr_charset,
			     aview,page,prompt_area, parent_message)
     struct hdr_menu_item *h;                          /* NOT USED */
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;
     struct mailer_info       * mailer_info;           /* NOT USED */
     charset_t                  hdr_charset;           /* NOT USED */
     struct AliasView         * aview;                 /* NOT USED */
     struct menu_context      * page;                  /* NOT USED */
     struct menu_context      * prompt_area;           /* NOT USED */
     struct header_rec        * parent_message;        /* NOT USED */
{
    if (HDRMENU_CONTEXT_magic != cnt->magic)
	panic("INPUT PANIC",__FILE__,__LINE__,
              "hdrproc_subject",
              "Bad magic number (hdrmenu_context)",0);
    
    switch (free_only) {
	charset_t do_utf7;

    case hdrproc_set_value:
	do_utf7 = want_convert_to_utf7(cnt->buffer);
	
	if (headers->subject)
	    free_string(&(headers->subject));
	
	if (do_utf7) {
	    
	    headers->subject = convert_string(do_utf7,cnt->buffer,1);
	    
	} else {	   
	    headers->subject = cnt->buffer;
	    cnt->buffer = NULL;
	}
	break;
    case hdrproc_free_only:
	break;
    }

    clear_hdrmenu_context(cnt);
        
    return 0;
}

/* Reply-To: -header -------------------------------------------------- */

static void inpval_reply_to (h,headers,cnt,hdr_charset)
     struct hdr_menu_item     * h;                      
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     charset_t                  hdr_charset;    /* NOT USED */
{    
    hdr_to_buffer(headers->reply_to,cnt,h->flags);
}

static struct string * expval_reply_to (h,headers,hdr_charset)
     struct hdr_menu_item *h;                       /* NOT USED */
     struct mailing_headers *headers;
     charset_t hdr_charset;                         /* NOT USED */
{
    return hdr_to_expval(headers->reply_to);
}

static int hdrproc_reply_to  (h,headers,cnt,free_only,
			      mailer_info,hdr_charset,
			      aview,page,prompt_area,parent_message)
     struct hdr_menu_item     * h;                  
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;
     struct mailer_info       * mailer_info;
     charset_t                  hdr_charset;       /* NOT USED */
     struct AliasView         * aview;
     struct menu_context      * page;              /* NOT USED */
     struct menu_context      * prompt_area;       /* NOT USED */
     struct header_rec        * parent_message;    /* NOT USED */
{
    return buffer_to_header(&headers->reply_to,cnt,
			    free_only,mailer_info,
			    aview,h->flags);
}

/* Action: -header ---------------------------------------------------- */

static void inpval_action (h,headers,cnt,hdr_charset)
     struct hdr_menu_item     * h;                 /* NOT USED */
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     charset_t                  hdr_charset;
{
    clear_hdrmenu_context(cnt);

    text_to_buffer(headers->action,&(cnt->buffer),hdr_charset); 
}

static struct string * expval_action (h,headers,hdr_charset)
     struct hdr_menu_item *h;                      /* NOT USED */
     struct mailing_headers *headers;
     charset_t hdr_charset;
{
    return text_to_expval(headers->action,hdr_charset);
}

static int hdrproc_action  (h,headers,cnt,
			    free_only,mailer_info,hdr_charset,
			    aview,page,prompt_area,parent_message)
     struct hdr_menu_item     * h;                  /* NOT USED */
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;
     struct mailer_info       * mailer_info;
     charset_t                  hdr_charset;
     struct AliasView         * aview;
     struct menu_context      * page;
     struct menu_context      * prompt_area;       /* NOT USED */
     struct header_rec        * parent_message;    /* NOT USED */
{
    int r;

    if (HDRMENU_CONTEXT_magic != cnt->magic)
	panic("INPUT PANIC",__FILE__,__LINE__,
              "hdrproc_action",
              "Bad magic number (hdrmenu_context)",0);
    
    r = buffer_to_text(&headers->action,&(cnt->buffer),free_only,hdr_charset);

    clear_hdrmenu_context(cnt);
    
    return r;
}

/* Expires: -header ---------------------------------------------------- */

static void inpval_expires (h,headers,cnt,hdr_charset)
     struct hdr_menu_item     * h;                /* NOT USED */
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     charset_t hdr_charset;
{
    clear_hdrmenu_context(cnt);
    

    if (headers->expires) {
	int days_ahead =
	    get_expanded_expires_days_ahead(headers->expires);
	
	DPRINT(Debug,16,(&Debug,"inpval_expires: days_ahead=%d\n",
			 days_ahead));
	
	if (days_ahead > 0) {
	    cnt->buffer = format_string(FRM("%d"),
				  days_ahead);
	} else
	    cnt->buffer = new_string(hdr_charset);
     } else
	cnt->buffer = new_string(hdr_charset);
}

static struct string * expval_expires (h,headers,hdr_charset)
     struct hdr_menu_item *h;                        /* NOT USED */
     struct mailing_headers *headers;
     charset_t hdr_charset;
{
    struct string * T = NULL;

    if (headers->expires) {
	T = get_expanded_expires_textual(headers->expires);

	if (!T) {
	    const struct tm * tm =
		get_expanded_expires_value(headers->expires);
	    enum print_time print_time
		= get_expanded_expires_print_time(headers->expires);

	    if (tm)
		T = days_ahead_text(tm,print_time);

	    if (T)
		set_expanded_expires_textual(headers->expires,
					     T);
	}	    
    }

    if (!T)
	T =  new_string(hdr_charset);

    DPRINT(Debug,16,(&Debug,"expval_expires=%S\n",
		     T));
    
    return T; 
}

/*
 * Process the expires header.  The value entered by the user is interpreted
 * as a number of days, and is expanded out to a date specification.  If
 * an error occurs a message is printed, the expanded value is cleared
 * out, and a -1 is returned.
 */


static int hdrproc_expires  (h,headers,cnt,
			     free_only,mailer_info,hdr_charset,
			     aview,page,prompt_area,parent_message)
     struct hdr_menu_item     * h;                 /* NOT USED */
     struct mailing_headers   * headers;           /* NOT USED */
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;
     struct mailer_info       * mailer_info;
     charset_t                  hdr_charset;       /* NOT USED */
     struct AliasView         * aview;             /* NOT USED */
     struct menu_context      * page;              /* NOT USED */
     struct menu_context      * prompt_area;       
     struct header_rec        * parent_message;    /* NOT USED */
{
    int ret_val = 0;

    if (HDRMENU_CONTEXT_magic != cnt->magic)
	panic("INPUT PANIC",__FILE__,__LINE__,
              "hdrproc_expires",
              "Bad magic number (hdrmenu_context)",0);
    
    switch (free_only) {
    case hdrproc_set_value: {

	long days;
	int fail_pos;

	struct string * X = NULL;
	struct tm * val;
	int LINES, COLUMNS;
	enum print_time print_time = Expires_have_time ?  print_date_and_time : print_date_only;

	menu_get_sizes(page,&LINES, &COLUMNS);

	/* initialize expanded date spec to empty */
	
	if (headers->expires)
	    free_expanded_expires(& (headers->expires));

	DPRINT(Debug,16,(&Debug,"hdrproc_expires: cnt->buffer=%S\n",
			 cnt->buffer));
	

	/* blank is ok */
	if (0 == string_len(cnt->buffer)) {
	    goto free_it;
	}

	/* verify the number of days is valid and in range */
	days = string_to_long(cnt->buffer,&fail_pos);
	if (days < 1 || fail_pos >= 0) {
	    menu_print_format_center(prompt_area,3, 
				     CATGETS(elm_msg_cat, ElmSet,
					     ElmHdrmenuExpiresNotNumber,
					     "Expiration must be specified as a number of days."));
	    ret_val = -1;
	    goto free_it;
	}	

	DPRINT(Debug,16,(&Debug,"hdrproc_expires: days=%d\n",
			 days));
	
	if (days > 8*7) {
	    menu_print_format_center(prompt_area,3, 
				     CATGETS(elm_msg_cat, ElmSet,
					     ElmHdrmenuExpiresOutOfRange,
					     "Expiration date must be within eight weeks of today."));
	    ret_val = -1;
	    goto free_it;
	}

	/* convert number of days to a date */
	val = days_ahead_tm(days);
	if (!val) {
	    ret_val = -1;
	    DPRINT(Debug,16,(&Debug,"hdrproc_expires: days_ahead_tm failed\n"));
	    goto free_it;
	}

	X = days_ahead_text(val,print_time);

	headers->expires
	    = new_expanded_expires(X,val,days,
				   print_time);

	if (X)
	    free_string(&X);
	ret_val = 0;
    }
	break;
    case hdrproc_free_only:
	break;
    }

 free_it:
    clear_hdrmenu_context(cnt);

    DPRINT(Debug,16,(&Debug,"hdrproc_expires=%d%s\n",
		     ret_val,
		     0 == ret_val ? " (OK)" : ""));
    
    return ret_val;
}

/* Priority: -header ---------------------------------------------------- */

static void inpval_priority (h,headers,cnt,hdr_charset)
     struct hdr_menu_item     * h;         /* NOT USED */
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     charset_t hdr_charset;
{
    clear_hdrmenu_context(cnt);

    text_to_buffer(headers->priority,&(cnt->buffer),hdr_charset); 
}

static struct string * expval_priority (h,headers,hdr_charset)
     struct hdr_menu_item *h;
     struct mailing_headers *headers;
     charset_t hdr_charset;
{
    return text_to_expval(headers->priority,hdr_charset);
}

static int hdrproc_priority(h,headers,cnt,
			    free_only,mailer_info,hdr_charset,
			    aview,page,prompt_area,parent_message)
     struct hdr_menu_item     * h;             /* NOT USED */
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;     /* NOT USED */
     struct mailer_info       * mailer_info;
     charset_t                  hdr_charset;
     struct AliasView         * aview;         /* NOT USED */
     struct menu_context      * page;          /* NOT USED */
     struct menu_context      * prompt_area;   /* NOT USED */
     struct header_rec *       parent_message; /* NOT USED */
{
    int r;

    if (HDRMENU_CONTEXT_magic != cnt->magic)
	panic("INPUT PANIC",__FILE__,__LINE__,
              "hdrproc_priority",
              "Bad magic number (hdrmenu_context)",0);
    
    r = buffer_to_text(&headers->priority,&(cnt->buffer),free_only,hdr_charset);

    clear_hdrmenu_context(cnt);
    
    return r;
}

/* Precedence: -header ---------------------------------------------------- */

static void inpval_precedence (h,headers,cnt,hdr_charset)
     struct hdr_menu_item     * h;           /* NOT USED */ 
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     charset_t                  hdr_charset;
{
    clear_hdrmenu_context(cnt);
    
    text_to_buffer(headers->precedence,&(cnt->buffer),hdr_charset); 
}

static struct string * expval_precedence (h,headers,hdr_charset)
     struct hdr_menu_item *h;
     struct mailing_headers *headers;
     charset_t hdr_charset;
{
    return text_to_expval(headers->precedence,hdr_charset);
}

/*
 * Process the precedence header.  The value entered by the user is
 * checked against the list of allowed precedences, if one exists.  If
 * the precedence has a priority assigned to it, then an empty priority
 * field will be filled in with that value.  If an error occurs a message
 * is printed, the precedence value is cleared out, and a -1 is returned.
 */

static int hdrproc_precedence(h,headers,cnt,
			      free_only,mailer_info,
			      hdr_charset,aview,page,prompt_area,
			      parent_message)
     struct hdr_menu_item     * h;              /* NOT USED */
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;
     struct mailer_info       * mailer_info;    /* NOT USED */
     charset_t                  hdr_charset;
     struct AliasView         * aview;          /* NOT USED */
     struct menu_context      * page;
     struct menu_context      * prompt_area;
     struct header_rec        * parent_message; /* NOT USED */
{
    int ret_val = 0;

    if (HDRMENU_CONTEXT_magic != cnt->magic)
	panic("INPUT PANIC",__FILE__,__LINE__,
              "hdrproc_precedence",
              "Bad magic number (hdrmenu_context)",0);
        
    switch (free_only) {
    case hdrproc_set_value: {
	char *buf, *bp;
	char *prec = NULL, *prio = NULL;
	int Lptr = string_len(cnt->buffer);
	struct string * X = NULL;
	int LINES, COLUMNS;

	menu_get_sizes(page,&LINES, &COLUMNS);

	/* empty is ok */
	if (0 == Lptr) {
	    if (headers->precedence)
		free(headers->precedence);
	    headers->precedence = NULL;
	    goto free_it;
	}

	if (can_ascii_string(cnt->buffer))
	    X = ascify_string(cnt->buffer);
	else
	    X = convert_string(hdr_charset,cnt->buffer,0);

	/* if there are no restrictions on precedence then anything is ok */
	if (allowed_precedences[0] == '\0') {
	    if (headers->precedence)
		free(headers->precedence);
	    headers->precedence = us2s(stream_from_string(X,0,NULL));
	    free_string(&X);

	    goto free_it;
	}

	/* the "allowed_precedences[]" format is: */
	/*   precedence[:priority-value] precedence[:priority-value] ... */
	bp = buf = safe_strdup(allowed_precedences);

	while ((prec = strtok(bp, " \t\n")) != NULL) {
	    struct string *T = NULL;
	    int r;

	    bp = NULL;
	    if ((prio = index(prec, ':')) != NULL)
		*prio++ = '\0';

	    /* TODO: This is ineffective 
	       TODO: Should be case-insensitive
	    */
	    T = new_string2(system_charset,s2us(prec));
	    r = string_cmp(cnt->buffer,T,
			   999 /* == Not equal if not comparable */ );

	    free_string(&T);

	    if (0 == r)
		break;
	}

	/* Do not 	
	   free(buf);
	   here because prio pointer must be valid!
	*/

	/* see if we reached the end of the list without a match */
	if (prec == NULL) {
	    menu_print_format_center(prompt_area,3, 
				     CATGETS(elm_msg_cat, ElmSet,
					     ElmHdrmenuPrecedenceBadValue,
					     "Unknown precedence value specified."));	    
	    ret_val = -1;
	    free_string(&X);
	    free(buf);
	    goto free_it;
	}

	if (headers->precedence)
	    free(headers->precedence);
	headers->precedence = us2s(stream_from_string(X,0,NULL));

	/* see if this precedence has an associated priority */
	if (prio != NULL && headers->priority == NULL) {
	    headers->priority = strmcpy(headers->priority,prio);
	    hdrmenu_put(&hmenu_priority, FALSE,headers,hdr_charset,page);
	}

	free_string(&X);
	free(buf);

	ret_val = 0;
    }
	break;
    case hdrproc_free_only:
	break;
    }

 free_it:
    clear_hdrmenu_context(cnt);

    return ret_val;
}

/* In-Reply-To: -header ---------------------------------------------------- */

static void inpval_in_reply_to (h,headers,cnt,hdr_charset)
     struct hdr_menu_item     * h;
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     charset_t                  hdr_charset;
{
    clear_hdrmenu_context(cnt);
    
    cnt->buffer = new_string(hdr_charset);	

    if (headers->in_reply_to) {
	int count = references_item_count(headers->in_reply_to);
	int i;
	int found = 0;

	for (i = 0; i < count; i++) {
	    const struct message_id * id = references_get_message_id(headers->in_reply_to,i);
	    const struct string * phrase = references_get_phrase(headers->in_reply_to,i);
	    const struct string * comment_tail = references_get_comment(headers->in_reply_to,i);

	    if (id) {
		char * XX                     = get_message_id_ascii_brackets(id);
		const struct string * comment_id = message_id_comment(id);

		if (XX) {
		    if (found)
			fill_ascii_to_string(cnt->buffer,1,' ');
		    found++;

		    add_ascii_to_string(cnt->buffer,s2us(XX));

		    free(XX);
		}

		if (comment_id) {
		    struct string * XX = string_quote_comment(comment_id);

		    if (found)
			fill_ascii_to_string(cnt->buffer,1,' ');
		    found++;
		    
		    fill_ascii_to_string(cnt->buffer,1,'(');
		    append_string(&(cnt->buffer),XX,1);
		    fill_ascii_to_string(cnt->buffer,1,')');

		    free_string(&XX);		    
		}
	    }

	    if (phrase) {
		if (found)
		    fill_ascii_to_string(cnt->buffer,1,' ');
		found++;

		append_quoted_to_string(&(cnt->buffer),phrase);
	    }

	    if (comment_tail) {
		struct string * XX = string_quote_comment(comment_tail);
		
		if (found)
		    fill_ascii_to_string(cnt->buffer,1,' ');
		found++;
		
		fill_ascii_to_string(cnt->buffer,1,'(');
		append_string(&(cnt->buffer),XX,1);
		fill_ascii_to_string(cnt->buffer,1,')');
		
		free_string(&XX);		    
	    }	   	    
	}
    } 
}

static struct string * expval_in_reply_to (h,headers,hdr_charset)
     struct hdr_menu_item *h;
     struct mailing_headers *headers;
     charset_t hdr_charset;
{

    struct string * m = new_string(hdr_charset);

    if (headers->in_reply_to) {
	int count = references_item_count(headers->in_reply_to);
	int i;

	int found = 0;
	
	for (i = 0; i < count; i++) {
	    const struct message_id * id = references_get_message_id(headers->in_reply_to,i);
	    const struct string * phrase = references_get_phrase(headers->in_reply_to,i);
	    const struct string * comment_tail = references_get_comment(headers->in_reply_to,
									i);
	    
	    if (id) {		
		char * XX                     = get_message_id_ascii_brackets(id);
		const struct string * comment_id = message_id_comment(id);

		if (XX) {
		    if (found)
			fill_ascii_to_string(m,1,' ');
		    found++;

		    add_ascii_to_string(m,s2us(XX));

		    free(XX);
		}

		/* We do not (backslash) quote displayed (only) values */
		if (comment_id) {
		    if (found)
			fill_ascii_to_string(m,1,' ');
		    found++;

		    fill_ascii_to_string(m,1,'(');
		    append_string(&m,comment_id,1);
		    fill_ascii_to_string(m,1,')');
		}
	    }

	    if (phrase) {
		enum phrase_display_mode_v pd = 
		    give_dt_enumerate_as_int(&phrase_display_mode);

		if (found)
			fill_ascii_to_string(m,1,' ');
		found++;

		switch (pd) {
		case phrase_display_plain:
		    append_string(&m,phrase,0);
		    break;

		default:
		    /* We do not (backslash) quote displayed (only) values */
		    fill_ascii_to_string(m,1,'"');
		    append_string(&m,phrase,1);
		    fill_ascii_to_string(m,1,'"');
		    break;
		}
	    }

	    if (comment_tail) {
		/* We do not (backslash) quote displayed (only) values */

		if (found)
		    fill_ascii_to_string(m,1,' ');
		found++;
		
		fill_ascii_to_string(m,1,'(');
		append_string(&m,comment_tail,1);
		fill_ascii_to_string(m,1,')');
		
	    }
	}
    }

    return m;
}

static  void append_comment_unquote P_((struct string **comment, const struct string *token));
static  void append_comment_unquote(comment,token)
     struct string **comment; 
     const struct string *token;
{
    int L = string_len(token);
    struct string * temp = new_string(get_string_type(token));
    int X;

    /* Remove ( ) and unquote \ */

    for (X = 0; X < L; X++) {
	uint16 code = give_unicode_from_string(token,X);

	if (0x005C /* \ */ == code) {
	    if (++X < L)
		code = give_unicode_from_string(token,X);
	    else 
		break;

	} else if (0x0028 /* ( */ == code) {
	    if (0 == X)
		continue;
	} else if (0x0029 /* ) */ == code) {
	    if (L-1 == X)
		continue;
	}

	add_unicode_to_string(temp,1,&code);
    }

    /* Does charset convert */
    append_string(comment,temp,0);
    free_string(&temp);
}


static void check_references P_((struct header_rec      * current_header,
				 struct mailing_headers *headers));
static void check_references(current_header,headers)
     struct header_rec      * current_header;
     struct mailing_headers * headers;
{

    if (! headers->in_reply_to ||
	1 != references_item_count(headers->in_reply_to)) {

    clear:
	if (headers->references) {
	    DPRINT(Debug,7,(&Debug,"check_references: Clearing References header field\n"));
	    
	    free_references(& headers->references);
	}

    } else {
	const struct message_id * ID = 
	    references_get_message_id(headers->in_reply_to,0);
	
	if (! ID)
	    goto clear;

#if DEBUG
	{
	    char * A = get_message_id_ascii_brackets(ID);
	    DPRINT(Debug,7,(&Debug,"check_references: in-reply-to message-id: %s\n",
			    A));
	    free(A);

	}
#endif


	if (headers->references) {
	    int count = references_item_count(headers->references);
	    const struct message_id * LID = NULL;

	    if (count > 0)
		LID = references_get_message_id(headers->references,count-1);
	    
	    if (!LID) {
		
		DPRINT(Debug,7,(&Debug,
				"check_references: no message-id found -- resetting References header field\n"));
		
		free_references(& headers->references);
		goto generate;
	    }

#if DEBUG
	    {
		char * A = get_message_id_ascii_brackets(LID);
		DPRINT(Debug,7,(&Debug,"check_references: references last message-id: %s\n",
				A));
		free(A);
	    }
#endif
	    
	    if (! same_message_id(ID,LID,1)) {
		DPRINT(Debug,7,(&Debug,
				"check_references: no same message-id -- resetting References header field\n"));
		
		free_references(& headers->references);
		goto generate;
		
	    }

	    DPRINT(Debug,9,(&Debug,
			    "check_references: References header field OK\n"));
	   
	} else {

	generate:

	    if (current_header && current_header->message_id &&
		same_message_id(ID,current_header->message_id,1)) {

		DPRINT(Debug,7,(&Debug,
				"check_references: Generating normal References header field\n"));

		generate_references(current_header,headers);

	    } else {
		DPRINT(Debug,7,(&Debug,
				"check_references: Generating dummy References header field\n"));

		headers->references = new_references(ID,NULL,NULL);		       

	    }
	}	
    }
}


static int hdrproc_in_reply_to (h,headers,cnt,free_only,mailer_info,
				hdr_charset,aview,page,prompt_area,parent_message)
     struct hdr_menu_item     * h;
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;
     struct mailer_info       * mailer_info;
     charset_t                  hdr_charset;
     struct AliasView         * aview;
     struct menu_context      * page;
     struct menu_context      * prompt_area;        
     struct header_rec        * parent_message;   
{
    int ret_val = 0;

    if (HDRMENU_CONTEXT_magic != cnt->magic)
	panic("INPUT PANIC",__FILE__,__LINE__,
              "hdrproc_in_reply_to",
              "Bad magic number (hdrmenu_context)",0);
    
    if (!free_only) {
	struct references * new_value = NULL;
	struct string_token * tokenized;
	int i;

	int LINES, COLUMNS;

	menu_get_sizes(page,&LINES, &COLUMNS);

	if (! cnt->buffer)
	    goto clear;

	tokenized = string_tokenize(cnt->buffer, TOK_mail);

	if (!tokenized)
	    goto clear;

	for (i = 0; tokenized[i].token; ) {
	    struct message_id * id  = NULL;
	    struct string * comment = NULL;
	    struct string * phrase  = NULL;

	    /* Eat comments */
	    while (tokenized[i].token &&
		   (0x0020 /* SPACE */ == tokenized[i].special ||
		    0x0028 /* ( */     == tokenized[i].special)) {
		
		if (0x0028 /* ( */     == tokenized[i].special) {
		    if (comment)
			fill_ascii_to_string(comment,1,' ');	       
		    append_comment_unquote(&comment,tokenized[i].token); 

		    DPRINT(Debug,15,(&Debug,
				     "hdrproc_in_reply_to: [%d] comment: %S\n",
				     i,tokenized[i].token));

		}
		i++;
			    
	    }
	    
	    if (! tokenized[i].token)
		goto done;

	    if (0x003C  /* < */  == tokenized[i].special) { 
		struct string * left   = NULL;
		char * left_asc        = NULL;
		struct string * domain = NULL;
		char * domain_asc      = NULL;

		DPRINT(Debug,15,(&Debug,
				 "hdrproc_in_reply_to: [%d] '<': %S\n",
				 i,tokenized[i].token));

		i++;

		while (tokenized[i].token && 
		       0x003E  /* > */ != tokenized[i].special  &&
		       0x0040  /* @ */ != tokenized[i].special) {
		    
		    if (0x0020 /* SPACE */ == tokenized[i].special) {
			/* Empty */

			DPRINT(Debug,15,(&Debug,
					 "hdrproc_in_reply_to: [%d] space: %S\n",
					 i,tokenized[i].token));


		    } else if (0x0028 /* ( */     == tokenized[i].special) {
			if (comment)
			    fill_ascii_to_string(comment,1,' ');	       
			append_comment_unquote(&comment,tokenized[i].token); 

			DPRINT(Debug,15,(&Debug,
					 "hdrproc_in_reply_to: [%d] comment: %S\n",
					 i,tokenized[i].token));


		    } else {
			if (!can_ascii_string(tokenized[i].token))
			    menu_print_format_center(prompt_area,3, 
						     CATGETS(elm_msg_cat, MeSet,
							     MeAsciiRequiredTok,
							     "Ascii required for token %S"),
						     tokenized[i].token);

			append_string(&left,tokenized[i].token,0);

			DPRINT(Debug,15,(&Debug,
					 "hdrproc_in_reply_to: [%d] left: %S\n",
					 i,tokenized[i].token));


		    }
		    i++;
		}
		       

		if (tokenized[i].token &&
		    0x0040  /* @ */ == tokenized[i].special) {

		    DPRINT(Debug,15,(&Debug,
				     "hdrproc_in_reply_to: [%d] '@': %S\n",
				     i,tokenized[i].token));

		    i++;
		    
		    while (tokenized[i].token && 
			   0x003E  /* > */ != tokenized[i].special) {
			
			if (0x0020 /* SPACE */ == tokenized[i].special) {
			    /* Empty */

			    DPRINT(Debug,15,(&Debug,
					     "hdrproc_in_reply_to: [%d] space: %S\n",
					     i,tokenized[i].token));


			} else if (0x0028 /* ( */     == tokenized[i].special) {
			    if (comment)
				fill_ascii_to_string(comment,1,' ');
			    append_comment_unquote(&comment,tokenized[i].token); 

			    DPRINT(Debug,15,(&Debug,
					     "hdrproc_in_reply_to: [%d] comment: %S\n",
					     i,tokenized[i].token));

			} else {
			    if (!can_ascii_string(tokenized[i].token))
				menu_print_format_center(prompt_area,3, 
							 CATGETS(elm_msg_cat, MeSet,
								 MeAsciiRequiredTok,
								 "Ascii required for token %S"),
							 tokenized[i].token);
			    
			    append_string(&domain,tokenized[i].token,0);

			    DPRINT(Debug,15,(&Debug,
					     "hdrproc_in_reply_to: [%d] domain: %S\n",
					     i,tokenized[i].token));

			}
			i++;
		    }

		} else {
		    menu_print_format_center(prompt_area,3, 
					     CATGETS(elm_msg_cat, ElmSet,
						     ElmHdrmenuMissAt,
						     "Missing @ after < on in-reply-to header."));
		    goto failure;
		} 
		    
		if (! tokenized[i].token ||
		    0x003E  /* > */ != tokenized[i].special) {

		    menu_print_format_center(prompt_area,3, 
					     CATGETS(elm_msg_cat, ElmSet,
						     ElmHdrmenuMissAngle,
						     "Missing > after < on in-reply-to header."));
		    
		failure:

		    if (left)
			free_string(&left);
		    if (domain)
			free_string(&domain);
		    if (comment)
			free_string(&comment);
			    
		    free_string_tokenized(&tokenized);
		    
		    if (new_value)
			free_references(& new_value);

		    ret_val = -1;
		    goto free_it;
		}
		
		DPRINT(Debug,15,(&Debug,
				 "hdrproc_in_reply_to: [%d] '>': %S\n",
				 i,tokenized[i].token));
		i++;


		if (left) {
		    struct string * X = NULL;
		    
		    if (can_ascii_string(left))
			X = ascify_string(left);
		    else
			X = convert_string(hdr_charset,left,0);
		    
		    left_asc = us2s(stream_from_string(X,0,NULL));	       
		}

		if (domain) {
		    struct string * X = NULL;
		    
		    if (can_ascii_string(domain))
			X = ascify_string(domain);
		    else
			X = convert_string(hdr_charset,domain,0);
		    
		    domain_asc = us2s(stream_from_string(X,0,NULL));	       
		}

		/* Eat comments */
		while (tokenized[i].token &&
		       (0x0020 /* SPACE */ == tokenized[i].special ||
			0x0028 /* ( */     == tokenized[i].special)) {
		    
		    if (0x0028 /* ( */     == tokenized[i].special) {
			if (comment)
			    fill_ascii_to_string(comment,1,' ');	       
			append_comment_unquote(&comment,tokenized[i].token); 

			DPRINT(Debug,15,(&Debug,
					 "hdrproc_in_reply_to: [%d] comment: %S\n",
					 i,tokenized[i].token));
		    }
		    i++;		    
		}
			
		if (comment) {
		    charset_t do_utf7 = want_convert_to_utf7(comment);

		    if (do_utf7) {
			struct string *tmp = convert_string(do_utf7,comment,1);

			free_string(&comment);
			comment = tmp;
		    }
		}

		if (left && domain) 	       
		    id = new_message_id(left_asc,domain_asc,comment);
		else {
		    menu_print_format_center(prompt_area,3, 
					     CATGETS(elm_msg_cat, ElmSet,
						     ElmHdrmenuBadMessageId,
						     "Bad message-id on in-reply-to header."));

		    
		    if (left) {
			free_string(&left);
			free(left_asc);
		    }
		    if (domain) {
			free_string(&domain);
			free(domain_asc);
		    }
		    if (comment)
			free_string(&comment);
		    
		    free_string_tokenized(&tokenized);
		    
		    if (new_value)
			free_references(& new_value);
		    
		    ret_val = -1;
		    goto free_it;
		}

		if (left) {
		    free_string(&left);
		    free(left_asc);
		}
		if (domain) {
		    free_string(&domain);
		    free(domain_asc);
		}
		if (comment)
		    free_string(&comment);				   		
	    }


	    while (tokenized[i].token &&
		      0x003C  /* < */  != tokenized[i].special) { 
		
		if (0x0020 /* SPACE */ == tokenized[i].special) {
		    if (phrase)
			fill_ascii_to_string(phrase,1,' ');

		    DPRINT(Debug,15,(&Debug,
				     "hdrproc_in_reply_to: [%d] space: %S\n",
				     i,tokenized[i].token));


		} else if (0x0028 /* ( */     == tokenized[i].special) {
		    if (comment)
			fill_ascii_to_string(comment,1,' ');
		    append_comment_unquote(&comment,tokenized[i].token); 

		    DPRINT(Debug,15,(&Debug,
				     "hdrproc_in_reply_to: [%d] comment: %S\n",
				     i,tokenized[i].token));


		} else {
		    struct string *XX = 
			unquote_string(tokenized[i].token,
				       unq_can_print_error,
				       NULL,0,NULL);
		    
		    append_string(&phrase,XX,1);
		    free_string(&XX);

		    DPRINT(Debug,15,(&Debug,
				     "hdrproc_in_reply_to: [%d] phrase: %S\n",
				     i,tokenized[i].token));
		}

		i++;
	    }
	    
	done:
	    if (id || phrase || comment) {
		
		if (phrase) {
		    charset_t do_utf7 = want_convert_to_utf7(phrase);

		    if (do_utf7) {
			struct string *tmp = convert_string(do_utf7,phrase,1);

			free_string(&phrase);
			phrase = tmp;
		    }
		}

		if (comment) {
		    charset_t do_utf7 = want_convert_to_utf7(comment);
		    
		    if (do_utf7) {
			struct string *tmp = convert_string(do_utf7,comment,1);
			
			free_string(&comment);
			comment = tmp;
		    }
		}
		
		if (!new_value)
		    new_value = new_references(id,phrase,comment);
		else
		    references_add_item(new_value,id,phrase,comment);
		
		if (phrase && !add_irt_phrase)
		    menu_print_format_center(prompt_area,3, 
					     CATGETS(elm_msg_cat, ElmSet,
						     ElmHdrmenuPhraseIRT,
						     "In-reply-to header should include only message-id"));
		
		if (id)
		    free_message_id(&id);
		if (phrase)
		    free_string(&phrase);
		if (comment)
		    free_string(&comment);
	    }	   
	}

	free_string_tokenized(&tokenized);

    clear:
	if (headers->in_reply_to)
	    free_references(& headers->in_reply_to);
	
	headers->in_reply_to = new_value;
	check_references(parent_message,headers);
    }

 free_it:
    clear_hdrmenu_context(cnt);

    return ret_val;
}

/* User defined header ---------------------------------------------------- */

static void inpval_userhdr(h,headers,cnt,hdr_charset)
     struct hdr_menu_item     * h;
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     charset_t                  hdr_charset;
{
    clear_hdrmenu_context(cnt);
    
    if (headers->user_header_count > 0 &&
	headers->user_header[0].value) {

	cnt->buffer = format_string(FRM("%s: %S"),
			     give_header_name(headers->user_header[0].name),
			     headers->user_header[0].value);
    } else
	cnt->buffer = new_string(hdr_charset);
}

static struct string * expval_userhdr(h,headers,hdr_charset)
     struct hdr_menu_item *h;
     struct mailing_headers *headers;
     charset_t hdr_charset;
{
    if (headers->user_header_count > 0 &&
	headers->user_header[0].value) 
	return format_string(FRM("%s: %S"),
			     give_header_name(headers->user_header[0].name),
			     headers->user_header[0].value);
    return new_string(hdr_charset);
}

/*
 * Process the user-defined header.  The value entered by the user is
 * verified for proper format.  If an error occurs a message is printed,
 * the expanded value is cleared out, and a -1 is returned.
 */

static int hdrproc_userhdr(h,headers,cnt,free_only,mailer_info,hdr_charset,
			   aview,page,prompt_area,parent_message)
     struct hdr_menu_item     * h;
     struct mailing_headers   * headers;
     struct hdrmenu_context   * cnt;
     enum free_only_t           free_only;
     struct mailer_info       * mailer_info;
     charset_t                  hdr_charset;
     struct AliasView         * aview;
     struct menu_context      * page;
     struct menu_context      * prompt_area;       
     struct header_rec        * parent_message;    /* NOT USED */
{
    int ret_val = 0;

    if (HDRMENU_CONTEXT_magic != cnt->magic)
	panic("INPUT PANIC",__FILE__,__LINE__,
              "hdrproc_userhdr",
              "Bad magic number (hdrmenu_context)",0);
        
    switch (free_only) {
    case  hdrproc_set_value: {
	header_ptr header_name;	
	int Xl = string_len(cnt->buffer);

	charset_t utf8 = MIME_name_to_charset("UTF-8",0);
	struct string * utf8_temp  = NULL;
	unsigned char * utf8_value = NULL;
	char * s;
	int LINES, COLUMNS;

	menu_get_sizes(page,&LINES, &COLUMNS);

	/* empty is ok */
	if (0 == Xl) {
	    int i;
	    for (i = 0; i < headers->user_header_count; i++) {
		if (headers->user_header[i].value)
		    free_string(&(headers->user_header[i].value));
		headers->user_header[i].name = NULL;
	    }
	    headers->user_header_count = 0;

	    goto free_it;
	}


	/* HACK: We use UTF-8 version of header value .... */
	
	if (!utf8) 
	    panic("CHARSET PANIC",__FILE__,__LINE__,"hdrproc_userhdr",
		  "UTF-8 not found",0);

	utf8_temp    = convert_string(utf8,cnt->buffer,0);
	utf8_value = stream_from_string(utf8_temp,0,NULL);

	
	/* make sure the header name doesn't begin with some strange 
	 * character -- Note that only US-ASCII is allowed according
	 * of standards on headers
	 */

	if (!isascii(utf8_value[0]) &&
	    !isalnum(utf8_value[0])) {
	    menu_print_format_center(prompt_area,3, 
				     CATGETS(elm_msg_cat, ElmSet,
					     ElmHdrmenuUserdefNotAlnum,
					     "The user-defined header must begin with a letter or number."));
	    ret_val = -1;

	    free(utf8_value);
	    free_string(&utf8_temp);

	    goto free_it;
	}
	
	/* locate the end of the header name */
	for (s = us2s(utf8_value) ; 
	     *s != ':' && isascii(*s) && isprint(*s) && !isspace(*s) ; 
	     ++s)
	    continue;
	
	    /* there needs to be a colon at the end of the header name */
	if (*s != ':') {
	    menu_print_format_center(prompt_area,3, 
				     CATGETS(elm_msg_cat, ElmSet,
					     ElmHdrmenuUserdefMissingColon,
					     "The user-defined header must have a colon after the field name."));
	    ret_val = -1;

	    free(utf8_value);
	    free_string(&utf8_temp);

	    goto free_it;
	}

	*s = '\0';
	s++;

	while (whitespace(*s))
	    s++;

	header_name = find_header(us2s(utf8_value),1);
		
	/* Allow user to paste mime encoded words to buffer ... */
	if (!add_to_mailing_header(headers,header_name,s,
				   1,utf8,1)) 
	    ret_val = -1;
	else
	    ret_val = 0;
    }
	break;
    case hdrproc_free_only:
	break;
    }

 free_it:
    clear_hdrmenu_context(cnt);
    
    return ret_val;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

