static char rcsid[] = "@(#)$Id: browser.c,v 2.50 2024/06/10 18:05:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.50 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_browser.h"
#include "s_elm.h"

#include "fbrowser.h"
#include "browser.h"



DEBUG_VAR(Debug,__FILE__,"ui");

#define GB_FLAGS     browser->gb_flags
#define GB_PAGE      browser->gb_page
#define GB_LINE      browser->gb_line

enum {  pv_buffer = 0,
	pv_text,
	pv_specify_text,
	pv_prev_fold,
	pv_defext,
	pv_fbrowser_filter,
	pv_defname,

	pv_COUNT
};

/* HELP TEXT -------------------------------------------------------------- */

#define ADD_TEXT(X) do { 						\
	I->browser->help_text[I->browser->help_text_count].text = X; \
	I->browser->help_text[I->browser->help_text_count].range = NULL; \
	I->browser->help_text_count++; } while(0)
#define ADD_TEXTR(X,R) do {					     \
	I->browser->help_text[I->browser->help_text_count].text = X; \
	I->browser->help_text[I->browser->help_text_count].range = R; \
	if (I->browser->help_text[I->browser->help_text_count].range) \
	    inc_pager_range_refcount(I->browser->help_text[I->browser->\
					help_text_count].range); \
	I->browser->help_text_count++; } while(0)


static void browser_add_help P_((struct enter_info *I));
static void browser_add_help(I)
     struct enter_info *I;
{
    int new_len;

    struct pager_range *range_word_wrap_max 
	= malloc_simple_pager_range(NULL,PR_WORD_WRAP|PR_MAX_WIDTH,0,0);

    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_add_help",
	      "Bad magic number",0);

    new_len = I->browser->help_text_count + 8;

    I->browser->help_text = 
	safe_array_realloc(I->browser->help_text,
			   new_len, sizeof(I->browser->help_text[0]));


    ADD_TEXTR(format_string(CATGETS(elm_msg_cat, ElmSet, 
				   ElmBrowserHelp1,
				   "You may get directory listing by giving directory name and pressing TAB key.")),
	     range_word_wrap_max);
	
    ADD_TEXT(new_string(display_charset));
    
    ADD_TEXTR(format_string(CATGETS(elm_msg_cat, ElmSet, 
				   ElmBrowserHelp2,
				   "Pressing LEFT key fills buffer with current directory name.")),
	     range_word_wrap_max);
    
    ADD_TEXTR(format_string(CATGETS(elm_msg_cat, 
				   ElmSet, ElmBrowserHelp3,
				   "Pressing RIGTH key fills buffer with current selection.")),
	     range_word_wrap_max);

    ADD_TEXTR(format_string(CATGETS(elm_msg_cat, ElmSet, 
				   ElmBrowserHelp4,
				   "Pressing UP or DOWN keys changes current selection on directory listing.")),
	     range_word_wrap_max);

    if (move_when_paged)
	ADD_TEXTR(format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmBrowserHelp5,
				       "Pressing PAGE UP or PAGE DOWN keys changes page and current selection on directory listing.")),
		 range_word_wrap_max);
    else
	ADD_TEXTR(format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmBrowserHelp6,
				       "Pressing PAGE UP or PAGE DOWN keys changes page on directory listing.")),
		 range_word_wrap_max);
	

    if (0 != (I->GB_FLAGS & GB_FBROWSER_AVAILABLE)) 
	ADD_TEXTR(format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmBrowserHelp7,
				       "Pressing F4 key toggles between this browser and Elm 2.5 style browser.")),
		 range_word_wrap_max);

    ADD_TEXTR(format_string(CATGETS(elm_msg_cat, ElmSet, 
				   ElmBrowserHelp8,
				   "Pressing F5 reloads directory.")),
	     range_word_wrap_max);

    if (I->browser->help_text_count > new_len)
	panic("BROWSER PANIC",__FILE__,__LINE__," browser_add_help",
	      "Overflow",0);

    free_pager_range(&range_word_wrap_max);
}

static void browser_add_folderhelp P_((struct enter_info *I));
static void browser_add_folderhelp(I)
     struct enter_info *I;
{
    int new_len;


    enum browser_wildcards_v bw = 
	give_dt_enumerate_as_int(&browser_wildcards);
    /*   0 == off
	 1 == on
	 2 == file browser
    */

    struct pager_range *range_word_wrap 
	= malloc_simple_pager_range(NULL,PR_WORD_WRAP,0,0);

    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__," browser_add_folderhelp",
	      "Bad magic number",0);

    new_len = I->browser->help_text_count + 22;

    I->browser->help_text = 
	safe_array_realloc(I->browser->help_text,
			   new_len, sizeof(I->browser->help_text[0]));


    if (0 != (GB_SAVE_COPY & I->GB_FLAGS)) {

	ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
					ElmBrowserHelp9,
					"Enter: <nothing> to not save a copy of the message,")));

	{
	    const char * sent_val = give_dt_estr_as_str(&sent_mail_e,
							"sentmail", 
							NULL,NULL);

	    if (sent_val)
		ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
					       ElmBrowserHelp10,
"       '<'       to save in your \"sent\" folder (%s)"),
					       sent_val));
	}


	ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmBrowserHelp11,
"       '='       to save by name (the folder name depends on whom the")));
	ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmBrowserHelp12,
"                     message is to, in the end),")));

	ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmBrowserHelp13,
"       '=?'      to save by name if the folder already exists,")));
	ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmBrowserHelp14,
"                     and if not, to your \"sent\" folder,")));
	ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmBrowserHelp15,
"       or a filename (a leading '=' denotes your folder directory).")));

    } else {
	ADD_TEXTR(format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmBrowserHelp16,
				       "%S Several options are available:"),
				I->pvector[pv_specify_text]),
		  range_word_wrap);
	
	{
	    const char * default_val = 
		give_dt_estr_as_str(&defaultfile_e,"incoming-mailbox",
				    NULL,NULL);  /* XXX */

	    if (default_val)
		ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
					       ElmBrowserHelp17,
"       '!'  will use your incoming mailbox (%s)"),			       
				       default_val));
	}
	
	
	{
	    const char * recvd_val = 
		give_dt_estr_as_str(&recvd_mail_e,"receivedmail",
				    NULL,NULL); /* XXXX */

	    if (recvd_val) {
		ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
					       ElmBrowserHelp18,
"       '>'  will use your \"received\" folder (%s)"),
				       recvd_val));

	    }
	}

	{
	    const char * sent_val = 
		give_dt_estr_as_str(&sent_mail_e,
				    "sentmail",
				    NULL,NULL);  /* XXXX */

	    if (sent_val) {
		ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
					       ElmBrowserHelp19,
"       '<'  will use your \"sent\" folder (%s)"),
					       sent_val));
			
	    }
	}

	if (I->pvector[pv_prev_fold]) {
	    	ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
					       ElmBrowserHelp20,
"       '!!' will use the previous folder (%S)"),
				       I->pvector[pv_prev_fold]));
	}
    
	ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmBrowserHelp21,
"       '@alias' will use the default folder for \"alias\"")));

	ADD_TEXT(new_string(display_charset));	 
	
	ADD_TEXTR(format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmBrowserHelp22,
				       "If you enter a filename elm will use that file.  If the file name begins \
with a prefix letter (for example '='), elm will look for the file in here:")),
		  range_word_wrap);

	ADD_TEXT(new_string(display_charset));

	{
	    /* give_dt_estr_as_str adds / to end */
	    const char * folders_val = give_dt_estr_as_str(&folders_e,
							   "maildir",
							   NULL,NULL);
	    if (folders_val) 
		ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
					       ElmBrowserHelp23,
"       '='  will look at your folder directory (%s)"),
				       folders_val));
	    else			     
		ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
					       ElmBrowserHelp24,
"       '='  will look at your folder directory")));
	}

	{
	    const struct remote_server * rs = 
		browser_get_remote_server(I->browser->dir_p);
	    
	    if (rs) {
		struct string * title = give_remote_server_title(rs);
		
		if (title) {
		    ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
						   ElmBrowserHelp25,
"       '&'  will look at current server (%S)"),
					   title));
		    
		    free_string(&title);
		} else
		    ADD_TEXT(format_string(CATGETS(elm_msg_cat, ElmSet, 
						   ElmBrowserHelp26,
"       '&'  will look at current remote server")));
	    }
	}

	if (bw) {
	    ADD_TEXT(new_string(display_charset));
	    ADD_TEXTR(format_string(CATGETS(elm_msg_cat, ElmSet, 
					   ElmBrowserHelp27,
					    "You may use wildcards (*?) in the name, similary as you do in the shell.")),
		      range_word_wrap);
	}
    }

    if (I->browser->help_text_count > new_len)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_add_folderhelp",
	      "Overflow",0);


    free_pager_range(&range_word_wrap);
}

static void browser_add_filehelp P_((struct enter_info *I));
static void browser_add_filehelp(I)
     struct enter_info *I;
{
    int bw = give_dt_enumerate_as_int(&browser_wildcards);

    struct pager_range *range_word_wrap 
	= malloc_simple_pager_range(NULL,PR_WORD_WRAP,0,0);

    int new_len;

    /*   0 == off
	 1 == on
	 2 == file browser
    */

    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__," browser_add_filehelp",
	      "Bad magic number",0);

    new_len = I->browser->help_text_count + 3;
    
    I->browser->help_text = 
	safe_array_realloc(I->browser->help_text,
			   new_len, sizeof(I->browser->help_text[0]));
       
    ADD_TEXTR(dup_string(I->pvector[pv_specify_text]),
	      range_word_wrap);
    
    ADD_TEXT(new_string(display_charset));

    if (bw) {

	ADD_TEXTR(format_string(CATGETS(elm_msg_cat, ElmSet, 
					ElmBrowserHelp28,
					"You may use wildcards (*?) in the name, similary as you do in the shell.")),
		  range_word_wrap);

    }

    if (I->browser->help_text_count > new_len)
	panic("BROWSER PANIC",__FILE__,__LINE__," browser_add_filehelp",
	      "Overflow",0);

    free_pager_range(&range_word_wrap);
}

static void browser_add_filehelp2 P_((struct enter_info *I));
static void browser_add_filehelp2(I)
     struct enter_info *I;
{

    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__," browser_add_filehelp2",
	      "Bad magic number",0);

    if (I->pvector[pv_defname]) {
	struct pager_range *range_word_wrap_max 
	    = malloc_simple_pager_range(NULL,PR_WORD_WRAP|PR_MAX_WIDTH,0,0);

	int new_len = I->browser->help_text_count + 1;
	
	I->browser->help_text = 
	    safe_array_realloc(I->browser->help_text,
			       new_len, sizeof(I->browser->help_text[0]));

	ADD_TEXTR(format_string(CATGETS(elm_msg_cat, ElmSet, 
				       ElmBrowserHelp29,
				       "Pressing F6 key fills buffer with default filename: %S"),
				I->pvector[pv_defname]),
		  range_word_wrap_max);	     
	
	if (I->browser->help_text_count > new_len)
	    panic("BROWSER PANIC",__FILE__,__LINE__," browser_add_filehelp2",
		  "Overflow",0);

	free_pager_range(&range_word_wrap_max);
    }
}


#undef ADD_TEXT
#undef ADD_TEXTR

/* -------------------------------------------------------------------------*/
	
static void adjust_comment_col P_((struct browser_edit *BE,
				   int l, int l1, int COLUMNS));
static void adjust_comment_col(BE,l,l1,COLUMNS)
     struct browser_edit *BE;
     int l; 
     int l1; 
     int COLUMNS;
{
    if (BROWSER_EDIT_magic != BE->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"adjust_comment_col",
	      "Bad magic number",0);

    if (BE->comment_col > COLUMNS) {

	if (COLUMNS > 60) {
	    int old UNUSED_VAROK = BE->comment_col;
	    
	    BE->comment_col =  COLUMNS - 30;
	    
	    DPRINT(Debug,10,
		   (&Debug, "adjust_comment_col: comment column %d => %d\n",
		    old,BE->comment_col));
	}
    }

    if (BE->comment_col > COLUMNS - l1 &&
	7 + l1 + l < COLUMNS) {
	int old UNUSED_VAROK = BE->comment_col;	

	BE->comment_col = COLUMNS - l1 -2;

	DPRINT(Debug,10,
	       (&Debug, "adjust_comment_col: comment column %d => %d\n",
		old,BE->comment_col));
    }
}

static void draw_comment P_((struct enter_info *I,
			     int line,
			     const struct string * comment1,
			     int l1, int COLUMNS));
static void draw_comment(I,line,comment1,l1,COLUMNS)
     struct enter_info   * I;
     int line;
     const struct string * comment1;
     int l1; 
     int COLUMNS;
{    
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"draw_comment",
	      "Bad magic number",0);

    if (l1 > 0 && I->browser->comment_col < COLUMNS-3) {
	
	int size = COLUMNS - I->browser->comment_col - 1;
	int visible;
	int X = 0;
	struct string * buffer1 = curses_printable_clip(comment1,&X,l1,&visible,size);
	    
	if (buffer1) {
	    int filllen = size - visible;
	    
	    if (filllen < 0)
		filllen = 0;
	    
	    menu_PutLineX(I->current_page,line,
			  I->browser->comment_col,
			  FRM(" %S%*s"),
			  buffer1,
			  filllen,"");
	    
	    free_string(&buffer1);
	} else 
	    goto no_comment;
    } else {
    no_comment:
	if (I->browser->comment_col < COLUMNS) {
	    int left = COLUMNS -  I->browser->comment_col;
	    
	    menu_PutLineX(I->current_page,line,
			  I->browser->comment_col,
			  FRM("%*s"),
			  left,"");
	}
    }
}

static void draw_line P_((struct enter_info *I,
			  int i,int lines_per_page));
static void draw_line(I,i,lines_per_page)
     struct enter_info *I;
     int i;
     int lines_per_page;
{
    int flags = 0;

    /* return reference to array -- do not free_string !!! */
    const struct string * name_line;
    const struct string * comment1 = NULL;
    int current = i == I->GB_LINE; 
    int c = '?';
    int n = ' ';
    int LINES,COLUMNS;
    int size;
    
    int X = 0;
    struct string * buffer1 = NULL;
    int l,l1 = 0;
    int visible;
    int use_comment = 0;

    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"draw_line",
	      "Bad magic number",0);

    name_line = 
	give_line_dir(I->browser->dir_p,i,&flags,
		      &comment1);
    l = string_len(name_line);

    if (comment1)
	l1 = string_len(comment1);

    menu_get_sizes(I->current_page,&LINES,&COLUMNS);

    adjust_comment_col(I->browser,l,l1,COLUMNS);

    
    switch (flags & (BROWSER_NODIR|BROWSER_NOFOLDER)) {
    case 0:                c = 'd'; break;
    case BROWSER_NODIR:    c = ' '; break;
    case BROWSER_NOFOLDER: c = 'D'; break;
    }
		
    if (ison(flags,BROWSER_HAVELOCKFILE))
	n = 'L';
    else if (ison(flags,BROWSER_SESSIONLOCK))
	n = 'S';
    else if (ison(flags,BROWSER_MARKED))
	n = 'N';

    if (current && !arrow_cursor)
	menu_StartXX(I->current_page,pg_STANDOUT);

 restart:
    if (I->browser->comment_col < COLUMNS && comment1) {
	size = I->browser->comment_col-5;
	use_comment = 1;
    } else {
	size = COLUMNS-5;
	use_comment = 0;
    }

    X = 0;
    buffer1 = curses_printable_clip(name_line,&X,l,&visible,size);

    if (buffer1) {
	int filllen = size - visible;

	if (filllen < 0)
	    filllen = 0;

	if (X < l && I->browser->comment_col < COLUMNS && use_comment) {
	    int old UNUSED_VAROK = I->browser->comment_col;

	    if (I->browser->comment_col < l+6)
		I->browser->comment_col = l+6;
	    else
		I->browser->comment_col += 5;
	    
	    DPRINT(Debug,10,
		   (&Debug, "draw_line: comment column %d => %d\n",
		    old,I->browser->comment_col));
	    
	    free_string(&buffer1);
	    goto restart;
	}

	menu_PutLineX(I->current_page,5 + i % lines_per_page,0,
		      FRM("%.2s%c%c %S%*s"),
		      current && arrow_cursor ? "->" : "  ",
		      n,
		      c,buffer1,
		      filllen,"");

	free_string(&buffer1);
    } else {
	menu_PutLineX(I->current_page,
		      5 + i % lines_per_page,0,
		      FRM("%.2s%c%c %*s"),
		      current && arrow_cursor ? "->" : "  ",
		      n,
		      c,
		      size,"");
    }

    if (use_comment) {
	draw_comment(I,5 + i % lines_per_page,comment1,l1,COLUMNS);
    }

    if (current && !arrow_cursor)
	menu_EndXX(I->current_page,pg_STANDOUT);
}

static void write_prompt P_((struct enter_info *I));
static void write_prompt(I)
     struct enter_info *I;
{
    int LINES,COLUMNS;
    
    menu_get_sizes(I->current_page,&LINES,&COLUMNS);

    show_last_error();

    menu_PutLineX(I->current_page,LINES-3,0, FRM("%S"),I->pvector[pv_text]);
    if (I->pvector[pv_buffer])
	menu_Write_to_screen(I->current_page,FRM("%S"),I->pvector[pv_buffer]);
    menu_CleartoEOLN(I->current_page);
}

int browser_expand(XXX,name,prev_folder,aview,pos)
     struct folder_browser * XXX;
     struct string **name;
     struct string ** prev_folder;
     struct AliasView *aview;
     int              *pos;
{
    int ret = 0;
    int s_len;

    if (*name && (s_len=string_len(*name)) > 0) {
	switch (give_unicode_from_string(*name,0)) {
	    struct  addr_list * address_l;
		
	case 0x0040: { /* '@' -- @alias ... */
	    int POS = 1;
	    struct string * name_rest = 
		clip_from_string(*name,&POS,s_len);

	    address_l = 
		get_alias_address_expanded(name_rest,
					   NULL,aview,
					   1 /* unquote */,
					   NULL);
						   
	    free_string(&name_rest);
	    

	    if (address_l) {

		int a_len = addr_list_item_count(address_l);
		
		if (1 == a_len) {
		    int group = -1;
		    const struct address * address = 
			addr_list_get_item(address_l,0,&group);
		
		    const char          * addr     = 
			address_get_ascii_addr(address);

		    if (addr) {
			char buffer[LONG_STRING];
			struct string *S1;
		
			/* get filename from address */
			get_return_name(addr, buffer, 
					TRUE, sizeof buffer);
		
			if (buffer[0]) {
			    S1 = format_string(FRM("=%s"),buffer);
			    
			    if (select_dir_item(XXX,&S1,pos)) {
				free_string(&S1);
				free_addr_list(&address_l);
				
				ret = 1;
				DPRINT(Debug,4,
				       (&Debug, 
					"-- browser_expand (@alias) OK\n"));
				goto out;
			    }
			}
			free_string(&S1);
		    }
		}

		free_addr_list(&address_l);
	    }
	    goto out;     /* Alias FAILED */
	}

	    
	case 0x0021: /*  ! */
	    
	    if (2 == s_len &&
		0x0021 /*  ! */ == give_unicode_from_string(*name,1)) {
		if (prev_folder && *prev_folder &&
		    select_dir_item(XXX, prev_folder,pos)) {
		    
		    ret = 1;
		    DPRINT(Debug,4,
			   (&Debug, 			    
			    "-- browser_expand (!! == previous folder) OK\n"));
		    goto out;
		}
		goto out;     /* previous folder */
	    }
	    break;  
	}
	if (select_dir_item(XXX,name,pos))
	    ret = 1;
    }

 out:
	
    DPRINT(Debug,7, (&Debug, 
		     "browser_expand=%d\n",ret));
    
    return ret;    
}

static int cf_english P_((struct enter_info *I));
static int cf_english(I)
     struct enter_info *I;
{
    int s_len;
    int LINES,COLUMNS;

    menu_get_sizes(I->current_page,&LINES,&COLUMNS);

    if (I->pvector[pv_buffer] && 
	0 == string_len(I->pvector[pv_buffer])) {
	free_string(&(I->pvector[pv_buffer]));
    }

    if (!I->pvector[pv_buffer]) {

	menu_PutLineX(I->current_page,LINES-3,0, FRM("%S"),I->pvector[pv_text]);
	menu_Write_to_screen(I->current_page,
			     CATGETS(elm_msg_cat, ElmSet, ElmNoSave, 
				     "<no save>"));
	menu_CleartoEOLN(I->current_page);
	menu_MoveCursor(I->current_page,
			LINES-3,string_len(I->pvector[pv_text]));
	return 1;

    } else if ((s_len=string_len(I->pvector[pv_buffer])) > 0) {

	switch (give_unicode_from_string(I->pvector[pv_buffer],0)) {

	case 0x003D:  /* '='  Handle save copy prompt! */
	    if (1 == s_len) {

		menu_PutLineX(I->current_page,
			      LINES-3,0, FRM("%S"),I->pvector[pv_text]);
		menu_Write_to_screen(I->current_page,
				     CATGETS(elm_msg_cat, ElmSet, 
					     ElmUncondSaveByName, 
					     "<unconditionally save by name>"));
		menu_CleartoEOLN(I->current_page);
		menu_MoveCursor(I->current_page,
				LINES-3,string_len(I->pvector[pv_text]));
		return 1;		

	    } else if (2 == s_len &&
		       give_unicode_from_string(I->pvector[pv_buffer],1) 
		       == 0x003F /* '?' */) {

		menu_PutLineX(I->current_page,
			      LINES-3,0, FRM("%S"),I->pvector[pv_text]);

		menu_Write_to_screen(I->current_page,
				     CATGETS(elm_msg_cat, ElmSet, 
					     ElmCondSaveByName, 
					     "<conditionally save by name>"));
		menu_CleartoEOLN(I->current_page);
		menu_MoveCursor(I->current_page,
				LINES-3,string_len(I->pvector[pv_text]));
		return 1;		

	    }
	    break;
	case 0x003C: /* '<' */
	    if (1 == s_len) {

		menu_PutLineX(I->current_page,
			      LINES-3,0, FRM("%S"),I->pvector[pv_text]);
		menu_Write_to_screen(I->current_page,
				     CATGETS(elm_msg_cat, ElmSet, 
					     ElmSentFolder, "<\"sent\" folder>"));
		menu_CleartoEOLN(I->current_page);
		menu_MoveCursor(I->current_page,
				LINES-3,string_len(I->pvector[pv_text]));
		return -1;  /* Special prompt but normal expansion */	    
	    }
	    break;
	}
    }
    return 0;
}

static void gb_initial_text P_((struct enter_info *I));
static void gb_initial_text(I)
     struct enter_info *I;
{
    int LINES,COLUMNS;
    int l,l1;
    struct string * S = NULL;

    menu_get_sizes(I->current_page,&LINES,&COLUMNS);

    if (0 == (I->GB_FLAGS & GB_MENU) &&
	COLUMNS > 60 &&
	(! I->pvector[pv_buffer] ||
	 0 == string_len(I->pvector[pv_buffer]))) 

	S = format_string(CATGETS(elm_msg_cat, ElmSet, 
				  ElmBrowserPressTabOrHelp,
				  "(Press TABulator for listing or use '?' for help)"));	           
    else       	
	S = format_string(CATGETS(elm_msg_cat, ElmSet, 
				  ElmUseForHelp,
				  "(Use '?' for help)"));	    


    /* FIXME: not complete correct -- this do not take account
              double wide characters
    */
    l1 = string_len(S);
    if (l1 < 15)
	l1 = 15;



    l = COLUMNS - l1-5;

    if (l < 45) {
	

	menu_MoveCursor(I->current_page,
			LINES-4, 0);
	menu_CleartoEOS(I->current_page);

	if (l < -5)
	    l = -5;

    } else {
	menu_MoveCursor(I->current_page,
			LINES-4, l);
	menu_CleartoEOS(I->current_page);
	
    }

    show_last_error();
    	
    menu_PutLineX(I->current_page,
		  LINES-4, l+5, FRM("%S"),
		  S);

    free_string(&S);
}

static void gb_initial_non_append P_((struct enter_info *I));
static void gb_initial_non_append(I)
     struct enter_info *I;
{
    int LINES,COLUMNS;

    menu_get_sizes(I->current_page,&LINES,&COLUMNS);

    menu_PutLineX(I->current_page,
		  LINES-3,0, FRM("%S"),I->pvector[pv_text]);

    if (I->pvector[pv_buffer]) {

	menu_Write_to_screen(I->current_page,
			     FRM("%S"),I->pvector[pv_buffer]);
	menu_CleartoEOLN(I->current_page);
	menu_MoveCursor(I->current_page,
			LINES-3,string_len(I->pvector[pv_text]));

    }
}

static void gb_initial_title P_((struct enter_info *I,struct string * buffer));
static void gb_initial_title(I,buffer)
     struct enter_info *I;
     struct string * buffer;
{

    menu_ClearScreen(I->current_page);

    menu_StartXX(I->current_page,pg_BOLD);
    menu_print_center(I->current_page,1,buffer);
    menu_EndXX(I->current_page,pg_BOLD);

}


static void gb_dir_title P_((struct enter_info *I,int *entries));
static void gb_dir_title(I,entries)
     struct enter_info *I;
     int *entries;
{
    struct string * buffer;

    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"gb_dir_title",
	      "Bad magic number",0);

    buffer = give_title_dir(I->browser->dir_p,entries);

    menu_PutLineX(I->current_page,3,0,FRM("%S"),buffer);  
    menu_CleartoEOLN(I->current_page);

    menu_MoveCursor(I->current_page,4,0);                 
    menu_CleartoEOLN(I->current_page);
	
    free_string(&buffer);    
}

static void sort_message P_((struct string *message));
static void sort_message(message)
     struct string *message;
{
    if (message)
	lib_transient(FRM("%S"),message);
    else
	clear_error();
}

static int browser_autofill P_((struct enter_info *I));
static int browser_autofill(I)
     struct enter_info *I;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_autofill",
	      "Bad magic number",0);

    if (0    == (I->GB_FLAGS & GB_FILL) &&
	0    != (I->GB_FLAGS & GB_MENU) &&
	NULL != I->pvector[pv_buffer]) {

	/* Act as cursor right is pressed? */
	
	struct string  *  old   = NULL;
	
	if (!give_edit_buffer(I->browser->dir_p,I->GB_LINE,
			      &old,0) ||
	    !old) {
	    
	    DPRINT(Debug,8, (&Debug,  
			     "-- browser_autofill: %S -- failed to get currect dir prefix\n",
			     I->pvector[pv_buffer]));
	    
	    goto fail_test;
	}
		
	/* If current editor prefix is not equivalent what fill be
	   filled, then not to automatic fill of filename 
	*/
	
	if (0 != string_cmp(old,I->pvector[pv_buffer],999)) {
	    
	    DPRINT(Debug,8, (&Debug,  
			     "-- browser_autofill: %S does not match prefix %S\n",
			     I->pvector[pv_buffer],old));
	    
	    goto fail_test;
	}

	/* Act as cursor right is pressed */
	if (give_edit_buffer(I->browser->dir_p,I->GB_LINE,
			     &(I->pvector[pv_buffer]),1)) {
	    I->GB_FLAGS |= GB_FILL;
	    
	    lib_transient(CATGETS(elm_msg_cat, ElmSet,
				  ElmBrowserFileNameConfirm,
				  "Press enter again to confirm filename."));
	    
	    free_string(&old);
	    return 1;     /*  reject ENTER */
	}
	
    fail_test:
	
	if (old)
	    free_string(&old);		
    }

    return 0;
}

typedef void gb_draw_line_f P_((struct enter_info *I,
				int i,int lines_per_page)); 

static void gb_draw_line_wrapper P_((struct enter_info *I,
				     gb_draw_line_f    * draw_line_p,
				     int i,int lines_per_page,
				     int entries));
static void gb_draw_line_wrapper(I,draw_line_p,i,lines_per_page,entries)
     struct enter_info *I;
     gb_draw_line_f    * draw_line_p;
     int i;
     int lines_per_page;
     int entries;
{
    if (i < entries)
	draw_line_p(I,i,lines_per_page);
    else {
	menu_MoveCursor(I->current_page,
			5 + i % lines_per_page,0);
	menu_CleartoEOLN(I->current_page);
    }	              	    
}

static int gb_lines_per_page P_((struct enter_info *I));
static int gb_lines_per_page(I)
     struct enter_info *I;
{
    int lines_per_page;
    int LINES,COLUMNS;   

    menu_get_sizes(I->current_page,&LINES,&COLUMNS);
    
    lines_per_page = LINES-10;

    if (lines_per_page < 1) {
	lines_per_page = 1;

	DPRINT(Debug,12, (&Debug,"gb_lines_per_page=%d: LINES=%d COLUMNS=%d\n",
			  lines_per_page,LINES,COLUMNS));
    }

    return lines_per_page;
}

static void gb_draw_menu P_((struct enter_info *I,
			     gb_draw_line_f  * draw_line_p));
static void gb_draw_menu(I, draw_line_p)
     struct enter_info *I;
     gb_draw_line_f  * draw_line_p;
{
    int entries;
    int lines_per_page = gb_lines_per_page(I);
    int i;


    gb_dir_title(I,&entries);


    if (I->GB_PAGE > entries / lines_per_page)
	I->GB_PAGE = entries / lines_per_page;
    if (I->GB_LINE > entries-1 && entries > 0)
	I->GB_LINE = entries-1;

    DPRINT(Debug,12, (&Debug,
		      "gb_draw_menu: %d entries, %d lines per page; line %d, page %d\n",
		      entries,lines_per_page,
		      I->GB_LINE,I->GB_PAGE));
    
    for (i = I->GB_PAGE * lines_per_page;
	 i < I->GB_PAGE * lines_per_page + lines_per_page;
	 i++) {
	gb_draw_line_wrapper(I,draw_line_p,i,lines_per_page,entries);
    }    
}


enum gb_result { gb_none, gb_sort_it, gb_repage };

static enum gb_result gb_tabaction P_((struct enter_info *I,int append_current,
				       gb_draw_line_f    * draw_line_p));
static enum gb_result gb_tabaction(I,append_current, draw_line_p)
     struct enter_info *I;
     int append_current;
     gb_draw_line_f    * draw_line_p;
{
    int match =  -1;         /* First match on autocomplete */

    enum browser_wildcards_v bw = 
	give_dt_enumerate_as_int(&browser_wildcards);
    /*   0 == off
	 1 == on
	 2 == file browser
    */
	

    DPRINT(Debug,8, (&Debug,  
		     "-- gb_tabaction: append_current=%d ch_count=%d\n",
		     append_current,I->ch_count));

    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"gb_tabaction",
	      "Bad magic number",0);

    I->GB_FLAGS &= ~GB_EXTENSION;
    I->GB_FLAGS &= ~GB_FILENAME;

    if (!append_current && I->ch_count == 0) {
	if (I->pvector[pv_buffer])
	    free_string(&(I->pvector[pv_buffer]));
	I->ch_count++;
    }
    
    clear_error();
    
    /* TAB does not give Elm 2.5 style file browser   */
    if (bw /* do wildcard match? */
	&& 
	dir_is_wildcard(I->browser->dir_p,&(I->pvector[pv_buffer]))) {
	DPRINT(Debug,4, (&Debug,  
			 "-- gb_tabaction: wildcarding\n"));
	
	I->GB_FLAGS &= ~GB_HELP;
	I->GB_FLAGS |= GB_MENU | GB_REDRAW | GB_FILL;
	
	I->GB_LINE = 0;
	I->GB_PAGE = 0;
	
	if (!give_edit_buffer(I->browser->dir_p,I->GB_LINE,
			      &(I->pvector[pv_buffer]),
			      I->GB_FLAGS & GB_FILL ? 1 : 0)) {
	    /* Ring a bell */
	    menu_Writechar(I->current_page,'\007');
	}
	
	return gb_sort_it;
    }
    
    if (browser_tabcomplete  /* tabcomplete prompt? */
	&&
	dir_autocomplete(I->browser->dir_p,&(I->pvector[pv_buffer]),
			 &match)) {

	DPRINT(Debug,4, (&Debug,  
			 "-- gb_tabaction: autocomplete\n"));
	
	if (match >= 0 && 
	    0 != (I->GB_FLAGS & GB_MENU) &&
	    I->GB_LINE != match) {
	    int oldline = I->GB_LINE;

	    int lines_per_page = gb_lines_per_page(I);

	    int entries = dir_give_count(I->browser->dir_p);

	    I->GB_LINE = match;

	    if (I->GB_LINE / lines_per_page != I->GB_PAGE) {
		I->GB_PAGE = I->GB_LINE / lines_per_page;
		
		return gb_repage;
	    }
	    
	    gb_draw_line_wrapper(I,draw_line_p,oldline,
				 lines_per_page,entries);
	    gb_draw_line_wrapper(I,draw_line_p,I->GB_LINE,
				 lines_per_page,entries);
	}

	return gb_none;
    }

    if (change_dir(I->browser->dir_p,&(I->pvector[pv_buffer]))) {
	I->GB_FLAGS |= GB_MENU | GB_REDRAW | GB_FILL;
	I->GB_FLAGS &= ~GB_HELP;
	
	I->GB_LINE = -1;
	I->GB_PAGE = 0;
		
	return gb_sort_it;
    } else {
	/* Ring a bell */
	menu_Writechar(I->current_page,'\007');
    }
    
    return gb_repage;
}

static int check_path P_((struct enter_info *I));
static int check_path(I)
     struct enter_info *I;
{
    struct string * base = NULL;
    
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"check_path",
	      "Bad magic number",0);

    /* Check that patch is correct ... */
    
    if (give_edit_buffer(I->browser->dir_p,I->GB_LINE,
			 &base,
			 I->GB_FLAGS & GB_FILL ? 1 : 0)) {
	
	if (base && I->pvector[pv_buffer] && 
	    0 != string_cmp(base,I->pvector[pv_buffer],99)) {
	    
	    DPRINT(Debug,8,(&Debug, "check_path: path not current %S <> %S, GB_FILL %s\n",
			    base,I->pvector[pv_buffer],
			    I->GB_FLAGS & GB_FILL ? "set" : "unset"));

	    free_string(&(I->pvector[pv_buffer]));
	    
	    I->pvector[pv_buffer] = base;
	    return 1;
	}
	
	free_string(&base);	
    }

    return 0;
}


static enum gb_result gb_left P_((struct enter_info *I));
static enum gb_result gb_left(I)
     struct enter_info *I;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"gb_left",
	      "Bad magic number",0);

    I->GB_FLAGS &= ~GB_EXTENSION;
    I->GB_FLAGS &= ~GB_FILENAME;

    clear_error();

    if (0 != (I->GB_FLAGS & GB_MENU)) {

	if (0 == (I->GB_FLAGS & GB_FILL)) {
	    int pos = -1;

	    if (check_path(I)) {

		DPRINT(Debug,8,
		       (&Debug, "gb_left: Changing directory canceled\n"));

		return gb_none;
	    }

	    if (!change_dir_up(I->browser->dir_p,
			       &(I->pvector[pv_buffer]),&pos)) {
		/* Ring a bell */
		menu_Writechar(I->current_page,'\007');
	    } else {
		int lines_per_page = gb_lines_per_page(I);
		
		I->GB_LINE = pos;
		
		if (-1 != I->GB_LINE) 
		    I->GB_PAGE = I->GB_LINE / lines_per_page;
		else
		    I->GB_PAGE = 0;

		DPRINT(Debug,8,(&Debug, 
				"gb_left: GB_LINE => %d, GB_LINE => %d\n",
				I->GB_LINE,I->GB_PAGE));


		I->GB_FLAGS |= GB_FILL;

		return gb_sort_it;
	    }

	    return gb_repage;
	} else {
	    int lines_per_page = gb_lines_per_page(I);

	    I->GB_FLAGS &= ~GB_FILL;
	    
	    if (!give_edit_buffer(I->browser->dir_p,I->GB_LINE,
				  &(I->pvector[pv_buffer]),
				  I->GB_FLAGS & GB_FILL ? 1 : 0)) {
		/* Ring a bell */
		menu_Writechar(I->current_page,'\007');
		return gb_repage;
	    }

	    /* Check if page is changed with Page Up 
	       or Page Down */

	    if (I->GB_LINE / lines_per_page != I->GB_PAGE) {
		I->GB_PAGE = I->GB_LINE / lines_per_page;
	    
		return gb_repage;
	    }
	}
    }
    
    return gb_none;
}

static enum gb_result gb_right P_((struct enter_info *I));
static enum gb_result gb_right(I)
     struct enter_info *I;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"gb_right",
	      "Bad magic number",0);

    I->GB_FLAGS &= ~GB_EXTENSION;
    I->GB_FLAGS &= ~GB_FILENAME;
    
    clear_error();
    
    if (0 != (I->GB_FLAGS & GB_MENU)) {
	int lines_per_page = gb_lines_per_page(I);

	if (0 != (I->GB_FLAGS & GB_FILL)) {

	    if (check_path(I)) {

		DPRINT(Debug,8,
		       (&Debug, "gb_right: Changing directory canceled\n"));

		return gb_none;
	    }
	    
	    if (!change_dir_to_entry(I->browser->dir_p,I->GB_LINE,
				     &(I->pvector[pv_buffer]))) {
		/* Ring a bell */
		menu_Writechar(I->current_page,'\007');
	    } else {
		I->GB_LINE = -1;
		I->GB_PAGE = 0;
		
		I->GB_FLAGS &= ~GB_FILL;
		
		return gb_sort_it;
	    }
	    
	    return gb_repage;
	    
	} else {
	    I->GB_FLAGS |= GB_FILL;
	    
	    if (!give_edit_buffer(I->browser->dir_p,I->GB_LINE,
				  &(I->pvector[pv_buffer]),
				  I->GB_FLAGS & GB_FILL ? 1 : 0)) {
		/* Ring a bell */
		menu_Writechar(I->current_page,'\007');
		return gb_repage;
	    }

	    /* Check if page is changed with Page Up 
	       or Page Down */

	    if (I->GB_LINE / lines_per_page != I->GB_PAGE) {
		I->GB_PAGE = I->GB_LINE / lines_per_page;
	    
		return gb_repage;
	    }
	}
    }
    
    return gb_none;
}

static enum gb_result gb_prev P_((struct enter_info *I,
				  gb_draw_line_f    * draw_line_p));
static enum gb_result gb_prev(I,draw_line_p)
     struct enter_info *I;
     gb_draw_line_f    * draw_line_p;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"gb_prev",
	      "Bad magic number",0);

    I->GB_FLAGS &= ~GB_EXTENSION;
    I->GB_FLAGS &= ~GB_FILENAME;
    
    clear_error();
    if (0 != (I->GB_FLAGS & GB_MENU)) {
	int lines_per_page = gb_lines_per_page(I);
	int oldline = I->GB_LINE;
	
	int entries = dir_give_count(I->browser->dir_p);
	
	I->GB_LINE--;
	if (I->GB_LINE < 0)
	    I->GB_LINE = 0;
	
	if (!give_edit_buffer(I->browser->dir_p,I->GB_LINE,
			      &(I->pvector[pv_buffer]),
			      I->GB_FLAGS & GB_FILL ? 1 : 0)) {
	    /* Ring a bell */
	    menu_Writechar(I->current_page,'\007');
	    
	    return gb_repage;
	}
	
	if (I->GB_LINE / lines_per_page != I->GB_PAGE ||
	    oldline / lines_per_page != I->GB_PAGE) {
	    I->GB_PAGE = I->GB_LINE / lines_per_page;
	    
	    return gb_repage;
	}
	
	gb_draw_line_wrapper(I,draw_line_p,oldline,
			     lines_per_page,entries);
	gb_draw_line_wrapper(I,draw_line_p,I->GB_LINE,
			     lines_per_page,entries);

    } else if (0 != (I->GB_FLAGS & GB_HELP)) {
	browser_change_prev_line(I);
	return gb_repage;
    }
    
    return gb_none;
}


static enum gb_result gb_next P_((struct enter_info *I,
				  gb_draw_line_f    * draw_line_p));
static enum gb_result gb_next(I,draw_line_p)
     struct enter_info *I;
     gb_draw_line_f    * draw_line_p;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"gb_next",
	      "Bad magic number",0);

    I->GB_FLAGS &= ~GB_EXTENSION;
    I->GB_FLAGS &= ~GB_FILENAME;
    
    clear_error();

    if (0 != (I->GB_FLAGS & GB_MENU)) {
	int lines_per_page = gb_lines_per_page(I);

	int oldline = I->GB_LINE;
	
	int entries = dir_give_count(I->browser->dir_p);
	
	if (entries > 0) {
	    I->GB_LINE++;
	    if (I->GB_LINE >= entries)
		I->GB_LINE = entries-1;
	    
	    if (!give_edit_buffer(I->browser->dir_p,I->GB_LINE,
				  &(I->pvector[pv_buffer]),
				  I->GB_FLAGS & GB_FILL ? 1 : 0)) {
		/* Ring a bell */
		menu_Writechar(I->current_page,'\007');

		return gb_repage;
	    }
	}
	
	if (I->GB_LINE / lines_per_page != I->GB_PAGE ||
	    oldline / lines_per_page != I->GB_PAGE) {
	    I->GB_PAGE = I->GB_LINE / lines_per_page;

	    return gb_repage;
	}
		
	gb_draw_line_wrapper(I,draw_line_p,oldline,
			     lines_per_page,entries);
	gb_draw_line_wrapper(I,draw_line_p,I->GB_LINE,
			     lines_per_page,entries);
	
    } else if (0 != (I->GB_FLAGS & GB_HELP)) {
	browser_change_next_line(I);
	return gb_repage;
    }
           
    return gb_none;
}

static void gb_move_to_page P_((struct enter_info *I));
static void gb_move_to_page(I)
     struct enter_info *I;
{
    int lines_per_page = gb_lines_per_page(I);
    int entries = dir_give_count(I->browser->dir_p);

    if (I->GB_LINE / lines_per_page != I->GB_PAGE) {
	
	I->GB_LINE = I->GB_PAGE * lines_per_page;
	
	if (I->GB_LINE < 0)
	    I->GB_LINE = 0;
	else if (I->GB_LINE >= entries)
	    I->GB_LINE = entries-1;
	
	
	if (!give_edit_buffer(I->browser->dir_p,I->GB_LINE,
			      &(I->pvector[pv_buffer]),
			      I->GB_FLAGS & GB_FILL ? 1 : 0)) {
	    /* Ring a bell */
	    menu_Writechar(I->current_page,'\007');
	}
    }
}

static enum gb_result gb_page_up P_((struct enter_info *I,
				  gb_draw_line_f    * draw_line_p));
static enum gb_result gb_page_up(I,draw_line_p)
     struct enter_info *I;
     gb_draw_line_f    * draw_line_p;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"gb_page_up",
	      "Bad magic number",0);

    I->GB_FLAGS &= ~GB_EXTENSION;
    I->GB_FLAGS &= ~GB_FILENAME;

    clear_error();

    if (0 != (I->GB_FLAGS & GB_MENU)) {
		
	if (I->GB_PAGE > 0) {
	    I->GB_PAGE--;

	    if (move_when_paged)
		gb_move_to_page(I);

	    return gb_repage;
	}

    } else if (0 != (I->GB_FLAGS & GB_HELP)) {
	browser_change_page_up(I);
	return gb_repage;
    }
    
    return gb_none;
}

static enum gb_result gb_page_down P_((struct enter_info *I,
				  gb_draw_line_f    * draw_line_p));
static enum gb_result gb_page_down(I,draw_line_p)
     struct enter_info *I;
     gb_draw_line_f    * draw_line_p;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"gb_page_down",
	      "Bad magic number",0);

    I->GB_FLAGS &= ~GB_EXTENSION;
    I->GB_FLAGS &= ~GB_FILENAME;

    clear_error();

    if (0 != (I->GB_FLAGS & GB_MENU)) {
	int lines_per_page = gb_lines_per_page(I);
	
	int entries = dir_give_count(I->browser->dir_p);

	if ((I->GB_PAGE+1) * lines_per_page < entries) {
	    I->GB_PAGE++;

	    if (move_when_paged)
		gb_move_to_page(I);

	    return gb_repage;
	}
	    	   	    
    } else if (0 != (I->GB_FLAGS & GB_HELP)) {
	browser_change_page_down(I);
	return gb_repage;
    }

    return gb_none;
}


static void gb_reload_dir P_((struct enter_info *I));
static void gb_reload_dir(I)
     struct enter_info *I;
{
    int pos = I->GB_LINE;
    
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"gb_reload_dir",
	      "Bad magic number",0);

    lib_transient(CATGETS(elm_msg_cat, ElmSet,
			  ElmBrowserReloadingDirectory,
			  "Reloading directory ..."));

    if (!reload_dir(I->browser->dir_p,&pos)) {
	/* Ring a bell */
	menu_Writechar(I->current_page,
		       '\007');
	
    }
    
    if (-1 != pos &&
	pos != I->GB_LINE ) {
		
	DPRINT(Debug,10,
	       (&Debug, 
		"gb_reload_dir: GB_LINE: %d => %d\n",
		I->GB_LINE,pos));
	
	I->GB_LINE = pos;

    } else if (-1 == I->GB_LINE) {
	
	DPRINT(Debug,10, (&Debug, 
			  "gb_reload_dir: Initial position\n"));
	I->GB_LINE = 0;
    }

    /* It is assumed  that gb_sort_dir is also called ... */
}

static void gb_sort_dir P_((struct enter_info *I, int get_buffer, int pos));
static void gb_sort_dir(I, get_buffer, pos)
     struct enter_info *I;
     int get_buffer;
     int pos;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"gb_sort_dir",
	      "Bad magic number",0);

    if (-1 == pos)
	pos = I->GB_LINE;
        
    folder_sort_dir(I->browser->dir_p, sort_message,&pos);

    if (-1 != pos &&
	pos != I->GB_LINE ) {
	
	int lines_per_page = gb_lines_per_page(I);
	
	DPRINT(Debug,10,
	       (&Debug, 
		"gb_sort_dir: GB_LINE: %d => %d\n",
		I->GB_LINE,pos));

	I->GB_LINE = pos;
	
	if (I->GB_LINE / lines_per_page != I->GB_PAGE) {
	    I->GB_PAGE = I->GB_LINE / lines_per_page;
	    
	    DPRINT(Debug,10,
		   (&Debug, 
		    "gb_sort_dir: page => %d\n",
		    I->GB_PAGE));
		    }		   

    } else if (-1 == I->GB_LINE) {

	DPRINT(Debug,10, (&Debug, 
			  "gb_sort_dir: Initial position\n"));
	I->GB_LINE = 0;
	I->GB_PAGE = 0;
    }
       

    if (get_buffer) {
	/* Refill edit buffer 
	 */
	
	if (!give_edit_buffer(I->browser->dir_p,I->GB_LINE,
			      &(I->pvector[pv_buffer]),
			  I->GB_FLAGS & GB_FILL ? 1 : 0)) {
	    /* Ring a bell */
	    menu_Writechar(I->current_page,'\007');
	}
    }
    
}

static void gb_redraw_dirlisting P_((struct enter_info *I, int *sort_it, 
				     int get_buffer,
				     gb_draw_line_f *draw_line_p,
				     int pos));
static void gb_redraw_dirlisting(I,sort_it,get_buffer, 
				 draw_line_p,pos)
     struct enter_info *I;
     int *sort_it; 
     int get_buffer;
     gb_draw_line_f *draw_line_p;
     int pos;
{
    int LINES,COLUMNS;

    menu_get_sizes(I->current_page,&LINES,&COLUMNS);


    if (0 != (I->GB_FLAGS & GB_MENU)) {
	int entries = 0;
	
	gb_dir_title(I,&entries);

	if (*sort_it) {
	    *sort_it = 0;
	    I->GB_FLAGS &= ~GB_SORT_IT;

	    gb_sort_dir(I,get_buffer,pos);
	}

	gb_draw_menu(I,draw_line_p);	
    }
}

static struct string **gb_browser P_((struct enter_info *I,
				      enum enter_mode em,
				      struct menu_context *base_page));

static struct string **gb_browser(I,em, base_page)
     struct enter_info *I;
     enum enter_mode em;
     struct menu_context *base_page;
{
    int sort_it =        ison(I->GB_FLAGS,GB_SORT_IT);
    int append_current = ison(I->flags,OE_APPEND_CURRENT);
    int LINES,COLUMNS;

    const char * s UNUSED_VAROK = NULL;

    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"gb_browser",
	      "Bad magic number",0);

    DPRINT(Debug,10,(&Debug, "gb_browser: em=%d",
		     em));
    if ((s = enter_mode_debug_name(em))) {
	DPRINT(Debug,10,(&Debug, " %s",s));
    }
    DPRINT(Debug,10,(&Debug, "\n"));

    menu_get_sizes(I->current_page,&LINES,&COLUMNS);

    /* These are output values */
    if (0 != (I->GB_FLAGS & GB_SWITCH_FBROWSER)) {
	    I->GB_FLAGS &=   ~GB_SWITCH_FBROWSER;

	    DPRINT(Debug,5,(&Debug,
			    "gb_browser: clearing stale GB_SWITCH_FBROWSER\n"));
    }
    if (0 != (I->GB_FLAGS & GB_SWITCH_FBROWSER_SELECT)) {
	I->GB_FLAGS &=   ~GB_SWITCH_FBROWSER_SELECT;
	DPRINT(Debug,5,(&Debug,
			"gb_browser: clearing stale GB_SWITCH_FBROWSER_SELECT\n"));
    }

    switch(em) {
    case em_redraw:
	I->GB_FLAGS |= GB_REDRAW ;     
    default:
	break;
    }

    switch(em) {
	int s_len;
    case em_redraw_initial:
	
	/* POP/IMAP client code will prompt messages to same area
	   so we re-print prompt every time ... */

	if (0 != (GB_SAVE_COPY & I->GB_FLAGS)) {
	    menu_MoveCursor(I->current_page,LINES-3,0);
	    menu_CleartoEOS(I->current_page);

	    show_last_error();	    

	} else {
	    gb_initial_text(I);
	}

	if (!append_current) {
	    if (0 == (GB_SAVE_COPY & I->GB_FLAGS) ||
		cf_english(I) == 0) {
		gb_initial_non_append(I);
	    }
	    I->ch_count = 0;

	    DPRINT(Debug,4,
		   (&Debug, 
		    "-- gb_browser(..,%d)=non null  (initial prompt)\n",
		    em));
	    return &(I->pvector[pv_buffer]);
	}
	/* FALLTHRU */

    case em_redraw: 
    repage:

	if (0 != (I->GB_FLAGS & GB_MENU))
	    I->GB_FLAGS |= GB_REDRAW;

	if (0 != (I->GB_FLAGS & GB_HELP))
	    I->GB_FLAGS |= GB_REDRAW;


	if (0 != (I->GB_FLAGS & GB_REDRAW)) {

	    struct string * buffer = 
		format_string(CATGETS(elm_msg_cat, ElmSet,
				      ElmFolderSelection,
				      "Folder selection [ELM %s]"),
			      version_buff);

	    if (I->current_page == base_page)
		I->current_page = new_menu_context();

	    gb_initial_title(I,buffer);
	    free_string(&buffer);
	}

 	if (0 != (I->GB_FLAGS & GB_HELP)) {	        
	    	        
	    if (I->browser->help_text_count < 1) {
	    
		browser_add_folderhelp(I);

		browser_add_help(I);

	    }

	    switch(browser_draw_help(I)) {
	    case br_help_redraw: 
		DPRINT(Debug,4,(&Debug, "-- gb_browser: redraw\n"));
		goto repage;
	    case br_help_none: /* OK */ break;
	    }

	} 

	gb_redraw_dirlisting(I,&sort_it,1,draw_line, -1);

	break;

    case em_enter: 

	clear_error();			   

	if (I->pvector[pv_buffer] && (s_len=string_len(I->pvector[pv_buffer])) > 0) {
	    enum browser_wildcards_v bw = 
		give_dt_enumerate_as_int(&browser_wildcards);
	    /*   0 == off
		 1 == on
		 2 == file browser
	    */

	    int flags;
	    int pos;

	    switch (give_unicode_from_string(I->pvector[pv_buffer],0)) {
	    case 0x003F: /* '?' */  
		if (1 == s_len) {

		    if (0 != (I->GB_FLAGS & GB_HELP))
			browser_change_help_page(I);
		    else {
			I->GB_FLAGS |= GB_HELP;
			I->GB_FLAGS &= ~GB_MENU;
			browser_reset_help_page(I);
		    }
		    goto repage;
		}
		break;
	    }
	    if (0 != (GB_SAVE_COPY & I->GB_FLAGS)) {
		int cf_r;
		if ((cf_r = cf_english(I)) > 0) {
		    DPRINT(Debug,4, (&Debug,  "-- gb_browser(..,%d)=NULL (save folder prompt)\n",
			       em));
		    error_sleep((sleepmsg + 1) / 2);

		    return NULL;
		}
		if (cf_r < 0) {     /* Show text of cf_english() little time */
		    error_sleep((sleepmsg + 1) / 2);
		}
	    }
	    
	    switch(bw) { /* do wildcard match? */
		struct string *filter;
		int changed;
		
	    case browser_wildcard_yes:
	    default:
		
	    no_fbrowser:
		
		if (dir_is_wildcard(I->browser->dir_p,&(I->pvector[pv_buffer]))) {
		    DPRINT(Debug,4, (&Debug,  
				     "-- gb_browser(..,%d): wildcarding\n",
				     em));
		    
		wildcard_oops:
		    I->GB_FLAGS &= ~GB_HELP;
		    I->GB_FLAGS |= GB_MENU | GB_REDRAW | GB_FILL;
		    
		    I->GB_LINE = 0;
		    I->GB_PAGE = 0;
		    
		    if (!give_edit_buffer(I->browser->dir_p,I->GB_LINE,
					  &(I->pvector[pv_buffer]),
					  I->GB_FLAGS & GB_FILL ? 1 : 0)) {
			/* Ring a bell */
			menu_Writechar(I->current_page,
				       '\007');
		    }
		    
		    sort_it++;
		    
		    goto repage;
		}
		break;
		
	    case browser_wildcard_file_browser: /* file browser */
		
		if (0 == (I->GB_FLAGS & GB_FBROWSER_AVAILABLE)) {
		    DPRINT(Debug,4, (&Debug,  
				     "-- gb_browser: fbrowser not available\n"));
		    goto no_fbrowser;
		}
		
		changed = 0;
		filter = dir_give_wildcard(I->browser->dir_p,&(I->pvector[pv_buffer]),
					   &changed);
		
		if (filter) {
		    
		    DPRINT(Debug,4, (&Debug,  
				     "-- gb_browser(..,%d): wildcard=%S\n",
				     em,filter));
		    
		    /* Call fbrowser ? */
		    
		    if (fbrowser_supported_on_dir(I->browser->dir_p)) {
			
			if (I->pvector[pv_fbrowser_filter])
			    free_string(& (I->pvector[pv_fbrowser_filter]));
			
			I->pvector[pv_fbrowser_filter] = filter;
			
			I->GB_FLAGS |= GB_SWITCH_FBROWSER;
			
			DPRINT(Debug,4, (&Debug,  
					 "-- gb_browser(..,%d)=NULL -- GB_SWITCH_FBROWSER\n",
					 em));    
			return NULL;
		    }
		    
		    free_string(&filter);
		}
		
		if (changed) {
		    DPRINT(Debug,4, (&Debug,  
				     "-- gb_browser(..,%d): something changed\n",
				     em));
		    goto wildcard_oops;
		}
		
		break;
	    case browser_wildcard_no:
		DPRINT(Debug,10, (&Debug,  
				  "-- gb_browser(..,%d): no browser wildcards\n",
				  em));
		break;
	    }

	    if (browser_autofill(I)) {
		DPRINT(Debug,4, (&Debug,  
				 "-- gb_browser(..,%d): autofilled\n",
				 em));

		break;   /*  reject ENTER */
	    }

	    pos = I->GB_LINE;

	    if (!browser_expand(I->browser->dir_p,&(I->pvector[pv_buffer]),&(I->pvector[pv_prev_fold]),
				I->browser->aview, &pos)) {
		
 		sort_it = 1;  /* directory listing have reloaded */
 		gb_redraw_dirlisting(I,&sort_it,0,draw_line, pos);
 		
		break;     /* Selection FAILED -- reject ENTER */
	    }

	    sort_it = 1;  /* directory listing have reloaded */
	    gb_redraw_dirlisting(I,&sort_it,0,draw_line, pos);

	    flags = give_dir_flags(I->browser->dir_p);
	    
	    DPRINT(Debug,4, (&Debug, 
			     "*** %S have flags:%s%s%s%s%s%s%s%s%s%s\n",
			     I->pvector[pv_buffer],
			     flags & BROWSER_NODIR    ?   " NODIR":    "",
			     flags & BROWSER_NOFOLDER ?   " NOFOLDER": "",
			     flags & BROWSER_MARKED   ?   " MARKED":   "",
			     flags & BROWSER_HAVELOCKFILE  ?   " HAVELOCKFILE":   "",
			     flags & BROWSER_SESSIONLOCK ? " SESSIONLOCK" : "",
				 
			     flags & BROWSER_MAILFILE ?   " MAILFILE": "",
			     flags & BROWSER_SELECTED ?   " SELECTED": "",
			     flags & BROWSER_EXIST    ?   " EXIST"   : "",
			     flags & BROWSER_DIRPREFIX ?   " DIRPREFIX"   : "",
			     !flags                   ?   " none"    : ""));

	    
	    if (0 != (flags & BROWSER_DIRPREFIX)) {
		
		
		if (0 == (flags & BROWSER_MAILFILE) &&

		    /* Folder == mailbox (normally file)*/
		    0 != (flags & BROWSER_NOFOLDER) &&   
		    0 == (flags & BROWSER_NODIR) &&   
		    0 != (I->GB_FLAGS & GB_FBROWSER_AVAILABLE) && 
		    fbrowser_supported_on_dir(I->browser->dir_p)) {

		    if (I->pvector[pv_fbrowser_filter])
			free_string(& (I->pvector[pv_fbrowser_filter]));

		    I->GB_FLAGS |= GB_SWITCH_FBROWSER_SELECT;

		    DPRINT(Debug,4, (&Debug,  
				     "-- gb_browser(..,%d)=NULL -- GB_SWITCH_FBROWSER_SELECT\n",
				     em));    
		    return NULL;


		}


		if ((I->GB_FLAGS & GB_FILENAME) == 0) {
		    I->GB_FLAGS |= GB_FILENAME;
		    
		    lib_transient(CATGETS(elm_msg_cat, ElmSet,
					  ElmBrowserFolderName,
					  "Enter folder name or press enter again to confirm directory prefix."));
		    
		    break;     /*  reject ENTER */
		}			    

	    }

	    if (0 != (flags & BROWSER_EXIST) &&
		0 != (I->GB_FLAGS & GB_MBOX_CHECK) &&
		fbrowser_supported_on_dir(I->browser->dir_p)) {

		DPRINT(Debug,10, (&Debug,  
				  "-- gb_browser(..,%d): Checking MBOX format\n",
				  em));

		if ( ! fbrowser_selection_is_mbox(I->browser->dir_p, 1)) {

		    DPRINT(Debug,4, (&Debug,  
				      "-- gb_browser(..,%d): Not a mailbox, rejecting ENTER\n",
				     em));
		    break;     /*  reject ENTER */
		}
		

	    }
				

	}
	DPRINT(Debug,4, (&Debug,  
			 "-- gb_browser(..,%d)=NULL\n",
			 em));    
	return NULL;

    case em_prev:	
	switch (gb_prev(I,draw_line)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;
	}
	break;

    case em_next:
	switch (gb_next(I,draw_line)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;

	}	
	break;

    case em_page_up:
	switch (gb_page_up(I,draw_line)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;

	}
	break;

    case em_page_down:
	switch (gb_page_down(I,draw_line)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;

	}
	break;
	
    case em_tabaction:

	DPRINT(Debug,4, (&Debug,  
			 "-- gb_browser(..,%d): tabaction\n",
			 em));

	switch (gb_tabaction(I,append_current,draw_line)) {
	case gb_sort_it:
	    sort_it++;

	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;
	}

	break;

    case em_left:

	switch (gb_left(I)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;
	}
	
	break;

    case em_right:

	switch (gb_right(I)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;
	}

	break;

    case em_F4_key:

	DPRINT(Debug,4, (&Debug,  
			 "-- gb_browser(..,%d): F4\n",
			 em));


	if (0 != (I->GB_FLAGS & GB_FBROWSER_AVAILABLE) && 
	    fbrowser_supported_on_dir(I->browser->dir_p)) {

	    const struct string * X = dir_give_filter(I->browser->dir_p);

	    if (I->pvector[pv_fbrowser_filter])
		free_string(& (I->pvector[pv_fbrowser_filter]));

	    if (X)
		I->pvector[pv_fbrowser_filter] = dup_string(X);
	    
	    I->GB_FLAGS |= GB_SWITCH_FBROWSER;
	    
	    DPRINT(Debug,4, (&Debug,  
			     "-- gb_browser(..,%d)=NULL -- GB_SWITCH_FBROWSER\n",
			     em));    
	    return NULL;	   
	}
	
	break;

    case em_F5_key: 

	DPRINT(Debug,4, (&Debug,  
			 "-- gb_browser(..,%d): F5\n",
			 em));

	gb_reload_dir(I);

	sort_it++;

	goto repage;


	default:
	    break;
    }


    write_prompt(I);	 
    
    DPRINT(Debug,4, (&Debug,  
		     "-- gb_browser(..,%d)=non null\n",
		     em));
    return &(I->pvector[pv_buffer]);
}

static enum browser_status run_browser P_((struct menu_context *page,
					   struct folder_browser *p,
					   struct string **buffer,
					   struct fbrowser_call *fc,
					   int save_copy,
					   int fb_rerun,
					   struct string *sepecify, 
					   struct string * prev_fold,
					   struct AliasView *aview,
					   const char *format, const char *msg,
					   va_list args));
static enum browser_status run_browser(page,
				       p,buffer,fc,save_copy,fb_rerun,
				       specify,prev_fold,
				       aview,
				       format,msg,args)
     struct menu_context *page;
     struct folder_browser *p;
     struct string **buffer;
     struct fbrowser_call *fc;
     int save_copy;
     int  fb_rerun;
     struct string *specify;
     struct string * prev_fold;
     struct AliasView *aview;
     const char *format; 
     const char *msg;
     va_list args;
{
    enum browser_status ret = browser_error;
    int code;

    struct string *text;
    struct string *vector[pv_COUNT];

    struct  browser_edit BE;
    struct enter_info INFO;
    int LINES,COLUMNS;
    int delay_redraw = 0;

    
    DPRINT(Debug,4,
	   (&Debug, "run_browser: Entering [%s]\n",format));

    zero_enter_info(&INFO);
    
    text = elm_smessage(0,format,msg,args);

    vector[pv_buffer]    = *buffer;
    vector[pv_text]      = text;
    vector[pv_specify_text] = specify;
    vector[pv_prev_fold] = prev_fold;   
    vector[pv_defext]  = NULL;        /* default_extension */
    vector[pv_defname] = NULL;

    vector[pv_fbrowser_filter] = NULL;

    INFO.browser     = &BE;
    INFO.pvector     = vector;

    init_browser_edit(&BE, p, aview);

    INFO.give_buffer  = gb_browser;
    INFO.alter_buffer = default_alter_buffer;
    INFO.full_page    = default_full_page;
    INFO.prompt_hint  = default_prompt_hint;

    if (save_copy)
	INFO.flags       = OE_REDRAW_MARK | OE_TABACTION |
	    OE_SIG_CHAR /* Ctrl-C */;
    else
	INFO.flags       = (*buffer ? 0 : OE_APPEND_CURRENT ) |
	    OE_REDRAW_MARK | OE_TABACTION | OE_SIG_CHAR /* Ctrl-C */;
    INFO.ch_count    = 0;
    INFO.builtin     = NULL;
    INFO.in_utils    = NULL;
    INFO.address     = NULL;

    INFO.current_page = page;
						
    if (save_copy)
	INFO.GB_FLAGS |= GB_SAVE_COPY;
    

    if (fc) {
	if (FBROWSER_CALL_magic != fc->magic)
	    panic("BROWSER PANIC",__FILE__,__LINE__,"run_browser",
		  "Bad magic number",0);
	           
	INFO.GB_FLAGS |= GB_FBROWSER_AVAILABLE;

	if (fb_rerun) {
	    INFO.flags   |= OE_APPEND_CURRENT;   
	    BE.gb_flags  |= GB_MENU|GB_REDRAW|GB_SORT_IT;  
	    BE.gb_line    = -1;
	}

	if (get_browser_sel_type(p) == selection_folder &&
	    0 == (fc -> options & FB_NOMBOX_CHECK)) {
	    
	    BE.gb_flags  |= GB_MBOX_CHECK;

	}
	    	
    } else if (fb_rerun)
	panic("BROWSER PANIC",__FILE__,__LINE__,"run_browser",
	      "fb_rerun set without fc",0);

    clear_selection_dir(p);
  
    do {
	code = enter_helper(&INFO,page,page);

	if (code == REDRAW_MARK) {
	    menu_ClearScreen(page);   
	    
	    menu_redraw_children(page);
	    
	    delay_redraw++;
	}

    } while (code == REDRAW_MARK);
    
    menu_get_sizes(page,&LINES,&COLUMNS);

    menu_MoveCursor(page,LINES-4,0);
    menu_CleartoEOS(page);
    show_last_error();

    if (0 != code) {
	DPRINT(Debug,4, (&Debug, "run_browser: Ctrl-C or error\n"));
	ret = browser_error;
    } else {
	int flags = give_dir_flags(p);

	if (0 != (BE.gb_flags & GB_SWITCH_FBROWSER)) {
	    
	    if (!fc)
		panic("BROWSER PANIC",__FILE__,__LINE__,"run_browser",
		      "fc (fbrowser_call) not set",0);	    

	    if (fc->filter)
		free_string(& (fc->filter));

	    
	    fc->filter =  vector[pv_fbrowser_filter];
	    vector[pv_fbrowser_filter] = NULL;
		
	    fc->flags = 0;

	    BE.gb_flags &= ~GB_SWITCH_FBROWSER;
         
	    ret = browser_fbrowser;

	} else if (0 != (BE.gb_flags & GB_SWITCH_FBROWSER_SELECT)) {

	    if (!fc)
		panic("BROWSER PANIC",__FILE__,__LINE__,"run_browser",
		      "fc (fbrowser_call) not set",0);

	    if (0 == (BROWSER_SELECTED & flags))
		panic("BROWSER PANIC",__FILE__,__LINE__,"run_browser",
		      "not selected",0);

	    if (fc->filter)
		free_string(& (fc->filter));

	    fc->flags = FBROWSER_USE_SELECTION;

	    BE.gb_flags &=  ~GB_SWITCH_FBROWSER_SELECT;

	    ret = browser_fbrowser;

	} else if (0 != (BROWSER_SELECTED & flags)) {

	   
	    ret = browser_select;

	}
    }

    *buffer           = vector[pv_buffer];

    free_string(&text);

    if (vector[pv_fbrowser_filter]) 
	free_string(& vector[pv_fbrowser_filter]);


    clear_browser_edit(&BE);

    if (delay_redraw || (INFO.GB_FLAGS & GB_REDRAW)) {
	DPRINT(Debug,10, (&Debug,  "run_browser: redraw set\n"));
	menu_trigger_redraw(page);	
    }

    DPRINT(Debug,10, (&Debug,  
		      "run_browser=%d\n",ret));

    return ret;
}


struct folder_info * folder_browser(
#if ANSI_C
				    struct menu_context *page,
				    struct folder_browser *p,
				    struct string **buffer,
				    struct fbrowser_call *fc,
				    struct AliasView *aview,
				    struct MailboxView *mailbox,
				    const char *format, const char *msg, ...
#else
				    page,
				    p,buffer,fc,aview,mailbox,format, msg, va_alist
#endif
				    )
#if !ANSI_C
     struct menu_context *page;
     struct folder_browser *p;
     struct string **buffer;
     struct fbrowser_call *fc;
     struct AliasView *aview;
     struct MailboxView *mailbox;
     const char *format; 
     const char *msg;
     va_dcl
#endif
{
    enum browser_status r  = browser_error;
    enum browser_status BS = browser_error;

    struct folder_info *ret = NULL;
    struct string * specify = 
	format_string(CATGETS(elm_msg_cat, ElmSet, ElmFolderChangeHelp,
			      "You must specify a folder to change to."));

    va_list vl;

    if (1) {
	Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
	r = run_browser(page,p,buffer,fc,0,0,specify,NULL,aview,format,msg,vl);
	va_end(vl);
    } else {
 RERUN:
	BS = browser_error;

	Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
	r = run_browser(page,p,buffer,fc,0,1,specify,NULL,aview,format,msg,vl);
	va_end(vl);
    }

    switch(r) {
    case browser_select:
	BS = browser_select;
	break;
	
    default:
	panic("BROWSER PANIC",__FILE__,__LINE__,"folder_browser",
		  "Bad return value of run_browser",0);
    case browser_error:
	ret = NULL;
	break;

    case browser_fbrowser:
    FBROWSER:
	if (!fc) 
	    panic("BROWSER PANIC",__FILE__,__LINE__,"folder_browser",
		  "fbrowser_call is not set",0);

	BS = fbrowser_hook_enter(page,p,mailbox,buffer,fc);

	break;
    }
	
    
    switch(BS) {
	int flags;

    case browser_select:
	DPRINT(Debug,5,
	       (&Debug,"folder_browser: browser select\n"));

	flags = give_dir_flags(p);

	if (0 == (BROWSER_SELECTED & flags))
	   panic("BROWSER PANIC",__FILE__,__LINE__,"folder_browser",
		 "Not selected",0);

	ret = folder_from_dir_item(p,TreatAsSpooled);
	break;
	
    case browser_error:
	DPRINT(Debug,5,
	       (&Debug,"folder_browser: browser error\n"));
	break;
	
    case browser_fbrowser:
	DPRINT(Debug,5,
	       (&Debug,"folder_browser: browser fbrowser\n"));
	goto FBROWSER;


    case browser_rerun:
	DPRINT(Debug,5,
	       (&Debug,"folder_browser: browser rerun\n"));
	clear_selection_dir(p);

	goto RERUN;

    default:
	DPRINT(Debug,1,
	       (&Debug,"folder_browser: browser %d\n",
		BS));
	panic("BROWSER PANIC",__FILE__,__LINE__,"folder_browser",
	      "Bad BS",0);
    }

    free_string(&specify);

    if (ret) {
	DPRINT(Debug,5,
	       (&Debug,"folder_browser=%p\n",ret));
    } else {
	DPRINT(Debug,5,
	       (&Debug,"folder_browser=NULL\n"));
    }
    
    return ret;
}

static void draw_fileline P_((struct enter_info *I,
			      int i,int lines_per_page));
static void draw_fileline(I,i,lines_per_page)
     struct enter_info *I;
     int i;
     int lines_per_page;
{
    int flags = 0;
    int LINES,COLUMNS;

    /* return reference to array -- do not free_string !!! */
    const struct string * name_line;
    const struct string * comment1 = NULL;
    int current = i == I->GB_LINE; 
    int c = '?';
    char n = ' ';
    int size;

    int X = 0;
    struct string * buffer1 = NULL;
    int l, l1=0;
    int visible;
    int use_comment = 0;

    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"draw_fileline",
	      "Bad magic number",0);

    name_line = 
	give_line_dir(I->browser->dir_p,i,&flags,&comment1);
    l = string_len(name_line);

     if (comment1)
	 l1 = string_len(comment1);

    menu_get_sizes(I->current_page,&LINES,&COLUMNS);

    adjust_comment_col(I->browser,l,l1,COLUMNS);

    if (0 == (flags & BROWSER_NODIR))
	c = 'D';
    else if (fbrowser_supported_on_dir(I->browser->dir_p)) {
	const struct fbentry * entry = 
	    fbrowser_give_entry(I->browser->dir_p,i);

	if (entry &&
#ifdef S_ISREG
	    S_ISREG(entry->mode)
#else
	    S_IFREG == (entry->mode & S_IFMT)
#endif
	    ) c = ' ';              /* Show regular files with space */
    }

		
    if (flags & BROWSER_HAVELOCKFILE)
	n = 'L';
    else if (flags & BROWSER_SESSIONLOCK)
	n = 'S';
    else if (flags & BROWSER_MARKED)
	n = 'N';

    if (current && !arrow_cursor)
	menu_StartXX(I->current_page,pg_STANDOUT);

    
 restart:
    if (I->browser->comment_col < COLUMNS && comment1) {
	size = I->browser->comment_col-5;
	use_comment = 1;
    } else {
	size = COLUMNS-5;
	use_comment = 0;
    }

    X = 0;
    buffer1 = curses_printable_clip(name_line,&X,l,&visible,size);
   
    if (buffer1) {
	int filllen = size - visible;

	if (filllen < 0)
	    filllen = 0;

	if (X < l && I->browser->comment_col < COLUMNS && use_comment) {
	    int old UNUSED_VAROK = I->browser->comment_col;

	    if (I->browser->comment_col < l+6)
		I->browser->comment_col = l+6;
	    else
		I->browser->comment_col += 5;
	    
	    DPRINT(Debug,10,
		   (&Debug, "draw_fileline: comment column %d => %d\n",
		    old,I->browser->comment_col));
	    
	    free_string(&buffer1);
	    goto restart;
	}


	menu_PutLineX(I->current_page,
		      5 + i % lines_per_page,0,
		      FRM("%.2s%c%c %S%*s"),
		      current && arrow_cursor ? "->" : "  ",
		      n,
		      c,buffer1,
		      filllen,"");

	free_string(&buffer1);
    } else {
	menu_PutLineX(I->current_page,
		      5 + i % lines_per_page,0,
		      FRM("%.2s%c%c %*s"),
		      current && arrow_cursor ? "->" : "  ",
		      n,
		      c,
		      size,"");
    }

    if (use_comment) {
	draw_comment(I,5 + i % lines_per_page,comment1,l1,COLUMNS);
    }

    if (current && !arrow_cursor)
	menu_EndXX(I->current_page,pg_STANDOUT);    

}

int browser_check_extension(filename,defext)
     const struct string  * filename;
     const struct string  * defext;
{

    int L  = string_len(defext);     /* default extension */
    int L1 = string_len(filename);    /* filename */
    int ok = 1;

    if (L >= L1)
	ok = 0;
    else {
	int pos = L1-L;
	
	
	if (give_unicode_from_string(filename,pos-1)
	    != 0x002E /* . */)
	    ok = 0;
	else {
	    int i;
	    
	    for (i = 0; i < L; i++) {
		if (give_unicode_from_string(filename,pos+i) 
		    !=
		    give_unicode_from_string(defext,i)) {
		    ok = 0;
		    break;
		}
	    }
	}
    }

    DPRINT(Debug,10, (&Debug, 
		      "browser_check_extension=%d: filename=%S defext=%S\n",
		      ok,filename,defext));

    return ok;
}




static struct string **gb_filebrowser P_((struct enter_info *I,
					  enum enter_mode em,
					  struct menu_context *base_page));
static struct string **gb_filebrowser(I,em,base_page)
     struct enter_info *I;
     enum enter_mode em;
     struct menu_context *base_page;
{
    int sort_it =  0 != (I->GB_FLAGS & GB_SORT_IT);
    int append_current = 0 != (I->flags & OE_APPEND_CURRENT);
    int LINES,COLUMNS;

    const char * s UNUSED_VAROK = NULL;
    
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"gb_filebrowser",
	      "Bad magic number",0);

    DPRINT(Debug,10,(&Debug, "gb_filebrowser: em=%d",
		     em));
    if ((s = enter_mode_debug_name(em))) {
	DPRINT(Debug,10,(&Debug, " %s",s));
    }
    DPRINT(Debug,10,(&Debug, "\n"));

    
    /* These are output values */
    if (0 != (I->GB_FLAGS & GB_SWITCH_FBROWSER)) {
	    I->GB_FLAGS &=   ~GB_SWITCH_FBROWSER;

	    DPRINT(Debug,5,(&Debug,
			    "gb_filebrowser: clearing stale GB_SWITCH_FBROWSER\n"));
    }
    if (0 != (I->GB_FLAGS & GB_SWITCH_FBROWSER_SELECT)) {
	I->GB_FLAGS &=   ~GB_SWITCH_FBROWSER_SELECT;
	DPRINT(Debug,5,(&Debug,
			"gb_filebrowser: clearing stale GB_SWITCH_FBROWSER_SELECT\n"));
    }

    menu_get_sizes(I->current_page,&LINES,&COLUMNS);

    /* Clear screen ... */
    if (em_redraw == em)           
	I->GB_FLAGS |= GB_REDRAW ;     
    
    switch(em) {
	int s_len;
    case em_redraw_initial:
	
	/* POP/IMAP client code will prompt messages to same area
	   so we re-print prompt every time ... */

	gb_initial_text(I);

	if (!append_current) {
	    gb_initial_non_append(I);
	    
	    I->ch_count = 0;
	    
	    DPRINT(Debug,4,
		   (&Debug, 
		    "-- gb_filebrowser(..,%d)=non null  (initial prompt)\n",
		    em));
	    return &(I->pvector[pv_buffer]);
	}
	/* FALLTHRU */

    case em_redraw: repage:

	if (0 != (I->GB_FLAGS & GB_MENU))
	    I->GB_FLAGS |= GB_REDRAW;
	
	if (0 != (I->GB_FLAGS & GB_HELP))
	    I->GB_FLAGS |= GB_REDRAW;

	if (0 != (I->GB_FLAGS & GB_REDRAW)) {
	    struct string * buffer = 
		format_string(CATGETS(elm_msg_cat, ElmSet,
				      ElmFileSelection,
				      "File selection [ELM %s]"),
			      version_buff);

	    if (I->current_page == base_page)
		I->current_page = new_menu_context();

	    gb_initial_title(I,buffer);
	    free_string(&buffer);
	}

	if (0 != (I->GB_FLAGS & GB_HELP)) {	        

	    if (I->browser->help_text_count < 1) {

		browser_add_filehelp(I);

		browser_add_help(I);

		browser_add_filehelp2(I);

	    } 

	    switch(browser_draw_help(I)) {
	    case br_help_redraw: 
		 DPRINT(Debug,4,(&Debug, "-- gb_filebrowser: redraw\n"));
		 goto repage;
	    case br_help_none: /* OK */ break;
	    }
	}

	gb_redraw_dirlisting(I,&sort_it,1,draw_fileline, -1);

	break;
    case em_enter: 

	clear_error();			   

	if (I->pvector[pv_buffer] && (s_len=string_len(I->pvector[pv_buffer])) > 0) {
	    int bw = give_dt_enumerate_as_int(&browser_wildcards);
	    /*   0 == off
		 1 == on
		 2 == file browser
	    */

	    int flags;
	    int pos;

	    switch (give_unicode_from_string(I->pvector[pv_buffer],0)) {
	    case 0x003F: /* '?' */  
		if (1 == s_len) {

		    if (0 != (I->GB_FLAGS & GB_HELP))
			browser_change_help_page(I);
		    else {
			I->GB_FLAGS |= GB_HELP;
			I->GB_FLAGS &= ~GB_MENU;
			browser_reset_help_page(I);
		    }

		    goto repage;
		}
		break;
	    }

	    if (bw) { /* do wildcard match? */
		
		switch(bw) {
		    struct string *filter;
		    int changed;
		    
		case browser_wildcard_yes:
		default:
		no_fbrowser:
		    
		    if (dir_is_wildcard(I->browser->dir_p,&(I->pvector[pv_buffer]))) {
			
			DPRINT(Debug,4, (&Debug,  
					 "-- gb_filebrowser(..,%d): wildcarding\n",
					 em));
			
		    wildcard_oops:
			I->GB_FLAGS &= ~GB_HELP;
			I->GB_FLAGS |= GB_MENU | GB_REDRAW | GB_FILL;
			
			I->GB_LINE = 0;
			I->GB_PAGE = 0;
			
			if (!give_edit_buffer(I->browser->dir_p,I->GB_LINE,
					      &(I->pvector[pv_buffer]),
					      I->GB_FLAGS & GB_FILL ? 1 : 0)) {
			    /* Ring a bell */
			    menu_Writechar(I->current_page,'\007');
			}
			
			sort_it++;
			
			goto repage;
		    }
		    break;
		    
		    
		case browser_wildcard_file_browser: /* file browser */
		    
		    if (0 == (I->GB_FLAGS & GB_FBROWSER_AVAILABLE)) {
			DPRINT(Debug,4, (&Debug,  
					 "-- gb_filebrowser: fbrowser not available\n"));
			goto no_fbrowser;
		    }


		    changed = 0;
		    filter = dir_give_wildcard(I->browser->dir_p,&(I->pvector[pv_buffer]),
					       &changed);
		    
		    if (filter) {
			
			DPRINT(Debug,4, (&Debug,  
					 "-- gb_filebrowser(..,%d): wildcard=%S\n",
					 em,filter));
			
			/* Call fbrowser ? */
						
			if (fbrowser_supported_on_dir(I->browser->dir_p)) {
			    
			    if (I->pvector[pv_fbrowser_filter])
				free_string(& (I->pvector[pv_fbrowser_filter]));

			    I->pvector[pv_fbrowser_filter] = filter;			  
			    
			    I->GB_FLAGS |= GB_SWITCH_FBROWSER;

			    DPRINT(Debug,4, (&Debug,  
					     "-- gb_filebrowser(..,%d)=NULL -- GB_SWITCH_FBROWSER\n",
					     em));    
			    return NULL;
			}
			

			free_string(&filter);
		    }
		    
		    if (changed) {
			DPRINT(Debug,4, (&Debug,  
					 "-- gb_filebrowser(..,%d): something changed\n",
					 em));
			goto wildcard_oops;
		    }
		    
		    break;
		}
		
	    }
	    
	    if (browser_autofill(I)) {
		DPRINT(Debug,4, (&Debug,  
				 "-- gb_filebrowser(..,%d): autofilled\n",
				 em));
		break;
	    }

	    pos = I->GB_LINE;

	    if (!select_dir_item(I->browser->dir_p,&(I->pvector[pv_buffer]),
				 &pos)) {

 		sort_it = 1;        /* Directory listing is reloaded */
 		gb_redraw_dirlisting(I,&sort_it,0,draw_fileline,pos);
 	       
		break;     /* Selection FAILED -- reject ENTER */
	    }

	    sort_it = 1;        /* Directory listing is reloaded */
	    gb_redraw_dirlisting(I,&sort_it,0,draw_fileline,pos);

	    flags = give_dir_flags(I->browser->dir_p);

	    DPRINT(Debug,4, (&Debug, 
			     "*** %S have flags:%s%s%s%s%s%s%s%s%s%s\n",
			     I->pvector[pv_buffer],
			     flags & BROWSER_NODIR    ?   " NODIR":    "",
			     flags & BROWSER_NOFOLDER ?   " NOFOLDER": "",
			     flags & BROWSER_MARKED   ?   " MARKED":   "",
			     flags & BROWSER_HAVELOCKFILE  ?   " HAVELOCKFILE":   "",
			     flags & BROWSER_SESSIONLOCK ? " SESSIONLOCK": "",

			     flags & BROWSER_MAILFILE ?   " MAILFILE": "",
			     flags & BROWSER_SELECTED ?   " SELECTED": "",
			     flags & BROWSER_EXIST    ?   " EXIST"   : "",
			     flags & BROWSER_DIRPREFIX ?   " DIRPREFIX"   : "",
			     !flags                   ?   " none"    : ""));

	    
	    if (flags & BROWSER_DIRPREFIX) {

		if (0 != (flags & BROWSER_NOFOLDER) &&      /* Folder == mailbox (normally file)*/
		    0 == (flags & BROWSER_NODIR) &&   
		    0 != (I->GB_FLAGS & GB_FBROWSER_AVAILABLE) && 
		    fbrowser_supported_on_dir(I->browser->dir_p)) {

		    if (I->pvector[pv_fbrowser_filter])
			free_string(& (I->pvector[pv_fbrowser_filter]));

		    I->GB_FLAGS |= GB_SWITCH_FBROWSER_SELECT;

		    DPRINT(Debug,4, (&Debug,  
				     "-- gb_filebrowser(..,%d)=NULL -- GB_SWITCH_FBROWSER_SELECT\n",
				     em));    
		    return NULL;

		}

		if ((I->GB_FLAGS & GB_FILENAME) == 0) {
		    I->GB_FLAGS |= GB_FILENAME;
		    
		    lib_transient(CATGETS(elm_msg_cat, ElmSet,
					  ElmBrowserFilename,
					  "Enter filename or press enter again to confirm directory prefix."));

		    break;     /*  reject ENTER */
		}
	    }


	    if ((I->GB_FLAGS & GB_EXTENSION) == 0 &&
		I->pvector[pv_defext] && I->pvector[pv_buffer]) {

		int ok = browser_check_extension(I->pvector[pv_buffer],
						 I->pvector[pv_defext]);
						
		    
		if (ok) {
		    DPRINT(Debug,4, (&Debug,  
				     "-- gb_filebrowser(..,%d): filename %S includes extension %S\n",
				     em,I->pvector[pv_buffer],I->pvector[pv_defext]));
		} else {
		    struct string *X;
		    
		    DPRINT(Debug,4, (&Debug,  
				     "-- gb_filebrowser(..,%d): filename %S do not include extension %S\n",
				     em,I->pvector[pv_buffer],I->pvector[pv_defext]));
		    
		    
		    I->GB_FLAGS |= GB_EXTENSION;
		    
		    X = format_string(FRM("%S.%S"),
				      I->pvector[pv_buffer],
					  I->pvector[pv_defext]);
		    
		    free_string(& (I->pvector[pv_buffer]));
		    I->pvector[pv_buffer] = X;
		    
		    lib_transient(CATGETS(elm_msg_cat, ElmSet,
					  ElmBrowserExtension,
					  "Press enter again to confirm appended extension: %S"),
				  I->pvector[pv_defext]);
		    
		    break;     /*  reject ENTER */
		}		   		
	    }
	

	    if (0 != (flags & BROWSER_EXIST) &&
		0 != (I->GB_FLAGS & GB_MBOX_CHECK) &&
		fbrowser_supported_on_dir(I->browser->dir_p)) {

		DPRINT(Debug,10, (&Debug,  
				  "-- gb_filebrowser(..,%d): Checking MBOX format\n",
				  em));

		if ( ! fbrowser_selection_is_mbox(I->browser->dir_p, 1)) {

		    DPRINT(Debug,4, (&Debug,  
				      "-- gb_filebrowser(..,%d): Not a mailbox, rejecting ENTER\n",
				     em));
		    break;     /*  reject ENTER */
		}	       
	    }
	}
	DPRINT(Debug,4, (&Debug,  
			 "-- gb_filebrowser(..,%d)=NULL\n",
			 em));    
	return NULL;

    case em_prev:
	switch (gb_prev(I,draw_fileline)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;
	}
	break;

    case em_next:
	switch (gb_next(I,draw_fileline)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;
	}
	break;

    case em_page_up:
	switch (gb_page_up(I,draw_line)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;

	}
	break;

    case em_page_down:
	switch (gb_page_down(I,draw_line)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;

	}
	break;
	
    case em_tabaction:
	switch (gb_tabaction(I,append_current,
			     draw_fileline)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;
	}
	break;
      
    case em_left:
	switch (gb_left(I)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;
	}

	break;
    case em_right:

	switch (gb_right(I)) {
	case gb_sort_it:
	    sort_it++;
	    
	case gb_repage:
	    goto repage;

	case gb_none:
	    /* Not used */
	    break;
	}
	
	break;

    case em_F4_key:

	DPRINT(Debug,4, (&Debug,  
			 "-- gb_filebrowser(..,%d): F4\n",
			 em));


	if (0 != (I->GB_FLAGS & GB_FBROWSER_AVAILABLE) && 
	    fbrowser_supported_on_dir(I->browser->dir_p)) {

	    const struct string * X = dir_give_filter(I->browser->dir_p);

	    if (I->pvector[pv_fbrowser_filter])
		free_string(& (I->pvector[pv_fbrowser_filter]));

	    if (X)
		I->pvector[pv_fbrowser_filter] = dup_string(X);
	    
	    I->GB_FLAGS |= GB_SWITCH_FBROWSER;
	    
	    DPRINT(Debug,4, (&Debug,  
			     "-- gb_filebrowser(..,%d)=NULL -- GB_SWITCH_FBROWSER\n",
			     em));    
	    return NULL;	   
	}
	
	break;

    case em_F5_key: 

	DPRINT(Debug,4, (&Debug,  
			 "-- gb_filebrowser(..,%d): F5\n",
			 em));

	gb_reload_dir(I);

	sort_it++;

	goto repage;

    case em_F6_key: {
	struct string * B = NULL;

	DPRINT(Debug,4, (&Debug,  
			 "-- gb_filebrowser(..,%d): F6\n",
			 em));

	/* if filename is NULL, dir_cat_filename returns just
	   current directory

	   Returns NULL if error (current directory not supported)
	*/

	B = dir_cat_filename(I->browser->dir_p, I->pvector[pv_defname]);

	if (B) {
	    if (I->pvector[pv_buffer])
		free_string(& (I->pvector[pv_buffer]));
	    
	    
	    I->pvector[pv_buffer] = B;
	}
	
    }
	break;

    default:
	break;

    case em_wrap:
    case em_editor_escape:
    case em_bs_prev:
	break;
    }

    write_prompt(I);	 
    
    DPRINT(Debug,4, (&Debug,  
		     "-- gb_filebrowser(..,%d)=non null\n",
		     em));
    return &(I->pvector[pv_buffer]);
}

int gen_browser(
#if ANSI_C
		 struct menu_context *page,
		 struct folder_browser *p,
		 struct string **buffer,
		 struct fbrowser_call *fc,
		 enum word_sel w,
		 struct string * prev_fold,
		 struct AliasView *aview,
		 struct MailboxView *mailbox,
		 const char *format, const char *msg, ...
#else
		 page,p,buffer,fc,redraw,w,prev_fold,
		 aview,mailbox,format, msg, va_alist
#endif
)
#if !ANSI_C
     struct menu_context *page;
     struct folder_browser *p;
     struct string **buffer;
     struct fbrowser_call *fc;
     enum word_sel w;
     struct string * prev_fold;
     struct AliasView *aview;
     struct MailboxView *mailbox;
     const char *format; 
     const char *msg;
     va_dcl
#endif
{
    va_list vl;
    struct string * specify = NULL;
    int save_copy = 0;

    int r = 0;
    enum browser_status r1 = browser_error;
    enum browser_status BS = browser_error;

    switch (w) {
    case word_open: 
	specify = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFileFSOpen,
					"You must specify a file or folder to open."));
	break;
    case word_change: 
	specify = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFileFSChange,
					"You must specify a file or folder to change to."));
	break;
    case word_save:
	specify = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFileFSSave,
					"You must specify a file or folder to save to."));
	break;
    case word_copy:
	specify = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFileFSCopy,
					"You must specify a file or folder to copy to."));
	break;
    case word_save_copy:
	specify = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFileFSSaveCopy,
					"You must specify a file or folder to save copy to."));
	save_copy = 1;
	break;
    case word_read:
	specify = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFileFSRead,
					"You must specify a file or folder to read."));
	break;
    }

    if (1) {

	if (fc) {
	    int rerun;

	    if (FBROWSER_CALL_magic != fc->magic)
		panic("BROWSER PANIC",__FILE__,__LINE__,"gen_browser",
		      "Bad magic number",0);

	    rerun = 0 != (fc->flags & FBROWSER_RERUN);

	    Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
	    
	    r1 = run_browser(page,p,buffer,fc,save_copy,rerun,specify,prev_fold,
			     aview,format,msg,vl); 

	    va_end(vl);
	    
	} else {

	    Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
	    
	    r1 = run_browser(page,p,buffer,fc,save_copy,0,specify,prev_fold,
			     aview,format,msg,vl);

	    va_end(vl);
	    
	}
	
    } else {
RERUN:
	BS = browser_error;

	Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
	
	r1 = run_browser(page,p,buffer,fc,save_copy,1,specify,prev_fold,
			 aview,format,msg,vl);

	va_end(vl);
	
    }

    switch(r1) {
    case browser_select:
	BS = r1;
	break;
	
    default:
	panic("BROWSER PANIC",__FILE__,__LINE__,"gen_browser",
		  "Bad return value of run_browser",0);
    case browser_error:
	BS = r1;
	break;

    case browser_fbrowser:
    FBROWSER:
	if (!fc) 
	    panic("BROWSER PANIC",__FILE__,__LINE__,"gen_browser",
		  "fbrowser_call is not set",0);
	
	BS = fbrowser_hook_enter(page,p,mailbox,buffer,fc);

	break;
    }


    switch(BS) {
	int flags;

    case browser_select:
	DPRINT(Debug,5,
	       (&Debug,"gen_browser: browser select\n"));

	flags = give_dir_flags(p);

	if (0 == (BROWSER_SELECTED & flags))
	   panic("BROWSER PANIC",__FILE__,__LINE__,"gen_browser",
		 "Not selected",0);
	r = 1;
	break;

    case browser_error:
	DPRINT(Debug,5,
	       (&Debug,"gen_browser: browser error\n"));
	r = 0;

	break;

    case browser_fbrowser:
	DPRINT(Debug,5,
	       (&Debug,"gen_browser: browser fbrowser\n"));
	goto FBROWSER;

	case browser_rerun:
	DPRINT(Debug,5,
	       (&Debug,"gen_browser: browser rerun\n"));

	clear_selection_dir(p);

	goto RERUN;

    default:
	DPRINT(Debug,1,
	       (&Debug,"gen_browser: browser %d\n",
		BS));
	panic("BROWSER PANIC",__FILE__,__LINE__,"gen_browser",
	      "Bad BS",0);
    }

    free_string(&specify);

    DPRINT(Debug,4, (&Debug, "gen_browser=%d\n",
		     r));

    return r;
}

int file_browser(
#if ANSI_C
		 struct menu_context *page,
		 struct folder_browser *p,
		 struct string **buffer,
		 struct fbrowser_call *fc,
		 enum word_sel w,
		 struct AliasView   *aview,
		 struct MailboxView *mailbox,
		 struct string * default_extension,
		 const char *format, const char *msg, ...
#else
		 page,p,buffer,fc,w,aview,mailbox,
		 default_extension,
		 format, msg, va_alist
#endif
)
#if !ANSI_C
     struct menu_context *page;
     struct folder_browser *p;
     struct string **buffer;
     struct fbrowser_call *fc;
     enum word_sel w;
     struct AliasView   *aview;
     struct MailboxView *mailbox;
     struct string * default_extension;
     const char *format; 
     const char *msg;
     va_dcl
#endif
{
    int ret = 0;
    int code;
    va_list vl;

    struct string *text;
    struct string *vector[pv_COUNT];     /* We use same vector than on folder browser
				            so that we can share routines ....
					 */
    struct  browser_edit BE;
    struct enter_info INFO;
    struct string * specify = NULL;
    int delay_redraw = 0;

    enum browser_status BS = browser_error;

    int LINES,COLUMNS;

    DPRINT(Debug,5,
	   (&Debug, "file_browser: Entering [%s]\n",format));

    zero_enter_info(&INFO);
    
    switch (w) {
    case word_open: 
	specify = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFileSOpen,
					"You must specify a file to open."));
	break;
    case word_change: 
	specify = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFileSChange,
					"You must specify a file to change to."));
	break;
    case word_save:
	specify = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFileSSave,
					"You must specify a file to save to."));
	break;
    case word_copy:
	specify = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFileSCopy,
					"You must specify a file to copy to."));
	break;
    case word_save_copy:
	specify = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFileSSaveCopy,
					"You must specify a file to save copy to."));
	break;
    case word_read:
	specify = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFileSRead,
					"You must specify a file to read."));
	break;
    }

    Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
    text = elm_smessage(0,format,msg,vl);
    va_end(vl);
    
    vector[pv_buffer]    = *buffer;
    vector[pv_text]      = text;
    vector[pv_specify_text]    = specify;
    vector[pv_prev_fold] = NULL;       /* prev_fold */
    vector[pv_defext]    = default_extension;
    vector[pv_defname]   = NULL;
    vector[pv_fbrowser_filter] = NULL;

    INFO.browser     = &BE;
    INFO.pvector     = vector;
    
    init_browser_edit(&BE, p, aview);

    INFO.give_buffer  = gb_filebrowser;
    INFO.alter_buffer = default_alter_buffer;
    INFO.full_page    = default_full_page;
    INFO.prompt_hint  = default_prompt_hint;

    INFO.flags       = (*buffer ? 0 : OE_APPEND_CURRENT ) |
	OE_REDRAW_MARK | OE_TABACTION | OE_SIG_CHAR;

    INFO.ch_count    = 0;
    INFO.builtin     = NULL;
    INFO.in_utils    = NULL;
    INFO.address     = NULL;
    INFO.current_page = page;

    if (fc) {
	if (FBROWSER_CALL_magic != fc->magic)
	    panic("BROWSER PANIC",__FILE__,__LINE__,"file_browser",
		  "Bad magic number",0);
	           
	INFO.GB_FLAGS |= GB_FBROWSER_AVAILABLE;

	if (get_browser_sel_type(p) == selection_folder &&
	    0 == (fc -> options & FB_NOMBOX_CHECK)) {
	    
	    BE.gb_flags  |= GB_MBOX_CHECK;

	}

	if (fc->default_extension)
	    free_string(& (fc->default_extension));

	if (default_extension)
	    fc->default_extension = dup_string(default_extension);

	vector[pv_defname] = fc->default_filename;
    }


    if (0) {
    RERUN:

	vector[pv_buffer]    = *buffer;
	
	BS = browser_error;
	INFO.flags   |= OE_APPEND_CURRENT;   
	BE.gb_flags  |= GB_MENU|GB_REDRAW|GB_SORT_IT;
	BE.gb_line    = -1;
    }

    clear_selection_dir(p);
    do {
	code = enter_helper(&INFO,page,page);

	menu_ClearScreen(page);   

	menu_redraw_children(page);

	delay_redraw++;
    } while (code == REDRAW_MARK);

    menu_get_sizes(page,&LINES,&COLUMNS);
    
    menu_MoveCursor(page,LINES-4,0);
    menu_CleartoEOS(page);
    show_last_error();
   
    menu_MoveCursor(page,LINES-4,0);
    menu_CleartoEOS(page);
    show_last_error();

    *buffer = vector[pv_buffer];

    if (0 != code) {
	DPRINT(Debug,4, (&Debug, "file_browser: Ctrl-C or error\n"));
	ret = 0;
    } else {
	int flags = give_dir_flags(p);
	
	if (0 != (BE.gb_flags & GB_SWITCH_FBROWSER)) {

	    
	    if (!fc)
		panic("BROWSER PANIC",__FILE__,__LINE__,"file_browser",
		      "fc (fbrowser_call) not set",0);	    

	    if (fc->filter)
		free_string(& (fc->filter));

	    fc->filter = vector[pv_fbrowser_filter];
	    vector[pv_fbrowser_filter] = NULL;

	    fc->flags = 0;

	    BE.gb_flags &= ~GB_SWITCH_FBROWSER;


	FBROWSER:
	    
	    BS = fbrowser_hook_enter(page,p,mailbox,buffer,fc);
   
	} else if (0 != (BE.gb_flags & GB_SWITCH_FBROWSER_SELECT)) {       
	    if (!fc)
		panic("BROWSER PANIC",__FILE__,__LINE__,"file_browser",
		      "fc (fbrowser_call) not set",0);
	    
	    if (0 == (BROWSER_SELECTED & flags))
		panic("BROWSER PANIC",__FILE__,__LINE__,"file_browser",
		      "not selected",0);
	    
	    if (fc->filter)
		free_string(& (fc->filter));

	    fc->flags = FBROWSER_USE_SELECTION;
	    
	    BE.gb_flags &=  ~GB_SWITCH_FBROWSER_SELECT;


	    BS = fbrowser_hook_enter(page,p,mailbox,buffer,fc);

	} else if (0 != (BROWSER_SELECTED & flags)) {
	    BS =  browser_select;
	}
    }

    switch(BS) {
	int flags;

    case browser_select:
	DPRINT(Debug,5,
	       (&Debug,"file_browser: browser select\n"));
	
	flags = give_dir_flags(p);

	if (0 == (BROWSER_SELECTED & flags))
	   panic("BROWSER PANIC",__FILE__,__LINE__,"file_browser",
		 "Not selected",0);

	ret = 1;
	break;
	
    case browser_error:
	DPRINT(Debug,5,
	       (&Debug,"file_browser: browser error\n"));
	
	ret = 0;
	break;
	
    case browser_fbrowser:
	DPRINT(Debug,5,
	       (&Debug,"file_browser: browser fbrowser\n"));
	goto FBROWSER;
	
    case browser_rerun:
	DPRINT(Debug,5,
	       (&Debug,"file_browser: browser rerun\n"));

	clear_selection_dir(p);

	goto RERUN;
	
    default:
	DPRINT(Debug,1,
	       (&Debug,"file_browser: browser %d\n",
		BS));
	panic("BROWSER PANIC",__FILE__,__LINE__,"file_browser",
	      "Bad BS",0);
    }
    
    free_string(&text);
    free_string(&specify);

    if (vector[pv_fbrowser_filter])
	free_string(& vector[pv_fbrowser_filter]);

    clear_browser_edit(&BE);

    if (delay_redraw || (INFO.GB_FLAGS & GB_REDRAW)) {
	DPRINT(Debug,10, (&Debug,  "file_browser: redraw set\n"));
	menu_trigger_redraw(page);	
    }

    DPRINT(Debug,10, (&Debug,  "file_browser=%d\n",ret));
    
    return ret;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
