static char rcsid[] = "@(#)$Id: aliases_map.c,v 2.15 2021/08/20 18:11:57 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.15 $   $Stat$
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_aliases.h"
#include "s_aliases.h"
#include "rc_imp.h"

DEBUG_VAR(Debug,__FILE__,"alias");


#define ALIASCLASS_magic	0xF60A

struct aliasclass {
    unsigned short magic;        /* ALIASCLASS_magic */


    enum aliasmode   { system_aliasmap, user_aliasmap, private_aliasmap,
		       system_oldmap,  user_oldmap } mode;

    struct aliases_map *  aliases_map;



    struct aliasview_record     ** found_aliases_map;
    int                            found_aliases_map_len;

    int                            need_update; 
    int                            need_write;

    /* Cheat ... */
    struct string              **  delete_list;
    int                            delete_list_len;

};        


static void reset_aliases_cache P_((struct aliasclass *map));
static void reset_aliases_deletions P_((struct aliasclass *map));

void am_free_aliasclass(map)
     struct aliasclass **map;
{
    if (ALIASCLASS_magic != (*map)->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_free_aliasclass",
	      "bad aliasclass magic",0);


    if ((*map)->aliases_map) {
	free_aliases_map(& ((*map)->aliases_map));	
    }

    reset_aliases_cache(*map);
    reset_aliases_deletions(*map);


    (*map)->magic = 0;       /* Invalidate */
    free(*map);
    *map = NULL;
}

struct aliasclass * am_malloc_aliasclass()
{
    struct aliasclass *ret = safe_malloc(sizeof(*ret));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)ret,sizeof (*ret));

    ret->mode        = private_aliasmap;
    ret->aliases_map = NULL;
    
    ret->found_aliases_map     = NULL;
    ret->found_aliases_map_len = 0;

    ret->need_update           = 0;
    ret->need_write            = 0;

    ret->delete_list           = NULL;
    ret->delete_list_len       = 0;

    ret->magic       = ALIASCLASS_magic;

    return ret;
}

static struct aliases_map * get_aliases_map P_((struct aliasclass *map));
static struct aliases_map * get_aliases_map(map)
     struct aliasclass *map;
{

    switch(map->mode) {
    case system_aliasmap:   

	if (map->aliases_map)
	    panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"get_aliases_map",
		  "Mode is not private",0);
	
	if (!system_aliases_map) {
	    DPRINT(Debug,4,(&Debug,
			    "get_aliases_map: Creating empty system_aliases_map\n"));

	    system_aliases_map =  new_aliases_map();
	}
       
	return system_aliases_map;

    case user_aliasmap:     

	if (map->aliases_map)
	    panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"get_aliases_map",
		  "Mode is not private",0);

	if (!user_aliases_map) {
	    DPRINT(Debug,4,(&Debug,
			    "get_aliases_map: Creating empty user_aliases_map\n"));

	    user_aliases_map =  new_aliases_map();
	}

	return user_aliases_map;

    case private_aliasmap: 
    case system_oldmap:
    case user_oldmap:
	return map->aliases_map;
    }

    panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"get_aliases_map",
	  "Bad mode",0);

    return NULL;
}

void am_set_standard_aliasmap(map,user)
     struct aliasclass *map;
     int user;
{
   if (ALIASCLASS_magic != map->magic)
       panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_set_standard_aliasmap",
	     "bad aliasclass magic",0);

   if (map->aliases_map) {
       panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_set_standard_aliasmap",
	     "aliases_map already set",0);
   }
      

   if (user) 
       map->mode = user_aliasmap;
   else
       map->mode = system_aliasmap;
   
}



void am_set_compat_aliasmap(map,am,mode)
     struct aliasclass  *map;
     struct aliases_map *am;
     enum compat_mode    mode;
{
   if (ALIASCLASS_magic != map->magic)
       panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_set_compat_aliasmap",
	     "bad aliasclass magic",0);

   if (map->aliases_map) {
       panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_set_compat_aliasmap",
	     "aliases_map already set",0);
   }

   map->aliases_map = am;
   switch (mode) {
   case compat_none:    map->mode = private_aliasmap; break;
   case compat_system:  map->mode = system_oldmap;    break;
   case compat_user:    map->mode = user_oldmap;      break;
   default:
       panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_set_compat_aliasmap",
	     "bad mode",0);
   }

}

int give_aliases_filename(map,am,fn,cs,pl, have_propline,change_info)
     struct aliasclass *map;
     struct aliases_map **am;
     char **fn;
     charset_t               * cs;
     struct editor_propline ** pl; 
     int                     * have_propline;
     struct file_changes    ** change_info;
{

   if (ALIASCLASS_magic != map->magic)
       panic("ALIAS VIEW PANIC",__FILE__,__LINE__," give_aliases_filename",
	     "bad aliasclass magic",0);


    *am = NULL;
    *fn = NULL;
    if (cs) 
	*cs = NULL;
    if (pl) 
	*pl = NULL;
    if (have_propline) 
	*have_propline = 0;
    if (change_info)
	*change_info = NULL;
    
    switch(map->mode) {
    case system_aliasmap:             /* Actually should not work .... */
	*am  =    system_aliases_map;
	*fn  =    system_aliases_file;
	if (cs) 
	    *cs  =    system_aliases_cs;
	if (pl)
	    *pl  =    system_aliases_pl;
	if (have_propline)
	    *have_propline = 1;

	break;
    case user_aliasmap:    
	*am  =    user_aliases_map;
	*fn  =    user_aliases_file;
	if (cs) 
	    *cs  =    user_aliases_cs;
	if (pl)
	    *pl  =    user_aliases_pl;
	if (have_propline) 
	    *have_propline = 1;
	if (change_info)
	    *change_info = &user_aliases_change;
	
	break;
    case  private_aliasmap: 
	*am = map->aliases_map;
	*fn = NULL;                    /* Filename is not known */

	DPRINT(Debug,6,(&Debug,
			"give_aliases_filename: Filename of aliases map is not known (private_aliasmap)\n"));
	
	return 0;

    case system_oldmap:
	*am = map->aliases_map;
	*fn = old_system_text_file;
	break;

    case user_oldmap:
	*am = map->aliases_map;
	*fn = old_user_text_file;
	break;

    default:
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"give_aliases_filename",
	      "Bad mode",0);

    }
    
    return 1;
}




static void reset_aliases_cache(map)
     struct aliasclass *map;
{
    if (map->found_aliases_map) {
	int i;
	
	for (i = 0; i < map->found_aliases_map_len; i++)
	    if (map->found_aliases_map[i])
		free_aliasview_record(& (map->found_aliases_map[i]));
	
	free(map->found_aliases_map);
	map->found_aliases_map = NULL;
    }

    map->found_aliases_map_len = 0; 
}


static void update_aliases_cache P_((struct aliasclass *map));
static void update_aliases_cache(map)
     struct aliasclass *map;
{
    struct aliasview_record ** old_aliases_map =
	map->found_aliases_map;
    int                        old_aliases_map_len =
	map->found_aliases_map_len;
    int i;
 
    map->found_aliases_map     = NULL;
    map->found_aliases_map_len = 0;

    if (! old_aliases_map || 
	!old_aliases_map_len )
	return;

    /* Preallocate new cache */

    map->found_aliases_map = 
	safe_calloc(old_aliases_map_len,
		    sizeof(map->found_aliases_map[0]));

    for (i = 0; i < old_aliases_map_len; i++)
	map->found_aliases_map[i] = NULL;    
    map->found_aliases_map_len = old_aliases_map_len;


    for (i = 0; i < old_aliases_map_len; i++) {
	int cindex;
	struct aliasview_record *S;

	if (!old_aliases_map[i])
	    continue;

	S = 
	    am_lookup_alias(map,
			    old_aliases_map[i]->alias_key,
			    &cindex);

	if (S) {

	    DPRINT(Debug,14,(&Debug,
			     "update_aliases_cache: \"%S\": %d => %d\n",
			     old_aliases_map[i]->alias_key,
			     i,
			     cindex));
	    
	    S->type   = old_aliases_map[i]->type;
	    S->status = old_aliases_map[i]->status;
	} else {
	    DPRINT(Debug,9,(&Debug,
			    "update_aliases_cache: \"%S\": %d => lost\n",
			    old_aliases_map[i]->alias_key,
			    i));
	}
	    
	free_aliasview_record( & old_aliases_map[i] );
    }


    free(old_aliases_map);
    old_aliases_map = NULL;
}

static void reset_aliases_deletions(map)
     struct aliasclass *map;
{
    if (map->delete_list) {
	int i;
	
	for (i = 0; i < map->delete_list_len; i++) {
	    if (map->delete_list[i])
		free_string(& ( map->delete_list[i] ));
	}

	free(map->delete_list);
	map->delete_list = NULL;
    }
    map->delete_list_len = 0;

}

void am_delete_alias(map,cindex)
     struct aliasclass *map;
     int cindex;
{
    struct aliasview_record * XX;


   if (ALIASCLASS_magic != map->magic)
       panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_delete_alias",
	     "bad aliasclass magic",0);


   XX = am_give_alias(map,cindex);

   if (!XX)
       return;

   /* Just cache deletions */

   map->delete_list = safe_array_realloc(map->delete_list,
					 (map->delete_list_len+1),
					 sizeof(map->delete_list[0]));   
   
   map->delete_list[map->delete_list_len++]
       = dup_string(aliasview_key(XX));  

   /* Mark for writing */
   map->need_write  = 1;

}

int am_deleted_alias(map,cindex)
     struct aliasclass *map;
     int cindex;
{
    int r;
    struct aliases_map *am = NULL;

    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_deleted_alias",
	      "bad aliasclass magic",0);
    
    am = get_aliases_map(map);
    
    if (!am) 
	return 0;
    
    r = aliases_map_deleted_alias(am,cindex);

    DPRINT(Debug,7,(&Debug,"am_deleted_alias(...,%d)=%d\n",
		    cindex,r));
    
    return r;
}

struct aliasview_record *am_give_alias(map,cindex)
     struct aliasclass *map;
     int cindex;
{
    struct aliasview_record *r;
    
    struct aliases_map *am = NULL;
    const struct address_alias * alias_value = NULL;
    const struct string        * alias_key   = NULL;

    int type = 0;

    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_give_alias",
	      "bad aliasclass magic",0);

    if (system_aliasmap == map->mode || 
	system_oldmap   == map->mode)
	type = SYSTEM;

    am = get_aliases_map(map);

    if (!am)
	return NULL;

    if (map->found_aliases_map_len <= cindex) {
	int i;

	map->found_aliases_map = safe_array_realloc(map->found_aliases_map,
						    (cindex+1),
						    sizeof(map->found_aliases_map[0]));
	
	for (i = map->found_aliases_map_len; i <= cindex; i++)
	    map->found_aliases_map[i] = NULL;

	map->found_aliases_map_len = cindex+1;
    }

    alias_value = aliases_map_get_alias(am,cindex,&alias_key);

    DPRINT(Debug,7,(&Debug,"am_give_alias(...,%d): ",cindex));
    if (alias_key) {
	DPRINT(Debug,7,(&Debug,"alias_key=%S ",alias_key));
    }
    DPRINT(Debug,7,(&Debug,"\n"));

    if (!alias_key)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_give_alias",
	      "No alias_key",0);

    if (! map->found_aliases_map[cindex])
	map->found_aliases_map[cindex] = 
	    malloc_aliasview_record(alias_key,alias_value,type,0);
    else
	replace_aliasview_record(map->found_aliases_map[cindex],
				 alias_key,alias_value);

    r = map->found_aliases_map[cindex];

    DPRINT(Debug,7,(&Debug,"am_give_alias(...,%d)=%p%s, alias_key=%S\n",
		    cindex,
		    r,
		    r ? " OK " : " (not found)",
		    alias_key));

    return r;
}

struct aliasview_record *am_lookup_alias(map,key,cindex)
     struct aliasclass *map;
     const struct string *key;
     int *cindex;
{
    struct aliasview_record    * ret         = NULL;

    struct aliases_map *am                   = NULL;
    const struct address_alias * tmp_value   = NULL;

    int index   = -1;

    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_lookup_alias",
	      "bad aliasclass magic",0);

    am = get_aliases_map(map);

    tmp_value = aliases_map_lookup_alias(am,key,&index);

    /* am_give_alias is used to instiate correct shared key pointer 
       for alias_key
    */

    if (index < 0) {

	if (tmp_value) {
	    DPRINT(Debug,1,(&Debug,
			    "am_lookup_alias: On value for alias %S is index %d, but alias found %p\n",
			    key,index,tmp_value));
		   
	}
	    
	DPRINT(Debug,9,(&Debug,
			"am_lookup_alias=NULL:  key=%S (index=%d)\n",
			key,index));
	       
	return NULL;
    }

    if (cindex)
	*cindex = index;

    ret = am_give_alias(map,index);

    if (!ret) {
	 DPRINT(Debug,1,(&Debug,
			 "am_lookup_alias: On value for alias %S is index %d, but alias not found by index\n",
			 key,index));
	 return NULL;
    }

    if (ret->alias_value != tmp_value) {
	DPRINT(Debug,1,(&Debug,
			"am_lookup_alias: alias %S, index %d: aliases_map_lookup_alias=%p != %p (found by index)\n",
			key,index, tmp_value,ret->alias_value));

	       }

    return ret;
}

void am_add_alias(map,rec)
     struct aliasclass        *map; 
     struct aliasview_record  *rec;
{
    struct aliases_map                 * am  = NULL;

    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_add_alias",
	      "bad aliasclass magic",0);
    
    if (ALIASVIEW_record_magic != rec->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_add_alias",
	      "bad record magic",0);

    am = get_aliases_map(map);
    
    if (!am) {
	DPRINT(Debug,1,(&Debug,
			"am_add_alias: \"%S\": No aliases map available\n",
			rec->alias_key));
	return;
    }

    aliases_map_add_alias(am,rec->alias_key,rec->alias_value);
   
    /* Mark for rebuild */
    map->need_update = 1;     
    
    /* Mark for writing */
    map->need_write  = 1;
}

struct aliasview_record * am_set_alias(map,old_index,buffer)
     struct aliasclass        *map;   
     int                       old_index;
     const struct alias_buffer      *buffer;
{
    struct aliasview_record    * ret         = NULL;
    int index = -1;
    struct aliases_map                 * am  = NULL;
    const struct address_alias * tmp_value   = NULL;



    
    if (ALIASVIEW_buffer_magic != buffer->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_set_alias",
	      "bad buffer magic",0);
    
    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_set_alias",
	      "bad aliasclass magic",0);
    
    am = get_aliases_map(map);
    
    if (!am) {
	DPRINT(Debug,1,(&Debug,
			"am_set_alias: \"%S\": No aliases map available\n",
			buffer->alias_key));
	return NULL;
    }

    aliases_map_add_alias(am,buffer->alias_key,buffer->alias_value);

    tmp_value = aliases_map_lookup_alias(am,buffer->alias_key,&index);

    if (!tmp_value) 
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_set_alias",
	      "Added alias not found",0);
	
    if (old_index >= 0 && index != old_index) {
	DPRINT(Debug,1,(&Debug,
			"am_set_alias: \"%S\": Alias position changed, removing old index %d\n",
			buffer->alias_key,old_index));

	/* NOTE: Removing index invalidates all indexes   ... */

	aliases_map_remove_index(am,old_index);

	update_aliases_cache(map);
    } 

  
    /* am_give_alias() and am_lookup_alias() does always lookup so 
       cache not needed to clear otherwise */


    /* Mark for rebuild */
    map->need_update = 1;     

    /* Mark for writing */
    map->need_write  = 1;

    ret = am_give_alias(map,index);

    if (!ret) {
	DPRINT(Debug,1,(&Debug,
			"am_set_alias: On value for alias %S is index %d, but alias not found by index\n",
			buffer->alias_key,index));
	return NULL;
    }

    if (ret->alias_value != tmp_value) {
	DPRINT(Debug,1,(&Debug,
			"am_set_alias: alias %S, index %d: aliases_map_lookup_alias=%p != %p (found by index)\n",
			buffer->alias_key,index, tmp_value,ret->alias_value));
	
    }

    return ret;
}



int am_count_aliases(map)
     struct aliasclass *map;
{
    struct aliases_map *am = NULL;
    int count;

    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_count_aliases",
	      "bad aliasclass magic",0);

    am = get_aliases_map(map);
    count = aliases_map_item_count(am);

    DPRINT(Debug,7,(&Debug,"am_count_aliases=%d\n",count));

    return count;
}

int am_update_aliases(map)
     struct aliasclass *map;
{
    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__," am_update_aliases",
	      "bad aliasclass magic",0);

    /* FIXME: aliasmap update from file not supported now */

    if (map->need_update) {
	map->need_update = 0;

	update_aliases_cache(map);

	DPRINT(Debug,7,(&Debug,"am_update_aliases=1: Aliases cache update\n"));

	return 1;
    }

    return 0;
}

void am_add_to_alias_stack(stack,map)
     struct alias_stack * stack;
     struct aliasclass *map;
{
    union alias_stack_item map_info;

    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_add_to_alias_stack",
	      "bad aliasclass magic",0);

    map_info.alias_map = get_aliases_map(map);

    add_lookup_to_alias_stack(stack,alias_simple_lookup,map_info);
}

int am_file_changed(map)
     struct aliasclass *map;
{
    int r = 0;

    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_file_changed",
	      "bad aliasclass magic",0);

    if (map->need_write) {
	DPRINT(Debug,14, (&Debug,
			  "am_file_changed: need_write set\n"));

	r = 1;
	
    } else {
	char * fn               = NULL;
	struct aliases_map * am = NULL;
	charset_t            cs = NULL;
	struct editor_propline * pl = NULL; 
	int                     have_propline = 0;
	struct file_changes     * file_information = NULL;

	int err = 0;
	enum file_changes_result res;

	if (!give_aliases_filename(map,&am,&fn,&cs,&pl,&have_propline,
			       &file_information)) {
	    
	    DPRINT(Debug,1,(&Debug,
			    "am_file_changed=0: Filename of aliases map is not known\n"));
	    
	    
	    return 0;
	    
	}

	if (!fn || !file_information) {

	    DPRINT(Debug,14,(&Debug,
			    "am_file_changed=0: No filename or file information\n"));


	    return r;
	}

	res = test_file_changes(fn, file_information,&err);

	DPRINT(Debug,14,(&Debug,
			 "am_file_changed: %s: test_file_changes gives %d",
			 fn,res));
	switch (res) {
	case file_changes_error:  DPRINT(Debug,14,(&Debug," file_changes_error"));
	    if (err) {
		DPRINT(Debug,14,(&Debug,"; error %s",
				 strerror(err)));
	    }

	    /* If no file, indicate that write needed */

	    if (ENOENT == err) {
		DPRINT(Debug,14,(&Debug,", setting need_write"));

		map->need_write = 1;
		r = 1;
	    }
	    break;
	case no_file_changes: DPRINT(Debug,14,(&Debug," no_file_changes"));
	    break;
	case have_file_changes: DPRINT(Debug,14,(&Debug," have_file_changes"));
	    r = 1;
	    break;
	}
	DPRINT(Debug,14,(&Debug,"\n"));

	
	if (file_changes_error == res && err && err != ENOENT) {
	    lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesFileError,
			      "Aliases file %s: %s"),
		      fn, strerror(err));
	}
	
    }

    DPRINT(Debug,14,(&Debug,
		     "am_file_changed=%d\n",r));
    
    return r;
}



int am_write_changed(map,rebuild_flag)
     struct aliasclass *map;
     int *rebuild_flag;
{
    char * fn               = NULL;
    struct aliases_map * am = NULL;
    charset_t            cs = NULL;
    struct editor_propline * pl = NULL; 
    int                     have_propline = 0;
    struct file_changes     * file_information = NULL;


    
    int err;
    char *tmp               = NULL;
    int ret                 = 0;
    FILE *F                 = NULL;
    int i;

    char * bck = NULL;
    
    int old_fd = -1;
    int old_fd_locked = 0;
    int loop_message_printed = 0;
    int loop_count = 0;

    FILE * old_F = NULL;

    int r_dump = 0;
    
    struct tm * the_time = NULL;
    time_t      now = 0;
    
    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_write_changed",
	      "bad aliasclass magic",0);


    
    if (!give_aliases_filename(map,&am,&fn,&cs,&pl,&have_propline,
			       &file_information)) {

	DPRINT(Debug,1,(&Debug,
			"am_write_changed=0: Filename of aliases map is not known\n"));


	return 0;

    }

    if (!fn) {
	DPRINT(Debug,1,(&Debug,
			"am_write_changed: filename is not set\n"));
	return 0;
    }

    
    if (system_oldmap == map->mode ||
	user_oldmap   == map->mode) {
	lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesOldFormatCanNotSaved,
			  "Old format aliases file can not be saved: %s"),
		  fn);
	return 0;
    }

  
    if (!am) {
	DPRINT(Debug,1,(&Debug,
			"am_write_changed: map is not set\n"));
	return 0;
    }

    if (((time_t) -1) != time(&now)) {
	
	the_time = localtime(&now);
	
	if (!the_time) {
	    err = errno;
	    
	    DPRINT(Debug,4, (&Debug,
			     "am_write_changed: localtime failed, time=%ld: %s\n",
			     (long)now,strerror(err)));
	}
	
    } else {
	err = errno;
	
	DPRINT(Debug,4, (&Debug,
			 "am_write_changed: Date not available, time: %s\n",
			 strerror(err)));	    
    }

    
    /* Similar than  load_aliases_map(), but opens current
       aliases file for  read write and uses exclusive lock
    */

    /* Similar than write_conf() but is specific to aliases */
    
    while (-1 == old_fd) {
	enum syscall_status  r;
	enum FLOCKING_status r1;
	struct stat old_fd_stat;
	struct stat filename_stat;

	int have_old_fd_stat = 0;
	int have_filename_stat = 0;

    	err = can_open(fn,"r+");
	
	old_fd_locked = 0;
	
	if (err) {
	    DPRINT(Debug,14,(&Debug,
			     "am_write_changed: %s: %s (can_open)\n",
			     fn,strerror(err)));


	    
	    goto have_old_fd_error;
	}

	old_fd = open(fn,O_RDWR);
	
	if (-1 == old_fd) {
	    err = errno;
	    
	    DPRINT(Debug,10,(&Debug,
			     "am_write_changed: %s: %s\n",
			     fn,strerror(err)));
	    
	have_old_fd_error:
	    if (EINTR == err) {
		goto oops;
	    }
	    
	    if (loop_message_printed) {
		lib_transient(CATGETS(elm_msg_cat, AliasesSet, AliasesWaitingUpdFail,
				      "Waiting aliases file %s to be updated... Fail: %s"),
			      fn,strerror(err));
	    }			      
	    
	    if (ENOENT == err) {
		goto no_merge;
	    }

	    if (loop_message_printed) {  /* HACK */
		sleep_message();
	    }

	    goto merge_failed_message;
	}

	r = fstat(old_fd,&old_fd_stat);

	switch (r) {
	    char *X;
	case syscall_success:
	    DPRINT(Debug,14,(&Debug,
			     "am_write_changed: fstat %d: %s succeed: dev %ld ino %ld size %ld modified %ld",
			     old_fd,fn,
			     (long)old_fd_stat.st_dev,
			     (long)old_fd_stat.st_ino,
			     (long)old_fd_stat.st_size,
			     (long)old_fd_stat.st_mtime));
	    
	    X = ctime(& (old_fd_stat.st_mtime));
	    if (X) { /* ctime() includes newline */
		DPRINT(Debug,14,(&Debug," -- %s",X));
	    } else {
		DPRINT(Debug,14,(&Debug,"\n"));
	    }

	    have_old_fd_stat = 1;

	    break;

	     break;
	case syscall_error:
	    err = errno;

	    DPRINT(Debug,14,(&Debug,
			     "am_write_changes: fstat %d: %s: %s\n",
			     old_fd,fn,strerror(err)));

	    if (EINTR == err) {
		goto oops;
	    }

	    if (loop_message_printed) {
		lib_transient(CATGETS(elm_msg_cat, AliasesSet,
				      AliasesWaitingUpdError,
				      "Waiting aliases file %s to be updated... Error: %s"),
			      fn,strerror(err));
	    }

	    /* Just read old file ? */
	    goto quit_locking_loop;	    
	}

    	/* When write_conf() and  am_write_changed()
	   merges changes from
	   <conf_file> and writes <conf_file>.N,
	   it locks  <conf_file> with exclusive (write
	   lock) althouh new file <conf_file>.N
	   is written and then renamed to <conf_file>
	*/

	r1 = filelock_fd(old_fd,FLOCKING_exclusive,
			 &conf_merge_locking,
			 fn,
			 FLOCKING_non_block,
			 &err);

	switch (r1) {
	case FLOCKING_FAIL:
	    DPRINT(Debug,14,(&Debug,
			     "am_write_changes: filelock_fd %d: %s locking failed: %s\n",
			     old_fd,fn,strerror(err)));
	    /* Just read old file ? */
	    goto quit_locking_loop;
	    
	case FLOCKING_OK:
	    DPRINT(Debug,14,(&Debug,
			     "am_write_changes: filelock_fd %d: %s locked\n",
			     old_fd,fn));
	    old_fd_locked = 1;
	    break;
	case FLOCKING_RETRY:
	    DPRINT(Debug,14,(&Debug,
			     "am_write_changes: filelock_fd %d: %s locking failed (retry)",
			     old_fd,fn));
	    if (err) {
		DPRINT(Debug,14,(&Debug,": %s",
				 strerror(err)));
	    }
	    DPRINT(Debug,14,(&Debug,"\n"));
	    
	    if (EINTR == err) {
		goto oops;
	    }
	    
	    goto wait_to_change;
	case FLOCKING_UNSUPPORTED:
	    DPRINT(Debug,14,(&Debug,
			     "am_write_changes: filelock_fd %d: %s locking not supported\n",
			     old_fd,fn));
	    /* Just read old file ? */
	    goto quit_locking_loop;	
	}

	r = stat(fn,&filename_stat);
	switch (r) {
	    char *X;
	case syscall_success:
	    DPRINT(Debug,14,(&Debug,
			     "am_write_changes: stat %s succeed: dev %ld ino %ld size %ld modified %ld",
			     fn,
			     (long)filename_stat.st_dev,
			     (long)filename_stat.st_ino,
			     (long)filename_stat.st_size,
			     (long)filename_stat.st_mtime));
	    X = ctime(& (filename_stat.st_mtime));
	    if (X) { /* ctime() includes newline */
		DPRINT(Debug,14,(&Debug," -- %s",X));
	    } else {
		DPRINT(Debug,14,(&Debug,"\n"));
	    }

	    have_filename_stat = 1;
	    
	    break;
	case syscall_error:
	    err = errno;

	    DPRINT(Debug,14,(&Debug,
			     "am_write_changes: stat %s: %s\n",
			     fn,strerror(err)));

	    if (EINTR == err) {
		goto oops;
	    }
	    
	    /* Just read old file ? */
	    goto quit_locking_loop;
	}
	
	if (!have_filename_stat || !have_old_fd_stat) {

	    DPRINT(Debug,14,(&Debug,
			     "am_write_changes: file %s - no stat?\n",
			     fn));

	    goto quit_locking_loop;
	} else if (filename_stat.st_dev   == old_fd_stat.st_dev  &&
	    filename_stat.st_ino   == old_fd_stat.st_ino  &&
	    filename_stat.st_size  == old_fd_stat.st_size &&
	    filename_stat.st_mtime == old_fd_stat.st_mtime) {

	    DPRINT(Debug,14,(&Debug,
			     "am_write_changes: file %s not changed (since open)\n",
			     fn));
	    
	    if (loop_message_printed) {
		lib_transient(CATGETS(elm_msg_cat, AliasesSet,
				      AliasesWaitingUpdOK,
				      "Waiting aliases file %s to be updated... OK"),
			      fn);
		loop_message_printed = 0;
	    }
	    
	    goto quit_locking_loop;

	} else {
	    int wait_it;
	    
	    DPRINT(Debug,14,(&Debug,
			     "am_write_changes: file %s changed after open\n",
			     fn));

	oops:
	    wait_it = 0;
	    
	    if (0) {
	    wait_to_change:
		wait_it = 1;
	    }

	    
	    if (loop_count++ > 10) {
		DPRINT(Debug,14,(&Debug,
				 "am_write_changes: file %s - try #%d -- quiting\n",
				 fn,loop_count));
		goto quit_locking_loop;
	    }

	    if (wait_it) {
		DPRINT(Debug,14,(&Debug,
				 "am_write_changes: file %s - try #%d -- waiting\n",
				 fn,loop_count));

		
		if (!loop_message_printed) {
		    lib_transient(CATGETS(elm_msg_cat, AliasesSet,
					  AliasesWaitingUpd,
					  "Waiting aliases file %s to be updated..."),
				  fn);
		    loop_message_printed = 1;
		    
		}
		
		if (POLL_method)
		    wait_for_timeout(5);
		else
		    sleep(5);
	    } else {
		DPRINT(Debug,14,(&Debug,
				 "am_write_changes: file %s - try #%d -- looping\n",
				 fn,loop_count));
	    }

	    if (0) {
	    quit_locking_loop:

		old_F = fdopen(old_fd,"r");

		if (old_F) {
		    DPRINT(Debug,14,(&Debug,
				     "am_write_changes: file %s - try #%d -- done\n",
				     fn,loop_count));

		    
		    break;
		}

		err = errno;

		DPRINT(Debug,14,(&Debug,"load_aliases_map: fdopen %d: %s: %s\n",
				 old_fd,fn,strerror(err)));

		if (EINTR != err) {
		merge_failed_message:

		    if (! map->need_write) {	    
			DPRINT(Debug,9,(&Debug,
					"am_write_changed: %s: Merge failed (%s) and write is not needed\n",
					fn,strerror(err)));
			ret =  1;
			goto quit;
		    }  

		    
		    if (the_time && !bck)
			bck = elm_message(FRM("%s.%04d-%02d-%02d.old"),
					  fn,
					  the_time->tm_year+1900,
					  the_time->tm_mon+1,
					  the_time->tm_mday);
		    
		    if (bck) {
			
			lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesMergeFailed,
					  "Aliases file %s merge failed: %s  (backup: %s)"),
				  fn,strerror(err),bck);		
		    } else {
			lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesMergeFailed1,
					  "Aliases file %s merge failed: %s  (no backup)"),
				  fn,strerror(err));					
		    }

		    goto no_merge;
		}
	    }

	    
	    if (old_fd_locked) { /* Just ignore error --
				     close() should release this anyway
				  */
		filelock_fd(old_fd,FLOCKING_release,&conf_merge_locking,
			    fn,FLOCKING_non_block,NULL);
	    }
	    
	    close(old_fd);  /* Result ignored */
	    old_fd = -1;
	}
    }

    if (old_F) {
	int errors = 0;
	int updateX = 0;
	
	int rx = merge_aliases_map(fn,old_F,am,&errors,&cs,&pl,NULL,&updateX,
				   file_information);

	DPRINT(Debug,14, (&Debug,
			  "am_write_changed: current file %s - merge_aliases_map=%d %s\n",
			  fn,
			  rx,
			  rx ? "succeed" : "failed"));

	if (errors) {
	    DPRINT(Debug,14, (&Debug,
			      "am_write_changed: current file %s - %d errors on merge_aliases_map\n",
			      fn,errors));
	}

	if (updateX) {
	    DPRINT(Debug,14, (&Debug,
			      "am_write_changed: current file %s - marking map for update / rebuild\n",fn));
	    map->need_update = 1;            	
	    *rebuild_flag = 1;
	}
	
	if (!rx) {

	    if (the_time && !bck)
		bck = elm_message(FRM("%s.%04d-%02d-%02d.old"),
				  fn,
				  the_time->tm_year+1900,
				  the_time->tm_mon+1,
				  the_time->tm_mday);
	    
	    if (bck) {
		
		lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesMergeFailed2,
				  "Aliases file %s merge failed (backup: %s)"),
			  fn,bck);		
	    } else {
		lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesMergeFailed3,
				  "Aliases file %s merge failed (no backup)"),
			  fn);					
	    }	    
	}
		
    } else {

    no_merge:
	
	DPRINT(Debug,14,(&Debug,
			  "am_write_changed: No merge\n"));
    }
    
    
    if (! map->need_write) {	    
	DPRINT(Debug,9,(&Debug,
			"am_write_changed: Write is not needed\n"));
	ret =  1;
	goto quit;
    }  

    

	
    if (map->delete_list_len) {
	int i;

	DPRINT(Debug,8,(&Debug,
			"am_write_changed: %d deletions\n",
			map->delete_list_len));

	for (i = 0; i < map->delete_list_len; i++) {
	    if (map->delete_list[i]) {
		int r;

		DPRINT(Debug,8,(&Debug,
				"am_write_changed: deleting %S ",
				map->delete_list[i]));

		r = aliases_map_remove_alias(am,map->delete_list[i]);
		
		DPRINT(Debug,8,(&Debug,"%s\n",
				r ? "OK" : "Not found"));		

		if (r)
		    *rebuild_flag = 1;

	    }
	}

	update_aliases_cache(map);

	reset_aliases_deletions(map);
	
	/* Mark for rebuild */
	map->need_update = 1;     
    }


    tmp = elm_message(FRM("%s.N"),fn);
    err = can_open(tmp,"w");

    if (err) {
	lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesTmpFileNotWriteable,
			  "Aliases temp file %s is not writeable: %s"),
		  tmp, strerror(err));
	
	goto fail;
    }

    F = fopen(tmp,"w");
    if (!F) {
	err = errno;

	lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesTmpFileNotWriteable,
			  "Aliases temp file %s is not writeable: %s"),
		  tmp, strerror(err));
	
	goto fail;
    }


    r_dump = dump_aliases_map(F,am,NULL,"ELM",version_buff,
			      cs,pl,file_information,&err);

    if (!r_dump) {
	lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesTmpFileNotWriteable,
			  "Aliases temp file %s is not writeable: %s"),
		  tmp, strerror(err));

	fclose(F); /* ignore error */
	goto fail;
    }

    if (EOF == fclose(F)) {
	err = errno;

	lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesTmpFileNotWriteable,
			  "Aliases temp file %s is not writeable: %s"),
		  tmp, strerror(err));
	
	goto fail;
    }

    if (bck) {
	enum syscall_status ra = link(fn,bck);

	 switch (ra) {
	 case syscall_success:
	     DPRINT(Debug,4, (&Debug,
			      "am_write_changed: Created backup %s => %s\n",
			      fn,bck));
	     break;
	 case syscall_error:
	     err = errno;
	     DPRINT(Debug,4, (&Debug,
			      "am_write_changed: Backup link %s => %s failed: %s\n",
			      fn,bck,
			      strerror(err)));

	     lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesBackupFailed,
			       "Aliases file backup %s => %s failed: %s"),
		       fn,bck,
		       strerror(err));

	     unlink(tmp);   /* Ignore error */
	     goto fail;
	 }
    }
    
        
    if (0 != rename(tmp,fn)) {
	
	lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesTmpFileNotRenamed,
			  "Aliases temp file is not renamed to %s: %s"),
		  fn,strerror(err));

	goto fail;	
    }


    lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesFileWritten,
		      "Aliases file %s is written"),
	      fn);
    ret = 1;
    map->need_write = 0;

    for (i = 0; i < map->found_aliases_map_len; i++) {

	/* Clear NEW flag */

	if (map->found_aliases_map[i])
	    aliasview_clearf_status(map->found_aliases_map[i],NEW);
    }

 fail:
 quit:
    if (tmp) {
	free(tmp);
	tmp = NULL;
    }
	
    if (bck) {
	free(bck);
	bck = NULL;
    }

    if (-1 != old_fd && old_fd_locked) { /* Just ignore error --
					    close() should release this anyway
					 */
	filelock_fd(old_fd,FLOCKING_release,&conf_merge_locking,
		    fn,FLOCKING_non_block,NULL);
    }

    if (old_F) {
	fclose(old_F);
	old_F = NULL;
	old_fd = -1;
    } else if (-1 != old_fd) {
	close(old_fd);
	old_fd = 1;
    }

    
    DPRINT(Debug,9,(&Debug,
		    "am_write_changed=%d\n",ret));
    
    return ret;
}

int am_reload_aliases(map)
     struct aliasclass *map;
{
    char * fn               = NULL;
    struct aliases_map * am = NULL;
    int errors = 0;
    struct aliases_map * tmp = NULL;
    int ret = 0;
    charset_t                cs = NULL;
    struct editor_propline * pl = NULL;
    int        have_propline = 0;
    struct file_changes    * change_info = NULL;

    
    struct editor_propline * tmp_pl = NULL;
    struct file_changes      tmp_change_info = NULL_file_changes;

    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_reload_aliases",
	      "bad aliasclass magic",0);


    if (!give_aliases_filename(map,&am,&fn,&cs,&pl,&have_propline,
			       &change_info)) {

	DPRINT(Debug,1,(&Debug,
			"am_reload_aliases=0: Filename of aliases map is not known\n"));
	return 0;

    }
    
    if (!fn) {
	DPRINT(Debug,1,(&Debug,
			"am_reload_aliases: filename is not set\n"));
	return 0;
    }

    if (have_propline && pl) {
	tmp_pl = copy_editor_propline(pl);
    }

    tmp = load_aliases_map(fn,&errors,&cs,
			   have_propline ? &tmp_pl : NULL,
			   NULL /* suggest_rewrite */,
			   &tmp_change_info);   

    if (!tmp || errors) {
	lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesFileNotReloaded,
			  "Failed to reload aliases file: %s"),
		  fn);

	goto failure;
    }

    switch(map->mode) {
    case system_aliasmap:             /* Actually should not work .... */

	if (system_aliases_map)
	    free_aliases_map(& system_aliases_map);
	system_aliases_map = tmp;
	tmp = NULL;

	if (system_aliases_pl)
	    free_editor_propline(& system_aliases_pl);
	system_aliases_pl = tmp_pl;
	tmp_pl = NULL;

	system_aliases_cs = cs;

	ret = 1;
	break;

    case user_aliasmap:    

	if (user_aliases_map)
	    free_aliases_map(&  user_aliases_map);
	user_aliases_map = tmp;
	tmp = NULL;

	if (user_aliases_pl)
	    free_editor_propline(& user_aliases_pl);
	user_aliases_pl = tmp_pl;
	tmp_pl = NULL;

	user_aliases_cs = cs;

	user_aliases_change = tmp_change_info;
	
	ret = 1;
	break;
    
    case system_oldmap:          /* Actually should not work .... */
    case user_oldmap:
    case private_aliasmap:       /* Should not be reached ... */

	if (map->aliases_map)
	    free_aliases_map(& map->aliases_map);
	
	map->aliases_map = tmp;
	tmp = NULL;
	ret = 1;
	break;
	       
    default:
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_reload_aliases",
	      "Bad mode",0);

    }
 
    if (ret)
	lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesFileReloaded,
			  "Aliases file %s is reloaded"),
		  fn);
 failure:
    if (tmp)
	free_aliases_map(&tmp);
    
    if (tmp_pl)
	free_editor_propline(&tmp_pl);

    DPRINT(Debug,9,(&Debug,
		    "am_reload_aliases=%d fn=%s\n",
		    ret, fn ? fn : "<none>"));

    return ret;
}


const char * am_is_user_editable(map)
     struct aliasclass *map;
{
    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_is_user_editable",
	      "bad aliasclass magic",0);

    if (user_aliasmap == map->mode)
	return user_aliases_file;

    return NULL;
}

int am_is_normal_target(map)
     struct aliasclass *map;
{
    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_is_normal_target",
	      "bad aliasclass magic",0);

    return 
	system_aliasmap == map->mode ||
	user_aliasmap   == map->mode;

}

int am_is_compat_map(map)
     struct aliasclass *map;
{
    if (ALIASCLASS_magic != map->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"am_is_compat_map",
	      "bad aliasclass magic",0);

    return 
	system_oldmap == map->mode ||
	user_oldmap   == map->mode;
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
