static char rcsid[] = "@(#)$Id: aliases.c,v 2.12 2021/07/14 07:19:22 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.12 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_aliases.h"

DEBUG_VAR(Debug,__FILE__,"alias");


static struct AliasView    ** aliasview_list = NULL;
static int                    aliasview_list_len = 0;


struct AliasView * new_aliasview() 
{
    int idx;
    struct AliasView **X;
    struct AliasView *r = safe_malloc(sizeof (*r));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)r,sizeof (*r));

    r->magic                = ALIASVIEW_magic;
    r->aliasclasscount      = 0;
    r->aliasclass           = NULL;

    r->view                 = NULL;
    r->view_len             = 0;

    r->normal_alias_len     = 0;

    for (idx = 0; idx < aliasview_list_len; idx++)
	if (!aliasview_list[idx]) {
	    DPRINT(Debug,7,(&Debug,
			       "malloc_view: Re-using aliasview index %d\n",
			       idx));
	    goto found;
	}

    X = safe_array_realloc(aliasview_list,(aliasview_list_len+1), sizeof (*X));
    X[aliasview_list_len] = NULL;
    aliasview_list = X;
    idx = aliasview_list_len++;

    DPRINT(Debug,7,(&Debug,
		       "malloc_view: Allocing new aliasview index %d\n",
		       idx));


 found:
    aliasview_list[idx] = r;

    return r;
}

static void reset_aliasview P_((struct AliasView *aview));
static void reset_aliasview(aview)
     struct AliasView *aview;
{

    if (aview->view) {
	free(aview->view);
	aview->view = NULL;
    }
    aview->view_len = 0;
    aview->normal_alias_len = 0;

}

static void free_aliasview1 P_((struct AliasView **x));
static void free_aliasview1(x)
     struct AliasView **x;
{
    struct AliasView *r = *x;
    if (r->aliasclass) {
	int i;

	for (i = 0; i < r->aliasclasscount; i++) {

	    am_free_aliasclass(& r->aliasclass[i]);

	}

	free(r->aliasclass); r->aliasclass = NULL;
    }
    r->aliasclasscount = 0;

    reset_aliasview(r);

    r->magic = 0;           /* Invalidate */
    free(r); r = NULL;

    *x = r;
}

void free_aliasview(x)
     struct AliasView **x;
{
    int i;
    struct AliasView *r = *x;

    if (r->magic != ALIASVIEW_magic) 
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"free_aliasview",
	      "bad aliasview magic",0);

    for ( i = 0; i < aliasview_list_len; i++) {
	if (aliasview_list[i] == r) {
	    
	    SIGDPRINT(Debug,7,(&Debug,
			       "free_aliasview: Aliasview index %d goes free\n",
			       i));
	    
	    aliasview_list[i] = NULL;
	    goto okei;
	}
    }

    panic("MBX VIEW PANIC",__FILE__,__LINE__,"free_aliasview",
	  "aliasview not found from list",0);
	
 okei:
    free_aliasview1(&r);

    *x = r;
}

void free_all_aliasviews()
{
    if (aliasview_list) {
	int i;
	
	for ( i = 0; i < aliasview_list_len; i++) {
	    if (aliasview_list[i]) {
		
		SIGDPRINT(Debug,7,(&Debug,"free_all_aliasviews: [%d]\n",
				   i));
		
		if (aliasview_list[i]->magic    != ALIASVIEW_magic)
		    panic("MBX VIEW PANIC",__FILE__,__LINE__,"free_all_aliasviews",
			  "Bad magic number",0);
		
		free_aliasview1(& (aliasview_list[i]));
	    }
	}
	
	free(aliasview_list);
	aliasview_list = NULL;
    }
    aliasview_list_len = 0;
}

struct aliasclass * add_aliasclass(x)
     struct AliasView *x; 
{
    x->aliasclass = safe_array_realloc(x->aliasclass,
				       ( x->aliasclasscount +1),
				       sizeof (x->aliasclass[0]));

    x->aliasclass[x->aliasclasscount] = am_malloc_aliasclass();

    return ( x->aliasclass[x->aliasclasscount++]);
}

/* Caller must free buffer */

struct alias_buffer * give_alias_buffer(rec)
     struct aliasview_record *rec;
{
    return malloc_alias_buffer(aliasview_key(rec),
			       aliasview_address_alias(rec));

}

/* Adds alias to aview, if old_alias == -1 adds new alias to top level
   may replace existing alias
   this do not check if alias is system alias ...
 */
struct aliasview_record * add_alias_to_aview(aview,buffer,old_alias)
     struct AliasView *aview;
     const struct alias_buffer *buffer;
     int old_alias;
{
    int cx      = 0;         /* Index 0 is now 'top level' */
    int old_idx = -1;

    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"add_alias_to_aview",
	      "bad aliasview magic",0);

    if (-1 != old_alias) {
	
	if ( old_alias < 0 ||  old_alias >= aview->view_len)
	    panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"add_alias_to_aview",
		  "bad old_alias",0);

	cx       = aview->view[old_alias].classnum;
	old_idx = aview->view[old_alias].index;
	
    }


    if (cx < 0 || cx >= aview->aliasclasscount)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"give_alias",
	      "add_alias_to_aview",0);

    return am_set_alias( aview->aliasclass[cx],old_idx,buffer);
}

/* makes alias to be NULL -- does not change index */
void delete_alias_from_aview(aview,index)
     struct AliasView *aview;
     int index;
{
    int cx;
    int cindex;

    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"delete_alias_from_aview",
	      "bad aliasview magic",0);

    if (index < 0 || index >= aview->view_len)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"delete_alias_from_aview",
	      "bad index",0);

    cx     = aview->view[index].classnum;
    cindex = aview->view[index].index;

    if (cx < 0 || cx >= aview->aliasclasscount)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"delete_alias_from_aview",
	      "bad class number",0);

    am_delete_alias( aview->aliasclass[cx],
		     cindex);
}



struct aliasview_record * give_alias(aview,index)
     struct AliasView *aview;
     int index;
{
    int cx;
    int cindex;

    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"give_alias",
	      "bad aliasview magic",0);
    
    if (index < 0 || index >= aview->view_len)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"give_alias",
	      "bad index",0);
    
    cx     = aview->view[index].classnum;
    cindex = aview->view[index].index;

    if (cx < 0 || cx >= aview->aliasclasscount)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"give_alias",
	      "bad class number",0);

    return am_give_alias( aview->aliasclass[cx],
			  cindex);
}

struct aliasview_record * lookup_alias(aview,key,index)
     struct AliasView *aview;
     const struct string *key;
     int *index;
{
    struct aliasview_record * result = NULL;

    int cx;
    int cindex;
    int i;

    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"lookup_alias",
	      "bad aliasview magic",0);
    
    *index = -1;

    for (cx = 0; cx < aview->aliasclasscount; cx++) {

	result = am_lookup_alias(aview->aliasclass[cx],
				 key,&cindex);

	if (result) {
	    DPRINT(Debug,9, (&Debug,
			     "lookup_alias: \"%S\" found\n",
			     key));

	    break;
	}

    }



    if (!result)
	return NULL;

    /* Calculate index */

    for (i = 0; i < aview->view_len; i++) {
	if (aview->view[i].classnum == cx &&
	    aview->view[i].index   == cindex) {
	    *index = i;

	    DPRINT(Debug,9, (&Debug,
			     "lookup_alias: ... index = %d\n",i));

	    break;
	}
    }


    return result;
}

void zero_alias_view(x)
     struct alias_view *x;
{
    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)x,sizeof (*x));
    
    x->classnum = 0;
    x->index    = 0;

}

void aliasview_give_index_number(aview,index,res)
     struct AliasView *aview;
     int index; 
     struct alias_view *res;
{
    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"aliasview_give_index_number",
	      "bad aliasview magic",0);
    
    if (index < 0 || index >= aview->view_len)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"aliasview_give_index_number",
	      "bad index",0);

    *res = aview->view[index];
}

int get_alias_count(aview)
     struct AliasView *aview;
{

    if (aview->normal_alias_len < 0) {
	DPRINT(Debug,9, (&Debug,
			 "get_alias_count=0: Rebuild is needed\n"));
	return 0;
    }

    DPRINT(Debug,9, (&Debug,
		     "get_alias_count=%d\n",
		     aview->normal_alias_len));

    return aview->normal_alias_len;
}


int get_total_alias_count(aview)
     struct AliasView *aview;
{
    DPRINT(Debug,9, (&Debug,
		     "get_total_alias_count=%d\n",
		     aview->view_len));

    return aview->view_len;
}

/*  current == index+1     value is used only by caller */
int get_alias_current(aview) 
     struct AliasView *aview;
{
    return aview->current;
}

void set_alias_current(aview,cur)
     struct AliasView *aview; 
     int cur;
{
    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"set_alias_current",
	      "bad aliasview magic",0);

    aview->current = cur;
}

int get_alias_selected(aview)
     struct AliasView *aview;
{
    return aview->selected;
}

void set_alias_selected(aview,sel)
     struct AliasView *aview; 
     int sel;
{
    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"set_alias_selected",
	      "bad aliasview magic",0);

    aview->selected = sel;
}


static void rebuild_aliasview P_((struct AliasView *aview));
static void rebuild_aliasview(aview)
     struct AliasView *aview;
{
    int len = 0;
    int i;
    int dup = 0;
    int x;

    reset_aliasview(aview);

    DPRINT(Debug,7, (&Debug,
		     "rebuild_aliasview: %d alias classes\n",
		     aview->aliasclasscount));

    for (i = 0; i < aview->aliasclasscount; i++) {
	len += am_count_aliases(aview->aliasclass[i]);
    }

    DPRINT(Debug,4, (&Debug,
		     "rebuild_aliasview: len=%d\n",len));

    if (len < 1)
	return;

    aview->view = safe_calloc(len, sizeof (aview->view[0]));

    /* reset duplicate indicators .... */
    for (i = 0; i < aview->aliasclasscount; i++) {
	int y;

	int count = am_count_aliases(aview->aliasclass[i]);
	
	for (y = 0; y < count; y++) {
	    struct aliasview_record *a;

	    a = am_give_alias(aview->aliasclass[i],y);

	    if (a) {
		if (ison(a->type,DUPLICATE)) {
		    
		    DPRINT(Debug,14, (&Debug,
				      "rebuild_aliasview: Initially clearing [%d/%d] duplicate mark\n",
				      i,y));
		    clearit(a->type,DUPLICATE);
		}

	    }
	}
	
    }


    /* build new view */

    x = 0;
    
    for (i = 0; i < aview->aliasclasscount && x < len; i++) {
	int y;

	int count = am_count_aliases(aview->aliasclass[i]);
	
	for (y = 0; y < count && x < len; y++) {
	    struct aliasview_record *a;

	    zero_alias_view(& aview->view[x]);

	    aview->view[x].classnum = i;
	    aview->view[x].index    = y;

	    a = am_give_alias(aview->aliasclass[i],y);

	    if (a) {
		int inner;

		if (ison(a->type,DUPLICATE)) {
		     DPRINT(Debug,14, (&Debug,
				       "rebuild_aliasview:  [%d/%d] \"%S\" is duplicate , not added as %d\n",
				       i,y,a->alias_key,x));
		     continue;
		}

		/* Check for duplicates ... */

       		if (ALIASVIEW_record_magic != a->magic)
		    panic("ALIAS VIEW PANIC",__FILE__,__LINE__," rebuild_aliasview",
			  "bad magic",0);
		
		for (inner = i+1; inner < aview->aliasclasscount; inner++) {
		    int idx1;
		    

		    struct aliasview_record *b  = am_lookup_alias(aview->aliasclass[inner],
								  a->alias_key,
								  &idx1);
		    
		    
		    if (b && 
			isoff(b->type,DUPLICATE)) {
			
			setit(b->type,DUPLICATE);
			
			DPRINT(Debug,14, (&Debug,
					  "rebuild_aliasview: Marking [%d/%d] \"%S\" as duplicate)\n",
					  inner,idx1,
					  a->alias_key));
			dup++;
		    }								 		    
		}

		DPRINT(Debug,14, (&Debug,
				  "rebuild_aliasview:  [%d/%d] \"%S\" => as %d\n",
				  i,y,a->alias_key,x));

		x++;

	    }
	}
    }

    DPRINT(Debug,4, (&Debug,
		     "rebuild_aliasview: %d normal aliases\n",x));
    aview->normal_alias_len = x;

    if (dup) {
	/* Add duplicates */

	for (i = 0; i < aview->aliasclasscount && x < len; i++) {
	    int y;
	    
	    int count = am_count_aliases(aview->aliasclass[i]);
	    
	    for (y = 0; y < count && x < len; y++) {

		struct aliasview_record *a = 
		    am_give_alias(aview->aliasclass[i],y);

		if (a) {
		    if (ison(a->type,DUPLICATE)) {
			
			zero_alias_view(& aview->view[x]);
			
			aview->view[x].classnum = i;
			aview->view[x].index    = y;
			x++;
		    }
		}
	    }
	}
    }


    aview->view_len = x;
    
    DPRINT(Debug,4, (&Debug,
		     "rebuild_aliasview: len=%d duplicates=%d\n",
		     x,dup));

}

static int update_expunge_mark P_((struct AliasView *aview));
static int update_expunge_mark(aview)
     struct AliasView *aview;
{
    int i;
    int r = 0;
    
    DPRINT(Debug,14, (&Debug,
		     "update_expunge_mark: %d alias classes\n",
		     aview->aliasclasscount));

    for (i = 0; i < aview->aliasclasscount; i++) {
	int y;

	int count = am_count_aliases(aview->aliasclass[i]);
	
	for (y = 0; y < count; y++) {

	    struct aliasview_record *a;
	    
	    a = am_give_alias(aview->aliasclass[i],y);
	    
	    if (a) {
		int expunge;

		expunge = am_deleted_alias(aview->aliasclass[i],y);

		if (expunge && isoff(a->type,EXPUNGED)) {
		    DPRINT(Debug,14, (&Debug,
				      "update_expunge_mark: Setting [%d/%d] expunged mark\n",
				      i,y));
		    setit(a->type,EXPUNGED);
		    r = 1;
		} else if (!expunge && ison(a->type,EXPUNGED)) {
		    DPRINT(Debug,14, (&Debug,
				      "update_expunge_mark: Clearing [%d/%d] expunged mark\n",
				      i,y));
		    clearit(a->type,EXPUNGED);
		    r = 1;
		}
	    }
	}
    }

    DPRINT(Debug,14, (&Debug,
		      "update_expunge_mark=%d\n",r));
    
    return r;
}


/* Return 1 if changes */
int update_aview(aview)
     struct AliasView *aview;
{
    int r = 0;
    int count = 0;

    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"update_aview",
	      "bad aliasview magic",0);
       
    if (aview->aliasclass) {
	int i;

	for (i = 0; i < aview->aliasclasscount; i++) {
		
	    if (am_update_aliases(aview->aliasclass[i])) {
		
		r = 1;		
	    }

	    count += am_count_aliases(aview->aliasclass[i]);
	}

	DPRINT(Debug,9, (&Debug,
			 "update_aview: count=%d\n",count));       
    }

    if (count != aview->view_len) {

	DPRINT(Debug,9, (&Debug,
			 "update_aview: count = %d != view_len = %d\n",
			 count,aview->view_len));

	r = 1;
    }

    if (aview->normal_alias_len < 0) {
	DPRINT(Debug,9, (&Debug,"update_aview: normal_alias_len was reset\n"));
	r = 1;
    }

    
    if (r) 
	rebuild_aliasview(aview);

    if (update_expunge_mark(aview)) {
	DPRINT(Debug,9, (&Debug,"update_aview: EXPUNGED flag was changed\n"));

	r = 1;
    }

    
    DPRINT(Debug,4, (&Debug, "update_aview=%d\n",r));

    return r;
}


struct sort_data2 {
    struct aliasclass *c;
    struct alias_view   w;
};

struct aliasview_record *  give_alias_s(s)
     struct sort_data2 *s;
{
    return am_give_alias(s->c, s->w.index);
}

void aliasview_give_index_number_s(s,res)
     struct sort_data2 *s;
     struct alias_view *res;
{
    *res = s->w;
}


void sort_aview (aview,func)
     struct AliasView     *aview;
     alias_compare_func   *func;
{
    int i;
    struct sort_data2 * array;

    /* Little dirty ... */
    typedef int (*compar) P_((const void *, const void *));
    compar X = (compar) func;

    int len = aview->view_len;
    
    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"sort_aview",
	      "bad aliasview magic",0);

    if (len < 2)
	return;

    array = safe_calloc(len, sizeof (array[0]));

    for (i = 0; i < len; i++) {
	int cx     = aview->view[i].classnum;

	if (cx < 0 || cx >= aview->aliasclasscount)
	    panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"sort_aview",
		  "bad class number",0);

	
	array[i].c           = aview->aliasclass[cx];
	array[i].w           = aview->view[i];	
    }

    qsort(array,len,sizeof (array[0]), X);
   

    for (i = 0; i < len; i++) {
	aview->view[i] = array[i].w;
    }

    free(array);
}


void add_aview_to_alias_stack(stack,aview)
     struct alias_stack * stack;
     struct AliasView *aview;
{
    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"add_aview_to_alias_stack",
	      "bad aliasview magic",0);
  

    if (aview->aliasclass) {
	int i;

	/* 0 is now top level */

	for (i = 0; i < aview->aliasclasscount; i++) {

	    am_add_to_alias_stack(stack,aview->aliasclass[i]);
	}

    }
}

int file_changed_aliasview(aview)
     struct AliasView *aview;
{
    int r = 0;

    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"write_aliasview",
	      "bad aliasview magic",0);

    if (aview->aliasclass) {
	int i;

	for (i = 0; i < aview->aliasclasscount; i++) {
	    		
	    if (am_file_changed(aview->aliasclass[i])) {
		DPRINT(Debug,14, (&Debug,
				  "file_changed_aliasview: class #%d file changed (or need_write set)\n",
				  i));
		r = 1;
	    }
	}	
    }
    
    DPRINT(Debug,14, (&Debug,
		      "file_changed_aliasview=%d\n",r));

    return r;
}


int write_aliasview(aview)
     struct AliasView *aview;
{

    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"write_aliasview",
	      "bad aliasview magic",0);

    if (aview->aliasclass) {
	int r = 1;
	int i;

	/* 0 is now top level */

	for (i = 0; i < aview->aliasclasscount; i++) {
	    int rebuild = 0;

	    if (! am_write_changed(aview->aliasclass[i],
				   &rebuild))
		r = 0;

	    
	    if (rebuild) {
		aview->normal_alias_len = -1;

		DPRINT(Debug,9, (&Debug,"write_aliasview: Marking for rebuild ....\n"));
	    }	       
	}

	DPRINT(Debug,9, (&Debug,"write_aliasview=%d\n",r));

	return r;
    }


    DPRINT(Debug,9, (&Debug,"write_aliasview=0\n"));

    return 0;
}

int start_edit_aliases(aview,edited_file)
     struct AliasView *aview;
     const char **edited_file;
{
    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"start_edit_aliases",
	      "bad aliasview magic",0);

    *edited_file = NULL;

    if (aview->aliasclass) {
	int i;

	for (i = 0; i < aview->aliasclasscount; i++) {
	    const char * res = am_is_user_editable(aview->aliasclass[i]);

	    if (res) {
		int rebuild = 0;

		if (! am_write_changed(aview->aliasclass[i], &rebuild))
		    goto failure;

		*edited_file = res;

		if (rebuild) {
		    aview->normal_alias_len = -1;
		    
		    DPRINT(Debug,9, (&Debug,
				     "start_edit_aliases: Marking for rebuild ....\n"));
		}

		return 1;
	    }

	}
    }
    

    failure:
	return 0;
}

void end_edit_aliases(aview)
     struct AliasView *aview;
{
    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"end_edit_aliases",
	      "bad aliasview magic",0);


    if (aview->aliasclass) {
	int i;

	for (i = 0; i < aview->aliasclasscount; i++) {
	    const char * res = am_is_user_editable(aview->aliasclass[i]);
	    
	    if (res) {
		if (am_reload_aliases(aview->aliasclass[i])) {
		    aview->normal_alias_len = -1;
		    
		    DPRINT(Debug,9, (&Debug,
				     "end_edit_aliases: Marking for rebuild ....\n"));
		

		}
	    }
	}
    }


}


/* Add standard alias databases to aview */
void add_files_to_aview(aview)
     struct AliasView *aview;
{
    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"add_files_to_aview",
	      "bad aliasview magic",0);

    /* User map */
    am_set_standard_aliasmap(add_aliasclass(aview),1);


    if (old_user_text_file[0] &&
	0 == access(old_user_text_file,READ_ACCESS)) {

	struct aliases_map * compat_aliases_user = NULL;

	DPRINT(Debug,7,(&Debug,
			"add_files_to_aview: Old format aliases file exists: %s\n",
			old_user_text_file));

	
	if (compat_aliases_user)
	    panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"add_files_to_aview",
		  "Old user aliases already readed",0);
 
	compat_aliases_user = load_old_aliases_map(old_user_text_file);
			
	if (compat_aliases_user)
	    am_set_compat_aliasmap(add_aliasclass(aview),
				   compat_aliases_user,
				   compat_user);

	/* am_set_compat_aliasmap moves ownership to alias_class */
    }

    /* system map */
    am_set_standard_aliasmap(add_aliasclass(aview),0);

    if (old_system_text_file[0] &&
	0 == access(old_system_text_file,READ_ACCESS)) {

	struct aliases_map * compat_aliases_system = NULL;

	DPRINT(Debug,7,(&Debug,
			"add_files_to_aview: Old format aliases file exists: %s\n",
			old_system_text_file));

	compat_aliases_system = load_old_aliases_map(old_system_text_file);

	if (compat_aliases_system)
	    am_set_compat_aliasmap(add_aliasclass(aview),
				   compat_aliases_system,
				   compat_system);     
    }

}

#define ALIASES_find_magic      0xF60d

struct aliases_find_state {
    unsigned short magic;         /* ALIASES_find_magic */

    struct AliasView *aview;
    int idx;

    struct aliasclass *target;
    struct aliasclass *source;

    struct aliases_map  * source_map;

    const char *  target_filename;
    char *  source_filename;

} * aliases_find_init(aview)
     struct AliasView *aview;
{
    struct aliases_find_state * ret = safe_malloc(sizeof (*ret));
    
    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)ret,sizeof (*ret));

    if (aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"aliases_find_init",
	      "bad aliasview magic",0);



    ret->magic   = ALIASES_find_magic;
    ret->aview   = aview;
    ret->idx = -1;

    ret->target  = NULL;
    ret->source  = NULL;

    ret->source_map = NULL;

    ret->target_filename = NULL;
    ret->source_filename = NULL;

    return ret;
}

void aliases_find_free(state)
     struct aliases_find_state **state;
{
    
    if (ALIASES_find_magic != (*state)->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"aliases_find_free",
	      "bad state magic",0);
	
    (*state)->magic = 0;    /* Invalidate */
    free(*state);
    *state = NULL;   
}

int aliases_find_compat(state,from_filename,to_filename)
     struct aliases_find_state *state;
     const char **from_filename; 
     const char **to_filename;
{
    if (ALIASES_find_magic != state->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"aliases_find_compat",
	      "bad state magic",0);

    *from_filename = NULL;
    *to_filename   = NULL;

    state->source_map = NULL;

    state->target_filename = NULL;
    state->source_filename = NULL;

    if (state->aview->magic != ALIASVIEW_magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"aliases_find_compat",
	      "bad aliasview magic",0);


    if (state->aview->aliasclass) {
	int i;

	/* This code assumes that target is before  compat map */
		   
	for (i = state->idx+1; i < state->aview->aliasclasscount; i++) {
	    
	    if (am_is_normal_target(state->aview->aliasclass[i]))
		state->target = state->aview->aliasclass[i];

	    if (am_is_compat_map(state->aview->aliasclass[i])) {

		state->idx    = i;
		state->source = state->aview->aliasclass[i];

		if (! give_aliases_filename(state->source,
					    & state->source_map,
					    & state->source_filename,
					    NULL,NULL,NULL,NULL)) {
		    DPRINT(Debug,1,(&Debug,
				    "aliases_find_compat: %d: source filename is not known\n",
				    i));
		    continue;		    
		}

		*from_filename = state->source_filename;
		
		if (! state->source_map) {
		    DPRINT(Debug,1,(&Debug,
				    "aliases_find_compat: %d: source map is not known\n",
				    i));
		    continue;
		}


		if (! state->target) {
		    DPRINT(Debug,1,(&Debug,
				    "aliases_find_compat: %d: No target seen\n",
				    i));

		    
		    *to_filename           = NULL;
		    state->target_filename = NULL;

		    return 1;
		}


		state->target_filename = am_is_user_editable(state->target);

		if (! state->target_filename) {
		    DPRINT(Debug,1,(&Debug,
				    "aliases_find_compat: %d: Target is not user editable\n",
				    i));
		}

		*to_filename   = state->target_filename;
		    		
		return 1;
	    }
	}
    }

    return 0;
}

void aliases_find_remove(state)
     struct aliases_find_state *state;
{
    int i;

    if (ALIASES_find_magic != state->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"aliases_find_remove",
	      "bad state magic",0);

     if (state->aview->magic != ALIASVIEW_magic)
	 panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"aliases_find_remove",
	       "bad aliasview magic",0);
  
     if (state->idx < 0 || state->idx >= state->aview->aliasclasscount)
	 panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"aliases_find_remove",
	       "bad index",0);

     am_free_aliasclass(& state->aview->aliasclass[state->idx]);

     DPRINT(Debug,5,(&Debug,
		     "aliases_find_remove: %d: Removed\n",
		     state->idx));

     for (i = state->idx; i < state->aview->aliasclasscount-1; i++) {
	 state->aview->aliasclass[i]   = state->aview->aliasclass[i+1];
	 state->aview->aliasclass[i+1] = NULL;
     }
     state->aview->aliasclasscount = i;
	 
     state->idx--;
     state->source_map = NULL;
     
     state->target_filename = NULL;
     state->source_filename = NULL;
}


void aliases_find_convert(state)
     struct aliases_find_state *state;
{
    int i, count;
    int rebuild = 0;

    if (ALIASES_find_magic != state->magic)
	panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"aliases_find_convert",
	      "bad state magic",0);

     if (state->aview->magic != ALIASVIEW_magic)
	 panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"aliases_find_convert",
	       "bad aliasview magic",0);
 
     count = am_count_aliases(state->source);

     /* am_delete_alias() is ignored */

     for (i = 0; i < count; i++) {
	 struct aliasview_record *a = 
	     am_give_alias(state->source,i);

	 am_add_alias(state->target,a);
     }

     if (state->aview->magic != ALIASVIEW_magic)
	 panic("ALIAS VIEW PANIC",__FILE__,__LINE__,"aliases_find_convert",
	       "bad aliasview magic",0);
     

     if (count > 0) {
	 state->aview->normal_alias_len = -1;
	 DPRINT(Debug,9, (&Debug,"aliases_find_convert: Marking for rebuild ....\n"));
     }

     if (am_write_changed(state->target, &rebuild)) {
	 char *tmp = elm_message(FRM("%s.OLD"),
				 state->source_filename);

	 if (rebuild) {
	     state->aview->normal_alias_len = -1;
	     DPRINT(Debug,9, (&Debug,"aliases_find_convert: Marking for rebuild ....\n"));
	 }


	 if (0 == rename(state->source_filename,tmp)) {

	     DPRINT(Debug,7,(&Debug,
			     "aliases_find_convert: %s renamed to %s\n",
			     state->source_filename,tmp));
	 }

	 if (old_system_text_file[0] && old_system_data_file[0] &&
	     0 == strcmp(state->source_filename,old_system_text_file))
	     delete_old_alias_files(old_system_data_file);

	 if (old_user_text_file[0] && old_user_data_file[0] &&
	     0 == strcmp(state->source_filename,old_user_text_file))
	     delete_old_alias_files(old_user_data_file);
	 	    
	 aliases_find_remove(state);
	 free(tmp);
     }
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
