static char rcsid[] = "@(#)$Id: alias_itemized.c,v 2.6 2014/07/14 15:07:29 hurtta Exp $";


/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.6 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/


#include "def_elm.h"
#include "s_aliases.h"

DEBUG_VAR(Debug,__FILE__,"alias");

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str)
     char *str;
{
    return (unsigned char *)str;
}


/* ------------------------------------------------------------------------- */

#define DISPLAY_addrlist_magic  0xF60c

struct display_list_addrs_state {
    unsigned short magic;           /* DISPLAY_addrlist_magic */

    const struct addr_list          * address_list;

    int idx;
    int current_group;
    int delim_idx;
};

struct display_list_addrs_state * display_list_addrs_iterator_alloc(address_list)
     const struct addr_list          * address_list;
{
    struct display_list_addrs_state * ret = safe_malloc(sizeof(*ret));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)ret,sizeof (*ret));
    
    ret->magic         = DISPLAY_addrlist_magic;

    ret->address_list  = address_list;
    ret->idx           = 0;
    ret->current_group = -1;
    ret->delim_idx     = 0;
    
    return ret;
}


void display_list_addrs_iterator_free(state)
    struct display_list_addrs_state **state;
{
    if (DISPLAY_addrlist_magic != (*state)->magic)
	panic("DISPLAY PANIC",__FILE__,__LINE__,"display_list_addrs_iterator_free",
	      "Bad magic number",0);

    (*state)->magic = 0;    /* Invalidate */

    free(*state);
    *state = NULL;
}


struct string * display_list_addrs_iterator(state)
   struct display_list_addrs_state * state;
{
    struct string * result = NULL;
    int addr_item_count = 0;
    
    if (DISPLAY_addrlist_magic != state->magic)
	panic("DISPLAY PANIC",__FILE__,__LINE__,"display_list_addrs_iterator",
	      "Bad magic number",0);

    addr_item_count = addr_list_item_count(state->address_list);


    /* Increments state->idx if address is consumed,
       returns NULL when all is processed

       state->idx may be > list length 
       
    */


    DPRINT(Debug,11, (&Debug, 
		      "display_list_addrs_iterator: idx=%d current_group=%d delim_idx=%d\n",
		      state->idx,state->current_group,state->delim_idx));

    if (state->idx < addr_item_count) {

	int group = -1;
	const struct address * address = 
	    addr_list_get_item(state->address_list,state->idx,&group);


	if (state->current_group != group &&
	    state->current_group != -1) {
	    result = new_string(display_charset);
	    
	    add_ascii_to_string(result,s2us(";"));
	    state->current_group = -1;
	    
	    goto READY;
	}


	if (state->delim_idx < state->idx) {
	    result = new_string(display_charset);

	    add_ascii_to_string(result,s2us(", "));
	    state->delim_idx++;

	    goto READY;
	}

	if (group >= 0 &&
	    state->current_group == -1) {

	    const struct string * groupname1 =
		addr_list_get_group(state->address_list,group);


	    result = new_string(display_charset);
	    
	    display_add_phrase(&result,groupname1);

	    add_ascii_to_string(result,s2us(": "));
	    state->current_group = group;	   

	    goto READY;
	}


	result = new_string(display_charset);

	display_add_one_address(&result,address);
	state->idx++;

	goto READY;

    } 


    /* End of list */
    if (state->current_group != -1) {
	result = new_string(display_charset);
	
	add_ascii_to_string(result,s2us(";"));
	state->current_group = -1;
	
	goto READY;
    }
    

    /* End of LIST */
    DPRINT(Debug,11, (&Debug, "display_list_addrs_iterator: End of list\n"));

 READY:
    
    if (result) {
	DPRINT(Debug,11, (&Debug, "display_list_addrs_iterator=%S\n",
			  result));
    }

   return result;
}


/* ------------------------------------------------------------------------- */

#define DISPLAY_aliasvector_magic  0xF60E

struct display_list_aliases_state {
    unsigned short magic;           /* DISPLAY_aliasvector_magic */

    const struct alias_vector      * alias_vector;

    int idx;
    int delim_idx;
};


struct display_list_aliases_state * display_list_aliases_iterator_alloc(alias_vector)
     const struct alias_vector     * alias_vector;
{
    struct  display_list_aliases_state * ret = safe_malloc(sizeof(*ret));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)ret,sizeof (*ret));
    
    ret->magic         = DISPLAY_aliasvector_magic;

    ret->alias_vector  = alias_vector;
    ret->idx           = 0;
    ret->delim_idx     = 0;
    
    return ret;
}

void display_list_aliases_iterator_free(state)
    struct display_list_aliases_state **state;
{
    if (DISPLAY_aliasvector_magic != (*state)->magic)
	panic("DISPLAY PANIC",__FILE__,__LINE__,
	      "display_list_aliases_iterator_free",
	      "Bad magic number",0);

    (*state)->magic = 0;    /* Invalidate */

    free(*state);
    *state = NULL;
}

struct string * display_list_aliases_iterator(state)
   struct display_list_aliases_state * state;
{
    struct string * result = NULL;
    int alias_item_count = 0;

    if (DISPLAY_aliasvector_magic != state->magic)
	panic("DISPLAY PANIC",__FILE__,__LINE__,
	      "display_list_aliases_iterator",
	      "Bad magic number",0);

    
    alias_item_count = alias_vector_item_count(state->alias_vector);

    /* Increments state->idx if alias is consumed,
       returns NULL when all is processed
       
       state->idx may be > list length 
       
    */

    DPRINT(Debug,11, (&Debug, 
		      "display_list_aliases_iterator: idx=%d delim_idx=%d\n",
		      state->idx,state->delim_idx));

    
    if (state->idx < alias_item_count) {

	const struct string * alias = 
	    alias_vector_get_alias(state->alias_vector,
				   state->idx);

	if (state->delim_idx < state->idx) {
	    result = new_string(display_charset);
	    
	    add_ascii_to_string(result,s2us(", "));
	    state->delim_idx++;
	    
	    goto READY;
	}
	
	/* We do not quote displayed (only) alias anyway */

	result = dup_string(alias);
	state->idx++;

	goto READY;

    } 
	
    /* End of list */

    DPRINT(Debug,11, (&Debug, "display_list_aliases_iterator: End of list\n"));
    
 READY:
    
    if (result) {
	DPRINT(Debug,11, (&Debug, "display_list_aliases_iterator=%S\n",
			  result));
    }
    
    return result;
}

/* ------------------------------------------------------------------------ */

#define DISPLAY_genlist_magic   0xF60F

struct display_list_state {

    unsigned short magic;           /* DISPLAY_genlist_magic */

    struct display_list_type   * list_type;

    union {
	struct display_list_addrs_state    * addrs;
	struct display_list_aliases_state  * aliases;
	void               * dummy;

    } u;

};

typedef void display_list_free_f P_((struct display_list_state *state));
typedef struct string * display_list_iterator_f P_((struct display_list_state *state));

#define DISPLAY_genlist_type_magic  0xF610

struct display_list_type {
    unsigned short magic;          /* DISPLAY_genlist_type_magic */

    display_list_free_f      * free_state;
    display_list_iterator_f  * iterator;
};



static struct display_list_state * alloc_display_list_state P_((
    struct display_list_type *list_type));
static struct display_list_state * alloc_display_list_state(list_type)
     struct display_list_type *list_type;
{
    struct display_list_state * ret = safe_malloc(sizeof(*ret));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)ret,sizeof (*ret));
    
    if (DISPLAY_genlist_type_magic != list_type->magic)
	 panic("DISPLAY PANIC",__FILE__,__LINE__,"alloc_display_list_state",
	       "Bad type magic number",0);
	 
	      
    ret->magic     = DISPLAY_addrlist_magic;
    ret->list_type = list_type;
    ret->u.dummy   = NULL;

    return ret;
}

void display_list_iterator_free(state)
     struct display_list_state **state;
{   
    if (DISPLAY_addrlist_magic != (*state)->magic)
	panic("DISPLAY PANIC",__FILE__,__LINE__,
	      "display_list_iterator_free",
	      "Bad magic number",0);
    
    if (DISPLAY_genlist_type_magic != (*state)->list_type->magic)
	panic("DISPLAY PANIC",__FILE__,__LINE__,
	      "display_list_iterator_free",
	      "Bad type magic number",0);

    (*state)->list_type->free_state(*state);

    if ((*state)->u.dummy)
	panic("DISPLAY PANIC",__FILE__,__LINE__,
	      "display_list_iterator_free",
	      "State not free'ed",0);

    (*state)->magic = 0;    /* Invalidate */
    
    free(*state);
    *state = NULL; 
}


struct string * display_list_iterator(state)
     struct display_list_state * state;
{
    struct string * item = NULL;

     if (DISPLAY_addrlist_magic != state->magic)
	panic("DISPLAY PANIC",__FILE__,__LINE__,
	      "display_list_iterator",
	      "Bad magic number",0);
    
     if (DISPLAY_genlist_type_magic != state->list_type->magic)
	 panic("DISPLAY PANIC",__FILE__,__LINE__,
	       "display_list_iterator",
	       "Bad type magic number",0);
     
     item = state->list_type->iterator(state);
     
     if (item) {
	 DPRINT(Debug,11, (&Debug, "display_list_iterator=%S\n",
			   item));
     }


     return item;
}

/* ----------------------------------------------------------------------- */

int genlist_display_list(page,prefix,line,reserve,state)
     struct menu_context * page;
     const struct string * prefix;
     int *line; 
     int reserve;
     struct display_list_state *state;
{
    int ret = 0;

    int LINE = *line;
    struct string * item = display_list_iterator(state);
    int LINES, COLUMNS;

    menu_get_sizes(page, &LINES, &COLUMNS);

     if (item) {
	 /* Address list with zero addresses is possible */
	 int col, basecol;
		
	 menu_PutLineX(page,LINE,0,FRM("%S"),prefix);
   
	 menu_GetXYLocation(page,&LINE,&col);
	 basecol = col;

	 do {
	     
	     int len = COLUMNS - col;
	     int l;
	     int X;
	     struct string *news  = NULL;
	     int visible = 0;
	     int last_line = 0;
	     
	     if (len < 1) {

	     REFLOW:
		 
		 DPRINT(Debug,5, (&Debug, "genlist_display_list: Next line: %S\n",
				  item));
		 
		 LINE++;
		 col = basecol;
		 
		 len = COLUMNS - col;
						
		 if (LINE >= LINES-reserve || len < 1) {

		     DPRINT(Debug,5, (&Debug, "genlist_display_list: Fails: %S\n",
				      item));
		     
		     free_string(&item);
		     break;
		 }
	     }
	     
	     /* check for overflow */
		    
	     l = string_len(item);
	     X = 0;

	     last_line = LINE >= LINES-reserve-1;
	     
	     news = curses_printable_clip(item,&X,l,&visible,
					  len - (last_line ? 3 : 1));
	     if (X < l) {  /* Overflow */
		 
		 DPRINT(Debug,5, (&Debug, "genlist_display_list: Wrap/Overflow: %S => %S (visible=%d)\n",
				  item, news,visible));
		 
		 if (col > basecol && !last_line) {
		     free_string(&news);			    
		     goto REFLOW;
		 }
		 
		 menu_PutLineX(page,LINE,col,FRM("%S"),news);
		 
		 free_string(&news);	

		 if (! last_line) {

		     menu_StartXX(page,pg_BOLD);
		     menu_Write_to_screen(page,FRM("\\"));
		     menu_EndXX(page,pg_BOLD);
		 
		     LINE++;
		     col = basecol;

		     while (X < l && LINE < LINES-reserve) {

			 last_line = LINE >= LINES-reserve-1;
	     
			 news = curses_printable_clip(item,&X,l,&visible,
						      len - (last_line ? 3 : 1));

			 DPRINT(Debug,5, (&Debug, "genlist_display_list: Wrap ... => %S (visible=%d)\n",
					  news,visible));

			 
			 menu_PutLineX(page,LINE,col,FRM("%S"),news);
			 
			 free_string(&news);
			 

			 if (X < l) {

			     if (last_line)
				 goto was_last_line;

			     menu_StartXX(page,pg_BOLD);
			     menu_Write_to_screen(page,FRM("\\"));
			     menu_EndXX(page,pg_BOLD);
			     
			     LINE++;
			     col = basecol;

			 } else
			     col += visible;

		     }

		     goto NEXT;

		 } else {

		 was_last_line:

		     menu_StartXX(page,pg_BOLD);
		     menu_Write_to_screen(page,FRM("..."));
		     menu_EndXX(page,pg_BOLD);

		     DPRINT(Debug,5, (&Debug, "genlist_display_list: last line: %S\n",
				      item));
		     
		     free_string(&item);
		     break;

		 }
	     }
	     
	     free_string(&news);
		    
	     menu_PutLineX(page,LINE,col,FRM("%S"),item);
	     
	     col += visible;
	     
	     
	 NEXT:
	     free_string(&item);

	     item = display_list_iterator(state);
	 } while (item);

	 ret = 1;
     } 

     *line = LINE;
     return ret;
}


struct string * genlist_display_list_string(state,len)
     struct display_list_state *state;
     int len;
{    
    struct string * ret = NULL;
    struct string * item = NULL;
    
    while (NULL != (item = display_list_iterator(state))) {
	
	int l;
	int X;
	struct string *tmp   = NULL;
	struct string *news  = NULL;
	int visible = 0;
	
	if (!ret) {
	    tmp = item;
	    item = NULL;
	} else {
	    tmp = cat_strings(ret,item,1);
	    free_string(&item);
	}
	
	/* check for overflow */
	
	l = string_len(tmp);
	X = 0;
	
	news = curses_printable_clip(tmp,&X,l,&visible,len-5);
	if (X < l) {  /* Overflow */
	    
	    DPRINT(Debug,5, (&Debug, "genlist_display_list_string: Overflow: %S\n",
			     tmp));
	    
	    free_string(&tmp);
	    
	    free_string(&ret);
	    ret = news;
	    add_ascii_to_string(ret,s2us("..."));
	    
	    DPRINT(Debug,5, (&Debug, "     ... returning: %S\n",
			     ret));

	    break;
	}

	free_string(&news);

	if (ret)
	    free_string(&ret);
	ret = tmp;

    }

    if (ret) {
	DPRINT(Debug,10, (&Debug, "genlist_display_list_string=%S\n",
			 ret));
    }


    return ret;
}
			      

/* ----------------------------------------------------------------------- */

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

S_(display_list_free_f genlist_addrlist_free)
static void genlist_addrlist_free P_((struct display_list_state *state));

static void genlist_addrlist_free(state)
     struct display_list_state *state;
{
    display_list_addrs_iterator_free(& state->u.addrs);
}

S_(display_list_iterator_f genlist_addrlist_iterator)
static struct string * genlist_addrlist_iterator 
   P_((struct display_list_state *state));

static struct string * genlist_addrlist_iterator(state)
     struct display_list_state *state;
{
    return display_list_addrs_iterator(state->u.addrs);
}

static struct display_list_type addrlist_list_type = {
    DISPLAY_genlist_type_magic,
    genlist_addrlist_free,
    genlist_addrlist_iterator
};

struct display_list_state * display_list_iterator_alloc_addrs(
               address_list)
     const struct addr_list          * address_list;
{
    struct display_list_state * ret = 
	alloc_display_list_state(& addrlist_list_type);
   
    ret->u.addrs = display_list_addrs_iterator_alloc(address_list);

    return ret;
}

/* ---------------------------------------------------------------------- */


S_(display_list_free_f genlist_aliases_free)
static void genlist_aliases_free P_((struct display_list_state *state));

static void genlist_aliases_free(state)
     struct display_list_state *state;
{
    display_list_aliases_iterator_free(& state->u.aliases);
}

S_(display_list_iterator_f genlist_aliases_iterator)
static struct string * genlist_aliases_iterator 
   P_((struct display_list_state *state));

static struct string * genlist_aliases_iterator(state)
     struct display_list_state *state;
{
    return display_list_aliases_iterator(state->u.aliases);
}

static struct display_list_type aliases_list_type = {
    DISPLAY_genlist_type_magic,
    genlist_aliases_free,
    genlist_aliases_iterator
};

struct display_list_state * display_list_iterator_alloc_aliases(
               alias_vector)
     const struct alias_vector      * alias_vector;
{
    struct display_list_state * ret = 
	alloc_display_list_state(& aliases_list_type);
   
    ret->u.aliases = display_list_aliases_iterator_alloc(alias_vector);

    return ret;
}

    


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

