static char rcsid[] = "@(#)$Id: alias.c,v 2.25 2022/07/14 14:16:01 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.25 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/alias.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************
 *  Incorparated Elm 2.5 code from src/alias.c. 
 *  That code was following copyright:
 *
 *  The Elm Mail System
 *
 *                      Copyright (c) 1988-1995 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** This file contains alias stuff

**/

#include "def_elm.h"
#include "s_elm.h"
#include "s_aliases.h"

DEBUG_VAR(Debug,__FILE__,"alias");

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static int resync_aliases P_((int newaliases,struct AliasView *aview,
			      struct menu_context  * page,
			      struct screen_parts *LOC));
static int resync_aliases(newaliases,aview,page,LOC)
     int                   newaliases;
     struct AliasView    * aview;
     struct menu_context * page;
     struct screen_parts * LOC;
{
    int r = delete_aliases(newaliases,TRUE, aview, page, LOC);
    
    DPRINT(Debug,14,(&Debug, "resync_aliases: delete_aliases gives %d\n",r));

    if (r < 0)
	goto fail;
    
    if (0 == r) {

	r = file_changed_aliasview(aview);

	DPRINT(Debug,14,(&Debug, "resync_aliases: file_changed_aliasview gives %d\n",r));
	    	    
    }
    
 fail:
    DPRINT(Debug,14,(&Debug, "resync_aliases=%d%s\n",
		     r,
		     r > 0 ? " (need resync)" : ""));
    
    return r;
}


  


static void install_aliases P_((struct AliasView *aview,
				struct screen_parts *LOC));

static int add_alias P_((int to_replace, 
			 struct AliasView *aview, 
			 struct menu_context  *page,
			 struct screen_parts *LOC));	

static int ask_accept P_((struct alias_buffer * buffer,
			  int replacement, 
			  struct AliasView *avies, 
			  struct menu_context  *page,
			  struct screen_parts *LOC));  

extern int  is_system;		/* system file updating?     */

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif



char *a_rev_alias_pad, *a_rev_alias_abr, *a_rev_alias_name,
	*a_rev_full_pad, *a_full_abr, *a_rev_full_name,
	*a_rev_text_pad, *a_text_abr, *a_rev_text_file,
	*a_alias_pad, *a_alias_abr, *a_alias_name,
	*a_full_pad, *a_full_name,
	*a_text_pad, *a_text_file,
    *a_system_flag;
	

static void set_aview_alias_mode P_((struct AliasView *aview,
				     struct screen_parts *LOC));
static void set_aview_alias_mode(aview, LOC)
     struct AliasView *aview;
     struct screen_parts *LOC;
{

    if (update_aview(aview)) {
	struct menu_common MENU;

	int tac = get_total_alias_count(aview);
	
	set_alias_current(aview,0);
	
	if (tac) {
	    sort_aliases(tac, sa_visible_sort,aview);
	    set_alias_current(aview,1);
	}

	set_mcommon_from_aliasview(&MENU, aview);
	copy_current(&MENU,LOC->header_page);
	
	get_page(&MENU, LOC->header_page);
    }
    
}


/* 0  == no not superceed
   1  == superceed
   -1 == cancel */

static int superceed_system P_((struct aliasview_record *a,
				struct AliasView *aview,
				struct menu_context *page,
				struct menu_context *prompt_area));
static int superceed_system(a, aview,page,prompt_area)
     struct aliasview_record *a;
     struct AliasView *aview;
     struct menu_context *page;
     struct menu_context *prompt_area;
{
    const struct string * key =  aliasview_key(a);
    int delay_redraw = 0;
    int ret = '\0';

    while(1) {
	int code;
		
	menu_PutLineX(prompt_area,1, 0, 
		      CATGETS(elm_msg_cat,
			      AliasesSet, AliasesAlreadySystemAlias,
			      "Already a system alias with name %S."),
		      key);

	code = prompt_letter(0,"",*def_ans_no,
			     PROMPT_yesno|PROMPT_redraw_mark|
			     PROMPT_ctrlL|PROMPT_cancel,
			     prompt_area,
			     CATGETS(elm_msg_cat,
				     AliasesSet, AliasesSuperceed,
				     "Superceed? (%c/%c) "), 
			     *def_ans_yes, *def_ans_no);
	
	if (('L'&31)    == code ||
	    REDRAW_MARK == code) {
	    menu_ClearScreen(page);   /* Reset possible redraw flag */
	    
	    /* Call refresh routines of children */
	    menu_redraw_children(page);
	    
	    if (menu_need_redraw(prompt_area))
		menu_ClearScreen(prompt_area);
	    
	    delay_redraw++;   /* Can't trigger redraw yet... */
	    continue;
	}
	
	if (TERMCH_interrupt_char == code || 
	    EOF == code) {
	    ret = -1;
	    break;
	}
		
	if (code != *def_ans_yes) {
	    ret = 0;
	    break;
	}

	ret = 1;
	break;
    }

    DPRINT(Debug,7,(&Debug, 
		    "superceed_system=%d\n",ret));

    if (delay_redraw)
	menu_trigger_redraw(page);    

    return ret;
}



/* 0 == not ok,
   1 == OK
   -1 == cancel */

static int verify_aliasname P_((struct alias_buffer *buffer,
				struct AliasView *aview,
				int to_replace,  int *replaces,
				struct menu_context *page,
				struct menu_context *prompt_area));
static int verify_aliasname(buffer,aview,to_replace,replaces,
			    page,prompt_area)
     struct alias_buffer *buffer;
     struct AliasView *aview;
     int to_replace;         /* Old alias index */
     int *replaces;          /* alias index for new name */
     struct menu_context *page;
     struct menu_context *prompt_area;
{
    const struct string *key   = alias_buffer_key(buffer); 
    struct aliasview_record *rec = NULL;
    int delay_redraw = 0;
    int ret = 0;

    if (!key) {
	DPRINT(Debug,2,(&Debug, "verify_aliasname: No aliasname\n"));

	lib_error(CATGETS(elm_msg_cat,AliasesSet, 
			  AliasesPleaseEnterAliasname,
			  "No alias name given. Please enter alias name.")); 
	return 0;
    }
    
    if (0 == string_len(key)) {
	DPRINT(Debug,2,(&Debug, "verify_aliasname: Empty aliasname\n"));

	lib_error(CATGETS(elm_msg_cat,AliasesSet, 
			  AliasesPleaseEnterAliasname,
			  "No alias name given. Please enter alias name.")); 
	return 0;
    }
    
    rec = lookup_alias(aview,key,replaces);

    if (!rec) {
	/* Only new names are checked */

	/* Assumes ASCII for string */
	int idx = string_have_ascii_sep(key,s2us("@<>\\\""));

	if (idx >= 0) {
	    struct string *A = clip_from_string(key,&idx,1);
	    
	    lib_error(CATGETS(elm_msg_cat,AliasesSet, 
			      AliasesBadnameNotAllowed,
			      "Character \"%S\" on alias name %S is not allowed."),
		      A,key);
	    
	    free_string(&A);
	    return 0;
	}

	DPRINT(Debug,2,(&Debug, 
			"verify_aliasname=1: Key \"%S\" not found\n",
			key));
	return 1;
    }

    if (*replaces == to_replace) {
	
	DPRINT(Debug,2,(&Debug, 
			"verify_aliasname=1: Key \"%S\" have old alias %d index\n",
			key,to_replace));
	return 1;
    }


    if (aliasview_have_type_mask(rec,SYSTEM)) {
	int r;

	DPRINT(Debug,2,(&Debug, 
			"verify_aliasname: Key \"%S\" index %d is system alias\n",
			key,*replaces));

	r = superceed_system(rec,aview,page,prompt_area);

	*replaces = -1;   /* System aliases are not overwitten */

	DPRINT(Debug,2,(&Debug, 
			"verify_aliasname=%d\n",r));
	return r;
    }

    while(1) {
	int code;

	menu_PutLineX(prompt_area,1,0, 
		      CATGETS(elm_msg_cat,
			      AliasesSet, AliasesAlreadyAliasName,
			      "Already a alias with name %S."), 
		      key);
    
	code = prompt_letter(0,"",*def_ans_no,
			       PROMPT_yesno|PROMPT_redraw_mark|
			       PROMPT_ctrlL|PROMPT_cancel,
			       prompt_area,
			       CATGETS(elm_msg_cat,
				       AliasesSet, AliasesReplaceExisting,
				       "Replace existing alias? (%c/%c) "),
			       *def_ans_yes, *def_ans_no);

	
	if (('L'&31)    == code ||
	    REDRAW_MARK == code) {
	    menu_ClearScreen(page);   /* Reset possible redraw flag */
	    
	    /* Call refresh routines of children */
	    menu_redraw_children(page);
	    
	    if (menu_need_redraw(prompt_area))
		menu_ClearScreen(prompt_area);
	    
	    delay_redraw++;   /* Can't trigger redraw yet... */
	    continue;
	}

	if (TERMCH_interrupt_char == code || 
	    EOF == code) {
	    ret = -1;
	    break;
	}
	

	if (code != *def_ans_yes) {
	    ret = 0;
	    break;
	}

	ret = 1;
	break;
    }


    DPRINT(Debug,2,(&Debug, 
		    "verify_aliasname=%d\n",ret));

    if (delay_redraw)
	menu_trigger_redraw(page);    

    return ret;
}



static int add_alias(to_replace, aview, page, LOC)
     int to_replace;
     struct AliasView *aview;
     struct menu_context  *page;
     struct screen_parts *LOC;
{
    /*
     *	Add an alias to the user alias text file.  If there
     *	are aliases tagged, the user is asked if he wants to
     *	create a group alias from the tagged files.
     *
     *	Return zero if alias not added in actuality.
     *
     *	If to_replace == -1, then we will ask for the new
     *	aliasname.
     *
     *	If to_replace >= 0, then we are replacing the alias
     *	denoted by to_replace.
     *
     *	Note that even if to_replace == -1, if the user types
     *	in the name of a current alias then we can still do
     *	a replacement.
     */
    
    int i, leftoff = 0, tagged = 0;
    
    int ch = *def_ans_no;
    int ac;

    struct alias_vector * alias_list = NULL;
    int delay_redraw  = 0;
    struct aliasview_record   * this_alias    = NULL;  /* Alias record for alias_modify() */
    struct alias_buffer       * buffer  = NULL;

    int ret = 0;
    int replaces = -1;    /* verify_aliasname gives */
    int r = 0;

    /*
     *	See if there are any tagged aliases.
     */
    ac = get_alias_count(aview);
    for (i=0; i < ac; i++) {

	struct aliasview_record * a = give_alias(aview,i);
	
	if (a &&
	    aliasview_ison_status(a,TAGGED)) {

	    const struct string   *key = aliasview_key(a);

	    if (!alias_list) { 
		leftoff = i;

		alias_list = new_alias_vector();
	    }

	    alias_vector_add_alias(alias_list,key);
	    
	    tagged++;
	}
    }
 
    if (tagged > 0) {
    ask_tagged:   

	if (tagged == 1) {
	    /*
	     * There is only on alias tagged.  Ask the question
	     * but the default response is NO.
	     */
	    
	    menu_PutLineX(LOC->prompt_page,1,0, CATGETS(elm_msg_cat,
							AliasesSet, 
							AliasesOneTagged,
							"There is 1 alias tagged..."));
	    menu_CleartoEOLN(LOC->prompt_page);
	    
	    
	    ch = prompt_letter(0,"",*def_ans_no,
			       PROMPT_yesno|PROMPT_redraw_mark|PROMPT_ctrlL|
			       PROMPT_cancel,LOC->prompt_page,
			       CATGETS(elm_msg_cat,
				       AliasesSet, AliasesCreateGroup,
				       "Create group alias? (%c/%c) "),
			       *def_ans_yes, *def_ans_no);

	} else if (tagged > 1) {
	    
	    /*
	     * If multiple tagged aliases then we assume the user
	     * wants to create a group alias.  The default response
	     * is YES.
	     */
	    
	    menu_PutLineX(LOC->prompt_page,
			  1,0, CATGETS(elm_msg_cat,
				       AliasesSet, AliasesManyTagged,
				       "There are %d aliases tagged..."),
			  tagged);
	    menu_CleartoEOLN(LOC->prompt_page);
	    
	    
	    ch = prompt_letter(0,"",*def_ans_yes,
			       PROMPT_yesno|PROMPT_redraw_mark|PROMPT_ctrlL|
			       PROMPT_cancel,LOC->prompt_page,
			       CATGETS(elm_msg_cat,
				       AliasesSet, AliasesCreateGroup,
				       "Create group alias? (%c/%c) "),
			       *def_ans_yes, *def_ans_no);
	    
	}
	    
	if (TERMCH_interrupt_char == ch || EOF == ch)
	    goto free_it;
	
	if (ch == ('L'&31) || ch == REDRAW_MARK) {
	    
	    menu_ClearScreen(page);   /* Clear possible redraw mark */
	    
	    /* Call refresh routines of children */
	    menu_redraw_children(page);
	    
	    if (menu_need_redraw(LOC->prompt_page))
		menu_ClearScreen(LOC->prompt_page);   /* but clear redraw mark from prompt_area*/
	    
	    /* NOTICE: using menu_trigger_redraw(page) on here
	       may cause redraw loop!
	    */
	    	    
	    delay_redraw++;
	    goto ask_tagged;
	}

	if (ch != *def_ans_yes) {
	    tagged = 0;
	    free_alias_vector(&alias_list);
	}

    }

    /*
     *	Only ask for an aliasname if we are NOT replacing the
     *	current alias.
     */
    if (to_replace >= 0) {
	this_alias = give_alias(aview,to_replace);

	if (this_alias) {
	    const struct string  * key UNUSED_VAROK = 
		aliasview_key(this_alias);

	    int alias_type             = aliasview_get_type(this_alias);


	    /*
	     *  First, see if what we are replacing is a SYSTEM
	     *  alias.  If so, we need to ask a question.
	     */
	    if (alias_type & SYSTEM) {
		int r;
			    
		DPRINT(Debug,3,
		       (&Debug, 
			"Aliasname [%S] is SYSTEM in add_alias\n", 
			key));
		
		/*
		 *  If they don't want to superceed the SYSTEM alias 
		 * then just return.
		 */
		r = superceed_system(this_alias, aview, page, LOC->prompt_page);

		if (r <= 0) {
		    goto free_it;
		}


		to_replace = -1;  /* Do not overwrite system alias */
	    } 
	}
    }
    

    if (this_alias) 
	buffer =  give_alias_buffer(this_alias);
    else 
	buffer = malloc_empty_alias_buffer();


    if (alias_list) {  /* Create (or change alias to) group alias */
	
	struct address_alias  * new_value =
	    new_group_address_alias(NULL,NULL,alias_list);

	set_alias_buffer_value(buffer,new_value);
	
    }

    
    if (this_alias)
	r = alias_modify(aview,buffer,0,page);
    else
	r = alias_modify(aview,buffer,1,page);

    if (r < 0)
	goto free_it;	

    /* If not modified, need not saved */
    if (!alias_list && 0 == r)
	goto free_it;



    while (1) {
	int v = verify_aliasname(buffer,aview,to_replace,&replaces,
				 page,LOC->prompt_page);

	if (0 == v) {
	    
	    /* new == 1:  prompt alias name */
	    r = alias_modify(aview,buffer,1,page); 

	    if (r < 0)
		goto free_it;	


	} else if (v < 0) {
	    goto free_it;
	} else
	    break;
    }

    if (ask_accept(buffer,replaces,
		   aview,  page, LOC)) {


	/* NOTE:  ask_accept() calls add_alias_to_aview() */

	/*
	 * We can only clear the tags after we know that the
	 * alias was added.  This allows the user to back out
	 * and rethink without losing the tags.
	 */
	if (tagged > 0) {
	    ac = get_alias_count(aview);
	    
	    for (i=leftoff; i < ac; i++) {
		struct aliasview_record * a = give_alias(aview,i);

		if (a &&
		    aliasview_ison_status(a, TAGGED)) {
		    struct menu_common MENU;

		    int vis;

		    aliasview_clearf_status(a,TAGGED);

		    set_mcommon_from_aliasview(&MENU,aview);
    
		    vis = compute_visible(i+1, &MENU);
		    menu_header_status_update(LOC->header_page,vis-1);

		}
	    }
	}


	

	ret = 1;

	if (!delay_redraw)
	    menu_trigger_redraw(LOC->header_page);
    }

 free_it:

    if (buffer)
	free_alias_buffer(&buffer);

    if (alias_list)
	free_alias_vector(&alias_list);
    

    if (delay_redraw)
	menu_trigger_redraw(page);
    

    DPRINT(Debug,9, (&Debug, "add_alias=%d\n",ret));

    return ret;
}

static int add_current_alias P_((struct MailboxView *mailbox,
				 struct AliasView *aview,
				 struct menu_context  *page,
				 struct screen_parts *LOC));
static int add_current_alias(mailbox,aview,  page, LOC)
     struct MailboxView *mailbox;
     struct AliasView *aview;
     struct menu_context  *page;
     struct screen_parts *LOC;
{
    /*
     *	Alias the current message to the specified name and
     *	add it to the alias text file, for processing as
     *	the user leaves the program.
     *
     *	Returns non-zero iff alias actually added to file.
     */
    
    struct header_rec *current_header;
    
    int current_mail_message = get_current(mailbox);
    int addr_count;

    struct alias_buffer       * buffer  = NULL;
    struct address_alias      * new_value = NULL;
	    

    int r = 0;
    int replaces = -1;
    int ret = 0;


    if (current_mail_message == 0) {
	DPRINT(Debug,3,
	       (&Debug,
		"Add current alias called without any current message!\n"));
	lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesNoMessage,
			  "No message to alias to!"));
	return(0);
    }

    current_header = give_header(mailbox,current_mail_message - 1);

    if (!current_header ||
	!current_header->from) 
	return 0;

    addr_count = addr_list_item_count(current_header->from);

    if (addr_count < 1)
	return 0;

    if (1 == addr_count) {
	int group = -1;   /* Groups are not allowed on From: header fields */

	const struct address * address = 
	    addr_list_get_item(current_header->from,0,&group);    

	const struct string * fullname  = address_get_phrase(address);

	if (fullname) {
	    const struct string * comment1  = address_get_comment(address);
	    int POS = 0;
	    struct string * firstname = NULL;
	    struct string * lastname = NULL;
	    uint16 delim = 0x0000;
	    int L;

		
	    /*
	     *	Try to break up the From: phrase into firstname, lastname, and
	     *	any other text.  This is based on the fact that many address
	     *	phrases are pretty straightforward.  This will break on many
	     *	situations.  Should handle:
	     *		Robert Howard
	     */
	    
	    
	    if (! get_word_from_string(fullname,&firstname,&POS,GWF_trim_space,
				       s2us(" \t,.:;"),&delim))
		goto fail_person;

	    if (delim &&
		delim != 0x0020 /* space */)
		goto fail_person;


	    if (! (L = get_word_from_string(fullname,&lastname,&POS,
					    GWF_trim_space,
					    s2us(" \t.,:;"),&delim)))
		goto fail_person;

	    if (delim &&
		delim != 0x0020 /* space */ &&
		delim != 0x002E /* . */)
		goto fail_person;


	    /*
	     *   If the second word is either a single
	     *   character or a character followed by '.' it was
	     *   probably a middle initial.  Add it to firstname
	     *   and shift.
	     */
	    if (1 == L) {
		add_ascii_to_string(firstname,s2us(" "));

		append_string(&firstname,lastname,1);
		
		if (delim == 0x002E /* . */) {
		    POS++;
		    add_ascii_to_string(firstname,s2us("."));
		}

		free_string(&lastname);

		L = get_word_from_string(fullname,&lastname,&POS,
					 GWF_trim_space,
					 s2us(" \t,.:;"),&delim);


	    } 

	    if (delim &&
		delim != 0x0020 /* space */)
		goto fail_person;

	    if (L < 1)
		goto fail_person;
	    if (delim == 0x002E /* . */)
		goto fail_person;


	    if (POS < string_len(fullname)) {

	    fail_person:

		/* Do not separate firstname and lastname */

		new_value = new_person_address_alias(NULL,NULL,NULL,address);


	    } else {

		const char * ascii_addr = address_get_ascii_addr(address);

		struct address * addr0 = NULL;

		if (!ascii_addr)
		    goto fail_person;

		/* Seperate firstname and lastname 
		 *
		 * NOTE: Address comment (comment1) is not used as
		 *       alias comment ...
		 */
		
		addr0 = new_address(ascii_addr,NULL,comment1);

		new_value = new_person_address_alias(firstname,lastname,
						     NULL,addr0);
		
		free_address(&addr0);
	    }

	    if (firstname)
		free_string(&firstname);

	    if (lastname)
		free_string(&lastname);

	} else {

	    /* Generate dummy person address */

	    new_value = new_person_address_alias(NULL,NULL,NULL,address);

	}
	    	    

    } else {

	new_value = new_list_address_alias(NULL,current_header->from);

    }


    buffer = malloc_empty_alias_buffer();
    set_alias_buffer_value(buffer,new_value);
    
    r = alias_modify(aview,buffer,1,page);
    
    if (r < 0)
	goto free_it;	

    while (1) {
	int v = verify_aliasname(buffer,aview,-1,&replaces,
				 page,LOC->prompt_page);

	if (0 == v) {
	    
	    /* new == 1:  prompt alias name */
	    r = alias_modify(aview,buffer,1,page); 

	    if (r < 0)
		goto free_it;	


	} else if (v < 0) {
	    goto free_it;
	} else
	    break;
    }

    if (ask_accept(buffer,replaces,
		   aview,  page, LOC)) {
	
	/* NOTE:  ask_accept() calls add_alias_to_aview() */

	ret = 1;
	
	menu_trigger_redraw(LOC->header_page);
    }

    

 free_it:

    if (new_value) 
	free_address_alias(&new_value);
	          
    if (buffer)
	free_alias_buffer(&buffer);

    DPRINT(Debug,9, (&Debug, "add_current_alias=%d\n",ret));

    return ret;
}



static void set_alias_screen P_((struct menu_context  *page, 
				   struct screen_parts *LOC,
				   struct menu_param  *LIST));
static void set_alias_screen(page,LOC, LIST)
     struct menu_context  *page;
     struct screen_parts *LOC;
     struct menu_param  *LIST;
{    
    int   LINES, COLUMNS;	

    menu_get_sizes(page,&LINES, &COLUMNS);

    /* 1)  Title part of screen */

    if (! LOC->title_page)
	LOC->title_page = new_menu_subpage(page,0,4,sb_update_title,LIST);
    else
	menu_subpage_relocate(LOC->title_page,page,0,4);

    /* 2) menu part */

     if (LOC->menu_page && LINES < 14)
	erase_menu_context (&(LOC->menu_page));
    else if (LOC->menu_page)
	menu_subpage_relocate(LOC->menu_page,page,LINES-8,4);
    else if (mini_menu && LINES > 14)
	LOC->menu_page = new_menu_subpage(page,LINES-8,4,
					  sb_alias_menu,LIST);
   
    mailbox_screen_common(page,LOC, LIST);		
}

static void check_alias_screen P_((struct screen_parts *LOC,
				     struct menu_param *list));
static void check_alias_screen(LOC,list)
     struct screen_parts *LOC;
     struct menu_param *list;
{
    /* 1) title page */
    if (menu_resized(LOC->title_page)) {
	DPRINT(Debug,1, (&Debug, "title page resized\n"));

    }
    if (menu_need_redraw(LOC->title_page)) {
	DPRINT(Debug,1, (&Debug, "title page redraw???\n"));
	sb_update_title(LOC->title_page,list);
    }

    if (LOC->menu_page) {
	/* 2) menu page */
	if (menu_resized(LOC->menu_page)) {
	    DPRINT(Debug,1, (&Debug, "menu page resized\n"));
	    
	}
	if (menu_need_redraw(LOC->menu_page)) {
	    DPRINT(Debug,1, (&Debug, "menu page redraw???\n"));
	    sb_show_menu(LOC->menu_page,list);
	}
    }

    /* 3) prompt part */
    if (menu_resized(LOC->prompt_page)) {
	DPRINT(Debug,1, (&Debug, "prompt page resized\n"));
    }
    if (menu_need_redraw(LOC->prompt_page)) {
	menu_ClearScreen(LOC->prompt_page);

	show_last_error();	/* for those operations that have to
				 * clear the footer except for a message.
				 */
    }

    /* 4) headers part */
    if (menu_resized(LOC->header_page)) {
	DPRINT(Debug,1, (&Debug, "header page resized\n"));
    }
    if (menu_need_redraw(LOC->header_page)) {
	DPRINT(Debug,1, (&Debug, "header page redraw\n"));
	menu_ClearScreen(LOC->header_page);
    }
}

static void alias_expand_prompt P_((struct menu_context  *page,
				    struct AliasView *aview,
				    struct screen_parts *LOC));

static void alias_expand_prompt(page,aview,LOC)
     struct menu_context  *page;
     struct AliasView *aview;
     struct screen_parts  *LOC ;
{
    int line;
    struct string * alias_name = NULL;
    int code;
    int delay_redraw = 0;


 redraw:
    menu_MoveCursor(LOC->prompt_page,1,0);
    menu_CleartoEOS(LOC->prompt_page);
    
    
    /* FIXME --optionally_enter*  should use prompt_page */
    line = menu_GetAbsLine(LOC->prompt_page,1);
 
    code = optionally_enter2(page,
			     &alias_name,
			     line,0,
			     OE_APPEND_CURRENT|OE_REDRAW_MARK|
			     OE_SIG_CHAR /* Ctrl-C */,
			     CATGETS(elm_msg_cat,
				     AliasesSet, 
				     AliasesFullyExpanded,
				     "Fully expand alias: "));

    if (REDRAW_MARK == code ||
	('L'&31)    == code) {
	menu_ClearScreen(page);   /* Clear possible redraw mark */
	
	/* Call refresh routines of children */
	menu_redraw_children(page);
	
	if (menu_need_redraw(LOC->prompt_page))
	    menu_ClearScreen(LOC->prompt_page);   /* but clear redraw mark from prompt_area*/
	
	/* NOTICE: using menu_trigger_redraw(page) on here
	   may cause redraw loop!
	*/

	delay_redraw++;
	goto redraw;
    }

    if (0 == code && alias_name) {
	alias_expansion_info(NULL,alias_name,page,aview);
    } 

    if (alias_name)
	free_string(&alias_name);

    if (delay_redraw)
	menu_trigger_redraw(page);
}


int alias(mailbox, aview)
     struct MailboxView *mailbox;
     struct AliasView *aview;
{
    
    /*
     *	Work with alias commands...
     */
   
    int retch = '\0';
 
    int ch;
    static int  newaliases = 0;
    int have_command = '\0';
    
    struct menu_context  *page = new_menu_context();
    
    
    struct screen_parts  LOC  = { NULL, NULL };
    struct menu_common MENU;
    struct menu_param  PARAM[elm_mp_COUNT+1] = { 
	{ mp_menu_common, { 0 } },
	{ mp_integer,     { 0 } },
	{ mp_integer,     { 0 } },
	{ mp_integer,     { 0 } },
	{ mp_integer,     { 0 } },
	{ mp_END,         { 0 } }
    };

    struct elm_commands *menu_cmds = give_alias_commands();

    static int first_time = TRUE;

/*
 *	We're going to try to match the way elm does it at
 * 	he main menu.  I probably won't be able to use any
 *	main menu routines, but I will "borrow" from them. RLH
 */

    if (first_time) {
	a_system_flag = catgets(elm_msg_cat, AliasesSet, AliasesSystemFlag,
				"(S)");
	a_rev_alias_pad = catgets(elm_msg_cat, AliasesSet, AliasesRevAliasPad,
				  "Reverse Alias Name      ");
	a_rev_alias_abr = catgets(elm_msg_cat, AliasesSet, AliasesRevAliasAbr,
				  "Reverse-Alias");
	a_rev_alias_name = catgets(elm_msg_cat, AliasesSet, AliasesRevAliasName,
				   "Reverse Alias Name");
	a_rev_full_pad = catgets(elm_msg_cat, AliasesSet, AliasesRevFullPad,
				 "Reverse Full (Real) Name");
	a_full_abr = catgets(elm_msg_cat, AliasesSet, AliasesRevFullAbr,
			     "Reverse-Name");
	a_rev_full_name = catgets(elm_msg_cat, AliasesSet, AliasesRevFullName,
				  "Reverse Full (Real) Name");
	a_rev_text_pad = catgets(elm_msg_cat, AliasesSet, AliasesRevTextPad,
				 "Reverse Text File       ");
	a_text_abr = catgets(elm_msg_cat, AliasesSet, AliasesRevTextAbr,
			     "Reverse-Text");
	a_rev_text_file = catgets(elm_msg_cat, AliasesSet, AliasesRevTextFile,
				  "Reverse Text File");
	a_alias_pad = catgets(elm_msg_cat, AliasesSet, AliasesAliasPad,
			      "Alias Name              ");
	a_alias_abr = catgets(elm_msg_cat, AliasesSet, AliasesAliasAbr,
			      "Alias");
	a_alias_name = catgets(elm_msg_cat, AliasesSet, AliasesAliasName,
			       "Alias Name");
	a_full_pad = catgets(elm_msg_cat, AliasesSet, AliasesFullPad,
			     "Full (Real) Name        ");
	a_full_abr = catgets(elm_msg_cat, AliasesSet, AliasesFullAbr,
			     "Name");
	a_full_name = catgets(elm_msg_cat, AliasesSet, AliasesFullName,
			      "Full (Real) Name");
	a_text_pad = catgets(elm_msg_cat, AliasesSet, AliasesTextPad,
			     "Text File               ");
	a_text_abr = catgets(elm_msg_cat, AliasesSet, AliasesTextAbr,
			     "Text");
	a_text_file = catgets(elm_msg_cat, AliasesSet, AliasesTextFile,
			      "Text File");
	first_time = FALSE;
    }

    set_mcommon_from_aliasview(&MENU, aview);
    mp_list_set_mcommon(PARAM,elm_mp_menu,&MENU);

    mp_list_set_integer(PARAM,elm_mp_modified,newaliases);
    set_alias_screen(page,&LOC,PARAM);

    set_aview_alias_mode(aview, &LOC);	

    copy_current(&MENU,LOC.header_page);
    alias_screen(aview, page);
    /* define_softkeys(ALIAS); */

    while (1) {
	
    resize_mark:
	menu_set_default(page);

	set_mcommon_from_aliasview(&MENU, aview);

	if (menu_resized(page)) {
	    int	LINES, COLUMNS;
		
	    set_alias_screen(page,&LOC,PARAM);
	
	    menu_get_sizes(page,&LINES, &COLUMNS);
		
	    menu_trigger_redraw(page);

	}

#ifdef BACKGROUD_PROCESSES      
	if (handle_sigchld)
	    sigchld_handler();
#endif

	if (update_aview(aview)) {
	    int tac = get_total_alias_count(aview);

	    sort_aliases(tac, sa_silent_sort,aview);

	    get_page(&MENU, LOC.header_page);

	    menu_trigger_redraw(LOC.header_page);
	    menu_trigger_redraw(LOC.title_page);
	}

	if (menu_need_redraw(page)) {	  /* Redraw screen if necessary */
	    DPRINT(Debug,7, (&Debug, 
			     "alias: pending redraw\n"));
	    
	    mp_list_set_integer(PARAM,elm_mp_modified,newaliases);
	    alias_screen(aview, page);
	     
	}
	check_alias_screen(&LOC, PARAM);
	

	if (have_command) {
	    ch = have_command;
	    have_command = '\0';

	    menu_PutLineX(LOC.prompt_page,0,0,
			  FRM("%S"),mcommon_give_item(&MENU, m_Prompt));
	    menu_CleartoEOS(LOC.prompt_page);
	    
	    if (isascii(ch) && isprint(ch)) {
		DPRINT(Debug,4,
		       (&Debug, "-- Cached command: %c [%d]\n",ch,ch));
	    } else {
		DPRINT(Debug,4,
		       (&Debug, "-- Cached command: [%d]\n",ch));
	    }

	} else {
	    
	    int lin,col;
	    
	    menu_PutLineX(LOC.prompt_page,0,0,
			  FRM("%S"),mcommon_give_item(&MENU, m_Prompt));
	    
	    menu_GetXYLocation(LOC.prompt_page,&lin,&col);
	    
	    menu_CleartoEOLN(LOC.prompt_page);

	    show_last_error();
	    menu_MoveCursor(LOC.prompt_page,lin,col);

	    ch = menu_ReadCh(LOC.prompt_page,
			     REDRAW_MARK|READCH_CURSOR|
			     READCH_resize);
	    menu_MoveCursor(LOC.prompt_page,lin,col); 
	    menu_CleartoEOS(LOC.prompt_page);
	
	
	    if (isascii(ch) && isprint(ch)) {
		DPRINT(Debug,4,
		       (&Debug, "-- Alias command: %c [%d]\n",ch,ch));
	    } else {
		DPRINT(Debug,4,
		       (&Debug, "-- Alias command: [%d]\n",ch));
	    }

	    set_error("");	/* clear error buffer */
	    
	    menu_MoveCursor(LOC.prompt_page,lin,col);
	}

	  switch (ch) {
	  case RESIZE_MARK:
	      DPRINT(Debug,4,  (&Debug," ... resizing\n"));
	      goto resize_mark;

	  case F3_KEY_MARK:
	    mini_menu = ! mini_menu;

	    if (LOC.menu_page && !mini_menu)
		erase_menu_context (&(LOC.menu_page));

	    DPRINT(Debug,10,(&Debug,"mini_menu = %d \n",mini_menu));

	    set_alias_screen(page,&LOC,PARAM);
		
	    menu_trigger_redraw(page);   /* ? Is needed? */
	    
	    break;
	      
	  case REDRAW_MARK:
	    case 'L'&31: 
		menu_trigger_redraw(page);
		break;


	  case '!' : 
	      menu_Writechar(LOC.prompt_page,'!');
	      if (ALLOW_subshell)
		  subshell(mailbox,page,LOC.prompt_page);
	      else
		  lib_error(CATGETS(elm_msg_cat, ElmSet, 
				    ElmNoSubshellCommand,
				    "Subshell command isn't configured in this version of ELM."));
	      break;


	  case '$':
	  case F5_KEY_MARK:
	      menu_Write_to_screen(LOC.prompt_page,
				   CATGETS(elm_msg_cat, AliasesSet, 
					   AliasesResync,
					   "Resynchronize aliases..."));
	      FlushBuffer();

	           /*
	            * Process deletions and then see if we need to
	            * write aliases
	            */
	      if (resync_aliases(newaliases,aview, page, &LOC) 
		  > 0) {

		  install_aliases(aview,&LOC);

		  newaliases = 0;
		  menu_trigger_redraw(page);
		  
	      }
	      break;

	  case 'a': 
	      menu_Write_to_screen(LOC.prompt_page,
				   CATGETS(elm_msg_cat, AliasesSet, 
				      AliasesAddCurrent,
				      "Add address from current message..."));
	      FlushBuffer();

	      clear_error();
	      if (add_current_alias(mailbox,aview, page, &LOC)) {
		  newaliases++;

		  mp_list_set_integer(PARAM,elm_mp_modified,newaliases);
		  menu_trigger_redraw(LOC.title_page);
		 
	      }
	      break;

	  case 'c': {
	      int alias_current = get_alias_current(aview);

	      if (alias_current > 0) {
		  menu_Write_to_screen(LOC.prompt_page,
				       CATGETS(elm_msg_cat,
					       AliasesSet, AliasesReplaceCurrent,
					       "Replace current alias in database..."));
		  FlushBuffer();

		  clear_error();
		  if (add_alias(alias_current-1, aview, 
				page, &LOC)) {
		      newaliases++;

		      mp_list_set_integer(PARAM,elm_mp_modified,newaliases);
		      menu_trigger_redraw(LOC.title_page);

		  }
	      }
	      else {
		  lib_error(CATGETS(elm_msg_cat,
				    AliasesSet, AliasesNoneToReplace,
				    "Warning: no aliases to replace!"));
	      }
	  }
	      break;

	  case 'e': 

	      menu_Write_to_screen(LOC.prompt_page,
				   CATGETS(elm_msg_cat, AliasesSet, AliasesEdit0,
					   "Edit"));
	      FlushBuffer();

	      /*
	       * Process aliases for deletions, etc.  You
	       * have to do this *before* checking current because
	       * all aliases could be marked for deletion.
	       */
	      if (resync_aliases(newaliases,aview, page, 
				 &LOC) < 0)
		  break;


	      /* edit_aliases_text() indirectly writes
		 aliases file if needed
	      */

	      if (edit_aliases_text(aview, page, &LOC)) {
		  newaliases = 0;
	      }
	      
	      break;

	    case 'm':{
		int alias_current = get_alias_current(aview);
		
		if (alias_current > 0) {
		    menu_Write_to_screen(LOC.prompt_page,
					 CATGETS(elm_msg_cat, AliasesSet, AliasesMail,
						 "Mail..."));
		    FlushBuffer();

		    a_sendmsg(TRUE,
			      mailbox, aview, 
			      page, &LOC);

		}
		else {
		    lib_error(CATGETS(elm_msg_cat,
				      AliasesSet, AliasesNoneToMail,
				      "Warning: no aliases to send mail to!"));
		}
	    } 
		break;

	  case 'n': 
	      menu_Write_to_screen(LOC.prompt_page,
				   CATGETS(elm_msg_cat, AliasesSet, AliasesAddNew,
					   "Add a new alias to database..."));
	      FlushBuffer();
		
		      clear_error();
		      if (add_alias(-1, aview, page, &LOC)) {
		          newaliases++;
		          
			  mp_list_set_integer(PARAM,elm_mp_modified,
					      newaliases);
			  menu_trigger_redraw(LOC.title_page);
		      }
		      break;

	    case 'q':
	    case 'Q':
	    case 'i':
	    case 'I':
	    case 'r':
	  case 'R': {
	      int r;
	      menu_Write_to_screen(LOC.prompt_page,
				   CATGETS(elm_msg_cat, AliasesSet, 
					   AliasesAddReturn,
					   "Return to main menu..."));
	      FlushBuffer();

	           /*
	            * leaving the alias system.  Must check for
	            * pending deletes, etc.  prompt is set to FALSE
	            * on uppercase letters so that deletions are
	            * NOT queried.
	            */

	      r = delete_aliases(newaliases, islower(ch), aview, page,
				 &LOC);
	      if (-1 == r) { /* Ctrl-C */
		  break;
	      }

	      if (r > 0) {
		  install_aliases(aview,&LOC);
		  newaliases = 0;
	      }
	      clear_error();

	      goto OUT;
	  }
	      
	  case RETURN:
	  case LINE_FEED:
	  case ' ':
	  case 'v':	{
		int alias_current = get_alias_current(aview);
		
		if (newaliases) {		/* Need this ?? */
		    int lin,col;
		    GetXYLocation(&lin,&col);

		    lib_error(CATGETS(elm_msg_cat,
				      AliasesSet, AliasesNotInstalled,
				      "Warning: new aliases not installed yet!"));
		    MoveCursor(lin,col);
		}
		
		if (alias_current > 0) {

		    have_command = show_alias_info(aview,page,
						   LOC.header_page);

		} else {
		    
		    lib_error(CATGETS(elm_msg_cat,
				      AliasesSet, AliasesNoneToView,
				      "Warning: no aliases to view!"));
		}
	  }
	      break;

	  case 'x':
	  case 'X': 
	      menu_Write_to_screen(LOC.prompt_page,
				   CATGETS(elm_msg_cat, AliasesSet, 
					   AliasesAddReturn,
					   "Return to main menu..."));
	      FlushBuffer();
	      
	      exit_alias(aview);
	      clear_error();
	      
	      goto OUT;

	  case 'f':
	  case 'F': {

	      int alias_current = get_alias_current(aview);
	      if (alias_current > 0) {
		  
		  if (ch == 'F')
		      alias_expand_prompt(page,aview,&LOC);
		  else {

		      have_command = 
			  show_alias_expanded(aview,page,
					      LOC.header_page);
		  }

	      } else 
		  lib_error(CATGETS(elm_msg_cat,
				    AliasesSet, AliasesNoneToView,
				    "Warning: no aliases to view!"));
	      
	  }
	      break;
	      
	 /*
	  * None of the menu specific commands were chosen, therefore
	  * it must be a "motion" command (or an error).
	  */
	    default	: 
		ch = motion(ch,&MENU,&LOC, page, menu_cmds);
	      		
		switch (ch) {
		case 0: /* OK */
		    break;
    
		case EOF: 
		    retch  = EOF;
		    goto OUT;

		default:
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmUnknownCommand,
				      "Unknown command. Use '?' for help."));		    
		}
	  }

	  if (menu_need_redraw(page)) {	  /* Redraw screen if necessary */
	      DPRINT(Debug,7, (&Debug, 
			       "alias: pending redraw\n"));

	      mp_list_set_integer(PARAM,elm_mp_modified,newaliases);
	      alias_screen(aview, page);
	  }

	  check_range(&MENU, &LOC);   
	  check_alias_screen(&LOC, PARAM);
	  
	  
    }			/* BIG while loop... */

 OUT:
    free_commands(&menu_cmds);   /* Decrement refcount */

    free_mailbox_screen(&LOC);
    
    erase_menu_context(&page);

    return retch;
}

static void install_aliases(aview,LOC)
     struct AliasView *aview;
     struct screen_parts *LOC;
{

    lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesUpdating,
		      "Updating aliases..."));

    if (write_aliasview(aview))
	lib_error(CATGETS(elm_msg_cat, AliasesSet, AliasesUpdatedOK,
			  "Aliases updated successfully."));

}

/* Cause that all unknown aliases are treated as local addresses */

#if ANSI_C
alias_stack_lookup_f lookup_dummy;
#endif
int lookup_dummy(value,alias,result_alias,result_list,expand_result)
     union alias_stack_item     value;
     const struct string       *alias;
     const struct address_alias **result_alias;
     /* Caller must free result_list */
     struct addr_list          **result_list;
     enum alias_address_expand_result * expand_result;
{
    /* Only ASCII accounts are accepted on here ... */

    if (!can_ascii_string(alias)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet,
			  ElmAsciiRequiredAddress,
			  "Ascii required for address %S"),
		  alias);
	return 0;
    }


    {
	struct address * address =  new_address_string (alias,NULL,NULL);

	*result_list = new_empty_addr_list(1);
	
	add_address_to_list(*result_list,address,-1);

	free_address(&address);       	
    }

    if (expand_result) {
	*expand_result = alias_expand_local_dummy;
	DPRINT(Debug,10,
	       (&Debug,"lookup_dummy: setting alias_expand_local_dummy\n"));
    }
    
    return 1;
}

static int ask_accept(buffer, replacement,aview,page,LOC)
     struct alias_buffer * buffer;
     int replacement;
     struct AliasView *aview;
     struct menu_context  *page;
     struct screen_parts *LOC;
{
    int delay_redraw = 0;
    int answer = 0;
    int ret    = 0;

    const struct string * alias_key = 
	alias_buffer_key(buffer);

    const struct address_alias * address_alias =
	alias_buffer_value(buffer);
   

    if (!address_alias) {

	DPRINT(Debug,2,(&Debug, "ask_accept: No address value\n"));

	lib_error(CATGETS(elm_msg_cat,AliasesSet, 
			  AliasesNoAliasvalue,
			  "No alias address (or value) given.")); 
	return 0;

    } else {

	struct alias_stack         * stack = new_alias_stack();
	struct addr_list           * addr  = NULL;  


	
	union alias_stack_item  mailer;
	mailer.dummy_arg = NULL;
		
	add_aview_to_alias_stack(stack,aview);
	add_lookup_to_alias_stack(stack,lookup_dummy,mailer);
	
	addr = aliases_expand_alias_proto(stack,address_alias);
	
	free_alias_stack(&stack);

 again:
	if (addr) {
	    struct string * str1 = NULL;
	    int LINES, COLUMNS;
	    	    
	    menu_get_sizes(LOC->prompt_page, &LINES, &COLUMNS);
	    	    
	    str1 = display_list_addrs(addr,COLUMNS-30);

	    if (!str1) {
		DPRINT(Debug,5,(&Debug,
				"ask_accept: Alias expanded, but display_list_addrs failed\n"));

		free_addr_list(&addr);
		
		goto fail_expand;
	    }
	    	    
	    menu_PutLineX(LOC->prompt_page,
			  2,0, 
			  CATGETS(elm_msg_cat, AliasesSet, 
				  AliasesAddressAsS,
				  "Messages addressed as: %S"),
			  str1);
	    
	    free_string(&str1);
	} else {
	fail_expand:
	    menu_PutLineX(LOC->prompt_page,
			  2,0, 
			  CATGETS(elm_msg_cat, AliasesSet, 
				  AliasesAddressAsFailed,
				  "Messages addressed as: (Alias expansion failed)"));
	}
    
	{	
	    const struct string       *phrase = NULL;
	    const struct alias_vector *vector = NULL;
	    
	    int LINES, COLUMNS;
	    menu_get_sizes(LOC->prompt_page, &LINES, &COLUMNS);
	    
	    if (address_alias_get_group(address_alias,&phrase,&vector)) {
		struct string *X = 
		    display_list_members(vector,COLUMNS-25);
		
		if (!X)
		    goto empty;
		
		menu_PutLineX(LOC->prompt_page,1,0, 
			      CATGETS(elm_msg_cat, AliasesSet, 
				      AliasesAddressTo,
				      "New alias: %S is '%S'."), 
			      alias_key, X);
		
		free_string(&X);
	    } else {
	    empty:
		
		menu_PutLineX(LOC->prompt_page,1,0, 
			      CATGETS(elm_msg_cat, AliasesSet, 
				      AliasesNewAliasName,
				      "New alias: %S"), 
			      alias_key);
	    }
	}
    
	
	menu_CleartoEOLN(LOC->prompt_page);
	/*
	 *	Kludge Alert:  Spaces are padded to the front of the prompt
	 *	to write over the previous question.  Should probably record
	 *	the end of the line, move to it, and CleartoEOLN() it.
	 */
	
	answer = prompt_letter(0,"",*def_ans_no,
			       PROMPT_yesno|PROMPT_redraw_mark|
			       PROMPT_ctrlL|PROMPT_cancel,
			       LOC->prompt_page,
			       CATGETS(elm_msg_cat, AliasesSet, AliasesAcceptNew,
				       "      Accept new alias? (%c/%c) "), 
			       *def_ans_yes, *def_ans_no);
	if (('L'&31)    == answer ||
	    REDRAW_MARK == answer) {
	    menu_ClearScreen(page);   /* Reset possible redraw flag */
	    
	    /* Call refresh routines of children */
	    menu_redraw_children(page);
	    
	    if (menu_need_redraw(LOC->prompt_page))
		menu_ClearScreen(LOC->prompt_page);
	    
	    delay_redraw++;   /* Can't trigger redraw yet... */
	    goto again;
	}
    
	
	if (addr)
	    free_addr_list(&addr);

	if (TERMCH_interrupt_char == answer || 
	    EOF == answer)
	    goto clean;
    
	if(answer == *def_ans_yes) {

	    /* !!! Actually adds alias */
	    struct aliasview_record *a =  
		add_alias_to_aview(aview,buffer,replacement);

	    if (a) {
		
		/*
		 *  First, clear flag if this is marked to be deleted.
		 *  This prevents the problem where they marked it for
		 *  deletion and then figured out that it could be
		 *  c)hanged but didn't explicitly U)ndelete it.  
		 */
		
		if (aliasview_ison_status(a, DELETED)) {
		    aliasview_clearf_status(a, DELETED);
		}
		
		/*
		 *  Changed aliases are given the NEW flag.
		 */
		aliasview_setf_status(a, NEW);
				
		if (replacement != -1) {

		    struct menu_common MENU;
		    int vis;
		    
		    set_mcommon_from_aliasview(&MENU,aview);
		    
		    vis = compute_visible(replacement+1, &MENU);
		    menu_header_status_update(LOC->header_page,vis-1);
		    
		}
			    
		ret = 1;
	    }
	}
    }

 clean:
    menu_ClearLine(LOC->prompt_page,1);
    menu_ClearLine(LOC->prompt_page,2);
    
    if (delay_redraw)
	menu_trigger_redraw(page);


    DPRINT(Debug,10, (&Debug, "ask_accept=%d: %S\n",ret,alias_key));

    return ret;
}



/* Check whether an address is aliased; if so return the alias, otherwise
 * return NULL. */
const struct string *address_to_alias(address, aview)
     const struct address  *address;
     struct AliasView *aview;
{
    int ac;
    int i;
    const char * addr0 = address_get_ascii_addr(address);

    if (!addr0)
	return NULL;

    DPRINT(Debug,8,(&Debug, 
		    "address_to_alias: searching %s\n",
		    addr0));
        
    ac = get_alias_count(aview);
    for (i = 0; i < ac; i++) {
	struct aliasview_record * a = give_alias(aview,i);

	if (a) {
	    const struct address_alias * alias =
		aliasview_address_alias(a);

	    const struct string *firstn     = NULL;
	    const struct string *lastn      = NULL;
	    const struct address *address1   = NULL;

	    if (address_alias_get_person(alias,&firstn,&lastn,&address1)) {
		
		const char * addr1 = address_get_ascii_addr(address1);

		/* XXXX   -- compare address is needed */

		if (addr1 && 0 == strcmp(addr0,addr1)) {		   
		    const struct string * ret = aliasview_key(a);

		    DPRINT(Debug,8,(&Debug, 
				    "address_to_alias=%S\n",
				    ret));

		    return ret;
		}		    	       
	    }
	}
    }

    DPRINT(Debug,8,(&Debug, 
		    "address_to_alias=NULL (not found)n"));

    return NULL;
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
