static char rcsid[] = "@(#)$Id: addr_util.c,v 2.31 2022/07/14 14:16:01 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.31 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/addr_util.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************
 *  Incorparated Elm 2.5 code from src/addr_util.c. 
 *  That code was following copyright:
 *
 *  The Elm Mail System
 *
 *                      Copyright (c) 1988-1995 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** This file contains addressing utilities 

**/

#include "def_elm.h"
#include "s_elm.h"
#include "s_me.h"

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

DEBUG_VAR(Debug,__FILE__,"addr");

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}


void add_one_alias_to_expanded(x,a)
     struct expanded_address *x;
     struct aliasview_record  *a;
{
    int NEWPOS = 0;
    const struct string   *s =  aliasview_key(a);
    
    if (x->addrs)
	NEWPOS = addr_list_item_count(x->addrs);
    
    /* Use quoted aliases ... */
    if (string_need_quote(s)) {
	struct string * s1 = string_quote_phrase(s);
	
	add_textual_addr_(x,s1,NEWPOS,0);
	
	free_string(&s1);
	
    } else
	add_textual_addr_(x,s,NEWPOS,0);
    
}

int aliases_to_expanded(x,aview)
     struct expanded_address *x;
     struct AliasView *aview;
{
    int i;
    int tagged = 0;
    int ac = get_alias_count(aview);

    int NEWPOS = 0;

    if (x->addrs)
	NEWPOS = addr_list_item_count(x->addrs);
    
    for (i=0; i < ac; i++) {
	struct aliasview_record  *a =  give_alias(aview,i);
	
	if (a &&
	    aliasview_ison_status(a,TAGGED)) {
	    
	    const struct string   *s =  aliasview_key(a);

	    /* Use quoted aliases ... */
	    if (string_need_quote(s)) {
		struct string * s1 = string_quote_phrase(s);
		
		add_textual_addr_(x,s1,NEWPOS,0);

		free_string(&s1);

	    } else
		add_textual_addr_(x,s,NEWPOS,0);

	    tagged++;
	}
    }
    
    if (tagged == 0) {
	int alias_current = get_alias_current(aview);
	struct  aliasview_record *a =  give_alias(aview,alias_current-1);
	
	if (a) {
	    const struct string   *s =  aliasview_key(a);
	    	    
	    /* Use quoted aliases ... */
	    if (string_need_quote(s)) {
		struct string * s1 = string_quote_phrase(s);
		
		add_textual_addr_(x,s1,NEWPOS,0);
		
		free_string(&s1);
		
	    } else
		add_textual_addr_(x,s,NEWPOS,0);

	}
    }

    return tagged;
}

int argv_to_expanded(result,argv,mailer_info,aview)
     struct expanded_address *result;
     char *argv[];
     struct mailer_info      *mailer_info;
     struct AliasView *aview;
{
    int i,res;
    free_expanded_address(result);

    DPRINT(Debug,6,
	   (&Debug, "argv_to_expanded:"));
    for (i = 0;  argv[i]; i++) {
	DPRINT(Debug,6,
	       (&Debug, " [%d]=%s",i,argv[i]));
    }
    DPRINT(Debug,6,
	   (&Debug,"\n"));

    for (i = 0;  argv[i]; i++) {
	/* Because this comes from argument vector,
	   we use system charset instead of display_charset
	*/

	struct string * s = new_string2(system_charset,
					s2us(argv[i]));
	add_textual_addr_(result,s,0,0);

	free_string(&s);
    }

    dump_expanded_address(6,"argv_to_expanded: before build_address_l()",
			  *result);

    /* And make address structures */
    res = build_address_l(result,mailer_info,aview);

    dump_expanded_address(6,"argv_to_expanded: (result)",*result);

    DPRINT(Debug,6,
	   (&Debug, "argv_to_expanded=%d\n",res));
    return res;
}


void addr_to_expanded(result,addrs,mailer_info,aview)
     struct expanded_address *result;
     struct addr_list *addrs;
     struct mailer_info      *mailer_info;
     struct AliasView *aview;
{
    int i;

    int alen = 0;
    int glen = 0;

    free_expanded_address(result);

    DPRINT(Debug,6, (&Debug ,"addr_to_expanded:"));
    if (!addrs) {
	DPRINT(Debug,6,
	       (&Debug ," -- NULL\n"));
    } else {
	
	alen = addr_list_item_count(addrs);
	glen = addr_list_group_count(addrs);

	DPRINT(Debug,6,(&Debug," (%d addrs, %d group) \n",alen,glen));

	for (i = 0; i < glen;  i++) {
	    const struct string * G UNUSED_VAROK = addr_list_get_group(addrs,i);

	    DPRINT(Debug,6,(&Debug,"addr_to_expanded: group [%d]: group=%S\n",
			    i,G));
	}

	for (i = 0; i < alen;  i++) {
	    
	    int group = -1;
	    const struct address * address = 
		addr_list_get_item(addrs,i,&group);
	    
	    const char          * addr     UNUSED_VAROK = 
		address_get_ascii_addr(address);
	    const struct string * fullname = address_get_phrase(address);
	    const struct string * comment  = address_get_comment(address);

	    DPRINT(Debug,6,(&Debug,"addr_to_expanded: addr [%d]: addr=%s",
			    i,addr ? addr : "<NULL>"));

	    if (fullname) {
		const char * A UNUSED_VAROK = get_string_MIME_name(fullname);
		const char * B UNUSED_VAROK = get_string_lang(fullname);

		DPRINT(Debug,6,(&Debug,", fullname=%S",
				fullname));
		DPRINT(Debug,11,(&Debug, " (cs=%s lang=%s)",		    
				 A ? A : "<none>",
				 B ? B : "<none>"));
	    }

	    if (comment) {
		const char * A UNUSED_VAROK = get_string_MIME_name(comment);
		const char * B UNUSED_VAROK = get_string_lang(comment);	    

		DPRINT(Debug,6,(&Debug,", comment=%S",comment));
		DPRINT(Debug,11,(&Debug, " (cs=%s lang=%s)",		    
				 A ? A : "<none>",
				 B ? B : "<none>"));	    
	    }

	    DPRINT(Debug,6,(&Debug,", group=%d\n",group));
	}

	for (i = 0; i < glen;  i++) {
	    const struct string * G = addr_list_get_group(addrs,i);
	    
	    add_group_phrase_(result,G);
	}
	
	for (i = 0; i < alen;  i++) {
	    
	    int group = -1;
	    const struct address * address = 
		addr_list_get_item(addrs,i,&group);	    	   
	    int pos = add_expanded_addr0_(result,address,group);

	    add_textual_addr_(result,NULL,pos,1);
	}
    }

    dump_expanded_address(7,"addr_to_expanded: before build_address_l()",
			  *result);

    /* And make address structures */
    build_address_l(result,mailer_info,aview);

    dump_expanded_address(6,"addr_to_expanded: (result)",*result);
}

void expanded_to_edit_buffer(buffer,expanded)
     struct string ** buffer;
     struct expanded_address expanded;
{
    struct textual *ptr;

    if (*buffer)
	free_string(buffer);

    dump_expanded_address(6,"expanded_to_edit_buffer",expanded);
 
    for (ptr = expanded.surface; 
	 ptr < expanded.surface + expanded.surface_len; 
	 ptr++) {
	if (*buffer) {


	    add_ascii_to_string(*buffer,s2us(", "));

	    append_string(buffer,ptr->Textual,0);
	    
	} else
	    *buffer = dup_string(ptr->Textual);
    }

    if (!*buffer)
	*buffer = new_string(display_charset);

    DPRINT(Debug,6,
	   (&Debug, "expanded_to_edit_buffer: (result) buffer=%S\n",
	    *buffer));
}

void update_textual_from_tokenized(expanded,result,tokenized,aview,buffer)
     const struct expanded_address *expanded;
     struct expanded_address *result;
     struct string_token * tokenized;
     struct AliasView *aview /* Used as flag */;
     const struct string *buffer /* For error message */;
{
    int i, next;
    
    struct textual *ptr = expanded->surface;
    
    free_expanded_address(result);

    for (i = 0; tokenized[i].token; i = next) {
	int NEWPOS = 0;
	struct string * surface = NULL;
	int q= 0, spacecount =  0, q_seen = 0;
	int groupq = 0;

	int Tlen = string_len(tokenized[i].token);
	
	if (result->addrs)
	    NEWPOS = addr_list_item_count (result->addrs);

	DPRINT(Debug,9,
	       (&Debug, 
		"update_textual_from_tokenized: [%d]=\"%S\" %04x\n",
		i,
		tokenized[i].token,
		tokenized[i].special));

	if (0x0020 /* SPACE  */ == tokenized[i].special ||
	    0x002C    /* ',' */ == tokenized[i].special) {
	    next = i+1;
	    continue;
	}
	
	for (next = i; tokenized[next].token; next++) {
	    	    
	    if (0x003C    /* '<' */  == tokenized[next].special) {
		q++;
		q_seen = 1;
	    } else if (0x003E    /* '>' */ == tokenized[next].special)
		q--;
	    else if (!q && !groupq && 
		     0x002C    /* ',' */ == tokenized[next].special)
		break;
	    else if (!q &&
		     0x003A    /* : */ == tokenized[next].special) {
		groupq++;
		q_seen = 1;
	    } else if (!q &&
		     0x003B    /* ; */ == tokenized[next].special)
		groupq--;
	    else if (!q && !groupq && q_seen && 
		     0x0020 /* SPACE  */ != tokenized[next].special &&
		     0x0028 /* '(' */ != tokenized[next].special)
		break;

	    else if (!q && 0x0020 /* SPACE  */  == tokenized[next].special &&
		     tokenized[next+1].token && 
		     0x0028    /* '(' */ == tokenized[next+1].special)
		/* Don't count whitespaces before comments */  ;

	    else if (!q && 0x0020 /* SPACE  */  == tokenized[next].special &&
		     !tokenized[next+1].token)
		/* Don't count whitespaces on end */  ;
	    
	    else if (!q && !groupq && q_seen &&
		     0x0020 /* SPACE  */ == tokenized[next].special)
		break;
	    else if (!q && !groupq && 
		     0x0020 /* SPACE  */ == tokenized[next].special)
		spacecount ++;

	    DPRINT(Debug,9,
		   (&Debug, 
		    "update_textual_from_tokenized+ [%d]=\"%S\" %04x  q=%d groupq=%d\n",
		    next,
		    tokenized[next].token,
		    tokenized[next].special,
		    q,groupq));
	    
	    append_string(&surface,tokenized[next].token,0);
	}

	DPRINT(Debug,9,
	       (&Debug, 
		"update_textual_from_tokenized- spacecount=%d, q=seen=%d\n",
		spacecount,q_seen));

  
	if (surface && 
	    ptr < expanded->surface + expanded->surface_len &&
	    0 == string_cmp(ptr->Textual,
			    surface,
			    -999 /* Unknow values are not equal */
			    )) {
	    int r UNUSED_VAROK;
	    int pos = NEWPOS;
	    int count = 0;
	    int j;
	    /* no change -- copy original */
      
	    for (j = 0; j < ptr->len; j++) {
		int group = -1;
		const struct address * address = 
		     addr_list_get_item(expanded->addrs,ptr->pos+j,&group);
		int p;

		if (group >= 0) {
		    const struct string * grp =
			addr_list_get_group(expanded->addrs,group);

		    
		    p = ADD_EXPANDED_G_p(result,address,grp);
		} else
		    p = ADD_EXPANDED0_p(result,address);

		if (0 == count)
		    pos = p;
		count++;
	    }
	    r=ADD_TEXTUAL_p(result,ptr->Textual,pos,count);
	    DPRINT(Debug,9,
		   (&Debug, 
		    "update_expanded_from_edit_buffer-> [%d] = %S (keep %d addresses)\n",
		    r,surface,count));
	    
	}

	/* Starting with - is alias expansion removing syntax */
	else if (aview &&
		 Tlen > 0 &&
		 give_unicode_from_string(tokenized[i].token,
					  0) == 0x002D /* '-' */ 
		 ) {
	    int r UNUSED_VAROK;
	    struct string * surface1 = NULL;
      
	    DPRINT(Debug,9,
		   (&Debug, 
		    "update_textual_from_tokenized= rescan from %d\n",i));
			   
	    for (next = i; tokenized[next].token; next++) {

		
		if (0x003C /* '<' */ == tokenized[next].special ||
		    0x003E /* '>' */ == tokenized[next].special ||
		    0x002C /* ',' */ == tokenized[next].special ||
		    0x0020 /* SPACE */ == tokenized[next].special)
		    break;

		DPRINT(Debug,9,
		       (&Debug, 
			"update_textual_from_tokenized+ [%d]=\"%S\" %04x\n",
			next,tokenized[next].token,tokenized[next].special));
		
		append_string(&surface1,tokenized[next].token,0);
	    }
	    
	    r=ADD_TEXTUAL_p(result,surface1,NEWPOS,0);
	    DPRINT(Debug,9,
		   (&Debug, 
		    "update_textual_from_tokenized-> [%d] = %S\n",
		    r,surface1));
	    
	    if (surface1)
		free_string(&surface1);
	    
	    /* word <address> OR word : list ; */
	} else if (spacecount <= 1 && q_seen) {
	    int r UNUSED_VAROK = ADD_TEXTUAL_p(result,surface,NEWPOS,0);

	    DPRINT(Debug,9,
		   (&Debug, "update_textual_from_tokenized-> [%d] = %S\n",
		    r,surface));
	} else {
	    struct string * surface1 = NULL;
	    int r UNUSED_VAROK;

	    DPRINT(Debug,9,
		   (&Debug, 
		    "update_textual_from_tokenized= rescan from %d\n",i));
	    
	    for (next = i; tokenized[next].token; next++) {

		
		if (!q && 
		    0x0020 /* SPACE */ == tokenized[next].special &&
		    tokenized[next+1].token && 
		    0x0028 /* '(' */ == tokenized[next+1].special)
		    /* Don't break on whitespaces before comments */  ;
		
		else if (0x003C /* '<' */ == tokenized[next].special ||
			 0x003E /* '>' */ == tokenized[next].special ||
			 0x002C /* ',' */ == tokenized[next].special ||
			 0x0020 /* SPACE */ == tokenized[next].special)
		    break;

		DPRINT(Debug,9,
		       (&Debug, 
			"update_textual_from_tokenized+ [%d]=\"%S\" %04x\n",
			next,
			tokenized[next].token,
			tokenized[next].special));

		append_string(&surface1,tokenized[next].token,0);
	    }

	    if (!surface1) {
		DPRINT(Debug,9,
		       (&Debug, "update_textual_from_tokenized-> parse failure\n"));

		if (buffer) {		
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnparseable,
				      "Unparseable (%S): %S"),
			      tokenized[next].token,buffer);
		}
		
		if (token_parsed == tokenized[next].status)
		    tokenized[next].status = token_fail;
		
		break;
	    }

	    r=ADD_TEXTUAL_p(result,surface1,NEWPOS,0);
	    DPRINT(Debug,9,
		   (&Debug, "update_textual_from_tokenized-> [%d] = %S\n",
		    r,surface1));
	    
	    if (surface1)
		free_string(&surface1);
	}

	if (ptr < expanded->surface + expanded->surface_len)
	    ptr++;
	
	if (i == next) {
	    if (buffer) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnparseable,
				  "Unparseable (%S): %S"),
			  tokenized[next].token,buffer);
	    }

	    if (token_parsed == tokenized[next].status)
		tokenized[next].status = token_fail;
	    
	    next++;
	}

	if (surface)
	    free_string(&surface);
    }

    dump_expanded_address(6,"update_textual_from_tokenized: without build_address_l",
			  *result);    
}


void update_expanded_from_edit_buffer(expanded,buffer,mailer_info,aview)
     struct expanded_address *expanded;
     const struct string     *buffer;
     struct mailer_info      *mailer_info;
     struct AliasView *aview;   /* NULL if aliases should be ignored */
{
    struct expanded_address result;
   
    struct string_token * tokenized = string_tokenize(buffer, TOK_mail);

    zero_expanded_address(&result);

    DPRINT(Debug,6,
	   (&Debug, "update_expanded_from_edit_buffer: buffer=%S\n",
	    buffer));
    dump_expanded_address(6,"update_expanded_from_edit_buffer: (initial)",
			  *expanded);

    update_textual_from_tokenized(expanded,&result,tokenized,aview,
				  buffer);
    
    build_address_l(&result,mailer_info,aview);

    free_expanded_address(expanded);
    *expanded = result;   /* put new (same?) result to place */

    if (tokenized)
	free_string_tokenized(&tokenized);

    dump_expanded_address(6,"update_expanded_from_edit_buffer: (result)",
			  *expanded);
}


void free_addr_util()
{
    DPRINT(Debug,20,(&Debug, "free_addr_util: called\n"));

    /* Now empty ... */
}





int build_address_is_local(domain)
     const char *domain;
{
    const char **mailname_list = give_dt_path_as_elems(&mailname,"mailname");
    
    int is_local = 0;
    
    if (mailname_list) {
	
	int i;
	
	for (i = 0; mailname_list[i]; i++) {
	    
	    if (0 == istrcmp(mailname_list[i],domain)) {
		DPRINT(Debug,12,(&Debug,
				 "build_address_is_local: %s is local [mailname[%d]=%s]\n",
				 domain,i,mailname_list[i]));
		
		is_local = 1;
		break;
	    }
	}
	
    } else {   /* mailname setting overrides hostname and hostfullname */
	
	is_local = 
	    0 == istrcmp(domain,hostfullname) ||
	    0 == istrcmp(domain,hostname);
	
	if (is_local) {
	    DPRINT(Debug,12,(&Debug,
			     "build_address_is_local: %s is local\n",domain));
	}
	
    }

    return is_local;
}

static int build_address_add_address P_((struct expanded_address * result,
					 const char              * addr,
					 const struct string     * fullname,
					 const struct string     * comment,
					 const struct string     * grp ));
static int build_address_add_address(result,addr,fullname,comment,grp)
     struct expanded_address * result;
     const char              * addr;
     const struct string     * fullname;
     const struct string     * comment;
     const struct string     * grp;
{
    charset_t utf7, utf8;
    int p;
    
    if ((utf8 = MIME_name_to_charset("UTF-8",0)) &&
	text_charset != utf8 &&
	0 != (CS_mapping 
	      & charset_properties(text_charset))) {
	struct string * A, *B;
	
	/* Convert strings to text_charset
	   so that utf-8 is not show on
	   MIME encoded words
	*/
	A = convert_string(text_charset,fullname,1);
	B = convert_string(text_charset,comment,1);
	
	p = add_expanded_addr_4(result,addr,A,B,grp);
	free_string(&A);
	free_string(&B);
	
    } else if (utf8 &&
	       convert_utf_header && 
	       (utf7 =  MIME_name_to_charset("UTF-7",0)) &&
	       0 != (CS_mapping & charset_properties(utf7))) {
	struct string * A, *B;
	
	/* If we are using utf-8 as text charset convert it to utf-7 */
	
	A = convert_string(utf7,fullname,1);
	B = convert_string(utf7,comment,1);
	
	p =  add_expanded_addr_4(result,addr,A,B,grp);  
	free_string(&A);
	free_string(&B);
	
    } else
	add_expanded_addr_4(result,addr,fullname,comment,grp);
    
    return p;
}

static int build_address_add_address2 P_((struct expanded_address * result,
					  const struct address    * address,
					  const struct string     * gname ));
static int build_address_add_address2(result,address,gname)
     struct expanded_address * result;
     const struct address    * address; 
     const struct string     * gname;
{
    int p;
    
    const char          * addr     = address_get_ascii_addr(address);
    const struct string * fullname = address_get_phrase(address);
    const struct string * comment  = address_get_comment(address);

    charset_t    do_utf7_fullname = NULL;
    charset_t    do_utf7_comment  = NULL;
    charset_t    do_utf7_gname  = NULL;

    if (fullname)
	do_utf7_fullname = want_convert_to_utf7(fullname);
    if (comment)
	do_utf7_comment  = want_convert_to_utf7(comment);
    if (gname)
	do_utf7_gname = want_convert_to_utf7(gname);
        
    if (do_utf7_fullname || do_utf7_comment || do_utf7_gname) {

	struct string * fullname_x  = NULL;
	struct string * comment_x   = NULL;
	struct string * gname_x   = NULL;

	if (do_utf7_fullname)
	    fullname_x = convert_string(do_utf7_fullname,fullname,1);
	if (do_utf7_comment)
	    comment_x = convert_string(do_utf7_comment,comment,1);
	if (do_utf7_gname)
	    gname_x = convert_string(do_utf7_gname,gname,1);
			
			
	if (gname) 
	    p = add_expanded_addr_4(result,addr,
				    do_utf7_fullname ? fullname_x : fullname,
				    do_utf7_comment  ? comment_x  : comment,
				    do_utf7_gname    ? gname_x    : gname);
	else
	    p = add_expanded_addr_(result,addr,
				   do_utf7_fullname ? fullname_x : fullname,
				   do_utf7_comment  ? comment_x  : comment);
			
	if (fullname_x)
	    free_string(& fullname_x);
	if (comment_x)
	    free_string(& comment_x);
	if (gname_x)
	    free_string(& gname_x);
	
    } else {
	
	if (gname) 
	    p = add_expanded_addr_g(result,address,gname);
	else
	    p = add_expanded_addr0_(result,address,-1);
    }
  
    return p;
}

/* Return 1 if verified */
static int build_address_verify_domain P_((int is_literal,
					   const char * whitelisted_name,
					   const char * domain,
					   const struct string * Textual,
					   const char * addr,
					   char               ** rewrite));
static int build_address_verify_domain(is_literal,
					whitelisted_name,
					domain,Textual,
				       addr,rewrite)
     int                   is_literal;
     const char          * whitelisted_name;
     const char          * domain;
     const struct string * Textual;
     const char          * addr;
     char               ** rewrite;
{
    int ok = 0;

    if (rewrite && *rewrite) {
	free(*rewrite);
	*rewrite = NULL;
    }
    
    if (!whitelisted_name && !is_literal) {
	const char * dummy_name =
	    matches_dummy_domain(domain);
		
	if (dummy_name) {
	    DPRINT(Debug,12,(&Debug,
			     "build_address_verify_domain: Domain %s matches to dummy domain %s\n",
			     domain,dummy_name));

	} else {
				    
	    int val = give_dt_enumerate_as_int(&verify_domain);
	    
	    if (val) {	    
		enum verify_domain_result r;
		char *rewrite2 = NULL;
		struct cancel_data  * main_cancel =
		    build_address_cancel_mode(domain);
		
		DPRINT(Debug,9,(&Debug, 
				"build_address_verify_domain: %s: verify-domain =  %d%s%s, verifying domain %s\n",
				addr,						
				val,
				val  < 0 ? ", uses shared library" : "",
				main_cancel ? ", have cancel mode" : "",
				domain));
		
		r = verify_mail_domain_util(domain,&rewrite2,&main_cancel);
		
		switch (r) {
		case verify_domain_not_found:
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmDomainUnknown,
				      "Domain %s of address %S is unknown."),
			      
			      domain,
			      Textual);
		    break;
		case verify_domain_no_mail:
		    
		    /* RFC 7505:  "Null MX" No Service Resource Record 
		       for Domains That Accept No Mail
		    */
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmDomainNoMail,
				      "Domain %s of address %S accepts no mail."),
			      rewrite2 ? rewrite2 : domain,
			      Textual);
		    break;
		case verify_domain_bad_syntax:
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmInvalidDomain,
				      "Invalid domain %s on address %S."),
			      domain,Textual);
		    break;
		    
		case verify_domain_ok:
		    ok = 1;
		    
		    break;
		case verify_domain_not_available:
		case verify_domain_fallback:
		case verify_domain_failure:
		    break;
		}
		
		
		if (main_cancel)
		    free_cancel(&main_cancel);
		
		if (rewrite) {
		    *rewrite = rewrite2;
		    rewrite2 = NULL;
		} else if (rewrite2) {
		    free(rewrite2);
		    rewrite2 = NULL;
		}
	    
	    } else {
		DPRINT(Debug,14,(&Debug, 
				 "build_address_verify_domain: %s: domain not verified, verify-domain =  %d (disabled)\n",
				 addr,val));
	    }

	}
	    
	
    } else {
	DPRINT(Debug,14,(&Debug, 
			 "build_address_verify_domain: %s: domain not verified",
			 addr));
	
	if (is_literal) {
	    DPRINT(Debug,14,(&Debug, ", literal %s",
			     domain));
	}
	if (whitelisted_name) {
	    DPRINT(Debug,14,(&Debug, ", whitelisted %s",
			     whitelisted_name));
	}				    
	DPRINT(Debug,14,(&Debug, "\n"));				    
    }


    DPRINT(Debug,14,(&Debug, 
		     "build_address_verify_domain=%d\n",ok));

    return ok;
}

int need_verify_ldomain(mailer_info,addr) 
    struct mailer_info      * mailer_info;
    const char              * addr;
{
    int ret = 0;
    
    enum verify_ldomain_v verify_ldomain =
	give_dt_enumerate_as_int(& verify_local_domain);

    if (!addr)
	addr = "(?)";
    
    switch(verify_ldomain) {
    case verify_ldomain_no: break;
	
    case verify_ldomain_remote_mailer:
					
	DPRINT(Debug,9,
	       (&Debug, 
		"need_verify_ldomain: %s: verify-local-domain %d\n", addr,verify_ldomain));
				    
	if (mailer_info &&
	    query_mailer_info(mailer_info,
			      MI_REMOTE_MAILER)) {
	    DPRINT(Debug,9,(&Debug,
			    "need_verify_ldomain: Need verify domain of local address %s, mailer is remote\n",
			    addr));
	    ret = 1;
	}
	break;
	
    case verify_ldomain_yes:

	DPRINT(Debug,9,
	       (&Debug, 
		"need_verify_ldomain: %s: verify-local-domain %d. Need verify domain of local address %s\n",
		addr,verify_ldomain,addr));
	ret = 1;
									    
	break;
	
    case NUM_verify_ldomain:
	break;
    }

    DPRINT(Debug,14,(&Debug,"need_verify_ldomain=%d\n",ret));
    
    return ret;
}

int build_address_l(expanded,mailer_info,aview)
    struct expanded_address * expanded;
    struct mailer_info      * mailer_info;
    struct AliasView        * aview;   /* NULL id aliases should NOT be expanded */
{
    struct textual *ptr;
    int expands = 0;
    struct expanded_address result;
    struct addr_list  *elim_list = NULL;
    charset_t utf8 = MIME_name_to_charset("UTF-8",0);

    
    if (!utf8) 
	panic("CHARSET PANIC",__FILE__,__LINE__,"build_address_l",
	      "UTF-8 not found",0);

    dump_expanded_address(9,"build_address_l",*expanded);

    zero_expanded_address(&result);

    /** 1) Look for addresses to be eliminated from aliases **/

    for (ptr = expanded->surface; 
	 ptr < expanded->surface + expanded->surface_len; 
	 ptr++) {

	if (ptr->Textual) {
	    int l = string_len(ptr->Textual);
	    
	    if (0 == l) {
		DPRINT(Debug,1,
		       (&Debug, 
			"build_address_l: SOFWARE ERROR: [%d]:  textual len %d\n",
			ptr,l));

		continue;
	    }

	    if (0x002D  /* - */ == give_unicode_from_string(ptr->Textual,0)) {
		int POS = 1;
		struct string * eliminated = 
		    clip_from_string(ptr->Textual,&POS,l);

		struct addr_list   * aliases2 = NULL;

		/*  expand alias and eliminate all members */

		if (NULL != (aliases2 = 
			     get_alias_address_expanded(eliminated,mailer_info,aview,
							1 /* Unquote */,NULL))) {

		    if (!elim_list) {
			elim_list = aliases2;
			aliases2 = NULL;
		    } else {
			append_addr_list(elim_list,aliases2);
			free_addr_list(&aliases2);
		    }

		} else {
		    
		    struct string * utf8_temp;
		    unsigned char * utf8_eliminated;
		    struct address * address;

		    /* HACK: We use UTF-8 version of eliminated .... */
		    utf8_temp       = convert_string(utf8,eliminated,0);
		    utf8_eliminated = stream_from_string(utf8_temp,0,NULL);

		    address = new_address(us2s(utf8_eliminated),
					  NULL,NULL);
		    
		    if (!elim_list)
			elim_list = new_empty_addr_list(1);
		    

		    add_address_to_list(elim_list,address,-1);
		    free_address(&address);

		    free_string(&utf8_temp);
		    free(utf8_eliminated);
		}		

		free_string(&eliminated);
	    }	    
	}	
    }

    /* 2) now rebuild list */

    for (ptr = expanded->surface; 
	 ptr < expanded->surface + expanded->surface_len; 
	 ptr++) {

	int NEWPOS = 0;

	struct addr_verify_result verify_result;
	enum mailer_errcode errcode = MAILER_NOT_AVAIL;

	if (result.addrs)
	    NEWPOS = addr_list_item_count(result.addrs);

	if (!ptr->Textual) {
	    
	    if (ptr->len == 1) {
		int group = -1;
		const struct address * address = 
		    addr_list_get_item(expanded->addrs,ptr->pos,&group);

		/* If we have no textual presentation we are converting
		 * from struct address
		 */
		struct string * temp = 
		    make_surface_addr(address);

		int pos;

		if (group >= 0) {
		    const struct string * grp =
			addr_list_get_group(expanded->addrs,group);

		    pos = ADD_EXPANDED_G(result,address,grp);
		} else		
		    pos = ADD_EXPANDED0(result,address);

		ADD_TEXTUAL(result,temp,pos,1);
	
		free_string(&temp);
	    } else {
		DPRINT(Debug,1,
		       (&Debug, 
			"build_address_l: SOFWARE ERROR: [%d]: textual=NULL, len=%d\n",
			ptr - expanded->surface, ptr->len));

		panic("ADDR PANIC",__FILE__,__LINE__,
		      "build_address_l",
		      "textual=NULL",0);
	    }

	} else {  /* have ptr->Textual */

	    struct addr_list * aliases = NULL;
	    enum alias_address_expand_result alias_result = alias_expand_fail;
	    struct string * utf8_temp;
	    unsigned char * utf8_textual;

	    DPRINT(Debug,10,(&Debug,"build_address_l: processing %S\n",
			     ptr->Textual));

	    /* HACK: We use UTF-8 version of Textual .... */

	    utf8_temp    = convert_string(utf8,ptr->Textual,0);
	    utf8_textual = stream_from_string(utf8_temp,0,NULL);

	    if (utf8_textual[0] == '-') {

		/* Is also on result's remove list */
		ADD_TEXTUAL(result,ptr->Textual,NEWPOS,0);

	    } else if (qstrpbrk(us2s(utf8_textual),"!@: <>,()") != NULL) {

		if (ptr->len == 1) {

		    int group = -1;
		    const struct address * address = 
			addr_list_get_item(expanded->addrs,ptr->pos,
					   &group);
					   
		    struct string * temp = make_surface_addr(address);
		    
		    if (temp && 0 == string_cmp(temp,ptr->Textual,999)) {
			int pos;
			/* no change -- preserve original address */


			DPRINT(Debug,10,(&Debug,
					 "build_address_l: ... no change\n"));
			
			if (group >= 0) {
			    const struct string * grp =
				addr_list_get_group(expanded->addrs,group);

			    pos = ADD_EXPANDED_G(result,address,grp);
			} else		       			
			    pos = ADD_EXPANDED0(result,address);
			
			ADD_TEXTUAL(result,temp,pos,1);
		
			free_string(&temp);
	
		    } else {
			if (temp) 
			    free_string(&temp);			
			goto no_surface;
		    }
		    
		} else {

		    /* generate addresses from textual 'surface' 
		       presentation */
		    int pos,count;
		    struct addr_list * address_list;
		    int idx, addr_item_count;

		no_surface:

		    pos = NEWPOS;
		    count = 0;

		    address_list =
			parse_header_address(NULL,us2s(utf8_textual), 
					     /* If user pastes encoded words 
						from somewhere decode them also.
					     */
					     is_rfc1522(us2s(utf8_textual)),
					     utf8,NULL);
		
		    addr_item_count = addr_list_item_count(address_list);

		    for (idx = 0; idx < addr_item_count; idx++) {
			int group = -1;
			const struct address * address = 
			    addr_list_get_item(address_list,idx,&group);

			const char          * addr     = address_get_ascii_addr(address);
			const struct string * fullname = address_get_phrase(address);
			const struct string * comment  = address_get_comment(address);

			int p;
			const char * sep = NULL;

			const struct string * grp = NULL;
			int is_literal = 0;
			const char * whitelisted_name = NULL;
			const char * reserved_name    = NULL;
			int ok_domain                 = 1;
			
			if (group >= 0) 
			    grp = addr_list_get_group(address_list,group);

			DPRINT(Debug,10,(&Debug,"build_address_l: ... got %s\n",
					 addr ? addr : "<<NONE>>"));			


			if (!addr)
			    continue;

			sep = qstrpbrk_c(addr,"!:@");
			
			if (sep) {
			    DPRINT(Debug,9,
				   (&Debug, 
				    "build_address_l: addr %s separator %c\n",
				    addr, *sep));
			}

                        if (sep && '@' == *sep && sep > addr) {

			    int is_local = 0;

			    ok_domain = build_address_classify_domain(sep+1,
								      & is_literal,
								      & whitelisted_name,
								      &  reserved_name );

			    build_address_classify_message(addr,sep+1,
							   is_literal,
							   whitelisted_name,
							   reserved_name);
			   			    
			    
			    if (! ok_domain) {

				DPRINT(Debug,9,
				       (&Debug, 
					"build_address_l: %s: skipping mail domain %s\n",
					addr,sep+1));
				
				continue;
			    }

			    if (build_address_is_local(sep+1)) {
				DPRINT(Debug,12,(&Debug,
						 "build_address_l: %s: %s is local\n",
						 addr,sep+1));
				is_local = 1;
			    }
			    

			    if (is_local) {

				
			    local_address_verify:
				
				if (verify_local_address) {

				    DPRINT(Debug,9,(&Debug,
						     "build_address_l: verifying local address %s\n",
						     addr));

				    if (mailer_info &&
					verify_mailer_domaddr(mailer_info,
							      addr,
							      &verify_result,
							      &errcode)) {
					
					struct address * address1 = NULL;

					DPRINT(Debug,5,
					       (&Debug, 
						"build_address_l: Address %s verified -- really %s\n",
						addr,verify_result.addr));

				       
					address1 = new_address(verify_result.addr,
							       fullname ? fullname : 
							       verify_result.fullname,
							       comment ? comment :
							       verify_result.comment);

					addr_list_replace_item(address_list,
							       idx,address1,
							       group);
					
					free_address(&address1);

					/* Replace invalidates pointers, so reload new pointers */
					
					address   = addr_list_get_item(address_list,idx,&group);
					addr      = address_get_ascii_addr(address);
					fullname  = address_get_phrase(address);
					comment   = address_get_comment(address);

					sep = qstrpbrk_c(addr,"!:@");
					
					if (sep) {
					    DPRINT(Debug,9,
						   (&Debug, 
						    "build_address_l: addr %s separator %c\n",
						    addr, *sep));
					    
					    if ('@' == *sep && sep > addr) {
						ok_domain = build_address_classify_domain(sep+1,
									  & is_literal,
									  & whitelisted_name,
									  &  reserved_name );


						build_address_classify_message(addr,sep+1,
									       is_literal,
									       whitelisted_name,
									       reserved_name);

						if (! ok_domain) {
						    
						    DPRINT(Debug,9,
							   (&Debug, 
							    "build_address_l: %s: skipping mail domain %s\n",
							    addr,sep+1));
						    
						    continue;
						}						
					    }					    					    
					}
										
				    } else {

					DPRINT(Debug,9, (&Debug, 
							 "build_address_l: %s: local verify failed, errcode = %d\n",
							 addr,errcode));
					
					switch (errcode) {
					    
					case MAILER_NOT_AVAIL:
					    lib_error(CATGETS(elm_msg_cat, 
							      ElmSet, 
							      ElmVerifyNot1,
							      "Unable to verify mail address %s"),
						      addr);
					    break;

					case MAILER_NOT_EXIST:
					    lib_error(CATGETS(elm_msg_cat, 
							      ElmSet, ElmVerifyBad1,
							      "Address %s is not e-mail address"),
						      addr);
					    break;

					case MAILER_OK:
					    break;
					}
				    }
				}

				if (sep && '@' == *sep && sep > addr) {


				    if ( need_verify_ldomain(mailer_info,addr)) {
					DPRINT(Debug,9,(&Debug,
							"build_address_l: Need verify domain %s from local address %s\n",
							sep+1,addr));
					goto do_verify_domain;
				    }
				    
				    
				} else {
				    DPRINT(Debug,14, (&Debug, "build_address_l: %s: No domain on local address\n",
						      addr));
				}
				
			    } else {  /* !is_local */
				
				char *rewrite;
				
			    do_verify_domain:
				rewrite = NULL;

				if (build_address_verify_domain(is_literal,whitelisted_name,
								sep+1,ptr->Textual,
								addr,
								&rewrite)) {

				    DPRINT(Debug,5, (&Debug, 
						     "Domain %s of address %S verified",
						     sep+1, ptr->Textual));
				    if (rewrite) {
					DPRINT(Debug,5, (&Debug, " -- really %s",
							 rewrite));
				    }
				    
				    DPRINT(Debug,5, (&Debug, "\n"));
				    
				    if (rewrite) {

					if (0 == strcmp(rewrite, sep+1)) {
					    DPRINT(Debug,10,(&Debug,
							     "build_address_l: Rewriting mail domain %s not needed, canceled.\n",
							     sep+1));
					    
					} else {
					    struct address * address1 = NULL;
					    int len = sep-addr;
					    char * addr1 = 
						elm_message(FRM("%.*s@%s"),
							    len,addr,
							    rewrite);
					    
					    DPRINT(Debug,10,(&Debug,
							     "build_address_l: Rewriting mail domain %s to %s on address %S\n",
							     sep+1,rewrite,ptr->Textual));
					    DPRINT(Debug,10,(&Debug,
							     "build_address_l: Rewriting address %s to %s\n",
							     addr,addr1));
					    
					    address1 = new_address(addr1,fullname,comment);
					    
					    addr_list_replace_item(address_list,
								   idx,address1,
								   group);   
					    
					    free_address(&address1);
					    free(addr1); addr1 = NULL;
					    
					    /* Replace invalidates pointers, so reload new pointers */
					    
					    address   = addr_list_get_item(address_list,idx,&group);
					    addr      = address_get_ascii_addr(address);
					    fullname  = address_get_phrase(address);
					    comment   = address_get_comment(address);
					}
				    }
				}
				
				if (rewrite)
				    free(rewrite);
				
			    }  /* !is_local */
			    
			} else {   /* Local address */

			    if (sep == NULL &&
				mailer_info &&
				query_mailer_info(mailer_info,MI_USE_DOMAIN)) {
				
				char * addr1              = safe_strdup(addr);
				struct address * address1 = NULL;
				
				const char * maildomain = hostfullname;
				const char * have_mailname = useful_mailname();		   
				
				if (have_mailname)
				    maildomain = have_mailname;
				else {
				    DPRINT(Debug,10,(&Debug,
						     "build_address_l: mailname not set or not useful, using hostfullname=%s\n",maildomain));
				}


				DPRINT(Debug,10,(&Debug,
						 "build_address_l: Adding domain %s to address %s\n",
						 maildomain, addr));
			       				
				addr1    = strmcat(addr1,"@");
				addr1    = strmcat(addr1,  maildomain);  

				address1 = new_address(addr1,fullname,comment);

				addr_list_replace_item(address_list,
						       idx,address1,
						       group);   
				
				free_address(&address1);
				free(addr1); addr1 = NULL;

				/* Replace invalidates pointers, so reload new pointers */
				
				address   = addr_list_get_item(address_list,idx,&group);
				addr      = address_get_ascii_addr(address);
				fullname  = address_get_phrase(address);
				comment   = address_get_comment(address);

				sep = qstrpbrk_c(addr,"!:@");
					
				if (sep) {
				    DPRINT(Debug,9,
					   (&Debug, 
					    "build_address_l: addr %s separator %c\n",
					    addr, *sep));

				    if ('@' == *sep && sep > addr) {
					ok_domain = build_address_classify_domain(sep+1,
										  & is_literal,
										  & whitelisted_name,
										  &  reserved_name );

					build_address_classify_message(addr,sep+1,
								       is_literal,
								       whitelisted_name,
								       reserved_name);
					
					if (! ok_domain) {
					    
					    DPRINT(Debug,9,
						   (&Debug, 
						    "build_address_l: %s: skipping mail domain %s\n",
						    addr,sep+1));
					    
					    continue;
					}						
				    }					    					    
				}				
			    }

			    goto local_address_verify;   /* HACK */
			}  /* Local address */

			
			p =  build_address_add_address(&result,
						       addr,fullname,
						       comment,grp);
			
			
			if (0 == count)
			    pos = p;
			count++;

		    } /* for */
		    
		    ADD_TEXTUAL(result,ptr->Textual,pos,count);

		    if (address_list)
			    free_addr_list(&address_list);
		}
	       
	    } else if (aview && (aliases = 
				 get_alias_address_expanded(ptr->Textual,mailer_info,
							    aview,
							    1 /* unquote */,
							    &alias_result  
							    ))) {
		
		int len = addr_list_item_count(aliases);
		int idx;

		int pos = NEWPOS;
		int count = 0;

		DPRINT(Debug,9, (&Debug, "build_address_l: %S: alias_result=%d",
				 ptr->Textual,
				 alias_result));
		
		switch (alias_result) {
		case alias_expand_fail:        DPRINT(Debug,10,(&Debug," alias_expand_fail"));        break;
		case alias_expand_ok:          DPRINT(Debug,10,(&Debug," alias_expand_ok"));
		    expands = 1;
		    break;
		case alias_expand_local_fill:  DPRINT(Debug,10,(&Debug," alias_expand_local_fill"));  break;
		case alias_expand_local_dummy: DPRINT(Debug,10,(&Debug," alias_expand_local_dummy")); break;
		}	
		DPRINT(Debug,9, (&Debug,"\n"));

		for (idx = 0; idx < len; idx++) {
		    int group;
		    const struct address * address = 
			addr_list_get_item(aliases,idx,&group);
		    const char * addr     = address_get_ascii_addr(address);		    
		    const char * sep = NULL;
		    
		    const struct string *gname = NULL;

		    int p;
		   		    
		    if (elim_list) {
			int elim_list_len = addr_list_item_count(elim_list);
			
			int elim_idx;

			/* scan words to be eliminated */
			for (elim_idx = 0; elim_idx < elim_list_len; elim_idx++) {
			    
			    int group = -1;
			    const struct address * elim_address = 
				addr_list_get_item(elim_list,elim_idx,&group);
			    
			    const char  * elim_addr   = address_get_ascii_addr(elim_address);

			    
			    if (addr && elim_addr && 
				0 == strcmp(addr,
					    elim_addr))
				break;
			} 

			if (elim_idx < elim_list_len)
			    /* Is on eliminating list -- don't add */
			    continue;
			
		    }

		    if (addr)
			sep = qstrpbrk_c(addr,"!:@");
		    		    
		    if (sep) {
			DPRINT(Debug,9, (&Debug,  "build_address_l: addr %s separator %c\n",
					 addr, *sep));
			
			if ('@' == *sep && sep > addr) {
			    int is_literal = 0;
			    const char * whitelisted_name = NULL;
			    const char *  reserved_name = NULL;
			    int verify_this_domain = 0;
			    
			    int ok_domain = build_address_classify_domain(sep+1,
									  & is_literal,
									  & whitelisted_name,
									  & reserved_name );
			    
			    build_address_classify_message(addr,sep+1,
							   is_literal,
							   whitelisted_name,
							   reserved_name);
			    
			    if (! ok_domain) {
				
				DPRINT(Debug,9, (&Debug, "build_address_l: %s: skipping mail domain %s\n",
						 addr,sep+1));
				
				continue;
			    }
			    
			    
			    switch (alias_result) {
			    case alias_expand_fail: /* Should not happen */   break;
			    case alias_expand_ok:
				
				if ( verify_alias_domain) {
				    if (build_address_is_local(sep+1)) {
					DPRINT(Debug,12,(&Debug,
							 "build_address_l: %s: %s is local\n",
						     addr,sep+1));
					
					goto handle_as_local;
				    }
				
				    verify_this_domain = 1;
				    
				    DPRINT(Debug,9,(&Debug,
						    "build_address_l: Need verify domain %s from alias address %s\n",
						    sep+1,addr));
				}
				break;
			    case alias_expand_local_fill: 
			    case alias_expand_local_dummy:
			    handle_as_local:

				if (need_verify_ldomain(mailer_info,addr)) {
				    verify_this_domain = 1;
				    
				    DPRINT(Debug,9,(&Debug,
						    "build_address_l: Need verify domain %s from local address %s\n",
						    sep+1,addr));
				}
				break;
			    }
			    
			    if (verify_this_domain) {
				char *rewrite = NULL;

				if (build_address_verify_domain(is_literal,whitelisted_name,
								sep+1,ptr->Textual,
								addr,
								&rewrite)) {

				    DPRINT(Debug,5, (&Debug,  "Domain %s of address or alias %S (%s) verified",
						     sep+1, ptr->Textual,addr));
				    if (rewrite) {
					DPRINT(Debug,5, (&Debug, " -- really %s",
							 rewrite));
				    }				    
				    DPRINT(Debug,5, (&Debug, "\n"));

				    if (rewrite) {
					struct address * address1 = NULL;
					const struct string * fullname = address_get_phrase(address);
					const struct string * comment  = address_get_comment(address);
					
					int len = sep-addr;
					char * addr1 = 
					    elm_message(FRM("%.*s@%s"),
							len,addr,
							rewrite);
					
					DPRINT(Debug,10,(&Debug,
							 "build_address_l: rewriting mail domain %s to %s on address or alias %S\n",
							 sep+1,rewrite,ptr->Textual));
					DPRINT(Debug,10,(&Debug,
							 "build_address_l: rewiting address %s to %s\n",
							 addr,addr1));
					 
					address1 = new_address(addr1,fullname,comment);
					
					addr_list_replace_item(aliases,idx,address1, group);
					free_address(&address1);
					free(addr1); addr1 = NULL;

					/* Replace invalidates pointers, so reload new pointers */
					address   = addr_list_get_item(aliases,idx,&group);
					addr      = address_get_ascii_addr(address);
					fullname  = address_get_phrase(address);
					comment   = address_get_comment(address);
				    }
				}
				if (rewrite)
				    free(rewrite);   
			    }
				
			}
		    }
		    
		    if (group >= 0) {
			gname = addr_list_get_group(aliases,group);
		    }

		    p = build_address_add_address2(&result,address,gname);

		    if (0 == count)
			pos = p;
		    count++;

		}

		free_addr_list(&aliases);

		ADD_TEXTUAL(result,ptr->Textual,pos,count);

	    } else if (mailer_info &&
		       verify_mailer_addr(mailer_info,us2s(utf8_textual),
					  &verify_result,
					  &errcode)) {
		int pos;
		const char * sep = NULL;
		
		DPRINT(Debug,5,
		       (&Debug, 
			"Address %S verified -- really %s\n",
			ptr->Textual,verify_result.addr));

		sep = qstrpbrk_c(verify_result.addr,"!:@");
		if (sep) {
		    DPRINT(Debug,9,
			   (&Debug, 
			    "build_address_l: addr %s separator %c\n",
			    verify_result.addr, *sep));
		    
		    if ('@' == *sep && sep > verify_result.addr) {
			int is_literal = 0;
			const char * whitelisted_name = NULL;
			const char * reserved_name    = NULL;

			int ok_domain = build_address_classify_domain(sep+1,
								      & is_literal,
								      & whitelisted_name,
								      &  reserved_name );
			
			build_address_classify_message(verify_result.addr,
						       sep+1,
						       is_literal,
						       whitelisted_name,
						       reserved_name);
			
			if (! ok_domain) {
			    
			    DPRINT(Debug,9,
				   (&Debug, 
				    "build_address_l: %s: skipping mail domain %s\n",
				    verify_result.addr,sep+1));
			    
			    goto skip_local;
			}

			if (need_verify_ldomain(mailer_info,verify_result.addr)) {

			    char *rewrite = NULL;
			    DPRINT(Debug,9,(&Debug,
					    "build_address_l: Need verify domain %s from local address %s\n",
					    sep+1,verify_result.addr));

			    if (build_address_verify_domain(is_literal,whitelisted_name,
							    sep+1,ptr->Textual,
							    verify_result.addr,
							    &rewrite)) {

				DPRINT(Debug,5, (&Debug,  "Domain %s of address %S (%s) verified",
						 sep+1, ptr->Textual,verify_result.addr));				
				if (rewrite) {
				    DPRINT(Debug,5, (&Debug, " -- really %s",
						     rewrite));
				}				    
				DPRINT(Debug,5, (&Debug, "\n"));

				if (rewrite) {
				    int len = sep-verify_result.addr;
				    char * addr1 = 
					    elm_message(FRM("%.*s@%s"),
							len,verify_result.addr,
							rewrite);

				    DPRINT(Debug,10,(&Debug,
						     "build_address_l: rewriting mail domain %s to %s on address %S\n",
						     sep+1,rewrite,ptr->Textual));

				    DPRINT(Debug,10,(&Debug,
						     "build_address_l: rewiting address %s to %s\n",
						     verify_result.addr,addr1));

				    free(verify_result.addr); verify_result.addr = addr1;
				    addr1 = NULL;
				}

			    }

			    if (rewrite)
				free(rewrite);
			}
			
		    }					    					    
		}
		
		/* Preserve possible comment and fullname
		 *
		 * Use dup_string() to avoid shared pointer so
		 * there no need to make spacial case for free_string()
		 */


		if (1 == ptr->len) {
		    int group = -1;

		    const struct address * address = 
			addr_list_get_item(expanded->addrs,ptr->pos,&group);
		    
		    const struct string * fullname = address_get_phrase(address);
		    const struct string * comment  = address_get_comment(address);

		    if (fullname && ! verify_result.fullname)
			verify_result.fullname = dup_string(fullname);
		    if (comment && ! verify_result.comment)
			verify_result.comment = dup_string(comment);
		}

		if (!verify_result.fullname)
		    verify_result.fullname = new_string(display_charset);
		if (!verify_result.comment)
		    verify_result.comment = new_string(display_charset);
	       
		pos = ADD_EXPANDED(result,verify_result.addr,
				   verify_result.fullname,
				   verify_result.comment);
		ADD_TEXTUAL(result,ptr->Textual,pos,1);


	    skip_local:
		/* ADD_EXPANDED makes copy, so original need to be free'ed 
		 */

		if (verify_result.fullname)
		    free_string(&verify_result.fullname);
		if (verify_result.comment)
		    free_string(&verify_result.comment);
		free(verify_result.addr); verify_result.addr = NULL;
		
	    } else { 

		struct string * gecos  = NULL; 
		struct string * comment = NULL;
		char  *         addr    = NULL;

		int pos; 

		DPRINT(Debug,10,(&Debug,
				 "build_address_l: local address %S verify failed\n",
				 ptr->Textual));

		if (verify_local_address) {
		    switch (errcode) {
		    case MAILER_NOT_AVAIL:
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmVerifyNot,
					  "Unable to verify mail address %S"),
				  ptr->Textual);
			break;
		    case MAILER_NOT_EXIST:
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmVerifyBad,
					  "Address %S is not e-mail address or Elm's alias"),
				  ptr->Textual);
			break;
		    default:
			break;
		    }
		}

		/* Preserve possible comment 
		 *
		 */

		if (1 == ptr->len) {
		    int group = -1;

		    const struct address * address = 
			addr_list_get_item(expanded->addrs,ptr->pos,&group);
		    
		    const struct string * fullname1 = address_get_phrase(address);
		    const struct string * comment1  = address_get_comment(address);
		    
		    if (fullname1)
			gecos = dup_string(fullname1);
		    if (comment1)
			comment = dup_string(comment1);
		}
		
		if (!gecos)
		    gecos = new_string(display_charset);
		if (!comment)
		    comment = new_string(display_charset);

		addr = safe_strdup(us2s(utf8_textual));
		
		if (mailer_info &&
		    query_mailer_info(mailer_info,MI_USE_DOMAIN)) {

		    const char * maildomain = hostfullname;

		    const char * have_mailname = useful_mailname();		   
		    
		    if (have_mailname)
			maildomain = have_mailname;
		    else {
			DPRINT(Debug,10,(&Debug,
					"build_address_l:: mailname not set or not useful, using hostfullname=%s\n"
					,maildomain));
		    }
		    
		    addr = strmcat(addr,"@");
		    addr = strmcat(addr,maildomain);
		}

		pos = ADD_EXPANDED(result,addr,gecos,comment);
		ADD_TEXTUAL(result,ptr->Textual,pos,1);
	    
		if (OPMODE_IS_CHECKMODE(opmode)) 
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmAliasUnknown,
				      "(alias \"%S\" is unknown)"), 
			      ptr->Textual);

		/* ADD_EXPANDED makes copy, so original need to be free'ed
		 */
		free_string(&gecos);
		free_string(&comment);
		free(addr); addr = NULL;
	    }

	    free_string(&utf8_temp);
	    free(utf8_textual);
	}
    }

    free_expanded_address(expanded);
    *expanded = result;   /* put new (same?) result to place */
   
    dump_expanded_address(9,"build_address_l: (result)",*expanded);

    if (elim_list)
	free_addr_list(&elim_list);

    DPRINT(Debug,9,
	   (&Debug,"build_address_l=%d\n",expands));

    return expands;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
