static char rcsid[] = "@(#)$Id: addr_prompt.c,v 2.6 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.6 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_elm.h"
#include "s_elm.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"addr");

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

#define ADDRESS_EDIT_magic	0xEB24

enum ae_mode {
    ae_prompt_area = 0,
    ae_full_page,
    ae_mode_COUNT
}; 

struct address_edit {
    unsigned short magic;  /* ADDRESS_EDIT_magic */

    struct ae_mode_rec {
    
	int line,col;

    } ae_mode[ae_mode_COUNT];
    
    struct AliasView        * aview /* May be NULL */;
    struct mailer_info      * mailer_info;
    struct expanded_address * expanded;
    
    struct string_token    * tokenized_buffer;

    struct string          * title;  /* Title on full page mode */

    struct menu_context     *prompt_area;  /* Original prompt area */

    int                     ae_line;  /* Current line -- -1 = back */
    int                     ae_page;  /* Current page              */
    int                     index_width;
    
    unsigned                confirm_err:1;
    unsigned                tabnotify:1;
    unsigned                enternotify:1;
    unsigned                show_aliases:1;
};

enum address_vector {
    av_question,
    av_buffer,

    av_COUNT
};

static void add_surface P_((struct string_token ** res_tok ,
			    size_t               * res_tok_len,
			    struct string       ** res_buffer,
			    struct string        * surface));
static void add_surface(res_tok,res_tok_len,res_buffer,surface)
     struct string_token ** res_tok;
     size_t               * res_tok_len;
     struct string       ** res_buffer;
     struct string        * surface;
{
    if (surface && string_len(surface) > 0) {
    
	struct string_token * tok =  string_tokenize(surface, TOK_mail);
	size_t x ;
	size_t new_len,i;

	DPRINT(Debug,10, (&Debug,"add_surface: surface=%S\n",
			  surface));
	
	for (x = 0; tok[x].token; x++) {
	    DPRINT(Debug,10, (&Debug,"           :  [%zu]=%S special=%x\n",
			      x,
			      tok[x].token,
			      tok[x].special ));
	}

	new_len = (* res_tok_len) + x;

	*res_tok = safe_array_realloc(*res_tok,new_len,sizeof ((*res_tok)[0]));

	for (i = 0; i < x; i ++) {
	    size_t a = (* res_tok_len) + i;
	    
	    (*res_tok)[a] = tok[i];
	    tok[i].special = 0;
	    tok[i].token   = NULL;
	    tok[i].status  = token_fail;	    
	}
	(* res_tok_len) = new_len;

	append_string(res_buffer,surface,0);

	free(tok); tok = NULL;
    } else {
	DPRINT(Debug,10, (&Debug,"add_surface: Empty surface\n"));
    }
}

/* Return 1 if changed */

static enum exp2tok_res {
    exp2tok_unchanged = 0,
    exp2tok_changed   = 1
    
} expanded_to_tokenized P_((struct string_token ** tokenized,
			    struct string       ** buffer,
			    struct expanded_address expanded));
static enum exp2tok_res expanded_to_tokenized(tokenized,buffer,expanded)
     struct string_token ** tokenized;
     struct string       ** buffer;
     struct expanded_address expanded;

{
    enum exp2tok_res ret = exp2tok_unchanged;
    
    struct textual *ptr;

    int first = 1;

    struct string_token * res_tok      = NULL;
    size_t                res_tok_len  = 0;

    struct string       * res_buffer   = NULL;

    struct string       * comma_str = new_string2(ASCII_SET,s2us(", "));
    size_t x;
    
    int old_tok_ok = NULL != (*tokenized);
    int tok_have_comma = 0;
    int tok_have_tail = 0;

    int changed_tok    = 0;
    int changed_buffer = 0;
    
    dump_expanded_address(10,"expanded_to_tokenized",expanded);

    for (ptr = expanded.surface; 
	 ptr < expanded.surface + expanded.surface_len; 
	 ptr++) {

	if (first)
	    first = 0;
	else {
	    add_surface(&res_tok,&res_tok_len,&res_buffer,comma_str);
	}

	add_surface(&res_tok,&res_tok_len,&res_buffer,ptr->Textual);
	

    }

    free_string(&comma_str);

    res_tok = safe_array_realloc(res_tok, res_tok_len+1,sizeof (res_tok[0]));
    res_tok[res_tok_len].special = 0;
    res_tok[res_tok_len].token   = NULL;
    res_tok[res_tok_len].status  = token_fail;
       
    DPRINT(Debug,10, (&Debug,"expanded_to_tokenized: token count %zu\n",res_tok_len));
        
    for (x = 0; x < res_tok_len || old_tok_ok; x++) {

	if (old_tok_ok && ! (*tokenized)[x].token)
	    old_tok_ok = 0;
	
	if (x < res_tok_len) {
	    DPRINT(Debug,10, (&Debug,"                       [%zu]=%S special=%x\n",
			      x,
			      res_tok[x].token,
			      res_tok[x].special 
			      ));
	    if (old_tok_ok) {

		if (0 != string_cmp(res_tok[x].token,(*tokenized)[x].token,0) ||
		    res_tok[x].special != (*tokenized)[x].special) {
		    DPRINT(Debug,10, (&Debug,"                       [%zu] changed, was %S special=%x\n",
				      x,
				      (*tokenized)[x].token,
				      (*tokenized)[x].special));
		    changed_tok = 1;
		}
		
	    } else {
		DPRINT(Debug,10, (&Debug,"                       [%zu] changed, was new token\n",
				  x));
	    }
	    
	    
	} else if (old_tok_ok) {

	    tok_have_tail = 1;
	    
	    switch((*tokenized)[x].special) {
	    case 0x002C    /* ',' */ :      tok_have_comma = 1;
	    case 0x0020    /* SPACE */   :

		if (!changed_tok) {
		    DPRINT(Debug,10, (&Debug,"                       [%zu] skipped %S special=%x\n",
				      x,
				      (*tokenized)[x].token,
				      (*tokenized)[x].special));
		    break;
		}
		
		/* FALLTHRU */
	    default:
		DPRINT(Debug,10, (&Debug,"                       [%zu] changed_tok, was %S special=%x\n",
				  x,
				  (*tokenized)[x].token,
				  (*tokenized)[x].special));
		changed_tok = 1;
		break;
	    }
	}
    }

    if (tok_have_tail && !tok_have_comma && !changed_tok) {
	DPRINT(Debug,10, (&Debug,"                       [%zu] no comma on tail\n",
			  x));
	changed_tok = 1;
	
    }
    

    if (res_buffer) {
	DPRINT(Debug,10, (&Debug,"                     : buffer=%S\n",res_buffer));

	if (*buffer) {
	    int changed1 = 0 != string_cmp(*buffer,res_buffer,0);

	    if (changed1) {
		
		int p = string_have_prefix(*buffer,res_buffer);

		if (p) {
		    int L = string_len(*buffer);
		    int have_comma = 0;
		    
		    DPRINT(Debug,10, (&Debug,"                     : prefix, was %S\n",*buffer));
		    
		    for (;p < L && !changed_buffer;p++) {
			const uint16 code = give_unicode_from_string(*buffer,p);

			switch (code) {
			case 0x002C    /* ',' */ : have_comma = 1;
			case 0x0020    /* SPACE */   :
			    /* Accept these */
			    break;
			default:
			    DPRINT(Debug,10, (&Debug,"                     : changed (#%d as %x), was %S\n",
					      p,code,*buffer));
			    changed_buffer = 1;
			    break;
			}	       	
		    }

		    if (!have_comma && !changed_buffer) {
			DPRINT(Debug,10, (&Debug,"                     : prefix, no comma on tail\n"));
			changed_buffer = 1;
		    }
					    		    
		} else {
		    DPRINT(Debug,10, (&Debug,"                     : changed, was %S\n",*buffer));
		    changed_buffer = 1;
		}
	    }

	    
	} else
	    changed_buffer = 1;
	
    } else {
	DPRINT(Debug,10, (&Debug,"                     : Empty buffer\n"));
	
	if (*buffer) {
	    int changed1 = 0 < string_len(*buffer);

	    if (changed1) {
		DPRINT(Debug,10, (&Debug,"                     : changed, was %S\n",*buffer));

		changed_buffer = 1;
	    }	    
	}	
    }	

    if (changed_buffer || changed_tok || !*tokenized || !*buffer) {

	DPRINT(Debug,10, (&Debug,
			  "                     : changed%s%s\n",
			  changed_buffer ? " buffer" : "",
			  changed_tok    ? " tokenized" : ""));
	
	if (*tokenized)
	    free_string_tokenized(tokenized);
	*tokenized = res_tok;

	if (*buffer)
	    free_string(buffer);
	*buffer = res_buffer;
	res_buffer = NULL;
	
	if (!*buffer)
	    *buffer = new_string(display_charset);

	ret = 1;
	
    } else {
	DPRINT(Debug,10, (&Debug,
			  "                     : Preserving old tokenized, and buffer=%S\n",
			  *buffer));
	free_string_tokenized(&res_tok);

	if (res_buffer) {
	    DPRINT(Debug,10, (&Debug,
			      "                     : Discarding %S\n",
			      res_buffer));
	    free_string(&res_buffer);
	}
	
    }

    DPRINT(Debug,10, (&Debug,"expanded_to_tokenized=%d",ret));
    switch (ret) {
    case exp2tok_unchanged: DPRINT(Debug,10, (&Debug," exp2tok_unchanged")); break;
    case exp2tok_changed:   DPRINT(Debug,10, (&Debug," exp2tok_changed"));   break;
    }
    DPRINT(Debug,10, (&Debug,"\n"));
    
    return ret;
}

static int ap_tokenized_have_error P_((struct string_token *tokenized));
static int ap_tokenized_have_error(tokenized)
     struct string_token *tokenized;
{
    if (tokenized) {
	size_t x;
	    
	for (x = 0; tokenized[x].token; x++) {
	    switch (tokenized[x].status) {
	    case token_fail:
	    case token_icomplete:
		return 1;
	    case token_parsed:
		break;
	    }
	}
    }
	
    return 0;
}


enum clear_prev_flag { cp_redraw_initial = 0, clear_to_prev, cp_redraw_page };

static void ap_print_prompt P_((struct enter_info *I,
			        enum clear_prev_flag clear_prev,
				int *have_err));
static void ap_print_prompt(I,clear_prev,have_err)
     struct enter_info *I;
     enum clear_prev_flag clear_prev;
     int               * have_err;
{
    struct address_edit * AE;
    int passwd;  /* Not used */
    int append_current;
    int old_line1,old_col1;     /* Old location before prompt was print */
    int start_line,start_col;   /* Position after prompt */
    int end_line,end_col;      /* Position after buffer text */
    enum ae_mode mx;
    int was_error = 0;
    
    if (ENTER_INFO_magic != I->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_print_prompt",
              "Bad magic number (enter_helper)",0);

    AE = I->address;
    if (ADDRESS_EDIT_magic != AE->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_print_prompt",
              "Bad magic number (address_edit)",0);

    menu_GetXYLocation(I->current_page,&old_line1, &old_col1);
    
    if (have_err) {
	*have_err = 0;
    }
    
    if (AE->prompt_area == I->current_page)
	mx = ae_prompt_area;
    else {
	mx = ae_full_page;	
    }
	
    passwd         = ison(I->flags,OE_PASSWD);
    append_current = ison(I->flags,OE_APPEND_CURRENT);

    
    menu_PutLineX(I->current_page,
		  AE->ae_mode[mx].line,
		  AE->ae_mode[mx].col,
		  FRM("%S"),I->pvector[av_question]);

    menu_GetXYLocation(I->current_page,&start_line, &start_col);
    
    if (!passwd) {  /* OE_PASSWD not used */
	int L = 0;
	int len = 0;

	if (I->pvector[av_buffer])
	    L = string_len(I->pvector[av_buffer]);
	    
	if (AE->tokenized_buffer) {
	    int x;
	    
	    for (x = 0; AE->tokenized_buffer[x].token; x++) {
		int f = 0;
		
		switch (AE->tokenized_buffer[x].status) {
		case token_fail: f |= pg_STANDOUT;
		    if (have_err) {
			*have_err = 1;
		    }
		    was_error = 1;
		    break;
		case token_parsed:
		    switch (AE->tokenized_buffer[x].special) {
		    case 0:     /* not special or space or quoted */
		    case 0x0020 /* SPACE  */:
			break;
		    default:
			f |= pg_BOLD;		
			break;
		    }
		    break;
		case token_icomplete: f |=  pg_UNDERLINE;
		    if (have_err) {
			*have_err = 1;
		    }
		    was_error = 1;
		    break;
		}
		menu_StartXX(I->current_page,f);
		menu_Write_to_screen(I->current_page,
				     FRM("%S"),AE->tokenized_buffer[x].token);
		menu_EndXX(I->current_page,f);

		len +=  string_len(AE->tokenized_buffer[x].token);
		
	    }

	    DPRINT(Debug,10, (&Debug,
			      "ap_print_prompt: tokenized_buffer %d items, total len %d\n",
			      x,len));
	}
	
	if (I->pvector[av_buffer] && len < L) {
	    struct string *S;

	    DPRINT(Debug,10, (&Debug,
			      "ap_print_prompt: tokenized_buffer gives %d < buffer %d, printing tail\n",
			      len,L));
	   
	    S = clip_from_string(I->pvector[av_buffer],&len,L);
	    
	    menu_Write_to_screen(I->current_page,FRM("%S"),S);
	    
	    free_string(&S);

	    DPRINT(Debug,10, (&Debug,
			      "ap_print_prompt: .. extended to %d\n",
			      len));

	    if (len != L)
		goto wrong_len;
	    
	} else if (I->pvector[av_buffer] && L != len) {

	wrong_len:
	    
	    DPRINT(Debug,10, (&Debug,
			      "ap_print_prompt: tokenized_buffer gives %d <> buffer %d\n",
			      len,L));
	    
	    menu_PutLineX(I->current_page,
			  AE->ae_mode[mx].line,
			  AE->ae_mode[mx].col,
			  FRM("%S%S"),
			  I->pvector[av_question],
			  I->pvector[av_buffer]);
	}
    }

    menu_GetXYLocation(I->current_page,&end_line, &end_col);
    menu_CleartoEOLN(I->current_page);

    if (AE->confirm_err && !was_error) {
	clear_error();

	DPRINT(Debug,10, (&Debug,
			  "ap_print_prompt: Clearing confirm_err\n"));
	
	AE->confirm_err = 0;
	menu_MoveCursor(I->current_page,end_line, end_col);	
    }
    
    switch (clear_prev) {
	int line;
    case cp_redraw_initial:
	if (!append_current) {
	    menu_MoveCursor(I->current_page,
			    start_line, start_col);
	}
	break;
    case clear_to_prev:
	/* Clear also next lines if cursor was there */

        line = end_line;

	 while (line < old_line1) {
	     line++;
	     menu_ClearLine(I->current_page,line);
	 }

	 menu_MoveCursor(I->current_page,end_line,end_col);
	 break;
    case cp_redraw_page:
	break;
    }    
}
				 
static void ap_update_expanded P_((struct address_edit * AE,
				   struct string *buffer));
static void ap_update_expanded(AE,buffer)
     struct address_edit * AE;
     struct string *buffer;
{
    struct expanded_address result;
    
    if (ADDRESS_EDIT_magic != AE->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_update_expanded",
              "Bad magic number (address_edit)",0);

    zero_expanded_address(&result);

    DPRINT(Debug,10, (&Debug,"ap_update_expanded: buffer=%S\n",buffer));
    
    dump_expanded_address(10,"ap_update_expanded -- old",*( AE->expanded ));
    
    if (AE->tokenized_buffer)
	free_string_tokenized(& (AE->tokenized_buffer));
    
    AE->tokenized_buffer = string_tokenize(buffer, TOK_mail);
    
    update_textual_from_tokenized(AE->expanded,& result,
				  AE->tokenized_buffer,
				  AE->aview, NULL /* No error messages */);

    free_expanded_address(AE->expanded);
    *( AE->expanded ) = result;        

    dump_expanded_address(10,"ap_update_expanded -- new",*( AE->expanded ));
}

static void ap_build_expanded P_((struct address_edit * AE,
				  struct string       * buffer,
				  struct mailer_info  * mailer_info,
				  struct AliasView    * aview));
static void ap_build_expanded(AE,buffer,mailer_info,aview)
     struct address_edit * AE;
     struct string *buffer;
     struct mailer_info      *mailer_info;
     struct AliasView *aview;   /* NULL if aliases should be ignored */
{
    struct expanded_address result;

    if (ADDRESS_EDIT_magic != AE->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_update_expanded",
              "Bad magic number (address_edit)",0);
		
    zero_expanded_address(&result);

    DPRINT(Debug,10, (&Debug,"ap_build_expanded: buffer=%S\n",buffer));
    
    dump_expanded_address(10,"ap_build_expanded -- old",*( AE->expanded ));
    
    if (AE->tokenized_buffer)
	free_string_tokenized(& (AE->tokenized_buffer));
    
    AE->tokenized_buffer = string_tokenize(buffer, TOK_mail);
    
    update_textual_from_tokenized(AE->expanded,&result,
				  AE->tokenized_buffer,
				  AE->aview,

				  buffer /* for error messages */);

    build_address_l(&result,mailer_info,aview);
            
    free_expanded_address(AE->expanded);
    *( AE->expanded ) = result;

    dump_expanded_address(10,"ap_update_expanded -- new",*( AE->expanded ));
}

static void ap_update_tokenized P_((struct address_edit * AE,
				   struct string *buffer));
static void ap_update_tokenized(AE,buffer)
     struct address_edit * AE;
     struct string *buffer;
{
    if (ADDRESS_EDIT_magic != AE->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_update_tokenized",
              "Bad magic number (address_edit)",0);

    DPRINT(Debug,10, (&Debug,"ap_update_tokenized: buffer=%S\n",buffer));
    
    if (AE->tokenized_buffer)
	free_string_tokenized(& (AE->tokenized_buffer));
    
    AE->tokenized_buffer = string_tokenize(buffer, TOK_mail);

    dump_expanded_address(10,"ap_update_tokenized -- clearing",*( AE->expanded ));
    
    /* No parsing now */
    free_expanded_address(AE->expanded);
}

enum ap_redraw_mode { ap_redraw_no_prompt = 0, ap_redraw_full };


static void ap_redraw_full_page P_((struct enter_info *I,
				    struct address_edit * AE,
				    struct menu_context  *base_page,
				    enum ap_redraw_mode  redraw_mode));
static void ap_resize_full_page  P_((struct enter_info *I,
				 struct address_edit * AE,
				 struct menu_context  *base_page));



static void ph_address_edit  P_((struct enter_info *I,
				 enum prompt_hint_mode phm,
				 struct menu_context *base_page));
static void ph_address_edit(I,phm,base_page)
     struct enter_info *I;
     enum prompt_hint_mode phm;
     struct menu_context *base_page;
{
    struct address_edit * AE;

    if (ENTER_INFO_magic != I->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ph_address_edit",
              "Bad magic number (enter_info)",0);

    AE = I->address;
    if (ADDRESS_EDIT_magic != AE->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ph_address_edit",
              "Bad magic number (address_edit)",0);

     DPRINT(Debug,10,(&Debug, "ph_address_edit: phm=%d",
		      phm));
    switch (phm) {
    case phm_clear:
	DPRINT(Debug,10,(&Debug, " phm_clear\n"));
	
	menu_CleartoEOLN(I->current_page);

	break;
	
    case phm_empty_buffer:
	DPRINT(Debug,10,(&Debug, " phm_empty_buffer\n"));

	if (AE->prompt_area == I->current_page &&
	    AE->title && ! AE->show_aliases &&
	    AE->aview && get_total_alias_count(AE->aview) > 0) {
	    
	    menu_Write_to_screen(I->current_page,
				 CATGETS(elm_msg_cat, MeSet,
					 MeAddrPromptHint,
					 "(Enter address or press TAB to open aliases menu.)"));
	}
	
	break;

    default:
	DPRINT(Debug,10,(&Debug, "\n"));
	
	default_prompt_hint(I,phm,base_page);
	break;
    }
    
}


static void fp_address_edit P_((struct enter_info *I,
				enum full_page_action action,
				struct menu_context *base_page));

static void fp_address_edit(I,action,base_page)
    struct enter_info *I;
    enum full_page_action action;
    struct menu_context *base_page;
{
    struct address_edit * AE;
    
    if (ENTER_INFO_magic != I->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "fp_address_edit",
              "Bad magic number (enter_helper)",0);

    AE = I->address;
    if (ADDRESS_EDIT_magic != AE->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "fp_address_edit",
              "Bad magic number (address_edit)",0);


    
    default_full_page(I,action,base_page);

    switch (action) {
    case fp_resized:

	DPRINT(Debug,10, (&Debug, "fp_address_edit: resize on full page\n"));

	ap_resize_full_page(I,AE,base_page);
	break;

    case fp_redraw:
	
	DPRINT(Debug,10, (&Debug, "fp_address_edit: redraw on full page\n"));

	ap_redraw_full_page(I,AE,base_page,ap_redraw_no_prompt);

	break;
    }
    
}


static void ab_address_edit P_((struct enter_info *I,
				struct string **buffer,
				enum alter_buffer_action action,
				struct menu_context *base_page));

static void ab_address_edit(I,buffer,action,base_page)
     struct enter_info *I;
     struct string **buffer;
     enum alter_buffer_action action;
     struct menu_context *base_page;
{
    struct address_edit * AE;
    
    if (ENTER_INFO_magic != I->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ab_address_edit",
              "Bad magic number (enter_helper)",0);

    AE = I->address;
    if (ADDRESS_EDIT_magic != AE->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ab_address_edit",
              "Bad magic number (address_edit)",0);
	    
    switch(action) {
	
    case ab_autocomma:
	ap_update_expanded(AE,*buffer);
			
	goto use_default;
	
    case ab_truncated:
	ap_update_tokenized(AE,*buffer);
	
	goto use_default;
		
    case ab_free_buffer:  /* Only free's buffer */
    case ab_erase:        /* Also initializes empty buffer */

	if (AE->tokenized_buffer)
	    free_string_tokenized(& (AE->tokenized_buffer));
	
	free_expanded_address(AE->expanded);

	/* FALLTHRU */
    default:
    use_default:
	
	default_alter_buffer(I,buffer,action,base_page);

	break;
    }
}

static void ap_resize_full_page(I,AE,base_page)
     struct enter_info   * I;
     struct address_edit * AE;
     struct menu_context * base_page;
{
    if (ENTER_INFO_magic != I->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_resize_full_page",
              "Bad magic number (enter_info)",0);

    if (ADDRESS_EDIT_magic != AE->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_resize_full_page",
              "Bad magic number (address_edit)",0);

    if (I->current_page != AE->prompt_area) {
	int LINES, COLUMNS;
   
	menu_get_sizes(I->current_page,&LINES, &COLUMNS);

	if (LINES > 2) {
	    
	    AE->ae_mode[ae_full_page].line  = LINES-2;
	    AE->ae_mode[ae_full_page].col = 0;

	    menu_trigger_redraw(I->current_page);
	} 
    }
}



static int ap_set_full_page  P_((struct enter_info *I,
				 struct address_edit * AE,
				 struct menu_context  *base_page));
static int ap_set_full_page(I,AE,base_page)
     struct enter_info   * I;
     struct address_edit * AE;
     struct menu_context * base_page;
{
    int r = 1;

    struct menu_context     *new_area = NULL;
    
    
    if (ENTER_INFO_magic != I->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_set_full_page",
              "Bad magic number (enter_info)",0);

     if (ADDRESS_EDIT_magic != AE->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_set_full_page",
              "Bad magic number (address_edit)",0);

     if (AE->prompt_area == I->current_page) {
	 DPRINT(Debug,10,(&Debug, "ap_set_full_page: Creating new menu context\n"));
	 
	 new_area = new_menu_context();
	 
	 r = menu_translate_pos(AE->prompt_area,
				AE->ae_mode[ae_prompt_area].line,
				AE->ae_mode[ae_prompt_area].col,
				new_area,
				& (AE->ae_mode[ae_full_page].line),
				& (AE->ae_mode[ae_full_page].col));

	 if (r) {
	     DPRINT(Debug,10,(&Debug, "ap_set_full_page: Changing current page\n"));
	     
	     I->current_page = new_area;
	     menu_set_default(I->current_page);
	 } 
     } else {
	 r = menu_translate_pos(AE->prompt_area,
				AE->ae_mode[ae_prompt_area].line,
				AE->ae_mode[ae_prompt_area].col,
				I->current_page,
				& (AE->ae_mode[ae_full_page].line),
				& (AE->ae_mode[ae_full_page].col));	 
     }

     if (!r) {
	 DPRINT(Debug,10,(&Debug, "ap_set_full_page: Prompt cordinates out of area\n"));
     }
     
     

     if (new_area && I->current_page != new_area) {
	 DPRINT(Debug,10,(&Debug, "ap_set_full_page: erasing new menu context\n"));
	 
	 erase_menu_context(&new_area);
	 menu_set_default(I->current_page);
     }

     DPRINT(Debug,10,(&Debug, "ap_set_full_page=%d\n",r));
     
     return r;
}


static int ap_lines_per_page P_((int prompt_line));
static int ap_lines_per_page(prompt_line)
     int prompt_line;
{
    int lines_per_page = prompt_line - 6;

    if (lines_per_page < 1) {
	lines_per_page = 1;

	DPRINT(Debug,12, (&Debug,
			  "ap_lines_per_page=%d: prompt_lines=%d\n",
			  lines_per_page,prompt_line));
    }

    return lines_per_page;
    
}

static void ap_draw_line P_((struct enter_info *I,
			     struct address_edit * AE,
			     int idx  /* -1 == back selection */));

static void ap_draw_line(I,AE,idx)
     struct enter_info *I;
     struct address_edit * AE;
     int idx  /* -1 == back selection */;
{
    int lin = 0;
    int L;
    int lines_per_page;
    int cur;
    int LINES, COLUMNS;
    
    struct aliasview_record * entry  = NULL;    
    struct string           * buffer = NULL;
    int                       count = 0;

    int old_line,old_col;
    
    if (ENTER_INFO_magic != I->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_draw_line",
              "Bad magic number (enter_info)",0);

     if (ADDRESS_EDIT_magic != AE->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_draw_line",
              "Bad magic number (address_edit)",0);

     menu_get_sizes(I->current_page,&LINES, &COLUMNS);
     
     L = AE->ae_mode[ae_full_page].line;
     lines_per_page = ap_lines_per_page(L);
     
     if (-1 == idx)
	 lin = 3;
     else
	 lin = 5 + idx % lines_per_page;

     if (lin >= LINES) {
	 DPRINT(Debug,10, (&Debug,
			   "ap_draw_line: idx=%d lin=%d >= LINES=%d\n",
			   idx,lin,LINES));
	 
	 return;
     }
     
     cur = idx == AE->ae_line;

     if (-1 == idx) {

	 buffer =  format_string(CATGETS(elm_msg_cat, MeSet,
					 MeAddrPromptBack,
					 "%s Back (close aliases list)"),
				 (cur && arrow_cursor)? "->" : "  ");
     } else if (AE->aview && (count = get_total_alias_count(AE->aview)) > idx &&
		(entry = give_alias(AE->aview,idx))) {

	 /* Printed index is idx+1 */
	 if (idx >= 9999 &&  AE->index_width < 5) {
	     AE->index_width = 5;
	     DPRINT(Debug,14, (&Debug,
			       "ap_draw_line: ... idx %d index_width %d\n",
			       idx,AE->index_width));
	 } else if (idx >= 999 && AE->index_width < 4) {
	     AE->index_width = 4;
	     DPRINT(Debug,14, (&Debug,
			       "ap_draw_line: ... idx %d index_width %d\n",
			       idx,AE->index_width));
	 }

	 buffer = build_alias_line(entry,idx+1,cur,I->current_page,AE->index_width);	 
     }


     menu_MoveCursor(I->current_page,lin,0);
     
     if (cur && has_highlighting && !arrow_cursor)
	 menu_StartXX(I->current_page,pg_STANDOUT);
     
     
     if (buffer) {
	 menu_Write_to_screen(I->current_page,FRM("%S"),
			      buffer);
     } else {
	 DPRINT(Debug,10, (&Debug,
			   "ap_draw_line: idx=%d lin=%d no line -- alias count = %d",
			   idx,lin,count));

	 if (AE->aview) {
	     DPRINT(Debug,10, (&Debug,", have alias view"));
	 }
	 DPRINT(Debug,10, (&Debug,"\n"));
	 
     }
          
     if (cur && has_highlighting && !arrow_cursor)
	 menu_EndXX(I->current_page,pg_STANDOUT);

     menu_GetXYLocation(I->current_page,&old_line, &old_col);

     if (old_line == lin && old_col < COLUMNS) {

	 if (old_col+50 < COLUMNS && -1 == idx && !cur) {

	     menu_Write_to_screen(I->current_page,CATGETS(elm_msg_cat, MeSet,
							  MeAddrPromptPressEnter,
							  ", or press ENTER to select alias"));
	 }
	 
	 menu_CleartoEOLN(I->current_page);
     }
     
     menu_Write_to_screen(I->current_page,FRM("\r\n"));

     if (buffer)
	 free_string(&buffer);
     
}


static void ap_redraw_full_page(I,AE,base_page, redraw_mode)
     struct enter_info   * I;
     struct address_edit * AE;
     struct menu_context * base_page;
     enum ap_redraw_mode  redraw_mode;
{
    int old_line,old_col;

    struct string * title = NULL;
    
    if (ENTER_INFO_magic != I->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_redraw_full_page",
              "Bad magic number (enter_info)",0);

     if (ADDRESS_EDIT_magic != AE->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "ap_redraw_full_page",
              "Bad magic number (address_edit)",0);
     
     menu_GetXYLocation(I->current_page,&old_line, &old_col);

     /* Reset redraw flag */

     menu_ClearScreen(I->current_page);

     if (AE->confirm_err || AE->tabnotify || AE->enternotify) {
	 
	 DPRINT(Debug,10, (&Debug,
			   "ap_redraw_full_page: Clearing"));
     
	 if (AE->confirm_err) {
	     DPRINT(Debug,10, (&Debug," confirm_err"));
	     AE->confirm_err = 0;
	 }
	 
	 if (AE->tabnotify) {
	     DPRINT(Debug,10, (&Debug,
			       " tabnotify"));
	     AE->tabnotify   = 0;  
	 }
	 
	 if (AE->enternotify) {
	     DPRINT(Debug,10, (&Debug,
			       " enternotify"));
	     AE->enternotify   = 0;  
	 }
	    
	 DPRINT(Debug,10, (&Debug,"\n"));
     }

     if (AE->title)
	 title = format_string(CATGETS(elm_msg_cat, MeSet,
				       MeAddrPromptTitle1,
				       "%S [ELM %s]"),
			       AE->title,version_buff);
     else
	 title = format_string(CATGETS(elm_msg_cat, MeSet,
				       MeAddrPromptTitleVer,
				       "ELM %s"),
			       version_buff);

     menu_StartXX(I->current_page,pg_BOLD);
     menu_print_center(I->current_page,1,title);
     menu_EndXX(I->current_page,pg_BOLD);


     if (AE->show_aliases) {
	 int L = AE->ae_mode[ae_full_page].line;
	 int lines_per_page = ap_lines_per_page(L);
	            
	 ap_draw_line(I,AE,-1);

	 if (AE->aview && L > 6) {
	     int i;
	     
	     for (i = AE->ae_page * lines_per_page;
		  i < AE->ae_page * lines_per_page + lines_per_page;
		  i++) {
		 ap_draw_line(I,AE,i);
	     }
	 
	 } else {
	     DPRINT(Debug,10, (&Debug,
			       "ap_redraw_full_page: No aliases list - prompt line %d\n",
			       L));
	 }
     }
     

     
     
     menu_MoveCursor(I->current_page,old_line,old_col);

     switch (redraw_mode) {
     case ap_redraw_no_prompt:     break;
     case ap_redraw_full:
	 ap_print_prompt(I,cp_redraw_page,NULL);
	 break;
     }

     free_string(&title);
}


static struct string **gb_address_edit P_((struct enter_info *I,
					   enum enter_mode em,
					   struct menu_context  *base_page));
static struct string **gb_address_edit(I,em,base_page)
     struct enter_info   * I;
     enum enter_mode       em;
     struct menu_context * base_page;
{
    const char * s UNUSED_VAROK = NULL;
    struct address_edit * AE;
    enum ae_mode mx UNUSED_VAROK;  /* Not used now */

    if (ENTER_INFO_magic != I->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "gb_address_edit",
              "Bad magic number (enter_info)",0);

    AE = I->address;
    if (ADDRESS_EDIT_magic != AE->magic)
        panic("INPUT PANIC",__FILE__,__LINE__,
              "gb_address_edit",
              "Bad magic number (address_edit)",0);


    DPRINT(Debug,10,(&Debug, "gb_address_edit: em=%d",
		     em));
    if ((s = enter_mode_debug_name(em))) {
	DPRINT(Debug,10,(&Debug, " %s",s));
    }

    if (AE->prompt_area == I->current_page) {
	DPRINT(Debug,10,(&Debug, " on prompt area\n"));
	mx = ae_prompt_area;
    } else {
	
	DPRINT(Debug,10,(&Debug, " on full page\n"));
	mx = ae_full_page;
    }
    
    if (AE->tabnotify) {
	int old_line,old_col;
	    
	menu_GetXYLocation(I->current_page,&old_line, &old_col);
	
	clear_error();
	
	DPRINT(Debug,10, (&Debug,
			  "gb_address_edit: Cleared tabnotify\n"));
	AE->tabnotify = 0;
	
	menu_MoveCursor(I->current_page,old_line,old_col);
    }

    if (AE->enternotify && em_redraw_initial != em) {
	int old_line,old_col;
	    
	menu_GetXYLocation(I->current_page,&old_line, &old_col);
	
	clear_error();
	
	DPRINT(Debug,10, (&Debug,
			  "gb_address_edit: Cleared enternotify\n"));
	AE->enternotify = 0;
	
	menu_MoveCursor(I->current_page,old_line,old_col);
    }
    
    
    switch(em) {
    case em_redraw_initial:  ap_print_prompt(I, cp_redraw_initial,NULL); break;
    case em_redraw:          ap_print_prompt(I, clear_to_prev,NULL);     break;
    case em_enter:	
	
	if (I->pvector[av_buffer]) {
	    int have_error = 0;

	    dump_expanded_address(10,"gb_address_edit -- enter -- old",*( AE->expanded ));
	    
	    ap_update_tokenized(AE,I->pvector[av_buffer]);
	    ap_print_prompt(I, clear_to_prev,&have_error);

	    if (have_error && ! AE->confirm_err) {
		int old_line,old_col;
		
		menu_GetXYLocation(I->current_page,&old_line, &old_col);

		lib_transient(CATGETS(elm_msg_cat, MeSet,
				      MeAddrPromptErrEnter,
				      "Have error. Press enter again to confirm address."));


		DPRINT(Debug,10, (&Debug,
				  "gb_address_edit: Setting confirm_err\n"));
		
		AE->confirm_err = 1;

		menu_MoveCursor(I->current_page,old_line,old_col);	
			
		DPRINT(Debug,10,(&Debug, "gb_address_edit: enter with error ... canceled\n"));
		break;
	    }	    
	}

	if (AE->confirm_err) {
	    int old_line,old_col;
	    
	    menu_GetXYLocation(I->current_page,&old_line, &old_col);
	    clear_error();

	    DPRINT(Debug,10, (&Debug,
			      "gb_address_edit: Clearing confirm_err\n"));
	    
	    AE->confirm_err = 0;
	    menu_MoveCursor(I->current_page,old_line,old_col);		    
	}

	if (ae_full_page == mx && AE->show_aliases) {
	    int count = AE->aview ? get_total_alias_count(AE->aview) : 0;
	    struct aliasview_record * entry  = NULL;    

	    struct expanded_address result;

	    zero_expanded_address(&result);
	    
	    DPRINT(Debug,10,(&Debug, "gb_address_edit: enter .. disabling show aliases\n"));

	    update_textual_from_tokenized(AE->expanded,&result,
					  AE->tokenized_buffer,
					  AE->aview,
					  I->pvector[av_buffer] /* for error message */);
	    
	    free_expanded_address(AE->expanded);
	    *( AE->expanded ) = result;  

	    dump_expanded_address(10,"gb_address_edit -- tabaction -- new",*( AE->expanded ));

	    
	    if (AE->ae_line >= 0 && AE->ae_line < count &&
		(entry = give_alias(AE->aview,AE->ae_line))) {

		DPRINT(Debug,10,(&Debug, "gb_address_edit: ... got alias\n"));

		add_one_alias_to_expanded(AE->expanded,entry);

		dump_expanded_address(10,"gb_address_edit: ... alias added",
				      *( AE->expanded ));

		
		switch (expanded_to_tokenized(& (AE->tokenized_buffer),
					      & (I->pvector[av_buffer]),
					      *( AE->expanded ))) {

		case exp2tok_changed:  /* Expected */

		    /* Moves cursor to correct place */
		    ap_print_prompt(I, clear_to_prev,NULL);
		    break;
		case exp2tok_unchanged:
		    /* No alias collected ? */
		    break;
		}

	    }
	    
	    AE->show_aliases = 0;
	    ap_redraw_full_page(I,AE,base_page, ap_redraw_full);
	    break;
	}
	
	DPRINT(Debug,10,(&Debug, "gb_address_edit: enter .. returning NULL\n"));
	
	return NULL;

    case em_prev:
	    if (ae_full_page == mx && AE->show_aliases) {
		int old_ae_line       = AE->ae_line;
		int old_ae_page       = AE->ae_page;
		int old_line,old_col;
	    
		menu_GetXYLocation(I->current_page,&old_line, &old_col);
		
		if (old_ae_line >= 0) {
		    int L              = AE->ae_mode[ae_full_page].line;
		    int lines_per_page = ap_lines_per_page(L);
		    
		    AE->ae_line--;

		    if (AE->ae_line >= 0)
			AE->ae_page = AE->ae_line / lines_per_page;

		    if (old_ae_page != AE->ae_page) {
			ap_redraw_full_page(I,AE,base_page,ap_redraw_full);
			
			DPRINT(Debug,10,(&Debug,
					 "gb_address_edit: em_prev .. changing line %d => %d, page %d => %d\n",
					 old_ae_line,AE->ae_line,old_ae_page,AE->ae_page));
		    } else {
			ap_draw_line(I,AE,old_ae_line);
			ap_draw_line(I,AE,AE->ae_line);
			
			DPRINT(Debug,10,(&Debug,
					  "gb_address_edit: em_prev .. changing line %d => %d\n",
					  old_ae_line,AE->ae_line));
		    }

		}
		menu_MoveCursor(I->current_page,old_line,old_col);		

	    }
	    
	break;

    case em_next:
	if (ae_full_page == mx && AE->show_aliases) {
	    int old_ae_line       = AE->ae_line;
	    int old_ae_page       = AE->ae_page;
	    int old_line,old_col;
	    
	    int count = AE->aview ? get_total_alias_count(AE->aview) : 0;
	    int L              = AE->ae_mode[ae_full_page].line;
	    int lines_per_page = ap_lines_per_page(L);
	    int N = 0;
	    
	    menu_GetXYLocation(I->current_page,&old_line, &old_col);

	    if (-1 == old_ae_line &&
		count > (N =  old_ae_page * lines_per_page)) {

		AE->ae_line = N;

		ap_draw_line(I,AE,old_ae_line);
		ap_draw_line(I,AE,AE->ae_line);
		
		DPRINT(Debug,10,(&Debug,
				 "gb_address_edit: em_next .. changing line %d => %d (page %d)\n",
				 old_ae_line,AE->ae_line,old_ae_page));
		
	    } else if (count > old_ae_line+1) {
		
		AE->ae_line++;

		if (AE->ae_line >= 0)
		    AE->ae_page = AE->ae_line / lines_per_page;
		
		if (old_ae_page != AE->ae_page) {
		    ap_redraw_full_page(I,AE,base_page,ap_redraw_full);
		    
		    DPRINT(Debug,10,(&Debug,
				     "gb_address_edit: em_next .. changing line %d => %d, page %d => %d\n",
				     old_ae_line,AE->ae_line,old_ae_page,AE->ae_page));
		} else {
		    ap_draw_line(I,AE,old_ae_line);
		    ap_draw_line(I,AE,AE->ae_line);
		    
		    DPRINT(Debug,10,(&Debug,
				     "gb_address_edit: em_next .. changing line %d => %d\n",
				     old_ae_line,AE->ae_line));
		}		
	    }
	    menu_MoveCursor(I->current_page,old_line,old_col);		
	    
	}
	
	break;

    case em_page_up:
	if (ae_full_page == mx && AE->show_aliases) {
	    int old_ae_line       = AE->ae_line;
	    int old_ae_page       = AE->ae_page;
	    int old_line,old_col;
	    
	    menu_GetXYLocation(I->current_page,&old_line, &old_col);

	    if (AE->ae_page > 0) {
		int L              = AE->ae_mode[ae_full_page].line;
		int lines_per_page = ap_lines_per_page(L);
		
		AE->ae_page--;

		if (-1 != old_ae_line &&
		    old_ae_line / lines_per_page != AE->ae_page) {

		    AE->ae_line = AE->ae_page * lines_per_page + lines_per_page -1;

		    ap_redraw_full_page(I,AE,base_page,ap_redraw_full);

		    DPRINT(Debug,10,(&Debug,
				     "gb_address_edit: em_page_up  .. changing line %d => %d, page %d => %d\n",
				     old_ae_line,AE->ae_line,old_ae_page,AE->ae_page));
		    
		} else {
		    ap_redraw_full_page(I,AE,base_page,ap_redraw_full);

		     DPRINT(Debug,10,(&Debug,
				     "gb_address_edit: em_page_up  .. page %d => %d\n",
				      old_ae_page,AE->ae_page));
		}
		
	    } else if (old_ae_line >= 0) {
		AE->ae_line = -1;
		
		ap_draw_line(I,AE,old_ae_line);
		ap_draw_line(I,AE,AE->ae_line);
		
		DPRINT(Debug,10,(&Debug,
				 "gb_address_edit: em_page_up .. changing line %d => %d\n",
				 old_ae_line,AE->ae_line));
		
	    }

	    menu_MoveCursor(I->current_page,old_line,old_col);		
	}
	break;

    case em_page_down:
	if (ae_full_page == mx && AE->show_aliases) {
	    int old_ae_line       = AE->ae_line;
	    int old_ae_page       = AE->ae_page;
	    int old_line,old_col;
	    int count = AE->aview ? get_total_alias_count(AE->aview) : 0;

	    int L              = AE->ae_mode[ae_full_page].line;
	    int lines_per_page = ap_lines_per_page(L);
	    
	    menu_GetXYLocation(I->current_page,&old_line, &old_col);

	    if (count/lines_per_page > old_ae_page) {

		AE->ae_page++;

		if (-1 != old_ae_line &&
		    old_ae_line / lines_per_page != AE->ae_page) {
		    
		    AE->ae_line = AE->ae_page * lines_per_page;

		    ap_redraw_full_page(I,AE,base_page,ap_redraw_full);

		    DPRINT(Debug,10,(&Debug,
				     "gb_address_edit: em_page_down  .. changing line %d => %d, page %d => %d\n",
				     old_ae_line,AE->ae_line,old_ae_page,AE->ae_page));
		    
		} else {
		    ap_redraw_full_page(I,AE,base_page,ap_redraw_full);

		     DPRINT(Debug,10,(&Debug,
				     "gb_address_edit: em_page_down  .. page %d => %d\n",
				      old_ae_page,AE->ae_page));
		}
		
	    } else if (count > old_ae_line+1) {

		AE->ae_line = count-1;

		ap_draw_line(I,AE,old_ae_line);
		ap_draw_line(I,AE,AE->ae_line);
		
		DPRINT(Debug,10,(&Debug,
				 "gb_address_edit: em_page_down .. changing line %d => %d (page %d)\n",
				 old_ae_line,AE->ae_line,old_ae_page));

		
	    }
	    
	    menu_MoveCursor(I->current_page,old_line,old_col);		
	}
	break;
    	    
    case em_tabaction:
	if (I->pvector[av_buffer]) {
	    struct expanded_address result;

	    int have_error = 0;
	    
	    zero_expanded_address(&result);

	    dump_expanded_address(10,"gb_address_edit -- tabaction -- old",*( AE->expanded ));
	    
	    ap_update_tokenized(AE,I->pvector[av_buffer]);
	    ap_print_prompt(I, clear_to_prev,&have_error);

	    if (have_error) {

		DPRINT(Debug,10,(&Debug, "gb_address_edit: error on tokenization\n"));
		
		goto print_error;
	    }
	    
	    update_textual_from_tokenized(AE->expanded,&result,
					  AE->tokenized_buffer,
					  AE->aview,
					  I->pvector[av_buffer] /* for error message */);
	    
	    free_expanded_address(AE->expanded);
	    *( AE->expanded ) = result;  

	    dump_expanded_address(10,"gb_address_edit -- tabaction -- new",*( AE->expanded ));
	    
	    if (ap_tokenized_have_error(AE->tokenized_buffer)) {
		/* Now have error -- reprompt */
		
		ap_print_prompt(I, clear_to_prev,&have_error);

		if (have_error) {
		    int old_line,old_col;

		    DPRINT(Debug,10,(&Debug, "gb_address_edit: error on update texttual\n"));

		    error_wait();
		    
		print_error:
		    
		    menu_GetXYLocation(I->current_page,&old_line, &old_col);
		    
		    lib_transient(CATGETS(elm_msg_cat, MeSet,
					  MeAddrPromptErrNotify,
					  "Address have error."));
		    
		    DPRINT(Debug,10, (&Debug,
				      "gb_address_edit: Setting confirm_err\n"));
		    
		    AE->confirm_err = 1;
		    
		    menu_MoveCursor(I->current_page,old_line,old_col);
		    DPRINT(Debug,10,(&Debug,
				     "gb_address_edit: tabaction with error ... canceled\n"));
		    break;
		}		
	    }

	    switch (expanded_to_tokenized(& (AE->tokenized_buffer),
					  & (I->pvector[av_buffer]),
					  *( AE->expanded ))) {
		int old_line,old_col;
		int count;
		
	    case exp2tok_changed:
		
		ap_print_prompt(I, clear_to_prev,NULL);
			       		
		menu_GetXYLocation(I->current_page,&old_line, &old_col);
		
		AE->tabnotify = 1;

		if (ae_prompt_area == mx && AE->aview && AE->title &&
		    (count = get_alias_count(AE->aview)) > 0
		    ) {
		    
		    lib_transient(CATGETS(elm_msg_cat, MeSet,
					  MeAddrPromptRepeatTAB,
					  "Address rewritten. Press TAB again to open aliases menu."));

		} else {
		
		    lib_transient(CATGETS(elm_msg_cat, MeSet,
					  MeAddrPromptRewrite,
					  "Address rewritten. "));

		}
		
		menu_MoveCursor(I->current_page,old_line,old_col);

		
		DPRINT(Debug,10,(&Debug,
				 "gb_address_edit: tabaction with prompt changed\n"));
		break;

	    case exp2tok_unchanged:
		DPRINT(Debug,10,(&Debug,
				 "gb_address_edit: No changes of buffer on parsing\n"));
		break;
		
	    }
			    
	} else {
	    /* No buffer, so no expanded address either */
	    
	    free_expanded_address(AE->expanded);
	}
	
	if (AE->confirm_err) {
	    int old_line,old_col;
	    
	    menu_GetXYLocation(I->current_page,&old_line, &old_col);
	    clear_error();
	    AE->confirm_err = 0;

	    DPRINT(Debug,10, (&Debug,
			      "gb_address_edit: Clearing confirm_err\n"));
	    
	    menu_MoveCursor(I->current_page,old_line,old_col);
	    
	} else {

	    if (ae_prompt_area == mx && AE->aview && AE->title) {
		int tac = get_total_alias_count(AE->aview);
		int old_line,old_col;

		menu_GetXYLocation(I->current_page,&old_line, &old_col);

		if (tac < 1) {
		    DPRINT(Debug,10,(&Debug,
				     "gb_address_edit: tabaction ... no aliases .. canceled\n"));

		    goto no_aliases;
		}

		sort_aliases(tac, sa_visible_sort,AE->aview);

		menu_MoveCursor(I->current_page,old_line,old_col);
		
		/* Open alias menu */

		if (!ap_set_full_page(I,AE,base_page)) {
		    
		    DPRINT(Debug,10,(&Debug,
				     "gb_address_edit: tabaction with error ... alias menu failed ... canceled\n"));
		    
		no_aliases:

		    AE->tabnotify = 1;
		    
		    lib_transient(CATGETS(elm_msg_cat, MeSet,
					  MeAddrPromptAliasMenuErr,
					  "Alias menu not available"));
		    
		    menu_MoveCursor(I->current_page,old_line,old_col);
		    break;
		}

		AE->show_aliases = 1;
		AE->ae_line = 0;
		AE->ae_page = 0;
		
		ap_redraw_full_page(I,AE,base_page, ap_redraw_full);

		DPRINT(Debug,10,(&Debug,
				 "gb_address_edit: tabaction .. showing aliases\n"));
		break;

	    }

	    if (ae_prompt_area != mx && AE->aview && AE->title && !AE->show_aliases) {
		AE->show_aliases = 1;
		AE->ae_line = 0;
		AE->ae_page = 0;
		
		ap_redraw_full_page(I,AE,base_page, ap_redraw_full);
		
		DPRINT(Debug,10,(&Debug,
				 "gb_address_edit: tabaction .. showing aliases\n"));
	    }

	    if (ae_prompt_area != mx && AE->show_aliases) {
		int old_ae_line       = AE->ae_line;
		int old_line,old_col;
	    
		menu_GetXYLocation(I->current_page,&old_line, &old_col);

		if (-1 == old_ae_line && AE->aview) {
		    int L              = AE->ae_mode[ae_full_page].line;
		    int lines_per_page = ap_lines_per_page(L);

		    AE->ae_line = AE->ae_page * lines_per_page;

		    ap_draw_line(I,AE,old_ae_line);
		    ap_draw_line(I,AE,AE->ae_line);
		    
		    DPRINT(Debug,10,(&Debug,
				     "gb_address_edit: tabaction .. changing line %d => %d\n",
				     old_ae_line,AE->ae_line));
		    
		} else if (old_ae_line >= 0) {
		    AE->ae_line = -1;

		    ap_draw_line(I,AE,old_ae_line);
		    ap_draw_line(I,AE,AE->ae_line);
		    
		    DPRINT(Debug,10,(&Debug,
				     "gb_address_edit: tabaction .. changing line %d => %d\n",
				     old_ae_line,AE->ae_line));
		    
		}

		menu_MoveCursor(I->current_page,old_line,old_col);		
	    }
	    
	}
	
	
	DPRINT(Debug,10,(&Debug, "gb_address_edit: tabaction ignored (now) \n"));
	
	break;
	
    default:
	
	DPRINT(Debug,10,(&Debug, "gb_address_edit: Action ignored (now)\n"));
    }

    DPRINT(Debug,10,(&Debug, "gb_address_edit: Returning non-NULL\n"));
    
    return &(I->pvector[av_buffer]);
}


    
int prompt_expanded_address P_((struct menu_context  *page,
				struct expanded_address *expanded,
				struct mailer_info      *mailer_info,
				struct string          **buffer /* for redraw */,
				struct AliasView *aview /* May be NULL */,
				struct string           * title  /* Title on full page mode */,
				struct menu_context     *prompt_area,
				/* Cordinates use prompt_area */
				int x, int y, int flags, 
				const char * format, const char *msg, ...));

int prompt_expanded_address (
#if ANSI_C			     
			     struct menu_context  *page,
			     struct expanded_address *expanded,
			     struct mailer_info      *mailer_info,
			     struct string          **buffer /* for redraw */,
			     struct AliasView *aview /* May be NULL */,
			     struct string           * title  /* Title on full page mode */,
			     struct menu_context     *prompt_area,
			     int x, int y, int flags, 
			     const char * format, const char *msg, ...
#else
			     page,expanded,mailer_info,buffer,
			     aview,title,
			     prompt_area,
			     x,y,flags,format,msg, va_alist
#endif
			     )
#if !ANSI_C
    struct menu_context  *page;
    struct expanded_address *expanded;
    struct mailer_info      *mailer_info;
    struct string          **buffer /* for redraw */;
    struct AliasView *aview /* May be NULL */;
    struct string               * title  /* Title on full page mode */;
    struct menu_context     *prompt_area;
    int x;
    int y;
    int flags;
    const char *format;
    const char *msg;
    va_dcl     
#endif
{
    int r = 0;
    struct string *question = NULL;

    struct address_edit AE;
    
    struct enter_info INFO;
    struct string *vector[av_COUNT];
    enum ae_mode mx;
    
    va_list vl;

    int confirm_err_loop = 0;
    
    DPRINT(Debug,5, (&Debug, "prompt_expanded_address: Entering [%s]\n",format));

    zero_enter_info(&INFO);
    
    Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */
    question = elm_smessage(0,format,msg,vl);
    va_end(vl);

    if(!(x >=0 && y >= 0))
	menu_GetXYLocation(prompt_area,&x, &y);

    DPRINT(Debug,10, (&Debug, "prompt_expanded_address: (x,y) = %d,%d (line,col)\n",
		      x,y));

    menu_MoveCursor(prompt_area,x,y);

    AE.magic         = ADDRESS_EDIT_magic;

    for (mx = 0; mx < ae_mode_COUNT; mx++) {
	AE.ae_mode[mx].line          = 0;
	AE.ae_mode[mx].col           = 0;
    }
    
    AE.ae_mode[ae_prompt_area].line          = x;
    AE.ae_mode[ae_prompt_area].col           = y;
    AE.aview         = aview;
    AE.mailer_info   = mailer_info;
    AE.expanded      = expanded;
    AE.title         = title;   /* Maybe NULL ? */
    AE.prompt_area   = prompt_area;
    AE.confirm_err   = 0;
    AE.tabnotify     = 0;
    AE.enternotify   = 0;
    AE.show_aliases  = 0;
    AE.ae_line       = 0; /* current line on alias list */
    AE.ae_page       = 0; /* current page on alias list */
    AE.index_width   = 3;

    if (aview) {
	
	int count;

	if (update_aview(aview)) {
	    DPRINT(Debug,10, (&Debug,
			      "prompt_expanded_address: Alias list updated\n"));
	}
	
	count = get_alias_count(aview);

	DPRINT(Debug,10, (&Debug,
			  "prompt_expanded_address: %d aliases\n",
			  count));
	
	if (count >= 10000)
	    AE.index_width = 5;
	else if (count >= 1000)
	    AE.index_width = 4;
    }
    
    AE.tokenized_buffer = NULL;
        
    if (!*buffer) {
	enum exp2tok_res r_tok;
	
	int i;
	
	DPRINT(Debug,10, (&Debug,
			  "prompt_expanded_address: surface len %d",
			  expanded->surface_len));
	for (i = 0; i < expanded->surface_len; i++) {
	    if (expanded->surface[i].Textual) {
		DPRINT(Debug,12, (&Debug,
				  "\nprompt_expanded_address [%d]: %S",
				  i,expanded->surface[i].Textual));
	    }
	}	
	DPRINT(Debug,10, (&Debug,"\n"));

	r_tok = expanded_to_tokenized(& (AE.tokenized_buffer),
				      buffer,
				      *expanded);

	DPRINT(Debug,10, (&Debug,
			  "prompt_expanded_address: expanded_to_tokenized %d",
			  r_tok ));
	switch (r_tok) {
	case exp2tok_unchanged: DPRINT(Debug,10, (&Debug," exp2tok_unchanged")); break;
	case exp2tok_changed:   DPRINT(Debug,10, (&Debug," exp2tok_changed"));   break;
	}
	if (*buffer) {
	    DPRINT(Debug,10, (&Debug, ",  buffer=%S",
			      *buffer));
	}
	DPRINT(Debug,10, (&Debug,"\n"));
	
    } else if (*buffer) {
	DPRINT(Debug,10, (&Debug,
			  "prompt_expanded_address: buffer=%S\n",
			  *buffer));

	AE.tokenized_buffer = string_tokenize(*buffer, TOK_mail);	
    }

    if (AE.tokenized_buffer) {
	size_t a;

	DPRINT(Debug,10, (&Debug,
			  "prompt_expanded_address: tokenized\n"));
	for (a = 0; AE.tokenized_buffer[a].token; a++) {
	    DPRINT(Debug,10, (&Debug, "                       : [%zu]=%S",
			      a,AE.tokenized_buffer[a].token));
	    if (AE.tokenized_buffer[a].special) {
		DPRINT(Debug,10, (&Debug, " special=%x",
				  AE.tokenized_buffer[a].special));
	    }
	    DPRINT(Debug,10, (&Debug,"\n"));
	}
	DPRINT(Debug,10, (&Debug, "                       : count %zu\n",a));
    }
    
    vector[av_question] = question;
    vector[av_buffer] = *buffer;

    
    INFO.address     = &AE;
    INFO.pvector     = vector;

    INFO.give_buffer  = gb_address_edit;
    INFO.alter_buffer = ab_address_edit;
    INFO.full_page    = fp_address_edit;
    INFO.prompt_hint  = ph_address_edit;
    INFO.flags       = flags;
    INFO.ch_count    = 0;
    
    INFO.in_utils    = NULL;
    INFO.builtin     = NULL;
    INFO.browser     = NULL;
    INFO.current_page = prompt_area;

    do {
	int have_err        = 0;
	enum exp2tok_res r1 = exp2tok_unchanged;  /* Not processed */
	
	if (confirm_err_loop) {
	    DPRINT(Debug,10, (&Debug,
			      "prompt_expanded_address: On confirm_err_loop"));

	    if (isoff(INFO.flags,OE_APPEND_CURRENT)) {
		INFO.flags |= OE_APPEND_CURRENT;
		DPRINT(Debug,10, (&Debug,", adding OE_APPEND_CURRENT"));
	    }	    	    
	    DPRINT(Debug,10, (&Debug,"\n"));
	}
	
	r = enter_helper(&INFO, prompt_area,page);
    
	*buffer = vector[av_buffer];

    
	if (0 == r && *buffer) {
	    ap_build_expanded(&AE,*buffer,mailer_info,aview);
	    
	} else if (!*buffer) {
	    free_expanded_address(expanded);
	}
	
	ap_print_prompt(&INFO,clear_to_prev,&have_err);

	if (0 == r && !have_err) {
	    r1 =  expanded_to_tokenized(&AE.tokenized_buffer,&vector[av_buffer],*expanded);

	    *buffer = vector[av_buffer];

	    switch (r1) {
	    case exp2tok_unchanged: break;
	    case exp2tok_changed:
		ap_print_prompt(&INFO,clear_to_prev,&have_err);
		break;
	    }	    
	}
	
	if (0 == r && have_err && !confirm_err_loop && !AE.confirm_err) {
	    int old_line,old_col;
	    
	    menu_GetXYLocation(INFO.current_page,&old_line, &old_col);
	    
	    error_wait();
	    confirm_err_loop++;

	    lib_transient(CATGETS(elm_msg_cat, MeSet,
				  MeAddrPromptErrEnter,
				  "Have error. Press enter again to confirm address."));

	    DPRINT(Debug,10, (&Debug,
			      "prompt_expanded_address: Setting confirm_err\n"));
	    
	    AE.confirm_err = 1;

	    if (AE.tabnotify) {
		DPRINT(Debug,10, (&Debug,
				  "prompt_expanded_address: Cleared tabnotify\n"));
		AE.tabnotify   = 0;  /* Was erased */
	    }

	    if (AE.enternotify) {
		DPRINT(Debug,10, (&Debug,
				  "prompt_expanded_address: Cleared enternotify\n"));
		AE.enternotify   = 0;  /* Was erased */
	    }
	    
	    menu_MoveCursor(INFO.current_page,old_line,old_col);
		
	} else if (exp2tok_changed == r1) {

	    int old_line,old_col;
	    
	    menu_GetXYLocation(INFO.current_page,&old_line, &old_col);
	    
	    error_wait();
	    confirm_err_loop++;

	    lib_transient(CATGETS(elm_msg_cat, MeSet,
				  MeAddrPromptRewriteEnter,
				  "Address rewritten. Press enter again to confirm address."));
	    
	    DPRINT(Debug,10, (&Debug,
			      "prompt_expanded_address: Setting enternotify\n"));
	    
	    AE.enternotify = 1;

	    if (AE.tabnotify) {
		DPRINT(Debug,10, (&Debug,
				  "prompt_expanded_address: Cleared tabnotify\n"));
		AE.tabnotify   = 0;  /* Was erased */
	    }
	    
	} else if (AE.confirm_err || confirm_err_loop || AE.tabnotify || AE.enternotify) {
	    int old_line,old_col;
	    
	    menu_GetXYLocation(INFO.current_page,&old_line, &old_col);
	
	    clear_error();

	    DPRINT(Debug,10, (&Debug,
			      "prompt_expanded_address: Clearing"));
	
	    if (AE.confirm_err) {
		DPRINT(Debug,10, (&Debug," confirm_err"));
		AE.confirm_err = 0;
	    }
	    if (confirm_err_loop) {
		DPRINT(Debug,10, (&Debug," confirm_err_loop"));
		confirm_err_loop = 0;
	    }
	    if (AE.tabnotify) {
		DPRINT(Debug,10, (&Debug,
				  " tabnotify"));
		AE.tabnotify   = 0;  
	    }
	    if (AE.enternotify) {
		DPRINT(Debug,10, (&Debug,
				  " enternotify"));
		AE.enternotify   = 0;  
	    }
	    
	    DPRINT(Debug,10, (&Debug,"\n"));
	
	    menu_MoveCursor(INFO.current_page,old_line,old_col);	
	}

    } while (confirm_err_loop && confirm_err_loop++ < 10);
	
    if (AE.tokenized_buffer)
	free_string_tokenized(& (AE.tokenized_buffer));

    if (AE.title && AE.title != title)
	free_string(& AE.title);

    
    free_string(&question);
    
    DPRINT(Debug,10, (&Debug, "prompt_expanded_address=%d",
		      r));

    switch (r) {
    case REDRAW_MARK:     DPRINT(Debug,10, (&Debug, " REDRAW_MARK"));    break;
    case OE_ALT_SOLIDUS:  DPRINT(Debug,10, (&Debug, " OE_ALT_SOLIDUS")); break;
    }
    if (*buffer) {
	DPRINT(Debug,10, (&Debug,", buffer=%S",
			  *buffer));
    }
    if (expanded->surface_len) {
	int i;
	DPRINT(Debug,10, (&Debug,
			  ", surface len %d",
			  expanded->surface_len));

	for (i = 0; i < expanded->surface_len; i++) {
	    if (expanded->surface[i].Textual) {
		DPRINT(Debug,12, (&Debug,
				  "\nprompt_expanded_address [%d]: %S",
				  i,expanded->surface[i].Textual));
	    }
	}	
    }
    DPRINT(Debug,10, (&Debug, "\n"));
    
    return r;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
