static char rcsid[] = "@(#)$Id: query.c,v 1.24 2022/06/26 14:16:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.24 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elmresolv.h"

/* For USHRT_MAX */
#include <limits.h>

DEBUG_VAR(Debug,__FILE__,"resolv");

#ifndef h_errno
extern int h_errno;
#endif
#include <errno.h>

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
extern int errno;
#endif

#define RESOLV_QUERY_HDR_magic	0xF917

/* This is used for messaging between 
   processes
*/
static const struct resolv_query_hdr {
    unsigned short magic;          /* RESOLV_QUERY_HDR_magic */
    unsigned short buflen;         /* 0 on failure (not -1)  */
    int          errno_val;        /* used on answer */
    int        h_errno_val;        /* used on answer */    
    size_t       querynum;         /* 0 is not used  */
    int         senderpid;
    enum query_msg {
	qm_none          = 0,     
	qm_query         = 1,
	qm_answer        = 2,
	qm_res_send_fail = 3,
	qm_running       = 4,
	qm_shutdown      = 5,
	qm_sizeupdate    = 6,

    }                msg;
    
} NULL_resolv_query_hdr = {
    RESOLV_QUERY_HDR_magic,
    0,0,0,0,-1,qm_none 

};

static struct resolv_query_hdr * new_resolv_query_hdr P_((void))
    UNUSED_FUNCOK;
static struct resolv_query_hdr * new_resolv_query_hdr()
{
    struct resolv_query_hdr *
	ret = safe_zero_alloc(sizeof (*ret));

    /* Struct assigment do not initalize whole 
       struct resolv_query_hdr
    */
    
    *ret = NULL_resolv_query_hdr;
    return ret;
}

union resolv_query_buf_p {
    struct resolv_query_hdr   * hdr;
    char                      * buffer;
};

#define RESOLV_SIZEUPDATE_magic  	0xF91C

static struct resolv_sizeupdate {
    unsigned short magic;           /* RESOLV_SIZEUPDATE_magic */
    unsigned short querysize;
    unsigned short anssize;
} NULL_resolv_sizeupdate = {
    RESOLV_SIZEUPDATE_magic,
    RESOLV_BUFSIZE,
    RESOLV_BUFSIZE
};

union resolv_sizeupdate_p {
    struct resolv_sizeupdate * update;
    unsigned char            * buffer;
};

#define DEFAULT_helper_int_handling  helper_ignore_interrupt

static enum helper_interrupt {
    helper_allow_interrupt,
    helper_ignore_interrupt,
    NUM_helper_interrupt
} helper_int_handling = DEFAULT_helper_int_handling;

#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET)

static enum read_status {
    rs_read_error  =  -3,
    rs_parse_error =  -2,
    rs_EOF         =  -1,
    rs_none        =  0,
    rs_got_message =  1,
    rs_incomplete  =  2,
    rs_continue    =  3 } read_message P_((int socket, struct resolv_query_hdr *result_hdr,
					   unsigned char * buffer, size_t buffer_size,
					   size_t *used_buffer,int *err,int mypid,
					   size_t *continue_pos));
static enum read_status read_message(socket,result_hdr,buffer,buffer_size, used_buffer, err,
				     mypid,continue_pos)
    int socket;
    struct resolv_query_hdr *result_hdr;
    unsigned char * buffer;
    size_t buffer_size;
    size_t *used_buffer;
    int *err;
    int mypid;
    size_t *continue_pos;
{
    enum read_status ret = rs_none;
    struct iovec vec[2];
    struct iovec vecx[2];
    struct msghdr M;
    size_t N = 0;
    int got_eof = 0;
    size_t read_only =  buffer_size;
    union resolv_query_buf_p HDR;
    
    if (err)
	*err = 0;

    HDR.hdr = result_hdr;
    
    /* bzero is defined on hdrs/defs.h */
    bzero((void *)&M, sizeof M);
    
    if (continue_pos && *continue_pos > 0 &&
	( *continue_pos < sizeof (result_hdr->magic) ||
	RESOLV_QUERY_HDR_magic == result_hdr->magic )
	) {

	N = *continue_pos;
	
	SIGDPRINT(Debug,14,(&Debug, "read_message: socket=%d ",
			    socket));
	if (mypid > 0) {
	    SIGDPRINT(Debug,14,(&Debug, "[pid %d] ",mypid));
	}

	SIGDPRINT(Debug,14,(&Debug, "continuing at pos %lu",
			    (unsigned long)N));
			    
	goto continue_middle;
    }
    
    
    bzero((void *)result_hdr,sizeof(*result_hdr));
    
    vec[0].iov_base = HDR.buffer;
    vec[0].iov_len  = sizeof(*result_hdr);
    vec[1].iov_base = buffer;
    vec[1].iov_len  = buffer_size;


    M.msg_name     = NULL;
    M.msg_namelen  = 0;
    M.msg_iov      = &(vec[0]);
    M.msg_iovlen   = 2;
    if (0 == buffer_size)
	M.msg_iovlen = 1;
    M.msg_control     = NULL;
    M.msg_controllen  = 0;
    M.msg_flags       = 0;     

    *used_buffer = 0;
    
     while (N < sizeof(*result_hdr) +  read_only &&
	    rs_none == ret &&
	    !got_eof) {
	int r;
	
	/*  MSG_EOR do not have set on Linux ; However manual page of recvmsg() says:

           "MSG_EOR
              indicates end-of-record; the data returned completed a record (generally used 
              with sockets of type SOCK_SEQPACKET)."
	*/
	
	if (0 != (M.msg_flags & MSG_EOR)) {
	    SIGDPRINT(Debug,14,(&Debug, "read_message: socket=%d ",
				socket));
	    if (mypid > 0) {
		SIGDPRINT(Debug,14,(&Debug, "[pid %d] ",mypid));
	    }
	    SIGDPRINT(Debug,14,(&Debug,
				" end of record, %zu bytes total\n",
				N));
	    break;
	}
	
	r = recvmsg(socket,&M,0);
    
	if (r > 0) {
	    SIGDPRINT(Debug,14,(&Debug, "read_message: socket=%d ",
				socket));
	    if (mypid > 0) {
		SIGDPRINT(Debug,14,(&Debug, "[pid %d] ",mypid));
	    }
	    SIGDPRINT(Debug,14,(&Debug, "got %d bytes ", r));
	    
	    N += r;
	    if (N > r) {
		SIGDPRINT(Debug,14,(&Debug, " (total %zu)",
				    N));
	    }

	    SIGDPRINT(Debug,14,(&Debug, ", flags %d", M.msg_flags));

	continue_middle:
	    
	    if (N < sizeof(*result_hdr)) {
		SIGDPRINT(Debug,14,(&Debug,", not full header (%zu)",
				    sizeof(*result_hdr)));
		
		vecx[0].iov_base = HDR.buffer + N;
		vecx[0].iov_len  = sizeof(*result_hdr) - N;
		vecx[1].iov_base = buffer;
		vecx[1].iov_len  = buffer_size;
		
		M.msg_iov      = &(vecx[0]);
		M.msg_iovlen   = 2;
		if (0 == buffer_size)
		    M.msg_iovlen = 1;
		
	    } else {
		/* (M.msg_flags & MSG_EOR) seems not work? */
		
		if (RESOLV_QUERY_HDR_magic != result_hdr->magic) {
		    SIGDPRINT(Debug,14,(&Debug,", bailing out\n"));
		    break;
		}

		SIGDPRINT(Debug,14,(&Debug,", got full header (%zu)", sizeof(*result_hdr)));

		if (read_only > result_hdr->buflen) {
		    read_only = result_hdr->buflen;
		    SIGDPRINT(Debug,14,(&Debug,", limiting buffer size %zu to %zu",
					buffer_size,
					read_only));
		}
		
		*used_buffer = N - sizeof(*result_hdr);
		
		if (N < sizeof(*result_hdr) +  read_only) {
		    size_t left = sizeof(*result_hdr) +  read_only - N;
		    SIGDPRINT(Debug,14,(&Debug,", not full buffer (%zu), %zu left",
					read_only,
					left));
		    
		    vecx[0].iov_base = buffer + (N - sizeof(*result_hdr));
		    vecx[0].iov_len  = left;
		    M.msg_iov      = &(vecx[0]);
		    M.msg_iovlen   = 1;	        
		}
	    }
		
	    SIGDPRINT(Debug,14,(&Debug,"\n"));
	    
	} else if (0 == r) {
	    got_eof =  1;

	    SIGDPRINT(Debug,14,(&Debug, "read_message: socket=%d ",
				socket));
	    if (mypid > 0) {
		SIGDPRINT(Debug,14,(&Debug, "[pid %d] ",mypid));
	    }
	    SIGDPRINT(Debug,14,(&Debug, "Got eof\n"));
	    
	} else if (-1 == r) {
	    int e = errno;
	    if (err)
		*err = e;
	    
	    if (continue_pos &&
		(EINTR  == e ||
		 EAGAIN == e ||
		 EWOULDBLOCK == e)
		) {
		
		ret = rs_continue;
		*continue_pos = N;
		
	    } else if (e != EINTR)
		ret = rs_read_error;
	    
	    SIGDPRINT(Debug,14,(&Debug, "read_message: socket=%d ",
				socket));
	    if (mypid > 0) {
		SIGDPRINT(Debug,14,(&Debug, "[pid %d] ",mypid));
	    }
	    SIGDPRINT(Debug,14,(&Debug, "error %d: %s\n",
				e,strerror(e)));
	}
    }
	  
    if (rs_none == ret) {
	if (N < sizeof(*result_hdr) ||
	    RESOLV_QUERY_HDR_magic != result_hdr->magic)
	    ret = got_eof ? rs_EOF : rs_parse_error;
	else if (0 != (M.msg_flags & MSG_TRUNC)) {
	    SIGDPRINT(Debug,14,(&Debug, "read_message: socket=%d ",
				socket));
	    if (mypid > 0) {
		SIGDPRINT(Debug,14,(&Debug, "[pid %d] ",mypid));
	    }
	    SIGDPRINT(Debug,14,(&Debug, " truncated, %zu bytes total\n",
				N));
	    ret = rs_incomplete;
	} else if (*used_buffer > result_hdr->buflen)
	    ret = rs_parse_error;
	else if (*used_buffer < result_hdr->buflen)
	    ret = rs_incomplete;
	else
	    ret = rs_got_message;

	if (continue_pos)
	    *continue_pos = 0;
    }
    
    SIGDPRINT(Debug,14,(&Debug, "read_message=%d ",ret));
    switch(ret) {
    case rs_read_error:  SIGDPRINT(Debug,14,(&Debug, "(read error)  ")); break;
    case rs_parse_error: SIGDPRINT(Debug,14,(&Debug, "(parse error) ")); break;
    case rs_EOF:         SIGDPRINT(Debug,14,(&Debug, "(EOF)         ")); break;
    case rs_none:        SIGDPRINT(Debug,14,(&Debug, "(none)        ")); break;
    case rs_got_message: SIGDPRINT(Debug,14,(&Debug, "(got message) ")); break;
    case rs_incomplete:  SIGDPRINT(Debug,14,(&Debug, "(incomplete)  ")); break;
    case rs_continue:    SIGDPRINT(Debug,14,(&Debug, "(continue)    ")); break;
    }
    SIGDPRINT(Debug,14,(&Debug, "socket=%d ",
			socket));
    
    if (mypid > 0) {
	SIGDPRINT(Debug,14,(&Debug, "[pid %d] ",mypid));
    }
    SIGDPRINT(Debug,14,(&Debug, "*used_buffer=%zu",*used_buffer));
    if (continue_pos && *continue_pos > 0) {
	SIGDPRINT(Debug,14,(&Debug, ", *continue_pos=%zu",
			    *continue_pos));
    }
    
    SIGDPRINT(Debug,14,(&Debug, "\n"));
    return ret;
}


static enum write_status {
    ws_write_error  =  -3,
    ws_none         = 0,
    ws_sent_message = 1,
    ws_incomplete   = 2,
    ws_continue     = 3 } write_message P_((int socket, struct resolv_query_hdr *result_hdr,
					    unsigned char * buffer, size_t buffer_len,
					    int *err,int mypid, size_t *continue_pos));

static enum write_status write_message(socket,result_hdr,buffer,buffer_len,err,
				       mypid,continue_pos)
     int socket;
     struct resolv_query_hdr *result_hdr;
     unsigned char * buffer;
     size_t buffer_len;
     int *err;
     int mypid;
     size_t *continue_pos;   
{
    enum write_status ret = ws_none;
    struct iovec vec[2];
    struct iovec vecx[2];
    struct msghdr M;
    size_t N = 0;
    union resolv_query_buf_p HDR;
    
    if ( RESOLV_QUERY_HDR_magic != result_hdr->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "write_message",
	      "Bad magic number (resolv_query_hdr)",0);
    
    if (result_hdr->buflen != buffer_len)
	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "write_message",
	      "Bad buf_len",0);

    if (err)
	*err = 0;

    HDR.hdr = result_hdr;
    
    /* bzero is defined on hdrs/defs.h */
    bzero((void *)&M, sizeof M);
       
    if (continue_pos && *continue_pos > 0) {

	N = *continue_pos;
	
	SIGDPRINT(Debug,14,(&Debug, "write_message: socket=%d ",
			    socket));
	if (mypid > 0) {
	    SIGDPRINT(Debug,14,(&Debug, "[pid %d] ",mypid));
	}

	SIGDPRINT(Debug,14,(&Debug, "continuing at pos %lu",
			    (unsigned long)N));

	goto continue_middle;
    }

    M.msg_name     = NULL;
    M.msg_namelen  = 0;
    M.msg_iov      = &(vec[0]);
    M.msg_control     = NULL;
    M.msg_controllen  = 0;
    M.msg_flags       = MSG_EOR;

    vec[0].iov_base = HDR.buffer;
    vec[0].iov_len  = sizeof(*result_hdr);
    vec[1].iov_base = buffer;
    vec[1].iov_len  = buffer_len;
    M.msg_iovlen   = 2;
    if (0 == buffer_len)
	M.msg_iovlen = 1;


    while (N < sizeof(*result_hdr) +  buffer_len &&
	   ws_none == ret) {
	int r;

	r = sendmsg(socket,&M,MSG_EOR);
	
	if (r > 0) {
	    SIGDPRINT(Debug,14,(&Debug, "write_message: socket=%d ",
				socket));
	    if (mypid > 0) {
		SIGDPRINT(Debug,14,(&Debug, "[pid %d] ",mypid));
	    }
	    SIGDPRINT(Debug,14,(&Debug, "wrote %d bytes ", r));

	    N += r;
	    if (N > r) {
		SIGDPRINT(Debug,14,(&Debug, " (total %zu)",
				    N));
	    }

	continue_middle:
	    
	    if (N < sizeof(*result_hdr)) {
		SIGDPRINT(Debug,14,(&Debug,", not full header (%zu)", sizeof(*result_hdr)));

		vecx[0].iov_base = HDR.buffer + N;
		vecx[0].iov_len  = sizeof(*result_hdr) - N;
		vecx[1].iov_base = buffer;
		vecx[1].iov_len  = buffer_len;
		
		M.msg_iov      = &(vecx[0]);
		M.msg_iovlen   = 2;
		if (0 == buffer_len)
		    M.msg_iovlen = 1;
		
	    } else if (N < sizeof(*result_hdr) + buffer_len) {
		size_t left = sizeof(*result_hdr) +  buffer_len - N;
		SIGDPRINT(Debug,14,(&Debug, ", not full buffer (%zu), %zu left",
				    buffer_len,
				    left));
		
		vecx[0].iov_base = buffer + (N - sizeof(*result_hdr));
		vecx[0].iov_len  = left;
		M.msg_iov      = &(vecx[0]);
		M.msg_iovlen   = 1;	 
	    }
	    SIGDPRINT(Debug,14,(&Debug, "\n"));
	    
	} else if (-1 == r) {
	    int e = errno;
	    if (err)
		*err = e;

	    if (continue_pos &&
		(EINTR  == e ||
		 EAGAIN == e ||
		 EWOULDBLOCK == e)
		) {
		
		ret = rs_continue;
		*continue_pos = N;
	    } else if (e != EINTR)
		ret =  ws_write_error;
	    
	    SIGDPRINT(Debug,14,(&Debug, "write_message: socket=%d ",
				socket));
	    if (mypid > 0) {
		SIGDPRINT(Debug,14,(&Debug, "[pid %d] ",mypid));
	    }
	    SIGDPRINT(Debug,14,(&Debug, "error %d: %s\n",
				e,strerror(e)));
	}
    }

    if (ws_none == ret) {
	SIGDPRINT(Debug,14,(&Debug, "write_message: socket=%d ",
			    socket));
	if (mypid > 0) {
	    SIGDPRINT(Debug,14,(&Debug, "[pid %d] ",mypid));
	}
	if (N ==  sizeof(*result_hdr) + buffer_len) {
	    ret = ws_sent_message;
	    SIGDPRINT(Debug,14,(&Debug, "complete"));
	} else if (N < sizeof(*result_hdr) + buffer_len) {
		SIGDPRINT(Debug,14,(&Debug, "incomplete (need %zu bytes)",
				    (sizeof(*result_hdr) + buffer_len)));
		ret = ws_incomplete;
	}
	SIGDPRINT(Debug,14,(&Debug, " %zu bytes total\n",
			    N));

	if (continue_pos)
	    *continue_pos = 0;
    }
    
    SIGDPRINT(Debug,14,(&Debug, "write_message=%d ",ret));
    switch(ret) {
    case ws_write_error:  SIGDPRINT(Debug,14,(&Debug, "(write error)  ")); break;
    case ws_none:         SIGDPRINT(Debug,14,(&Debug, "(none)         ")); break;
    case ws_sent_message: SIGDPRINT(Debug,14,(&Debug, "(sent message) ")); break;
    case ws_incomplete:   SIGDPRINT(Debug,14,(&Debug, "(incomplete)   ")); break;
    case ws_continue:     SIGDPRINT(Debug,14,(&Debug, "(continue)     ")); break;	
    }
    SIGDPRINT(Debug,14,(&Debug, "socket=%d ",
			socket));
    
    if (mypid > 0) {
	SIGDPRINT(Debug,14,(&Debug, "[pid %d] ",mypid));
    }

    if (continue_pos && *continue_pos > 0) {
	SIGDPRINT(Debug,14,(&Debug, ", *continue_pos=%lu",
			    (unsigned long)*continue_pos));
    }
    SIGDPRINT(Debug,14,(&Debug,"\n"));
    
    return ret;
}


static void resolv_send_process P_((int socket));
static void resolv_send_process(socket)
     int socket;
{
    int pid = getpid();
    struct resolv_query_hdr hdr;
    int r;
    enum write_status ws;
    int err = 0;
    unsigned char query_buffer1[RESOLV_BUFSIZE];
    unsigned char answer_buffer1[RESOLV_BUFSIZE];

    unsigned char * query_buffer = query_buffer1;
    size_t          query_buffer_size = sizeof query_buffer1;

    unsigned char * answer_buffer = answer_buffer1;
    size_t          answer_buffer_size = sizeof answer_buffer1;
    
    size_t query_buflen;
    enum read_status rs;
    
    bzero((void *)&hdr,sizeof(hdr));

    SIGDPRINT(Debug,12,(&Debug, "resolv_send_process: [pid %d] Starting\n",pid));

    /* Disable most of installed signal handlers */
    switch (helper_int_handling) {
    case helper_allow_interrupt:  signal(SIGINT,  SIG_DFL); break;
    case helper_ignore_interrupt: signal(SIGINT,  SIG_IGN); break;
    case NUM_helper_interrupt:    /* Not used */            break;
    }
    signal(SIGQUIT, SIG_DFL);
    signal(SIGTERM, SIG_DFL);
    signal(SIGHUP,  SIG_DFL);
    signal(SIGUSR1, SIG_DFL);
    signal(SIGUSR2, SIG_DFL);
    signal(SIGALRM, SIG_DFL);
    signal(SIGPIPE, SIG_DFL);
#ifdef SIGTSTP
    signal(SIGTSTP, SIG_DFL);
    signal(SIGCONT, SIG_DFL);
#endif
#ifdef SIGWINCH
    signal(SIGWINCH, SIG_DFL);
#endif
	   
    errno   = 0;
    h_errno = 0;

    hdr =  NULL_resolv_query_hdr;
    hdr.senderpid   = pid;
    
    r = res_init();
    if (r < 0) {
	hdr.errno_val   = errno;
	hdr.h_errno_val = h_errno;
	hdr.msg         = qm_shutdown;
	
	SIGDPRINT(Debug,12,(&Debug,
			    "resolv_send_process: [pid %d] res_init() failed, errno=%d h_errno=%d\n",
			    pid,errno,h_errno));
	
	ws = write_message(socket,&hdr,NULL,0,&err,pid,NULL);
	return;
    } else if (0 == r) {
	hdr.msg         = qm_running;
	ws = write_message(socket,&hdr,NULL,0,&err,pid,NULL);
	if (ws_sent_message != ws)
	    goto quitting;
    }

    while (rs_got_message == (rs = read_message(socket,&hdr,
						query_buffer,
						query_buffer_size,
						&query_buflen,
						&err,pid,NULL))) {

	SIGDPRINT(Debug,14,(&Debug, "resolv_send_process: [pid %d] got message %d",
			    pid,hdr.msg));
	switch(hdr.msg) {
	case qm_none:          SIGDPRINT(Debug,14,(&Debug, " (none)"));          break;
	case qm_query:         SIGDPRINT(Debug,14,(&Debug, " (query)"));         break;
	case qm_answer:        SIGDPRINT(Debug,14,(&Debug, " (answer)"));        break;
	case qm_res_send_fail: SIGDPRINT(Debug,14,(&Debug, " (res_send fail)")); break;
	case qm_running:       SIGDPRINT(Debug,14,(&Debug, " (running)"));       break;
	case qm_shutdown:      SIGDPRINT(Debug,14,(&Debug, " (shutdown)"));      break;
	case qm_sizeupdate:    SIGDPRINT(Debug,14,(&Debug, " (sizeupdate)"));    break;
	}
	if (hdr.senderpid > 0) {
	    SIGDPRINT(Debug,14,(&Debug, ", sender pid %d",hdr.senderpid));
	}
	SIGDPRINT(Debug,14,(&Debug, "\n"));
	
	switch(hdr.msg) {
	    int    anslen;

	    union resolv_sizeupdate_p size_update;
	    
	case qm_none:
	    break;
	    
	case qm_query:

	    errno   = 0;
	    h_errno = 0;
	    anslen = res_send(query_buffer,query_buflen,
			      answer_buffer, answer_buffer_size);

	    hdr.magic       = RESOLV_QUERY_HDR_magic;
	    hdr.errno_val   = errno;
	    hdr.h_errno_val = h_errno;
	    hdr.senderpid   = pid;
	    
	    if (anslen < 0) {
		err = errno;	       
		hdr.buflen      = 0;
		hdr.msg         = qm_res_send_fail;

		SIGDPRINT(Debug,14,(&Debug,
				    "resolv_send_process: [pid %d] res_send() failed, errno=%d h_errno=%d\n",
				    pid,errno,h_errno));
		
		ws = write_message(socket,&hdr,NULL,0,&err,pid,NULL);
		if (ws_sent_message != ws)
		    goto quitting;
	    } else if (anslen <= USHRT_MAX) {
		hdr.buflen      = anslen;
		hdr.msg         = qm_answer;

		SIGDPRINT(Debug,14,(&Debug,
				    "resolv_send_process: [pid %d] res_send() succeed, answer %d bytes\n",
				    pid,anslen));
		
		ws = write_message(socket,&hdr,answer_buffer,anslen,&err,pid,NULL);
		if (ws_sent_message != ws)
		    goto quitting;
	    } else {
		SIGDPRINT(Debug,14,(&Debug,
				    "resolv_send_process: [pid %d] res_send() succeed, answer %d bytes, too long, truncated to %d\n",
				    pid,anslen,USHRT_MAX));
		
		hdr.buflen      = USHRT_MAX;
		hdr.msg         = qm_answer;
		hdr.errno_val   = EMSGSIZE;
		ws = write_message(socket,&hdr,answer_buffer,USHRT_MAX,&err,pid,NULL);
		if (ws_sent_message != ws)
		    goto quitting;
	    }
	    break;
	case qm_running:
	    break;
	case qm_answer:
	case qm_res_send_fail:
	case qm_shutdown:
	    goto quitting;
	case qm_sizeupdate:
	    size_update.buffer = query_buffer;

	    if (query_buflen == sizeof (struct resolv_sizeupdate) &&
		RESOLV_SIZEUPDATE_magic == size_update.update->magic) {

		if (size_update.update->querysize >= sizeof (struct resolv_sizeupdate)) {

		    if (query_buffer_size != size_update.update->querysize) {
		    
			SIGDPRINT(Debug,14,(&Debug, "resolv_send_process: [pid %d] Changing querysize %lu => %d\n",
					    pid,(unsigned long)query_buffer_size,size_update.update->querysize));
			
			query_buffer_size = size_update.update->querysize;
			if (query_buffer == query_buffer1)
			    query_buffer = safe_malloc(query_buffer_size);
			else
			    query_buffer = safe_realloc(query_buffer,query_buffer_size);
		    }
		    
		} else {
		    SIGDPRINT(Debug,14,(&Debug, "resolv_send_process: [pid %d] Bad size update message, querysize %d\n",
					pid,size_update.update->querysize));
		}
		

		if (size_update.update->anssize >= sizeof (struct resolv_sizeupdate)) {

		    if (answer_buffer_size != size_update.update->anssize) {
			SIGDPRINT(Debug,14,(&Debug, "resolv_send_process: [pid %d] Changing anssize  %lu => %d\n",
					    pid,(unsigned long)answer_buffer_size,size_update.update->anssize));
			
			answer_buffer_size = size_update.update->anssize;
			if (answer_buffer == answer_buffer1)
			    answer_buffer = safe_malloc(answer_buffer_size);
			else
			    answer_buffer = safe_realloc(answer_buffer,answer_buffer_size);
		    }
		    
		} else {
		    SIGDPRINT(Debug,14,(&Debug, "resolv_send_process: [pid %d] Bad size update message, anssize %d\n",
					pid,size_update.update->anssize));
		}
		
	    } else {
		SIGDPRINT(Debug,14,(&Debug, "resolv_send_process: [pid %d] Bad size update message\n",
				    pid));
	    }
	    break;
	}
    }

    SIGDPRINT(Debug,14,(&Debug, "resolv_send_process: [pid %d] got %d",
			pid,rs));
    switch(rs) {
    case rs_read_error:  SIGDPRINT(Debug,14,(&Debug, " (read error, error=%d)  ",err)); break;
    case rs_parse_error: SIGDPRINT(Debug,14,(&Debug, " (parse error) ")); break;
    case rs_EOF:         SIGDPRINT(Debug,14,(&Debug, " (EOF)         ")); break;
    case rs_none:        SIGDPRINT(Debug,14,(&Debug, " (none)        ")); break;
    case rs_got_message: SIGDPRINT(Debug,14,(&Debug, " (got message) ")); break;
    case rs_incomplete:  SIGDPRINT(Debug,14,(&Debug, " (incomplete)  ")); break;
    case rs_continue:    SIGDPRINT(Debug,14,(&Debug, " (continue)  ")); break;
    }
    SIGDPRINT(Debug,14,(&Debug, "\n"));
    
    
 quitting:
    hdr.magic       = RESOLV_QUERY_HDR_magic;
    hdr.errno_val   = err;
    hdr.h_errno_val = 0;
    hdr.msg         = qm_shutdown;
    hdr.buflen      = 0;
    hdr.senderpid   = pid;
    ws = write_message(socket,&hdr,NULL,0,&err,pid,NULL);

    SIGDPRINT(Debug,12,(&Debug, "resolv_send_process: [pid %d] Stopping\n",pid));

    if (query_buffer != query_buffer1)
	free(query_buffer);
    if (answer_buffer != answer_buffer1)
	free(answer_buffer);
}

#endif

    
#define RESOLV_MESSAGE_magic    0xF918

static struct resolv_message {
    unsigned short  magic;        /* RESOLV_MESSAGE_magic */
    unsigned short  message_len;
    unsigned char * message;
    size_t          message_size;   /* alloced space of message for sizeupdate */
    ns_msg          parsed_message;

    unsigned int    message_parsed  :1;
} *  new_resolv_message P_((void)) UNUSED_FUNCOK;

static struct resolv_message * new_resolv_message()
{
    struct resolv_message * ret =
	safe_zero_alloc(sizeof (*ret));

    ret->magic          = RESOLV_MESSAGE_magic;
    ret->message_len    = 0;
    ret->message        = NULL;
    ret->message_size   = 0;
    ret->message_parsed = 0;
    
    return ret;
}

static void resize_resolv_message P_((struct resolv_message *msg,
				      size_t newsize))
    UNUSED_FUNCOK;
static void resize_resolv_message(msg,newsize)
     struct resolv_message *msg;
     size_t newsize;
{
    size_t old_size;
    
    if (RESOLV_MESSAGE_magic != msg->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "resize_resolv_message",
	      "Bad magic number (resolv_message)",0);

    old_size = msg->message_size;
    
    msg->message_size = newsize;
    msg->message = safe_realloc(msg->message,
				msg->message_size);
    msg->message_len    = 0;
    msg->message_parsed = 0;

    if (old_size < msg->message_size)
	bzero(msg->message+old_size,
	      msg->message_size - old_size);
}


static void free_resolv_message P_((struct resolv_message **msg))
     UNUSED_FUNCOK;
static void free_resolv_message(msg)
     struct resolv_message **msg;
{
    if (RESOLV_MESSAGE_magic != (*msg)->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "free_resolv_message",
	      "Bad magic number (resolv_message)",0);

    if ((*msg)->message) {
	free ((*msg)->message);
	(*msg)->message = NULL;
    }

    (*msg)->magic = 0;  /* Invalidate */
    free(*msg);
    *msg = NULL;
}

#define RESOLV_QUERY_ITEM_magic	0xF919

static struct resolv_query_item {
    unsigned short magic;       /* RESOLV_QUERY_ITEM_magic */

    size_t                    querynum;
    struct schedule_timelimit query_time;
    char                    * name;               /* name for error messages */
    struct resolv_message   * query_buffer;
    struct resolv_query_hdr * query_hdr;
    
    struct resolv_message   * answer_buffer;
    struct resolv_query_hdr * answer_hdr;

    int                       last_errno;

    struct resolv_cache      * cache_answer_in;    /* 'answer' for fill_resolv_cache()     */
    struct resolv_cache      * cache_answer_ret;   /* Return value of  fill_resolv_cache() */
    char                     * answer_search_name; /* Return 'search_name' for fill_resolv_cache() */
    enum looking_up            looking_up;         /* Return 'looking_up' for fill_resolv_cache() */
    
    unsigned int             fill_have_error:1;    /* Return 'have_error' for fill_resolv_cache() */
    unsigned int             transient_error:1;    /* Return 'have_error' for fill_resolv_cache() */
    unsigned int             other:1;              /* have other data for name */

    unsigned int      is_search      :1;           /* Is search, do not report not found errors yet */
    
    unsigned int      query_sent     :1;
    unsigned int      valid_answer   :1;
    unsigned int      failed_answer  :1;
    unsigned int      invalid_answer :1;  /* Answer not stored */
    unsigned int      shutdown       :1;  /* shutdown seen     */

    int               refcount;
    
}           ** query_list      = NULL;    /* Indezed by querynum , 0 is not used */
static size_t  query_list_len   = 0;      /* MAX querynum + 1    */

static struct resolv_query_item * new_resolv_query_item P_((size_t  querynum, const char *name)) UNUSED_FUNCOK;
static struct resolv_query_item * new_resolv_query_item(querynum,name) 
     size_t       querynum;
     const char * name;
{
    struct resolv_query_item *ret =
	safe_zero_alloc(sizeof (*ret));

    if (query_list_len > querynum && query_list[querynum])
	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "new_resolv_query_item",
	      "querynum is already used",0);

    ret->magic        = RESOLV_QUERY_ITEM_magic;
    ret->querynum     = querynum;
    ret->query_time   = NO_schedule_timelimit;
    ret->name         = name ? safe_strdup(name) : NULL;
    ret->query_buffer = NULL;
    ret->query_hdr    = NULL;
    ret->last_errno   = 0;
    ret->cache_answer_in = NULL;
    ret->cache_answer_ret = NULL;
    ret->answer_search_name = NULL;
    ret->looking_up    = l_up_none;
    ret->fill_have_error = 0;
    ret->transient_error = 0;
    ret->other           = 0;
    ret->is_search       = 0;
    ret->query_sent      = 0;
    ret->valid_answer    = 0;
    ret->failed_answer   = 0;
    ret->invalid_answer  = 0;
    ret->shutdown        = 0;

    ret->refcount        = 1;         

    SIGDPRINT(Debug,16,(&Debug,
			"new_resolv_query_item=%p; querynum=%zu\n",
			ret,ret->querynum));
    
    return ret;
}

/* Decrements refcount */

static void free_resolv_query_item P_(( struct resolv_query_item **ptr));
static void free_resolv_query_item(ptr)
     struct resolv_query_item **ptr;
{
    if (RESOLV_QUERY_ITEM_magic != (*ptr)->magic)
    	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "free_resolv_query_item",
	      "Bad magic number (resolv_query_item)",0);

    SIGDPRINT(Debug,16,(&Debug,
			"free_resolv_query_item: [%p] querynum=%zu, refcount=%d\n",
			*ptr,(*ptr)->querynum,(*ptr)->refcount));

    
    if ((*ptr)->refcount < 1)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_resolv_query_item",
	      "Bad refcount",0);
    
    (*ptr)->refcount--;

    SIGDPRINT(Debug,16,(&Debug,
			"free_resolv_query_item: [%p] refcount => %d",
			*ptr,(*ptr)->refcount));
    
    if ((*ptr)->refcount > 0) {
	/* Just reset this reference */

	SIGDPRINT(Debug,16,(&Debug,", returning\n"));
	
	*ptr = NULL;
	return;
    }
    
    SIGDPRINT(Debug,16,(&Debug,", freeing"));

    if ((*ptr)->name) {
	SIGDPRINT(Debug,16,(&Debug,", name=%s",
			    (*ptr)->name));
	free((*ptr)->name);
	(*ptr)->name = NULL;
    }

    SIGDPRINT(Debug,16,(&Debug,"\n"));
	    
    if ((*ptr)->query_buffer)
	free_resolv_message(& ((*ptr)->query_buffer));

    if ((*ptr)->query_hdr) {
	free ((*ptr)->query_hdr);
	(*ptr)->query_hdr = NULL;
    }

    if ((*ptr)->answer_buffer)
	free_resolv_message(& ((*ptr)->answer_buffer));

    if ((*ptr)->answer_hdr) {
	free ((*ptr)->answer_hdr);
	(*ptr)->answer_hdr = NULL;
    }

    if ((*ptr)->cache_answer_in)
	free_resolv_cache(& ((*ptr)->cache_answer_in));
    
    if ((*ptr)->cache_answer_ret)	
	free_resolv_cache(& ((*ptr)->cache_answer_ret));
    
    if ((*ptr)->answer_search_name) {
	free((*ptr)->answer_search_name);
	(*ptr)->answer_search_name = NULL;
    }

    SIGDPRINT(Debug,16,(&Debug,
			"free_resolv_query_item: [%p]",
			*ptr));
    
    (*ptr)->magic = 0;   /* Invalidate */

    free(*ptr);
    *ptr = NULL;

    SIGDPRINT(Debug,16,(&Debug," done\n"));
}

static void inc_resolv_query_item_refcount P_(( struct resolv_query_item *ptr));
static void inc_resolv_query_item_refcount(ptr)
     struct resolv_query_item *ptr;
{
    if (RESOLV_QUERY_ITEM_magic != ptr->magic)
    	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "inc_resolv_query_item_refcount",
	      "Bad magic number (resolv_query_item)",0);
    
    ptr->refcount++;

    SIGDPRINT(Debug,16,(&Debug,
			"inc_resolv_query_item_refcount: [%p] querynum=%zu, refcount => %d\n",
			ptr,ptr->querynum,ptr->refcount));
    
}

/* ------------------------------------------------------------------------------- */

#define RESOLV_MESSAGE_STATE_magic	0xF91B

static struct resolv_message_state {
    unsigned short magic;             /* RESOLV_MESSAGE_STATE_magic */

    struct resolv_query_hdr  read_hdr;
    unsigned char          * read_buffer;
    size_t                   read_buffer_len;
    size_t                   read_buffer_size;                     
    size_t                   read_pos;

    struct resolv_query_hdr  write_hdr;
    unsigned char          * write_buffer;
    size_t                   write_buffer_len;
    size_t                   write_buffer_size;   
    size_t                   write_pos;

    size_t                   querynum;          /* Last or current query */
    
    unsigned int             write_prepared:1;
    unsigned int             broken_pipe:1;     /* Broken pipe when writing,
						   EOF when reading 
					         */
    unsigned int             query_in_progress:1; /* Query 'querynum' in progress */
    
    int refcount;
} * new_resolv_message_state P_((int pid)) UNUSED_FUNCOK;

static struct resolv_message_state * new_resolv_message_state(pid)
     int pid;
{
    struct resolv_message_state *ret =
	safe_zero_alloc(sizeof (*ret));

    ret->magic = RESOLV_MESSAGE_STATE_magic;

    ret->read_hdr.magic       = RESOLV_QUERY_HDR_magic;
    ret->read_hdr.buflen      = 0;
    ret->read_hdr.errno_val   = 0;
    ret->read_hdr.h_errno_val = 0;
    ret->read_hdr.querynum    = 0;
    ret->read_hdr.senderpid   = pid;
    ret->read_hdr.msg         = qm_none;

    ret->read_buffer          = NULL;
    ret->read_buffer_size     = 0;
    ret->read_buffer_len = 0;
    ret->read_pos        = 0;

    ret->write_hdr.magic       = RESOLV_QUERY_HDR_magic;
    ret->write_hdr.buflen      = 0;
    ret->write_hdr.errno_val   = 0;
    ret->write_hdr.h_errno_val = 0;
    ret->write_hdr.querynum    = 0;
    ret->write_hdr.senderpid   = pid;
    ret->write_hdr.msg         = qm_none;
    
    ret->write_buffer          = NULL;
    ret->write_buffer_len      = 0;
    ret->write_buffer_size     = 0;
    ret->write_pos             = 0;
    ret->write_prepared        = 0;
    ret->broken_pipe           = 0;
    ret->query_in_progress     = 0;
    ret->querynum              = 0;

    ret->refcount = 1;
    
    return ret;
}

/* Decrements refcount */
static void free_resolv_message_state P_((struct resolv_message_state **ptr));
static void free_resolv_message_state(ptr)
     struct resolv_message_state **ptr;
{
    if (RESOLV_MESSAGE_STATE_magic != (*ptr)->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_resolv_message_state",
	      "Bad magic number (resolv_message_state)",0);

    if ((*ptr)->refcount < 1)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_resolv_message_state",
	      "Bad refcount",0);

    (*ptr)->refcount--;
    if ((*ptr)->refcount > 0) {
	/* Just reset this reference */

	*ptr = NULL;
	return;
    }

    if ((*ptr)->read_buffer) {
	free((*ptr)->read_buffer);
	(*ptr)->read_buffer = NULL;
    }
    
    if ((*ptr)->write_buffer) {
	free((*ptr)->write_buffer);
	(*ptr)->write_buffer = NULL;
    }

    (*ptr)->read_hdr.magic       = 0;
    (*ptr)->write_hdr.magic       = 0;
    (*ptr)->magic = 0; /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}

/* Decrements refcount */
static void inc_resolv_message_state_refcount P_((struct resolv_message_state *ptr))
    UNUSED_FUNCOK;
static void inc_resolv_message_state_refcount(ptr)
     struct resolv_message_state *ptr;
{
    if (RESOLV_MESSAGE_STATE_magic != ptr->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"inc_resolv_message_state_refcount",
	      "Bad magic number (resolv_message_state)",0);

    ptr->refcount ++;
}



/* --------------------------------------------------------------------------------*/

#define RESOLV_PROCESS_magic	0xF91A

static struct resolv_process {
    unsigned short magic;       /* RESOLV_PROCESS_magic */

    int            pid;

    enum process_state {
	ps_none     = 0,
	ps_forked   = 1,
	ps_running  = 2  /* Got qm_running   */,
	ps_shutdown = 3  /* Got qm_shutdown  */,
	ps_exited   = 4  /* Got exit status  */,
	ps_died     = 5  /* Got signal       */
    }           state;

    struct resolv_message_state  * message;
    int                            socket;
    
    int refcount;
}            ** process_list     = NULL;
static size_t   process_list_len = 0;


static struct resolv_process * new_resolv_process P_((int pid)) UNUSED_FUNCOK;
static struct resolv_process * new_resolv_process(pid)
     int pid;
{
    struct resolv_process * ret =
	safe_zero_alloc(sizeof (*ret));
    
    ret->magic  = RESOLV_PROCESS_magic;
    ret->pid    = pid;
    ret->state  = ps_none;
    ret->message = NULL;
    ret->socket  = -1;
    ret->refcount = 1;

    SIGDPRINT(Debug,16,(&Debug,
			"new_resolv_process=%p; pid %d\n",
			ret,pid));
    
    return ret;
}

static void free_resolv_process P_((struct resolv_process **ptr));
static void free_resolv_process(ptr)
     struct resolv_process **ptr;
{    
    if (RESOLV_PROCESS_magic != (*ptr)->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_resolv_process",
	      "Bad magic number (resolv_process)",0);
    
    SIGDPRINT(Debug,16,(&Debug,
			"free_resolv_process: [%p] refcount=%d",
			*ptr,(*ptr)->refcount));
    
    if (-1 != (*ptr)->pid) {
	SIGDPRINT(Debug,16,(&Debug,", pid=%d",
			    (*ptr)->pid));
    }
    if (-1 != (*ptr)->socket) {
	SIGDPRINT(Debug,16,(&Debug,", socket=%d",
			    (*ptr)->socket));
    }
    SIGDPRINT(Debug,16,(&Debug,"\n"));
    
    if ((*ptr)->refcount < 1)
	panic("RESOLV PANIC",__FILE__,__LINE__,"free_resolv_process",
	      "Bad refcount",0);
    (*ptr)->refcount--;

    SIGDPRINT(Debug,16,(&Debug,
			"free_resolv_process: [%p] refcount => %d",
			*ptr,(*ptr)->refcount));
    
    if ((*ptr)->refcount > 0) {
	/* Just reset this reference */

	SIGDPRINT(Debug,16,(&Debug,", returning\n"));
	
	*ptr = NULL;
	return;
    }

    SIGDPRINT(Debug,16,(&Debug,", freeing\n"));
    
    if ((*ptr)->message) {
	free_resolv_message_state(& ((*ptr)->message));
    }

    if (-1 != (*ptr)->socket) {
	int r;

	DPRINT(Debug,12,(&Debug,
			 "free_resolv_process: [%p] socket=%d -- clear action\n",
			 *ptr,(*ptr)->socket));
	
	/* No longer schedule for this! */
	clear_action((*ptr)->socket);

	DPRINT(Debug,12,(&Debug,"free_resolv_process: [%p] Closing socket %d (to process pid %d)",
			 *ptr,(*ptr)->socket,(*ptr)->pid));
	
	r = close((*ptr)->socket);
	if (-1 == r) {
	    int err UNUSED_VAROK = errno;

	    DPRINT(Debug,12,(&Debug,", failed; errno %d (%s)",
			     err,strerror(err)));
	} else if (r == 0) {
	    DPRINT(Debug,12,(&Debug,", succeed"));
	}
	DPRINT(Debug,12,(&Debug,".\n"));
	
	(*ptr)->socket = -1;
    }

    DPRINT(Debug,16,(&Debug,
		     "free_resolv_process: [%p] ",*ptr));
    
    (*ptr)->magic = 0; /* Invalidate */
    free(*ptr);
    *ptr = NULL;

    DPRINT(Debug,16,(&Debug,"done\n"));
}

static void inc_resolv_process_refcount P_((struct resolv_process *ptr));
static void inc_resolv_process_refcount(ptr)
     struct resolv_process *ptr;
{
    if (RESOLV_PROCESS_magic != ptr->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"inc_resolv_process_refcount",
	      "Bad magic number (resolv_process)",0);

    ptr->refcount++;

    SIGDPRINT(Debug,16,(&Debug,
			"inc_resolv_process_refcount: [%p] refcount => %d\n",
			ptr,ptr->refcount));    
}

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

static int parse_resolv_message1 P_((struct resolv_message * buffer,
				     int                   * errno_val));
static int parse_resolv_message1(buffer,errno_val)
     struct resolv_message * buffer;
     int                   * errno_val;
{
    int ret = 0;
    int res;
    
    if (RESOLV_MESSAGE_magic != buffer->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "parse_resolv_message1",
	      "Bad magic number (resolv_message)",0);
    
    res = ns_initparse(buffer->message,buffer->message_len,
		       & (buffer-> parsed_message));
    
    if (res < 0) {
	int err = errno;
	
	DPRINT(Debug,14,(&Debug,
			 "parse_resolv_message1: ns_initparse() failed, len %u",
			 (unsigned) buffer->message_len));
	    
	if (err) {
	    DPRINT(Debug,14,(&Debug,", errno = %d (%s)",
			     err, strerror(err)));
	    if (errno_val)
		*errno_val = err;
	}
		
	DPRINT(Debug,14,(&Debug,"\n"));
	    
    } else {
	DPRINT(Debug,14,(&Debug,
			 "parse_resolv_message1: ns_initparse() succeed, len %u\n",
			 (unsigned) buffer->message_len));
	buffer->message_parsed = 1;
	ret = 1;
    }

    DPRINT(Debug,14,(&Debug,
		     "parse_resolv_message1=%d",ret));
    if (errno_val && *errno_val) {
	DPRINT(Debug,14,(&Debug,
			 "; *errno_val=%d",*errno_val));
    }
    DPRINT(Debug,14,(&Debug,"\n"));
    
    return ret;

}

#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET)

static int parse_resolv_message P_((struct resolv_message * buffer,
				    const unsigned char     answer_buffer[],
				    const size_t            answer_buflen,
				    int                   * errno_val));
static int parse_resolv_message(buffer,answer_buffer,answer_buflen,errno_val)
     struct resolv_message * buffer;
     const unsigned char     answer_buffer[];
     const size_t            answer_buflen;
     int                   * errno_val;
{
    int ret = 0;
    
    errno = 0;
    if (errno_val) 
	*errno_val = 0;
    
    if (RESOLV_MESSAGE_magic != buffer->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,
	      " parse_resolv_message",
	      "Bad magic number (resolv_message)",0);
    
    if (answer_buflen > 0 && answer_buffer &&
	answer_buflen <= buffer->message_size &&
	answer_buflen <= USHRT_MAX &&
	buffer->message ) {
        
	memcpy(buffer->message, answer_buffer, answer_buflen);
	buffer->message_len = answer_buflen;


	ret = parse_resolv_message1(buffer,errno_val);
    } else {
	DPRINT(Debug,14,(&Debug,
			 "parse_resolv_message: len %zu\n",answer_buflen));
	if (errno_val)
	    *errno_val = EMSGSIZE;
	
    }
	
    return ret;
}

static int same_question_rr P_((ns_rr *rr1, ns_rr *rr2));
static int same_question_rr(rr1,rr2)
     ns_rr *rr1;
     ns_rr *rr2;
{
  char *name1 = ns_rr_name((*rr1));
  char *name2 = ns_rr_name((*rr2));
  ns_class recclass1 = ns_rr_class((*rr1));
  ns_class recclass2 = ns_rr_class((*rr2));
  ns_type rectype1 =  ns_rr_type((*rr1));
  ns_type rectype2 =  ns_rr_type((*rr2));
  
  if (!  name1 || ! name2) {
      DPRINT(Debug,14,(&Debug, "same_question_rr: no name\n"));
      return 0;
  }
  
  if (0 != strcmp(name1,name2)) {
      DPRINT(Debug,14,(&Debug, "same_question_rr: name mismatch: %s <> %s\n",
		       name1,name2));
      return 0;
  }
  
  if (recclass1 != recclass2) {
      DPRINT(Debug,14,(&Debug, "same_question_rr: %s: class mismatch: %d <> %d\n",
		       name1,recclass1,recclass2));
      return 0;
  }

  if (rectype1 !=  rectype2) {
      DPRINT(Debug,14,(&Debug,"same_question_rr: %s: rectype mismatch: %d <> %d\n",
		       name1,rectype1,rectype2));
      return 0;
  }

  return 1;
}


static void process_answer P_((const int                        pid,
			       const struct resolv_query_hdr *  hdr,
			       const unsigned char              answer_buffer[],
			       const size_t                     answer_buflen,
			       struct resolv_query_item       * qitem));
static void process_answer(pid,hdr,answer_buffer,answer_buflen,qitem)
     const int                        pid;
     const struct resolv_query_hdr *  hdr;
     const unsigned char              answer_buffer[];
     const size_t                     answer_buflen;
     struct resolv_query_item       * qitem;
{
    
    if (RESOLV_QUERY_HDR_magic != hdr->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "process_answer",
	      "Bad magic number (resolv_query_hdr)",0);

    if (qitem) {
	
	if (RESOLV_QUERY_ITEM_magic != qitem->magic)
	    panic("RESOLV PANIC",__FILE__,__LINE__,"process_answer",
		  "Bad magic number (resolv_query_item)",0);


	if (qitem->answer_hdr) {
	    if (RESOLV_QUERY_HDR_magic != qitem->answer_hdr->magic)
		panic("RESOLV PANIC",__FILE__,__LINE__,
		      "process_answer",
		      "Bad magic number resolv_query_hdr)",0);
	}

	if (hdr->errno_val) {
	    DPRINT(Debug,14,(&Debug, "process_answer: [pid %d] %lu %s: copying error %d (%s)\n",
			     pid,
			     (unsigned long)(qitem->querynum),
			     qitem->name ? qitem->name : "<NO NAME>",
			     hdr->errno_val,strerror(hdr->errno_val)));
	    qitem->last_errno = hdr->errno_val;	    
	}
	
	if (qitem->answer_buffer) {
	    int                        err_rmesg = 0;
	    int ok;
	    
	    if (RESOLV_MESSAGE_magic != qitem->answer_buffer->magic)
		panic("RESOLV PANIC",__FILE__,__LINE__,
		      "process_answer",
		      "Bad magic number (resolv_message)",0);
	   	    
	    ok = parse_resolv_message(qitem->answer_buffer,
				      answer_buffer,answer_buflen,&err_rmesg);

	    if (!ok) {
		DPRINT(Debug,14,(&Debug, "process_answer: [pid %d] %lu %s: failed to parse message (len %lu)",
				 pid,
				 (unsigned long)(qitem->querynum),
				 qitem->name ? qitem->name : "<NO NAME>",
				 (unsigned long) answer_buflen));
	
		if (err_rmesg) {
		    DPRINT(Debug,14,(&Debug, ", error %s (errno %d)",
				     strerror(err_rmesg),err_rmesg));
		    
		    qitem->last_errno = err_rmesg;
		}
	
		DPRINT(Debug,14,(&Debug, "\n"));
	
		qitem->invalid_answer = 1;

		if (qitem->name) {
		    if (err_rmesg) {
			lib_error(CATGETS(elm_msg_cat, 
					  ResolvSet,
					  ResolvLookFParseAnswerE,
					  "Looking up %s: Failed to parse answer: %s"),
				  qitem->name,
				  strerror(err_rmesg));
		    } else
			lib_error(CATGETS(elm_msg_cat, 
					  ResolvSet,
					  ResolvLookFParseAnswer,
					  "Looking up %s: Failed to parse answer"),
				  qitem->name);

		    qitem->looking_up = l_up_error;
		}
		
		
	    } else if (qitem->query_buffer) {
		char * refname = NULL;
		
		if (RESOLV_MESSAGE_magic != qitem->query_buffer->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,
			  "process_answer",
			  "Bad magic number (resolv_message)",0);
		
		if (ok) {
		    u_int16_t query_id  = ns_msg_id(qitem->query_buffer->parsed_message);
		    u_int16_t answer_id = ns_msg_id(qitem->answer_buffer->parsed_message);
		    
		    if (query_id != answer_id) {
			
			DPRINT(Debug,14,(&Debug,
					 "process_answer: [pid %d] %lu %s: query id %d <=> answer id %d\n",
					 pid,
					 (unsigned long)(qitem->querynum),
					 qitem->name ? qitem->name : "<NO NAME>",
					 query_id,answer_id));
			
				ok = 0;
		    }
		}
			
		if (ok) {
		    int count_query_qd  = ns_msg_count(qitem->query_buffer->parsed_message,
						       ns_s_qd);
		    int count_answer_qd = ns_msg_count(qitem->answer_buffer->parsed_message,
						       ns_s_qd);
		    
		    if (count_query_qd != count_answer_qd) {
			DPRINT(Debug,14,(&Debug,
					 "process_answer: [pid %d] %lu %s: query %d <=> answer %d  records on question section\n",
					 pid,
					 (unsigned long)(qitem->querynum),
					 qitem->name ? qitem->name : "<NO NAME>",
					 count_query_qd,count_answer_qd));
			ok = 0;
		    } else {
			int recnum;
			
			DPRINT(Debug,14,(&Debug,
					 "process_answer: [pid %d] %lu %s: %d records on question section\n",
					 pid,
					 (unsigned long)(qitem->querynum),
					 qitem->name ? qitem->name : "<NO NAME>",
					 count_query_qd));
			
			for (recnum = 0; recnum < count_query_qd && ok; recnum++) {
			    ns_rr   rrquestion;
			    ns_rr   rranswer;
			    int res;
			    
			    errno = 0;
			    res =  ns_parserr(&(qitem->query_buffer->parsed_message),
					      ns_s_qd,recnum,&rrquestion);
			    
			    if (res < 0) {
				int errnum = errno;
				DPRINT(Debug,14,(&Debug,
						 "process_answer: [pid %d] %lu %s: failed to parse record #%d on question section of query",
						 pid,
						 (unsigned long)(qitem->querynum),
						 qitem->name ? qitem->name : "<NO NAME>",
						 recnum));
				
				if (errnum) {
				    DPRINT(Debug,14,(&Debug, ", error %s (errno %d)",
						     strerror(errnum),errnum));
				    qitem->last_errno = errnum;
				}
				
				DPRINT(Debug,14,(&Debug,"\n"));
				ok = 0;
			    }
			    
			    res =  ns_parserr(&(qitem->answer_buffer->parsed_message),
					      ns_s_qd,recnum,&rranswer);
			    
			    if (res < 0) {
				int errnum = errno;
				DPRINT(Debug,14,(&Debug,
						 "process_answer: [pid %d] %lu %s: failed to parse record #%d on question section of answer",
						 pid,
						 (unsigned long)(qitem->querynum),
						 qitem->name ? qitem->name : "<NO NAME>",
						 recnum));
				if (errnum) {
				    DPRINT(Debug,14,(&Debug, ", error %s (errno %d)",
						     strerror(errnum),errnum));
				    qitem->last_errno = errnum;
				}
				
				DPRINT(Debug,14,(&Debug,"\n"));
				ok = 0;				    
			    }
			    
			    if (ok && !same_question_rr(&rrquestion,&rranswer)) {
				DPRINT(Debug,14,(&Debug,
						 "process_answer: [pid %d] %lu %s: record #%d differ on between question section of query and answer\n",
						 pid,
						 (unsigned long)(qitem->querynum),
						 qitem->name ? qitem->name : "<NO NAME>",
						 recnum));
				ok = 0;
			    }

			    if (ok && 1 == count_query_qd) {
				const char * X = ns_rr_name(rranswer);				
				if (X) {
				    refname = strmcpy(refname,X);
				    DPRINT(Debug,14,(&Debug,
						     "process_answer: [pid %d] %lu %s: reference name is %s\n",
						     pid,
						     (unsigned long)(qitem->querynum),
						     qitem->name ? qitem->name : "<NO NAME>",
						     refname));
				}
			    }
			}			       
		    }
		}
		
		if (ok) {
		    ns_rcode response_code =
			ns_msg_getflag(qitem->answer_buffer->parsed_message,
				       ns_f_rcode);
		    unsigned short aa =
			ns_msg_getflag(qitem->answer_buffer->parsed_message,
				       ns_f_aa);
		    
		    DPRINT(Debug,14,(&Debug,
				     "process_answer: [pid %d] %lu %s: query name=%s, authorative=%u, response code=%d",
				     pid,
				     (unsigned long)(qitem->querynum),
				     qitem->name ? qitem->name : "<NO NAME>",
				     refname ? refname : "<NO NAME>",
				     (unsigned)aa,
				     response_code));
		    switch (response_code) {
		    case ns_r_noerror: DPRINT(Debug,14,(&Debug," (no error)\n"));    break;
		    case ns_r_formerr: DPRINT(Debug,14,(&Debug," (format error)\n"));  
			if (qitem->name) {

			    if (refname && 0 != strcmp(qitem->name,refname))
				lib_error(CATGETS(elm_msg_cat, 
						  ResolvSet,
						  ResolvLookFormErrR,
						  "Looking up %s [%s]: format error"),
					  qitem->name,refname);				
			    else
				lib_error(CATGETS(elm_msg_cat, 
						  ResolvSet,
						  ResolvLookFormErr,
						  "Looking up %s: format error"),
					  qitem->name);
			    qitem->looking_up = l_up_error;
			} else if (refname) {
			    lib_error(CATGETS(elm_msg_cat, 
					      ResolvSet,
					      ResolvLookFormErr,
					      "Looking up %s: format error"),
				      refname);
			    qitem->looking_up = l_up_error;
			}
			
			qitem->fill_have_error = 1;			    
			break;
		    case ns_r_servfail: DPRINT(Debug,14,(&Debug," (server failure)\n"));  
			if (qitem->name) {
			    if (refname && 0 != strcmp(qitem->name,refname))
				lib_error(CATGETS(elm_msg_cat, 
						  ResolvSet,
						  ResolvLookServFailR,
						  "Looking up %s [%s]: server failure"),
					  qitem->name,refname);
			    else
				lib_error(CATGETS(elm_msg_cat, 
						  ResolvSet,
						  ResolvLookServFail,
						  "Looking up %s: server failure"),
					  qitem->name); 
			    qitem->looking_up = l_up_error;
			} else if (refname) {
			    lib_error(CATGETS(elm_msg_cat, 
					      ResolvSet,
					      ResolvLookServFail,
					      "Looking up %s: server failure"),
				      refname);
			    qitem->looking_up = l_up_error;
			}
			break;
		    case ns_r_nxdomain: DPRINT(Debug,14,(&Debug," (name error)"));

			if (qitem->is_search) {
			    DPRINT(Debug,14,(&Debug," -- using search list for name, do not print error yet.\n"));
			    qitem->transient_error = 1;
			} else {
			    DPRINT(Debug,14,(&Debug,"\n"));
			    if (aa) {
				if (qitem->name) {
				    if (refname && 0 != strcmp(qitem->name,refname))
					lib_error(CATGETS(elm_msg_cat, 
							  ResolvSet,
							  ResolvLookNotFoundR,
							  "Looking up %s [%s]: not found"),
						  qitem->name,refname);
				    else
					lib_error(CATGETS(elm_msg_cat, 
							  ResolvSet,
							  ResolvLookNotFound,
							  "Looking up %s: not found"),
						  qitem->name);
				    qitem->looking_up = l_up_error;
				} else if (refname) {
				    lib_error(CATGETS(elm_msg_cat, 
						      ResolvSet,
						      ResolvLookNotFound,
						      "Looking up %s: not found"),
					      refname);
				    qitem->looking_up = l_up_error;
				}
			    } else {
				if (qitem->name) {
				    if (refname && 0 != strcmp(qitem->name,refname))
					lib_transient(CATGETS(elm_msg_cat, 
							      ResolvSet,
							      ResolvLookTryAgainR,
							      "Looking up %s [%s]: not found yet"),
						      qitem->name,refname);				
				    else
					lib_transient(CATGETS(elm_msg_cat, 
							      ResolvSet,
							      ResolvLookTryAgain,
							      "Looking up %s: not found yet"),
						      qitem->name);
				    qitem->looking_up = l_up_error;
				} else if (refname) {
				    lib_transient(CATGETS(elm_msg_cat, 
							  ResolvSet,
							  ResolvLookTryAgain,
							  "Looking up %s: not found yet"),
						  refname);
				}
				qitem->fill_have_error = 1;
			    }
			}
			break;
		    case ns_r_notimpl: DPRINT(Debug,14,(&Debug," (unimplemented)\n"));
			if (qitem->name) {
			    if (refname && 0 != strcmp(qitem->name,refname))
				lib_error(CATGETS(elm_msg_cat, 
						  ResolvSet,
						  ResolvLookNotImplR,
						  "Looking up %s [%s]: not implemented"),
					  qitem->name,refname);
			    else
				lib_error(CATGETS(elm_msg_cat, 
						  ResolvSet,
						  ResolvLookNotImpl,
						  "Looking up %s: not implemented"),
					  qitem->name);
			    qitem->looking_up = l_up_error;
			} else if (refname) {
			    lib_error(CATGETS(elm_msg_cat, 
					      ResolvSet,
					      ResolvLookNotImpl,
					      "Looking up %s: not implemented"),
				      refname);
			    qitem->looking_up = l_up_error;
			}
			
			qitem->fill_have_error = 1;
			break;
		    case ns_r_refused: DPRINT(Debug,14,(&Debug," (operation refused)\n"));
			if (qitem->name) {
			    if (refname && 0 != strcmp(qitem->name,refname))
				lib_error(CATGETS(elm_msg_cat, 
						  ResolvSet,
						  ResolvLookOpRefR,
						  "Looking up %s [%s]: operation refused"),
					  qitem->name,refname);			    
			    else
				lib_error(CATGETS(elm_msg_cat, 
						  ResolvSet,
						  ResolvLookOpRef,
						  "Looking up %s: operation refused"),
					  qitem->name);
			    qitem->looking_up = l_up_error;
			} else if (refname) {
			    lib_error(CATGETS(elm_msg_cat, 
					      ResolvSet,
					      ResolvLookOpRef,
					      "Looking up %s: operation refused"),
				      refname);
			    qitem->looking_up = l_up_error;
			}
			
			qitem->fill_have_error = 1;
			break;
		    default: DPRINT(Debug,14,(&Debug,"\n")); break;
		    }
		    		    		   
		    if (!qitem->answer_hdr)			
			qitem->answer_hdr = safe_malloc(sizeof (* (qitem->answer_hdr)));		    
		    * (qitem->answer_hdr) = *hdr;
		    
		    if (qitem->cache_answer_ret) {
			DPRINT(Debug,14,(&Debug,
					 "process_answer: [pid %d] %lu %s: clearing previous answer\n",
					 pid,
					 (unsigned long)(qitem->querynum),
					 qitem->name ? qitem->name : "<NO NAME>"));
			free_resolv_cache(&  (qitem->cache_answer_ret));
		    }

		    if (ns_r_noerror == response_code) {
			int anscount = ns_msg_count(qitem->answer_buffer->parsed_message,
						    ns_s_an);
			
			if (!anscount) {
			    DPRINT(Debug,14,(&Debug,
					     "process_answer: [pid %d] %lu %s: Didn't got answer but name found\n",
					     pid,
					     (unsigned long)(qitem->querynum),
					     qitem->name ? qitem->name : "<NO NAME>"));
			    qitem->other = 1;
			} else {
			    DPRINT(Debug,14,(&Debug,
					     "process_answer: [pid %d] %lu %s: %d recors on answer section\n",
					      pid,
					     (unsigned long)(qitem->querynum),
					     qitem->name ? qitem->name : "<NO NAME>",
					     anscount));
			}
			
			qitem->valid_answer = 1;
		    
			if (qitem->cache_answer_in) {
			    int have_error = 0;
			    
			    qitem->cache_answer_ret =
				fill_resolv_cache(qitem->cache_answer_in,
						  &(qitem->answer_buffer->parsed_message),
						  &(qitem->query_time),
						  &(qitem->answer_search_name),
						  &(qitem->looking_up),
						  &have_error);
			    
			    if (have_error) {
				DPRINT(Debug,14,(&Debug,
						 "process_answer: [pid %d] %lu %s: fill_resolv_cache have error\n",
						 pid,
						 (unsigned long)(qitem->querynum),
						 qitem->name ? qitem->name : "<NO NAME>"));
				qitem->fill_have_error = 1;
			    }
			    
			} else {
			    DPRINT(Debug,14,(&Debug,
					     "process_answer: [pid %d] %lu %s: no answer record for fill_resolv_cache\n",
					 pid,
					     (unsigned long)(qitem->querynum),
					     qitem->name ? qitem->name : "<NO NAME>"));
			    qitem->fill_have_error = 1;
			}

		    } else
			qitem->failed_answer = 1;	    
			    
		} else {
		    DPRINT(Debug,14,(&Debug, "process_answer: [pid %d] %lu %s: will not store answer\n",
				     pid,
				     (unsigned long)(qitem->querynum),
				     qitem->name ? qitem->name : "<NO NAME>"));
		    qitem->invalid_answer = 1;
		}

		if (refname) {
		    DPRINT(Debug,14,(&Debug, "process_answer: [pid %d] %lu %s: query name %s\n",
				     pid,
				     (unsigned long)(qitem->querynum),
				     qitem->name ? qitem->name : "<NO NAME>",
				     refname));
		    free(refname);
		    refname = NULL;
		}
		
	    } else {
		DPRINT(Debug,14,(&Debug, "process_answer: [pid %d] %lu %s: query not stored\n",
				 pid,
				 (unsigned long)(qitem->querynum),
				 qitem->name ? qitem->name : "<NO NAME>"));
		
		qitem->invalid_answer = 1;
	    }

	    if (!ok) {
		DPRINT(Debug,14,(&Debug, "process_answer: [pid %d] %lu %s: clearing answer\n",
				 pid,
				 (unsigned long)(qitem->querynum),
				 qitem->name ? qitem->name : "<NO NAME>"));

		qitem->answer_buffer->message_len    = 0;
		qitem->answer_buffer->message_parsed = 0;

		if (qitem->answer_hdr)
		    *(qitem->answer_hdr) =  NULL_resolv_query_hdr;

		/* bzero is defined on hdrs/defs.h */
		if (qitem->answer_buffer->message && qitem->answer_buffer->message_size)
		    bzero(qitem->answer_buffer->message,qitem->answer_buffer->message_size);
	    }

	    
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "process_answer: [pid %d] %lu %s: no answer buffer\n",
			     (unsigned long)(qitem->querynum),
			     qitem->name ? qitem->name : "<NO NAME>"));
	    qitem->invalid_answer = 1;
	}
    } else {
	DPRINT(Debug,14,(&Debug, "process_answer: [pid %d] no query item\n",
			 pid));

    }
}

S_(free_action_data_f resolv_free_state)
static void resolv_free_state P_((union action_routine_data * data,int fd,
				  int badpid));
static void resolv_free_state(data,fd,badpid)
     union action_routine_data * data;
     int fd;
     int badpid;
{
    free_resolv_message_state(&(data->resolv));
}

S_(inc_action_data_refcount_f resolv_inc_state_refcount)
static void resolv_inc_state_refcount P_((union action_routine_data data));
static void resolv_inc_state_refcount(data)
     union action_routine_data data;
{
    inc_resolv_message_state_refcount(data.resolv);
}

S_(badpid_action_f resolv_badpid_action)
static enum badpid_status resolv_badpid_action
   P_((int fd,
       union action_routine_data         data,
       int badpid));
static enum badpid_status resolv_badpid_action(fd,data,badpid)
     int fd;
     union action_routine_data         data;
     int badpid;
{
    struct resolv_message_state * state = data.resolv;

    int pid = getpid();

    int r;
    size_t i;
    int found = 0;

    DPRINT(Debug,14,(&Debug, "resolv_badpid_action: [pid %d] socket %d is owned by pid %d\n",
		     pid,fd,badpid));
    
    if (RESOLV_MESSAGE_STATE_magic != state->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"resolv_badpid_action",
	      "Bad magic number (resolv_message_state)",0);

    state->broken_pipe = 1; /* Broken pipe when writing,
			       EOF when reading 
			    */

    for (i = 0 ; i < process_list_len; i++) {

	if (process_list[i]) {
		    
	    if (RESOLV_PROCESS_magic != process_list[i]->magic)
		panic("RESOLV PANIC",__FILE__,__LINE__,"select_resolv_process",
		      "Bad magic number (resolv_badpid_action)",0);

	    if (process_list[i]->socket == fd) {
		DPRINT(Debug,14,(&Debug, "resolv_badpid_action: may delete process index %lu (pid %d, socket %d)\n",
				 (unsigned long)i,process_list[i]->pid,
				 process_list[i]->socket));

		free_resolv_process(& (process_list[i]));
		found = 1;
	    }
	}
    }

    if (!found) {
	DPRINT(Debug,14,(&Debug,
			 "resolv_badpid_action: Not found socket %d from process_list (%lu entries)\n",
			 fd,(unsigned long)process_list_len));
	r = close(fd);

	if (-1 == r) {
	    int e = errno;
	    
	    DPRINT(Debug,14,(&Debug,
			     "resolv_badpid_action: socket %d close failed: error %d: %s\n",
			     fd,e,strerror(e)));
	} else if (r == 0) {
	DPRINT(Debug,14,(&Debug, "resolv_badpid_action: socket %d closed on process %d\n",
			 fd,pid));
	}
    }
    
    return badpid_remove;
}

S_(action_routine resolv_read_message)
enum action_status resolv_read_message P_((int fd,
					   union action_routine_data         data,
					   const struct schedule_timelimit * now));
enum action_status resolv_read_message(fd,data,now)
     int fd;
     union action_routine_data         data;
     const struct schedule_timelimit * now;
{
    struct resolv_message_state * state = data.resolv;
    int pid = getpid();
    int err = 0;
    enum action_status st = action_continue;

    enum read_status rs;

    DPRINT(Debug,14,(&Debug, "resolv_read_message: [pid %d] socket %d readable\n",
		     pid,fd));

    if (RESOLV_MESSAGE_STATE_magic != state->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"resolv_read_message",
	      "Bad magic number (resolv_message_state)",0);
    
    rs = read_message(fd,
		      &(state->read_hdr),
		      state->read_buffer,
		      state->read_buffer_size,
		      &(state->read_buffer_len),
		      &err,pid,
		      &(state->read_pos));

    DPRINT(Debug,14,(&Debug, "resolv_read_message: [pid %d] fd %d: got %d",
		     pid,fd,rs));
    switch(rs) {
    case rs_read_error:  SIGDPRINT(Debug,14,(&Debug, " (read error, error=%d)  ",err));
	if (err != EINTR && err != EAGAIN && err != EWOULDBLOCK)
	    st = action_disable;
	break;
    case rs_parse_error: SIGDPRINT(Debug,14,(&Debug, " (parse error) ")); break;
    case rs_EOF:         SIGDPRINT(Debug,14,(&Debug, " (EOF)         "));
	if (! state->broken_pipe) {
	    SIGDPRINT(Debug,14,(&Debug, ", marking broken pipe"));
	    state->broken_pipe = 1;   /* Writing will cause EPIPE error */
	}
	st = action_disable;
	break;
    case rs_none:        SIGDPRINT(Debug,14,(&Debug, " (none)        ")); break;
    case rs_got_message: SIGDPRINT(Debug,14,(&Debug, " (got message) ")); break;
    case rs_incomplete:  SIGDPRINT(Debug,14,(&Debug, " (incomplete)  ")); break;
    case rs_continue:    SIGDPRINT(Debug,14,(&Debug, " (continue)  ")); break;
    }
    SIGDPRINT(Debug,14,(&Debug, "\n"));

    if (state->query_in_progress) {
	DPRINT(Debug,14,(&Debug,
			 "resolv_read_message: [pid %d] fd %d: resetting 'query in progress' (querynum %lu)\n",
			 pid,fd,(unsigned long)(state->querynum)));
	state->query_in_progress = 0;
    }
    
    if (rs_got_message == rs) {
	struct resolv_query_hdr  * hdr    = &(state->read_hdr);
	struct resolv_process    * proc   = NULL;
	struct resolv_query_item * qitem  = NULL;
	
	DPRINT(Debug,14,(&Debug, "resolv_read_message: [pid %d] fd %d: got message %d",
			 pid,fd,hdr->msg));
	switch(hdr->msg) {
	case qm_none:          SIGDPRINT(Debug,14,(&Debug, " (none)"));          break;
	case qm_query:         SIGDPRINT(Debug,14,(&Debug, " (query)"));         break;
	case qm_answer:        SIGDPRINT(Debug,14,(&Debug, " (answer)"));        break;
	case qm_res_send_fail: SIGDPRINT(Debug,14,(&Debug, " (res_send fail)")); break;
	case qm_running:       SIGDPRINT(Debug,14,(&Debug, " (running)"));       break;
	case qm_shutdown:      SIGDPRINT(Debug,14,(&Debug, " (shutdown)"));      break;
	case qm_sizeupdate:    SIGDPRINT(Debug,14,(&Debug, " (sizeupdate)"));    break;   
	}
	if (hdr->senderpid > 0) {
	    size_t i;
	    
	    SIGDPRINT(Debug,14,(&Debug, ", sender pid %d",hdr->senderpid));

	    for (i = 0; i < process_list_len; i++) {
		if (process_list[i]) {
		    if (RESOLV_PROCESS_magic != process_list[i]->magic)
			panic("RESOLV PANIC",__FILE__,__LINE__,"resolv_read_message",
			      "Bad magic number (resolv_process)",0);

		    if (hdr->senderpid == process_list[i]->pid) {
			SIGDPRINT(Debug,14,(&Debug, "; found #%lu: state=%d",
					    (unsigned long)i,
					    process_list[i]->state));
			
			switch(process_list[i]->state) {
			case ps_none:     SIGDPRINT(Debug,14,(&Debug, " (none)    ")); break;
			case ps_forked:   SIGDPRINT(Debug,14,(&Debug, " (forked)  ")); break;
			case ps_running:  SIGDPRINT(Debug,14,(&Debug, " (running) ")); break;
			case ps_shutdown: SIGDPRINT(Debug,14,(&Debug, " (shutdown)")); break;
			case ps_exited:   SIGDPRINT(Debug,14,(&Debug, " (exited)  ")); break;
			case ps_died:     SIGDPRINT(Debug,14,(&Debug, " (died)    ")); break; 
			}

			if (state == process_list[i]->message) {
			    proc = process_list[i];
			    break;

			} else {
			    SIGDPRINT(Debug,14,(&Debug, " <- not current"));
			}
		    }			
		}
	    }
	}

	if (hdr->querynum > 0) {
	    SIGDPRINT(Debug,14,(&Debug, ", querynum=%lu", (unsigned long)(hdr->querynum)));

	    if (hdr->querynum <  query_list_len &&
		query_list[hdr->querynum]) {

		if (RESOLV_QUERY_ITEM_magic != query_list[hdr->querynum]->magic)
			panic("RESOLV PANIC",__FILE__,__LINE__,"resolv_read_message",
			      "Bad magic number (resolv_query_item)",0);	     		
		qitem = query_list[hdr->querynum];
	    } else {
		SIGDPRINT(Debug,14,(&Debug, " (invalid)"));
	    }	       
	}
	SIGDPRINT(Debug,14,(&Debug, "\n"));

	switch(hdr->msg) {
	    union resolv_sizeupdate_p size_update;
	    
	case qm_none:    break;
	case qm_query:   break;
	case qm_answer:
	    if (state->read_buffer_len) {
		process_answer(pid,			       
			       hdr,
			       state->read_buffer,
			       state->read_buffer_len,
			       qitem);
	    } else if (qitem)  {
		DPRINT(Debug,14,(&Debug, "process_answer: [pid %d] %lu %s: no answer",
				 pid,
				 (unsigned long)(qitem->querynum),
				 qitem->name ? qitem->name : "<NO NAME>"));
		
		if (hdr->errno_val) {
		    DPRINT(Debug,14,(&Debug,"copying error %d (%s)",
				     hdr->errno_val,strerror(hdr->errno_val)));
		    qitem->last_errno = hdr->errno_val;
		}
		DPRINT(Debug,14,(&Debug,"\n"));
		
		qitem->invalid_answer = 1;
	    }
		
	    break;

	case  qm_res_send_fail:
	    if (qitem) {
		int have_already_answer = 1;
		
		if (qitem->answer_hdr) {
		    if (RESOLV_QUERY_HDR_magic != qitem->answer_hdr->magic)
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "resolv_read_message",
			      "Bad magic number ( resolv_query_hdr)",0);

		    if (qm_none != qitem->answer_hdr->msg) {
			DPRINT(Debug,14,(&Debug,
					 "resolv_read_message: [pid %d] fd %d: lu% %s: have already valid answer header\n",
					 pid,fd,
					 (unsigned long)(qitem->querynum),
					 qitem->name ? qitem->name : "<NO NAME>"));
		    } else
			have_already_answer = 0;
		    
		    
		} else
		    have_already_answer = 0;

		if (qitem->answer_buffer) {
		    if (RESOLV_MESSAGE_magic != qitem->answer_buffer->magic)
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "resolv_read_message",
			      "Bad magic number (resolv_message)",0);

		    if (qitem->answer_buffer->message &&
			qitem->answer_buffer->message_len &&
			qitem->answer_buffer->message_parsed) {
			
			DPRINT(Debug,14,(&Debug,
					 "resolv_read_message: [pid %d] fd %d: %lu %s: have already valid answer message\n",
					 pid,fd,
					 (unsigned long)(qitem->querynum),
					 qitem->name ? qitem->name : "<NO NAME>"));
			
		    } else
			have_already_answer = 0;

		} else
		    have_already_answer = 0;


		if (have_already_answer &&
		    qitem->valid_answer) {

		    DPRINT(Debug,14,(&Debug,
				     "resolv_read_message: [pid %d] fd %d: lu% %s: ignoring res_send failing, seems already got valid answer\n",
				     pid,fd,
				     (unsigned long)(qitem->querynum),
				     qitem->name ? qitem->name : "<NO NAME>"));
		} else {
		    /* Store error message */
		    
		    qitem->failed_answer = 1;
		    
		    if (!qitem->answer_hdr)	
			qitem->answer_hdr    = safe_malloc(sizeof (* (qitem->answer_hdr)));
		    * (qitem->answer_hdr) = *hdr;
		    
		    DPRINT(Debug,14,(&Debug,
				     "resolv_read_message: [pid %d] fd %d: %lu %s: res_send failed\n",
				     pid,fd,
				     (unsigned long)(qitem->querynum),
				     qitem->name ? qitem->name : "<NO NAME>"));
		}		
	    } else {
		    DPRINT(Debug,14,(&Debug,
				     "resolv_read_message: [pid %d] fd %d: res_send failed, no query item\n",
				     pid,fd));

	    }
	    break;

	case qm_running:

	    if (proc) {
		if (ps_forked == proc->state) {

		    proc->state = ps_running;
		    DPRINT(Debug,14,(&Debug,
				     "resolv_read_message: [pid %d] fd %d: marking pid %d running\n",
				     pid,fd,proc->pid));
		} else {
		    DPRINT(Debug,14,(&Debug,
				     "resolv_read_message: [pid %d] fd %d: pid %d have unexpected state %d (got running)\n",
				     pid,fd,proc->pid,proc->state));
		}
	    }
	    break;

	case qm_shutdown:
	    if (proc) {
		if (ps_forked  == proc->state ||
		    ps_running == proc->state) {
		    proc->state = ps_shutdown;
		    
		    DPRINT(Debug,14,(&Debug,
				     "resolv_read_message: [pid %d] fd %d: marking pid %d shutdown\n",
				     pid,fd,proc->pid));
		} else if (ps_exited ==  proc->state) {
		    DPRINT(Debug,14,(&Debug,
				     "resolv_read_message: [pid %d] fd %d: pid %d already exited (got shutdown)\n",
				     pid,fd,proc->pid));
		} else {
		    DPRINT(Debug,14,(&Debug,
				     "resolv_read_message: [pid %d] fd %d: pid %d have unexpected state %d (got shutdown)\n",
				     pid,fd,proc->pid,proc->state));
		}
	    }

	    if (qitem) {
		DPRINT(Debug,14,(&Debug,
				 "resolv_read_message: [pid %d] fd %d: lu% %s: query got shutdown?\n",
				 pid,fd,
				 (unsigned long)(qitem->querynum),
				 qitem->name ? qitem->name : "<NO NAME>"));
		qitem->shutdown = 1;
	    }
	    
	    break;
	case qm_sizeupdate:
	    size_update.buffer =  state->read_buffer;
	    
	    if (state->read_buffer_len == sizeof (struct resolv_sizeupdate) &&
		RESOLV_SIZEUPDATE_magic == size_update.update->magic) {

		if (size_update.update->querysize >= sizeof (struct resolv_sizeupdate)) {

		    if (state->write_prepared &&
			size_update.update->querysize < state->write_buffer_len) {
			SIGDPRINT(Debug,14,(&Debug,
					    "resolv_read_message: [pid %d] fd %d: Ignoring querysize %lu => %d (write buffer size) -- write buffer len = %lu\n",
					    pid,fd,(unsigned long)(state->write_buffer_size),
					    size_update.update->querysize,
					    (unsigned long)(state->write_buffer_len)));
		    
		    } else if (size_update.update->querysize != state->write_buffer_size) {
			SIGDPRINT(Debug,14,(&Debug,
					    "resolv_read_message: [pid %d] fd %d: Changing querysize %lu => %d (write buffer size)\n",
					    pid,fd,(unsigned long)(state->write_buffer_size),size_update.update->querysize));
			state->write_buffer_size = size_update.update->querysize;
			state->write_buffer = safe_realloc(state->write_buffer,state->write_buffer_size);

		    }
		    
		} else {
		    SIGDPRINT(Debug,14,(&Debug, "resolv_read_message: [pid %d] fd %d: Bad size update message, querysize %d\n",
					pid,fd,size_update.update->querysize));
		}

		if (size_update.update->anssize >= sizeof (struct resolv_sizeupdate)) {
		    
		    if (size_update.update->anssize != state->read_buffer_size) {
			SIGDPRINT(Debug,14,(&Debug, "resolv_read_message: [pid %d] fd %d: Changing anssize  %lu => %d (read buffer size)\n",
					    pid,fd,(unsigned long)(state->read_buffer_size),
					    size_update.update->anssize));
			
			state->read_buffer_size = size_update.update->anssize;
			state->read_buffer = safe_realloc(state->read_buffer,state->read_buffer_size);
		    }
		   
		}  else {
		    SIGDPRINT(Debug,14,(&Debug, "resolv_read_message: [pid %d] fd %d: Bad size update message, anssize %d\n",
					pid,fd,size_update.update->anssize));
		}
		
	    } else {
		SIGDPRINT(Debug,14,(&Debug, "resolv_read_message: [pid %d] fd %d: Bad size update message\n",
				    pid,fd));
	    }

	    break;   
	}	
    }
	
    DPRINT(Debug,14,(&Debug, "resolv_read_message: [pid %d] result %d",pid,st));
    switch(st) {
    case action_disable:  DPRINT(Debug,14,(&Debug, " (disable)"));  break;
    case action_continue: DPRINT(Debug,14,(&Debug, " (continue)")); break;
    }
    DPRINT(Debug,14,(&Debug, "\n"));
    return st;
}

S_(action_routine resolv_write_message)
enum action_status resolv_write_message P_((int fd,
					   union action_routine_data         data,
					   const struct schedule_timelimit * now));
enum action_status resolv_write_message(fd,data,now)
     int fd;
     union action_routine_data         data;
     const struct schedule_timelimit * now;
{
    struct resolv_message_state * state = data.resolv;
    int pid = getpid();
    int err = 0;
    enum action_status st = action_continue;

    enum write_status ws;
    
    DPRINT(Debug,14,(&Debug, "resolv_write_message: [pid %d] socket %d writable\n",
		     pid,fd));

    if (RESOLV_MESSAGE_STATE_magic != state->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"resolv_write_message",
	      "Bad magic number (resolv_message_state)",0);

    if (!state->write_prepared) {
	size_t    previous = state->querynum;
	
	if (previous >= query_list_len) {
	    previous = 0;
	    state->querynum = 0;
	    DPRINT(Debug,14,(&Debug,
			     "resolv_write_message: [pid %d] socket %d -- resetting search of query to %zu\n",
			     pid,fd,previous)); 
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "resolv_write_message: [pid %d] socket %d -- starting search of query from %zu\n",
			     pid,fd,previous)); 
	}
	
	do {


	    if (state->querynum < query_list_len && query_list[state->querynum]) {
		if (RESOLV_QUERY_ITEM_magic != query_list[state->querynum]->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,"resolv_read_message",
			  "Bad magic number (resolv_query_item)",0);	

		if (! query_list[state->querynum]->query_sent &&
		    query_list[state->querynum]->query_buffer &&
		    query_list[state->querynum]->answer_buffer &&
		    query_list[state->querynum]->query_hdr) {
		    DPRINT(Debug,14,(&Debug, "resolv_write_message: [pid %d] socket %d -- selected query %zu\n",
				     pid,fd,state->querynum));
		    break;
		}
	    }

	    state->querynum++;
	    if (state->querynum >= query_list_len) {
		size_t X UNUSED_VAROK = state->querynum;
		state->querynum = 0;

		DPRINT(Debug,14,(&Debug, "resolv_write_message: [pid %d] socket %d -- reached query number %zu -- wrapping to %zu\n",
				 pid,fd,X,state->querynum));

	    }
	    
	} while (state->querynum != previous);

	if (state->querynum < query_list_len && query_list[state->querynum]) {

	    if (RESOLV_QUERY_ITEM_magic != query_list[state->querynum]->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,"resolv_read_message",
			  "Bad magic number (resolv_query_item)",0);
	    
	    if (! query_list[state->querynum]->query_sent &&
		query_list[state->querynum]->query_buffer &&
		query_list[state->querynum]->answer_buffer &&
		query_list[state->querynum]->query_hdr) {
		size_t querysize = RESOLV_BUFSIZE;
		size_t anssize = RESOLV_BUFSIZE;
		int queryupdate, ansupdate;
		
		if (RESOLV_MESSAGE_magic != query_list[state->querynum]->query_buffer->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,
			  "resolv_write_message",
			  "Bad magic number (resolv_message) of query_buffer",0);
		
		if (RESOLV_MESSAGE_magic != query_list[state->querynum]->answer_buffer->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,
			  "resolv_write_message",
			  "Bad magic number (resolv_message) of answer_buffer",0);

		if (RESOLV_QUERY_HDR_magic != query_list[state->querynum]->query_hdr->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,
			  "resolv_write_message",
			  "Bad magic number (resolv_query_hdr)",0);
		
		querysize = query_list[state->querynum]->query_buffer->message_size;
		anssize = query_list[state->querynum]->answer_buffer->message_size;
		
		queryupdate = ( querysize >= sizeof (struct resolv_sizeupdate) &&
				querysize != state->write_buffer_size );
		ansupdate   = ( anssize >= sizeof (struct resolv_sizeupdate) &&
				anssize !=  state->read_buffer_size ); 
		
		if (queryupdate || ansupdate) {
		    struct resolv_query_hdr  Z;
		    struct resolv_sizeupdate U = NULL_resolv_sizeupdate;
		    union resolv_sizeupdate_p update_buffer;
		    size_t wsize,rsize;

		    /* Struct assigment do not initalize whole 
		       struct resolv_query_hdr
		    */
		    
		    bzero((void *)&Z,sizeof Z);  
		    
		    Z = NULL_resolv_query_hdr;


		    DPRINT(Debug,14,(&Debug,
				     "resolv_write_message: [pid %d] socket %d: need update sizes: write buffer size %zu %s query buffer size %zu, read buffer size %zu %s answer buffer size %zu\n",
				     pid,fd,
				     state->write_buffer_size,
				     queryupdate ? "=>" : "~",
				     querysize,
				     state->read_buffer_size,
				     ansupdate ? "=>" : "~",
				     anssize));

		    wsize = queryupdate ? querysize : state->write_buffer_size;
		    rsize = ansupdate   ? anssize   : state->read_buffer_size;
		    
		    if (wsize < sizeof U)
			wsize = sizeof U;
		    if (rsize < sizeof U)
			rsize = sizeof U;
		    
		    if (wsize != state->write_buffer_size) {
			size_t old_size = state->write_buffer_size;
			
			 DPRINT(Debug,14,(&Debug,
					  "resolv_write_message: [pid %d] socket %d: changing write buffer size %zu => %zu\n",
					  pid,fd,
					  state->write_buffer_size,
					  wsize));

			 state->write_buffer_size = wsize;
			 state->write_buffer = safe_realloc(state->write_buffer,
							    state->write_buffer_size);
			 if (old_size < state->write_buffer_size)
			     bzero(state->write_buffer+old_size,
				   state->write_buffer_size - old_size);
		    }
		    U.querysize = state->write_buffer_size;

		    if (rsize != state->read_buffer_size) {
			size_t old_size = state->read_buffer_size; 
			DPRINT(Debug,14,(&Debug,
					 "resolv_write_message: [pid %d] socket %d: changing read buffer size %zu => %zu\n",
					 pid,fd,
					 state->read_buffer_size,
					 rsize));

			state->read_buffer_size = rsize;
			state->read_buffer = safe_realloc(state->read_buffer,state->read_buffer_size);
			if (old_size < state->read_buffer_size)
			    bzero(state->read_buffer+old_size,
				  state->read_buffer_size - old_size);
		    }
		    U.anssize   = state->read_buffer_size;
		    
		    update_buffer.update = &U;
		    Z.buflen    = sizeof U;
		    Z.senderpid = pid;
		    Z.msg       = qm_sizeupdate;

		    state->write_hdr = Z;
		    memcpy(state->write_buffer, update_buffer.buffer,sizeof U);
		    state->write_buffer_len = sizeof U;
		    state->write_pos = 0;
		    state->write_prepared = 1;
		    
		} else {
		    DPRINT(Debug,14,(&Debug,
				     "resolv_write_message: [pid %d] socket %d: writing query\n",
				     pid,fd));

		    if (state->write_buffer_size < query_list[state->querynum]->query_buffer->message_len ||
			query_list[state->querynum]->query_buffer->message_len != 
			query_list[state->querynum]->query_hdr->buflen)
			panic("RESOLV PANIC",__FILE__,__LINE__,
			      "resolv_write_message",
			      "Bad message_len",0);

		    state->write_hdr = * (query_list[state->querynum]->query_hdr);
		    if (query_list[state->querynum]->query_buffer->message_len &&
			query_list[state->querynum]->query_buffer->message) {
			memcpy(state->write_buffer,
			       query_list[state->querynum]->query_buffer->message,
			       query_list[state->querynum]->query_buffer->message_len);
			state->write_buffer_len = query_list[state->querynum]->query_buffer->message_len;
		    } else
			state->write_buffer_len = 0;

		    state->write_pos = 0;
		    state->write_prepared = 1;
		    query_list[state->querynum]->query_sent = 1;
		}
	    }
	}
    }
    
    if (state->write_prepared) {

	ws =  write_message(fd,&(state->write_hdr),state->write_buffer,state->write_buffer_len,
			    &err,pid,&(state->write_pos));
	
	DPRINT(Debug,14,(&Debug, "resolv_write_message: [pid %d] socket %d, got %d",
			 pid,fd,ws));
	switch (ws) {
	case ws_write_error:  SIGDPRINT(Debug,14,(&Debug, " (write error, error=%d)  ",err));
	    if (err != EINTR && err != EAGAIN && err != EWOULDBLOCK)
		st = action_disable;

	    if (EPIPE == err) {
		if (! state->broken_pipe) {
		    SIGDPRINT(Debug,14,(&Debug, ", marking broken pipe"));
		    state->broken_pipe = 1;
		}
	    }
	    if (state->querynum < query_list_len && query_list[state->querynum]) {

		if (RESOLV_QUERY_ITEM_magic != query_list[state->querynum]->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,"resolv_read_message",
			  "Bad magic number (resolv_query_item)",0);

		if (query_list[state->querynum]->query_sent) {
		    query_list[state->querynum]->last_errno = err;
		    SIGDPRINT(Debug,14,(&Debug, " -- query %zu errno %d set",
					state->querynum,
					err));
		}
	    }
	    break;
	case ws_none:         SIGDPRINT(Debug,14,(&Debug, " (none)          ")); break;
	case ws_sent_message: SIGDPRINT(Debug,14,(&Debug, " (sent message)  "));

	    if (state->querynum < query_list_len && query_list[state->querynum]) {
		
		if (RESOLV_QUERY_ITEM_magic != query_list[state->querynum]->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,"resolv_read_message",
			  "Bad magic number (resolv_query_item)",0);

		if (query_list[state->querynum]->query_sent) {
		    SIGDPRINT(Debug,14,(&Debug, " -- query %lu done",
					(unsigned long)(state->querynum)));
		    st = action_disable;

		    if (!state->query_in_progress) {
			SIGDPRINT(Debug,14,(&Debug, ", setting 'query in progress'"));
			state->query_in_progress = 1;
		    }		    
		}
	    }
	    state->write_prepared = 0;
	    break;
	case ws_incomplete:   SIGDPRINT(Debug,14,(&Debug, " (incomplete)     ")); break;
	case ws_continue:     SIGDPRINT(Debug,14,(&Debug, " (continue)       ")); break;
	}
	DPRINT(Debug,14,(&Debug, "\n"));
    } else {
	st = action_disable;
	DPRINT(Debug,14,(&Debug, "resolv_write_message: [pid %d] socket %d: No query available\n",
			 pid,fd));
    }

	
    DPRINT(Debug,14,(&Debug, "resolv_write_message: [pid %d] socket %d: result %d",pid,fd,st));
    switch(st) {
    case action_disable:  DPRINT(Debug,14,(&Debug, " (disable)"));  break;
    case action_continue: DPRINT(Debug,14,(&Debug, " (continue)")); break;
    }
    DPRINT(Debug,14,(&Debug, "\n"));
    return st;
}
#endif

/* -------------------------------------------------------------------------------------
   http://pubs.opengroup.org/onlinepubs/009695399/functions/xsh_chap02_10.html quote:

   The SOCK_SEQPACKET socket type is similar to the SOCK_STREAM type,
   and is also connection-oriented. The only difference between these
   types is that record boundaries are maintained using the
   SOCK_SEQPACKET type. A record can be sent using one or more output
   operations and received using one or more input operations, but a
   single operation never transfers parts of more than one
   record. Record boundaries are visible to the receiver via the
   MSG_EOR flag in the received message flags returned by the
   recvmsg() function. It is protocol-specific whether a maximum
   record size is imposed.

   --------------------------------------------------------------------------------------  */

S_(end_handler end_send_process)
static void end_send_process P_((union any_fd fd, char * title, 
				 struct run_state *rs, int ret, int exit_stat))
 UNUSED_FUNCOK;
static void end_send_process(fd,title,rs,ret,exit_stat)
     union any_fd fd;
     char * title;
     struct run_state *rs;
     int ret;
     int exit_stat;
{
    struct resolv_process   *proc = fd.resolv_process;

    if (RESOLV_PROCESS_magic != proc->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"end_send_process",
	      "Bad magic number (resolv_process)",0);

    DPRINT(Debug,16,(&Debug,
		     "end_send_process: resolv_process=%p\n",
		     fd.resolv_process));
   
    if (proc->state >= ps_forked && proc->state <= ps_shutdown) {
	if (ret < 0) {
	    proc->state = ps_died;
	    DPRINT(Debug,12,(&Debug,
			     "end_send_process: marking pid %d died\n",
			     proc->pid));
	} else {
	    proc->state = ps_exited;
	    DPRINT(Debug,12,(&Debug,
			     "end_send_process: marking pid %d exited\n",
			     proc->pid));

	}
    } else {
	DPRINT(Debug,12,(&Debug,
			 "end_send_process: pid %d have unexpected state %d\n",
			 proc->pid,proc->state));
    }
}

S_(free_any_fd_f free_send_process)
static void free_send_process P_((union any_fd *fd)) UNUSED_FUNCOK;
static void free_send_process(fd)
     union any_fd *fd;
{
    DPRINT(Debug,16,(&Debug,
		     "free_send_process: resolv_process=%p\n",
		     fd->resolv_process));

    free_resolv_process(& (fd->resolv_process));
}

S_(inc_any_fd_refcount_f inc_send_process_refount)
static void inc_send_process_refount  P_((union any_fd fd)) UNUSED_FUNCOK;
static void inc_send_process_refount(fd)
     union any_fd fd;
{
    DPRINT(Debug,16,(&Debug,
		     "inc_send_process_refount: resolv_process=%p\n",
		     fd.resolv_process));

    inc_resolv_process_refcount(fd.resolv_process);
}

static int resolv_process_ok P_((struct resolv_process *pr));
static int resolv_process_ok(pr)
     struct resolv_process *pr;
{
    int XX = 0;
    
    if (RESOLV_PROCESS_magic != pr->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"resolv_process_ok",
	      "Bad magic number (resolv_process)",0);
    
    DPRINT(Debug,14,(&Debug, "resolv_process_ok: [%p] ",pr));
    
    if (-1 != pr->pid) {
	DPRINT(Debug,14,(&Debug, "pid %d, ",pr->pid));
    }
    
    DPRINT(Debug,14,(&Debug, "state %d",pr->state));
    switch (pr->state) {
    case ps_none:    DPRINT(Debug,14,(&Debug, " (none)"));  break;
    case ps_forked:  DPRINT(Debug,14,(&Debug, " (forked)"));  break;
    case ps_running: DPRINT(Debug,14,(&Debug, " (running)"));
	if (-1 != pr->pid) {
	    XX  = 1;
	}
	break;
    case ps_shutdown: DPRINT(Debug,14,(&Debug, " (shutdown)")); break;
    case ps_exited:  DPRINT(Debug,14,(&Debug, " (exited)")); break;
    case ps_died:    DPRINT(Debug,14,(&Debug, " (died)")); break;
    }
    
    if (-1 != pr->socket) {
	DPRINT(Debug,14,(&Debug, ", socket %d",
			 pr->socket));
	
    } else
	XX = 0;	
        
    DPRINT(Debug,14,(&Debug, " => result %d%s\n",
		     XX,XX ? " OK ": ""));
   
    return XX;
}

#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET) && defined(BACKGROUD_PROCESSES)
static struct resolv_process * give_resolv_process P_((int pid0));
static struct resolv_process * give_resolv_process(pid0)
     int pid0;
{
    struct resolv_process * ret = NULL;

    int com_socket_pair[2];
    int r = socketpair(AF_UNIX,SOCK_SEQPACKET,0,com_socket_pair);

    if (0 == r) {
	struct resolv_message_state * st;
	char title[80];
	union any_fd FD;
	int pid;
	int r;
	
	union action_routine_data    data;
	
	DPRINT(Debug,12,(&Debug, "give_resolv_process: Got socket %d <-> %d\n",
			 com_socket_pair[0],com_socket_pair[1]));

	pid = fork();
	/* Add one worker process */
	
	if (0 == pid) {   /* Child */
	    int p =  getpid();
	    
	    DPRINT(Debug,12,(&Debug,
			     "give_resolv_process: on CHILD (pid %d)\n",
			     p));

	    clear_other_actions(p);
	    
	    close(com_socket_pair[0]);
	    resolv_send_process(com_socket_pair[1]);
	    
	    exit(0);
	    
	} else if (-1 == pid) {
	    int err = errno;

	    DPRINT(Debug,12,(&Debug,
			     "give_resolv_process: fork() failed: %s\n",
			     strerror(err)));
	    
	    lib_error(CATGETS(elm_msg_cat, 
			      ResolvSet,
			      ResolvFailedChildResSend,
			      "Failed to create child prosess for res_send: %s"),
			      strerror(err));
	    goto fail;	      
	}

	DPRINT(Debug,12,(&Debug,
			 "give_resolv_process: #%lu, pid %d created\n",
			 (unsigned long)process_list_len,
			 pid));
	
	close(com_socket_pair[1]);
	st = new_resolv_message_state(pid0);
	data.resolv = st;

	if (-1 == (r = fcntl(com_socket_pair[0],F_SETFL,O_NONBLOCK))) {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,12,(&Debug,
			     "give_resolv_process: fcntl [F_SETFL O_NONBLOCK] failed: %s\n",
			     strerror(err)));
	} else if (0 == r) {
	    DPRINT(Debug,12,(&Debug,
			     "give_resolv_process: fcntl(%d,F_SETFL,O_NONBLOCK) done\n",
			     com_socket_pair[0]));
	}

	if (-1 == (r = fcntl(com_socket_pair[0],F_SETFD,FD_CLOEXEC))) {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,12,(&Debug,
			     "give_resolv_process: fcntl [F_SETFD FD_CLOEXEC] failed: %s\n",
			     strerror(err)));
	} else if (0 == r) {
	    DPRINT(Debug,12,(&Debug,
			     "give_resolv_process: fcntl(%d,F_SETFD,FD_CLOEXEC) done\n",
			     com_socket_pair[0]));
	}
	
	change_action2(com_socket_pair[0],-1,resolv_read_message,
		       no_action_routine,no_action_routine,
		       resolv_free_state,resolv_inc_state_refcount,
		       resolv_badpid_action,
		       data);
		
	ret = new_resolv_process(pid);
	ret->state = ps_forked;
	ret->message = st;
	ret->socket  = com_socket_pair[0];
		
	process_list = safe_array_realloc(process_list,
					  process_list_len+1,
					  sizeof (process_list[0]));
		
	process_list[process_list_len++] = ret;

	elm_sfprintf(title, sizeof title,
		     CATGETS(elm_msg_cat, 
			     ResolvSet,
			     ResolvLookupHelper,
			     "Lookup (res_send) helper (pid %d)"),
		     pid);
	FD.resolv_process = ret;
	add_foreign_background(pid,FD,title,end_send_process,
			       free_send_process,inc_send_process_refount);

	inc_resolv_process_refcount(ret);
    } else if (-1 == r) {
	int err = errno;

	DPRINT(Debug,12,(&Debug,
			 "give_resolv_process: socketpair() failed: %s\n",
			 strerror(err)));
	
	lib_error(CATGETS(elm_msg_cat, 
			  ResolvSet,
			  ResolvFailedSocketPair,
			  "Failed to create socket pair: %s"),
		  strerror(err));
    }
	
 fail:
    if (ret) {
	DPRINT(Debug,12,(&Debug,
			 "give_resolv_process=%p: refcount=%d\n",
			 ret,ret->refcount));
    } else {
	DPRINT(Debug,12,(&Debug,
			 "give_resolv_process=NULL\n"));
    }
    
    return ret;
}

#endif

S_(have_qm_f have_non_blocking_qm)
static  int have_non_blocking_qm P_((void));
static int have_non_blocking_qm()
{
    int ret = 0;
    int pid0 = getpid();
    
    if (process_list) {
	size_t i;
	
	for (i = 0; i < process_list_len && !ret; i++) {
	    if (process_list[i]) {
		int XX = 0;
		if (RESOLV_PROCESS_magic != process_list[i]->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,"have_non_blocking_qm",
			  "Bad magic number (resolv_process)",0);

		XX = resolv_process_ok(process_list[i]);
		
		DPRINT(Debug,12,(&Debug, "have_non_blocking_qm: #%zu",
				 i));		
		if (XX) {
		    DPRINT(Debug,12,(&Debug, " -- OK"));
		    ret = 1;
		}
		DPRINT(Debug,12,(&Debug, "\n"));

		if (XX && process_list[i]->message) {
		     union action_routine_data    data UNUSED_VAROK;
		     
		     if (RESOLV_MESSAGE_STATE_magic != process_list[i]->message->magic)
			 panic("RESOLV PANIC",__FILE__,__LINE__,"have_non_blocking_qm",
			       "Bad magic number (resolv_message_state)",0);

		     if (process_list[i]->message->broken_pipe) {
			 DPRINT(Debug,12,(&Debug,
					  "                    : #%zu -- broken pipe (or EOF)\n",
					  i));

			 /* Not handled here */
		     }

		     if (process_list[i]->message->write_prepared) {
			 DPRINT(Debug,12,(&Debug,
					  "                    : #%zu -- write prepared (busy)\n",
					  i));
		     }

		     if (process_list[i]->message->query_in_progress) {
			 DPRINT(Debug,12,(&Debug,
					  "                    : #%zu -- query in progress (busy)\n",
					  i));
		     }		    
		     
#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET)
		     data.resolv = process_list[i]->message;
		     /* This should be no-op */
		     change_action2(process_list[i]->socket,-1,
				    resolv_read_message,
				    process_list[i]->message->write_prepared ?
				    resolv_write_message : no_action_routine,
				    no_action_routine,
				    resolv_free_state,resolv_inc_state_refcount,
				    resolv_badpid_action,
				    data);
#endif		     

		} else if (process_list[i]->message) {
		    
		    if (RESOLV_MESSAGE_STATE_magic != process_list[i]->message->magic)
			panic("RESOLV PANIC",__FILE__,__LINE__,"have_non_blocking_qm",
			      "Bad magic number (resolv_message_state)",0);

		    if (process_list[i]->message->broken_pipe) {
			DPRINT(Debug,12,(&Debug,
					 "                    : #%zu -- broken pipe (or EOF)\n",
					 i));
		    }

		    if (process_list[i]->message->write_prepared) {
			DPRINT(Debug,12,(&Debug,
					 "                    : #%zu -- write prepared (busy)\n",
					 i));
		    }

		    if (process_list[i]->message->query_in_progress) {
			DPRINT(Debug,12,(&Debug,
					 "                    : #%zu -- query in progress (busy)\n",
					 (unsigned long)i));
		    }		    
		}

	    } else {
		DPRINT(Debug,15,(&Debug,
				 "have_non_blocking_qm: #%zu -- unused entry\n",
				 i));		
	    }
	}	
    }

    if (ret) {
	DPRINT(Debug,12,(&Debug, "have_non_blocking_qm: Available\n"));
    }
    
    if (!ret) {

	if (qm_non_blocking == query_mode) {
#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET) && defined(BACKGROUD_PROCESSES)

	    struct resolv_process * pr = give_resolv_process(pid0);
	    if (pr) {
		int r = res_init();

		if (0 == r) {

		    
		    DPRINT(Debug,12,(&Debug, "have_non_blocking_qm: Started\n"));
		    ret = 1;

		} else if (r < 0) {
		    int err = errno;

		    DPRINT(Debug,12,(&Debug,
				     "have_non_blocking_qm: res_init failed: errno = %d (%s)\n",
				     err,strerror(err)));
		    
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvResInitFailed,
				      "Failed to initialize with res_init: %s"),
			      strerror(err));				      
		}

		if (RESOLV_PROCESS_magic != pr->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,"have_non_blocking_qm",
			  "Bad magic number (resolv_process)",0);

		
		DPRINT(Debug,12,(&Debug,
				 "have_non_blocking_qm: helper process [%p], pid %d\n",
				 pr,pr->pid));
		
		free_resolv_process(&pr);
	    } else {
		DPRINT(Debug,12,(&Debug, "have_non_blocking_qm: Failed to add process for res_send()\n"));

	    }
#else
	    DPRINT(Debug,12,(&Debug, "have_non_blocking_qm: Not supported\n"));	    
#endif
	} else {
	    DPRINT(Debug,12,(&Debug, "have_non_blocking_qm: query-mode=%d",
			     query_mode));
	    if (query_mode >= 0 && query_mode < NUM_query_mode) {
		DPRINT(Debug,12,(&Debug, " %s",
				 QUERY_MODE[query_mode]));
	    }
	    DPRINT(Debug,12,(&Debug, "\n"));
	}
    }
           
    DPRINT(Debug,12,(&Debug, "have_non_blocking_qm=%d (pid %d)\n",ret,pid0));
    
    return ret;
}

static int prepare_write_message P_((struct resolv_message_state *st, int socket,
				      struct resolv_query_hdr *result_hdr,
				      unsigned char * buffer, size_t buffer_len))
     UNUSED_FUNCOK;
static int prepare_write_message(st,socket,result_hdr,buffer,buffer_len)
     struct resolv_message_state *st;
     int socket;
     struct resolv_query_hdr *result_hdr;
     unsigned char * buffer;
     size_t buffer_len;
{
    int ret = 0;
    int pid = getpid();
    
    if (RESOLV_MESSAGE_STATE_magic != st->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,"prepare_write_message",
	      "Bad magic number (resolv_message_state)",0);

     if (RESOLV_QUERY_HDR_magic != result_hdr->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "prepare_write_message",
	      "Bad magic number (resolv_query_hdr)",0);
    
    if (result_hdr->buflen != buffer_len)
	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "prepare_write_message",
	      "Bad buf_len",0);

    if (st->write_prepared && st->write_pos) {
	SIGDPRINT(Debug,14,(&Debug,
			    "prepare_write_message: [pid %d] socket=%d on middle of write\n",
			    pid,socket));
	ret = 0;
    } else {
	union action_routine_data    data UNUSED_VAROK;
    
	if (buffer_len > st->write_buffer_size) {
	    struct resolv_query_hdr  Z = NULL_resolv_query_hdr;
	    struct resolv_sizeupdate U = NULL_resolv_sizeupdate;
	    union resolv_sizeupdate_p update_buffer;
	    
	    unsigned short A = buffer_len;
	    if (A < RESOLV_BUFSIZE)
		A = RESOLV_BUFSIZE;
	    if (A < sizeof U)
		A = sizeof U;
	    
	    SIGDPRINT(Debug,14,(&Debug,
				"prepare_write_message: [pid %d] socket=%d: Changing write buffer size %lu => %d (need write %lu bytes)\n",
				pid,socket,(unsigned long)(st->write_buffer_size),A,(unsigned long)buffer_len));
	    
	    st->write_buffer_size = A;
	    st->write_buffer = safe_realloc(st->write_buffer,st->write_buffer_size);

	    U.querysize = st->write_buffer_size;
	    U.anssize   = st->read_buffer_size;
	    update_buffer.update = &U;

	    Z.buflen    = sizeof U;
	    Z.senderpid = pid;
	    Z.msg       = qm_sizeupdate;

	    st->write_hdr = Z;
	    memcpy(st->write_buffer, update_buffer.buffer,sizeof U);
	    st->write_buffer_len = sizeof U;

	    ret = 0;
	} else {
	    st->write_hdr = *result_hdr;
	    
	    SIGDPRINT(Debug,14,(&Debug,
				"prepare_write_message: [pid %d] socket=%d Preparing write %lu bytes (buffer size %lu)\n",
				pid,socket,(unsigned long)buffer_len,(unsigned long)(st->write_buffer_size)));
	    if (buffer_len)  {
		memcpy(st->write_buffer,buffer,buffer_len);
		st->write_buffer_len = buffer_len;
	    } else {
		st->write_buffer_len = 0;
	    }

	    ret = 1;
	}
	
	st->write_pos = 0;
	st->write_prepared = 1;
	data.resolv = st;

#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET)	
	change_action2(socket,-1,resolv_read_message,resolv_write_message,
		       no_action_routine,
		       resolv_free_state,resolv_inc_state_refcount,
		       resolv_badpid_action,
		       data);
#else
	ret = 0;
#endif
    }

    SIGDPRINT(Debug,14,(&Debug, "prepare_write_message=%d: [pid %d] socket=%d\n",
			ret,pid,socket));
    
    return ret;
}

void free_non_blocking_qm()
{
    int mypid UNUSED_VAROK = getpid();
    /* Free resources alloced by  have_non_blocking_qm() */

    if (process_list) {
	size_t i;
	int have_write UNUSED_VAROK = 0;
	
	for (i = 0; i <  process_list_len; i++) {

	    if (process_list[i]) {
		int shut UNUSED_VAROK = 0;
		if (RESOLV_PROCESS_magic != process_list[i]->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,"free_non_blocking_qm",
			  "Bad magic number (resolv_process)",0);
		
		DPRINT(Debug,12,(&Debug,"free_non_blocking_qm: process #%zu: pid=%d, state=%d",
				 i,
				 process_list[i]->pid,
				 process_list[i]->state));
		
		switch(process_list[i]->state) {
		case ps_none:     SIGDPRINT(Debug,12,(&Debug, " (none)    ")); break;
		case ps_forked:   SIGDPRINT(Debug,12,(&Debug, " (forked)  "));
		    shut = 1;
		    break;
		case ps_running:  SIGDPRINT(Debug,12,(&Debug, " (running) "));
		    shut = 1;
		    break;
		case ps_shutdown: SIGDPRINT(Debug,12,(&Debug, " (shutdown)")); break;
		case ps_exited:   SIGDPRINT(Debug,12,(&Debug, " (exited)  ")); break;
		case ps_died:     SIGDPRINT(Debug,12,(&Debug, " (died)    ")); break; 
		}
		
		if (-1 != process_list[i]->socket) {
		    DPRINT(Debug,12,(&Debug,", socket=%d",
				     process_list[i]->socket));
		} else
		    shut = 0;
		
		if (process_list[i]->message) {
		    if (RESOLV_MESSAGE_STATE_magic != process_list[i]->message->magic)
			panic("RESOLV PANIC",__FILE__,__LINE__,"free_non_blocking_qm",
			      "Bad magic number (resolv_message_state)",0);
		    
		    if (! process_list[i]->message->write_prepared) {
			DPRINT(Debug,12,(&Debug,", write available"));
		    } else
			shut = 0;
		} else
		    shut = 0;
		DPRINT(Debug,12,(&Debug,"\n"));
		
#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET)
		if (shut) {
		    int r;
		    struct resolv_query_hdr Z;

		    /* Struct assigment do not initalize whole 
		       struct resolv_query_hdr
		    */
		    
		    bzero((void *)&Z,sizeof Z);  
		    
		    Z = NULL_resolv_query_hdr;
		    
		    Z.senderpid = mypid;
		    Z.msg       = qm_shutdown;
		    
		    r = prepare_write_message(process_list[i]->message,process_list[i]->socket,
					  &Z,NULL,0);

		    if (!r) {
			enum wait_for_status rz;
					    
			DPRINT(Debug,12,(&Debug,"free_non_blocking_qm: process #%d: Pausing for socket %d\n",
					 i,process_list[i]->socket));
					 
			if ((rz = wait_for_action_or_timeout(resolv_write_message,1)) > wait_for_none)
			    r = prepare_write_message(process_list[i]->message,process_list[i]->socket,
						      &Z,NULL,0);
			else {
			    switch (rz) {
				int err UNUSED_VAROK;
			    case wait_for_done:   break;
			    case wait_for_none:
				DPRINT(Debug,12,(&Debug,
						 "free_non_blocking_qm:  unexpected result from wait_for_action_or_timeout\n"));
				break;
			    case wait_for_error:
				err = errno;
				DPRINT(Debug,12,(&Debug,"free_non_blocking_qm:  wait  errno=%d, %s\n",
						 err,strerror(err)));
				break;
			    }
			}
		    }
		    
		    if (r)
			have_write = 1;
		}
#endif
	    }
	}

#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET)
	if (have_write)  {
	    DPRINT(Debug,12,(&Debug,
			     "free_non_blocking_qm: Waiting for write\n"));
	    wait_for_action_or_timeout(resolv_write_message,0);

	    /* Not really work */
	    
	    DPRINT(Debug,12,(&Debug,
			     "free_non_blocking_qm: Waiting for read\n"));

	    wait_for_action_or_timeout(resolv_read_message,1);	    
	}
#endif

#ifdef BACKGROUD_PROCESSES
	if (handle_sigchld) {
	    DPRINT(Debug,12,(&Debug,"free_non_blocking_qm: checking for died processes.\n"));
	    sigchld_handler();
	}

	
	for (i = 0; i <  process_list_len; i++) {

	    if (process_list[i]) {
		if (RESOLV_PROCESS_magic != process_list[i]->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,"free_non_blocking_qm",
			  "Bad magic number (resolv_process)",0);
		
		if (process_list[i]->state >= ps_forked &&
		    process_list[i]->state <= ps_exited) {
		    int r;
		    
		    DPRINT(Debug,12,(&Debug,"free_non_blocking_qm: process #%zu: pid=%d, ",
				     i,
				     process_list[i]->pid));
		    
		    r = kill (process_list[i]->pid,SIGTERM);
		    if (0 == r) {
			int s;
			int exit_code;
			DPRINT(Debug,12,(&Debug,"sent SIGTERM, waiting for termination\n"));
			
			s =  wait_background_end(process_list[i]->pid,&exit_code);
			
			DPRINT(Debug,12,(&Debug,"free_non_blocking_qm: process #%lu: pid=%d, ",
					 (unsigned long)i,
					 process_list[i]->pid));
			switch (s) {
			case 1: DPRINT(Debug,12,(&Debug,"exited with status %d",exit_code)); break;
			case 2: DPRINT(Debug,12,(&Debug,"no process")); break;
			}
			if (s < 0) {
			    DPRINT(Debug,12,(&Debug,"Dies with signal %d%s%s",
					     -s,
					     -s == SIGTERM ? " (SIGTERM)" : "",
					     -s == SIGINT  ? " (SIGINT)" : ""
					     ));
			}			
		    } else if (-1 == r) {
			int err UNUSED_VAROK = errno;
			SIGDPRINT(Debug,12,(&Debug, "error %d: %s",
					    err,strerror(err)));
		    }
		    DPRINT(Debug,12,(&Debug,"\n"));
		}
	    }
	}
	if (handle_sigchld) {
	    DPRINT(Debug,12,(&Debug,"free_non_blocking_qm: checking for died processes.\n"));
	    sigchld_handler();
	}     	    
#endif

	for (i = 0; i <  process_list_len; i++) {
	    
	    if (process_list[i]) {
				
		if (RESOLV_PROCESS_magic != process_list[i]->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,"free_non_blocking_qm",
			  "Bad magic number (resolv_process)",0);

		DPRINT(Debug,12,(&Debug,"free_non_blocking_qm: process #%zu: ",
				 i));

		if (-1 != process_list[i]->socket) {
		    enum syscall_status r;
		    
		    DPRINT(Debug,12,(&Debug,"socket=%d  -- clear action\n",
				     process_list[i]->socket));

		    /* No longer schedule for this! */
		    clear_action(process_list[i]->socket);
		    
		    DPRINT(Debug,12,(&Debug,"free_non_blocking_qm: process #%zu: closing socket %d",
				     i,process_list[i]->socket));

		    r = close(process_list[i]->socket);
		    switch (r) {
			int err UNUSED_VAROK;
		    case syscall_error: /*-1 */		         
			err = errno;
			DPRINT(Debug,12,(&Debug,", failed; errno %d (%s)",
					 err,strerror(err)));
			break;
		    case syscall_success: /* 0 */
			DPRINT(Debug,12,(&Debug,", succeed"));
			break;
		    }
	
		    DPRINT(Debug,12,(&Debug,".\n"));
	
		    process_list[i]->socket = -1;		    
		} else {
		    DPRINT(Debug,12,(&Debug,"no socket\n"));
		}		
	    }
	}

	
	for (i = 0; i <  process_list_len; i++) {
	    
	    if (process_list[i]) {
		DPRINT(Debug,12,(&Debug,"free_non_blocking_qm: process #%zu: Freeing\n",
				 i));

		free_resolv_process(& (process_list[i]));

	    }
	}
	free(process_list);
	process_list = NULL;
	
    }    
    process_list_len = 0;

    if (query_list) {
	size_t i;

	for (i = 0; i < query_list_len; i++) {
	    if (query_list[i]) {
		DPRINT(Debug,12,(&Debug,"free_non_blocking_qm: query item #%zu: Freeing\n",
				 i));

		free_resolv_query_item(&  (query_list[i]));
	    }
	}
	free(query_list);
	query_list = NULL;
    }
    query_list_len   = 0;

}

#define MAX_PROCESS 2
static int prx_max_processes = MAX_PROCESS;

#define MAX_SKIP_PROCESS 5
static int  prx_max_skip_process = MAX_SKIP_PROCESS;

static const int MAX_prx_processes = (MAX_PROCESS + MAX_SKIP_PROCESS > 100) ? (MAX_PROCESS + MAX_SKIP_PROCESS) : 100;



static void  select_resolv_process P_((int pid,size_t querynum,const char *name));
static void  select_resolv_process(pid,querynum,name)
     int          pid;
     size_t       querynum;
     const char * name;
{
    struct resolv_process * prx_static[MAX_PROCESS];
    struct resolv_process ** prx = prx_static;
    size_t                  prx_len = sizeof (prx_static) / sizeof (prx_static[0]);
    size_t                  prx_count = 0;
    size_t t UNUSED_VAROK;

    DPRINT(Debug,16,(&Debug,
		     "select_resolv_process: prx_max_processes=%d prx_max_skip_process=%d process_list_len=%zu\n",
		     prx_max_processes,
		     prx_max_skip_process,
		     process_list_len
		     ));

#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET)    
    if (process_list) {
        static size_t prindex = 0;
	size_t previous_prindex = prindex;
	size_t i;
	int have_forked = 0;
	int check_pipe = 0;
	
	int skip_count = 0;
       
	for (i = 0; i < process_list_len; i++) {
	    if (process_list[i]) {
		if (RESOLV_PROCESS_magic != process_list[i]->magic)
		    panic("RESOLV PANIC",__FILE__,__LINE__,"select_resolv_process",
			  "Bad magic number (resolv_process)",0);

		if (ps_forked == process_list[i]->state &&
		    -1 !=  process_list[i]->pid) {
		    DPRINT(Debug,14,(&Debug,
				     "select_resolv_process: process index %zu ([%p], pid %d) have state forked\n",
				     i,process_list[i],
				     process_list[i]->pid));
		    have_forked = 1;
		    break;
		} else if (ps_shutdown < process_list[i]->state &&
			   -1 !=  process_list[i]->pid) {
		    DPRINT(Debug,14,(&Debug,
				     "select_resolv_process: process index %zu (pid %d) have process ended",
				     i,process_list[i],
				     process_list[i]->pid));
		    if (process_list[i]->message) {
			if (RESOLV_MESSAGE_STATE_magic != process_list[i]->message->magic)
			    panic("RESOLV PANIC",__FILE__,__LINE__,"select_resolv_process",
				  "Bad magic number (resolv_message_state)",0);
			
			if (process_list[i]->message->broken_pipe) {
			    DPRINT(Debug,14,(&Debug, "-- broken pipe (or EOF)"));			    
			} else {
			    DPRINT(Debug,14,(&Debug, "-- EOF not read yet"));
			    check_pipe = 1;
			}
		    }
		    DPRINT(Debug,14,(&Debug, "\n"));		    
		}
	    }
	}

	if (have_forked) {
	    DPRINT(Debug,14,(&Debug,
			     "select_resolv_process: [pid %d] waiting running message from forked process.\n",
			     pid));
	    wait_for_action_or_timeout(resolv_read_message,1);
	} else if (check_pipe) {
	     DPRINT(Debug,14,(&Debug, "select_resolv_process: [pid %d] polling EOF from  pipe\n",
			      pid));
	     wait_for_action_or_timeout(resolv_read_message,0);
	}
	
	do {
	    prindex++;
	    
	    if (prindex >= process_list_len) {
		size_t X UNUSED_VAROK = prindex;
		
		prindex  = 0;
		DPRINT(Debug,14,(&Debug,
				 "select_resolv_process: [pid %d] query %zu %s: reached process index %zu -- wrapping to %zu\n",
				 pid,
				 querynum,
				 name,
				 X,prindex));
	    }

	    if (prindex  < process_list_len &&
		prx_count < prx_max_processes) {

		if (prx_count > prx_len)
		    panic("RESOLV PANIC",__FILE__,__LINE__,
			  "select_resolv_process",
			  "Overflow",0);
		else if (prx_count == prx_len) {

		    if (prx == prx_static &&
			prx_max_processes > prx_count) {
			size_t a;
			
			DPRINT(Debug,14,(&Debug,
					 "select_resolv_process: allocating list, len %zu",
					 prx_len));
			
			prx_len = prx_max_processes;
			if (prx_len > process_list_len+1 &&
			    process_list_len+1 > prx_count) {
			    DPRINT(Debug,14,(&Debug, " (max %d)",
					     prx_max_processes));
			    prx_len = process_list_len+1;
			}

			DPRINT(Debug,14,(&Debug, " => %zu\n",
					 prx_len));
			
			prx = safe_calloc(prx_len,sizeof(prx[0]));
			for (a = 0;
			     a < prx_count &&
				 a < sizeof (prx_static) / sizeof (prx_static[0]);
			     a++)
			    prx[a] = prx_static[a];
						
		    } else {
			DPRINT(Debug,14,(&Debug,
					 "select_resolv_process: allocating list, len %zu -- something wrong\n",
					 prx_len));
			break;
		    }
		}
	       
		if (process_list[prindex]) {
		    
		    if (RESOLV_PROCESS_magic != process_list[prindex]->magic)
			panic("RESOLV PANIC",__FILE__,__LINE__,"select_resolv_process",
			      "Bad magic number (resolv_process)",0);
		    
		    if (resolv_process_ok(process_list[prindex])) {
			
			DPRINT(Debug,14,(&Debug,
					 "select_resolv_process: [pid %d] query %zu %s: process index %zu is available ([%p], pid %d, socket %d)\n",
					 pid,
					 querynum,
					 name,
					 prindex,
					 process_list[prindex],
					 process_list[prindex]->pid,
					 process_list[prindex]->socket));

			
			if (process_list[prindex]->message) {
			    if (RESOLV_MESSAGE_STATE_magic != process_list[prindex]->message->magic)
				panic("RESOLV PANIC",__FILE__,__LINE__,"select_resolv_process",
				      "Bad magic number (resolv_message_state)",0);
			    
			    if (process_list[prindex]->message->broken_pipe) {
				int r;
				int may_delete = 0;
				
				DPRINT(Debug,14,(&Debug,
						 "select_resolv_process: process #%zu: ([%p], pid %d, socket %d) -- broken pipe (or EOF)",
						 prindex,
						 process_list[prindex],
						 process_list[prindex]->pid,
						 process_list[prindex]->socket));
				
				r = kill(process_list[prindex]->pid,
					 SIGTERM);
				
				if (0 == r) {
				    int s;
				    int exit_code;
				    
				    DPRINT(Debug,14,(&Debug,"; sent SIGTERM, waiting for termination\n"));
				    
				    s =  wait_background_end(process_list[prindex]->pid,&exit_code);
				    
				    DPRINT(Debug,14,(&Debug,
						     "select_resolv_process: process #%zu: [%p] pid=%d, ",
						     prindex,
						     process_list[prindex],
						     process_list[prindex]->pid));
				    switch (s) {
				    case 1:
					DPRINT(Debug,14,(&Debug,"exited with status %d",exit_code));
					may_delete = 1;
					break;
				    case 2: DPRINT(Debug,14,(&Debug,"no process"));
					may_delete = 1;
					break;
				    }
				    if (s < 0) {
					DPRINT(Debug,14,(&Debug,"Dies with signal %d%s%s",
						     -s,
							 -s == SIGTERM ? " (SIGTERM)" : "",
							 -s == SIGINT  ? " (SIGINT)" : ""
							 ));
					may_delete = 1;
				    }			
				} else if (-1 == r) {
				    int err = errno;
				    SIGDPRINT(Debug,14,(&Debug, "; tried send SIGTERM, got error %d: %s",
							err,strerror(err)));
				    
				    if (ESRCH == err)
					may_delete = 1;
				}
				
				if (may_delete)
				    goto cleanup_process;

				DPRINT(Debug,14,(&Debug,"\n"));
				
				goto skip_it;
			    }
				
			    if (process_list[prindex]->message->write_prepared ||
				process_list[prindex]->message->query_in_progress) {
				DPRINT(Debug,14,(&Debug,
						 "select_resolv_process: process #%zu: ([%p], pid %d, socket %d)%s%s\n",
						 prindex,
						 process_list[prindex],
						 process_list[prindex]->pid,
						 process_list[prindex]->socket,
						 process_list[prindex]->message->write_prepared ? 
						 " -- write prepared (busy)" : "",
						 process_list[prindex]->message->query_in_progress ?
						 " -- query in progress (busy)" : ""));

				if (skip_count >= prx_max_skip_process) {
				    DPRINT(Debug,14,(&Debug,
						     "select_resolv_process: Already %d skipped, using anyway\n",
						     skip_count));
				} else {
				    skip_count ++;
				    goto skip_it;
				}
			    }
			    
			}
			    
			prx[prx_count] = process_list[prindex];
			inc_resolv_process_refcount(prx[prx_count]);
			prx_count++;
		    } else if (ps_shutdown < process_list[prindex]->state &&
			       -1 !=  process_list[prindex]->pid) {
			DPRINT(Debug,14,(&Debug,
					 "select_resolv_process: [pid %d] query %zu %s: process index %zu not available ([%p], pid %d, socket %d)",
					 pid,
					 querynum,
					 name,
					 prindex,
					 process_list[prindex],
					 process_list[prindex]->pid,
					 process_list[prindex]->socket));
			
			if (process_list[prindex]->message) {
			    if (RESOLV_MESSAGE_STATE_magic != process_list[prindex]->message->magic)
				panic("RESOLV PANIC",__FILE__,__LINE__,"select_resolv_process",
				      "Bad magic number (resolv_message_state)",0);
			    
			    if (process_list[prindex]->message->broken_pipe) {
				DPRINT(Debug,14,(&Debug," -- broken pipe (or EOF)"));
				goto cleanup_process;
			    }
			    DPRINT(Debug,14,(&Debug,", ignored now\n"));
			    
			} else 
			    goto cleanup_process;
		    } else if (-1 ==  process_list[prindex]->pid) {
			DPRINT(Debug,14,(&Debug,
					 "select_resolv_process: [pid %d] query %zu %s: process index %zu ([%p]) not available (no process?)",
					 pid,
					 querynum,
					 name,
					 prindex,
					 process_list[prindex]));

		    cleanup_process:
			SIGDPRINT(Debug,14,(&Debug,
					    "; may delete process #%zu index entry\n",
					    prindex));
			
			free_resolv_process(& (process_list[prindex]));
			
		    }
		} else {
		    DPRINT(Debug,15,(&Debug,
				     "select_resolv_process: [pid %d] query %zu %s: process index %zu unused entry\n",
				     pid,
				     querynum,
				     name,
				     prindex));
		}
	    }

	skip_it: ;
	    
	} while (prindex != previous_prindex &&
		 prx_count < prx_max_processes);
    }
    
    if (prx_count < prx_max_processes &&
	prx_count < prx_len) {
	
	prx[prx_count] = give_resolv_process(pid);
	
	DPRINT(Debug,14,(&Debug,
			 "select_resolv_process: [pid %d] query %zu %s: added process ([%p], pid %d, socket %d)\n",
			 pid,
			 querynum,
			 name,
			 prx[prx_count],
			 prx[prx_count]->pid,
			 prx[prx_count]->socket));
	
	prx_count++;
    }

    for (t = 0; t < prx_count; t++) {
	
	union action_routine_data    data;
	
	if (RESOLV_PROCESS_magic != prx[t]->magic)
	    panic("RESOLV PANIC",__FILE__,__LINE__,"select_resolv_process",
		  "Bad magic number (resolv_process)",0);

	if (prx[t]->message) {
	    if (RESOLV_MESSAGE_STATE_magic != prx[t]->message->magic)
		panic("RESOLV PANIC",__FILE__,__LINE__,"select_resolv_process",
		      "Bad magic number (resolv_message_state)",0);
	    
	} else if (! prx[t]->message) {
	    DPRINT(Debug,14,(&Debug,
			     "select_resolv_process: [pid %d] query %zu %s: adding message state for [%p] pid %d (socket %d)\n",
			     pid,
			     querynum,			     
			     name,
			     prx[t],
			     prx[t]->pid,
			     prx[t]->socket));
	    prx[t]->message =
		new_resolv_message_state(pid /* my pid -- senderpid */);
	} 

	data.resolv = prx[t]->message;

	change_action2(prx[t]->socket,-1,resolv_read_message,resolv_write_message,
		       no_action_routine,
		       resolv_free_state,resolv_inc_state_refcount,
		       resolv_badpid_action,
		       data);

	DPRINT(Debug,14,(&Debug,
			 "select_resolv_process: [pid %d] query %zu %s: scheduled write for [%p] pid %d socket %d\n",
			 pid,
			 querynum,
			 name,
			 prx[t],
			 prx[t]->pid,
			 prx[t]->socket));
	
	free_resolv_process(& prx[t]);	
    }
#endif

    if (prx != prx_static)
	free(prx);
}

static struct resolv_query_item *query_one_name P_((struct resolv_cache             * record,
						    const char                      * name,
						    const char                      * domain,
						    const struct schedule_timelimit * now,
						    ns_type                           q_type,
						    enum looking_up                 * looking_up,
						    int                             * err_p,
						    int                               is_search
						    ));
static struct resolv_query_item *query_one_name(record,name,domain,now,q_type,
						looking_up,err_p, is_search)
     struct resolv_cache             * record;
     const char                      * name;
     const char                      * domain;
     const struct schedule_timelimit * now;
     ns_type                           q_type;
     enum looking_up                 * looking_up;
     int                             * err_p;
     int                               is_search;
{
    struct resolv_query_item * ret = NULL;
    int pid = getpid();
    
    static size_t querynum  = 0;
    size_t previous_querynum = querynum;

    char * name_buffer = NULL;
    size_t   qsize = RESOLV_BUFSIZE;
    int      retry = 0;
    
    *err_p = 0;
    
    do {

	querynum++;
	if (querynum >=  query_list_len) {
	    size_t X UNUSED_VAROK = querynum;

	    querynum = 0;
	    DPRINT(Debug,14,(&Debug, "query_one_name: [pid %d] name %s domain %s -- reached query number %zu -- wrapping to %zu\n",
			     pid,name,
			     domain ? domain : "<none>",
			     X,querynum));
	}

	/* 0 is not used */
	if (querynum > 0 && querynum <  query_list_len &&
	    ! query_list[querynum]) {
	    
	    DPRINT(Debug,14,(&Debug, "query_one_name: [pid %d] name %s domain %s --  query number %zu is available\n",
			     pid,name,
			     domain ? domain : "<none>",querynum));
	    break;
	}
	    

    } while (querynum != previous_querynum);
    
    if (querynum < 1 ||  querynum >=  query_list_len ||
	query_list[querynum]) {
	size_t nsize = query_list_len +3;
	/* Increment query list */

	query_list =  safe_array_realloc(query_list,nsize,sizeof (query_list[0]));
	
	DPRINT(Debug,14,(&Debug,
			 "query_one_name: [pid %d] name %s domain %s -- enlarging query_list %zu => %zu\n",
			 pid,name,
			 domain ? domain : "<none>",
			 query_list_len, nsize));

	querynum = query_list_len;
	if (!querynum)
	    querynum++;

	while (query_list_len < nsize)
	    query_list[query_list_len++] = NULL;

    }
    
    if (querynum < 1 ||  querynum >=  query_list_len ||
	query_list[querynum]) 
	panic("RESOLV PANIC",__FILE__,__LINE__,"query_one_name",
	       "Bad magic number (resolv_process) querynum",0);


    ret = query_list[querynum] = new_resolv_query_item(querynum,name);
    inc_resolv_query_item_refcount(ret);
    ret->query_time      = *now;
    ret->query_buffer    = new_resolv_message();
    ret->query_hdr       = new_resolv_query_hdr();
    if (is_search) {
	DPRINT(Debug,14,(&Debug,
			 "query_one_name: [%p] [pid %d] name %s domain %s -- setting 'is search'\n",
			 ret,pid,name,
			 domain ? domain : "<none>"));

	ret->is_search = 1;
    }

    ret->answer_buffer   = new_resolv_message();
    resize_resolv_message(ret->answer_buffer,RESOLV_BUFSIZE);  
    ret->answer_hdr      =  new_resolv_query_hdr();
    ret->cache_answer_in = record;
    inc_resolv_cache_refcount(ret->cache_answer_in);
    if (looking_up)
	ret->looking_up      = *looking_up;
          
    if (domain && domain[0]) {
	size_t a = strlen(name);
	size_t d = strlen(domain);
	size_t len = a+d+2;
	size_t z = a;

	name_buffer = safe_malloc(len);

	  memcpy(name_buffer,name,a);
	  if ('.' != domain[0])
	      name_buffer[z++] = '.';
	  memcpy(name_buffer+z,domain,d);
	  name_buffer[z+d] = '\0';

	  DPRINT(Debug,14,(&Debug, "query_one_name: [%p] [pid %d] query %zu: name %s domain %s => %s\n",
			   ret,
			   pid,
			   querynum,
			   name,
			   domain ? domain : "<none>",
			   name_buffer));
	  
    }

    do {
	int qlen;
	
	resize_resolv_message(ret->query_buffer,qsize);

	DPRINT(Debug,14,(&Debug, "query_one_name: [%p] [pid %d] query %zu %s: query buffer size %u\n",
			 ret,pid,
			 querynum,
			 name_buffer ? name_buffer : name,
			 (unsigned) ret->query_buffer->message_size));
	
	errno = 0;

	qlen =  res_mkquery(QUERY,name_buffer ? name_buffer : name,
			    ns_c_in,q_type,
			    NULL,0   /* Only used with IQUERY */,
			    NULL /* Only used with dynamic update */,
			    ret->query_buffer->message,
			    ret->query_buffer->message_size);

	if (qlen < 0 ) {
	    int err = errno;
	    
	    DPRINT(Debug,14,(&Debug, "query_one_name: [%p] [pid %d] query %zu %s: res_mkquery: err = %d (%s)\n",
			     ret,pid,
			     querynum,
			     name_buffer ? name_buffer : name,
			     err,strerror(err)));

	     if (EMSGSIZE == err &&  qsize < MAX_resolv_bufsize) {

		 if (qsize < sizeof (struct resolv_sizeupdate))
		     qsize = sizeof (struct resolv_sizeupdate);
		 else
		     qsize <<=1;

		 if (qsize > MAX_resolv_bufsize)
		     qsize = MAX_resolv_bufsize;

		 retry = 1;
		 
	     } else {
		 *err_p = err;

		 free_resolv_query_item(&ret);
		 goto failure;
		 
	     }	   		   
	} else if (qlen <= USHRT_MAX) {
	    ret->query_buffer->message_len = qlen;

	    
	    DPRINT(Debug,14,(&Debug, "query_one_name: [%p] [pid %d] query %zu %s: res_mkquery: message len %u\n",
			     ret,pid,
			     querynum,
			     name_buffer ? name_buffer : name,
			     (unsigned) ret->query_buffer->message_len));

	    retry  = 0;
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "query_one_name: [%p] [pid %d] query %lu %s: query len %d, too long\n",
			     ret,pid,
			     (unsigned long)querynum,
			     name_buffer ? name_buffer : name,
			     qlen));
	    *err_p = EMSGSIZE;

	    free_resolv_query_item(&ret);
	    goto failure;
	}

	if (retry) {
	    DPRINT(Debug,14,(&Debug, "query_one_name: [%p] [pid %d] query %zu %s: res_mkquery: retrying with size %zu\n",
			     ret,pid,
			     querynum,
			     name_buffer ? name_buffer : name,
			     qsize));
	}

    } while (retry);

    ret->query_hdr->buflen     = ret->query_buffer->message_len;
    ret->query_hdr->querynum   = querynum;
    ret->query_hdr->senderpid  = pid;
    ret->query_hdr->msg        = qm_query;
    
    if (!parse_resolv_message1(ret->query_buffer,err_p)) {
	DPRINT(Debug,14,(&Debug,
			 "query_one_name: [%d] [pid %d] query %zu %s: failed to parse query\n",
			 ret,pid,
			 querynum,
			 name_buffer ? name_buffer : name));
	free_resolv_query_item(&ret);
	goto failure;
    }
    
    select_resolv_process(pid,querynum,name_buffer ? name_buffer : name);
    
 failure:
    if (name_buffer)
	free(name_buffer);

    if (ret) {
	DPRINT(Debug,14,(&Debug, "query_one_name=%p: [pid %d]; querynum=%zu",
			 ret,pid,ret->querynum));
    } else {    
	DPRINT(Debug,14,(&Debug, "query_one_name=NULL: [pid %d]",pid));
    }

    if (*err_p) {
	DPRINT(Debug,14,(&Debug, "; *err_p=%d",*err_p));
    }
    
    DPRINT(Debug,14,(&Debug, "\n"));
	
    return ret;
}


/* Returns 1 is answer is already ready */
static int resolv_answer_ready P_((struct resolv_query_item * qitem,
				   int                      * resize,
				   int                        pid));
static int resolv_answer_ready(qitem,resize,pid)
     struct resolv_query_item * qitem;
     int                      * resize;
     int                        pid;
{
    int done = 0;

    if (RESOLV_QUERY_ITEM_magic != qitem->magic)
    	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "resolv_answer_ready",
	      "Bad magic number (resolv_query_item)",0);

    DPRINT(Debug,16,(&Debug,
		     "resolv_answer_ready: [pid %d] start --------------------------\n",
		     pid));
    DPRINT(Debug,16,(&Debug,
		     "                   : querynum                 = %lu\n",
		     (unsigned long)qitem->querynum));
    DPRINT(Debug,16,(&Debug,
		     "                   : name                     = %s\n",
		     qitem->name ? qitem->name : "<none>"));
    DPRINT(Debug,16,(&Debug,
		     "                   : last_errno               = %d\n",
		     qitem->last_errno));
    DPRINT(Debug,16,(&Debug,
		     "                   : looking_up               = %d",
		     qitem->looking_up));
    switch (qitem->looking_up) {
    case l_up_none:             DPRINT(Debug,16,(&Debug," (none)"));            break;
    case l_up_printed:          DPRINT(Debug,16,(&Debug," (printed)"));         break;
    case l_up_error:            DPRINT(Debug,16,(&Debug," (error)"));           break;
    case l_up_truncated_retry:  DPRINT(Debug,16,(&Debug," (truncated retry)")); break;
    case l_up_stalled:          DPRINT(Debug,16,(&Debug," (stalled)"));         break;
    }
    DPRINT(Debug,16,(&Debug,"\n"));
    DPRINT(Debug,16,(&Debug,
		     "                   : fill_have_error          = %d\n",
		     qitem->fill_have_error));
    DPRINT(Debug,16,(&Debug,
		     "                   : transient_error          = %d\n",
		     qitem->transient_error));
    DPRINT(Debug,16,(&Debug,
		     "                   : other                    = %d\n",
		     qitem->other));
    DPRINT(Debug,16,(&Debug,
		     "                   : is_search                = %d\n",
		     qitem->is_search));
    DPRINT(Debug,16,(&Debug,
		     "                   : query_sent               = %d\n",
		     qitem->query_sent));
    DPRINT(Debug,16,(&Debug,
		     "                   : valid_answer             = %d\n",
		     qitem->valid_answer));
    DPRINT(Debug,16,(&Debug,
		     "                   : invalid_answer           = %d\n",
		     qitem->invalid_answer));
    DPRINT(Debug,16,(&Debug,
		     "                   : shutdown                 = %d\n",
		     qitem->shutdown));
    DPRINT(Debug,16,(&Debug,
		     "resolv_answer_ready: [pid %d] --------------------------------\n",
		     pid));
        
    if (qitem->last_errno) {
	DPRINT(Debug,14,(&Debug,
			 "resolv_answer_ready: [pid %d] query %lu %s: done, error %d (%s)\n",
			 pid,(unsigned long)qitem->querynum,
			 qitem->name ? qitem->name : "<none>",
			 qitem->last_errno,strerror(qitem->last_errno)));
	
	if (resize && EMSGSIZE == qitem->last_errno && qitem->answer_buffer) {
	    struct resolv_message   * answer_buffer = qitem->answer_buffer;
	    
	    if (RESOLV_MESSAGE_magic != answer_buffer->magic)
		panic("RESOLV PANIC",__FILE__,__LINE__,
		      "resolv_answer_ready",
		      "Bad magic number (resolv_message)",0);

	    if (answer_buffer->message_size < MAX_resolv_bufsize) {
		DPRINT(Debug,14,(&Debug,
				 "resolv_answer_ready:  [pid %d] query %lu %s: message size %lu, can resized\n",
				 pid,(unsigned long)qitem->querynum,
				 qitem->name ? qitem->name : "<none>",
				 (unsigned long)answer_buffer->message_size));
		*resize = 1;
	    } else
		goto no_resize;
	    
	} else {
	no_resize:
	    if (qitem->looking_up < l_up_error && qitem->name) {
		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvLookError,
				  "Looking up %s: %s"),
			  qitem->name,strerror(qitem->last_errno));
		qitem->looking_up = l_up_error;
	    }
	}
	done = 1;
	
    } else if (qitem->fill_have_error ||
	       qitem->transient_error ||
	       qitem->valid_answer    ||
	       qitem->failed_answer   ||
	       qitem->invalid_answer  ||
	       qitem->shutdown) {
	DPRINT(Debug,14,(&Debug,
			 "resolv_answer_ready: [pid %d] query %lu %s: done%s%s%s%s%s\n",
			 pid,(unsigned long)qitem->querynum,
			 qitem->name ? qitem->name : "<none>",
			 qitem->fill_have_error ? ", fill have error" : "",
			 qitem->transient_error ? ", (fill have) transient error" : "",
			 qitem->valid_answer    ? ", valid answer"    : "",
			 qitem->invalid_answer  ? ", invalid answer"  : "",
			 qitem->shutdown        ? ", shutdown"        : ""));
	
	done = 1;
    } else if (qitem->answer_buffer) {
	struct resolv_message   * answer_buffer = qitem->answer_buffer;
	
	if (RESOLV_MESSAGE_magic != answer_buffer->magic)
	    panic("RESOLV PANIC",__FILE__,__LINE__,
		  "resolv_answer_ready",
		  "Bad magic number (resolv_message)",0);
	
	if (answer_buffer->message_parsed) {
	    DPRINT(Debug,14,(&Debug,
			     "resolv_answer_ready: [pid %d] query %lu %s: done, message parsed\n",
			     pid,(unsigned long)qitem->querynum,
			     qitem->name ? qitem->name : "<none>"));
	    
	    done=1;
	}
    }

     DPRINT(Debug,14,(&Debug,
		      "resolv_answer_ready=%d (%s):  [pid %d] query %lu %s",
		      done,
		      done ? "done" : "not done",
		       pid,(unsigned long)qitem->querynum,
		      qitem->name ? qitem->name : "<none>"));

     if (resize) {
	 DPRINT(Debug,14,(&Debug,"; *resize=%d",
			  *resize));
     }
     
     DPRINT(Debug,14,(&Debug,"\n"));
     
    return done;

}

#define STALLED_SECONDS_DEFAULT (5 * 60)
static const int MAX_stalled_seconds = (STALLED_SECONDS_DEFAULT > (60 *60)) ? STALLED_SECONDS_DEFAULT : (60 *60);
static const int MIN_stalled_seconds = (STALLED_SECONDS_DEFAULT < 30) ? STALLED_SECONDS_DEFAULT : 30;

static char resolv_stalled_seconds_value[SHORT];   /* cache exact string */
static int  resolv_stalled_seconds 
#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET) && defined(BACKGROUD_PROCESSES)
  = STALLED_SECONDS_DEFAULT /* deadline 5 minutes from reference time of query */
#endif
    ;     


/* return 1 if got answer */
static int resolv_wait_answer P_((struct resolv_query_item * qitem,
				  int                      * resize,
				  struct cancel_data      ** cancel_p));
static int resolv_wait_answer(qitem, resize, cancel_p)
     struct resolv_query_item * qitem;
     int                      * resize;
     struct cancel_data      ** cancel_p
     /*  May be NULL, Used if dns lookup was cancelable
	 -- not really canceled, just
	 stop waiting of result  
     */;
{

    char *X = NULL;
    int pid = getpid();
    struct schedule_timelimit wait_deadline = NO_schedule_timelimit;
    struct schedule_timelimit current       = NO_schedule_timelimit;
    int done = 0;
    
    if (RESOLV_QUERY_ITEM_magic != qitem->magic)
    	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "resolv_wait_answer",
	      "Bad magic number (resolv_query_item)",0);

    if (resize)
	*resize = 0;
    
    X = schedule_timeout_string(& (qitem->query_time));
    if (X) {
	DPRINT(Debug,14,(&Debug,
			 "resolv_wait_answer: [pid %d] query %lu %s: query time %s\n",
			 pid,(unsigned long)qitem->querynum,
			 qitem->name ? qitem->name : "<NONE>",X));
	free(X); X = NULL;
    }

    if (schedule_set_next_timeout0(&wait_deadline,& (qitem->query_time),resolv_stalled_seconds,
				   se_use_time_MAX)) {

	X = schedule_timeout_string(&wait_deadline);

	if (X) {
	    DPRINT(Debug,14,(&Debug,
			     "resolv_wait_answer: [pid %d] query %lu %s: wait deadline %s\n",
			     pid,(unsigned long)qitem->querynum,
			     qitem->name ? qitem->name : "<NONE>",X));	    
	    free(X); X = NULL;
	}
    }

    done = resolv_answer_ready(qitem,resize,pid);
    
#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET)
    while (!done) {
	enum wait_for_status r;

	errno = 0;

	if (cancel_p) {
	  	    
	    if (l_up_none == qitem->looking_up && qitem->name) {
		if (!*cancel_p || is_schedule_cancel(*cancel_p,NULL)) {
		    r =  wait_for_action_or_timeout_settime_c(resolv_read_message,1,&current,
							      *cancel_p);
		} else {
		    DPRINT(Debug,14,(&Debug,
				     "resolv_wait_answer: 'Looking up' already printed.\n"));
		    qitem->looking_up = l_up_printed;
		    
		    r =  wait_for_action_or_deadline_settime_c(resolv_read_message,&wait_deadline,
							       &current,*cancel_p);
		}
	    } else {
		r =  wait_for_action_or_deadline_settime_c(resolv_read_message,&wait_deadline,
							   &current,*cancel_p);
	    }
	    
	} else {
	    if (l_up_none == qitem->looking_up && qitem->name) 
		r =  wait_for_action_or_timeout_settime(resolv_read_message,1,&current);
	    else
		r =  wait_for_action_or_deadline_settime(resolv_read_message,&wait_deadline,
							 &current);
	}
	
	if (r > wait_for_none) {

	    done = resolv_answer_ready(qitem,resize,pid);
	    
	    if (!done) {
		if (l_up_none == qitem->looking_up && qitem->name)  {

		    if (qitem->looking_up < l_up_error && qitem->name) {
			if (cancel_p && *cancel_p) {
			    if (is_schedule_cancel(*cancel_p,NULL))
				set_cancel_message(*cancel_p,
						   -1 /* Show immediately */,
						   CATGETS(elm_msg_cat, 
							   ResolvSet,
							   ResolvLookingUp,
							   "Looking up %s ..."),
						   qitem->name);
			    else {
				DPRINT(Debug,14,(&Debug,
						 "resolv_wait_answer: 'Looking up' already printed.\n"));
			    }
				
			} else
			    lib_transient(CATGETS(elm_msg_cat, 
						  ResolvSet,
						  ResolvLookingUp,
						  "Looking up %s ..."),
					  qitem->name);
		    }
		    
		    qitem->looking_up = l_up_printed;

		    /* Messages may wait events -- so recheck status .... */		    
		    done = resolv_answer_ready(qitem,resize,pid);
		    
		} else if (schedule_timeout_reached(&wait_deadline,&current)) {
		    if (qitem->looking_up < l_up_error && qitem->name) {
			lib_transient(CATGETS(elm_msg_cat, 
					      ResolvSet,
					      ResolvLookingUpStalled,
					      "Looking up %s: Stalled (giving up)"),
				      qitem->name);
			
			qitem->looking_up = l_up_stalled;					    

			/* Messages may wait events -- so recheck status .... */		    
			done = resolv_answer_ready(qitem,resize,pid);
		    }

		    DPRINT(Debug,14,(&Debug,
				     "resolv_wait_answer: [pid %d] query %lu %s: timeout\n",
				     pid,(unsigned long)qitem->querynum,
				     qitem->name ? qitem->name : "<none>"));
		    goto timeout;
		}
	    }
	} else {
	    switch (r) {
		int err;
	    case wait_for_done:     break;
	    case wait_for_none:
		DPRINT(Debug,14,(&Debug,
				 "resolv_wait_answer: [pid %d] query %lu %s: unexpected result from  wait_for_action_or_...\n",
				 pid,(unsigned long)qitem->querynum,
				 qitem->name ? qitem->name : "<none>"));
		break;
	    case wait_for_error:
		err = errno;
		
		DPRINT(Debug,14,(&Debug,
				 "resolv_wait_answer: [pid %d] query %lu %s: errno=%d, %s\n",
				 pid,(unsigned long)qitem->querynum,
				 qitem->name ? qitem->name : "<none>",
				 err,strerror(err)));
		if (err == EINTR && cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		    DPRINT(Debug,12,(&Debug,
				     "bgconnect_wait_delay: wait canceled\n"));
		    
		    goto out;
		}
		break;
	    }
	}
    }
#endif

#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET)    
 out:
 timeout:
#endif
    X = schedule_timeout_string(&current);

    DPRINT(Debug,14,(&Debug,
		     "resolv_wait_answer=%d%s [pid %d] query %lu %s",
		     done,
		     done ? " (done) " : "",
		     pid,(unsigned long)qitem->querynum,
		     qitem->name ? qitem->name : "<none>"));
    if (X) {
	DPRINT(Debug,14,(&Debug," at %s",
			 X));
	free(X);
    }

    DPRINT(Debug,14,(&Debug,"\n"));
    return done;
}

/* input / output flags */
static const struct check_query_answer_f {
    unsigned int       have_error :1;
    unsigned int       other      :1;
    unsigned int       no_name    :1;
    unsigned int       end_search :1;    
} NULL_check_query_answer_f = {
    0,0,0,0
};

static struct resolv_cache * check_query_answer P_((struct resolv_query_item    * query,
						    const char                  * name,
						    enum looking_up             * looking_up_p,
						    char                       ** search_name_p,
						    struct check_query_answer_f * flags_p));
static struct resolv_cache * check_query_answer(query,name,looking_up_p,
						search_name_p,flags_p)
     struct resolv_query_item    * query;
     const char                  * name;
     enum looking_up             * looking_up_p;
     char                       ** search_name_p;
     struct check_query_answer_f * flags_p;
{
    struct resolv_cache       * ret   = NULL;

    if (RESOLV_QUERY_ITEM_magic != query->magic)
	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "check_query_answer",
	      "Bad magic number (resolv_query_item)",0);


    DPRINT(Debug,14,(&Debug,
		     "check_query_answer: %s: query %lu %s: IN flags have_error:%d other:%d no_name:%d end_search:%d\n",
		     name,
		     (unsigned long)query->querynum,
		     query->name ? query->name : "<none>",
		     flags_p->have_error,
		     flags_p->other,
		     flags_p->no_name,
		     flags_p->end_search));
    
    if (query->answer_buffer) {
	struct resolv_message   * answer_buffer = query->answer_buffer;
	
	if (RESOLV_MESSAGE_magic != answer_buffer->magic)
	    panic("RESOLV PANIC",__FILE__,__LINE__,
		  "check_query_answer",
		  "Bad magic number (resolv_message)",0);
	
	if (answer_buffer->message_parsed) {
	    ns_rcode response_code =
		ns_msg_getflag(answer_buffer->parsed_message,
			       ns_f_rcode);
	    unsigned short aa UNUSED_VAROK =
		ns_msg_getflag(answer_buffer->parsed_message,
			       ns_f_aa);
	    
	    DPRINT(Debug,14,(&Debug,
			     "check_query_answer: %s: query %lu %s: answer buffer: resonse code=%d, authorative=%d (%s)\n",
			     name,
			     (unsigned long)query->querynum,
			     query->name ? query->name : "<none>",
			     response_code,aa,
			     aa ? "yes" : "no"));
			    
	    if (ns_r_nxdomain == response_code) {
				
		DPRINT(Debug,14,(&Debug,
				 "check_query_answer: %s: query %lu %s: no name\n",
				 name,(unsigned long)query->querynum,
				 query->name ? query->name : "<none>"));
		flags_p->no_name = 1;
	    }			   			       			    			    
	}
    }
    
    if (query->valid_answer) {
	DPRINT(Debug,14,(&Debug,
			 "check_query_answer: %s: query %lu %s: valid answer set\n",
			 name,(unsigned long)query->querynum,
			 query->name ? query->name : "<none>"));
	
	if (query->other) {   /* Name found, no more search */
	    flags_p->end_search = 1;
	    flags_p->other      = 1;
	}
			    
	ret = query-> cache_answer_ret;
	if (ret) {
	    inc_resolv_cache_refcount(ret);

	    DPRINT(Debug,14,(&Debug,
			     "check_query_answer: %s: query %lu %s: got resolv cache record\n",
			     name,(unsigned long)query->querynum,
			     query->name ? query->name : "<none>"));
	}	

	if (search_name_p && query->answer_search_name) {

	    if (*search_name_p) {
		DPRINT(Debug,14,(&Debug,
				 "check_query_answer: %s: query %lu %s: changing *search_name_p: %s => %s\n",
				 name,(unsigned long)query->querynum,
				 query->name ? query->name : "<none>",
				 *search_name_p,
				 query->answer_search_name));
		
		*search_name_p = strmcpy(*search_name_p,query->answer_search_name);
	    } else {
		DPRINT(Debug,14,(&Debug,
				 "check_query_answer: %s: query %lu %s: setting *search_name_p: %s\n",
				 name,(unsigned long)query->querynum,
				 query->name ? query->name : "<none>",
				 query->answer_search_name));
		
		*search_name_p = safe_strdup(query->answer_search_name);
	    }
	}	    
    }
        
    if (query->failed_answer) {
	
	struct resolv_query_hdr * answer_hdr = query->answer_hdr;

	DPRINT(Debug,14,(&Debug,
			 "check_query_answer: %s: query %lu %s: failed answer set\n",
			 name,(unsigned long)query->querynum,
			 query->name ? query->name : "<none>"));
	
	if (answer_hdr) {
	    if ( RESOLV_QUERY_HDR_magic != answer_hdr->magic)
		panic("RESOLV PANIC",__FILE__,__LINE__,
		      "check_query_answer",
		      "Bad magic number (resolv_query_hdr)",0);
	    
	    DPRINT(Debug,14,(&Debug,
			     "check_query_answer: %s: query %lu %s: answer hdr: h_errno=%d, errno=%d (%s)\n",
			     name,(unsigned long)query->querynum,
			     query->name ? query->name : "<none>",
			     answer_hdr->h_errno_val,answer_hdr->errno_val,
			     strerror(answer_hdr->errno_val)));
	    
	    switch(answer_hdr->h_errno_val) {
	    case HOST_NOT_FOUND:
		if (query->looking_up < l_up_error)  {
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookNotFound,
				      "Looking up %s: not found"),
			      name);
		    
		    query->looking_up = l_up_error;
		    if (looking_up_p)
			*looking_up_p = l_up_error;
		}
		
		flags_p->end_search = 1;
		break;
			
	    case NO_DATA:
		flags_p->other = 1;
		break;
		
	    case TRY_AGAIN:
		
		if (query->looking_up < l_up_error)  {
		    if (answer_hdr->errno_val)
			lib_error(CATGETS(elm_msg_cat, 
					  ResolvSet,
					  ResolvLookErrno,
					  "Looking up %s: %s"),
				  name,strerror(answer_hdr->errno_val));
		    else				      
			lib_transient(CATGETS(elm_msg_cat, 
					      ResolvSet,
					      ResolvLookTryAgain,
					      "Looking up %s: not found yet"),
				      name);
		    
		    query->looking_up = l_up_error;
		}

		flags_p->have_error= 1;
		flags_p->end_search = 1;
		break;
				
	    case NO_RECOVERY:
				
		if (query->looking_up < l_up_error)  {
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvLookBadName,
				      "Looking up %s: bad name or refused query"),
			      name);
		    
		    query->looking_up = l_up_error;
		}

		flags_p->have_error = 1;
		flags_p->end_search = 1;
		break;
				
	    case NETDB_INTERNAL:
	    default:

		if (answer_hdr->errno_val) {
		    
		    if (query->looking_up < l_up_error)  {
			lib_error(CATGETS(elm_msg_cat, 
					  ResolvSet,
					  ResolvLookErrno,
					  "Looking up %s: %s"),
				  name,strerror(answer_hdr->errno_val));
			query->looking_up = l_up_error;
		    }
		    
		    flags_p->have_error = 1;
		    flags_p->end_search = 1;
		
		} else if (answer_hdr->h_errno_val) {

		    if (query->looking_up < l_up_error)  {
			lib_error(CATGETS(elm_msg_cat, 
					  ResolvSet,
					  ResolvLookFailed,
					  "Looking up %s: failed"),
				  name);
			
			query->looking_up = l_up_error;
		    }
		    
		    flags_p->have_error = 1;
		    flags_p->end_search = 1;
		    
		} else {
		    DPRINT(Debug,14,(&Debug,
				     "check_query_answer: %s: query %lu %s: no error data on failed answer\n",
				     name,(unsigned long)query->querynum,
				     query->name ? query->name : "<none>"));
		}
		break;
	    }	
	}
    }

    if (query->transient_error) {
	DPRINT(Debug,14,(&Debug,
			 "check_query_answer: %s: query %lu %s: fill have transient error set\n",
			 name,(unsigned long)query->querynum,
			 query->name ? query->name : "<none>"));
    }
    
    if (query->fill_have_error) {

	DPRINT(Debug,14,(&Debug,
			 "check_query_answer: %s: query %lu %s: fill have error set\n",
			 name,(unsigned long)query->querynum,
			 query->name ? query->name : "<none>"));

	flags_p->have_error = 1;
	flags_p->end_search = 1;
	
	if (query->looking_up < l_up_error)  {
	    lib_error(CATGETS(elm_msg_cat, 
			      ResolvSet,
			      ResolvLookErrorProcess,
			      "Looking up %s: Failed to process answer"),
		      name);
			    
	    query->looking_up = l_up_error;			    
	}
			    

    }

    if (looking_up_p)
	*looking_up_p = query->looking_up;

    DPRINT(Debug,14,(&Debug,
		     "check_query_answer: %s: query %lu %s: OUT flags have_error:%d other:%d no_name:%d end_search:%d\n",
		     name,
		     (unsigned long)query->querynum,
		     query->name ? query->name : "<none>",
		     flags_p->have_error,
		     flags_p->other,
		     flags_p->no_name,
		     flags_p->end_search));
    
    return ret;
}

/* return 1 if still have answer */
static int resize_query_answer P_((struct resolv_query_item * query,
				   int                      * resize,
				   struct cancel_data      ** cancel_p));
static int resize_query_answer(query,resize,cancel_p)
     struct resolv_query_item * query;
     int                      * resize;
     struct cancel_data      ** cancel_p         
     /*  May be NULL, Used if dns lookup was cancelable
	 -- not really canceled, just
	 stop waiting of result  
     */;
{
    size_t asize = RESOLV_BUFSIZE;
    int need_recheck = 0;
    int answer_ok = 1;
    int pid = getpid();
     
    if (RESOLV_QUERY_ITEM_magic != query->magic)
    	panic("RESOLV PANIC",__FILE__,__LINE__,
	      "resize_query_answer",
	      "Bad magic number (resolv_query_item)",0);

    do {    
	int need_resize = 0;
	int anslen = 0;
	
	if (resize) {	
	    need_resize = *resize;
	    
	    if (need_resize) {
		DPRINT(Debug,14,(&Debug,
				 "resize_query_answer: [pid %d] query %lu %s: resizing request %s set\n",
				 pid,
				 (unsigned long)query->querynum,
				 query->name ? query->name : "<none>",
				 need_recheck ? "still" : "already"));
		*resize = 0;
	    }
	}

	need_recheck = 0;
	
	if (query->answer_buffer) {
	    struct resolv_message   * answer_buffer = query->answer_buffer;
	    
	    if (RESOLV_MESSAGE_magic != answer_buffer->magic)
		panic("RESOLV PANIC",__FILE__,__LINE__,
		      "check_query_answer",
		      "Bad magic number (resolv_message)",0);
	    
	    asize = answer_buffer->message_size;
	    
	    DPRINT(Debug,14,(&Debug,
			     "resize_query_answer: [pid %d] query %lu %s: answer size %lu\n",
			     pid,
			     (unsigned long)query->querynum,
			     query->name ? query->name : "<none>",
			     (unsigned long)asize));
	    
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "resize_query_answer: [pid %d] query %lu %s: no answer buffer?\n",
			     pid,
			     (unsigned long)query->querynum,
			     query->name ? query->name : "<none>"));
	    
	    query->answer_buffer =  new_resolv_message();
	    need_resize = 1;
	}
	
	if (query->answer_hdr) {
	    
	    struct resolv_query_hdr * answer_hdr = query->answer_hdr;
	    
	    if (RESOLV_QUERY_HDR_magic != answer_hdr->magic)
		panic("RESOLV PANIC",__FILE__,__LINE__,
		      "resize_query_answer",
		      "Bad magic number (resolv_query_hdr)",0);
	    
	    if (answer_hdr->errno_val) {
		DPRINT(Debug,14,(&Debug,
				 "resize_query_answer: [pid %d] query %lu %s: answer hdr errno %d (%s), answer buffer message size %lu\n",
				 pid,
				 (unsigned long)query->querynum,
				 query->name ? query->name : "<none>",
				 answer_hdr->errno_val,
				 strerror(answer_hdr->errno_val),
				 (unsigned long)query->answer_buffer->message_size));
		
		if (EMSGSIZE == answer_hdr->errno_val &&
		    query->answer_buffer->message_size < MAX_resolv_bufsize) {
		    DPRINT(Debug,14,(&Debug,
				     "resize_query_answer: [pid %d] query %lu %s: answer buffer message size %lu, can resized\n",
				     pid,
				     (unsigned long)query->querynum,
				     query->name ? query->name : "<none>",
				     (unsigned long)query->answer_buffer->message_size));
		    need_resize = 1;

		    anslen = query->answer_buffer->message_len;
		}
	    }	 
	}

	
	if (need_resize) {
	    
	    if (asize < MAX_resolv_bufsize) {
		size_t old UNUSED_VAROK = asize;
		
		if (asize < sizeof (struct resolv_sizeupdate))
		     asize = sizeof (struct resolv_sizeupdate);
		else
		    asize <<= 1;

		if (asize >  MAX_resolv_bufsize)
		    asize = MAX_resolv_bufsize;

		DPRINT(Debug,14,(&Debug,
				 "resize_query_answer: [pid %d] query %lu %s: resizing answer buffer message size %lu => %lu\n",
				 pid,
				 (unsigned long)query->querynum,
				 query->name ? query->name : "<none>",
				 (unsigned long)old,
				 (unsigned long)asize));

		if (anslen) {
		    lib_transient(CATGETS(elm_msg_cat, 
					  ResolvSet,
					  ResolvLookTruncatedRetry,
					  "Looking up %s: Answer truncated (%d bytes), retrying..."),
				  query->name ? query->name : "<none>",
				  anslen);
		    query->looking_up =  l_up_truncated_retry;
		}
		
		resize_resolv_message(query->answer_buffer,asize);

		/* Reset answer data */
		if (query->answer_hdr)
		    *(query->answer_hdr) =  NULL_resolv_query_hdr;
		query->last_errno = 0;
		if (query->cache_answer_ret)
		    free_resolv_cache(& (query->cache_answer_ret));

		query->fill_have_error   = 0;
		query->transient_error   = 0;
		query->other             = 0;
		query->query_sent        = 0;    /* cause resent */
		query->valid_answer      = 0;
		query->failed_answer     = 0;
		query-> invalid_answer   = 0;
		query->shutdown          = 0;

		select_resolv_process(pid,query->querynum,
				      query->name ? query->name : "<none>");

		
		answer_ok = resolv_wait_answer(query,resize,cancel_p);

		if (answer_ok) {
		    DPRINT(Debug,14,(&Debug,
				     "resize_query_answer: [pid %d] query %lu %s: got answer after resent\n",
				     pid,
				     (unsigned long)query->querynum,
				     query->name ? query->name : "<none>"));
		    need_recheck = 1;
		} else
		    need_recheck = 0;
		
	    } else 
		need_recheck = 0;
	    		
	} else
	    need_recheck = 0;

	if (need_recheck) {
	    DPRINT(Debug,14,(&Debug,
			     "resize_query_answer: [pid %d] query %lu %s: rechecking\n",
			     pid,
			     (unsigned long)query->querynum,
			     query->name ? query->name : "<none>"));
	}
		
    } while (need_recheck);
    
    DPRINT(Debug,14,(&Debug,
		     "resize_query_answer%d%s: [pid %d] query %lu %s\n",
		     answer_ok,
		     answer_ok ? " (answer ok)" : " (no answer)",
		     pid,
		     (unsigned long)query->querynum,
		     query->name ? query->name : "<none>"));

    
    return answer_ok;
}


static void end_query_helper P_((struct resolv_query_item   ** query_p,
				 const int                     err,
				 const char                  * name,
				 struct resolv_cache        ** ret_p,
				 enum looking_up             * looking_up_p,
				 char                       ** search_name_p,
				 struct check_query_answer_f * flags_p,
				 struct cancel_data         ** cancel_p  
				 /*  May be NULL, Used if dns lookup was cancelable
				     -- not really canceled, just
				     stop waiting of result  
				 */));
static void end_query_helper(query_p,err,name,ret_p,looking_up_p,search_name_p,
			     flags_p,cancel_p)
     struct resolv_query_item   ** query_p;
     const int                     err;
     const char                  * name;
     struct resolv_cache        ** ret_p;
     enum looking_up             * looking_up_p;
     char                       ** search_name_p;
     struct check_query_answer_f * flags_p;
     struct cancel_data         ** cancel_p
     /*  May be NULL, Used if dns lookup was cancelable
	 -- not really canceled, just
	 stop waiting of result  
     */;
{

    if (*query_p) {
	int resize = 0;
	
	if (RESOLV_QUERY_ITEM_magic != (*query_p)->magic)
	    panic("RESOLV PANIC",__FILE__,__LINE__,
		  "end_query_helper",
		  "Bad magic number (resolv_query_item)",0);

	DPRINT(Debug,16,(&Debug,
			 "end_query_helper: %s: got query\n",
			 name));
	
	if (*ret_p)
	    panic("RESOLV PANIC",__FILE__,__LINE__,
		  "end_query_helper",
		  "result already set",0);
	
	if (resolv_wait_answer(*query_p,&resize,cancel_p)) {

	    if (resize_query_answer(*query_p,&resize,cancel_p)) {		    
		*ret_p = check_query_answer(*query_p,name,looking_up_p,
					    search_name_p,flags_p);
	    } else
		goto end_search;
	} else {
	end_search:
	    if (cancel_p && *cancel_p &&
		is_canceled(*cancel_p)) {
		
		DPRINT(Debug,14,(&Debug,
				 "end_query_helper: %s: DNS lookup canceled\n",
				 name));
		
		flags_p->end_search = 1;
		flags_p->have_error = 1;   /* Not actually used when is_canceled()
					      is set, in that case error is set directly
					      on caller
					   */
		goto fail;
		
	    } else {
		DPRINT(Debug,14,(&Debug,
				 "end_query_helper: %s: no answer\n",
				 name));
		flags_p->end_search = 1; /* No answer or no answer after retry*/
	    }
	}
	    		
    } else if (err) {
	DPRINT(Debug,14,(&Debug,
			 "end_query_helper: %s:  error %d (%s)\n",
			 name,err,strerror(err)));
	
	lib_error(CATGETS(elm_msg_cat, 
			  ResolvSet,
			  ResolvLookError,
			  "Looking up %s: %s"),
		  name,strerror(err));
	
	if (looking_up_p)
	    *looking_up_p = l_up_error;
	
	flags_p->end_search = 1;
    } else {
	DPRINT(Debug,16,(&Debug,
			 "end_query_helper: %s: no query\n",
			 name));

	flags_p->no_name = 1;  /* searching only when NRXDOMAIN was previous result */
    }

 fail:
    
    if (*query_p)
	free_resolv_query_item(query_p);
    
    DPRINT(Debug,16,(&Debug,
		     "end_query_helper: %s: handled%s%s%s%s\n",
		     name,
		     flags_p->end_search ? ", end search" : "",
		     flags_p->no_name    ? ", no name"    : "",
		     flags_p->have_error ? ", have error" : "",
		     *ret_p              ? ", answer set" : ""));
}

S_(lookup_resolv_cache_qm_f lookup_resolv_cache_nonblocked)
struct resolv_cache * lookup_resolv_cache_nonblocked
    P_((struct resolv_cache             * record,
	const char                      * name,
	const struct schedule_timelimit * now,
	const ns_type                     q_type,
	const int                         is_search,
	enum looking_up                 * looking_up,
	char                           ** search_name,
	int                             * have_error,
	int                             * other,
	struct cancel_data             ** cancel_p));
static struct resolv_cache * lookup_resolv_cache_nonblocked(record,name,now,
						     q_type,is_search,
						     looking_up_p,search_name_p,
						     have_error_p,other_p,
						     cancel_p)
     struct resolv_cache             * record;
     const char                      * name;
     const struct schedule_timelimit * now;
     const ns_type                     q_type;
     const int                         is_search;
     enum looking_up                 * looking_up_p;
     char                           ** search_name_p;
     int                             * have_error_p;
     int                             * other_p;
     struct cancel_data             ** cancel_p
     /*  May be NULL, Used if dns lookup was cancelable
	 -- not really canceled, just
	 stop waiting of result  
     */;
{
    struct resolv_cache       * ret   = NULL;
    int err = 0;    
    struct resolv_query_item  * query = NULL;
    int root_searched = 0;    
    int dots = -1;
    
    struct check_query_answer_f flags = NULL_check_query_answer_f;

    if (cancel_p && !*cancel_p) {
	
	enum name_lookup_cancel_v cancel_mode = give_dt_enumerate_as_int(&name_lookup_cancel);
	
	DPRINT(Debug,14,(&Debug,
			 "lookup_resolv_cache_nonblocked: %s: name-lookup-cancel = %d\n",
			 name,cancel_mode));

	switch (cancel_mode) {
	case name_lookup_cancel_enabled:
	case name_lookup_cancel_auto:       /* "auto" applies to non-blocking queries */
	     *cancel_p =
		new_schedule_cancel(100  /* print after 100 ms */,
				    CATGETS(elm_msg_cat, 
					    ResolvSet,
					    ResolvLookingUp,
					    "Looking up %s ..."),
				    name);
	     break;
	case name_lookup_cancel_disabled:
	    case NUM_name_lookup_cancel:
	    break;
	}    
    }


    if (have_error_p)
	flags.have_error = 0 != *have_error_p;
    if (other_p)
	flags.other      = 0 != *other_p;
	
    if (is_search) {
	const char *c;
	const char * found_dot = NULL;

	dots = 0;

	for (c = name;*c; c++) {
	    if ('.' == *c) {
		dots++;
		found_dot = c;
	    }
	}
	
	DPRINT(Debug,14,(&Debug,"lookup_resolv_cache_nonblocked: %s: %d dots\n",
			 name,dots));

	/* Need to similar than res_search */
	
	if (found_dot && !found_dot[1]) {
	    DPRINT(Debug,14,(&Debug,"lookup_resolv_cache_nonblocked: %s: trailing dot\n",
			     name));

	    query = query_one_name(record,name,NULL,now,q_type,
				   looking_up_p,&err,
				   0  /* not search */);
	    root_searched = 1;

	     
	} else {
	    char * default_domain_searched = NULL;   

	    if (!dots && 0 != (_res.options & RES_DEFNAMES)) {

		DPRINT(Debug,14,(&Debug,
				 "lookup_resolv_cache_nonblocked: %s: No dots, (RES_DEFNAMES) searching from default domain: %s\n",
				 name,_res.defdname));

		default_domain_searched = _res.defdname;

		if ( ! _res.defdname[0] || 0 == strcmp(_res.defdname,".")) {
		    DPRINT(Debug,14,(&Debug,
				     "lookup_resolv_cache_nonblocked:  ... default domain is root\n"));
		    root_searched = 1;
		}

		query = query_one_name(record,name,_res.defdname,now,q_type,
				       looking_up_p,&err,
				       1  /* have search */);

	    } else if (dots >= _res.ndots) {

		DPRINT(Debug,14,(&Debug,
				 "lookup_resolv_cache_nonblocked: %s: %d dots, doing absolute query (ndots:%d)\n",
				 name,dots,_res.ndots));

		query = query_one_name(record,name,NULL,now,q_type,
				       looking_up_p,&err,
				       (_res.dnsrch[0] != NULL) /* maybe have search */);
		root_searched = 1;
	    }
	    	    
	    end_query_helper(&query,err,name,&ret,looking_up_p,
			     search_name_p,&flags,cancel_p);
			       
	    if (!ret && cancel_p && *cancel_p &&
		is_canceled(*cancel_p)) {

		DPRINT(Debug,14,(&Debug,
				 "lookup_resolv_cache_nonblocked: %s: DNS lookup canceled\n",
				 name));

		/* Flag this as error, so query is marked as failed */
		if (have_error_p && !*have_error_p) {
		    DPRINT(Debug,14,(&Debug, "lookup_resolv_cache_nonblocked: %s: marking as failed\n",
				     name));
		    *have_error_p = 1;
		}
		
		goto fail;
		
	    } else if (!flags.end_search && !ret &&
		!query &&
		flags.no_name /* searching only when NRXDOMAIN was previous result */
		&&		
		0 != (_res.options & RES_DNSRCH) &&
		(dots ||
		 0 != (_res.options & RES_DEFNAMES))
		) {
		
		int x;

		
		DPRINT(Debug,14,(&Debug,
				 "lookup_resolv_cache_nonblocked: %s: %d dots%s, (RES_DNSRCH) using search list\n",
				 name,dots,
				 0 != (_res.options & RES_DEFNAMES) ? " (RES_DEFNAMES)" : ""));
		
		for (x = 0 ; 
		     x < (sizeof _res.dnsrch) / sizeof (_res.dnsrch[0]) &&
			 _res.dnsrch[x] && 
			 !flags.end_search && !ret &&
			 !query &&
			 flags.no_name /* searching only when NRXDOMAIN was previous result */;
		     x++) {

		    if (default_domain_searched &&
			0 == strcmp(default_domain_searched,
				    _res.dnsrch[x])) {
			
			DPRINT(Debug,14,(&Debug,
					 "lookup_resolv_cache_nonblocked: %s: skipping %d: %s (default domain already searched)\n",
					 name, x, _res.dnsrch[x]));

		    } else {
			int is_root =
			    ( ! _res.dnsrch[x][0] || 0 == strcmp( _res.dnsrch[x],"."));

			if (root_searched && is_root) {
			    DPRINT(Debug,14,(&Debug,
					     "lookup_resolv_cache_nonblocked: %s: skipping %d: %s (root already searched)\n",
					     name,x, _res.dnsrch[x]));
				   
			} else {
			    DPRINT(Debug,14,(&Debug,
					     "lookup_resolv_cache_nonblocked: %s: searching %d: %s%s\n",
					     name,x, _res.dnsrch[x],
					     is_root ? " (is root)" : ""));

			    query = query_one_name(record,name,_res.dnsrch[x],
						   now,q_type,
						   looking_up_p,&err,
						   1 /* have search */);
			    
			    if (is_root)
				root_searched = 1;

			    end_query_helper(&query,err,name,&ret,looking_up_p,
					     search_name_p,&flags,cancel_p);

			    if (!ret && cancel_p && *cancel_p &&
				is_canceled(*cancel_p)) {

				DPRINT(Debug,14,(&Debug,
						 "lookup_resolv_cache_nonblocked: %s: DNS lookup canceled\n",
						 name));
				
				/* Flag this as error, so query is marked as failed */
				if (have_error_p && !*have_error_p) {
				    DPRINT(Debug,14,(&Debug, "lookup_resolv_cache_nonblocked: %s: marking as failed\n",
						     name));
				    *have_error_p = 1;
				}
		
				goto fail;
			    }
			}
		    }
		}		       
	    }
	}
	
    } else
	flags.no_name = 1;  /* searching only when NRXDOMAIN was previous result */
    
    /* dots == -1   not search
       dots >= 0    search 
    */

    if (!flags.end_search && !ret &&
	!query && !root_searched &&
	flags.no_name /* searching only when NRXDOMAIN was previous result */
#ifdef RES_NOTLDQUERY
	&& (dots != 0 || 0 == (_res.options &  RES_NOTLDQUERY) )
#endif		
	) {
	
	DPRINT(Debug,14,(&Debug,
			 "lookup_resolv_cache_nonblocked: %s: ",
			 name));
	if (dots < 0) {
	    DPRINT(Debug,14,(&Debug, "not search"));
	} else {
	    DPRINT(Debug,14,(&Debug, "search, dots %d",
			     dots));
	}
	DPRINT(Debug,14,(&Debug, ", doing absolute query\n"));
	
	query = query_one_name(record,name,NULL,now,q_type,
			       looking_up_p,&err,
			       0 /* is not search */);

	end_query_helper(&query,err,name,&ret,looking_up_p,
			 search_name_p,&flags,cancel_p);

	if (!ret && cancel_p && *cancel_p &&
	    is_canceled(*cancel_p)) {
	    
	    DPRINT(Debug,14,(&Debug,
				 "lookup_resolv_cache_nonblocked: %s: DNS lookup canceled\n",
			     name));
	    
	    /* Flag this as error, so query is marked as failed */
	    if (have_error_p && !*have_error_p) {
		DPRINT(Debug,14,(&Debug, "lookup_resolv_cache_nonblocked: %s: marking as failed\n",
				 name));
		*have_error_p = 1;
	    }
	    
	    goto fail;
	}
    }
	
    if (have_error_p)
	*have_error_p = flags.have_error;
    else if (flags.have_error && (!looking_up_p &&
				  *looking_up_p <  l_up_error)) {
	lib_error(CATGETS(elm_msg_cat, 
			  ResolvSet,
			  ResolvLookingUpSomeFailure,
			  "Looking up %s: some failure?"),
		  name);
	
	if (looking_up_p)
	    *looking_up_p = l_up_error;
    }
    
    if (other_p)
	*other_p      = flags.other;
    else if (flags.other && (!looking_up_p &&
			     *looking_up_p <  l_up_error)) {
	
	lib_transient(CATGETS(elm_msg_cat, 
			      ResolvSet,
			      ResolvLookNoData,
			      "Looking up %s: no data"),
		      name);
	
	if (looking_up_p)
	    *looking_up_p = l_up_error;

    }
    
 fail:
    
    if (query) {
	free_resolv_query_item(&query);
    }

    DPRINT(Debug,14,(&Debug,
		     "lookup_resolv_cache_nonblocked=%d (%s): %s",
		     ret,
		     ret ? "have result" : "no result",
		     name));
    
    if (search_name_p && *search_name_p) {
	DPRINT(Debug,14,(&Debug,"; search name = %s",
			 *search_name_p));
    }
    
    DPRINT(Debug,14,(&Debug,"\n"));
    return ret;
} 


struct resolv_query_mode nonblocked_query_mode = {
    RESOLV_QUERY_MODE_magic,	
    have_non_blocking_qm,
    lookup_resolv_cache_nonblocked
};


const char * QUERY_MODE[NUM_query_mode] = { "blocking",
					    "non-blocking" };

enum query_mode_val query_mode =
#if defined(RESOLV_SENDOK) && defined(RESOLV_SEQPACKET) && defined(BACKGROUD_PROCESSES)
    qm_non_blocking
#else
    qm_blocking
#endif
    ;

const struct resolv_query_mode * query_modes [NUM_query_mode]  = {
    &blocked_query_mode,
    &nonblocked_query_mode
};

static const char * HELPER_INTERRUPT[NUM_helper_interrupt] = {
    "allow-interrupt",
    "ignore-interrupt",
};
    
static char * query_mode_func_return_buffer = NULL;

void free_resolv_config()
{
    if (query_mode_func_return_buffer) {
	free(query_mode_func_return_buffer);
	query_mode_func_return_buffer = NULL;
    }
}

enum query_mode_param_kw {
    qmp_helper,
    qmp_helper_skip_limit,    
    qmp_parallel_lookup_max_time,
    NUM_query_mode_param
};

static const char * QUERY_MODE_PARAM[NUM_query_mode_param] = {
    "helper",
    "helper-skip-limit",
    "parallel-lookup-max-time"
};

int query_mode_func(value,enter,lineno,filename,read_flags)
     char **value;
     int   enter;
     int   lineno;
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ok = 0;

    if (enter) {
	char * WALK;
	char * temp = safe_strdup(*value);
	char * f = mime_parse_content_opts(temp, &WALK);

	int x;

	if (!f) {
	    goto fail;
	}

	if (conf_select_value(f, QUERY_MODE,NUM_query_mode,
			      &x)) {
		ok         = 1;
		query_mode = x;
	}

	if (ok) {	    
#ifndef RESOLV_SENDOK
	    if (query_mode) {
		ok = 0;
	    }
#endif
#ifndef RESOLV_SEQPACKET
	    if (query_mode) {
		ok = 0;
	    }
#endif
#ifndef BACKGROUD_PROCESSES
	    if (query_mode) {
		ok = 0;
	    }
#endif
	    if (!ok) {
		if (filename)
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvParallelLookupNotAvailFile,
				      "Parallel DNS lookup is not availeble (query-mode keyword %s on line %d on file %s)"),
			      f,lineno,filename);
		else
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvParallelLookupNotAvail,
				      "Parallel DNS lookup is not availeble (query-mode keyword %s)"),
			      f);
		    
		query_mode = 0;
	    }
	    

	} else {

	    if (filename)
		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvUnknowKeywordPLFile,
				  "Unknown query-mode keyword %s on line %d on file %s"),
			  f,lineno,filename);
	    else
		lib_error(CATGETS(elm_msg_cat, 
				  ResolvSet,
				  ResolvUnknowKeywordPL,
				  "Unknown query-mode keyword %s"),
			  f);	    
	}

	while (NULL != (f = mime_parse_content_opts(NULL, &WALK))) {
	    /* modifies string f, return param value or NULL */
	    char * q = conf_split_param_value(f);
	    int p;

	    if (conf_select_value(f,QUERY_MODE_PARAM,
				  NUM_query_mode_param,
				  &p)) {
		enum query_mode_param_kw param = p;

		switch (param) {
		case qmp_helper:
		    if (q) {
			errno = 0;
			if (conf_select_value(q,HELPER_INTERRUPT,
					      NUM_helper_interrupt,
					      &x)) {
			    helper_int_handling = x;
			} else {
			    char * end = NULL;
			    long val = strtol(q,&end,10);
			    int err;
			    
			    err = errno;
			    
			    if (0 != err   ||
				'\0' != *end ||
				val < 1      ||
				val > MAX_prx_processes) {
				
				ok = 0;
				
				if (err) {
				    if (filename)			    
					lib_error(CATGETS(elm_msg_cat, 
							  ResolvSet,
							  ResolvBadValueKwPLFileErr,
							  "Bad value %s on query-mode keyword %s on line %d on file %s: %s"),
						  q,f,lineno,filename,
						  strerror(err));
				    else
					lib_error(CATGETS(elm_msg_cat, 
							  ResolvSet,
							  ResolvBadValueKwPLErr,
							  "Bad value %s on query-mode keyword %s: %s"),
						  q,f,
						  strerror(err));
				    
				} else {
				    if (filename)			    
					lib_error(CATGETS(elm_msg_cat, 
							  ResolvSet,
							  ResolvBadValueKwPLFile,
							  "Bad value %s on query-mode keyword %s on line %d on file %s"),
						  q,f,lineno,filename);
				    else
					lib_error(CATGETS(elm_msg_cat, 
							  ResolvSet,
						      ResolvBadValueKwPL,
							  "Bad value %s on query-mode keyword %s"),
						  q,f);
				}
				
				if (end > q) {  /* If not keyword */
				    if (val < 1)
					prx_max_processes = 1;
				    else if (val > MAX_prx_processes)
					prx_max_processes = MAX_prx_processes;
				    else
					prx_max_processes = val;
				}
				
			    } else
				prx_max_processes = val;
			}
			
		    } else
			goto value_missing;
		    break;

		case qmp_helper_skip_limit:
		    errno = 0;
		    if (q) {
			char * end = NULL;
			long val = strtol(q,&end,10);
			int err;
			
			err = errno;

			if (0 != err   ||
			    '\0' != *end ||
			    val < 0      ||
			    val > MAX_prx_processes) {

			    ok = 0;

			    if (err) {
				if (filename)			    
				    lib_error(CATGETS(elm_msg_cat, 
						      ResolvSet,
						      ResolvBadValueKwPLFileErr,
						      "Bad value %s on query-mode keyword %s on line %d on file %s: %s"),
					      q,f,lineno,filename,
					      strerror(err));
				else
				    lib_error(CATGETS(elm_msg_cat, 
						      ResolvSet,
						      ResolvBadValueKwPLErr,
						      "Bad value %s on query-mode keyword %s: %s"),
					      q,f,
					      strerror(err));
				
			    } else {
				if (filename)			    
				    lib_error(CATGETS(elm_msg_cat, 
						      ResolvSet,
						      ResolvBadValueKwPLFile,
						      "Bad value %s on query-mode keyword %s on line %d on file %s"),
					      q,f,lineno,filename);
				else
				    lib_error(CATGETS(elm_msg_cat, 
						      ResolvSet,
						      ResolvBadValueKwPL,
						      "Bad value %s on query-mode keyword %s"),
					      q,f);
			    }

			    if (end > q) {  /* If not keyword */
				if (val < 0)
				    prx_max_skip_process = 0;
				else if (val > MAX_prx_processes)
				    prx_max_processes = MAX_prx_processes;
				else
				    prx_max_skip_process = val;
			    }

			} else
			    prx_max_skip_process = val;

		    } else 
			goto value_missing;
		    break;
		    
		case qmp_parallel_lookup_max_time:
		    errno = 0;
		    if (q) {
			char * end = NULL;
			long val = strtol(q,&end,10);
			int err = errno;
			
			resolv_stalled_seconds_value[0] = '\0';
			
			if (0 == err) {
			    if ('\0' == *end ||
				0 == strcmp(end,"s")) {   /* seconds */
				
				if (val >= MIN_stalled_seconds && val <=  MAX_stalled_seconds) {
				    resolv_stalled_seconds = val;
				    
				    if (strlen(q) < sizeof resolv_stalled_seconds_value)
					strfcpy(resolv_stalled_seconds_value,q,
						sizeof resolv_stalled_seconds_value);
				} else {
				    
				    if (val < MIN_stalled_seconds)
					resolv_stalled_seconds = MIN_stalled_seconds;
				    else
					resolv_stalled_seconds = MAX_stalled_seconds;
				    
				    goto bad_value;
				}
				
			    } else if ( 0 == strcmp(end,"m")) {   /* minutes */
				const int factor = 60;
				
				/* Check overflow first */
				if (val >= MIN_stalled_seconds / factor &&
				    val <= MAX_stalled_seconds / factor) {
				    
				    /* Check exact value when multiplation does not overflow */
				    
				    int res = val * factor;
				    
				    
				    if (res >= MIN_stalled_seconds && res <=  MAX_stalled_seconds) {
					resolv_stalled_seconds = res;
					
					if (strlen(q) < sizeof resolv_stalled_seconds_value)
					    strfcpy(resolv_stalled_seconds_value,q,
						    sizeof resolv_stalled_seconds_value);
				    } else {
					
					if (res < MIN_stalled_seconds)
					    resolv_stalled_seconds = MIN_stalled_seconds;
					else
					    resolv_stalled_seconds = MAX_stalled_seconds;
					
					goto bad_value;
				    }
				    
				} else {
				    if (val < MIN_stalled_seconds / factor)
					resolv_stalled_seconds = MIN_stalled_seconds;
				    else
					resolv_stalled_seconds = MAX_stalled_seconds;
				    
				    goto bad_value;
				}
				
			    } else {
			    bad_value:
				ok = 0;
				
				if (filename)			    
				    lib_error(CATGETS(elm_msg_cat, 
						      ResolvSet,
						      ResolvBadValueKwPLFile,
						      "Bad value %s on query-mode keyword %s on line %d on file %s"),
					      q,f,lineno,filename);
				else
				    lib_error(CATGETS(elm_msg_cat, 
						      ResolvSet,
						      ResolvBadValueKwPL,
						      "Bad value %s on query-mode keyword %s"),
					      q,f);
			    }
			    
			} else {
			    ok = 0;
			    
			    if (filename)			    
				lib_error(CATGETS(elm_msg_cat, 
						  ResolvSet,
						  ResolvBadValueKwPLFileErr,
						  "Bad value %s on query-mode keyword %s on line %d on file %s: %s"),
					  q,f,lineno,filename,
					  strerror(err));
			    else
				lib_error(CATGETS(elm_msg_cat, 
						  ResolvSet,
						  ResolvBadValueKwPLErr,
						  "Bad value %s on query-mode keyword %s: %s"),
					  q,f,
					  strerror(err));
			}
			
		    } else {
		    value_missing:
			ok = 0;
			
			if (filename)
			    lib_error(CATGETS(elm_msg_cat, 
					      ResolvSet,
					      ResolvMissingValueKwPLFile,
					      "Missing value on query-mode keyword %s on line %d on file %s"),
				      f,lineno,filename);
			else
			    lib_error(CATGETS(elm_msg_cat, 
					      ResolvSet,
					      ResolvMissingValueKwPL,
					      "Missing value on query-mode keyword %s"),
				      f);
			
		        
		    }
		    break;

		case NUM_query_mode_param:    /* Not used */    break;
		}
		
	    } else {
		
		if (filename)
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvUnknowKeywordPLFile,
				      "Unknown query-mode keyword %s on line %d on file %s"),
			      f,lineno,filename);
		else
		    lib_error(CATGETS(elm_msg_cat, 
				      ResolvSet,
				      ResolvUnknowKeywordPL,
				      "Unknown query-mode keyword %s"),
			      f);	    
		
		ok = 0;
	    }
	}
	
    fail:
	free(temp);
	

    } else {

	if (query_mode >= 0 && query_mode < NUM_query_mode) {
	    query_mode_func_return_buffer =
		strmcpy(query_mode_func_return_buffer,
			QUERY_MODE[query_mode]);
	    ok = 1;
	} else
	    goto fail1;

	if (query_mode ||
	    helper_int_handling != DEFAULT_helper_int_handling)  {
	    if (helper_int_handling >= 0 &&
		helper_int_handling < NUM_helper_interrupt) {
		
		query_mode_func_return_buffer =
		    strmcat(query_mode_func_return_buffer,"; ");
		query_mode_func_return_buffer =
		    strmcat(query_mode_func_return_buffer,
			    QUERY_MODE_PARAM[qmp_helper]);
		query_mode_func_return_buffer =
		    strmcat(query_mode_func_return_buffer,"=");
		query_mode_func_return_buffer =
		    strmcat(query_mode_func_return_buffer,
			    HELPER_INTERRUPT[helper_int_handling]);
		
	    } else
		ok = 0;
	}
       	
	if (query_mode || 
	    prx_max_processes != MAX_PROCESS) {
	    if (prx_max_processes > 0) {
		
		char * t =  elm_message(FRM("; %s=%d"),
					QUERY_MODE_PARAM[qmp_helper],
					prx_max_processes);
		
		query_mode_func_return_buffer =
		    strmcat(query_mode_func_return_buffer,t);
		free(t);
	    } else
		ok = 0;
	}

	if (query_mode || 
	    prx_max_skip_process != MAX_SKIP_PROCESS
	    ) {
	    
	    if (prx_max_skip_process >= 0) {
		char * t =  elm_message(FRM("; %s=%d"),
					QUERY_MODE_PARAM[qmp_helper_skip_limit],
					prx_max_skip_process);
		
		query_mode_func_return_buffer =
		    strmcat(query_mode_func_return_buffer,t);
		free(t);
	    } else
		ok = 0;
	}
	
	if (resolv_stalled_seconds_value[0]) {
	    
	    query_mode_func_return_buffer =
		    strmcat(query_mode_func_return_buffer,"; ");
	    query_mode_func_return_buffer =
		strmcat(query_mode_func_return_buffer,
			QUERY_MODE_PARAM[qmp_parallel_lookup_max_time]);
	    query_mode_func_return_buffer =
		strmcat(query_mode_func_return_buffer,"=");
	    query_mode_func_return_buffer =
		strmcat(query_mode_func_return_buffer,
			resolv_stalled_seconds_value);
	    
	} else if (query_mode || 
		   resolv_stalled_seconds != STALLED_SECONDS_DEFAULT) {

	    if (resolv_stalled_seconds > 0) {
		if (resolv_stalled_seconds >= 60 &&
		    resolv_stalled_seconds % 60 == 0) {
		    int min =  resolv_stalled_seconds / 60;
		    
		    char * t =  elm_message(FRM("; %s=%dm"),
					    QUERY_MODE_PARAM[qmp_parallel_lookup_max_time],
					    min);
		    
		    query_mode_func_return_buffer =
			strmcat(query_mode_func_return_buffer,t);
		    free(t);
		} else {
		    char * t =  elm_message(FRM("; %s=%ds"),
					    QUERY_MODE_PARAM[qmp_parallel_lookup_max_time],
					    resolv_stalled_seconds);
		    
		    query_mode_func_return_buffer =
			strmcat(query_mode_func_return_buffer,t);
		    free(t);		 
		}
	    } else
		ok = 0;
	}

	*value = query_mode_func_return_buffer;
    }

 fail1:
    return ok;
}
/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

