static char rcsid[] = "@(#)$Id: iconv.c,v 2.16 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.16 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elmiconv.h"

DEBUG_VAR(Debug,__FILE__,"iconv");

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

/* UCS-2BE = UCS-2 big endian */
static const char * UNICODE = "UCS-2BE";

static const char * alternative[] = { "UCS-2", "UCS-2BE", NULL };

#define RAW_OVERHEAD     128
#define RAW_PER_UNICODE  8

/* State can not work correctly, but ... */
struct state_generic_1 {
    iconv_t to_unicode;

    char    raw_bytes[RAW_OVERHEAD + RAW_PER_UNICODE];
    int     raw_bytes_count;

    uint16  unicode_value;
};

static struct  map_info *cs_find_iconv P_((const char * map_name));

S_(cs_init_string cs_init_iconv)
static void cs_init_iconv(str)
     struct string *str;
{
    str->p->len = 0;
    str->p->private_flag = 0;
    str->p->a.bytes      = NULL;

    /* private_flag = 0 :    Use bytes (raw form)
       private_flag = 1 :    Use words (unicode)
    */
    
    /* If not map -- try create map with charset name as map name */
    if (! str->string_type->map_info && str->string_type->MIME_name)
	str->string_type->map_info = 
	    cs_find_iconv(str->string_type->MIME_name);

}

S_(cs_free_string cs_free_iconv)
static void cs_free_iconv(str)
     struct string *str;
{
    
    if (str->p->private_flag) {
	/* use words (unicode) */
	if (str->p->a.words) {
	    free(str->p->a.words);
	    str->p->a.words = NULL;
	}
	str->p->len = 0;
    } else {
	/* Use bytes (raw form) */
	if (str->p->a.bytes) {
	    free(str->p->a.bytes);
	    str->p->a.bytes = NULL;
	}
	str->p->len = 0;
    }
}

/* Gnu LIBICONV uses const, IRIX 6.5 does not use const on argument ... 
 */
#if defined(_LIBICONV_VERSION)
#define GNUIconst const
#elif __GLIBC__ == 2 && __GLIBC_MINOR__ < 2
/* - Also GNU libc 2.x uses const on argument
   -      GNU libc 2.2 no longer uses const for iconv(3) args
 */
#define GNUIconst const
#elif defined(sun)
/* - Also SunOS 5.10 uses const on argument
 */
#define GNUIconst const
#else
#define GNUIconst
#endif

static void convert_to_raw P_((const struct string *str));
static void convert_to_raw(str)
     const struct string *str;
{
    uint16        *words;
    int w_len;

    if (!str->p->private_flag)
    	panic("STRING PANIC",__FILE__,__LINE__,"convert_to_raw",
	      "Not in unicode form",0);

    words = str->p->a.words;
    w_len = str->p->len;
    str->p->a.words = NULL;

    str->p->private_flag = 0;   /* Indicate raw mode (bytes) */
    str->p->a.bytes      = NULL;
    str->p->len          = 0;

    if (words) {
	int x;

	DPRINT(Debug,61,(&Debug,
			 "convert_to_raw: words="));
	for (x = 0; x < w_len; x++) {
	    DPRINT(Debug,61,(&Debug," %04X",
			     words[x]));
	}
	DPRINT(Debug,61,(&Debug," (len=%d)\n",w_len));

	if (str->string_type->map_info) {
	    iconv_t XX = iconv_open(str->string_type->map_info->map_name,
				    UNICODE);
	    
	    if (XX && XX != (iconv_t)(-1)) {
		/* Guess enough space ... */
		int max_bytes = RAW_OVERHEAD + w_len * RAW_PER_UNICODE;
		size_t out_left = max_bytes;
		char * out_ptr;

		/* do UCS-2BE coding for words */
		size_t in_left = w_len * 2,i;
		char * s = safe_malloc(in_left);
		GNUIconst char * in_ptr = s;   

		DPRINT(Debug,62,(&Debug,
				 "convert_to_raw: ucs="));
		for (i = 0; i < w_len; i++) {
		    s[i*2]    = words[i] / 256;
		    s[i*2+1]  = words[i] % 256;

		    DPRINT(Debug,62,(&Debug," x%02X x%02X",
				     (unsigned char)s[i*2],
				     (unsigned char)s[i*2+1]));
		}
		DPRINT(Debug,62,(&Debug,"\n"));

		str->p->a.bytes = safe_malloc(max_bytes);
		out_ptr = us2s(str->p->a.bytes);

		DPRINT(Debug,62,(&Debug,
				 "convert_to_raw: raw="));
		while (in_left > 0) {
		    char * d = out_ptr;
		    size_t r = iconv(XX,
				     &in_ptr,&in_left,
				     &out_ptr,&out_left);
		    
		    if (r == (size_t)(-1)) {
			int y UNUSED_VAROK = (in_ptr-s)/2;
			DPRINT(Debug,4,(&Debug,
					"\nconvert_to_raw: %s: iconv failed at [%d]: errno = %d  [bytes=x%02Xx%02X]\n",
					str->string_type->MIME_name ? 
					str->string_type->MIME_name :
					"<no MIME name>",
					y,errno,
					(unsigned char)in_ptr[0],
					(unsigned char)in_ptr[1]));
			break;
		    }
		    while (d < out_ptr) {
			DPRINT(Debug,62,(&Debug,
					 " x%02X",(unsigned char)*d));
			
			d++;
		    }
		    if (r > 0) {
			DPRINT(Debug,61,(&Debug,
					 "\nconvert_to_raw: %s: %d non-reversible conversions\n",
					 str->string_type->MIME_name ? 
					 str->string_type->MIME_name :
					 "<no MIME name>",
					 r));
			DPRINT(Debug,62,(&Debug,
					 "convert_to_raw: raw="));
		    }
		}
		DPRINT(Debug,62,(&Debug,"\n"));

		str->p->len = out_ptr - us2s(str->p->a.bytes);

		DPRINT(Debug,61,(&Debug,
				 "convert_to_raw: %s: %d characters produces %d bytes raw encoding\n",
				 str->string_type->MIME_name ? 
				 str->string_type->MIME_name :
				 "<no MIME name>",
				 w_len,str->p->len));

		free(s);
		iconv_close(XX);
	    } else {
		DPRINT(Debug,4,(&Debug,
				"convert_to_raw: %s: No conversion from %s to %s available -- unicode characters lost\n",
				str->string_type->MIME_name ? 
				str->string_type->MIME_name :
				"<no MIME name>",
				UNICODE,
				str->string_type->map_info->map_name));
	    }
	} else {
	    DPRINT(Debug,4,(&Debug,
			    "convert_to_raw: %s: No map -- unicode characters lost\n",
			    str->string_type->MIME_name ? 
			    str->string_type->MIME_name :
			    "<no MIME name>"));
	}

	free(words);
    }
}

static void convert_to_unicode P_((const struct string *str));
static void convert_to_unicode(str)
     const struct string *str;
{
    unsigned char *bytes;
    int           b_len;
    
    if (str->p->private_flag)
    	panic("STRING PANIC",__FILE__,__LINE__,"convert_to_unicode",
	      "Not in raw form",0);

    bytes = str->p->a.bytes;
    b_len = str->p->len;
    str->p->a.bytes = NULL;

    str->p->private_flag = 1;   /* Indicate unicode */
    str->p->a.words = NULL;
    str->p->len     = 0;

    if (bytes) {
	int x;

	DPRINT(Debug,61,(&Debug,
			 "convert_to_unicode: raw="));
	for (x = 0; x < b_len; x++) {
	    DPRINT(Debug,61,(&Debug," x%02X",
			     bytes[x]));
	}
	DPRINT(Debug,61,(&Debug," (len=%d)\n",b_len));

	if (str->string_type->map_info) {
	    iconv_t XX = iconv_open(UNICODE,
				    str->string_type->map_info->map_name);
				    	    
	    if (XX && XX != (iconv_t)(-1)) {
		/* Guess enough space ... */
                int max_bytes = b_len *2;  /* Two bytes per target unicode */
                size_t out_left = max_bytes;

		char * s = safe_malloc(max_bytes);
		char * out_ptr = s;

		size_t in_left         = b_len;
		GNUIconst char * bytes_s = us2s(bytes);
		GNUIconst char * in_ptr = bytes_s;

		DPRINT(Debug,62,(&Debug,
				 "convert_to_raw: ucs="));

		while (in_left > 0) {
		    char * d = out_ptr;

		    size_t r = iconv(XX,
				     &in_ptr,&in_left,
				     &out_ptr,&out_left);
		 
		    if (r == (size_t)(-1)) {
			int y UNUSED_VAROK = in_ptr-bytes_s;
			DPRINT(Debug,4,(&Debug,
					"\nconvert_to_unicode: %s: iconv failed at [%d]: errno = %d [byte=x%02X]\n",
					str->string_type->MIME_name ? 
					str->string_type->MIME_name :
					"<no MIME name>",
					y,errno,
					(unsigned char)in_ptr[0]));
			break;
		    }
		    while (d < out_ptr) {
			DPRINT(Debug,62,(&Debug,
					 " x%02X",(unsigned char)*d));
			
			d++;
		    }
		    if (r > 0) {
			DPRINT(Debug,61,(&Debug,
					 "\nconvert_to_raw: %s: %d non-reversible conversions\n",
					 str->string_type->MIME_name ? 
					 str->string_type->MIME_name :
					 "<no MIME name>",
					 r));
			DPRINT(Debug,62,(&Debug,
					 "convert_to_unicode: ucs="));
		    }
		}
		DPRINT(Debug,62,(&Debug,"\n"));

		str->p->len = (out_ptr - s) / 2;
		
		DPRINT(Debug,61,(&Debug,
				 "convert_to_unicode: %s: %d raw bytes produces %d unicode characters\n",
				 str->string_type->MIME_name ? 
				 str->string_type->MIME_name :
				 "<no MIME name>",
				 b_len,str->p->len));

		if (str->p->len > 0) {
		    int i;
		    /* Now convert unicode data to words */

		    str->p->a.words = safe_array_realloc(str->p->a.words,
							 str->p->len,
							 sizeof (uint16));
		    
		    DPRINT(Debug,61,(&Debug,
				     "convert_to_unicode: words="));
		    for (i = 0; i < str->p->len; i++) {
			/* Convert to unsigned so arithemitic works */
			unsigned char a1 = s[i*2];
			unsigned char a2 = s[i*2+1];

			str->p->a.words[i] = a1 * 256 + a2;

			DPRINT(Debug,61,(&Debug,
					 " %04X",str->p->a.words[i]));
		    }
		    DPRINT(Debug,61,(&Debug,"\n"));
		}

		free(s);
		iconv_close(XX);
	    } else {
		DPRINT(Debug,4,(&Debug,
				"convert_to_unicode: %s: No conversion from %s to %s available -- raw characters lost\n",
				str->string_type->MIME_name ? 
				str->string_type->MIME_name :
				"<no MIME name>",
				str->string_type->map_info->map_name,
				UNICODE));				
	    }

	} else {
	    DPRINT(Debug,4,(&Debug,
			    "convert_to_unicode: %s: No map -- raw characters lost\n",
			    str->string_type->MIME_name ? 
			    str->string_type->MIME_name :
			    "<no MIME name>"));	
	}
	free(bytes);
    }
}

S_(cs_add_streambyte_to_string cs_add_streambyte_to_iconv)
static int cs_add_streambyte_to_iconv(str,ch)
     struct string *str;
     int ch;
{
#if 0
    /* We not currently use state for checking encoding ... */
    if (!str->p->state)
	str->p->state = new_state_1(str->string_type);
#endif

    if (str->p->private_flag)
	convert_to_raw(str);

    /* NOT test is this valid encoding ... */

    str->p->a.bytes = safe_realloc(str->p->a.bytes,
				   str->p->len+1);
    str->p->a.bytes[str->p->len++] = ch;

    /* So we return success even when encoding is not correct ... */
    return 1;
}

S_(cs_add_intdata_to_string cs_add_intdata_to_iconv)
static void cs_add_intdata_to_iconv(str,data)
     struct string *str;
     const struct string *data;
{
    /* In both str and data should be private_flag 
       set because caller of this is called check_length
       -- so next tests are NO-OPes.
    */

    if (!str->p->private_flag)
	convert_to_unicode(str);

    if (!data->p->private_flag)
	convert_to_unicode(data);
    
    if (data->p->len > 0) {
	int i;

	/* realloc with size 0 is equivalent of free and may 
	   corrupt memory ...
	*/
	
	/* NOTE:  str->p->a.words is not NUL terminated        */
	str->p->a.words = safe_array_realloc(str->p->a.words,
					     (str->p->len+data->p->len),
					     sizeof (uint16));
	
	for (i = 0; i < data->p->len; i++) {
	    str->p->a.words[str->p->len++] = data->p->a.words[i];
	}	
    }
}

/* Does compression, recalculation and so on ... 
   so 'const' is not very const
 */
S_(cs_check_length_string cs_check_length_iconv)
static void cs_check_length_iconv(str)
     const struct string *str;
{    
   /* First convert to unicode so length is correctly expressed
      in terms of characters (instead of terms of bytes)
   */
    
    if (!str->p->private_flag)
	convert_to_unicode(str);

    /* Then canonify */
    compress_unicode(str->p->a.words, &(str->p->len));
}

S_(cs_give_unicode_from_string cs_give_unicode_from_iconv)
static uint16 cs_give_unicode_from_iconv(str,pos,found)
     const struct string *str;
     int pos;
     int *found;
{
    if (!str->p->private_flag) {
	DPRINT(Debug,1,(&Debug,
			"cs_give_unicode_from_iconv: private_flag not set -- cs_check_length_iconv not called??\n"));
	*found = 0;
	return 0x003F;  /* '?' */
    }

    if (pos < 0 || pos >= str->p->len)
	panic("STRING PANIC",__FILE__,__LINE__,"cs_give_unicode_from_iconv",
	      "Index out of array",0);

    *found = 1;
    return str->p->a.words[pos];
}


S_(cs_unicode_vector_from_string cs_unicode_vector_from_iconv)
static enum unicode_vector_mode cs_unicode_vector_from_iconv P_((struct string_unicode_vector *ret,
								 const struct string *s,
								 int *errcount)); 
static enum unicode_vector_mode cs_unicode_vector_from_iconv(ret,s,errcount)
     struct string_unicode_vector *ret;
     const struct string *s;
     int *errcount;
{
    
    if (!s->p->private_flag) {
	convert_to_unicode(s);
    
	/* Then canonify */
	compress_unicode(s->p->a.words, &(s->p->len));
    }
   
    if (s->p->len > 0) {      
	size_t len = (size_t)s->p->len;
	size_t l = 0;
	int i;
	
	/* DOES NOT TAKE ACCOUNT CONVERSION ERRROS !! */

	ret->first_failure = len;
	ret->vector        = safe_array_realloc(ret->vector,len,
						sizeof (ret->vector[0]));

	for (i = 0; i < s->p->len; i++) {
	    
	    if (l >= len) 
		panic("STRING PANIC",__FILE__,__LINE__," cs_unicode_vector_from_iconv",
		      "Overflow",0);

	    ret->vector[l] = s->p->a.words[i];

	     l++;
	}
	ret->vector_len = l;
	
	if (errcount)
	    *errcount =0;

	return unicode_vector_done;
    }
    
    if (ret->vector) {
	free(ret->vector);
	ret->vector = NULL;
    }
    ret->vector_len = 0;
    ret->first_failure = 0;

    if (errcount)
	*errcount = 0;
    
    return unicode_vector_empty;
}


S_(cs_add_unicodedata_to_string cs_add_unicodedata_to_iconv)
static void cs_add_unicodedata_to_iconv(str,len,data,failcount)
     struct string *str;
     int len; 
     const uint16 *data;
     int *failcount;
{
    if (!str->p->private_flag)
	convert_to_unicode(str);


    /* FIXME:    --- no failure indication */
    *failcount = 0;

    if (len > 0) {
	int i;
	/* realloc with size 0 is equivalent of free and may 
	   corrupt memory ...
	*/
	
	/* NOTE:  str->p->a.words is not NUL terminated        */
	str->p->a.words = safe_array_realloc(str->p->a.words,
					     (str->p->len+len), sizeof (uint16));

	for (i = 0; i < len; i++) {
	    str->p->a.words[str->p->len++] = data[i];
	}
    }
}

S_(cs_cmp_string cs_cmp_iconv)
static int cs_cmp_iconv(str1,str2)
     const struct string *str1;
     const struct string *str2;
{
    int i;

    /* In both str1 and str2 should be private_flag 
       set because caller of this is called check_length
       -- so next tests are NO-OPes.
    */

    if (!str1->p->private_flag)
	convert_to_unicode(str1);
    if (!str2->p->private_flag)
	convert_to_unicode(str2);

    for (i = 0; i < str1->p->len && i < str2->p->len; i++) {
	if (str1->p->a.words[i] < str2->p->a.words[i])
	    return -1;
	if (str1->p->a.words[i] > str2->p->a.words[i])
	    return 1;
    }
    
    if (i < str1->p->len)
	return 1;
    if (i < str2->p->len)
	return -1;
    return 0;
}

S_(cs_stream_from_string cs_stream_from_iconv)
static unsigned char * cs_stream_from_iconv(str,printable,terminal,reslen)
     const struct string *str;
     int printable;
     screen_info_p terminal;
     int *reslen;
{
    unsigned char * ret = NULL;

    *reslen = 0;

    if (!printable) {
	int l = 0,i;

	/* Because not printable characters is not
	   requested, can return just raw encoding 
	*/
	
	if (str->p->private_flag)
	    convert_to_raw(str);

	ret = safe_malloc(str->p->len+1);

	for (i = 0; i < str->p->len; i++) {
	    ret[l++] = str->p->a.bytes[i];
	}

	ret[l] = '\0';
	*reslen = l;
    } else {
	int l = 0;

	/* Because printable characters are requested, we need convert
	   first to unicode so we can get printable information
	*/

	if (!str->p->private_flag)
	    convert_to_unicode(str);

	/* First we need just pick printable characters */
	if (str->p->len >0) {
	    int i;
	    uint16  * words = safe_calloc(str->p->len,sizeof (uint16));

	    size_t in_left = str->p->len * 2;
	    char * s = safe_malloc(in_left);
	    GNUIconst char * in_ptr = s;   

	    int max_bytes = RAW_OVERHEAD + str->p->len * RAW_PER_UNICODE;
	    size_t out_left = max_bytes;
	    char * out_ptr;
	    
	    ret = safe_malloc(max_bytes);
	    out_ptr = us2s(ret);

	    out_left--;  /* Reserve space for trailing \0 */

	    for (i = 0; i < str->p->len; i++) {
		if (!unicode_ch(str->p->a.words[i],UOP_printable))
		    words[i] = 0x003F;  /* '?' */
		else
		    words[i] = str->p->a.words[i];
	    }		

	    /* do UCS-2BE coding for words */

	    for (i = 0; i < str->p->len; i++) {
		s[i*2]    = words[i] / 256;
		s[i*2+1]  = words[i] % 256;
	    }

	    if (str->string_type->map_info) {
		iconv_t XX = iconv_open(str->string_type->map_info->map_name,
					UNICODE);

		if (XX && XX != (iconv_t)(-1)) {

		    while (in_left > 0) {
			size_t r = iconv(XX,
					 &in_ptr,&in_left,
					 &out_ptr,&out_left);
			
			if (r == (size_t)(-1)) {
			    int y UNUSED_VAROK = (in_ptr-s)/2;
			    DPRINT(Debug,4,(&Debug,
					    "cs_stream_from_iconv: %s: iconv failed at [%d]: errno = %d\n",
					    str->string_type->MIME_name ? 
					    str->string_type->MIME_name :
					    "<no MIME name>",
					    y,errno));
			    break;
			}
			if (r > 0) {
			    DPRINT(Debug,61,(&Debug,
					     "cs_stream_from_iconv: %s: %d non-reversible conversions\n",
					     str->string_type->MIME_name ? 
					     str->string_type->MIME_name :
					     "<no MIME name>",
					     r));
			}
		    }
		    
		    l = out_ptr - us2s(ret); 
		    
		    DPRINT(Debug,61,(&Debug,
				     "cs_stream_from_iconv: %s: %d characters produces %d bytes raw encoding\n",
				     str->string_type->MIME_name ? 
				     str->string_type->MIME_name :
				     "<no MIME name>",
				     str->p->len,l));
		    
		} else {
		    DPRINT(Debug,4,(&Debug,
				    "cs_stream_from_iconv: %s: No conversion from %s to %s available\n",
				    str->string_type->MIME_name ? 
				    str->string_type->MIME_name :
				    "<no MIME name>",
				    UNICODE,
				    str->string_type->map_info->map_name));
		}
	    } else {
		DPRINT(Debug,4,(&Debug,
				"cs_stream_from_iconv: %s: No map\n",
				str->string_type->MIME_name ? 
				str->string_type->MIME_name :
				"<no MIME name>"));		
	    }
	    
	    free(words);
	    free(s);

	} else  /* Just empty return */
	    ret = safe_malloc(1);

	ret[l] = '\0';
	*reslen = l;       
    }

    return ret;
}

S_(cs_can_ascii_string cs_can_ascii_iconv)
static int cs_can_ascii_iconv(str)
     const struct string *str;
{
    int i;
    
    if (!str->p->private_flag)
	convert_to_unicode(str);

    for (i = 0; i < str->p->len; i++) {
	/* str->p->a.words is unsigned */
	if (str->p->a.words[i] > 127)
	    return 0;
    }
    return 1;
}

S_(cs_streamclip_from_string cs_streamclip_from_iconv)
static unsigned char * cs_streamclip_from_iconv(str,pos,len,terminal,printable_len)
     const struct string *str;
     int *pos; 
     int len;
     screen_info_p terminal;                         /* NOT USED */
     struct cs_printable_len *printable_len;         /* NOT USED */
{
    unsigned char *ret = NULL;
    int l = 0;

    /* Because positions are in terms of characters we need convert
       data to unicode -- also this requires only printable characters       
    */
    
    if (!str->p->private_flag)
	convert_to_unicode(str);
    
    if (*pos < 0)
	panic("STRING PANIC",__FILE__,__LINE__,"cs_streamclip_from_iconv",
	      "Index out of array",0);

    if (len < 0)
	panic("STRING PANIC",__FILE__,__LINE__,"cs_streamclip_from_iconv",
	      "Negative size",0);

    /* First we need just pick printable characters */
    if (str->p->len >0) {
	int i;
	uint16  * words = safe_calloc(str->p->len,sizeof (uint16));
		
	int max_bytes;
	size_t out_left;
	char * out_ptr;
	int COUNT;
       
	for (i = 0; i < len && *pos < str->p->len; i++, (*pos)++) {
	    if (!unicode_ch(str->p->a.words[*pos],UOP_printable))
		words[i] = 0x003F;  /* '?' */
	    else
		words[i] = str->p->a.words[*pos];
	}		
	COUNT = i;    /* Length of clip */

	max_bytes = RAW_OVERHEAD + COUNT * RAW_PER_UNICODE;
	out_left = max_bytes;
	ret = safe_malloc(max_bytes);
	out_ptr = us2s(ret);
	
	out_left--;  /* Reserve space for trailing \0 */

	if (COUNT > 0) {
	    size_t in_left;
	    char * s;
	    GNUIconst char * in_ptr;
	    	    
	    in_left = COUNT * 2;
	    s = safe_malloc(in_left);
	    in_ptr = s;   
	    	   	    
	    /* do UCS-2BE coding for words */
	    
	    for (i = 0; i < COUNT; i++) {
		s[i*2]    = words[i] / 256;
		s[i*2+1]  = words[i] % 256;
	    }
	    
	    if (str->string_type->map_info) {
		iconv_t XX = iconv_open(str->string_type->map_info->map_name,
					UNICODE);
		
		if (XX && XX != (iconv_t)(-1)) {
		    
		    while (in_left > 0) {
			size_t r = iconv(XX,
					 &in_ptr,&in_left,
					 &out_ptr,&out_left);
			
			if (r == (size_t)(-1)) {
			    int y UNUSED_VAROK = (in_ptr-s)/2;
			    DPRINT(Debug,4,(&Debug,
					    "cs_streamclip_from_iconv: %s: iconv failed at [%d]: errno = %d\n",
					    str->string_type->MIME_name ? 
					    str->string_type->MIME_name :
					    "<no MIME name>",
					    y,errno));
			    break;
			}
			if (r > 0) {
			    DPRINT(Debug,61,(&Debug,
					     "cs_streamclip_from_iconv: %s: %d non-reversible conversions\n",
					     str->string_type->MIME_name ? 
					     str->string_type->MIME_name :
					     "<no MIME name>",
					     r));
			}
		    }
		    
		    l = out_ptr - us2s(ret); 
		    
		    DPRINT(Debug,61,(&Debug,
				     "cs_streamclip_from_iconv: %s: %d characters produces %d bytes raw encoding\n",
				     str->string_type->MIME_name ? 
				     str->string_type->MIME_name :
				     "<no MIME name>",
				     COUNT,l));
		    
		} else {
		    DPRINT(Debug,4,(&Debug,
				    "cs_streamclip_from_iconv: %s: No conversion from %s to %s available\n",
				    str->string_type->MIME_name ? 
				    str->string_type->MIME_name :
				    "<no MIME name>",
				    UNICODE,
				    str->string_type->map_info->map_name));
		}
	    } else {
		DPRINT(Debug,4,(&Debug,
				"cs_streamclip_from_iconv: %s: No map\n",
				str->string_type->MIME_name ? 
				str->string_type->MIME_name :
				"<no MIME name>"));		
	    }
	    
	    free(s);
	} 

	free(words);
	
    } else  /* Just empty return */
	ret = safe_malloc(1);
    
    ret[l] = '\0';
    
    return ret;
}

S_(cs_clip_from_string cs_clip_from_iconv)
static void cs_clip_from_iconv(ret,str,pos,len)
     struct string *ret;
     const struct string *str;
     int *pos; 
     int len;
{
    int i;
    /* Because positions are in terms of characters we need convert
       data to unicode 
    */

    if (!str->p->private_flag)
	convert_to_unicode(str);

    /* Clear ret also ... */
    cs_free_iconv(ret);
    ret->p->private_flag = 1;
    ret->p->a.words = NULL;
    ret->p->len = 0;

    if (*pos < 0)
	panic("STRING PANIC",__FILE__,__LINE__,"cs_clip_from_iconv",
	      "Index out of array",0);

    if (len < 0)
	panic("STRING PANIC",__FILE__,__LINE__,"cs_clip_from_iconv",
	      "Negative size",0);

    if (len > 0) {
	
	/* NOTE:  str->p->a.words is not NUL terminated        */
	ret->p->a.words = safe_calloc(len, sizeof (uint16));
	ret->p->len = 0;
	
	for (i = 0; i < len && *pos < str->p->len; i++, (*pos)++) {   
	    ret->p->a.words[ret->p->len++] = str->p->a.words[*pos];
	}
    } 
}     

S_(cs_find_pattern_from_string cs_find_pattern_from_iconv)
static int cs_find_pattern_from_iconv(str,pattern,ignore_case)
     const struct string *str;
     const struct string *pattern;
     int ignore_case;
{
    DPRINT(Debug,63,(&Debug, 
		     "cs_find_pattern_from_iconv=-1\n"));	
    return -1;   /* Use UNICODE comparision on upper level instead */
}
     
S_(cs_add_streambytes_to_string cs_add_streambytes_to_iconv)
static int cs_add_streambytes_to_iconv(str,count,data,errors)
     struct string *str;
     int count;
     const unsigned char *data;
     int *errors;
{
#if 0
    /* We not currently use state for checking encoding ... */
    if (!str->p->state)
	str->p->state = new_state_1(str->string_type);
#endif

    if (count < 0)
	panic("STRING PANIC",__FILE__,__LINE__,"cs_add_streambytes_to_iconv",
	      "Negative length",0);
    
    *errors = 0;

    if (str->p->private_flag)
	convert_to_raw(str);
    
    /* NOT test is this valid encoding ... */
    
    if (count > 0) {
	int i;
	/* realloc with size 0 is equivalent of free and may 
	   corrupt memory ...
	*/
	
	/* NOTE:  str->p->a.bytes is not NUL terminated        */
	str->p->a.bytes = safe_realloc(str->p->a.bytes,str->p->len+count);
	for (i = 0; i < count; i++)
	    str->p->a.bytes[str->p->len++] = data[i];
	return count;
    }
    return 0;  
}

static void map_init_bad P_((struct  map_info *map));
static void map_init_bad(map)
     struct  map_info *map;
{
    panic("STRING PANIC",__FILE__,__LINE__,"map_init_bad",
	  "map_init_bad called",0);
}

S_(cs_find_map_type cs_find_iconv)
static struct  map_info *cs_find_iconv(map_name)
     const char * map_name;
{
    static struct map_info **genmaps = NULL;
    static int               genmap_count = 0;
    struct map_info *ret = NULL;

    iconv_t XX;

    int i;

    for (i =0; i < genmap_count; i++)
	if (0 == strcmp(genmaps[i]->map_name,map_name))
	    return genmaps[i];

    XX = iconv_open(UNICODE,map_name);
    
    if (!XX || XX == (iconv_t)(-1)) {
	for (i = 0; alternative[i]; i++) {
	    XX = iconv_open(alternative[i],map_name);
	    if (XX && XX != (iconv_t)(-1)) {
		DPRINT(Debug,1,(&Debug, 
				"Using %s instead of %s for conversions\n",
				alternative[i],UNICODE));
		UNICODE = alternative[i];
		break;
	    }
	}
    }

    if (XX && XX != (iconv_t)(-1)) {

	DPRINT(Debug,4,(&Debug, 
			"Creating map %s: conversion from %s to %s is available\n",
			map_name,map_name,UNICODE));

	ret = safe_malloc(sizeof (struct map_info));
	ret -> map_type = &cs_iconv;
	ret -> map_name = safe_strdup(map_name);
	ret -> map_initialized = 1;
	ret -> map_init_it = map_init_bad;

	ret -> b.dummy = NULL;
	ret -> c.dummy = NULL;
	ret -> d.dummy = NULL;

	genmaps = safe_array_realloc(genmaps,
				     (genmap_count + 1),
				     sizeof (genmaps[0]));
        genmaps[genmap_count++] = ret;

	iconv_close(XX);
    }
    return ret;
}

S_(cs_remove_control cs_remove_control_iconv)
static void cs_remove_control_iconv(str)
     const struct string *str;
{
    int i;
    /* Convert to unicode so that we do not remove control
       characters which are part of encoding
    */
    
    if (!str->p->private_flag)
	convert_to_unicode(str);
    
    for (i = 0; i < str->p->len; i++) {
	if (str->p->a.words[i] < 32 ||
	    str->p->a.words[i] == 127)
	    str->p->a.words[i] = 32;
	else {
	    if (!unicode_ch(str->p->a.words[i],UOP_noctrl))
		str->p->a.words[i] = 32;	    
	}
    }    
}

S_(cs_add_state_to_string cs_add_state_to_iconv)
static void cs_add_state_to_iconv(str,ch)
     struct string *str;
     struct charset_state *ch;
{
    if (str->p->private_flag) {
	/* Add unicode value */

	
	/* NOTE:  str->p->a.words is not NUL terminated        */
	str->p->a.words = safe_array_realloc(str->p->a.words,
					     (str->p->len+1), sizeof (uint16));
	
	str->p->a.words[str->p->len++] = ch->a.g1->unicode_value;

    } else {
	int i;
	/* Add raw encoding */

	if (ch->a.g1->raw_bytes_count < 1)
	    return;
	
	/* NOTE:  str->p->a.bytes is not NUL terminated        */
	str->p->a.bytes = safe_realloc(str->p->a.bytes,
				       str->p->len+
				       ch->a.g1->raw_bytes_count);
	for (i = 0; i < ch->a.g1->raw_bytes_count; i++)
            str->p->a.bytes[str->p->len++] = ch->a.g1->raw_bytes[i];
    }
}

S_(cs_init_state cs_init_s_iconv)
static void cs_init_s_iconv(st)
     struct charset_state *st;
{

    st->a.g1 = safe_malloc(sizeof (*st->a.g1));
    
    /* defined in hdrs/defs.h */
    bzero((void *)st->a.g1, sizeof (*st->a.g1));

    st->a.g1->to_unicode = NULL;
    st->a.g1->raw_bytes_count = 0;
    st->a.g1->unicode_value   = 0;

    /* If not map -- try create map with charset name as map name */
    if (! st->charset->map_info && st->charset->MIME_name)
	st->charset->map_info = cs_find_iconv(st->charset->MIME_name);

    if (st->charset->map_info) {
	st->a.g1->to_unicode = iconv_open(UNICODE,
					   st->charset->map_info->map_name);
	if (NULL          == st->a.g1->to_unicode ||
	    (iconv_t)(-1) == st->a.g1->to_unicode) {
	    
	    DPRINT(Debug,4,(&Debug,
			     "iconv_open(\"%s\",\"%s\") failed: No conversion from %s to %s available\n",
			     UNICODE,st->charset->map_info->map_name,
			    st->charset->map_info->map_name,UNICODE));


	    st->a.g1->to_unicode = NULL;
	}
    }

    DPRINT(Debug,48,(&Debug,"cs_init_s_iconv: %p\n",st->a.g1));
}

S_(cs_free_state cs_free_s_iconv)
static void cs_free_s_iconv(st)
     struct charset_state *st;
{
    if (st->a.g1) {

	DPRINT(Debug,48,(&Debug,"cs_free_s_iconv: %p\n",st->a.g1));
	
	if (st->a.g1->to_unicode) {
	    iconv_close(st->a.g1->to_unicode);
	    st->a.g1->to_unicode = NULL;
	}
 
	free(st->a.g1);
	st->a.g1 = NULL;
    }
}

S_(cs_add_streambyte_to_state cs_add_streambyte_to_s_iconv)
static int cs_add_streambyte_to_s_iconv(st,ch)
     struct charset_state *st; 
     int ch;
{
    char outbuf[2];
    size_t out_left  = sizeof outbuf;
    char * outptr = &(outbuf[0]);

    GNUIconst char *inptr;
    size_t inleft;
    size_t r;
    
    if (ch < 0 || ch > 0xFF)
	panic("STRING PANIC",__FILE__,__LINE__,"cs_add_streambyte_to_s_iconv",
	      "Value not in range 0-255",0);

    if (st->a.g1->raw_bytes_count >=
	sizeof (st->a.g1->raw_bytes)) {
	DPRINT(Debug,4,(&Debug,
			"cs_add_streambyte_to_s_iconv: %p: Overlow (already %d bytes)\n",
			st->a.g1,
			st->a.g1->raw_bytes_count));
	return 0;    /* Overflow */
    }

    DPRINT(Debug,20,(&Debug,
		     "cs_add_streambyte_to_s_iconv: %p: byte index %d\n",
		     st->a.g1,
		     st->a.g1->raw_bytes_count));

    st->a.g1->raw_bytes[st->a.g1->raw_bytes_count++] = ch;
    
    if(!st->a.g1->to_unicode) {
	DPRINT(Debug,4,(&Debug,
			"cs_add_streambyte_to_s_iconv: %p: no state available\n",
			st->a.g1));
	return 0;    
    }

    inptr   = st->a.g1->raw_bytes;
    inleft  = st->a.g1->raw_bytes_count;

    r = iconv(st->a.g1->to_unicode,
	      &inptr,&inleft,
	      &outptr,&out_left);

    if ((size_t)(-1) == r) {
	int err = errno;
	
	if (EILSEQ == err) {
        
	    DPRINT(Debug,4,(&Debug,
			    "cs_add_streambyte_to_s_iconv: %p: iconv failed: Bad sequence",
			    st->a.g1
			    ));

	    if (inleft > 0) {
		size_t l;
		const char *x;

		DPRINT(Debug,4,(&Debug,", %zu bytes:",inleft));

		for (l = inleft, x = inptr; l > 0; l--, x++) {
		    unsigned char a = *x;
		    DPRINT(Debug,4,(&Debug," 0x%02x",
				    a));

		}
	    }
	    DPRINT(Debug,4,(&Debug,"\n"));
	    
	    return 0;
	}
	if (EINVAL == err) {
	    DPRINT(Debug,62,(&Debug, "cs_add_streambyte_to_s_iconv: %p: Incomplete sequence -- OK",
			     st->a.g1));
	    if (inleft > 0) {
		size_t l;
		const char *x;

		DPRINT(Debug,62,(&Debug,", %zu bytes:",inleft));

		for (l = inleft, x = inptr; l > 0; l--, x++) {
		    unsigned char a = *x;
		    DPRINT(Debug,62,(&Debug," 0x%02x",
				    a));

		}
	    }
	    DPRINT(Debug,62,(&Debug,"\n"));


	    return 1;
	}
	DPRINT(Debug,4,(&Debug,
			"cs_add_streambyte_to_s_iconv: %p: iconv failed: errno = %d\n",
			st->a.g1,err));
	return 0;
    }
    if (r > 0) {
	DPRINT(Debug,61,(&Debug,
			 "cs_add_streambyte_to_s_iconv: %p: %zu non-reversible conversions\n",
			 st->a.g1,r));
    }


    /* outptr  should either be
       &outbuf[2] or  &outbuf[0]
    */
    
    if (outptr == &outbuf[2]) {
	unsigned char a1;
	unsigned char a2;
    
	if (inleft > 0) {
	    DPRINT(Debug,4,(&Debug,
			    "cs_add_streambyte_to_s_iconv: %p: %zu bytes left!!\n",
			    st->a.g1,inleft));
	    return 0;
	}

	/* Convert to unsigned so arithemitic works */
	a1 = outbuf[0];
	a2 = outbuf[1];
	
	st->a.g1->unicode_value = a1 * 256 + a2;
	st->ready         = 1;

    } else {

	if (outptr != &outbuf[0]) {
	    DPRINT(Debug,4,(&Debug,
			    "cs_add_streambyte_to_s_iconv: %p: BAd outptr ?\n",
			    st->a.g1));
	    return 0;
	}		

	/* Shift sequence? */
	    
	if (inleft > 0) {
	    
	    DPRINT(Debug,4,(&Debug,
			    "cs_add_streambyte_to_s_iconv: %p: Shifting %zu bytes\n",
			    st->a.g1,inleft));
	    
	    
	    if (inleft >= sizeof (st->a.g1->raw_bytes)) {
		DPRINT(Debug,4,(&Debug,
				"cs_add_streambyte_to_s_iconv: Overlow (buffer %zu bytes)\n",
				sizeof (st->a.g1->raw_bytes)));
		return 0;		
	    }
		
	    memmove(st->a.g1->raw_bytes,inptr,inleft);	    	    
	}
	    
	st->a.g1->raw_bytes_count = inleft;		
    }
    
    return 1;
}

S_(cs_soft_reset_state cs_soft_reset_s_iconv)
static void cs_soft_reset_s_iconv(st)
     struct charset_state *st;
{
    static int warned = 0;

    DPRINT(Debug,4,(&Debug,
		    "cs_soft_reset_s_iconv: %p: %d bytes count\n",
		    st->a.g1,st->a.g1->raw_bytes_count));

    
    st->a.g1->raw_bytes_count = 0;
    st->a.g1->unicode_value   = 0;

    if (st->ready) {
	st->ready                 = 0;

	/* Assuming that this is enough */
	return;
    }
    
    /* soft reset should reset only partially state, so ... */
    
    if (!warned) {
	warned = 1;
	lib_error(CATGETS(elm_msg_cat,
			  IconvSet,
			  IconvUnsupportedKeyboardInput,
			  "Iconv charsets are unsupported as keyboard input!"));
    }

    if(!st->a.g1->to_unicode)
	return;

    /* May reset too much */
    iconv(st->a.g1->to_unicode,NULL,NULL,NULL,NULL);
}

S_(cs_give_unicode_from_state cs_give_unicode_from_s_iconv)
static uint16 cs_give_unicode_from_s_iconv(st,found)
     struct charset_state *st;
     int *found;
{
    *found =1;
    return st->a.g1->unicode_value;
}

S_(cs_state_same_char cs_s_iconv_same_char)
static int cs_s_iconv_same_char(A,B,ignore_case)
     struct charset_state *A;
     struct charset_state *B;
     int ignore_case;
{
    if (A->a.g1->unicode_value == B->a.g1->unicode_value)
	return 1;
    
    if (ignore_case) 
	return -1;  /* Use UNICODE values for comparision */
    
    return 0;
}

S_(cs_state_printable cs_s_iconv_printable)
static int cs_s_iconv_printable(st)
     struct charset_state *st;
{
    return unicode_ch(st->a.g1->unicode_value,UOP_printable) != 0;
}

/* If character corresponds one byte on stream, returns it.
 * Otherwise returns 0. This is used implement ReadCh().
 * It is assumed that returned character corresponds to
 * code character set (and perhaps also US-ASCII)
 */
S_(cs_state_is_onebyte cs_s_inconv_is_onebyte)
static int cs_s_inconv_is_onebyte(st)
     struct charset_state *st;
{
    if (1 == st->a.g1->raw_bytes_count)
	return st->a.g1->raw_bytes[0];
    return 0;
}

static int cs_iconv_properties P_((charset_t st));
static int cs_iconv_properties(st)
     charset_t st;
{
    int prop = 0;
    
    /* If not map -- try create map with charset name as map name */
    if (! st->map_info && st->MIME_name)
	st->map_info = cs_find_iconv(st->MIME_name);

    if (st->map_info) {
	iconv_t XX = iconv_open(st->map_info->map_name,
				UNICODE);
	if (XX && XX != (iconv_t)(-1)) {

	    /* We know charset */
	    prop |= CS_printable | CS_mapping;
	    iconv_close(XX);

	} else {
	    DPRINT(Debug,7,(&Debug,
			    "cs_iconv_properties: %s: No conversion from %s to %s available\n",
			    st->MIME_name ? 
			    st->MIME_name :
			    "<no MIME name>",
			    UNICODE,
			    st->map_info->map_name));
	}
    } else {
	DPRINT(Debug,7,(&Debug,
			"cs_iconv_properties: %s: No map\n",
			st->MIME_name ? 
			st->MIME_name :
			"<no MIME name>"));	
    }

    return prop;
}

S_(cs_estimate_clip_string cs_estimate_clip_unsupported)
static int cs_estimate_clip_unsupported(str,pos,len,terminal,printable_len)
     const struct string *str;
     int pos; 
     int len;      /* UPPER LIMIT */
     screen_info_p terminal;
     struct cs_printable_len *printable_len;
{
    panic("STRING PANIC",__FILE__,__LINE__,"cs_estimate_clip_unsupported",
	  "cs_estimate_clip_unsupported() called",0);

    return -1;
}

static int cs_iso2022_info_set_iconv P_((struct charcode_info *new_vals,
					  struct setlist       *new_setlist,
					  int setcount));
static int cs_iso2022_info_set_iconv(new_vals, new_setlist,setcount)
     struct charcode_info *new_vals;
     struct setlist       *new_setlist;
     int setcount;
{
    int ptr_94 = -1;
    int ptr_96 = -1;
    int banks[ISO2022_BANK_NUM];

    if (!cs_info_set_scan(new_vals,new_setlist,setcount,&ptr_94,&ptr_96,
			  banks)) {
	return 0;
    }
	
    new_vals->iso2022_info = loc_setlist(*new_setlist);
    new_vals->flags       &= ~SET_nodata;
	
    return 1;
}

S_(cs_give_bytecode_from_string cs_give_bytecode_from_iconv)
unsigned char cs_give_bytecode_from_iconv(str,pos,found)
   const struct string *str;
   int pos; 
   int *found;
{
    if (pos < 0 || pos >= str->p->len)
	panic("STRING PANIC",__FILE__,__LINE__,"cs_give_bytecode_from_iconv",
	      "Index out of array",0);

    if (str->p->private_flag) {
	*found = 0;
	return 0;
    } else {
	unsigned char ch;

	/* Use bytes (raw form) */

	ch = str->p->a.bytes[pos];
	*found = 1;
	return ch;
    }
}

S_(cs_have_unicode cs_iconv_have_unicode)
static enum charset_unicode_stat cs_iconv_have_unicode P_((charset_t set,
							     int unicode));
static enum charset_unicode_stat cs_iconv_have_unicode(set,unicode)
     charset_t set;
     int unicode;
{
     enum charset_unicode_stat ret = charset_missing_unicode;
    
    if (unicode < 0 || unicode > 0xFFFF)
	ret = charset_unicode_bad;
    else if (UNICODE_BAD_CHAR == unicode)
	ret = charset_unicode_unknown;
    else {
	static int warn UNUSED_VAROK = 1;
	
	DPRINT(Debug,warn,(&Debug,
			   "cs_iconv_have_unicode: Not supported\n"));

	warn = 20;
	ret = charset_have_unicode;
    }
    
    return ret;
}



struct charset_type cs_iconv    =  { "iconv",
				     cs_init_iconv,
				     cs_free_iconv,
				     cs_add_streambyte_to_iconv,
				     cs_add_intdata_to_iconv,
				     cs_check_length_iconv,
				     cs_give_unicode_from_iconv,
				     cs_add_unicodedata_to_iconv,
				     cs_cmp_iconv,
				     cs_stream_from_iconv,
				     cs_can_ascii_iconv,  
				     cs_streamclip_from_iconv,
				     cs_clip_from_iconv,  
				     cs_find_pattern_from_iconv,
				     cs_add_streambytes_to_iconv,
				     cs_find_iconv,
				     cs_remove_control_iconv,   
				     cs_add_state_to_iconv,
				     cs_init_s_iconv,
				     cs_free_s_iconv,
				     cs_add_streambyte_to_s_iconv,
				     cs_soft_reset_s_iconv,
				     cs_give_unicode_from_s_iconv,
				     cs_s_iconv_same_char,
				     cs_s_iconv_printable,
				     cs_s_inconv_is_onebyte,
				     cs_iconv_properties,
				     cs_estimate_clip_unsupported,
				     cs_iso2022_info_set_iconv,
				     cs_give_bytecode_from_iconv,
				     cs_digest_feed_from_null,
				     cs_unicode_vector_from_iconv,
				     cs_iconv_have_unicode,
				     
				     NULL /* Next charset type on this module */
};

static int auto_charset = 0;



#if ANSI_C
CS_charset_type_f CS_charset_type;
#endif
struct charset_type * CS_charset_type P_((const char *name));
struct charset_type * CS_charset_type(name)
     const char *name;
{
    struct  charset_type * walk;

    for (walk = &cs_iconv;
	 walk; 
	 walk = walk -> next_type) {
	if (0 == strcmp(name,walk->type_name))
	    break;

    }
    
    if (walk) {
	DPRINT(Debug,8,(&Debug,
			"CS_charset_type: module iconv provides charset type=%s\n",
			walk->type_name));
	return walk;
    }
    
    DPRINT(Debug,10,(&Debug,
		     "CS_charset_type: charset type %s not found from module iconv\n",
		     name));
    
    return NULL;
}



#if ANSI_C
CS_auto_charset_f CS_auto_charset;
#endif
int CS_auto_charset  P_((const char *name,
			 struct  charset_type **type,
			 struct  map_info **map));
int CS_auto_charset(name,type,map)
     const char *name;
     struct  charset_type **type;
     struct  map_info **map;
{
    *type = NULL;
    *map = NULL;

    if (auto_charset) {
	*map = cs_find_iconv(name);

	if (*map) {
	    *type = &cs_iconv;
	    return 1;
	}
    }
    return 0;
}

#include "rc_imp.h"
#include "save_opts.h"

static ZZZ_SAVE_TYPE save_info_data[] = {
    { "auto-charset",   ZZZ_DT_BOL(&auto_charset),   ZZZ_TAIL },
};

#if ANSI_C
provides_RC_options_f provides_RC_options2;
#endif
struct rc_save_info_rec * provides_RC_options2(count,s_size)
     size_t *count; 
     size_t *s_size;
{
    *s_size = sizeof (save_info_data[0]);

    *count = (sizeof save_info_data) / *s_size;
    return (struct rc_save_info_rec *) save_info_data;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
