static char rcsid[] = "@(#)$Id: mime_selector.c,v 2.25 2022/08/19 16:13:04 hurtta Exp $";
 
/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.25 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/
 
#include "def_melib.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"mime");                                   

void mime_selector_free (p)
     struct mime_selected_handler **p;
{
    
    if (MIME_selector_magic != (*p)->magic) 
        panic("MAILCAP PANIC",__FILE__,__LINE__,"mime_selector_free",
              "Bad magic number",0);                                 
    
    (*p)->handler       = NULL;
    (*p)->entry         = NULL;
    (*p)->use_entry     = 0;
    (*p)->EC_decoder    = null_EC_decoder;
    (*p)->SG_decoder    = null_SG_decoder;
    (*p)->selected_alternative = NULL;
    (*p)->start_part           = NULL;

    if ((*p)->other_entries) {
	free((*p)->other_entries);
	(*p)->other_entries = NULL;
    }
    (*p)->num_other_entries = 0;
    
    
    (*p)->magic = 0;         /* Invalidate */
    
    free(*p);
    *p = NULL;
} 

static void mime_selector_reset P_((struct mime_selected_handler *handler));
static void mime_selector_reset(handler)
    struct mime_selected_handler *handler;
{
    if (MIME_selector_magic != handler->magic) 
        panic("MAILCAP PANIC",__FILE__,__LINE__,"mime_selector_reset",
              "Bad magic number",0);


    handler->handler       = NULL;   /* Not selected yet */
    handler->entry         = NULL;
    handler->use_entry     = 0;

    if (handler->other_entries) {
	free(handler->other_entries);
	handler->other_entries = NULL;
    }
    handler->num_other_entries = 0;
    
    handler->EC_decoder    = null_EC_decoder;
    handler->SG_decoder    = null_SG_decoder;
    handler->selected_alternative    = NULL;
    handler->start_part           = NULL;
    
}

static  struct mime_selected_handler * malloc_mime_selector P_((void));
static  struct mime_selected_handler * malloc_mime_selector()
{
    struct mime_selected_handler *handler
	= safe_zero_alloc(sizeof (*handler));

    handler->magic = MIME_selector_magic;
    
    handler->handler       = NULL;   /* Not selected yet */
    handler->entry         = NULL;
    handler->use_entry     = 0;

    handler->other_entries = NULL;    
    handler->num_other_entries = 0;
    
    handler->EC_decoder    = null_EC_decoder;
    handler->SG_decoder    = null_SG_decoder;
    handler->selected_alternative    = NULL;
    handler->start_part           = NULL;
    
    return handler;
}


/* old mime_notplain() reinvented here 
   returns mask: 

        NOTPLAIN_need_metamail           0x01
	NOTPLAIN_need_mailcap            0x02
	NOTPLAIN_canuse_mailcap          0x04
*/

#if DEBUG
char * mime_debug_classify_f(c) 
     int c;
{
    static char buffer[160];

    if (!c)
	return "none";

    if (c < 0)
	return "failed";

    buffer[0] = '\0';
    buffer[2] = '\0';   /* In case there is unknown bits set */

    if (c & NOTPLAIN_need_metamail)
	strfcat(buffer,", NOTPLAIN_need_metamail",sizeof buffer);
 
    if (c & NOTPLAIN_need_mailcap)
	strfcat(buffer,", NOTPLAIN_need_mailcap",sizeof buffer);

    if (c & NOTPLAIN_canuse_mailcap)
	strfcat(buffer,", NOTPLAIN_canuse_mailcap",sizeof buffer);

    if (c & NOTPLAIN_is_fallback)
	strfcat(buffer,", NOTPLAIN_is_fallback",sizeof buffer);

    if (c & NOTPLAIN_metamail_blacklisted)
	strfcat(buffer,", NOTPLAIN_metamail_blacklisted",sizeof buffer);

    if (c & NOTPLAIN_type_not_found)
    	strfcat(buffer,", NOTPLAIN_type_not_found",sizeof buffer);

    if (c & NOTPLAIN_mailcap_skipped)
	strfcat(buffer,", NOTPLAIN_mailcap_skipped",sizeof buffer);
    
    return buffer +2;

}
#endif

static int mime_classify_metamail P_((mime_t *p,
				      struct header_rec * hdr));
static int mime_classify_metamail(p,hdr) 
     mime_t *p;
     struct header_rec * hdr;
{
    int  submask        = 0;

    if (p->magic != MIME_magic)
	panic("MIME PANIC",__FILE__,__LINE__,"mime_classify_metamail",
	      "Bad magic number",0);
    
    if (p->TYPE) {
	struct media_type_handle * H = NULL;
	int is_default = 0;	
	
	/* check if metamail need to be blacklisted: NOTPLAIN_metamail_blacklisted */
		
	while (walk_mt_handler(p->TYPE,&H,&is_default,handle_metamail_mailcap_entry)) {
	    
	    struct  metamail_mailcap_entry * f;
	    
	    if (H->type != handle_metamail_mailcap_entry)
		panic("MIME PARSER PANIC",__FILE__,__LINE__,
		      "mime_classify_metamail",
		      "Bad handler type",0);
	    
	    DPRINT(Debug,11,(&Debug, 
			     "mime_classify_metamail [%s/%s] metamail mailcap = %p (media_type_handle %p) default:%d\n",
			     get_major_type_name(p->TYPE), 
			     get_subtype_name(p->TYPE),
			     H->p.metamail_mailcap,
			     H,
			     is_default));
	    
	    f = H->p.metamail_mailcap;
	    
	    if (metamail_mailcap_is_blacklisted(f)) {
		DPRINT(Debug,11,(&Debug, 
				 "mime_classify_metamail [%s/%s] have blacklisted metamail mailcap view command\n",
				 get_major_type_name(p->TYPE), 
				 get_subtype_name(p->TYPE)));
		
		
		/* Mark first entry */

		if (! p->metamail_blacklisted)
		    p->metamail_blacklisted = f;

		submask |= NOTPLAIN_metamail_blacklisted;

	    }
	}	
    }


    if (p->mime_flags != (p->mime_flags | submask)) {
	int a = p->mime_flags | submask;
	
	
	DPRINT(Debug,11,(&Debug, 
			 "mime_classify_metamail: changing mime_flags from %d (%s) ",
			 p->mime_flags,mime_debug_classify_f(p->mime_flags)));

	/* Do not call mime_debug_classify_f() twise on same output, return is
	   statically alloced! 
	*/
	DPRINT(Debug,11,(&Debug,
			 "to %d (%s) for %s/%s\n",
			 a,mime_debug_classify_f(a),
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE)));

	p->mime_flags = a;

	if (p->mime_flags != a)
	    mime_panic(__FILE__,__LINE__,"mime_classify_metamail",
		       "mime_flags too narrow?");

    }


    DPRINT(Debug,11,(&Debug, 
		     "mime_classify_metamail(%p) = %d (%s)\n",
		     p,
		     submask, mime_debug_classify_f(submask)));
    
    return submask;
}

int mime_classify_media(p,hdr,mode)
     mime_t                 * p;
     struct header_rec     * hdr;
     enum mime_classify_mode mode;
{
    int Flags;
    struct media_type_handle * H = NULL;
    int is_default = 0;
    int is_fallback = 0;
    
    int  submask        = 0;

    DPRINT(Debug,11,(&Debug, "mime_classify_media(%p): mode=%d",
		     p,mode));
    switch (mode) {
    case mime_classify_all:
	DPRINT(Debug,11,(&Debug, " mime_classify_all"));
	break;
    case mime_classify_skip_mailcap:
	DPRINT(Debug,11,(&Debug, " mime_classify_skip_mailcap"));
	break;
    }
    DPRINT(Debug,11,(&Debug, "\n"));
    
    
    if (p->magic != MIME_magic)
	panic("MIME PANIC",__FILE__,__LINE__,"mime_classify_media",
	      "Bad magic number",0);

    
    if (! p->TYPE) {
	DPRINT(Debug,1,(&Debug, 
			 "mime_classify_media(%p)  NOT PARSED\n",
			p));
	submask = NOTPLAIN_need_metamail;
	goto bad_encoding;
    }

    DPRINT(Debug,11,(&Debug, 
		     "mime_classify_media(%p) << type: %p=%s/%s - disposition %d (%s)\n",
		     p,
		     p->TYPE,
		     get_major_type_name(p->TYPE),
		     get_subtype_name(p->TYPE),
		     p->disposition,DISPOSITION(p->disposition)));
	   	  
    Flags = get_type_flags(p->TYPE);

    if (Flags) {
	DPRINT(Debug,11,(&Debug, "       Flags: %s%s%s%s%s%s%s\n",
			 (Flags & MIME_RFC822) ? " MIME_RFC822" : "",
			 (Flags & MIME_MIXED)  ? " MIME_MIXED" : "",
			 (Flags & MIME_DIGEST) ? " MIME_DIGEST" : "",
			 (Flags & MIME_ALTERNATIVE) ? " MIME_ALTERNATIVE" : "",
			 (Flags & MIME_SIGNED) ? " MIME_SIGNED" : "",
			 (Flags & MIME_ENCRYPTED) ? " MIME_ENCRYPTED" : "",
			 (Flags & MIME_REPORT)    ? " MIME_REPORT" : ""
			 ));
    }


    submask |= mime_classify_metamail(p,hdr);

    if (! is_valid_encoding(p->encoding)) {
	DPRINT(Debug,11,(&Debug, 
			 "mime_classify_media(%p) -- type: %p=%s/%s; encoding=%s (%d)\n",
			 p,
			 p->TYPE,
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE),
			 ENCODING(p->encoding),p->encoding));
	submask |= NOTPLAIN_need_metamail;
	goto bad_encoding;
    }


    /* so that handle_pager is available */
    melib_register_decoders();

    if (p->handler_data) {
	if (MIME_selector_magic != p->handler_data->magic) 
	    panic("MAILCAP PANIC",__FILE__,__LINE__,"mime_classify_media",
		  "Bad magic number",0);                                 	
    } else {
	p->handler_data        =  malloc_mime_selector();	
    }


    mime_selector_reset(p->handler_data);    

    /* RESET loop variables */
    H = NULL;
    is_default = 0;

    while(walk_mt_handler(p->TYPE,&H,&is_default,handle_pager)) {
	int r;

	if (H->type != handle_pager) {
	    mime_panic(__FILE__,__LINE__,"mime_classify_media",
		       "Unexpected handler type");
	    continue; /* not reached */
	}

	DPRINT(Debug,11,(&Debug, 
			 "mime_classify_media [%s/%s] func = %p, selector(pager) = %p (media_type_handle %p) default:%d\n",
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE),
			 H->p.pager->func,
			 H->p.pager,
			 H,
			 is_default));
	
	r = H->p.pager->selector(p,hdr);

	if (r >= 0) {

	    p->handler_data->handler = H->p.pager;
	    
	    is_fallback  = r & NOTPLAIN_is_fallback;

	    submask |= (r & ~NOTPLAIN_is_fallback);

	    DPRINT(Debug,11,(&Debug, 
			     "mime_classify_media: internal pager found for %s/%s, r=%d (%s)\n",
			     get_major_type_name(p->TYPE), 
			     get_subtype_name(p->TYPE),
			     r, mime_debug_classify_f(r)));

	    if (is_fallback) {
		DPRINT(Debug,11,(&Debug, 
				 "mime_classify_media: ... but continuing search\n",
				 get_major_type_name(p->TYPE), 
				 get_subtype_name(p->TYPE)));

	    } else
		goto found;
	}

    }

    /* Check mailcap */

    /* RFC 1343 is not completely clear, but it is assumed that "test"
       entry does _not_ need mail data ,,,

       therefore we do NOT run STATE_in_decode on here ....
    */

    /* RESET loop variables */
    H = NULL;
    is_default = 0;

    while (walk_mt_handler(p->TYPE,&H,&is_default,handle_mailcap_entry)) {
	struct mailcap_entry *f;
	
	if (H->type != handle_mailcap_entry)
	    panic("MIME PARSER PANIC",__FILE__,__LINE__,
		  "mime_classify_media",
		  "Bad handler type",0);


	DPRINT(Debug,11,(&Debug, 
			 "mime_classify_media [%s/%s] mailcap = %p (media_type_handle %p) default:%d\n",
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE),
			 H->p.mailcap,
			 H,
			 is_default));

	switch (mode) {
	case mime_classify_all:
       	    
	    f = H->p.mailcap;
	    
	    if (mailcap_is_valid_view(f,p)) {

		if (p->handler_data->entry) {
		    DPRINT(Debug,11,(&Debug, 
				     "mime_classify_media: other mailcap pager found for %s/%s\n",
				     get_major_type_name(p->TYPE), 
				     get_subtype_name(p->TYPE)));
		} else {
		
		    submask               |= NOTPLAIN_need_mailcap;
		    p->handler_data->entry = f;
		    
		    DPRINT(Debug,11,(&Debug, 
				     "mime_classify_media: mailcap pager found for %s/%s\n",
				     get_major_type_name(p->TYPE), 
				     get_subtype_name(p->TYPE)));
		}

		if (mailcap_select_other) {

		    /* Also include original entry! */
		    		    
		    p->handler_data->other_entries =
			safe_array_realloc(p->handler_data->other_entries,
					   p->handler_data->num_other_entries +1,
					   sizeof (p->handler_data->other_entries[0]));

		    p->handler_data->other_entries[p->handler_data->num_other_entries++]
			= f;
		    
		} else {
		    goto found;
		}
		    
	    }
	    break;
	case mime_classify_skip_mailcap:
	    submask               |=   NOTPLAIN_mailcap_skipped;
	    
	    DPRINT(Debug,11,(&Debug, 
			     "mime_classify_media: mailcap pager skipped for %s/%s\n",
			     get_major_type_name(p->TYPE), 
			     get_subtype_name(p->TYPE)));
	    goto skipped;
	}    	    
    }

    if (p->handler_data->entry) {
	DPRINT(Debug,11,(&Debug, 
			 "mime_classify_media: mailcap pager was found for %s/%s\n",
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE)));
	
    } else {
    
	/* Not found */
	submask |= NOTPLAIN_type_not_found | NOTPLAIN_need_metamail; 
    }
	
 skipped:        
 found:

    
    if ((submask | is_fallback) != p->mime_flags && submask >= 0) {
	int a = (submask | is_fallback);

	DPRINT(Debug,11,(&Debug, 
			 "mime_classify_media: changing mime_flags from %d (%s) ",
			 p->mime_flags,mime_debug_classify_f(p->mime_flags)));

	/* Do not call mime_debug_classify_f() twice on same output, return is
	   statically alloced! 
	*/
	DPRINT(Debug,11,(&Debug,
			 "to %d (%s) for %s/%s\n",
			 a,mime_debug_classify_f(a),
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE)));

	p->mime_flags = a;

	if (p->mime_flags != a)
	    mime_panic(__FILE__,__LINE__,"mime_classify_media",
		       "mime_flags too narrow?");
    }
    
 bad_encoding:

    DPRINT(Debug,11,(&Debug, 
		     "mime_classify_media(%p) = %d (%s)\n",
		     p,
		     submask, mime_debug_classify_f(submask)));
    
    return submask;
}

/* mime_classify_subparts replaces can_handle() 

   returns mask: 

        NOTPLAIN_need_metamail           0x01
	NOTPLAIN_need_mailcap            0x02
	NOTPLAIN_canuse_mailcap          0x04

	NOTPLAIN_metamail_blacklisted    0x10
*/

int mime_classify_subparts(ptr,hdr)     
     mime_t *ptr;
     struct header_rec * hdr;
{
    int i;
    int count;
    
    int  submask        = 0;

    if (!ptr->parser_data) {
	DPRINT(Debug,9,(&Debug, 
			"mime_classify_subparts: No parser data, type %p %s/%s\n",
			ptr->TYPE,
			get_major_type_name(ptr->TYPE), 
			get_subtype_name(ptr->TYPE)));
			
	
	submask  = NOTPLAIN_need_metamail;

	goto out;
    }

    count = mime_parser_subparts(ptr->parser_data);

    for (i = 0; i < count; i++) {
	 mime_t *att = 	mime_parser_index(ptr->parser_data,i);

	 if (att->magic != MIME_magic)
	     mime_panic(__FILE__,__LINE__,"mime_classify_subparts",
			"Bad magic number (subpart)");
	 
	 if (att->disposition == DISP_INLINE) {      
	     int f = mime_classify_media(att, hdr, mime_classify_all);

	     if (f < 0) {
		 DPRINT(Debug,9,(&Debug, 
				 "mime_classify_subparts -- Failed: %p=%s/%s (f=%d)\n",
				 att->TYPE,
				 get_major_type_name(att->TYPE), 
				 get_subtype_name(att->TYPE),
				 f));
		 submask = -1;
		 goto out;
	     }

	     DPRINT(Debug,9,(&Debug, 
			     "mime_classify_subparts:     f=%d (%s): %p=%s/%s\n",
			     f,
			     mime_debug_classify_f(f),
			     att->TYPE,
			     get_major_type_name(att->TYPE), 
			     get_subtype_name(att->TYPE)));

	     submask |= f;

	 } else {
	     int f = mime_classify_metamail(att,hdr);

	     DPRINT(Debug,9,(&Debug, 
			     "mime_classify_subparts:  f=%d (%s) Attachment: %p=%s/%s\n",
			     f,
			     mime_debug_classify_f(f),
			     att->TYPE,
			     get_major_type_name(att->TYPE),
			     get_subtype_name(att->TYPE)));

	     if (f >= 0)
		 submask |= f;
	 }
    }
    
 out:

    DPRINT(Debug,11,(&Debug, 
		     "mime_classify_subparts(%p) = %d (%s)  -- type %p=%s/%s\n",
		     ptr,
		     submask,mime_debug_classify_f(submask),
		     ptr->TYPE,
		     get_major_type_name(ptr->TYPE), 
		     get_subtype_name(ptr->TYPE)));
    
    return submask;
}

int mime_classify_start_page(ptr, ret,hdr)     
     mime_t *ptr;
     mime_t **ret;
     struct header_rec * hdr;
{
    int count;
    
    int  submask        = 0;

    const char *startid = get_mime_param_compat(ptr->TYPE_opts,"start");
    mime_t * start_part = NULL;
    
    
    if (ret)
	*ret = NULL;
    
    if (!ptr->parser_data) {
	DPRINT(Debug,9,(&Debug, 
			"mime_classify_start_page: No parser data, type %p %s/%s\n",
			ptr->TYPE,
			get_major_type_name(ptr->TYPE), 
			get_subtype_name(ptr->TYPE)));
	
	
	submask  = NOTPLAIN_need_metamail;
	
	goto out;
    }

    count = mime_parser_subparts(ptr->parser_data);

    if (count < 1) {
	DPRINT(Debug,9,(&Debug, 
			"mime_classify_start_page: No parts, type %p %s/%s\n",
			ptr->TYPE,
			get_major_type_name(ptr->TYPE), 
			get_subtype_name(ptr->TYPE)));
	submask = -1;
	goto out;	
    }

    if (startid) {
	int i;
	
	struct message_id * Start_ID =
	    parse_header_message_id(NULL,startid,
				    0 /* demime -- should not include comment */,
				    hdr ? hdr->header_charset : ASCII_SET,
				    NULL);

	if (! Start_ID) {
	    DPRINT(Debug,9,(&Debug, 
			    "mime_classify_start_page: Failed to parse start=%Q,  type %p %s/%s\n",
			    startid,
			    ptr->TYPE,
			    get_major_type_name(ptr->TYPE), 
			    get_subtype_name(ptr->TYPE)));
	    submask = -1;
	    goto out;	
	}

	for (i = 0; i < count; i++) {
	    mime_t * att = mime_parser_index(ptr->parser_data,i);

	    if (att->magic != MIME_magic)
		mime_panic(__FILE__,__LINE__,"mime_classify_start_page",
			   "Bad magic number");

	    if (att->content_id &&
		same_message_id(Start_ID,att->content_id,
				1 /* Ignore comment*/)) {
		DPRINT(Debug,9,(&Debug, 
				"mime_classify_start_page: start part #%d/%d\n",
				i,count));
		start_part = att;
		break;
	    }
	}
	    
	free_message_id(& Start_ID);
    } else {
	mime_t * att = mime_parser_index(ptr->parser_data,0);

	if (att->magic != MIME_magic)
	    mime_panic(__FILE__,__LINE__,"mime_classify_start_page",
		       "Bad magic number");
	
	DPRINT(Debug,9,(&Debug, 
			"mime_classify_start_page: Using first part as start page\n"));
	start_part = att;
    }

    if (!start_part) {
	DPRINT(Debug,9,(&Debug, 
			"mime_classify_start_page: start part %s not found, type %p %s/%s\n",
			startid ? startid : "(no id)",
			ptr->TYPE,
			get_major_type_name(ptr->TYPE), 
			get_subtype_name(ptr->TYPE)));
	submask = -1;
	goto out;	


    } else if (ret)
	*ret = start_part;

    /* Use only start page for classify */
    
    submask = mime_classify_media(start_part,hdr,
				  /* Check start_page->disposition ? */
				  mime_classify_all);
    if (submask < 0) {
	DPRINT(Debug,9,(&Debug, 
			"mime_classify_start_page: Failed to classify start part %p -- type %p=%s/%s\n",
			start_part,
			start_part->TYPE,
			get_major_type_name(start_part->TYPE), 
			get_subtype_name(start_part->TYPE)));
    }
    
 out:

    DPRINT(Debug,11,(&Debug, 
		     "mime_classify_start_page(%p) = %d (%s)  -- type %p=%s/%s\n",
		     ptr,
		     submask,mime_debug_classify_f(submask),
		     ptr->TYPE,
		     get_major_type_name(ptr->TYPE), 
		     get_subtype_name(ptr->TYPE)));
    
    return submask;
}

/* mime_classify_best_alternative replaces best_alternative() 

   returns mask: 

        NOTPLAIN_need_metamail           0x01
	NOTPLAIN_need_mailcap            0x02
	NOTPLAIN_canuse_mailcap          0x04

	NOTPLAIN_metamail_blacklisted    0x10
*/

static int mask_score P_((int x));
static int mask_score(x) 
     int x;
{
    int r = 0;

    if (0 == x)
	r = 10;
    else {

	r = 4;

	if (x & NOTPLAIN_need_metamail)       r -=4;
	else if (x & NOTPLAIN_is_fallback)    r -=3;
	else if (x & NOTPLAIN_need_mailcap)   r -=2;
	else if (x & NOTPLAIN_canuse_mailcap) r -=1;

    }

    DPRINT(Debug,11,(&Debug,"  Mask %d (%s) score = %d\n",
		     x,mime_debug_classify_f(x),r));
    return r;
}

int mime_classify_best_alternative(ptr, ret,hdr)     
     mime_t *ptr;
     mime_t **ret;
     struct header_rec * hdr;
{
    mime_t * r = NULL;
    
    int i;
    int count;
    

    int  submask        = 0;
    int score           = 0;

    int blacklist       = 0;
    
    if (!ptr->parser_data) {
	submask  = NOTPLAIN_need_metamail;

	DPRINT(Debug,9,(&Debug, 
			"mime_classify_best_alternative: No parser data, type %p %s/%s\n",
			ptr->TYPE,
			get_major_type_name(ptr->TYPE), 
			get_subtype_name(ptr->TYPE)));

	
	goto out;
    }
    count = mime_parser_subparts(ptr->parser_data);
    if (count < 1) {
	submask  = NOTPLAIN_need_metamail;

	DPRINT(Debug,9,(&Debug, 
			"mime_classify_best_alternative: %d parts, type %p %s/%s\n",
			count,
			ptr->TYPE,
			get_major_type_name(ptr->TYPE), 
			get_subtype_name(ptr->TYPE)));

	
	goto out;
    }
    
    r = mime_parser_index(ptr->parser_data,0);
    submask  = NOTPLAIN_need_metamail;   /* DEFAULT -- recalculated later */
    
    for (i = 0; i < count; i++) {
	mime_t * att = mime_parser_index(ptr->parser_data,i);
	int f;
	int s;
	
	if (att->magic != MIME_magic)
	    mime_panic(__FILE__,__LINE__,"mime_classify_best_alternative",
		       "Bad magic number");
	
	/* We will ignore here Content-Disposition: -header, 
	   because it does not make sense inside on multipart/alternative
	*/
	
	f = mime_classify_media(att,hdr,mime_classify_all);
	if (f < 0) {
	    DPRINT(Debug,11,(&Debug, 
			     "mime_classify_best_alternative -- Failed: %p=%s/%s (f=%d)\n",
			     att->TYPE,
			     get_major_type_name(att->TYPE), 
			     get_subtype_name(att->TYPE),
			     f));
	    continue;	
	}
	
	blacklist |=  ( f & NOTPLAIN_metamail_blacklisted );


	s = mask_score(f);
	
	if ( s > score)  {
	    r       = att;
	    submask = f;
	    score   = s;
	    
	    DPRINT(Debug,11,(&Debug, 
			     "-- mime_classify_best_alternative, found: %p=%s/%s  [%d] flags=%d (%s), score=%d\n",
			     att->TYPE,
			     get_major_type_name(att->TYPE), 
			     get_subtype_name(att->TYPE),
			     i,submask,mime_debug_classify_f(submask),
			     score
			     ));
	}
    }
    
    
 out:
    
    submask  |= blacklist;

    DPRINT(Debug,11,(&Debug, 
		     "mime_classify_best_alternative(%p) = %d (%s)",
		     ptr,
		     submask, mime_debug_classify_f(submask)));
    if (r) {
	DPRINT(Debug,11,(&Debug, 
			 ", found=%s/%s",
			 get_major_type_name(r->TYPE), 
			 get_subtype_name(r->TYPE)));
    }
    
    
    DPRINT(Debug,11,(&Debug, "\n"));
    
    if (ret)
	*ret = r;
    return submask;
}

void mime_decode (ptr, state_in, state_out, decode_opt, defcharset, mss, badtype,
		  bodydefcharset)
     mime_t *ptr;
     struct in_state *state_in;
     struct out_state *state_out;
     charset_t defcharset;
     const struct decode_opts *decode_opt;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    int j;
    /* This routine calls the appropriate routine to decode the data
     * described in "ptr".
     */
    
    const charset_t *dV =  get_out_state_charset_vector(state_out);

    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
				     0);

    if (!ptr->TYPE) {
	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeBadCT,
			     "Bad content-type"));
	state_nlputs(" ]\n",state_out);
	
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeBadCT,
			  "Bad content-type"));
	goto out;
	
    }

    DPRINT(Debug,11,(&Debug, 
		     "mime_decode: state: offset=%ld, length=%ld, type=%s/%s, mime_flags=%d (%s), encoding=%d (%s), dispostition=%d (%s) -- ",
		     (long) ptr -> offset, (long) ptr -> length,
		     get_major_type_name(ptr->TYPE), 
		     get_subtype_name(ptr->TYPE),
		     ptr->mime_flags,
		     mime_debug_classify_f(ptr->mime_flags),
		     ptr->encoding,ENCODING(ptr->encoding),
		     ptr->disposition,DISPOSITION(ptr->disposition)
		     ));


    for (j = 0; dV[j]; j++) {
	const char * MIME_name_d UNUSED_VAROK = get_charset_MIME_name(dV[j]);
	
	DPRINT(Debug,11,(&Debug, 
			 "'display charset[%d]'=%s ",j,
			 MIME_name_d ? MIME_name_d : "<no MIME name>"));
    }

    {
	const char * MIME_name UNUSED_VAROK = 
	    get_charset_MIME_name(defcharset);

	DPRINT(Debug,11,(&Debug, 
			 ", header charset='%s'\n",
			 MIME_name ? MIME_name : "<no MIME name>"));
    }    

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"mime_decode",
		   "Bad magic number");
    
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"mime_decode",
		   "mime_decode: unsupported input state");
    }
    
    if (ptr->disposition == DISP_INLINE) {

	if (!ptr->handler_data) {

	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_puts("[ ",state_out);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeNoHandler,
				 "mime_decode: no handler selected"));
	    state_nlputs(" ]\n",state_out);

	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeNoHandler,
			      "mime_decode: no handler selected"));
	    goto out;
	}

	if (MIME_selector_magic != ptr->handler_data->magic)
	    mime_panic(__FILE__,__LINE__,"mime_decode",
		       "Bad magic number (handler_data)");
    
	
	if (in_state_fseek(state_in,ptr->offset) != 0) {
	    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	       conversions */

	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_puts("[ ",state_out);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeMimeSeekFailed,
				 "mime_decode: seek failed"));
	    state_nlputs(" ]\n",state_out);

	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeMimeSeekFailed,
			      "mime_decode: seek failed"));
	    goto out;
	}


	/* Use handler if it is not just fallback routine */
	if (ptr->handler_data->handler &&
	    0 == (ptr->mime_flags & NOTPLAIN_is_fallback)) {

	    DPRINT(Debug,11,(&Debug, 
			     "mime_decode: using builtin handler  \n"));

	    ptr->handler_data->handler->func (ptr, state_in, state_out,  decode_opt,
					      defcharset, mss, badtype, bodydefcharset);    
	} else if (ptr->handler_data->entry) {
	   	    
	    if (ptr->handler_data->use_entry) {
		
		const char * file_name   = NULL; /* reference to STATE_in_decode data */
		
		struct in_state * newstate2 = new_in_state(STATE_in_decode);
		int flags = STATE_DECODE_buffer; 
		int errors = 0;
		
		DPRINT(Debug,11,(&Debug, 
				 "mime_decode: using mailcap handler  \n"));
		
		if (set_in_state_decode_helper(ptr,state_in,newstate2,flags,
					       &file_name)) {
		    
		    int x;
		    int errors UNUSED_VAROK;
		    FILE * file_handle = NULL;

		    struct delay_unlink * file_name_unlink  = NULL; /* refcounted reference to STATE_in_decode data */
		    
		    if (check_type_pattern) {
			if (!check_type_magic(ptr,newstate2, state_out,
					      decode_opt, badtype)) {
			    DPRINT(Debug,11,(&Debug, 
					     "mime_decode: mime_type_rejected\n"));
			    goto FAILTYPE;
			}
		    }
		    
		    /* Print total 10 error messages */
		    errors = print_in_errors(newstate2,state_out,decode_opt);

		    file_handle = file_handle_from_state_decode(newstate2);

		    if (!file_handle)
			goto FAILTYPE;

		    if (mailcap_tempfile_lifetime) {
			/* Increments refcount, possibly also creates struct delay_unlink */ 
			file_name_unlink    = delay_unlink_from_state_decode(newstate2,
									     mailcap_tempfile_lifetime);
		    }
		    
		    x = 
			run_mailcap_view(file_name,file_handle,state_out,
					 ptr->handler_data->entry,ptr,
					 file_name_unlink );    
		    
		    if (!x) {
			DPRINT(Debug,11,(&Debug, 
					 "mime_decode: ... mailcap handler failed\n"));
		    }

		    /* file_handle_from_state_decode dup()ed original handle */
		    fclose(file_handle);

		FAILTYPE:

		    if(file_name_unlink)
			free_delay_unlink(& file_name_unlink);

			
		} else {
		    DPRINT(Debug,11,(&Debug, 
				     "mime_decode: ... decoding failed for mailcap handler\n"));
		}

		/* Print total 10 error messages */
		errors += print_in_errors(newstate2,state_out,decode_opt);

		DPRINT(Debug,11,(&Debug, 
				 "mime_decode: %d errors\n"));

		free_in_state(&newstate2);      	      

	    } else {

		DPRINT(Debug,11,(&Debug, 
				 "mime_decode: ... mailcap handler skipped\n"));

		goto is_null;
	    }

	} else { 
	is_null:

	    if (ptr->handler_data->handler) { 

		DPRINT(Debug,11,(&Debug, 
				 "mime_decode: using builtin (fallback) handler  \n"));

		ptr->handler_data->handler->func (ptr, state_in, state_out, 
						  decode_opt,
						  defcharset, mss, badtype,
						  bodydefcharset);    

	    } else {

		DPRINT(Debug,11,(&Debug, 
				 "mime_decode: using builtin null handler  \n"));
		
		null_decode(ptr, state_in, state_out, decode_opt,
			    defcharset, mss); 
	    }
	}

    } else { 

	if (ptr->disposition == DISP_AUTOATTACH) {

	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeUnsupportedAttach,
				 "[ Type %.15s/%.30s treated as attachment, skipping... ]\n"),
			 get_major_type_name(ptr->TYPE),
			 get_subtype_name(ptr->TYPE));

	    if (decode_opt->displaying) {
		/* \n resets this */
		set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
		state_printf(state_out,
			     CATGETS(elm_msg_cat, MeSet, MeDecodeUseVtosave,
				     "[ Use 'v' to view or save this part. ]\n"));
	    }

	} else {

	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */); 
	    if (decode_opt->displaying)      
		state_printf(state_out,
			     CATGETS(elm_msg_cat, MeSet, MeDecodeAttachUseV,
				     "[ Attachment, skipping...  Use 'v' to view this part. ]\n"));
	    else
		state_printf(state_out,
			     CATGETS(elm_msg_cat, MeSet, MeDecodeAttachSkipping,
				     "[ Attachment, skipping... ]\n"));
	}
    }
    
    DPRINT(Debug,11,(&Debug, 
		     "mime_decode <- END\n"));

 out:
    free_pager_range(&title_range);
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
