static char rcsid[] = "@(#)$Id: write_rc.c,v 2.17 2025/05/29 05:42:57 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.17 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************
 * Some code based on Elm 2.4 src/save_opts.c
 * That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

#include "elm_defs.h"
#include "rc_imp.h"
#include "save_opts.h"
#include "s_elmrc.h"

#include <errno.h> 
#ifndef ANSI_C
extern int errno;
#endif

DEBUG_VAR(Debug,__FILE__,"config");

charset_t rcset;             /* What charset use for writing */
static const char * CHARSETOPT = "@charset";

static enum copy_result {
    copy_res_quit = -1,
    copy_res_next = 0   /* go to next (continue) on main loop              */,
    copy_res_ok         /* buffer have next line                           */,
} copy_part_header P_((FILE       * targetfile_F  ,
		       FILE       * commentfile_F ,
		       int          global        ,
		       char      ** buffer_p      ,
		       int        * len_p         ,
		       int        * skip_p        ,
		       char      ** section_p     ,
		       size_t     * section_len_p));
static enum copy_result copy_part_header(targetfile_F,commentfile_F,global,
					 buffer_p,len_p,skip_p,
					 section_p,section_len_p)
     FILE       * targetfile_F  ;
     FILE       * commentfile_F ;
     int          global        ;
     char      ** buffer_p      ;
     int        * len_p         ;
     int        * skip_p        ;
     char      ** section_p     ;
     size_t     * section_len_p ;
{
    enum copy_result  cr = copy_res_ok;
    
    char *buffer       = * buffer_p;
    int len            = * len_p;
    int skip           = * skip_p;
    char * section     = * section_p;
    size_t section_len = * section_len_p;
    
    if (0 == len) {
	cr = copy_res_next;
	goto done;
    }
    
    if ('#' == buffer[0] || whitespace(buffer[0])) {
	
	if (strncmp(buffer, "#$HDR", 5) == 0) {
	    skip = 0;
	    cr = copy_res_next;
	    goto done;
	}
		
	if (skip) {
	    cr = copy_res_next;
	    goto done;
	}

	if (len > 0 && '#' == buffer[0]) {
	    /* Copy initial comments */
	    fwrite(buffer,1,(size_t)len,targetfile_F);
	}
	putc('\n',targetfile_F);
		
	cr = copy_res_next;
	goto done;
    }
    skip = 1;

    /* trim trailing space */
    while (len > 0 && whitespace(buffer[len-1]))
	buffer[--len] = '\0';
    if (0 == len) {
	cr = copy_res_next;
	goto done;
    }

    /* get section header */ 
    if ('$' == buffer[0]) {

	if (section)
	    free(section);
	section     = NULL;
	section_len = 0;
	
	if (0 == strcmp(buffer,"$user")) {
	    /*  User comment */
	    
	    skip = global;
	    cr   = copy_res_next;
	    goto done;
	} else if (0 == strcmp(buffer,"$global")) {

	    skip = !global;
	    cr   = copy_res_next;
	    goto done;
	} else if (0 == strcmp(buffer,"$usersection")) {

	    if (global) {
		cr   = copy_res_next;
		goto done;
	    }
	    
	} else if (0 == strcmp(buffer,"$globalsection")) {

	    if (!global) {
		cr   = copy_res_next;
		goto done;
	    }
	
	}

	/* malloc_gets do not return \n
	   
	   return -1 if limit exceeded, buffer is still alloced
	   return -2 if error or if feof() is true before reading anything
	*/
	
	while (0  < (len = 
		     malloc_gets(&buffer, MAX_LINE, 
				 commentfile_F))) {
	    size_t L = len;
	    
	    if ('#' != buffer[0])
		break;
	    
	    section = safe_realloc(section,section_len+L+2);

	    memcpy(section+section_len,buffer,L);
	    section_len += L;
	    
	    section[section_len++] = '\n';
	    section[section_len] = '\0';
	}
	if (len < 0) {
	    cr   = copy_res_quit;
	    goto done;
	}
	
	skip = 1;
	cr   = copy_res_next;
    }
    
 done:
    * buffer_p      = buffer;
    * len_p         = len;
    * skip_p        = skip;
    * section_p     = section;
    * section_len_p = section_len;
    
    return cr; 
}

static enum copy_result copy_comment P_((FILE       * targetfile_F  ,
					 FILE       * commentfile_F ,
					 char      ** buffer_p      ,
					 int        * len_p         ,
					 char      ** section_p     ,
					 size_t     * section_len_p));
static enum copy_result copy_comment(targetfile_F,commentfile_F,
				     buffer_p,len_p,
				     section_p,section_len_p)
     FILE       * targetfile_F ;
     FILE       * commentfile_F ;
     char      ** buffer_p      ;
     int        * len_p         ;
     char      ** section_p     ;
     size_t     * section_len_p ;
{
    enum copy_result  cr = copy_res_ok;
    
    char *buffer       = * buffer_p;
    int len            = * len_p;
    char * section     = * section_p;
    size_t section_len = * section_len_p;

    if (section) {
	putc('\n',targetfile_F);
	
	fwrite(section,1,section_len,targetfile_F);
	
	free(section);
	section = NULL;
	
	section_len = 0;
    }

    putc('\n',targetfile_F);
    
    /* Copy comments 
       
       malloc_gets do not return \n
       
       return -1 if limit exceeded, buffer is still alloced
       return -2 if error or if feof() is true before reading anything
    */
    while (0  < (len = 
		 malloc_gets(&buffer,MAX_LINE, 
				     commentfile_F))) {
	if ('#' != buffer[0])
	    break;
		
	fwrite(buffer,1,(size_t)len,targetfile_F);
	putc('\n',targetfile_F);
    }
    if (len < 0) {
	cr =  copy_res_quit;
      
    }
    
    if (copy_res_ok == cr && len > 0) {
	/* trim trailing space */
	while (len > 0 && whitespace(buffer[len-1]))
	    buffer[--len] = '\0';
    }
    
    * buffer_p      = buffer;
    * len_p         = len;
    * section_p     = section;
    * section_len_p = section_len;
     
    return cr; 
}

static void write_option P_((FILE                    * targetfile_F,
			     const char              * part_tag,
			     int                       global,
			     charset_t                 rcset,
			     struct rc_save_info_rec * rc_options,
			     size_t                    rc_option_count,
			     size_t                    opt,
			     int                       have_comment));
static void write_option(targetfile_F,part_tag,global,rcset,
			 rc_options, rc_option_count,opt, have_comment)
     FILE                    * targetfile_F;
     const char              * part_tag;
     int                       global;
     charset_t                 rcset;
     struct rc_save_info_rec * rc_options;
     size_t                    rc_option_count;
     size_t                    opt;
     int                       have_comment;
{
    int local_value;
    
    if (opt >= rc_option_count)
	panic("RC PANIC",__FILE__,__LINE__,"write_option",
	      "Bad opt", 0);
        
    if (global)
	local_value = rc_options[opt].flags & FL_CHANGED;
    else
	local_value = rc_options[opt].flags & FL_LOCAL;

    if (RCTYPE_magic != rc_options[opt].dt_type->magic)
		panic("RC PANIC",__FILE__,__LINE__,"write_rc",
		      "Bad config item type",0);

    DPRINT(Debug,15,(&Debug,
		     "Writing %s option %s ... %s%s\n",
		     part_tag ? part_tag : "main",
		     rc_options[opt].name,
		     local_value ? "(local value)" : "",
		     have_comment ? "" : " (no comment)"));
	    
    rc_options[opt].dt_type->print_value(targetfile_F,
				       & rc_options[opt],
				       !local_value,
					 0 != ( rc_options[opt].flags & 
					      FL_ASCIIFIRST)
				       ? ASCII_SET : rcset);
    
    rc_options[opt].flags |= FL_SAVED;
}

void write_rc_part(targetfile_F,
		   part_tag,commentfile_F,
		   global,rcset,rc_options,rc_option_count)
     FILE       * targetfile_F;
     const char * part_tag          /* NULL for main part */;
     FILE       * commentfile_F     /* NULL if no commentfile */;    
     int          global;
     charset_t    rcset;
     struct rc_save_info_rec * rc_options;
     size_t                    rc_option_count;
{
    size_t x;

    for (x = 0; x < rc_option_count; x++)
	rc_options[x].flags &= ~FL_SAVED;
    
    if (commentfile_F) {
	char *buffer = NULL;
	int len;
	int skip = 1;

	char * section     = NULL;
	size_t section_len = 0;
	
	/* malloc_gets do not return \n

	   return -1 if limit exceeded, buffer is still alloced
	   return -2 if error or if feof() is true before reading anything
	*/
	while (0  <= (len = 
		      malloc_gets(&buffer, MAX_LINE, 
				  commentfile_F))) {
	    
	    int y = 0;
	    enum copy_result  cr;

	redo:
	    /* get top header 
	       get possible section header
	     */
	    switch ((cr = copy_part_header(targetfile_F,
					   commentfile_F,
					   global,
					   & buffer,
					   & len,
					   & skip,
					   & section,
					   & section_len))) {
	    case copy_res_quit: goto quit_copy;
	    case copy_res_next: continue;
	    case copy_res_ok:   /* Nothing */      break;
	    }
		    	    

	    for (x = 0; x < rc_option_count; x++) {
		y = strcmp(buffer, rc_options[x].name);
		if (y <= 0)
		    break;
	    }
	    
	    if (x >= rc_option_count || y != 0) {
		
		DPRINT(Debug,15,(&Debug,
				 "Skipping %s option %s ... \n",
				 part_tag ? part_tag : "main",
				 buffer));
		
		continue;
	    }

	    if (global && 
		! (rc_options[x].flags & FL_CHANGED) &&
		! (rc_options[x].flags & FL_SYS) &&
		! (rc_options[x].flags & FL_SHOWSYS)) {
		
		DPRINT(Debug,15,(&Debug,
				 "Skipping %s user option %s ... \n",
				 part_tag ? part_tag : "main",
				 buffer));
		
		continue;
	    }
	    
	    if (!global &&
		(rc_options[x].flags & FL_SYS)) {
		
		DPRINT(Debug,15,(&Debug,
				 "Skipping %s system option %s ... \n",
				 part_tag ? part_tag : "main",
				 buffer));
		
		continue;
	    }
	    
	    /* Copy comments */
	    switch ((cr = copy_comment(targetfile_F,
				       commentfile_F,
				       & buffer,
				       & len,
				       & section,
				       & section_len))) {
	    case copy_res_quit: goto quit_copy;
	    case copy_res_next: /* Do not print option? */ continue;
	    case copy_res_ok:   /* Nothing */      break;
	    }

	    write_option(targetfile_F,part_tag,global,rcset,
			 rc_options,rc_option_count,x,1);
	    

	    /* Copy extra comments */
	    skip = 0;
	    
	    if (len > 0) {
		/* If non-empty non-comment line readed, do not skip it */
		goto redo;
	    }
	}

    quit_copy:
	if (section)
	    free(section);
	section     = NULL;
	section_len = 0;
	
	if (-1 == len) {
	    DPRINT(Debug,1,(&Debug, 
			    "%s commentile: Too long line: %s\n",
			    part_tag ? part_tag : "Main",			    
			    buffer ? buffer : "<no buffer>"));
	}

	if (ferror(commentfile_F)) {
	    int err UNUSED_VAROK = errno;

	    DPRINT(Debug,1,(&Debug, 
			    "%s commentile: %s\n",
			    part_tag ? part_tag : "Main",
			    strerror(err)));
	}
	
	if (buffer) {
	    free(buffer);
	    buffer = NULL;
	}	
    }


    if (rc_options) {
	
	/* print options without documentation */
	
	for (x = 0; x < rc_option_count; x++) {
	    
	    if (rc_options[x].flags & FL_SAVED)
		continue;
	    
	    /* This is written to top of file */
	    if (0 == strcmp(rc_options[x].name,CHARSETOPT))
		continue;
	    
	    if (global && 
		! (rc_options[x].flags & FL_CHANGED) &&
		! (rc_options[x].flags & FL_SYS) &&
		! (rc_options[x].flags & FL_SHOWSYS)) {
		DPRINT(Debug,15,(&Debug,
				 "Skipping %s user option %s ... \n",
				 part_tag ? part_tag : "main",
				 rc_options[x].name));
		
		continue;
	    }
	    
	    if (!global &&
		(rc_options[x].flags & FL_SYS)) {
		DPRINT(Debug,15,(&Debug,
				 "Skipping system option %s ... \n",
				 rc_options[x].name));
		
		continue;
	    }

	    putc('\n',targetfile_F);

	    write_option(targetfile_F,part_tag,global,rcset,
			 rc_options, rc_option_count,x,0);
	}
    }
}

static void write_cached_option P_((FILE                      * targetfile_F,
				    const char                * part_tag,
				    enum record_mode            lcl,
				    charset_t                   rcset,
				    struct collected_keywords * keywords,
				    size_t                      keywords_len,
				    size_t                      kw,
				    int                         have_comment));
static void write_cached_option(targetfile_F,part_tag,
				lcl,rcset,keywords,keywords_len,
				kw,have_comment)
     FILE                      * targetfile_F;
     const char                * part_tag;
     enum record_mode            lcl;
     charset_t                   rcset;
     struct collected_keywords * keywords;
     size_t                      keywords_len;
     size_t                      kw;
     int                         have_comment;
{
    size_t x;
    int comment_option = 0;
    int unsupported    = 0;
    
    for (x = kw+1; x < keywords_len; x++) {
	if (0 == strcmp(keywords[kw].keyword,
			keywords[x].keyword)) {
	    
	    keywords[kw].seen = 1;
	    
	    if (lcl != keywords[x].lcl) {
		DPRINT(Debug,15,(&Debug,
				 "Skipping %s ",
				 part_tag ? part_tag : "main"));
		
		switch (keywords[x].lcl) {
		case SYSTEM_RC:
		    DPRINT(Debug,15,(&Debug,"system"));
		    break;
		case LOCAL_RC:
		    DPRINT(Debug,15,(&Debug,"user"));
		    break;
		case RC_MODE_COUNT:
		    DPRINT(Debug,15,(&Debug,"(RC_MODE_COUNT)"));
		    break;
		}
		DPRINT(Debug,15,(&Debug,
				 " option %s ... \n",
				 keywords[x].keyword));
		keywords[x].seen = 1;
		
	    } else
		kw = x;	    
	}
    }
    
    if (kw >= keywords_len ||
	lcl != keywords[kw].lcl)
	panic("RC PANIC",__FILE__,__LINE__,"write_cached_option",
	      "Bad kw", 0);

    DPRINT(Debug,15,(&Debug,
		     "writing %s option %s ... %s\n",
		     part_tag ? part_tag : "main",
		     keywords[kw].keyword,
		     have_comment ? "" : " (no comment)"));
    
    comment_option = 0;
    if (keywords[kw].failed)
	comment_option = 1;
    
    for (x = 0; x <  keywords[kw].rest_count; x++) {
	char * buffer2 = NULL;
	int   buffer2_len = 0;
	
	int ok = sconvert_to_rcset(keywords[kw].rest[x],
				   keywords[x].keyword,
				   rcset,&buffer2,&buffer2_len);
	
	if (comment_option || !ok || !buffer2)
	    fprintf(targetfile_F, "### ");
	if (!ok)
	    keywords[kw].failed = 1;
	
	if (!buffer2) {
	    fprintf(targetfile_F, "(not set)\n");
	    continue;
	}
	
	if (x == 0) {
	    if ('!' == keywords[kw].keyword[0]) {
		/* unsupported */
		fprintf(targetfile_F, "# %s = %s\n", keywords[kw].keyword, buffer2);
		unsupported = 1;
	    } else
		fprintf(targetfile_F, "%s = %s\n", keywords[kw].keyword, buffer2);
	} else {
	    if (unsupported)
		putc('#',targetfile_F);
	    
	    fprintf(targetfile_F, "\t%s\n", buffer2);
	}
	free(buffer2);
	buffer2 = NULL;	    		
    }
    
    if (keywords[kw].failed)
	elm_fprintf(targetfile_F, 
		    CATGETS(elm_msg_cat, ElmrcSet, ElmrcFailConvCharset,
			    "# Failed to convert charset of option %s\n"),
		    keywords[kw].keyword);
    
}

void write_rc_part_cached(targetfile_F, part_tag,commentfile_F,global,
			  rcset,keywords,keywords_len)
     FILE       * targetfile_F;
     const char * part_tag;
     FILE       * commentfile_F     /* NULL if no commentfile */;
     int          global;
     charset_t    rcset;
     struct collected_keywords * keywords;
     size_t                      keywords_len;
{
    size_t kw;
    enum record_mode lcl = global ? SYSTEM_RC : LOCAL_RC;

    for (kw = 0; kw < keywords_len; kw++)
	keywords[kw].seen = 0;

    if (commentfile_F) {
	char *buffer = NULL;
	int len;

	int skip = 1;

	char * section     = NULL;
	size_t section_len = 0;
		
	/* malloc_gets do not return \n

	   return -1 if limit exceeded, buffer is still alloced
	   return -2 if error or if feof() is true before reading anything
	*/
	while (0  <= (len = 
		      malloc_gets(&buffer, MAX_LINE, 
				  commentfile_F))) {
	    
	    enum copy_result  cr;

	redo:
	    /* get top header 
	       get possible section header
	     */
	    switch ((cr = copy_part_header(targetfile_F,
					   commentfile_F,
					   global,
					   & buffer,
					   & len,
					   & skip,
					   & section,
					   & section_len))) {
	    case copy_res_quit: goto quit_copy;
	    case copy_res_next: continue;
	    case copy_res_ok:   /* Nothing */      break;
	    }
		    	    
	    for (kw = 0; kw < keywords_len; kw++) {
		if (0 == strcmp(buffer,keywords[kw].keyword)) {
		    keywords[kw].seen = 1;

		    if (lcl != keywords[kw].lcl) {
			DPRINT(Debug,15,(&Debug,
					 "Skipping %s ",
					 part_tag ? part_tag : "main"));
			
			switch (keywords[kw].lcl) {
			case SYSTEM_RC:
			    DPRINT(Debug,15,(&Debug,"system"));
			    break;
			case LOCAL_RC:
			    DPRINT(Debug,15,(&Debug,"user"));
			    break;
			case RC_MODE_COUNT:
			    DPRINT(Debug,15,(&Debug,"(RC_MODE_COUNT)"));
			    break;
			}
			DPRINT(Debug,15,(&Debug,
					 " option %s ... \n",
					 keywords[kw].keyword));
			
		    } else
			goto found_keyword;
		}
	    }
	    
	    DPRINT(Debug,15,(&Debug,
			     "Skipping %s option %s ... \n",
			     part_tag ? part_tag : "main",
			     buffer));
		
	    continue;

	found_keyword:
	    /* Copy comments */
	    switch ((cr = copy_comment(targetfile_F,
				       commentfile_F,
				       & buffer,
				       & len,
				       & section,
				       & section_len))) {
	    case copy_res_quit: goto quit_copy;
	    case copy_res_next: /* Do not print option? */ continue;
	    case copy_res_ok:   /* Nothing */      break;
	    }


	    write_cached_option(targetfile_F,part_tag,lcl,
				rcset,
				keywords,keywords_len,
				kw,1);
					   	    
	    /* Copy extra comments */
	    skip = 0;
	    
	    if (len > 0) {
		/* If non-empty non-comment line readed, do not skip it */
		goto redo;
	    }
	}

    quit_copy:
	if (section)
	    free(section);
	section     = NULL;
	section_len = 0;
	
	if (-1 == len) {
	    DPRINT(Debug,1,(&Debug, 
			    "%s commentile: Too long line: %s\n",
			    part_tag ? part_tag : "Main",			    
			    buffer ? buffer : "<no buffer>"));
	}

	if (ferror(commentfile_F)) {
	    int err UNUSED_VAROK = errno;

	    DPRINT(Debug,1,(&Debug, 
			    "%s commentile: %s\n",
			    part_tag ? part_tag : "Main",
			    strerror(err)));
	}
	
	if (buffer) {
	    free(buffer);
	    buffer = NULL;
	}		
    }

    if (keywords) {

	/* print options without documentation */
	
	for (kw = 0; kw < keywords_len; kw++) {

	    if (!keywords[kw].seen) {

		keywords[kw].seen = 1;
		if (lcl == keywords[kw].lcl) {

		    putc('\n',targetfile_F);
		    
		    write_cached_option(targetfile_F,part_tag,lcl,
					rcset,
					keywords,keywords_len,
					kw,0);
		    
		} else {
		    DPRINT(Debug,15,(&Debug,
				     "Skipping %s ",
				     part_tag ? part_tag : "main"));
		    
		    switch (keywords[kw].lcl) {
		    case SYSTEM_RC:
			DPRINT(Debug,15,(&Debug,"system"));
			break;
		    case LOCAL_RC:
			DPRINT(Debug,15,(&Debug,"user"));
			break;
		    case RC_MODE_COUNT:
			DPRINT(Debug,15,(&Debug,"(RC_MODE_COUNT)"));
			break;
		    }
		    DPRINT(Debug,15,(&Debug,
				     " option %s ... \n",
				     keywords[kw].keyword));
		    
		}		
	    }		
	}
    }
}

int write_rc(targetfile,commentfile,global,actor,username,version_buff)
     char          * targetfile   ; 
     FILE          * commentfile  ; 
     int             global       ;
     const char    * actor        ;
     struct string * username     ;
     char          * version_buff ;
{
    int close_comment_file = 0;
    char *tmp  = NULL;
    int res = 0;
    int errcode;

    const char * sysnam = NULL;
    int default_commentfile UNUSED_VAROK = 0;  /* Use default commentile  */

    FILE *f = NULL;

    enum editor_propline_v propline_mode = 
	give_dt_enumerate_as_int(&editor_elmrc_propline);

    if (!rcset) {
	DPRINT(Debug,10,(&Debug, 
			 "write_rc: rcset is not set, using system charset\n"));

	rcset = system_charset;
    }

    sysnam = get_charset_MIME_name(rcset);

    if (!commentfile) {
	commentfile = fopen(ELMRC_INFO,"r");

	if (!commentfile) {
	    int err = errno;
	
	    if (ENOENT != err) 
		lib_error(CATGETS(elm_msg_cat, ElmrcSet, 
				  ElmrcCommentFileError,
				  "Comment file %s not readable: %s\n"),
			  ELMRC_INFO,strerror(err));

	    lib_error(CATGETS(elm_msg_cat, ElmrcSet, 
			      ElmrcSavingWithoutComments,
			      "Warning: saving without comments! Can't get to %s."),
		      ELMRC_INFO);
	    
	    DPRINT(Debug,8,(&Debug,
			    "no comment file %s: %s\n",
			    ELMRC_INFO,
			    strerror(err)));
	} else 
	    close_comment_file++;

	default_commentfile = 1;
    }


    tmp = elm_message(FRM("%s.N"),targetfile);
    
    errcode = can_open(tmp,"w");

    if (errcode) {
	lib_error(CATGETS(elm_msg_cat, ElmrcSet, ElmrcFileNotWriteable,
			  "File %s is not writeable: %s"),
		  tmp, strerror(errcode));
	
	res = 0;
	goto fail;
    }


    f = fopen(tmp,"w");
    if (!f) {
	int errcode = errno;
	lib_error(CATGETS(elm_msg_cat, ElmrcSet, ElmrcFileNotWriteable,
			  "File %s is not writeable: %s"),
		  tmp, strerror(errcode));
	
	res = 0;
	goto fail;
    }
    
    /* Must have first line */
    if (propline_mode != editor_propline_ignore)
	write_editor_propline(current_editor_propline,propline_mode,
			      f,"# ","\n",rcset);

    if (!global)
	elm_fprintf(f, 
		    CATGETS(elm_msg_cat, ElmrcSet, ElmrcOptionsFile,
			    "#\n# .elm/elmrc - options file for the ELM mail system\n#\n"));

    if (sysnam) {
	/* Store used character set */
	
	elm_fprintf(f, 
		    CATGETS(elm_msg_cat, ElmrcSet, ElmrcCharsetExplain,
			    "# Character set used on this file\n"));

	fprintf(f,"%s = %s\n",CHARSETOPT,sysnam);
    }


    if (username) {
	struct string * S =
	    format_string(CATGETS(elm_msg_cat, ElmrcSet, ElmrcSavedAutoFor1X,
				  "# Saved automatically by %s %s for %S\n#\n\n"),
			  actor, version_buff, username);
	char * buffer = NULL;
	int buffer_len = 0;
	int ok UNUSED_VAROK = 
	    sconvert_to_rcset(S,NULL,rcset,&buffer,&buffer_len);

	fwrite(buffer,1,(size_t)buffer_len,f);

	free(buffer);
	free_string(&S);

    } else
	elm_fprintf(f, 
		    CATGETS(elm_msg_cat, ElmrcSet, ElmrcSavedAutoWrite1,
			    "# Saved automatically by %s %s\n#\n\n"), 
		    actor,version_buff);

    write_rc_part(f,NULL,commentfile,
		  global,rcset,
		  save_info,NUMBER_OF_SAVEABLE_OPTIONS);
			     
#ifdef USE_DLOPEN
    print_local_shared_options(f,global,rcset,
			       default_commentfile);
#endif
    
    if (EOF == fclose(f)) {
	int errcode = errno;
	lib_error(CATGETS(elm_msg_cat, ElmrcSet, ElmrcFileNotWriteable,
			  "File %s is not writeable: %s"),
		  tmp, strerror(errcode));
	
	res = 0;
	goto fail;
    }
    
    if (0 != rename(tmp,targetfile)) {
	int errcode = errno;
	lib_error(CATGETS(elm_msg_cat, ElmrcSet, ElmrcFileNotRenamed,
			  "Failed to rename temporary file to %s: %s"),
		  targetfile, strerror(errcode));
	
	res = 0;
	goto fail;
    }
        
    res = 1;

 fail:
    if (tmp)
	free(tmp);

    if (close_comment_file)
	fclose(commentfile);

    return res;
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

