static char rcsid[] = "@(#)$Id: stringbuffer.c,v 2.13 2023/12/13 16:55:32 hurtta Exp $";

/*************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.13 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ************************************************************************/

#include "elm_defs.h"
#include "s_me.h"
#include "sb_imp.h"
#include "pg_range_imp.h"
#include "pg_lineext_imp.h"

DEBUG_VAR(Debug,__FILE__,"charset");
DEBUG_VAR(DebugIO,__FILE__,"stateio");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

static struct stringbuffer *malloc_buffer P_((struct sb_type *tag));
static struct stringbuffer *malloc_buffer(tag)
     struct sb_type *tag;
{
    struct stringbuffer *ret = safe_zero_alloc(sizeof (*ret));

    if (SB_TYPE_magic != tag->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"malloc_buffer",
	      "Bad magic number (buffer type)",0);

    ret->magic               = STRINGBUFFER_magic;

    ret->refcount            = 1;

    ret->buffer_type         = tag;
    ret->a.dummy             = NULL;

    ret->have_next_lineext   = 0;
    
    if (!ret->buffer_type->sb_init_it(ret)) {
	DPRINT(Debug,68,(&Debug,
			 "-- failed to init membuffer (type=%p)\n",
			 tag));
	ret->buffer_type->sb_free_it(ret);
	ret->buffer_type = NULL;
	ret->magic       = 0; /* Invalidate */
	free(ret);
	return NULL;
    }

    return ret;
}

struct stringbuffer * create_membuffer()
{
    struct stringbuffer *ret = malloc_buffer(&sb_in_mem);
    
    if (!ret) {
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"create_membuffer",
	      "Initialization failure",0);	
    }

    DPRINT(Debug,68,(&Debug,
		     "create_membuffer=%p (type=%p)\n",
		     ret,ret->buffer_type));
    
    return ret;
}

struct stringbuffer * create_filebuffer()
{
    struct stringbuffer *ret = malloc_buffer(&sb_in_file);
    
    if (!ret) {
	int err = errno;
	lib_error(CATGETS(elm_msg_cat, MeSet, MeCantCreaTempFIleUsingMemory,
			  "Sorry, can't create temp file (error: %s) using memory"),
		  strerror(err));
	ret = malloc_buffer(&sb_in_mem);
	if (!ret) 
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"create_membuffer",
		  "Initialization failure",0);	
    }

    DPRINT(Debug,68,(&Debug,
		     "create_membuffer=%p (type=%p)\n",
		     ret,ret->buffer_type));
    
    
    return ret;
}



void free_stringbuffer(ptr)
     struct stringbuffer **ptr;
{
    if (STRINGBUFFER_magic != (*ptr)->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"free_stringbuffer",
	      "Bad magic number (buffer type)",0);

    if (SB_TYPE_magic != (*ptr)->buffer_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"free_stringbuffer",
	      "Bad magic number (buffer type)",0);

    DPRINT(Debug,68,(&Debug,
		     "free_stringbuffer(*ptr=%p) (type=%p)\n",
		     (*ptr),(*ptr)->buffer_type));

    (*ptr)->refcount--;

    if ((*ptr)->refcount > 0) {  /* Do not free */
		
        (*ptr) = NULL;           /* Refefence count for this pointer is     */
	return;			 /* decrement, so this must have be reset   */
	                                 
    }
	    
    (*ptr)->buffer_type->sb_free_it(*ptr);

    (*ptr)->buffer_type = NULL;
    (*ptr)->magic       = 0;     /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}

void inc_stringbuffer_refcount(buffer)
     struct stringbuffer *buffer;
{
    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "inc_stringbuffer_refcount",
	      "Bad magic number (buffer type)",0);
    
    buffer->refcount++;
    
}

void add_line_to_stringbuffer(buffer,string,pg_flags,range,eoln)
     struct stringbuffer *buffer;
     const struct string *string;
     int pg_flags;
     struct pager_range *range; 
     int eoln;
{
    const char * MIME_name                    UNUSED_VAROK =  
	get_string_MIME_name(string);
    int len                                   UNUSED_VAROK = 
	string_len(string);
    const struct  charset_type * charset_type UNUSED_VAROK =
	get_string_charset_type(string);
    struct pgrng_bfr_binding  * binding = NULL;

    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "add_line_to_stringbuffer",
	      "Bad magic number (buffer type)",0);

    if (SB_TYPE_magic != buffer->buffer_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "add_line_to_stringbuffer",
	      "Bad magic number (buffer type)",0);

    DPRINT(Debug,68,(&Debug,
		     "add_line_to_stringbuffer(%p,%p,pg_flags=%d,eoln=%d) (type=%p, string type=('%s'; type=%p)) -- string len = %d%s\n",
		     buffer,string,
		     pg_flags,
		     eoln,
		     buffer->buffer_type,
		     MIME_name ? MIME_name : "<none>",
		     charset_type,len,
		     buffer->have_next_lineext ? ", continuing line" : ", at boln"));
#if 0
    DPRINT(Debug,69,(&Debug," .... string=%S\n",string));
#endif

    if (range) {
	binding = give_pager_range_stringbuffer(range);

	if (!binding)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "add_line_to_stringbuffer",
		  "pager_range not added to stringbuffer",0);

    }

    if (buffer->have_next_lineext) 
	buffer->buffer_type->sb_add_lineext_to_it(buffer,string,pg_flags,range,
						  binding);
    else 
	buffer->buffer_type->sb_add_line_to_it(buffer,string,pg_flags,range,
					       binding);

#ifdef DEBUG
    {   
	int linenum = buffer->buffer_type->sb_linecount_it(buffer) -1;

	DPRINT(DebugIO,12,(&DebugIO,
			   "add_line_to_stringbuffer [%d] %s",
			   linenum,
			   buffer->have_next_lineext ? "[cont]" : ""));

	if (pg_flags) {
	    DPRINT(DebugIO,12,(&DebugIO," pg_flags=%d",
			       pg_flags));
	}

	if (range) {
	    int flags = get_pager_range_flags(range);
	    int ql    = get_pager_range_quote_level(range);

	    DPRINT(DebugIO,12,(&DebugIO,
			       " range %p flags=%d quote level=%d",
			       range,flags,ql));
	} 

	DPRINT(DebugIO,12,(&DebugIO," %d characters %s\n",
			   string_len(string),
			   eoln ? "[eoln]" : "[no eoln]"));
    }
#endif

    
    buffer->have_next_lineext = !eoln;

    if (binding)
	free_pgrng_bfr_binding(&binding);
}

int linecount_stringbuffer(ptr)
     const struct stringbuffer *ptr;
{
    int ret;


    if (STRINGBUFFER_magic != ptr->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "linecount_stringbuffer",
	      "Bad magic number (buffer type)",0);

    if (SB_TYPE_magic != ptr->buffer_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"linecount_stringbuffer",
	      "Bad magic number (buffer type)",0);

    ret=ptr->buffer_type->sb_linecount_it(ptr);

    DPRINT(Debug,68,(&Debug,
		     "linecount_stringbuffer(%p)=%d (type=%p)\n",
		     ptr,ret,ptr->buffer_type));

    return ret;
}

struct pager_param_value * get_pager_param_from_stringbuffer
            (buffer,
	     range,
	     opcode,
	     next_pos,
	     found_pos)
     struct stringbuffer *buffer;
     struct pager_range     * range;
     enum pager_param_opcode  opcode;
     int                    * next_pos /* iterator */;
     int                    * found_pos /* -1 if not found */;
{
    struct pager_param_value *ret = NULL;
    struct pgrng_bfr_binding  * binding = NULL;
    
    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "get_line_from_stringbuffer",
	      "Bad magic number (buffer type)",0);

    if (SB_TYPE_magic != buffer->buffer_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "get_line_from_stringbuffer",
	      "Bad magic number (buffer type)",0);

    if (range) {
	binding = give_pager_range_stringbuffer(range);

	if (!binding)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "get_line_from_stringbuffer",
		  "pager_range not added to stringbuffer",0);

    }

    if (found_pos)
	*found_pos = -1;
    
    
    ret = buffer->buffer_type->sb_get_pager_param_from_it(buffer,range,
							  opcode,next_pos,
							  binding,
							  found_pos);

    if (binding)
	free_pgrng_bfr_binding(&binding);
    						    
    return ret;
}

struct string *get_line_from_stringbuffer(buffer,ptr,pg_flags,
					  range_changed,range,
					  first_extension)
     struct stringbuffer *buffer;
     int ptr;
     int *pg_flags;
     enum range_changed *range_changed;
     struct pager_range **range;  
     struct pager_lineext **first_extension;
{
    struct string *ret;
    int PG_FLAGS = 0;
    struct pager_range * new_range = NULL;

    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "get_line_from_stringbuffer",
	      "Bad magic number (buffer type)",0);

    if (SB_TYPE_magic != buffer->buffer_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "get_line_from_stringbuffer",
	      "Bad magic number (buffer type)",0);

    if (range_changed)
	*range_changed = rc_same_range;

    if (first_extension && *first_extension)
	free_pager_lineext(first_extension);

    ret = buffer->buffer_type->sb_get_line_from_it(buffer,ptr,&PG_FLAGS,
						   range ? &new_range : NULL,
						   first_extension);

    if (range && *range != new_range) {
	if (range_changed) {
	    DPRINT(DebugIO,10,(&DebugIO, 
			       "get_line_from_stringbuffer: range changed"));
	    
	    if (*range) {
		int flags UNUSED_VAROK = get_pager_range_flags(*range);
		int ql    UNUSED_VAROK = get_pager_range_quote_level(*range);
		DPRINT(DebugIO,10,(&DebugIO, 
				   ", old range %p flags=%d quote level=%d",
				   *range,flags,ql));
	    }
	    if (new_range) {
		int flags UNUSED_VAROK = get_pager_range_flags(new_range);
		int ql    UNUSED_VAROK = get_pager_range_quote_level(new_range);
		DPRINT(DebugIO,10,(&DebugIO, 
				   ", new range %p flags=%d quote level=%d",
				   new_range,flags,ql));
	    }
	    	    
	    *range_changed = get_pager_range_change(*range,new_range);
	    switch (*range_changed) {
	    case rc_same_range: DPRINT(DebugIO,10,(&DebugIO,", same range?"));
		break;
	    case rc_newline: DPRINT(DebugIO,10,(&DebugIO,", newline"));
		break;
	    case rc_same_line: DPRINT(DebugIO,10,(&DebugIO,", same line"));
		break;
	    default: DPRINT(DebugIO,10,(&DebugIO,", bad value (%d)",
					*range_changed));
		break;
	    }

	    DPRINT(DebugIO,10,(&DebugIO, "\n"));

	}

	if (*range)
	    free_pager_range(range);
	
	*range = new_range;
	new_range = NULL;
    }

    if (new_range)
	free_pager_range(&new_range);


    {
	const char * MIME_name                    UNUSED_VAROK =  
	    get_string_MIME_name(ret);
	const struct  charset_type * charset_type UNUSED_VAROK =
	    get_string_charset_type(ret);

	DPRINT(Debug,68,(&Debug,
			 "get_line_from_stringbuffer(%p,%d)=%p (type=%p; string type=('%s'; type=%p), pg_flags=%d)\n",
			 buffer,ptr,ret,
			 buffer->buffer_type,
			 MIME_name ? 
			 MIME_name : "<none>",
			 charset_type,
			 PG_FLAGS));
    }

    if (pg_flags)
	*pg_flags = PG_FLAGS;

#if 0
    DPRINT(Debug,69,(&Debug," .... return=%S\n",ret));
#endif

    return ret;
}

struct string *walk_lineext_on_stringbuffer(buffer,walk_extension,pg_flags,range)
     struct stringbuffer  *  buffer;
     struct pager_lineext ** walk_extension;
     int                  *  pg_flags;
     struct pager_range   ** range;
{
    struct string *ret;
    int PG_FLAGS = 0;
    struct pager_range * new_range = NULL;

    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "walk_lineext_on_stringbuffer",
	      "Bad magic number (stringbufer)",0);

    if (SB_TYPE_magic != buffer->buffer_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "walk_lineext_on_stringbuffer",
	      "Bad magic number (buffer type)",0);

    ret = buffer->buffer_type->
	sb_walk_lineext_on_it(buffer,walk_extension,
			      &PG_FLAGS,
			      range ? &new_range : NULL);

    if (range && *range != new_range) {
	if (*range)
	    free_pager_range(range);
	
	*range = new_range;
	new_range = NULL;
    }

    if (new_range)
	free_pager_range(&new_range);


    {
	const char * MIME_name                    UNUSED_VAROK =  
	    get_string_MIME_name(ret);
	const struct  charset_type * charset_type UNUSED_VAROK =
	    get_string_charset_type(ret);

	DPRINT(Debug,68,(&Debug,
			 "walk_lineext_on_stringbuffer(%p,...)=%p (type=%p; string type=('%s'; type=%p), pg_flags=%d)\n",
			 buffer,ret,
			 buffer->buffer_type,
			 MIME_name ? 
			 MIME_name : "<none>",
			 charset_type,
			 PG_FLAGS));
    }

    if (pg_flags)
	*pg_flags = PG_FLAGS;

#if 0
    DPRINT(Debug,69,(&Debug," .... return=%S\n",ret));
#endif

    return ret;    
}

/* Allocates new struct pager_range -- not used */
struct pager_range * stringbuffer_add_simple_pager_range(buffer,inherit,
							 range_flags,
							 quote_level,
							 indent)
     struct stringbuffer *buffer;
     struct pager_range *inherit;
     int range_flags;
     int quote_level;
     int indent;
{
    struct pager_range * ret = NULL;

    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "stringbuffer_add_pager_range",
	      "Bad magic number (buffer type)",0);

    ret = malloc_simple_pager_range(inherit,range_flags,quote_level,
				    indent);

    stringbuffer_add_pager_range(buffer,ret);

    return ret;
}

void stringbuffer_add_pager_range(buffer,range)
     struct stringbuffer *buffer;
     struct pager_range *range;
{
    struct pgrng_bfr_binding *binding = new_pgrng_bfr_binding();
    struct pager_range * inherited, *child;
   
    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "stringbuffer_add_pager_range",
	      "Bad magic number (buffer type)",0);

    set_pager_range_stringbuffer(range,binding);

    child = range;
    inc_pager_range_refcount(child);

    /* Check parents */
    while (NULL != (inherited = give_parent_pager_range(child))) {
	struct pgrng_bfr_binding * bng_inherited =
	    give_pager_range_stringbuffer(inherited);
	struct stringbuffer  * buf_inherited;

	if (! bng_inherited) 
	    stringbuffer_add_pager_range(buffer,inherited);
	if (! bng_inherited) 
	    bng_inherited = give_pager_range_stringbuffer(inherited);
	
	buf_inherited =
	    pgrng_bfr_bnd_get_stringbuffer(bng_inherited,
					   NULL);

	if (! buf_inherited)
	    stringbuffer_add_pager_range_b(buffer,
					   bng_inherited,
					   inherited);

	if (! buf_inherited)
	    buf_inherited = pgrng_bfr_bnd_get_stringbuffer(bng_inherited,
							   NULL);

	if (buf_inherited  != buffer)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "stringbuffer_add_pager_range",
		  "Parent/inherited pager_range added to other stringbuffer",0);
	
	free_stringbuffer(&buf_inherited);

	free_pgrng_bfr_binding(&bng_inherited);
	free_pager_range(&child);
	child = inherited;
    }
    free_pager_range(&child);

    stringbuffer_add_pager_range_b(buffer,binding,range);
 
    free_pgrng_bfr_binding(&binding);
}

void stringbuffer_add_pager_range_b(buffer,binding,range)
     struct stringbuffer *buffer;
     struct pgrng_bfr_binding *binding;
     struct pager_range *range;
{
    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "stringbuffer_add_pager_range_b",
	      "Bad magic number (buffer type)",0);

    if (SB_TYPE_magic != buffer->buffer_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "stringbuffer_add_simple_pager_range_b",
	      "Bad magic number (buffer type)",0);
    
    buffer->buffer_type->sb_add_pager_range_to_it(buffer,binding,range);

}


/* Caller must call free_pager_range() 
   used by get_pager_range_serialization_from_file()
*/
struct pager_range * stringbuffer_get_pager_range(buffer,index)
     struct stringbuffer *buffer; 
     int index; 
{
    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "stringbuffer_get_pager_range",
	      "Bad magic number (buffer type)",0);

    if (SB_TYPE_magic != buffer->buffer_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "stringbuffer_get_pager_range",
	      "Bad magic number (buffer type)",0);

    return buffer->buffer_type->sb_get_pager_range_from_it(buffer,index);

}

enum stringbuffer_entity_result  stringbuffer_add_entity(buffer,oe,pg_flags,
								  range)
     struct stringbuffer *buffer;
     struct out_entity *oe;
     int pg_flags;
     struct pager_range *range; 

{
    struct pgrng_bfr_binding  * binding = NULL;

    enum stringbuffer_entity_result ret =  sb_entity_fallback ;
    
    
    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "stringbuffer_add_entity",
	      "Bad magic number (buffer type)",0);

    if (SB_TYPE_magic != buffer->buffer_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "stringbuffer_add_entity",
	      "Bad magic number (buffer type)",0);

 
    if (range) {
	binding = give_pager_range_stringbuffer(range);

	if (!binding)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "stringbuffer_add_entity",
		  "pager_range not added to stringbuffer",0);

    }

    /* If have_next_lineext is not set, this is beginning of line
       and need add dummy buffer 
    */

    if (! buffer->have_next_lineext) {
	struct string *dummy = new_string(display_charset);

	buffer->buffer_type->sb_add_line_to_it(buffer,dummy,pg_flags,
					       range,
					       binding);

	buffer->have_next_lineext = 1;

	free_string(&dummy);
    }

    /* entity is processed as lineext */
    ret = buffer->buffer_type->sb_add_entity_to_it(buffer,oe,pg_flags,
						   range,
						   binding);

    if (ret > sb_entity_failed) {
	const struct string  *ev =
	    out_entity_text_value(oe);

	int eoln = 0;	
	
        if (ev) {
	    int len;

	    if ((len = string_len(ev)) > 0) {   /* Check LF from end */
		uint16 c = give_unicode_from_string(ev,len-1);
		
		if (0x000A == c)
		    eoln = 1;
	    }
	}

	if ( 0x000A /*   LF  '\n' */ ==
	     out_entity_unicode_value(oe)) {
	    
	    eoln = 1;
	}
    
	buffer->have_next_lineext = !eoln;  /* entity is not end of line? */
    }
	
    if (binding)
	free_pgrng_bfr_binding(&binding);

    return ret;
}

void stringbuffer_add_pager_param(buffer,range,opcode,param_value)
     struct stringbuffer *buffer;
     struct pager_range *range;
     enum pager_param_opcode opcode;
     struct pager_param_value * param_value;
{
    struct pgrng_bfr_binding  * binding = NULL;

    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "stringbuffer_add_pager_param",
	      "Bad magic number (buffer type)",0);
    
    if (SB_TYPE_magic != buffer->buffer_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "stringbuffer_add_pager_param",
	      "Bad magic number (buffer type)",0);

    binding = give_pager_range_stringbuffer(range);
    
    if (!binding)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "stringbuffer_add_pager_param",
	      "pager_range not added to stringbuffer",0);

    
    buffer->buffer_type->sb_add_pager_param_to_it(buffer,binding,
						  range,opcode,
						  param_value);
    
    free_pgrng_bfr_binding(&binding);
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

