static char rcsid[] = "@(#)$Id: string_sort.c,v 2.3 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.3 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"
#include "cs_imp.h"

DEBUG_VAR(Debug,__FILE__,"charset");

#define STRING_SORT_magic	0xF310

struct string_sort {
    unsigned short magic;   /* STRING_SORT_magic */

    int refcount;
    
    struct string *original_string;

    enum unicode_vector_mode     mode;
    struct string_unicode_vector vector;

    unsigned char * stream_bytes;
    int             stream_bytes_len;
    
} * new_string_sort(s)
     struct string *s;
{
    int failures = 0;
    
    struct string_sort * res =
	safe_zero_alloc(sizeof (*res));

    if (STRING_magic != s->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"new_string_sort",
	      "Bad magic number",0);

    if (CS_charset_magic != s->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"new_string_sort",
	    "Bad magic number (string type)",0);

    /* Check lengths -- do compressions and so on */
    s->string_type->charset_type->cs_check_length_it(s);
    
    res->magic = STRING_SORT_magic;
    res->refcount = 1;
    res->original_string = s;
    inc_string_refcount(res->original_string);
    
    res->vector.vector        = NULL;
    res->vector.vector_len    = 0;
    res->vector.first_failure = 0;


    
    res->mode =
	s->string_type->charset_type->cs_unicode_vector_from_it(& (res->vector),
						  s,&failures);

    res->stream_bytes     = NULL;
    res->stream_bytes_len = -1;
    

    switch (res->mode) {

    case unicode_vector_done:

	if (! res->vector.vector) 
	    panic("STRING PANIC",__FILE__,__LINE__,"new_string_sort",
		      "No vector",0);
		
	if (failures > 0) {
	case unicode_vector_fail:

	    /* If there is failures, compare streams --- */

	    res->stream_bytes =		
		s->string_type->charset_type->
		cs_stream_from_it(s,0,NULL,& (res->stream_bytes_len));

	}
	break;
    case unicode_vector_empty:
	break;

    }
    
    return res;
}

void inc_string_sort_refcount(ptr)
     struct string_sort *ptr;
{
    if (STRING_SORT_magic != ptr->magic)
    	panic("STRING PANIC",__FILE__,__LINE__,
	      "inc_string_sort_refcount",
	      "Bad magic number",0);
    
    ptr->refcount++;
}

void free_string_sort(ptr)
     struct string_sort **ptr;
{
    if (STRING_SORT_magic != (*ptr)->magic)
    	panic("STRING PANIC",__FILE__,__LINE__,
	      "free_string_sort",
	      "Bad magic number",0);

    if ((*ptr)->refcount < 1)
    	panic("STRING PANIC",__FILE__,__LINE__,
	      "free_string_sort",
	      "Bad refcount",0);
    
    (*ptr)->refcount--;
    
    if ((*ptr)->refcount > 0) {

	*ptr = NULL;
	return;
    }

    free_string(& ((*ptr)->original_string));

    if ((*ptr)->vector.vector) {
	free((*ptr)->vector.vector);
	(*ptr)->vector.vector = NULL;
    }
    (*ptr)->vector.vector_len = 0;
    
    if ((*ptr)->stream_bytes) {
	free((*ptr)->stream_bytes);
	(*ptr)->stream_bytes = 0;
    }
    (*ptr)->stream_bytes_len = -1;


    free(*ptr);
    *ptr = NULL;

}

int string_sort_cmp(ptr1,ptr2)
     const struct string_sort *ptr1;
     const struct string_sort *ptr2;
{    
    if (STRING_SORT_magic != ptr1->magic)
    	panic("STRING PANIC",__FILE__,__LINE__,
	      "string_sort_cmp",
	      "Bad magic number (ptr1)",0);
    if (STRING_SORT_magic != ptr2->magic)
    	panic("STRING PANIC",__FILE__,__LINE__,
	      "string_sort_cmp",
	      "Bad magic number (ptr2)",0);
    
    /* top level sorting is:
       - no unicode
       - empty string
       - unicode string (also with strings which start with known unicode)
    */

    if (ptr1->mode < ptr2->mode)
	return -1;
    if (ptr1->mode > ptr2->mode)
	return 1;

    if (ptr1->mode != ptr2->mode)
    	panic("STRING PANIC",__FILE__,__LINE__,
	      "string_sort_cmp",
	      "Mode differ between ptr1 and ptr2",0);
    
    switch (ptr1->mode) {
	size_t max_len;
	size_t i;

    case unicode_vector_empty:
	/* Both strings are empty */
	return 0;
	
    case unicode_vector_done:
	/* Use first unicode */

	max_len = ptr1->vector.vector_len;
	if (max_len > ptr2->vector.vector_len)
	    max_len = ptr2->vector.vector_len;
	if (max_len > ptr1->vector.first_failure)
	    max_len = ptr1->vector.first_failure;
	if (max_len > ptr2->vector.first_failure)
	    max_len = ptr2->vector.first_failure;
	
	for (i = 0; i < max_len; i++) {

	    if (ptr1->vector.vector[i] < ptr2->vector.vector[i])
		return -1;
	    if (ptr1->vector.vector[i] > ptr2->vector.vector[i])
		return 1;	    	    
	}

	if (i == ptr1->vector.vector_len &&
	    i == ptr2->vector.vector_len)
	    return 0;

	if (i == ptr1->vector.vector_len &&
	    i < ptr2->vector.vector_len)
	    return -1;

	if (i < ptr1->vector.vector_len &&
	    i == ptr2->vector.vector_len)
	    return 1;

	if (i == ptr1->vector.first_failure &&
	    i < ptr2->vector.first_failure)
	    return -1;

	if (i < ptr1->vector.first_failure &&
	    i == ptr2->vector.first_failure)
	    return 1;

	/* FALLTRU */
    case digest_feed_fallback_stream:

	if (ptr1->stream_bytes_len < 0 &&
	    ptr2->stream_bytes_len >= 0)
	    return -1;

	if (ptr1->stream_bytes_len >= 0 &&
	    ptr2->stream_bytes_len < 0)
	    return 1;

	if (! ptr1->stream_bytes &&
	    ptr2->stream_bytes)
	    return -1;

	if (ptr1->stream_bytes &&
	    ! ptr2->stream_bytes)
	    return 1;

	if (ptr1->stream_bytes &&
	    ptr2->stream_bytes &&
	    ptr1->stream_bytes_len >= 0 &&
	    ptr2->stream_bytes_len >= 0) {

	    int max_bytes_len;
	    int bi;
	    
	    max_bytes_len = ptr1->stream_bytes_len;
	    if (max_bytes_len > ptr2->stream_bytes_len)
		max_bytes_len = ptr2->stream_bytes_len;
	    
	    for (bi = 0; bi < max_bytes_len; bi++) {
		if (ptr1->stream_bytes[bi] < ptr2->stream_bytes[bi])
		    return -1;
		if (ptr1->stream_bytes[bi] > ptr2->stream_bytes[bi])
		    return -1;		
	    }

	    if (bi == ptr1->stream_bytes_len &&
		bi < ptr2->stream_bytes_len)
		return -1;
	    if (bi < ptr1->stream_bytes_len &&
		bi == ptr2->stream_bytes_len)
		return 1;

	    if (bi == ptr1->stream_bytes_len &&
		bi == ptr2->stream_bytes_len) {
		
		if (STRING_magic != ptr1->original_string->magic)
		    panic("STRING PANIC",__FILE__,__LINE__,"string_sort_cmp",
			  "Bad magic number (ptr1, original_string)",0);
		if (STRING_magic != ptr2->original_string->magic)
		    panic("STRING PANIC",__FILE__,__LINE__,"string_sort_cmp",
			  "Bad magic number (ptr2, original_string)",0);
		
		if (CS_charset_magic != ptr1->original_string->string_type->magic) 
		    panic("STRING PANIC",__FILE__,__LINE__,"string_sort_cmp",
			  "Bad magic number (ptr1,string type)",0);
		if (CS_charset_magic != ptr2->original_string->string_type->magic) 
		    panic("STRING PANIC",__FILE__,__LINE__,"string_sort_cmp",
			  "Bad magic number (ptr,string type)",0);

		/* If bytes are same and string types are same,
		   then strings are same 
		*/
		
		if (ptr1->original_string->string_type ==
		    ptr2->original_string->string_type)
		    return 0;

		/* If bytes are same and MIME_name is same, 
		   then assume that strings are same 
		*/
		
		
		if (! ptr1->original_string->string_type->MIME_name &&
		    ptr2->original_string->string_type->MIME_name)
		    return -1;
		
		if (ptr1->original_string->string_type->MIME_name &&
		    ! ptr2->original_string->string_type->MIME_name)
		    return 1;
	    
	   
		if (ptr1->original_string->string_type->MIME_name &&
		    ptr2->original_string->string_type->MIME_name) {
		    
		    return strcmp(ptr1->original_string->string_type->MIME_name,
				  ptr2->original_string->string_type->MIME_name);
		}

		/* If bytes are same and codeset is same, 
		   then assume that strings are same 
		*/

		if (! ptr1->original_string->string_type->codeset &&
		    ptr2->original_string->string_type->codeset)
		    return -1;
		
		if (ptr1->original_string->string_type->codeset &&
		    ! ptr2->original_string->string_type->codeset)
		    return 1;
	    
	   
		if (ptr1->original_string->string_type->codeset &&
		    ptr2->original_string->string_type->codeset) {
		    
		    return strcmp(ptr1->original_string->string_type->codeset,
				  ptr2->original_string->string_type->codeset);
		}		
	    }	    
	}

	break;
    }

    /* Otherwise just compare original pointers */

    if (ptr1->original_string <  ptr2->original_string)
	return -1;
    if (ptr1->original_string >  ptr2->original_string)
	return 1;
	 
    /* In here this should be same string */
    
    return 0;
}


/* Increments refcount, caller must call free_string() */
struct string * give_string_from_string_sort(ptr)
     struct string_sort *ptr;
{
    struct string * ret;
    
    if (STRING_SORT_magic != ptr->magic)
    	panic("STRING PANIC",__FILE__,__LINE__,
	      "give_string_from_string_sort",
	      "Bad magic number",0);

    ret = ptr->original_string;

    inc_string_refcount(ret);

    return ret;
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
