static char rcsid[] = "@(#)$Id: schedule.c,v 2.33 2022/07/14 14:16:00 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.33 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"
#include "schedule_time.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"net");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

volatile int wait_can_signal = 0;

enum action_status no_action_routine(fd,data,now)
     int fd; 
     union action_routine_data         data;
     const struct schedule_timelimit * now;
{
    return action_disable;
}

static enum action_status ANY_ACTION P_((int fd,
					 union action_routine_data data,
					 const struct schedule_timelimit *now))
    UNUSED_FUNCOK;
static enum action_status ANY_ACTION(fd,data,now)
     int                               fd; 
     union action_routine_data         data;
     const struct schedule_timelimit * now;
{
    return action_disable;
}

void no_free_action_data(data,fd,badpid)
     union action_routine_data * data;
     int fd;
     int badpid;
{
    data->data = NULL;
}

void no_inc_action_data_refcount(data)
     union action_routine_data data;
{

}

enum schedule_return no_schedule_routine(fd,data,now)
     int                         fd; 
     union schedule_routine_data data;
     const struct schedule_timelimit * now;
{
    return schedule_done;
}

void no_free_schedule_data(data,fd,badpid)
     union schedule_routine_data * data;
     int fd;
     int badpid;
{
    data->data = NULL;
}

void no_inc_schedule_data_refcount(data)
     union schedule_routine_data data;
{

}

enum badpid_status no_badpid_action(fd,data,badpid)
     int fd;
     union action_routine_data         data;
     int badpid;
{
    return badpid_disable;
}

enum badpid_status badpid_remove_action(fd,data,badpid)
     int fd;
     union action_routine_data         data;
     int badpid;
{
    return badpid_remove;
}

static struct actions {
    
    int    fd; 
    int    timeout_sec;
    int                          caller_pid;
    
    action_routine             * read_act; 
    action_routine             * write_act; 		       
    action_routine             * timeout_act;
    free_action_data_f         * free_action_data;
    inc_action_data_refcount_f * inc_action_data_refcount;
    badpid_action_f            * badpid_act;
    union action_routine_data    data;

    struct schedule_timelimit next_timeout;
    unsigned int busy  :1;           /* Prevent nested calls --
				      * when read_act(), write_act() or
				      * timeout_act() calls wait_for_timeout()
				      * or wait_for_action() directly or
				      * indirectly (for example with lib_error)
				      */
    
    unsigned int badfd : 1;          /* FD is not valid */
    

    schedule_routine             * schedule_act;
    free_schedule_data_f         * free_schedule_data;
    inc_schedule_data_refcount_f * inc_schedule_data_refcount; 
    union schedule_routine_data    schedule_data;

}         * actions       = NULL;
static int  actions_count = 0; 

static void zero_actions_item  P_((struct actions *item));
static void zero_actions_item(item)
     struct actions *item;
{
    item->fd                = -1;
    item->timeout_sec       = 0;
    item->caller_pid        = getpid();
    item->next_timeout      = NO_schedule_timelimit;
    item->busy              = 0;
    item->badfd             = 1;
    item->timeout_sec       = 0;
    item->read_act          = no_action_routine;
    item->write_act         = no_action_routine;
    item->timeout_act       = no_action_routine;
    item->free_action_data         = no_free_action_data;
    item->inc_action_data_refcount = no_inc_action_data_refcount;
    item->badpid_act               = badpid_remove_action;
    item->data                     = NO_action_routine_data;
    
    item->schedule_act               = no_schedule_routine;
    item->free_schedule_data         = no_free_schedule_data;
    item->inc_schedule_data_refcount = no_inc_schedule_data_refcount;
    item->schedule_data = NO_schedule_routine_data;	
}

static void remove_action0 P_((int ptr,
			       int badpid));
static void remove_action0(ptr,badpid)
     int ptr;
     int badpid;
{
    int i;
    
    if (ptr < 0 || ptr >=  actions_count)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "remove_action0",
	      "Bad index",0);
    
    DPRINT(Debug,9,(&Debug,
		    "remove_action0: action %d fd=%d removing (badpid=%d)\n",
		    ptr, actions[ptr].fd,badpid));

    if (actions[ptr].data.data) {
	DPRINT(Debug,12,(&Debug,
			 "remove_action0: action %d fd=%d freeing basic data %p\n",
			 ptr, actions[ptr].fd,actions[ptr].data.data));
	 actions[ptr].free_action_data(& (actions[ptr].data),
				       actions[ptr].fd,
				       badpid);
    }

    if (actions[ptr].schedule_data.data) {
	DPRINT(Debug,9,(&Debug,
			"remove_action0: action %d fd=%d freeing schedule data %p\n",
			ptr,actions[ptr].fd,
			actions[ptr].schedule_data.data));
	
	actions[ptr].free_schedule_data(& (actions[ptr].schedule_data),
					actions[ptr].fd,
					badpid);
    }
    
    zero_actions_item(&(actions[ptr]));
    
    for (i = ptr+1; i < actions_count; i++) {
	actions[i-1] = actions[i];
	zero_actions_item(& (actions[i]));
    }
    actions_count--;
}

static void remove_action P_((int ptr));
static void remove_action(ptr)
     int ptr;
{
    int p = getpid();
    int badpid = 0;
    
    if (ptr < 0 || ptr >=  actions_count)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "remove_action",
	      "Bad index",0);
    
    DPRINT(Debug,4,(&Debug,
		    "** Removing action %d  (fd=%d)", ptr,
		    actions[ptr].fd));

    if (actions[ptr].caller_pid != p) {
	DPRINT(Debug,4,(&Debug," -- caller pid %d is not our (%d)",
			actions[ptr].caller_pid,p));
	badpid = actions[ptr].caller_pid;
    }
    
    DPRINT(Debug,4,(&Debug,"\n"));

    remove_action0(ptr,badpid);
}

static void schedule_remove_idx P_((int ptr));


void clear_other_actions(mypid)
     int mypid;
{
    int ptr;
    int have_busy = 0;
    
    for (ptr = 0; ptr < actions_count; ptr++) {
	if (actions[ptr].busy) {
	    DPRINT(Debug,12,(&Debug,
			     "clear_other_actions: action %d (fd=%d) is BUSY\n",
			     ptr,actions[ptr].fd));
	    have_busy = 1;
	}
    }

    for (ptr = actions_count; ptr > 0;) {
	ptr--;

	DPRINT(Debug,12,(&Debug,
			 "clear_other_actions: action %d (fd=%d) caller pid %d",
			 ptr,actions[ptr].fd,actions[ptr].caller_pid));
	if (actions[ptr].caller_pid == mypid) {
	    DPRINT(Debug,12,(&Debug,"  OK\n"));
	} else {
	    enum badpid_status val;
	    int was_busy = actions[ptr].busy;
	    int badpid = actions[ptr].caller_pid;
	    
	    DPRINT(Debug,12,(&Debug," NOT OUR (%d)\n",
			     mypid));

	    DPRINT(Debug,20,(&Debug,
			     "clear_other_actions: running badpid_act for %d (action %d)\n",
			     actions[ptr].fd,ptr));

	    actions[ptr].busy = 1;

	    /* XXXXX Process even when busy !!!! */
	    
	    val = actions[ptr].badpid_act(actions[ptr].fd,
					  actions[ptr].data,
					  badpid);

	    actions[ptr].busy = was_busy;
	    
	    DPRINT(Debug,20,(&Debug,
			     "clear_other_actions: [%d].badpid_act=%d",ptr,val));
	    switch (val) {
	    case badpid_disable:
		actions[ptr].badpid_act = no_badpid_action;
		DPRINT(Debug,20,(&Debug," (badpid_disable)"));
		break;
	    case badpid_ignore:
		DPRINT(Debug,20,(&Debug," (badpid_ignore)"));
		break;
	    case badpid_remove:
		DPRINT(Debug,20,(&Debug," (badpid_remove)"));
		break;
	    case badpid_change:		
		actions[ptr].caller_pid = mypid;
		DPRINT(Debug,20,(&Debug," (badpid_change)"));
		break;
	    }
	    DPRINT(Debug,20,(&Debug,"\n"));

	    if (badpid_remove == val) {
		if (have_busy)
		    schedule_remove_idx(ptr);
		else
		    remove_action0(ptr,badpid);
	    }
	}	    
    }    
}

const union action_routine_data   NO_action_routine_data;
const union schedule_routine_data NO_schedule_routine_data;

struct action_setups {
    action_routine               * read_act;
    action_routine               * write_act;
    action_routine               * timeout_act;
    int                            timeout_sec;
    free_action_data_f           * free_action_data;
    inc_action_data_refcount_f   * inc_action_data_refcount;
    badpid_action_f              * badpid_act;
    union action_routine_data      basic_data;
};

struct schedule_setup {
    schedule_routine             * schedule_rut;
    free_schedule_data_f         * free_schedule_data;
    inc_schedule_data_refcount_f * inc_schedule_data_refcount;
    union schedule_routine_data    schedule_data;
};

void setup_actions0(fd,is_badpid,basic_setup,sched_setup)
     int                            fd;
     int                            is_badpid;
     const struct action_setups   * basic_setup;
     const struct schedule_setup  * sched_setup;
{
    int ptr;
    int badpid    = 0;
    int is_remove = 0;
    
    struct action_setups  BASIC;
    struct schedule_setup SCHED;

    struct stat  ST;
    int r;

    static int report_time_MAX = 1;

    if (report_time_MAX) {
	char * X = ctime(&time_MAX);
	
	report_time_MAX = 0;
	
	if (X) {
	    DPRINT(Debug,1,(&Debug,
			    "Using time_MAX = %ld  = %s",
			    (long)time_MAX,X));
	} else {
	    DPRINT(Debug,1,(&Debug,
			    "Using time_MAX = %ld\n",
			    (long)time_MAX));
	}
	DPRINT(Debug,1,(&Debug,"\ntime_t size is %ld bytes\n",
			(long)sizeof(time_t)));
    }
    
    for (ptr = 0; ptr < actions_count; ptr++)
	if (actions[ptr].fd == fd)
	    break;
    
    if (ptr == actions_count) {	

	if (basic_setup ||
	    sched_setup) {

	    if (basic_setup && 
		no_action_routine == basic_setup->read_act    &&
		no_action_routine == basic_setup->write_act   &&
		no_action_routine == basic_setup->timeout_act &&
		sched_setup &&
		no_schedule_routine == sched_setup->schedule_rut) {
		DPRINT(Debug,9,(&Debug,
				"setup_actions0: routines for fd=%d already cleared -- actions_count=%d\n",
				fd,actions_count));
		return;
	    }
	} else {
	    DPRINT(Debug,9,(&Debug,
			    "setup_actions0: Nothing setup for fd=%d -- not found; actions_count=%d\n",
			    fd,actions_count));
	    return;
	}

	actions = safe_array_realloc(actions, (++actions_count),
				     sizeof (struct actions));

	zero_actions_item(& (actions[ptr]));

	DPRINT(Debug,4,(&Debug,
			"setup_actions0: ** Adding action %d for fd=%d; actions_count=%d\n",
			ptr,fd,actions_count));

	actions[ptr].fd          = fd;
    }

    if (!basic_setup) {
	DPRINT(Debug,9,(&Debug,
			"setup_actions0: action %d fd=%d, no basic_setup -- using current values.\n",
			ptr,fd));

	/* bzero is defined on hdrs/defs.h */
	bzero((void *)&BASIC, sizeof BASIC);

	BASIC.read_act                    = actions[ptr].read_act;
	BASIC.write_act                   = actions[ptr].write_act;
	BASIC.timeout_act                 = actions[ptr].timeout_act;
	BASIC.timeout_sec                 = actions[ptr].timeout_sec;
	BASIC.free_action_data            = actions[ptr].free_action_data;
	BASIC.inc_action_data_refcount    = actions[ptr].inc_action_data_refcount;
	BASIC.badpid_act                  = actions[ptr].badpid_act;
	BASIC.basic_data                  = actions[ptr].data;

	basic_setup = &BASIC;
    }

    if (!sched_setup) {
	DPRINT(Debug,9,(&Debug,
			"setup_actions0: action %d fd=%d, no sched_setup -- using current values.\n",
			ptr,fd));

	/* bzero is defined on hdrs/defs.h */
	bzero((void *)&SCHED, sizeof SCHED);

	SCHED.schedule_rut                = actions[ptr].schedule_act;
	SCHED.free_schedule_data          = actions[ptr].free_schedule_data;
	SCHED.inc_schedule_data_refcount  = actions[ptr].inc_schedule_data_refcount;
	SCHED.schedule_data               = actions[ptr].schedule_data;

	sched_setup = &SCHED;	
    }

    if (no_action_routine   == basic_setup->read_act &&
	no_action_routine   == basic_setup->write_act &&
	no_action_routine   == basic_setup->timeout_act &&
	no_schedule_routine == sched_setup->schedule_rut) {

	DPRINT(Debug,9,(&Debug,
			"setup_actions0: action %d fd=%d --  is remove request\n",
			ptr,fd));

	is_remove = 1;
    }

    if (is_badpid) {
	DPRINT(Debug,9,(&Debug,
			"setup_actions0: action %d fd=%d -- assuming badpid %d request\n",
			ptr,fd,is_badpid));
	badpid = is_badpid;
    } else {
	int p = getpid();

	if (actions[ptr].caller_pid != p) {
	    if (is_remove) {
		DPRINT(Debug,9,(&Debug,
				"setup_actions0: action %d fd=%d -- caller pid %d is not our (%d)\n",
				ptr,fd,actions[ptr].caller_pid,p));
		
		badpid = actions[ptr].caller_pid;
	    } else {
		DPRINT(Debug,9,(&Debug,
				"setup_actions0: action %d fd=%d -- changing caller pid %d => %d\n",
				ptr,fd,actions[ptr].caller_pid,p));
		
		actions[ptr].caller_pid = p;	    
	    }
	} else {
	    DPRINT(Debug,18,(&Debug,
			     "setup_actions0: action %d fd=%d -- not badpid request -- caller pid %d is our\n",
			     ptr,fd,actions[ptr].caller_pid));
	}
    }

    if (actions[ptr].timeout_sec != basic_setup->timeout_sec ||
	! schedule_have_timelimit(& (actions[ptr].next_timeout))
	) {
	char * X;

	if (no_action_routine == basic_setup->timeout_act &&
	    -1 ==  basic_setup->timeout_sec ) {
	    
	    /* Not actually needed, because timeout is considered only
	       when no_action_routine != timeout_act */
	    actions[ptr].next_timeout = NO_schedule_timelimit;

	    DPRINT(Debug,9,(&Debug,
			    "setup_actions0: action %d fd=%d -- no timeout (no action)\n",
			    ptr,fd));
	    
	} else if (! schedule_set_next_timeout(& (actions[ptr].next_timeout),
					       basic_setup->timeout_sec)) {

	    DPRINT(Debug,9,(&Debug,
			    "setup_actions0: action %d fd=%d -- time overflow or no current time\n",
			    ptr,fd));
	}

	X =  schedule_timeout_string(& actions[ptr].next_timeout);

	DPRINT(Debug,9,(&Debug,
			"setup_actions0: action %d fd=%d",
			ptr,fd));
	if (-1 !=  basic_setup->timeout_sec || no_action_routine != basic_setup->timeout_act ) {
	    DPRINT(Debug,9,(&Debug," timeout %d seconds",
			    basic_setup->timeout_sec));
	}
	if (no_action_routine == basic_setup->timeout_act) {
	    DPRINT(Debug,9,(&Debug, ", no timeout (no action)"));
	}
	if (X) {
	    DPRINT(Debug,9,(&Debug, ", timeout => %s",
			    X));
	    free(X);
	    X = NULL;
	}
	DPRINT(Debug,9,(&Debug,"\n"));
    } else if (-1 !=  basic_setup->timeout_sec ||
	       schedule_have_timelimit(& (actions[ptr].next_timeout))) {
	char * X;
	
	DPRINT(Debug,18,(&Debug,
			 "setup_actions0: action %d fd=%d -- timeout %d is not changed",
			 ptr,fd,basic_setup->timeout_sec));
	X =  schedule_timeout_string(& actions[ptr].next_timeout);
	if (X) {
	    DPRINT(Debug,18,(&Debug, ", timeout => %s",
			    X));
	    free(X);
	    X = NULL;
	}
	DPRINT(Debug,18,(&Debug,"\n"));
    }

    if (actions[ptr].data.data != basic_setup->basic_data.data) {
	if (actions[ptr].data.data) {
	    DPRINT(Debug,9,(&Debug,
			    "setup_actions0: action %d fd=%d  freeing previous basic data %p\n",
			    ptr,fd,actions[ptr].data.data));
	    
	    actions[ptr].free_action_data(& (actions[ptr].data),
					  fd,badpid);
	}
	
	actions[ptr].data        = basic_setup->basic_data;	
	if (actions[ptr].data.data) {
	    DPRINT(Debug,9,(&Debug,
			    "setup_actions0: action %d fd=%d  adding new basic data %p\n",
			    ptr,fd,actions[ptr].data.data));

	    basic_setup->inc_action_data_refcount(actions[ptr].data);
	}
    } else if (actions[ptr].data.data) {
	DPRINT(Debug,18,(&Debug,
			"setup_actions0: action %d fd=%d basic data %p is not changed\n",
			 ptr,fd,actions[ptr].data.data));
    }

    if (actions[ptr].schedule_data.data != sched_setup->schedule_data.data) {
	if (actions[ptr].schedule_data.data) {
	    DPRINT(Debug,9,(&Debug,
			    "setup_actions0: action %d fd=%d  freeing previous schedule data %p\n",
			    ptr,fd,actions[ptr].schedule_data.data));
	    actions[ptr].free_schedule_data(& (actions[ptr].schedule_data),
					    fd,badpid);
	}
	
	actions[ptr].schedule_data =  sched_setup->schedule_data;
	if (actions[ptr].schedule_data.data) {
	    DPRINT(Debug,9,(&Debug,
			     "setup_actions0: action %d fd=%d  adding new schedule data %p\n",
			    ptr,fd,actions[ptr].schedule_data.data));
	    sched_setup->inc_schedule_data_refcount(actions[ptr].schedule_data);
	}
    } else if (actions[ptr].schedule_data.data) {
	DPRINT(Debug,18,(&Debug,
			 "setup_actions0: action %d fd=%d schedule data %p is not changed\n",
			 ptr,fd,actions[ptr].schedule_data.data));
    }
    
    if (actions[ptr].fd != fd)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "setup_actions0",
	      "Bad fd",0);
    
    r = fstat(fd,&ST);
    if (-1 == r) {
	int err UNUSED_VAROK = errno;
	
	DPRINT(Debug,2,(&Debug,
			"setup_actions0: action %d fd=%d: stat errno=%d (%s)\n",
			ptr,fd,err,strerror(err)));
	
	if (errno == EBADF) {
	    actions[ptr].badfd = 1;
	    DPRINT(Debug,2,(&Debug,
			    "setup_actions0: action %d fd=%d: setting badfd\n",
			    ptr,fd));
	}
	
    } else if (0 == r) {
	if (actions[ptr].badfd && !badpid) {
	    DPRINT(Debug,2,(&Debug,
			    "setup_actions0: action %d fd=%d: stat OK, clearing badfd\n",
			    ptr,fd));
	    actions[ptr].badfd = 0;
	} else {
	    DPRINT(Debug,18,(&Debug,
			     "setup_actions0: action %d fd=%d: stat OK\n",
			     ptr,fd));
	}
    }
    
    actions[ptr].timeout_sec                = basic_setup->timeout_sec;
    actions[ptr].read_act                   = basic_setup->read_act;
    actions[ptr].write_act                  = basic_setup->write_act;
    actions[ptr].timeout_act                = basic_setup->timeout_act;
    actions[ptr].free_action_data           = basic_setup->free_action_data;
    actions[ptr].inc_action_data_refcount   = basic_setup->inc_action_data_refcount;
    actions[ptr].badpid_act                 = basic_setup->badpid_act;
    
    actions[ptr].schedule_act               = sched_setup->schedule_rut;
    actions[ptr].free_schedule_data         = sched_setup->free_schedule_data;
    actions[ptr].inc_schedule_data_refcount = sched_setup->inc_schedule_data_refcount;
    
    if (is_remove) {
	
	if (actions[ptr].busy) {
	    DPRINT(Debug,9,(&Debug,
			    "setup_actions0: action %d fd=%d -- busy: remove skipped\n", 
			    ptr, actions[ptr].fd));
	} else
	    remove_action0(ptr,badpid);
    }	     
}



void change_action (fd,timeout_sec,read_act,write_act,timeout_act,
                    free_action_data,inc_action_data_refcount,
                    data)
     int                       fd; 
     int                       timeout_sec;
     action_routine          * read_act; 
     action_routine          * write_act;       
     action_routine          * timeout_act;
     free_action_data_f      * free_action_data;
     inc_action_data_refcount_f * inc_action_data_refcount;
     union action_routine_data data;
{
    if (no_action_routine == read_act &&
	no_action_routine == write_act &&
	no_action_routine == timeout_act &&
	no_inc_action_data_refcount == inc_action_data_refcount
	)
	change_action2 (fd,timeout_sec,read_act,write_act,timeout_act,
			free_action_data,inc_action_data_refcount,
			no_badpid_action,
			data);
    else
	change_action2 (fd,timeout_sec,read_act,write_act,timeout_act,
			free_action_data,inc_action_data_refcount,
			badpid_remove_action,
			data);
}


void change_action2 (fd,timeout_sec,read_act,write_act,timeout_act,
		     free_action_data,inc_action_data_refcount,
		     badpid_act,
		     data)
     int                       fd; 
     int                       timeout_sec;
     action_routine          * read_act; 
     action_routine          * write_act; 	
     action_routine          * timeout_act;
     free_action_data_f      * free_action_data;
     inc_action_data_refcount_f * inc_action_data_refcount;
     badpid_action_f            * badpid_act;
     union action_routine_data data;
{
    struct action_setups  BASIC;

    /* bzero is defined on hdrs/defs.h */
    bzero((void *)&BASIC, sizeof BASIC);
    
    BASIC.read_act                    = read_act;
    BASIC.write_act                   = write_act;
    BASIC.timeout_act                 = timeout_act;
    BASIC.timeout_sec                 = timeout_sec;
    BASIC.free_action_data            = free_action_data;
    BASIC.inc_action_data_refcount    = inc_action_data_refcount;
    BASIC.badpid_act                  = badpid_act;
    BASIC.basic_data                  = data;

    setup_actions0(fd,
		   0 /* No  badpid so far */,
		   &BASIC,
		   NULL /*No schedule setup */);

}

void set_schedule_action (fd,routine,
			  free_schedule_data,inc_schedule_data_refcount,
			  data)
     int                         fd;
     schedule_routine          * routine;
     free_schedule_data_f         * free_schedule_data;
     inc_schedule_data_refcount_f * inc_schedule_data_refcount;
     union schedule_routine_data data;
{
    struct schedule_setup SCHED;
    
    /* bzero is defined on hdrs/defs.h */
    bzero((void *)&SCHED, sizeof SCHED);

    SCHED.schedule_rut                = routine;
    SCHED.free_schedule_data          = free_schedule_data;
    SCHED.inc_schedule_data_refcount  = inc_schedule_data_refcount;
    SCHED.schedule_data               = data;

    setup_actions0(fd,
		   0 /* No  badpid so far */,
		   NULL /* No basic setup */,
		   &SCHED);    
}			     

static void schedule_remove_idx(ptr)
     int ptr;
{
    if (ptr < 0 || ptr >=  actions_count)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "schedulre_remove_idx",
	      "Bad index",0);

    DPRINT(Debug,18,(&Debug,
		     "schedulre_remove_idx: Resetting action %d (fd=%d)\n",
		     ptr,actions[ptr].fd));

    actions[ptr].read_act     = no_action_routine;
    actions[ptr].write_act    = no_action_routine;
    actions[ptr].timeout_act  = no_action_routine;
    actions[ptr].schedule_act = no_schedule_routine;
}

static void clear_action_idx P_((int ptr, int is_badpid));
static void clear_action_idx(ptr,is_badpid)
     int ptr;
     int is_badpid;
{
    int badpid = 0;
    
    if (ptr < 0 || ptr >=  actions_count)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "clear_action_idx",
	      "Bad index",0);

    DPRINT(Debug,18,(&Debug,
		     "clear_action_idx: Clearing action %d (fd=%d)",
		     ptr,actions[ptr].fd));

    if (is_badpid) {
	DPRINT(Debug,18,(&Debug,", assuming badpid %d request",
			 is_badpid));
	badpid = is_badpid;
    } else {
	int p = getpid();
	if (actions[ptr].caller_pid != p) {
	    DPRINT(Debug,18,(&Debug,", caller pid %d is not our (%d)",
			     actions[ptr].caller_pid,p));
	    badpid = actions[ptr].caller_pid;
	}
    }

    if (actions[ptr].busy) {

	 struct action_setups  BASIC;
	 struct schedule_setup SCHED;

	 DPRINT(Debug,18,(&Debug,"; busy\n"));
	 
	 /* bzero is defined on hdrs/defs.h */
	 bzero((void *)&BASIC, sizeof BASIC);
	 
	 BASIC.read_act                    = no_action_routine;
	 BASIC.write_act                   = no_action_routine;
	 BASIC.timeout_act                 = no_action_routine;
	 BASIC.timeout_sec                 = -1;
	 BASIC.free_action_data            = no_free_action_data;
	 BASIC.inc_action_data_refcount    = no_inc_action_data_refcount;
	 BASIC.badpid_act                  = no_badpid_action;
	 BASIC.basic_data                  = NO_action_routine_data;;

	 /* bzero is defined on hdrs/defs.h */
	 bzero((void *)&SCHED, sizeof SCHED);
	 
	 SCHED.schedule_rut                = no_schedule_routine;
	 SCHED.free_schedule_data          = no_free_schedule_data;
	 SCHED.inc_schedule_data_refcount  = no_inc_schedule_data_refcount;
	 SCHED.schedule_data               = NO_schedule_routine_data;

	 setup_actions0(actions[ptr].fd,
			badpid,
			&BASIC,
			&SCHED);
	 
    } else {
	DPRINT(Debug,18,(&Debug,"; may remove\n"));
	remove_action0(ptr,badpid);
    }
}

void clear_action0(fd,is_badpid)
     int fd; 
     int is_badpid;
{
    int ptr;

    for (ptr = 0; ptr < actions_count; ptr++)
	if (actions[ptr].fd == fd)
	    break;

    if (ptr < actions_count) {
	DPRINT(Debug,9,(&Debug,
			"clear_action0: action %d fd=%d -- clearing action",
			ptr,fd));
	if (is_badpid) {
	    DPRINT(Debug,9,(&Debug,", assuming badpid %d request",
			    is_badpid));
	}
	DPRINT(Debug,9,(&Debug,"\n"));
	
	
	clear_action_idx(ptr,is_badpid);

    } else {
	DPRINT(Debug,9,(&Debug,
			"clear_action0: No action for (fd=%d); actions_count=%d\n",
			fd,actions_count));
    }

}

void clear_action(fd)
     int fd;
{
    clear_action0(fd,
		  0 /* Assume no badpid */);
}


/* -------------------------------------------------------------------- */

#define TIMED_TASK_HANDLE_magic 0xF102
struct timed_task_handle {
    unsigned short magic;     /* TIMED_TASK_HANDLE_magic */

    struct timed_task          * task;
    timed_task_free_f          * free_task;
    inc_timed_task_refcount_f  * inc_task_refcount;
    timed_task_action_f        * task_action;

    struct timed_task_handle   ** free_schedule_cleanup;
    
    struct schedule_timelimit    deadline;
    unsigned int                 fired:1;   /* task_action called         */
    unsigned int                 busy:1;    /* task_action() on execution */
    unsigned int                 delete:1;  /* delete after task_action() */
					    /*  ---                       */
					    /* used if alter_timed_task() */
					    /* called when busy           */
} ** timed_task_handles = NULL;
int  timed_task_handle_count = 0;

static struct timed_task_handle ANY_TASK_dummy = {
    TIMED_TASK_HANDLE_magic,
    NULL,
    NO_timed_task_free,
    NO_inc_timed_task_refcount,
    NO_timed_task_action,
    NULL,

    {  /* NO_schedule_timelimit */
	0,
#ifdef USE_GETTIMEOFDAY
	{ 0 , 0 },
#endif
    
#ifdef USE_CLOCK_MONOTONIC
	{ 0 , 0 },
#endif
    },

    0,0,0
};

const struct timed_task_handle * ANY_TASK = &ANY_TASK_dummy;

static void free_timed_task_handle P_((struct timed_task_handle **handle,int found));
static void free_timed_task_handle(handle,found)
     struct timed_task_handle **handle;
     int found;
{
    if (TIMED_TASK_HANDLE_magic != (*handle)->magic)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "free_timed_task_handle",
	      "Bad magic number",0);

    if (timed_task_handles) {

	if (found >= 0 && found < timed_task_handle_count &&
	    *handle == timed_task_handles[found]) {

	    DPRINT(Debug,9,(&Debug,
			    "free_timed_task_handle: timed task %d -- task handle %p\n",
			    found,*handle));
	    
	} else {
	    
	    int i;
	    found = -1;
	    
	    for (i = 0; i < timed_task_handle_count; i++) {
		if (*handle == timed_task_handles[i]) {
		    found = i;
		    
		    DPRINT(Debug,9,(&Debug,
				    "free_timed_task_handle: timed task %d -- task handle %p\n",
				    i,*handle));
		}
	    }
	}
    } else {
	found = -1;
    }
    
    if ((*handle)->busy) {
	
	if ((*handle)->delete)
	    panic("SCHEDULE PANIC",__FILE__,__LINE__,
		  "free_timed_task_handle",
		  "Busy and delete set",0);

	DPRINT(Debug,9,(&Debug,
			"free_timed_task_handle: busy, delete pending (task handle %p)\n",
			*handle));
	    
	(*handle)->delete = 1;
	(*handle) = NULL;
	return;
    }

    if ((*handle)->delete) {
	DPRINT(Debug,9,(&Debug,
			"free_timed_task_handle: delete pending -- task handle %p\n",
			*handle));
    } else if (-1 == found) {
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "free_timed_task_handle",
	      "Not on timed_task_handles",0);
    }

    if ((*handle)->task) {
	(*handle)->busy = 1;
	(*handle)->free_task(&  ((*handle)->task));

	if (!(*handle)) {
	    DPRINT(Debug,9,(&Debug,
			    "free_timed_task_handle: [%d] OOPS --- handle was reset by ->free_task()\n",
			    found));
	    goto no_handle;
	}
	
	(*handle)->busy = 0;
    }
	
    if ((*handle)->task) {
	DPRINT(Debug,9,(&Debug,
			"free_timed_task_handle: [%d], task %p not free'ed,  task handle %p\n",
			found,(*handle)->task,*handle));
    }

    if ((*handle)->free_schedule_cleanup) {
	if ((*handle)->free_schedule_cleanup == handle) {
	    DPRINT(Debug,9,(&Debug,
			    "free_timed_task_handle:  [%d], task hadle %p, free_schedule_cleanup points to this variable %p\n",
			    found,(*handle),handle));
	} else {
	    DPRINT(Debug,9,(&Debug,
			    "free_timed_task_handle:  [%d], task hadle %p, free_schedule_cleanup points to %p  != this variable %p\n",
			    found,(*handle),(*handle)->free_schedule_cleanup,handle));
	}
	(*handle)->free_schedule_cleanup = NULL;
    }
    
    (*handle)->deadline = NO_schedule_timelimit;

    (*handle)->magic = 0;  /* Invalidate */
    free(*handle);
    *handle = NULL;

 no_handle:
    /* If handle != &timed_task_handles[found] 
       clear pointer now, when it is not cleared above
    */
    
    if (found >= 0 && timed_task_handles[found]) {
	DPRINT(Debug,9,(&Debug,
			"free_timed_task_handle: clearing timed task %d -- task handle %p\n",
			found,timed_task_handles[found]));
	
	timed_task_handles[found] = NULL;
    }
}

static void remove_timed_task P_((int ptr));
static void remove_timed_task(ptr)
     int ptr;
{
    int i;
    
    if (ptr < 0 || ptr >=  timed_task_handle_count)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "remove_timed_task",
	      "Bad index",0);

    DPRINT(Debug,4,(&Debug,
		    "** Removing timed task %d",
		    ptr));
    if (timed_task_handles[ptr]) {
	DPRINT(Debug,4,(&Debug," (task handle %p)",
			timed_task_handles[ptr]));
    } else {
	DPRINT(Debug,4,(&Debug," (no task handle)"));
    }
    DPRINT(Debug,4,(&Debug,"\n"));
    
    
    if (timed_task_handles[ptr])
	free_timed_task_handle(& (timed_task_handles[ptr]),ptr);

    for (i = ptr+1; i < timed_task_handle_count; i++) {
	timed_task_handles[i-1] = timed_task_handles[i];
	timed_task_handles[i] = NULL;
    }
	
    timed_task_handle_count--;
}

static int add_timed_task_handle P_((struct timed_task_handle * rec));
static int add_timed_task_handle(rec)
     struct timed_task_handle * rec;
{
    int found = -1;

    timed_task_handles =
	safe_array_realloc(timed_task_handles, timed_task_handle_count+1,
			   sizeof (timed_task_handles[0]));
    
    timed_task_handles[timed_task_handle_count] = rec;
    found = timed_task_handle_count;
    timed_task_handle_count++;

    return found;
}


static struct timed_task_handle * malloc_timed_task_handle P_((int *ptr));
static struct timed_task_handle * malloc_timed_task_handle(ptr)
     int *ptr;
{
    struct timed_task_handle * ret = safe_zero_alloc(sizeof (*ret));

    ret->task              = NULL;
    ret->free_task         = NO_timed_task_free;
    ret->inc_task_refcount = NO_inc_timed_task_refcount;
    ret->task_action       = NO_timed_task_action;
    ret->free_schedule_cleanup = NULL;
    ret->deadline          = NO_schedule_timelimit;

    ret->fired  = 0;
    ret->busy   = 0;
    ret->delete = 0;

    ret->magic = TIMED_TASK_HANDLE_magic;
    
    if (*ptr >= 0 && *ptr < timed_task_handle_count &&
	timed_task_handles && !timed_task_handles[*ptr]) {

	DPRINT(Debug,9,(&Debug,
			"malloc_timed_task_handle: reusing timed task %d -- task handle %p\n",
			*ptr,ret));
	
	timed_task_handles[*ptr] = ret;
    } else {

	*ptr = add_timed_task_handle(ret);
	
	DPRINT(Debug,9,(&Debug,
			"malloc_timed_task_handle: Adding timed task %d -- task handle %p\n",
			*ptr,ret));
    }
    
    return ret;
}

void timed_task_alter_cleanup(handle,allow_cleanup)
     struct timed_task_handle  ** handle;
     enum timed_task_cleanup allow_cleanup;
{
    int found = -1;
    
    if (TIMED_TASK_HANDLE_magic != (*handle)->magic)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "timed_task_alter_cleanup",
	      "Bad magic number",0);

    DPRINT(Debug,9,(&Debug,
		    "timed_task_alter_cleanup: timed task handle %p",
		    (*handle)));
    
    if (timed_task_handles) {
	int i;
	
	for (i = 0; i < timed_task_handle_count; i++) {
	    if (*handle == timed_task_handles[i]) {
		found = i;
		
		DPRINT(Debug,9,(&Debug,
				" -- timed task %d",
				i));
	    }
	}
    }	   	
        
    if ((*handle)->free_schedule_cleanup) {
	DPRINT(Debug,9,(&Debug,", free_schedule_cleanup variable %p",
			(*handle)->free_schedule_cleanup));
	if ((*handle)->free_schedule_cleanup != handle) {
	    DPRINT(Debug,9,(&Debug," != current variable %p",
			    handle));
	}
    }
    if ((*handle)->busy) {
	DPRINT(Debug,9,(&Debug,", busy"));
    }    
    DPRINT(Debug,9,(&Debug,"\n"));
    
    if ((*handle)->free_schedule_cleanup &&
	(*handle)->free_schedule_cleanup != handle) 
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "timed_task_alter_cleanup",
	      "free_schedule_cleanup already set for different variable!",0);

    switch (allow_cleanup) {
    case  tt_discard_free_schedule_cleanup:
	if ((*handle)->free_schedule_cleanup) {
	    DPRINT(Debug,9,(&Debug,
			    "timed_task_alter_cleanup: timed task handle %p -- clearing free_schedule_cleanup\n",
			    (*handle)));
	    (*handle)->free_schedule_cleanup = NULL;
	}
	break;
    case tt_allow_free_schedule_cleanup:
	if (-1 == found) {
	    found = add_timed_task_handle(*handle);
	    
	    DPRINT(Debug,9,(&Debug,
			    "timed_task_alter_cleanup: timed task handle %p -- added timed task %d\n",
			    *handle,found));
	    
	}

	if (! (*handle)->free_schedule_cleanup) {
	    DPRINT(Debug,9,(&Debug,
			    "timed_task_alter_cleanup: timed task handle %p -- setting free_schedule_cleanup = variable %p\n",
			    (*handle),handle));
	    
	    (*handle)->free_schedule_cleanup = handle;
	}
	break;
    }
}

/* Return 1 if timed_task is sceduled 
   Return 0 if there is no POLL_method
*/
int alter_timed_task(handle,task,free_task,inc_task_refcount,task_action,deadline)
     struct timed_task_handle       ** handle;
     struct timed_task               * task   /* may be NULL */;   
     timed_task_free_f               * free_task;
     inc_timed_task_refcount_f       * inc_task_refcount;
     timed_task_action_f             * task_action;
     const struct schedule_timelimit * deadline;
{
    int ret = 0;
    int found = -1;
    
    if (*handle) {
	if (TIMED_TASK_HANDLE_magic != (*handle)->magic)
	    panic("SCHEDULE PANIC",__FILE__,__LINE__,
		  "alter_timed_task",
		  "Bad magic number",0);

	if (timed_task_handles) {
	    int i;
	    
	    for (i = 0; i < timed_task_handle_count; i++) {
		if (*handle == timed_task_handles[i]) {
		    found = i;
		    
		    DPRINT(Debug,9,(&Debug,
				    "alter_timed_task: timed task %d -- task handle %p\n",
				    i,*handle));
		}
	    }
	}	   	
    }

    
    if ((NULL                       == task &&
	 NO_timed_task_free         == free_task &&
	 NO_inc_timed_task_refcount == inc_task_refcount &&
	 NO_timed_task_action       == task_action) ||
	!deadline
	) {
	
	DPRINT(Debug,9,(&Debug,
			"alter_timed_task: Timed task removal request\n"));

    remove_task:
	/*  free_timed_task_handle() sets (*handle)->delete if (*handle)->busy is set */
	
	if (*handle)
	    free_timed_task_handle(handle,found);

	DPRINT(Debug,9,(&Debug,
			"alter_timed_task=%d (delete)\n",ret));
	
	return ret;
    }

    if (!schedule_have_timelimit(deadline)) {
	DPRINT(Debug,9,(&Debug,
			"alter_timed_task: No deadline\n"));

	goto remove_task;
    }
    
    if (*handle && -1 == found)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "alter_timed_task",
	      "Not on timed_task_handles",0);
    else if (!*handle) {

	if (POLL_method) {	
	    *handle = malloc_timed_task_handle(&found);

	    DPRINT(Debug,9,(&Debug,
			    "alter_timed_task: timed task %d -- task handle %p\n",
			    found,*handle));
	} else {
	    DPRINT(Debug,9,(&Debug,
			    "alter_timed_task=%d (No POLL_method)\n",
			    ret));

	    return ret;
	}
    } 

    if ((*handle)->task != task) {
	if (*handle && (*handle)->task)
	    (*handle)->free_task(& ((*handle)->task));
	
	if (*handle && (*handle)->task) {
	    DPRINT(Debug,9,(&Debug,
			    "alter_timed_task: Prevous task not free'ed() !\n"));
	}

	(*handle)->task = task;
	if ((*handle)->task)
	    inc_task_refcount((*handle)->task);
    }

    (*handle)->free_task         = free_task;
    (*handle)->inc_task_refcount = inc_task_refcount;
    (*handle)->task_action       = task_action;    
    (*handle)->deadline          = *deadline;
    (*handle)->fired             = 0;
    ret = 1;

    DPRINT(Debug,9,(&Debug,
		    "alter_timed_task=%d\n",
		    ret));
    
    return ret;
}

void NO_timed_task_free(task)
     struct timed_task  ** task;
{
    DPRINT(Debug,4,(&Debug,"NO_timed_task_free:  timed_task %p\n",
		    *task));

    *task = NULL;
}

void NO_inc_timed_task_refcount(task)
     struct timed_task   * task;
{
    DPRINT(Debug,4,(&Debug,"NO_inc_timed_task_refcount:  timed_task %p\n",
		    task));
}


enum timed_task_r NO_timed_task_action(task,now)
     struct timed_task   * task;
     const struct schedule_timelimit * now;
{
    DPRINT(Debug,4,(&Debug,"NO_timed_task_action:  timed_task %p\n",
		    task));

    return timed_task_disable;
}


/* -------------------------------------------------------------------- */


int have_actions()
{
    int count = actions_count + timed_task_handle_count;

    return count;
}

void free_schedule()
{
    DPRINT(Debug,12,(&Debug,"free_schedule: %d actions\n",
		     actions_count));

    
    if (actions_count > 0) {
	int ptr;

	for (ptr = actions_count-1; ptr >= 0; ptr--) {
	    if (actions[ptr].busy) {
		DPRINT(Debug,12,(&Debug,"free_schedule: action %d (fd=%d) busy\n",
				 ptr,actions[ptr].fd));
	    } else {
		DPRINT(Debug,12,(&Debug,"free_schedule: removing action %d (fd=%d)\n",
				 ptr,actions[ptr].fd));

		remove_action(ptr);
	    }
	}
    }

    if (0 == actions_count && actions) {
	DPRINT(Debug,12,(&Debug,"free_schedule: freeing actions array\n"));
	free(actions);
	actions = NULL;
    }

    if (timed_task_handle_count > 0) {
	int ptr;

	for (ptr = timed_task_handle_count-1; ptr >= 0; ptr--) {
	    if (timed_task_handles[ptr]) {
		if (TIMED_TASK_HANDLE_magic != timed_task_handles[ptr]->magic)
		    panic("SCHEDULE PANIC",__FILE__,__LINE__,
			  "free_schedule",
			  "Bad magic number",0);


		if (timed_task_handles[ptr]->busy) {
		    DPRINT(Debug,12,(&Debug,
				     "free_schedule: timed task %d (task handle %p) busy\n",
				     ptr,timed_task_handles[ptr]));
		} else if (timed_task_handles[ptr]->free_schedule_cleanup) {

		    struct timed_task_handle   ** free_schedule_cleanup
			= timed_task_handles[ptr]->free_schedule_cleanup;

		    
		    if (*(timed_task_handles[ptr]->free_schedule_cleanup) ==
			timed_task_handles[ptr]) {
			
			DPRINT(Debug,12,(&Debug,
					 "free_schedule: removing timed task %d (task handle %p)\n",
					 ptr,timed_task_handles[ptr]));
			remove_timed_task(ptr);

			if (NULL == *free_schedule_cleanup) {
			    DPRINT(Debug,12,(&Debug,
					     "free_schedule: timed task %d -- schedule cleanup variable %p was reset\n",
					     ptr,free_schedule_cleanup));
			    
			} else {
			    DPRINT(Debug,12,(&Debug,
					     "free_schedule: timed task %d -- claring schedule cleanup variable %p -- is task handle %p\n",
					     ptr,free_schedule_cleanup,*free_schedule_cleanup));
			    *free_schedule_cleanup = NULL;
			}
		    } else {
			DPRINT(Debug,12,(&Debug,
					 "free_schedule: timed task %d -- schedule cleanup variable %p is task handle %p!= current task handle %p\n",
					 ptr,free_schedule_cleanup,*free_schedule_cleanup,
					 timed_task_handles[ptr]));
			goto cleanup_data;
		    }
		} else {

		cleanup_data:
		    if (timed_task_handles[ptr]->task) {
			DPRINT(Debug,12,(&Debug,
					 "free_schedule: timed task %d (task handle %p) -- removing only time task %p (data)\n",
					 ptr,timed_task_handles[ptr],timed_task_handles[ptr]->task));
			timed_task_handles[ptr]->busy = 1;
			timed_task_handles[ptr]->free_task(& (timed_task_handles[ptr]->task));
			if (timed_task_handles[ptr]) {
			    timed_task_handles[ptr]->busy = 0;
			    timed_task_handles[ptr]->task = NULL;
			} else {
			    DPRINT(Debug,12,(&Debug,
					     "free_schedule: timed task %d -- OOPS ->free_task() free'ed timed task handle?\n",
					     ptr));
			}
		    }
		    
		}
	    }
	}
    }

    if (0 == timed_task_handle_count && timed_task_handles) {
	DPRINT(Debug,12,(&Debug,"free_schedule: freeing timed_task_handles array\n"));
	free(timed_task_handles);
	timed_task_handles = NULL;
    }    
}


static enum real_wait_status {
    real_wait_next_timeout = -2,
    real_wait_error        = -1,
    real_wait_none         = 0,
    real_wait_succeed      = 1	
} real_wait P_((action_routine                  * action,
		const struct schedule_timelimit * wait_timeout,
		const struct schedule_timelimit * next_timeout,
		struct cancel_data              * cd,
		struct timed_task_handle        * task,
		struct schedule_timelimit       * resulttime,
		int                             * err_p,
		int                               wait_flags
		));

static void real_wait_check_timeouts P_((const struct schedule_timelimit * current,
					 int                             * done_p,
					 enum real_wait_status           * result_p, 
					 action_routine                  * action,
					 struct timed_task_handle        * task));
static void real_wait_check_timeouts(current,done_p,result_p,action,task)
     const struct schedule_timelimit * current;
     int                             * done_p;
     enum real_wait_status           * result_p;
     action_routine                  * action;
     struct timed_task_handle        * task;
{
    enum real_wait_status result = *result_p;
    int done                     = *done_p;
    int i;
    int empty_slot = 0;

    
    DPRINT(Debug,20,(&Debug,
		     "real_wait_check_timeouts: %d actions\n",
		     actions_count));
    
    for (i = 0; i < actions_count; i++) {

	if (schedule_timeout_reached(& (actions[i].next_timeout),
				     current)) {
	    
	    if (actions[i].busy) {
		DPRINT(Debug,20,(&Debug,
				 "real_wait_check_timeouts: %d busy (action %d) -- timout skipped\n",
				 actions[i].fd,i));
	    } else {		
		enum action_status	 val;
		actions[i].busy = 1;
		
		DPRINT(Debug,20,(&Debug,
				 "real_wait_check_timeouts: running timeout_act for %d (action %d)\n",
				 actions[i].fd,i));
		val = actions[i].timeout_act(actions[i].fd,
					     actions[i].data,
					     current);
		DPRINT(Debug,20,(&Debug,
				 "real_wait_check_timeouts: [%d].timeout_act=%d",i,val));
		if (action == actions[i].timeout_act ||
		    action == ANY_ACTION) {
		    done = 1;
		    result = real_wait_succeed;
		    DPRINT(Debug,20,(&Debug," <wait done> "));
		}
		switch (val) {
		case action_disable:
		    actions[i].timeout_act  = no_action_routine;
		    actions[i].next_timeout = NO_schedule_timelimit;
		    DPRINT(Debug,20,(&Debug," (action_disable)"));
		    break;
		case action_continue:
		    DPRINT(Debug,20,(&Debug," (action_continue)"));
		    break;			    			    
		}
		DPRINT(Debug,20,(&Debug,"\n"));

		if (no_action_routine != actions[i].timeout_act) {
		    
		    if (! schedule_set_next_timeout0(& actions[i].next_timeout,
						     current,
						     actions[i].timeout_sec,

						     se_use_time_MAX
						     )) {
			DPRINT(Debug,20,(&Debug, 
					 "real_wait_check_timeouts: [%d] time overflow or bad time\n",i));
			
		    }
		}		
		actions[i].busy = 0;
	    }
	}
    }

    DPRINT(Debug,20,(&Debug,
		     "real_wait_check_timeouts: %d timed tasks\n",
		     timed_task_handle_count));

    for (i = 0; i < timed_task_handle_count; i++) {
	if (timed_task_handles[i]) {
	    if (TIMED_TASK_HANDLE_magic != timed_task_handles[i]->magic)
		panic("SCHEDULE PANIC",__FILE__,__LINE__,
		      "real_wait_check_timeout",
		      "Bad magic number (timed_task_handle)",0);

	    if (timed_task_handles[i]->fired) {
		    DPRINT(Debug,20,(&Debug,
				     "real_wait_check_timeouts: timed task %d busy (task handle %p) -- already fired/activated or disabled\n",
				     i,timed_task_handles[i]));

	    } else if (schedule_timeout_reached(&(timed_task_handles[i]->deadline),
					 current)) {

		if (timed_task_handles[i]->busy) {
		    DPRINT(Debug,20,(&Debug,
				     "real_wait_check_timeouts: timed task %d busy (task handle %p) -- deadline skipped\n",
				     i,timed_task_handles[i]));

		} else {
		    enum timed_task_r r;
		    
		    timed_task_handles[i]->busy  = 1;
		    timed_task_handles[i]->fired = 1;
		    DPRINT(Debug,20,(&Debug,
				     "real_wait_check_timeouts: running task_action for timed task %d (task handle %p)\n",
				     i,timed_task_handles[i]));
		    
		    r = timed_task_handles[i]->task_action(timed_task_handles[i]->task,
							   current);

		    if (task == timed_task_handles[i] ||
			task == ANY_TASK) {
			done = 1;
			result = 1;
		    }

		    switch (r) {
		    case timed_task_default: DPRINT(Debug,20,(&Debug,
							      "real_wait_check_timeouts: timed task %d (task handle %p) action done%s%s\n",
							      i,timed_task_handles[i],
							      timed_task_handles[i]->fired  ? "" : ", rescheduled",
							      timed_task_handles[i]->delete ? ", delete pending" : ""));
			break;
		    case timed_task_retry:
			DPRINT(Debug,20,(&Debug,
					 "real_wait_check_timeouts: timed task %d (task handle %p) action rescheduling%s\n",
					 i,timed_task_handles[i],
					 timed_task_handles[i]->delete ? ", delete pending" : ""));
			timed_task_handles[i]->fired = 0;
			break;
		    case timed_task_disable:
			DPRINT(Debug,20,(&Debug,
					 "real_wait_check_timeouts: timed task %d (task handle %p) action disabling (marking as fired/activated)%s\n",
					 i,timed_task_handles[i],
					 timed_task_handles[i]->delete ? ", delete pending" : ""));
			timed_task_handles[i]->fired = 1;
			break;
		    }
		    
		    timed_task_handles[i]->busy = 0;
		    if (timed_task_handles[i]->delete) {
			DPRINT(Debug,20,(&Debug,
				     "real_wait_check_timeouts: timed task %d delete pending (task handle %p) -- removing\n",
				     i,timed_task_handles[i]));

			free_timed_task_handle(& (timed_task_handles[i]),i);
		    }
		}

	    }
	} else {
	    empty_slot++;
	}	
    }
    
    DPRINT(Debug,20,(&Debug,
		     "real_wait_check_timeouts: %d empty slots on timed tasks, result = %d",
		     empty_slot,result));
    switch (result) {
    case real_wait_next_timeout: DPRINT(Debug,20,(&Debug," (real_wait_next_timeout)"));     break;
    case real_wait_error:        DPRINT(Debug,20,(&Debug," (real_wait_error)"));            break;
    case real_wait_none:         DPRINT(Debug,20,(&Debug," (real_wait_none)"));             break;
    case real_wait_succeed:      DPRINT(Debug,20,(&Debug," (real_wait_succeed)"));          break;
    }    
    DPRINT(Debug,20,(&Debug,", done = %d\n",done));
    
    *result_p = result;
    *done_p   = done;
}


    
static void real_wait_check_endtimeout P_((const struct schedule_timelimit  *
					   current,
					   int *done_p,
					   enum real_wait_status *result_p,
					   const struct schedule_timelimit  *
					   wait_timeout,
					   const struct schedule_timelimit *
					   next_timeout));
static void real_wait_check_endtimeout(current,done_p,result_p,
				       wait_timeout,next_timeout)
    const struct schedule_timelimit * current;
    int                             * done_p;
    enum real_wait_status           * result_p;
    const struct schedule_timelimit * wait_timeout;
    const struct schedule_timelimit * next_timeout;
{
    enum real_wait_status result = *result_p;
    int done                     = *done_p;

    if (schedule_timeout_reached(next_timeout,current)) {
	done = 1;
	if (real_wait_none  == result)
	    result = real_wait_next_timeout;

	DPRINT(Debug,20,(&Debug,
			 "real_wait_check_endtimeout: next timeout reached\n"));
    }

    if (schedule_timeout_reached(wait_timeout,current)) {
	done = 1;
	result = real_wait_succeed;
	
	DPRINT(Debug,20,(&Debug,
			 "real_wait_check_endtimeout: wait timeout reached\n"));
    }

    DPRINT(Debug,20,(&Debug,
		     "real_wait_check_endtimeout: result = %d",
		     result));
    switch (result) {
    case real_wait_next_timeout: DPRINT(Debug,20,(&Debug," (real_wait_next_timeout)"));     break;
    case real_wait_error:        DPRINT(Debug,20,(&Debug," (real_wait_error)"));            break;
    case real_wait_none:         DPRINT(Debug,20,(&Debug," (real_wait_none)"));             break;
    case real_wait_succeed:      DPRINT(Debug,20,(&Debug," (real_wait_succeed)"));          break;
    }    
    DPRINT(Debug,20,(&Debug,", done = %d\n",done));

    
    *result_p = result;
    *done_p   = done;
}





#ifdef  HAVE_SELECT
/* --------------------- select() ------------------------------------------ */

static enum real_wait_status real_wait_select P_((action_routine                  * action,
						  const struct schedule_timelimit * wait_timeout,
						  const struct schedule_timelimit * next_timeout,
						  struct cancel_data              * cd,
						  struct timed_task_handle        * task,
						  /* Updates *resulttime */
						  struct schedule_timelimit       * resulttime,
						  int                             * err_p,
						  int                               wait_flags));
static enum real_wait_status real_wait_select(action,wait_timeout,next_timeout,cd,task,resulttime,
					      err_p,wait_flags)
    action_routine                  * action;
    const struct schedule_timelimit * wait_timeout;
    const struct schedule_timelimit * next_timeout;
    struct cancel_data              * cd;
    struct timed_task_handle        * task;
    /* Updates *resulttime */
    struct schedule_timelimit       * resulttime;
    int                             * err_p;
    int                               wait_flags;
{
    enum real_wait_status result = real_wait_none;
    int done = 0;
    int err = 0;
    char * WA, * NA;
    
    char *Y UNUSED_VAROK = "action";
    char *Z UNUSED_VAROK = "timed task";
    
    if (action == no_action_routine)
	Y = "no action";
    else if (action == ANY_ACTION)
	Y = "any action";
    if (!task)
	Z = "no timed task";
    if (task == ANY_TASK)
	Z = "any timed task";
	        
    WA = schedule_timeout_string(wait_timeout);
    NA = schedule_timeout_string(next_timeout);
    
    DPRINT(Debug,20,(&Debug,
		     "real_wait_select: %s %p or %s %p, wait_timeout=%s, next_timeout=%s (actions_count=%d, %d timed tasks)\n", 	       
		     Y,action,
		     Z,task,
		     WA ? WA : "???",
		     NA ? NA : "???",
		     actions_count,
		     timed_task_handle_count));

    if (WA) {
	free(WA);
	WA = NULL;
    }

    if (NA) {
	free(NA);
	NA = NULL;
    }

    do {
	int i, n = 0;
	fd_set read_set;
	fd_set write_set;
	fd_set exp_set;
	struct timeval timeout, *timeout_val = NULL;
	int ret;
	struct schedule_timelimit  current = NO_schedule_timelimit;
	char * current_s = NULL;
	int cancel_delay = 0;

	FD_ZERO(&read_set);
	FD_ZERO(&write_set);
	FD_ZERO(&exp_set);
	
	DPRINT(Debug,20,(&Debug,
			 "real_wait_select: actions:"));
	for (i = 0; i < actions_count; i++) {
	    int ok = 1;
	    if (actions[i].fd >= FD_SETSIZE) {
		DPRINT(Debug,20,(&Debug,
			     " (fd=%d TOO BIG, FDSETSIZE=%d)",
				 actions[i].fd,FD_SETSIZE));
		continue;
	    }
	    
	    if (n <= actions[i].fd)
		n = actions[i].fd+1;

	    DPRINT(Debug,20,(&Debug,
			     " (fd=%d",actions[i].fd));

	    if (actions[i].badfd) {
		DPRINT(Debug,20,(&Debug,
				 " badfd!"));
		ok = 0;
	    }

	    if (actions[i].busy) {
		DPRINT(Debug,20,(&Debug,
				 " busy!"));
		ok = 0;
	    }

	    if (ok) {
		FD_SET(actions[i].fd,&exp_set);
		
		if (actions[i].read_act != no_action_routine) {
		    FD_SET(actions[i].fd,&read_set);
		    DPRINT(Debug,20,(&Debug,
				     " READ"));
		}
		
		if (actions[i].write_act != no_action_routine) {
		    FD_SET(actions[i].fd,&write_set);
		    DPRINT(Debug,20,(&Debug,
				     " WRITE"));
		}	
		if (actions[i].timeout_act != no_action_routine) {
		    DPRINT(Debug,20,(&Debug,
					 " TIMEOUT"));
		}	    
	    }
	    DPRINT(Debug,20,(&Debug, ")"));
	}
	DPRINT(Debug,20,(&Debug, "\n"));	
	
	if (schedule_timeout_to_timeval(next_timeout,&timeout)) {

	    DPRINT(Debug,20,(&Debug,
			     "real_wait_select: diff=%ld.%06ld\n",
			     (long)timeout.tv_sec,
			     timeout.tv_usec));	    
	    timeout_val        = &timeout;
	}

	if (cd) {
	    int delay_msec;

	    if (is_schedule_cancel(cd,&delay_msec)) {
		
		DPRINT(Debug,20,(&Debug, "real_wait_select: schedule cancel %d milliseconds\n",
				 delay_msec));

		if (delay_msec < 0)
		    delay_msec = 0;

		if (! timeout_val ||
		    timeout_val->tv_sec > delay_msec/1000
		    ||
		    ( timeout_val->tv_sec == delay_msec/1000 &&
		      timeout_val->tv_usec > delay_msec % 1000 )
		    ) {

		    timeout.tv_usec = 1000 * (  delay_msec % 1000 );
		    timeout.tv_sec  = delay_msec/1000;

		    cancel_delay = 1;
		    timeout_val        = &timeout;
		}
	    }	    
	}

	if (handle_sigchld && (WAIT_FOR_handle_sigchld & wait_flags)) {
	    done = 1;
	    
	    DPRINT(Debug,20,(&Debug,
			     "real_wait_select: WAIT_FOR_handle_sigchld -- handle_sigchld set, wait done\n"));
	    result = real_wait_succeed;
	    timeout.tv_usec = 0;
	    timeout.tv_sec  = 0;
	    timeout_val        = &timeout;
	    cancel_delay = 0;
	}

	DPRINT(Debug,20,(&Debug, "real_wait_select: n = %d, read set=", n));
	for (i = 0; i < n; i++) {
	    if (FD_ISSET(i,&read_set)) {
		DPRINT(Debug,20,(&Debug, " %d", i));
	    }
	}
	DPRINT(Debug,20,(&Debug, ", write set="));
	for (i = 0; i < n; i++) {
	    if (FD_ISSET(i,&write_set)) {
		DPRINT(Debug,20,(&Debug, " %d", i));
	    }
	}
	DPRINT(Debug,20,(&Debug,", exception set="));
	for (i = 0; i < n; i++) {
	    if (FD_ISSET(i,&exp_set)) {
		DPRINT(Debug,20,(&Debug, " %d", i));
	    }
	}
	DPRINT(Debug,20,(&Debug, ", timeout="));
	if (timeout_val) {
	    DPRINT(Debug,20,(&Debug, " (sec=%ld usec=%ld)\n", 
			     (long) timeout_val->tv_sec,
			     (long) timeout_val->tv_usec));
	} else {
	    DPRINT(Debug,20,(&Debug, "NULL\n")); 
	}
	
	wait_can_signal = 1;
#ifdef _HPUX_SOURCE
	ret = select(n,(int *)&read_set,(int *)&write_set,
		     (int *)&exp_set,timeout_val);
#else
	ret = select(n,&read_set,&write_set,&exp_set,timeout_val);
#endif
	wait_can_signal = 0;

	if (ret < 0) {
	    err = errno;
	    DPRINT(Debug,20,(&Debug,
			     "real_wait_select select: %s (errno %d)\n",
			     strerror(err),err));
	    result = real_wait_error;

	    if (EINTR == err && cd && is_canceled(cd)) {
		DPRINT(Debug,20,(&Debug,
				 "real_wait_select: wait canceled\n"));		    	
	    }
	    
	    break;
	}

	schedule_set_current_time(&current);
	current_s = schedule_timeout_string(&current);
	DPRINT(Debug,20,(&Debug,
			 "real_wait_select: select=%d, current=%s\n",
			 ret,
			 current_s ? current_s : "??"));
	if (current_s) {
	    free(current_s);
	    current_s = NULL;
	}
	*resulttime = current;
	
#ifdef USE_DLOPEN
	if (ret > 0) {
	    /* If we just timed out, it is quite predictable
	       so no seeding 
	    */
	    union xxx_rand {
		struct collection {
		    struct timeval XX;   /* someone may modify this */
		    struct schedule_timelimit X;
		    fd_set read_set;
		    fd_set write_set;
		    int r;
		} X;
		char bytes [sizeof (struct collection)];
	    } A;

	    /* Try avoid valgrind uninitialised byte
	       varning */
	    bzero((void *)&A,sizeof A);
	    
	    if (timeout_val)
		A.X.XX = *timeout_val;
	    A.X.X         = current;
	    memcpy( &(A.X.read_set), &(read_set), sizeof read_set);
	    memcpy( &(A.X.write_set), &(write_set), sizeof write_set);
	    A.X.r         = ret;

	    seed_rand_bits(A.bytes, sizeof A, 
			   4 /* entropy bits */ );
	}
#endif

	if (0 == ret && cancel_delay && cd) 
	    schedule_cancel_timeout(cd);
	
	if (ret > 0) {
	    for (i = 0; i < actions_count; i++) {
		if (actions[i].fd >= FD_SETSIZE) {
		    DPRINT(Debug,20,(&Debug, 
				     "real_wait_select: fd=%d too big  FDSETSIZE=%d (action %d)\n",
				     actions[i].fd,FD_SETSIZE,i));
		    continue;
		}
		
		if (!actions[i].busy) {
		    actions[i].busy = 1;
		    		    
		    if (FD_ISSET(actions[i].fd,&read_set) ||
			FD_ISSET(actions[i].fd,&exp_set)) {
			enum action_status val; 
			DPRINT(Debug,20,(&Debug, 
					 "real_wait_select: running read_act for %d (action %d)\n",
					 actions[i].fd,i
					 ));
			
			val  = actions[i].read_act(actions[i].fd,
						   actions[i].data,
						   &current);
			DPRINT(Debug,20,(&Debug,
					 "real_wait_select: [%d].read_act=%d",i,val));
			
			if (action == actions[i].read_act ||
			    action == ANY_ACTION) {
			    done = 1;
			    result = real_wait_succeed;
			    DPRINT(Debug,20,(&Debug," <wait done> "));
			}
			switch (val) {
			case action_disable:
			    actions[i].read_act = no_action_routine;
  			    DPRINT(Debug,20,(&Debug," (action_disable)"));
			    break;
			case action_continue:
			    DPRINT(Debug,20,(&Debug," (action_continue)"));
			    break;			    			    
			}
			DPRINT(Debug,20,(&Debug,"\n"));
		    }
		    
		    if (FD_ISSET(actions[i].fd,&write_set) ||
			FD_ISSET(actions[i].fd,&exp_set)) {
			enum action_status val;
			DPRINT(Debug,20,(&Debug,
					 "real_wait_select: running write_act for %d (action %d)\n",
					 actions[i].fd,i));
			val = actions[i].write_act(actions[i].fd,
						   actions[i].data,
						   &current);
			DPRINT(Debug,20,(&Debug,
					 "real_wait_select: [%d].write_act=%d",i,val));
			if (action == actions[i].write_act ||
			    action == ANY_ACTION) {
			    done = 1;
			    result =  real_wait_succeed;
			    DPRINT(Debug,20,(&Debug," <wait done> "));
			}
			switch (val) {
			case action_disable:
			    actions[i].write_act = no_action_routine;
			    DPRINT(Debug,20,(&Debug," (action_disable)"));
			    break;
			case action_continue:
			    DPRINT(Debug,20,(&Debug," (action_continue)"));
			    break;		 
			}
			DPRINT(Debug,20,(&Debug,"\n"));
		    }
		    actions[i].busy = 0;
		}
		if (FD_ISSET(actions[i].fd,&exp_set)) {
		    DPRINT(Debug,20,(&Debug,
				     "real_wait_select: [%d] exception\n",i));
		    actions[i].write_act = no_action_routine;
		    actions[i].read_act = no_action_routine;
		    done = 1;
		    result = real_wait_succeed;
		}
	    }
	}

	real_wait_check_timeouts(&current,&done,&result,action,task);

	real_wait_check_endtimeout(&current,&done,&result,
				   wait_timeout,next_timeout);

	DPRINT(Debug,20,(&Debug,
			 "real_wait_select: done=%d, result=%d",done,result));
	switch (result) {
	case real_wait_next_timeout: DPRINT(Debug,20,(&Debug," (real_wait_next_timeout)"));     break;
	case real_wait_error:        DPRINT(Debug,20,(&Debug," (real_wait_error)"));            break;
	case real_wait_none:         DPRINT(Debug,20,(&Debug," (real_wait_none)"));             break;
	case real_wait_succeed:      DPRINT(Debug,20,(&Debug," (real_wait_succeed)"));          break;
	}    
	DPRINT(Debug,20,(&Debug,"\n"));
	
    } while(!done);

    
    DPRINT(Debug,20,(&Debug,
		     "real_wait_select=%d",result));
    switch (result) {
    case real_wait_next_timeout: DPRINT(Debug,20,(&Debug," (real_wait_next_timeout)"));     break;
    case real_wait_error:        DPRINT(Debug,20,(&Debug," (real_wait_error)"));            break;
    case real_wait_none:         DPRINT(Debug,20,(&Debug," (real_wait_none)"));             break;
    case real_wait_succeed:      DPRINT(Debug,20,(&Debug," (real_wait_succeed)"));          break;
    }    
    if (err) {
	DPRINT(Debug,20,(&Debug,", errno %d (%s)",
			 err,strerror(err)));
    }
    DPRINT(Debug,20,(&Debug,"\n"));
    errno = err;
    if (err_p)
	*err_p = err;
    
    return result;
}

#endif


#ifdef POLL
/* --------------------- poll() -------------------------------------------- */
 
static enum real_wait_status real_wait_poll P_((action_routine                  * action,
						const struct schedule_timelimit * wait_timeout, 
						const struct schedule_timelimit * next_timeout,
						struct cancel_data              * cd,
						struct timed_task_handle        * task,
						/* Updates *resulttime */
						struct schedule_timelimit       * resulttime,
						int                             * err_p,
						int                               wait_flags
						));

static int real_wait_poll(action,wait_timeout,next_timeout,cd,task,resulttime,err_p,wait_flags)
    action_routine                  * action;
    const struct schedule_timelimit * wait_timeout;
    const struct schedule_timelimit * next_timeout;
    struct cancel_data              * cd;
    struct timed_task_handle        * task;
    /* Updates *resulttime */
     struct schedule_timelimit      * resulttime;
    int                             * err_p;
    int                               wait_flags;
{
    enum real_wait_status result = 0;
    int done = 0;
    int err = 0;
    struct pollfd  * polls = NULL;
    int            * translate = NULL;
    int polls_count = 0;
    char * WA, * NA;
    char *Y UNUSED_VAROK = "action";
    char *Z UNUSED_VAROK = "timed task";
     
    if (action == no_action_routine)
	Y = "no action";
    else if (action == ANY_ACTION)
	Y = "any action";
    if (!task)
	Z = "no timed task";
    if (task == ANY_TASK)
	Z = "any timed task";
    
    WA = schedule_timeout_string(wait_timeout);
    NA = schedule_timeout_string(next_timeout);
    
    DPRINT(Debug,20,(&Debug,
		     "real_wait_poll: %s %p or %s %p, wait_timeout=%s, next_timeout=%s (actions_count=%d, %d timed tasks)\n",
		     Y,action,
		     Z,task,
		     WA ? WA : "???",
		     NA ? NA : "???",
		     actions_count,
		     timed_task_handle_count));

    if (WA) {
	free(WA);
	WA = NULL;
    }

    if (NA) {
	free(NA);
	NA = NULL;
    }

    do {
	int i,pl;
	int ret;
	struct schedule_timelimit  current = NO_schedule_timelimit;
	char * current_s = NULL;
	int timeout = -1;
	int cancel_delay = 0;
	int used_polls = 0;

	if (polls_count < actions_count) {
	    polls = safe_array_realloc(polls,
				       actions_count, sizeof(struct pollfd));
	    polls_count = actions_count;
	    bzero(polls,actions_count * sizeof(struct pollfd));

	    translate = safe_array_realloc(translate,
					   actions_count, sizeof(translate[0]));
	    bzero(translate,actions_count * sizeof (translate[0]));
	    
	    DPRINT(Debug,20,(&Debug,
			     "real_wait_poll: polls=%p, polls_count=%d\n",
			     polls,polls_count));	    
	}

	DPRINT(Debug,20,(&Debug,
			 "real_wait_poll: actions:"));
	for (i = 0; i < actions_count; i++) {
	    int ok = 1;
	    
	    DPRINT(Debug,20,(&Debug, " (fd=%d",actions[i].fd));

	    if (actions[i].badfd) {
		DPRINT(Debug,20,(&Debug,
				 " badfd!"));
		ok = 0;
	    } 

	    if (actions[i].busy) {
		DPRINT(Debug,20,(&Debug, " busy!"));
		ok = 0;
	    }

	    if (ok) {
		DPRINT(Debug,20,(&Debug,
				 " idx=%d => polls[%d] ",
				 i,used_polls));
		
		if (used_polls >= polls_count)
		    panic("SCHEDULE PANIC",__FILE__,__LINE__,
			  "real_wait_poll",
			  "used_polls overflow!",0);
		
		translate[used_polls]     = i;
		polls[used_polls].fd      = actions[i].fd;
		polls[used_polls].events  = 0;
		polls[used_polls].revents = 0;
		
		if (actions[i].read_act != no_action_routine) {
		    polls[used_polls].events |= POLLIN | POLLHUP;
		    DPRINT(Debug,20,(&Debug, " READ"));
		}
		
		if (actions[i].write_act != no_action_routine) {
		    polls[used_polls].events |= POLLOUT;
		    DPRINT(Debug,20,(&Debug, " WRITE"));
		}	
		if (actions[i].timeout_act != no_action_routine) {
		    DPRINT(Debug,20,(&Debug, " TIMEOUT"));
		}
		
		used_polls++;	    
	    }
	    DPRINT(Debug,20,(&Debug, ")"));
	}
        DPRINT(Debug,20,(&Debug, "\n"));	

	if (schedule_timeout_to_timeout_msec(next_timeout,
					     &timeout)) {
					    
	    DPRINT(Debug,20,(&Debug, "real_wait_poll: diff=%d.%03d\n",
			     timeout/1000,timeout%1000));
	    
	}

	if (cd) {
	    int delay_msec;

	    if (is_schedule_cancel(cd,&delay_msec)) {
		
		DPRINT(Debug,20,(&Debug, "real_wait_poll: schedule cancel %d milliseconds\n",
				 delay_msec));
		
		if (delay_msec < 0)
		    delay_msec = 0;

		if (timeout > delay_msec ||
		    -1 == timeout) {
		    cancel_delay = 1;
		    timeout = delay_msec;		    
		}
	    }
	}

	if (handle_sigchld && (WAIT_FOR_handle_sigchld & wait_flags)) {
	    done = 1;
	    
	    DPRINT(Debug,20,(&Debug,
			     "real_wait_poll: WAIT_FOR_handle_sigchld -- handle_sigchld set, wait done\n"));
	    result = real_wait_succeed;
	    timeout = 0;
	    cancel_delay = 0;
	}

	DPRINT(Debug,20,(&Debug,"real_wait_poll: used_polls = %d, polls=", 
			 used_polls));
	for (pl = 0; pl < used_polls; pl++)
	    DPRINT(Debug,10,(&Debug, " {fd=%d events=%d}", 
			     polls[pl].fd, polls[pl].events));

	DPRINT(Debug,20,(&Debug, ", timeout=%d\n", 
			 timeout));
	
	wait_can_signal = 1;
	ret = poll(polls,used_polls,timeout);
	wait_can_signal = 0;

	if (ret < 0) {
	    err = errno;
	    DPRINT(Debug,20,(&Debug,
			"real_wait_poll poll: %s (errno %d)\n",
			strerror(err),err));
	    result = real_wait_error;

	    if (EINTR == err) {
		if (cd && is_canceled(cd)) {
		    DPRINT(Debug,20,(&Debug,
				     "real_wait_poll: wait canceled\n"));		    
		}
	    }
	    
	    break;
	}

	schedule_set_current_time(&current);
	current_s = schedule_timeout_string(&current);
	DPRINT(Debug,20,(&Debug, "real_wait_poll: poll=%d, current=%s\n",
			 ret,
			 current_s ? current_s : "??"));
	if (current_s) {
	    free(current_s);
	    current_s = NULL;
	}
	*resulttime = current;

#ifdef USE_DLOPEN
	if (ret > 0) {
	    /* If we just timed out, it is quite predictable
	       so no seeding 
	    */
	    union xxx_rand {
                struct collection {
		    int    r;
		    struct schedule_timelimit t;
		} X;
		char bytes [sizeof (struct collection)];
	    } A;

	    /* Try avoid valgrind uninitialised byte
	       varning */
	    bzero((void *)&A,sizeof A);
	    
	    A.X.r = ret;
	    A.X.t = current;

	    
	    seed_rand_bits(A.bytes, sizeof A, 
			   4 /* entroby bits */ );

	}
#endif
	
	if (0 == ret && cancel_delay && cd) 
	    schedule_cancel_timeout(cd);

	if (ret > 0) {
	    for (pl = 0; pl < used_polls; pl++) {
		i = translate[pl];

		DPRINT(Debug,20,(&Debug, 
				 "real_wait_poll: polls[%d] => action %d\n",
				 pl,i));

		if (i < 0 || i >= actions_count ||
		    polls[pl].fd != actions[i].fd)
		    panic("SCHEDULE PANIC",__FILE__,__LINE__,
			      "real_wait_poll",
			      "Bad translate table",0);
				
		if (polls[pl].revents) {
		    DPRINT(Debug,20,(&Debug, 
				     "real_wait_poll: polls[%d]: fd=%d revents=%d\n",
				     pl,polls[pl].fd,polls[pl].revents));
		}
		if (!actions[i].busy) {
		    actions[i].busy = 1;
		    if (0 != (polls[pl].revents & (POLLIN|POLLHUP|POLLERR))) {
			enum action_status val; 
			DPRINT(Debug,20,(&Debug,
					 "real_wait_poll: running read_act for %d (actions[%d], polls[%d])\n",
					 actions[i].fd,i,pl));
			val  = actions[i].read_act(actions[i].fd,
						   actions[i].data,
						   &current);
		       DPRINT(Debug,20,(&Debug,
					"real_wait_poll: actions[%d].read_act=%d",i,val));
			if (action == actions[i].read_act ||
			    action == ANY_ACTION) {
			    done = 1;
			    result = real_wait_succeed;
			    DPRINT(Debug,20,(&Debug," <wait done> "));
			}
			switch (val) {
			case action_disable:
			    actions[i].read_act = no_action_routine;
			    DPRINT(Debug,20,(&Debug," (action_disable)"));
			    break;
			case action_continue:
			    DPRINT(Debug,20,(&Debug," (action_continue)"));
			    break;		       
			}
			DPRINT(Debug,20,(&Debug,"\n"));
		    }
		    
		    if (0 != (polls[pl].revents & (POLLOUT|POLLERR))) {
			enum action_status val;
			DPRINT(Debug,20,(&Debug,
					 "real_wait_poll: running write_act for %d (actions[%d], polls[%d])\n",
					 actions[i].fd,i,pl));
			val = actions[i].write_act(actions[i].fd,
						   actions[i].data,
						   &current);
			DPRINT(Debug,20,(&Debug,
					 "real_wait_poll: actions[%d].write_act=%d",i,val));
			if (action == actions[i].write_act ||
			    action == ANY_ACTION) {
			    done = 1;
			    result = real_wait_succeed;
			    DPRINT(Debug,20,(&Debug," <wait done> "));
			}
			switch (val) {
			case action_disable:
			    actions[i].write_act = no_action_routine;
			    DPRINT(Debug,20,(&Debug," (action_disable)"));
			    break;
			case action_continue:
			    DPRINT(Debug,20,(&Debug," (action_continue)"));
			    break;			    	      
			}
			DPRINT(Debug,20,(&Debug,"\n"));
		    }
		    actions[i].busy = 0;
		}
		if (polls[pl].revents & POLLERR) {
		    DPRINT(Debug,20,(&Debug,
				     "real_wait_poll: actions[%d] POLLERR (polls[%d])\n",i,pl));
		    actions[i].write_act = no_action_routine;
		    actions[i].read_act = no_action_routine;
		    done = 1;
		    result = real_wait_succeed;
		}
		if (polls[pl].revents & POLLNVAL) {
		    DPRINT(Debug,20,(&Debug,
				     "real_wait_poll: actions[%d] POLLNVAL (polls[%d] fd %d)\n",
				     i,pl,polls[pl].fd));
		    actions[i].badfd = 1;
		    done = 1;
		    result = real_wait_error;
		    err = EBADF;           /* Simulate error */
		}
	    }
	}
	
	real_wait_check_timeouts(&current,&done,&result,action,task);

	real_wait_check_endtimeout(&current,&done,&result,
				   wait_timeout,next_timeout);
	
	DPRINT(Debug,20,(&Debug,
			 "real_wait_poll: done=%d, result=%d",done,result));
	switch (result) {
	case real_wait_next_timeout: DPRINT(Debug,20,(&Debug," (real_wait_next_timeout)"));     break;
	case real_wait_error:        DPRINT(Debug,20,(&Debug," (real_wait_error)"));            break;
	case real_wait_none:         DPRINT(Debug,20,(&Debug," (real_wait_none)"));             break;
	case real_wait_succeed:      DPRINT(Debug,20,(&Debug," (real_wait_succeed)"));          break;
	}    
	DPRINT(Debug,20,(&Debug,"\n"));
	
    } while(!done);


    if (polls)
	free(polls);
    if (translate)
       free(translate);

    DPRINT(Debug,20,(&Debug,
		     "real_wait_poll=%d",result));
    switch (result) {
    case real_wait_next_timeout: DPRINT(Debug,20,(&Debug," (real_wait_next_timeout)"));     break;
    case real_wait_error:        DPRINT(Debug,20,(&Debug," (real_wait_error)"));            break;
    case real_wait_none:         DPRINT(Debug,20,(&Debug," (real_wait_none)"));             break;
    case real_wait_succeed:      DPRINT(Debug,20,(&Debug," (real_wait_succeed)"));          break;
    }    
    if (err) {
	DPRINT(Debug,20,(&Debug,", errno %d (%s)",
			 err,strerror(err)));
    }
    DPRINT(Debug,20,(&Debug,"\n"));
    errno = err;
    if (err_p)
	*err_p = err;

    return result;
}

#endif

static enum real_wait_status real_wait(action,wait_timeout,next_timeout,cd,task,resulttime,err_p,
				       wait_flags)
     action_routine                  * action;
     const struct schedule_timelimit * wait_timeout;
     const struct schedule_timelimit * next_timeout;
     struct cancel_data              * cd;
     struct timed_task_handle        * task;
     /* Updates *resulttime */
     struct schedule_timelimit       * resulttime;
     int                             * err_p;
     int                               wait_flags;
{
    enum real_wait_status ret = 0;

    switch (POLL_method) {
	unsigned int sleep_sec;
	
#ifdef HAVE_SELECT
    case POLL_method_select:
	ret = real_wait_select(action,wait_timeout,
			       next_timeout,cd,task,
			       resulttime,err_p,wait_flags);
	break;
#endif
#ifdef POLL
    case POLL_method_poll:
	ret = real_wait_poll(action,wait_timeout,
			     next_timeout,cd,task,
			     resulttime,err_p,wait_flags);
	break;
#endif
    default:
	if (handle_sigchld && (WAIT_FOR_handle_sigchld & wait_flags)) {

	    DPRINT(Debug,20,(&Debug,
			     "real_wait: WAIT_FOR_handle_sigchld -- handle_sigchld set, wait done\n"));
	    
	} else if (schedule_timeout_to_timeout_sec(next_timeout,
						   & sleep_sec)) {
	    
	    DPRINT(Debug,20,(&Debug,
			     "real_wait: not available, diff=%u\n",
			     sleep_sec));

	    if ( sleep_sec > 0)
		sleep(sleep_sec);
	}

	schedule_set_current_time(resulttime);
		
	errno = ENOSYS;
	if (err_p)
	    *err_p = ENOSYS;
	
	ret = real_wait_error;
	DPRINT(Debug,20,(&Debug,
			 "real_wait: not available, POLL_method=%d\n",
			 POLL_method));
    }

    DPRINT(Debug,20,(&Debug,
		     "real_wait=%d",
		     ret));
    switch (ret) {
    case real_wait_next_timeout: DPRINT(Debug,20,(&Debug," (real_wait_next_timeout)"));     break;
    case real_wait_error:        DPRINT(Debug,20,(&Debug," (real_wait_error)"));            break;
    case real_wait_none:         DPRINT(Debug,20,(&Debug," (real_wait_none)"));             break;
    case real_wait_succeed:      DPRINT(Debug,20,(&Debug," (real_wait_succeed)"));          break;
    }    
    if (err_p && *err_p) {
	DPRINT(Debug,20,(&Debug,", errno %d (%s)",
			 *err_p,strerror(*err_p)));
    }
    DPRINT(Debug,20,(&Debug,"\n"));

    return ret;
}

static int wait_level = 0;

/* ----------- GENERIC ---------------------------------------------------- */

/* Should return 
      -1 on error
       1 if action or timeout occured
*/
static enum wait_for_status wait_for_something P_((action_routine            * action,
						   int                         seconds,
						   struct cancel_data        * cd,
						   struct timed_task_handle  * task,
						   const struct schedule_timelimit * deadline,
						   /* Updates *resulttime */
						   struct schedule_timelimit * resulttime,
						   int                       * err_p,
						   int                         wait_flags
						   ));
static int wait_for_something(action,seconds,cd,task,deadline,resulttime,err_p,wait_flags)
     action_routine           * action;
     int                        seconds; 
     struct cancel_data       * cd;
     struct timed_task_handle * task;
     const struct schedule_timelimit * deadline;
     struct schedule_timelimit * resulttime;
     int                       * err_p;
     int                         wait_flags;
{
    
    struct schedule_timelimit next_timeout = NO_schedule_timelimit;
    struct schedule_timelimit wait_timeout = NO_schedule_timelimit;
    struct schedule_timelimit current      = NO_schedule_timelimit;
    enum real_wait_status result = real_wait_none;
    enum wait_for_status  ret    = wait_for_none;
    int err = 0;
    int done = 0;
    int i;

    static int show_wait_error = 1;

    int current_wait_level = wait_level++;

    DPRINT(Debug,10,(&Debug,"wait_for_something%*s: current wait level=%d, %s cancel data\n",
		     2*current_wait_level,"",
		     current_wait_level,
		     cd ? "hav" : "no"));
    
    schedule_set_current_time(&current);
        
    if (seconds >= 0 &&
	schedule_set_next_timeout0(& wait_timeout,
				   &current,
				   seconds,
				   se_use_time_MAX)) {
	char * X = schedule_timeout_string(& wait_timeout);

	if (X) {
	    DPRINT(Debug,10,(&Debug,"wait_for_something%*s: wait_timeout=%s\n",
			     2*current_wait_level,"",
			     X));
	    
	    free(X);
	}
    }

    if (schedule_have_timelimit(deadline) &&
	schedule_shorten_next_timeout(& wait_timeout,deadline)) {
	char * X = schedule_timeout_string(& wait_timeout);

	if (X) {
	    DPRINT(Debug,10,(&Debug,"wait_for_something%*s: uses deadline, wait_timeout=%s\n",
			     2*current_wait_level,"",
			     X));
	    
	    free(X);
	}
    }
    
    do {
	int have_busy = 0;
	int p = getpid();
	
	next_timeout = wait_timeout;
		
	for (i = 0; i < actions_count; i++) {

	    DPRINT(Debug,12,(&Debug,
			     "wait_for_something%*s: action %d (fd=%d) caller pid %d",
			     2*current_wait_level,"",
			     i,actions[i].fd,actions[i].caller_pid));
	    
	    if (actions[i].caller_pid == p) {
		DPRINT(Debug,12,(&Debug,"  OK\n"));
	    } else {
		enum badpid_status val;
		int was_busy = actions[i].busy;

		DPRINT(Debug,12,(&Debug," NOT OUR (%d)\n",
				 p));

		DPRINT(Debug,20,(&Debug,
				 "wait_for_something%*s: running badpid_act for %d (action %d)\n",
				 2*current_wait_level,"",
				 actions[i].fd,i));

		actions[i].busy = 1;

		/* XXXXX Process even when busy !!!! */
		
		val = actions[i].badpid_act(actions[i].fd,
					    actions[i].data,
					    actions[i].caller_pid);
		
		actions[i].busy = was_busy;

		DPRINT(Debug,20,(&Debug,
				 "wait_for_something%*s: [%d].badpid_act=%d",
				 2*current_wait_level,"",
				 i,val));
		switch (val) {
		case badpid_disable:
		    actions[i].badpid_act = no_badpid_action;
		    DPRINT(Debug,20,(&Debug," (badpid_disable)"));
		    break;
		case badpid_ignore:
		    DPRINT(Debug,20,(&Debug," (badpid_ignore)"));
		    break;
		case badpid_remove:
		    DPRINT(Debug,20,(&Debug," (badpid_remove)"));
		    break;
		case badpid_change:		
		    actions[i].caller_pid = p;
		    DPRINT(Debug,20,(&Debug," (badpid_change)"));
		    break;
		}
		DPRINT(Debug,20,(&Debug,"\n"));
		if (badpid_remove == val)
		    schedule_remove_idx(i);		
	    }
	    
	    if (actions[i].busy) {
		have_busy = 1;
	    } else {
		actions[i].busy = 1;
		
		DPRINT(Debug,20,(&Debug,
				 "wait_for_something%*s: running .schedule_act for %d (action %d)\n",
				 2*current_wait_level,"",
				 actions[i].fd,i));
		
		switch(actions[i].schedule_act(actions[i].fd,
					       actions[i].schedule_data,
					       &current)) {
		case schedule_remove:
		    DPRINT(Debug,4,(&Debug,
				    "wait_for_something%*s: Schedule action %d (fd=%d): Remove pending\n",
				    2*current_wait_level,"",
			      i,actions[i].fd)); 

		    schedule_remove_idx(i);
		    break;
		case schedule_reconsider:
		    DPRINT(Debug,4,(&Debug,
				    "wait_for_something%*s: Schedule action %d (fd=%d): Reconsider pending\n",
				    2*current_wait_level,"",
				    i,actions[i].fd));

		    schedule_set_current_time(& next_timeout);
		    current = next_timeout;

		    break;
		case schedule_have_action:
		    DPRINT(Debug,4,(&Debug,
				    "wait_for_something: Schedule action %d (fd=%d): Action filled\n",
				    i,actions[i].fd));
		    schedule_set_current_time(& next_timeout);
		    current = next_timeout;
		    
		    if (ANY_ACTION == action ||
			( no_action_routine != action &&
			  (action == actions[i].read_act ||
			   action == actions[i].write_act ||
			   action == actions[i].timeout_act)
			  )
			) {
			DPRINT(Debug,4,(&Debug,
					"wait_for_something%*s: Schedule action %d (fd=%d): Ready\n",
					2*current_wait_level,"",
					i,actions[i].fd));
			done = 1;
		    }
                    break;
		default:
		    break;
		}

		actions[i].busy = 0;
	    }
	}
	
	for (i = 0; i < actions_count; i++) {
	    if (!actions[i].busy &&
		no_action_routine != actions[i].timeout_act) {

		if (schedule_shorten_next_timeout(&next_timeout,
						  & actions[i].next_timeout)) {
		    char * X = schedule_timeout_string(& next_timeout);

		    if (X) {
			DPRINT(Debug,10,(&Debug,
					 "wait_for_something%*s: Action %d (fd=%d): setting or shortened next_timeout to %s\n",
					 2*current_wait_level,"",
					 i,actions[i].fd,X));			
			free(X);
		    }		    
		}
	    }
	}

	for (i = 0; i < timed_task_handle_count; i++) {
	     if (timed_task_handles[i]) {
		if (TIMED_TASK_HANDLE_magic != timed_task_handles[i]->magic)
		    panic("SCHEDULE PANIC",__FILE__,__LINE__,
			  "wait_for_something",
			  "Bad magic number (timed_task_handle)",0);

		if (timed_task_handles[i]->busy)
		    have_busy = 1;
		else if (NO_timed_task_action != timed_task_handles[i]->task_action) {
		    if (timed_task_handles[i]->fired) {
			DPRINT(Debug,10,(&Debug,
					 "wait_for_something%*s: timed task %d (task handle %p): already fired/activated or disabled\n",
					 2*current_wait_level,"",
					 i,timed_task_handles[i]));
			
		    } else if (schedule_shorten_next_timeout(&next_timeout,
						      &(timed_task_handles[i]->deadline))) {

			char * X = schedule_timeout_string(& next_timeout);

			if (X) {
			    DPRINT(Debug,10,(&Debug,
					     "wait_for_something%*s: timed task %d (task handle %p): setting or shortened next_timeout to %s\n",
					     2*current_wait_level,"",
					     i,timed_task_handles[i],X));			
			    free(X);
			}
		    }
		}

	    }
	}
	
	if (schedule_invalid_next_timeout(& next_timeout)){
	    char * X = schedule_timeout_string(& next_timeout);
	    
	    DPRINT(Debug,1,(&Debug,
			    "wait_for_something%*s: Next timeout %s -- setting to current time\n",
			    2*current_wait_level,"",
			    X ? X : "n/a"));

	    if (X) {
		free(X);
		X = NULL;
	    }
	    schedule_set_current_time(& next_timeout);
	    current = next_timeout;
	}

	
	result = real_wait(action,&wait_timeout,&next_timeout,cd,task,

			   /* Updates current to time where wait ended */
			   
			   &current,&err, wait_flags);
	
	if (have_busy) {
	    DPRINT(Debug,4,(&Debug,
			    "wait_for_something%*s: Was busy actions or timed tasks, removals skipped\n",
			    2*current_wait_level,""
			    ));
	} else {

	    if (actions_count > 0) {
		int w;

		for (w = actions_count; w > 0; w=i) {
		    i = w-1;
		    
		    if (no_action_routine == actions[i].read_act &&
			no_action_routine == actions[i].write_act &&
			no_action_routine == actions[i].timeout_act &&
			actions[i].schedule_act == no_schedule_routine) {
			if (actions[i].busy) {
			    DPRINT(Debug,4,(&Debug,
					    "wait_for_something%*s: Action %d busy (fd=%d): remove skipped\n",
					    2*current_wait_level,"",
					    i, actions[i].fd)); 
			} else {		       
			    remove_action(i);
			    break;
			}
		    }
		}
	    }


	    
	    if (timed_task_handle_count > 0) {
		int w;
		
		for (w = timed_task_handle_count; w > 0 ; w=i) {
		    i = w-1;
		    
		    if (!timed_task_handles[i]) {
			/* remove empty slots */
			
			remove_timed_task(i);
		    }
		}
	    }
	}
	
	if (real_wait_error == result) {
	    switch (err) {
		int a;
		
	    case 0:
		break;
	    case EINTR:
		if (cd) {
		    if (is_canceled(cd)) {
			DPRINT(Debug,4,(&Debug,
					"wait_for_something%*s: ** waiting canceled on interrupt\n",
					2*current_wait_level,""
					));
			done = 1;
		    } else {
			DPRINT(Debug,12,(&Debug,
					 "wait_for_something%*s: seen EINTR and have cancel data, but waiting not canceled\n",
					 2*current_wait_level,""
					 ));
		    }
		} else {
		    DPRINT(Debug,12,(&Debug,
				     "wait_for_something%*s: seen EINTR but not cancel data\n",
				     2*current_wait_level,""
				     ));
		}
		if (wait_flags &  WAIT_FOR_intr) {
		     DPRINT(Debug,4,(&Debug,
				     "wait_for_something%*s: ** waiting interrupted\n",
				     2*current_wait_level,""
				     ));
		    done = 1;
		}		
		break;
	    case EBADF:

		 DPRINT(Debug,4,(&Debug,
				 "wait_for_something%*s: Bad file handle\n",
				 2*current_wait_level,""
				 ));

		 for (a = 0; a < actions_count; a++) {

		     if (actions[a].badfd) {
			 DPRINT(Debug,4,(&Debug,
					 "wait_for_something%*s: action %d (fd=%d): marked for bad fd\n",
					 2*current_wait_level,"",
					 a,actions[a].fd));
		     } else {		     
			 struct stat  ST;
			 int r;
			 			 			 
			 r = fstat(actions[a].fd,&ST);
			 
			 if (-1 == r) {
			     int err1 = errno;
			     
			     DPRINT(Debug,4,(&Debug,
					     "wait_for_something%*s: action %d (fd=%d): stat errno=%d (%s)\n",
					     2*current_wait_level,"",
					     a,actions[a].fd,err1,strerror(err1)));
			     
			     if (err1 == EBADF)
				 actions[a].badfd = 1;
			 }
		     }
		 }

		 /* FALLTHRU */
	    default:
		DPRINT(Debug,4,(&Debug,
				"wait_for_something%*s: Quiting; Unexpected error:  errno=%d (%s)\n",
				2*current_wait_level,"",
				err,strerror(err)));
		done = 1;

		if (show_wait_error) {
		    show_wait_error = 0;

		    lib_transient(CATGETS(elm_msg_cat, MeSet,
					  MeScheduleErrorWait,
					  "Unexpected error when waiting event: %s (errno=%d)"),
				  strerror(err),err);
		    
		}
		 
		break;

		
	    }
	}

	DPRINT(Debug,12,(&Debug,
			 "wait_for_something%*s: loop end, result=%d",
			 2*current_wait_level,"",
			 result));
	switch (result) {
	case real_wait_next_timeout: DPRINT(Debug,12,(&Debug," (real_wait_next_timeout)"));     break;
	case real_wait_error:        DPRINT(Debug,12,(&Debug," (real_wait_error"));
	    if (err) {
		DPRINT(Debug,12,(&Debug,"; errno %d (%s)",
				 err,strerror(err)));
	    }
	    DPRINT(Debug,12,(&Debug,")"));

	    break;
	case real_wait_none:         DPRINT(Debug,12,(&Debug," (real_wait_none)"));             break;
	case real_wait_succeed:      DPRINT(Debug,12,(&Debug," (real_wait_succeed)"));          break;
	}    
	DPRINT(Debug,12,(&Debug,", done=%d\n",
			 done));
	
    } while (result < real_wait_none
	     && !done);
    
    if (resulttime) {
	char * current_s =  schedule_timeout_string(&current);

	if (current_s) {
	    DPRINT(Debug,10,(&Debug,"wait_for_something%*s: wait end time is %s\n",
			     2*current_wait_level,"",
			     current_s));
	    free(current_s);
	}
	
	*resulttime = current;
    }

    DPRINT(Debug,10,(&Debug,
		     "wait_for_something%*s: result=%d",
		     2*current_wait_level,"",
		     result));
    switch (result) {
    case real_wait_next_timeout: DPRINT(Debug,10,(&Debug," (real_wait_next_timeout)"));     break;
    case real_wait_error:        DPRINT(Debug,10,(&Debug," (real_wait_error)"));
	ret = wait_for_error;
	break;
    case real_wait_none:         DPRINT(Debug,10,(&Debug," (real_wait_none)"));
	ret = wait_for_none;
	break;
    case real_wait_succeed:      DPRINT(Debug,10,(&Debug," (real_wait_succeed)"));
	ret = wait_for_done;
	break;
    }        
    DPRINT(Debug,10,(&Debug,"\n"));
    
    if (wait_level != current_wait_level+1) {
	DPRINT(Debug,10,(&Debug,
			 "wait_for_something%*s: wait_level = %d != %d\n",
			 2*current_wait_level,"",
			 wait_level, current_wait_level+1));
    }
    
    wait_level = current_wait_level;

    DPRINT(Debug,10,(&Debug,"wait_for_something%*s=%d",
		     2*current_wait_level,"",
		     ret));
    switch (ret) {
    case wait_for_error:  DPRINT(Debug,10,(&Debug," (wait_for_error)"));      break;
    case wait_for_none:   DPRINT(Debug,10,(&Debug," (wait_for_none)"));       break;
    case wait_for_done:   DPRINT(Debug,10,(&Debug," (wait_for_done)"));       break;
    }
    if (err) {
	DPRINT(Debug,10,(&Debug,"; errno %d (%s)",
			 err,strerror(err)));
    }
    DPRINT(Debug,10,(&Debug,"\n"));

    if (err_p)
	*err_p = err;
    errno = err;
    
    return ret;
}

/* Got alarm --
   Perhaps jumped out from wait_* routines() 
   or arrived via setjump
*/
void wait_got_jump()
{
    int i;
    
    DPRINT(Debug,4,(&Debug,
		    "wait_got_jump:      INTERRUPTED\n"));
    DPRINT(Debug,4,(&Debug,
		    "                    wait level %d\n",
		    wait_level));

    wait_level      = 0;
    wait_can_signal = 0;

    for (i = 0; i < actions_count; i++) {

	if (actions[i].busy) {
	    DPRINT(Debug,20,(&Debug,
			     "wait_got_jump: action for %d (action %d) was busy, resetting\n",
			     actions[i].fd,i));
	    actions[i].busy = 0;
	}
    }

    for (i = 0; i < timed_task_handle_count; i++) {
	if (timed_task_handles[i]) {
	    if (TIMED_TASK_HANDLE_magic != timed_task_handles[i]->magic)
		panic("SCHEDULE PANIC",__FILE__,__LINE__,
		      "wait_got_jump",
		      "Bad magic number (timed_task_handle)",0);
	    
	    if (timed_task_handles[i]->busy) {
		DPRINT(Debug,20,(&Debug,
				 "wait_got_jump: timed task %d  (task handle %p) was busy, resetting\n",
				 i,timed_task_handles[i]));

		timed_task_handles[i]->busy = 1;
	    }
	}
    }
}

enum wait_for_status wait_for_action(action)
     action_routine * action;
{
    enum wait_for_status ret;
    int err = 0;
    
    DPRINT(Debug,4,(&Debug,
		    "wait_for_action:    START | action=%p\n"));
    ret=wait_for_something(action,-1,NULL,NULL,&NO_schedule_timelimit,
			   NULL,&err,0);
    DPRINT(Debug,4,(&Debug,
		    "wait_for_action=%d END (errno %d)\n",
		    ret,err)); 

    errno = err;
    return ret;
}

enum wait_for_status wait_for_action_c(action,cd)
     action_routine * action;
     struct cancel_data *cd;
{
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,
		    "wait_for_action_c:    START | action=%p\n",action)); 
    ret=wait_for_something(action,-1,cd,NULL,&NO_schedule_timelimit,NULL,
			   &err,0);
    DPRINT(Debug,4,(&Debug,
		    "wait_for_action_c=%d END (errno %d)\n",
		    ret,err)); 

    errno = err;
    return ret;
}


enum wait_for_status wait_for_timeout(seconds)
     int seconds;
{
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,
		    "wait_for_timeout:    START | seconds=%d\n",seconds)); 
    ret = wait_for_something(no_action_routine,seconds,NULL,NULL,&NO_schedule_timelimit,
			     NULL,&err,0);
    DPRINT(Debug,4,(&Debug,
		    "wait_for_timeout=%d END (errno %d) \n",
		    ret,err));
    errno = err;
    return ret;
}

enum wait_for_status wait_for_timeout_c(seconds,cd)
     int seconds;
     struct cancel_data *cd;
{
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,
		    "wait_for_timeout_c:    START | seconds=%d\n",seconds)); 
    ret = wait_for_something(no_action_routine,seconds,cd,NULL,&NO_schedule_timelimit,
			     NULL,&err,0);
    DPRINT(Debug,4,(&Debug,
		    "wait_for_timeout_c=%d END (errno %d) \n",
		    ret,err));
    errno = err;
    return ret;
}

enum wait_for_status wait_for_timeout_f(seconds,wait_flags,err_p)
     int   seconds;
     int   wait_flags;
     int * err_p;
{
    enum wait_for_status ret;
    int err = 0;

     DPRINT(Debug,4,(&Debug,
		     "wait_for_timeout_f:    START | seconds=%d waitflags=%d",
		     seconds,wait_flags)); 
     if (WAIT_FOR_intr & wait_flags) {
	 DPRINT(Debug,4,(&Debug,"  WAIT_FOR_intr"));
     }
     if (WAIT_FOR_handle_sigchld & wait_flags) {
	 DPRINT(Debug,4,(&Debug," WAIT_FOR_handle_sigchld"));
     }     
     DPRINT(Debug,4,(&Debug,"\n"));
     
     ret = wait_for_something(no_action_routine,seconds,NULL,NULL,&NO_schedule_timelimit,
			      NULL,&err,wait_flags);
     
     DPRINT(Debug,4,(&Debug,
		    "wait_for_timeout_f=%d END (errno %d) \n",
		    ret,err));
    errno = err;
    if (err_p)
	*err_p = err;
    return ret;    
}

/* Updates *resulttime */
enum wait_for_status wait_for_action_or_timeout_settime(action,seconds,resulttime)
     action_routine * action;
     int              seconds;
     struct schedule_timelimit *resulttime;
{
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,
		    "wait_for_action_or_timeout_settime:    START | action=%p seconds=%d\n",
		    action,seconds)); 
    ret = wait_for_something(action,seconds,NULL,NULL,&NO_schedule_timelimit,
			     resulttime,&err,0);
    DPRINT(Debug,4,(&Debug,
		    "wait_for_action_or_timeout_settime=%d END (errno %d)\n",
		    ret,err));
    errno = err;
    return ret;

}

/* Updates *resulttime */
enum wait_for_status wait_for_action_or_timeout_settime_c(action,seconds,resulttime,cd)
     action_routine * action;
     int              seconds;
     struct schedule_timelimit *resulttime;
     struct cancel_data *cd;
{
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,
		    "wait_for_action_or_timeout_settime_c:    START | action=%p seconds=%d\n",
		    action,seconds)); 
    ret = wait_for_something(action,seconds,cd,NULL,&NO_schedule_timelimit,
			     resulttime,&err,0);
    DPRINT(Debug,4,(&Debug,
		    "wait_for_action_or_timeout_settime_c=%d END (errno %d)\n",
		    ret,err));
    errno = err;
    return ret;

}


enum wait_for_status  wait_for_action_or_timeout(action,seconds)
     action_routine * action;
     int seconds;
{
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,
		    "wait_for_action_or_timeout:    START | action=%p seconds=%d\n",
		    action,seconds)); 
    ret = wait_for_something(action,seconds,NULL,NULL,&NO_schedule_timelimit,
			     NULL,&err,0);
    DPRINT(Debug,4,(&Debug,
		    "wait_for_action_or_timeout=%d END (errno %d)\n",
		    ret,err));

    errno = err;
    return ret;
}

enum wait_for_status wait_for_action_or_timeout_f(action,seconds,wait_flags,err_p)
     action_routine * action;
     int seconds /* -1 no timeout */;
     int   wait_flags;
     int * err_p;
 {
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,
		    "wait_for_action_or_timeout_f:    START | action=%p seconds=%d waitflags=%d",
		    action,seconds,wait_flags)); 
     if (WAIT_FOR_intr & wait_flags) {
	 DPRINT(Debug,4,(&Debug,"  WAIT_FOR_intr"));
     }
     if (WAIT_FOR_handle_sigchld & wait_flags) {
	 DPRINT(Debug,4,(&Debug," WAIT_FOR_handle_sigchld"));
     }     
     DPRINT(Debug,4,(&Debug,"\n"));
     
     ret = wait_for_something(action,seconds,NULL,NULL,&NO_schedule_timelimit,
			      NULL,&err,wait_flags);
     
     DPRINT(Debug,4,(&Debug,
		     "wait_for_action_or_timeout_f=%d END (errno %d)\n",
		     ret,err));

    errno = err;
    if (err_p)
	*err_p = err;
    return ret;    
}
	 

enum wait_for_status wait_for_action_or_timeout_c(action,seconds,cd)
     action_routine * action;
     int seconds;
     struct cancel_data *cd;
{
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,
		    "wait_for_action_or_timeout_c:    START | action=%p seconds=%d\n",
		    action,seconds,&err)); 
    ret = wait_for_something(action,seconds,cd,NULL,&NO_schedule_timelimit,
			     NULL,&err,0);
    DPRINT(Debug,4,(&Debug,
		    "wait_for_action_or_timeout_c=%d END (errno %d)\n",
		    ret,err));
    errno = err;
    return ret;
}

enum wait_for_status wait_for_any_action()
{
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,"wait_for_any_action:    START\n")); 
    ret=wait_for_something(ANY_ACTION,-1,NULL,NULL,&NO_schedule_timelimit,
			   NULL,&err,0);
    DPRINT(Debug,4,(&Debug,"wait_for_any_action=%d END (errno %d)\n",
		    ret,err)); 

    errno = err;
    return ret;
}

enum wait_for_status wait_for_any_action_settime(resulttime)
     struct schedule_timelimit *resulttime;
{
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,"wait_for_any_action_settime:    START\n")); 
    ret=wait_for_something(ANY_ACTION,-1,NULL,NULL,&NO_schedule_timelimit,
			   resulttime, &err,0);
    DPRINT(Debug,4,(&Debug,"wait_for_any_action_settime=%d END (errno %d)\n",
		    ret,err)); 

    errno = err;
    return ret;
}



enum wait_for_status wait_for_any_action_c(cd)
     struct cancel_data *cd;
{
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,"wait_for_any_action_c:    START\n")); 
    ret=wait_for_something(ANY_ACTION,-1,cd,NULL,&NO_schedule_timelimit,
			   NULL,&err,0);
    DPRINT(Debug,4,(&Debug,"wait_for_any_action_c=%d END (errno %d)\n",
		    ret,err)); 

    errno = err;
    return ret;
}

/* Updates *resulttime */
enum wait_for_status wait_for_any_action_settime_c(resulttime,cd)
     struct schedule_timelimit * resulttime;
     struct cancel_data        * cd;
{
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,"wait_for_any_action_settime_c:    START\n")); 
    ret=wait_for_something(ANY_ACTION,-1,cd,NULL,&NO_schedule_timelimit,
			   resulttime,&err,0);
    DPRINT(Debug,4,(&Debug,"wait_for_any_action_settime_c=%d END (errno %d)\n",
		    ret,err)); 

    errno = err;
    return ret;
}

enum wait_for_status wait_for_any_action_or_timeout(seconds)
     int seconds;
{
    enum wait_for_status ret;
    int err = 0;
    
    DPRINT(Debug,4,(&Debug,"wait_for_any_action_or_timeout:    START | seconds=%d\n",seconds)); 
    ret = wait_for_something(ANY_ACTION,seconds,NULL,NULL,&NO_schedule_timelimit,
			     NULL,&err,0);
    DPRINT(Debug,4,(&Debug,"wait_for_any_action_or_timeout=%d END (errno %d)\n",
		    ret,err));

    errno = err;
    return ret;
}

enum wait_for_status wait_for_any_action_or_timeout_c(seconds,cd)
     int seconds;
     struct cancel_data *cd;
{
    enum wait_for_status ret;
    int err = 0;
    
    DPRINT(Debug,4,(&Debug,"wait_for_any_action_or_timeout_c:    START | seconds=%d\n",seconds)); 
    ret = wait_for_something(ANY_ACTION,seconds,cd,NULL,&NO_schedule_timelimit,
			     NULL, &err,0);
    DPRINT(Debug,4,(&Debug,"wait_for_any_action_or_timeout_c=%d END (errno %d)\n",
		    ret,err));
    errno = err;
    return ret;
}

/* Return 1 if task occured */
enum wait_for_status wait_for_task(task)
     struct timed_task_handle  * task;
{
    enum wait_for_status ret;
    int err = 0;

    DPRINT(Debug,4,(&Debug,"wait_for_task:    START | timed_task_handle=%p",task));

    if (task) {
	if (TIMED_TASK_HANDLE_magic != task->magic)
	    panic("SCHEDULE PANIC",__FILE__,__LINE__,
		  "wait_for_task",
		  "Bad magic number",0);
 
	DPRINT(Debug,4,(&Debug," timed_task %p",
			task->task));
    }
    DPRINT(Debug,4,(&Debug,"\n"));
    
    ret = wait_for_something(ANY_ACTION,-1,NULL,task,&NO_schedule_timelimit,
			     NULL,&err,0);
    DPRINT(Debug,4,(&Debug,"wait_for_task=%d END (errno %d)\n",
		    ret,err)); 

    errno = err;
    return ret;
}

enum wait_for_status wait_for_action_or_deadline_c(action,deadline,cd)
     action_routine                  * action;
     const struct schedule_timelimit * deadline;
     struct cancel_data *cd;
{
    enum wait_for_status ret;
    int err = 0;
    char *X =  schedule_timeout_string(deadline);
    
    DPRINT(Debug,4,(&Debug,"wait_for_action_or_deadline_c:    START | deadline=%s\n",
		    X ? X : "n/a"));
    ret = wait_for_something(ANY_ACTION,-1,cd,NULL,deadline,NULL,&err,0);
    DPRINT(Debug,4,(&Debug,"wait_for_action_or_deadline_c=%d END (errno %d) \n",
		    ret,err)); 

    if (X)
	free(X);
    
    errno = err;
    return ret;
}

/* Updates *resulttime */
enum wait_for_status wait_for_action_or_deadline_settime(action,deadline,resulttime)
     action_routine                  * action;
     const struct schedule_timelimit * deadline;
     struct schedule_timelimit       * resulttime;
{
    enum wait_for_status ret;
    int err = 0;
    char *X =  schedule_timeout_string(deadline);
    
    DPRINT(Debug,4,(&Debug,"wait_for_action_or_deadline_settime:    START | deadline=%s\n",
		    X ? X : "n/a"));
    ret = wait_for_something(ANY_ACTION,-1,NULL,NULL,deadline,resulttime,&err,0);
    DPRINT(Debug,4,(&Debug,"wait_for_action_or_deadline_settime=%d END (errno %d) \n",
		    ret,err)); 

    if (X)
	free(X);
    
    errno = err;
    return ret;

}

/* Updates *resulttime */
enum wait_for_status  wait_for_action_or_deadline_settime_c(action,deadline,resulttime,cd)
     action_routine                  * action;
     const struct schedule_timelimit * deadline;
     struct schedule_timelimit       * resulttime;
     struct cancel_data              * cd;
{
    enum wait_for_status ret;
    int err = 0;
    char *X =  schedule_timeout_string(deadline);
    
    DPRINT(Debug,4,(&Debug,"wait_for_action_or_deadline_settime_c:    START | deadline=%s\n",
		    X ? X : "n/a"));
    ret = wait_for_something(ANY_ACTION,-1,cd,NULL,deadline,resulttime,&err,0);
    DPRINT(Debug,4,(&Debug,"wait_for_action_or_deadline_settime_c=%d END (errno %d) \n",
		    ret,err)); 

    if (X)
	free(X);

    errno = err;
    return ret;
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
