static char rcsid[] = "@(#)$Id: sb_file.c,v 2.19 2023/12/13 16:55:32 hurtta Exp $";

/*************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.19 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ************************************************************************/

#include "elm_defs.h"
#include "s_me.h"
#include "sb_imp.h"
#include "pg_range_imp.h"
#include "pg_lineext_imp.h"

/* Needs STRING_magic */
#include "elmme-magic.h"
#include "pg_params_imp.h"

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

DEBUG_VAR(Debug,__FILE__,"charset");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

#define SB_file_hdl_magic		0xEB18


static struct sb_file_hdl {
    unsigned short magic;       /* SB_file_hdl_magic */

    char * filename;
    FILE * fh;

} * malloc_sb_file_hdl P_((int *error));

static struct sb_file_hdl * malloc_sb_file_hdl(error)
     int *error;
{
    struct sb_file_hdl * ret = safe_zero_alloc(sizeof (*ret));
    const char *tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    static int count =  0;
    int i;
    int err = 0;
    
    if (!tmp)
	tmp = "/tmp/";
   
    for (i = 0; i < 10; i++) {
	count++;
	
	ret->filename = elm_message(FRM("%selmsb-%d-%d"),
				    tmp, getpid (),
				    count);
	
	ret->fh = safeopen_rdwr(ret->filename,&err);
	if (!ret->fh) {
	    DPRINT(Debug,25,(&Debug,
			     "stringbuffer: safeopen_rdwr: %s: %s (errno %d)\n",
			     ret->filename,
			     strerror(err),err));
	    free(ret->filename);
	    ret->filename = NULL;

	    if (error)
		*error = err;
	} else
	    break;
	
    } 

    if (!ret->fh) {
	free(ret);

	return NULL;
    }

    if (error)
	*error = 0;

    if (0 == unlink (ret->filename)) {
	DPRINT(Debug,25,(&Debug,"malloc_sb_file_hdl: %s: Unlinked\n",
			 ret->filename));
    } else {
	int err UNUSED_VAROK = errno;
	
	DPRINT(Debug,1,(&Debug,
			  "malloc_sb_file_hdl: %s: %s (errno=%d)\n",
			  ret->filename,strerror(err),err));		
    }

    ret->magic = SB_file_hdl_magic;

    return ret;
}

static void free_sb_file_hdl P_((struct sb_file_hdl **hdl));
static void free_sb_file_hdl(hdl)
     struct sb_file_hdl **hdl;
{
    if (SB_file_hdl_magic != (*hdl)->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"free_sb_file_hdl",
	      "Bad magic number (sb_file_hdl)",0);

    if ((*hdl)->fh) {
	fclose((*hdl)->fh);
	(*hdl)->fh = NULL;
    }

    if ((*hdl)->filename) {
	free((*hdl)->filename);
	(*hdl)->filename = NULL;
    }

    (*hdl)->magic = 0;  /* Invalidate */
    free(*hdl);
    *hdl = NULL;
}

/* ----------------------------------------------------------- */

#define SB_file_ext_offsets_magic	0xEB19

struct sb_file_ext_offsets {
    unsigned short            magic;  /* SB_file_ext_offsets */

    int refcount;

    long * lineext_offsets;
    int    lineext_count;
} * malloc_sb_file_ext_offsets P_((long offset));

struct sb_file_ext_offsets * malloc_sb_file_ext_offsets(offset) 
     long offset;
{
    struct sb_file_ext_offsets * ret =  safe_zero_alloc(sizeof (* ret));

    ret->lineext_offsets = 
	safe_malloc(sizeof (ret->lineext_offsets[0]));
    ret->lineext_offsets[0] = offset;
    ret->lineext_count = 1;

    ret->refcount = 1;
    ret->magic = SB_file_ext_offsets_magic;

    return ret;
}

static void add_sb_file_ext_offset P_((struct sb_file_ext_offsets *offsets,
				       long offset));
static void add_sb_file_ext_offset(offsets,offset)
     struct sb_file_ext_offsets *offsets;
     long offset;
{
    int off_ptr;

    if (SB_file_ext_offsets_magic != offsets->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"add_sb_file_ext_offset",
	      "Bad magic number",0);

    if (offsets->lineext_count < 1) {
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"add_sb_file_ext_offset",
	      "No offset",0);
    }

    off_ptr = offsets->lineext_count;
    
    offsets->lineext_offsets =
	safe_array_realloc(offsets->lineext_offsets, (offsets->lineext_count +1),
			   sizeof (offsets->lineext_offsets[0]));
    
    offsets->lineext_offsets[off_ptr] = offset;
    offsets->lineext_count++;
    
}
    
static void inc_sb_file_ext_offsets_refcount P_((struct sb_file_ext_offsets *offsets));
static void inc_sb_file_ext_offsets_refcount(offsets)
     struct sb_file_ext_offsets *offsets;
{
    if (SB_file_ext_offsets_magic != offsets->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"inc_sb_file_ext_offsets_refcount",
	      "Bad magic number",0);

    offsets->refcount++;
}

static void free_sb_file_ext_offsets  P_((struct sb_file_ext_offsets **offsets));
static void free_sb_file_ext_offsets(offsets)
     struct sb_file_ext_offsets **offsets;
{
    if (SB_file_ext_offsets_magic != (*offsets)->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "free_sb_file_ext_offsets",
	      "Bad magic number",0);

    (*offsets)->refcount--;

    if ((*offsets)->refcount > 0) {  /* Do not free */
	(*offsets) = NULL;           /* Refefence count for this pointer is     */
	return;			     /* decrement, so this must have be reset   */	 
    }

    if ((*offsets)->lineext_offsets) {
	free((*offsets)->lineext_offsets);
	(*offsets)->lineext_offsets = NULL;
    }
    (*offsets)->lineext_count = 0;

    (*offsets)->magic = 0;  /* Invalidate */
    free(*offsets);
    *offsets = NULL;
}

/* ----------------------------------------------------------- */

#define SB_in_file_magic     0xEB09

struct sb_in_file {
    unsigned short magic;       /* SB_in_file_magic */

    struct sb_file_hdl * hdl;

    struct sb_line_offsets {
	long offset;

	struct sb_file_ext_offsets * ext_offsets;

    } * offsets;
    int offset_count;

    struct sb_pager_range {
	struct pgrng_bfr_binding *pager_range;  /* cache */
	long offset;

	long * param_value_offsets ;
	int param_value_count;
	
    } * pager_range;
    int pager_range_count;


};

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

#if 0
static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}
#endif

S_(sb_init_buffer sb_init_file)
static int sb_init_file P_((struct stringbuffer *buffer));
static int sb_init_file(buffer)
     struct stringbuffer *buffer;
{
    int err = 0;

    buffer->a.file = safe_zero_alloc(sizeof (* (buffer->a.file)));
    
    buffer->a.file->magic = SB_in_file_magic;

    buffer->a.file->offset_count = 0;


    buffer->a.file->pager_range = NULL;
    buffer->a.file->pager_range_count = 0;
    
    buffer->a.file->hdl  = malloc_sb_file_hdl(&err);
    if (!buffer->a.file->hdl) {
	
	DPRINT(Debug,25,(&Debug,
			 "stringbuffer: %s (errno %d)\n",
			 strerror(err),err));
	return 0;
    }

    DPRINT(Debug,1,(&Debug,
		    "stringbuffer: using temp file (%s)\n",
		    buffer->a.file->hdl->filename));
    return 1;
}

S_(sb_free_buffer sb_free_file)
static void sb_free_file P_((struct stringbuffer *buffer));
static void sb_free_file(buffer)
     struct stringbuffer *buffer;
{
    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"sb_free_file",
	      "Bad magic number",0);

    if (buffer->a.file->hdl)
	free_sb_file_hdl(& (buffer->a.file->hdl));

    if (buffer->a.file->offsets) {
	int i;

	for (i = 0; i < buffer->a.file->offset_count; i++) {
	    if (buffer->a.file->offsets[i].ext_offsets) 
		free_sb_file_ext_offsets(& (buffer->a.file->offsets[i].ext_offsets));
	}

	free(buffer->a.file->offsets);
	buffer->a.file->offsets = NULL;
    }
    buffer->a.file->offset_count = 0;

    if (buffer->a.file->pager_range) {
	int i;

	for (i = 0; i < buffer->a.file->pager_range_count; i++) {
	    if (buffer->a.file->pager_range[i].pager_range) {
		pgrng_bfr_bnd_clear_stringbuffer(& (buffer->a.file->
						    pager_range[i].pager_range),
						 buffer,i);

	    }
	    
	    if (buffer->a.file->pager_range[i].param_value_offsets) {		
		free(buffer->a.file->pager_range[i].param_value_offsets);
		buffer->a.file->pager_range[i].param_value_offsets = NULL;		
	    }
	    buffer->a.file->pager_range[i].param_value_count = 0;
	    
	}
	free(buffer->a.file->pager_range);
	buffer->a.file->pager_range = NULL;
    }
    buffer->a.file->pager_range_count = 0;


    buffer->a.file->magic = 0; /* Invalidate */
    free(buffer->a.file);
    buffer->a.file = NULL;
}

void put_string_serialization_to_file(fh,Str)
     FILE *fh; 
     const struct string *Str;
{
    if (RAW_BUFFER == get_string_type(Str)) {
	char * s;
	int l1;

	bytestream_from_string(Str,&s,&l1);
	
	fprintf(fh,"R%d:",l1);
	fputc('\0',fh);  /* Terminating \0 of null charset */

	/* Include ending \0 to string */
	fwrite(s,1,l1+1,fh);
	free(s);

    } else {
	const char * MIME_name = get_string_MIME_name(Str);
	
	if (MIME_name) {
	    char * s;
	    int l1;
	    int l2 = strlen(MIME_name);
	    
	    bytestream_from_string(Str,&s,&l1);
	    
	    fprintf(fh,"%d:%d:",l2,l1);	
	    /* Include ending \0 to string */
	    fwrite(MIME_name,1,l2+1,fh);
	    /* Include ending \0 to string */
	    fwrite(s,1,l1+1,fh);
	    free(s);
	} else { 
	    /* No charset name available ... */
	    struct string *str = convert_string(system_charset,Str,0);
	    char * s;
	    int l1;
	    
	    bytestream_from_string(str,&s,&l1);
	    
	    fprintf(fh,":%d:",l1);
	    fputc('\0',fh);  /* Terminating \0 of null charset */
	    /* Include ending \0 to string */
	    fwrite(s,1,l1+1,fh);
	    free(s);
	    free_string(&str);
	}
    }
}

static void put_out_entity_serialization_to_file P_((FILE *fh,
						 const struct out_entity *Ent));
static void put_out_entity_serialization_to_file(fh,Ent)
     FILE *fh; 
     const struct out_entity *Ent;
{
    int                    pg_flags = 0;
    const struct string  * reference_key = out_entity_reference_key(Ent,
								    &pg_flags);
    const struct string  * entity_value  = out_entity_text_value(Ent);
    const uint16           unicode_value = out_entity_unicode_value(Ent);

    fputc('$',fh);

    put_string_serialization_to_file(fh,reference_key);

    if (entity_value) {
	fputc('"',fh);
	put_string_serialization_to_file(fh,entity_value);
    }

    if (unicode_value != UNICODE_BAD_CHAR) {
	fprintf(fh,"#%d",unicode_value);
    }

    if (pg_flags) {
	fprintf(fh,",%d",pg_flags);
    }

    fputc(';',fh);    
}

S_(sb_add_line_to_stringbuffer sb_add_line_to_file)
static void sb_add_line_to_file P_((struct stringbuffer *buffer,
				    const struct string *string,
				    int pg_flags,
				    struct pager_range *range,
				    struct pgrng_bfr_binding  * binding));
static void sb_add_line_to_file(buffer,string,pg_flags,range,
				binding)
     struct stringbuffer *buffer;
     const struct string *string;
     int pg_flags;
     struct pager_range *range;               /* NOT USED */
     struct pgrng_bfr_binding  * binding;
{
    int ptr;
    enum syscall_status r;

    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_line_to_file",
	      "Bad magic number",0);

    ptr = buffer->a.file->offset_count;

    if (SB_file_hdl_magic != buffer->a.file->hdl->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_line_to_file",
	      "Bad magic number (sb_file_hdl)",0);


    r = fseek(buffer->a.file->hdl->fh,0,SEEK_END);
    
    switch (r) {
	int err UNUSED_VAROK;
    case syscall_success:
	DPRINT(Debug,14,(&Debug,
			 "stringbuffer: fseek to end: %s\n",
			 buffer->a.file->hdl->filename));
	break;
    case syscall_error:
	err  = errno;
	DPRINT(Debug,1,(&Debug,
			"stringbuffer: fseek (flush) failure: %s: %s (errno %d)\n",
			buffer->a.file->hdl->filename,
			strerror(err),err));
	
	return;
    }
    
    buffer->a.file->offsets = 
	safe_array_realloc(buffer->a.file->offsets,
			   (buffer->a.file->offset_count +1),
			   sizeof (buffer->a.file->offsets[0]));

    /* defined in hdrs/defs.h */
    bzero((void *)& (buffer->a.file->offsets[ptr]),
	  sizeof (buffer->a.file->offsets[ptr]));
    buffer->a.file->offsets[ptr].offset = 
	ftell(buffer->a.file->hdl->fh);
    buffer->a.file->offsets[ptr].ext_offsets = NULL;
    buffer->a.file->offset_count++;
    
    if (binding) {
	int range_index;
	struct stringbuffer *bf = 
	    pgrng_bfr_bnd_get_stringbuffer(binding,&range_index);

	if (bf != buffer)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_add_line_to_file",
		  "pager_range (binding) added to wrong stringbuffer",0);

	if (range_index < 0 || 
	    range_index >=  buffer->a.file->pager_range_count ||
	    buffer->a.file->pager_range[range_index].pager_range != binding)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_add_line_to_file",
		  "bad range_index",0);
	    
	fprintf(buffer->a.file->hdl->fh,"r%d,",range_index);

	free_stringbuffer(&bf);
    }
    
    fprintf(buffer->a.file->hdl->fh,"%d",pg_flags);
    /* : starts string */
    fputc(':', buffer->a.file->hdl->fh);
    put_string_serialization_to_file(buffer->a.file->hdl->fh,string);
}


static void sb_plext_write_prefix  P_((struct stringbuffer *buffer,
				       int pg_flags,
				       struct pager_range *range,
				       struct pgrng_bfr_binding  * binding));
static void sb_plext_write_prefix(buffer,pg_flags,range,binding)
     struct stringbuffer *buffer;
     int pg_flags;
     struct pager_range *range;
     struct pgrng_bfr_binding  * binding;
{
    int ptr;

    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_plext_prefix_prefix",
	      "Bad magic number (stringbuffer)",0);

    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_plext_write_prefix",
	      "Bad magic number (sb_in_file)",0);

    if (buffer->a.file->offset_count < 0) {
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_plext_prefix_prefix",
	      "No line",0);
    }

    ptr     = buffer->a.file->offset_count-1;

    if (! buffer->a.file->offsets[ptr].ext_offsets)
	buffer->a.file->offsets[ptr].ext_offsets = 
	    malloc_sb_file_ext_offsets(ftell(buffer->a.file->hdl->fh));
    else
	add_sb_file_ext_offset(buffer->a.file->offsets[ptr].ext_offsets,
			       ftell(buffer->a.file->hdl->fh));
 
    if (binding) {
	int range_index;
	struct stringbuffer *bf = 
	    pgrng_bfr_bnd_get_stringbuffer(binding,&range_index);

	if (bf != buffer)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_plext_prefix_prefix",
		  "pager_range (binding) added to wrong stringbuffer",0);

	if (range_index < 0 || 
	    range_index >=  buffer->a.file->pager_range_count ||
	    buffer->a.file->pager_range[range_index].pager_range != binding)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_plext_prefix_prefix",
		  "bad range_index",0);
	    
	fprintf(buffer->a.file->hdl->fh,"r%d,",range_index);

	free_stringbuffer(&bf);
    }
   
    fprintf(buffer->a.file->hdl->fh,"%d",pg_flags);
}


S_(sb_add_lineext_to_stringbuffer sb_add_lineext_to_file)
static void sb_add_lineext_to_file P_((struct stringbuffer *buffer,
				      const struct string *string,
				      int pg_flags,
				       struct pager_range *range,
				      struct pgrng_bfr_binding  * binding));
static void sb_add_lineext_to_file(buffer,string,pg_flags,range,binding)
     struct stringbuffer *buffer;
     const struct string *string;
     int pg_flags;
     struct pager_range *range;
     struct pgrng_bfr_binding  * binding;
{

    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_lineext_to_file",
	      "Bad magic number",0);
    
    sb_plext_write_prefix(buffer,pg_flags,range,binding);
    
    if (SB_file_hdl_magic != buffer->a.file->hdl->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_lineext_to_file",
	      "Bad magic number (sb_file_hdl)",0);
        
    /* : starts string */
    fputc(':', buffer->a.file->hdl->fh);
    put_string_serialization_to_file(buffer->a.file->hdl->fh,string);
}


S_(sb_linecount_stringbuffer sb_linecount_file)
static int sb_linecount_file P_((const struct stringbuffer *ptr));
static int sb_linecount_file(ptr)
     const struct stringbuffer *ptr;
{
    if (SB_in_file_magic != ptr->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_linecount_file",
	      "Bad magic number",0);

    return ptr->a.file->offset_count;
}

static struct pager_range * sb_get_pager_range_from_file
  P_((struct stringbuffer * buffer,int index));


#define ADD_DBUG(ch) do {						\
	if (debug_len+1 < sizeof debug_buffer && EOF != ch) {		\
	    debug_buffer[debug_len++] = ch;				\
	    debug_buffer[debug_len] = '\0';				\
	}								\
    } while(0)


struct string * get_string_serialization_from_file(fh)
     FILE *fh;
{
    struct string *ret = NULL;

    char * bigs = NULL;
    int setlen = 0;
    int strlen = 0;
    charset_t sn = system_charset;
    int ERRORS = 0;

    unsigned char debug_buffer[100];
    int debug_len = 0;
    
    int c = fgetc(fh);
    size_t rlen,x;

    ADD_DBUG(c);
    
    if ('R' == c) {
	sn = RAW_BUFFER;
    } else {
	while (c >= '0' && c <= '9') {
	    setlen = setlen * 10 + c - '0';
	    c = fgetc(fh);
	    ADD_DBUG(c);
    
	}
	if (':' != c) {
	    DPRINT(Debug,1,(&Debug,
			    "get_string_serialization_from_file: charset name len do not end with ':'\n"));
	    
	    goto failure;
	}
    }

    c = fgetc(fh);
    ADD_DBUG(c);

    while (c >= '0' && c <= '9') {
	strlen = strlen * 10 + c - '0';
	c = fgetc(fh);
	ADD_DBUG(c);
    
    }
    if (':' != c) {
	DPRINT(Debug,1,(&Debug,
			"get_string_serialization_from_file: string len do not end with ':'\n"));

	goto failure;
    }

    bigs = safe_malloc(setlen+1+strlen+1);
    rlen = fread(bigs,1,setlen+1+strlen+1,fh);
    for (x = 0; x < rlen; x++)
	ADD_DBUG(bigs[x]);
    
    if (rlen !=
	setlen+1+strlen+1) {

	goto failure;
    }
    if (bigs[setlen] != '\0' ||
	bigs[setlen+1+strlen] != '\0') {

	goto failure;
    }
	    
    if (setlen != 0) {
	sn = MIME_name_to_charset(bigs,0);
	if (!sn) { 
	    DPRINT(Debug,1,(&Debug,
			    "get_string_serialization_from_file: Charset '%s' is not known (impossible?)\n",
			    bigs));

	    goto failure;
	}
    }

    ret = new_string(sn);
    
    add_streambytes_to_string(ret,strlen,s2us(&(bigs[setlen+1])),&ERRORS);
    
    if (ERRORS > 0) {
	DPRINT(Debug,1,(&Debug,
			"get_string_serialization_from_file: %d errors when adding string grom file\n",
			ERRORS));
    }

 failure:
    if (bigs)	    
	free(bigs);

    if (!ret) {
	DPRINT(Debug,10,(&Debug,"get_string_serialization_from_file: failure, buffer="));
	DEBUG_PRINT_BUFFER(Debug,10,debug_len, debug_buffer);
	DPRINT(Debug,10,(&Debug,"\n"));
    }
		 
    return ret;
}

struct out_entity * get_out_entity_serialization_from_file P_((FILE *fh));
struct out_entity * get_out_entity_serialization_from_file(fh)
     FILE *fh;
{
    struct out_entity * ret = NULL;
    struct string  * reference_key = NULL;
    struct string  * entity_value  = NULL;
    uint16           unicode_value = UNICODE_BAD_CHAR;
    int              pg_flags = 0;
    
    int c = fgetc(fh);

    if ('$' == c) {

        reference_key = get_string_serialization_from_file(fh);

	if (!reference_key) {
	    DPRINT(Debug,1,(&Debug,
			    "get_out_entity_serialization_from_file: out_entity reference key failed\n"));
	    goto failure;
	}
	
    } else {
	DPRINT(Debug,1,(&Debug,
			"get_out_entity_serialization_from_file: out_entity reference key not start with '$'\n"));
	goto failure;
    }

    c = fgetc(fh);

    if ('"' == c) {

	entity_value =  get_string_serialization_from_file(fh);

	if (!entity_value) {
	    DPRINT(Debug,1,(&Debug,
			    "get_out_entity_serialization_from_file: out_entity entity value failed\n"));
	    goto failure;
	}

	c = fgetc(fh);


    }

    if ('#' == c) {
	unicode_value = 0;

	c = fgetc(fh);

	while (c >= '0' && c <= '9') {
	    unicode_value  = unicode_value  * 10 + c - '0';
	    c = fgetc(fh);
		
	}	
    }

    if (',' == c) {
	pg_flags = 0;

   	c = fgetc(fh);

	while (c >= '0' && c <= '9') {
	    pg_flags  = pg_flags  * 10 + c - '0';
	    c = fgetc(fh);
	}	
    }

    if (';' != c) {
	DPRINT(Debug,1,(&Debug,
			"get_out_entity_serialization_from_file: out_entity does not ent with ';'\n"));
	goto failure;

    }

    ret =  new_out_entity(reference_key,entity_value,
			  unicode_value,pg_flags);
			  
    
 failure:

    if (reference_key)
	free_string(&reference_key);
    if (entity_value)
	free_string(&entity_value);

    if (!ret) {
	DPRINT(Debug,10,(&Debug,"get_out_entity_serialization_from_file: failure\n"));
    }
    
    return ret;
}

static struct pager_lineext * sb_file_copy_extension
     P_((struct stringbuffer *buffer,
	 int ptr));

S_(sb_get_line_from_stringbuffer sb_get_line_from_file)
static struct string *sb_get_line_from_file
     P_((struct stringbuffer *buffer,
	 int ptr, int *pg_flags,
	 struct pager_range **range,
	 struct pager_lineext **first_extension));
static struct string *sb_get_line_from_file(buffer,ptr,pg_flags,range,
					    first_extension)
     struct stringbuffer *buffer;
     int ptr;
     int *pg_flags;
     struct pager_range **range;
     struct pager_lineext **first_extension;
{
    int flags = 0;
    struct string * res = NULL;
    int c;
    int range_index = 0;
    int have_range = 0;
    enum syscall_status r;
    
    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_line_from_file",
	      "Bad magic number",0);

    if (ptr < 0 || ptr >= buffer->a.file->offset_count)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_line_from_file",
	      "Bad index",0);

    if (SB_file_hdl_magic != buffer->a.file->hdl->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_line_from_file",
	      "Bad magic number (sb_file_hdl)",0);

    r = fseek(buffer->a.file->hdl->fh,
	      buffer->a.file->offsets[ptr].offset,
	      SEEK_SET);
    switch(r) {
	int err UNUSED_VAROK;
    case syscall_success:
	DPRINT(Debug,14,(&Debug,
			 "stringbuffer: fseek to %ld: %s\n",
			 buffer->a.file->offsets[ptr].offset,
			 buffer->a.file->hdl->filename));
	break;
    case syscall_error:
	err = errno;

	DPRINT(Debug,1,(&Debug,
			"stringbuffer: fseek (flush) failure: %s: %s (errno %d)\n",
			buffer->a.file->hdl->filename,
			strerror(err),err));

	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_line_from_file",
	      "Failed to flush or seek file",0);
    }

    c = fgetc(buffer->a.file->hdl->fh);

    if (range && *range)
	free_pager_range(range);

    if (c == 'r') {
	have_range = 1;

	c = fgetc(buffer->a.file->hdl->fh);
	while (c >= '0' && c <= '9') {
	    range_index = range_index * 10 + c - '0';
	    c = fgetc(buffer->a.file->hdl->fh);
	}

	if (',' != c) {
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_get_line_from_file",
		  "Bad data read from file",0);
	}
	c = fgetc(buffer->a.file->hdl->fh);
    }

    while (c >= '0' && c <= '9') {
	flags = flags * 10 + c - '0';
	c = fgetc(buffer->a.file->hdl->fh);
    }
    if (':' != c) {
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_line_from_file",
	      "Bad data read from file",0);
    }
    *pg_flags = flags;

    res = get_string_serialization_from_file(buffer->a.file->hdl->fh);
    if (!res) {
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_line_from_file",
	      "Bad data read from file",0);
    }

    if (first_extension) {
	if (*first_extension)
	    free_pager_lineext(first_extension);

	if (buffer->a.file->offsets[ptr].ext_offsets) 
	    *first_extension = sb_file_copy_extension(buffer,ptr);	
    }

    /* May read from file */

    if (range && have_range)
	*range = sb_get_pager_range_from_file(buffer,range_index);
    
    return res;
}


S_(sb_walk_lineext_on_stringbuffer sb_walk_lineext_on_file)
static struct string *sb_walk_lineext_on_file
       P_((struct stringbuffer *buffer,
	   struct pager_lineext **walk_extension,
	   int *pg_flags,
	   struct pager_range **range));

S_(sb_add_pager_range_to_stringbuffer sb_add_pager_range_to_file)
static void sb_add_pager_range_to_file
        P_((struct stringbuffer * buffer,
	    struct pgrng_bfr_binding * binding,
	    struct pager_range * range));
static void sb_add_pager_range_to_file(buffer,binding,range)
     struct stringbuffer * buffer;
     struct pgrng_bfr_binding * binding;
     struct pager_range * range;
{
    int ptr;

    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_pager_range_to_file",
	      "Bad magic number",0);

    ptr = buffer->a.file->pager_range_count;

    buffer->a.file->pager_range = 
	safe_array_realloc(buffer->a.file->pager_range,
			   (buffer->a.file->pager_range_count +1),
			   sizeof (buffer->a.file->pager_range[0]));

    if (SB_file_hdl_magic != buffer->a.file->hdl->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_pager_range_to_file",
	      "Bad magic number (sb_file_hdl)",0);

    buffer->a.file->pager_range[ptr].offset = 
	ftell(buffer->a.file->hdl->fh);
    buffer->a.file->pager_range[ptr].pager_range = binding;

    buffer->a.file->pager_range[ptr].param_value_offsets = NULL;
    buffer->a.file->pager_range[ptr].param_value_count = 0;
    
    pgrng_bfr_bnd_set_stringbuffer(buffer->a.file->pager_range[ptr].
				   pager_range,
				   buffer,ptr);
    buffer->a.file->pager_range_count++;

    /* May need update offset, because 
       put_pager_range_serialization_to_file()
       may call this recursively */
    put_pager_range_serialization_to_file(buffer->a.file->hdl->fh,
					  range,buffer,
					  & buffer->a.file->pager_range[ptr].offset);
}


S_(sb_get_pager_range_from_stringbuffer sb_get_pager_range_from_file)
static struct pager_range * sb_get_pager_range_from_file(buffer,index)
     struct stringbuffer * buffer;
     int index;
{
    struct pager_range *ret = NULL;

    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_pager_range_from_file",
	      "Bad magic number",0);

    if (index < 0 || index >= buffer->a.file->pager_range_count)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_pager_range_from_file",
	      "Bad index",0);
	
    if (! buffer->a.file->pager_range[index].pager_range) {
	buffer->a.file->pager_range[index].pager_range =
	    new_pgrng_bfr_binding();
	pgrng_bfr_bnd_set_stringbuffer(buffer->a.file->pager_range[index].pager_range,
				       buffer,index);
    }
	
    ret = pgrng_bfr_bnd_get_range(buffer->a.file->pager_range[index].pager_range);

    if (!ret) {
	enum syscall_status r;
	
	if (SB_file_hdl_magic != buffer->a.file->hdl->magic)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_get_pager_range_from_file",
		  "Bad magic number (sb_file_hdl)",0);

	r = fseek(buffer->a.file->hdl->fh,
		  buffer->a.file->pager_range[index].offset,
		  SEEK_SET);
	
	switch (r) {
	    int err UNUSED_VAROK;
	    
	case syscall_success:
	    DPRINT(Debug,14,(&Debug,
			     "stringbuffer: fseek to %ld: %s\n",
			     buffer->a.file->pager_range[index].offset,
			     buffer->a.file->hdl->filename));
	    break;
	case syscall_error:	    
	    err = errno;

	    DPRINT(Debug,1,(&Debug,
			    "stringbuffer: fseek (flush) failure: %s: %s (errno %d)\n",
			    buffer->a.file->hdl->filename,
			    strerror(err),err));
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_get_pager_range_from_file",
		  "Failed to flush or seek file",0);
	}

	ret = get_pager_range_serialization_from_file(buffer->a.file->hdl->fh,buffer);

	if (!ret)
	   panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		 "sb_get_pager_range_from_file",
		 "No pager_range on file",0); 

	set_pager_range_stringbuffer(ret,
				     buffer->a.file->pager_range[index].pager_range);
    }

    return ret;
}

/* entity is processed as lineext */
S_(sb_add_entity_to_stringbuffer sb_add_entity_to_file)
static  enum stringbuffer_entity_result sb_add_entity_to_file
   P_((struct stringbuffer *buffer,
       const struct out_entity *oe,
       int pg_flags,struct pager_range *range,
       struct pgrng_bfr_binding  * binding));
static  enum stringbuffer_entity_result sb_add_entity_to_file(buffer,oe,pg_flags,range,
							      binding)
     struct stringbuffer *buffer;
     const struct out_entity *oe;
     int pg_flags;
     struct pager_range *range;
     struct pgrng_bfr_binding  * binding;
{

    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_entity_to_file",
	      "Bad magic number",0);
    
    sb_plext_write_prefix(buffer,pg_flags,range,binding);
    
    if (SB_file_hdl_magic != buffer->a.file->hdl->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_entity_to_file",
	      "Bad magic number (sb_file_hdl)",0);

    /* & starts out_entity */
    fputc('&', buffer->a.file->hdl->fh);
    put_out_entity_serialization_to_file(buffer->a.file->hdl->fh,oe);

    return sb_entity_succeed;
}

S_(sb_add_pager_param_to_stringbuffer sb_add_pager_param_to_file)
static void sb_add_pager_param_to_file 
    P_((struct stringbuffer       * buffer,
	struct pgrng_bfr_binding  * binding,
	struct pager_range        * range,
	enum pager_param_opcode     opcode,
	struct pager_param_value  * param_value));
static void sb_add_pager_param_to_file (buffer,binding,range,opcode,
				       param_value)
     struct stringbuffer       * buffer;
     struct pgrng_bfr_binding  * binding;
     struct pager_range        * range;
     enum pager_param_opcode     opcode;
     struct pager_param_value  * param_value;
{

    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_pager_param_to_file",
	      "Bad magic number",0);


    if (binding) {
	int range_index;
	struct stringbuffer *bf = 
	    pgrng_bfr_bnd_get_stringbuffer(binding,&range_index);
	struct sb_pager_range * sb_range;

	int ptr;
	
	if (bf != buffer)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_add_pager_param_to_file",
		  "pager_range (binding) added to wrong stringbuffer",0);

	if (range_index < 0 || 
	    range_index >=  buffer->a.file->pager_range_count ||
	    buffer->a.file->pager_range[range_index].pager_range != binding)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_add_pager_param_to_file",
		  "bad range_index",0);
	sb_range = &(buffer->a.file->pager_range[range_index]);

	ptr = sb_range->param_value_count;

	sb_range->param_value_offsets =
	    safe_array_realloc(sb_range->param_value_offsets,
			       sb_range->param_value_count+1,
			       sizeof (sb_range->param_value_offsets[0]));

	sb_range->param_value_offsets[ptr] =
	    ftell(buffer->a.file->hdl->fh);

	sb_range->param_value_count++;

	pgrng_binding_add_value(binding,param_value,ptr);

	put_pager_param_serialization_to_file(buffer->a.file->hdl->fh,
					      param_value);

	free_stringbuffer(&bf);

    } else
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_pager_param_to_file",
	      "pager_range (binding) missing",0);

	
}

S_(sb_get_pager_param_from_stringbuffer sb_get_pager_param_from_file)
static struct pager_param_value * sb_get_pager_param_from_file
     P_((struct stringbuffer      * buffer,
	 struct pager_range       * range,
	 enum pager_param_opcode    opcode,
	 int                      * next_pos,
	 struct pgrng_bfr_binding * binding,
	 int                      * found_pos));
static struct pager_param_value * sb_get_pager_param_from_file(buffer,range,
							       opcode,next_pos,
							       binding,found_pos)
     struct stringbuffer      * buffer;
     struct pager_range       * range;
     enum pager_param_opcode    opcode;
     int                      * next_pos;
     struct pgrng_bfr_binding * binding;
     int                      * found_pos;
{
    struct pager_param_value * ret = NULL;
    int                        min_pos = 0;
    
    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_pager_param_from_file",
	      "Bad magic number",0);

    if (found_pos)
	 *found_pos = -1;

    if (next_pos) {
	min_pos = *next_pos;

	if (min_pos < 0)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_get_pager_param_from_file",
		  "Bad start index",0);
	
	*next_pos = -1;
    }
    
    if (binding) {
	int range_index;
	struct stringbuffer *bf = 
	    pgrng_bfr_bnd_get_stringbuffer(binding,&range_index);
	struct sb_pager_range * sb_range;

	if (bf != buffer)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_get_pager_param_from_file",
		  "pager_range (binding) added to wrong stringbuffer",0);

	if (range_index < 0 || 
	    range_index >=  buffer->a.file->pager_range_count ||
	    buffer->a.file->pager_range[range_index].pager_range != binding)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_get_pager_param_from_file",
		  "bad range_index",0);
	sb_range = &(buffer->a.file->pager_range[range_index]);

	if (sb_range->param_value_offsets) {
	    int i;
	
	    for (i = min_pos; i < sb_range->param_value_count; i++) {

		struct pager_param_value *
		    param_value =
		    pgrng_binding_get_value(binding,i);

		if (!param_value) {

		    enum syscall_status r;

		    r = fseek(buffer->a.file->hdl->fh,
			      sb_range->param_value_offsets[i],
			      SEEK_SET);

		    switch (r) {
			int err UNUSED_VAROK;
		    case syscall_success:
			DPRINT(Debug,14,(&Debug,
					 "stringbuffer: fseek to %ld: %s\n",
					 sb_range->param_value_offsets[i],
					 buffer->a.file->hdl->filename));
			break;
		    case syscall_error:
			err = errno;

			DPRINT(Debug,1,(&Debug,
					"stringbuffer: fseek (flush) failure: %s: %s (errno %d)\n",
					buffer->a.file->hdl->filename,
					strerror(err),err));

			panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
			      "sb_get_pager_param_from_file",
			      "Failed to seek pager param range from file",0);
		    }

		    param_value =
			get_pager_param_serialization_from_file(buffer->a.file->hdl->fh);

		    if (!param_value)
			   panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
				 "sb_get_pager_param_from_file",
				 "No pager_range on file",0);

		    pgrng_binding_add_value(binding,param_value,i);		    
		}
					    

		if (match_pager_param_value(param_value,opcode)) {
		    ret =  param_value;

		    if (next_pos)
			*next_pos = i+1;
		    if (found_pos)
			*found_pos = i; 
		    break;
		}

		free_pager_param_value(& param_value);		
	    }		
	}

	free_stringbuffer(&bf);

    } else
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_pager_param_from_file",
	      "pager_range (binding) missing",0);
	

    /* Does not need:
       inc_pager_param_value_refcount(ret);
    */
    
    return ret;
}


struct sb_type   sb_in_file = {
    SB_TYPE_magic,

    sb_init_file,
    sb_free_file,
    sb_add_line_to_file,
    sb_linecount_file,
    sb_get_line_from_file,
    sb_add_pager_range_to_file,
    sb_get_pager_range_from_file,
    sb_add_lineext_to_file,
    sb_walk_lineext_on_file,
    sb_add_entity_to_file,
    sb_add_pager_param_to_file,
    sb_get_pager_param_from_file
};

/* ------------------------------------------------------------------ */

#define PLEXT_IN_FILE_magic  0xEB14

struct plext_in_file {
    unsigned short            magic;       /* PLEXT_IN_FILE_magic */

    struct sb_file_ext_offsets * ext_offsets;

    int     off_ptr;
};

S_(plext_init_extension plext_init_file)
static void plext_init_file P_((struct pager_lineext *ext,
				struct stringbuffer * stringbuffer));
static void plext_init_file(ext,stringbuffer)
     struct pager_lineext * ext;
     struct stringbuffer * stringbuffer; /* This is stored by caller */
{

    ext->a.file = safe_zero_alloc(sizeof (* ext->a.file));

    ext->a.file->ext_offsets = NULL;
    ext->a.file->off_ptr = -1;

    ext->a.file->magic = PLEXT_IN_FILE_magic;

    
}

S_(plext_dest_extension plext_dest_file)
static void plext_dest_file P_((struct pager_lineext *ext,
				struct stringbuffer * stringbuffer));
static void plext_dest_file(ext,stringbuffer)
     struct pager_lineext * ext;
     struct stringbuffer * stringbuffer; /* This is stored by caller */
{
    if (PLEXT_IN_FILE_magic != ext->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"plext_dest_file",
	      "Bad magic number",0);

    if (ext->a.file->ext_offsets)
	free_sb_file_ext_offsets(& (ext->a.file->ext_offsets));

    ext->a.file->magic = 0;          /* Invalidate */
    free(ext->a.file);
    ext->a.file = NULL;

}

/* Returns next character */

static int sb_plext_read_prefix P_((struct stringbuffer *buffer,
				    struct pager_lineext * this_lineext,
				    int * off_ptr_p,
				    int * have_range_p,
				    int * range_index_p,
				    int * flags_p));
static int sb_plext_read_prefix(buffer,this_lineext,off_ptr_p,
				have_range_p, range_index_p, flags_p)
     struct stringbuffer  * buffer;
     struct pager_lineext * this_lineext;
     int * off_ptr_p;
     int * have_range_p;
     int * range_index_p;
     int * flags_p;
{
    int c;

    int off_ptr       = * off_ptr_p;
    int have_range    = * have_range_p;
    int range_index   = * range_index_p;
    int flags         = * flags_p;

    enum syscall_status r;

    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_plext_read_prefix",
	      "Bad magic number (stringbuffer)",0);

    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_plext_read_prefix",
	      "Bad magic number (sb_in_file)",0);

    if (PAGER_LINEEXT_magic != this_lineext->magic)
        panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
              "sb_plext_read_prefix",
              "Bad magic number (pager_lineext)",0);

    if (PLEXT_IN_FILE_magic != this_lineext->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_plext_read_prefix",
	      "Bad magic number (plext_in_file)",0);

    if (SB_file_ext_offsets_magic !=
	this_lineext->a.file->ext_offsets->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_plext_read_prefix",
	      "Bad magic number (sb_file_ext_offsets)",0);

    if (this_lineext->a.file->off_ptr < 0 ||
	this_lineext->a.file->off_ptr >= 
	this_lineext->a.file->ext_offsets->lineext_count ||
	! this_lineext->a.file->ext_offsets->lineext_offsets)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_plext_read_prefix",
	      "Bad extension",0);
    
    off_ptr = this_lineext->a.file->off_ptr;

    if (this_lineext->a.file->ext_offsets->lineext_offsets[off_ptr] < 0L)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_plext_read_prefix",
	      "Bad extension offset",0);

    if (SB_file_hdl_magic != buffer->a.file->hdl->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_plext_read_prefix",
	      "Bad magic number (sb_file_hdl)",0);


    r = fseek(buffer->a.file->hdl->fh,
	      this_lineext->a.file->ext_offsets->lineext_offsets[off_ptr],
	      SEEK_SET);

    switch (r) {
	int err UNUSED_VAROK;
    case syscall_success:
	DPRINT(Debug,14,(&Debug,
			 "stringbuffer: fseek to %ld: %s\n",
			 this_lineext->a.file->ext_offsets->lineext_offsets[off_ptr],
			 buffer->a.file->hdl->filename));
	break;
    case syscall_error:
	err = errno;
	
	DPRINT(Debug,1,(&Debug,
			"stringbuffer: fseek (flush) failure: %s: %s (errno %d)\n",
			buffer->a.file->hdl->filename,
			strerror(err),err));
	
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_plext_read_prefix",
	      "Failed to seek extension from file",0);
    }
    
    c = fgetc(buffer->a.file->hdl->fh);
    
    if (c == 'r') {
	have_range = 1;

	c = fgetc(buffer->a.file->hdl->fh);
	if (c >= '0' && c <= '9') {
	    range_index = 0;
	    while (c >= '0' && c <= '9') {
		range_index = range_index * 10 + c - '0';
		c = fgetc(buffer->a.file->hdl->fh);
	    }
	}
	    
	if (',' != c) {
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_plext_read_prefix",
		  "Bad data read from file",0);
	}
	c = fgetc(buffer->a.file->hdl->fh);
    }

    if (c >= '0' && c <= '9') {
	flags = 0;
	
	while (c >= '0' && c <= '9') {
	    flags = flags * 10 + c - '0';
	    c = fgetc(buffer->a.file->hdl->fh);
	}
    }
    
    * off_ptr_p     = off_ptr;
    * have_range_p  = have_range;    
    * range_index_p = range_index;
    * flags_p       =  flags;
   
    return c;
}


static struct pager_lineext * sb_file_copy_extension2
    P_((const struct  pager_lineext *extension,
	int idx,
	struct stringbuffer *buffer));


S_(plext_walk_alt_extension plext_walk_alt_file)
static struct string_alt * plext_walk_alt_file P_((struct pager_lineext *ext,
						   struct pager_lineext **
						   walk_extension,
						   int *pg_flags,
						   struct pager_range **range));
static struct string_alt * plext_walk_alt_file(ext,walk_extension,pg_flags,
					       range)
     struct pager_lineext *ext;
     struct pager_lineext **walk_extension;
     int *pg_flags;
     struct pager_range **range;
{
    struct stringbuffer *buffer;
    struct string_alt * res = NULL;

    int off_ptr = 0;
    int have_range = 0;
    int c;
    int range_index = 0;
    int flags = 0;
    struct pager_lineext * this_lineext = NULL;

    
    if (PAGER_LINEEXT_magic != ext->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "plext_walk_alt_file",
	      "Bad magic number (pager_lineext)",0);

    buffer = ext->stringbuffer;

    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "plext_walk_alt_file",
	      "Bad magic number (stringbuffer)",0);

    if (&sb_in_file != buffer->buffer_type)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "plext_walk_alt_file",
	      "Bad buffer type",0);

    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_walk_lineext_on_file",
	      "Bad magic number (sb_in_file)",0);

    if (PLEXT_IN_FILE_magic != (*walk_extension)->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_walk_lineext_on_file",
	      "Bad magic number (plext_in_file)",0);

    if (SB_file_ext_offsets_magic !=
	(*walk_extension)->a.file->ext_offsets->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_walk_lineext_on_file",
	      "Bad magic number (sb_file_ext_offsets)",0);

    this_lineext = *walk_extension;
    *walk_extension = NULL;

    if (range && *range)
	free_pager_range(range);

    c = sb_plext_read_prefix(buffer,this_lineext,
			     &off_ptr,&have_range,
			     &range_index,&flags);
    
    *pg_flags = flags;

    if (SB_file_hdl_magic != buffer->a.file->hdl->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "plext_walk_alt_file",
	      "Bad magic number (sb_file_hdl)",0);
    
    switch (c) {
	struct string     * text;
	struct out_entity * oe;
	
    case ':':  /* struct string */
	
	text = get_string_serialization_from_file(buffer->a.file->hdl->fh);
	if (!text) {
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "plext_walk_alt_file",
		  "Bad data read (string serialization failed) from file",0);
	}

	res = new_string_alt_text(text);
	free_string(&text);

	break;

    case '&': /* struct out_entity */

	oe = get_out_entity_serialization_from_file(buffer->a.file->hdl->fh);
	if (!oe) {
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "plext_walk_alt_file",
		  "Bad data read (out_entity serialization failed) from file",0);
	}
	
	res =  new_string_alt_entity(oe);
	free_out_entity(&oe);
	
	break;
    default:
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_walk_lineext_on_file",
	      "Bad data read from file",0);
	break;
    }
    
    if (off_ptr+1 < this_lineext->a.file->ext_offsets->lineext_count)
	*walk_extension = sb_file_copy_extension2(this_lineext,off_ptr+1,
						  buffer);
    
    /* May read from file */

    if (range && have_range)
	*range = sb_get_pager_range_from_file(buffer,range_index);    

    
    free_pager_lineext(&this_lineext);
    
    return res;
}

static struct lineext_type plext_in_file = {
    LINEEXT_TYPE_magic,
    
    plext_init_file,
    plext_dest_file,
    plext_walk_stringbuffer,
    plext_walk_alt_file
};


static struct pager_lineext * sb_file_copy_extension(buffer,ptr)
     struct stringbuffer *buffer; 
     int ptr;
{
    struct pager_lineext *first_extension = 
	malloc_pager_lineext(&plext_in_file,buffer);
        
    first_extension->a.file->ext_offsets =
	buffer->a.file->offsets[ptr].ext_offsets;

    inc_sb_file_ext_offsets_refcount(first_extension->a.file->ext_offsets);

    first_extension->a.file->off_ptr = 0;

    return first_extension;
}

static struct pager_lineext * sb_file_copy_extension2(extension,idx,
						      buffer)
     const struct  pager_lineext *extension;
     int idx;
     struct stringbuffer *buffer;
{
    struct pager_lineext *next_extension = 
	malloc_pager_lineext(&plext_in_file,
			     buffer);
    
    next_extension->a.file->ext_offsets	= extension->a.file->ext_offsets;

    inc_sb_file_ext_offsets_refcount(next_extension->a.file->ext_offsets);	

    next_extension->a.file->off_ptr = idx;

    return next_extension;
}

static struct string *sb_walk_lineext_on_file(buffer,walk_extension,
					      pg_flags,range)
     struct stringbuffer *buffer;
     struct pager_lineext **walk_extension;
     int *pg_flags;
     struct pager_range **range;
{
    struct string * res = NULL;

    int off_ptr = 0;
    int have_range = 0;
    int c;
    int range_index = 0;
    int flags = 0;
    struct pager_lineext * this_lineext = NULL;

    if (SB_in_file_magic != buffer->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_walk_lineext_on_file",
	      "Bad magic number (sb_in_file)",0);

    if (PLEXT_IN_FILE_magic != (*walk_extension)->a.file->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_walk_lineext_on_file",
	      "Bad magic number (plext_in_file)",0);

    if (SB_file_ext_offsets_magic !=
	(*walk_extension)->a.file->ext_offsets->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_walk_lineext_on_file",
	      "Bad magic number (sb_file_ext_offsets)",0);

    this_lineext = *walk_extension;
    *walk_extension = NULL;

    if (range && *range)
	free_pager_range(range);

    c = sb_plext_read_prefix(buffer,this_lineext,
			     &off_ptr,&have_range,
			     &range_index,&flags);

    *pg_flags = flags;

    if (SB_file_hdl_magic != buffer->a.file->hdl->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_walk_lineext_on_file",
	      "Bad magic number (sb_file_hdl)",0);

    
    switch (c) {
	struct out_entity * oe;
	const struct string  * text;
	
    case ':':  /* struct string */
	
	res = get_string_serialization_from_file(buffer->a.file->hdl->fh);
	if (!res) {
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_walk_lineext_on_file",
		  "Bad data read (string serialization failed) from file",0);
	}

	break;

    case '&': /* struct out_entity */

	oe = get_out_entity_serialization_from_file(buffer->a.file->hdl->fh);
	if (!oe) {
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_walk_lineext_on_file",
		  "Bad data read (out_entity serialization failed) from file",0);
	}

	text = out_entity_text_value(oe);
	if (text) {

	    res = dup_string(text);

	} else {
	    const uint16 code = out_entity_unicode_value(oe);

	    if (UNICODE_BAD_CHAR != code &&
		charset_have_unicode ==
		string_type_have_unicode(display_charset,
					 code)) {

		res = new_string(display_charset);
		add_unicode_to_string(res,1,&code);
		
	    } else {

		int flags2 = 0;

		 const struct string  * refkey =  
		     out_entity_reference_key(oe,&flags2);

		 res = dup_string(refkey);
		 *pg_flags |= flags2;
	    }	               
	}
	
	free_out_entity(&oe);

	break;
    default:
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_walk_lineext_on_file",
	      "Bad data read from file",0);
	break;
    }
	
    if (off_ptr+1 < this_lineext->a.file->ext_offsets->lineext_count)
	*walk_extension = sb_file_copy_extension2(this_lineext,off_ptr+1,
						  buffer);

    
    /* May read from file */

    if (range && have_range)
	*range = sb_get_pager_range_from_file(buffer,range_index);    

    free_pager_lineext(&this_lineext);

    return res;
}



/* ------------------------------------------------------- */




/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

