static char rcsid[] = "@(#)$Id: safemalloc.c,v 2.8 2020/01/18 20:07:40 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.8 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 lib/safemalloc.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1992 USENET Community Trust
 *****************************************************************************/

#include "elm_defs.h"

/*
 * These routines perform dynamic memory allocation with error checking.
 * The "safe_malloc_fail_handler" vector points to a routine that is invoked
 * if memory allocation fails.  The default error handler displays a message
 * and aborts the program.
 */


DEBUG_VAR(Debug,__FILE__,"mem");

void dflt_safe_malloc_fail_handler(proc, len)
     char *proc;
     size_t len;
{    
#if DEBUG
    panic_dprint("ERROR - out of memory [%s failed allocating %zu bytes]\n",
		 proc, len);
#endif

    fprintf(stderr,
	    "error - out of memory [%s failed allocating %lu bytes]\n",
	    proc, (unsigned long)len);
    exit(1);
}

void (*safe_malloc_fail_handler) P_((char *proc,
				     size_t len)) = 
    dflt_safe_malloc_fail_handler;

malloc_t safe_malloc(len)
     size_t len;
{
    malloc_t p;

    if (len < 1) {
	DPRINT(Debug,1,(&Debug,
			"safe_malloc(%zu)    BAD SIZE\n",
			len));
	(*safe_malloc_fail_handler)("safe_malloc", len);
    }

    if ((p = malloc(len)) == NULL) {
	DPRINT(Debug,1,(&Debug,
			"safe_malloc(%zu)    FAILED\n",
			len));
	(*safe_malloc_fail_handler)("safe_malloc", len);
    }

    DPRINT(Debug,100,(&Debug,
		      "safe_malloc(%zu)=%p\n",
		      len,p));
    return p;
}

malloc_t safe_zero_alloc(len)
     size_t len;
{
    malloc_t p;

    if (len < 1) {
	DPRINT(Debug,1,(&Debug,
			"safe_zero_alloc(%zu)    BAD SIZE\n",
			len));
	(*safe_malloc_fail_handler)("safe_zero_alloc", len);
    }

    if ((p = malloc(len)) == NULL) {
	DPRINT(Debug,1,(&Debug,
			"safe_zero_alloc(%zu)    FAILED\n",
			len));
	(*safe_malloc_fail_handler)("safe_zero_alloc", len);
    }

    /* bzero is defined hdrs/elm_defs.h */
    bzero(p,len);

    DPRINT(Debug,100,(&Debug,
		      "safe_zero_alloc(%zu)=%p\n",
		      len,p));
    return p;
}

malloc_t safe_realloc(p, len)
     malloc_t p;
     size_t len;
{
    malloc_t po UNUSED_VAROK = p;

    if (len < 1) {
	DPRINT(Debug,1,(&Debug,
			"safe_realloc(%p,%lu)    BAD SIZE\n",
			po,(unsigned long)len));
	(*safe_malloc_fail_handler)("safe_realloc", len);
    }
    

    if ((p = (p == NULL ? malloc(len) : realloc(p, len))) 
	== NULL) {
	DPRINT(Debug,1,(&Debug,
			"safe_realloc(%p,%lu)     FAILED\n",
			po,(unsigned long)len));
	(*safe_malloc_fail_handler)("safe_realloc", len);
    }
    DPRINT(Debug,100,(&Debug,
		      "safe_realloc(%p,%lu)=%p\n",
		      po,(unsigned long)len,p));
    return p;
}

/* <inttypes.h> includes <stdint.h> */
#if defined(I_STDINT) && !defined(SIZE_MAX)
/* For SIZE_MAX */
#include <stdint.h>
#endif

#ifdef  SIZE_MAX
const size_t size_MAX = SIZE_MAX;
#else

/* maximun possible value for size_t -- there is not constant for that 
   this assumes that size_t is unsigned
*/
const size_t size_MAX = ~(size_t)0;
#endif


malloc_t safe_calloc (nmemb,size)
     size_t nmemb; 
     size_t size;
{
    malloc_t p;
    size_t Len = size_MAX;

    if (size != 0 && nmemb > size_MAX / size) {
	DPRINT(Debug,1,(&Debug,
			"safe_calloc(%lu,%lu)    OVERFLOW size_MAX=%lu \n",
			(unsigned long)nmemb,(unsigned long)size,
			(unsigned long)size_MAX));

	(*safe_malloc_fail_handler)("safe_calloc",Len);

    } else
	Len = nmemb * size;  
    
    if (nmemb < 1 || size < 1) {
	DPRINT(Debug,1,(&Debug,
			"safe_calloc(%lu,%lu)    BAD SIZE\n",
			(unsigned long)nmemb,(unsigned long)size));
	
	(*safe_malloc_fail_handler)("safe_calloc",Len);
    }

    if ((p = calloc(nmemb,size)) == NULL) {
	DPRINT(Debug,1,(&Debug,
			"safe_calloc(%lu,%lu)    FAILED\n",
			(unsigned long)nmemb,(unsigned long)size)); 
	
	(*safe_malloc_fail_handler)("safe_calloc",Len);
    }

    DPRINT(Debug,100,(&Debug,
		      "safe_calloc(%lu,%lu)=%p      Len=%lu\n",
		      (unsigned long)nmemb,(unsigned long)size,p,
		      (unsigned long)Len));

    return p;
}

/* Return 1 if safe result */

int safe_array_size(result_size,nmemb,size)
     size_t *result_size;
     size_t nmemb;
     size_t size;
{
    int ret = 1;
    size_t Len = size_MAX;

    if (size != 0 && nmemb > size_MAX / size) {
	DPRINT(Debug,1,(&Debug,
			"safe_array_size(%p,%lu,%lu)    OVERFLOW size_MAX=%lu \n",
			result_size,(unsigned long)nmemb,(unsigned long)size,
			(unsigned long)size_MAX));

	ret = 0;
	
    } else
	Len = nmemb * size;  
    
    if (nmemb < 1 || size < 1) {
	DPRINT(Debug,1,(&Debug,
			"safe_array_size(%p,%lu,%lu)    BAD SIZE\n",
		       result_size,(unsigned long)nmemb,(unsigned long)size));
	
	ret = 0;
    }

    DPRINT(Debug,14,(&Debug,
		     "safe_array_size(%p,%lu,%lu)=%d",
		     result_size,(unsigned long)nmemb,(unsigned long)size,
		     ret));

    if (result_size) {
	*result_size = Len;
	
	
	DPRINT(Debug,14,(&Debug,", *result_size=%zu",
			 *result_size));	
    }

    DPRINT(Debug,14,(&Debug,"\n"));

    return ret;
}



malloc_t safe_array_realloc(p,nmemb,size)
     malloc_t p;
     size_t nmemb; 
     size_t size;
{
    size_t Len = size_MAX;
    malloc_t po UNUSED_VAROK = p;

    if (size != 0 && nmemb > size_MAX / size) {
	DPRINT(Debug,1,(&Debug,
			"safe_array_realloc(%p,%lu,%lu)    OVERFLOW size_MAX=%lu \n",
			po,(unsigned long)nmemb,(unsigned long)size,
			(unsigned long)size_MAX));

	(*safe_malloc_fail_handler)("safe_array_realloc",Len);

    } else
	Len = nmemb * size;  

    if (nmemb < 1 || size < 1) {
	DPRINT(Debug,1,(&Debug,
			"safe_array_realloc(%p,%lu,%lu)    BAD SIZE\n",
			po,(unsigned long)nmemb,(unsigned long)size));
	
	(*safe_malloc_fail_handler)("safe_array_realloc",Len);
    }

    if ((p = (p == NULL ? malloc(Len) : realloc(p, Len))) 
	== NULL) {

	DPRINT(Debug,1,(&Debug,
			"safe_array_realloc(%p,%lu,%lu)    FAILED\n",
			po,(unsigned long)nmemb,(unsigned long)size));

	(*safe_malloc_fail_handler)("safe_array_realloc",Len);
    }

    DPRINT(Debug,100,(&Debug,
		      "safe_array_realloc(%p,%lu,%lu)=%p      Len=%lu\n",
		      po,(unsigned long)nmemb,(unsigned long)size,p,
		      (unsigned long)Len));
    return p;
}

void safe_stralloc_append(str,str_len,append,append_len)
     char   ** str;
     size_t  * str_len;
     char    * append;
     size_t    append_len;
{
    size_t Len = size_MAX;
    
    if (size_MAX - append_len <= *str_len) {
	DPRINT(Debug,1,(&Debug,
                        "safe_stralloc_append: *str_len=%lu append_len=%lu OVERFLOW size_MAX=%lu\n",
                        (unsigned long)*str_len,
			(unsigned long)append_len,
                        (unsigned long)size_MAX));

        (*safe_malloc_fail_handler)("safe_stralloc_append",Len);
	
    } else
	Len = *str_len +  append_len +1;

    if ((*str = (*str == NULL ? malloc(Len) : realloc(*str, Len)))
	== NULL) {

	DPRINT(Debug,1,(&Debug,
                        "safe_stralloc_append: *str_len=%lu append_len=%lu FAILED Len=%lu\n",
                        (unsigned long)*str_len,
			(unsigned long)append_len,
			(unsigned long)Len));
	
	(*safe_malloc_fail_handler)("safe_stralloc_append",Len);
    }

    memcpy((*str) + (*str_len), append,append_len);    
    
    (*str_len) += append_len;
    (*str)[*str_len] = '\0';
}



char *safe_strdup(s)
     const char *s;
{
    char *p;
    size_t Len = strlen(s)+1;

    if ((p = (char *) malloc(Len)) == NULL) {

	DPRINT(Debug,1,(&Debug,
			"safe_strdup(%s)      FAILED, len = %lu\n",
			s,(unsigned long)Len));
	(*safe_malloc_fail_handler)("safe_strdup", Len);
    }
    DPRINT(Debug,100,(&Debug,
		      "safe_strdup(%s)=%p\n",s,p));
    return strfcpy(p, s, Len);
}



#ifdef DEBUG
#undef free
void safe_free(x)
     malloc_t x;
{
    DPRINT(Debug,100,(&Debug,
		"(safe)free(%p) ",x));


    if (x == NULL) {
	DPRINT(Debug,1,(&Debug,
			"** FREE OF NULL POINTER\n"));
    }

    free(x);
    DPRINT(Debug,100,(&Debug,
		      "\n"));
}
#endif

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
