static char rcsid[] = "@(#)$Id: remote_mbx.c,v 2.68 2022/11/02 16:26:29 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.68 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"
#include "schedule_time.h"
#include "connection_imp.h"
#include "ss_imp.h"
#ifdef USE_DLOPEN
#include "shared_imp.h"
#endif
#include "save_opts.h"
#include "s_me.h"
#include "s_elm.h"

/* For INT_MAX */
#include <limits.h>

DEBUG_VAR(Debug,__FILE__,"net");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif


void zero_Read_Buffer(buffer)
     struct Read_Buffer *buffer;
{
    buffer->read_buffer = NULL;
    buffer->read_len    = 0;
}

void free_Read_Buffer(buffer)
     struct Read_Buffer *buffer;
{
    if (buffer->read_buffer) {
	free(buffer->read_buffer);
	buffer->read_buffer = NULL;
    }
    buffer->read_len = 0;
}

/* For non-literal we read with quite short block because
   many answers are quite small
*/

#define READ_BLOCK   1024

int ReadFromSocket(fd,buffer,wanted)
     int fd; 
     struct Read_Buffer *buffer;
     int wanted;
{
    int n;
    
    size_t read_limit = READ_BLOCK;
    char * tmp;
    size_t new_len;
    

    if (wanted > 0 && wanted < size_MAX)
	read_limit = wanted;

    if (buffer -> read_len >= size_MAX-read_limit) {
	DPRINT(Debug,49,(&Debug,
			 "ReadFromSocket: read_len=%zu wanted=%d (size_MAX=%zu) => ",
			 buffer -> read_len,wanted,size_MAX));

	read_limit = size_MAX - buffer -> read_len;
	if (read_limit > READ_BLOCK)
	    read_limit = READ_BLOCK;

	DPRINT(Debug,49,(&Debug,"read_limit=%zu\n",
			 read_limit));

	if (read_limit < 1) {
	     DPRINT(Debug,49,(&Debug,
			      "ReadFromSocket: Bad read limit\n"));

	     n = -1;
	     errno = ERANGE;
	     goto fail;
	}	    
    }
	
    new_len = buffer -> read_len + read_limit;
    tmp = realloc(buffer -> read_buffer,new_len);
    if (!tmp && read_limit > READ_BLOCK) {
	read_limit = READ_BLOCK;
	
	new_len = buffer -> read_len + read_limit;
	tmp = realloc(buffer -> read_buffer,new_len);
	if (tmp) {
	    DPRINT(Debug,49,(&Debug,
			     "ReadFromSocket: realloc failure => read_limit=%zu\n",
			     read_limit));
	}
    }

    if (tmp) {
	buffer -> read_buffer = tmp;
	n = read(fd, buffer -> read_buffer + buffer -> read_len, read_limit);
    } else {
	int err = errno;

	DPRINT(Debug,10,(&Debug,
			 "ReadFromSocket: Failed to reallocate %zu bytes: %s (errno=%d)\n",
			 new_len,strerror(err),err));
	
	n = -1;
	errno = err;
    }

 fail:
    
    return n;
    
}

int find_crlf(buffer, add_null)
     struct Read_Buffer *buffer;
     int add_null;
{
    char * p = buffer->read_buffer;
    
    if (buffer->read_len > 0) {
	size_t i,max;
	
	max = buffer->read_len -1;
	
	if (max > INT_MAX-2) {
	    DPRINT(Debug,49,(&Debug,
			     "find_crlf: limit %zu => ",max));
	    max = INT_MAX-2;
	    
	    DPRINT(Debug,49,(&Debug,"%zu\n",max));
	}
	
	
	for (i = 0; i < max; i++) {
	    if ('\r' == p[i] && '\n' == p[i+1]) {
		if (add_null) {
		    p[i] = '\0';
		    p[i+1] = '\0';
		}
		DPRINT(Debug,49,(&Debug,
				 "find_crlf=%d (%zu total)\n",
				 i+2,buffer->read_len));
		return i+2;
	    }
	}
    }
    
    DPRINT(Debug,49,(&Debug,
		     "find_crlf=0 (%zu total)\n",
		     buffer->read_len));
    return 0;
}

void cut_line(buffer, len)
     struct Read_Buffer *buffer;
     size_t len;
{
    char * p = buffer->read_buffer;

    if (buffer->read_len < len)
    	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "cut_line",
	      "len too big",0);

    buffer->read_len -= len;
    DPRINT(Debug,49,(&Debug,
		     "cut_line: %zu chars consumed, %zu left\n",
		     len,buffer->read_len));
    if (buffer->read_len)
	memmove(p, p+len, buffer->read_len);
}     

void zero_Write_Buffer(buffer)
     struct Write_Buffer *buffer;
{
    buffer->write_buffer = NULL;
    buffer->write_len    = 0;
}

void free_Write_Buffer(buffer)
     struct Write_Buffer *buffer;
{
    if (buffer->write_buffer) {
	free(buffer->write_buffer);
	buffer->write_buffer = NULL;
    }
    buffer->write_len = 0;
}

int WriteToSocket(fd,buffer)
     int fd; 
     struct Write_Buffer *buffer;
{
    int n = write(fd,buffer->write_buffer, buffer->write_len);
    
    return n;
}

void cut_Write_Buffer(buffer,n)
     struct Write_Buffer *buffer; 
     int n;
{
    if (n > 0) {
	char * p = buffer->write_buffer;
	
	buffer->write_len -= n;
	if (buffer->write_len)
	    memmove(p, p+n, buffer->write_len);

	DPRINT(Debug,13,(&Debug,
			 "cut_Write_Buffer: Written %d bytes (%d left)\n",
			 n,buffer->write_len));
    }
}

void add_to_Write_Buffer(buffer,str,l)
     struct Write_Buffer *buffer;
     char **str;
     int l;
{
    if (!buffer->write_len) {
	if (buffer->write_buffer) 
	    free(buffer->write_buffer);
	buffer->write_buffer = *str;
	buffer->write_len    = l;
    } else if (l > 0) {
	buffer->write_buffer = safe_realloc(buffer->write_buffer,
					    buffer->write_len+l);
	memmove(buffer->write_buffer + buffer->write_len,*str,l);
	buffer->write_len += l;
	free(*str);
    }
    *str = NULL;
}

/* Seems that h_errno is macro on AIX */
#ifndef h_errno
extern int h_errno;
#endif

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

FILE * transaction_file = NULL;

int set_transaction_file(filename)
     const char *filename;
{
    FILE * f;
    int err;
    int fd;
    time_t now = 0;
    enum syscall_status  r;
    
    if (userid == -1 || groupid == -1) {
	/* save original user and group ids */
	userid  = getuid();
	groupid = getgid();	

	DPRINT(Debug,1,(&Debug,"set_transaction_file: %s: Saving userid/groupid=%d/%d\n",
			filename,userid,groupid));

    }

    err = can_open(filename,"a");
    if (0 != err) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			  "File %.50s is not writeable: %s"),
		  filename, strerror(err));
	return 0;
    }

    fd = open(filename,O_CREAT|O_WRONLY|O_APPEND,00600);
    if (-1 == fd) {
	int err = errno;
	lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			  "File %.50s is not writeable: %s"),
		  filename, strerror(err));
	return 0;
    }
    elm_chown(filename, userid, groupid, NULL); /* file owned by user */

#ifdef FD_CLOEXEC
    r = fcntl(fd, F_SETFD, FD_CLOEXEC);
#else
    r = fcntl(fd, F_SETFD, 1);
#endif    

    switch (r) {
	int err UNUSED_VAROK;
    case syscall_error /* -1 */:
	err = errno;
        
	DPRINT(Debug,14,(&Debug, 
			 "set_transaction_file: Failed to set close-on-exec flag for %s: %s\n",
			 filename,
			 strerror(err)));
	break;
    case syscall_success /* 0 */:
	DPRINT(Debug,14,(&Debug, 
			 "set_transaction_file: Set close-on-exec flag for %s\n",
			 filename));
	break;
    }
            
    f = fdopen(fd,"a");
    if (!f) {
	close(fd);
	return 0;
    }

    if (transaction_file) {
	fprintf(transaction_file,
		"\n===== Changing logging to file %s\n",
		filename);
	fclose(transaction_file);
    }
    transaction_file = f;
    
#ifdef SETLINEBUF
    setlinebuf(transaction_file);
#endif
		
    if (((time_t) -1) != time(&now)) {
	struct tm * zz = localtime(&now);
	
	if (!zz)
	    goto no_time15;
	
	fprintf(transaction_file,
		"%d [-] %02d:%02d:%02d --- starting transaction log\n",
		getpid(),
		zz->tm_hour,
		zz->tm_min,
		zz->tm_sec);
    } else {
    no_time15:
	
	fprintf(transaction_file,
		"%d [-] --- starting transaction log\n",
		getpid());
    }
    
    DPRINT(Debug,1,(&Debug,
		    "set_transaction_file: %s: transaction file opened\n",
		    filename));

    return 1;
}

void free_transaction_file()
{
    if (transaction_file) {
	time_t now = 0;
		
	if (((time_t) -1) != time(&now)) {
	    struct tm * zz = localtime(&now);
	    
	    if (!zz)
		goto no_time14;

	    fprintf(transaction_file,
		    "%d [-] %02d:%02d:%02d --- closing transaction log\n",
		    getpid(),
		    zz->tm_hour,
		    zz->tm_min,
		    zz->tm_sec);
	} else {
	no_time14:

	    fprintf(transaction_file,
		    "%d [-] --- closing transaction log\n",
		    getpid());
	}

	fclose(transaction_file);
	transaction_file = NULL;
    }
}

/* Value on arrays is either  AF_INET or AF_INET6 */
static int    ipv_option_array [ NUM_protocols ];
static size_t ipv_option_array_len = 0;

static enum ipv_option ipv_option_copy [ NUM_protocols ];

#ifdef REMOTE_MBX
static struct interface_info ** interface_list    = NULL;
static int                      interface_count   = 0;
static int                      interface_flags   = 0;

size_t get_ipv_option(ipv_option_af,ipv_option_c)
     const int             ** ipv_option_af  /* NULL, if not required */;
     const enum ipv_option ** ipv_option_c   /* NULL, if not required */;
{
    if (ipv_option_af)
	*ipv_option_af = ipv_option_array;
    if (ipv_option_c)
	*ipv_option_c  = ipv_option_copy;

    return ipv_option_array_len;
}
#endif

enum ipv_opt_res add_ipv_option(opt,printerr)
     enum ipv_option opt;
     enum ipv_option_print printerr;
{
    int af               = 0;  /* AF_UNSPEC -- but not necessary defined? */
    enum ipv_opt_res res = ipv_opt_unsupported;
    int err UNUSED_VAROK = 0;
    
    /*  AF_INET is defined on sys/socket.h 
	Configure tested both <sys/socket.h> and  <sys/un.h>
	with HAVE_SOCKET
     */

    DPRINT(Debug,12,(&Debug, "add_ipv_option: opt=%d",
		    opt));
    switch (opt) {
    case ipv4_option:  DPRINT(Debug,12,(&Debug," ipv4_option")); break;
    case ipv6_option:  DPRINT(Debug,12,(&Debug," ipv6_option")); break;
    }
    /* Assuming that option letter is same than opt value */
    if (isascii(opt) && isprint(opt)) {
	DPRINT(Debug,12,(&Debug, " '%c'",opt));
    }
    DPRINT(Debug,12,(&Debug, ", printerr=%d",printerr));
    switch(printerr) {
    case ipv4_option_silent:   DPRINT(Debug,12,(&Debug," ipv4_option_silent")); break;
    case ipv4_option_printerr: DPRINT(Debug,12,(&Debug," ipv4_option_printerr")); break;
    }
    DPRINT(Debug,12,(&Debug, "\n"));

#ifdef HAVE_SOCKET
    af =  AF_UNSPEC;
#endif
    switch (opt) {
#ifdef HAVE_SOCKET
    case ipv4_option:  af = AF_INET;  break;
#ifdef HAVE_IN6
    case ipv6_option:  af = AF_INET6; break;
#endif
#endif
    default:
	DPRINT(Debug,12,(&Debug, "add_ipv_option: Unsupported option\n"));
	
	res = ipv_opt_unsupported;
	
	switch(printerr) {
	case ipv4_option_silent: break;
	case ipv4_option_printerr:
	    if (isascii(opt) && isprint(opt)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnsupportedIPvOption,
				  "Option -%c not supported - unupported IP protocol."),
			  opt);
	    }	    
	    break;
	}

	goto fail;
    }
	        
    /* if current protocol is same than
       last added, accept this silently
       without adding anything
    */

    if (ipv_option_array_len > 0 &&
	ipv_option_array[ipv_option_array_len-1] == af) {

	DPRINT(Debug,12,(&Debug,
			 "add_ipv_option: Option #%zu is same than current option %d (OK)\n",
			 ipv_option_array_len-1,af));

	res = ipv_opt_succeed;
	goto ipv_ok;
    } else {
	/* If already on list, return error 
	   order can not changed
	 */

	size_t i;

	for (i = 0; i < ipv_option_array_len; i++) {
	    if (ipv_option_array[i] == af) {
		DPRINT(Debug,12,(&Debug,
				 "add_ipv_option: Option #%zu is same than current address family %d (ERROR)\n",
				 i,af));

		res = ipv_opt_already;

		switch(printerr) {
		case ipv4_option_silent: break;
		case ipv4_option_printerr:
		    if (isascii(opt) && isprint(opt)) {
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmDuplicateIPvOption,
					  "Option -%c already given - duplicate IP protocol."),
				  opt);
		    }	    
		    break;
		}
		goto fail;
	    }
	}	
    }

 ipv_ok:

    /* Check interface */

#if defined(HAVE_IFADDRS) && defined(REMOTE_MBX)

    if (update_interface_list(INTERFACE_ADDRLIST,&err)) {
	
	if (isoff(interface_flags,INTERFACE_ADDRLIST)) {
	    DPRINT(Debug,12,(&Debug,
			     "add_ipv_option: Failed to get interface address list\n"));

	    res = ipv_opt_succeed;
	    
	} else {
	    int found = 0;

	    if (interface_list && interface_count > 0) {
		int i_idx;
		
		DPRINT(Debug,12,(&Debug,
				 "add_ipv_option: %d interfaces\n",interface_count));
		
		for (i_idx = 0; i_idx < interface_count && !found; i_idx++) {
		    
		    if (interface_list[i_idx]) {
			if (INTERFACE_INFO_magic != interface_list[i_idx]->magic)
			    panic("RC PANIC",__FILE__,__LINE__,
				  "add_ipv_option",
				  "Bad magic number (interface_info)",0);
			
			if (! (interface_list[i_idx]->valid_addrlist)) {
			    DPRINT(Debug,12,(&Debug,
					     "add_ipv_option: interface \"%s\" skipped, not on address list.\n",
					     interface_list[i_idx]->name));
			    continue;			    
			}
			
			if (interface_list[i_idx]->addr_list) {
			    int a_idx; 
			    
			    DPRINT(Debug,12,(&Debug, 
					     "add_ipv_option: interface \"%s\": %d addresses\n",
					     interface_list[i_idx]->name,
					     interface_list[i_idx]->addr_count));
			    
			    for (a_idx = 0; a_idx < interface_list[i_idx]->addr_count;
				 a_idx++) {
				
				if (interface_list[i_idx]->addr_list[a_idx]) {
				    
				    if (INTERFACE_ADDR_magic !=
					interface_list[i_idx]->addr_list[a_idx]->magic)
					panic("RC PANIC",__FILE__,__LINE__,
					      "add_ipv_option",
					      "Bad magic number (interface_addr)",
					      0);
				    
				    if (! (interface_list[i_idx]->addr_list[a_idx]->valid_address)) {
					DPRINT(Debug,12,(&Debug,
							 "add_ipv_option: interface \"%s\", address #%d skipped, not on address list.\n",
							 interface_list[i_idx]->name,a_idx));
					continue;
				    }
				    
				    if (interface_list[i_idx]->addr_list[a_idx]->address.sa) {
					
					if (interface_list[i_idx]->addr_list[a_idx]->address.sa->
					    sa_family == af) {
					    
					    DPRINT(Debug,12,(&Debug,
							     "add_ipv_option: interface \"%s\", address #%d have current address family %d\n",
							     interface_list[i_idx]->name,a_idx,af));
					    
					    found = 1;
					    res = ipv_opt_succeed;
					    break;
					}				    
				    }
				}
			    }			
			}		    
		    }		
		}
				
	    } else {
		DPRINT(Debug,12,(&Debug,
				 "add_ipv_option: No interfaces\n"));
	    }

	    if (!found) {
		res =  ipc_opt_noactice;

		switch(printerr) {
		case ipv4_option_silent: break;
		case ipv4_option_printerr:
		    if (isascii(opt) && isprint(opt)) {
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadIPvOption,
					  "Option -%c not working - no active interface address for IP protocol."),
				  opt);
		    }
		    break;
		}
	    }	   
	}
	
    } else {
	DPRINT(Debug,12,(&Debug,
			 "add_ipv_option: update_interface_list failed: (errno=%d) %s\n",
			 err,strerror(err)));
	res = ipv_opt_succeed;
    }
    
#else
    /* Can't check interfaces */
    
    res = ipv_opt_succeed;

#endif
    if (res >= ipv_opt_succeed) {
	
	if (ipv_option_array_len >= NUM_protocols) {
	    /* Should not be possible !!! */

	    DPRINT(Debug,12,(&Debug,
			     "add_ipv_option: ipv_option_array_len=%zu >= %d\n",
			     ipv_option_array_len,NUM_protocols));

	    res = ipv_opt_unsupported;
	    goto fail;
	}

	ipv_option_array[ipv_option_array_len] = af;
	ipv_option_copy [ipv_option_array_len] = opt;
			 
	ipv_option_array_len++;
    }
	
 fail:

    DPRINT(Debug,12,(&Debug,
		     "add_ipv_option=%d",
		     res));
    switch (res) {
    case ipv_opt_already:     DPRINT(Debug,12,(&Debug," ipv_opt_already"));     break;
    case ipv_opt_unsupported: DPRINT(Debug,12,(&Debug," ipv_opt_unsupported")); break;
    case ipv_opt_succeed:     DPRINT(Debug,12,(&Debug," ipv_opt_succeed"));     break;
    case ipc_opt_noactice:    DPRINT(Debug,12,(&Debug," ipc_opt_noactice"));    break;
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    return res;
}

#ifdef REMOTE_MBX

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static int parse_delay P_(( struct sec_ms *delay,
			    char delay_value[],
			    size_t delay_value_size,
			    char *f,
			    char *q /* May be NULL */,
			    int lineno, const char *filename));
static int parse_delay(delay,delay_value,delay_value_size,f,q,
		       lineno,filename)
     struct sec_ms * delay;
     char            delay_value[];
     size_t          delay_value_size;
     char          * f;
     char          * q /* May be NULL */;
     int             lineno;
     const char    * filename;
{
    int ok = 1;
       
    errno = 0;
    if (q) {
	char * end = NULL;
	long val = strtol(q,&end,10);
	int err = errno;
	
	delay_value[0] = '\0';
	
	if (0 == err) {
	    if ('\0' == *end ||
		0 == strcmp(end,"s")) {
		
		if (val > 0 && val <= INT_MAX) {
		    delay->timeout_sec = val;
		    delay->timeout_ms  = 0;
		    
		    if (strlen(q) < delay_value_size)
			strfcpy(delay_value,q, delay_value_size);
		    
		} else {
		    if (filename)
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPCFile,
					  "Bad value %s on connect-mode keyword %s on line %d on file %s"),
			      q,f,lineno,filename);
		    else
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPC,
					  "Bad value %s on connect-mode keyword %s"),
				  q,f);
		    ok = 0;
		    if (val <= 0) {
			delay->timeout_sec = 1;
			delay->timeout_ms  = 0;
		    } else {
			delay->timeout_sec = INT_MAX;
			delay->timeout_ms  = 0;
		}
		}
	    } else if (0 == strcmp(end,"ms")) {
		if (val > 0 || val/1000 <= INT_MAX) {
		    delay->timeout_sec = val / 1000;
		    delay->timeout_ms  = val % 1000;
		    
		    if (strlen(q) < delay_value_size)
			strfcpy(delay_value, q, delay_value_size);
		    
		} else {
		    if (filename)
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPCFile,
				      "Bad value %s on connect-mode keyword %s on line %d on file %s"),
				  q,f,lineno,filename);
		    else
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPC,
					  "Bad value %s on connect-mode keyword %s"),
				  q,f);
		    ok = 0;
		    if (val <= 0) {
			delay->timeout_sec = 0;
			delay->timeout_ms  = 100;
		    } else {
			delay->timeout_sec = INT_MAX;
			delay->timeout_ms  = 0;
		    }
		}
	    } else
		goto bad_long_value;
	} else {
	    double vald;
	    
	bad_long_value:
	    
	    errno = 0;
	    vald = strtod(q,&end);
	    err = errno;
	    
	    if (0 == err) {
		if ('\0' == *end ||
		    0 == strcmp(end,"s")) {
		    
		    if (vald > 0 && vald <= INT_MAX) {
			int secs = vald;
			double left = vald-secs;
			unsigned short msec = left * 1000;
			
			if (msec > 999)
			    msec = 999;
			
			delay->timeout_sec = secs;
			delay->timeout_ms  = msec;
			
			if (strlen(q) < delay_value_size)
			    strfcpy(delay_value, q, delay_value_size);
			
		    } else {
			if (filename)
			    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPCFile,
					      "Bad value %s on connect-mode keyword %s on line %d on file %s"),
				      q,f,lineno,filename);
			else
			    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPC,
					      "Bad value %s on connect-mode keyword %s"),
				      q,f);
		    
			ok = 0;
			
			if (vald <= 0) {
			    delay->timeout_sec = 0;
			    delay->timeout_ms  = 100;
			} else {
			    delay->timeout_sec = INT_MAX;
			    delay->timeout_ms  = 0;
			}
		    }
		} else {

		    ok = 0;
		    
		    if (filename)
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPCFile,
					  "Bad value %s on connect-mode keyword %s on line %d on file %s"),
				  q,f,lineno,filename);
		    else
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPC,
					  "Bad value %s on connect-mode keyword %s"),
				  q,f);			
		    
		    
		}
		
	    } else {
		ok = 0;
		
		if (filename)
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPCFileErr,
				      "Bad value %s on connect-mode keyword %s on line %d on file %s: %s"),
			      q,f,lineno,filename,
			      strerror(err));
		else
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPCErr,
				      "Bad value %s on connect-mode keyword %s: %s"),
			      q,f,
			      strerror(err));			
	    }
	}
    } else {
	ok = 0;
	
	if (filename)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMissingValueKwPCFile,
			      "Missing value on connect-mode keyword %s on line %d on file %s"),
		      f,lineno,filename);
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMissingValueKwPC,
			      "Missing value on connect-mode keyword %s"),
		      f);
	
    }
    
    return ok;
}

enum connect_mode_val {
    connect_mode_blocking = 0,
    connect_mode_nonblocking,
    connect_mode_parallel,
    NUM_connect_mode
};

static void dump_delay P_((char **return_buffer,
			   const char *name,
			   struct sec_ms *delay,
			   char delay_value[],
			   enum connect_mode_val mode_val
			   ));
static void dump_delay(return_buffer,name,delay,delay_value, mode_val)
     char         ** return_buffer;
     const char    * name;
     struct sec_ms * delay;
     char            delay_value[];
     enum connect_mode_val mode_val;
{
    if (delay_value[0]) {

	(*return_buffer) = strmcat((*return_buffer),"; ");
	(*return_buffer) = strmcat((*return_buffer),name);
	(*return_buffer) = strmcat((*return_buffer),"=");
	(*return_buffer) = strmcat((*return_buffer),delay_value);
	    	    
    } else if (mode_val ||
	       delay->timeout_sec ||
	       delay->timeout_ms) {
	    
	if (0 <= delay->timeout_sec &&
	    0 == delay->timeout_ms) {
	    char * t =  elm_message(FRM("; %s=%ds"),
				    name,
				    delay->timeout_sec);
	    (*return_buffer) = strmcat((*return_buffer),t);
	    free(t);
	} else if (0 == delay->timeout_sec &&
		   0  < delay->timeout_ms) {
	    char * t =  elm_message(FRM("; %s=%dms"),
				    name,
				    delay->timeout_ms);
	    (*return_buffer) = strmcat((*return_buffer),t);
	    free(t);
	} else if (0   <= delay->timeout_sec &&
		   999 >= delay->timeout_ms) {
	    char * t =  elm_message(FRM("; %s=%d.%03us"),
				    name,
				    delay->timeout_sec,
				    (unsigned) delay->timeout_ms);
	    (*return_buffer) = strmcat((*return_buffer),t);
	    free(t);
	}
    }    
}


static const char * CONNECT_MODE[NUM_connect_mode] = {
    "blocking",
    "non-blocking",
    "parallel",
};

static enum connect_mode_val  connect_mode
#ifdef HAVE_SOERROR
= connect_mode_parallel
#endif
    ;


static char connect_mode_addrdelay_value[SHORT];   /* cache exact string */
static struct sec_ms connect_mode_addrdelay
#ifdef HAVE_SOERROR
= { 2, 0 }
#endif
    ;

static char connect_mode_portdelay_value[SHORT];   /* cache exact string */
static struct sec_ms connect_mode_portdelay
#ifdef HAVE_SOERROR
= { 0 , 600 }
#endif
    ;

static char connect_mode_display_time_value[SHORT];   /* cache exact string */
static struct sec_ms connect_mode_display_time
#ifdef HAVE_SOERROR
= { 0 , 300 }
#endif
    ;


/* Arbitary low value */
static const int MAX_connect_mode_limit = 99;
static int connect_mode_limit 
#ifdef HAVE_SOERROR
= 8
#endif
    ;


/* modifies string f, return param value or NULL */
char * conf_split_param_value(f)
     char * f;
{
    char * q = strchr(f,'=');

    DPRINT(Debug,13,(&Debug,"conf_split_param_value: f=\"%s\"\n",
		     f));
    
    if (q) {
	char * x = q;
	
	while (x > f && ' ' == *(x-1))
	    x--;
	*x = '\0';
	
	q++;
	
	while (' ' == *q)
	    q++;

	DPRINT(Debug,13,(&Debug,"conf_split_param_value=\"%s\", f=\"%s\"\n",
			 q,f));
    }
    
    return q;
}

/* Return 1 if value found */
int conf_select_value(f,array,array_len,value)
     const char *f;
     const char *array[];
     const int array_len;
     int *value  /* Return value */;
{
    int x;
    
    DPRINT(Debug,13,(&Debug,"conf_select_value: f=\"%s\"\n",
		     f));
    
    for (x = 0;
	 x < array_len &&
	     array[x];
	 x++) {
	DPRINT(Debug,14,(&Debug,"conf_select_value: #%d=\"%s\"\n",
			 x,array[x]));
	
	if (0 == strcmp(f,array[x])) {
	    *value = x;
	    DPRINT(Debug,13,(&Debug,"conf_select_value=1, *value=%d\n",
			     *value));

	    return 1;
	}
    }

    return 0;
}


static char * connect_mode_func_return_buffer = NULL;

enum connect_mode_param_kw {
    cmp_delay,
    cmp_address_delay,
    cmp_port_delay,
    cmp_display_time,
    cmp_concurrency_limit,
    NUM_connect_mode_param
};

static const char * CONNECT_MODE_PARAM[NUM_connect_mode_param] = {
    "delay",
    "address-delay",
    "port-delay",
    "display-time",
    "concurrency-limit"
};
    

#ifdef ANSI_C
option_func connect_mode_func;
#endif
int connect_mode_func(value,enter,lineno,filename,
		      read_flags)
     char      ** value;
     int          enter;
     int          lineno;
     const char * filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ok = 0;

    static char delay_value[SHORT];   /* cache exact string */
    static struct sec_ms delay;

    
    if (enter) {
	char * WALK;
	char * temp = safe_strdup(*value);
	char * f = mime_parse_content_opts(temp, &WALK);
	int x;

	int copy_from_delay   =  0;
	int addrdelay_set     = 0;
	int portdelay_set     = 0;

	
	if (!f) {
	    goto fail;
	}
	
	if (conf_select_value(f,CONNECT_MODE,NUM_connect_mode,
			      &x)) {
	    ok           = 1;
	    connect_mode = x;
	}

	if (ok) {

#ifndef HAVE_SOERROR
	    if (connect_mode) {
		ok = 0;
	    }
#endif
	    if (!ok) {
		if (filename)
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmParallelConnectNotAvailFIle,
				      "Parallel connect() is not availeble (connect-mode keyword %s on line %d on file %s)"),
			  f,lineno,filename);
		else
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmParallelConnectNotAvail,
				      "Parallel connect() is not availeble (connect-mode keyword %s)"),
			      f);
		connect_mode = 0;
	    }

	} else {
	    
	    if (filename)
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnknowKeywordPCFile,
				  "Unknown connect-mode keyword %s on line %d on file %s"),
			  f,lineno,filename);
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnknowKeywordPC,
				  "Unknown connect-mode keyword %s"),
			  f);	    	
	}

	while (NULL != (f = mime_parse_content_opts(NULL, &WALK))) {
	    /* modifies string f, return param value or NULL */
	    char * q = conf_split_param_value(f);
	    int p;
	    
	    if (conf_select_value(f,CONNECT_MODE_PARAM,
				  NUM_connect_mode_param,
				  &p)) {
		enum connect_mode_param_kw param = p;

		switch (param) {
		case cmp_delay: 
		    if (! parse_delay(&delay,
				      delay_value,
				      sizeof delay_value,
				      f,q,lineno,filename))
			ok = 0;
		    else
			copy_from_delay = 1;
		    break;
		    
		case cmp_address_delay:
		    if (! parse_delay(&connect_mode_addrdelay,
				      connect_mode_addrdelay_value,
				      sizeof connect_mode_addrdelay_value,
				      f,q,lineno,filename))
		    ok = 0;
		    else
			addrdelay_set = 1;
		    break;

		case cmp_port_delay:
		    if (! parse_delay(&connect_mode_portdelay,
				      connect_mode_portdelay_value,
				      sizeof connect_mode_portdelay_value,
				      f,q,lineno,filename))
			ok = 0;
		    else
			portdelay_set = 1;
		    break;

		case cmp_display_time:		
		    if (! parse_delay(&connect_mode_display_time,
				      connect_mode_display_time_value,
				      sizeof connect_mode_display_time_value,
				      f,q,lineno,filename))
			ok = 0;
		    break;

		case cmp_concurrency_limit:
		    if (q) {
			char * end = NULL;
			long val;
			int err;
			
			errno = 0;
			val = strtol(q,&end,10);
			err = errno;
			
			if (0 != err   ||
			    '\0' != *end ||
			    val < 0      ||
			    val > MAX_connect_mode_limit)   {
			    
			    ok = 0;
			    
			    if (err) {
				if (filename)
				    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPCFileErr,
						      "Bad value %s on connect-mode keyword %s on line %d on file %s: %s"),
					      q,f,lineno,filename,
					      strerror(err));
				else
				    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPCErr,
						      "Bad value %s on connect-mode keyword %s: %s"),
					      q,f,
					      strerror(err));			
				
			    } else {
				if (filename)
				    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPCFile,
						      "Bad value %s on connect-mode keyword %s on line %d on file %s"),
					      q,f,lineno,filename);
				else
				    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueKwPC,
						      "Bad value %s on connect-mode keyword %s"),
					      q,f);
			    }
			    
			    if (val < 0)
				connect_mode_limit = 0;
			    else if (val > MAX_connect_mode_limit)
				connect_mode_limit = MAX_connect_mode_limit;
			    else
				connect_mode_limit = val;
			    
			} else
			    connect_mode_limit = val;
		    } else {
			
			ok = 0;
			
			if (filename)
			    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMissingValueKwPCFile,
					      "Missing value on connect-mode keyword %s on line %d on file %s"),
				      f,lineno,filename);
			else
			    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMissingValueKwPC,
					      "Missing value on connect-mode keyword %s"),
				      f);
			
		    }
		    break;

		case NUM_connect_mode_param:   /* Not used */
		    break;
		}
		
	    } else {
		if (filename)
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnknowKeywordPCFile,
				      "Unknown connect-mode keyword %s on line %d on file %s"),
			      f,lineno,filename);
		else
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnknowKeywordPC,
				      "Unknown connect-mode keyword %s"),
			      f);
		ok = 0;
	    }
	}

	if (copy_from_delay && !addrdelay_set) {
	    DPRINT(Debug,12,(&Debug,"Copying connect-mode delay=(%d sec, %u ms) to address-delay\n",
			     delay.timeout_sec,
			     (unsigned) delay.timeout_ms));

	    connect_mode_addrdelay = delay;
	    strfcpy(connect_mode_addrdelay_value,delay_value, sizeof connect_mode_addrdelay_value);
	    
	}

	if (copy_from_delay && !portdelay_set) {
	    DPRINT(Debug,12,(&Debug,"Copying connect-mode delay=(%d sec, %u ms) to port-delay\n",
			     delay.timeout_sec,
			     (unsigned) delay.timeout_ms));

	    connect_mode_portdelay = delay;
	    strfcpy(connect_mode_portdelay_value,delay_value, sizeof connect_mode_portdelay_value);	    
	}
	
    fail:
	free(temp);
    } else {
	/* static pointer to buffer accross invocations */
 	
	if (connect_mode >= 0 && connect_mode < NUM_connect_mode) {
	    connect_mode_func_return_buffer =
		strmcpy(connect_mode_func_return_buffer,
			CONNECT_MODE[connect_mode]);
	    ok = 1;
	} else
	    goto fail1;

	if (0 == strcmp(delay_value,
			connect_mode_addrdelay_value) &&
	    same_sec_ms(&delay,
			&connect_mode_addrdelay)

	    &&

	    0 == strcmp(delay_value,
			connect_mode_portdelay_value) &&
	    same_sec_ms(&delay,
			&connect_mode_portdelay)

	    ) {
	    
	    dump_delay(&connect_mode_func_return_buffer,
		       CONNECT_MODE_PARAM[cmp_delay],
		       &delay,
		       delay_value,
		       connect_mode);
	} else {
		
	    dump_delay(&connect_mode_func_return_buffer,
		       CONNECT_MODE_PARAM[ cmp_address_delay],
		       &connect_mode_addrdelay,
		       connect_mode_addrdelay_value,
		       connect_mode);

	    dump_delay(&connect_mode_func_return_buffer,
		       CONNECT_MODE_PARAM[cmp_port_delay],
		       &connect_mode_portdelay,
		       connect_mode_portdelay_value,
		       connect_mode);
	    
	}
	    	
	dump_delay(&connect_mode_func_return_buffer,
		   CONNECT_MODE_PARAM[cmp_display_time],
		   &connect_mode_display_time,
		   connect_mode_display_time_value,
		   connect_mode);

	if (connect_mode ||
	    connect_mode_limit > 0) {
	    char * t =  elm_message(FRM("; %s=%d"),
				    CONNECT_MODE_PARAM[cmp_concurrency_limit],
				    connect_mode_limit);

	    connect_mode_func_return_buffer =
		strmcat(connect_mode_func_return_buffer,t);
	    free(t);	    
	}

	*value =  connect_mode_func_return_buffer;
    }

 fail1:
    return ok;
}

void free_remote_mbx_config()

{

    if (connect_mode_func_return_buffer) {
	DPRINT(Debug,10,(&Debug,"free_remote_mbx_config: freeing connect-mode return buffer %s\n",
			 connect_mode_func_return_buffer));
	free(connect_mode_func_return_buffer);
	connect_mode_func_return_buffer = NULL;
    }
    
}

static void init_remote_account P_((struct remote_account *ra));
static void init_remote_account(ra)
     struct remote_account *ra;
{
    ra->magic                 = REMOTE_ACCOUNT_magic;

    ra->addrsize              = 0;
    ra->hostaddr.dummy        = NULL;
    ra->hostname              = NULL;
    ra->service               = NULL;
    ra->stream                = NULL;
    ra->username              = NULL;
    ra->host                  = NULL;

    ra->socknamesize          = 0;
    ra->socknameaddr.dummy    = NULL;
}

void zero_remote_account(ra)
     struct remote_account *ra;
{
    /* bzero is defined hdrs/elm_defs.h */
    bzero ((void *)ra, sizeof (struct remote_account));

    init_remote_account(ra);
}

struct remote_account * malloc_remote_account(username,host)
     const char *username;
     const char *host;
{
    struct remote_account *ra = safe_zero_alloc(sizeof(*ra));

    init_remote_account(ra);

    if (username)
	ra->username = strmcpy(ra->username,username);
    if (host)
	ra->host = strmcpy(ra->host,host);

    return ra;
}

int remote_account_OK(st_size)
     size_t st_size;
{

    if (st_size == sizeof(struct remote_account))
	return 1;

    DPRINT(Debug,1,(&Debug,
		    "remote_account_OK: Size mismatch st_size=%ld != %ld\n",
		    (long)st_size,(long)sizeof(struct remote_account)));

    return 0;
}

void clear_remote_account0(ra,seen_badpid)
     struct remote_account *ra;
     int seen_badpid;
{
    if (ra->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "clear_remote_account0",
	      "Bad magic number",0);

    DPRINT(Debug,20,(&Debug,
		     "clear_remote_account0: ra=%p",
		     ra));
    if (ra->hostaddr.dummy) {
	free(ra->hostaddr.dummy);
	ra->hostaddr.dummy = NULL;
    }
    ra->addrsize      = 0;

    if (ra->hostname) {
	DPRINT(Debug,20,(&Debug," hostname=%s",
			 ra->hostname));

	free(ra->hostname);
	ra->hostname = NULL;
    }
    if (seen_badpid) {
	DPRINT(Debug,20,(&Debug,", not our connection, owner pid %d",
			 seen_badpid));	    
    }    
    
    DPRINT(Debug,20,(&Debug,"\n"));
    
    if (ra->service)
	free_service_entry(& (ra->service));

    if (ra->stream) 
	FreeStreamStack0(& (ra->stream),
			 seen_badpid,
			 1 /* Force clearing of stack even when this is not last reference */);
    if (ra->username) {
	free(ra->username);
	ra->username = NULL;
    }
    if (ra->host) {
	free(ra->host);
	ra->host = NULL;
    }

    if (ra->socknameaddr.dummy) {
	free(ra->socknameaddr.dummy);
	ra->socknameaddr.dummy = NULL;
    }
    ra->socknamesize = 0;
}

void clear_remote_account(ra)
     struct remote_account *ra;
{
    if (ra->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "clear_remote_account",
	      "Bad magic number",0);
    
    clear_remote_account0(ra,
			  0 /* Assume no badpid */);
    
}

void free_remote_account(ra)
     struct remote_account **ra;
{
    clear_remote_account(*ra);

    (*ra)->magic = 0; /* Invalidate */
    free(*ra);
    (*ra) = NULL;
}

enum connect_result {
    connect_fail_abort = -2,
    connect_fail   = -1,
    connect_none   = 0,
    connect_OK     = 1,
    connect_in_progress = 2
};

#define BGCONNECT_INFO_magic	0xF524

static struct bgconnect_info {
    unsigned short     magic;       /* BGCONNECT_INFO_magic */
    int                socket;

    int                refcount;
    
    size_t             addrsize;
    
    union SOCKADDR_ptr hostaddr;
    char             * hostname;    /* name corresponding to hostaddr */ 
    char             * addrstring;  /* hostaddr as string             */

    int                portgroup;
    
    struct schedule_timelimit time_added;
    
    enum connect_result result;
    int                 error;
    
} * malloc_backgroud_connect_info P_((int socket,struct remote_account * ra,
				      const char *addrstring, int portgroup)) UNUSED_FUNCOK;

static struct bgconnect_info * malloc_backgroud_connect_info(socket,ra,addrstring,portgroup)
     int                     socket;
     struct remote_account * ra;
     const char            * addrstring;
     int                     portgroup;
{
    struct bgconnect_info * ret = safe_zero_alloc(sizeof (*ret));
    char * X = NULL;

    ret->magic   = BGCONNECT_INFO_magic;
    ret->socket  = socket;
    ret->refcount = 1;
    
    ret->hostaddr.dummy = NULL;
    ret->addrsize       = 0;
    ret->hostname       = NULL;

    if (ra) {
	if (ra->magic  != REMOTE_ACCOUNT_magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "malloc_backgroud_connect_info",
		  "Bad magic number (remote_account)",0);

	if (! set_SOCKADDR_ptr_helper(& (ret->addrsize),
				      & (ret->hostaddr),
				      ra->addrsize,
				      ra->hostaddr)){
	    DPRINT(Debug,12,(&Debug,"malloc_backgroud_connect_info: Failed to set address\n"));
	}

	/* hostname correspong to IP-address */
	if (ra->hostname)
	    ret->hostname = strmcpy(ret->hostname,
				    ra->hostname);
	else if (ra->host)
	    ret->hostname = strmcpy(ret->hostname,
				    ra->host);
    }

    ret->addrstring = addrstring ? safe_strdup(addrstring) : NULL;
    ret->portgroup  = portgroup;
    
    schedule_set_current_time(&(ret->time_added));
    
    ret->result = connect_none;
    ret->error  = 0;

       
    DPRINT(Debug,12,(&Debug,
		     "malloc_backgroud_connect_info=%p (socket=%d, hostname=%s, address=%s; portgroup=%d)",
		     ret, ret->socket,ret->hostname ? ret->hostname : "<not set>",
		     ret->addrstring ? ret->addrstring : "?",
		     portgroup));

    X =  schedule_timeout_string(&(ret->time_added));
    if (X) {
	DPRINT(Debug,12,(&Debug,", add time: %s",
			 X));
	
	free(X);
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    return ret;
}

static void free_backgroud_connect_info P_((struct bgconnect_info **info));
static void free_backgroud_connect_info(info)
     struct bgconnect_info **info;
{
    if (BGCONNECT_INFO_magic != (*info)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_backgroud_connect_info",
	      "Bad magic number",0);

    if ((*info)->refcount < 1)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_backgroud_connect_info",
	      "Bad refcount",0);

    (*info)->refcount--;
    
    DPRINT(Debug,12,(&Debug,
		     "free_backgroud_connect_info: %p => refcount=%d\n",
		     *info,(*info)->refcount));
    
    if ((*info)->refcount > 0) {
	/* Just remove this reference */

	*info = NULL;
	return;
    }
	   
	   
    if (-1 != (*info)->socket) {
	DPRINT(Debug,12,(&Debug,
			 "free_backgroud_connect_info:  closing socket (%d)\n",
			 (*info)->socket));
	close((*info)->socket);
	(*info)->socket = -1;
    }

    if ((*info)->hostaddr.dummy) {
	free((*info)->hostaddr.dummy);
	(*info)->hostaddr.dummy = NULL;
    }
    (*info)->addrsize      = 0;

    if ((*info)->hostname) {
	free((*info)->hostname);
	(*info)->hostname = NULL;
    }

    if ((*info)->addrstring) {
	free((*info)->addrstring);
	(*info)->addrstring = 0;
    }
    
    (*info)->magic = 0;  /* Invalidate */
    free (*info);
    *info = NULL;
}

static void inc_backgroud_connect_info_refcount P_((struct bgconnect_info *info));
static void inc_backgroud_connect_info_refcount(info)
     struct bgconnect_info *info;
{
    if (BGCONNECT_INFO_magic != info->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "inc_backgroud_connect_info_refcount",
	      "Bad magic number",0);

    info->refcount++;
}

/* if returns 0 -- action is removed from list */
S_(action_routine bgconnect_socket_writable)
static enum action_status bgconnect_socket_writable P_((int fd,   union action_routine_data data,
							const struct schedule_timelimit * now)) UNUSED_FUNCOK;
static enum action_status bgconnect_socket_writable(fd,data,now)
     int                       fd;
     union action_routine_data data;
     const struct schedule_timelimit * now;
{
    struct bgconnect_info *info = data.connect;
    socklen_t len               = sizeof (info->error);
    
    if (BGCONNECT_INFO_magic != info->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "bgconnect_socket_writable",
	      "Bad magic number",0);

#ifdef HAVE_SOERROR
    if (-1 == getsockopt(fd, SOL_SOCKET, SO_ERROR, &(info->error), &len)) {

	int err = errno;

	lib_error(CATGETS(elm_msg_cat, MeSet,MeFailedToReadError,
			  "Failed to read error: %s"),
		  strerror(err));

	return action_disable;
    }

    if (info->error) {
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_socket_writable: fd=%d connect() error: %s\n",
			 fd,strerror(info->error)));
    }
    
    switch (info->error) {
    case EINPROGRESS:   /* Should not happen? */
	
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_socket_writable: Still on progress\n"));
	
	info->result =  connect_in_progress;
	return action_continue;
	
    case EINTR:         /* Should not happen? */
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_socket_writable: Interrupt?\n"));
	info->result = connect_fail;
	break;

    case ECONNREFUSED: 
    case ETIMEDOUT:
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_socket_writable: Refusing connection or timeout. Not fatal.\n"));
	info->result = connect_none;
	break;
    case 0:
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_socket_writable: Connection succeed\n"));
	info->result = connect_OK;
	break;
	
    default:
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_socket_writable: Treating as fatal error\n"));
	info->result = connect_fail_abort;
	break;
    }
#else
    DPRINT(Debug,1,(&Debug,
		    "bgconnect_socket_writable: Not supported\n"))
#endif


    return action_disable;
}

S_(badpid_action_f bgconnect_socket_badpid)
static enum badpid_status bgconnect_socket_badpid P_((int fd,
			       union action_routine_data         data,
			       int badpid));
static enum badpid_status bgconnect_socket_badpid(fd,data,badpid)
     int fd;
     union action_routine_data         data;
     int badpid;
{
    struct bgconnect_info *info = data.connect;

     if (BGCONNECT_INFO_magic != info->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "bgconnect_socket_badpid",
	      "Bad magic number",0);

     DPRINT(Debug,12,(&Debug,
		      "bgconnect_socket_badpid: Treating as fatal error\n"));
     info->result = connect_fail_abort;

     return badpid_remove       /* remove action */;
}


S_(free_action_data_f bgconnect_free_data)
static void bgconnect_free_data P_((union action_routine_data * data,
				    int fd,
				    int badpid)) UNUSED_FUNCOK;
static void bgconnect_free_data(data,fd,badpid)
     union action_routine_data * data;
     int fd;
     int badpid;
{
    free_backgroud_connect_info(& (data->connect));
}

S_(inc_action_data_refcount_f bgconnect_inc_data_refcount)
static void bgconnect_inc_data_refcount P_((union action_routine_data data)) UNUSED_FUNCOK;
static void bgconnect_inc_data_refcount(data)
     union action_routine_data data;
{
    inc_backgroud_connect_info_refcount(data.connect);
}

static enum connect_result connect_one_IN P_((struct remote_account  * ra,
					      int                    * cur_socket,
					      int                    * last_error,
					      struct cancel_data     * can,
					      int                      report_port,
					      struct bgconnect_info ** nonblocking_connect,
					      int                      portgroup));
static enum connect_result connect_one_IN(ra,cur_socket,last_error,can,
					  report_port,nonblocking_connect,
					  portgroup)
     struct remote_account * ra;
     int                   * cur_socket;
     int                   * last_error;
     struct cancel_data    * can;
     int                     report_port;
     struct bgconnect_info ** nonblocking_connect;
     int                      portgroup;
{
    enum connect_result  r = connect_none;

    char mybuffer[256] UNUSED_VAROK;
    const char * s     UNUSED_VAROK;
    int r1;
    
    if (ra->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "connect_one_IN",
	      "Bad magic number",0);

#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
    
    if (!ra->hostaddr.dummy) {
	DPRINT(Debug,12,(&Debug,
			 "connect_one_IN: No address\n"));
	return connect_fail;
    }

    s = give_SOCKADDR_ptr_as_string(ra->hostaddr,ra->addrsize,
				    mybuffer,sizeof mybuffer);
			    
    DPRINT(Debug,12,(&Debug,
		     "connect_one_IN: family=%d size=%ld",
		     ra->hostaddr.sa->sa_family,
		     (long)(ra->addrsize)));
    if (portgroup) {
	DPRINT(Debug,12,(&Debug," (portgroup=%d)",
			 portgroup));
    }
    
    if (s) {
	DPRINT(Debug,12,(&Debug," addr=%s",s));
    }
    if (ra->hostname) {
	DPRINT(Debug,12,(&Debug," hostname=%s",ra->hostname));
    }

    switch (ra->hostaddr.sa->sa_family) {
    case AF_INET:
	DPRINT(Debug,12,(&Debug," IPv4 port=%d\n",
			 ntohs(ra->hostaddr.sin->sin_port)));
	
	if (report_port)
	    cancel_progress(can,CATGETS(elm_msg_cat, MeSet,MeConPort,
					"Trying port %d"),
			    ntohs(ra->hostaddr.sin->sin_port));

	break;
#ifdef HAVE_IN6
    case AF_INET6: 
	DPRINT(Debug,12,(&Debug," IPv6 port=%d\n",
			 ntohs(ra->hostaddr.sin6->sin6_port)));


	if (report_port)
	    cancel_progress(can,CATGETS(elm_msg_cat, MeSet,MeConPort,
					"Trying port %d"),
			    ntohs(ra->hostaddr.sin6->sin6_port));
	
	break;
#endif
    default:
	DPRINT(Debug,12,(&Debug," ????\n"));
	break;
    }
    
    /* r == 0: not succeed
       r <  0: fatal error
       r >  0: succeed
    */


 retry:
    if (-1 != (*cur_socket)) {
	/* Some systems require that socket is closed even
	   after unsuccesfully connect() -- it can not be reused
	*/
	DPRINT(Debug,12,(&Debug,
			 "connect_one_IN:  closing socket (%d) for reopening\n",
			 (*cur_socket)));
	close(*cur_socket);
	(*cur_socket) = -1;
    }
    
    (*cur_socket) = socket(ra->hostaddr.sa->sa_family,
			   SOCK_STREAM,IPPROTO_TCP);
    if (-1 == (*cur_socket)) {
	int err =errno;
	lib_error(CATGETS(elm_msg_cat, MeSet,MeFailedToCreate,
			  "Failed to create socket: %s"),
		  strerror(err));
	r = connect_fail;
	goto clean;
    }
    DPRINT(Debug,12,(&Debug,
		     "connect_one_IN: socket=%d\n",(*cur_socket)));
    
    if (nonblocking_connect) {

	if (*nonblocking_connect) {
	    DPRINT(Debug,12,(&Debug,
			     "connect_one_IN: Freeing old background connect info %p\n",
			     *nonblocking_connect));
	    free_backgroud_connect_info(nonblocking_connect);
	}
	
	/* We set folder to non-blocking after connect, because
	 * non-blocking connect is little complicate to use
	 */
	
	if (-1 == (r1 = fcntl(*cur_socket,F_SETFL,O_NONBLOCK))) {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,12,(&Debug,
			     "connect_one_IN: fcntl [F_SETFL O_NONBLOCK] failed: %s\n",
			     strerror(err)));
	} else if (0 == r1) {
	    DPRINT(Debug,12,(&Debug,
			     "connect_one_IN: fcntl(%d,F_SETFL,O_NONBLOCK) done\n",
			     *cur_socket));
	}
    }

    if (-1 == (r1 = fcntl(*cur_socket,F_SETFD,FD_CLOEXEC))) {
	int err UNUSED_VAROK = errno;
	DPRINT(Debug,12,(&Debug,
			 "connect_one_IN: fcntl [F_SETFD FD_CLOEXEC] failed: %s\n",
			 strerror(err)));
	
    } else if (0 == r1) {
	DPRINT(Debug,12,(&Debug,
			 "connect_one_IN: fcntl(%d,F_SETFD,FD_CLOEXEC) done\n",
			 *cur_socket));
    }
    
    if (-1 == connect((*cur_socket),ra->hostaddr.sa,
		      ra->addrsize)) {
	(*last_error) = errno;
	DPRINT(Debug,12,(&Debug,
		    "connect_one_IN: connect failed: %s\n",
		    strerror(*last_error)));
	if (EINPROGRESS == *last_error) {
	    r = connect_in_progress;

	    if (nonblocking_connect) {
		union action_routine_data data;
		
		*nonblocking_connect = malloc_backgroud_connect_info(*cur_socket,
								     ra,s,portgroup);
		(*nonblocking_connect)->result = r;
		(*nonblocking_connect)->error  = *last_error;
		
		DPRINT(Debug,12,(&Debug,
				 "connect_one_IN: Moving socket (%d) to background connect info %p\n",
				 (*cur_socket),(*nonblocking_connect)));


		data.connect = *nonblocking_connect;
		
		change_action2(*cur_socket,-1,
			       no_action_routine,bgconnect_socket_writable,no_action_routine,
			       bgconnect_free_data,bgconnect_inc_data_refcount,
			       bgconnect_socket_badpid,
			       data);
			      
		*cur_socket = -1;
	    }
	}
	
	if (connect_none == r && EINTR == *last_error) {
	    
	    if (is_canceled(can)) {
		DPRINT(Debug,12,(&Debug,"connect canceled...\n"));
		r = connect_fail;
	    } else {
		DPRINT(Debug,12,(&Debug,"Retrying connect...\n"));
		goto retry;
	    }
	    
	}
    } else
	r = connect_OK;
		
    /* If got connection to host and and it refused
       connection, try another ports
    */
    if (r <= connect_none) { 
	switch (*last_error) {
	case ECONNREFUSED: 
	case ETIMEDOUT:
	    DPRINT(Debug,12,(&Debug,
			     "connect_one_IN: Refusing connection or timeout. Not fatal.\n"));
	    break;
	default:
	    DPRINT(Debug,12,(&Debug,
			     "connect_one_IN: Treating as fatal error\n"));
	    r = connect_fail_abort;		   
	    break;
	}
    }

 clean:
#else
    /* NOT SUPPORTED */
    r = connect_fail;
#endif
    
    DPRINT(Debug,12,(&Debug,
		     "connect_one_IN=%d",r));
    switch (r) {
    case connect_fail_abort:  DPRINT(Debug,12,(&Debug," (connect_fail_abort)"));  break;
    case connect_fail:        DPRINT(Debug,12,(&Debug," (connect_fail)"));        break;
    case connect_none:        DPRINT(Debug,12,(&Debug," (connect_none)"));        break;
    case connect_OK:          DPRINT(Debug,12,(&Debug," (connect_OK)"));          break;
    case connect_in_progress: DPRINT(Debug,12,(&Debug," (connect_in_progress)")); break;
    }
    if (-1 != *cur_socket) {
	DPRINT(Debug,12,(&Debug," *cur_socket=%d",*cur_socket));
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return r;
}

#define BGCONNECT_LIST_magic	0xF525
static struct bgconnect_list {
    unsigned short     magic;       /* BGCONNECT_LIST_magic */

    struct service_entry *se;

    struct bgconnect_list_entry {
	struct bgconnect_info     * connect;
	int                         port;
	const struct service_type * s_type;

    } * list;
    int list_len;
} * malloc_bgconnect_list P_((struct service_entry *se));
static struct bgconnect_list * malloc_bgconnect_list(se)
     struct service_entry *se;
{
    struct bgconnect_list * ret = safe_zero_alloc(sizeof (*ret));

    ret->magic = BGCONNECT_LIST_magic;

    ret->se = se;
    if (ret->se)
	inc_service_entry_refcount(ret->se);

    ret->list     = NULL;
    ret->list_len = 0;

    return ret;
}

static void free_bgconnect_list P_((struct bgconnect_list **list));
static void free_bgconnect_list(list)
     struct bgconnect_list **list;
{

    if (BGCONNECT_LIST_magic != (*list)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_bgconnect_list",
	      "Bad magic number",0);

    if ((*list)->se)
	free_service_entry(& ((*list)->se));

    if ((*list)->list) {
	int i;

	for (i = 0; i < (*list)->list_len; i++) {

	    if ((*list)->list[i].connect)
		free_backgroud_connect_info(& ((*list)->list[i].connect));

	}

	free((*list)->list);
	(*list)->list = NULL;
    }
    (*list)->list_len = 0;

    (*list)->magic = 0;  /* Invalidate */
    free(*list);
    *list = NULL;
}

static void bgconnect_add_to_list P_((struct bgconnect_list *list, struct bgconnect_info *info,
				      int port,const struct service_type * s_type));
static void bgconnect_add_to_list(list,info,port,s_type)
     struct bgconnect_list *list;
     struct bgconnect_info *info;
     int port;
     const struct service_type * s_type; /* maybe NULL */
{
    if (BGCONNECT_LIST_magic != list->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "bgconnect_add_to_list",
	      "Bad magic number (bgconnect_list)",0);
    if (BGCONNECT_INFO_magic != info->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "bgconnect_add_to_list",
	      "Bad magic number (bgconnect_info)",0);

    list->list = safe_array_realloc(list->list,list->list_len+1,sizeof(list->list[0]));
    list->list[list->list_len].connect = info;
    inc_backgroud_connect_info_refcount(list->list[list->list_len].connect);
    list->list[list->list_len].port = port;
    list->list[list->list_len].s_type = s_type;
    
    DPRINT(Debug,12,(&Debug,"bgconnect_add_to_list: %s #%d: %s/%s, port %d",
		     list->se         ? list->se->official_name : "--",
		     list->list_len,
		     info->hostname   ? info->hostname          : "--",
		     info->addrstring ? info->addrstring        : "--",
		     port));

    if (list->list[list->list_len].s_type) {
	DPRINT(Debug,12,(&Debug,"; type %p %s defport %d",
			 list->list[list->list_len].s_type,
			 service_type_name(list->list[list->list_len].s_type),
			 service_type_defport(list->list[list->list_len].s_type)
			 ));		
    }
    
    DPRINT(Debug,12,(&Debug,"\n"));
    
    
    list->list_len++; 
}

/* Returns 0 if interrupted */

static int  bgconnect_wait_delay  P_((struct bgconnect_list *list,
				      struct bgconnect_info *info,
				      struct cancel_data    * can,
				      struct sec_ms         * delay));
static int  bgconnect_wait_delay(list,info,can,delay)
     struct bgconnect_list *list;
     struct bgconnect_info *info;
     struct cancel_data    * can;
     struct sec_ms         * delay;
{
    int ret = 1;

    struct schedule_timelimit delay_deadline  =  NO_schedule_timelimit;

    if (BGCONNECT_LIST_magic != list->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "bgconnect_wait_delay",
	      "Bad magic number (bgconnect_list)",0);

    if (BGCONNECT_INFO_magic != info->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "bgconnect_wait_delay",
	      "Bad magic number (bgconnect_info)",0);

    if (schedule_set_next_timeout2a(&delay_deadline,
				    &(info->time_added),
				    delay,se_use_zero)) {
	int have_result = 0;
	struct schedule_timelimit now  =  NO_schedule_timelimit;
	
	 char * A =  schedule_timeout_string(& delay_deadline);
	 if (A) {
	     DPRINT(Debug,12,(&Debug,
			      "bgconnect_wait_delay: deadline for wait: %s\n",
			      A));
	     
	     free(A);
	 }
    
	 do {
	     int x;
	     
	     for (x = 0; x < list->list_len; x++) {
		 if (list->list[x].connect) {
		     if (BGCONNECT_INFO_magic != list->list[x].connect->magic) 
			 panic("CONNECTION PANIC",__FILE__,__LINE__,
			       "bgconnect_wait_delay",
			       "Bad magic number (bgconnect_info)",0);

		     if (connect_OK == list->list[x].connect->result) {
			 
			 DPRINT(Debug,12,(&Debug,
					  "bgconnect_wait_delay: %s found #%d: connect_OK: %s/%s, port %d\n",
					  list->se         ? list->se->official_name : "--",		     
					  x,
					  list->list[x].connect->hostname   ? list->list[x].connect->hostname   : "--",
					  list->list[x].connect->addrstring ? list->list[x].connect->addrstring : "--",
					  list->list[x].port));
			 have_result = 1;		     
		     }
		 }
	     }

	     if (!have_result) {
		 enum wait_for_status r =
		     wait_for_action_or_deadline_settime_c(bgconnect_socket_writable,
							   &delay_deadline,&now,
							   can);

		 switch (r) {
		     int err;
		 case wait_for_done:                           break;
		 case wait_for_none:
		     DPRINT(Debug,10,(&Debug,
				      "bgconnect_wait_delay: wait interrupted\n"));
		     break;
		 case wait_for_error:
		     err = errno;
		     
		     DPRINT(Debug,12,(&Debug,
				      "bgconnect_wait_delay: errno=%d, %s\n",
				      err,strerror(err)));
		     
		     if (err == EINTR && can && is_canceled(can)) {
			 DPRINT(Debug,12,(&Debug,
					  "bgconnect_wait_delay: wait canceled\n"));
			 
			 
			 ret = 0;
			 goto out;
		     }
		     break;
		 }
		 
	     } else
		 schedule_set_current_time(&now);
	     
	 } while (ret && !have_result &&
		  schedule_valid_until_ok(& delay_deadline,
					  & now));
    }

 out:
     DPRINT(Debug,12,(&Debug,
		      "bgconnect_wait_delay=%d\n",
		      ret));
    
    return ret;
}

static enum connect_result bgconnect_wait_this P_((struct bgconnect_list *list, struct bgconnect_info *info,
						   int                   * last_error,
						   struct cancel_data    * can,
						   int                   * cur_socket));
static enum connect_result bgconnect_wait_this(list,info,last_error,can,cur_socket)
     struct bgconnect_list *list /* can be NULL ? */;
     struct bgconnect_info *info;
     int                   * last_error;
     struct cancel_data    * can;
     int                   * cur_socket;
{

    enum connect_result  r = connect_none;
    enum wait_for_status r1 =  wait_for_none;
    
    if (BGCONNECT_INFO_magic != info->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "bgconnect_wait_this",
	      "Bad magic number (bgconnect_info)",0);

    DPRINT(Debug,12,(&Debug,
		     "bgconnect_wait_this: info=%p (socket=%d, hostname=%s, address=%s)\n",
		     info,info->socket,info->hostname ? info->hostname : "<not set>",
		     info->addrstring ? info->addrstring : "?" ));
    
        
    while ((connect_in_progress == info->result) &&
	   (-1 == *cur_socket ||
	    *cur_socket == info->socket)
	   &&
	   wait_for_none < (r1 = wait_for_action_c(bgconnect_socket_writable,can))) {
		
    }

    switch (r1) {
	int err UNUSED_VAROK;
    case wait_for_done:          break;
    case wait_for_none:
	DPRINT(Debug,10,(&Debug,
			 "bgconnect_wait_this: wait interrupted\n"));
	break;
    case wait_for_error:
	err = errno;
	
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_wait_this: errno=%d, %s\n",
			 err,strerror(err)));
	break;
    }

    r           = info->result;
    *last_error = info->error;
    
    if (connect_in_progress != r &&
	connect_none        != r &&
	-1                  == *cur_socket) {
	
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_wait_this: Moving socket %d from background connect info %p\n",
			 info->socket,info));
	*cur_socket  = info->socket;
	info->socket = -1;
    }
    
    if (list) {
	const char * official_name UNUSED_VAROK = NULL;
	
	if (BGCONNECT_LIST_magic != list->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "bgconnect_wait_this",
		  "Bad magic number (bgconnect_list)",0);

	if (list->se) {
	    if ( SERVICE_ENTRY_magic != list->se->magic)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "bgconnect_wait_this",
		      "Bad magic (service_entry)",0);

	    official_name = list->se->official_name;
	}
	
	if (connect_in_progress != r &&
	    connect_none        != r &&
	    -1 ==  info->socket) {
	    int x;
	    
	    for (x = 0; x < list->list_len; x++) {
		if (list->list[x].connect) {
		    if (BGCONNECT_INFO_magic != list->list[x].connect->magic) 
			panic("CONNECTION PANIC",__FILE__,__LINE__,
			      "bgconnect_wait_this",
			      "Bad magic number (bgconnect_info)",0);
		    
		    if (info == list->list[x].connect) {
			
			DPRINT(Debug,12,(&Debug,
					 "bgconnect_wait_this: %s found #%d: %s/%s, port %d\n",
					 official_name ? official_name : "--",		     
					 x,
					 list->list[x].connect->hostname   ? list->list[x].connect->hostname   : "--",
					 list->list[x].connect->addrstring ? list->list[x].connect->addrstring : "--",
					 list->list[x].port));
			
			/* Decrements refcount, but caller should still keep that */
			
			free_backgroud_connect_info(& (list->list[x].connect));
		    }
		}
	    }
	} 
    }
	
    DPRINT(Debug,12,(&Debug,
		     "bgconnect_wait_this=%d",r));
    switch (r) {
    case connect_fail_abort:  DPRINT(Debug,12,(&Debug," (connect_fail_abort)"));  break;
    case connect_fail:        DPRINT(Debug,12,(&Debug," (connect_fail)"));        break;
    case connect_none:        DPRINT(Debug,12,(&Debug," (connect_none)"));        break;
    case connect_OK:          DPRINT(Debug,12,(&Debug," (connect_OK)"));          break;
    case connect_in_progress: DPRINT(Debug,12,(&Debug," (connect_in_progress)")); break;
    }
    if (-1 != *cur_socket) {
	DPRINT(Debug,12,(&Debug,", *cur_socket=%d",*cur_socket));
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return  r;
}


static int bgconnect_found_complete P_((struct bgconnect_list * list, int *inp, int * conc,
					struct schedule_timelimit *last_started,
					int portgroup, int *portc,
					char **oneaddr,
					int  *oneport,
					struct schedule_timelimit *portgroup_last_started,
					char **onehname
					));
static int bgconnect_found_complete(list,inp,conc,last_started, portgroup, portc,oneaddr,
				    oneport,portgroup_last_started,onehname)
     struct bgconnect_list * list;
     int                   * inp;
     int                   * conc;
     struct schedule_timelimit *last_started;
     int                     portgroup;
     int                   * portc;
     char                 ** oneaddr;   /* malloced */
     int                   * oneport;
     struct schedule_timelimit *portgroup_last_started;
     char                 ** onehname;
{
    int x;
    int found       = -1;
    int inprogress  = 0;
    int concurrent  = 0;
    int portcount   = 0;
    const char * compaddr = NULL;
    int    compfail = 0;
    const char *comphname = NULL;
    int    compfailhname = 0;
    int    theport  = 0;
    int    portfail = 0;
    const char * official_name UNUSED_VAROK = NULL;
    
    if (BGCONNECT_LIST_magic != list->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "bgconnect_found_complete",
	      "Bad magic number (bgconnect_list)",0);
    
    if (list->se) {
	if ( SERVICE_ENTRY_magic != list->se->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "bgconnect_found_complete",
		  "Bad magic (service_entry)",0);
	
	official_name = list->se->official_name;
    }
    
    if (last_started)
	*last_started = NO_schedule_timelimit;
    if (portgroup_last_started)
	*portgroup_last_started = NO_schedule_timelimit;
	 
    for (x = 0; x < list->list_len; x++) {
	if (list->list[x].connect) {
	    int same_portgroup;
	    
	    if (BGCONNECT_INFO_magic != list->list[x].connect->magic) 
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "bgconnect_found_complete",
		      "Bad magic number (bgconnect_info)",0);

	    same_portgroup = portgroup && 
		(portgroup == list->list[x].connect->portgroup);
	    
	    if (-1 != list->list[x].connect->socket)
		concurrent++;

	    if (last_started &&
		schedule_extend_valid_until(last_started,
					    & (list->list[x].connect->time_added))) {
		char * A =  schedule_timeout_string(last_started);
		if (A) {
		    DPRINT(Debug,12,(&Debug,
				     "bgconnect_found_complete: %s #%d: extended last started to %s\n",
				     official_name ? official_name : "--",		     
				     x,A));		    
		    free(A);
		}
	    }

	    if (portgroup_last_started && same_portgroup &&
		schedule_extend_valid_until(portgroup_last_started,
					    & (list->list[x].connect->time_added))) {

		char * A =  schedule_timeout_string(portgroup_last_started);
		if (A) {
		    DPRINT(Debug,12,(&Debug,
				     "bgconnect_found_complete: %s #%d: extended portgroup %d last started to %s\n",
				     list->se         ? list->se->official_name : "--",		     
				     x,portgroup,A));		    
		    free(A);
		}
	    }
		
	    if (connect_OK == list->list[x].connect->result) {
		
		DPRINT(Debug,12,(&Debug,
				 "bgconnect_found_complete: %s found #%d: connect_OK: %s/%s, port %d\n",
				 official_name ? official_name : "--",		     
				 x,
				 list->list[x].connect->hostname   ? list->list[x].connect->hostname   : "--",
				 list->list[x].connect->addrstring ? list->list[x].connect->addrstring : "--",
				 list->list[x].port));
		if (-1 == found)
		    found = x;
		break;
	    }
	    if (connect_in_progress == list->list[x].connect->result) {
		
		DPRINT(Debug,12,(&Debug,
				 "bgconnect_found_complete: %s  #%d: in progress: %s/%s, port %d; %sportgroup %d\n",
				 official_name ? official_name : "--",		     
				 x,
				 list->list[x].connect->hostname   ? list->list[x].connect->hostname   : "--",
				 list->list[x].connect->addrstring ? list->list[x].connect->addrstring : "--",
				 list->list[x].port,
				 same_portgroup ? "same " : "",
				 list->list[x].connect->portgroup));		
		inprogress++;

		if (list->list[x].port) {
		    if (theport) {
			if (theport != list->list[x].port)
			    portfail = 1;
		    } else
			theport = list->list[x].port;
		} else
		    portfail = 1;
		
		if (same_portgroup)
		    portcount++;

		if (list->list[x].connect->addrstring) {
		    if (compaddr) {
			if (0 != strcmp(compaddr,list->list[x].connect->addrstring))
			    compfail = 1;
		    } else
			compaddr = list->list[x].connect->addrstring;
			
		    
		} else
		    compfail = 1;

		if (list->list[x].connect->hostname) {
		    if (comphname) {
			if (0 != strcmp(comphname,list->list[x].connect->hostname))
			    compfailhname = 1;
		    } else
			comphname = list->list[x].connect->hostname;
		} else
		    compfailhname = 1;
		    
	    }		 
	}
    }
    
    *inp   = inprogress;
    *conc  = concurrent;
    *portc = portcount;

    if (compaddr && !compfail) {
	*oneaddr = strmcpy(*oneaddr,compaddr);
    } else if (*oneaddr) {
	free(*oneaddr);
	*oneaddr = NULL;
    }

    if (comphname && !compfailhname) {
	*onehname = strmcpy(*onehname,comphname);
    } else if (*onehname) {
	free(*onehname);
	*onehname = NULL;
    }

    if (!portfail)
	*oneport = theport;
    else
	*oneport = 0;
    
    return found;
}


enum bgconnect_wait {
    bgwait_none_completed = -1,
    bgwait_none_success = 0,
    bwwait_succees,
    bgwait_completed,
    bgwait_space_or_completed
};

/* Returns connect_none, if connection is not picked from backround list */
static enum connect_result bgconnect_got_connection P_((struct bgconnect_list * list,
							struct remote_account * ra  /* to update */,
							int                   * cur_socket,
							int                   * got_port,
							const struct service_type ** got_type,
							int                   * last_error,
							struct cancel_data    * can,
							enum bgconnect_wait     wait,
							char                 ** addrstring,
							struct sec_ms         * delay_noresult,
							int                   portgroup,
							struct sec_ms         * delay_portgroup_noresult
							));
static enum connect_result bgconnect_got_connection(list,ra,cur_socket,got_port,got_type,
						    last_error,can,wait,
						    addrstring,delay_noresult,portgroup,
						    delay_portgroup_noresult)
     struct bgconnect_list * list;
     struct remote_account * ra  /* to update */;
     int                   * cur_socket;
     int                   * got_port;
     const struct service_type ** got_type;
     int                   * last_error;
     struct cancel_data    * can;
     enum bgconnect_wait     wait;
     char                 ** addrstring /* update */;
     struct sec_ms         * delay_noresult;
     int                     portgroup;
     struct sec_ms         * delay_portgroup_noresult;
{
    enum connect_result  r = connect_none;
    const char * official_name UNUSED_VAROK = NULL;
    
    if (BGCONNECT_LIST_magic != list->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "bgconnect_got_connection",
	      "Bad magic number (bgconnect_list)",0);

    if (ra->magic  != REMOTE_ACCOUNT_magic)
	 panic("CONNECTION PANIC",__FILE__,__LINE__,
	       "bgconnect_got_connection",
	       "Bad magic number (remote_account)",0);
    
    if (list->se) {
	if ( SERVICE_ENTRY_magic != list->se->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "bgconnect_found_complete",
		  "Bad magic (service_entry)",0);
	
	official_name = list->se->official_name;
    }

    switch (wait) {
    case bgwait_none_completed:
 	DPRINT(Debug,12,(&Debug,
			 "bgconnect_got_connection: not waiting, get any completed\n"));
	break;
    case bgwait_none_success:
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_got_connection: not waiting, get succeed\n"));
	break;
    case bwwait_succees:
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_got_connection: wait for succeed\n"));
	break;

    case bgwait_completed:
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_got_connection: wait for any completed\n"));
	break;
    case bgwait_space_or_completed:
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_got_connection: wait for space (< %d concurrent) or any completed\n",
			 connect_mode_limit));
	break;
    }
       
    *got_port   = PORT_end;
    if (got_type)
	*got_type = NULL;
    *last_error = 0;
    
     if (-1 != (*cur_socket)) {
	/* Some systems require that socket is closed even
	   after unsuccesfully connect() -- it can not be reused
	*/
	DPRINT(Debug,12,(&Debug,
			 "bgconnect_got_connection: closing socket (%d)\n",
			 (*cur_socket)));
	close(*cur_socket);
	(*cur_socket) = -1;
     }

     if (*addrstring) {
	 free(*addrstring);
	 *addrstring = NULL;
     }
     
     if (list->list) {
	 int inprogress  = 0;
	 int concurrent  = 0;
	 int need_spc = 0;
	 int need_wait = 0;
	 int portcount = 0;
	 int was_portgroup = 0;
	 char * oneaddr = NULL;
	 char * onehname = NULL;
	 int theport = 0;

	 struct schedule_timelimit last_started =  NO_schedule_timelimit;
	 struct schedule_timelimit portgroup_last_started =  NO_schedule_timelimit;
	 struct schedule_timelimit delay_deadline  =  NO_schedule_timelimit;
	 struct schedule_timelimit now = NO_schedule_timelimit;
	 int delay_set = 0;
	 
	 int found =  bgconnect_found_complete(list,&inprogress,&concurrent,
					       &last_started,portgroup,&portcount,
					       &oneaddr,&theport,&portgroup_last_started,
					       &onehname);
	 
	 if (-1 == found  && inprogress > 0) {
	     enum wait_for_status r1;

	     DPRINT(Debug,12,(&Debug,
			      "bgconnect_got_connection: %d in progress, probing\n",
			      inprogress));

	     if  (wait_for_none < (r1 = wait_for_action_or_timeout_c(bgconnect_socket_writable,
								     0 /* NO WAIT */,
								     can))) {

		 found =  bgconnect_found_complete(list,&inprogress,&concurrent,
						   NULL,portgroup,&portcount,
						   &oneaddr,&theport,&portgroup_last_started,
						   &onehname);
		 
		 
	     } else {
		 switch (r1) {
		     int err UNUSED_VAROK;
		 case wait_for_done:            break;
		 case wait_for_none:
		     DPRINT(Debug,11,(&Debug,
				      "bgconnect_got_connection: wait interrupted\n"));
		     break;
		 case wait_for_error:
		     err = errno;
		     
		     DPRINT(Debug,12,(&Debug,
				      "bgconnect_got_connection: errno=%d, %s\n",
				      err,strerror(err)));
		 }
	     }
	 }

	 if (delay_portgroup_noresult && schedule_have_timelimit(&portgroup_last_started) &&
	     schedule_set_next_timeout2a(&delay_deadline,&portgroup_last_started,
					 delay_noresult,se_use_zero)) {
	 
	     char * A =  schedule_timeout_string(& delay_deadline);
	     if (A) {
		 DPRINT(Debug,12,(&Debug,
				  "bgconnect_got_connection: portgroup %d: deadline for no connection: %s\n",
				  portgroup, A));

		 free(A);
	     }

	     schedule_set_current_time(&now);
	     delay_set = 1;

	 } else if (delay_noresult && schedule_have_timelimit(&last_started) &&
		    schedule_set_next_timeout2a(&delay_deadline,&last_started,
							   delay_noresult,se_use_zero)) {

	     char * A =  schedule_timeout_string(& delay_deadline);
	     if (A) {
		 DPRINT(Debug,12,(&Debug,
				  "bgconnect_got_connection: deadline for no connection: %s\n",
				  A));

		 free(A);
	     }

	     schedule_set_current_time(&now);
	     delay_set = 1;
	 }
	     	 	 
	 while (-1 == found && (
				(need_wait = (delay_set &&
					      schedule_valid_until_ok(& delay_deadline,
								      & now)))
				||
				((bwwait_succees <= wait && inprogress > 0)		
				 && (bgwait_space_or_completed != wait ||
				     (need_spc = (connect_mode_limit > 0 &&
						  concurrent >= connect_mode_limit))
				     )
				 )
				)
				
		) {
	     
	     enum wait_for_status r;
	     
	     DPRINT(Debug,12,(&Debug,
			      "bgconnect_got_connection: %d in progress, waiting\n",
			      inprogress));

	     if (need_spc) {
		 DPRINT(Debug,12,(&Debug,
				  "bgconnect_got_connection: Need space for connection, %d connections, max %d connections\n",
				  concurrent,connect_mode_limit));
	     }

	     if (need_wait) {
		 DPRINT(Debug,12,(&Debug,
				  "bgconnect_got_connection: Need wait to deadline if connection do not found\n"));
	     }
	     
	     if (can) {
		 if (portgroup) {

		     if (1 == portcount && 1 == inprogress && theport) {
			 cancel_progress(can, CATGETS(elm_msg_cat, MeSet,MeTheConPortInProg,
						      " port %d in progress"),
					 theport);
			 
			 was_portgroup = 1;
		     } else if (portcount > 0 || was_portgroup) {
			 cancel_progress(can, CATGETS(elm_msg_cat, MeSet,MeConPortInProg,
						      " %2d ports in progress"),
					 portcount);
			 was_portgroup = 1;
		     }

		 } else if (oneaddr) {

		     if (onehname && official_name && 0 != strcmp(onehname,official_name)) {
			 if (1 == inprogress && theport) 
			     cancel_progress(can, CATGETS(elm_msg_cat, MeSet,MeConAd2ThePortInProg,
							  " [%s/%s] port %d in progress"),
					     onehname,oneaddr,theport);
			 else
			     cancel_progress(can, CATGETS(elm_msg_cat, MeSet,MeConAd2InProg,
							  " %2d attempts to [%s/%s] in progress"),
					     inprogress,onehname,oneaddr);


		     } else {
			 if (1 == inprogress && theport) 
			     cancel_progress(can, CATGETS(elm_msg_cat, MeSet,MeConAdThePortInProg,
							  " [%s] port %d in progress"),
					     oneaddr,theport);
			 
			 else
			     cancel_progress(can, CATGETS(elm_msg_cat, MeSet,MeConAdInProg,
							  " %2d attempts to [%s] in progress"),
					     inprogress,oneaddr);
		     }
		 } else if (onehname && official_name && 0 != strcmp(onehname,official_name)) {

		     if (theport)
			 cancel_progress(can, CATGETS(elm_msg_cat, MeSet,MeConNamePortInProg,
						  " %2d attempts to [%s/*] port %d in progress"),
					 inprogress,onehname,theport);
		     else
			 cancel_progress(can, CATGETS(elm_msg_cat, MeSet,MeConNameInProg,
						      " %2d attempts to [%s/*] in progress"),
					 inprogress,onehname);

		 } else if (theport) {
		      cancel_progress(can, CATGETS(elm_msg_cat, MeSet,MeConPortInprogress,
						      " %2d attempts to port %d in progress"),
				      inprogress,theport);		     
		 } else {		 
		     if (connect_mode_limit > 1)
			 cancel_progress(can, CATGETS(elm_msg_cat, MeSet,MeConInprogressLim,
						      " %2d attempts in progress (limit %d)"),
					 inprogress,connect_mode_limit);
		     else		 
			 cancel_progress(can, CATGETS(elm_msg_cat, MeSet,MeConInprogress,
						      " %2d attempts in progress"),
					 inprogress);
		 }

	     }
	     
	     if (wait_for_none < (r = (need_wait ?
				       wait_for_action_or_deadline_c(bgconnect_socket_writable,
								     &delay_deadline,can) :
				       wait_for_action_c            (bgconnect_socket_writable,can))
				  )) {
		 found =  bgconnect_found_complete(list,&inprogress,&concurrent,NULL,
						   portgroup,&portcount,&oneaddr,&theport,
						   NULL,&onehname);
		 
	     } else {
		 switch (r) {
		     int err;
		 case wait_for_done:              break;
		 case wait_for_none:
		     DPRINT(Debug,12,(&Debug,
				      "bgconnect_got_connection: wait interrupted\n"));
		     break;
		 case  wait_for_error:
		     err = errno;
		     
		     DPRINT(Debug,12,(&Debug,
				      "bgconnect_got_connection: errno=%d, %s\n",
				      err,strerror(err)));
		     
		     if (err == EINTR && can && is_canceled(can)) {
			 DPRINT(Debug,12,(&Debug,
					  "bgconnect_got_connection: %d in progress, wait canceled\n",
					  inprogress));
			 goto out;
		     }
		     break;
		 }
		 
		 DPRINT(Debug,12,(&Debug,
				  "bgconnect_got_connection: %d in progress, quiting\n",
				  inprogress));
		 break;
	     }
	     

	     if (delay_noresult)
		 schedule_set_current_time(&now); 
	 }

     out:
	 if (-1 == found && (bgwait_completed <= wait || bgwait_none_completed == wait)) {
	     int found_ok = -1;
	     int found_failed = -1;
	     int x;

	     DPRINT(Debug,12,(&Debug,
			      "bgconnect_got_connection: checking completed, %d connection attempts added to list, %d concurrent, portgroup %d\n",
			      list->list_len,concurrent,portgroup));
	     
	     for (x = 0; x < list->list_len; x++) {
		 if (list->list[x].connect) {
		     if (BGCONNECT_INFO_magic != list->list[x].connect->magic) 
			 panic("CONNECTION PANIC",__FILE__,__LINE__,
			       "bgconnect_got_connection",
			       "Bad magic number (bgconnect_info)",0);
		     
		     if (connect_OK == list->list[x].connect->result) {
			 
			 DPRINT(Debug,12,(&Debug,
					  "bgconnect_got_connection: %s found #%d: connect_OK: %s/%s, port %d\n",
					  official_name ? official_name : "--",		     
					  x,
					  list->list[x].connect->hostname   ? list->list[x].connect->hostname   : "--",
					  list->list[x].connect->addrstring ? list->list[x].connect->addrstring : "--",
					  list->list[x].port));
			 
			 found_ok = x;
			 break;
		     }

		     if (connect_none >= list->list[x].connect->result) {
			 int matchfail =
			     0 == portgroup || portgroup == list->list[x].connect->portgroup;
			 DPRINT(Debug,12,(&Debug,
					  "bgconnect_got_connection: %s found #%d: failed: %s/%s, port %d; portgroup %d%s\n",
					  official_name ? official_name : "--",
					  x,
					  list->list[x].connect->hostname   ? list->list[x].connect->hostname   : "--",
					  list->list[x].connect->addrstring ? list->list[x].connect->addrstring : "--",
					  list->list[x].port,
					  list->list[x].connect->portgroup,
					  matchfail ? " OK" : "skipping"));

			 if ((found_failed < 0 ||
			      list->list[x].connect->result < list->list[found_failed].connect->result)
			     && 
			     matchfail)
			     found_failed = x;
		     }
		 }
	     }

	     if (found_ok >= 0)
		 found = found_ok;
	     else if (found_failed >= 0)
		 found = found_failed;
	 }

	 if (found >= 0) {
	     char * s UNUSED_VAROK = "";

	     if (BGCONNECT_INFO_magic != list->list[found].connect->magic) 
		 panic("CONNECTION PANIC",__FILE__,__LINE__,
		       "bgconnect_got_connection",
		       "Bad magic number (bgconnect_info)",0);

	     
	     switch (list->list[found].connect->result) {
	     case connect_fail_abort:  s = " (connect_fail_abort)";  break;
	     case connect_fail:        s = " (connect_fail)";        break;
	     case connect_none:        s = " (connect_none)";        break;
	     case connect_OK:          s = " (connect_OK)";          break;
	     case connect_in_progress: s = " (connect_in_progress)"; break;
	     }

	     DPRINT(Debug,12,(&Debug,
			      "bgconnect_got_connection: %s found #%d: status %d%s: %s/%s, port %d, socket %d (portgroup %d)",
			      official_name ? official_name : "--",		     
			      found,
			      list->list[found].connect->result,s,
			      list->list[found].connect->hostname   ? list->list[found].connect->hostname   : "--",
			      list->list[found].connect->addrstring ? list->list[found].connect->addrstring : "--",
			      list->list[found].port,
			      list->list[found].connect->socket,
			      list->list[found].connect->portgroup));
	     if (list->list[found].s_type) {
		 DPRINT(Debug,12,(&Debug,"; type %p %s defport %d",
				  list->list[found].s_type,
				  service_type_name(list->list[found].s_type),
				  service_type_defport(list->list[found].s_type)));
	     }		 
	     DPRINT(Debug,12,(&Debug,"\n"));
	     
	     r = list->list[found].connect->result;

	     if (!  set_SOCKADDR_ptr_helper(& (ra->addrsize),
					    & (ra->hostaddr),
					    list->list[found].connect->addrsize,
					    list->list[found].connect->hostaddr)) {

		 DPRINT(Debug,12,(&Debug,
				  "bgconnect_got_connection: Failed to set address.\n"));
	     }
	     
	     ra->hostname = strmcpy(ra->hostname,
				    list->list[found].connect->hostname);
	     
	     *cur_socket = list->list[found].connect->socket; list->list[found].connect->socket = -1;
	     *got_port   = list->list[found].port;
	     if (got_type)
		 *got_type = list->list[found].s_type;
	     
	     *addrstring = list->list[found].connect->addrstring; list->list[found].connect->addrstring = NULL;
	     *last_error = list->list[found].connect->error;
	     
	     
	     /* There should not be action anyway */
	     if (-1 != *cur_socket)
		 clear_action(*cur_socket);
	     free_backgroud_connect_info(& (list->list[found].connect));
	 } 
	 
	 if (oneaddr)
	     free(oneaddr);
	 if (onehname)
	     free(onehname);

     }
     
    DPRINT(Debug,12,(&Debug,
		     "bgconnect_got_connection=%d",r));
    switch (r) {
    case connect_fail_abort:  DPRINT(Debug,12,(&Debug," (connect_fail_abort)"));  break;
    case connect_fail:        DPRINT(Debug,12,(&Debug," (connect_fail)"));        break;
    case connect_none:        DPRINT(Debug,12,(&Debug," (connect_none)"));        break;
    case connect_OK:          DPRINT(Debug,12,(&Debug," (connect_OK)"));          break;
    case connect_in_progress: DPRINT(Debug,12,(&Debug," (connect_in_progress)")); break;
    }
    if (-1 != *cur_socket) {
	DPRINT(Debug,12,(&Debug,", *cur_socket=%d",*cur_socket));
    }
    if (PORT_end != *got_port) {
	DPRINT(Debug,12,(&Debug,", *got_port=%d",*got_port));
    }
    if (*last_error) {
	DPRINT(Debug,12,(&Debug,", *last_error=%d",*last_error));
    }
    if (*addrstring) {
	DPRINT(Debug,12,(&Debug,", *addrstring=%s",*addrstring));
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return r;
}

static void update_ra_sockname P_((int cur_socket, struct remote_account *ra));
static void update_ra_sockname(cur_socket,ra)
     int cur_socket;
     struct remote_account *ra;
{
    
#ifdef HAVE_GETSOCKNAME
    SOCKADDR sockname;
    GETSOCKNAME_SOCKLEN socknamelen = sizeof (sockname);
    int r;
    
    /* getsockname can return local address of machine
       only after socket is connected. Before that
       routing is not done and source address is not
       selected. before that wildcard address may be
       returned.
    */

    bzero(&sockname,sizeof (sockname));
    
    r = getsockname(cur_socket,&sockname.sa,&socknamelen);
    
    if (0 == r) {
	char mybuffer2[256];
	
	const char * s2 = give_addr_as_string(&sockname,
					      mybuffer2,sizeof mybuffer2);
	
	
	DPRINT(Debug,12,(&Debug,
			 "update_ra_sockname: getsockname succeed"));
	
	if (s2) {
	    DPRINT(Debug,12,(&Debug,": %s",
			     s2));
	}
	DPRINT(Debug,12,(&Debug,"\n"));
	
	if (socknamelen > sizeof (sockname)) {
	    DPRINT(Debug,12,(&Debug,
			     "update_ra_sockname: getsockname buffer too small, need %d bytes (was %zu)\n",
			     socknamelen,sizeof (sockname)));
	    socknamelen =sizeof (sockname);
	}
	
	
	if (set_address_from_gen_helper2(&(ra->socknamesize),
					 &(ra->socknameaddr),
					 &sockname,
					 socknamelen)) {
	    
	    DPRINT(Debug,12,(&Debug,
			     "update_ra_sockname: Address stored\n"));
	}

	if (transaction_file && s2) {
	    
	    time_t now = 0;

	    if (((time_t) -1) != time(&now)) {
		struct tm * zz = localtime(&now);

		if (!zz)
		    goto no_time1;
	    
		fprintf(transaction_file,
			"%d [%d] %02d:%02d:%02d === local address %s",
			getpid(),cur_socket,
			zz->tm_hour,
			zz->tm_min,
			zz->tm_sec,
			s2);
	    } else {
	    no_time1:
		fprintf(transaction_file,
			"%d [%d] === local address %s",
			getpid(),cur_socket,
			s2);

	    }
	    
	    switch (sockname.sa.sa_family) {
	    case AF_INET:
		fprintf(transaction_file," IPv4 port %d",
			ntohs(sockname.sin.sin_port));
		break;
#ifdef HAVE_IN6
	    case AF_INET6: 
		fprintf(transaction_file," IPv6 port %d",
			ntohs(sockname.sin6.sin6_port));
		break;
#endif
	    }

	    fprintf(transaction_file,"\n");
	}

	
    } else if (-1 == r) {
	int err UNUSED_VAROK = errno;
	
	DPRINT(Debug,12,(&Debug,
			 "update_ra_sockname: getsockname failed: %s (errno=%d)\n",
			 strerror(err),err));		}
    
    
#endif

}

static void connect_error_clean_1 P_((const char             * host,
				      int                      was_canceled,
				      struct cancel_data     * bgcan,
				      int                      last_error,
				      int                    * cur_socket_p));

static void connect_error_clean_1(host,was_canceled,bgcan,last_error, cur_socket_p)
     const char             * host;
     int                      was_canceled;
     struct cancel_data     * bgcan;
     int                      last_error;
     int                    * cur_socket_p;
{
    if (*cur_socket_p != -1) {
	DPRINT(Debug,12,(&Debug,
			 "connect_error_clean: Closing socket (%d) after failure\n",
			 *cur_socket_p));
	close(*cur_socket_p);
	*cur_socket_p = -1;	    
    }

    if (was_canceled || (bgcan && is_canceled(bgcan)))
	lib_error(CATGETS(elm_msg_cat, MeSet,MeConnectCanceled,
			  "Connect %s canceled."),
		  host);
    else if (last_error)
	lib_error(CATGETS(elm_msg_cat, MeSet,MeConnectFailed,
			  "Failed to connect %s: %s"),
		  host,strerror(last_error));
    else
	lib_error(CATGETS(elm_msg_cat, MeSet,MeConnectFailedX,
			  "Failed to connect %s"),
		  host);

}

static void connect_set_socket P_((int cur_socket));
static void connect_set_socket(cur_socket)
     int cur_socket;
{

    enum syscall_status  r1;

    /* We set socket to non-blocking after connect, because
     * non-blocking connect is not always used
     */

    r1 = fcntl(cur_socket,F_SETFL,O_NONBLOCK);
    switch (r1) {
	int err UNUSED_VAROK;
    case syscall_error /* -1 */:
	err= errno;
	DPRINT(Debug,12,(&Debug,
			 "connect_set_socket: fcntl [F_SETFL O_NONBLOCK] failed: %s\n",
			 strerror(err)));
	break;
    case syscall_success /* 0 */:
	DPRINT(Debug,12,(&Debug,
			 "connect_set_socket: fcntl(%d,F_SETFL,O_NONBLOCK) done\n",
			 cur_socket));
	break;
    }

#ifdef FD_CLOEXEC
    r1 = fcntl(cur_socket,F_SETFD,FD_CLOEXEC);
#else
    r1 = fcntl(cur_socket,F_SETFD,1);
#endif
    switch (r1) {
	int err UNUSED_VAROK;
    case syscall_error /* -1 */:
	err = errno;
	DPRINT(Debug,12,(&Debug,
			 "connect_set_socket: fcntl [F_SETFD FD_CLOEXEC] failed: %s\n",
			 strerror(err)));
	break;
    case syscall_success /* 0 */:
	DPRINT(Debug,12,(&Debug,
			 "connect_set_socket: fcntl(%d,F_SETFD,FD_CLOEXEC) done\n",
			 cur_socket));
	break;
    }
}

int connect_remote_account_1(ra,ra_from)
     struct remote_account *ra;  
     const struct remote_account *ra_from;
{
    int cur_socket   = -1;
    int last_error   = 0;
    int ok = 0;
    int was_canceled = 0;
    int port = 0;
    struct bgconnect_list  * bglist = NULL;
    struct cancel_data     * bgcan = NULL;
    
    if (ra->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "connect_remote_account_1",
	      "Bad magic number",0);

    
    if (ra->stream) {
	DPRINT(Debug,12,(&Debug,
			 "connect_remote_account_1: Closing for reopening\n"));
	FreeStreamStack2(& (ra->stream),
			 1 /* Force clearing of stack even when this is not last reference */);
    }

    if (ra->service) {
	DPRINT(Debug,12,(&Debug,
			 "connect_remote_account_1: Removing old service entry: %p\n",
			 ra->service));	 
	free_service_entry(& (ra->service));
    }

    if (ra_from->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "connect_remote_account_1",
	      "Bad (ra_from) magic number",0);
       
    if (! ra_from->hostaddr.dummy ||
	! ra_from->addrsize        ||
	! ra_from->username             ||
	! ra_from->host)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "connect_remote_account_1",
	      "No data (ra_from)",0);

    if (connect_mode) {
	bglist = malloc_bgconnect_list(ra_from->service);
	bgcan  = new_schedule_cancel(100 /* msec */,
				     CATGETS(elm_msg_cat, MeSet,MeConnectingSimple,
				    "Connecting to %s"),
			    ra_from->host);
    }
    
    if (! set_SOCKADDR_ptr_helper(& (ra->addrsize),
				  & (ra->hostaddr),
				  ra_from->addrsize,
				  ra_from->hostaddr)) {
	ok = 0;
	goto clean;
    }

    /* name corresponding to hostaddr */
    ra->hostname = NULL;
    if (ra_from->hostname)
	ra->hostname = strmcpy(ra->hostname,
			       ra_from->hostname);
    else if (ra->hostname) {
	free(ra->hostname);
	ra->hostname = NULL;
    }
    
    /* Copy username@host */
    ra->username = strmcpy(ra->username, ra_from->username);
    ra->host     = strmcpy(ra->host,     ra_from->host);

    if (ra->socknameaddr.dummy) {
	free(ra->socknameaddr.dummy);
	ra->socknameaddr.dummy = NULL;
    }
    ra->socknamesize =  0;
    
    switch (ra->hostaddr.sa->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
    case AF_INET:        
	port = ntohs(ra->hostaddr.sin->sin_port);
	ok = 1;
	break;

#ifdef HAVE_IN6
    case AF_INET6: 	
	port = ntohs(ra->hostaddr.sin6->sin6_port);
	ok = 1;    
	break;
#endif

#endif
    default:
	lib_error(CATGETS(elm_msg_cat, MeSet,MeUnsupportedAddrType,
			  "Name %s have odd type address"),
		  ra->hostname ? ra->hostname : ra->host);
	break;
    }

    if (ok) {
	struct cancel_data     * can       = NULL;
	struct bgconnect_info  * bgconnect = NULL;
	enum connect_result r;
	char mybuffer[256];

	const char * s = give_SOCKADDR_ptr_as_string(ra->hostaddr,ra->addrsize,
						     mybuffer,sizeof mybuffer);
	
	if (ra->hostname && 0 != strcmp(ra->host,ra->hostname))
	    can = new_cancel(CATGETS(elm_msg_cat, MeSet,MeConnectingPort2,
				 "Connecting to %s [%s/%s], port %d ..."),
			     ra->host,
			     ra->hostname,
			     s ? s : "???",
			     port);
	else
	    can = new_cancel(CATGETS(elm_msg_cat, MeSet,MeConnectingPort1,
				     "Connecting to %s [%s], port %d ..."),
			     ra->host,
			     s ? s : "???",
			     port);
	
	r = connect_one_IN(ra,&cur_socket,&last_error,can,0,
			   connect_mode ? &bgconnect : NULL,0);

    redo_status:
	switch(r) {
	    
	default:   ok = 0; break;

	case connect_in_progress:
	    DPRINT(Debug,12,(&Debug,
			     "connect_remote_account_1: Connection in progress to %s [%s/%s], port %d\n",
			     ra->host,
			     ra->hostname ? ra->hostname : "???",
			     s ? s : "???",
			     port));

	    if (bgconnect && bglist)
		bgconnect_add_to_list(bglist,bgconnect,port,
				      ra_from->service->service
				      /* service type */);  /* increments refcount */

	    if (bgconnect) {
		r =  bgconnect_wait_this(bglist,bgconnect,&last_error,can,
					 &cur_socket);
		if ( connect_in_progress != r)
		    goto redo_status;
	    }
	    ok = 0;    
	    break;
	    
	case connect_OK:
	    
	    DPRINT(Debug,12,(&Debug,
			     "connect_remote_account_1: Connection succeed %s [%s/%s], port %d\n",
			     ra->host,
			     ra->hostname ? ra->hostname : "???",
			     s ? s : "???",
			     port));
	    ok = 1;
	    
	    if (transaction_file) {

		time_t now = 0;
		
		if (((time_t) -1) != time(&now)) {
		    struct tm * zz = localtime(&now);
		    
		    if (!zz)
			goto no_time2;
	    
		    fprintf(transaction_file,
			    "%d [%d] %02d:%02d:%02d === CONNECT %s [%s/%s], port %d\n",
			    getpid(),cur_socket,
			    zz->tm_hour,
			    zz->tm_min,
			    zz->tm_sec,
			    ra->host,
			    ra->hostname ? ra->hostname : "???",
			    s ? s : "???",
			    port);
		} else {
		no_time2:
		    fprintf(transaction_file,
			    "%d [%d] === CONNECT %s [%s/%s], port %d\n",
			    getpid(),cur_socket,
			    ra->host,
			    ra->hostname ? ra->hostname : "???",
			    s ? s : "???",
			    port);		    
		}
	    }

	    update_ra_sockname(cur_socket,ra);	    
	} 
	    

	was_canceled = is_canceled(can);
	free_cancel(&can);

	if (bgconnect) 
	    free_backgroud_connect_info(& bgconnect);  /* decrements refcount */
    }

    if (!ok) {	
	connect_error_clean_1(ra->host,was_canceled,bgcan,last_error,& cur_socket);

	ok = 0;
	goto clean;
    }

    connect_set_socket(cur_socket);
 
    ra->stream      = returnSimpleStream(cur_socket);
    ra->service     = ra_from->service;
    if (ra->service)
	inc_service_entry_refcount(ra->service);

 clean:
    if (bglist)
	free_bgconnect_list(&bglist);
    if (bgcan)
	free_cancel(&bgcan);
    
    DPRINT(Debug,12,(&Debug,
		     "connect_remote_account_1=%d\n",ok)); 
    return ok;
}

int set_address_from_gen_helper2(addrsize,ptr,generic,genericsize)
     size_t             * addrsize;
     union SOCKADDR_ptr * ptr;
     const SOCKADDR     * generic;
     size_t               genericsize;
{
    int ret = 0;

    switch (generic->sa.sa_family) {
	size_t s;
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)

    case AF_INET:
	if (genericsize < sizeof(generic->sin)) {
	    DPRINT(Debug,11,(&Debug,
			     "set_address_from_gen_helper2: size %zu < AF_INET size %zu\n",
			     genericsize,sizeof(generic->sin)));
	    goto fail;
	} else if (genericsize > sizeof(generic->sin)) {
	    DPRINT(Debug,11,(&Debug,
			     "set_address_from_gen_helper2: size %zu > AF_INET size %zu (extra data?)\n",
			     genericsize,sizeof(generic->sin)));
	}
	ptr->sin    = safe_realloc(ptr->sin,
				   sizeof(generic->sin));
	*(ptr->sin) = generic->sin;
	*addrsize   = sizeof(generic->sin);
	ret = 1;
	break;
	
#ifdef HAVE_IN6
    case AF_INET6: 
	if (genericsize < sizeof(generic->sin6)) {
	    DPRINT(Debug,11,(&Debug,
			     "set_address_from_gen_helper2: size %zu < AF_INET6 size %zu\n",
			     genericsize,sizeof(generic->sin6)));
	    goto fail;
	} else if (genericsize > sizeof(generic->sin6)) {
	    DPRINT(Debug,11,(&Debug,
			     "set_address_from_gen_helper2: size %zu > AF_INET6 size %zu (extra data?)\n",
			     genericsize,sizeof(generic->sin6)));
	}
	
	ptr->sin6  = safe_realloc(ptr->sin6,
				  sizeof(generic->sin6));
	*(ptr->sin6) = generic->sin6;
	*addrsize         =  sizeof(generic->sin6);
	ret = 1;
	break;
#endif
#endif
	
    default:
	DPRINT(Debug,11,(&Debug,
			 "set_address_from_gen_helper2: Unknown address family %d\n",
			 generic->sa.sa_family));

	if (genericsize < sizeof (generic->sa.sa_family)) {
	    DPRINT(Debug,11,(&Debug,
			     "set_address_from_gen_helper2: size %zu < null size %zu\n",
			     genericsize,sizeof (generic->sa.sa_family)));
	    goto fail;
	    
	} else if (genericsize > sizeof (*generic)) {
	    DPRINT(Debug,11,(&Debug,
			     "set_address_from_gen_helper2: size %zu > storage %zu (all data not stored?)\n"));
	    s = sizeof (*generic);
	} else
	    s = genericsize;
	
	ptr->dummy    = safe_realloc(ptr->dummy,s);
	memcpy(ptr->dummy,generic,s);
	*addrsize     = s;
	break;
    }
	
    
    if (!ret) {
    fail:
	if (ptr->generic)
	    free(ptr->generic);
	ptr->generic = NULL;
	*addrsize = 0;
    }

    DPRINT(Debug,11,(&Debug,
		     "set_address_from_gen_helper2=%d; *addrsize=%zu\n",ret,*addrsize));
        
    return ret;
}
     

void set_address_from_gen_helper(addrsize,ptr,generic)
     size_t             * addrsize;
     union SOCKADDR_ptr * ptr;
     const SOCKADDR     * generic;
{

    switch (generic->sa.sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)

    case AF_INET:
	ptr->sin    = safe_realloc(ptr->sin,
				   sizeof(generic->sin));
	*(ptr->sin) = generic->sin;
	*addrsize   = sizeof(generic->sin);
	break;
#ifdef HAVE_IN6
    case AF_INET6: 
	ptr->sin6  = safe_realloc(ptr->sin6,
				  sizeof(generic->sin6));
	*(ptr->sin6) = generic->sin6;
	*addrsize         =  sizeof(generic->sin6);
	break;
#endif
#endif
    default:
	DPRINT(Debug,11,(&Debug,
			 "set_address_from_gen_helper: Unknown address family %d\n",
			 generic->sa.sa_family));

	ptr->generic    = safe_realloc(ptr->generic,
				       sizeof (*generic));
	*(ptr->generic) = *generic;
	*addrsize       =  sizeof(*generic);
	break;
    }
}

/* Return 1 if succeed, 0 if wrong size or unsupported family */
int set_SOCKADDR_from_data(target_addrsize,target,
			   addrtype,addrlength,addrdata)
     size_t             * target_addrsize;
     union SOCKADDR_ptr * target;
     int                  addrtype;
     int                  addrlength;
     char               * addrdata;     
{
    switch (addrtype) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
    case AF_INET:
	if (sizeof (target->sin->sin_addr) != addrlength) {
	    DPRINT(Debug,11,(&Debug,
			     "set_SOCKADDR_from_data=0: Bad IPv4-addr length %ld (should be %ld)\n",
			     (long)addrlength,
			     (long)sizeof (target->sin->sin_addr)));
	    goto fail;
	}
	target->sin      = safe_realloc(target->sin,
					sizeof(* (target->sin)));
	/* Make comparision easier */
	bzero((void *)(target->sin),sizeof(* (target->sin)));
	target->sin->sin_family = addrtype;
	target->sin->sin_port   = htons(PORT_end);    /* No port given */
	memcpy(&(target->sin->sin_addr),addrdata,addrlength);
	*target_addrsize = sizeof(* (target->sin));
	break;
#ifdef HAVE_IN6
    case AF_INET6:
	if (sizeof (target->sin6->sin6_addr) != addrlength) {
	    DPRINT(Debug,11,(&Debug,
			     "set_SOCKADDR_from_data=0: Bad IPv6-addr length %ld (should be %ld)\n",
			     (long)addrlength,
			     (long)sizeof (target->sin6->sin6_addr)));
	    goto fail;
	}
	target->sin6  = safe_realloc(target->sin6,
				     sizeof(* (target->sin6)));
	/* Make comparision easier */
	bzero((void *)(target->sin6),sizeof(* (target->sin6)));
	target->sin6->sin6_family = addrtype;
	target->sin6->sin6_port   = htons(PORT_end);    /* No port given */

	/* target->sin6->sin6_flowinfo  left to zero
	   target->sin6->sin6_scope_id  left to zero
	*/
	
	memcpy(&(target->sin6->sin6_addr),addrdata,addrlength);
	*target_addrsize = sizeof(* (target->sin6));
	break;
#endif
#endif
    default:
	DPRINT(Debug,11,(&Debug,
			 "set_SOCKADDR_from_data=0: Unknown address family %d, addr length %d\n",
			 addrtype,addrlength));

	/* NOTE: We do not know on what components arbitary
	   socket address is constructed so we can't
	   fill it
	*/
	goto fail;
    }

    return 1;

 fail:   /* Clear target on failure */
    *target_addrsize = 0;
    if (target->dummy) {
	free(target->dummy);
	target->dummy = NULL;
    }
    return 0;
}


/* Return 1 if succeed, 0 if too small*/
int set_SOCKADDR_ptr_helper(target_addrsize,target,
				   source_addrsize,source)
     size_t                 * target_addrsize;
     union SOCKADDR_ptr     * target;
     const size_t             source_addrsize;
     const union SOCKADDR_ptr source;
{
    /* sa_family is assumed to be first field !! */
    if (source_addrsize < sizeof (source.sa->sa_family)) {
	DPRINT(Debug,11,(&Debug,
			 "set_SOCKADDR_ptr_helper=0; size=%ld\n",
			 (long)source_addrsize));
	goto fail;
    }

    switch (source.sa->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
    case AF_INET:
	if (source_addrsize < sizeof (* (source.sin))) {
	    DPRINT(Debug,11,(&Debug,
			     "set_SOCKADDR_ptr_helper=0; size=%ld\n",
			     (long)source_addrsize));
	    goto fail;
	}

	target->sin      = safe_realloc(target->sin,
					sizeof(* (source.sin)));
	*(target->sin)   = *(source.sin);
	*target_addrsize = sizeof(* (source.sin));
	break;
#ifdef HAVE_IN6
    case AF_INET6: 
	if (source_addrsize < sizeof (* (source.sin6))) {
	    DPRINT(Debug,11,(&Debug,
			     "set_SOCKADDR_ptr_helper=0; size=%ld\n",
			     (long)source_addrsize));
	    goto fail;
	}
	target->sin6  = safe_realloc(target->sin6,
				     sizeof(* (source.sin6)));
	*(target->sin6) = * (source.sin6);
	*target_addrsize  =  sizeof(* (source.sin6));
	break;
#endif
#endif
    default:
	DPRINT(Debug,11,(&Debug,
			 "set_SOCKADDR_ptr_helper: Unknown address family %d, size %ld\n",
			 source.sa->sa_family,
			 (long)source_addrsize));
	
	/* Copy all information */
	target->dummy = safe_realloc(target->dummy,source_addrsize);	    
	memcpy(target->dummy,source.dummy,source_addrsize);
	*target_addrsize = source_addrsize;
	break;
    }

    return 1;

 fail:   /* Clear target on failure */
    *target_addrsize = 0;
    if (target->dummy) {
	free(target->dummy);
	target->dummy = NULL;
    }
    return 0;
}
					    

/* Return 1 if succeed, 0 if too small */
static int ra_set_address_from_se_address P_((struct remote_account *ra,
					      struct service_entry_addr *ad,
					      const char * official_name));
static int ra_set_address_from_se_address(ra,ad,official_name)
     struct remote_account *ra;
     struct service_entry_addr *ad;
     const char * official_name;
{
    int ret ;

    if (ra->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "ra_set_address_from_se_address",
	      "Bad magic number",0);
    
    if (ad->hostname)
	ra->hostname = strmcpy(ra->hostname,ad->hostname);
    else
	ra->hostname = strmcpy(ra->hostname,official_name);

    ret = set_SOCKADDR_ptr_helper(& (ra->addrsize),
				   & (ra->hostaddr),
				   ad->addrsize,
				   ad->address);

    if (!ret) {
	free(ra->hostname);
	ra->hostname = NULL;
    }	

    return ret;
}

#if NOTUSED
static void ra_set_address_from_generic P_((struct remote_account *ra,
					   const SOCKADDR *generic));
static void ra_set_address_from_generic(ra,generic)
     struct remote_account *ra;
     const SOCKADDR *generic;
{
    if (ra->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "ra_set_address_from_generic",
	      "Bad magic number",0);
 
    set_address_from_gen_helper(& ( ra->addrsize ),
				& ( ra->hostaddr ),
				generic);   
}
#endif

static void connect_set_bgcan P_((struct bgconnect_list ** bglist_p,
				  struct cancel_data    ** bgcan_p,

				  struct service_entry   * se,
				  struct cancel_data     * main_cancel));
static void connect_set_bgcan(bglist_p,bgcan_p,se,main_cancel)
     struct bgconnect_list ** bglist_p;
     struct cancel_data    ** bgcan_p;

     struct service_entry   * se;
     struct cancel_data     * main_cancel;
{

    if (connect_mode) {
	*bglist_p = malloc_bgconnect_list(se);

	if (main_cancel) {
	    set_cancel_message(main_cancel,
			       100 /* msec */,
			       CATGETS(elm_msg_cat, MeSet,MeConnectingSimple,
				       "Connecting to %s"),
			       se->official_name);
	    *bgcan_p = main_cancel;
	} else {
	    *bgcan_p  = new_schedule_cancel(100 /* msec */,
					 CATGETS(elm_msg_cat, MeSet,MeConnectingSimple,
						 "Connecting to %s"),
					 se->official_name);
	}
    } else if (main_cancel) {
	set_cancel_message(main_cancel,
			   -1,
			   CATGETS(elm_msg_cat, MeSet,MeConnectingSimple,
				   "Connecting to %s"),
			   se->official_name);
	*bgcan_p = main_cancel;
    }
}

static void connect_ra_cleanup P_((struct remote_account      * ra));
static void connect_ra_cleanup(ra)
     struct remote_account      * ra;
{
    if (ra->stream) {
	DPRINT(Debug,12,(&Debug,
			 "connect_ra_cleanup: Closing for reopening\n"));
	FreeStreamStack2(& (ra->stream),
			 1 /* Force clearing of stack even when this is not last reference */);
    }

    if (ra->service) {
	DPRINT(Debug,12,(&Debug,
			 "connect_ra_cleanup: Removing old service entry: %p\n",
			 ra->service));	 
	free_service_entry(& (ra->service));
    }


}


/* Return ok */
static int connect_wait_bglist P_((struct remote_account  * ra,
				   struct service_entry   * se,
				   struct bgconnect_list  * bglist,
				   struct cancel_data     * bgcan,
				   int                    * last_error_p,
				   int                    * cur_socket_p,
				   const struct service_type ** got_type,
				   int                        * got_port));
static int connect_wait_bglist(ra,se,bglist,bgcan,last_error_p,cur_socket_p,
			       got_type,got_port)
     struct remote_account      * ra;
     struct service_entry       * se;
     struct bgconnect_list      * bglist;
     struct cancel_data         * bgcan;
     int                        * last_error_p;
     int                        * cur_socket_p;
     const struct service_type ** got_type;
     int                        * got_port;
{
    int ok = 0;
    
    enum connect_result         r = connect_in_progress;
    char             * addrstring = NULL;
       	
 redo_wait:
    if (bgcan &&  is_canceled(bgcan)) {
	DPRINT(Debug,12,(&Debug,
			 "connect_wait_bglist: Connection canceled, quitting\n"));
	
    } else { 	    
	r = bgconnect_got_connection(bglist,ra,cur_socket_p,
				     got_port,got_type,
				     last_error_p,bgcan,
				     bgwait_completed,&addrstring,NULL,0,NULL);
	    
	switch(r) {
	default:
	    DPRINT(Debug,12,(&Debug,
			     "connect_wait_bglist: status %d\n",
			     r));
		
	    break;
		
	case connect_in_progress:
	    DPRINT(Debug,12,(&Debug,
			     "connect_wait_bglist: Connection in progress to %s [%s/%s], port %d\n",		 
			     se->official_name,
			     ra->hostname ? ra->hostname : "???",
			     addrstring ? addrstring : "???",
			     *got_port));
	    goto redo_wait;
	    	    
	case connect_OK:
		
	    DPRINT(Debug,12,(&Debug,
			     "connect_wait_bglist: Connection succeed %s [%s/%s], port %d\n",
			     se->official_name,
			     ra->hostname ? ra->hostname : "???",
			     addrstring ? addrstring : "???",
			     *got_port));
	    ok = 1;
		
	    if (transaction_file) {
		time_t now = 0;
		
		if (((time_t) -1) != time(&now)) {
		    struct tm * zz = localtime(&now);
		    
		    if (!zz)
			goto no_time4;
		    
		    fprintf(transaction_file,
			    "%d [%d] %02d:%02d:%02d === CONNECT %s [%s/%s], port %d\n",
			    getpid(),*cur_socket_p,
			    zz->tm_hour,
			    zz->tm_min,
			    zz->tm_sec,
			    se->official_name,
			    ra->hostname ? ra->hostname : "???",
			    addrstring ? addrstring : "???",
			    *got_port);
		} else {
		no_time4:
			
		    fprintf(transaction_file,
			    "%d [%d] === CONNECT %s [%s/%s], port %d\n",
			    getpid(),*cur_socket_p,
			    se->official_name,
			    ra->hostname ? ra->hostname : "???",
			    addrstring ? addrstring : "???",
			    *got_port);
		    
		}
	    }

	    update_ra_sockname(*cur_socket_p,ra);
	    
	}	    	    
    }
    
    if (addrstring)
	free(addrstring);
    
    return ok;
}

static void connect_error_clean P_((struct service_entry   * se,
				    int                      was_canceled,
				    struct cancel_data     * bgcan,
				    int                      last_error,
				    int                    * cur_socket_p));

static void connect_error_clean(se,was_canceled,bgcan,last_error,cur_socket_p)
     struct service_entry   * se;
     int                      was_canceled;
     struct cancel_data     * bgcan;
     int                      last_error;
     int                    * cur_socket_p;
{
    connect_error_clean_1(se->official_name,was_canceled,bgcan,last_error,cur_socket_p);
}


static int connect_result_ok P_((struct service_entry   * se,
				 struct remote_account  * ra,
				 int                      idx,
				 enum connect_result      r,
				 char                   * addrstring,
				 int                      got_port,
				 int                      cur_socket
				 ));
static int connect_result_ok(se,ra,idx,r,addrstring,got_port,cur_socket)
     struct service_entry   * se;
     struct remote_account  * ra;
     int                      idx;
     enum connect_result      r;
     char                   * addrstring;
     int                      got_port;
     int                      cur_socket;
{
    int ok = 0;
    
    switch(r) {
    default:
	DPRINT(Debug,12,(&Debug,
			 "connect_result_ok: %d: status %d\n",
			 idx,r));
	
	break;
	
    case connect_in_progress:
	DPRINT(Debug,12,(&Debug,
			 "connect_result_ok: %d: Connection in progress to %s [%s/%s], port %d\n",
			 idx,
			 se->official_name,
			 ra->hostname ? ra->hostname : "???",
			 addrstring ? addrstring : "???",
			 got_port));

	break;
	
    case connect_OK:
	
	DPRINT(Debug,12,(&Debug,
			 "connect_result_ok: %d: Connection succeed %s [%s/%s], port %d\n",
			 idx,
			 se->official_name,
			 ra->hostname ? ra->hostname : "???",
			 addrstring ? addrstring : "???",
			 got_port));
	ok = 1;
	    
	if (transaction_file) {
	    time_t now = 0;
	    
	    if (((time_t) -1) != time(&now)) {
		struct tm * zz = localtime(&now);
		
		if (!zz)
		    goto no_time3;
		
		fprintf(transaction_file,
			"%d [%d] %02d:%02d:%02d === CONNECT %s [%s/%s], port %d\n",
			getpid(),cur_socket,
			zz->tm_hour,
			zz->tm_min,
			zz->tm_sec,
			se->official_name,
			ra->hostname ? ra->hostname : "???",
			addrstring ? addrstring : "???",
			got_port);
	    } else {
	    no_time3:
		fprintf(transaction_file,
			"%d [%d] === CONNECT %s [%s/%s], port %d\n",
			getpid(),cur_socket,
			se->official_name,
			ra->hostname ? ra->hostname : "???",
			addrstring ? addrstring : "???",
			got_port);

	    }
	}

	update_ra_sockname(cur_socket,ra);
	
    }

    return ok;
}

static void connect_socket_reset P_((int  * cur_socket_p,
				     int idx));
static void connect_socket_reset(cur_socket_p,idx)
     int  * cur_socket_p;
     int idx;
{
    if (*cur_socket_p != -1) {
	
	DPRINT(Debug,12,(&Debug,
			 "connect_socket_reset: %d: closing socket (%d)\n",
			 idx,
			 *cur_socket_p));
	
	close(*cur_socket_p);
	*cur_socket_p = -1;	    
    }
}

static int connect_setup_address P_((struct remote_account      * ra,
				     struct service_entry       * se,
				     int idx));
static int connect_setup_address(ra,se,idx)
     struct remote_account      * ra;
     struct service_entry       * se;
     int idx;
{
    if (idx < 0 || idx >= se->addr_count) {
	DPRINT(Debug,12,(&Debug,
			 "connect_setup_address: %d: bad index\n",
			 idx));
	return 0;
    }
    
    if (!se->addr_list[idx].address.dummy) {
	DPRINT(Debug,12,(&Debug,
			 "connect_setup_address: %d: No address\n",idx));
	return 0;
    }
	    
    if (! ra_set_address_from_se_address(ra,& (se->addr_list[idx]),
					 se->official_name)) {
	DPRINT(Debug,12,(&Debug,
			 "connect_setup_address: %d: bad address\n",idx));
	return 0;
    }					     

    if (! ra->hostaddr.sa)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "connect_remote_account",
	      "No address",0);

    return 1;    
}

static int connect_copy_sockaddr P_((SOCKADDR               * A_p,
				     struct remote_account  * ra,
				     char                  ** addrstring_p,
				     int                    * port_p,
				     char                     buffer[],
				     size_t                   buffer_size,
				     const char            ** s_p
				     ));
static int connect_copy_sockaddr(A_p,ra,addrstring_p,port_p,buffer,buffer_size,s_p)
     SOCKADDR               * A_p;
     struct remote_account  * ra;
     char                  ** addrstring_p;
     int                    * port_p;
     char                     buffer[];
     size_t                   buffer_size;
     const char            ** s_p;
{

    if (! set_SOCKADDR_from_ptr_helper(A_p,
				       ra->addrsize,
				       ra->hostaddr)) {
	DPRINT(Debug,11,(&Debug,"connect_copy_sockaddr: bad address\n"));
	
	return 0;
    }
    		    
    *s_p = give_addr_as_string(A_p,buffer,buffer_size);
    *port_p = give_addr_port(A_p);
		    
    if (*s_p)
	*addrstring_p = strmcpy(*addrstring_p,*s_p);
    else if (*addrstring_p) {
	free(*addrstring_p);
	*addrstring_p = NULL;
    }

    return 1;
}


static enum connect_result connect_handle_tail P_((struct remote_account      * ra,
						   const char                 * s,
						   struct cancel_data         * can,
						   int                          this_port,
						   int                          portgroup,
						   int                        * cur_socket_p,
						   struct bgconnect_list      * bglist,
						   const struct service_type ** got_type,
						   int                        * got_port,
						   int                        * last_error_p,
						   const struct service_type  * st
						   ));
static enum connect_result connect_handle_tail(ra,s,can,this_port,portgroup,
					       cur_socket_p, bglist,got_type,
					       got_port,last_error_p,st)
     struct remote_account      * ra;
     const char                 * s;
     struct cancel_data         * can;
     int                          this_port;
     int                          portgroup;
     int                        * cur_socket_p;
     struct bgconnect_list      * bglist;
     const struct service_type ** got_type;
     int                        * got_port;
     int                        * last_error_p;
     const struct service_type  * st;
{
    enum connect_result r = connect_none;

    struct bgconnect_info  * bgconnect = NULL;

    DPRINT(Debug,14,(&Debug,
		     "connect_handle_tail: connecting %s port %d service %p %s\n",
		     s ? s : "???",
		     this_port,st,service_type_name(st)));
		         
    r = connect_one_IN(ra,cur_socket_p,last_error_p,
		       can,1,
		       connect_mode ? &bgconnect : NULL,
		       portgroup);
			
    switch(r) {
    default:                                  break;
	
    case connect_in_progress:
	if (bgconnect && bglist)
	    bgconnect_add_to_list(bglist,bgconnect,this_port,
				  st);  /* increments refcount */
	
	*got_port = this_port;
	if (got_type)
	    *got_type = st;
	
	if (bgconnect &&
	    connect_mode_nonblocking ==
	    connect_mode) {
	    r =  bgconnect_wait_this(bglist,bgconnect,
				     last_error_p,can,
				     cur_socket_p);
				
	} else if (bgconnect && bglist &&
		   ( connect_mode_display_time.timeout_sec  ||
		     connect_mode_display_time.timeout_ms)) {
				
	    if (!  bgconnect_wait_delay(bglist,bgconnect,can,
					&connect_mode_display_time)) {
		DPRINT(Debug,11,(&Debug,"connect_handle_tail: ... wait canceled \n"));
				    
	    }				
	}
			    			    
	break;
			    
    case connect_OK:
	
	*got_port = this_port;
	if (got_type)
	    *got_type = st;
	
	break;
    }

    if (bgconnect) 
	free_backgroud_connect_info(& bgconnect);  /* decrements refcount */

    DPRINT(Debug,12,(&Debug,
		     "connect_handle_tail=%d",r));
    switch (r) {
    case connect_fail_abort:  DPRINT(Debug,12,(&Debug," (connect_fail_abort)"));  break;
    case connect_fail:        DPRINT(Debug,12,(&Debug," (connect_fail)"));        break;
    case connect_none:        DPRINT(Debug,12,(&Debug," (connect_none)"));        break;
    case connect_OK:          DPRINT(Debug,12,(&Debug," (connect_OK)"));          break;
    case connect_in_progress: DPRINT(Debug,12,(&Debug," (connect_in_progress)")); break;
    }
    if (-1 != *cur_socket_p) {
	DPRINT(Debug,12,(&Debug," *cur_socket_p=%d",*cur_socket_p));
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    

    return r;
}


static enum connect_result connect_handle_port P_((struct remote_account      * ra,
						    struct service_entry       * se,
						    const char                 * s,
						    struct cancel_data        ** can_p,
						    int                          this_port,
						    int                          idx,
						    int                        * cur_socket_p,
						    struct bgconnect_list      * bglist,
						    const struct service_type ** got_type,
						    int                        * got_port,
						    int                        * last_error_p
						    ));
static enum connect_result connect_handle_port(ra,se,s,can_p,this_port,idx,cur_socket_p,
						bglist,got_type,got_port,
						last_error_p)
     struct remote_account      * ra;
     struct service_entry       * se;
     const char                 * s;
     struct cancel_data        ** can_p;
     int                          this_port;
     int                          idx;
     int                        * cur_socket_p;
     struct bgconnect_list      * bglist;
     const struct service_type ** got_type;
     int                        * got_port;
     int                        * last_error_p;
{
    enum connect_result r = connect_none;
    
    if (ra->hostname && 
	0 != strcmp(se->official_name,ra->hostname))
	*can_p = new_cancel(CATGETS(elm_msg_cat, MeSet,
				    MeConnectingPort3,
				    "Connecting to %s [%s/%s], port %d ... (%d)"),	 
			    se->official_name,
			    ra->hostname,
			    s ? s : "???",
			    this_port,
			    idx);
    else
	*can_p = new_cancel(CATGETS(elm_msg_cat, MeSet,
				    MeConnectingPort,
				    "Connecting to %s [%s], port %d ... (%d)"),		  
			    se->official_name,
			    s ? s : "???",
			    this_port,
			    idx);

    r = connect_handle_tail(ra,s,*can_p,this_port,0,
			    cur_socket_p, bglist,got_type,
			    got_port,last_error_p,se->service);


    DPRINT(Debug,12,(&Debug,
		     "connect_handle_port=%d",r));
    switch (r) {
    case connect_fail_abort:  DPRINT(Debug,12,(&Debug," (connect_fail_abort)"));  break;
    case connect_fail:        DPRINT(Debug,12,(&Debug," (connect_fail)"));        break;
    case connect_none:        DPRINT(Debug,12,(&Debug," (connect_none)"));        break;
    case connect_OK:          DPRINT(Debug,12,(&Debug," (connect_OK)"));          break;
    case connect_in_progress: DPRINT(Debug,12,(&Debug," (connect_in_progress)")); break;
    }
    if (-1 != *cur_socket_p) {
	DPRINT(Debug,12,(&Debug," *cur_socket_p=%d",*cur_socket_p));
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return r;
}

static void connect_host_message P_((struct remote_account      * ra,
				     struct service_entry       * se,
				     const char                 * s,
				     struct cancel_data        ** can_p,
				     int                          idx));
static void connect_host_message(ra,se,s,can_p,idx)
     struct remote_account      * ra;
     struct service_entry       * se;
     const char                 * s;
     struct cancel_data        ** can_p;
     int                          idx;
{
    if (ra->hostname && 
	0 != strcmp(se->official_name,ra->hostname))
	*can_p = new_cancel(CATGETS(elm_msg_cat, MeSet,
				    MeConnecting2,
				    "Connecting to %s [%s/%s]... (%d)"),
			    se->official_name,
			    ra->hostname,
			    s ? s : "???",
			    idx);
    else
	*can_p = new_cancel(CATGETS(elm_msg_cat, MeSet,
				 MeConnecting,
				 "Connecting to %s [%s]... (%d)"),
			    se->official_name,
			    s ? s : "???",
			    idx);
}


static void connect_copy_addrstring P_((char  ** addrstring_p,
					 const char               * s));
static void connect_copy_addrstring(addrstring_p,s)
     char  ** addrstring_p;
     const char               * s;
{
    /* bgconnect_got_connection resets addrstring */
    if (s)
	*addrstring_p = strmcpy(*addrstring_p,s);
    else if (*addrstring_p) {
	free(*addrstring_p);
	*addrstring_p = NULL;
    }
}

    

static enum connect_result connect_handle_portlist P_((struct remote_account    * ra,
							struct service_entry     * se,
							const char               * s,
							struct cancel_data       * can,
							int                        portcount,
							int                      * portlist,
							int                        portgroup,
							char                    ** addrstring_p,
							int                      * cur_socket_p,
							struct bgconnect_list   *  bglist,
							const struct service_type ** got_type,
							int                      * got_port,
							int                      * last_error_p));
static enum connect_result connect_handle_portlist(ra,se,s,can,
						   portcount,portlist,
						   portgroup,
						   addrstring_p,
						   cur_socket_p,bglist,got_type,
						   got_port,last_error_p)
     struct remote_account    * ra;
     struct service_entry     * se;
     const char               * s;
     struct cancel_data       * can;
     int                        portcount;
     int                      * portlist;
     int                        portgroup;
     char                    ** addrstring_p;
     int                      * cur_socket_p;
     struct bgconnect_list    * bglist;
     const struct service_type ** got_type;
     int                      * got_port;
     int                      * last_error_p;
{
    enum connect_result r = connect_none;

    int idx2;
    int next_idx2 = 0;
    
    for (idx2 = 0; idx2 < portcount && (connect_none == r || connect_in_progress == r) ;
	 idx2 = next_idx2) {

	if (bglist)
	    r = bgconnect_got_connection(bglist,ra,cur_socket_p,
					 got_port,got_type,
					 last_error_p,can,
					 bgwait_space_or_completed  /* XXXX perhaps other? */,
					 addrstring_p,
					 &connect_mode_addrdelay,portgroup,
					 &connect_mode_portdelay);

	if (connect_none != r) {
	    DPRINT(Debug,12,(&Debug,
			     "connnect_handle_portlist: got background connection, %s\n",
			     *addrstring_p ? *addrstring_p : "--"));
	    
	} else if (is_canceled(can)) {
	    DPRINT(Debug,12,(&Debug,
			     "connnect_handle_portlist: Connection canceled, quitting\n"));
	    break;
	} else {
				    
	    /* bgconnect_got_connection resets addrstring */
	    connect_copy_addrstring(addrstring_p,s);
	    
	    next_idx2 = idx2+1;
	    
	    set_addr_port(ra->hostaddr.generic,
			  portlist[idx2]);

	    r = connect_handle_tail(ra,s,can,portlist[idx2],portgroup,
				    cur_socket_p, bglist,got_type,
				    got_port,last_error_p,se->service);
   	    
	}
    }


    DPRINT(Debug,12,(&Debug,
		     "connect_handle_portlist=%d",r));
    switch (r) {
    case connect_fail_abort:  DPRINT(Debug,12,(&Debug," (connect_fail_abort)"));  break;
    case connect_fail:        DPRINT(Debug,12,(&Debug," (connect_fail)"));        break;
    case connect_none:        DPRINT(Debug,12,(&Debug," (connect_none)"));        break;
    case connect_OK:          DPRINT(Debug,12,(&Debug," (connect_OK)"));          break;
    case connect_in_progress: DPRINT(Debug,12,(&Debug," (connect_in_progress)")); break;
    }
    if (-1 != *cur_socket_p) {
	DPRINT(Debug,12,(&Debug," *cur_socket_p=%d",*cur_socket_p));
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return r;
}

#define ENUM_SOCKET_ADDRESS_magic	0xF922

static struct enum_socket_address {
    unsigned short           magic;          /* ENUM_SOCKET_ADDRESS_magic */

    struct service_entry   * se;

    int                      ipv_option_index;   /* -1 if regulal index is used */    
    int                      master_idx;         /* count of used address       */

    int                      se_idx;             /* Running service index       */

    int                    * used_idx;           /* -1 ... se address not yet used 
						    indexed  by se_idx
						 */
    size_t                   used_idx_len;       /* initial_se->addr_count */      

    enum enum_sock_mode {
	enum_socket_address_next,  /* this is next to used address */
	enum_socket_address_used,  /* this address is used      */
	enum_socket_address_end

    }                        mode;
    
} NULL_enum_socket_address = {  
    ENUM_SOCKET_ADDRESS_magic,

    NULL, 
    -1,
    0,
    0,

    NULL,
    0,
    enum_socket_address_end  /* no addresses */
};

static void init_enum_socket_address  P_((struct enum_socket_address * walker,
					  struct service_entry       * se));
static void init_enum_socket_address(walker,se)
     struct enum_socket_address * walker;
     struct service_entry   * se;
{
    bzero (walker, sizeof (*walker));
    walker->magic = ENUM_SOCKET_ADDRESS_magic;

    walker->used_idx     = NULL;
    walker->used_idx_len = 0;
    walker->mode         = enum_socket_address_end;
    walker->se_idx       = 0;

    if (se && SERVICE_ENTRY_magic != se->magic)
        panic("CONNECTION PANIC",__FILE__,__LINE__,
              "init_enum_socket_address",
	      "Bad magic (service_entry)",0);
       
    walker->se = se;
    if (walker->se) {
	inc_service_entry_refcount(walker->se);

	if (walker->se->addr_count > 0) {
	    size_t i;
	    walker->used_idx_len = walker->se->addr_count;

	    walker->used_idx =
		safe_calloc(walker->used_idx_len,
			    sizeof (walker->used_idx[0]));

	    for (i = 0; i < walker->used_idx_len; i++)
		walker->used_idx[i] = -1;  /* Not used yet */

	    walker->mode = enum_socket_address_next;	    
	}
    }

    if (ipv_option_array_len > 0) {
	walker->ipv_option_index = 0;	
    } else {
	walker->ipv_option_index = -1;
    }
    walker->master_idx = 0;
    
}

static void clear_enum_socket_address P_((struct enum_socket_address * walker));
static void clear_enum_socket_address(walker)
     struct enum_socket_address * walker;
{
    if (ENUM_SOCKET_ADDRESS_magic != walker->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
              "clear_enum_socket_address",
	      "Bad magic (enum_socket_address)",0);

    if (walker->se)
	free_service_entry(& (walker->se));

    if (walker->used_idx) {
	free(walker->used_idx);
	walker->used_idx = NULL;
    }

    *walker = NULL_enum_socket_address;
}

static void step_enum_socket_address P_((struct enum_socket_address * walker));
static void step_enum_socket_address(walker)
     struct enum_socket_address * walker;
{
    if (ENUM_SOCKET_ADDRESS_magic != walker->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
              "step_enum_socket_address",
	      "Bad magic (enum_socket_address)",0);

    DPRINT(Debug,14,(&Debug,
		     "step_enum_socket_address: ipv_option_index=%d master_idx=%d se_idx=%d mode=%d",
		     walker->ipv_option_index,walker->master_idx, walker->se_idx,
		     walker->mode));
    switch (walker->mode) {
    case enum_socket_address_next: DPRINT(Debug,14,(&Debug," enum_socket_address_next")); break;
    case enum_socket_address_used: DPRINT(Debug,14,(&Debug," enum_socket_address_used")); break;
    case enum_socket_address_end:  DPRINT(Debug,14,(&Debug," enum_socket_address_end"));  break;
    }
    DPRINT(Debug,14,(&Debug," start\n"));
    
    if (walker->se) {
	int retry = 0;
	
	if (SERVICE_ENTRY_magic != walker->se->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "step_enum_socket_address",
		  "Bad magic (service_entry)",0);

	if (walker->se->official_name) {
	    DPRINT(Debug,14,(&Debug,
			     "step_enum_socket_address: official_name=%s\n",
			     walker->se->official_name));
	}
		
	do {
	    DPRINT(Debug,14,(&Debug,
			     "step_enum_socket_address: %s%s: ipv_option_index=%d master_idx=%d se_idx=%d mode=%d",
			     walker->se->official_name ?
			     walker->se->official_name :
			     "",
			     retry ? " retry" : "",
			     walker->ipv_option_index,walker->master_idx, walker->se_idx,
			     walker->mode));
	    switch (walker->mode) {
	    case enum_socket_address_next: DPRINT(Debug,14,(&Debug," enum_socket_address_next")); break;
	    case enum_socket_address_used: DPRINT(Debug,14,(&Debug," enum_socket_address_used")); break;
	    case enum_socket_address_end:  DPRINT(Debug,14,(&Debug," enum_socket_address_end"));  break;
	    }
	    DPRINT(Debug,14,(&Debug,"\n"));
	    
	    retry = 0;
	
	    switch (walker->mode) {
		int af;
	    case enum_socket_address_used:
		DPRINT(Debug,14,(&Debug,
				 "step_enum_socket_address: address used\n"));
		/* FALLTRU */		
	    case enum_socket_address_next:
		
		if (-1 == walker->ipv_option_index)
		    af = AF_UNSPEC;
		else if (walker->ipv_option_index >= ipv_option_array_len) {
		    
		    size_t skipped = 0;
		    
		    if (walker->used_idx) {
			size_t i;
			
			for (i = 0; i < walker->used_idx_len; i++)
			    if (-1 == walker->used_idx[i])
				skipped++;
		    }
		    
		    DPRINT(Debug,14,(&Debug,
				     "step_enum_socket_address: End of ipv_option_array, skipped=%zu\n",
				     skipped));
		    
		    if (skipped) {
			char option[NUM_protocols+1];
			size_t i;
			size_t a = 0;
			
			
			for (i = 0; i < NUM_protocols && i < ipv_option_array_len; i++) {
			    if (isascii(ipv_option_copy[i]) && isprint(ipv_option_copy[i])) {
				option[a++] = ipv_option_copy[i];
			    }
			}
			
			option[a] = '\0';
			if (a > 0) {
			    
			    if (1 == skipped)			    
				lib_error(CATGETS(elm_msg_cat, MeSet,
						  MeConnectingSkip1Opt,
						  "Connecting to %s ... Option -%s skipped %zu address"),
					  walker->se->official_name,
					  option,skipped);
			    else			    
				lib_error(CATGETS(elm_msg_cat, MeSet,
						  MeConnectingSkipOpt,
						  "Connecting to %s ... Option -%s skipped %zu addresses"),
					  walker->se->official_name,
					  option,skipped);
			}		
		    }

		    walker->mode = enum_socket_address_end;
		    break;
		    
		} else if (walker->ipv_option_index >= 0) {
		    af =  ipv_option_array[walker->ipv_option_index];
		    
		} else {

		    DPRINT(Debug,1,(&Debug,
				    "step_enum_socket_address: walker->ipv_option_index=%d\n",
				    walker->ipv_option_index));
		    
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "step_enum_socket_address",
			  "Bad walker->ipv_option_index",0);
		    break;
		}
		
		if (walker->se_idx >= walker->se->addr_count) {

		    DPRINT(Debug,14,(&Debug,
				     "step_enum_socket_address: End of addr_list array\n"));
		    
		    if (-1 == walker->ipv_option_index)
			walker->mode = enum_socket_address_end;
		    else {
			walker->se_idx = 0;
			walker->ipv_option_index++;
			retry = 1;
		    }
		} else if (walker->se_idx >= 0) {

		    if (walker->se->addr_list) {

			if (enum_socket_address_used == walker->mode) {

			    DPRINT(Debug,14,(&Debug,
					     "step_enum_socket_address: %d: address #%d: Incrementing address\n",
					     walker->master_idx,
					     walker->se_idx));
			    
			    walker->se_idx++;
			    retry = 1;
			    walker->mode = enum_socket_address_next;
			    walker->master_idx++;   /* Count transitions */
			} else if (enum_socket_address_next == walker->mode) {
			    struct service_entry_addr * A =
				&(walker->se->addr_list[walker->se_idx]);

			    DPRINT(Debug,14,(&Debug,
					     "step_enum_socket_address: %d: address #%d: Looking this address\n",
					     walker->master_idx,
					     walker->se_idx));

			    
			    if (A->address.sa) {
				char mybuffer[256];		    
				const char * s UNUSED_VAROK
				    = give_SOCKADDR_ptr_as_string(A->address,
								  A->addrsize,
								  mybuffer,
								  sizeof mybuffer);
				
				DPRINT(Debug,14,(&Debug,
						 "step_enum_socket_address: %d: address #%d family %d",
						 walker->master_idx,
						 walker->se_idx,
						 A->address.sa->sa_family));

				if (s) {
				    DPRINT(Debug,14,(&Debug," %s", s));
				}
						 								
				if (AF_UNSPEC == af ||
				    af == A->address.sa->sa_family) {

				    /* Either any address family matches
				       or this is required address family */

				    DPRINT(Debug,14,(&Debug," matches"));
				    
				    if (walker->used_idx &&
					walker->se_idx < walker->used_idx_len) {

					if (walker->master_idx ==
					    walker->used_idx[walker->se_idx]) {
					    
					    /* This was found earlier */
					    
					    DPRINT(Debug,14,(&Debug," OK -- found earlier\n"));

					    
					    break;
					} else if (-1 != walker->used_idx[walker->se_idx]) {

					    /* Already used */

					    DPRINT(Debug,14,(&Debug," -- already used on %d\n",
							     walker->used_idx[walker->se_idx]));

					    
					    walker->se_idx++;
					    retry = 1;
					} else {

					    /* Found this */

					    DPRINT(Debug,14,(&Debug," OK -- found\n"));
					    					    
					    walker->used_idx[walker->se_idx] =
						walker->master_idx;
					}
				    }

				    /* maybe OK */
				    
				} else {
				    /* Filtered address family */

				    DPRINT(Debug,14,(&Debug," filtered\n"));
				    
				    walker->se_idx++;
				    retry = 1;
				}

			    } else {
				/* Bad address */

				DPRINT(Debug,14,(&Debug,
						 "step_enum_socket_address: %d: address #%d no address\n",
						 walker->master_idx,
						 walker->se_idx));
				
				walker->se_idx++;
				retry = 1;
			    }
			    
			} else
			    break;
												
		    } else {

			DPRINT(Debug,14,(&Debug,
					 "step_enum_socket_address: No address list\n"));
			
			walker->mode = enum_socket_address_end;
			break;
		    }
			
		} else {

		    DPRINT(Debug,1,(&Debug,
				    "step_enum_socket_address:  walker->se_idx=%d\n",
				    walker->se_idx));
		    
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "step_enum_socket_address",
			  "Bad walker->se_idx",0);
		    break;

		}
				
	    case enum_socket_address_end:

		DPRINT(Debug,14,(&Debug,
				 "step_enum_socket_address: address end\n"));
		
		break;
	    }
	    
	} while (retry);

    } else {
	DPRINT(Debug,14,(&Debug,
			 "step_enum_socket_address: No se\n"));
	
	walker->mode = enum_socket_address_end;
    }


    DPRINT(Debug,14,(&Debug,
		     "step_enum_socket_address: ipv_option_index=%d master_idx=%d se_idx=%d mode=%d",
		     walker->ipv_option_index,walker->master_idx, walker->se_idx,
		     walker->mode));
    switch (walker->mode) {
    case enum_socket_address_next: DPRINT(Debug,14,(&Debug," enum_socket_address_next")); break;
    case enum_socket_address_used: DPRINT(Debug,14,(&Debug," enum_socket_address_used")); break;
    case enum_socket_address_end:  DPRINT(Debug,14,(&Debug," enum_socket_address_end"));  break;
    }
    DPRINT(Debug,14,(&Debug," done\n"));
    
}



static int have_enum_socket_address P_((struct enum_socket_address * walker));
static int have_enum_socket_address(walker)
    struct enum_socket_address * walker;
{
    int ret = 0;

    if (ENUM_SOCKET_ADDRESS_magic != walker->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
              "have_enum_socket_address",
	      "Bad magic (enum_socket_address)",0);
    
    step_enum_socket_address(walker);

    DPRINT(Debug,14,(&Debug,
		     "have_enum_socket_address: ipv_option_index=%d master_idx=%d se_idx=%d mode=%d",
		     walker->ipv_option_index,walker->master_idx, walker->se_idx,
		     walker->mode));
    switch (walker->mode) {
    case enum_socket_address_next: DPRINT(Debug,14,(&Debug," enum_socket_address_next")); break;
    case enum_socket_address_used: DPRINT(Debug,14,(&Debug," enum_socket_address_used")); break;
    case enum_socket_address_end:  DPRINT(Debug,14,(&Debug," enum_socket_address_end"));  break;
    }
    DPRINT(Debug,14,(&Debug," after step\n"));

    if (walker->se) {
	
	if (SERVICE_ENTRY_magic != walker->se->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "have_enum_socket_address",
		  "Bad magic (service_entry)",0);

	if (walker->se->official_name) {
	    DPRINT(Debug,14,(&Debug,
			     "have_enum_socket_address: official_name=%s\n",
			     walker->se->official_name));
	}
    
	switch (walker->mode) {
	case enum_socket_address_next:

	    if (walker->se_idx < walker->se->addr_count) {
		DPRINT(Debug,14,(&Debug,
				 "have_enum_socket_address: have address se_idx=%d < addr_count=%d\n",
				 walker->se_idx,walker->se->addr_count));

		ret = 1;
	    }
	    break;
	case enum_socket_address_used:
	    DPRINT(Debug,14,(&Debug,
			     "have_enum_socket_address: address used se_idx=%d\n",
			     walker->se_idx));
	    break;
	case enum_socket_address_end:
	    DPRINT(Debug,14,(&Debug,
			     "have_enum_socket_address: end of addresses\n"));
	    break;
	}

    } else {
	DPRINT(Debug,14,(&Debug,
			 "have_enum_socket_address: No se\n"));
    }

    DPRINT(Debug,14,(&Debug,
		     "have_enum_socket_address=%d\n",ret));
    
    return ret;
}

static int idx_enum_socket_address P_((struct enum_socket_address * walker));
static int idx_enum_socket_address(walker)
     struct enum_socket_address * walker;
{
    int ret;
    
    
    if (ENUM_SOCKET_ADDRESS_magic != walker->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
              "idx_enum_socket_address",
	      "Bad magic (enum_socket_address)",0);
 
    DPRINT(Debug,14,(&Debug,
		     "idx_enum_socket_address: ipv_option_index=%d master_idx=%d se_idx=%d mode=%d",
		     walker->ipv_option_index,walker->master_idx, walker->se_idx,
		     walker->mode));
    switch (walker->mode) {
    case enum_socket_address_next: DPRINT(Debug,14,(&Debug," enum_socket_address_next")); break;
    case enum_socket_address_used: DPRINT(Debug,14,(&Debug," enum_socket_address_used")); break;
    case enum_socket_address_end:  DPRINT(Debug,14,(&Debug," enum_socket_address_end"));  break;
    }
    DPRINT(Debug,14,(&Debug,"\n"));



    ret = walker->master_idx;
    
    DPRINT(Debug,14,(&Debug,
		     "idx_enum_socket_address=%d\n",ret));

    return ret;
}

/* Returns se_idx or -1 */
static int advance_enum_socket_address P_((struct enum_socket_address * walker));
static int advance_enum_socket_address(walker)
     struct enum_socket_address * walker;
{
    int ret = -1;

    if (ENUM_SOCKET_ADDRESS_magic != walker->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
              "advance_enum_socket_address",
	      "Bad magic (enum_socket_address)",0);

    DPRINT(Debug,14,(&Debug,
		     "advance_enum_socket_address: ipv_option_index=%d master_idx=%d se_idx=%d mode=%d",
		     walker->ipv_option_index,walker->master_idx, walker->se_idx,
		     walker->mode));
    switch (walker->mode) {
    case enum_socket_address_next: DPRINT(Debug,14,(&Debug," enum_socket_address_next")); break;
    case enum_socket_address_used: DPRINT(Debug,14,(&Debug," enum_socket_address_used")); break;
    case enum_socket_address_end:  DPRINT(Debug,14,(&Debug," enum_socket_address_end"));  break;
    }
    DPRINT(Debug,14,(&Debug,"\n"));

    if (walker->se) {
	
	if (SERVICE_ENTRY_magic != walker->se->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "advance_enum_socket_address",
		  "Bad magic (service_entry)",0);

	if (walker->se->official_name) {
	    DPRINT(Debug,14,(&Debug,
			     "advance_enum_socket_address: official_name=%s\n",
			     walker->se->official_name));
	}

	switch (walker->mode) {
	case enum_socket_address_next:

	    if (walker->se_idx < walker->se->addr_count) {

		DPRINT(Debug,14,(&Debug,
				 "advance_enum_socket_address: have address se_idx=%d < addr_count=%d  -- marking uesed\n",
				 walker->se_idx,walker->se->addr_count));

		ret = walker->se_idx;
		walker->mode =  enum_socket_address_used;
	    }
	    break;
	case enum_socket_address_used:
	    DPRINT(Debug,14,(&Debug,
			     "advance_enum_socket_address: address used se_idx=%d\n",
			     walker->se_idx));
	    break;
	case enum_socket_address_end:
	    DPRINT(Debug,14,(&Debug,
			     "advance_enum_socket_address: end of addresses\n"));
	    break;
	}
    } else {
	DPRINT(Debug,14,(&Debug,
			 "advance_enum_socket_address: No se\n"));
    }

    DPRINT(Debug,14,(&Debug,
		     "advance_enum_socket_address=%d\n",ret));


    return ret;
}

/* also idx_enum_socket_address -- but may advance step */
static int next_enum_socket_address P_((struct enum_socket_address * walker,
					int *next_idx));
static int next_enum_socket_address(walker,next_idx)
     struct enum_socket_address * walker;
     int *next_idx;
{
    int ret = 0;

    if (ENUM_SOCKET_ADDRESS_magic != walker->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
              "next_enum_socket_address",
	      "Bad magic (enum_socket_address)",0);
        
    step_enum_socket_address(walker);

    DPRINT(Debug,14,(&Debug,
		     "next_enum_socket_address: ipv_option_index=%d master_idx=%d se_idx=%d mode=%d",
		     walker->ipv_option_index,walker->master_idx, walker->se_idx,
		     walker->mode));
    switch (walker->mode) {
    case enum_socket_address_next: DPRINT(Debug,14,(&Debug," enum_socket_address_next")); break;
    case enum_socket_address_used: DPRINT(Debug,14,(&Debug," enum_socket_address_used")); break;
    case enum_socket_address_end:  DPRINT(Debug,14,(&Debug," enum_socket_address_end"));  break;
    }
    DPRINT(Debug,14,(&Debug," after step\n"));

    if (next_idx)
	*next_idx = walker->master_idx;
    
    switch (walker->mode) {
    case enum_socket_address_next:
	ret = 1;
	break;
    case enum_socket_address_used:
    case enum_socket_address_end:
	ret = 0;
    }

    DPRINT(Debug,14,(&Debug,
		     "next_enum_socket_address=%d",
		     ret));
    if (next_idx) {
	DPRINT(Debug,14,(&Debug,"; *next_idx=",
			 *next_idx));
    }    
    DPRINT(Debug,14,(&Debug,"\n"));

    return ret;
}


int connect_remote_account_2(ra,initial_se,main_cancel,
			     result_se,got_type,got_port,

			     /* For enumerate_service_type */			     
			     est)
     
     struct remote_account      * ra;
     struct service_entry       * initial_se;
     struct cancel_data         * main_cancel  /* Used if  dns lookup was  cancelable */     ;

     struct service_entry      ** result_se    /* Increments refcount                 */     ; 
     const struct service_type ** got_type;
     int                        * got_port;


     /* For enumerate_service_type */
     struct enum_service_type *est;
{
    int ok = 0;
    struct bgconnect_list  * bglist = NULL;
    struct cancel_data     * bgcan = NULL;

    int last_error    = 0;
    int cur_socket    = -1;
    int was_canceled  = 0;

    struct service_entry       * this_se = NULL; /* not refcounted */
    const struct service_type  * this_type = NULL;
    int                          this_port = 0;
    struct enum_socket_address   walker = NULL_enum_socket_address;
    

    int portgroup = 0;  /* loop number, when scanning different ports on same address */

    
    if (ra->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "connect_remote_account",
	      "Bad magic number",0);

    if ( SERVICE_ENTRY_magic != initial_se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "connect_remote_account",
	      "Bad magic (service_entry)",0);

    
    if (got_type)
	*got_type = NULL;
    if (got_port)
	*got_port = PORT_end;

    if (result_se && *result_se) {
	DPRINT(Debug,12,(&Debug,
			 "connect_remote_account_2: Removing old result service entry: %p\n",
			 *result_se));	 
	
	free_service_entry(result_se);
    }
	
    connect_set_bgcan(&bglist,&bgcan,initial_se,main_cancel);

    connect_ra_cleanup(ra);
    
    DPRINT(Debug,12,(&Debug,"connect_remote_account_2: %s: address count %d\n",
		     initial_se->official_name,
		     initial_se->addr_count));

    for (init_enum_socket_address(&walker,initial_se);
	 have_enum_socket_address(&walker);
	 ) {
	enum connect_result r = connect_none;
	char             * addrstring = NULL;
	int next_display_idx;
	
	connect_socket_reset(&cur_socket,
			     idx_enum_socket_address(&walker));

	if (bglist)
	    r = bgconnect_got_connection(bglist,ra,&cur_socket,&this_port,&this_type,
					 &last_error,bgcan,
					 bgwait_space_or_completed,&addrstring,
					 &connect_mode_addrdelay,0,NULL);


	if (connect_none != r) {
	    DPRINT(Debug,12,(&Debug,
			     "connect_remote_account_2: got background connection, %s\n",
			     addrstring ? addrstring : "--"));
	} else if (bgcan &&  is_canceled(bgcan)) {
	    DPRINT(Debug,12,(&Debug,
			     "connect_remote_account_2: Connection canceled, quitting\n"));
	    break;
	} else {
	    int se_idx = advance_enum_socket_address(&walker);
	    int display_idx = idx_enum_socket_address(&walker);

	    if (se_idx < 0) {
		DPRINT(Debug,12,(&Debug,
				 "connect_remote_account_2: %d (count %d) => end of addresses\n",
				 display_idx,initial_se->addr_count));
		continue;
	    }
	    
	    DPRINT(Debug,12,(&Debug,
			     "connect_remote_account_2: %d (count %d) => se_idx=%d\n",
			     display_idx,initial_se->addr_count,se_idx));

	    
	    /* Copy all information */
	    if (! connect_setup_address(ra,initial_se,se_idx))
		continue;

	    switch (ra->hostaddr.sa->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
	    case AF_INET: 
#ifdef AF_INET6
	    case AF_INET6: 	
#endif
		{
		    struct cancel_data     * can       = NULL;
		    SOCKADDR A;
		    int port;
		    const char *s;
		    char mybuffer[256];
		    
		    if (! connect_copy_sockaddr(& A,ra,&addrstring,&port,
						mybuffer, sizeof mybuffer,
						&s))
			continue;

		    /* r == 0: not succeed
		       r <  0: fatal error
		       r >  0: succeed
		    */

		    if (PORT_end != port) {
		    
			DPRINT(Debug,11,(&Debug,"connect_remote_account_2: address record gives port %d\n",
					 port));

			r = connect_handle_port(ra,initial_se,s,&can,port,
						display_idx,
						&cur_socket,
						bglist,
						&this_type,
						&this_port,&last_error);
						
		    } else {

			connect_host_message(ra,initial_se,s,&can,port);
			
			if (initial_se->port_count > 0) {
			    
			    portgroup++;
			    
			    DPRINT(Debug,11,(&Debug,
					     "connect_remote_account_2: service list gives portlist (%d ports) => portgroup %d\n",
					     initial_se->port_count,portgroup));
			    
			    
			    r = connect_handle_portlist(ra,
							 initial_se,s,can,
							 initial_se->port_count,
							 initial_se->port_list,
							 portgroup,
							 &addrstring,&cur_socket,
							 bglist,&this_type,
							 &this_port,&last_error);


			    /* Do not use default portlist if port was 
			       given by name */
			} else if (0 == (initial_se->flags & SE_rescan_name_port)) {

			    portgroup++;
			     
			    DPRINT(Debug,11,(&Debug,
					     "connect_remote_account_2: service list does not give portlist -- trying service type enumeration  => portgroup %d\n",
					     portgroup));

			    if (est) {
				
				while (connect_none == r || connect_in_progress == r) {

				    if (bglist)
					r = bgconnect_got_connection(bglist,ra,&cur_socket,
								     &this_port,
								     &this_type,
								     &last_error,can,
								     bgwait_space_or_completed  /* XXXX perhaps other? */,
								     &addrstring,
								     &connect_mode_addrdelay,portgroup,
								     &connect_mode_portdelay);

				    if (connect_none != r) {
					DPRINT(Debug,12,(&Debug,
							 "connect_remote_account_2: got background connection, %s\n",
							 addrstring ? addrstring : "--"));
					
				    } else if (is_canceled(can)) {
					DPRINT(Debug,12,(&Debug,
							 "connect_remote_account_2: Connection canceled, quitting\n"));
					break;
				    } else {
					const struct service_type * st =
					    enumerate_service_type(est);
					PORTS port;

					if (!st) {
					    DPRINT(Debug,12,(&Debug,
							     "connect_remote_account_2: End of serice enumeration\n"));
					}

					
					/* bgconnect_got_connection resets addrstring */
					connect_copy_addrstring(&addrstring,s);

					port = service_type_defport(st);
					
					DPRINT(Debug,12,(&Debug,
							 "connect_remote_account_2: next %p %s port %d\n",
							 st,
							 service_type_name(st),
							 port));
					
					set_addr_port(ra->hostaddr.generic,port);
					
					r = connect_handle_tail(ra,s,can,port,portgroup,
								&cur_socket, bglist,
								&this_type,
								&this_port,&last_error,st);
				    }				    
				}

			    } else {
				DPRINT(Debug,11,(&Debug,
						 "connect_remote_account_2: No service type enumeration given\n"));
			    }
				
			}
		    }
			
		    was_canceled = is_canceled(can);
		    free_cancel(&can);
		}	       
		break;
#endif
	    default:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeUnsupportedAddrType,
				  "Name %s have odd type address"),
			  ra->hostname ? ra->hostname : initial_se->official_name);
		break;
	    }
	}

	if (connect_result_ok(initial_se,ra,
			      idx_enum_socket_address(&walker),
			      r,addrstring,this_port,cur_socket)) {
	    ok = 1;
	}
	
	if (addrstring)
	    free(addrstring);

	
	if (ok) {
	    DPRINT(Debug,20,(&Debug,
			     "connect_remote_account_2: Connection succeed -- quiting loop\n"));
	    break;
	}

	if (next_enum_socket_address(&walker,&next_display_idx)) {
	    DPRINT(Debug,12,(&Debug,
			     "connect_remote_account_2: continuing to next address #%d\n",
			     next_display_idx));
	}		
    }


    if (!ok && bglist) {
	
	DPRINT(Debug,12,(&Debug,
			 "connect_remote_account_2: was last address, waiting completion\n"));

	ok = connect_wait_bglist(ra,initial_se,bglist,bgcan,&last_error,&cur_socket,
				 &this_type, &this_port);		    
    }

    if (!ok) {

	connect_error_clean(initial_se,was_canceled,bgcan,last_error,&cur_socket);
	
	ok = 0;
	goto clean;
    }
    
    if (this_type) {

	DPRINT(Debug,12,(&Debug,
			 "connect_remote_account_2: type %p %s defport=%d\n",
			 this_type,service_type_name(this_type),
			 service_type_defport(this_type)));
	
	if (this_type == initial_se->service) {
	    DPRINT(Debug,12,(&Debug,"connect_remote_account_2: type is same than service entry type\n"));

	    this_se = initial_se;
	} else {
	    /* Increments refcount */
	    this_se = convert_service_entry(initial_se,this_type);
	}
	
    } else {
	DPRINT(Debug,12,(&Debug,
			 "connect_remote_account_2: type not set\n"));
	
	this_se = initial_se;
    }
	
    connect_set_socket(cur_socket);
    
    ra->stream = returnSimpleStream(cur_socket);
    ra->service = this_se;
    inc_service_entry_refcount(ra->service);
    if (result_se) {
	*result_se = this_se;
	inc_service_entry_refcount(*result_se);
    }

    if (got_type)
	*got_type = this_type;
    if (got_port)
	*got_port = this_port;
    
 clean:
    clear_enum_socket_address(&walker);
    
    if (bglist)
	free_bgconnect_list(&bglist);
    if (bgcan && bgcan != main_cancel)
	free_cancel(&bgcan);
    if (this_se && this_se != initial_se)
	free_service_entry(&this_se);
    
    DPRINT(Debug,12,(&Debug,
		     "connect_remote_account_2=%d\n",ok)); 

    return ok;
}



int connect_remote_account(ra,got,se,default_portlist,force_port,main_cancel)
     struct remote_account *ra;     
     int *got;
     struct service_entry *se;    
     const PORTS default_portlist[];
     const PORTS force_port;
     struct cancel_data *main_cancel;
{
    int ok = 0;
    int last_error   = 0;
    int cur_socket   = -1;
    int was_canceled = 0;

    struct bgconnect_list  * bglist = NULL;
    struct cancel_data     * bgcan = NULL;
    
    int portgroup = 0;  /* loop number, when scanning different ports on same address */

    struct enum_socket_address   walker = NULL_enum_socket_address;
    
    if (ra->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "connect_remote_account",
	      "Bad magic number",0);

    if ( SERVICE_ENTRY_magic != se->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "connect_remote_account",
	      "Bad magic (service_entry)",0);

    connect_set_bgcan(&bglist,&bgcan,se,main_cancel);
    
    connect_ra_cleanup(ra);
    
    DPRINT(Debug,12,(&Debug,"connect_remote_account: %s: address count %d\n",
		     se->official_name,
		     se->addr_count));

    for (init_enum_socket_address(&walker,se);
	 have_enum_socket_address(&walker);
	 ) {
	enum connect_result r = connect_none;
	char             * addrstring = NULL;
	int next_display_idx;

	connect_socket_reset(&cur_socket,
			     idx_enum_socket_address(&walker));
			     	
	if (bglist)
	    r = bgconnect_got_connection(bglist,ra,&cur_socket,got,NULL /* service type */,
					 &last_error,bgcan,
					 bgwait_space_or_completed,&addrstring,
					 &connect_mode_addrdelay,0,NULL);

	if (connect_none != r) {
	    DPRINT(Debug,12,(&Debug,
			     "connect_remote_account: got background connection, %s\n",
			     addrstring ? addrstring : "--"));
	} else if (bgcan &&  is_canceled(bgcan)) {
	    DPRINT(Debug,12,(&Debug,
			     "connect_remote_account: Connection canceled, quitting\n"));
	    break;
	} else {
	    int se_idx = advance_enum_socket_address(&walker);
	    int display_idx = idx_enum_socket_address(&walker);

	    if (se_idx < 0) {
		DPRINT(Debug,12,(&Debug,
				 "connect_remote_account: %d (count %d) => end of addresses\n",
				 display_idx,se->addr_count));
		continue;
	    }
	    
	    DPRINT(Debug,12,(&Debug,
			     "connect_remote_account: %d (count %d) => se_idx=%d\n",
			     display_idx,se->addr_count,se_idx));

	    /* Copy all information */
	    if (! connect_setup_address(ra,se,se_idx))
		continue;
	    	    	    
	    switch (ra->hostaddr.sa->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
	    case AF_INET: 
#ifdef AF_INET6
	    case AF_INET6: 	
#endif
		{
		    struct cancel_data     * can       = NULL;
		    SOCKADDR A;
		    int port;
		    const char *s;
		    char mybuffer[256];

		    if (! connect_copy_sockaddr(& A,ra,&addrstring,&port,
						mybuffer, sizeof mybuffer,
						&s))
			continue;
		    		    
		    /* r == 0: not succeed
		       r <  0: fatal error
		       r >  0: succeed
		    */
		    
		    if (force_port != PORT_end) {
			
			DPRINT(Debug,11,(&Debug,
					 "connect_remote_account:force_port = %d\n", force_port));
			
			if (PORT_end != port &&
			    port != force_port)  {
			    
			    DPRINT(Debug,11,(&Debug,
					     "connect_remote_account:   but address gives port %d\n",
					     port));
			    continue;
			}

			set_addr_port(ra->hostaddr.generic,force_port);

			r = connect_handle_port(ra,se,s,&can,force_port,
						display_idx,
						&cur_socket,
						bglist,
						NULL /* service type */,
						got,&last_error);
						 
		    } else if (PORT_end != port) {
		    
			DPRINT(Debug,11,(&Debug,"connect_remote_account: address record gives port %d\n",
					 port));

			r = connect_handle_port(ra,se,s,&can,port,
						display_idx,
						&cur_socket,
						bglist,
						NULL /* service type */,
						got,&last_error);

		    } else {

			connect_host_message(ra,se,s,&can,port);
						
			if (se->port_count > 0) {
			   						       		       
			    portgroup++;
			    
			    DPRINT(Debug,11,(&Debug,
					     "connect_remote_account: service list gives portlist (%d ports) => portgroup %d\n",
					     se->port_count,portgroup));


			    r = connect_handle_portlist(ra,se,s,can,
							se->port_count,se->port_list,
							portgroup,
							&addrstring,&cur_socket,
							bglist,NULL /* service type */,
							got,&last_error);
			    			    
			    /* Do not use default portlist if port was 
			       given by name */
			} else if (0 == (se->flags & SE_rescan_name_port)) {

			    int portcount = 0;
			    
			    portgroup++;
			     
			    DPRINT(Debug,11,(&Debug,
					     "connect_remote_account: service list does not give portlist -- trying default portlist  => portgroup %d\n",
					     portgroup));

			    for (portcount = 0; default_portlist[portcount] != PORT_end; portcount++)
				;

			    if (portcount > 0) {
				int                      * portlist =
				    safe_calloc(portcount,sizeof portlist[0]);
				int idx2;
						
				DPRINT(Debug,11,(&Debug,"connect_remote_account: portcount = %d, ports",portcount));

				for (idx2 = 0; idx2 < portcount; idx2++) {
				    portlist[idx2] = default_portlist[idx2];
				    DPRINT(Debug,11,(&Debug," %d",portlist[idx2]));
				}
				DPRINT(Debug,11,(&Debug,"\n"));

				r = connect_handle_portlist(ra,se,s,can,
							    portcount,portlist,
							    portgroup,
							    &addrstring,&cur_socket,
							    bglist,NULL /* service type */,
							    got,&last_error);

				free(portlist); portlist = NULL;
			    } else {
				DPRINT(Debug,11,(&Debug,"connect_remote_account: No default ports\n"));
			    }
			}
		    }	    
		    
		    was_canceled = is_canceled(can);
		    free_cancel(&can);
		}	       
		break;
#endif
	    default:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeUnsupportedAddrType,
				  "Name %s have odd type address"),
			  ra->hostname ? ra->hostname : se->official_name);
		break;
	    }
	}

	if (connect_result_ok(se,ra,
			      idx_enum_socket_address(&walker),
			      r,addrstring,*got,cur_socket)) {
	    ok = 1;
	}
	
	if (addrstring)
	    free(addrstring);
	
	if (ok) {
	    DPRINT(Debug,20,(&Debug,
			     "connect_remote_account: Connection succeed -- quiting loop\n"));
	    break;
	}

	if (next_enum_socket_address(&walker,&next_display_idx)) {
	    DPRINT(Debug,12,(&Debug,
			     "connect_remote_account: continuing to next address #%d\n",
			     next_display_idx));
	}
    }

    if (!ok && bglist) {
	
	DPRINT(Debug,12,(&Debug,
			 "connect_remote_account: was last address, waiting completion\n"));

	ok = connect_wait_bglist(ra,se,bglist,bgcan,&last_error,&cur_socket,
				 NULL /* service type */, got);
		    
    }

    if (!ok) {

	connect_error_clean(se,was_canceled,bgcan,last_error,&cur_socket);
	      
	ok = 0;
	goto clean;
    }

    connect_set_socket(cur_socket);
    
    ra->stream = returnSimpleStream(cur_socket);
    ra->service = se;
    inc_service_entry_refcount(ra->service);

 clean:
    clear_enum_socket_address(&walker);
    
    if (bglist)
	free_bgconnect_list(&bglist);
    if (bgcan && bgcan != main_cancel)
	free_cancel(&bgcan);
    
    DPRINT(Debug,12,(&Debug,
		     "connect_remote_account=%d\n",ok)); 
    return ok;
}

/* -1 == name not found or bad syntax
    0 == not a remote address
    1 == name found 
    2 == no lookup (se == NULL)
*/
int split_remote_name(name,X,se,rest,lookup_flags,cancel_p)
     const char *name;
     struct remote_account *X;
     struct service_entry **se;
     char **rest;
     int lookup_flags;
     struct cancel_data         ** cancel_p
     /* May be NULL, Used if dns lookup was cancelable */;

{
    char *sep;
    int ret = 0;
    
    DPRINT(Debug,12,(&Debug,
		     "split_remote_name: name=\"%s\", lookup_flags=%d\n",
		     name,lookup_flags));
    
    if (!se) {

	DPRINT(Debug,12,(&Debug,
			 "split_remote_name: lookup disabled\n"));
    } else {

	if (*se) {
	    if (SERVICE_ENTRY_magic != (*se)->magic)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "split_remote_name",
		      "Bad magic (service_entry)",0);
	    
	    free_service_entry(se);
	}
    }

    *rest = NULL;
    zero_remote_account(X);
    
    *rest = strpbrk(name,"/:");
    sep  = strchr(name,'@');

    if (sep && (!*rest || *rest > sep)) {
	char * sep2;
       
	if (sep == name || sep[1] == '\0' ||
	    *rest == sep+1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet,MeBadRemoteMailbox,
			      "Bad remote mailbox: %s"),
		      name);
	    ret = -1;
	    goto clean;
	}

	/* Test for '@' on hostname -- if there is, then
	   assume that first '@' was on username
	*/
	sep2 = strchr(sep+1,'@');
	if (sep2 && (!*rest || *rest > sep2+1)) {
	    DPRINT(Debug,12,(&Debug,
			     "split_remote_name: Found second '@' from hostname, Assuming that first '@' was on username\n"));
	    sep = sep2;
	}
	
	if (*rest) {
	    const char * START = sep+1;
	    X->host = safe_malloc(*rest - START+1);
	    memcpy(X->host,START,*rest - START);
	    X->host[*rest - START] = '\0';
	} else
	    X->host = safe_strdup(sep+1);	   

	X->username = safe_malloc(sep - name+1);
	memcpy(X->username,name,sep - name);
	X->username[sep - name] = '\0';

	if (se) {
	    if (0 == lookup_flags) {
		lookup_flags = *rest ? STFLAG_browser : STFLAG_mbox;
		
		DPRINT(Debug,12,(&Debug,
				 "split_remote_name: (%s) Using lookup_flags=%d\n",
				 name,lookup_flags));
	    }

	    *se = give_service_entry_can(X->host, lookup_flags,
					 itls_ignore,NULL,
					 cancel_p);

	    if (!*se) {
		if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		    DPRINT(Debug,12,(&Debug,
				     "split_remote_name: DNS lookup canceled (%s).\n",
				     X->host));
		} else {
		    DPRINT(Debug,12,(&Debug,
				     "split_remote_name: lookup failed (%s)\n",
				     X->host));
		}
		
		ret = -1;
		goto clean;
	    }
	    
	    /* Canonify host name ... */
	    X->host = strmcpy(X->host,
			      (*se)->official_name);

	    ret = 1;
	} else
	    ret = 2;


	DPRINT(Debug,12,(&Debug,
			 "split_remote_name: username=%s, host=%s\n",
			 X->username ? X->username : "<NULL>",
			 X->host ? X->host : "<NULL>"));


    } 
    
 clean:
    DPRINT(Debug,12,(&Debug,
		     "split_remote_name=%d: *rest=%s\n",
		     ret,
		     *rest ? *rest : "<NULL>"));

    return ret;
}		

static int remote_account_verify_cert P_((struct remote_account        * ra,
					  const struct string          * Server,
					  const struct string          * server));
static int remote_account_verify_cert(ra,Server,server)
     struct remote_account        * ra;
     const struct string          * Server;
     const struct string          * server;
{
    int ret = 1;
    int r = 0;
    
    StreamInfo(ra->stream,SS_verify,&r,NULL,NULL);
    
    if (r < 0) {
	
	lib_error(CATGETS(elm_msg_cat, MeSet, MeRAFailedVerifyCert,
			  "Failed to verify %S %s certificate (required)."),
		  server,ra->host);
	ret = 0;
	goto failure;
    } else if (0 == r) {
	int bits;
	
	StreamInfo(ra->stream,SS_ssf,&bits,NULL,NULL);
	
	if (bits > 0)
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeRANotVerifiedCert,  
			      "%S %s certificate not verified (required)."),
		      Server,ra->host);
	else
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeRANotEncryptedCert,
			      "%S %s connection not encrypted (certificate required)."),
		      Server,ra->host);
	ret = 0;
	goto failure;
    }

 failure:
    return ret;
}



/* 1 verify OK
   0 verify failed
  -1 no connection or other failure
*/

int remote_account_verify_peer(ra,data,cb,Server,server)
     struct remote_account        * ra;
     union ra_verify_peer_cb_data * data;
     ra_verify_peer_cb_f          * cb;
     const struct string          * Server;
     const struct string          * server;
{
    int ret = 1;
    int need_callback = 1;    
    int r;

    int cert_checked = 0;
    int name_checked = 0;
    
    if (REMOTE_ACCOUNT_magic != ra->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "remote_account_verify_peer",
	      "Bad magic number",0);

    if (! ra->stream) {
	DPRINT(Debug,8,(&Debug,
			"remote_account_verify_peer: No connection\n"));

	ret = -1;
	goto failure;
    }

    
    if (ra->service) {

	if (SERVICE_ENTRY_magic != ra->service->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "remote_account_verify_peer",
		  "Bad magic number (service_entry)",0);
    
	if (isoff(ra->service->flags,SE_ver_tls_cert) &&
	    isoff(ra->service->flags,SE_require_tls_name) &&
	    ra->service->req_tls_peername_count < 1) {

	    DPRINT(Debug,8,(&Debug,
			    "remote_account_verify_peer: No verify required per service entry.\n"));

	    goto global_checks;
	}

	if (need_callback) {
	    r = cb(ra,data,Server,server);
	    
	    if (r <= 0) {
		DPRINT(Debug,8,(&Debug,"remote_account_verify_peer: callback=%d\n",
				r));
		ret = r;
		goto failure;
	    }
	    need_callback = 0;
	}

	if (ison(ra->service->flags,SE_ver_tls_cert)) {

	    int r1 = remote_account_verify_cert(ra,Server,server);

	    if (r1 < 1) {
		ret = r1;
		goto failure;

	    } else
		cert_checked = 1;
	}

	if (ison(ra->service->flags,SE_require_tls_name) ||
	    ra->service->req_tls_peername_count > 0) {

	    int i;
	    int start = 0;

	    struct string * returned_name = NULL;
	    struct string * returned_host_name = NULL;
	    
	    struct string * official_name = NULL;
	    const struct string *require_name = NULL;


	    int found = 0;

	    if (ison(ra->service->flags,SE_require_tls_name)) {
		start = -1;
		official_name = new_string2(system_charset,
					    s2us(ra->service->
						 official_name));
	    }

	    for (i = start; i < ra->service->req_tls_peername_count; i++) {
		int r;
		int is_ip_literal = 0;
		
		if (i < 0) {
		    require_name = official_name;
		    is_ip_literal = NULL != ra->service->ip_literal_address.dummy;
		} else
		    require_name =  ra->service->req_tls_peername_list[i];

		r = StreamVerifyName(ra->stream,SS_peer_cn,require_name,
				     & returned_name);

		DPRINT(Debug,8,(&Debug,
				"remote_account_verify_peer: %d: StreamVerifyName SS_peer_cn require_name=%S",
				i,require_name));
		if (returned_name) {
		    DPRINT(Debug,8,(&Debug, " returned_name=%S",
				    returned_name));
		}

		
		if (r) {
		    DPRINT(Debug,8,(&Debug, " -- succeed (return %d)\n",r));
		    
		    found = 1;
		    name_checked = 1;
		    break;
		}
		DPRINT(Debug,8,(&Debug, " -- failed\n",r));


		if (is_ip_literal) {
		    DPRINT(Debug,8,(&Debug,
				    "remote_account_verify_peer: %d: %S marked as ip literal -- no SS_check_host\n",
				    i,require_name));

		    /* XXXX Need SS_check_ip */
		     

		} else {

		    r = StreamVerifyName(ra->stream,SS_check_host,require_name,
					 & returned_host_name);
		    DPRINT(Debug,8,(&Debug,
				    "remote_account_verify_peer: %d: StreamVerifyName SS_check_host require_name=%S",
				    i,require_name));
		    if (returned_host_name) {
			DPRINT(Debug,8,(&Debug, " returned_host_namename=%S",
					returned_host_name));
		    }
		    if (r) {
			DPRINT(Debug,8,(&Debug, " -- succeed (return %d)\n",r));
		    
			found = 1;
			name_checked = 1;
			break;
		    }
		    DPRINT(Debug,8,(&Debug, " -- failed\n",r));
		}
		
	    }
	 
	    if (!found) {
		ret = 0;

		if (start < ra->service->req_tls_peername_count -1 ||
		    !require_name) {
		    /* Several possible names */

		    if (returned_name) {
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeRACertCNWrongOther,
					  "%S %s certificate CN is %S. Other certificate is required."),
				  Server,ra->host,returned_name);
		    } else {
			int bits;
		
			StreamInfo(ra->stream,SS_ssf,&bits,NULL,NULL);

			if (bits > 0)
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeRACertCNWrongCert,
					      "%S %s certificate CN is wrong."),
				      Server,ra->host);
			else
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeRANotEncryptedCN,
					      "%S %s connection not encrypted, certificate required."),
				      Server,ra->host);
		    }

		} else {
		    if (returned_name) {
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeRACertCNWrongRequired,
					  "%S %s certificate CN is %S, but %S required."),
				  Server,ra->host,returned_name,
				  require_name);
		    } else {
			int bits;
			
			StreamInfo(ra->stream,SS_ssf,&bits,NULL,NULL);

			if (bits > 0)
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeRACertCNRequired,
					      "%S %s certificate CN %S required."),
				      Server,ra->host,require_name);
			else
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeRANotEncryptedRequired,
					      "%S %s connection not encrypted, but certificate CN %S required."),
				      Server,ra->host,require_name);
		    }

		}
	    }

	    if (returned_host_name)
		free_string(&returned_host_name);	    
	    if (returned_name)			       
		free_string(&returned_name);
	    if (official_name)
		free_string(&official_name);

	    if (!ret)
		goto failure;
	    	    
	}

    global_checks:
	if (ison(ra->service->flags,SE_nover_tls_cert) &&
	    ! cert_checked) {
	    DPRINT(Debug,8,(&Debug,
			    "remote_account_verify_peer: verify-tls-certificate=off given\n"));
	    cert_checked = -1; /* Disabled */
	}

	
	if (ison(ra->service->flags,SE_nouse_tls_checks)) {  /* Not implemented */
	    DPRINT(Debug,8,(&Debug,
			    "remote_account_verify_peer: use-tls-checks=no given\n"));

	    goto failure;
	}
	
    }


    if (dt_flag_is_set(&use_tls,use_tls_verify_cert) ||
	dt_flag_is_set(&use_tls,use_tls_require_name)) {

	if (need_callback) {
	    r = cb(ra,data,Server,server);
	    
	    if (r <= 0) {
		DPRINT(Debug,8,(&Debug,"remote_account_verify_peer: callback=%d\n",
				r));
		ret = r;
		goto failure;
	    }
	    need_callback = 0;
	}

	if (dt_flag_is_set(&use_tls,use_tls_verify_cert) &&
	    ! cert_checked
	    ) {
	    int r1 = remote_account_verify_cert(ra,Server,server);
	    
	    if (r1 < 1) {
		ret = r1;
		goto failure;
		
	    }	    
	}

	if (dt_flag_is_set(&use_tls,use_tls_require_name) &&
	    ! name_checked
	    ) {
	    struct string * host_name = new_string2(system_charset,s2us(ra->host));
	    struct string * returned_name = NULL;
	    struct string * returned_host_name = NULL;
	    
	    int r =  StreamVerifyName(ra->stream,SS_peer_cn,host_name,
				      & returned_name);

	    DPRINT(Debug,8,(&Debug,
			    "remote_account_verify_peer: StreamVerifyName SS_peer_cn host_name=%S",
			    host_name));

	    if (returned_name) {
		DPRINT(Debug,8,(&Debug, " returned_name=%S",
				returned_name));
	    }
	    
	    
	    if (r) {
		DPRINT(Debug,8,(&Debug, " -- succeed (return %d)\n",r));
		    
	    } else {
		SOCKADDR ip_literal;

		DPRINT(Debug,8,(&Debug, " -- failed\n",r));
		
		if (get_ip(&ip_literal,ra->host)) {
		    DPRINT(Debug,8,(&Debug,
				    "remote_account_verify_peer: %s is ip literal -- no SS_check_host\n",
				    ra->host));
		    
		    /* XXXX Need SS_check_ip */
		    
		} else {
		     r = StreamVerifyName(ra->stream,SS_check_host,host_name,
					  & returned_host_name);

		     DPRINT(Debug,8,(&Debug,
				     "remote_account_verify_peer: StreamVerifyName SS_check_host host_name=%S",
				     host_name));

		     if (returned_host_name) {
			 DPRINT(Debug,8,(&Debug, " returned_host_namename=%S",
					 returned_host_name));
		     }
		     
		     if (r) {
			 DPRINT(Debug,8,(&Debug, " -- succeed (return %d)\n",r));
		     } else {
			 DPRINT(Debug,8,(&Debug, " -- failed\n",r));
		     }
		}		
	    }
	    
	    if (! r) {
		if (returned_name) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeRACertCNWrongRequired,
				      "%S %s certificate CN is %S, but %S required."),
			      Server,ra->host,returned_name,
			      host_name);
		} else {
		    int bits;
		    
		    StreamInfo(ra->stream,SS_ssf,&bits,NULL,NULL);
		    
		    if (bits > 0)
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeRACertCNRequired,
					  "%S %s certificate CN %S required."),
				  Server,ra->host,host_name);
		    else
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeRANotEncryptedRequired,
					  "%S %s connection not encrypted, but certificate CN %S required."),
				  Server,ra->host,host_name);
		}	    		
	    }

	    if (returned_host_name)
		free_string(&returned_host_name);	    
	    if (returned_name)			       
		    free_string(&returned_name);
	    if (host_name)
		free_string(&host_name);

	    if (!r) {
		ret = 0;
		goto failure;
	    }	    	    
	}
    }


    
 failure:
    DPRINT(Debug,8,(&Debug,"remote_account_verify_peer=%d%s\n",
		    ret,
		    ret > 0 ? " (succeed)" : ""));

    return ret;
}

enum itls_status remote_account_init_tls(ra) 
     struct remote_account *ra;
{
    enum itls_status ret = itls_none;

    if (REMOTE_ACCOUNT_magic != ra->magic) {
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "remote_account_init_tls",
	      "Bad magic (remote_account)",0);
    }

    if (ra->service) {

	const char * init_tls = NULL;

	if (SERVICE_ENTRY_magic != ra->service->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "remote_account_init_tls",
		  "Bad magic (service_entry)",0);

	init_tls =  have_init_tls(ra->service);

	if (init_tls) {
#ifdef USE_DLOPEN
	    ret = initial_TLS_ra(ra,init_tls);
#else
	    DPRINT(Debug,9,(&Debug,
			    "remote_account_init_tls:  no shared routines available\n"));
	    ret = itls_unsupported;
#endif

	    if (itls_unsupported == ret) {
		lib_error(CATGETS(elm_msg_cat, MeSet,MeConTLSUnsupported,
				  "%s connection to %s does not support SSL/TLS."),
			  init_tls,ra->host);

	    }
	}
    }

    
    return ret;
}

#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
static int same_sockaddr_in P_((struct sockaddr_in * addr1,
				struct sockaddr_in * addr2,
				int                      addr_only /* ignore port */));
static int same_sockaddr_in(addr1,addr2,addr_only)
     struct sockaddr_in * addr1;
     struct sockaddr_in * addr2;
     int                      addr_only /* ignore port */;
{
    if (AF_INET != addr1->sin_family)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "same_sockaddr_in",
	      "Bad sin_family (addr1)",0);	
    if (AF_INET != addr2->sin_family)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "same_sockaddr_in",
	      "Bad sin_family (addr2)",0);

    if (0 != memcmp(& (addr1->sin_addr),
		    & (addr2->sin_addr),
		    sizeof addr1->sin_addr)) {
	DPRINT(Debug,16,(&Debug,
			 "same_sockaddr_in=0; addr1->sin_addr != addr2->sin_addr\n"));
	return 0;
    }
    if (addr1->sin_port != addr2->sin_port) {
	if (addr_only) {
	    DPRINT(Debug,17,(&Debug,
			     "same_sockaddr_in: addr1->sin_port != addr2->sin_port (ignored)\n"));
	} else {
	    DPRINT(Debug,16,(&Debug,
			     "same_sockaddr_in=0; addr1->sin_port != addr2->sin_port\n"));
	    return 0;
	}
    }

    return 1;
}

#ifdef HAVE_IN6
static int same_sockaddr_in6 P_((struct sockaddr_in6 * addr1,
				 struct sockaddr_in6 * addr2,
				 int                  addr_only /* ignore port */));
static int same_sockaddr_in6(addr1,addr2,addr_only)
     struct sockaddr_in6 * addr1;
     struct sockaddr_in6 * addr2;
     int                      addr_only /* ignore port */;
{
    if (AF_INET6 != addr1->sin6_family)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "same_sockaddr_in6",
	      "Bad sin6_family (addr1)",0);	
    if (AF_INET6 != addr2->sin6_family)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "same_sockaddr_in6",
	      "Bad sin6_family (addr2)",0);

    if (0 != memcmp(& (addr1->sin6_addr),
		    & (addr2->sin6_addr),
		    sizeof addr1->sin6_addr)) {
	DPRINT(Debug,16,(&Debug,
			 "same_sockaddr_in6=0; addr1->sin6_addr != addr2->sin6_addr\n"));
	return 0;
    }
    if (addr1->sin6_port != addr2->sin6_port) {
	if (addr_only) {
	    DPRINT(Debug,17,(&Debug,
			     "same_sockaddr_in: addr1->sin6_port != addr2->sin6_port (ignored)\n"));

	} else {
	    DPRINT(Debug,16,(&Debug,
			     "same_sockaddr_in=0; addr1->sin6_port != addr2->sin6_port\n"));
	    return 0;
	}	    
    }

#ifdef HAVE_SCOPE
    if (addr1->sin6_scope_id != addr2->sin6_scope_id) {
	DPRINT(Debug,16,(&Debug,
			 "same_sockaddr_in=0; addr1->sin6_scope_id != addr2->sin6_scope_id\n"));
	return 0;
    }

    if (addr1->sin6_flowinfo != addr2->sin6_flowinfo) {
	if (addr_only) {
	    DPRINT(Debug,17,(&Debug,
			     "same_sockaddr_in: addr1->sin6_flowinfo != addr2->sin6_flowinfo (ignored)\n"));

	} else {
	    DPRINT(Debug,16,(&Debug,
			     "same_sockaddr_in=0; addr1->sin6_flowinfo != addr2->sin6_flowinfo\n"));
	    return 0;
	}
    }    
#endif

    return 1;
}
#endif
#endif

static int same_SOCKADDR_ptr_sa P_((const union SOCKADDR_ptr addr,
				    size_t                   addrsize,
				    const struct sockaddr         * sa))
    UNUSED_FUNCOK;
static int same_SOCKADDR_ptr_sa(addr,addrsize,sa)
     const union SOCKADDR_ptr addr;
     size_t                   addrsize;
     const struct sockaddr  * sa;
{
    size_t size;

    switch (sa->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
    case AF_INET:  size = sizeof (struct sockaddr_in); break;
#ifdef HAVE_IN6
    case AF_INET6: size = sizeof (struct sockaddr_in6); break;
#endif
#endif
    default:       size = sizeof (struct sockaddr);     break;
    }

    if (addr.sa->sa_family != sa->sa_family) {
	DPRINT(Debug,11,(&Debug,
			 "same_SOCKADDR_ptr_sa=0; addr family %d, sa family %d\n",
			 addr.sa->sa_family,sa->sa_family));
	return 0;
    }
    
    if (size == addrsize &&
	0 == memcmp(sa,addr.dummy,size))
	return 1;

      switch (sa->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
      case AF_INET:
	  if (same_sockaddr_in(addr.sin,
			       (struct sockaddr_in  *)sa,
			       0 /* check port */)) {
	      DPRINT(Debug,11,(&Debug,
			       "same_SOCKADDR_ptr_sa=1; same_sockaddr_in matches; size=%zu does not match\n",
			       size));
	      return 1;
	  }
	  break;
#ifdef HAVE_IN6
      case AF_INET6:
	  if (same_sockaddr_in6(addr.sin6,
				(struct sockaddr_in6  *)sa,
				0 /* check port */)) {
	      DPRINT(Debug,11,(&Debug,
			       "same_SOCKADDR_ptr_sa=1; same_sockaddr_in6 matches; size=%zu does not match\n",
			       size));
	      return 1;
	  } 
	  break;
#endif
#endif
      default:
	  DPRINT(Debug,11,(&Debug,
			   "same_SOCKADDR_ptr_sa: Unknown family %d\n",
			   sa->sa_family));
	  break;
      }			

    return 0;
}

int same_SOCKADDR_ptr_generic(addr,addrsize,generic)
     const union SOCKADDR_ptr addr;
     size_t                   addrsize;
     const SOCKADDR         * generic;
{
    return same_SOCKADDR_ptr_sa(addr,addrsize,& (generic->sa));
}

int same_SOCKADDR_ptr(addr1,addr1size,addr2,addr2size,addr_only)
     const union SOCKADDR_ptr addr1;
     size_t                   addr1size;
     const union SOCKADDR_ptr addr2;
     size_t                   addr2size;
     int                      addr_only /* ignore port */;
{
    size_t size;
    
    if (addr1size <  sizeof (addr1.sa->sa_family)) {
	DPRINT(Debug,11,(&Debug,
			 "same_SOCKADDR_ptr=0; addr1size=%zu\n",
			 addr1size));

	return 0;
    }

    if (addr2size <  sizeof (addr2.sa->sa_family)) {
	DPRINT(Debug,11,(&Debug,
			 "same_SOCKADDR_ptr=0; addr2size=%zu\n",
			 addr1size));

	return 0;
    }

    if (addr1.sa->sa_family != addr2.sa->sa_family) {
	DPRINT(Debug,11,(&Debug,
			 "same_SOCKADDR_ptr=0; addr1 family %d, addr family %d\n",
			 addr1.sa->sa_family,
			 addr2.sa->sa_family));
	return 0;
    }
    
    switch (addr1.sa->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
    case AF_INET:  size = sizeof (struct sockaddr_in); break;
#ifdef HAVE_IN6
    case AF_INET6: size = sizeof (struct sockaddr_in6); break;
#endif
#endif
    default:       size = sizeof (struct sockaddr);     break;
    }

    if (addr1size <  size) {
	DPRINT(Debug,11,(&Debug,
			 "same_SOCKADDR_ptr=0; addr1size=%zu\n",
			 addr1size));

	return 0;

    }
    
    if (addr2size < size) {
	DPRINT(Debug,11,(&Debug,
			 "same_SOCKADDR_ptr=0; addr2size=%zu\n",
			 addr2size));

	return 0;
    }

    if (addr1size == addr2size &&
	0 == memcmp(addr1.dummy,addr2.dummy,addr1size))
	return 1;

    if (0 == memcmp(addr1.dummy,addr2.dummy,size)) {
	DPRINT(Debug,11,(&Debug,
			 "same_SOCKADDR_ptr=1; addr1size=%zu, addr2size=%zu; size %zu matches (family %d)\n",
			 addr1size,addr2size,
			 size,addr1.sa->sa_family));
			 
	return 1;
    }

    switch (addr1.sa->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
      case AF_INET:
	  if (same_sockaddr_in(addr1.sin,addr2.sin,addr_only)) {
	      DPRINT(Debug,addr_only ? 16 : 11,
		     (&Debug, 
		      "same_SOCKADDR_ptr=1; same_sockaddr_in matches; size=%zu does not match, addr_only=%d\n",
		      size,addr_only));
	      return 1;
	  }
	  break;
#ifdef HAVE_IN6
      case AF_INET6:
	  if (same_sockaddr_in6(addr1.sin6,addr2.sin6,addr_only)) {
	      DPRINT(Debug,addr_only ? 16 : 11,
		     (&Debug,
		      "same_SOCKADDR_ptr=1; same_sockaddr_in6 matches; size=%zu does not match, addr_only=%d\n",
		      size,addr_only));
	      return 1;
	  } 
	  break;
#endif
#endif
      default:
	  DPRINT(Debug,11,(&Debug,
			   "same_SOCKADDR_ptr: Unknown family %d\n",
			   addr1.sa->sa_family));
	  break;
      }			
    
    
    DPRINT(Debug,16,(&Debug,
		     "same_SOCKADDR_ptr=0; addr1size=%zu, addr2size=%zu; family size %zu, family %d\n",
		     addr1size,addr2size,
		     size,addr1.sa->sa_family));
    return 0;
}


static void set_SOCKADDR_ptr_sa P_((union SOCKADDR_ptr * target,
				     size_t             * target_addrsize,
				     struct sockaddr    * sa))
     UNUSED_FUNCOK;
static void set_SOCKADDR_ptr_sa(target,target_addrsize,sa)
     union SOCKADDR_ptr * target;
     size_t             * target_addrsize;
     struct sockaddr    * sa;
{
    switch (sa->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
    case AF_INET:
	target->sin = safe_realloc(target->sin,
				   sizeof(* (target->sin)));
	memcpy(target->sin,sa,sizeof(* (target->sin)));
	*target_addrsize = sizeof(* (target->sin));
	break;
#ifdef HAVE_IN6
    case AF_INET6:
	target->sin6 = safe_realloc(target->sin6,
				    sizeof (* (target->sin6)));
	memcpy(target->sin6,sa,sizeof (* (target->sin6)));
	*target_addrsize = sizeof (* (target->sin6));
	break;
#endif
#endif
    default:   /* This is too small for INET6 family, but it is handled
		  separately! */
	DPRINT(Debug,11,(&Debug,
			 "set_SOCKADDR_ptr_sa: Unknown address family %d\n",
			 sa->sa_family));

	target->sa = safe_realloc(target->sa,
				  sizeof (* (target->sa)));
	*(target->sa) = *sa;
	*target_addrsize = sizeof (* (target->sa));
	break;
    }
}


static int update_ilist_name P_((const char *name,
				 int * added_if))
     UNUSED_FUNCOK;
static int update_ilist_name(name,added_if)
     const char *name;
     int * added_if;
{
    int loc;

    DPRINT(Debug,10,(&Debug,"update_ilist_name: searching %s\n",
		     name));

    for (loc = 0; loc < interface_count; loc++) {
	if (interface_list[loc]) {
	    if (INTERFACE_INFO_magic != interface_list[loc]->magic)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "update_ilist_name",
		      "Bad magic number (interface_info)",0);	
	    
	    if (0 == strcmp(interface_list[loc]->name,
			    name)) {
		DPRINT(Debug,10,(&Debug,
				 "update_ilist_name: found #%d: %s\n",
				 loc,interface_list[loc]->name));
		goto found_interface;
	    }	    
	}
    }
    
    if (loc != interface_count)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "update_ilist_name",
	      "bad index (interface_info)",0);

    interface_list = safe_array_realloc(interface_list,
					interface_count+1,
					sizeof(interface_list[0]));

    interface_list[loc] = safe_zero_alloc(sizeof (* (interface_list[loc])));
    interface_list[loc]->magic = INTERFACE_INFO_magic;
    interface_list[loc]->name  = safe_strdup(name);

    interface_list[loc]->addr_list  = NULL;
    interface_list[loc]->addr_count = 0;

    interface_list[loc]->if_index = 0;

    interface_list[loc]->valid_addrlist = 0;
    interface_list[loc]->valid_index    = 0;


    DPRINT(Debug,10,(&Debug,
		     "update_ilist_name: added #%d: %s\n",
		     loc,interface_list[loc]->name));

    interface_count++;
    if (added_if)
	(* added_if)++;
       
 found_interface:

    return loc;
}


#ifdef I_NET_IF
#include <net/if.h>

/* https://www.freebsd.org/cgi/man.cgi?getifaddrs
 *
 * If both <net/if.h> and <ifaddrs.h> are being included, <net/if.h> must be
 * included before <ifaddrs.h>.
 */
#endif

#ifdef HAVE_IFADDRS
#include <ifaddrs.h>


static void update_ilist_ifaddr P_((struct ifaddrs *ifaddr,
				    int * added_if,
				    int * added_ad,
				    int * changed_flags
				    ));
static void update_ilist_ifaddr(ifaddr,added_if,added_ad,changed_flags)
     struct ifaddrs * ifaddr;
     int            * added_if;
     int            * added_ad;
     int            * changed_flags;
{
    int loc = update_ilist_name(ifaddr->ifa_name,added_if);
    int fchanged   = 0;
    
    interface_list[loc]->valid_addrlist = 1;

     DPRINT(Debug,12,(&Debug,
		      "update_ilist_ifaddr: #%d: ",loc));

     if (! ifaddr->ifa_addr) {
	 /* ifa_addr may contain NULL pointer */
	 DPRINT(Debug,12,(&Debug,"no address\n"));
	 return;
     }
     DPRINT(Debug,12,(&Debug,"address family %d\n",
		      ifaddr->ifa_addr->sa_family));
     
    switch (ifaddr->ifa_addr->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
	int aloc;
    case AF_INET: 
#ifdef AF_INET6
    case AF_INET6:
#endif
	
	for (aloc = 0; aloc < interface_list[loc]->addr_count; aloc++) {
	    if (interface_list[loc]->addr_list[aloc]) {
		if (INTERFACE_ADDR_magic !=
		    interface_list[loc]->addr_list[aloc]->magic)
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "update_ilist_ifaddr",
			  "Bad magic number (interface_addr)",
			  0);
		
		if (same_SOCKADDR_ptr_sa(interface_list[loc]->
					 addr_list[aloc]->address,
					 interface_list[loc]->
					 addr_list[aloc]->addrsize,
					 ifaddr->ifa_addr)) {

		    char mybuffer[256];		    
		    const char * s UNUSED_VAROK
			= give_SOCKADDR_ptr_as_string(interface_list[loc]->
						      addr_list[aloc]->address,
						      interface_list[loc]->
						      addr_list[aloc]->addrsize,
						      mybuffer,sizeof mybuffer);		    

		    DPRINT(Debug,10,(&Debug,
				     "update_ilist_ifaddr: #%d: found addr #%d: %s\n",
				     loc,aloc,
				     s ? s : "???"));
		    
		    goto found_address;
		}		
	    }
	}
	
	if (aloc != interface_list[loc]->addr_count)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "update_ilist_ifaddr",
		  "bad index (interface_addr)",0);
	
	interface_list[loc]->addr_list = 
	    safe_array_realloc(interface_list[loc]->addr_list,
			       interface_list[loc]->addr_count+1,
			       sizeof(interface_list[loc]->addr_list[0]));
	
	interface_list[loc]->addr_list[aloc] =
	    safe_zero_alloc(sizeof (* (interface_list[loc]->addr_list[aloc])));
	interface_list[loc]->addr_list[aloc]->magic = INTERFACE_ADDR_magic;
    
	interface_list[loc]->addr_list[aloc]->address.dummy    = NULL;
	interface_list[loc]->addr_list[aloc]->addrsize = 0;

	interface_list[loc]->addr_list[aloc]->interface_up  = 0;
	interface_list[loc]->addr_list[aloc]->interface_loopback = 0;
	interface_list[loc]->addr_list[aloc]->valid_address = 0;
	
	set_SOCKADDR_ptr_sa(& (interface_list[loc]->addr_list[aloc]->
			       address),
			    & (interface_list[loc]->addr_list[aloc]->
			       addrsize),
			    ifaddr->ifa_addr);

	{
	    char mybuffer[256];		    
	    const char * s UNUSED_VAROK
		= give_SOCKADDR_ptr_as_string(interface_list[loc]->
					      addr_list[aloc]->address,
					      interface_list[loc]->
					      addr_list[aloc]->addrsize,
					      mybuffer,sizeof mybuffer);

	    DPRINT(Debug,10,(&Debug,
			     "update_ilist_ifaddr: #%d: added addr #%d: %s\n",
			     loc,aloc,
			     s ? s : "???"));

	}

	interface_list[loc]->addr_count++;
	if (added_ad)
	    (* added_ad)++;

 found_address:
	interface_list[loc]->addr_list[aloc]->valid_address = 1;

	{
	    int isup       = 1;     /* If IFF_UP is not defined, 
				       assume that interface is UP
				    */
	    
#ifdef IFF_UP
	    isup = ifaddr->ifa_flags & IFF_UP;
#endif
	
	    if (isup && ! interface_list[loc]->addr_list[aloc]->interface_up) {
		DPRINT(Debug,10,(&Debug,
				 "update_ilist_ifaddr: #%d: addr #%d: Setting interface_up\n",
				 loc,aloc));
		interface_list[loc]->addr_list[aloc]->interface_up = 1;
		fchanged = 1;
	    } else if (!isup && interface_list[loc]->addr_list[aloc]->interface_up) {
		DPRINT(Debug,10,(&Debug,
				 "update_ilist_ifaddr: #%d: addr #%d: Clearing interface_up\n",
				 loc,aloc));
		interface_list[loc]->addr_list[aloc]->interface_up = 0;
		fchanged = 1;
	    }
	}
	
	{
	    int isloopback = 0;

#ifdef IFF_LOOPBACK
	    isloopback = ifaddr->ifa_flags & IFF_LOOPBACK;

#endif

	    if (isloopback &&
		! interface_list[loc]->addr_list[aloc]->interface_loopback) {
		DPRINT(Debug,10,(&Debug,
				 "update_ilist_ifaddr: #%d: addr #%d: Setting interface_loopback\n",
				 loc,aloc));
		interface_list[loc]->addr_list[aloc]->interface_loopback = 1;
		fchanged = 1;
	    } else if (!isloopback &&
		       interface_list[loc]->addr_list[aloc]->interface_loopback) {
		DPRINT(Debug,10,(&Debug,
				 "update_ilist_ifaddr: #%d: addr #%d: Clearing interface_loopback\n",
				 loc,aloc));
		interface_list[loc]->addr_list[aloc]->interface_loopback = 0;
		fchanged = 1;
	    }			    
	}

	if (fchanged && changed_flags)
	    (*changed_flags)++;
	    	
#endif
	break;
    }		   
}
#endif

#ifdef HAVE_NAMEINDEX
static void update_ilist_index P_((struct if_nameindex *ifidx,
				   int * added_if,
				   int * added_id));
static void update_ilist_index(ifidx,added_if, added_id)
     struct if_nameindex *ifidx;
     int * added_if;
     int * added_id;
{
    int loc = update_ilist_name(ifidx->if_name,added_if);

    if (! (interface_list[loc]->if_index) &&
	added_id)
	(* added_id)++;

    interface_list[loc]->if_index    = ifidx->if_index;
    interface_list[loc]->valid_index = 1;

    DPRINT(Debug,10,(&Debug,
		     "update_ilist_ifaddr: #%d: interface index %d\n",
		     loc,interface_list[loc]->if_index));
}

#endif


static void free_interface_addr P_((struct interface_addr **addr));
static void free_interface_addr(addr)
     struct interface_addr **addr;
{
    if (INTERFACE_ADDR_magic != (*addr)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_interface_addr",
	      "Bad magic number (interface_addr)",
	      0);

    if ((*addr)->address.dummy) {
	free((*addr)->address.dummy);
	(*addr)->address.dummy = NULL;
    }

    (*addr)->addrsize = 0;
    (*addr)->interface_up  = 0;
    (*addr)->interface_loopback = 0;
    (*addr)->valid_address = 0;

    (*addr)->magic = 0;  /* Invalidate */

    free(*addr);
    *addr = NULL;
}

static void free_interface_info P_((struct interface_info **info));
static void free_interface_info(info)
     struct interface_info **info;
{
    if (INTERFACE_INFO_magic != (*info)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_interface_info",
	      "Bad magic number (interface_info)",0);

    if ((*info)->name) {
	free((*info)->name);
	(*info)->name = NULL;
    }
	
    if ((*info)->addr_list) {
	int j;
	
	for (j = 0; j < (*info)->addr_count; j++) {
	    if ((*info)->addr_list[j]) 
		free_interface_addr(& ((*info)->addr_list[j]));	    
	}
	
	free((*info)->addr_list);
	(*info)->addr_list = NULL;
    }
    (*info)->addr_count = 0;	    

    (*info)->magic = 0;  /* Invalidate */
    free (*info);
    *info = NULL;
}

void free_interface_list() 
{
    if (interface_list) {

	int i;

	for (i = 0; i < interface_count; i++) {
	    if (interface_list[i]) 
		free_interface_info(& (interface_list[i]));
	}

	free(interface_list);
	interface_list = NULL;
    }

    interface_count = 0;
    interface_flags = 0;    
}


int update_interface_list(flags, save_errno)
     int flags;
     int *save_errno;
{
    int status = 1;
    int added_if     = 0;
    int removed_if   = 0;
    int added_ad     = 0;
    int removed_ad   = 0;
    int added_id     = 0;
    int removed_id   = 0;
    int changed_flags   = 0;

    if (interface_list) {
	int i;

	DPRINT(Debug,10,(&Debug,
			 "update_interface_list: start with %d interfaces\n",
			 interface_count));

	for (i = 0; i < interface_count; i++) {
	    if (interface_list[i]) {
		if (INTERFACE_INFO_magic != interface_list[i]->magic)
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "update_interface_list",
			  "Bad magic number (interface_info)",0);

		if (ison(flags,INTERFACE_ADDRLIST)) {

		    interface_list[i]->valid_addrlist = 0;

		    if (interface_list[i]->addr_list) {

			int j;

			for (j = 0; j < interface_list[i]->addr_count;
			     j++) {

			    if (interface_list[i]->addr_list[j]) {
				if (INTERFACE_ADDR_magic !=
				    interface_list[i]->addr_list[j]->magic)
				    panic("CONNECTION PANIC",__FILE__,__LINE__,
					  "update_interface_list",
					  "Bad magic number (interface_addr)",
					  0);

				interface_list[i]->addr_list[j]->
				    valid_address = 0;

			    }
			    
			}
		    }
		}

		if (ison(flags,INTERFACE_INDEX))
		    interface_list[i]->valid_index = 0;
	    }
	}
    }
    

    if (ison(flags,INTERFACE_ADDRLIST)) {
#ifdef HAVE_IFADDRS
	struct ifaddrs * head = NULL;

	if (0 == getifaddrs(&head)) {

	    DPRINT(Debug,10,(&Debug,
			     "update_interface_list: getifaddrs succeed.\n"));

	    interface_flags |= INTERFACE_ADDRLIST;

	    if (head) {
		struct ifaddrs * walk;

		for (walk = head; walk; walk = walk->ifa_next) {
		    update_ilist_ifaddr(walk,&added_if,&added_ad,
					&changed_flags);
		}
	    } else {
		DPRINT(Debug,10,(&Debug,
				 "update_interface_list: getifaddrs returned no interfaces\n"));
	    }

	} else {
	    int err = errno;

	    if (save_errno)
		*save_errno = err;

	    DPRINT(Debug,10,(&Debug,
			     "update_interface_list: getifaddrs failed: errno=%d (%s)\n",
			     err,strerror(err)));
		   
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeGetIfaddsFailed,
			      "getifaddrs failed: %s"),
		      strerror(err));

	    status = 0;
	}

	if (head)
	    freeifaddrs(head);
#else
	DPRINT(Debug,10,(&Debug,
			"update_interface_list: getifaddrs() is not supported\n"));
	status = 0;
#endif
    }

    if (ison(flags,INTERFACE_INDEX)) {
#ifdef HAVE_NAMEINDEX
	struct if_nameindex *names = if_nameindex();

	if (names) {	    
	    int i;

	    DPRINT(Debug,10,(&Debug,
			     "update_interface_list: if_nameindex succeed.\n"));

	    interface_flags |= INTERFACE_INDEX;

	    for (i = 0; names[i].if_name; i++) {

		update_ilist_index(&names[i],&added_if,&added_id);

	    }

	    if_freenameindex(names);
	} else {
	    int err = errno;
	    
	    if (save_errno)
		*save_errno = err;	    

	    DPRINT(Debug,10,(&Debug,
			     "update_interface_list: if_nameindex failed: errno=%d (%s)\n",
			     err,strerror(err)));
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeGetINameIndexFailed,
			      "if_nameindex failed: %s"),
		      strerror(err));

	    status = 0;	    
	}
#else
	DPRINT(Debug,10,(&Debug,
			 "update_interface_list: if_nameindex()() is not supported\n"));
	status = 0;
#endif
    }


    if (status && interface_list) {
	int trg = 0;
	int i;

	/* Do cleanup */	
	for (i = 0; i < interface_count; i++) {

	    if (interface_list[i]) {
		int clean_all = 0;

		if (INTERFACE_INFO_magic != interface_list[i]->magic)
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "update_interface_list",
			  "Bad magic number (interface_info)",0);
		
		clean_all = !(interface_list[i]->valid_addrlist);

		if (interface_list[i]->addr_list) {

		    int j;
		    int t = 0;

		    for (j = 0; j < interface_list[i]->addr_count;
			 j++) {

			if (interface_list[i]->addr_list[j]) {
			    if (INTERFACE_ADDR_magic !=
				interface_list[i]->addr_list[j]->magic)
				panic("CONNECTION PANIC",__FILE__,__LINE__,
				      "update_interface_list",
				      "Bad magic number (interface_addr)",
				      0);

			    if (! (interface_list[i]->addr_list[j]->
				   valid_address) || clean_all) {
				
				free_interface_addr(& (interface_list[i]->addr_list[j]));

				removed_ad++;

			    } else if (j != t) {	
			
				interface_list[i]->addr_list[t]
				    = interface_list[i]->addr_list[j];
				interface_list[i]->addr_list[j] = NULL;
				t++;
			    } else
				t++;

			}
		    }

		    interface_list[i]->addr_count = t;

		    if (0 == interface_list[i]->addr_count) {
			free(interface_list[i]->addr_list);
			interface_list[i]->addr_list = NULL;
		    }

		    DPRINT(Debug,10,(&Debug,
				     "update_interface_list: #%d: %d addresses\n",
				     i,interface_list[i]->addr_count));
		}

		if (! (interface_list[i]->valid_index) &&
		    interface_list[i]->if_index > 0) {
		    
		    interface_list[i]->if_index = 0;
		    removed_id++;
		}

		if (! (interface_list[i]->valid_index) &&
		    ! (interface_list[i]->valid_addrlist) &&
		    ! (interface_list[i]->addr_list)) {
		
		    if (interface_list[i]->name) {
			free(interface_list[i]->name);
			interface_list[i]->name = NULL;
		    }

		    interface_list[i]->magic = 0;  /* Invalidate */

		    free(interface_list[i]);
		    interface_list[i] = NULL;
		    removed_if++;

		} else if (i != trg) {

		    interface_list[trg] = interface_list[i];
		    interface_list[i] = NULL;
		    trg++;
		} else
		    trg++;
	    }	    
	}

	interface_count = trg;

	if (0 == interface_count) {
	    free(interface_list);
	    interface_list = NULL;
	}
    }

    if (added_if || added_ad || added_id || removed_if ||
	removed_ad || removed_id || changed_flags) {

	if (0 == (interface_flags & INTERFACE_CHANGE)) {
	    DPRINT(Debug,10,(&Debug,
			     "update_interface_list: Setting  INTERFACE_CHANGE\n"));
	    interface_flags |= INTERFACE_CHANGE;
	}	
    }

    if (0 != (flags & INTERFACE_CHANGE) &&
	0 != (interface_flags & INTERFACE_CHANGE)) {

#ifdef USE_DLOPEN
	DPRINT(Debug,10,(&Debug,
			 "update_interface_list: Notifying interface change\n"));
	shared_notify_if_change();	
#endif

	DPRINT(Debug,10,(&Debug,
			 "update_interface_list: Clearing INTERFACE_CHANGE\n"));

	interface_flags &= ~INTERFACE_CHANGE;	
    }

    
    DPRINT(Debug,10,(&Debug,
		     "update_interface_list=%d: %d interfaces",
		     status,interface_count));
    
    if (added_if) {
	DPRINT(Debug,10,(&Debug,", %d interfaces added",
			 added_if));
    }
    if (added_ad) {
	DPRINT(Debug,10,(&Debug,", %d addresses added",
			 added_ad));
    }
    if (added_id) {
	DPRINT(Debug,10,(&Debug,", %d indexes added",
			 added_id));
    }

    if (removed_if) {
	DPRINT(Debug,10,(&Debug,", %d interfaces removed",
			 removed_if));
    }
    if (removed_ad) {
	DPRINT(Debug,10,(&Debug,", %d addresses removed",
			 removed_ad));
    }
    if (removed_id) {
	DPRINT(Debug,10,(&Debug,", %d indexes removed",
			 removed_id));
    }
    if (changed_flags) {
	DPRINT(Debug,10,(&Debug,", %d addresses wth flags changed",
			 changed_flags));
    }

    DPRINT(Debug,10,(&Debug,"\n"));
    
    return status;
}

/* Hook for resolv -module : *count == -1 if not supported */
struct interface_info ** get_interfaces (interface_info_size,
					 interface_addr_size,
					 count,flags) 
     size_t interface_info_size;
     size_t interface_addr_size;
     int *  count;
     int *  flags;
{
    if (interface_info_size != sizeof (** interface_list)) {
	DPRINT(Debug,1,(&Debug,
			"get_interfaces: Bad interface_info_size %d (should be %d)\n",
			interface_info_size, sizeof (* interface_list)));
	
	*count = -1;
	return NULL;
    }

    if (interface_addr_size != sizeof (struct interface_addr)) {
	DPRINT(Debug,1,(&Debug,
			"get_interfaces: Bad interface_addr_size %d (should be %d)\n",
			interface_addr_size, sizeof (struct interface_addr)));
	
	*count = -1;
	return NULL;
    }

    if (*flags != ( *flags & interface_flags) ||
	0 != (INTERFACE_CHANGE & *flags)	
	) {
	int err UNUSED_VAROK = 0;

	if (0 != (INTERFACE_CHANGE & *flags)) {
	    DPRINT(Debug,10,(&Debug,
			     "get_interfaces: Report interface changes\n"));
	}
	
	if (!update_interface_list(*flags,&err)) {
	    
	    DPRINT(Debug,10,(&Debug,
			     "get_interfaces: Failed to update interfaces for flags %04x",
			     *flags));

	    if (err) {
		DPRINT(Debug,10,(&Debug,", error %s",
				 strerror(err)));
	    }

	    DPRINT(Debug,10,(&Debug,"\n"));

	    *flags = interface_flags;
	    *count = -1;

	    return interface_list;
	}
    }

    *flags = interface_flags;
    *count = interface_count;

    DPRINT(Debug,10,(&Debug,
		     "get_interfaces: Returning %d interfaces\n",
		     *count));

    return interface_list;
}

#endif

    
/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
