static char rcsid[] = "@(#)$Id: read_rc.c,v 2.149 2024/06/10 18:05:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.149 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************
 *
 * Most of code copied from Elm 2.4 src/read_rc.c. It have following copyright:
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

#define SAVE_OPTS
#include "elm_defs.h"
#include "rc_imp.h"
#include "save_opts.h"
#include "s_elm.h"
#include "s_me.h"
#include "unidata.h"
#include "cs_imp.h"
#include "schedule_time.h"
#include "connection_imp.h"

#include "patchlevel.h"

#include <errno.h> 
#ifndef ANSI_C
extern int errno;
#endif

/* For SHRT_MAX */
#include <limits.h>

DEBUG_VAR(Debug,__FILE__,"config");

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

#ifdef PWDINSYS
#  include <sys/pwd.h>
#else
#  include <pwd.h>
#endif

#ifndef I_UNISTD 
char *getlogin();
unsigned short getgid(), getuid(); 
struct passwd *getpwuid();
#endif

# ifdef DOUNAME
#  include <sys/utsname.h>
# endif

struct hist_record {
    char *tag;
    struct rc_save_info_rec * entry;
    int changed;


    char  ** old_line;
    int   line_count;
};

static struct elmrc_recorder {
    struct hist_record *recs;
    int  rec_count;
    
} * recorders[RC_MODE_COUNT];

struct elmrc_recorder  * enable_recorder(mode)
     enum record_mode mode;
{
    if (mode < 0 || mode >= RC_MODE_COUNT) 
	panic("RC PANIC",__FILE__,__LINE__,"enable_recorder",
	      "Bad mode",0);

    if (! recorders[mode] ) {
	struct elmrc_recorder * X = safe_zero_alloc(sizeof (*X));

	X->recs      = NULL;
	X->rec_count = 0;

	recorders[mode] = X;
    }


    return recorders[mode];
}

static struct hist_record * record1 P_((struct elmrc_recorder  *recorder,
					struct rc_save_info_rec *X,
					const char *tag));

static struct hist_record * record1(recorder,X,tag)
     struct elmrc_recorder  *recorder;
     struct rc_save_info_rec *X;
     const char *tag;
{				       
    struct hist_record *r;


    recorder->recs = safe_array_realloc(recorder->recs,
					( recorder->rec_count + 1 ),
					sizeof (recorder->recs[0]));

    r = & (recorder->recs[recorder->rec_count]);

    bzero((void *)r, sizeof (*r));

    r->entry = X;
    r->tag   = NULL;

    if (tag)
	r->tag = safe_strdup(tag);

    r->old_line    = 0;
    r->line_count  = 0;
    r->changed     = 0;

    recorder->rec_count++;

    return r;
}

struct hist_record * rc_need_hist_record(mode,X,tag)
     enum record_mode mode;
     struct rc_save_info_rec * X;
     const char *tag;
{
    int i;

    if (! recorders[mode] ) 
	return NULL;

    /* If old value is already recorded, 
       do not add 'new' old value
    */

    for (i = 0; i < recorders[mode]->rec_count; i++) 
	if (recorders[mode]->recs[i].entry == X)
	    return NULL;
    

    return record1(recorders[mode],X,tag);
}					    

void hist_record_add_line(rec,rest)
     struct hist_record *rec; 
     char *rest;
{
    rec->old_line = safe_array_realloc(rec->old_line,
				       ( rec->line_count + 1),
				       sizeof(rec->old_line[0]));
    
    rec->old_line [ rec->line_count ] = safe_strdup(rest);
    rec->line_count++;
}

void hist_record_first_line(rec,rest)
     struct hist_record *rec; 
     char *rest;
{
    if (rec->line_count)
	panic("RC PANIC",__FILE__,__LINE__,"hist_record_first_line",
	      "Bad record",0);
    
    rec->old_line = safe_malloc(sizeof(rec->old_line[0]));
    
    rec->old_line[0] = safe_strdup(rest);
    
    rec->line_count = 1;
}

void hist_record_changed(rec)
     struct hist_record *rec;
{
    rec->changed = 1;
}


int seed_history(F,filename,recorder)
     FILE *F; 
     char * filename;
     struct elmrc_recorder  *recorder;
{
    int ok = 1;
    char * buffer = NULL;
    int r;  /* length of line or error/eof status */
    int lineno = 0;
    char current_tag[SLEN];

    struct rc_save_info_rec * prev_type = NULL;
    struct hist_record      * rec       = NULL;
    int ignored = 0;

    current_tag[0] = '\0';

    /* malloc_gets does realloc, so buffer need not
       be freed on loop 
    */
    while (0 <= ( r = malloc_gets(&buffer,MAX_LINE,F))) {
	char   * tag         = NULL;
	char   keyword[SLEN];
	char   * rest        = NULL;
	int    restlen = 0;
	int    assig         = 0;
	
	int    negate = 0;

	lineno++;

	if (!buffer)           /* EOF           */
	    break;

	if (buffer[0] == '#'   /* comment       */
	    || r < 1)          /* empty line    */
	    continue;

	rest = rc_split_keyword(buffer,r,keyword, sizeof keyword,
				&tag,&assig, &restlen);


	if (tag && !keyword[0]) {
	    strfcpy(current_tag, tag, sizeof current_tag);

	    if (rest[0] && '#' != rest[0]) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIgnoredLine,
				  "Rest of line ignored after \"%s:\" on line %d on file %s"),
			  tag, lineno,filename);
		ok = 0;
	    }

	} else if (!keyword[0]) {
	    if (prev_type != NULL) {
		
		if (RCTYPE_magic != prev_type->dt_type->magic)
		    panic("RC PANIC",__FILE__,__LINE__,"seed_history",
			  "Bad prev_type",0);
		
		if (rec) {
		    rec->old_line = safe_array_realloc(rec->old_line,
						       ( rec->line_count + 1),
						       sizeof(rec->old_line[0]));
		    
		    rec->old_line [ rec->line_count ] = safe_strdup(rest);
		    rec->line_count++;

		    /* Preserve seeded data when printing again */
		    rec->changed++;
		}
	    }

	} else if (assig) {

	    if (tag || current_tag[0]) {
#ifdef USE_DLOPEN
		struct rc_save_info_rec * rc_options      = NULL;
		size_t                    rc_option_count   = 0;
		enum shared_have_code     have_code = have_code_invalid;
#endif		

		if (!tag) 
		    tag = current_tag;
		else
		    ignored = 0;
		
#ifdef USE_DLOPEN
		if (give_options(tag,&rc_options,&rc_option_count,&have_code)) {
		    
		    prev_type = rc_locate_option(rc_options,rc_option_count,
					      keyword,&negate);
		    ignored = 0;
		    
		    if (!prev_type) {
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeyTagInElmrc,
					  "I can't understand keyword \"%s\" (section %s) in line %d in \"%s\" file"),
				  keyword, tag,lineno,filename);
			ok = 0;
			continue;
		    }
		    
		} else {
		    switch(have_code) {

		    case  have_code_invalid:
			if (!ignored) {
			    ignored = 1;
			    
			    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIgnoredSectionInv,
					      "Keywords on section \"%s\" ignored starting from line %d on file %s, tag %s: is invalid."),
				      tag, lineno,filename,
				      tag);
			    ok = 0;
			    
			}
			break;

		    case have_code_loaded:

			/* Should not happed ! */
			/* FALLTROUGH */

		    case have_code_delayed:
		    case have_code_not:
			
			if (!ignored) {
			    ignored = 1;
			    
			    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIgnoredSection,
				      "Keywords on section \"%s\" ignored starting from line %d on file %s"),
				      tag, lineno,filename);
			    ok = 0;
			}
			break;
		    }
		}
		    
#else
		if (!ignored) {
		    ignored = 1;
		    
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSharedIgnoredSection,
				      "Shared libraries not supported. Section \"%s\" ignored starting from line %d on file %s"),
			      tag, lineno,filename);
		    ok = 0;
		}
#endif
		
	    } else {
		
		prev_type = rc_locate_option(save_info,
					     NUMBER_OF_SAVEABLE_OPTIONS,
					     keyword,&negate);
		
		if (!prev_type) {	       
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeyInElmrc,
				      "I can't understand keyword \"%s\" in line %d in \"%s\" file"),
			      keyword, lineno,filename);
		    ok = 0;
		    continue;
		}
		
	    }
	    
	    rec = record1(recorder,prev_type,tag);
	    if (rec) {
		if (rec->line_count)
		    panic("RC PANIC",__FILE__,__LINE__,"seed_history",
			  "Bad prev_type",0);
		
		rec->old_line = safe_malloc(sizeof(rec->old_line[0]));
		
		rec->old_line[0] = safe_strdup(rest);
		
		rec->line_count = 1;
		rec->changed++;
	    }
	} else {
	    prev_type = NULL;

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadLineInElmrc,
			      "Keyword \"%s\" without = -character in line %d in \"%s\" file"),
		      keyword, lineno,filename);
	    ok = 0;

	}
    }

    if (-1 == r) {	
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmTooLongLine,
			  "%.30s: %d: Too long line: %.30s..."),
		  filename,lineno+1,
		  buffer ? buffer : "<not available>");
	ok = 0;
    }

    if (ferror(F)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorReading,
			  "Error reading from %.50s"),
		  filename);
	ok = 0;
    }

    if (buffer)
	free(buffer);


    return ok;
}

static void print_history_record P_((FILE *F,struct hist_record *x));
static void print_history_record(F,x)
     FILE *F;
     struct hist_record *x;
{
    int i;

    fprintf(F,"\n%s = %s\n",
	    x->entry->name,
	    x->old_line[0]);

    for (i = 1; i < x->line_count; i++) {
	fprintf(F,"\t%s\n",
		x->old_line[i]);
    }
}

void print_history_changed(F,recorder)
     FILE *F; 
     struct elmrc_recorder  *recorder;
{
    int i;

    /* First print lines without tag */
    char * last_tag = NULL;


    for (i = 0; i < recorder->rec_count; i++) {

	if (recorder->recs[i].changed &&
	    NULL == recorder->recs[i].tag) {
	    print_history_record(F,& (recorder->recs[i]));
	}

    }

    /* Then print values with tag */

    for (i = 0; i < recorder->rec_count; i++) {

	if (recorder->recs[i].changed &&
	    NULL != recorder->recs[i].tag) {

	    if (!last_tag || 
		0 != strcmp(last_tag,
			    recorder->recs[i].tag)) {
		last_tag = recorder->recs[i].tag;

		fprintf(F,"\n\n%s:\n",
			last_tag);
	    }

	    print_history_record(F,& (recorder->recs[i]));
	}
    }
}

void mark_XX(x, no_history_record)
     struct rc_save_info_rec *x;
     int no_history_record;
{
    enum record_mode mode; 

    int found = 0;

    x->flags |= FL_LOCAL;
    x->flags |= FL_CHANGED;

    DPRINT(Debug,10,(&Debug,
		     "mark_XX: Setting %p %s changed\n",
		     x,x->name));
	
    if (no_history_record)
	return;

    for (mode = SYSTEM_RC; mode < RC_MODE_COUNT; mode++) {
    
	if (recorders[mode]) {
	    int i;

	    for (i = 0; i < recorders[mode]->rec_count; i++) 
		if (recorders[mode]->recs[i].entry == x) {
		    recorders[mode]->recs[i].changed++;
		    found++;

		    DPRINT(Debug,10,(&Debug,
				     "mark_XX: recorders[%d]->recs[%d].entry == %p\n",
				     mode,i,x));

		}
	} else {
	    DPRINT(Debug,10,(&Debug,
			     "mark_XX: no mode=%d\n",mode));
	}
    }
	
    if (!found) {
	DPRINT(Debug,10,(&Debug,
			 "mark_XX: no change record\n"));
    }

}

void mark_local_changed(A,no_history_record)
     void *A;
     int no_history_record;
{
    int x;
    

    for (x = 0; x < NUMBER_OF_SAVEABLE_OPTIONS; x++) {
        if (save_info[x].val.dummy == A) {

	    DPRINT(Debug,10,(&Debug,
			     "mark_local_changed: %d: %p\n",
			     x,A));

	    mark_XX(& (save_info[x]), no_history_record);

	}
    }

#ifdef USE_DLOPEN
    mark_shared_changed(A, no_history_record);
#endif

    mark_delayed_changed(A, no_history_record);

}




void mark_flocal_changed(A, no_history_record)
     option_func *A;
     int no_history_record;
{
    int x;
    
    for (x = 0; x < NUMBER_OF_SAVEABLE_OPTIONS; x++) {
        if (save_info[x].func_val == A) {

	    DPRINT(Debug,10,(&Debug,
			     "mark_flocal_changed: %d: %p\n",
			     x,A));

	    mark_XX(& (save_info[x]), no_history_record);

	}
    }

#ifdef USE_DLOPEN
    mark_fshared_changed(A, no_history_record);
#endif

    mark_fdelayed_changed(A, no_history_record);
}



#ifdef HAVE_CODESET
#include <langinfo.h>
#endif

static int lineno = 0;
static int errors = 0;

static void do_expand_env P_((char *, char *, char *, unsigned));

static charset_t set_elmrc_charset P_((const char *filename,int *errors));

int parse_elmrc(F,global,filename,read_flags)
     FILE *F;
     int global;
     char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    struct parsed_rc *rc = NULL;
    charset_t   rcset = NULL;
    enum record_mode lcl = global ? SYSTEM_RC : LOCAL_RC;

    DPRINT(Debug,8,(&Debug, 
		    "parse_elmrc: global=%d read_flags = %d%s%s%s%s, filename=%s\n",
		    global,
		    read_flags,
		    (read_flags & READ_FLAG_GLOBAL_ONLY) ? " READ_FLAG_GLOBAL_ONLY" : "",
		    (read_flags & READ_FLAG_IGNORE_MISSING) ? " READ_FLAG_IGNORE_MISSING" : "",
		    (read_flags & READ_FLAG_UNIDATA_INIT) ? " READ_FLAG_UNIDATA_INIT" : "",
		    (read_flags & READ_FLAG_BINDATA_INIT) ? " READ_FLAG_BINDATA_INIT" : "",
		    filename
		    ));

    
    errors = 0;

    elmrc_charset[0] = '\0';   /* Reset old value */

    rc = do_rc_parse(F, lcl, filename,&errors,
		     read_flags);

    rcset = set_elmrc_charset(filename,&errors);

    if (rc) {
	do_rc_process(rc, lcl,filename,
		      rcset,&errors,
		      read_flags);
	free_parsed_rc(&rc);
    } else
	errors++;

    DPRINT(Debug,8,(&Debug, 
		    "parse_elmrc=%d (%s)\n",
		    errors,
		    errors ? "errors" : "OK"
		    ));

    return errors;
}
#ifndef ok_rc_char
#define ok_rc_char(c)   (isalnum(c) || c == '-' || c == '_')
#endif

char old_system_text_file[SLEN]     = OLD_SYSTEM_TEXT_FILE;
char old_system_data_file[SLEN]     = OLD_SYSTEM_DATA_FILE;
char system_aliases_file[SLEN]      = SYSTEM_ALIASES_FILE;
char system_rc_file[SLEN]       = SYSTEM_RC_FILE;
char system_mime_types[SLEN]    = SYSTEM_MIME_TYPES;

char system_iso2022_sets[SLEN]  = SYSTEM_ISO2022_SETS;
struct iso2022_map_conf  * system_iso2022_map_conf;

char system_terminal_info[SLEN] = SYSTEM_TERMINAL_INFO;

char system_mail_services[SLEN] = SYSTEM_MAIL_SERVICES;
charset_t                   system_mail_services_cs;
struct editor_propline    * system_mail_services_pl;
struct mail_services_conf * system_mail_services_conf;

char system_mlist_file[SLEN]    = SYSTEM_MLIST_FILE;
char hostdomfile[SLEN]          = HOSTDOMFILE;
char system_mime_charsets[SLEN] = SYSTEM_MIME_CHARSETS;

char system_hash_marks[SLEN]    = SYSTEM_HASHMARK_FILE;
char system_tagfilter_entfile[SLEN] = SYSTEM_TAGFILTER_ENTFILE;

ESTR map_txtdir_e = {            /* init_defaults */
    ESTR_meta|ESTR_slash,
    NULL,
    NULL,
    0,0,NULL,0
};
ESTR map_bindir_e = {            /* init_defaults */
    ESTR_meta|ESTR_slash,
    NULL,
    NULL,
    0,0,NULL,0
};

char unidata_path_global[SLEN];   /* Used before user elm.rc is read */
char unidata_path[SLEN];        /* init_defaults */
char raw_unidata_path[SLEN] =  "unidata.bin";     /* init_defaults */

char bindata_path_global[SLEN];   /* Used before user elm.rc is read */
char bindata_path[SLEN];  /* init_defaults */
char raw_bindata_path[SLEN] =  "bindata.bin";     /* init_defaults */

int  unstable_reverse_thread = 0;

char user_rc_file[SLEN]   = ELMRCFILE;              /* user_init fixes */
char old_user_text_file[SLEN] = OLD_USER_ALIAS_TEXT;     /* user_init fixes */
char old_user_data_file[SLEN] = OLD_USER_ALIAS_DATA;     /* user_init fixes */
char user_aliases_file[SLEN] = USER_ALIASES_FILE;        /* user_init fixes */
char user_mime_types[SLEN] = USER_MIME_TYPES;       /* user_init fixes */
char user_mailheaders[SLEN] = MAILHEADERS;          /* user_init fixes */
char user_mime_charsets[SLEN] = USER_MIME_CHARSETS; /* user_init fixes */

char user_iso2022_sets[SLEN]  = USER_ISO2022_SETS;  /* user_init fixes */
struct iso2022_map_conf      * user_iso2022_map_conf;

char user_terminal_info[SLEN] = USER_TERMINAL_INFO; /* user_init fixes */
char user_mlist_file[SLEN]    = USER_MLIST_FILE;    /* user_init fixes */

char user_mail_services[SLEN] = USER_MAIL_SERVICES; /* user_init fixes */
charset_t                   user_mail_services_cs;
struct editor_propline    * user_mail_services_pl;
struct mail_services_conf * user_mail_services_conf;

char user_hash_marks[SLEN] =   USER_HASHMARK_FILE;  /* user_init fixes */
char user_tagfilter_entfile[SLEN] =  USER_TAGFILTER_ENTFILE; /* user_init fixes */
char user_last_read_def[SLEN]  =  USER_LAST_READ;  /* user_init fixes */
char user_last_read_dot[SLEN]  = ".elm-last-read"; /* user_init fixes */

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

/* exported for lib/mbox/init.c */
char * MAILBOX_LOCKING[NUM_mailbox_locking_v+1] = {
    "lockfile", "flock", "fcntl",
    "prefer",
    NULL
}; 


nl_catd  elm_msg_cat; 	 /* message catalog	    */
                         /* WARNING: Elm's version uses 0 as failure
                                     System uses (nl_catd) -1 as failure !
			 */


char username[SLEN];	/* return address name!    */
char home[SLEN];		/* home directory of user  */
int userid = -1;			/* uid for current user	      */
int groupid = -1;			/* groupid for current user   */

int mailgroupid;		/* groupid for current user   */
int have_saved_ids;             /* sysconf _SC_SAVED_IDS      */

const char * have_dummy_domain   /* DUMMYDOMAIN  -- ".localdomain" */
#ifdef DUMMYDOMAIN
  = DUMMYDOMAIN
#endif
    ;


/* Really constants currently:  ----------------  */

const int ALLOW_mailbox_editing  /* set if 
				    ALLOW_MAILBOX_EDITING
				    defined */
#ifdef ALLOW_MAILBOX_EDITING
                    = 1

#endif
    ;

const int ALLOW_subshell       /* Set if ALLOW_SUBSHELL defined */
#ifdef ALLOW_SUBSHELL
                    = 1
#endif
    ;

const int ENABLE_calendar      /* Set if ENABLE_CALENDAR defined */
#ifdef ENABLE_CALENDAR
                     = 1
#endif
    ;

const int have_ISPELL          /* set if ISPELL defined */
#ifdef ISPELL
                    = 1
#endif
    ;
const char ISPELL_path[]    /* set if ISPELL_PATH defined */
#ifdef ISPELL_PATH
                     = ISPELL_PATH
#else
                     = "/bin/false"
#endif
    ;

const char ISPELL_options[]  /* set if ISPELL_OPTIONS defined */
#ifdef ISPELL_OPTIONS
                    = ISPELL_OPTIONS

#else
                    = ""
#endif
    ;

const int have_MMDF             /* set if MMDF defined        */
#ifdef MMDF 
                    = 1
#endif
    ;

const enum poll_method POLL_method /* set if POLL_METHOD defined */
#ifdef POLL_METHOD
                    = POLL_METHOD
#endif
    ;

const int use_PGP                  /* set if USE_PGP defined     */
#ifdef USE_PGP
                    = 1
#endif
    ;

const int CHOWN_neg1    /* Set if CHOWN_NEG1 defined   */
#ifdef CHOWN_NEG1
                    = 1
#endif
    ;


const char dsn_mail_tag[] = "[dsn]";

/* ------------- Variables in elmrc ------------------------------- */

int  add_irt_phrase;          /* In-reply-to: Add phare to in-reply-to ?
			           True = Do not follow RFC 2822 
				*/


static char *ADD_SENDER_HEADER[NUM_add_sender+1] =
    { "no", "yes", "auto", NULL };
ENUMERATE add_sender_header = {  /* If set add Sender: header if user specifies
				    From: -header
				 */
    add_sender_auto,
    NUM_add_sender, &(ADD_SENDER_HEADER[0]),
    NULL,
    1   	/* allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL
};


static char *ADDRESS_LOOKUP[NUM_addr_lookup+1] =
    { "normal"             /* same as gethostbyaddr     */,
      "shared"             /* RESERVED                  */,
      "gethostbyaddr"      /* gethostbyaddr             */,
      "getnameinfo"        /* getnameinfo               */,
      NULL };

ENUMERATE address_lookup = { /* Should try some odd address lookup ? 
			      */
    addr_lookup_normal       /* the default */,
    NUM_addr_lookup,&(ADDRESS_LOOKUP[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

int allow_charset_switching;   /* flag: allow changing charset of */
				 /* terminal if terminal supports   */
int allow_setuid;

static struct sortval ASR_srtval[]={
    {"alias", ALIAS_SORT},
    {"name", NAME_SORT},
    {"text", TEXT_SORT},
    {NULL, 0} 
};

SORT alias_sortby = {            /* how to sort aliases...   */
    NAME_SORT,	           

    ASR_srtval
};

int alias_group_phrase = 1;      /* Use phrases on group aliases */ 

static  struct keyword_val ALTEDITOR_KW[] = {
    { "EDITOR",       alteditor_kw_EDITOR, 0 },
    { "use-$EDITOR",  alteditor_KW_EDITOR, 1 },    /* bitmask */
    { "use-$VISUAL",  alteditor_KW_VISUAL, 1 },    /* bitmask */
    { NULL,           alteditor_kw_NO,     0 }
};
    
/* Returns 1 if success, 0 failure, -1 if expansion is not string */
S_(keyword_expand_func alteditor_expand_func)
static int alteditor_expand_func P_((struct dt_estr_info *kw,
				     struct dt_keyword_info *ki,
				     unsigned int keyword,
				     unsigned int continue_mask)); 
static struct dt_keyword_info alteditor_keyword_info = {
    alteditor_expand_func,
    ALTEDITOR_KW
};

ESTR alternative_editor_e = {    /* alternative editor...    */
    ESTR_none|ESTR_print_expanded,
    NULL,NULL,
    alteditor_kw_EDITOR, alteditor_KW_EDITOR|alteditor_KW_VISUAL,
    & alteditor_keyword_info,
    0
};

PATH alternative_addresses = {	/* how else do we get mail? */
    PATH_sep_comma| PATH_ascii_only,
    NULL, 0,NULL
};

int always_del;		/* flag: always delete marked msgs? */
int always_keep = 1;		/* flag: always keep unread msgs?   */
int always_store;		/* flag: always store read msgs?    */
int ask_delete = 1;             /* confirm delete on resync? */
int ask_keep = 1;              /* ask to keep unread msgs? */
int ask_reply_copy = 1;        /* ask to copy mssg into reply? */
int ask_store = 1;             /* ask to store read mail? */
int arrow_cursor;		/* flag: use "->" cursor regardless?*/

ESTR attachment_dir_e = {   /* {doc} -prefix */
    0,                      /* Note that default assigment do not add / to end */
    NULL,
    NULL,
    0,0,NULL,0
};

int attachment_fix_extension = 1; /* flag: fix extension of attashment when saving */

ESTR calendar_file_e; 	                /* name of file for clndr  */

static  struct keyword_val DEFAULTFILE_KW[] = {
     { "use-$MAIL",   defaultfile_KW_MAIL, 1 },    /* bitmask */
     { "local-file", FOLDER_is_local_file, 1 },    /* bitmask */
     { NULL,         0, 0  }
};

/* Returns 1 if success, 0 failure, -1 if expansion is not string */
S_(keyword_expand_func alteditor_expand_func)
static int defaultfile_expand_func P_((struct dt_estr_info *kw,
				       struct dt_keyword_info *ki,
				       unsigned int keyword,
				       unsigned int continue_mask));

static struct dt_keyword_info defaultfile_keyword_info = {
    defaultfile_expand_func,
    DEFAULTFILE_KW
};


ESTR defaultfile_e = { 	                /* name of default folder */
    ESTR_none|ESTR_meta,
    NULL,NULL,
    0,defaultfile_KW_MAIL,
    &defaultfile_keyword_info,
    0
};

ESTR dead_letter_dir_e; 

int prompt_for_cc = 1;		/* flag: ask user for "cc:" value?  */

int pgp_askpgpsig; /* Should pgp ask userid to sign messages with? */

static char * PGP_SIGN_TYPES[NUM_pgp_sign_type+1] = { "application/pgp", "text/plain", 
						      "text/x-pgp", NULL };

ENUMERATE pgp_sign_type= {     /* 0 = application/pgp  */
    pgp_sign_application_pgp,  /* 1 = text/plain       */
    NUM_pgp_sign_type,         /* 2 = text/x-pgp       */ 
    &(PGP_SIGN_TYPES[0]),
    NULL,
    0   	/* no boolean  */,
    NULL	/* not delayed */,
    NULL
};

static char * PGP_ENCRYPT_TYPES[NUM_pgp_encrypt_type+1] =
    { "application/pgp", "text/plain", 
      "text/x-pgp", NULL };

ENUMERATE pgp_encrypt_type= {   /* 0 = application/pgp */
    pgp_encrypt_application_pgp,/* 1 = text/plain      */
    NUM_pgp_encrypt_type,       /* 2 = text/x-pgp      */
    &(PGP_ENCRYPT_TYPES[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

int pgp_interactive;            /* Should pgp/gpg encryption/signing to be
			            run always on interactive mode */

static char * PGP_VERSIONS[PGP_NUM+1] = { "default", "pgp2", "pgp5", "gpg", NULL };
ENUMERATE send_pgp_version = {  /* preferred pgp version*/
    pgp_none,
    PGP_NUM, &(PGP_VERSIONS[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};
	
struct string * attribution_s;	        /* attribution string for replies   */
struct string * fwdattribution_s;	/* attribution string for forwarded 
					   mssgs */

static char * AUTO_ATTACH[NUM_auto_attachment+1] = { "none", "application", "non-text", NULL };
ENUMERATE  auto_attachment = {   /* 0 = none,                     */
    auto_attachment_application, /* 1 = application,              */
    NUM_auto_attachment,         /* 2 = non-text   		  */
    &(AUTO_ATTACH[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

int auto_copy;		        /* flag: automatically copy source? */
int reply_copy = 1;		/* flag: copy message into reply */

int auto_iso_8859 = 1;          /* flag: Should ISO-8859-X charsets defined
				   on demand ?                      */
#ifdef BACKGROUD_PROCESSES       
int background_wait_time = 2;    /* If > 0 background mailer after  */
				 /* this number of seconds          */
#endif

static char * BROWSER_WILDCARD[NUM_browser_wildcards+1] = { "off", "on", "file browser" };
ENUMERATE browser_wildcards = {  /* do wildcard match?         */
    browser_wildcard_yes,
    NUM_browser_wildcards, &(BROWSER_WILDCARD[0]),
    NULL,
    1   	/* allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL
};

int browser_tabcomplete = 1;    /* flag:  tabcomplete prompt?       */
int builtin_lines= -3;		/* int: if < 0 use builtin if message*/
                                /* shorter than LINES+builtin_lines */
                                /* else use pager. If > 0 use builtin*/
                                /* if message has fewer than # of lines*/
int charset_convert_ok;         /* flag: no warning about conversion? */
int check_type_pattern = 1;     /* flag: check magic numbers when paging */
char raw_text_charset[SLEN] = "DISPLAY";    /* name of character set     */
charset_t text_charset;   	    /* pointer of character set  */
char raw_default_mimetext_charset[SLEN] = "US-ASCII"; 
                                                /* name of character set    */
charset_t default_mimetext_charset;      /* pointer to character set */
char raw_default_nomime_charset[SLEN] = "SYSTEM"; /* name of character set  */
charset_t default_nomime_charset;      /* pointer to character set   */

static char * FOLDER_STATUS[NUM_folder_status+1] = { "Read", "Old", "New", NULL };
ENUMERATE def_folder_status = {      /* 0 = Read                    */
    FOLDER_STATUS_OLD,               /* 1 = Old                     */
    NUM_folder_status,               /*	2 = New                     */
    &(FOLDER_STATUS[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

FLAGS conf_merge_locking = {
    0L 
#ifdef USE_FLOCK_LOCKING
    | FLAGVAL(lock_flock_flag)
#endif
#ifdef USE_FCNTL_LOCKING
    | FLAGVAL(lock_fcntl_flag)
#endif
    ,0L,0L,
    NUM_mailbox_locking_v,&(MAILBOX_LOCKING[0]),
    0L
#ifdef COMPILE_FLOCK_LOCKING
    | FLAGVAL(lock_flock_flag)
#endif
#ifdef COMPILE_FCNTL_LOCKING
    | FLAGVAL(lock_fcntl_flag)
#endif
    , 1,NULL
};

char config_options[SLEN];/* which options are in o)ptions    */

int confirm_append;		/* flag: confirm append to folder?  */
int confirm_create;		/* flag: confirm create new folder? */
int confirm_files;		/* flag: confirm files for append?  */
int confirm_folders;	        /* flag: confirm folders for create?*/
int confirm_print;              /* flag: confirm printing of messages ?    */
int confirm_tag_save = 1;       /* flag: confirm saving of tagged messages */
int edit_flush = 1;	        /* flag: flush input after extern edit? */
int convert_comment = 1;        /* flag: convert comment to fullname ?  */

static char *CONVERT_CR_HACK[NUM_convert_cr_hack+1] =
    { "off",
      "on-with-error",
      "on-silent",
      NULL };

ENUMERATE convert_cr_to_newline_hack = {
    convert_cr_off,
    NUM_convert_cr_hack,&(CONVERT_CR_HACK[0]),
    NULL,
    1   	/* allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL
};

int convert_utf_header = 1; /* flag: convert utf-8 mime encoded words to utf-7 when sending?   */

int auto_copy_sent = -1;	  /* flag: automatically copy sent mail? */
char display_locale[SLEN] = "NONE";  /* LC_CTYPE locale (character set) */
char display_codeset[SLEN] = "";     /* nl_langinfo(CODESET) for current locale (character set) */

charset_t wanted_display_charset;
char raw_display_charset[SLEN]   = "SYSTEM";	
                               /* the wanted display charset set */

int DSN_success;               /* flag: Ask successfull DSNes      */

static  struct keyword_val DSNFLD_KW[] = {
     { "local-file", FOLDER_is_local_file, 1 },    /* bitmask */
     { NULL,         0, 0  }
};

static struct dt_keyword_info dsnfld_expand_info = {
    null_keyword_expand,
    DSNFLD_KW
};

ESTR dsn_mail_e = {  /* folder for storing DSN mail	*/
    ESTR_none|ESTR_meta|ESTR_initial,
    default_dsnmail,
    NULL,
    0,0,
    &dsnfld_expand_info,
    0
};   


char e_editor[SLEN];	/* "~e" editor...                   */

static  struct keyword_val EDITOR_KW[] = {
    { "builtin",    editor_kw_builtin,   0 },
    { "internal",   editor_kw_builtin,   0 },
    { "use-$EDITOR", editor_KW_EDITOR,  1 },    /* bitmask */
    { "use-$VISUAL", editor_KW_VISUAL,  1 },    /* bitmask */
    { NULL,          editor_kw_NO }
};

/* Returns 1 if success, 0 failure, -1 if expansion is not string */
S_(keyword_expand_func editor_expand_func)
static int editor_expand_func P_((struct dt_estr_info *kw,
				  struct dt_keyword_info *ki,
				  unsigned int keyword,
				  unsigned int continue_mask));
static struct dt_keyword_info editor_keyword_info = {
    editor_expand_func,
    EDITOR_KW
};


ESTR editor_e = {                 /* editor for outgoing mail         */
    ESTR_none|ESTR_print_expanded,
    NULL,NULL,
    0, alteditor_KW_EDITOR|alteditor_KW_VISUAL,
    & editor_keyword_info,
    0
};

static char * EDITOR_PROPLINE [NUM_editor_propline+1] = 
    { "auto", "emacs", "ignore", NULL };
ENUMERATE editor_elmrc_propline = {
    editor_propline_auto,
    NUM_editor_propline,
    &(EDITOR_PROPLINE[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};
ENUMERATE editor_ms_propline = {
    editor_propline_auto,
    NUM_editor_propline,
    &(EDITOR_PROPLINE[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};



int mime_rfc1522_filename = 1;     /* flag: mime-encoded-filename-hack
				      If this is set, then MIME encoded 
				      words (RFC 2047 or RFC 1522) 
				      filename parameter is detected. 
				      However RFC 2231 style encoding 
				      is preferred if 
				      mime_parameters > 0 (below)
				   */

static char * MPAR[NUM_mime_parameters+1] = { "plain", "encoded", "plain-and-encoded",NULL };
ENUMERATE mime_parameters = {
    mime_parameters_plain_and_encoded,
    NUM_mime_parameters, &(MPAR[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

static char * ENV_FROM_SOURCES[NUM_env_from_source+1] = { "forward-from", "from", "return-path", 
							  NULL };
ENUMERATE env_from_source = {  /* 0 == forward-from,                */
    env_source_forward_from,   /* 1 == from,                        */
    NUM_env_from_source,       /* 2 == return-path                  */
    &(ENV_FROM_SOURCES[0])  ,
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

char escape_char = TILDE_ESCAPE;/* '~' or something else..          */
int force_name;		     /* flag: save by name forced?	    */

static char * FRAGMENTH_list[NUM_fragment_handling+1] = {
    "none", "manual", "auto",NULL
};
ENUMERATE fragment_handling = { /* 0 == none,                       */
    fragment_handling_manual,   /* 1 == manual,                     */
    NUM_fragment_handling,      /* 2 == auto                        */
    &(FRAGMENTH_list[0])     ,
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};
struct string * full_username_s; /* Full username - gecos           */

static char * HDR_PHRASE_DISPLAY_MODES[NUM_phrase_display_mode+1] = {
    "plain", "quoted", NULL
};
ENUMERATE phrase_display_mode = {    /* 0 == plain, 1 = quoted */
    phrase_display_plain,
    NUM_phrase_display_mode, &(HDR_PHRASE_DISPLAY_MODES[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

char hostdomain[SLEN];	/* name of domain we're in          */
char hostfullname[SLEN];	/* name of FQDN we're in            */
char hostname[SLEN];	/* name of machine we're on         */
#ifdef DISP_HOST
int  menu_display_host = TRUE;  /* flag: Display host on mailbox menu* */
#else
int  menu_display_host = FALSE;  /* flag: Display host on mailbox menu? */
#endif
int keep_empty_files;	        /* flag: leave empty folder files?  */
int pgp_keeppassfor = 300;      /* 5 minutes                        */

static struct sortval DIR_srtval[]={
    {"none", LD_NONE},
    {"name", LD_NAME_SORT},
    {"mtime", LD_MTIME_SORT},
    {NULL, 0} 
};

SORT local_dir_sortby = {            /* how to sort files...  */
    LD_NAME_SORT,	           
    DIR_srtval
};

int local_fast_lookup;          /* flag: directory listing not needed 
				 *       for lookup 
				 */

int fb_show_dotfiles;           /* flag: Show .dotfiles on Elm 2.5 
                                 *       style browser
				 */

int fb_show_protection = 1;     /* flag: Show permission and owner columns 
				 *  on Elm 2.5 style browser 
				 */

int fb_show_title = -1;         /* flag: Show title of columns on Elm 2.5 
				 *       style browser 
				 */

#define DEFAULT_SORT_ORDER    SHRT_MAX
SORT fb_sort_order = {            /* how to sort files...  */
    DEFAULT_SORT_ORDER,	           
    DIR_srtval
};


FLAGS folder_locking = {
    0L 
#ifdef USE_FLOCK_LOCKING
    | FLAGVAL(lock_flock_flag)
#endif
#ifdef USE_FCNTL_LOCKING
    | FLAGVAL(lock_fcntl_flag)
#endif
    ,0L,0L,
    NUM_mailbox_locking_v,&(MAILBOX_LOCKING[0]),
    0L
#ifdef COMPILE_FLOCK_LOCKING
    | FLAGVAL(lock_flock_flag)
#endif
#ifdef COMPILE_FCNTL_LOCKING
    | FLAGVAL(lock_fcntl_flag)
#endif
    , 1,NULL
};


#ifdef PIDCHECK
static char * LLP_list[NUM_lockfile_pidcheck+1] = {
    "no", "yes", "hostname", 
    "ignore-hostname", "require-hostname", NULL
};
ENUMERATE local_lockfile_pidcheck = {     /* 0 == no               */
    pidcheck_hostname,			  /* 1 == yes              */
    NUM_lockfile_pidcheck, &(LLP_list[0]),/* 2 == hostname         */
					  /* 3 == ignore-hostname  */
                                          /* 4 == require-hostname */
    NULL,
    1   	/* allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL
};
#endif

char local_sessionlock_dir[SLEN] = default_temp;

static char * LSL_list[NUM_local_seslck_use_home+1] = {
    "never", "spool", "always", "non-spool",
    NULL
};
ENUMERATE local_sessionlock_use_home = {  /* 0 == never,          */
    local_seslck_use_home_never,          /* 1 == spool           */
    NUM_local_seslck_use_home,            /* 2 == always          */ 
    &(LSL_list[0]),			  /* 3 == non-spool  	  */
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};					  


char raw_local_fs_charset[SLEN] = "SYSTEM"; /* filesystem charset */
charset_t local_fs_charset;     /* filesystem charset */
ESTR local_signature_e;         /* local msg signature file     */

#if defined(WCHAR) && defined(__STDC_ISO_10646__)
PATH locale_charsets;           /* Use type=unknown for these charsets
			         * if system charset
			         */
#endif

#ifdef SYSCALL_LOCKING
int lockfolders  = 1;           /* Lock folder when open */
int lock_in_copy = 1;           /* Lock folder when copied to it    */
#else
int lockfolders;                /* Lock folder when open */
int lock_in_copy;               /* Lock folder when copied to it    */
#endif

int long_encoded_headers = 1;   /* Produce over 80 character mime encoded
				   headers */

static char * MSERV_LOOKUP[NUM_mserv_lookup+1] = {
    "hostname",
    "shared",
    "gethostbyname",
    "getaddrinfo",
    NULL
};

ENUMERATE mail_services_lookup = { /* Should try some odd hostname lookup ? */
    mserv_lookup_hostname,
    NUM_mserv_lookup,&(MSERV_LOOKUP[0]),
    NULL,
    0  		/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

ESTR folders_e = 	{       /* folder home directory            */
    ESTR_slash,
    NULL,
    NULL,
    0,0,NULL,0
};   

int Expires_have_time;   /* Expires: -header field generated from
			    Message Header Edit Screen have
			    date and time; if not set, only
			    date in shown.
			 */

ESTR extra_mailbox_dir_e = {
    ESTR_slash,
    NULL,
    NULL,
    0,0,NULL,0
};

FLAGS mailbox_locking = {
    0L
#ifdef USE_DOTLOCK_LOCKING
    | FLAGVAL(lock_lockfile_flag)
#endif 
#ifdef USE_FLOCK_LOCKING
    | FLAGVAL(lock_flock_flag)
#endif
#ifdef USE_FCNTL_LOCKING
    | FLAGVAL(lock_fcntl_flag)
#endif
    ,0L,0L,
    NUM_mailbox_locking_v,&(MAILBOX_LOCKING[0]),
    FLAGVAL(lock_lockfile_flag)
#ifdef COMPILE_FLOCK_LOCKING
    | FLAGVAL(lock_flock_flag)
#endif
#ifdef COMPILE_FCNTL_LOCKING
    | FLAGVAL(lock_fcntl_flag)
#endif
    , 1,NULL
};


PATH mailname = {  /* Name used after the @  -character on addresses and 
		      aliases of that. */
    PATH_sep_comma| PATH_ascii_only,
    NULL, 0,NULL
};

ESTR mailhome_dir_e  = {         /* local mail spool directory        */
    ESTR_none|ESTR_slash|ESTR_initial,
    MAILHOMEDIR,
    MAILHOMEDIR,
    0,0,NULL,0
};


FLAGS mailhome_locking = {
    0L
#ifdef USE_DOTLOCK_LOCKING
    | FLAGVAL(lock_lockfile_flag)
#endif 
#ifdef USE_FLOCK_LOCKING
    | FLAGVAL(lock_flock_flag)
#endif
#ifdef USE_FCNTL_LOCKING
    | FLAGVAL(lock_fcntl_flag)
#endif
    ,0L,0L,
    NUM_mailbox_locking_v,&(MAILBOX_LOCKING[0]),
    FLAGVAL(lock_lockfile_flag)
#ifdef COMPILE_FLOCK_LOCKING
    | FLAGVAL(lock_flock_flag)
#endif
#ifdef COMPILE_FCNTL_LOCKING
    | FLAGVAL(lock_fcntl_flag)
#endif
    , 1,NULL
};

int mail_permissions = 0600;	  /* int: permissions for mailbox files */
int mini_menu_rc             = 1; /* flag: menu specified?	    */

static char * hide_hacks[NUM_message_hide_hack+1] = { "none", "FOLDER INTERNAL DATA", NULL };
ENUMERATE message_hide_hack = { /* 0 == none                          */
    message_hide_none,          /* 1 = FOLDER INTERNAL DATA           */
    NUM_message_hide_hack, &(hide_hacks[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

ESTR metamail_path_e = {        /* Metamail path or "none"          */
     ESTR_none|ESTR_metamail,   /* if no metamail                   */
     NULL,
     NULL,
     0,0,NULL,0
};
PATH metamail_mailcaps = {   /* mailcaps for metamail */
    PATH_file|PATH_sep_colon|PATH_expand,
    NULL, 0,NULL
};

PATH internal_mailcaps = {   /* mailcaps for internal handler */
    PATH_file|PATH_sep_colon|PATH_expand,
    NULL, 0,NULL
};

PATH internal_mailcap_t_programs;
int internal_mailcap_t_prompt;

PATH incomingfolders = {      /* treate as mailbox */
    PATH_file|PATH_expand|PATH_sep_comma,
    NULL, 0, NULL
};

#ifdef HAVE_SCOPE
static char * IPV6_SCOPE_ID[NUM_ipv6_scope_id_v+1] = {
    "auto", "link-local-if-index", NULL
};

FLAGS ipv6_scope_id = {
    0L 
#if defined (SCOPE_ID_DEFAULT)
    | SCOPE_ID_DEFAULT
#elif defined(HAVE_IFADDRS) && defined(HAVE_NAMEINDEX)
    | FLAGVAL(scope_auto_flag)
#endif
    ,0L,0L,
    NUM_ipv6_scope_id_v,&(IPV6_SCOPE_ID[0]),
    0L
#if defined(HAVE_IFADDRS) && defined(HAVE_NAMEINDEX)
    | FLAGVAL(scope_auto_flag)
#endif
#ifdef HAVE_NAMEINDEX
    | FLAGVAL(scope_link_local_if_index_flag)
#endif
    , 1,NULL
};

#endif


static char * MAIL_MENU_TIME[NUM_mail_menu_time_v+1] = {
    "year-separator",
    "auto-separator",
    "today",
    "today-separator",
    "year-or-time",
    NULL
};
FLAGS mail_menu_time = {
    FLAGVAL(mmtime_aseparator_flag) |
    FLAGVAL(mmtime_today_flag),
    0L,0L,
    
    NUM_mail_menu_time_v, &(MAIL_MENU_TIME[0]),
    FLAGVAL(mmtime_yseparator_flag) |
    FLAGVAL(mmtime_aseparator_flag) |
    FLAGVAL(mmtime_today_flag)      |
    FLAGVAL(mmtime_todaysep_flag)   |
    FLAGVAL(mmtime_year_time_flag),
    1,NULL
};

PATH mailcap_bl_programs;       /* Do not execute program or call metamail */
int mailcap_select_other = TRUE; /* flag: Show o)ther alternative on 
				    "Mailcap program selection" ? 
				 */


int mailcap_tempfile_lifetime;  /* Time (seconds) to keep mailcap temorary files 
				   init_defaults() changes this if POLL_method
				   is set
				*/

int metoo;			/* flag: copy me on mail to alias?  */
int mime_body_keywords = TRUE;  /* flag: parse body looking for encode */
				/* keywords */

static char * MIMEFORWARD[NUM_mimeforward+1] =  { "off", "on", "auto",NULL };
ENUMERATE mimeforward   = {     /* Forward as Message/rfc822  */
    mimeforward_auto,
    NUM_mimeforward, &(MIMEFORWARD[0]),
    NULL,
    1   	/* allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL    
};
    
int askmimeforward = 1;         /* flag: prompt about messaga/rfc822 forward */
int move_when_paged;	        /* flag: move when '+' or '-' used? */
int names_only;		        /* flag: display user names only?   */

#ifdef REMOTE_MBX
int name_lookup_cache_time = 320;  /*  DEFAULT_CACHE_TIME
				       defaul cache time for mail
				       services (seconds) */

static char * NAME_LOOKUP_CANCEL[NUM_name_lookup_cancel+1] = { "disabled", "enabled", "auto",
							       NULL };

ENUMERATE name_lookup_cancel = { /* Allow Ctrl-C on DNS lookups */
    name_lookup_cancel_auto,
    NUM_name_lookup_cancel, &(NAME_LOOKUP_CANCEL[0]),
    NULL,
    1   	/* allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL    
};
#endif

static char * NOENCODINGS[NUM_allow_no_encoding+1] = { "pass-7bit", "pass-8bit", "pass-binary", 
						       NULL };
ENUMERATE allow_no_encoding = { /* 1: Allow 8bit without -B8BITMIME */
    allow_pass_7bit,            /* 2: Allow binary without -BBINARYMIME and */
    NUM_allow_no_encoding,      /*    and 8bit without -B8BITMIME   */
    &(NOENCODINGS[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

int allow_no_hdrencoding;       /* TRUE, if header encoding is      */
				/* not required                     */
int noheader = 1;		/* flag: copy + header to file?     */
int noheaderfwd;		/* flag: copy + header to file(fwd)? */
int pagealternative=1;          /* Page if alternative have one know*/
                                /* subtype ?                        */
int page_known_charsets=1;      /* Page if charset is not unknown?  */
int pagemultipart;               /* Page unknown mime subparts?      */
int pagesigned;                  /* Page unknown signed protocols?   */

static char * PAGERELATED[NUM_pagerelated+1] = { "no", "yes", "single-part", NULL };
ENUMERATE pagerelated = {
    pagerelated_singlepart,
    NUM_pagerelated, &(PAGERELATED[0]),
    NULL           /* shared       */,
    1              /* have boolean */,
    NULL   	   /* not delayed */,
    NULL           /* value as string */
};

static  struct keyword_val PAGER_KW[] = {
    { "builtin",    pager_kw_builtin,   0 },
    { "builtin+",   pager_kw_builtin,   0 },
    { "builtin++",  pager_kw_builtin,   0 },
    { "internal",   pager_kw_builtin,   0 },
    { "use-$PAGER", pager_KW_PAGER,  1 },    /* bitmask */
    { NULL, pager_kw_NO }
};

/* Returns 1 if success, 0 failure, -1 if expansion is not string */
S_(keyword_expand_func pager_expand_func)
static int pager_expand_func P_((struct dt_estr_info *kw,
				 struct dt_keyword_info *ki,
				 unsigned int keyword,
				 unsigned int continue_mask));


static struct dt_keyword_info pager_keyword_info = {
    pager_expand_func,
    PAGER_KW
};


ESTR pager_e = {                 /* what pager to use                */
    ESTR_none|ESTR_print_expanded,
    NULL,
    NULL,

#ifdef USE_BUILTIN_PAGER  
    pager_kw_builtin,     0,
#else
    0,                    pager_KW_PAGER,
#endif
    & pager_keyword_info,
    0
};

static char * HEADER_DISPLAY_MODES[NUM_header_display_mode+1] = { "plain", "bold", NULL };
ENUMERATE pg_header_display_mode = {
    hdr_display_bold,
    NUM_header_display_mode, &(HEADER_DISPLAY_MODES[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

int pager_indicate_wrapping = 1; /* Indicate wrapping with \ on builtin
				    pager                               */

int pager_paragraph_width = 70;  /* Indicate preferred width for flowed 
				    (word wrapped) paragraph. */

static char *PAGER_TIME_DISPLAY[NUM_pager_time_display+1] = {
    "legacy"			/* (same than month-day-year-12h)   */,
    "default"			/* (same than month-day-year-HH-MM) */,
    "month-day-year-12h"	/* month day, Year 12h (am/pm)
				   strftime %b %d, %Y %r            */,
    "month-day-year-24h"	/* month day, Year 24h
				   strftime %b %d, %Y %T             */,
    "month-day-year-HH-MM"      /* month day, Year HH:MH
				   strftime %b %d, %Y %R             */,
    "preferred"			/* Local preferred format
				   strftime %c                       */,
    "day-month-year-12h"	/* day month year 12h (am/pm)
				   strftime %d %b %Y %r              */,
    "day-month-year-24h"	/* day month year 24h
				   strftime %d %b %Y %T              */,
    "day-month-year-HH-MM"	/* day month year  HH:MM
				   strftime %d %b %Y %R              */,
    "YYYY-MM-DD-HH-MM"          /* numeric YYYY-MM-DD HH:MM          */,
    "YYYY-MM-DD-HH-MM-SS"       /* numeric YYYY-MM-DD HH:MM:SS       */,

    NULL
};
ENUMERATE pager_time_display = {
    ptd_default,
    NUM_pager_time_display, &(PAGER_TIME_DISPLAY[0]),
    NULL,
    0   	/* do not allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL
};


ESTR pgp2_path_e = {            /* Pgp path or "none" if no pgp     */
    ESTR_none,
    NULL,
    NULL,
    0,0,NULL,0
};
ESTR pgp5_dir_e = {   /* pgp5 binary directory or "none" if no pgp5 */
    ESTR_slash|ESTR_none,
    NULL,
    NULL,
    0,0,NULL,0
};
ESTR gpg_path_e  = {
    ESTR_none,
    NULL,
    NULL,
    0,0,NULL,0
};

int pgp_keeppass;             	/* Should Elm keep the passphrase in*/
int pgp_noarmor=1;      	/* Should Elm display text before PGP armor */

int point_to_new = 1;		/* flag: start pointing at new msg? */
int point_to_flagged;           /* flag: start pointing at flagged msg */
char allowed_precedences[SLEN];	/* list of precedences user */
                                /* may specify                      */
ESTR printout_e = {	        /* how to print messages            */
    ESTR_none,
    NULL,
    NULL,
    0,0,NULL,0
};
char printhdrs[SLEN];

static char * PROGRAM_SND_IDENT[NUM_program_snd_ident_v+1] = {
    "X-Mailer", "version",
    "User-Agent", NULL
};
FLAGS program_snd_ident = {    /* program-identification */
#ifndef NO_XHEADER
    FLAGVAL(snd_ident_x_mailer) |
#endif
    FLAGVAL(snd_ident_version),
    0L,0L,
    NUM_program_snd_ident_v,&(PROGRAM_SND_IDENT[0]),
    FLAGVAL(snd_ident_x_mailer) |
    FLAGVAL(snd_ident_version) |
    FLAGVAL(snd_ident_user_agent),
    1,NULL
};


int prompt_after_pager = 1;     /* flag: prompt after pager exits   */
int prompt_after_metamail = 1;  /* flag: prompt after metamail exits     */
int  prompt_metamail = 1;     /* flag: prompt for calling of metamil */
int quote_forward;		/* flag: fwd'd msgs quoted like replies */
struct string * quote_prefix;     /* prefix char(s) for msgs          */
int readdatapercentinc = 5;	/* data percent increment during new mbox read */

ESTR readmsg_e = {	        /* path to readmsg program	*/
    ESTR_meta,
    NULL,
    NULL,
    0,0,NULL,0
};

int readmsginc = 100;		/* increment of msg cnt when reading*/
                                /* new mbox                         */

static  struct keyword_val RECVD_KW[] = {
     { "local-file", FOLDER_is_local_file, 1 },    /* bitmask */
     { NULL,         0, 0  }
};

static struct dt_keyword_info recvd_expand_info = {
    null_keyword_expand,
    RECVD_KW
};

ESTR recvd_mail_e = {	/* folder for storing received mail */
    ESTR_none|ESTR_meta|ESTR_initial,
    default_recvdmail,
    NULL,
    0,0,
    &recvd_expand_info,
    0
};

ESTR remote_signature_e;        /* remote msg signature file    */
int req_mime_bodyencoding = 1;  /*                                  */
int req_mime_hdrencoding;       /*                                  */
int resolve_mode = 1;		/* flag: delete saved mail?	    */
int save_by_name = 1;		/* flag: save mail by login name?   */
int save_by_alias = 1;	/* save mail by alias of login name? */

static  struct keyword_val SENT_KW[] = {
     { "local-file", FOLDER_is_local_file, 1 },    /* bitmask */
     { NULL,         0, 0  }
};

static struct dt_keyword_info sent_expand_info = {
    null_keyword_expand,
    SENT_KW
};

ESTR sent_mail_e  = {	        /* name of file to save copies to   */
    ESTR_none|ESTR_meta|ESTR_initial,
    default_sentmail,
    NULL,
    0,0,
    &sent_expand_info,
    0
};

int send_mime_plain  = 1;       /* flag: send plain text as MIME?   */

ESTR shell_e  = {	        /* current system shell             */
    ESTR_bin,
    NULL,
    NULL,
    0,0,NULL,0
};

int  set_window_title;          /* flag: set (xterm?) window title and 
				   icon name */

int showto;                     /*                                  */
int show_reply = 1;	        /* flag: show 'r' for replied mail  */
int show_mlists;	        /* show mailing list info?          */

static char * HEADER_ERROR_MODES[NUM_show_header_errors+1] = 
    { "off", "on", "store", NULL };
ENUMERATE show_header_errors_e = {
    show_hdr_error_store,
    NUM_show_header_errors, &(HEADER_ERROR_MODES[0]),
    NULL,
    1   	/* allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL
};

static char * MAIL_QUOTA_MODES[NUM_show_header_errors+1] =
    { "off", "on-open", NULL };
ENUMERATE show_mail_quota_e = {
    show_mquota_ON_open,
    NUM_show_header_errors, &(MAIL_QUOTA_MODES[0]),
    NULL,
    1   	/* allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL    
};

int sig_dashes = 1;		/* flag: include dashes above sigs? */
int sleepmsg = 2;		/* time to sleep for messages being */
                                /* overwritten on screen            */

static struct sortval SRT_srtval[]={
    {"sent", SENT_DATE},
    {"thread", THREAD},
    {"received", RECEIVED_DATE},
    {"recieved", RECEIVED_DATE},
    {"rec", RECEIVED_DATE},
    {"from", SENDER},
    {"sender", SENDER},
    {"size", SIZE},
    {"lines", SIZE},
    {"subject", SUBJECT},
    {"mailbox", MAILBOX_ORDER},
    {"folder", MAILBOX_ORDER},
    {"status", STATUS},
    {NULL, 0} 
};

SORT sortby = {                 /* how to sort incoming mail...     */
    REVERSE SENT_DATE,	

    SRT_srtval
};

static char * SPECIAL_USE_DOMAINS[] = {
    /* https://www.iana.org/assignments/special-use-domain-names/special-use-domain-names.xhtml
       Special-Use Domain Names

       List only these which application software should recognize
       as special.

       If domain name is given as ".domain" then it blacklists names under that
       domain. If name is given as "domain" then also "domain" itself (and 
       names under it) are blacklisted. 

    */

    /* "example"             [RFC6761] ; Application software SHOULD NOT 
       recognize example names as special and SHOULD use example names as 
       they would other domain names.*/
    /* "example.com"         [RFC6761] ;  -"- */
    /* "example.net"         [RFC6761] ;  -"- */
    /* "example.org"         [RFC6761] ;  -"- */
    "invalid"           /*   [RFC6761] ; Application software MAY recognize 
       "invalid" names as special or MAY pass them to name resolution APIs as 
       they would for other domain names.*/,
    /* "local"               [RFC6762] ; Application software may use these 
       names as they would other similar DNS names, and is not required to 
       recognize the names and treat them specially.*/
    /* "localhost"           [RFC6761] ;  Application software MAY recognize 
       localhost names as special, or MAY pass them to name resolution APIs 
       as they would for other domain names.*/
    ".onion"            /*   [RFC7686] ; Applications that do not implement 
       the Tor protocol SHOULD generate an error upon the use of .onion and
       SHOULD NOT perform a DNS lookup. */,
    /* "test"                [RFC6761] ;  Application software SHOULD NOT 
       recognize test names as special */
    NULL
};

#define NUM_special_use_domains (sizeof (SPECIAL_USE_DOMAINS) / \
  sizeof(SPECIAL_USE_DOMAINS[0]))-1

PATH special_bl_domains = {     /* "special-use-domains-blacklist" */
    PATH_ascii_only|PATH_initial,NULL,
    NUM_special_use_domains,
    &(SPECIAL_USE_DOMAINS[0])
};

#ifdef REMOTE_MBX
static char * SPECIAL_DOMAINS_LOOKUP[NUM_special_domains_v+1] = {
    "static-hosts", NULL
};
FLAGS special_domains_lookup = {
    FLAGVAL(special_dom_static_hosts_flag),
    0L,0L,
    NUM_special_domains_v,&(SPECIAL_DOMAINS_LOOKUP[0]),
    FLAGVAL(special_dom_static_hosts_flag),
    1,NULL
};
int static_hosts_max_names = 10000  /* threshold for /etc/hosts caching */;
#endif

static struct sortval THREAD_srtval[]={
    {"sent",    SENT_DATE},
    {"subject", SUBJECT},
    {NULL, 0} 
};


SORT thread_sortby = {          /* how to sort threads of messages */
    REVERSE SENT_DATE,	

    THREAD_srtval
} ;      

long elm_timeout = 600L;	/* timeout (secs) on main prompt    */
int sort_thread_max_time = 30;  /* Number of days which can considered 
				    be same thread */
char to_chars[SLEN] = DEFAULT_TO_CHARS;	/* chars to indicate who mail is to */

ESTR temp_dir_e  = {         /* name of temp directory           */
    ESTR_slash,
    NULL,
    NULL,
    0,0,NULL,0
};


int title_messages = 1;		/* flag: title message display?     */


static char * USERLEVELS[NUM_user_level+1] = { "beginner", "intermediate", "expert", NULL };
ENUMERATE user_level = { 	/* flag: how good is the user?      */
    user_level_beginner,
    NUM_user_level, &(USERLEVELS[0]),
    NULL,
    0   	/* no boolean */,
    NULL   	/* not delayed */,
    NULL
};

int use_char_set_field_hack;   /* flag: Recognise non-standard header 
				  field Char-Set */
int use_ct_cs_for_subj_hack;   /* flag: Process 8-bit subject with using 
				  charset parameter from
				  header field Content-Type.
			       */

int use_tite = 1;		/* flag: use termcap/terminfo ti/te?*/

static char * USE_TLS[NUM_use_tls_v+1] = { "implicit-tls",
					   "starttls",
					   "verify-tls-certificate",
					   "require-tls-peer-name",
					   "display-check-host",
					   NULL };

FLAGS use_tls = {
    0L
#if defined(USE_DLOPEN) & defined(REMOTE_MBX)
    | FLAGVAL(use_tls_starttls)
#endif
    /* use_tls_implicit is enabled by use-config-library = tls 
       use_tls_starttls is here default because that was
       default without explicit option earlier. It is only 
       used by use-connect-library = tls
     */

    ,0L,0L,
    NUM_use_tls_v, &(USE_TLS[0]),
    
    0L 
#if defined(USE_DLOPEN) & defined(REMOTE_MBX)
    | FLAGVAL(use_tls_implicit) | FLAGVAL(use_tls_starttls)
    | FLAGVAL(use_tls_verify_cert) | FLAGVAL(use_tls_require_name)
    | FLAGVAL(use_tls_display_host)
#endif

    , 1,NULL
};

int utf7_encode_optional = 1;   /* flag: Should utf7 optional direct 
				   characters to be encoded?        */

static char * VERIFY_DOMAIN[NUM_verify_domain+1] = {
    "no",
#ifdef I_NETDB
    "yes",
    "hostname",
#endif
    "shared",
    NULL
};

int verify_alias_domain=1;         /* flag: Should we try check existence
				      of mail domain from alias 
				      expanded addresses ? */

ENUMERATE verify_domain = {      /* flag: Should we try check existence   */
    verify_domain_no,	         /*	  of mail domain                  */
    NUM_verify_domain,&(VERIFY_DOMAIN[0]),
    NULL,
    1   	/* allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL
};

int verify_local_address;       /* flag: Should we try check existence
				    of local user
				 */

static char * VERIFY_LDOMAIN[NUM_verify_ldomain+1] = {
    "no",
    "yes",
    "if-remote-mailer",
    NULL
};

ENUMERATE verify_local_domain = {  /* flag: Should we try check existence of */
    verify_ldomain_remote_mailer,  /*       "mailname" or "hostfullname" ?   */
    NUM_verify_ldomain, &(VERIFY_LDOMAIN[0]),
    NULL,
    1   	/* allow boolean ON, OFF, TRUE, FALSE, YES and NO */,
    NULL   	/* not delayed */,
    NULL    
};

char v_editor[SLEN];	        /* "~v" editor...                   */
int elm_filter = 1;		/* flag: weed out header lines?	    */
PATH weedlist = {
    PATH_sep_comma|PATH_quote|PATH_convert_underline,
    NULL, 0,NULL
};
PATH weedlist_no_title = {
    PATH_sep_comma|PATH_quote|PATH_convert_underline,
    NULL, 0,NULL
};


#ifdef REMOTE_MBX
static struct sortval IMAP_srtval[]={
    {"none", ID_NONE},
    {"name", ID_NAME_SORT},
    {NULL, 0} 
};

SORT imap_dir_sortby = {            /* how to sort imap folders...  */
    ID_NONE,	           
    IMAP_srtval
};

char raw_imap_charset[SLEN] = "SYSTEM";  /* IMAP foldername charset */
int imap_fast_lookup;                /* flag: skip some directory listing */
int imap_idle_alive_interval = 25;  /* Idle timeout on seconds for IMAP connection;
				       NOOP command interval
				  */
int imap_max_dl_size = 4*1024;      /* Maximum size of message to be
				     * downloaded 
				     */
charset_t imap_charset = NULL;           /* IMAP foldername charset */
int IMAP_connection_cache = 1; /* flag: call cache_connection() ? */
int IMAP_name_convention = 1; /* flag: Should imap intearnational folder
				 convention be used? */
int IMAP_use_examine   = 1;  /* flag: Should EXAMINE command to be used?    */
int IMAP_show_greeting = 1;  /* flag: Should untagged OK messages be shown? */
int IMAP_show_warning  = 1;  /* flag: Should untagged NO messages be shown? */
int IMAP_show_error    = 1;  /* flag: Should untagged BAD messages be shown? */
int pop_idle_alive_interval = 25;  /* Idle timeout on seconds for POP connection;
				      NOOP command interval
				  */
int pop_max_dl_size = 4*1024;      /* Maximum size of message to be
				     * downloaded 
				     */
int POP_show_greeting  = 1;  /* flag: Should POP server greeting be shown?  */
#endif

/* ---------------------------------------------------------------- */

static void do_expand_env P_((char  *descr, char *dest, char *src,
			      unsigned	destlen));
static void do_expand_env(descr, dest, src, destlen)
     char	*descr, *dest, *src;
     unsigned	destlen;
{
    if (expand_env(dest, src, destlen) != 0) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCannotInitErrorExpanding,
			  "Cannot initialize \"%s\" - error expanding \"%s\"."),
		  descr, src);
	errors++;
	dest[0] = '\0';
    }
}

static void do_expand_meta P_((char  *descr, char *dest, char *src,
			      unsigned	destlen));
static void do_expand_meta(descr, dest, src, destlen)
    char	*descr, *dest, *src;
    unsigned	destlen;
{
    if (expand_meta(dest, src, destlen) != 0) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCannotInitErrorExpanding,
			  "Cannot initialize \"%s\" - error expanding \"%s\"."),
		  descr, src);
	errors++;
	dest[0] = '\0';
    }
}

static void expand_map_bin P_((char  *descr, char *dest, char *src,
			       unsigned	destlen,
			       int global_mode_or_read_flags));
static void expand_map_bin(descr, dest, src, destlen,
			   global_mode_or_read_flags)
    char	*descr, *dest, *src;
    unsigned	destlen;
    int global_mode_or_read_flags;
{
    if (strcmp(src,"none") == 0 || src[0] == '\0') {
	
	strfcpy(dest,"none",destlen);
	
    } else if (NULL == strpbrk(src,"/$\\;{}()")) {

	/* give_dt_estr_as_str adds / to end */
	const char * map_bindir_val = give_dt_estr_as_str(&map_bindir_e,
							  "map-bin-dir",NULL,NULL);

	if (!map_bindir_val) {
	    dest[0] = '\0';
	    return;
	}

	if (0 != access(map_bindir_val,ACCESS_EXISTS)) {
	    int err = errno;

	    if (ENOENT == err &&
		0 != (global_mode_or_read_flags & READ_FLAG_IGNORE_MISSING)) {
	    
		DPRINT(Debug,2,(&Debug,
				"Ignoring missing %s\n",
				map_bindir_val));
		
	    } else {
		lib_error(CATGETS(elm_msg_cat, ElmSet,ElmNoBinMapDirX,
				  "%s: no map-bin-dir exist: %s: %s"),
			      descr,
			  map_bindir_val,strerror(err));
		
		/* No error counter incrementing because 
		   map-bin-dir can be modified only on system 
		   elm.rc
		*/
	    }
	}
	elm_sfprintf(dest,destlen,
		     FRM("%s/%s"),
		     map_bindir_val,src);
	
    } else
	do_expand_meta(descr, dest, src, destlen);


    DPRINT(Debug,10,(&Debug, "%s expanded: %s => %s (destlen %d)\n",
		     descr,src,dest,destlen));


}

/* sets locale and elm_msg_cat */
static int locale_init_called = 0;
static int locale_error = 0;


void locale_init () {
#ifdef I_LOCALE
    char *res;
#endif
    
    DPRINT(Debug,9,(&Debug, 
		    "locale_init: called = %d\n",
		    locale_init_called));
    
#ifdef I_LOCALE

    if (setlocale(LC_ALL, "") == NULL) {
	lib_error(FRM("Elm: Unsupported locale (check $LANG)\n"));
	if (setlocale(LC_CTYPE,"") != NULL) {
	    lib_error(FRM("Elm: ... but succeed setting of LC_CTYPE\n"));
	} else {
	    lib_error(FRM("Elm: ... check also $LC_CTYPE\n"));
	}
	locale_error = 1;
    }
#endif
  
    /* WARNING: Elm's version uses 0 as failure
     *  System uses (nl_catd) -1 as failure !
     */


    elm_msg_cat       = catopen("elmme+2.5", 0);
    if (0            == elm_msg_cat ||
	(nl_catd) -1 == elm_msg_cat) {
	DPRINT(Debug,9,(&Debug, 
			"locale_init: catopen elmme+2.5 failed\n"));
    }
    /* NOTE:    elm_msg_cat    'leaks' on end of program   ... */

#ifdef  I_LOCALE
    res = setlocale(LC_CTYPE,NULL);
    if (!res) {
	lib_error(FRM("Elm: Unable to get current locale (LC_CTYPE)\n"));
	locale_error = 1;
    } else
	strfcpy(display_locale,res, sizeof display_locale);
#endif
  
#ifdef HAVE_CODESET
    res = nl_langinfo(CODESET);

  if (!res) {
      lib_error(FRM("Elm: Unable to get CODESET for current locale\n"));
      locale_error = 1;
  } else	  
      strfcpy(display_codeset,res, sizeof display_codeset);
#endif

  locale_init_called++;
}

enum matches { match_exact = 0, 
	       match_codeset,
	       match_charset, 
	       match_lang_country, match_lang, MATCH_COUNT };

struct charset_map_item  * system_charset_map = NULL;
struct charset_map_item  * user_charset_map   = NULL;

static int  map_system_charset_called = 0;  /* HACK */

static void map_system_charset P_((const char * lc_ctype, 
				   const char * codeset,
				   int global_only));
static void map_system_charset(lc_ctype, codeset,global_only)
     const char * lc_ctype;
     const char * codeset;
     int global_only;
{  
    const char *lang = NULL;
    const char *lang_country = NULL;
    const char *charset = NULL;
    charset_t d_matches[MATCH_COUNT];
    int i,j;

    char * default_system_charset = DEFAULT_SYSTEM_CHARSET;
    char * c1;

    char lbuf[3], cbuf[6];

    DPRINT(Debug,3,(&Debug,
		    "map_system_charset: global_only=%d\n",global_only));

    map_system_charset_called = 1;

    if (!getenv("NOMETAMAIL") && (c1 = getenv("MM_CHARSET")) &&
	c1[0]) {

	default_system_charset  = c1;
	DPRINT(Debug,9,(&Debug, 
			"map_system_charset: using $MM_CHARSET (%s) as default system charset\n",
			default_system_charset));
    }

    for (i = 0; i < MATCH_COUNT; i++)
	d_matches[i] = NULL;

    if (isascii(lc_ctype[0]) && isalpha(lc_ctype[0]) &&
	isascii(lc_ctype[1]) && isalpha(lc_ctype[1]) &&
	('.' == lc_ctype[2] || '_' == lc_ctype[2] || '\0' == lc_ctype[2])) {
	strncpy(lbuf,lc_ctype,3);
	lbuf[2] = '\0';
	lang = lbuf;
	
	if ('_' == lc_ctype[2]) {
	    if (isascii(lc_ctype[3]) && isalpha(lc_ctype[3]) &&
		isascii(lc_ctype[4]) && isalpha(lc_ctype[4]) &&
		('.' == lc_ctype[5] || '\0' == lc_ctype[5])) {

		strncpy(cbuf,lc_ctype,6);
		cbuf[5] = '\0';
		lang_country = cbuf;
		
		if ('.' == lc_ctype[5])
		    charset = lc_ctype+6;
	    }
	} else if ('.' == lc_ctype[2])
	    charset = lc_ctype+3;      
    }
    
    if (codeset[0]) {
	DPRINT(Debug,10,(&Debug,
			 "map_system_charset: codeset: %s\n", 
			 codeset));
	d_matches[match_codeset] = codeset_name_to_charset(codeset);
	if (!d_matches[match_codeset]) {
	    DPRINT(Debug,1,(&Debug,
			    "map_system_charset: codeset name %s unknown!\n",codeset)); 
	}
    }

    if (lang) {
	DPRINT(Debug,10,(&Debug,
			 "map_system_charset: %s -> lang: %s\n", 
			 lc_ctype,lang));
    } 
    if (lang_country) {
	DPRINT(Debug,10,(&Debug,
			 "map_system_charset: %s -> lang_country: %s\n", 
			 lc_ctype,lang_country));
    }
    if (charset) {
	DPRINT(Debug,10,(&Debug,
			 "map_system_charset: %s -> charset: %s\n", 
			 lc_ctype,charset));
    }

    for (i = global_only ? 1 : 0; i < 3; i++) {
	if ( i < 2) {
	    int k;
	    struct charset_map_item * F = user_charset_map;
	    if (1 == i)
		F = system_charset_map;

	    if (!F)
		continue;

	    for (k = 0; F[k].charset; k++) {
		if (F[k].match) {
		    if (istrcmp (F[k].match, lc_ctype) == 0) {
			d_matches[match_exact] = F[k].charset;
			
			DPRINT(Debug,3,(&Debug,
					"map_system_charset: user defined \"%s\" as %s\n", 
					lc_ctype, 
					d_matches[match_exact]->MIME_name ?
					d_matches[match_exact]->MIME_name :
					"<no MIME name>"));
		    }
		    if (charset && strncmp(F[k].match,"*.",2) == 0  &&
			istrcmp (F[k].match+2, charset) == 0) {
			
			d_matches[match_charset] = F[k].charset;
			
			DPRINT(Debug,3,(&Debug,
					"map_system_charset: user defined \"%s\" (*.%s) as %s\n", 
					lc_ctype, charset,
					d_matches[match_charset]->MIME_name ?
					d_matches[match_charset]->MIME_name :
					"<no MIME name>"));	      
		    }
		    if (lang_country && istrcmp (F[k].match, lang_country) == 0) {
			d_matches[match_lang_country] = F[k].charset;

			DPRINT(Debug,3,(&Debug,
					"map_system_charset: user defined \"%s\" (%s) as %s\n", 
					lc_ctype, lang_country, 
					d_matches[match_lang_country]->MIME_name ?
					d_matches[match_lang_country]->MIME_name :
					"<no MIME name>")); 
			
		    }
		    
		    if (lang_country && istrcmp (F[k].match, lang) == 0) {
			d_matches[match_lang] = F[k].charset;
			
			DPRINT(Debug,3,(&Debug,
					"map_system_charset: user defined \"%s\" (%s) as %s\n", 
					lc_ctype, lang, 
					d_matches[match_lang]->MIME_name ?
					d_matches[match_lang]->MIME_name :
					"<no MIME name>"));
		    }
		}
	    }

	    for (j = 0; j < MATCH_COUNT; j++)
		if (d_matches[j]) {
		    system_charset = d_matches[j];
		  
		    DPRINT(Debug,3,(&Debug,
				    "map_system_charset: using match %d: \"%s\" (codeset \"%s\") as %s\n", 
				    j,lc_ctype,codeset,
				    system_charset->MIME_name ?
				    system_charset->MIME_name :
				    "<no MIME name>"));
		    return;
		}
	} else {
	    int val;
	    char buffer[STRING];
	
	    if (d_matches[match_codeset]) {
		system_charset = d_matches[match_codeset];
		DPRINT(Debug,3,(&Debug,
				"map_system_charset: using match codeset: \"%s\" (codeset \"%s\") as %s\n", 
				lc_ctype,codeset,
				system_charset->MIME_name ?
				system_charset->MIME_name :
				"<no MIME name>"));
		return;
	    }
  
	    /*  1) ------ guess based on codeset ------------------------ */

	    if (codeset[0] &&
		     0 == strincmp(codeset,"ISO-8859-",9) &&
		     0 < (val = atoi(codeset+9))) {
		elm_sfprintf(buffer, sizeof buffer,
			     FRM("ISO-8859-%d"),
			     val);
		system_charset = MIME_name_to_charset(buffer,
						      CHARSET_create);
	    } else if (codeset[0] &&
		       0 == strincmp(codeset,"ISO_8859-",9) &&
		       0 < (val = atoi(codeset+9))) {
		elm_sfprintf(buffer, sizeof buffer,
			     FRM("ISO-8859-%d"),
			     val);
		system_charset = MIME_name_to_charset(buffer,
						      CHARSET_create);
	    } else if (codeset[0] &&
		       0 == strincmp(codeset,"ISO8859-",8) &&
		       0 < (val = atoi(codeset+8))) {
		elm_sfprintf(buffer, sizeof buffer,
			     FRM("ISO-8859-%d"),
			     val);
		system_charset = MIME_name_to_charset(buffer,
						      CHARSET_create);
	    } else if (codeset[0] &&
		       0 == strincmp(codeset,"ISO8859",7) &&
		       isascii(codeset[7]) && isdigit(codeset[7]) && 
		       '\0' == codeset[8]) {
		elm_sfprintf(buffer, sizeof buffer,
			     FRM("ISO-8859-%c"),
			     codeset[7]);
		system_charset = MIME_name_to_charset(buffer,
						      CHARSET_create);

	    /*  2) ------ guess based on lc_ctype ------------------------ */

	    } else if (0 == strcmp(lc_ctype,"C"))
		system_charset = MIME_name_to_charset("US-ASCII",
						      CHARSET_create); 
	    else if (0 == istrcmp(lc_ctype,"ASCII") ||
		     0 == istrcmp(lc_ctype,"US-ASCII"))
		system_charset = MIME_name_to_charset("US_ASCII",
						      CHARSET_create);
	    else if (0 == strincmp(lc_ctype,"ISO-8859-",9) &&
		     0 < (val = atoi(lc_ctype+9))) {
		elm_sfprintf(buffer, sizeof buffer,
			     FRM("ISO-8859-%d"),
			     val);
		system_charset = MIME_name_to_charset(buffer,CHARSET_create);
	    } else if (0 == strincmp(lc_ctype,"ISO_8859-",9) &&
		       0 < (val = atoi(lc_ctype+9))) {
		elm_sfprintf(buffer, sizeof buffer,
			     FRM("ISO-8859-%d"),
			     val);
		system_charset = MIME_name_to_charset(buffer,CHARSET_create);
	    } else if (0 == strincmp(lc_ctype,"ISO8859-",8) &&
		       0 < (val = atoi(lc_ctype+8))) {
		elm_sfprintf(buffer, sizeof buffer,
			     FRM("ISO-8859-%d"),
			     val);
		system_charset = MIME_name_to_charset(buffer,CHARSET_create);

	    /*  3) ------ guess based on charset part of locale ------------------------ */

	    } else if (charset && (0 == istrcmp(charset,"ASCII") ||
				   0 == istrcmp(charset,"US-ASCII")))
		system_charset = MIME_name_to_charset("US-ASCII",
						      CHARSET_create);
	    else if (charset &&
		     0 == strincmp(charset,"ISO-8859-",9) &&
		     0 < (val = atoi(charset+9))) {
		elm_sfprintf(buffer, sizeof buffer,
			     FRM("ISO-8859-%d"),
			     val);
		system_charset = MIME_name_to_charset(buffer,
						      CHARSET_create);
	    } else if (charset &&
		       0 == strincmp(charset,"ISO_8859-",9) &&
		       0 < (val = atoi(charset+9))) {
		elm_sfprintf(buffer, sizeof buffer,
			     FRM("ISO-8859-%d"),
			     val);
		system_charset = MIME_name_to_charset(buffer,
						      CHARSET_create);
	    } else if (charset &&
		       0 == strincmp(charset,"ISO8859-",8) &&
		       0 < (val = atoi(charset+8))) {
		elm_sfprintf(buffer, sizeof buffer,
			     FRM("ISO-8859-%d"),
			     val);
		system_charset = MIME_name_to_charset(buffer,
						      CHARSET_create);
	    } else if (charset &&
		       0 == strincmp(charset,"ISO8859",7) &&
		       isascii(charset[7]) && isdigit(charset[7]) && 
		       '\0' == charset[8]) {
		elm_sfprintf(buffer, sizeof buffer,
			     FRM("ISO-8859-%c"),
			     charset[7]);
		system_charset = MIME_name_to_charset(buffer,
						      CHARSET_create);
	    } else {
	    
		if (global_only) {

		    DPRINT(Debug,3,(&Debug,
				    "map_system_charset: global_only=%d: Failed silently\n",
				    global_only));

		} else {
  
		    lib_error(FRM("Elm: Unable to map %.20s locale (LC_CTYPE) to MIME charset."),
			      lc_ctype);
		    
		    if (codeset[0]) {
			system_charset = MIME_name_to_charset(codeset,
							      CHARSET_create);
			lib_error(FRM("     Codeset name %s was unknown. Treating %s as MIME name."),
				  codeset,codeset);
			lib_error(FRM("     Check %.100s or %.100s"),
				  system_mime_charsets, user_mime_charsets);
			
		    } else {
			system_charset = MIME_name_to_charset(default_system_charset,
							      CHARSET_create);
			lib_error(FRM("     Using default (%.20s). Check %.100s or\n      %.100s"),
				  default_system_charset,system_mime_charsets,
				  user_mime_charsets);
		    }
		    locale_error = 1;
		}
	    }

	    DPRINT(Debug,3,(&Debug,
			    "map_system_charset: default \"%s\" as \"%s\"\n", 
			    lc_ctype, system_charset->MIME_name));
	}
    }
}

static int user_rc_set = 0;
/* shell value from passwd */
static char shell0[STRING] = "";


void user_init() 
{
    char *cp;
    long rl; /* sysconf() returns long */
    
    /** initialize the whole ball of wax.
     **/
    struct passwd *pass;
    
    if (!locale_init_called)
	locale_init ();
    
    /* save original user and group ids */
    userid  = getuid();
    groupid = getgid();	
    
    /* GNU library seems give _SC_SAVED_IDS as enumerated
       value instead of #define so we can not test if
       it is defined (as macro)
    */
#if defined(_SC_SAVED_IDS) || defined(_POSIX_VERSION)
    
    errno = 0;  /* sysyconf may return -1 also
		   when it is not error 
		*/
    
    rl = sysconf(_SC_SAVED_IDS);
    switch (rl) {
	int err;
    case -1:
	err = errno;

	if (err) {
	    DPRINT(Debug,1,(&Debug,
			    "user_init: sysconf(_SC_SAVED_IDS) failed: (errno=%d) %s\n",
			    err,strerror(err)));	
	    
	    have_saved_ids = 0;
	    break;
	}
	/* FALLTHRU */
	
    default:
	DPRINT(Debug,20,(&Debug,"user_init: sysconf(_SC_SAVED_IDS) returns %ld\n",
			 rl));
	have_saved_ids = rl > 0;
	DPRINT(Debug,1,(&Debug,
			"*** have_saved_ids=%d\n",have_saved_ids));
	break;
    }
#else
    have_saved_ids = 0;
#endif
    mailgroupid = getegid();
    
    /* Get username (logname), home (login directory), and full_username
     * (part of GCOS) field from the password entry for this user id.
     * Full_username will get overridden by fullname in elmrc, if defined.
     *
     * For those sites that have various user names with the same user
     * ID, use the passwd entry corresponding to the user name as long 
     * as it matches the user ID.  Otherwise fall back on the entry 
     * associated with the user ID alone.
     */

    if((cp = getenv("LOGNAME")) == NULL)
	cp = getenv("USER");

    errno = 0;             /* Errno is not necessary set on error */

    if(cp != NULL && (pass = getpwnam(cp)) != NULL &&
       pass->pw_uid == userid) {

	DPRINT(Debug,1,(&Debug,"Found username %s ($LOGNAME or $USER)\n",
			cp));

    } else if((pass = getpwuid(userid)) == NULL) {
	int err = errno;

	if (err) 
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmPasswordEntryError,
			      "Failed to get your password entry: %s"),
		      strerror(err));
	else	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmYouHaveNoPasswordEntry,
			      "You have no password entry!"));

	exit(1);
    }

    strfcpy(username, pass->pw_name, sizeof username);

    if (pass->pw_dir)
	strfcpy(home, pass->pw_dir, sizeof home);
    else if (NULL != (cp = getenv("HOME")) &&
	     '/' == cp[0])
	strfcpy(home,cp,sizeof home);
    else
	strfcpy(home,"/",sizeof home);
    
    if (pass->pw_shell)
	strfcpy(shell0, pass->pw_shell, sizeof shell0);

    /* Null shell in /etc/passwd means /bin/sh */
    if ('\0' == shell0[0]) 
	strfcpy(shell0, "/bin/sh", sizeof shell0);

    /* utils/from.c calls this again for resetting */
    if (full_username_s) 
	free_string(&full_username_s);

    if((cp = get_fullname1(pass, username)) != NULL)
	full_username_s = new_string2(system_charset,s2us(cp));
    else
	full_username_s = new_string2(system_charset,s2us(username));
    /* fall back on logname */


    if (!user_rc_set)
	elm_sfprintf(user_rc_file, sizeof user_rc_file,
		     FRM("%s/%s"), home, ELMRCFILE);

    elm_sfprintf(old_user_text_file, sizeof old_user_text_file,
		 FRM("%s/%s"), home, OLD_USER_ALIAS_TEXT);
    elm_sfprintf(old_user_data_file, sizeof old_user_data_file,
		 FRM("%s/%s"), home, OLD_USER_ALIAS_DATA);
    elm_sfprintf(user_aliases_file, sizeof user_aliases_file,
		 FRM("%s/%s"), home, USER_ALIASES_FILE);

    elm_sfprintf(user_mime_types, sizeof user_mime_types,
		 FRM("%s/%s"), home, USER_MIME_TYPES);
    elm_sfprintf(user_mailheaders, sizeof user_mailheaders,
		 FRM("%s/%s"), home, MAILHEADERS);
    elm_sfprintf(user_mime_charsets, sizeof user_mime_charsets,
		 FRM("%s/%s"), home, USER_MIME_CHARSETS);
    elm_sfprintf(user_iso2022_sets, sizeof user_iso2022_sets,
		 FRM("%s/%s"), home, USER_ISO2022_SETS);
    elm_sfprintf(user_terminal_info, sizeof user_terminal_info,
		 FRM("%s/%s"), home, USER_TERMINAL_INFO);
    elm_sfprintf(user_mail_services, sizeof user_mail_services,
		 FRM("%s/%s"), home, USER_MAIL_SERVICES);
    elm_sfprintf(user_mlist_file, sizeof user_mlist_file,
		 FRM("%s/%s"), home, USER_MLIST_FILE);
    elm_sfprintf(user_hash_marks, sizeof user_hash_marks,
		 FRM("%s/%s"), home, USER_HASHMARK_FILE);
    elm_sfprintf(user_tagfilter_entfile, sizeof user_tagfilter_entfile,
		 FRM("%s/%s"), home, USER_TAGFILTER_ENTFILE);
    
    elm_sfprintf(user_last_read_def,sizeof user_last_read_def,
		 FRM("%s/%s"), home, USER_LAST_READ);
    elm_sfprintf(user_last_read_dot,sizeof user_last_read_dot,
		 FRM(".elm-last-read-%d"),userid);
    
#if DEBUG    
    /* Open debug files and print spooled debug output */
    debug_user_init();
   
    DPRINT(Debug,1,(&Debug,"  username = %s\n",username));
    DPRINT(Debug,1,(&Debug,"  home     = %s\n",home));
    if (full_username_s) {
	DPRINT(Debug,1,(&Debug,"  fullname = %S\n",full_username_s));
    }

#endif

}

void set_user_rc_location(p)
     char *p;
{
    strfcpy(user_rc_file, p, sizeof user_rc_file);
    user_rc_set = 1;
}


#ifdef I_NETINET_IN
#include <netinet/in.h>
#endif
#ifdef I_ARPA_INET
#include <arpa/inet.h>
#endif

#ifdef REMOTE_MBX
/* Seems that h_errno is macro on AIX */
#ifndef h_errno
extern int h_errno;
#endif

static char * SOCKADDR_ptr_to_mail_domain P_((const union SOCKADDR_ptr address));
static char * SOCKADDR_ptr_to_mail_domain(address)
     const union SOCKADDR_ptr address;
{
    switch (address.sa->sa_family) {
#if defined(I_NETINET_IN) && defined(I_ARPA_INET)
    case AF_INET: {
	const char * s;
	
#ifdef USE_INET_XTOX_PN
	char mybuffer[INET_ADDRSTRLEN];

	/* inet_pton() and inet_ntop() are available */

	s = inet_ntop(address.sa->sa_family,
		      &(address.sin->sin_addr),
		      mybuffer,sizeof mybuffer);

	if (!s) {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,12,(&Debug,
			     "SOCKADDR_ptr_to_mail_domain: inet_ntop failed (errno=%d) %s\n",
			     err,strerror(err)));
	    
	    return NULL;
	}
#else
	s = inet_ntoa(address.sin->sin_addr);

	if (!s) {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,12,(&Debug,
			     "SOCKADDR_ptr_to_mail_domain: inet_ntoa failed (errno=%d) %s\n",
			     err,strerror(err)));
	    
	    return NULL;
	}	

	DPRINT(Debug,12,(&Debug,
			 "SOCKADDR_ptr_to_mail_domain: inet_ntoa: %s\n",
			 s));
	
#endif
	
	return  elm_message(FRM("[%s]"),s);

    }
	break;

#if defined(HAVE_IN6) && defined (USE_INET_XTOX_PN)
    case AF_INET6: {
	const char * s;
	char mybuffer[INET6_ADDRSTRLEN]; /* no scope */

#ifdef HAVE_SCOPE
	if (address.sin6->sin6_scope_id) {
	    DPRINT(Debug,12,(&Debug,
			     "SOCKADDR_ptr_to_mail_domain: IPv6 address with non-zero scope id ignored.\n"));
	    return NULL;
	}
#endif	

	s = inet_ntop(address.sa->sa_family,
		      &(address.sin6->sin6_addr),
		      mybuffer,sizeof mybuffer);


	if (!s) {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,12,(&Debug,
			     "SOCKADDR_ptr_to_mail_domain: inet_ntop failed (errno=%d) %s\n",
			     err,strerror(err)));
	    
	    return NULL;
	}

	/* See RFC 5321 for IPv6 address literal on email */
	
	return elm_message(FRM("[IPv6:%s]"),s);

    }
	break;
#endif
#endif
	
    }

    DPRINT(Debug,12,(&Debug,
		     "SOCKADDR_ptr_to_mail_domain: Unsupported address family %d\n",
		     address.sa->sa_family));

    return NULL;
}

#endif

static char **name_buffer = NULL;
static int  name_buffer_count = 0;

static int lookup_domain_name P_((char * name,const char ***haliases));
static int lookup_domain_name(name,haliases)
     char * name;
     const char ***haliases;
{
    if (haliases && *haliases)
	(*haliases)[0] = NULL;
   
#ifdef REMOTE_MBX
    
    if (name) {
	struct hostent *he = gethostbyname(name);
	if (!he) {
	    DPRINT(Debug,3,(&Debug,
			    "lookup_domain_name: %s: h_errno=%d\n",
			    name,h_errno));
	    switch(h_errno) {
	    case TRY_AGAIN:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeTemporary,
				  "Address for %s is not yet found..."),
			  name);
	    }
	} else {
	    int ok = 0;
	    int idx;
	    char *p;

	    int count = 0;


	    if (NULL != (p = 
			 strchr(he->h_name ,'.'))) {
		if (!hostdomain[0])
		    strfcpy(hostdomain,p,sizeof hostdomain);
		if (!hostfullname[0]) 
		    strfcpy(hostfullname, he->h_name, sizeof hostfullname);
		ok = 1;

	    } else if (!hostname[0]) {
		strfcpy(hostname, he->h_name, sizeof hostname);
		DPRINT(Debug,3,(&Debug,
				"lookup_domain_name: \"%s\" -> hostname=\"%s\"\n",
				name,hostdomain,hostname));
	    }

	    if (haliases) {
		*haliases = safe_array_realloc(*haliases,
					       2,sizeof ((*haliases)[0]));

		(*haliases)[0] = he->h_name;
		count = 1;
	    }

	    for (idx = 0; he->h_aliases[idx]; idx++) {
		if (!ok &&
		    NULL != (p = 
			     strchr(he->h_aliases[idx],'.'))) {
		    if (!hostdomain[0])
			strfcpy(hostdomain,p,sizeof hostdomain);
		    if (!hostfullname[0]) 
			strfcpy(hostfullname, he->h_aliases[idx], 
				sizeof hostfullname);
		    ok = 1;
		} 

		if (haliases) {
		    *haliases = safe_array_realloc(*haliases,
						   ( count + 2), sizeof ((*haliases)[0]));
		    
		    (*haliases)[count] = he->h_aliases[idx];
		    count++;
		    
		}
	    }
	    if (ok) {
		DPRINT(Debug,3,(&Debug,
				"lookup_domain_name: \"%s\" -> hostdomain=\"%s\", hostfullname=\"%s\"\n",
				name,hostdomain,hostfullname));
	    }

	    if (haliases) {
		for (idx = 0; he->h_addr_list[idx]; idx++) {
	
		    size_t             addrsize = 0;
		    union SOCKADDR_ptr addr;
		    addr.dummy = NULL;
		    
		    if (set_SOCKADDR_from_data(&addrsize,
					       &addr,
					       he->h_addrtype,
					       he->h_length,
					       he->h_addr_list[idx])) {
			
			char * mail_domain = SOCKADDR_ptr_to_mail_domain(addr);

			if (mail_domain) {
			    
			    name_buffer =  safe_array_realloc(name_buffer,
							      ( name_buffer_count + 1),
							      sizeof (name_buffer[0]));
			    
			    name_buffer[name_buffer_count] = mail_domain;

			    *haliases = safe_array_realloc(*haliases,
							   ( count + 2), sizeof ((*haliases)[0]));
			    
			    (*haliases)[count] = 
				name_buffer[name_buffer_count++];
			    count++;
			    
			} else {
			    DPRINT(Debug,12,(&Debug,
					     "lookup_domain_name: \"%s\" %d: unsupported address\n",
					     name,idx));
			}


		    } else {
		
			DPRINT(Debug,12,(&Debug,
					 "lookup_domain_name: \"%s\": Name %s have odd type address (family %d, length %d)",
					 name,he->h_name,he->h_addrtype,he->h_length));

			if (addr.dummy)
			    free(addr.dummy);
			break;
		    }
			    
		    if (addr.dummy)
			free(addr.dummy);
		}
	    }

	    if (haliases && *haliases ) {
		int i;

		(*haliases)[count] = NULL;
		
		for (i = 0; i < count; i++) {
		    DPRINT(Debug,7,(&Debug, 
				    "lookup_domain_name: \"%s\" -> host alias[%d]=\"%s\"\n",
				    name,i,(*haliases)[i]));
		}
	    }

	    DPRINT(Debug,7,(&Debug,
			    "lookup_domain_name=%d\n",
			    ok));

	    return ok;
	}
    }
#endif    

    return 0;
}

static void generate_hostfullname P_((int global_mode));
static void generate_hostfullname(global_mode) 
     int global_mode;
{
    int hostlen, domlen;
    int error = 0;

    /*
     * now the tough part:
     *	we need to make three variables out of this stuff:
     *	hostname = just the hostname, as in bangpaths,
     *		this is whatever the user gave us so far,
     *		we wont change this one
     *	hostdomain = this is the domain considered local to this
     *		machine, and should be what we got above.
     *	hostfullname = this is the full FQDN of this machine,
     *		and is a strange combination of the first two.
     *	if tail(hostname) == hostdomain
     *		then hostfullname = hostname
     *			ie: node.ld.domain.type, ld.domain.type -> node.ld.domain.type
     *	else if hostname == hostdomain + 1
     *		then hostfullname = hostname
     *			ie: domain.type, .domain.type -> domain.type
     *	
     *	else hostfullname = hostname + hostdomain
     *			ie: host, .domain.type -> host.domain.type
     * lost yet?
     */
    
    DPRINT(Debug,3,(&Debug,
		    "hostname=\"%s\",hostdomain=\"%s\" ",
		    hostname,hostfullname));
    
    hostlen = strlen(hostname);
    domlen = strlen(hostdomain);
    if (hostlen >= domlen) {
	if (istrcmp(&hostname[hostlen - domlen], hostdomain) == 0)
	    strfcpy(hostfullname, hostname, sizeof hostfullname);
	else if (hostlen + domlen < sizeof hostfullname) {
	    strfcpy(hostfullname, hostname, sizeof hostfullname);
	    strfcat(hostfullname, hostdomain, sizeof hostfullname);
	} else {	       
	    strfcpy(hostfullname, hostname, sizeof hostfullname);

	    error = 1;
	}
    } else {
	if (istrcmp(hostname, hostdomain + 1) == 0)
	    strfcpy(hostfullname, hostname, sizeof hostfullname);
	else if (hostlen + domlen < sizeof hostfullname) {
	    strfcpy(hostfullname, hostname, sizeof hostfullname);
	    strfcat(hostfullname, hostdomain, sizeof hostfullname);
	} else {
	    strfcpy(hostfullname, hostname, sizeof hostfullname);

	    error = 1;
	}
    }
    
    DPRINT(Debug,3,(&Debug,
		    "-> hostfullname=\"%s\"\n",
		    hostfullname));

    if (error && (0 != (global_mode & READ_FLAG_GLOBAL_ONLY) || 
		  0 == access(hostdomfile,WRITE_ACCESS))) {
	
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmFullHostNameTooLong,
			  "Full hostname \"%s%s\" too long (%d, max %d), using: %s"),
		  hostname,hostdomain,
		  hostlen + domlen,
		  sizeof hostfullname -1,
		  hostfullname);
    } 
}

static int init_defaults_called = 0;

static init_default_hook **hooklist = NULL;
static int                hooklist_len = 0;

void add_init_default_hook(hook)
     init_default_hook *hook;
{
    
    DPRINT(Debug,3,(&Debug,"add_init_default_hook: -- procedure %p\n",hook));

    if (init_defaults_called) {
	DPRINT(Debug,3,(&Debug,"add_init_default_hook: executing immediately -- %p\n",
			hook));

	hook(&errors);
    }

    hooklist = safe_array_realloc(hooklist,(hooklist_len+1), sizeof(hooklist[0]));

    hooklist[hooklist_len] = hook;

    DPRINT(Debug,3,(&Debug,"add_init_default_hook: stored hooklist entry #%d  -- %p\n",
		    hooklist_len,hook));

    hooklist_len++;
}

static unsigned long name_resolution_val = 0L;

int have_name_resolution(tag)
     enum name_resolution tag;
{
    if (tag < 0 || tag >= nr_COUNT)
	panic("RC PANIC",__FILE__,__LINE__,"have_name_resolution",
	      "Bad tag", 0);
    
    return 0 != ( name_resolution_val & ( 1 << tag ));
}

#if ANSI_C
struct name_resolution_rec;
#endif
typedef void name_resolution_init_f P_((struct name_resolution_rec *rec,
					enum name_resolution tag));

S_(name_resolution_init_f nr_null_init)
static void nr_null_init P_((struct name_resolution_rec *rec,
			  enum name_resolution tag));
static void nr_null_init(rec,tag)
     struct name_resolution_rec *rec;
     enum name_resolution tag;
{
    /* nothing */
}

S_(name_resolution_init_f nr_not_avail)
static void nr_not_avail P_((struct name_resolution_rec *rec,
			     enum name_resolution tag));
static void nr_not_avail(rec,tag)
     struct name_resolution_rec *rec;
     enum name_resolution tag;
{
    /* nothing */
}

S_(name_resolution_init_f nr_GETHOSTNAME)
static void nr_GETHOSTNAME P_((struct name_resolution_rec *rec,
			       enum name_resolution tag));
S_(name_resolution_init_f nr_UNAME)
static void nr_UNAME P_((struct name_resolution_rec *rec,
			 enum name_resolution tag));
S_(name_resolution_init_f nr_GETDOMAINNAME)
static void nr_GETDOMAINNAME P_((struct name_resolution_rec *rec,
				 enum name_resolution tag));
#ifdef MAILNAME_FILE
S_(name_resolution_init_f nr_MAILNAME)
static void nr_MAILNAME P_((struct name_resolution_rec *rec,
			    enum name_resolution tag));
#endif
#if defined(HAVE_IFADDRS) && defined(REMOTE_MBX)
S_(name_resolution_init_f nr_GETIFADDRS)
static void nr_GETIFADDRS P_((struct name_resolution_rec *rec,
			      enum name_resolution tag));
#endif
static struct name_resolution_rec {
    const char * tag;
    
    char * result;
    int errno_val;
    name_resolution_init_f *init;
} name_resolutions[nr_COUNT] = {  
    /* NOTE: Must be on same order than on enum name_resolution */

    /* nr_gethostname */
    { "gethostname", NULL, 0,
#ifdef GETHOSTNAME
      nr_GETHOSTNAME
#else
      nr_not_avail
#endif
    },

    /* nr_uname */
    { "uname", NULL, 0,
#ifdef DOUNAME
      nr_UNAME
#else
      nr_not_avail
#endif
    },

    /* nr_lookup */
    { "lookup", NULL, 0,
#ifdef I_NETDB     
      nr_null_init
#else
      nr_not_avail
#endif
    },

    /* nr_getdomainname */
    { "getdomainname", NULL, 0,
#ifdef GETDOMAINNAME
      nr_GETDOMAINNAME
#else
      nr_not_avail
#endif
    },

    /* nr_mailname */
    { "mailname-file", NULL, 0,
#ifdef MAILNAME_FILE
      nr_MAILNAME
#else
      nr_not_avail
#endif
    },

    /* nr_getifaddrs */
    { "getifaddrs", NULL, 0,
#if defined(HAVE_IFADDRS) && defined(REMOTE_MBX)
      nr_GETIFADDRS
#else
      nr_not_avail
#endif
    },
};

const char * name_resolution_tag(tag)
     enum name_resolution tag;
{
    if (tag < 0 || tag >= nr_COUNT)
	panic("RC PANIC",__FILE__,__LINE__,"name_resolution_tag",
	      "Bad tag", 0);
    
    return name_resolutions[tag].tag;
}

const char * name_resolution_gives(tag)
     enum name_resolution tag;
{
    if (tag < 0 || tag >= nr_COUNT)
	panic("RC PANIC",__FILE__,__LINE__,"name_resolution_tag",
	      "Bad tag", 0);
    
    return name_resolutions[tag].result;
}

int name_resolution_errno(tag)
     enum name_resolution tag;
{
    if (tag < 0 || tag >= nr_COUNT)
	panic("RC PANIC",__FILE__,__LINE__,"name_resolution_errno",
	      "Bad tag", 0);

    return name_resolutions[tag].errno_val;
}

#ifdef GETHOSTNAME
static void nr_GETHOSTNAME(rec,tag) 
     struct name_resolution_rec *rec;
     enum name_resolution tag;
{
    int len = SLEN;
    char * buffer;

#if HOST_NAME_MAX
    if (len < HOST_NAME_MAX)
	len = HOST_NAME_MAX;
#endif    

    buffer = safe_zero_alloc(len+1);
    
    /* Truncation of result may occur without error */

    if (gethostname(buffer,len) < 0) {
	int err = errno;
	DPRINT(Debug,8,(&Debug,
			"gethostname failed: (errno=%d) %s\n",
			err,strerror(err)));	

	rec->errno_val = err;

	/* glibc uses ENAMETOOLONG */
#ifdef 	 ENAMETOOLONG
	if (ENAMETOOLONG == err)
	    goto truncated;
#endif

    } else {
    truncated:
	buffer[len] = '\0';

	DPRINT(Debug,8,(&Debug,
			"gethostname gives: %s\n",buffer));

	if (buffer[0])
	    rec->result = strmcpy(rec->result,buffer);

    }

    free(buffer);
}
#endif

#ifdef DOUNAME
static void nr_UNAME(rec,tag) 
     struct name_resolution_rec *rec;
     enum name_resolution tag;
{
    struct utsname u_name;

    if (uname(&u_name) < 0) {
	int err = errno;
	DPRINT(Debug,1,(&Debug,
			"uname failed: (errno=%d) %s\n",
			err,strerror(err)));	
	rec->errno_val = err;

    } else {
	DPRINT(Debug,8,(&Debug,
			"uname gives: %s\n",u_name.nodename));

	if (u_name.nodename[0])
	    rec->result = strmcpy(rec->result,u_name.nodename);
    }
}
#endif

#ifdef GETDOMAINNAME
static void nr_GETDOMAINNAME(rec,tag)
     struct name_resolution_rec *rec;
     enum name_resolution tag;
{
    int len = SLEN;
    char * buffer;

#if HOST_NAME_MAX
    if (len < HOST_NAME_MAX)   /* Not correct */
	len = HOST_NAME_MAX;
#endif    
    
    buffer = safe_zero_alloc(len+1);
    
    /* Can truncation of result occur without error ? */

    if (getdomainname(buffer,len) < 0) {
	int err = errno;
	DPRINT(Debug,8,(&Debug,
			"getdomainname failed: (errno=%d) %s\n",
			err,strerror(err)));	

	rec->errno_val = err;

	/* ENAMETOOLONG not used ? */
#ifdef 	 ENAMETOOLONG
	if (ENAMETOOLONG == err)
	    goto truncated;
#endif
		
    } else {
    truncated:
	buffer[len] = '\0';

	DPRINT(Debug,8,(&Debug,
			"getdomainname gives: %s\n",buffer));

	if (buffer[0])
	    rec->result = strmcpy(rec->result,buffer);

    }

    free(buffer);
}
#endif

#if defined(HAVE_IFADDRS) && defined(REMOTE_MBX)
static void nr_GETIFADDRS(rec,tag)
     struct name_resolution_rec *rec;
     enum name_resolution tag;
{
    if (update_interface_list(INTERFACE_ADDRLIST,
			      & (rec->errno_val))) {
	DPRINT(Debug,8,(&Debug,"getiffaddrs succeed\n"));
    } else {
	DPRINT(Debug,8,(&Debug,"getiffaddrs failed: (errno=%d) %s\n",
			rec->errno_val,strerror(rec->errno_val)));	
    }
}
#endif

#ifdef MAILNAME_FILE
static void nr_MAILNAME(rec,tag)
     struct name_resolution_rec *rec;
     enum name_resolution tag;
{
    int len = SLEN;

    FILE    *fp = fopen(MAILNAME_FILE,"r");

#if HOST_NAME_MAX
    if (len < HOST_NAME_MAX)   /* Not correct */
	len = HOST_NAME_MAX;
#endif    


    if (fp) {
	int r = malloc_gets(& (rec->result), len, fp);

	if (-1 == r) {
	    DPRINT(Debug,8,(&Debug,
			    "%s truncated (max=%d): %s\n",
			    MAILNAME_FILE,len,
			    rec->result ? rec->result : "(none)"));
	} else if (-1 == r) {
	    int err = errno;
	    DPRINT(Debug,8,(&Debug,
			    "%s read failed: (errno=%d) %s\n",
			    MAILNAME_FILE,err,strerror(err)));
	    rec->errno_val = err;

	} else if (0 == r) {
	    DPRINT(Debug,8,(&Debug,
			    "%s is empty\n",MAILNAME_FILE));

	}

	if (rec->result) {
	    DPRINT(Debug,8,(&Debug,
			    "%s gives: %s\n",MAILNAME_FILE,rec->result));
	}

	fclose(fp);
    } else {
	int err = errno;
	DPRINT(Debug,8,(&Debug,
			"%s open failed: (errno=%d) %s\n",
			MAILNAME_FILE,err,strerror(err)));
	rec->errno_val = err;

    }
}
#endif

/* static pointer to buffer accross invocations */
static char * nameres_return_buffer = NULL;

int nameresfunc(value,enter,lineno,filename,read_flags)
     char **value; 
     int enter;
     int lineno; 
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    static unsigned long incremental_add = 0L;
    static unsigned long incremental_sub = 0L;

    int ok = 1;    

    if (enter) {
	char *ptr, *next = NULL;
	int absolute = 0;

	incremental_add = 0L;
	incremental_sub = 0L;
	
	for (ptr = *value; ptr && *ptr; ptr = next) {
	    int len;
	    int mode = 0;
	    enum name_resolution tag;

	    while (whitespace(*ptr))
		ptr++;

	    next = strpbrk(ptr,"\t #");

	    if (!next)
		len = strlen(ptr);
	    else {
		len = next - ptr;
		
		if ('#' == *next)
		    next = NULL;
	    }

	    if (len < 1)
		continue;
	    
	    if ('+' == *ptr || '-' == *ptr) {
		mode = *ptr;
		ptr++;
		len--;
	    }

	    if (mode && absolute) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNameResolutionMix,
				  "Mixing of incremental mode and absolute mode on \"name-resolution\" in line %d in \"%s\" file is not supported: %c%.*s"),
			  lineno,filename,mode,len,ptr);
		ok = 0;

	    } else if (!mode && !absolute) {
		
		if (incremental_add || incremental_sub) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmNameResolutionMix1,
				      "Mixing of incremental mode and absolute mode on \"name-resolution\" in line %d in \"%s\" file is not supported: %.*s"),
			      lineno,filename,len,ptr);
		    ok = 0;
		}

		absolute = 1;
		name_resolution_val = 0L;

		/* This is only valid as first value */
		if (0 == strcmp(ptr,"none")) 		    
		    continue;                   		
	    }
		
	    for (tag = 0; tag < nr_COUNT; tag++) {
		const char * T = name_resolution_tag(tag);

		if (0 == strncmp(ptr,T,len) && '\0' == T[len])
		    break;
	    }
	    
	    if (tag < nr_COUNT) {
		unsigned long val = 1 << tag;
		
		if (mode != '-' && 
		    nr_not_avail == name_resolutions[tag].init) {

		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmNameResolutionUnsupportedkw,
				      "Unsupported keyword on \"name-resolution\" in line %d in \"%s\" file: %.*s"),
			      lineno,filename,len,ptr);
		    ok = 0;	

		} else { 
		    switch (mode) {			
		    case '+':
			incremental_add      |= val;
			name_resolution_val  |= val;
			break;
		    case '-':
			incremental_sub      |= val;
			name_resolution_val  &= ~val;
			break;
		    case 0:
			name_resolution_val  |= val;
			break;
		    }
		}
	    } else {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNameResolutionBadkw,
				  "Bad keyword on \"name-resolution\" in line %d in \"%s\" file: %.*s"),
			  lineno,filename,len,ptr);
		ok = 0;	
	    }
	}
       
    } else {
	enum name_resolution tag;

	if (nameres_return_buffer)
	    free(nameres_return_buffer);
	nameres_return_buffer = NULL;

	if (incremental_add || incremental_sub) {

	    for (tag = 0; tag < nr_COUNT; tag++) {
		const char * T = name_resolution_tag(tag);
		unsigned long val = 1 << tag;

		if (val == (incremental_add & val)) {

		    char * buf = 
			elm_message(FRM("%s%s+%s"),
				    nameres_return_buffer ? nameres_return_buffer : "",
				    nameres_return_buffer ? " " : "",
				    T);
		    
		    if (nameres_return_buffer)
			free(nameres_return_buffer);
		    nameres_return_buffer = buf;		    
		} 

		if (val == (incremental_sub & val)) {
		    
		    char * buf = 
			elm_message(FRM("%s%s-%s"),
				    nameres_return_buffer ? nameres_return_buffer : "",
				    nameres_return_buffer ? " " : "",
				    T);
		    
		    if (nameres_return_buffer)
			free(nameres_return_buffer);
		    nameres_return_buffer = buf;		    
		} 
	    }

	    /* If on incremental mode, show real value after comment */

	    nameres_return_buffer = 
		strmcat(nameres_return_buffer," # ");
	}

	if (!name_resolution_val && !nameres_return_buffer) {
	    *value = "none";
	} else {

	    if (!name_resolution_val) {
		nameres_return_buffer = 
		    strmcat(nameres_return_buffer," none");
	    }
		
	    for (tag = 0; tag < nr_COUNT; tag++) {
	
		if (have_name_resolution(tag)) {
		    const char * T = name_resolution_tag(tag);
		    char * buf = 
			elm_message(FRM("%s%s%s"),
				    nameres_return_buffer ? nameres_return_buffer : "",
				    nameres_return_buffer ? " " : "",
				    T);
		    
		    if (nameres_return_buffer)
			free(nameres_return_buffer);
		    nameres_return_buffer = buf;		    		    
		}
	    }

	    *value = nameres_return_buffer;	    
	}
    }
   
    return ok;
}

/* Returns name only if it is ok, may update error counter and print error message */

const char * name_resolution_okname(tag,global_mode)
     enum name_resolution tag;
     int global_mode;
{
    const char * R = name_resolution_gives(tag);
    const char * T = name_resolution_tag(tag);
    const unsigned char *x;
    int l;

    if (!R)
	return NULL;

    if (! R[0])
	return NULL;

    /* For some reason getdomainname() returns "(none)" 
       Silently ignore it. 
      
       ( ) are comment characters on email addresses
    */
    if ('(' == R[0] && (2 < (l = strlen(R))) && ')' == R[l-1]) {
	DPRINT(Debug,4,(&Debug,
			"name_resolution_okname(%d,%d): %s: ignoring value %s\n",
			tag,global_mode,T,R));

	return NULL;
    }

    for (x = cs2us(R); *x; x++) {
	if (!isascii(*x)) { 
	    
	    if (0 != (global_mode & READ_FLAG_GLOBAL_ONLY) || 
		0 == access(system_rc_file,WRITE_ACCESS)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNameAsciiOnly,
				  "Only ASCII characters (not 0x%02x) are allowed on value \"%s\" given by %s"),
			  *x,R,T);
		errors++;	
	    }

	    return NULL;
	}
	
	if (!isgraph(*x)) {
	    if (0 != (global_mode & READ_FLAG_GLOBAL_ONLY) || 
		0 == access(system_rc_file,WRITE_ACCESS)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNameGraphOnly,
				  "Only visible characters (not 0x%02x) are allowed on value \"%s\" given by %s"),
			  *x,R,T);
		errors++;	
	    }

	    return NULL;
	}
	
	/* '.' is only RFC 822 special which is allowed */
	if ('.' != *x &&
	    rfc822_special(*x)) {

	    if (0 != (global_mode & READ_FLAG_GLOBAL_ONLY) || 
		0 == access(system_rc_file,WRITE_ACCESS)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNameNormalOnly,
				  "Only normal characters (not special %c) are allowed on value \"%s\" given by %s"),
			  *x,R,T);
		errors++;	
	    }
	    
	    return NULL;
	}
	    
    }

    return R;
}

static void add_no_duplicate P_((struct dt_path_info *ptr,
				 const char *fieldname,
				 const char *val,
				 int global_mode));
static void add_no_duplicate(ptr,fieldname,val,global_mode)
     struct dt_path_info *ptr; 
     const char *fieldname; 
     const char *val;
     int global_mode;
{
    const char ** list = 
	give_dt_path_as_elems(ptr,fieldname);
    
    if (list) {
	int i;
	for (i = 0; list[i]; i++) 
	    if (0 == strcmp(list[i],val)) {
		DPRINT(Debug,12,(&Debug,"add_no_duplicate: \"%s\": duplicate \"%s\"\n",
				 fieldname,val));
		return;
	    }
    }
    
    /* Fake recursive call */
	
    expand_dt_path(ptr,fieldname,val,2,global_mode);

    DPRINT(Debug,12,(&Debug,"add_no_duplicate: \"%s\": added \"%s\"\n",
		     fieldname,val));
}

static void setup_names P_((int rc_open_errno, int global_mode));
static void setup_names(rc_open_errno,global_mode)
     int rc_open_errno;
     int global_mode;
{
    const char **haliases = NULL; 
    
    const char ** mailname_list = 
	give_dt_path_as_elems(&mailname,"mailname");
    int fill_mailname UNUSED_VAROK = 
	have_name_resolution(nr_getifaddrs) && !mailname_list;

    int need_add_none = 0;
 
    /*
     * Get the host name as per configured behavior.
     */

    /* hostdomain, hostfullname, hostname, name-resolution, mailname
       are set with FL_ASCIIFIRST so they are set when this is called
    */

    if (!hostname[0]) {

	enum name_resolution vector[] = { nr_gethostname, nr_uname, nr_BAD };
	int i;

	for (i = 0; vector[i] != nr_BAD; i++) {
	    enum name_resolution tag = vector[i];

	    if (have_name_resolution(tag)) {
		const char * R = name_resolution_okname(tag,global_mode);
		const char * T = name_resolution_tag(tag);
		int E = name_resolution_errno(tag);
		
		if (E && (0 != (global_mode & READ_FLAG_GLOBAL_ONLY) || 
			  0 == access(system_rc_file,WRITE_ACCESS))) {
		    
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmHostnameErrno,
				      "Hostname query by %s failed: %s"),
			      T,strerror(E));
		    
		    errors++;		
		}
		
		if (R) {
		    int L;
		    strfcpy(hostname,R, sizeof hostname);
		    
		    if ((L = strlen(R)) > sizeof  hostname -1 &&
			(0 != (global_mode & READ_FLAG_GLOBAL_ONLY) || 
			 0 == access(system_rc_file,WRITE_ACCESS))) {
			
			lib_error(CATGETS(elm_msg_cat, ElmSet, 
					  ElmHostnameTooLong,
					  "Hostname given by %s is too long (%d, max %d): %s"),
				  T,L,sizeof  hostname -1, R);
			
			errors++;		       
		    }
		    
		    break;
		}
	    }
	}
    }
    
    if (hostfullname[0] && !hostdomain[0]) {
	char * p = strchr(hostfullname,'.');

	if (p && NULL != strchr(p+1,'.')) {
	    
	    strfcpy(hostdomain,p,sizeof hostdomain);

	    DPRINT(Debug,3,(&Debug,
			    "hostfullname=\"%s\" -> hostdomain=\"%s\"\n",
			    hostfullname,hostdomain));

	    if (!hostname[0]) {
		int L = p - hostfullname;

		strfcpy(hostname,hostfullname,sizeof hostname);
		if (L < sizeof hostname)
		    hostname[L] = '\0';

		DPRINT(Debug,3,(&Debug,
				" ... -> hostname=\"%s\"\n",
				hostname));

	    }
	}	
    }

    if ((hostname[0] || hostfullname[0]) &&
	have_name_resolution(nr_lookup)) {
	int fqdn_found UNUSED_VAROK = 0;

	if (!hostdomain[0] || !hostfullname[0] || ! mailname_list) {

	    if (hostfullname[0])
		fqdn_found = lookup_domain_name(hostfullname,&haliases);
	    else if (hostname[0])
		fqdn_found = lookup_domain_name(hostname,&haliases);
	}
    }

    /*
     * now get the domain name, used to build the full name
     */
    if (!hostdomain[0]) {
	if (gethostdomain(hostdomain, sizeof(hostdomain), global_mode) <0) {
	    DPRINT(Debug,1,(&Debug, 
			    "Default domain name is not available\n"));

	    hostdomain[0] = '\0';
	}

    } else {
	if ('.' != hostdomain[0]) {
	    char buffer[SLEN];
	    strfcpy(buffer,hostdomain,sizeof buffer);
	    strfcpy(hostdomain,".",sizeof hostdomain);
	    strfcat(hostdomain,buffer,sizeof hostdomain);
	    DPRINT(Debug,3,(&Debug,
			    "hostdomain=\"%s\" -> hostdomain=\"%s\"\n",
			    buffer,hostdomain));
	}
    }

    if (hostname[0] && hostdomain[0] && !hostfullname[0]) {
	generate_hostfullname(global_mode);

    }

    /* If global elm.rc gives
            mailname = none
       then
           mailname_list != NULL 
       but
           mailname_list[0] == NULL
    */
    
    if (mailname_list && mailname_list[0] && NULL == strchr(mailname_list[0],
							    '.')) {
	int i,l;

	const char * R = mailname_list[0];
	
	/* Ignore "(none)"  */

	if ('(' == R[0] && (2 < (l = strlen(R))) && ')' == R[l-1]) {
	    
	    DPRINT(Debug,10,(&Debug,
			     "ignoring mailname value %s\n",R));
	    
	} else if (hostfullname[0]) {
	    char *x;
	    int dot = 0;

	    for (x = hostfullname; *x; x++) {
		if (!isascii(*x)) 
		    break;
		if ('.' == *x)
		    dot = 1;
	    }

	    if (! *x && dot) {

		if (0 != (global_mode & READ_FLAG_GLOBAL_ONLY) || 0 == access(system_rc_file,WRITE_ACCESS)) {

		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailnameQualified,
				      "First hostname on \"mailname\" on RC file %s should be qualified: %s"),
			      system_rc_file,mailname_list[0]);
		    
		    errors++;		       
		}

		mailname_list = NULL;  /* Returned pointer is invalid after reallocation */
		
		mailname.list = safe_array_realloc(mailname.list,
						   (mailname.nlen+2), 
						   sizeof (mailname.list[0]));
		
		for (i = mailname.nlen; i > 0; i--)
		    mailname.list[i] = mailname.list[i-1];
		mailname.list[0] = safe_strdup(hostfullname);
		mailname.nlen++;
		mailname.list[mailname.nlen] = NULL;
		
		if (mailname.unexpanded)
		    free(mailname.unexpanded);
		mailname.unexpanded = NULL;

		mailname_list = 
		    give_dt_path_as_elems(&mailname,"mailname");	
		
	    } else
		goto print_mailname_error;

	} else {
	print_mailname_error:

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailnameQualified,
			      "First hostname on \"mailname\" on RC file %s should be qualified: %s"),
		      system_rc_file,mailname_list[0]);
	    
	    errors++;		       
	}


    } else if (! mailname_list && 
	       ( hostfullname[0] || 
		 (have_name_resolution(nr_mailname) && 
		  name_resolution_okname(nr_mailname,
					 global_mode))
		 )
	       ) { 

	/* nr_BAD             is -1 
	   nr_DEFAULT_names   is -2
	   nr_DEFAULT_aliases is -3
	*/
		
	int vector[] = { nr_mailname, nr_DEFAULT_names, 
			 nr_gethostname, nr_uname, 
			 nr_DEFAULT_aliases, nr_BAD };
	int i;
		   
	for (i = 0; vector[i] != nr_BAD; i++) {
	    enum name_resolution tag = vector[i];
	
	    switch (tag) {

	    case nr_DEFAULT_names:

		if (hostfullname[0])
		    add_no_duplicate(&mailname,"mailname",hostfullname,
				     global_mode);
	
		if (hostname[0])
		    add_no_duplicate(&mailname,"mailname",hostname,
				     global_mode);
	
		break;
		
	    case nr_DEFAULT_aliases:
		if (haliases) {
		    int j;

		    for (j = 0; haliases[j]; j++) {
			add_no_duplicate(&mailname,"mailname",haliases[j],
					 global_mode);	
		    }
		}
		break;

	    default:
		if (have_name_resolution(tag)) {
		    const char * R = name_resolution_okname(tag,global_mode);
		    const char * T = name_resolution_tag(tag);
		    int E = name_resolution_errno(tag);
		    
		    if (E && (0 != (global_mode & READ_FLAG_GLOBAL_ONLY) || 
			      0 == access(system_rc_file,WRITE_ACCESS))) {
			
			lib_error(CATGETS(elm_msg_cat, ElmSet, 
					  ElmMailnameErrno,
					  "Mailname query by %s failed: %s"),
				  T,strerror(E));
			
			errors++;		
		    }
		    
		    if (R) {
			add_no_duplicate(&mailname,"mailname",R,
					 global_mode);	
		    }
		}
	    }
	}

	mailname_list = 
	    give_dt_path_as_elems(&mailname,"mailname");	
    }

    if (mailname_list && mailname_list[0] && mailname_list[0][0]) {
	DPRINT(Debug,10,(&Debug,"Usefull mailname is %s\n",
			 mailname_list[0]));	    
    } else {
	need_add_none = 1;
	
	if (fill_mailname) {
	    if (0 != (global_mode & READ_FLAG_GLOBAL_ONLY) || 0 == access(system_rc_file,WRITE_ACCESS)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmDefineMailname,
				  "Give \"mailname\" on RC file: %s"),
			  system_rc_file);
	    }
	}	
    }
    
#ifdef REMOTE_MBX
    if (fill_mailname) {
	int query_flags = INTERFACE_ADDRLIST;
	int interfaces_count = 0;
       
	struct interface_info ** interfaces = 
	    get_interfaces(sizeof (struct interface_info),
			   sizeof (struct interface_addr),
			   &interfaces_count,&query_flags);

	if (0 == (query_flags & INTERFACE_ADDRLIST)) {
	    DPRINT(Debug,10,(&Debug,"Failed to get interface address list\n"));
	}
			   
	if (interfaces) {
	    int i_idx;

	    for (i_idx = 0; i_idx < interfaces_count; i_idx++) {

		if (interfaces[i_idx]) {
		    if (INTERFACE_INFO_magic != interfaces[i_idx]->magic)
			panic("RC PANIC",__FILE__,__LINE__,
			      "setup_names",
			      "Bad magic number (interface_info)",0);
		    
		    if (! (interfaces[i_idx]->valid_addrlist)) {
			DPRINT(Debug,12,(&Debug,"Interface \"%s\" skipped, not on address list.\n",
					 interfaces[i_idx]->name));
			continue;			    
		    }

		    if (0 == strcmp(interfaces[i_idx]->name,"lo")) {
			DPRINT(Debug,12,(&Debug,"Interface \"%s\" is ignored, when adding addresses to \"mailname\".\n",
					 interfaces[i_idx]->name));
			continue;
		    }


		    if (interfaces[i_idx]->addr_list) {
			int a_idx;
			
			for (a_idx = 0; a_idx < interfaces[i_idx]->addr_count;
			     a_idx++) {
			
			    if (interfaces[i_idx]->addr_list[a_idx]) {
				
				char * mail_domain = NULL;

				if (INTERFACE_ADDR_magic !=
				    interfaces[i_idx]->addr_list[a_idx]->magic)
				    panic("RC PANIC",__FILE__,__LINE__,
					  "setup_names",
					  "Bad magic number (interface_addr)",
					  0);

				if (!  (interfaces[i_idx]->addr_list[a_idx]->valid_address)) {
				    DPRINT(Debug,12,(&Debug,"Interface \"%s\", address #%d skipped, not on address list.\n",
						     interfaces[i_idx]->name,a_idx));
				    continue;
				}

				if (interfaces[i_idx]->addr_list[a_idx]->interface_loopback) {
				    DPRINT(Debug,12,(&Debug,"Interface \"%s\", address #%d skipped. Is on loopback interface.\n",
						     interfaces[i_idx]->name,a_idx));
				    continue;
				}
				
				if (interfaces[i_idx]->
				    addr_list[a_idx]->address.dummy)
				    mail_domain = 
					SOCKADDR_ptr_to_mail_domain(interfaces[i_idx]->
								    addr_list[a_idx]->address);

				if (mail_domain) {

				    if ( need_add_none ) {
					DPRINT(Debug,12,(&Debug,"Added dummy (none) to \"mailname\"\n"));
					
					/* Fake recursive call */
					expand_dt_path(&mailname,"mailname","(none)",2,
						       global_mode);

					need_add_none = 0;
				    }
				    
				    add_no_duplicate(&mailname,"mailname",mail_domain,
						     global_mode);
				    free(mail_domain);
				}
			    }
			}
		    }
		}
	    }

	}

    }
#endif

    
    if (!hostname[0] || !hostdomain[0] || !hostfullname[0]) {
	
	if (rc_open_errno) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedSystemRC,
			      "Failed to open RC file %s: %s"),
		      system_rc_file,strerror(rc_open_errno));    
	    errors++;
	}
	
	if (0L == name_resolution_val &&
	    (0 != (global_mode & READ_FLAG_GLOBAL_ONLY) || 0 == access(system_rc_file,WRITE_ACCESS))) {

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCheckNameresolution,
			      "Check \"name-resolution\" on RC file: %s"),
		      system_rc_file);
	}

	if (!hostname[0] || !hostdomain[0]) {

	    static const char LOCALDOMAIN[] = ".localdomain";
	    

	    if (hostfullname[0]) {
		char * p = strchr(hostfullname,'.');
		if (!hostname[0]) 
		    strfcpy(hostname,hostfullname,sizeof hostname);
		if (!hostdomain[0]) {
		    if (p) {
			strfcpy(hostdomain,p,sizeof hostdomain);
		    } else if (have_dummy_domain && have_dummy_domain[0]) {
			DPRINT(Debug,9,(&Debug,"setup_names: Using dummy domain %s\n",have_dummy_domain));
			
			strfcpy(hostdomain,have_dummy_domain,sizeof hostdomain);
		    } else {
			strfcpy(hostdomain,LOCALDOMAIN,sizeof hostdomain);
			have_dummy_domain = LOCALDOMAIN;

			DPRINT(Debug,9,(&Debug,"setup_names: Using dummy domain %s\n",have_dummy_domain));
		    }
		}

	    } else {

		if (0 != (global_mode & READ_FLAG_GLOBAL_ONLY) || 0 == access(system_rc_file,WRITE_ACCESS)) 
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmDefineHostname,
				      "Give \"hostname\", \"hostfullname\" and \"hostdomain\" on RC file: %s"),
			      system_rc_file);

		if (!hostname[0]) 
		    strfcpy(hostname,"localhost",sizeof hostname);
		if (!hostfullname[0]) {
		    int hostlen, domlen;

		    hostlen = strlen(hostname);
		    domlen = strlen(hostdomain);

		    strfcpy(hostfullname,hostname,sizeof hostfullname);
		    /* Hostdomain may be empty, in that case hostfullname
		       is just hostname
		    */
		    
		    if (hostlen + domlen < sizeof hostfullname)
			strfcat(hostfullname,hostdomain,
				sizeof hostfullname);
		    else if (0 != (global_mode & READ_FLAG_GLOBAL_ONLY) || 
			     0 == access(hostdomfile,WRITE_ACCESS)) {
			
			lib_error(CATGETS(elm_msg_cat, ElmSet, 
					  ElmFullHostNameTooLong,
					  "Full hostname \"%s%s\" too long (%d, max %d), using: %s"),
				  hostname,hostdomain,
				  hostlen + domlen,
				  sizeof hostfullname -1,
				  hostfullname);
		    } 		    		
		}
		if (!hostdomain[0]) 
		    strfcpy(hostdomain,".localdomain",sizeof hostdomain);		
		errors++;       
	    }
	}   
    }

    if (haliases)
	free(haliases);
}

static void name_resolution_register_if_ok P_((enum name_resolution tag));
static void name_resolution_register_if_ok(tag)
     enum name_resolution tag;
{
    if (0 == name_resolution_errno(tag))
	name_resolution_val |= 1 << tag;
}

/* Returns 1 if success, 0 failure, -1 if expansion is not string */
static int defaultfile_expand_func(kw,ki,keyword,continue_mask)
     struct dt_estr_info *kw;
     struct dt_keyword_info *ki;
     unsigned int keyword;
     unsigned int continue_mask;
{
    char * cp;

    if (0 != (continue_mask &  defaultfile_KW_MAIL) &&
	NULL != (cp = getenv("MAIL"))) {

	estr_set_expanded(kw,cp);

	DPRINT(Debug,12,(&Debug, 
			 "defaultfile_expand_func=1: continue_mask=x%04x $MAIL=%s\n",
			 continue_mask,cp));
	
	kw->continue_mask |= FOLDER_is_local_file;
	
	DPRINT(Debug,12,(&Debug, 
			 "defaultfile_expand_func: Setting FOLDER_is_local_file\n"));
	return 1;
    }

    return 0;
}

/* Returns 1 if success, 0 failure, -1 if expansion is not string */
static int pager_expand_func(kw,ki,keyword,continue_mask)
     struct dt_estr_info *kw;
     struct dt_keyword_info *ki;
     unsigned int keyword;
     unsigned int continue_mask;
{
    char *cp;

    if (0 != (continue_mask & pager_KW_PAGER) &&
	NULL != (cp = getenv("PAGER"))) {

	estr_set_expanded(kw,cp);

	DPRINT(Debug,12,(&Debug, 
			 "pager_expand_func=1: continue_mask=x%04x $PAGER=%s\n",
			 continue_mask,cp));
	
	return 1;
    }

    switch(keyword) {
    case pager_kw_builtin:

	kw->expanded = strmcpy(kw->expanded,"builtin");
	kw->expanded_keyword = keyword;

	DPRINT(Debug,12,(&Debug, 
			 "pager_expand_func=1: keyword=%d expanded=%s\n",
			 keyword,kw->expanded));
		
	return 1;
    }


    return 0;
}

/* Returns 1 if success, 0 failure, -1 if expansion is not string */
static int editor_expand_func(kw,ki,keyword,continue_mask)
     struct dt_estr_info *kw;
     struct dt_keyword_info *ki;
     unsigned int keyword;
     unsigned int continue_mask;
{
    char *cp;

    if (0 != (continue_mask & editor_KW_EDITOR) &&
	NULL != (cp = getenv("EDITOR"))) {

	estr_set_expanded(kw,cp);

	DPRINT(Debug,12,(&Debug, 
			 "editor_expand_func=1: continue_mask=x%04x $EDITOR=%s\n",
			 continue_mask,cp));
	
	return 1;
    }

    if (0 != (continue_mask & editor_KW_VISUAL) &&
	NULL != (cp = getenv("VISUAL"))) {

	estr_set_expanded(kw,cp);

	DPRINT(Debug,12,(&Debug, 
			 "editor_expand_func=1: continue_mask=x%04x $VISUAL=%s\n",
			 continue_mask,cp));
	
	return 1;
    }


    switch(keyword) {
    case editor_kw_builtin:

	kw->expanded = strmcpy(kw->expanded,"builtin");
	kw->expanded_keyword = keyword;

	DPRINT(Debug,12,(&Debug, 
			 "editor_expand_func=1: keyword=%d expanded=%s\n",
			 keyword,kw->expanded));
		
	return 1;
    }


    return 0;
}

/* Returns 1 if success, 0 failure, -1 if expansion is not string */
static int alteditor_expand_func(kw,ki,keyword,continue_mask)
     struct dt_estr_info *kw;
     struct dt_keyword_info *ki;
     unsigned int keyword;
     unsigned int continue_mask;
{

    char *cp;

    if (0 != (continue_mask & alteditor_KW_EDITOR) &&
	NULL != (cp = getenv("EDITOR"))) {

	estr_set_expanded(kw,cp);

	DPRINT(Debug,12,(&Debug, 
			 "alteditor_expand_func=1: continue_mask=x%04x $EDITOR=%s\n",
			 continue_mask,cp));
	
	return 1;
    }

    if (0 != (continue_mask & alteditor_KW_VISUAL) &&
	NULL != (cp = getenv("VISUAL"))) {

	estr_set_expanded(kw,cp);

	DPRINT(Debug,12,(&Debug, 
			 "alteditor_expand_func=1: continue_mask=x%04x $VISUAL=%s\n",
			 continue_mask,cp));
	
	return 1;
    }


    switch(keyword) {
	const char * editor_val;
	unsigned int editor_keyword, editor_mask;

    case alteditor_kw_EDITOR:
	
	editor_val =  give_dt_estr_as_str(&editor_e,"editor",
					  &editor_keyword,
					  &editor_mask);
	
	if (!editor_val)
	    editor_val = "EDITOR";
	
	kw->expanded_keyword = 
	    editor_keyword ? editor_keyword : keyword;

	kw->expanded = strmcpy(kw->expanded,editor_val);
					  
	DPRINT(Debug,12,(&Debug, 
			 "alteditor_expand_func=1: keyword=%d expanded=%s expanded_keyword=%d\n",
			 keyword,kw->expanded,
			 kw->expanded_keyword));
	return 1;
    }

    return 0;
}

char elmrc_charset[SLEN];
static charset_t set_elmrc_charset(filename,errors)
     const char *filename;
     int *errors;
{
    if (elmrc_charset[0]) {
	
	charset_t cs = MIME_name_to_charset(elmrc_charset,0);
	
	if (cs) {
	    const char * csn = get_charset_MIME_name(cs);
	    
	    if (csn) {
		DPRINT(Debug,20,(&Debug,
				 "set_elmrc_charset: %s: Using %s (%s) as charset\n",
				 filename,elmrc_charset,csn));
	    } else {
		DPRINT(Debug,20,(&Debug,
				 "set_elmrc_charset: %s: Using %s as charset\n",
				 filename,elmrc_charset,csn));
	    }
	    	    
	    if (!(CS_mapping & charset_properties(cs))) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmUnsupportedRCCharset,
				  "Charset %s is unsupported at @charset option in \"%s\" file"),
			  csn ? csn : elmrc_charset,filename);
		
		(*errors)++;
	    }
	    
	    return cs;
	}
	
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmUnsupportedRCCharset,
			  "Charset %s is unsupported at @charset option in \"%s\" file"),
		  elmrc_charset,
		  filename);
	
	(*errors)++;
    }

    /* Return NULL if @charset was not set on this file */

    return NULL;
}

#ifdef HAVE_SCOPE
/* This is called before parse_service_entries() */

static void fail_scope_id P_((int global_mode_or_read_rc_flag,
			      enum record_mode is_local))
    UNUSED_FUNCOK;
static void fail_scope_id(global_mode_or_read_rc_flag,is_local)
     int global_mode_or_read_rc_flag;
     enum record_mode is_local;
{
    switch (is_local) {
    case SYSTEM_RC:
	if (0 != (global_mode_or_read_rc_flag & READ_FLAG_GLOBAL_ONLY) || 
	    0 == access(system_rc_file,WRITE_ACCESS)) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmScopeIDFailSys,
			      "ipv6-scope-id = %s failed. Set ipv6-scope-id = none or ipv6-scope-id = %s on %s"),
		      IPV6_SCOPE_ID[scope_auto_flag],
		      IPV6_SCOPE_ID[scope_link_local_if_index_flag],
		      system_rc_file);
	}
	break;
    case LOCAL_RC:
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmScopeIDFailUser,
			  "ipv6-scope-id = %s failed. Set ipv6-scope-id = none or ipv6-scope-id = %s"),
		  IPV6_SCOPE_ID[scope_auto_flag],
		  IPV6_SCOPE_ID[scope_link_local_if_index_flag]);
	
	break;
    case RC_MODE_COUNT:
	break;
    }
}


static void probe_ipv6_scope_id P_((int global_mode_or_read_rc_flag,
				    enum record_mode is_local));
static void probe_ipv6_scope_id(global_mode_or_read_rc_flag,
				is_local)
     int global_mode_or_read_rc_flag;
     enum record_mode is_local;
{
#ifdef REMOTE_MBX
    int query_flags = INTERFACE_ADDRLIST|INTERFACE_INDEX;

    int interfaces_count = 0;
       
    struct interface_info ** interfaces = 
	get_interfaces(sizeof (struct interface_info),
		       sizeof (struct interface_addr),
		       &interfaces_count,&query_flags);

    if (0 == (query_flags & INTERFACE_ADDRLIST) ||
	0 == (query_flags & INTERFACE_INDEX)) {

	if (0 == (query_flags & INTERFACE_ADDRLIST)) {
	    DPRINT(Debug,8,(&Debug, "probe_ipv6_scope_id: INTERFACE_ADDRLIST not set\n"));
	}
	if (0 == (query_flags & INTERFACE_INDEX)) {
	    DPRINT(Debug,8,(&Debug, "probe_ipv6_scope_id: INTERFACE_INDEX not set\n"));
	}

	fail_scope_id(global_mode_or_read_rc_flag,is_local);
    }

    if (interfaces) {
	int i_idx;

	int ok_count   = 0;
	int bad_count  = 0;
	int fail_count = 0;
	int seen_ipv6  = 0;

	for (i_idx = 0; i_idx < interfaces_count; i_idx++) {
	    
	    if (interfaces[i_idx]) {
		if (INTERFACE_INFO_magic != interfaces[i_idx]->magic)
		    panic("RC PANIC",__FILE__,__LINE__,
			  "probe_ipv6_scope_id",
			  "Bad magic number (interface_info)",0);
		
		if (interfaces[i_idx]->valid_index) {
		    DPRINT(Debug,12,(&Debug,"probe_ipv6_scope_id: interface \"%s\" index = %u\n",
				     interfaces[i_idx]->name,
				     interfaces[i_idx]->if_index));
		}

		if (! (interfaces[i_idx]->valid_addrlist)) {
		    DPRINT(Debug,12,(&Debug,"probe_ipv6_scope_id: interface \"%s\" skipped, not on address list.\n",
				     interfaces[i_idx]->name));
		    continue;			    
		}


		if (interfaces[i_idx]->addr_list) {
		    int a_idx;
		    
		    for (a_idx = 0; a_idx < interfaces[i_idx]->addr_count;
			 a_idx++) {
			

			if (interfaces[i_idx]->addr_list[a_idx]) {
			    
			    if (INTERFACE_ADDR_magic !=
				interfaces[i_idx]->addr_list[a_idx]->magic)
				panic("RC PANIC",__FILE__,__LINE__,
				      "probe_ipv6_scope_id",
				      "Bad magic number (interface_addr)",
				      0);

			    if (!  (interfaces[i_idx]->addr_list[a_idx]->valid_address)) {
				DPRINT(Debug,12,(&Debug,
						 "probe_ipv6_scope_id: interface \"%s\", address #%d skipped, not on address list.\n",
						 interfaces[i_idx]->name,a_idx));
				    continue;
				}
			    
			    if (interfaces[i_idx]->addr_list[a_idx]->address.sa) {
				char mybuffer[256];		
				const char * s UNUSED_VAROK
				    = give_addr_as_string(interfaces[i_idx]->addr_list[a_idx]->
							  address.generic,
							  mybuffer, sizeof mybuffer);
				
				switch(interfaces[i_idx]->addr_list[a_idx]->address.
				       sa->sa_family) {
				    int is_link_local;
				    
				case AF_INET6:
				    seen_ipv6++;

				    is_link_local =
					IN6_IS_ADDR_LINKLOCAL(& (interfaces[i_idx]->
								 addr_list[a_idx]->
								 address.sin6->sin6_addr));
				    
				    DPRINT(Debug,8,(&Debug, 
						    "probe_ipv6_scope_id: interface \"%s\", address #%d: %s is %s\n",
						    interfaces[i_idx]->name,a_idx,s,
						    is_link_local ? 
						    "link local" : 
						    "not link local"));
				    
				    if (is_link_local) {
					if (interfaces[i_idx]->valid_index &&
					    interfaces[i_idx]->if_index ==
					    interfaces[i_idx]->addr_list[a_idx]->
					    address.sin6->sin6_scope_id) {
					    DPRINT(Debug,8,(&Debug, 
							    "probe_ipv6_scope_id: ... scope id %u is interface index\n",
							    (unsigned) (interfaces[i_idx]->
									addr_list[a_idx]->
									address.sin6->sin6_scope_id)));
					    ok_count++;
					} else if (interfaces[i_idx]->valid_index) {
					    DPRINT(Debug,8,(&Debug, 
							    "probe_ipv6_scope_id: ... scope id %u != interface index %u\n",
							    (unsigned) (interfaces[i_idx]->
								    addr_list[a_idx]->
								    address.sin6->sin6_scope_id),
							    interfaces[i_idx]->if_index));
					    bad_count++;
					} else
					    fail_count++;
				    }
				    break;
				}							   
			    }
			}
		    }
		}
	    }
	}

	DPRINT(Debug,12,(&Debug,"probe_ipv6_scope_id: ok %d, bad %d, fail %d, %d IPv6 addresses\n",
			 ok_count,bad_count,fail_count,seen_ipv6));

	if (fail_count) {
	    fail_scope_id(global_mode_or_read_rc_flag,is_local);
	   
	} else if (bad_count) {
	    ipv6_scope_id.val &= ~ (FLAGVAL(scope_link_local_if_index_flag));

	    DPRINT(Debug,12,(&Debug,"probe_ipv6_scope_id: no %s\n",
			     IPV6_SCOPE_ID[scope_link_local_if_index_flag]));
	} else if (ok_count) {
	    ipv6_scope_id.val |= FLAGVAL(scope_link_local_if_index_flag);
		
	    DPRINT(Debug,12,(&Debug,"probe_ipv6_scope_id: added %s\n",
			     IPV6_SCOPE_ID[scope_link_local_if_index_flag]));
		
	} else if (seen_ipv6) {  /* Silent failure if no IPv6 addresses */
	    fail_scope_id(global_mode_or_read_rc_flag,is_local);
	}
    }

#endif
}

#endif


int init_defaults(global_mode) 
     int global_mode;
{
    
    char     *cp;
    FILE * file;

    int rc_open_errno = 0;

    struct parsed_rc *rc = NULL;
    charset_t   rcset = NULL;

    int i;

    /* Establish some defaults in case elmrc is incomplete or not there.
     * Defaults for other elmrc options were established in their
     * declaration - in elm.h.  
   */
    
    DPRINT(Debug,8,(&Debug, 
		    "init_defaults: global_mode = %d%s%s%s%s\n",
		    global_mode,
		    (global_mode & READ_FLAG_GLOBAL_ONLY) ? " READ_FLAG_GLOBAL_ONLY" : "",
		    (global_mode & READ_FLAG_IGNORE_MISSING) ? " READ_FLAG_IGNORE_MISSING" : "",
		    (global_mode & READ_FLAG_UNIDATA_INIT) ? " READ_FLAG_UNIDATA_INIT" : "",
		    (global_mode & READ_FLAG_BINDATA_INIT) ? " READ_FLAG_BINDATA_INIT" : ""
		    ));

    if (init_defaults_called) {
	DPRINT(Debug,1,(&Debug,"init_defaults called twice!!\n"));	
    }
    init_defaults_called = 1;

    errors = 0;

    for (i = 0; i < nr_COUNT; i++) {
	/* If duplicate call */
	if (name_resolutions[i].result)
	    free(name_resolutions[i].result);
	name_resolutions[i].result = NULL;
	name_resolutions[i].errno_val = 0;

	if (name_resolutions[i].init != nr_null_init &&
	    name_resolutions[i].init != nr_not_avail) {

	    DPRINT(Debug,9,(&Debug, 
			    "init_defaults: Initializing name resolution %s\n",
			    name_resolutions[i].tag));

	    name_resolutions[i].init(& (name_resolutions[i]), i);
	}
 
    }

    if (!quote_prefix)
	quote_prefix = 
	    new_string2(ASCII_SET,s2us("> "));  /* prefix char(s) for msgs          */

    /* Setup default resolutions */
#ifdef GETHOSTNAME
    name_resolution_register_if_ok(nr_gethostname);
#endif

#ifdef DOUNAME
    name_resolution_register_if_ok(nr_uname);
#endif

#ifdef I_NETDB
    name_resolution_register_if_ok(nr_lookup);
#endif

#ifdef USEGETDOMAINNAME
    name_resolution_register_if_ok(nr_getdomain);
#endif
    
#ifdef USEMAILNAME
    name_resolution_register_if_ok(nr_mailname);
#endif

#ifdef USEGETIFADDRS
    name_resolution_register_if_ok(nr_getifaddrs);
#endif

    allowed_precedences[0] = '\0';
    /* no defaults for those */

    set_dt_estr(&shell_e,shell0,"SHELL",0,0);
    /* Shell is set from /etc/passwd in user_init () */

#ifdef USE_BUILTIN_PAGER
    set_dt_estr(&pager_e, "builtin++", NULL, pager_kw_builtin,0);
#else
    set_dt_estr(&pager_e,default_pager,NULL, 0,pager_KW_PAGER);
#endif    

    set_dt_estr(&temp_dir_e,default_temp,"TMPDIR",0,0);
    set_dt_estr(&editor_e,default_editor,NULL,0,
		editor_KW_EDITOR|editor_KW_VISUAL);
    set_dt_estr(&alternative_editor_e,"EDITOR",NULL,
		alteditor_kw_EDITOR,
		alteditor_KW_EDITOR|alteditor_KW_VISUAL);

    if ((cp = getenv("MAILCAPS")) == NULL) {
	/* RFC 1524:

   For UNIX, a path search of mailcap files is specified.  The default
   path search is specified as including at least the following:

   $HOME/.mailcap:/etc/mailcap:/usr/etc/mailcap:/usr/local/etc/mailcap

	*/
	char *X[] = { "/etc/mailcap", "/usr/etc/mailcap",
		      "/usr/local/etc/mailcap", NULL };
	char *G = elm_message(FRM("%s/.mailcap"),home);
	char **X1;
	
	/* Fake recursive call -- expand_dt_path ignores non-existiant
	   entries.
	 */
	DPRINT(Debug,9,(&Debug,
			"Adding default entry to metamail-mailcaps: %s\n",
			G));
	expand_dt_path(&metamail_mailcaps,"metamail-mailcaps",
		       G,2,global_mode);
	for (X1 = X; *X1; X1++) {
	    DPRINT(Debug,9,(&Debug,
			    "Adding default entry to metamail-mailcap: %s\n",
			    *X1));
	    expand_dt_path(&metamail_mailcaps,"metamail-mailcaps",
			   *X1,2,global_mode);	    
	}

	DPRINT(Debug,9,(&Debug,
			"Adding default entry to internal-mailcaps: %s\n",
			G));
	expand_dt_path(&internal_mailcaps,"internal-mailcaps",
		       G,2,global_mode);
	for (X1 = X; *X1; X1++) {
	    DPRINT(Debug,9,(&Debug,
			    "Adding default entry to internal-mailcap: %s\n",
			    *X1));
	    expand_dt_path(&internal_mailcaps,"internal-mailcaps",
			   *X1,2,global_mode);	    
	}
	
	free(G);
    } else {
	DPRINT(Debug,9,(&Debug,
			"Adding unexpanded entry to metamail-mailcaps: $MAILCAPS\n"
			));

	metamail_mailcaps.unexpanded = safe_strdup("$MAILCAPS");


	DPRINT(Debug,9,(&Debug,
			"Adding unexpanded entry to internal-mailcaps: $MAILCAPS\n"
			));

	internal_mailcaps.unexpanded = safe_strdup("$MAILCAPS");
    }

    folders_e.unexpanded    = elm_message(FRM("~/%s"), default_folders);
    map_txtdir_e.unexpanded = safe_strdup("{lib}/elm.map.txt");
    map_bindir_e.unexpanded = safe_strdup("{lib}/elm.map.bin");

    readmsg_e.unexpanded = safe_strdup("{bin}/readmsg");
       
    calendar_file_e.unexpanded = 
	elm_message(FRM("~/%s"), dflt_calendar_file);

    dead_letter_dir_e.unexpanded = safe_strdup("~/Canceled.mail.dir");

    strfcpy(e_editor, emacs_editor, sizeof e_editor);
    strfcpy(v_editor, default_editor, sizeof v_editor);

    if (use_PGP) {
#ifdef GPG_PATH
	set_dt_estr(&gpg_path_e,GPG_PATH,NULL,0,0);
#endif
#ifdef PGP2_PATH
	set_dt_estr(&pgp2_path_e,PGP2_PATH,NULL,0,0);
#endif
#ifdef PGP5_DIR
	set_dt_estr(&pgp5_dir_e,PGP5_DIR,NULL,0,0);
#endif
    }

    set_dt_estr(&metamail_path_e,METAMAIL_PATH,NULL,0,0);            
    set_dt_estr(&printout_e,DEFAULT_PRINTOUT,NULL,0,0);            

    {
	char *X[] = { "/usr/bin/wine", "wine", "/bin/sh", "sh", NULL };

	/* However do not add "true" or "false" to blacklist. */

	char *ignore[] = { "/bin/false", "false", "/bin/true", 
			   "true", NULL };

	char **X1;
	char *Y;
	
	/* Fake recursive call -- expand_dt_path ignores non-existiant
	   entries.
	 */

	for (X1 = X; *X1; X1++) {
	    DPRINT(Debug,9,(&Debug,
			    "Adding default entry to mailcap-blacklist-programs: %s\n",
			    *X1));
	    expand_dt_path(&mailcap_bl_programs,"mailcap-blacklist-programs",
			   *X1,2,global_mode);	    
	}

	if (POLL_method)
	    mailcap_tempfile_lifetime = 5;  /* Keep temorray files 5 seconds,
					       if this option is available
					    */
	
#ifdef GETUSERSHELL
	/* Also include shells from /etc/shells  */

	while (NULL != (Y = getusershell())) {
	    char * Y1;
	    for (X1 = X; *X1; X1++) {
		if (0 == strcmp(*X1,Y))
		    break;
	    }

	    if (!*X1) {
		for (X1 = ignore; *X1; X1++) {
		    if (0 == strcmp(*X1,Y))
			break;
		}
	    }

	    /* And if not in default list */
	    if (!*X1) {
		DPRINT(Debug,9,(&Debug,
				"Adding default entry to mailcap-blacklist-programs: %s\n",
				Y));
		expand_dt_path(&mailcap_bl_programs,"mailcap-blacklist-programs",
			       Y,2,global_mode);	    


	    }

	    Y1 = strrchr(Y,'/');
	    if (Y1) {
		Y1++;

		for (X1 = X; *X1; X1++) {
		    if (0 == strcmp(*X1,Y1))
			break;
		}

		if (!*X1) {
		    for (X1 = ignore; *X1; X1++) {
			if (0 == strcmp(*X1,Y1))
			    break;
		    }
		}
				
		/* And if not in default list */
		if (!*X1) {
		    DPRINT(Debug,9,(&Debug,
				    "Adding default entry to mailcap-blacklist-programs: %s\n",
				    Y1));
		    expand_dt_path(&mailcap_bl_programs,"mailcap-blacklist-programs",
				   Y1,2,global_mode);	    
		    		    
		}
	    }
	}
	endusershell();
#endif
    }

    elmrc_charset[0] = '\0';   /* Reset old @charset value */

    /* try system-wide rc file */
    file = fopen(system_rc_file, "r");
    if ( file != NULL ) {

	/* This process only options marked with FL_ASCIIFIRST */

	rc = do_rc_parse(file, SYSTEM_RC,system_rc_file,&errors,
			 global_mode);

	fclose(file);
	
    } else {
	rc_open_errno = errno;

	DPRINT(Debug,1,(&Debug,
			"Opening %s failed:  (errno=%d) %s\n",
			system_rc_file,rc_open_errno,strerror(rc_open_errno)));
    }
       
    /* hostdomain, hostfullname, hostname, name-resolution, mailname
       are set with FL_ASCIIFIRST so they are set when setup_names() is called
    */
    setup_names(rc_open_errno, global_mode);

    /* Also unidata  ... notice temporary variable */
    expand_map_bin("unidata", unidata_path_global, raw_unidata_path,
		   sizeof(unidata_path_global),global_mode);

    /* Also bindata  ... notice temporary variable */
    expand_map_bin("bindata", bindata_path_global, raw_bindata_path,
		   sizeof(bindata_path_global),global_mode);

    system_iso2022_map_conf =  parse_iso2022_map(system_iso2022_sets,
						 &errors,ml_global);
    system_charset_map = load_charset_map(system_mime_charsets,&errors);

    map_system_charset(display_locale,display_codeset,1);

    /* Returns NULL, if @charset is not set */
    rcset = set_elmrc_charset(system_rc_file,&errors);
    
    if (rc) {    /* Process most of  elm.rc options */
	do_rc_process(rc, SYSTEM_RC,system_rc_file,rcset,
		      &errors, global_mode /* also read_flags */);
	free_parsed_rc(&rc);

	if (process_delayed_enumerated(NULL)) {
	    DPRINT(Debug,9,(&Debug,
			    "Delayed enumerated values OK or not used on %s\n",
			    system_rc_file));
	} else {
	    DPRINT(Debug,9,(&Debug,
			    "Bad delayed  enumerated values on  on %s\n",
			    system_rc_file));
	    errors += 1;
	}

    } else if (!rc_open_errno)
	errors++;

#ifdef HAVE_SCOPE
    if (dt_flag_is_set(&ipv6_scope_id,scope_auto_flag))
	probe_ipv6_scope_id(global_mode,SYSTEM_RC);
#endif

#ifdef REMOTE_MBX
    
    system_mail_services_conf = 
	parse_service_entries(system_mail_services,&errors,
			      &system_mail_services_cs, 
			      &system_mail_services_pl);
#endif

    for (i = 0; i < hooklist_len; i++) {
	DPRINT(Debug,3,(&Debug,
			"init_defaults: executing hooklist entry #%d (total %d) -- %p\n",
			i, hooklist_len, hooklist[i]));
	hooklist[i](&errors);
    }

    DPRINT(Debug,3,(&Debug, 
		    "init_defaults=%d (errors)\n",errors));

    return errors;
}

enum charset_keyword special_charset_keyword(value)
     const char *value;
{
    if (0 == strcmp(value,"SYSTEM"))
	return cskw_SYSTEM;
    else if (0 == strcmp(value,"DISPLAY"))
	return cskw_DISPLAY;
    else if (0 == strcmp(value,"TEXT"))
	return cskw_TEXT;
    else
	return cskw_none;
}

void post_process_charset_options()
{
    switch (special_charset_keyword(raw_text_charset)) {
    case cskw_SYSTEM:   text_charset = system_charset;  break;
    case cskw_DISPLAY:  text_charset = display_charset; break;
    case cskw_TEXT:     
	text_charset = display_charset;
	strfcpy(raw_text_charset,"DISPLAY",sizeof raw_text_charset);
	break;
    default:
	break;
    }

    switch (special_charset_keyword(raw_display_charset)) {
    case cskw_SYSTEM:   wanted_display_charset = system_charset;  break;
    case cskw_DISPLAY:  wanted_display_charset = display_charset; break;
    case cskw_TEXT:     wanted_display_charset = text_charset;    break;
    default:                                                      break;
    }

    if (!wanted_display_charset ||
	0 == (charset_properties(wanted_display_charset) & CS_mapping)) {
	if (0 != charset_properties(system_charset)) 
	    wanted_display_charset = system_charset;
	else
	    wanted_display_charset = ASCII_SET;

	DPRINT(Debug,2,(&Debug,
			"Setting (wanted) display-charset to %s instead.\n",
			wanted_display_charset->MIME_name ?
			wanted_display_charset->MIME_name :
			"<no MIME name>"));
    }
    if (!text_charset || !text_charset->MIME_name) {
	text_charset = ASCII_SET;
	DPRINT(Debug,2,(&Debug,
			"Setting text-charset to %s instead.\n",
			text_charset->MIME_name ?
			text_charset->MIME_name :
			"<no MIME name"));
    }

    switch (special_charset_keyword(raw_local_fs_charset)) {
    case cskw_SYSTEM:   local_fs_charset = system_charset;  break;
    case cskw_DISPLAY:  local_fs_charset = display_charset; break;
    case cskw_TEXT:     local_fs_charset = text_charset;    break;
    default:                                                break;
    }

    switch (special_charset_keyword(raw_default_nomime_charset)) {
    case cskw_SYSTEM:   default_nomime_charset = system_charset;  break;
    case cskw_DISPLAY:  default_nomime_charset = display_charset; break;
    case cskw_TEXT:     default_nomime_charset = text_charset;    break;
    default:                                                      break;
    }

    switch (special_charset_keyword(raw_default_mimetext_charset)) {
    case cskw_SYSTEM:   default_mimetext_charset = system_charset;  break;
    case cskw_DISPLAY:  default_mimetext_charset = display_charset; break;
    case cskw_TEXT:     default_mimetext_charset = text_charset;    break;
    default:                                                        break;
    }

#ifdef REMOTE_MBX
    switch (special_charset_keyword(raw_imap_charset)) {
    case cskw_SYSTEM:   imap_charset = system_charset;  break;
    case cskw_DISPLAY:  imap_charset = display_charset; break;
    case cskw_TEXT:     imap_charset = text_charset;    break;
    default:                                            break;
    }
#endif
	
}


static init_default_hook **hooklist_rc = NULL;
static int                hooklist_rc_len = 0;

static int read_rc_file_called = 0;

void add_read_rc_file_hook(hook)
     init_default_hook *hook;
{
    DPRINT(Debug,3,(&Debug,"add_read_rc_file_hook: -- procedure %p\n",hook));


    if (read_rc_file_called) {
	DPRINT(Debug,3,(&Debug,"add_read_rc_file_hook: executing immediately -- %p\n",
			hook));

	hook(&errors);
    }

    hooklist_rc = safe_array_realloc(hooklist_rc,
				     (hooklist_rc_len+1), 
				     sizeof(hooklist_rc[0]));

    hooklist_rc[hooklist_rc_len] = hook;

    DPRINT(Debug,3,(&Debug,"add_read_rc_file_hook: stored hooklist_rc entry #%d  -- %p\n",
		    hooklist_rc_len,hook));

    hooklist_rc_len++;
}



int read_rc_file(read_flags)
     int read_flags;
{
    /** this routine does all the actual work of reading in the
	.rc file... **/

    int i;
    struct parsed_rc *rc = NULL;
    int open_error = 0;
    charset_t   rcset = NULL;

    DPRINT(Debug,8,(&Debug, 
		    "read_rc_file: read_flags = %d%s%s%s%s\n",
		    read_flags,
		    (read_flags & READ_FLAG_GLOBAL_ONLY) ? " READ_FLAG_GLOBAL_ONLY" : "",
		    (read_flags & READ_FLAG_IGNORE_MISSING) ? " READ_FLAG_IGNORE_MISSING" : "",
		    (read_flags & READ_FLAG_UNIDATA_INIT) ? " READ_FLAG_UNIDATA_INIT" : "",
		    (read_flags & READ_FLAG_BINDATA_INIT) ? " READ_FLAG_BINDATA_INIT" : ""
		    ));

    
    errors = 0;
    
    if (read_rc_file_called) {
	DPRINT(Debug,1,(&Debug,"read_rc_file called twice!\n"));
    }
    read_rc_file_called = 1;

    elmrc_charset[0] = '\0';   /* Reset old value */

    /* Look for the elmrc file */
    open_error = can_open(user_rc_file, "r");
    if (open_error) {

	if (ENOENT == open_error &&
	    0 != (read_flags & READ_FLAG_IGNORE_MISSING)) {
	    
	    DPRINT(Debug,2,(&Debug,
			    "Ignoring missing %s\n",
			    user_rc_file));
	    
	} else if (user_rc_set) 
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedUserRC,
			      "Failed to open RC file %s: %s"),
		      user_rc_file,strerror(open_error));    

	DPRINT(Debug,2,(&Debug,
			"Warning:User has no \"%s\" file (can_open)\n\n",
			user_rc_file));

    } else {
       
	FILE *file = NULL;
    
	if ((file = fopen(user_rc_file, "r")) == NULL) {
	    open_error = errno;


	if (ENOENT == open_error &&
	    0 != (read_flags & READ_FLAG_IGNORE_MISSING)) {
	    
	    DPRINT(Debug,2,(&Debug,
			    "Ignoring missing %s\n",
			    user_rc_file));
	    
	} else if (user_rc_set) 
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedUserRC,
				  "Failed to open RC file %s: %s"),
			  user_rc_file,strerror(open_error));    


	    DPRINT(Debug,2,(&Debug,
			    "Warning:Open \"%s\" failed\n\n",
			user_rc_file));

	} else {

	    /* This process only options marked with FL_ASCIIFIRST */

	    rc = do_rc_parse(file, LOCAL_RC, 
			     user_rc_set ? user_rc_file: ELMRCFILE,
			     &errors,read_flags);

	    fclose(file);
	}
    }


    /* unidata may be needed ... */
    expand_map_bin("unidata", unidata_path, raw_unidata_path,
		   sizeof(unidata_path),read_flags);

    /* Also bindata  ... */
    expand_map_bin("bindata", bindata_path, raw_bindata_path,
		   sizeof(bindata_path),read_flags);

    user_iso2022_map_conf = parse_iso2022_map(user_iso2022_sets,&errors, 
					      ml_user);
    user_charset_map   = load_charset_map(user_mime_charsets,&errors);

    rcset = set_elmrc_charset(user_rc_set ? user_rc_file: ELMRCFILE,
			      &errors);

    if (rc) { /* Process most of  elm.rc options */
	do_rc_process(rc,  LOCAL_RC, 
		      user_rc_set ? user_rc_file: ELMRCFILE,
		      rcset,&errors,
		      read_flags);
	free_parsed_rc(&rc);

	if (process_delayed_enumerated(NULL)) {
	    DPRINT(Debug,9,(&Debug,
			    "Delayed enumerated values OK or not used on %s\n",
			    user_rc_file));
	} else {
	    DPRINT(Debug,9,(&Debug,
			    "Bad delayed  enumerated values on  on %s\n",
			    user_rc_file));
	    errors += 1;
	}
	
    } else if (!open_error)
	errors++;

#ifdef HAVE_SCOPE
    if (dt_flag_is_set(&ipv6_scope_id,scope_auto_flag))
	probe_ipv6_scope_id(read_flags,LOCAL_RC);
#endif

    post_init_check(read_flags);

#ifdef REMOTE_MBX
    user_mail_services_conf = 
	parse_service_entries(user_mail_services,&errors,
			      &user_mail_services_cs, &user_mail_services_pl);
#endif 

    for (i = 0; i < hooklist_rc_len; i++) {

	DPRINT(Debug,3,(&Debug,
			"read_rc_file: executing hooklist_rc entry #%d (total %d) -- %p\n",
			i, hooklist_rc_len, hooklist_rc[i]));

	hooklist_rc[i](&errors);
    }
    
    DPRINT(Debug,3,(&Debug, 
		    "read_rc_file=%d (errors)\n",errors));

    return errors;
}

static void check_minimum_value P_((int *var, const char *varname,
				    const int minimum_value)) UNUSED_FUNCOK;
static void check_minimum_value(var,varname,minimum_value)
     int *var;
     const char *varname;
     const int minimum_value;
{
    if (*var < minimum_value) {
	errors++;
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmChangingMinimumValue,
			  "Changing %s value %d to %d"),
		  varname,
		  *var, minimum_value);

	*var = minimum_value;
	mark_local_changed(var,0);	
    }    
}

void post_init_check(read_flags) 
     int read_flags; 
{

    /* (read_flags & READ_FLAG_IGNORE_MISSING)
       
       is used only for global data
    */
    
    struct stat st_buffer;
    const char *t, *t1;

    DPRINT(Debug,8,(&Debug, 
		    "post_init_check: read_flags = %d%s%s%s%s\n",
		    read_flags,
		    (read_flags & READ_FLAG_GLOBAL_ONLY) ? " READ_FLAG_GLOBAL_ONLY" : "",
		    (read_flags & READ_FLAG_IGNORE_MISSING) ? " READ_FLAG_IGNORE_MISSING" : "",
		    (read_flags & READ_FLAG_UNIDATA_INIT) ? " READ_FLAG_UNIDATA_INIT" : "",
		    (read_flags & READ_FLAG_BINDATA_INIT) ? " READ_FLAG_BINDATA_INIT" : ""
		    ));

    
    t  = give_dt_estr_as_str(&folders_e,"maildir",NULL,NULL);
    t1 = give_dt_estr_as_str(&extra_mailbox_dir_e,
			     "extra-mailbox-dir",NULL,NULL);
       
    if (t1 && t && 0 == strcmp(t1,t)) {
	errors++;
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmExtraNoFolders,
			  "\"extra-mailbox-dir\" should not be \"folders\" or \"maildir\" directory."));	    

	free(extra_mailbox_dir_e.unexpanded);
	extra_mailbox_dir_e.unexpanded = NULL;

	mark_local_changed(& extra_mailbox_dir_e,0);
    }

    if (! defaultfile_e.unexpanded) {
	/* Determine the default mail file name.
	 * 
	 */

	/* May be "none" */
	const char *t2 = 
	    give_dt_estr_as_str(&mailhome_dir_e,"mailhome-dir",NULL,NULL);

	if (t2) {
	    char *X = NULL;

	    if (0 != (read_flags & READ_FLAG_GLOBAL_ONLY) || 
		0 == access(system_rc_file,WRITE_ACCESS)) {

		/* There is automatically added / to end of 
		   "mailhome-dir" -value, so this can only
		   be directory
		*/
		
		
		if (0 != stat(t2,&st_buffer)) {
		    int err = errno;

		    /* DO not check
		       (read_flags & READ_FLAG_IGNORE_MISSING)

		       because this is system directory
		    */

		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmCantAccessMailhomeDir,
				      "Can't access \"mailhome-dir\" %s: %s"),
			      t2,strerror(err));
		    
		    if (! (ESTR_initial &  mailhome_dir_e.flags) &&
			0 != strcmp(t2,MAILHOMEDIR)) {
			
			errors++;				
			free(mailhome_dir_e.unexpanded);
			mailhome_dir_e.unexpanded = MAILHOMEDIR;
			mailhome_dir_e.flags |= ESTR_initial;
			
			mark_local_changed(& mailhome_dir_e,0);
			
			/* does not reset expanded value now */
		    }
		}
	    }
	    
	    X = elm_message(FRM("%s%s"), t2, username);
	    set_dt_estr(&defaultfile_e,X,NULL,0,
			defaultfile_KW_MAIL);
	    free(X);
	} else {
	    DPRINT(Debug,9,(&Debug,"mailhome-dir is set to none\n"));

	    set_dt_estr(&defaultfile_e,NULL,NULL,0,
			defaultfile_KW_MAIL);
	}

    }

    t = give_dt_estr_as_str(&dead_letter_dir_e,"dead-letter-dir",NULL,NULL);
			    
    /* dead-letter-dir is create automatically, if it does not exists */
    if (t &&
	0 == stat(t,&st_buffer) &&
#ifdef S_ISDIR
	!S_ISDIR(st_buffer.st_mode)
#else
	S_IFDIR != (st_buffer.st_mode & S_IFMT)
#endif
	) {
	errors++;
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadDeadLetterDir,
			  "\"dead-letter-dir\" should be directory."));

	free(dead_letter_dir_e.unexpanded);
	dead_letter_dir_e.unexpanded = NULL;

	mark_local_changed(& dead_letter_dir_e,0);
    }
    
    if (! attachment_dir_e.unexpanded) {
	char * tmp = elm_message(FRM("%s/Documents"),home);
	
	if (0 == access(tmp,ACCESS_EXISTS)) {
	    attachment_dir_e.expanded = tmp;
	    attachment_dir_e.unexpanded = safe_strdup("~/Documents");

	} else
	    free(tmp);
    } 
                      
    if ((elm_timeout != 0) && (elm_timeout < 10)) {
	errors++;
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmTimeoutLTTen,
			  "Warning: timeout is set to less than 10 seconds. Ignored."));
	elm_timeout = 0;
    }
    
    if (readdatapercentinc < 1) {
	errors++;
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmReadDataIncrement,
			  "Warning: readdatapercentinc is set to less than 1.  Ignored."));
	readdatapercentinc = 1;

	mark_local_changed(& readdatapercentinc,0);
    }

    if (readmsginc < 1) {
	errors++;
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmReadMessageIncrement,
			  "Warning: readmsginc is set to less than 1.  Ignored."));
	readmsginc = 1;

	mark_local_changed(& readmsginc,0);
    }

    if (sleepmsg < 0) {
	errors++;
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSleepMessageInvalid,
			  "Warning: sleepmsg is set to less than 0.  Setting to 0."));
	sleepmsg = 0;

	mark_local_changed(& sleepmsg,0);
    }
    
#ifdef BACKGROUD_PROCESSES       
    if (background_wait_time  < 0) {
	errors++;
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBackgroundLLT,
			  "Warning: background_wait_time is set less than 0. Setting to 0."));
	background_wait_time = 0;

	mark_local_changed(& background_wait_time,0); 
    }
#endif
        
    if (-1 == auto_copy_sent) {
	if (dt_estr_is_disabled(&sent_mail_e)) {
	    DPRINT(Debug,5,(&Debug,"'copy' defaults to OFF (sentmail is disabled)\n"));
	    auto_copy_sent = 0;
	} else {
	    DPRINT(Debug,5,(&Debug,"'copy' defaults to ON\n"));
	    auto_copy_sent = 1;
	}
    }


    expand_map_bin("unidata", unidata_path, raw_unidata_path,
		   sizeof(unidata_path),read_flags);

    expand_map_bin("bindata", bindata_path, raw_bindata_path,
		   sizeof(bindata_path), read_flags);
    
    map_system_charset(display_locale,display_codeset,0);

       
#if defined(WCHAR) && defined(__STDC_ISO_10646__)
    change_system_charset();
#endif 

    /* Current display charset is locale charset */
    display_charset = system_charset;
    
    if (!special_charset_keyword(raw_text_charset)) {

	if (! load_charset_map_info(&text_charset,
				    raw_text_charset) ) {

	    char buffer[LONG_STRING];

	    do_expand_env("text-charset", buffer, 
			  raw_text_charset, 
			  sizeof(buffer));
	    text_charset = MIME_name_to_charset(buffer,CHARSET_create);
	}

	if (!(CS_mapping & charset_properties(text_charset))) {
	    errors++;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnknownTextCharset,
			      "text-charset %s is unknown, conversion to text-charset does not work"),
		      text_charset->MIME_name ? 
		      text_charset->MIME_name :
		      raw_text_charset);
	    strfcpy(raw_text_charset,"DISPLAY",sizeof raw_text_charset);
	    text_charset = display_charset;

	    mark_local_changed(& raw_text_charset,0);
	}        
    } 
    
    if (!special_charset_keyword(raw_display_charset)) {

	if (! load_charset_map_info(& wanted_display_charset,
				    raw_display_charset )) {

	    char buffer[LONG_STRING];
	    do_expand_env("display-charset", buffer, 
			  raw_display_charset, 
			  sizeof(buffer));
	    wanted_display_charset = MIME_name_to_charset(buffer,CHARSET_create);
	}

	if (!(CS_mapping & charset_properties(wanted_display_charset))) {
	    errors++;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmUnknownDisplayCharset,
			      "display-charset %s is unsupported, charset switching does not work"),
		      wanted_display_charset->MIME_name ? 
		      wanted_display_charset->MIME_name :
		      raw_display_charset);
	    strfcpy(raw_display_charset,"SYSTEM",
		    sizeof raw_display_charset);
	    wanted_display_charset = system_charset;

	    mark_local_changed(& raw_display_charset,0);
	}
    }

    if (!special_charset_keyword(raw_local_fs_charset)) {

	if (! load_charset_map_info(&local_fs_charset,
				    raw_local_fs_charset) ) {
	    char buffer[LONG_STRING];
	    do_expand_env("local-fs-charset", buffer, 
			  raw_local_fs_charset, 
			  sizeof(buffer));
	    local_fs_charset = MIME_name_to_charset(buffer,CHARSET_create);
	}

	if (!(CS_mapping & charset_properties(local_fs_charset))) {
	    errors++;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmUnknownFSCharset,
			      "local-fs-charset %s is unsupported, file access does not work"),
		      local_fs_charset->MIME_name ? 
		      local_fs_charset->MIME_name :
		      raw_local_fs_charset);
	    strfcpy(raw_local_fs_charset,"SYSTEM",
		    sizeof raw_local_fs_charset);
	    local_fs_charset = system_charset;		

	    mark_local_changed(& raw_local_fs_charset,0);
	}
    }
    
    if (!special_charset_keyword(raw_default_nomime_charset)) {

	if (! load_charset_map_info(& default_nomime_charset,
				    raw_default_nomime_charset) ) {
	    char buffer[LONG_STRING];

	    do_expand_env("default-nomime-charset", buffer, 
			  raw_default_nomime_charset, 
			  sizeof(buffer));
	    default_nomime_charset = MIME_name_to_charset(buffer,CHARSET_create);
	}

	if (!(CS_mapping & charset_properties(default_nomime_charset))) {
	    errors++;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmUnknownDef1Charset,
			      "default-nomime-charset %s is unsupported"),
		      default_nomime_charset->MIME_name ? 
		      default_nomime_charset->MIME_name :
		      raw_default_nomime_charset);

		strfcpy(raw_default_nomime_charset,"SYSTEM",
			sizeof raw_default_nomime_charset);
		default_nomime_charset = system_charset;		

		mark_local_changed(&  raw_default_nomime_charset,0);
	}
    }

    if (!special_charset_keyword(raw_default_mimetext_charset)) {

	if (! load_charset_map_info(& default_mimetext_charset,
				    raw_default_mimetext_charset) ) {

	    char buffer[LONG_STRING];
	    do_expand_env("default-mime-text-charset", buffer, 
			  raw_default_mimetext_charset, 
			  sizeof(buffer));
	    default_mimetext_charset = MIME_name_to_charset(buffer,CHARSET_create);
	}

	if (!(CS_mapping & charset_properties(default_mimetext_charset))) {
	    errors++;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmUnknownDef2Charset,
			      "default-mime-text-charset %s is unsupported"),
		      default_mimetext_charset->MIME_name ? 
		      default_mimetext_charset->MIME_name :
		      raw_default_mimetext_charset);

		strfcpy(raw_default_mimetext_charset,"US-ASCII",
			sizeof raw_default_mimetext_charset);
		default_mimetext_charset = ASCII_SET;		

		mark_local_changed(&  raw_default_mimetext_charset,0);
	}
    }

    
#ifdef REMOTE_MBX
    if (!special_charset_keyword(raw_imap_charset)) {

	if (! load_charset_map_info(& imap_charset,
				    raw_imap_charset) ) {

	    char buffer[LONG_STRING];
	    do_expand_env("imap-charset", buffer, 
			  raw_imap_charset, 
			  sizeof(buffer));
	    imap_charset = MIME_name_to_charset(buffer,CHARSET_create);
	}

	if (!(CS_mapping & charset_properties(imap_charset))) {
	    errors++;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmUnknownIMAPCharset,
			      "imap-charset %s is unsupported"),
		      imap_charset->MIME_name ? 
		      imap_charset->MIME_name :
		      raw_imap_charset);

	    strfcpy(raw_imap_charset,"SYSTEM",
		    sizeof raw_imap_charset);
	    imap_charset = system_charset;		

	    mark_local_changed(&  raw_imap_charset,0);
	}
    }

    check_minimum_value(&pop_idle_alive_interval,
			"pop-idle-alive-interval",
			MIN_POP_TIMEOUT);
    check_minimum_value(&imap_idle_alive_interval,
			"imap-idle-alive-interval",
			MIN_IMAP_TIMEOUT);
    check_minimum_value(&name_lookup_cache_time,
			"name-lookup-cache-time",
			MIN_NAME_CACHE_TIME);
			
#endif
    
    post_process_charset_options();
    
    if (!(CS_mapping & charset_properties(system_charset))) {
	/* Don't increment errors variable because this comes
	 * from locale (via mime.charsets) and not from elmrc
	 */
	
	if (system_charset->MIME_name)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmUnsupportedLocale,
			      "WARNING: Locale %s (charset %s) is unsupported, will cause problems!"), 
		      display_locale,system_charset->MIME_name);
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmUnsupportedLocale2,
			      "WARNING: Locale %s is unsupported, will cause problems!"), 
		      display_locale);
	locale_error = 1;
    } 
    
    if (locale_error == 1) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmLocaleProblem,
			  "Problem with locale (system character set)! Elm ME+ will behave erratically."));
	/* Long wait that message will be noticed ... */
	if (POLL_method)
	    wait_for_timeout(10);
	else
	    sleep(10);
	
	locale_error++;
    }
    
    if (
	(
	 (wanted_display_charset != system_charset &&	     
	  0 != (CS_mapping & charset_properties(wanted_display_charset)) &&
	  (!wanted_display_charset->MIME_name ||
	   0 != istrcmp(wanted_display_charset->MIME_name,"US-ASCII")) 
#ifndef ASCII_CTYPE
	  && (!wanted_display_charset->MIME_name ||
	      0 != istrcmp(wanted_display_charset->MIME_name,"ISO-8859-1"))
#endif
	  )
	 || allow_charset_switching
	 ) &&
	0 != strcmp(unidata_path,"none")) {
	
	/* We are going to need unidata information --
	   therefore check availivility ! */
	
	if (0 == (read_flags & READ_FLAG_UNIDATA_INIT) 
	    &&
	    !default_unidata()) {
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet,ElmNoUnicodeData,
			      "No Unicode data available: unidata: %s"),
		      raw_unidata_path);
	    
	    errors++;
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet,ElmUnidataSetting,
			      "Setting unidata=none, displaycharset=SYSTEM, allow-charset-switching=OFF"));
	    
	    strfcpy(raw_unidata_path,"none",sizeof raw_unidata_path);
	    strfcpy(unidata_path,"none",sizeof unidata_path);
	    
	    strfcpy(raw_display_charset,"SYSTEM", 
		    sizeof raw_display_charset);
	    wanted_display_charset = system_charset;		

	    allow_charset_switching = 0;

	    mark_local_changed(& raw_unidata_path,0);
	    mark_local_changed(& raw_display_charset,0);
	    mark_local_changed(& allow_charset_switching,0);
	}	
    }
    
    if (0 != strcmp(bindata_path,"none") &&
	0 != strcmp(bindata_path_global,bindata_path)) {

	if (0 == (read_flags & READ_FLAG_BINDATA_INIT) 
	    &&
	    !default_bindata()) {
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet,ElmBinaryMapData,
			      "No binary map data available: bindata: %s"),
		      raw_bindata_path);
	    
	    errors++;

	    strfcpy(raw_bindata_path,"none",sizeof raw_bindata_path);
	    strfcpy(bindata_path,"none",sizeof bindata_path);

	    mark_local_changed(& raw_bindata_path,0);
	}
    }
    
    if (use_PGP &&
	pgp_none == give_dt_enumerate_as_int(&send_pgp_version)) {
	const char * gpg_path_val  = give_dt_estr_as_str(&gpg_path_e, "gpg", 
							 NULL,NULL);
	const char * pgp2_path_val = give_dt_estr_as_str(&pgp2_path_e, "pgp2",
							 NULL,NULL);
	const char * pgp5_dir_val  = give_dt_estr_as_str(&pgp5_dir_e, 
							 "pgp5-dir",NULL,NULL);

	if (gpg_path_val &&  '/' == gpg_path_val[0]) 
	    send_pgp_version.val = gpg;
	if (pgp5_dir_val && '/' == pgp5_dir_val[0]) 
	    send_pgp_version.val = pgp5;
	if (pgp2_path_val && '/' == pgp2_path_val[0]) 
	    send_pgp_version.val = pgp2;
    }
    
    if (DEFAULT_SORT_ORDER == fb_sort_order.val) {
	fb_sort_order.val = local_dir_sortby.val;

	DPRINT(Debug,10,(&Debug, 
			 "file-browser-sortby = %d (from local-dir-sortby)\n",
			 fb_sort_order.val));

    }

    if (-1 == fb_show_title) {
	fb_show_title = user_level.val == user_level_beginner;

	DPRINT(Debug,10,(&Debug, 
			 "file-browser-show-title = %d (userlevel = %d)\n",
			 fb_show_title, user_level.val));
    }

    if (0L == mailhome_locking.val) {

	if (mailbox_locking.val)
	    mailhome_locking.val = mailbox_locking.val;
	else
	    mailhome_locking.val = 0L
#ifdef USE_DOTLOCK_LOCKING
		| FLAGVAL(lock_lockfile_flag)
#endif 
#ifdef USE_FLOCK_LOCKING
		| FLAGVAL(lock_flock_flag)
#endif
#ifdef USE_FCNTL_LOCKING
		| FLAGVAL(lock_fcntl_flag)
#endif
		;

	if (mailhome_locking.val_as_string) {
	    free (mailhome_locking.val_as_string);
	    mailhome_locking.val_as_string = NULL;
	}

	/* This still may be empty, if no defaults
	   are given on compile time!
	*/

	if (0 != (read_flags & READ_FLAG_GLOBAL_ONLY) || 
	    0 == access(system_rc_file,WRITE_ACCESS)) {
	    
	    const char *s1 = give_dt_flag_as_str(&mailhome_locking);

	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoLocking,
			      "No locking given on %s, using: %s"),
		      "mailhome-locking",s1);

	    if (mailhome_locking.val) {
		mark_local_changed(& mailhome_locking,0);
		errors++;
	    }
	}
    }

    if (0L == mailbox_locking.val) {
	const char *s1;

	if (mailhome_locking.val)
	    mailbox_locking.val = mailhome_locking.val;
	else
	    mailbox_locking.val = 0L
#ifdef USE_DOTLOCK_LOCKING
		| FLAGVAL(lock_lockfile_flag)
#endif 
#ifdef USE_FLOCK_LOCKING
		| FLAGVAL(lock_flock_flag)
#endif
#ifdef USE_FCNTL_LOCKING
		| FLAGVAL(lock_fcntl_flag)
#endif
		;


	if (mailbox_locking.val_as_string) {
	    free (mailhome_locking.val_as_string);
	    mailbox_locking.val_as_string = NULL;
	}

	/* This still may be empty, if no defaults
	   are given on compile time!
	*/

	s1 = give_dt_flag_as_str(&mailbox_locking);

	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmNoLocking,
			  "No locking given on %s, using: %s"),
		  "mailbox-locking",s1);
	
	if (mailbox_locking.val) {
	    mark_local_changed(& mailbox_locking,0);
	    errors++;
	}
    }

    if (lockfolders &&
	0L == folder_locking.val) {
	const char *s1;

	if (mailbox_locking.val)
	    folder_locking.val = 
		mailbox_locking.val & ~ (FLAGVAL(lock_lockfile_flag));
	else
	    folder_locking.val = 0L
#ifdef USE_FLOCK_LOCKING
		| FLAGVAL(lock_flock_flag)
#endif
#ifdef USE_FCNTL_LOCKING
		| FLAGVAL(lock_fcntl_flag)
#endif
		;	

	if (folder_locking.val_as_string) {
	    free (folder_locking.val_as_string);
	    folder_locking.val_as_string = NULL;
	}

	s1 = give_dt_flag_as_str(&folder_locking);

	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmNoLocking,
			  "No locking given on %s, using: %s"),
		  "folder-locking",s1);

	if (folder_locking.val) {
	    mark_local_changed(& folder_locking,0);
	    errors++;
	}
    }

    if (address_lookup.val > 0) {
	int valid_lookup = 1;

	switch (address_lookup.val) {
	case addr_lookup_shared:
	    valid_lookup = 0;   /* RESERVED */
	    break;
	case addr_lookup_getnameinfo:
#ifndef HAVE_GETNAMEINFO
	    valid_lookup = 0;
#endif
	    break;
	}
	
	if (!valid_lookup && 
	    address_lookup.val < NUM_addr_lookup) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmAddressLookUnsupported,
			      "Unsupported lookup method: address-lookup: %s"),
		      ADDRESS_LOOKUP[address_lookup.val]);
	    
	    errors++;
	    address_lookup.val = 0;
	    mark_local_changed(& address_lookup,0 );
	}		
    }
    

    /* Negative values indicate that value is tagged (shared value) */
    if (mail_services_lookup.val > 0) {
	int valid_lookup = 1;

	switch (mail_services_lookup.val) {
	case mserv_lookup_shared:
#ifndef USE_DLOPEN
	    valid_lookup = 0;
#endif
	    break;
	    
	case mserv_lookup_getaddrinfo:
#ifndef HAVE_ADDRINFO
	    valid_lookup = 0;
#endif
	    break;
	}
           
	if (!valid_lookup && 
	    mail_services_lookup.val < NUM_mserv_lookup) {

	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmMailServLookUnsupported,
			      "Unsupported lookup method: mail-services-lookup: %s"),
		      MSERV_LOOKUP[mail_services_lookup.val]);

	    errors++;
	    mail_services_lookup.val = 0;
	    mark_local_changed(& mail_services_lookup,0 );
	}
    }

#ifdef USE_DLOPEN
    post_init_shared_options(&errors,read_flags);

    post_init_change_config(&errors,read_flags,NULL  /* no tag */,
			    save_info,
			    NUMBER_OF_SAVEABLE_OPTIONS,
			    sizeof (save_info[0]));
#endif
}



/* Return rest */

char * rc_split_keyword(buffer,line_len,keyword,size,tag,
			    assig,restlen)
     char *buffer; 
     int line_len;				
     char *keyword; 
     int size;
     char **tag;
     int *assig;
     int *restlen;
{
    int i;
    char * c;

    DPRINT(Debug,10,(&Debug,
		     "rc_split_keyword len=%d, buffer=%s\n",
		     line_len,buffer));

    *assig = 0;

    i = 0;
    for (c = buffer; c - buffer < line_len && i < size; c++) {
	if ('_' == *c)
	    keyword[i++] = '-';

	/* allow @charset */
	else if (0 == i && '@' == *c)
	    keyword[i++] = '@';

	else if (!*tag && ':' == *c) {
	    /* Mark tag from buffer */
	    *c = '\0';
	    *tag = buffer;
	    i = 0;
	} else if (!isascii(*c))
	    break;
	else if (!ok_rc_char(*c))
	    break;
	else {
	    keyword[i++] = tolower((unsigned char)*c);
	}
    }

    keyword[i] = '\0';
	
    while (whitespace(*c)) 
	c++;
    
    if ('=' == *c) {
	c++;
	*assig = 1;
    }
    while (whitespace(*c)) 
	c++;

    DPRINT(Debug,10,(&Debug,
		     "rc_split_keyword: keyword(i=%d)=%s\n",
		     i,keyword));

    if (*tag) {
	DPRINT(Debug,10,(&Debug,
			 "rc_split_keyword: tag=%s\n",*tag));	
    }

    *restlen = line_len - (c - buffer);

    DPRINT(Debug,10,(&Debug,
		     "rc_split_keyword=%s, *restlen=%d\n",c,*restlen));	

    return c;
}

/* ------------------------------------------------------------------------  
 *
 * set the named parameter according to save_info structure.
 */



/* read_rc.c. -- return NULL on failure 
                 otherwise pointer to option
 */


struct rc_save_info_rec *rc_locate_option(rc_options,rc_option_count,
					  word1,negate)
     struct rc_save_info_rec * rc_options;
     size_t                    rc_option_count;
     const char              * word1; 
     int                     * negate;
{
    int y = -1;
    size_t x;
    
    if (negate) {
	if ('!' == *word1) {
	    *negate = 1;
	    word1++;

	    DPRINT(Debug,18,(&Debug,"rc_locate_option: nagated option = %s\n",
			     word1));
	}  
    }

    for (x=0; x < rc_option_count; ++x) {
	y = strcmp(word1, rc_options[x].name);
	if (y <= 0) 
	    break;
    }

    if (y != 0) {

	DPRINT(Debug,18,(&Debug,"rc_locate_option: option = %s, not found\n",
			 word1));
	return NULL;
    }

    DPRINT(Debug,18,(&Debug,"rc_locate_option: option = %s, found [%lu]\n",
		     word1,
		     (unsigned long)x));

    return & (rc_options[x]);
}

static int rc_eval_tail_verify P_((struct rc_save_info_rec * ret,
				   enum record_mode          lcl,
				   int                       lineno,
				   const char              * filename));
static int rc_eval_tail_verify(ret,lcl,lineno,filename)
     struct rc_save_info_rec * ret;
     enum record_mode          lcl;
     int                       lineno;
     const char              * filename;
{
    switch (lcl) {
    case  LOCAL_RC:
	ret->flags |= FL_LOCAL;
	break;
    case SYSTEM_RC:
    case RC_MODE_COUNT:
	break;
    }
    ret->flags |= FL_CHANGED;
    
    if ( ret->conditional_option.conditional_option_p &&
	 ! *(ret->conditional_option.conditional_option_p)) {

	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmDisabledKeyInElmrc,
			  "Keyword \"%s\" is not available in line %d in \"%s\" file"),
		  ret->name,lineno,filename);
	
	return 0;
    }

    return 1;
}

int rc_eval_tail_cline(ret,lcl,value,lineno,filename,read_flags)
     struct rc_save_info_rec * ret;
     enum record_mode          lcl;
     struct string           * value;
     int                       lineno; 
     const char              * filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int code = rc_eval_tail_verify(ret,lcl,lineno,filename);
    
    if (RCTYPE_magic != ret->dt_type->magic)
	panic("RC PANIC",__FILE__,__LINE__,"rc_eval_tail_cline",
	      "Bad config item type",0);

    if (code) {
	DPRINT(Debug,10,(&Debug,"%s:%d: [cont] %s=%S\n",
			 filename,lineno,ret->name,value));
	
	code = ret->dt_type->parse_cline(ret,
					 lcl,
					 value,
					 lineno,
					 filename,
					 read_flags);
    }

    return code;
}

int rc_eval_tail(ret,lcl,value,lineno,filename,
		 negate,read_flags)
     struct rc_save_info_rec *ret;
     enum record_mode          lcl;
     struct string *value;
     int lineno; 
     const char *filename; 
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int code = rc_eval_tail_verify(ret,lcl,lineno,filename);
    
    if (RCTYPE_magic != ret->dt_type->magic)
	panic("RC PANIC",__FILE__,__LINE__,"rc_eval_tail",
	      "Bad config item type",0);

    if (code) {
	DPRINT(Debug,10,(&Debug,"%s:%d: [%s] %s%s=%S\n",
			 filename,lineno,
			 negate ? "nega" : "line",
			 negate ? "!" : "",
			 ret->name,value));
	
	code = ret->dt_type->parse_line(ret,
					lcl,
					value,
					lineno,
					filename,
					negate,
					read_flags) ;
    }

    return code;
}

int rc_eval_tail_line(ret,lcl,value,lineno,filename,line_mode,
		      read_flags)
     struct rc_save_info_rec *ret;
     enum record_mode          lcl;
     struct string *value;
     int lineno;
     const char *filename;
     enum rec_line_mode line_mode;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int code = rc_eval_tail_verify(ret,lcl,lineno,filename);

    if (RCTYPE_magic != ret->dt_type->magic)
	panic("RC PANIC",__FILE__,__LINE__,"rc_eval_tail_line",
	      "Bad config item type",0);

    if (code) {
	switch (line_mode) {
	case line_normal:
	    
	    DPRINT(Debug,10,(&Debug,"%s:%d: [line] %s=%S\n",
			     filename,lineno,ret->name,value));
	    
	    code = ret->dt_type->parse_line(ret,
					    lcl,
					    value,
					    lineno,
					    filename,
					    0,
					    read_flags) ;
	    break;
	case  line_negate: 
	    DPRINT(Debug,10,(&Debug,"%s:%d: [nega] !%s=%S\n",
			     filename,lineno,ret->name,value));
	    
	    code = ret->dt_type->parse_line(ret,
					    lcl,
					    value,
					    lineno,
					    filename,
					    1,
					    read_flags);
	    break;
	    
	case line_continue:
	    DPRINT(Debug,10,(&Debug,"%s:%d: [cont] %s=%S\n",
			     filename,lineno,ret->name,value));
	    
	    code = ret->dt_type->parse_cline(ret,
					     lcl,
					     value,
					     lineno,
					     filename,
					     read_flags);
	    break;
	}
    }
    
    return code;
}

struct rc_save_info_rec * do_set(lcl, word1, word2, filename,
				 read_flags)
     enum record_mode   lcl;
     char             * word1;
     struct string    * word2;
     const char       * filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int negate = 0;

    struct rc_save_info_rec * ret =  rc_locate_option(save_info,
						   NUMBER_OF_SAVEABLE_OPTIONS,
						   word1,
						   &negate);

    if (!ret) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeyInElmrc,
			  "I can't understand keyword \"%s\" in line %d in \"%s\" file"),
		  word1, lineno,filename);
	
	return NULL;
    }

    
    if (!rc_eval_tail(ret,lcl,word2,lineno,filename,negate,
		      read_flags))
	return NULL;

    return ret;
}

static const char WEED_NO_DEFAULT[] = "*clear-weed-list*";
	

int matches_weedlist(buffer,have_title)
     char *buffer;
     int have_title;
{
	/** returns true iff the first 'n' characters of 'buffer' 
	    match an entry of the weedlist **/
	
	int i;

	static char *default_list_title[] = 
	    { ">From", "In-Reply-To:",
	      "References:", "Newsgroups:", "Received:",
	      "Apparently-To:", "Message-Id:", "Content-Type:",
	      "Content-Length:", "MIME-Version:",
	      "Content-Transfer-Encoding:",
	      "Content-Language:", "Content-Disposition:",
	      "From", "X-Mailer:", "User-Agent:", "Status:",
	      "X-ELM-",
	      "X-UIDL", /* Generated by some POP deamons */ 
	      "Return-Path:",
	      "List-ID:", 
	      "List-Help:", "List-Subscribe:", "List-Unsubscribe:",
	      "List-Post:", "List-Owner:", "List-Archive:",
	      "Auto-Submitted:",

	      "Resent-Message-ID:",   /* Other Resent- header fields
					 should be visible */
	      
	      "Precedence:", "Errors-To:", 

	      /* RFC 6376: DomainKeys Identified Mail (DKIM) Signatures */
	      "DKIM-Signature:", 
	      
	      /* Microsoft Exchange: */
	      "Thread-Topic:", "Thread-Index:", "X-MS-Has-Attach:",
	      "X-MS-TNEF-Correlator:","X-MS-Exchange-",
	      "X-Auto-Response-Suppress:",
	      "X-Originating-IP:", "Accept-Language:",
	      
	      /* Sendmail and Elm ME+ 2.5 */
	      "X-MIME-Autoconverted:",

	      /* Postfix */
	      "Delivered-To:",

	      /* Communigate Pro */
	      "X-Autogenerated:",

	      "*end-of-defaults*", NULL
	    };

	static char *default_list_notitle[] = 
	    { "In-Reply-To:",
	      "References:", "Newsgroups:", "Received:",
	      "Apparently-To:", "Message-Id:", "Content-Type:",
	      "Content-Length:", "MIME-Version:",
	      "Content-Transfer-Encoding:",
	      "Content-Language:", "Content-Disposition:",
	      "X-Mailer:", "User-Agent:", "Status:",
	      "X-ELM-",
	      "X-UIDL", /* Generated by some POP deamons */ 
	      "Return-Path:",
	      "List-ID:", 
	      "List-Help:", "List-Subscribe:", "List-Unsubscribe:",
	      "List-Post:", "List-Owner:", "List-Archive:",
	      "Auto-Submitted:",

	      "Resent-Message-ID:",   /* Other Resent- header fields
					 should be visible */
	      
	      "Precedence:", "Errors-To:", 

	      /* RFC 6376: DomainKeys Identified Mail (DKIM) Signatures */
	      "DKIM-Signature:", 
	      
	      /* Microsoft Exchange: */
	      "Thread-Topic:", "Thread-Index:", "X-MS-Has-Attach:",
	      "X-MS-TNEF-Correlator:","X-MS-Exchange-",
	      "X-Auto-Response-Suppress:",
	      "X-Originating-IP:", "Accept-Language:",
	      
	      /* Sendmail and Elm ME+ 2.5 */
	      "X-MIME-Autoconverted:",

	      /* Postfix */
	      "Delivered-To:",

	      /* Communigate Pro */
	      "X-Autogenerated:",
	      
	      "*end-of-defaults*", NULL
	    };


	const char **list = NULL;
	static char **default_list = NULL;

	DPRINT(Debug,12,(&Debug,"matches_weedlist: %s title, buffer=%s\n",
			 have_title ? "have" : "no",
			 buffer));

	if (have_title) {
	    list = give_dt_path_as_elems(&weedlist,"weedout");
	    default_list = default_list_title;
	} else {
	    list = give_dt_path_as_elems(& weedlist_no_title,
					 "weedout-without-title");
	    default_list= default_list_notitle;
	}

	/* If fist elem is "*clear-weed-list*" then default list is not 
	   used
	*/

	if (!list || !list[0] ||
	    0 != strcmp(list[0],WEED_NO_DEFAULT)) {
	    

	    for (i = 0; default_list[i]; i++) {
		if (strincmp(buffer, default_list[i], strlen(default_list[i])) == 0)  {
		    DPRINT(Debug,12,(&Debug,
				     "matches_weedlist=1, matches to default: %s\n",
				     default_list[i]));
		    return(1);
		}
	    }	    
	}

	if (list) {
	    for (i=0;list[i]; i++) {
		if (strincmp(buffer, list[i], strlen(list[i])) == 0) {
		    return(1);
		    DPRINT(Debug,12,(&Debug,
				     "matches_weedlist=1, matches to user's list: %s\n",
				     list[i]));

		}
	    }
	}

	DPRINT(Debug,12,(&Debug,
			 "matches_weedlist=0, no match\n"));
	
	return(0);
}


char * mode_to_str(mode)
     int mode;
{
	static char modestr[9+1];	/* "rwxrwxrwx\0" */
	char *s = &modestr[0];
	char *modecharp = "rwxrwxrwx";
	int modebit = 0400;

	while (modebit != 0) {
	    if (mode & modebit)
	        *s++ = *modecharp;
	    else
		*s++ = '-';
	    modecharp++;
	    modebit >>= 1;
	}
	*s = '\0';
	return(modestr);
}

int null_option_func(value,enter, lineno, filename,
		     read_flags)
     char **value; 
     int enter;
     int lineno; 
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    return 0;
}

static free_rc_hook **hooklist_free = NULL;
static int            hooklist_free_len = 0;

void add_free_rc_hook(hook)
     free_rc_hook *hook;
{
    DPRINT(Debug,3,(&Debug,"add_free_rc_hook: -- procedure %p\n",hook));
    
    hooklist_free = safe_array_realloc(hooklist_free,
				       (hooklist_free_len+1),
				       sizeof(hooklist_free[0]));
    
    hooklist_free[hooklist_free_len] = hook;

    DPRINT(Debug,3,(&Debug,"add_free_rc_hook: stored hooklist entry #%d  -- %p\n",
		    hooklist_free_len,hook));

    hooklist_free_len++;
}

void free_rc_options() 
{
    int i;

    if (nameres_return_buffer)
	free(nameres_return_buffer);
    nameres_return_buffer = NULL;

    for (i = 0; i < nr_COUNT; i++) {

	if (name_resolutions[i].result)
	    free(name_resolutions[i].result);
	name_resolutions[i].result = NULL;
    }

    for (i = 0; i < NUMBER_OF_SAVEABLE_OPTIONS; i++) {

	if (RCTYPE_magic != save_info[i].dt_type->magic) 
	    panic("RC PANIC",__FILE__,__LINE__,"free_rc_options",
		  "Bad config item type",0);
	    
	DPRINT(Debug,35,(&Debug,
			 "free_rc_options: Freeing save_info[%d] %p %s\n",
			 i, & (save_info[i]), save_info[i].name));

	save_info[i].dt_type->free_option(& (save_info[i]));
    }

    free_delayed_rc_options();
    free_delayed_enumerated_list();
    free_delayed_blocks();
    
    if (system_mail_services_pl)
	free_editor_propline(&system_mail_services_pl);
    if (user_mail_services_pl)
	free_editor_propline(&user_mail_services_pl);
    
#ifdef REMOTE_MBX
    if (system_mail_services_conf)
	free_mail_services_conf(& system_mail_services_conf);
    if (user_mail_services_conf)
	free_mail_services_conf(& user_mail_services_conf);
    free_service_list();
    free_static_hosts();
#endif
    free_transaction_file();
	
    if (system_iso2022_map_conf)
	free_iso2022_map_conf(& system_iso2022_map_conf);
    if (user_iso2022_map_conf)
	free_iso2022_map_conf(& user_iso2022_map_conf);

    free_iso2022_map_list();

    if (name_buffer) {
	int i;

	for (i = 0; i < name_buffer_count; i++) {
	    if (name_buffer[i]) {
		free (name_buffer[i]);
		name_buffer[i] = NULL;
	    }	    
	}

	free(name_buffer);
	name_buffer = NULL;
    }
    name_buffer_count = 0;

    if (hooklist_free) {
	int i;

	for (i = hooklist_free_len; i > 0; ) {
	    i--;

	    DPRINT(Debug,3,(&Debug,
			    "free_rc_options: executing hooklist_free entry #%d (total %d) -- %p\n",
			    i, hooklist_free_len, hooklist_free[i]));

	    hooklist_free[i]();
	}

	free(hooklist_free);
	hooklist_free = NULL;
    }
    hooklist_free_len = 0;

    if (hooklist_rc) {
	free(hooklist_rc);
	hooklist_rc = NULL;
    }
    hooklist_rc_len = 0;

    if (hooklist) {
	free(hooklist);
	hooklist = 0;
    }
    hooklist_len = 0;
    
#ifdef REMOTE_MBX
    free_interface_list();
    free_remote_mbx_config();
#endif

#ifdef USE_DLOPEN
    free_shared_rc_options();
#endif

    free_schedule();
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
