static char rcsid[] = "@(#)$Id: rc_handle.c,v 2.42 2025/05/29 05:42:56 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.42 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ****************************************************************************
 *
 * Some of code comes from Elm 2.4 src/read_rc.c and src/save_opts.c. These
 * are following copyright:
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

#include "elm_defs.h"
#include "rc_imp.h"
#include "save_opts.h"
#ifdef USE_DLOPEN
#include "shared_imp.h"
#endif
#include "s_elm.h"
#include "s_elmrc.h"

/* For INT_MAX, INT_MIN */
#include <limits.h>

DEBUG_VAR(Debug,__FILE__,"config");

static const char ** ps2cps P_((char **str));
static const char ** ps2cps(str)
     char **str;
{
    return (const char **)str;
}

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

/* Return 1 if no errors */
int convert_to_system_charset(value,lineno,filename,buffer,buffer_len)
     struct string *value;
     int lineno;
     const char *filename;
     char **buffer;
     int *buffer_len;
{
    int ret = 1;
    const char *csn = get_string_MIME_name(value);
    int ERRORS = 0;
    struct string *RS = convert_string2(system_charset,value,&ERRORS);

    if (ERRORS) {
	const char * sysnam = get_charset_MIME_name(system_charset);
	
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmConvertCharsetErrors,
			  "There is %d errors when converting line %d from charset %s to charset %s in \"%s\" file"),
		  ERRORS,lineno,
		  csn ? csn : "<no MIME name>",
		  sysnam ? sysnam  : "<no MIME name>",
		  filename);

	ret = 0;
    }

    if (*buffer) {
	free(*buffer);
	*buffer  = NULL;
    }

    /* result is malloced */
    bytestream_from_string(RS,buffer,buffer_len);

    free_string(&RS);

    DPRINT(Debug,20,(&Debug, 
		     "convert_to_system_charset [%s:%d] = %d, *buffer_len=%d\n",
		     filename,lineno,ret,*buffer_len));

    return ret;
}

int sconvert_to_rcset(S,fieldname,rcset,buffer,buffer_len)
     struct string *S;
     const char *fieldname;
     charset_t rcset;
     char **buffer;
     int *buffer_len;
{
    int ret = 1;
    const char *csn = get_string_MIME_name(S);
    const char * rcsetnam = get_charset_MIME_name(rcset);
    int ERRORS;
    struct string * res_S = convert_string2(rcset,S,&ERRORS);

    if (ERRORS) {
	if (fieldname)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmConvKeyCharsetErrors,
			      "There is %d errors when converting key \"%s\" from charset %s to charset %s"),
		      ERRORS,fieldname,
		      csn ? csn : "<no MIME name>",
		      rcsetnam ? rcsetnam : "<no MIME name>");

	ret = 0;
    }	      
    
    if (*buffer) {
	free(*buffer);
	*buffer  = NULL;
    }
    
    /* result is malloced */
    bytestream_from_string(res_S,buffer,buffer_len);

    free_string(&res_S);

    return ret;
}

/* in_buffer_len = -1 ... uses  strlen() 
   0 == failure, 1 = ok
*/					 
int convert_to_rcset(in_buffer,in_buffer_len,fieldname,rcset,buffer,buffer_len)
     const char * in_buffer;
     int in_buffer_len;
     const char *fieldname;
     charset_t rcset;
     char **buffer;
     int *buffer_len;
{
    int ret = 1;
    const char * sysnam   = get_charset_MIME_name(system_charset);
    struct string * S = new_string(system_charset);
    int ERRORS;
    int rs;

    if (in_buffer_len < 0)
	in_buffer_len = strlen(in_buffer);

    rs = add_streambytes_to_string(S,in_buffer_len,cs2us(in_buffer),&ERRORS);

    if (rs < in_buffer_len) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedKeyAsCharset,
			  "Failed to parse key \"%s\" as charset %s"),
		  fieldname,sysnam);
	ret = 0;
    }

    if (ERRORS) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmKeyAsCharsetErrors,
			  "There is %d errors when parsing key \"%s\" as charset %s"),
		  ERRORS,fieldname,sysnam);
	ret = 0;
    }

    if (! sconvert_to_rcset(S,fieldname,rcset,buffer,buffer_len))
	ret = 0;
   
    free_string(&S);

    return ret;
} 



S_(rc_parse_line dt_SYN_parse_line)
static int dt_SYN_parse_line P_((struct rc_save_info_rec *r,
				 enum record_mode          lcl,
				 struct string *value, int lineno, 
				 const char *filename,
				 int negate,
				 int read_flags /* READ_FLAG_IGNORE_MISSING */));
static int dt_SYN_parse_line(r,lcl,value,lineno,filename, negate,
			     read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    int ret = 0;

    if (negate) {
	char buffer[SLEN];
        buffer[0] = '!';
        strfcpy(buffer+1, r->val.str, sizeof buffer -1);
	
	if ( do_set(lcl,buffer,value,filename,
		    read_flags))
            ret = 1;
    } else { 
	if ( do_set(lcl, r->val.str, value, filename,
		    read_flags))
	    ret = 1;
    }

    DPRINT(Debug,20,(&Debug, 
		     "dt_SYN_parse_line [%s:%d] = %d\n",
		     filename,lineno,ret));

    return ret;
}

S_(rc_parse_cline BAD_parse_cline)
static int BAD_parse_cline P_((struct rc_save_info_rec *r,
			       enum record_mode          lcl,
			       struct string *value, int lineno, 
			       const char *filename,
			       int read_flags /* READ_FLAG_IGNORE_MISSING */
			       ));
static int BAD_parse_cline(r,lcl,value,lineno,filename,
			   read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadContinuationInElmrc,
		      "Key \"%s\" does not accept continuation lines in line %d in \"%s\" file"),
	      r->name,lineno,filename);
    return 0;    
}

S_(rc_print_value NO_print_value)
static void NO_print_value P_((FILE *F,
			       struct rc_save_info_rec *r,
			       int comment,
			       charset_t rcset));
static void NO_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
{
    /* Empty */
}

S_(rc_get_value NO_get_value)
static char * NO_get_value P_((struct rc_save_info_rec *r));
static char * NO_get_value(r)
     struct rc_save_info_rec *r;
{
    return "*bad*";
}

S_(rc_free_option NO_free_option)
static void NO_free_option  P_((struct rc_save_info_rec *r));
static void NO_free_option(r)
     struct rc_save_info_rec *r;
{
    /* Empty */
}


struct rc_type rc_DT_SYN = { RCTYPE_magic,
			     dt_SYN_parse_line, BAD_parse_cline,
			     NO_print_value, NO_get_value,  
			     NO_free_option };

S_(rc_parse_line dt_STR_parse_line)
static int dt_STR_parse_line P_((struct rc_save_info_rec *r,
				 enum record_mode          lcl,
				 struct string *value, int lineno, 
				 const char *filename,
				 int negate,
				 int read_flags /* READ_FLAG_IGNORE_MISSING */
				 ));
static int dt_STR_parse_line(r,lcl,value,lineno,filename,
			     negate,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    int ret;
    char * buffer = NULL;
    int buffer_len = 0;

    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    }

    ret = convert_to_system_charset(value,lineno,filename,&buffer,&buffer_len);

    if (buffer_len >= r->size_val) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLongValueInElmrc,
			  "Value of \"%s\" in line %d in \"%s\" file is too long"),
		  r->name,lineno,filename);
	ret = 0;
	goto out;
    }
	
    strfcpy(r->val.str, buffer, r->size_val);

    if (r->flags & FL_NOSPC) {
	char *s;
	for (s = r->val.str; *s; ++s)
	    if (*s == '_') *s=' ';
    }

 out:
    free(buffer);

    DPRINT(Debug,20,(&Debug, 
		     "dt_STR_parse_line [%s:%d] = %d\n",
		     filename,lineno,ret));

    return ret;
}


/* Returns pointer to static area */

S_(rc_get_value dt_STR_get_value)
static char * dt_STR_get_value P_((struct rc_save_info_rec *r));
static char * dt_STR_get_value(r)
     struct rc_save_info_rec *r;
{    
    if (r->flags & FL_NOSPC) {

	static char buffer[SLEN];

	char *t, *s;
	for (t = r->val.str, s = buffer; 
	     *t && s - buffer < sizeof buffer -1; 
	     ++t, ++s)
	    if ((*s = *t) == ' ') 
		*s='_';
	*s = '\0';

	return buffer;
    } else
	return r->val.str;
}

S_(rc_print_value dt_STR_print_value)
static void dt_STR_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;   
{
    char * s0 = dt_STR_get_value(r);
    char * s = s0;
    char * buffer = NULL;
    int   buffer_len = 0;
    int ok = 1;

    if (rcset != system_charset) {
	ok = convert_to_rcset(s,-1,r->name,rcset,&buffer,&buffer_len);	
	s = buffer;
    }

    if (comment || !ok)
	fprintf(F, "### ");
    fprintf(F, "%s = %s\n", r->name, s);
    
    DPRINT(Debug,9,(&Debug, 
		    " option \"%s\", value=\"%s\"%s\n",
		    r->name,s0, 
		    ok ? "" : ", failure"));
    
    if (buffer)
	free(buffer);
    buffer = NULL;

    if (!ok)
	 elm_fprintf(F, 
		     CATGETS(elm_msg_cat, ElmrcSet, ElmrcFailConvCharset,
			     "# Failed to convert charset of option %s\n"),
		     r->name);
}


struct rc_type rc_DT_STR = { RCTYPE_magic,
			     dt_STR_parse_line, BAD_parse_cline,
			     dt_STR_print_value, dt_STR_get_value,
			     NO_free_option };


S_(rc_parse_line dt_NUM_parse_line)
static int dt_NUM_parse_line P_((struct rc_save_info_rec *r,
				 enum record_mode          lcl,
				 struct string *value, int lineno, 
				 const char *filename,
				 int negate,
				 int read_flags /* READ_FLAG_IGNORE_MISSING */
				 ));
static int dt_NUM_parse_line(r,lcl,value,lineno,filename,
			     negate,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    int badpos = -1;
    long l = string_to_long(value,&badpos);
    
    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    }

    if (badpos >= 0 ||
	l > (long)INT_MAX ||
	l < (long)INT_MIN) {

	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmBadNumValueInElmrc,
			  "Bad numeric value \"%S\" on key \"%s\"  in line %d in \"%s\" file"),
		  value, r->name,lineno,filename);

	if (l > (long)INT_MAX)
	    l = (long)INT_MAX;
	else if (l < (long)INT_MIN)
	    l = (long)INT_MIN;

	*(r->val.num) = l;
	return 0;
    }
    
    *(r->val.num) = l;

    return 1;
}

S_(rc_print_value dt_NUM_print_value);
static void dt_NUM_print_value P_((FILE *F,
				   struct rc_save_info_rec *r,
				   int comment,charset_t rcset));
static void dt_NUM_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
{
    if (comment)
	fprintf(F, "### ");

    fprintf(F, "%s = %d\n", r->name, 
	    *r->val.num);
    
    DPRINT(Debug,9,(&Debug, 
		    " option \"%s\", value=%d\n",
		    r->name,*r->val.num ));
}

/* Returns static pointer */
static char * NUM_buffer = NULL;

S_(rc_get_value dt_NUM_get_value)
static char * dt_NUM_get_value P_((struct rc_save_info_rec *r));
static char * dt_NUM_get_value(r)
     struct rc_save_info_rec *r;
{    
    if (NUM_buffer)
	free(NUM_buffer);

    NUM_buffer = elm_message(FRM("%d"), *r->val.num);

    return NUM_buffer;
}

S_(rc_free_option dt_NUM_free_option)
void dt_NUM_free_option  P_((struct rc_save_info_rec *r));
void dt_NUM_free_option(r)
     struct rc_save_info_rec *r;
{
    if (NUM_buffer) {
	free(NUM_buffer);
	NUM_buffer = NULL;
    }
}

static int is_it_on P_((struct string *word));
static int is_it_on(word)
    struct string *word;
{
    int r = 0;

    /** Returns TRUE if the specified word is either 'ON', 'YES'
	or 'TRUE', and FALSE otherwise.   
    **/

    r = string_matches_ascii(word,s2us("on"),  SMA_ignore_case,SMA_op_match_prefix) ||
	string_matches_ascii(word,s2us("yes"), SMA_ignore_case,SMA_op_match_prefix) ||
	string_matches_ascii(word,s2us("true"),SMA_ignore_case,SMA_op_match_prefix);

    DPRINT(Debug,20,(&Debug, "is_it_on(%S)=%d\n",word,r));

    return r;
}

static int is_it_off P_((struct string *word));
static int is_it_off(word)
    struct string *word;
{
    int r = 0;

    /** Returns TRUE if the specified word is either 'OFF', 'NO'
	or 'FALSE', and FALSE otherwise.   
    **/

    r = string_matches_ascii(word,s2us("off"),  SMA_ignore_case,SMA_op_match_prefix) ||
	string_matches_ascii(word,s2us("no"), SMA_ignore_case,SMA_op_match_prefix) ||
	string_matches_ascii(word,s2us("false"),SMA_ignore_case,SMA_op_match_prefix);

    DPRINT(Debug,20,(&Debug, "is_it_off(%S)=%d\n",word,r));

    return r;
}

struct rc_type rc_DT_NUM = { RCTYPE_magic,
			     dt_NUM_parse_line, BAD_parse_cline,
			     dt_NUM_print_value, dt_NUM_get_value,
			     dt_NUM_free_option };

S_(rc_parse_line dt_BOL_parse_line)
static int dt_BOL_parse_line P_((struct rc_save_info_rec *r,
				 enum record_mode          lcl,
				 struct string *value, int lineno, 
				 const char *filename,
				 int negate,
				 int read_flags /* READ_FLAG_IGNORE_MISSING */
				 ));
static int dt_BOL_parse_line(r,lcl,value,lineno,filename,
			     negate,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    int val = is_it_on(value);

    if (!val && !is_it_off(value)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadBooleanInElmrc,
			  "Value \"%S\" treated as OFF (FALSE) in line %d in \"%s\" file"),
		  value, lineno, filename);

    }

    if (negate) {
	if (r->flags & FL_OR)
	    *(r->val.bol) |= !val;
	else if (r->flags & FL_AND)
	    *(r->val.bol) &= !val;
	else
	    *(r->val.bol) = !val;    
    } else {
	if (r->flags & FL_OR)
	    *(r->val.bol) |= val;
	else if (r->flags & FL_AND)
	    *(r->val.bol) &= val;
	else
	    *(r->val.bol) = val;    
    }
    return 1;
}

S_(rc_print_value dt_BOL_print_value)
static void dt_BOL_print_value P_((FILE *F,
				   struct rc_save_info_rec *r,
				   int comment,charset_t rcset));
static void dt_BOL_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
{
    char * s;
    if (comment)
	fprintf(F, "### ");

    s = *r->val.bol ? "ON" : "OFF";

    fprintf(F, "%s = %s\n", r->name, s);
    
    DPRINT(Debug,9,(&Debug, 
		    " option \"%s\", value=%s\n",
		    r->name,s ));
}

S_(rc_get_value dt_BOL_get_value)
static char * dt_BOL_get_value P_((struct rc_save_info_rec *r));
static char * dt_BOL_get_value(r)
     struct rc_save_info_rec *r;
{
    return *r->val.bol ? "ON" : "OFF";
}

struct rc_type rc_DT_BOL = { RCTYPE_magic,
			     dt_BOL_parse_line, BAD_parse_cline,
			     dt_BOL_print_value, dt_BOL_get_value,
			     NO_free_option };

S_(rc_parse_line dt_CHR_parse_line)
static int dt_CHR_parse_line P_((struct rc_save_info_rec *r,
				 enum record_mode          lcl,
				 struct string *value, int lineno, 
				 const char *filename,
				 int negate,
				 int read_flags /* READ_FLAG_IGNORE_MISSING */
				 ));
static int dt_CHR_parse_line(r,lcl,value,lineno,filename,
			     negate, read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    }

    /* This assumes that (system) charset is ascii compatibile 
     */

    if (string_len(value) > 0) {
	uint16 val = give_unicode_from_string(value,0);

	if (val < 128) {
	    *(r->val.chr) = val;
	    return 1;
	}
    }

    return 0;
}

S_(rc_print_value dt_CHR_print_value)
static void dt_CHR_print_value P_((FILE *F,
				   struct rc_save_info_rec *r,
				   int comment,
				   charset_t rcset));
static void dt_CHR_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
{
    if (comment)
	fprintf(F, "### ");

    fprintf(F, "%s = %c\n", r->name, *r->val.chr);

    DPRINT(Debug,9,(&Debug, 		    
		    " option \"%s\", value=%c\n",
		    r->name, *r->val.chr));
}

S_(rc_get_value dt_CHR_get_value)
static char * dt_CHR_get_value P_((struct rc_save_info_rec *r));
static char * dt_CHR_get_value(r)
     struct rc_save_info_rec *r;
{
    static char buffer[2];

    buffer[0] = *r->val.chr;
    buffer[1] = '\0';

    return buffer;
}

struct rc_type rc_DT_CHR = { RCTYPE_magic,
			     dt_CHR_parse_line, BAD_parse_cline,
			     dt_CHR_print_value, dt_CHR_get_value,
			     NO_free_option };

S_(rc_parse_line dt_SORT_parse_line)
static int dt_SORT_parse_line P_((struct rc_save_info_rec *r,
				  enum record_mode          lcl,
				  struct string *value, int lineno, 
				  const char *filename,
				  int negate,
				  int read_flags /* READ_FLAG_IGNORE_MISSING */
				  ));
static int dt_SORT_parse_line(r,lcl,value,lineno,filename,
			      negate,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    char * buffer = NULL;
    int buffer_len = 0;
    char *s;
    int f,y;
    int ret = convert_to_system_charset(value,lineno,filename,
					&buffer,&buffer_len);
    
    /* shift_lower returns ponter to static buffer */
    s = shift_lower(buffer);
    f = 1;

    if (strncmp(s, "rev-", 4) == 0 ||
	strncmp(s, "reverse-", 8) == 0) {
	f = -f;
	s = index(s, '-') + 1;
    }
    
    if (negate)
	f = -f;

    for (y= 0; r->val.sort->sortval[y].kw; y++) {
	if (equal(s, r->val.sort->sortval[y].kw))
	    break;
    }
    if (r->val.sort->sortval[y].kw) {
	r->val.sort->val = f > 0 ? 
	    r->val.sort->sortval[y].sv : 
	    - r->val.sort->sortval[y].sv;
    } else {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmBadSortKeyInElmrc,
			  "I can't understand sort key \"%S\" in line %d in \"%s\" file"),
		  value, lineno, filename);
	ret = 0;
    }

    free(buffer);

    DPRINT(Debug,20,(&Debug, 
		     "dt_SORT_parse_line [%s:%d] = %d\n",
		     filename,lineno,ret));
    return ret;
}

static char * give_sort_value P_((struct dt_sort_info *ptr));
static char * give_sort_value(ptr)
     struct dt_sort_info *ptr;
{
    int y;

    for (y= 0; ptr->sortval[y].kw; y++) {
	if (ptr->sortval[y].sv == ptr->val) 
	    return ptr->sortval[y].kw;
	else if (ptr->sortval[y].sv == - ptr->val) {

	    static char * buffer = NULL;
	    
	    buffer = strmcpy(buffer,"reverse-");
	    buffer = strmcat(buffer, ptr->sortval[y].kw);

	    return buffer;
	}
    }
    return "*bad*";

}

S_(rc_get_value dt_SORT_get_value)
static char * dt_SORT_get_value P_((struct rc_save_info_rec *r));
static char * dt_SORT_get_value(r)
     struct rc_save_info_rec *r;
{
    return give_sort_value(r->val.sort);
}

const char * give_dt_sort_as_str(ptr)
     struct dt_sort_info *ptr;
{
    char * s = give_sort_value(ptr);

    DPRINT(Debug,9,(&Debug, 
		    "give_dt_sort_as_str=%s\n",
		    s ? s : "<NULL>"));

    return s;
}

int give_dt_sort_as_int(ptr)
     struct dt_sort_info *ptr;
{
    return ptr->val;
}

void set_dt_sort_as_int(ptr,val)
     struct dt_sort_info *ptr; 
     int val;
{
    ptr->val = val;
}

S_(rc_print_value dt_SORT_print_value)
static void dt_SORT_print_value P_((FILE *F,
				    struct rc_save_info_rec *r,
				    int comment, charset_t rcset));
static void dt_SORT_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
     
{
    char * s0 = dt_SORT_get_value(r);
    char * s = s0;
    char * buffer = NULL;
    int   buffer_len = 0;
    int ok = 1;

    if (rcset != system_charset) {
	ok = convert_to_rcset(s,-1,r->name,rcset,&buffer,&buffer_len);	
	s = buffer;
    }

    if (comment || !ok)
	fprintf(F, "### ");

    fprintf(F, "%s = %s\n", r->name, s);

    DPRINT(Debug,9,(&Debug, 
		    " option \"%s\", value=\"%s\"%s\n",
		    r->name,s0, 
		    ok ? "" : ", failure"));

    if (buffer)
	free(buffer);
    buffer = NULL;

    if (!ok)
	elm_fprintf(F, 
		    CATGETS(elm_msg_cat, ElmrcSet, ElmrcFailConvCharset,
			    "# Failed to convert charset of option %s\n"),
		    r->name);
}

struct rc_type rc_DT_SORT = { RCTYPE_magic,
			      dt_SORT_parse_line, BAD_parse_cline,
			      dt_SORT_print_value, dt_SORT_get_value,
			      NO_free_option };


S_(rc_parse_line dt_MLT_parse_line)
static int dt_MLT_parse_line P_((struct rc_save_info_rec *r,
				 enum record_mode          lcl,
				 struct string *value, int lineno, 
				 const char *filename,
				 int negate,
				 int read_flags /* READ_FLAG_IGNORE_MISSING */
				 ));
static int dt_MLT_parse_line(r,lcl,value,lineno,filename, negate,
			     read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    int ok = 1;
    char **s;
    
    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    }
    
    for (s = r->val.mlt; *s; ++s) {
	if (!do_set(lcl, *s, value, filename,
		    read_flags))
	    ok = 0;
    }

    return ok;
}

struct rc_type rc_DT_MLT = { RCTYPE_magic,
			     dt_MLT_parse_line, BAD_parse_cline,
			     NO_print_value, NO_get_value,
			     NO_free_option };

S_(rc_parse_line dt_PRM_parse_line)
static int dt_PRM_parse_line P_((struct rc_save_info_rec *r,
				 enum record_mode          lcl,
				 struct string *value, int lineno, 
				 const char *filename,
				 int negate,
				 int read_flags /* READ_FLAG_IGNORE_MISSING */));
static int dt_PRM_parse_line(r,lcl,value,lineno,filename, negate,
			     read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    int m = 0;
    uint16 modecharp[] = {
	0x0072 /* r */,
	0x0077 /* w */,
	0x0078 /* x */,
	0x0072 /* r */,
	0x0077 /* w */,
	0x0078 /* x */,
	0x0072 /* r */,
	0x0077 /* w */,
	0x0078 /* x */,
	0 };
    int modebit = 0400;
    int i;
    int len = string_len(value);


    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    }

    for (i = 0; i < len && modecharp[i]; i++) {
	uint16 c = give_unicode_from_string(value,i);

	if (c == modecharp[i])
	    m |= modebit;
	else if (c != 0x002D /* - */) {

	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmBadModeInElmrc,
			      "I can't understand file permissions \"%S\" in line %d in \"%s\" file"),
		      value, lineno,filename);
	    return 0;
	}
	modebit >>= 1;
    }

    if (i < len) {	
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadModeInElmrc,
			  "I can't understand file permissions \"%S\" in line %d in \"%s\" file"),
		  value, lineno,filename);
	return 0;
    }
    *(r->val.num) = m;

    return 1;
}

S_(rc_print_value dt_PRM_print_value)
static void dt_PRM_print_value P_((FILE *F,
				   struct rc_save_info_rec *r,
				   int comment,charset_t rcset));
static void dt_PRM_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;   /* mode is ASCII only */
{
    char *s = mode_to_str(*(r->val.num));

    if (comment)
	fprintf(F, "### ");
    
    fprintf(F, "%s = %s\n", r->name, s);

    DPRINT(Debug,9,(&Debug, 
		    " option \"%s\", value=\"%s\"\n",
		    r->name,s ));

}

S_(rc_get_value dt_PRM_get_value)
static char * dt_PRM_get_value P_((struct rc_save_info_rec *r));
static char * dt_PRM_get_value(r)
     struct rc_save_info_rec *r;
{
    return mode_to_str(*(r->val.num));
}

struct rc_type rc_DT_PRM = { RCTYPE_magic,
			     dt_PRM_parse_line, BAD_parse_cline,
			     dt_PRM_print_value, dt_PRM_get_value,
			     NO_free_option };

S_(rc_parse_line dt_FUNC_parse_line)
static int dt_FUNC_parse_line P_((struct rc_save_info_rec *r,
				  enum record_mode          lcl,
				  struct string *value, int lineno, 
				  const char *filename,
				  int negate,
				  int read_flags /* READ_FLAG_IGNORE_MISSING */
				  ));
static int dt_FUNC_parse_line(r,lcl,value,lineno,filename, negate,
			      read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    char * buffer = NULL;
    int buffer_len = 0;
    int ret;

    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    }

    ret = convert_to_system_charset(value,lineno,filename,
				    &buffer,&buffer_len);

    if (!r->func_val(&buffer,1,lineno,filename,
		     read_flags)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueInElmrc,
			  "Value of \"%s\" in line %d in \"%s\" file is bad"),
		  r->name,lineno,filename);
	ret = 0;
    }
    
    free(buffer);

    DPRINT(Debug,20,(&Debug, 
		     "dt_FUNC_parse_line [%s:%d] = %d\n",
		     filename,lineno,ret));

    return ret;
}

S_(rc_print_value dt_FUNC_print_value)
static void dt_FUNC_print_value P_((FILE *F,
				    struct rc_save_info_rec *r,
				    int comment, charset_t rcset));
static void dt_FUNC_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
{
    char *s0 = NULL;
    char * s = NULL;
    char * buffer = NULL;
    int   buffer_len = 0;
    int ok = 1;
    int okset = 1;

    ok = r->func_val(&s0,0,0,NULL,0);
    s = s0;

    if (s && rcset != system_charset) {
	okset = convert_to_rcset(s,-1,r->name,rcset,&buffer,&buffer_len);
	ok = okset && ok;
	s = buffer;
    }

    if (s) {
	if (comment || !ok)
	    fprintf(F, "### ");
	
	fprintf(F, "%s = %s\n", r->name, s);

	DPRINT(Debug,9,(&Debug, 
			" option \"%s\", value=\"%s\"%s\n",
			r->name,s0, 
			ok ? "" : ", failure" ));

    } else {
	fprintf(F, "### %s (not set)\n", r->name);

	DPRINT(Debug,9,(&Debug, 
			" option \"%s\", value not returned%s\n",
			r->name, 
			ok ? "" : ", failure"));
    }

    if (!okset)
	elm_fprintf(F, 
		    CATGETS(elm_msg_cat, ElmrcSet, ElmrcFailConvCharset,
			    "# Failed to convert charset of option %s\n"),
		    r->name);
    
    if (buffer)
	free(buffer);
    buffer = NULL;
}

S_(rc_get_value dt_FUNC_get_value)
static char * dt_FUNC_get_value P_((struct rc_save_info_rec *r)); 
static char * dt_FUNC_get_value(r)
     struct rc_save_info_rec *r;
{
    char *s = NULL;

    r->func_val(&s,0,0,NULL,0);

    return s;
}


struct rc_type rc_DT_FUNC = { RCTYPE_magic,
			      dt_FUNC_parse_line, BAD_parse_cline,
			      dt_FUNC_print_value, dt_FUNC_get_value,
			      NO_free_option };

S_(rc_parse_line dt_LONG_parse_line)
static int dt_LONG_parse_line P_((struct rc_save_info_rec *r,
				  enum record_mode          lcl,
				  struct string *value, int lineno, 
				  const char *filename,
				  int negate,
				  int read_flags /* READ_FLAG_IGNORE_MISSING */
				  ));
static int dt_LONG_parse_line(r,lcl,value,lineno,filename,
			      negate,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{   
    int badpos;
    long l;
    
    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    }
    
    l =  string_to_long(value,&badpos);
    *(r->val.l_num) = l;
    
    return 1;
}

S_(rc_print_value dt_LONG_print_value)
static void dt_LONG_print_value P_((FILE *F,
				    struct rc_save_info_rec *r,
				    int comment,charset_t rcset));
static void dt_LONG_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
{
    if (comment)
	fprintf(F, "### ");
    
    fprintf(F, "%s = %ld\n", r->name, *r->val.l_num);
    
    DPRINT(Debug,9,(&Debug, 
		    " option \"%s\", value=%ld\n",
		    r->name,*r->val.l_num));
}


static char * LONG_buffer = NULL;

S_(rc_get_value dt_LONG_get_value)
static char * dt_LONG_get_value P_((struct rc_save_info_rec *r));
static char * dt_LONG_get_value(r)
     struct rc_save_info_rec *r;
{
    
    if (LONG_buffer)
	free(LONG_buffer);
    LONG_buffer = elm_message(FRM("%ld"), *r->val.l_num);

    return LONG_buffer;
}

S_(rc_free_option dt_LONG_free_option)
static void dt_LONG_free_option  P_((struct rc_save_info_rec *r));
static void dt_LONG_free_option(r)
     struct rc_save_info_rec *r;
{
    if (LONG_buffer) {
	free(LONG_buffer);
	LONG_buffer = NULL;
    }
}

struct rc_type rc_DT_LONG = { RCTYPE_magic,
			      dt_LONG_parse_line, BAD_parse_cline,
			      dt_LONG_print_value, dt_LONG_get_value,
			      dt_LONG_free_option };


static void malloc_dt_path P_((struct dt_path_info *ptr));
static void malloc_dt_path(ptr)
     struct dt_path_info *ptr;
{
    if (ptr->flags & PATH_initial) {
	/* value is not allocated */

	ptr->flags &= ~PATH_initial;

	if (ptr->unexpanded)
	    ptr->unexpanded = safe_strdup(ptr->unexpanded);

	if (ptr->list) {
	    char **initial_list = ptr->list;
	    int i;

	    ptr->list = safe_calloc((ptr->nlen+1), sizeof (ptr->list[0]));

	    for (i = 0; i < ptr->nlen; i++) 
		ptr->list[i] = safe_strdup(initial_list[i]);
		
	    ptr->list[ptr->nlen] = NULL;
	} 
    }
}


/* Appends expanded value to list 
   0 == failure
   1 == succeed
*/
int expand_dt_path(ptr,fieldname,val, recursive,read_flags)
     struct dt_path_info *ptr; 
     const char *fieldname; 
     const char *val;
     int recursive;   /* 0 if not recursive call */
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ret = 1;
    
    int  is_file         = ptr->flags & PATH_file;
    int  is_dir          = ptr->flags & PATH_dir;
    int  is_quote        = (ptr->flags & PATH_quote)  && 0 == recursive;
    int  is_expand       = (ptr->flags & PATH_expand) && 0 == recursive;
    int  is_ascii_only   =  ptr->flags & PATH_ascii_only;
    int  is_sep_comma    = ptr->flags & PATH_sep_comma;
    int  is_sep_colon    = ptr->flags & PATH_sep_colon;
    int  is_sep_space    =                             0 == recursive;
    int  is_quote_single = (ptr->flags & PATH_quote_single) && 0 == recursive;
    int  convert_underline = (ptr->flags & PATH_convert_underline) 
	&& 0 == recursive;

    const char *p;
    const char * start_val =  NULL;
    int  q = 0;
    int quit_flag = 0;

    malloc_dt_path(ptr);

    if (!recursive) {
	DPRINT(Debug,8,(&Debug, "expand_dt_path: %s: %s\n",
			fieldname,val));

	if (0 == strcmp(val,"none") && 0 == ptr->nlen) {

	    /* Set dummy value indicating none ... */
	    ptr->list    = safe_realloc(ptr->list,sizeof (ptr->list[0]));
	    ptr->list[0] = NULL;
	    ptr->nlen    = 0;

	    return ret;
	}


    } else {
	DPRINT(Debug,11,(&Debug, "expand_dt_path: %s: %s (recursive=%d)\n",
			 fieldname,val,recursive));
    }

    for (p = val; !quit_flag && (*p || start_val); p++) {
	char * value = NULL;

	/* Quarantee that we do not advance bast of \0 */
	quit_flag = !*p;

	/* Recognize \ escape only in quoted strings */
	if (q && '\\' == *p) {
	    p++;
	    if (!*p) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmTrailingBackslashInElmrc,		   
				  "Trailing backslash (\\) on key \"%s\" value \"%s\""),			  
			  fieldname,val);
		ret = 0;
	    }
	    continue;
	}

	if (!start_val) {
	    if (q || !*p) {
		panic("RC PANIC",__FILE__,__LINE__,"expand_dt_path",
		      "Bad state on !start_val",0);
	    }

	    if (is_quote && '"' == *p) {
		start_val = p+1;
		q         = '"';
		continue;
	    }

	    if (is_quote_single && '\'' == *p) {
		start_val = p+1;
		q         = '\'';
		continue;
	    }

	    /* Empty values not allowed */
	    if ( (is_sep_comma && ',' == *p)  ||	       
	   	 (is_sep_colon && ':' == *p)  ||	      
		 (is_sep_space && ' ' == *p)  ||	       
		 (is_sep_space && '\t' == *p) 
		 )
		continue;

	    start_val = p;


	} else if (q) {

	    if (q == *p || '\0' == *p) {
		int Len = p - start_val;
		const char *s;
		char * t;

		if (!*p) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmUnbalancedQuoteInElmrc,
				      "Unbalanced quote (%c) on key \"%s\" value \"%s\""),
			      q,fieldname,val);
		    ret = 0;
		}

		DPRINT(Debug,11,(&Debug, 
				 "expand_dt_path (clip len=%d)=%.*s\n",
				 Len,Len,start_val));

		value = safe_malloc( Len +1 );
		
		for (s = start_val, t = value;
		     t-value < Len && s < p;) {
		    if ('\\' == *s)
			s++;
		    *t++ = *s++;
		}
		*t = '\0';
		q  = 0;

		DPRINT(Debug,11,(&Debug, 
				 "expand_dt_path    (unquoted)=%s\n",
				 value));

		start_val = NULL;	    
		goto expand_value;
	    }

	} else {

	    if ( (is_sep_comma && ',' == *p)  ||
		 (is_sep_colon && ':' == *p)  ||
		 (is_sep_space && ' ' == *p)  ||
		 (is_sep_space && '\t' == *p) || 
		 '\0'  == *p ) {

		int Len = p - start_val;
		
		DPRINT(Debug,11,(&Debug, 
				 "expand_dt_path (clip len=%d)=%.*s\n",
				 Len,Len,start_val));

		value = safe_malloc( Len +1 );

		memcpy(value,start_val,Len);
		value[Len] = '\0';

		if (convert_underline) {
		    int i;

		    for (i = 0; i < Len; i++)
			if ('_' == value[i])
			    value[i] = ' ';
		}

		start_val = NULL;	    
		goto expand_value;
	    }
	    
	    if (is_quote && '"' == *p) {

		int Len = p - start_val;

		/* Split on quote */
		q = '"';

		DPRINT(Debug,11,(&Debug, 
				 "expand_dt_path (clip len=%d)=%.*s\n",
				 Len,Len,start_val));

		value = safe_malloc( Len +1 );

		memcpy(value,start_val,Len);
		value[Len] = '\0';

		DPRINT(Debug,9,(&Debug, 
				"expand_dt_path : Starting next quoted value...\n"));
		start_val = p+1;	    
		goto expand_value;
	    }

	    if (is_quote_single && '\'' == *p) {

		int Len = p - start_val;

		/* Split on quote */
		q = '\'';
		
		DPRINT(Debug,11,(&Debug, 
				 "expand_dt_path (clip len=%d)=%.*s\n",
				 Len,Len,start_val));

		value = safe_malloc( Len +1 );

		memcpy(value,start_val,Len);
		value[Len] = '\0';

		DPRINT(Debug,9,(&Debug, 
				"expand_dt_path : Starting next quoted value...\n"));
		start_val = p+1;	    
		goto expand_value;
	    }

	}

	
	if (0) {
	expand_value:

	    if (is_ascii_only) {
		unsigned char *x;

		for (x = s2us(value); *x; x++) {
		    if (!isascii(*x)) {
			if (0 == recursive) {
			    lib_error(CATGETS(elm_msg_cat, ElmSet, 
					      ElmAsciiOnlyAllowed,
					      "Only ASCII characters (not 0x%02x) are allowed on key \"%s\" value \"%s\""),
				      *x,fieldname,value);
			    ret = 0;
			}
			break;
		    }
		}

		if (*x) {
		    
		    free(value); value = NULL;
		    
		    continue;
		}
	    }
	
	    if (is_expand) {
		char buffer[1024];
		int v;

		if (is_file || is_dir) {
		    
		    /* HACK -- we can't use expand_meta() because
		       it do not return 1 when value is changed
		    */
		    if ('=' == value[0]) {
			
			/* give_dt_estr_as_str adds / to end */
			const char * t = give_dt_estr_as_str(&folders_e,
							     "maildir",
							     NULL,NULL);

			if (t) {
			    char * t1 = safe_strdup(t);
			
			    t1 = strmcat(t1,value+1);
			    free(value);

			    value = t1;	
			} else {
			    ret = 0;
			}		
		    }
		}

		v = expand_path(buffer,value,sizeof buffer);

		if (v < 0) {		
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmBadVariableInElmrc,
				      "Bad variable on key \"%s\" value \"%s\""),
			      fieldname,val);
		    ret = 0;
		} else if (v > 0) { 

		    DPRINT(Debug,11,(&Debug, 
				     "expand_dt_path    (expanded)=%s\n",
				     buffer));

		    if (recursive < 2 &&
			(is_sep_colon || is_sep_comma)) {
			
			if (!expand_dt_path(ptr,fieldname,buffer,recursive+1,
					    read_flags))
			    ret = 0;

			free(value); value = NULL; 
			continue;
		    }

		    if (is_ascii_only) {
			unsigned char *x;
			
			for (x = s2us(buffer); *x; x++) {
			    if (!isascii(*x)) {
				lib_error(CATGETS(elm_msg_cat, ElmSet, 
						  ElmAsciiOnlyAllowed2,
						  "Only ASCII characters (not 0x%02x) are allowed on key \"%s\" value \"%s\" (expanded from \"%s\")"),
					  *x,fieldname,buffer,value);
				break;
			    }
			}

			if (*x) {
		
			    free(value); value = NULL;
		    
			    continue;
			}
		    }
				    
		    value = strmcpy(value,buffer);
		}
	    }

	    if (is_file || is_dir) {
		struct stat S;
	    
		if ('/' !=  value[0]) {		    

		    if (0 == recursive) {

			DPRINT(Debug,9,(&Debug, 
					"expand_dt_path  not absolute=%s\n",
					value));
	    
			lib_error(CATGETS(elm_msg_cat, ElmSet, 
					  ElmNotAbsoluteInElmrc,	  
					  "Path component \"%s\" not absolute on key \"%s\" value \"%s\""),
				  value,fieldname,val);
			ret = 0;
		    }
		    free(value); value = NULL;
		    continue;
		} 

		if (-1 == stat(value,&S)) {
		    int err = errno;
		    
		    DPRINT(Debug,9,(&Debug, 
				    "expand_dt_path  not accessib=%s\n",
				    value));
		    
		    if (ENOENT == err &&
			0 != (read_flags & READ_FLAG_IGNORE_MISSING)) {
			
			DPRINT(Debug,2,(&Debug,
					"Ignoring missing %s\n",
					value));
			
		    } else {		    
			if (0 == recursive) {						
			    lib_error(CATGETS(elm_msg_cat, ElmSet, 
					      ElmNotAccessibleInElmrc,	  
					      "Path component \"%s\" not accessible on key \"%s\" value \"%s\""),
				      value,fieldname,val);
			    ret = 0;
			}
		    }
		    free(value); value = NULL;
		    continue;
		} 
		
		if (is_dir &&
#ifdef S_ISDIR
		    !S_ISDIR(S.st_mode)
#else
		    S_IFDIR != (S.st_mode & S_IFMT)
#endif
		    ) {

		    DPRINT(Debug,9,(&Debug, 
				    "expand_dt_path  not director=%s\n",
				    value));

		    if (0 == recursive) {
			lib_error(CATGETS(elm_msg_cat, ElmSet, 
					  ElmNotDirInElmrc,	  
					  "Path component \"%s\" not a directory on key \"%s\" value \"%s\""),
				  value,fieldname,val);
			ret = 0;
		    }
		    free(value); value = NULL;
		    continue;
		    
		}

		if (is_file &&
#ifdef S_ISREG
		    !S_ISREG(S.st_mode)
#else
		    S_IFREG != (S.st_mode & S_IFMT)
#endif
		    ) {

		    DPRINT(Debug,9,(&Debug, 
				    "expand_dt_path  not a file  =%s\n",
				    value));

		    if (0 == recursive) {
			lib_error(CATGETS(elm_msg_cat, ElmSet, 
					  ElmNotFileInElmrc,	  
					  "Path component \"%s\" not a file on key \"%s\" value \"%s\""),
				  value,fieldname,val);
			ret = 0;
		    }
		    free(value); value = NULL;
		    continue;
		    
		}
	    }

	    DPRINT(Debug,9,(&Debug, 
			    "expand_dt_path          [%d]=%s\n",
			    ptr->nlen,value));
	    
	    ptr->list = safe_array_realloc(ptr->list,
					   (ptr->nlen+1), sizeof (ptr->list[0]));
	    ptr->list[ptr->nlen++] = value;
	}	
    }

    if (start_val || q) {
	panic("RC PANIC",__FILE__,__LINE__,"expand_dt_path",
	      "Bad state on end",0);
    }

    if (!recursive) {
	DPRINT(Debug,9,(&Debug, 
			"expand_dt_path=%d\n",
			ret));
    } else {
	DPRINT(Debug,11,(&Debug, 
			 "expand_dt_path=%d (recursive=%d)\n",
			 ret,recursive));
    }


    if (ptr->nlen > 0) {
	/* Add trailing NULL */
	ptr->list = safe_array_realloc(ptr->list,
				       (ptr->nlen+1), sizeof (ptr->list[0]));
	ptr->list[ptr->nlen] = NULL;
    }

    return ret;
}



/* Pointer to internal list, do NOT free */
const char ** give_dt_path_as_elems(ptr, fieldname)
     struct dt_path_info *ptr;
     const char * fieldname;
{
    if (!ptr->list && ptr->unexpanded)
	expand_dt_path(ptr,fieldname,ptr->unexpanded,0,0);
    return  ps2cps(ptr->list);
}

/* Result is malloced, caller must free */
char * give_dt_path_as_str(ptr,fieldname)
     struct dt_path_info *ptr;
     const char * fieldname;
{
    char *res,*p;
    int sep;
    int quote;

    int i,L = 1;

    if (!ptr->list && ptr->unexpanded)
	expand_dt_path(ptr,fieldname,ptr->unexpanded,0,0);
    if (!ptr->list)
	return NULL;

    if (ptr->list && 0 == ptr->nlen) {
	res = safe_strdup("none");

	DPRINT(Debug,9,(&Debug, 
			"give_dt_path_as_str>> %s=%s\n",
			fieldname,res));
	
	return res;
    }


    for (i = 0; i < ptr->nlen; i++) 
	L += strlen(ptr->list[i]) +1;

    if (ptr->flags & PATH_sep_comma) sep = ',';
    else if (ptr->flags & PATH_sep_colon) sep = ':';
    else sep = ' ';

    if (ptr->flags & PATH_quote) quote = '"';
    else if (ptr->flags & PATH_quote_single) quote = '\'';
    else quote = 0;

    if (quote)
	L += ptr->nlen * 2;

    res = safe_malloc(L);
   
    for (i = 0, p = res; i < ptr->nlen;i++) {
	int l = strlen(ptr->list[i]);
	
	if (p+l+1 + (quote ? 2 : 0) >= res+L)
	    panic("RC PANIC",__FILE__,__LINE__,"give_dt_path_as_str",
		  "Overflow",0);
	if (i > 0)
	    *p++ = sep;

	if (quote)
	    *p++ = quote;

	memcpy(p,ptr->list[i],l);
	p += l;	    

	if (quote)
	    *p++ = quote;

    }
    *p = '\0';

    DPRINT(Debug,9,(&Debug, 
		    "give_dt_path_as_str (len=%d)>> %s=%s\n",L,
		    fieldname,res));

    return res;
}

S_(rc_parse_line dt_PATH_parse_line)
static int dt_PATH_parse_line P_((struct rc_save_info_rec *r,
				  enum record_mode          lcl,
				  struct string *value, int lineno, 
				  const char *filename,
				  int negate,
				  int read_flags /* READ_FLAG_IGNORE_MISSING */
				  ));
static int dt_PATH_parse_line(r,lcl,value,lineno,filename,
			      negate,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ret;
    char * buffer = NULL;
    int buffer_len = 0;

    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    }

    malloc_dt_path(r->val.path);

    ret = convert_to_system_charset(value,lineno,filename,&buffer,&buffer_len);

    r->val.path->unexpanded = strmcpy(r->val.path->unexpanded,buffer);

    /* Free old value */
    if (r->val.path->list) {
	int i;

	for (i = 0; i < r->val.path->nlen; i++) {
	    if (r->val.path->list[i]) {
		r->val.path->list[i] = NULL;
	    }
	}
	free(r->val.path->list);
	r->val.path->list = NULL;
    }
    r->val.path->nlen = 0;

    free(buffer);

    DPRINT(Debug,20,(&Debug, 
		     "dt_PATH_parse_line [%s:%d] = %d\n",
		     filename,lineno,ret));

    return ret;
}

S_(rc_parse_cline dt_PATH_parse_cline)
static int dt_PATH_parse_cline P_((struct rc_save_info_rec *r,
				   enum record_mode          lcl,
				   struct string *value, int lineno, 
				   const char *filename,
				   int read_flags /* READ_FLAG_IGNORE_MISSING */
				   ));
static int dt_PATH_parse_cline(r,lcl,value,lineno,filename,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    char * buffer = NULL;
    int buffer_len = 0;
    int ret = convert_to_system_charset(value,lineno,filename,
					&buffer,&buffer_len);

    malloc_dt_path(r->val.path);

    /* First line is unexpanded so expand it .... */
    if (r->val.path->unexpanded) {
	
	if (!expand_dt_path(r->val.path,
			    r->name,r->val.path->unexpanded,0,
			    read_flags)) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueInElmrc,
			      "Value of \"%s\" in line %d in \"%s\" file is bad"),
		      r->name,lineno-1,filename);
	    ret = 0;
	}

	free(r->val.path->unexpanded); r->val.path->unexpanded = NULL;	
    }

    if (!expand_dt_path(r->val.path,r->name,buffer,0,
			read_flags)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadValueInElmrc,
			  "Value of \"%s\" in line %d in \"%s\" file is bad"),
		  r->name,lineno,filename);
	    ret = 0;
    }
    
    free(buffer);

    DPRINT(Debug,20,(&Debug, 
		     "dt_PATH_parse_cline [%s:%d] = %d\n",
		     filename,lineno,ret));

    return ret;
}

S_(rc_print_value dt_PATH_print_value)
static void dt_PATH_print_value P_((FILE *F,
				    struct rc_save_info_rec *r,
				    int comment,charset_t rcset));
static void dt_PATH_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
{

    if (r->val.path->unexpanded) {
	char * s0 = r->val.path->unexpanded;
	char * s = s0;
	char * buffer = NULL;
	int   buffer_len = 0;
	int ok = 1;
	
	if (rcset != system_charset) {
	    ok = convert_to_rcset(s,-1,r->name,rcset,&buffer,&buffer_len);	
	    s = buffer;
	}

	if (comment || !ok)
	    fprintf(F, "### ");
	fprintf(F, "%s = %s\n", r->name, s);

	DPRINT(Debug,9,(&Debug, 
			" option \"%s\", value=\"%s\"%s\n",
			r->name,s0, 
			ok ? "" : ", failure"));

	if (buffer)
	    free(buffer);
	buffer = NULL;

	if (!ok)
	    elm_fprintf(F, 
			CATGETS(elm_msg_cat, ElmrcSet, ElmrcFailConvCharset,
				"# Failed to convert charset of option %s\n"),
			r->name);

	
    } else {
	int i;
	int len = strlen(r->name) + 3;
	int fail = 0;
	
	if (comment)
	    fprintf(F, "### ");
	
	fprintf(F, "%s = ",r->name);

	if (r->val.path->list && 0 == r->val.path->nlen) {
	    fprintf(F, "none");

	    DPRINT(Debug,9,(&Debug, 
			    " option \"%s\", value=none\n",
			    r->name,r->val.path->unexpanded));


	} else {
	    DPRINT(Debug,9,(&Debug, 
			    " option \"%s\", value have %d elements\n",
			    r->name,r->val.path->nlen));
	}

	for (i = 0; i < r->val.path->nlen; i++) {
	    char * s = r->val.path->list[i];
	    int slen = strlen(s);
	    char * buffer = NULL;
	    int   buffer_len = 0;
	    int ok = 1;

	    if (rcset != system_charset) {
		ok = convert_to_rcset(s,slen,r->name,rcset,&buffer,&buffer_len);	
		s = buffer;
		slen = buffer_len;

		if (!ok)
		    fail = 1;
	    }
	    
	    /* NOTE: space is always valid separator even when
	       PATH_sep_comma or PATH_sep_colon is set */

	    if (i > 0 || !ok) {
		if (len + slen > 70 || !ok) {
		    fprintf(F, "\n");
		    if (comment || !ok)
			fprintf(F, "### ");
		    fprintf(F, "\t");	
		    len = 8;
		} else 
		    fprintf(F, " ");		
	    }
	    if (r->val.path->flags & PATH_quote)
		elm_fprintf(F,FRM("%Q"),s);
	    else if (r->val.path->flags & PATH_quote_single) {
		fputc('\'',F);
		fputs(s,F);
		fputc('\'',F);
	    } else
		fputs(s,F);
	    len += slen + 1;

	    if (buffer)
		free(buffer);
	    buffer = NULL;

	}

	fprintf(F, "\n");

	if (fail)
	    elm_fprintf(F, 
			CATGETS(elm_msg_cat, ElmrcSet, ElmrcFailConvCharset,
				"# Failed to convert charset of option %s\n"),
			r->name);

    }
}

S_(rc_free_option dt_PATH_free_option)
static void dt_PATH_free_option  P_((struct rc_save_info_rec *r));
static void dt_PATH_free_option(r)
     struct rc_save_info_rec *r;
{
    if (r->val.path->flags & PATH_initial)
	return;

    if (r->val.path->unexpanded) {
	free(r->val.path->unexpanded); 
	r->val.path->unexpanded = NULL;
    }
    
    if (r->val.path->list) {
	int i;
	
	for (i = 0; i < r->val.path->nlen; i++) {
	    if (r->val.path->list[i]) {
		free(r->val.path->list[i]);
		r->val.path->list[i] = NULL;
	    }
	}
	free(r->val.path->list);
	r->val.path->list = NULL;

    }
    r->val.path->nlen = 0;
}

struct rc_type rc_DT_PATH = { RCTYPE_magic,
			      dt_PATH_parse_line, dt_PATH_parse_cline,
			      dt_PATH_print_value, NO_get_value,
			      dt_PATH_free_option };

S_(rc_parse_line dt_OBS_parse_line)
static int dt_OBS_parse_line P_((struct rc_save_info_rec *r,
				 enum record_mode          lcl,
				 struct string *value, int lineno, 
				 const char *filename,
				 int negate,
				 int read_flags /* READ_FLAG_IGNORE_MISSING */
				 ));
static int dt_OBS_parse_line(r,lcl,value,lineno,filename,
			     negate,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    /* This message is not perhaps very visible, because
       Elm starts anyway, but it was requested on that way.
    */

    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmObsoleteInElmrc,

		      "Elmrc variable \"%s\" is obsolete and should be removed from line %d in \"%s\" file"),
	      r->name,lineno,filename);

    return 1; /* Mark as succeed */
}


struct rc_type rc_DT_OBS = { RCTYPE_magic,
			     dt_OBS_parse_line, BAD_parse_cline,
			     NO_print_value, NO_get_value,
			     NO_free_option };


static void metamail_expand(struct dt_estr_info *ptr);
static void metamail_expand(ptr)
     struct dt_estr_info *ptr;
{
    ptr->expanded_keyword  = 0;

    if (getenv("NOMETAMAIL")) {
	if (ptr->expanded)
	    free(ptr->expanded);
	ptr->expanded = NULL;		
    } else {
	ptr->expanded = strmcpy(ptr->expanded,"metamail");
    }
}

static void malloc_dt_estr P_((struct dt_estr_info *ptr));
static void malloc_dt_estr(ptr)
     struct dt_estr_info *ptr;
{
    if (ptr->flags & ESTR_initial) {
	/* value is not allocated */

	ptr->flags &= ~ESTR_initial;

	if (ptr->unexpanded)
	    ptr->unexpanded = safe_strdup(ptr->unexpanded);

	if (ptr->expanded)
	    ptr->expanded = safe_strdup(ptr->expanded);
    }
}

void estr_set_expanded(ptr,buffer)
     struct dt_estr_info *ptr; 
     const char *buffer;
{
    ptr->expanded_keyword = 0;    /* No keyword value */

    malloc_dt_estr(ptr);

    /* Fix shell if needed */
    if (ptr->flags & ESTR_bin) { 
	if (buffer[0] != '/') {
	    char * A = elm_message(FRM("/bin/%s"), buffer);

	    if (ptr->expanded)
		free(ptr->expanded);
	    ptr->expanded = A;
	    return;
	}
    }

    ptr->expanded = strmcpy(ptr->expanded,buffer);

    /* Fix temp_dir if needed */
    if (ptr->flags & ESTR_slash) { 
	int l = strlen(buffer);

	if (0 == l || buffer[l-1] != '/') 
	    ptr->expanded = strmcat(ptr->expanded,"/");
    }
}

/* Expands ptr->unexpanded, ptr->expanded_keyword */
int expand_dt_estr(ptr,fieldname)
     struct dt_estr_info *ptr; 
     const char *fieldname; 
{
    char buffer[1024];

    int x;
    
    DPRINT(Debug,8,(&Debug, "expand_dt_estr: %s: ..->unexpanded=%s\n",
		    fieldname,ptr->unexpanded));

    malloc_dt_estr(ptr);

    ptr->expanded_keyword = 0;

    if (ptr->kw &&
	(ptr->continue_mask || ptr->is_keyword)) {
	int r;

	ptr->expanded_keyword = ptr->is_keyword;

	DPRINT(Debug,8,(&Debug, 
			"   ->is_keyword=%d continue_mask=x%04x\n",
			ptr->is_keyword,ptr->continue_mask));
	
	r = ptr->kw->expand(ptr,ptr->kw,ptr->is_keyword,ptr->continue_mask);
	
	DPRINT(Debug,8,(&Debug, 
			"   expand result code=%d\n",r));

	if (r < 0) {
	    if (ptr->expanded)
		free(ptr->expanded);
	    ptr->expanded = NULL;

	    DPRINT(Debug,8,(&Debug, 
			    "expand_dt_estr=-1 no string expansion available\n"));
	    return -1;
	}

	if (0 == r && ptr->is_keyword) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmBadKeywordExpansionInElmrc,
			      "Bad keyword expansion on key \"%s\" value \"%s\""),
		      fieldname,ptr->unexpanded);

	    return 0;
	}

	if (0 < r) {

	    DPRINT(Debug,8,(&Debug, 
			    "expand_dt_estr=1 (keyword expansion) expanded=%s\n",
			    ptr->expanded ? ptr->expanded : "<NULL>"));
	    
	    return 1;
	    
	}
    }

    if (ptr->flags & ESTR_none) {
	if (0 == strcmp("none",ptr->unexpanded)) {

	    if (ptr->expanded)
		free(ptr->expanded);
	    ptr->expanded = NULL;
	    return 1;
	}
    }

    if (ptr->flags & ESTR_metamail) {
	if (0 == strcmp("metamail",ptr->unexpanded)) {
	    metamail_expand(ptr);
	    return 1;
	}
    }

    if (ptr->flags & ESTR_meta)
	x = expand_meta(buffer, ptr->unexpanded, sizeof buffer);
    else 
	x = expand_env(buffer, ptr->unexpanded, sizeof buffer);

    if (x != 0) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmBadVariableInElmrc,
			  "Bad variable on key \"%s\" value \"%s\""),
		  fieldname,ptr->unexpanded);

	DPRINT(Debug,8,(&Debug, 
			"expand_dt_estr=0 expanded=%s\n",
			ptr->expanded ? ptr->expanded : "<NULL>"));

	return 0;
    }

    estr_set_expanded(ptr,buffer);
    
    DPRINT(Debug,8,(&Debug, "expand_dt_estr=1 expanded=%s\n",
		    ptr->expanded));

    return 1;
}

/* Retuns 1 if sepecial value "none" */

int dt_estr_is_disabled(ptr)
     struct dt_estr_info *ptr;
{
    if (ptr->flags & ESTR_none) {
	if (0 == strcmp("none",ptr->unexpanded)) {
	    return 1;
	}
    }

    return 0;
}

static void dt_generate_keyword_value P_((struct dt_estr_info *estr,
					  struct dt_keyword_info *info));

const char * give_dt_estr_as_str(ptr,fieldname,keyword,cmask)
     struct dt_estr_info *ptr;
     const char * fieldname;
     unsigned int *keyword;
     unsigned int *cmask;
{
    if (keyword)
	*keyword = 0;
    if (cmask)
	*cmask   = 0;
    
    malloc_dt_estr(ptr);

    if (! ptr->unexpanded && ptr->kw) 
	dt_generate_keyword_value(ptr,ptr->kw);

    if (!ptr->expanded && ptr->unexpanded)
	expand_dt_estr(ptr,fieldname);

    DPRINT(Debug,9,(&Debug, 
		    "give_dt_estr_as_str>> %s=%s",
		    fieldname,
		    ptr->expanded ? ptr->expanded : "<NULL>"));

    if (keyword) {
	*keyword = ptr->expanded_keyword;
	DPRINT(Debug,9,(&Debug, " *keyword=%d",*keyword));
    }
    if (cmask) {
	*cmask = ptr->continue_mask;
	DPRINT(Debug,9,(&Debug, " *cmask=x%04x",*cmask));
    }

    DPRINT(Debug,9,(&Debug,"\n"));

    return ptr->expanded;
}

static struct keyword_val * dt_keyword_lookup P_((struct dt_keyword_info *info,
						  const char *value,
						  int continue_line));


int null_keyword_expand(kw,ki,keyword,continue_mask)
     struct dt_estr_info *kw;
     struct dt_keyword_info *ki;
     unsigned int keyword;
     unsigned int continue_mask;
{
    int i;

    if (continue_mask) {
	DPRINT(Debug,12,(&Debug, 
			 "null_keyword_expand=0: continue_mask=x%04x\n",
			 continue_mask));

	return 0;
    }

    malloc_dt_estr(kw);

    for (i = 0; ki->keyword_val[i].kw; i++) {
	if (! ki->keyword_val[i].is_continue_mask &&
	    ki->keyword_val[i].val == keyword) {

	    estr_set_expanded(kw,ki->keyword_val[i].kw);

	    DPRINT(Debug,12,(&Debug, 
			     "null_keyword_expand=1: keyword=%d kw=%s expanded=%s\n",
			     keyword,ki->keyword_val[i].kw,kw->expanded));

	    kw->expanded_keyword = keyword;
	    return 1;
	}
    }

    return 0;
}
				    
/* Set initial value */
void set_dt_estr(ptr, const_val, def_env, is_keyword,continue_mask)
     struct dt_estr_info *ptr;
     const char *const_val;
     char *def_env;
     unsigned int is_keyword;  
     unsigned int  continue_mask;  
{
    char *cp;
 
    if (ptr->flags & ESTR_initial) {
	/* value is not allocated */

	ptr->flags &= ~ESTR_initial;

	ptr->unexpanded = NULL;
	ptr->expanded   = NULL;
    }
   
    if (ptr->kw && !is_keyword && const_val) {
	struct keyword_val *kw = dt_keyword_lookup(ptr->kw,const_val,0);

	if (kw && ! kw->is_continue_mask)
	    is_keyword = kw->val;
    }

    ptr->is_keyword       = is_keyword;
    ptr->continue_mask    = continue_mask;
 
    if (def_env &&
	(cp = getenv(def_env)) &&
	cp[0]) {
	char *x = elm_message(FRM("$%s"),def_env);

	ptr->unexpanded = strmcpy(ptr->unexpanded,x);

	estr_set_expanded(ptr,cp);
	
	free(x);
    } else if (const_val) {
	
	if (ptr->flags & ESTR_none) {
	    if (0 == strcmp("none",const_val)) {
		ptr->unexpanded = strmcpy(ptr->unexpanded,const_val);
		
		if (ptr->expanded)
		    free(ptr->expanded);
		ptr->expanded = NULL;
		ptr->expanded_keyword  = 0;

		return;
	    }
	}

	if (ptr->flags & ESTR_metamail) {
	    if (0 == strcmp("metamail",const_val)) {
		ptr->unexpanded = strmcpy(ptr->unexpanded,const_val);
		
		metamail_expand(ptr);
		return;
	    }
	}

	ptr->unexpanded = strmcpy(ptr->unexpanded,const_val);

	estr_set_expanded(ptr,const_val);
    } else {

	if (ptr->flags & ESTR_none) {

	    ptr->unexpanded = strmcpy(ptr->unexpanded,"none");
	    
	    if (ptr->expanded)
		free(ptr->expanded);
	    ptr->expanded = NULL;
	    ptr->expanded_keyword  = 0;
	    
	    return;	    
	}

    }

    if ((ptr->is_keyword || ptr->continue_mask) && 
	ptr->kw) {
	ptr->kw->expand(ptr,ptr->kw,ptr->is_keyword,ptr->continue_mask);
    } 
}

static struct keyword_val * dt_keyword_lookup(info,value,continue_line)
     struct dt_keyword_info *info;
     const char *value;
     int continue_line;
{
    struct keyword_val * bad_key = NULL;
    int i;

    for (i = 0; info->keyword_val[i].kw; i++) {
	
	if (0 == strcmp(value,info->keyword_val[i].kw)) {
	    
	    if (info->keyword_val[i].is_continue_mask == continue_line)
		return &( info->keyword_val[i] );

	    bad_key = &( info->keyword_val[i] );
	}
    }

    return bad_key;
}


static void dt_generate_keyword_value(estr,info)
     struct dt_estr_info *estr;
     struct dt_keyword_info *info;
{
    int i;
    
    for (i = 0; info->keyword_val[i].kw; i++) {
	
	if (! info->keyword_val[i].is_continue_mask &&
	    estr->is_keyword == info->keyword_val[i].val) {

	    if (estr->flags & ESTR_initial) {
		/* value is not allocated */
		
		estr->flags &= ~ESTR_initial;

		estr->unexpanded = NULL;
		estr->expanded   = NULL;
	    }

	    estr->unexpanded = strmcpy(estr->unexpanded,
				       info->keyword_val[i].kw);
					      
	    /* Free old value */
	    if (estr->expanded) {
		free(estr->expanded);
		estr->expanded = NULL;
	    }

	    return;
	}
    }
}


S_(rc_parse_line dt_ESTR_parse_line)
static  int dt_ESTR_parse_line P_((struct rc_save_info_rec *r,
				   enum record_mode          lcl,
				   struct string *value, int lineno, 
				   const char *filename,
				   int negate,
				   int read_flags /* READ_FLAG_IGNORE_MISSING */
				   ));
static int dt_ESTR_parse_line(r,lcl,value,lineno,filename,
			      negate,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    struct string * Value = value;
    int ok = 1;
    int buffer_len = 0;

    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    }

    if (r->val.estr->flags & ESTR_initial) {
	/* value is not allocated */

	r->val.estr->flags &= ~ESTR_initial;

	r->val.estr->unexpanded = NULL;
	r->val.estr->expanded   = NULL;
    }

    if (0 != (r->val.estr->flags & ESTR_print_expanded)) {
	/* # truncates value */
	
	int Len = 
	    locate_unicode_from_string(value,0x0023 /* #  */);

	if (Len >= 0) {
	    int POS = 0;

	    while (Len > 0) {
		uint16 val = give_unicode_from_string(value,Len-1);

		if (unicode_ch(val,UOP_space))
		    Len--;
		else
		    break;
	    }

	    if (Len < 1) {   
		/* No value */
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadHashInElmrc,
				  "Key \"%s\" ignores value after # character in line %d in \"%s\" file"),
			  r->name,lineno,filename);
		return 0;
	    }

	    Value = clip_from_string(value,&POS,Len);
	}
    }

    ok = convert_to_system_charset(Value,lineno,filename,
				   & (r->val.estr->unexpanded),
				   & buffer_len);

    
    /* Free old value */
    if (r->val.estr->expanded) {
	free(r->val.estr->expanded);
	r->val.estr->expanded = NULL;
    }
    r->val.estr->expanded_keyword = 0;

    if (r->val.estr->kw) {
	struct keyword_val *kw = 
	    dt_keyword_lookup(r->val.estr->kw,
			      r->val.estr->unexpanded,0);

	if (r->val.estr->continue_mask) {
	    DPRINT(Debug,12,(&Debug, 
			     "dt_ESTR_parse_line: option \"%s\", resetting continue_mask\n",
			     r->name));
	}

	r->val.estr->is_keyword = 0;            /* No keyword */
	r->val.estr->continue_mask = 0;  	/* resets also mask */

	if (kw) {

	    if (kw->is_continue_mask) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEstrKWMInElmrc,
				  "Key \"%s\" keyword \"%s\" is valid only on continuation line in line %d in \"%s\" file"),
			  r->name,
			  r->val.estr->unexpanded,
			  lineno,filename);	  
		ok = 0;
	    } else {
		r->val.estr->is_keyword = kw->val;	

		DPRINT(Debug,12,(&Debug, 
				 "dt_ESTR_parse_line: option \"%s\", keyword %d\n",
				 r->name,kw->val));

	    }

	} 
    }

    if (Value != value)
	free_string(&Value);
    
    return ok;
}

S_(rc_parse_cline dt_PATH_parse_cline)
static int dt_ESTR_parse_cline P_((struct rc_save_info_rec *r,
				   enum record_mode          lcl,
				   struct string *value, int lineno, 
				   const char *filename,
				   int read_flags /* READ_FLAG_IGNORE_MISSING */
				   ));
static int dt_ESTR_parse_cline(r,lcl,value,lineno,filename,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ok = 1;

    struct string * Value = value;
    int i;
    char *ptr, *next = NULL;
    struct dt_keyword_info *info;
    int buffer_len = 0;
    char * buffer = NULL;
    

    if (! r->val.estr->kw) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadContinuationInElmrc,
			  "Key \"%s\" does not accept continuation lines in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;  
    }
    info = r->val.estr->kw;

    malloc_dt_estr(r->val.estr);

    if (0 != (r->val.estr->flags & ESTR_print_expanded)) {
	/* # truncates value */
	
	int Len = 
	    locate_unicode_from_string(value,0x0023 /* #  */);

	if (Len >= 0) {
	    int POS = 0;

	    while (Len > 0) {
		uint16 val = give_unicode_from_string(value,Len-1);

		if (unicode_ch(val,UOP_space))
		    Len--;
		else
		    break;
	    }

	    if (Len < 1) {   
		/* No value */
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadHashInElmrc,
				  "Key \"%s\" ignores value after # character in line %d in \"%s\" file"),
			  r->name,lineno,filename);
		return 0;
	    }

	    Value = clip_from_string(value,&POS,Len);
	}
    }

    ok = convert_to_system_charset(Value,lineno,filename,
				   & buffer,
				   & buffer_len);

    for (ptr = buffer; ptr && *ptr; ptr = next) {
	int len;
	struct keyword_val *kw = NULL;

	while (whitespace(*ptr))
                ptr++;

	if (!*ptr)
	    break;

	next = strpbrk(ptr,"\t ");
	
	if (!next)
	    len = (buffer + buffer_len) - ptr;
	else 
	    len = next - ptr;

	for (i = 0; info->keyword_val[i].kw; i++) {
	    const char * T = info->keyword_val[i].kw;
	    
	    if (0 == strncmp(ptr,T,len) && '\0' == T[len]) {
		
		kw =  &( info->keyword_val[i] );
		
		if (info->keyword_val[i].is_continue_mask)
		    break;
	    }
	}

	if (kw && kw->is_continue_mask) {

	    r->val.estr->continue_mask |= kw->val;

	} else {

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEstrKWCInElmrc,
			      "Key \"%s\" keyword \"%.*s\" is not valid on continuation line in line %d in \"%s\" file"),
		      r->name,len,ptr,lineno,filename);	  
	    ok = 0;
	    
	}
    }	

    DPRINT(Debug,12,(&Debug, 
		     "dt_ESTR_parse_cline: option \"%s\", continue_mask=x%04x\n",
		     r->name,r->val.estr->continue_mask));

    if (Value != value)
	free(Value);

    if (buffer)
	free(buffer);

    return ok;
}

S_(rc_get_value dt_ESTR_get_value)
static char * dt_ESTR_get_value P_((struct rc_save_info_rec *r));
static char * dt_ESTR_get_value(r)
     struct rc_save_info_rec *r;
{    
    /* This does not give continution line value */

    if (! r->val.estr->unexpanded && r->val.estr->kw)
	dt_generate_keyword_value(r->val.estr,r->val.estr->kw);

    return r->val.estr->unexpanded;
}

S_(rc_print_value dt_ESTR_print_value)
static void dt_ESTR_print_value P_((FILE *F,
				    struct rc_save_info_rec *r,
				    int comment,
				    charset_t rcset));
static void dt_ESTR_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
{  

    if (! r->val.estr->unexpanded && r->val.estr->kw)
	dt_generate_keyword_value(r->val.estr,r->val.estr->kw);

    if (r->val.estr->unexpanded) {
	char * s0 = r->val.path->unexpanded;
	char * s = s0;
	char * buffer = NULL;
	int   buffer_len = 0;
	int ok = 1;

	if (rcset != system_charset) {
	    ok = convert_to_rcset(s,-1,r->name,rcset,&buffer,&buffer_len);	
	    s = buffer;
	}	

	if (comment || !ok)
	    fprintf(F, "### ");
	fprintf(F, "%s = %s", r->name, s);


	 if (0 != (r->val.estr->flags & ESTR_print_expanded) &&
	     ( r->val.estr->expanded || 
	       expand_dt_estr(r->val.estr,r->name) > 0)) {
	     
	     int ok1 = 1;
	     char * s1 = r->val.estr->expanded;

	     if (rcset != system_charset) {
		 ok1 = convert_to_rcset(s1,-1,r->name,rcset,
					&buffer,&buffer_len);

		 s1 = buffer;
	     }


	     if (s1 && NULL == strchr(s1,'\n') &&
		 0 != strcmp(r->val.estr->unexpanded, r->val.estr->expanded)
		 && ok1) {


		 /* # truncates value */
		 
		 fprintf(F, " # %s",r->val.estr->expanded);
	     }
	 }

	fputc('\n',F);
	
	DPRINT(Debug,9,(&Debug, 
			" option \"%s\", value=\"%s\"%s\n",
			r->name,s0,ok ? "" : ", failure"));


	if (buffer)
	    free(buffer);
	buffer = NULL;

	if (!ok)
	    elm_fprintf(F, 
			CATGETS(elm_msg_cat, ElmrcSet, ElmrcFailConvCharset,
				"# Failed to convert charset of option %s\n"),
			r->name);
	
    } else {

	fprintf(F, "### %s (not set)\n", r->name);

	DPRINT(Debug,9,(&Debug, 
			" option \"%s\", value not set\n",
			r->name));

	return;
    }

    if (r->val.estr->kw && r->val.estr->continue_mask) {
	struct dt_keyword_info *info = r->val.estr->kw;
	
	unsigned int mask = r->val.estr->continue_mask;

	if (comment)
	    fprintf(F, "### ");

	while (mask) {
	    struct keyword_val *kw = NULL;
	    int i;

	    for (i = 0; info->keyword_val[i].kw; i++) {

		if (!info->keyword_val[i].is_continue_mask)
		    continue;

		if (info->keyword_val[i].val ==
		    (info->keyword_val[i].val & mask) &&
		    (! kw || info->keyword_val[i].val > kw->val))
		    kw = & ( info->keyword_val[i] );

	    }

	    if (!kw)
		break;

	    fprintf(F, " %s",kw->kw);

	    mask &= ~ ( kw->val );
	}

	if (mask) 
	   fprintf(F, " # x%04x",mask); 
	
	fputc('\n',F);
    }
}

S_(rc_free_option dt_ESTR_free_option)
static void dt_ESTR_free_option  P_((struct rc_save_info_rec *r));
static void dt_ESTR_free_option(r)
     struct rc_save_info_rec *r;
{
    if (r->val.estr->flags & ESTR_initial) 
	return;

    if (r->val.estr->unexpanded) {
	free(r->val.estr->unexpanded);
	r->val.estr->unexpanded = NULL;
    }

     if (r->val.estr->expanded) {
	free(r->val.estr->expanded);
	r->val.estr->expanded = NULL;
    }
}

struct rc_type rc_DT_ESTR = { RCTYPE_magic,
			      dt_ESTR_parse_line, dt_ESTR_parse_cline,
			      dt_ESTR_print_value, dt_ESTR_get_value,
			      dt_ESTR_free_option };

#define DT_ENUM_DELAYED_magic 	0xF407

static struct dt_enum_delayed {
    unsigned short magic;   /* DT_ENUM_DELAYED_magic */

    char                     * tag;
    char                     * fieldname;
    char                     * new_value;
    int                        lineno;
    char                     * filename;
    char                     * val_as_string;
    
    int refcount;
} * new_dt_enum_delayed P_((const char  * tag,
			    const char  * fieldname,
			    const char  * new_value,
			    int           lineno,
		            const char  * filename,
			    const char  * val_as_string
			    ));
static struct dt_enum_delayed * new_dt_enum_delayed(tag,fieldname,new_value,
						    lineno,filename,val_as_string)
     const char  * tag;
     const char  * fieldname;
     const char  * new_value;
     int           lineno;
     const char  * filename;
     const char  * val_as_string;
{
    struct dt_enum_delayed * ret = safe_zero_alloc(sizeof (*ret));

    ret->tag         = safe_strdup(tag);
    ret->fieldname   = fieldname ? safe_strdup(fieldname) : NULL;
    ret->new_value   = safe_strdup(new_value);
    ret->lineno      = lineno;
    ret->filename    = filename ? safe_strdup(filename) : NULL;
    ret->val_as_string = safe_strdup(val_as_string);
    
    ret -> magic     =  DT_ENUM_DELAYED_magic;
    ret -> refcount  = 1;
    return ret;
}

static void free_dt_enum_delayed P_((struct dt_enum_delayed **ptr));
static void free_dt_enum_delayed(ptr)
     struct dt_enum_delayed **ptr;
{
    if (DT_ENUM_DELAYED_magic != (*ptr)->magic)
	panic("RC PANIC",__FILE__,__LINE__,
	      "free_dt_enum_delayed",
	      "Bad magic number (dt_enum_delayed)",0);

    if ((*ptr)->refcount < 1) {
	panic("RC PANIC",__FILE__,__LINE__,
	      "free_dt_enum_delayed",
	      "Bad refcount",0);

    }

    (*ptr)->refcount--;

    DPRINT(Debug,12,(&Debug, 
		     "free_dt_enum_delayed: item %p => refcount=%d\n",
		     *ptr,(*ptr)->refcount));
    
    
    if ((*ptr)->refcount > 0) {

	(*ptr) = NULL;
	return;
    }

    DPRINT(Debug,12,(&Debug, 
		     "free_dt_enum_delayed: Freeing item %p\n",
		     *ptr));
    
    if ((*ptr)->tag) {
	free((*ptr)->tag);
	(*ptr)->tag = NULL;
    }

    if ((*ptr)->fieldname) {
	free((*ptr)->fieldname);
	(*ptr)->fieldname = NULL;
    }

    if ((*ptr)->new_value) {
	free((*ptr)->new_value);
	(*ptr)->new_value = NULL;
    }

    if ((*ptr)->filename) {
	free((*ptr)->filename);
	(*ptr)->filename = NULL;
    }

    if ((*ptr)->val_as_string) {
	free((*ptr)->val_as_string);
	(*ptr)->val_as_string = NULL;
    }
    
    (*ptr)->magic = 0;  /* Invalidate */
    free(*ptr);
    *ptr = NULL;       
}

static void remove_delayed_enumerate2 P_((struct dt_enum_delayed *delayed));
static int get_enum_shared_value P_((struct dt_enum_shared *found_list,
				     const char * value,
				     int        * e_val_p));

static enum dt_enum_delayed_result { dt_enum_failed    = -1,
				     dt_enum_delayed   = 0,
				     dt_enum_resolved  = 1
} eval_enum_delayed P_((struct dt_enumerate_info *ptr));
static enum dt_enum_delayed_result  eval_enum_delayed(ptr)
     struct dt_enumerate_info *ptr;
{
    enum dt_enum_delayed_result ret = dt_enum_failed;
    
    struct dt_enum_shared * found_list = NULL;
    enum shared_have_code have_code  UNUSED_VAROK = have_code_invalid;
    
    if (DT_ENUM_DELAYED_magic != ptr->delayed->magic)
	panic("RC PANIC",__FILE__,__LINE__,
	      "eval_enum_delayed",
	      "Bad magic number (dt_enum_delayed)",0);

#ifdef USE_DLOPEN
    
    found_list = shared_fill_dt_enumerate_values(ptr,
						 ptr->delayed->tag,
						 &have_code,
						 ptr->delayed->fieldname);
#endif

    if (found_list) {
	int e_val = 0; 
	
	if (DT_ENUM_SHARED_magic != found_list->magic)
	    panic("RC PANIC",__FILE__,__LINE__,
		  "eval_enum_delayed",
		  "Bad magic number (dt_enum_shared)",0);

	if (get_enum_shared_value(found_list, ptr->delayed->new_value,&e_val)) {

	    if (ptr->val_as_string)
		free(ptr->val_as_string);
	    ptr->val_as_string =
		ptr->delayed->val_as_string;
	    ptr->delayed->val_as_string = NULL;

	    ptr->val =  e_val;
	    
	    ret = dt_enum_resolved;

	} else {

	    if (ptr->delayed->lineno > 0 &&
		ptr->delayed->filename) {

		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEKeyInElmrc,
				  "I can't understand %s key \"%s\" in line %d in \"%s\" file"),
			  ptr->delayed->fieldname,
			  ptr->delayed->val_as_string,
			  ptr->delayed->lineno,
			  ptr->delayed->filename);
			  		
	    }

	    mark_local_changed(ptr,0);
	    
	    ret = dt_enum_failed;
	}

	remove_delayed_enumerate2(ptr->delayed);
	free_dt_enum_delayed(& (ptr->delayed));
	    
    } else 
	ret = dt_enum_delayed;
    
    return ret;
}

int give_dt_enumerate_as_int(ptr)
     struct dt_enumerate_info *ptr;
{

    if (ptr->delayed) {
	enum dt_enum_delayed_result X =
	    eval_enum_delayed(ptr);
	
	if (dt_enum_delayed == X) {
	    if (DT_ENUM_DELAYED_magic != ptr->delayed->magic)
		panic("RC PANIC",__FILE__,__LINE__,
		      "give_dt_enumerate_as_int",
		      "Bad magic number (dt_enum_delayed)",0);

	    if (ptr->delayed->lineno > 0 &&
		ptr->delayed->filename) {
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEKeyIgnored,
				  "Ignoring %s key \"%s\" in line %d in \"%s\" file, code not loaded?"),
			  ptr->delayed->fieldname,
			  ptr->delayed->val_as_string,
			  ptr->delayed->lineno,
			  ptr->delayed->filename);
	    }
	}
    }
    
    DPRINT(Debug,9,(&Debug, 
		    "give_dt_enumerate_as_int=%d\n",
		    ptr->val));

    return ptr->val;
}

/* *translated_value is relative to *tag */

struct dt_enum_shared * give_dt_shared_value(ptr,tag,translated_value)
     struct dt_enumerate_info *ptr;
     const char **tag;
     int * translated_value;
{
    struct dt_enum_shared * found_list;

    if (tag)
	*tag = NULL;
    if (translated_value)
	*translated_value = 0;

    if (ptr->delayed) {

	enum dt_enum_delayed_result X =
	    eval_enum_delayed(ptr);
	
	if (dt_enum_delayed == X) {
	    if (DT_ENUM_DELAYED_magic != ptr->delayed->magic)
		panic("RC PANIC",__FILE__,__LINE__,
		      "give_dt_shared_value",
		      "Bad magic number (dt_enum_delayed)",0);

	    if (ptr->delayed->lineno > 0 &&
		ptr->delayed->filename) {
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEKeyIgnored,
				  "Ignoring %s key \"%s\" in line %d in \"%s\" file, code not loaded?"),
			  ptr->delayed->fieldname,
			  ptr->delayed->val_as_string,
			  ptr->delayed->lineno,
			  ptr->delayed->filename);
	    }
	}
    }
     
    if (ptr->val >= 0) {
	if (translated_value)
	    *translated_value = ptr->val;

	return NULL;
    }
    
    for (found_list = ptr->first_shared; 
	 found_list;
	 found_list = found_list->next_shared) {
	
	if (DT_ENUM_SHARED_magic != found_list->magic)
	    panic("RC PANIC",__FILE__,__LINE__,
		  "give_dt_shared_value",
		  "Bad magic number (dt_enum_shared)",0);
	
	if (ptr->val >= found_list->start_value &&
	    ptr->val < found_list->start_value + found_list->nlen) {

	    if (translated_value)
		*translated_value = ptr->val - found_list->start_value;

	    if (tag)
		*tag = found_list->tag;

	    return found_list;
	}
    }

    if (translated_value)
	*translated_value = ptr->val;

    return NULL;
}


static char * enumerate_as_str_helper P_((struct dt_enumerate_info *ptr));
static char * enumerate_as_str_helper(ptr) 
     struct dt_enumerate_info *ptr;
{
    struct dt_enum_shared * found_list;

    if (ptr->delayed) {

	enum dt_enum_delayed_result X =
	    eval_enum_delayed(ptr);

	if (dt_enum_delayed == X) {
	    if (DT_ENUM_DELAYED_magic != ptr->delayed->magic)
		panic("RC PANIC",__FILE__,__LINE__,
		      "enumerate_as_str_helper",
		      "Bad magic number (dt_enum_delayed)",0);

	    return ptr->delayed->val_as_string;
	}	
    }
	
    
    if (ptr->val >= 0 && 
	ptr->val < ptr->nlen) {

	return ptr->list[ptr->val];
    }

    if (ptr->val_as_string) {
	return ptr->val_as_string;
    }

    for (found_list = ptr->first_shared; 
	 found_list;
	 found_list = found_list->next_shared) {
	
	if (DT_ENUM_SHARED_magic != found_list->magic)
	    panic("RC PANIC",__FILE__,__LINE__,
		  "enumerate_as_str_helper",
		  "Bad magic number (dt_enum_shared)",0);
	
	if (ptr->val >= found_list->start_value &&
	    ptr->val < found_list->start_value + found_list->nlen) {
	    
	    char * s = found_list->list[ptr->val - found_list->start_value];

	    ptr->val_as_string = 
		elm_message(FRM("%s:%s"),
			    found_list->tag,s);

	    return ptr->val_as_string;
	}
    }

    ptr->val_as_string = elm_message(FRM("%d"), ptr->val);

    return ptr->val_as_string;
}


/* give_dt_enumerate_as_str / set_dt_enumerate_as_str do not support
   boolean values */

const char * give_dt_enumerate_as_str(ptr)
     struct dt_enumerate_info *ptr;
{   
    char * str = enumerate_as_str_helper(ptr);

    DPRINT(Debug,9,(&Debug, 
		    "give_dt_enumerate_as_str=%s\n",
		    str ? 
		    str : "<NULL>"));
    return str;
}

static struct delayed_enumerated {
    struct dt_enumerate_info * ptr;
    struct dt_enum_delayed   * delayed;

}    * delayed_enumerated_list  = NULL;
static size_t delayed_enumerated_count = 0;

void free_delayed_enumerated_list()
{
    DPRINT(Debug,10,(&Debug,"free_delayed_enumerated_list: %lu entries\n",
		     (unsigned long) delayed_enumerated_count));

    if (delayed_enumerated_list) {
	size_t i;
	    
	for (i = 0; i <  delayed_enumerated_count; i++) {
	    if (delayed_enumerated_list[i].delayed) {
		
		DPRINT(Debug,10,(&Debug,"free_delayed_enumerated_list: Freeing #%lu\n",
				 (unsigned long)i));
		
		if (DT_ENUM_DELAYED_magic != delayed_enumerated_list[i].delayed->magic)
		    panic("RC PANIC",__FILE__,__LINE__,
			  "free_delayed_enumerated_list",
			  "Bad magic number (dt_enum_delayed)",0);
		free_dt_enum_delayed(& delayed_enumerated_list[i].delayed);
	    }
	}

	free(delayed_enumerated_list);
	delayed_enumerated_list = NULL;
    }
    delayed_enumerated_count = 0;
}


union delayed_param {
    struct dt_enum_delayed   * delayed;
    struct dt_enumerate_info * ptr;
    const char               * tag;
};

typedef int rd_enumerated_match_f P_((struct delayed_enumerated *X,
				      union delayed_param      param,
				      int *ok_p));

static int remove_delayed_enumerate3 P_((rd_enumerated_match_f * func,
					  union delayed_param     param));

static int remove_delayed_enumerate3(func,param)
     rd_enumerated_match_f * func;
     union delayed_param     param;
{
    int ok = 1;
    
    size_t i,x = 0;
        
    for (i = 0; i <  delayed_enumerated_count; i++) {
	
	if (func(&(delayed_enumerated_list[i]),
		 param,&ok)) {
	    
	    /* Removing */
	    
	    DPRINT(Debug,10,(&Debug,"remove_delayed_enumerate3: #%lu removing",
			     (unsigned long)i));
	    
	    if (delayed_enumerated_list[i].delayed) {
		if (DT_ENUM_DELAYED_magic != delayed_enumerated_list[i].delayed->magic)
		    panic("RC PANIC",__FILE__,__LINE__,
			  "remove_delayed_enumerate3",
			  "Bad magic number (dt_enum_delayed)",0);
		if (delayed_enumerated_list[i].delayed->fieldname) {
		    DPRINT(Debug,10,(&Debug," \"%s\"",
				     delayed_enumerated_list[i].delayed->fieldname));
		}
		
		if (delayed_enumerated_list[i].delayed->tag) {
		    DPRINT(Debug,10,(&Debug," tag %s",
				     delayed_enumerated_list[i].delayed->tag));
		}
		
		if (delayed_enumerated_list[i].delayed->new_value) {
		    DPRINT(Debug,10,(&Debug," new value \"%s\"",
				     delayed_enumerated_list[i].delayed->new_value));
		}
		
		DPRINT(Debug,10,(&Debug,"\n"));
		
		free_dt_enum_delayed(& delayed_enumerated_list[i].delayed);
	    }
	} else {
	    if (x != i) {
		delayed_enumerated_list[x] = delayed_enumerated_list[i];
		delayed_enumerated_list[i].delayed = NULL;
		delayed_enumerated_list[i].ptr = NULL;
	    }
	    x++;	    
	}	
    }
    
    DPRINT(Debug,10,(&Debug,"remove_delayed_enumerated3=%d: delayed_enumerated_count: %lu => %lu\n",
		     ok,
		     (unsigned long)delayed_enumerated_count,
		     (unsigned long)x));
    delayed_enumerated_count = x;	 


    return ok;
}

static void report_enumerate_unknown P_((struct dt_enumerate_info *enum_info,
					 const char *fieldname,
					 const char *val_as_string,
					 int lineno,
					 const char *filename));
static void report_enumerate_unknown(enum_info,fieldname,val_as_string,lineno,filename)
     struct dt_enumerate_info *enum_info;
     const char *fieldname;
     const char *val_as_string;
     int lineno;
     const char *filename;
{
    int i;
    struct dt_enum_shared * found_list;
	
    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEKeyInElmrc,
		      "I can't understand %s key \"%s\" in line %d in \"%s\" file"),
	      fieldname,val_as_string,lineno,filename);

    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEKeyValues,
		      "Possible values are:"));
    for (i = 0; i < enum_info->nlen; i++)
	lib_error(FRM(" - %s"),
		  enum_info->list[i]);

    /* Not necessary give all shared values (gives only tried tags) */
    
    for (found_list = enum_info->first_shared; 
	 found_list;
	 found_list = found_list->next_shared) {
	
	if (DT_ENUM_SHARED_magic != found_list->magic)
	    panic("RC PANIC",__FILE__,__LINE__,
		  "report_enumerate_unknown",
		  "Bad magic number (dt_enum_shared)",0);
	
	for (i = 0; i < found_list->nlen; i++)
	    lib_error(FRM(" - %s:%s"),
		      found_list->tag,
		      found_list->list[i]);
	
    }

    if (enum_info->have_boolean) 
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEKeyBoolean,
			  "Also values ON, OFF, TRUE, FALSE, YES and NO are allowed."));        
}

static struct dt_enum_shared * locate_enum_shared P_((struct dt_enumerate_info *ptr,
						      const char *tag));

S_(rd_enumerated_match_f match_delayed_enum_process)
static int match_delayed_enum_process  P_((struct delayed_enumerated *X,
					 union delayed_param      param,
					 int                    * ok_p));

static int match_delayed_enum_process(X,param,ok_p)
     struct delayed_enumerated *X;
     union delayed_param      param;
     int *ok_p;
{
    int delete_this = 0;

    if (X->delayed) {
	if (DT_ENUM_DELAYED_magic != X->delayed->magic)
	    panic("RC PANIC",__FILE__,__LINE__,
		  "match_delayed_enum_process",
		  "Bad magic number (dt_enum_delayed)",0);

	if (! X->delayed->tag ||
	    ! X->delayed->new_value)
	    delete_this = 1; 
	else if (! param.tag || 
		 0 == strcmp(X->delayed->tag,
			     param.tag)) {

	    if (X->ptr) {
		struct dt_enum_shared * found_list =
		    locate_enum_shared(X->ptr,X->delayed->tag);

		if (found_list) {
		    DPRINT(Debug,20,(&Debug,
				     "match_delayed_enum_process: tag %s values already loaded.\n",
				     X->delayed->tag));

		} else {
		    enum shared_have_code have_code = have_code_invalid;

#ifdef USE_DLOPEN
		    found_list =
			shared_fill_dt_enumerate_values(X->ptr,
							X->delayed->tag,
							&have_code,
							X->delayed->fieldname /* Not used */);
#endif
		    if (X->delayed->fieldname &&
			X->delayed->val_as_string &&
			X->delayed->lineno &&
			X->delayed->filename) {
			
			switch(have_code) {
			case  have_code_invalid:
			    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEKeyInvalidTag,
					      "Ignoring %s key \"%s\" in line %d in \"%s\" file, tag %s: is not valid."),
				      X->delayed->fieldname,
				      X->delayed->val_as_string,
				      X->delayed->lineno,
				      X->delayed->filename,
				      X->delayed->tag);
			    if (ok_p)
				*ok_p = 0;
			    break;
			case have_code_delayed:
			case have_code_not:			    
			    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEKeyNotLoadedTag,
					      "Ignoring %s key \"%s\" in line %d in \"%s\" file, code for tag %s: is not loaded."),
				      X->delayed->fieldname,
				      X->delayed->val_as_string,
				      X->delayed->lineno,
				      X->delayed->filename,
				      X->delayed->tag);
			    if (ok_p)
				*ok_p = 0;
			    break;
			case have_code_loaded:
			    if (! found_list) {
				lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEKeyNoListTag,
						  "Ignoring %s key \"%s\" in line %d in \"%s\" file, tag %s: do not have enumerated values for %s -field."),
					  X->delayed->fieldname,
					  X->delayed->val_as_string,
					  X->delayed->lineno,
					  X->delayed->filename,
					  X->delayed->tag,
					  X->delayed->fieldname);
				if (ok_p)
				    *ok_p = 0;	      
			    }
			    break;
			}			
		    }
		}
		    
		if (found_list) {
		    
		    int e_val = 0;
		    
		    if (DT_ENUM_SHARED_magic != found_list->magic)
			panic("RC PANIC",__FILE__,__LINE__,
			      "match_delayed_enum_process",
			      "Bad magic number (dt_enum_shared)",0);
		    
		    if (get_enum_shared_value(found_list, X->delayed->new_value,&e_val)) {

			if (X->ptr->val_as_string)
			    free(X->ptr->val_as_string);
			X->ptr->val_as_string =
			    X->delayed->val_as_string;
			X->delayed->val_as_string = NULL;

			X->ptr->val =  e_val;

			if (X->ptr->delayed)
			    free_dt_enum_delayed(& (X->ptr->delayed));
			goto found;
		    }
		}

		if (ok_p)
		    *ok_p = 0;
		
		if (X->delayed->lineno > 0 &&
		    X->delayed->filename) {
		    report_enumerate_unknown(X->ptr,
					     X->delayed->fieldname,
					     X->delayed->val_as_string,
					     X->delayed->lineno,
					     X->delayed->filename);
		
		}
		
		mark_local_changed(X->ptr,0);
		if (X->ptr->delayed)
		    free_dt_enum_delayed(& (X->ptr->delayed));
	    }
	    
	found:
	    delete_this = 1; 
        }
	
    } else
	delete_this = 1;
    
    return delete_this;
}

/* return 0 if errors */
int process_delayed_enumerated(tag)
     const char *tag;
{
    union delayed_param      param;
    int ok = 1;

    param.tag = tag;
    
    ok = remove_delayed_enumerate3(match_delayed_enum_process,
				   param);

    return ok;
}


S_(rd_enumerated_match_f match_delayed_enumerate2)
static int match_delayed_enumerate2  P_((struct delayed_enumerated *X,
					 union delayed_param      param,
					 int                    * ok_p));
static int match_delayed_enumerate2(X,param,ok_p)
     struct delayed_enumerated *X;
     union delayed_param      param;
     int *ok_p;
{
    return param.delayed == X->delayed;
}


static void remove_delayed_enumerate2(delayed)
    struct dt_enum_delayed   * delayed;
{
    union delayed_param      param;
    
    if (DT_ENUM_DELAYED_magic != delayed->magic)
	panic("RC PANIC",__FILE__,__LINE__,
	      "remove_delayed_enumerate",
	      "Bad magic number (dt_enum_delayed)",0);

    param.delayed = delayed;

    remove_delayed_enumerate3(match_delayed_enumerate2,
			      param);
    
}

S_(rd_enumerated_match_f match_delayed_enumerate)
static int match_delayed_enumerate  P_((struct delayed_enumerated *X,
					union delayed_param      param,
					int *ok_p));
static int match_delayed_enumerate(X,param,ok_p)
     struct delayed_enumerated *X;
     union delayed_param      param;
     int *ok_p;
{

    return param.ptr == X->ptr;

}


static void remove_delayed_enumerate P_((struct dt_enumerate_info *ptr));
static void remove_delayed_enumerate(ptr)
     struct dt_enumerate_info *ptr;
{
    union delayed_param      param;

    param.ptr = ptr;

    remove_delayed_enumerate3(match_delayed_enumerate,
			      param);
}

static void add_delayed_enumerate P_((struct dt_enumerate_info *ptr,
				      struct dt_enum_delayed   * delayed));
static void add_delayed_enumerate(ptr,delayed)
     struct dt_enumerate_info *ptr;
     struct dt_enum_delayed   * delayed;     
{
    if (DT_ENUM_DELAYED_magic != delayed->magic)
	panic("RC PANIC",__FILE__,__LINE__,
	      " add_delayed_enumerate",
	      "Bad magic number (dt_enum_delayed)",0);
    
    delayed_enumerated_list =
	safe_array_realloc(delayed_enumerated_list,
			   delayed_enumerated_count+1,
			   sizeof (delayed_enumerated_list[0]));

    delayed_enumerated_list[delayed_enumerated_count].ptr       = ptr;
    delayed_enumerated_list[delayed_enumerated_count].delayed   = delayed;
    delayed_enumerated_list[delayed_enumerated_count].delayed->refcount++;

    DPRINT(Debug,10,(&Debug,"add_delayed_enumerate: added #%lu",
		     (unsigned long)delayed_enumerated_count));

    if (delayed_enumerated_list[delayed_enumerated_count].delayed->fieldname) {
	DPRINT(Debug,10,(&Debug," \"%s\"",
			 delayed_enumerated_list[delayed_enumerated_count].delayed->fieldname));
    }
    
    if (delayed_enumerated_list[delayed_enumerated_count].delayed->tag) {
	DPRINT(Debug,10,(&Debug," tag %s",
			 delayed_enumerated_list[delayed_enumerated_count].delayed->tag));
    }
    
    if (delayed_enumerated_list[delayed_enumerated_count].delayed->new_value) {
	DPRINT(Debug,10,(&Debug," new value \"%s\"",
			 delayed_enumerated_list[delayed_enumerated_count].delayed->new_value));
    }   
    DPRINT(Debug,10,(&Debug,"\n"));
			     
    delayed_enumerated_count++;
}

static struct dt_enum_shared * locate_enum_shared(ptr,tag)
     struct dt_enumerate_info * ptr;
     const char               * tag;
{
    struct dt_enum_shared * found_list;

    for (found_list = ptr->first_shared; 
	 found_list;
	 found_list = found_list->next_shared) {
	
	if (DT_ENUM_SHARED_magic != found_list->magic)
	    panic("RC PANIC",__FILE__,__LINE__,
		  "locate_enum_shared",
		  "Bad magic number (dt_enum_shared)",0);
	
	if (0 == strcmp(tag,found_list->tag)) 
	    break;	    
    }
    
    return found_list;
}

static int get_enum_shared_value(found_list,value,e_val_p)
     struct dt_enum_shared *found_list;
     const char * value;
     int        * e_val_p;
{
    int ci = -1;
    int e_val = 0;
    size_t i;
    
    if (DT_ENUM_SHARED_magic != found_list->magic)
	panic("RC PANIC",__FILE__,__LINE__,
	      "get_enum_shared_value",
	      "Bad magic number (dt_enum_shared)",0);

    if (found_list->start_value + found_list->nlen > 0)
	panic("RC PANIC",__FILE__,__LINE__,
	      "get_enum_shared_value",
	      "Bad start_value",0);
    
    for (i = 0; i < found_list->nlen; i++) {
	if (0 == strcmp(found_list->list[i],value)) {
	    e_val = found_list->start_value + i;
	    break;
	} else if (0 == istrcmp(found_list->list[i],value))   
	    ci = found_list->start_value + i;
    }
    
    if (!e_val && ci >= 0)
	e_val = ci;

    if (!e_val) {
	DPRINT(Debug,20,(&Debug, 
			 "get_enum_shared_value: bad tagged (%s) value: %s\n",
			 found_list->tag,
			 value));
	return 0;
    }

    *e_val_p = e_val;

    DPRINT(Debug,20,(&Debug, 
		     "get_enum_shared_value: succeed tagged (%s) value %s as %d\n",
		     found_list->tag,
		     value,e_val));

    return 1;
}

int set_dt_enumerate_as_str(ptr,str,fieldname,lineno,filename)
     struct dt_enumerate_info *ptr;
     char *str;
     const char * fieldname;
     int lineno;
     const char *filename;

{
    int ci = -1, i;
    int e_val = 0;    /* First value is 'unknown' */

    char * have_tag = strchr(str,':');

    if (ptr->delayed) {
	remove_delayed_enumerate(ptr);
    	
	free_dt_enum_delayed(& (ptr->delayed));
    }
	    
    if (have_tag) {
	int len = have_tag - str;
	char * tag;
	const char * value = have_tag+1;  /* Value without tag */

	struct dt_enum_shared * found_list;

	if (!len) {
	    DPRINT(Debug,20,(&Debug, 
			     "set_dt_enumerate_as_str(ptr=%p,str=\"%s\") empty tag not allowed",
			     ptr,str));
	    if (fieldname) {
		DPRINT(Debug,20,(&Debug,"; fieldname=\"%s\"",
				 fieldname));	 
	    }
	    DPRINT(Debug,20,(&Debug, "\n"));
	    return 0;

	}

	tag = safe_malloc(len+1);
	memcpy(tag,str,len);
	tag[len] = 0;

	found_list =  locate_enum_shared(ptr,tag);

	if (!found_list) {
	    enum shared_have_code have_code = have_code_invalid;
	    DPRINT(Debug,20,(&Debug, 
			     "set_dt_enumerate_as_str(ptr=%p,str=\"%s\") need fill values for tag %s",
			     ptr,str,tag));   
	    if (fieldname) {
		DPRINT(Debug,20,(&Debug,"; fieldname=\"%s\"",
				 fieldname));	 
	    }
	    DPRINT(Debug,20,(&Debug, "\n"));

	    
#ifdef USE_DLOPEN
	    found_list = shared_fill_dt_enumerate_values(ptr,tag,
							 &have_code,fieldname);
#endif
	    
	    if (!found_list && have_code_not == have_code) {
				
		ptr->delayed =
		    new_dt_enum_delayed(tag,fieldname,value /* tag not included */,
					lineno,filename,str);
		add_delayed_enumerate(ptr,ptr->delayed);

		DPRINT(Debug,20,(&Debug, 
				 "set_dt_enumerate_as_str(ptr=%p,str=\"%s\"): delaying evalution of values",
				 ptr,str));
		if (fieldname) {
		    DPRINT(Debug,20,(&Debug,"; fieldname=\"%s\"",
				     fieldname));	 
		}
		DPRINT(Debug,20,(&Debug, "\n"));

		free(tag);
		return 1;
	    }	    
	}

	if (found_list) {

	    if (DT_ENUM_SHARED_magic != found_list->magic)
		panic("RC PANIC",__FILE__,__LINE__,
		      "set_dt_enumerate_as_str",
		      "Bad magic number (dt_enum_shared)",0);

	    
	    if (! get_enum_shared_value(found_list,value,&e_val)) {
		DPRINT(Debug,20,(&Debug, 
				 "set_dt_enumerate_as_str(ptr=%p,str=\"%s\") bad tagged value",
				 ptr,str));
		if (fieldname) {
		    DPRINT(Debug,20,(&Debug,"; fieldname=\"%s\"",
				     fieldname));	 
		}
		DPRINT(Debug,20,(&Debug, "\n"));

		free(tag);
		return 0;
	    }	    	
	}

	free(tag);
       
    } else {
	for (i = 0; i < ptr->nlen; i++) {
	    if (0 == strcmp(ptr->list[i],str)) {
		e_val = i;
		break;
	    } else if (0 == istrcmp(ptr->list[i],str))
		ci = i;
	}

	if (!e_val && ci >= 0)
	    e_val = ci;
	
	if (i == ptr->nlen) {
	    char *p;	    
	    
	    if ((e_val = strtol(str,&p,10)) < 0 || *p != '\0') {

		DPRINT(Debug,20,(&Debug, 
				 "set_dt_enumerate_as_str(ptr=%p,str=\"%s\") bad numeric value",
			     ptr,str));
		if (fieldname) {
		    DPRINT(Debug,20,(&Debug,"; fieldname=\"%s\"",
				     fieldname));	 
		}
		DPRINT(Debug,20,(&Debug, "\n"));

		return 0;	    
	    }
	}
    }

    ptr->val_as_string = strmcpy(ptr->val_as_string,
				 str);
    ptr->val = e_val;

    DPRINT(Debug,20,(&Debug, 
		     "set_dt_enumerate_as_str(ptr=%p,str=\"%s\") succeed, value %d",
		     ptr,str,e_val));
    if (fieldname) {
	DPRINT(Debug,20,(&Debug,"; fieldname=\"%s\"",
			 fieldname));	 
    }
    DPRINT(Debug,20,(&Debug, "\n"));
	    
    return 1;
}

S_(rc_parse_line dt_ENUM_parse_line)
static int dt_ENUM_parse_line P_((struct rc_save_info_rec *r,
				  enum record_mode          lcl,
				  struct string *value, int lineno, 
				  const char *filename,
				  int negate,
				  int read_flags /* READ_FLAG_IGNORE_MISSING */
				  ));
static int dt_ENUM_parse_line(r,lcl,value,lineno,filename,
			      negate, read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    int ret;
    char * buffer = NULL;
    int buffer_len = 0;
   
    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    }
    
    ret = convert_to_system_charset(value,lineno,filename,&buffer,&buffer_len);

    if (!set_dt_enumerate_as_str(r->val.enumerate,buffer,
				 r->name,lineno,
				 filename)) {

	/* give_dt_enumerate_as_str / set_dt_enumerate_as_str do not support
	   boolean values 

	   is_it_on and is_it_off match to prefix 
	*/

	if (r->val.enumerate->have_boolean) {
	    if (is_it_on(value))  {
		r->val.enumerate->val = 1;	       
		r->val.enumerate->val_as_string = 
		    strmcpy(r->val.enumerate->val_as_string,buffer);
		goto clean;
	    } else if (is_it_off(value)) {
		r->val.enumerate->val = 0;
		r->val.enumerate->val_as_string = 
		    strmcpy(r->val.enumerate->val_as_string,buffer);
		goto clean;
	    }
	}

	report_enumerate_unknown(r->val.enumerate,
				 r->name,buffer, lineno,filename);
	
			
	ret = 0;
    }

 clean:
    free(buffer);

    return ret;
}

S_(rc_get_value dt_ENUM_get_value)
static char * dt_ENUM_get_value P_((struct rc_save_info_rec *r));
static char * dt_ENUM_get_value(r)
     struct rc_save_info_rec *r;
{
    return enumerate_as_str_helper(r->val.enumerate);
}

S_(rc_print_value dt_ENUM_print_value)
static void dt_ENUM_print_value P_((FILE *F,
				    struct rc_save_info_rec *r,
				    int comment, charset_t rcset));
static void dt_ENUM_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
{  
    struct dt_enum_shared * found_list;

    if (comment)
	fprintf(F, "### ");

    if (r->val.enumerate->delayed) {
	if (DT_ENUM_DELAYED_magic != r->val.enumerate->delayed->magic)
	    panic("RC PANIC",__FILE__,__LINE__,
		  "dt_ENUM_print_value",
		  "Bad magic number (dt_enum_delayed)",0);
	
	fprintf(F, "%s = %s:%s\n", r->name,
		r->val.enumerate->delayed->tag,
		r->val.enumerate->delayed->new_value);
	
	DPRINT(Debug,9,(&Debug, 			
			" option \"%s\", delayed value=\"%s:%s\"\n",
			r->name,
			r->val.enumerate->delayed->tag,
			r->val.enumerate->delayed->new_value));
	return;
    } 
    
    if (r->val.enumerate->val >= 0 && 
        r->val.enumerate->val < r->val.enumerate->nlen) {

	char * s = r->val.enumerate->list[r->val.enumerate->val];

	fprintf(F, "%s = %s\n", r->name, s);

       	DPRINT(Debug,9,(&Debug, 			
			" option \"%s\", value=\"%s\"\n",
			r->name,s ));

	return;
    } 

    for (found_list = r->val.enumerate->first_shared; 
	 found_list;
	 found_list = found_list->next_shared) {
	
	if (DT_ENUM_SHARED_magic != found_list->magic)
	    panic("RC PANIC",__FILE__,__LINE__,
		  "dt_ENUM_print_value",
		  "Bad magic number (dt_enum_shared)",0);

	if (r->val.enumerate->val >= found_list->start_value &&
	    r->val.enumerate->val < 
	    found_list->start_value + found_list->nlen) {
	    
	    char * s = found_list->list[r->val.enumerate->val - 
					found_list->start_value];
	    
	    fprintf(F, "%s = %s:%s\n", 
		    r->name, found_list->tag,s);

	    DPRINT(Debug,9,(&Debug, 			
			    " option \"%s\", value=\"%s:%s\"\n",
			    r->name,found_list->tag,s ));

	    return;
	}
    }

    fprintf(F, "%s = %d\n", r->name, 
	    r->val.enumerate->val);
    
    DPRINT(Debug,9,(&Debug, 
		    " option \"%s\", value=%d\n",
		    r->name,r->val.enumerate->val ));
}

S_(rc_free_option dt_ENUM_free_option)
static void dt_ENUM_free_option  P_((struct rc_save_info_rec *r));
static void dt_ENUM_free_option(r)
     struct rc_save_info_rec *r;
{
    struct dt_enum_shared * found_list, *next;

    if (r->val.enumerate->delayed)
	free_dt_enum_delayed(& (r->val.enumerate->delayed));
    
    if (r->val.enumerate->val_as_string) {
	free(r->val.enumerate->val_as_string);
	r->val.enumerate->val_as_string = NULL;
    }

    for (found_list = r->val.enumerate->first_shared;
	 found_list;
	 found_list = next) {
	next = found_list->next_shared;
	
	if (DT_ENUM_SHARED_magic != found_list->magic)
	    panic("RC PANIC",__FILE__,__LINE__,
		  "dt_ENUM_free_option",
		  "Bad magic number (dt_enum_shared)",0);
	
	found_list->free_this(& found_list);
    }
}

struct rc_type rc_DT_ENUM = { RCTYPE_magic,
			      dt_ENUM_parse_line, BAD_parse_cline,
			      dt_ENUM_print_value, dt_ENUM_get_value,
			      dt_ENUM_free_option };


S_(rc_parse_line dt_STRING_parse_line)
static int dt_STRING_parse_line P_((struct rc_save_info_rec *r,
				    enum record_mode          lcl,
				    struct string *value, int lineno, 
				    const char *filename,
				    int negate,
				    int read_flags /* READ_FLAG_IGNORE_MISSING */
				    ));
static int dt_STRING_parse_line(r,lcl,value,lineno,filename,
				negate,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    int ret = 1;
    charset_t rc = get_string_type(value);

    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    }

    /* Other  values are converted to system charset */
    if (!rcset) 
	rcset = system_charset;
    if (rcset != rc) {
    
	if (0 != (CS_mapping & charset_properties(rcset)) &&
	    (charset_superset_of(rcset,rc) ||
	     ISO2022_superset_of(rcset,rc) ||
	     0 != (charset_properties(rcset) & 
		   CS_universal_set))) {
	    
	    DPRINT(Debug,20,(&Debug,"dt_STRING_parse_line: Current rcset handles value\n"));

	} else if (0 != (CS_mapping & charset_properties(rc))) {

	    if (charset_superset_of(rc,rcset) ||
		ISO2022_superset_of(rc,rcset) ||
		0 != (charset_properties(rcset) & 
		      CS_universal_set)) {
		DPRINT(Debug,20,(&Debug,"dt_STRING_parse_line: Changing charset to value's charset\n"));
		rcset = rc;
	    } else {
		charset_t utf8cs = MIME_name_to_charset("UTF-8",0);
		
		if (!utf8cs)
		    panic("CHARSET PANIC",__FILE__,__LINE__,"dt_STRING_parse_line",
			  "UTF-8 not found",0);
		
		DPRINT(Debug,20,(&Debug,"dt_STRING_parse_line: Changing charset to UTF-8\n"));
		
		rcset = utf8cs;
	    }
	} else {

	    /* If value's charset is unsupported then it
	       is not possible to convert to any charset
	    */

	    DPRINT(Debug,20,(&Debug,"dt_STRING_parse_line: value's charset is unsupported\n"));
	}
    }

    if (* (r->val.String))
	free_string(r->val.String);

    if (r->flags & FL_NOSPC) {
	int len = string_len(value);
	int start_pos = 0;
       
	*(r->val.String) = new_string(get_string_type(value));
	
	while (start_pos < len) {
	    int i;
	    int Len;
	    int ERRORS = 0;

	    for (i = start_pos; i < len; i++) {
		uint16 code = give_unicode_from_string(value,i);

		if (0x005F /* _ */ == code)
		    break;
	    }

	    Len = i - start_pos;
	    if (Len > 0) {
		string_copy_character(r->val.String,value,
				      &start_pos,Len,&ERRORS);
		
		if (start_pos < i || ERRORS) {
		    if (ret)
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorsBadChars,
					  "There is bad characters on line %d in \"%s\" file"),
				  lineno,filename);
		    
		    ret = 0;
		}
	    }

	    if (start_pos < i) {
		uint16 code = UNICODE_BAD_CHAR;

		add_unicode_to_string(* (r->val.String),1,&code);
		start_pos++;
	    } 

	    if (start_pos < len) {
		uint16 code = give_unicode_from_string(value,start_pos);
		if (0x005F /* _ */ == code) {
		    code = 0x0020 /* SPACE */;

		    add_unicode_to_string(* (r->val.String),1,&code);
		    start_pos++;
		}
	    }		
	}

    } else
	* (r->val.String) = dup_string(value);

    DPRINT(Debug,20,(&Debug, 
		     "dt_STRING_parse_line [%s:%d] = %d\n",
		     filename,lineno,ret));

    return ret;
}

/* Returns static pointer */
static char * STRING_buffer = NULL;
static int    STRING_buffer_len = 0;

S_(rc_get_value dt_STRING_get_value)
static char * dt_STRING_get_value P_((struct rc_save_info_rec *r));
static char * dt_STRING_get_value(r)
     struct rc_save_info_rec *r;
{    
    int ok UNUSED_VAROK = 0;

    if (* (r->val.String)) { 
	ok = sconvert_to_rcset(* (r->val.String),r->name,system_charset,
			       &STRING_buffer,&STRING_buffer_len);

	if (r->flags & FL_NOSPC) {
	    char * t;

	    for (t = STRING_buffer; t < STRING_buffer + STRING_buffer_len; t++)
		if (' ' == *t)
		    *t = '_';
	} 

	return STRING_buffer;
    }
    
    return "";
}

S_(rc_print_value dt_STRING_print_value)
static void dt_STRING_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;   
{
    if (* (r->val.String)) {

	char * buffer = NULL;
	int   buffer_len = 0;
	int ok = sconvert_to_rcset(* (r->val.String),r->name,rcset,
				   &buffer,&buffer_len);

	if (r->flags & FL_NOSPC) {
	    char * t;

	    for (t = buffer; t < buffer + buffer_len; t++)
		if (' ' == *t)
		    *t = '_';
	} 

	if (comment || !ok)
	    fprintf(F, "### ");
	fprintf(F, "%s = %s\n", r->name, buffer);

	DPRINT(Debug,9,(&Debug, 
			" option \"%s\", value=\"%S\"%s\n",
			r->name,*(r->val.String), 
			ok ? "" : ", failure"));
	
	if (!ok)
	    elm_fprintf(F, 
			CATGETS(elm_msg_cat, ElmrcSet, ElmrcFailConvCharset,
				"# Failed to convert charset of option %s\n"),
			r->name);

	if (buffer) {
	    free(buffer);
	    buffer  = NULL;
	}

    } else {
	fprintf(F, "### %s (not set)\n", r->name);
	
	DPRINT(Debug,9,(&Debug, 
			" option \"%s\", value not set\n",
			r->name));
    }
}


S_(rc_free_option dt_STRING_free_option)
static void dt_STRING_free_option  P_((struct rc_save_info_rec *r));
static void dt_STRING_free_option(r)
     struct rc_save_info_rec *r;
{
    if (STRING_buffer) {
	free(STRING_buffer);
	STRING_buffer = NULL;
    }
    STRING_buffer_len = 0;

    if (* (r->val.String))
	free_string(r->val.String);
}

struct rc_type rc_DT_STRING = { RCTYPE_magic,
				dt_STRING_parse_line, BAD_parse_cline,
				dt_STRING_print_value, dt_STRING_get_value,
				dt_STRING_free_option };

static int dt_process_flags P_((char * buffer, int buffer_len,
				int lineno, 
				const char *filename,
				struct dt_flags_info *flag,
				int set_string_value,
				struct rc_save_info_rec *r));
static int dt_process_flags(buffer,buffer_len,lineno,filename,
			    flag,set_string_value,r)
     char * buffer; 
     int buffer_len;
     int lineno; 
     const char *filename;
     struct dt_flags_info *flag;
     int set_string_value;  /* Not continuation */
     struct rc_save_info_rec *r;
{
    int ok = 1;    
    char *ptr, *next = NULL;
    int cut_len = 0;

    if (flag->val_as_string) {
	free(flag->val_as_string);
	flag->val_as_string = NULL;
    }

    for (ptr = buffer; ptr && *ptr; ptr = next) {
	int len  = 0;
	int mode = 0;
	int tag;
	
	while (ptr < buffer + buffer_len &&
	       whitespace(*ptr))
	    ptr++;
	
	if (ptr < buffer && !*ptr) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmCharacterNotAllowed,
			      "Character 0x%02x is not allowed on key \"%s\" in line %d in \"%s\" file"),
		      *ptr,r->name,lineno,filename);
	    ok = 0;
	    ptr++;
	}
	    
	for (next = ptr; next < buffer + buffer_len; next++) {
	    if ('\0' == *next /* Should not happen */ ||
		whitespace(*next)) {

		break;
	    }
	
	    if ('#' == *next) {
		len = next - ptr;
		next = NULL;
		break;
	    }
	}

	if (next) 
	    len = next - ptr;

	if (len < 1)
	    continue;
	cut_len = (ptr - buffer) + len;

	if ('+' == *ptr || '-' == *ptr) {
	    mode = *ptr;
	    ptr++;
	    len--;
	}

	if (mode && flag->absolute) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmRcFlagMix,
			      "Mixing of incremental mode and absolute mode on \"%s\" in line %d in \"%s\" file is not supported: %c%.*s"),
		      r->name,lineno,filename,mode,len,ptr);
	    ok = 0;

	} else if (!mode && !flag->absolute) {

	    if (flag->incremental_add || flag->incremental_sub) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmRcFlagMix1,
				  "Mixing of incremental mode and absolute mode on \"%s\" in line %d in \"%s\" file is not supported: %.*s"),
			  r->name,lineno,filename,len,ptr);
		ok = 0;
	    }

	    flag->absolute = 1;
	    flag->val = 0L;

	    /* This is only valid as first value */
	    if (0 == strcmp(ptr,"none")) 		    
		continue;                   		
	}
	
	for (tag = 0; tag < flag->nlen; tag++) {
	    if (0 == strncmp(ptr,flag->flagval[tag],len) &&
		'\0' == flag->flagval[tag][len])
		break;
	}

	if (flag->flagval[tag]) {
	    unsigned long val = FLAGVAL(tag);

	    if ('-' != mode && 	0 == (flag->allowed_flags & val)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmRcFlagUnsupportedkw,
				  "Unsupported keyword on \"%s\" in line %d in \"%s\" file: %.*s"),
			  r->name,
			  lineno,filename,len,ptr);
		ok = 0;	

	    } else {
		switch (mode) {			
		case '+':
		    flag->incremental_add      |= val;
		    flag->val                  |= val;
		    break;
		case '-':
		    flag->incremental_sub      |= val;
		    flag->val                  &= ~val;
		    break;
		case 0:
		    flag->val  |= val;
		    break;
		}
	    }
	} else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmRcFlagBadkw,
			      "Bad keyword on \"%s\" in line %d in \"%s\" file: %.*s"),
		      r->name,lineno,filename,len,ptr);
	    ok = 0;	
	}
    }

    /* Save string form as value */

    if (ok && cut_len > 0 && flag->absolute && set_string_value) {
	
	flag->val_as_string = safe_malloc(cut_len+1);

	memcpy(flag->val_as_string,buffer,cut_len);
	flag->val_as_string[cut_len] = '\0';

    }

    return ok;
}

S_(rc_parse_line dt_FLAGS_parse_line)
static int dt_FLAGS_parse_line P_((struct rc_save_info_rec *r,
				   enum record_mode          lcl,
				   struct string *value, int lineno, 
				   const char *filename,
				   int negate,
				   int read_flags /* READ_FLAG_IGNORE_MISSING */
				   ));
static int dt_FLAGS_parse_line(r,lcl,value,lineno,filename,
			       negate, read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int negate;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    struct dt_flags_info *flag = r->val.flags;

    int ret;
    char * buffer = NULL;
    int buffer_len = 0;

    if (negate) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadNegate,
			  "!%s is not supported in line %d in \"%s\" file"),
		  r->name,lineno,filename);
	return 0;
    } 
   
    flag->incremental_add = 0L;
    flag->incremental_sub = 0L;
    flag->absolute        = 0;


    ret = convert_to_system_charset(value,lineno,filename,&buffer,
				    &buffer_len);

    if (! dt_process_flags(buffer,buffer_len,lineno,filename,flag,ret,r))
	ret = 0;

    free(buffer);

    return ret;
}

S_(rc_parse_cline dt_FLAGS_parse_cline)
static int dt_FLAGS_parse_cline P_((struct rc_save_info_rec *r,
				    enum record_mode          lcl,
				    struct string *value, int lineno, 
				    const char *filename,
				    int read_flags /* READ_FLAG_IGNORE_MISSING */
				    ));
static int dt_FLAGS_parse_cline(r,lcl,value,lineno,filename,read_flags)
     struct rc_save_info_rec *r;
     enum record_mode          lcl;
     struct string *value; 
     int lineno; 
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{     
    struct dt_flags_info *flag = r->val.flags;

    int ret;
    char * buffer = NULL;
    int buffer_len = 0;

    ret = convert_to_system_charset(value,lineno,filename,&buffer,
				    &buffer_len);
    
    if (! dt_process_flags(buffer,buffer_len,lineno,filename,flag,0,r))
	ret = 0;

    free(buffer);

    return ret;
}

S_(rc_print_value dt_FLAGS_print_value)
static void dt_FLAGS_print_value P_((FILE *F,
			       struct rc_save_info_rec *r,
			       int comment,
			       charset_t rcset));
static void dt_FLAGS_print_value(F,r,comment,rcset)
     FILE *F;
     struct rc_save_info_rec *r;
     int comment;
     charset_t rcset;
{
    struct dt_flags_info *flag = r->val.flags;

    int len = strlen(r->name) + 3;
    int count = 0;
    int fail = 0;
    
    if (comment)
	fprintf(F, "### ");
    
    fprintf(F, "%s = ",r->name);
  
    if (! flag->absolute &&
	( flag->incremental_add || flag->incremental_sub)) {
	int tag;
	
	for (tag = 0; tag < flag->nlen; tag++) {
	    unsigned long val = FLAGVAL(tag);

	    if (val == (flag->incremental_add & val) ||
		val == (flag->incremental_sub & val)) {

		char * s = flag->flagval[tag];
		int slen = strlen(s);
		char * buffer = NULL;
		int   buffer_len = 0;
		int ok = 1;
		
		if (rcset != system_charset) {
		    ok = convert_to_rcset(s,slen,r->name,rcset,&buffer,
					  &buffer_len);	
		    s = buffer;
		    slen = buffer_len;

		    if (!ok)
			fail = 1;
		}
	       
		if (count > 0 || !ok) {
		    if (len + slen > 70 || !ok) {
			fputc('\n',F);
			if (comment || !ok)
			    fprintf(F, "### ");
			fputc('\t',F);
			len = 8;
		    } else {
			fputc(' ',F);		
			len += 1;
		    }
		}
		
		if (val == (flag->incremental_add & val)) {
		    fprintf(F, "+%s",s);
		    len += slen + 1;
		}
		
		if (val == (flag->incremental_sub & val)) {
		    fprintf(F, "-%s",s);
		    len += slen + 1;
		}

		count++;
		if (buffer)
		    free(buffer);
		buffer = NULL;
	    }
	}


	/* If on incremental mode, show real value after comment */
	fputs(" # ",F);
	len += 3;
	comment = 1;

    }

    if (! flag->val) {
	fprintf(F, "none");
	len += 4;
    } else if (flag->val_as_string) {
	char * s = flag->val_as_string;
	int slen = strlen(s);
	char * buffer = NULL;
	int   buffer_len = 0;
	int ok = 1;

	if (len + slen > 70)
	    goto generate;
	
	if (rcset != system_charset) {
	    ok = convert_to_rcset(s,slen,r->name,rcset,&buffer,&buffer_len);	
	    s = buffer;
	    slen = buffer_len;	    
	}
	
	if (!ok) {
	    if (buffer)
		free(buffer);
	    buffer = NULL;

	    goto generate;
	}

	fputs(s,F);
	len += slen;

	if (buffer)
	    free(buffer);
	buffer = NULL;
    } else {
	int tag;

    generate:

	for (tag = 0; tag < flag->nlen; tag++) {
	    unsigned long val = FLAGVAL(tag);

	    if (val == (flag->val & val)) {

		char * s = flag->flagval[tag];
		int slen = strlen(s);
		char * buffer = NULL;
		int   buffer_len = 0;
		int ok = 1;
		
		if (rcset != system_charset) {
		    ok = convert_to_rcset(s,slen,r->name,rcset,&buffer,
					  &buffer_len);	
		    s = buffer;
		    slen = buffer_len;

		    if (!ok)
			fail = 1;
		}
		
		if (count > 0 || !ok) {
		    if (len + slen > 70 || !ok) {
			fputc('\n',F);
			if (comment || !ok)
			    fprintf(F, "### ");
			fputc('\t',F);
			len = 8;
		    } else {
			fputc(' ',F);
			len += 1;
		    }
		}

		fputs(s,F);
		len += slen;

		count++;
		if (buffer)
		    free(buffer);
		buffer = NULL;
	    }
	}
    }

    fputc('\n',F);

    if (fail)
	 elm_fprintf(F, 
		     CATGETS(elm_msg_cat, ElmrcSet, ElmrcFailConvCharset,
			     "# Failed to convert charset of option %s\n"),
		     r->name);

}

static char * give_dt_flag_as_str1 P_((struct dt_flags_info *ptr));
static char * give_dt_flag_as_str1(ptr)
     struct dt_flags_info *ptr;
{
    if (! ptr->val)
	return "none";
    
    if (!ptr->val_as_string) {
	int tag;
	
	for (tag = 0; tag < ptr->nlen; tag++) {
	    unsigned long val = FLAGVAL(tag);
	    
	    if (val == (ptr->val & val)) {
		
		if (ptr->val_as_string)
		    ptr->val_as_string =
			strmcat(ptr->val_as_string," ");
		ptr->val_as_string =
		    strmcat(ptr->val_as_string,
			    ptr->flagval[tag]);		
	    }
	}
    }
    
    return ptr->val_as_string;
}


S_(rc_get_value dt_FLAGS_get_value)
static char * dt_FLAGS_get_value P_((struct rc_save_info_rec *r));
static char * dt_FLAGS_get_value(r)
     struct rc_save_info_rec *r;
{
    struct dt_flags_info *flag = r->val.flags;

    return give_dt_flag_as_str1(flag);
}

S_(rc_free_option dt_FLAGS_free_option)
static void dt_FLAGS_free_option  P_((struct rc_save_info_rec *r));
static void dt_FLAGS_free_option(r)
     struct rc_save_info_rec *r;
 {
     struct dt_flags_info *flag = r->val.flags;

     if (flag->val_as_string)
	 free(flag->val_as_string);
     flag->val_as_string = NULL;
 }

struct rc_type rc_DT_FLAGS = { RCTYPE_magic,
				dt_FLAGS_parse_line, dt_FLAGS_parse_cline,
				dt_FLAGS_print_value, dt_FLAGS_get_value,
				dt_FLAGS_free_option };


int dt_flag_is_set(ptr,tag)
     struct dt_flags_info *ptr; 
     int tag;
{
    long val;

    if (tag < 0 || tag >= ptr->nlen)
	panic("RC PANIC",__FILE__,__LINE__,"dt_flag_is_set",
	      "Bad tag",0);
		
    val = FLAGVAL(tag);

    return (val == (ptr->val & val));
}

const char * dt_flag_tag(ptr,tag)
     struct dt_flags_info *ptr;
     int tag;
{
    if (tag < 0 || tag >= ptr->nlen)
	panic("RC PANIC",__FILE__,__LINE__,"dt_flag_is_set",
	      "Bad tag",0);

    return ptr->flagval[tag];
}

const char * give_dt_flag_as_str(ptr)
     struct dt_flags_info *ptr;
{
    return give_dt_flag_as_str1(ptr);
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
