static char rcsid[] = "@(#)$Id: pager_range.c,v 2.10 2024/10/13 10:57:10 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.10 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"
#include "pg_range_imp.h"

/* Needs STRING_magic */
#include "elmme-magic.h"
#include "pg_params_imp.h"

DEBUG_VAR(DebugIO,__FILE__,"stateio");

#define PAGER_RANGE_magic    0xEB07

struct pager_range {
    unsigned short              magic;    /* PAGER_RANGE_magic */

    struct pager_range *inherited;

    int           refcount;
    unsigned int  range_flags: NUM_pr_flag_bits;
    
    int quote_level;

    int indent;                     /* Absolute value; Adjusted
				       from inherted value if 
				       PR_INDENT_ABSOLUTE is not set
				    */

    enum pager_range_opcode    opcode;

    /* Also gives struct pager_param_value */
    struct pgrng_bfr_binding * stringbuffer;  
};

struct pager_range * malloc_simple_pager_range(inherit,range_flags,
					       quote_level,indent)
     struct pager_range *inherit; 
     int range_flags;
     int quote_level;
     int indent;
{
    struct pager_range *ret = safe_zero_alloc(sizeof (*ret));

    if (range_flags < 0 ||
	quote_level < 0 ||
	(indent < 0 && ison(range_flags,PR_INDENT_ABSOLUTE)))
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "malloc_simple_pager_range",
	      "Bad options",0);

    
    ret->magic = PAGER_RANGE_magic;
    ret->refcount = 1;
    ret->range_flags = range_flags;
    ret->quote_level = quote_level;

    ret->indent      = indent;
    ret->opcode      = pr_opcode_none;

    /* Not actually needed */
    ret->inherited   = inherit;
    if (ret->inherited) {
	inc_pager_range_refcount(ret->inherited);

	
	if (isoff(range_flags,PR_FLAGS_ABSOLUTE)) {
	    int temp = ret->inherited->range_flags;

	    /* Incompatible */
	    if (ison(ret->range_flags,PR_COLLAPSE_NOINHERIT))
		clearit(temp,PR_PREFORMAT);
	    
	    
	    clearit(temp,PR_FLAGS_ABSOLUTE);
	    clearit(temp,PR_INDENT_ABSOLUTE);
	    clearit(temp,PR_COLLAPSE_NOINHERIT);
	    
	    setit(ret->range_flags,temp);
	}
	    
	ret->quote_level  += ret->inherited->quote_level;       

	if (isoff(range_flags,PR_INDENT_ABSOLUTE))
	    ret->indent +=  ret->inherited->indent;
    }
   
    ret->stringbuffer  = NULL;

    return ret;
}

void  free_pager_range(range)
     struct pager_range **range;
{
    if (PAGER_RANGE_magic != (*range)->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,"free_pager_range",
	      "Bad magic number",0);

    (*range)->refcount--;

    if ((*range)->refcount > 0) {  /* Do not free */
	
        (*range) = NULL;           /* Refefence count for this pointer is decrement, */
	                           /* so this must have be reset                     */
        return;
    }

    if ((*range)->inherited)
	free_pager_range(& ((*range)->inherited));	

    if ((*range)->stringbuffer)
	pgrng_bfr_bnd_clear_range(& ((*range)->stringbuffer), *range);

    (*range)->magic = 0;    /* Invalidate */
    free(*range);
    *range = NULL;
}

void inc_pager_range_refcount(range)
     struct pager_range *range;
{
    if (PAGER_RANGE_magic != range->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,"inc_pager_range_refcount",
	      "Bad magic number",0);

    range->refcount++;
}

void set_pager_range_opcode(range,opcode)
     struct pager_range *range;
     enum pager_range_opcode  opcode;
{
    if (PAGER_RANGE_magic != range->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,"set_pager_range_opcode",
	      "Bad magic number",0);

    range->opcode = opcode;
}

/* Caller must call free_pager_range() */
struct pager_range * give_parent_pager_range(range)
     const struct pager_range *range;
{
   if (PAGER_RANGE_magic != range->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "give_parent_pager_range",
	      "Bad magic number",0);

     if (range->inherited)
	 inc_pager_range_refcount(range->inherited);

     return range->inherited;
}

int get_pager_range_flags(range)
     const struct pager_range *range;
{
    if (PAGER_RANGE_magic != range->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,"get_pager_range_flags",
	      "Bad magic number",0);
    return range->range_flags;
}

/* Quote level of text/plain; format=flowed */
int get_pager_range_quote_level(range)
     const struct pager_range *range;
{
    if (PAGER_RANGE_magic != range->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "get_pager_range_quote_level",
	      "Bad magic number",0);

    return range->quote_level;
}

/* Indent column 0 .. (columns)-1  ... may be out of range */
int get_pager_range_indent(range)
     const struct pager_range *range;
{
    if (PAGER_RANGE_magic != range->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "get_pager_range_indent",
	      "Bad magic number",0);

    return range->indent;
}

enum pager_range_opcode  get_pager_range_opcode(range)
     const struct pager_range *range;
{
    if (PAGER_RANGE_magic != range->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "get_pager_range_opcode",
	      "Bad magic number",0);

    return range->opcode;
}

/* range1 and range2 may be NULL */
enum range_changed get_pager_range_change(range1,range2) 
     const struct pager_range * range1;
     const struct pager_range * range2;
{
    enum range_changed ret = rc_same_range;

    if (range1 &&
	PAGER_RANGE_magic != range1->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "get_pager_range_change",
	      "Bad magic number (range1)",0);

    if (range2 &&
	PAGER_RANGE_magic != range2->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "get_pager_range_change",
	      "Bad magic number (range2)",0);

    if (range1 == range2)
	return ret;        /* same range, no changes */
    
    if (range1 && range2) {

	/* No wrapping if direct inheritance and 
	   child have PR_SAME_LINE */

	if (range1 == range2->inherited &&
	    ison(range2->range_flags,PR_SAME_LINE)) {
	    ret = rc_same_line;
	    goto skip;
	}

	if (range2 == range1->inherited &&
	    ison(range1->range_flags,PR_SAME_LINE)) {
	    ret = rc_same_line;
	    goto skip;
	}

	/* No wrapping if both have PR_SAME_LINE */


	if (isoff(range1->range_flags,PR_SAME_LINE) ||
	    isoff(range2->range_flags,PR_SAME_LINE)) {
	    ret = rc_newline;

	    goto skip;
	}		
    }


    
    /* No wrapping if both have PR_SAME_LINE 
       This point range1 != range2
     */

    if ((range1 && isoff(range1->range_flags,PR_SAME_LINE)) ||
	(range2 && isoff(range2->range_flags,PR_SAME_LINE))) {
	ret = rc_newline;
    } else {
	ret = rc_same_line;
    }
    

 skip:

    /* changing of quote level requires new line */
    if (range1 && range2 && range1->quote_level != range2->quote_level)
	ret = rc_newline;    
    if (range1 && !range2 && 0 != range1->quote_level)
	ret = rc_newline;
    if (!range1 && range2 && 0 != range2->quote_level)
	ret = rc_newline;

    return ret;
}

/* Caller must call free_pgrng_bfr_binding() */
struct pgrng_bfr_binding * give_pager_range_stringbuffer(range)
     const struct pager_range *range;
{
    if (PAGER_RANGE_magic != range->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "give_pager_range_stringbuffer",
	      "Bad magic number",0);

    if (range->stringbuffer)
	inc_pgrng_bfr_binding_refcount(range->stringbuffer);

    return range->stringbuffer;
}


/* Calls pgrng_bfr_bnd_set_range(), caller must handle 
 * free_pgrng_bfr_binding()
 */
void set_pager_range_stringbuffer(range,buffer)
     struct pager_range *range;
     struct pgrng_bfr_binding *buffer;
{
   if (PAGER_RANGE_magic != range->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "give_pager_range_stringbuffer",
	      "Bad magic number",0);

   if (range->stringbuffer)
       pgrng_bfr_bnd_clear_range(& (range->stringbuffer), range);
    
    range->stringbuffer = buffer;
    pgrng_bfr_bnd_set_range(buffer,range);
}					    


static int estimate_col P_((const struct string *line));
static int estimate_col(line)
     const struct string *line;
{
    const int tabspacing = 8;
    int L = string_len(line);
    int col = 0;
    int i;

    for ( i = 0; i < L; i++) {
	uint16 ch  = give_unicode_from_string(line,i);	

	switch(ch) {
	    case 0x0009:  /* HT */
		col = ((col / tabspacing) +1) * tabspacing;
		break;

	case 0x000C:   /* FF */
	case 0x000D:   /* CR */
	case 0x000B:   /* VT */
	case 0x000A:   /* LF */
	    col = 0;   /* not always correct */
	    break;
	case UNICODE_SOFT_HYPHEN:
	    break;
	    
	default:
	    col++;    /* not always correct */
	    break;
	}
    }
    
    return col;

}

/* If line is not changed, returns original pointer */

struct string * pager_range_change_line(range,line,filter_on_BOL,
					filter_on_BOR)
     const struct pager_range *range;
     struct string *line;
     int filter_on_BOL;  /* filter is on beginning of line  */
     int filter_on_BOR;   /* filter is on beginning of range */
{
    /* Used by flush_filter() and flush_filter_so_dir() */
    struct string * result_prefix = NULL;

        
    if (PAGER_RANGE_magic != range->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pager_range_change_line",
	      "Bad magic number",0);

    if (filter_on_BOL) {
	/* Original flowed quote */
	int quote_level;
	uint16 first_char =  0x0;

	quote_level = range->quote_level;
	
	if (line && string_len(line) > 0)
	    first_char = give_unicode_from_string(line,0);
	
	if (line) {
	    DPRINT(DebugIO,35,(&DebugIO,
			       "pager_range_change_line: (len=%d) line=%S\n",string_len(line),line));
	}
	
	DPRINT(DebugIO,30,(&DebugIO,
			   "pager_range_change_line: range_flags = %d, quote_level = %d, first_char=04x\n",
			   range->range_flags,
			   quote_level,
			   first_char));
	
	if (0 != (range->range_flags & PR_REPLY_QUOTE))  /* Normal reply quote */
	    quote_level++;
	
	if (quote_level && quote_prefix) {
	    int lastX = -1, i;
	    uint16 repchar = 0x0;
	    int POS = 0;
	    
	    int len = string_len(quote_prefix);
	    
	    DPRINT(DebugIO,35,(&DebugIO, 
			       "pager_range_change_line: (len=%d) prefix=%S\n",
			       len,quote_prefix));
	    
	    for (i = 0; i < len; i++) {
		uint16 ch  = give_unicode_from_string(quote_prefix,i);
		
		if (0 != unicode_ch(ch,UOP_noctrl) &&
		    0 == unicode_ch(ch,UOP_space)) {
		    repchar = ch;
		    lastX = i;
		}
	    }
	    
	    if (len > 0)
		first_char = give_unicode_from_string(quote_prefix,0);
	    
	    if (lastX >= 0) {
		DPRINT(DebugIO,30,(&DebugIO,
				   "pager_range_change_line: lastX=%d, repchar=%04x\n",
				   lastX,repchar));
		
		result_prefix = clip_from_string(quote_prefix,&POS,lastX+1);
		
		if (POS <= lastX) {
		    DPRINT(DebugIO,30,(&DebugIO,
				       "pager_range_change_line: failure, pos=%d\n",
				       POS));
		    goto failure;
		}
	    } else {
		DPRINT(DebugIO,30,(&DebugIO,
				   "pager_range_change_line: No printable characters on prefix\n"));
	    }
	    
	    if (quote_level > 1 && repchar) {
		int x;
		
		if (! result_prefix)
		    result_prefix = new_string(system_charset);
		
		for (x = 0; x < quote_level - 1; x++)
		    add_unicode_to_string(result_prefix,1,&repchar);
	    }
	    
	    if (POS < len) {
		struct string * tmp =  clip_from_string(quote_prefix,&POS,
							len-lastX);
		
		append_string(&result_prefix,tmp,0);
		
		free_string(&tmp);
	    }
	    
	failure:;
	}
	
	if (0 != (range->range_flags & PR_QUOTE_L)) {   /* Quotation for buffer */
	    int need_quote = 0x005B /* [ */ == first_char;
	    
	    DPRINT(DebugIO,30,(&DebugIO,
			       "pager_range_change_line: need_quote=%d first_char=%04x\n",
			       need_quote,first_char));
	    
	    if (need_quote) {
		if (! result_prefix)
		    result_prefix = new_string(system_charset);
		add_unicode_to_string(result_prefix,1,&first_char);		
	    }	
	}
	
	if (range->indent > 0) {
	    int needed = range->indent;
	    
	    /* Not really work */
	    if (result_prefix) {
		int col = estimate_col(result_prefix);
		
		needed -= col;
	    }
	    
	    if (needed > 0) {
		if (! result_prefix)
		    result_prefix = new_string(system_charset);
		fill_ascii_to_string(result_prefix,needed,' ');
	    }
	}
	
    } else if (filter_on_BOR) {
	
	if (range->indent > 0) {

	    DPRINT(DebugIO,35,(&DebugIO,
			       "pager_range_change_line: range indent %d on middle of line not supported\n",
			       range->indent));

	    /* XXXX Need know previous content of line */

	}

    } else {
	DPRINT(DebugIO,35,(&DebugIO,
			   "pager_range_change_line: Not filter_on_BOL or filter_or_BOR\n"));
    }
    
    if (result_prefix) {
	if (line) 
	    append_string(&result_prefix,line,0);
	
	/* Caller frees original string when *res changed */
	
	DPRINT(DebugIO,35,(&DebugIO,
			   "pager_range_change_line: (len=%d) returning changed line=%S\n",
			   string_len(result_prefix),result_prefix));
	return result_prefix;	
    }		  


    DPRINT(DebugIO,35,(&DebugIO,
		       "pager_range_change_line: Returning original line\n"));
    return line;
}

#define ADD_DBUG(ch) do {						\
	if (debug_len+1 < sizeof debug_buffer && EOF != ch) {		\
	    debug_buffer[debug_len++] = ch;				\
	    debug_buffer[debug_len] = '\0';				\
	}								\
    } while(0)


struct pager_range *get_pager_range_serialization_from_file(fh,buffer)
     FILE *fh;
     struct stringbuffer * buffer;
{
    int ch = fgetc(fh);
    int inherited_index = 0;
    int have_inherited = 0;
    int range_flags = 0;
    int quote_level = 0;
    int indent      = 0;
    enum pager_range_opcode  opcode = pr_opcode_none;

    unsigned char debug_buffer[100];
    int debug_len = 0;

    
    
    struct pager_range *ret = NULL;
    struct pager_range *inherited = NULL;

    ADD_DBUG(ch);
    
    if ('R' != ch) {
	if (EOF != ch) 
	    ungetc(ch,fh);

	DPRINT(DebugIO,30,(&DebugIO,
			   "get_pager_range_serialization_from_file: Not pager range: %c\n",
			   ch));

	return NULL;
    }

    ch = fgetc(fh);
    ADD_DBUG(ch);
    
    if ('I' == ch) {
	have_inherited = 1;

	ch = fgetc(fh);
	ADD_DBUG(ch);

	while (ch >= '0' && ch <= '9') {
	    inherited_index = inherited_index * 10 + ch - '0';
	    ch = fgetc(fh);
	    ADD_DBUG(ch);
	}

	if (',' != ch) {

	    DPRINT(DebugIO,10,(&DebugIO,"get_pager_range_serialization_from_file: Bad data '%c' (expected ',') buffer=",
			       ch));
	    DEBUG_PRINT_BUFFER(DebugIO,10,debug_len, debug_buffer);
	    DPRINT(DebugIO,10,(&DebugIO,"\n"));
	    
	    panic("PAGER RANGE PANIC",__FILE__,__LINE__,
		  "get_pager_range_serialization_from_file",
		  "Bad data read from file",0);
	}
	ch = fgetc(fh);
	ADD_DBUG(ch);

    }

    while (ch >= '0' && ch <= '9') {
	range_flags = range_flags * 10 + ch - '0';
	ch = fgetc(fh);
	ADD_DBUG(ch);
    }

    if (',' != ch) {
	DPRINT(DebugIO,10,(&DebugIO,
			   "get_pager_range_serialization_from_file: Bad data '%c' (expected ',') buffer=",
			   ch));
	DEBUG_PRINT_BUFFER(DebugIO,10,debug_len, debug_buffer);
	DPRINT(DebugIO,10,(&DebugIO,"\n"));
	
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "get_pager_range_serialization_from_file",
	      "Bad data read from file",0);
    }
    ch = fgetc(fh);
    ADD_DBUG(ch);
    
    while (ch >= '0' && ch <= '9') {
	quote_level = quote_level * 10 + ch - '0';
	ch = fgetc(fh);
	ADD_DBUG(ch);
    }

    if ('i' == ch) {
	int sig = 1;

	ch = fgetc(fh);
	ADD_DBUG(ch);
	/* Relative values can be negative */

	if ('-' == ch) {
	    sig = -1;
	    ch = fgetc(fh);
	    ADD_DBUG(ch);
	}

	while (ch >= '0' && ch <= '9') {
	    indent = indent * 10 + ch - '0';
	    ch = fgetc(fh);
	    ADD_DBUG(ch);
	}

	indent *= sig;
    }

    if ('o' == ch) {
	int val = 0;
	int sig = 1;    /* enum value can be negative */
	
	ch = fgetc(fh);
	ADD_DBUG(ch);
	
	if ('-' == ch) {
	    sig = -1;
	    ch = fgetc(fh);
	    ADD_DBUG(ch);
	}
	
	while (ch >= '0' && ch <= '9') {
	    val = val * 10 + ch - '0';
	    ch = fgetc(fh);
	    ADD_DBUG(ch);
	}

	opcode = val * sig;
    }
    
    if ('\n' != ch) {
	
	DPRINT(DebugIO,10,(&DebugIO,
			   "get_pager_range_serialization_from_file: Bad data %d (expected LF) buffer=",
			   ch));
	DEBUG_PRINT_BUFFER(DebugIO,10,debug_len, debug_buffer);
	DPRINT(DebugIO,10,(&DebugIO,"\n"));
		
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "get_pager_range_serialization_from_file",
	      "Bad data read from file",0);
    }

    /* Recursive call AFTER data is readed */
    if (have_inherited)
	inherited = stringbuffer_get_pager_range(buffer,
						 inherited_index);
    
   
    /* indent is relative if PR_INDENT_ABSOLUTE is not set */

    ret = malloc_simple_pager_range(inherited,range_flags,
				    quote_level,indent);

    if (opcode)
	set_pager_range_opcode(ret,opcode);
    
    if (inherited)
	free_pager_range(&inherited);

    return ret;
}

void put_pager_range_serialization_to_file(fh,range,buffer,pos)
     FILE *fh;
     struct pager_range *range;
     struct stringbuffer *buffer;
     long *pos;
{
    int indent;

    if (PAGER_RANGE_magic != range->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "put_pager_range_serialization_to_file",
	      "Bad magic number",0);

    indent = range->indent;

    if (range->inherited) {
	struct stringbuffer *binherited;
	int index;

	if (PAGER_RANGE_magic != range->inherited->magic)
	    panic("PAGER RANGE PANIC",__FILE__,__LINE__,
		  "put_pager_range_serialization_to_file",
		  "Bad (parent) magic number",0);

	if (! range->inherited->stringbuffer)
	    stringbuffer_add_pager_range(buffer,range->inherited);

	binherited = 
	    pgrng_bfr_bnd_get_stringbuffer(range->inherited->stringbuffer,
					   &index);
	
	/* May write to file */
	if (!binherited)
	    stringbuffer_add_pager_range_b(buffer,
					   range->inherited->stringbuffer,
					   range->inherited);

	if (!binherited)
	    binherited = 		
		pgrng_bfr_bnd_get_stringbuffer(range->inherited->stringbuffer,
					       &index);
	
	if (binherited != buffer)
	    panic("PAGER RANGE PANIC",__FILE__,__LINE__,
		  "put_pager_range_serialization_to_file",
		  "stringbuffer differ on parent/inherited pager_range",0);

	if (index <0)
	    panic("PAGER RANGE PANIC",__FILE__,__LINE__,
		  "put_pager_range_serialization_to_file",
		  "bad index on parent/inherited pager_range",0);


	if (pos)
	    *pos = ftell(fh);
	fputc('R', fh);
	fprintf(fh,"I%d,", index);

	if (!(range->range_flags & PR_INDENT_ABSOLUTE)) {
	    
	    indent -= range->inherited->indent;
	}

	free_stringbuffer(&binherited);
	
    } else {
	if (pos)
	    *pos = ftell(fh);
	fputc('R', fh);
    }

    fprintf(fh,"%d,%d", 
	    range->range_flags,
	    range->quote_level);

    if (indent != 0) {
	/* This stores relative indent if
	 *  PR_INDENT_ABSOLUTE is not set 
	 */

	fprintf(fh,"i%d",
		indent);		
    }

    if (range->opcode !=  pr_opcode_none) {
	fprintf(fh,"o%d",
		range->opcode);		

    }
    
    fputc('\n', fh);
}


/* ------------------------------------------------ */

struct pager_param_value * malloc_pager_param_value(opcode,value)
     enum pager_param_opcode opcode;
     union pager_param_uvalue value;
{
    enum pg_param_value_type value_type = * (value.magic);
    struct pager_param_value *ret = NULL;
    
    int ok = 0;
    
    DPRINT(DebugIO,14,(&DebugIO,"malloc_pager_param_value: opcode=%d",
		       opcode));

    switch (opcode) {
    case pp_opcode_ANY:  /* Not really opcode */
	DPRINT(DebugIO,14,(&DebugIO," pp_opcode_ANY"));
	break;
    case pp_opcode_builtin:
	DPRINT(DebugIO,14,(&DebugIO," pp_opcode_builtin"));
	break;
    }
    
    DPRINT(DebugIO,14,(&DebugIO," value type = %d",value_type));
    switch (value_type) {
    case  pg_pv_text:
	DPRINT(DebugIO,14,(&DebugIO," pg_pv_text"));
	ok = 1;
	break;
    }
    DPRINT(DebugIO,14,(&DebugIO,"\n"));
    
    if (ok) {
	ret =
	    safe_zero_alloc(sizeof (*ret));

	ret->magic    =  PAGER_PARAM_VALUE_magic;
	ret->refcount = 1;
	ret->backlink = NULL;		
	ret->opcode   = opcode;
	
	ret->value.magic  = NULL;

	 switch (value_type) {
	 case  pg_pv_text:
	     ret->value.text = NULL;
	     if (value.text)
		 ret->value.text = dup_string(value.text);
	     break;	     
	 }
    } else {
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,"malloc_pager_param_value",
	      "Bad magic number (pg_param_value)",0);
    }
    
    return ret;
}

void inc_pager_param_value_refcount(value)
     struct pager_param_value * value;
{
    if (PAGER_PARAM_VALUE_magic != value->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,"inc_pager_param_value_refcount",
	      "Bad magic number (pg_param_value)",0);

    value->refcount++;
}

const union pager_param_uvalue get_pager_param_uvalue(value,res_type)
     struct pager_param_value * value;
     enum pager_param_type    * res_type;
{
    enum pg_param_value_type value_type = 0;
    int ok = 0;

    if (PAGER_PARAM_VALUE_magic != value->magic)
    	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "get_pager_param_uvalue",
	      "Bad magic number (pager_param_value)",0);

    if (value->value.magic) {
	value_type = * (value->value.magic);

	switch (value_type) {
	case  pg_pv_text:
	    ok = 1;
	    if (res_type)
		*res_type = pp_type_text;
	    break;
	}

    } else {
	ok = 1;
	if (res_type)
	    *res_type = pp_type_none;
    }

    if (!ok)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "get_pager_param_uvalue",
	      "Bad magic number (pg_param_value)",0);
    
    return value->value;
}


static void clear_pager_param_value P_((struct pager_param_value **value));
static void clear_pager_param_value(value)
     struct pager_param_value **value;
{
    int ok = 0;
    enum pg_param_value_type value_type = 0;
    
    if (PAGER_PARAM_VALUE_magic != (*value)->magic)
    	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "clear_pager_param_value",
	      "Bad magic number (pager_param_value)",0);
    
    if ((*value)->backlink ||
	(*value)->refcount > 0) {
	
	(*value) = NULL;
	return;
    }

    (*value)->opcode = pp_opcode_ANY; /* Bad opcode */

    if ((*value)->value.magic) {
	value_type = * ((*value)->value.magic);

	switch (value_type) {
	case  pg_pv_text:
	    free_string(& ((*value)->value.text));
	    ok = 1;
	    break;
	}

    } else {
	ok = 1;
    }

    if (!ok)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "clear_pager_param_value",
	      "Bad magic number (pg_param_value)",0);

    (*value)->magic = 0;  /* Invalidate */
    free(*value);
    (*value) = NULL;
}


void reset_pager_param_value_bcklink(value,backlink)
     struct pager_param_value **value;
     struct pgrng_bfr_binding   * backlink;
{
    
    if (PAGER_PARAM_VALUE_magic != (*value)->magic)
    	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "reset_pager_param_value_bcklink",
	      "Bad magic number (pager_param_value)",0);

    if ((*value)->backlink &&
	(*value)->backlink != backlink) {
	
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "reset_pager_param_value_bcklink",
	      "Bad badlink",0);

	
    }
    (*value)->backlink = NULL;
    
    clear_pager_param_value(value);
}

/* Clear union pg_param_value value if refcount is zero */
void free_pager_param_value(value)
     struct pager_param_value **value;
{

    if (PAGER_PARAM_VALUE_magic != (*value)->magic)
    	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "free_pager_param_value",
	      "Bad magic number (pager_param_value)",0);

    if ((*value)->refcount < 1)
    	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "free_pager_param_value",
	      "Bad refcount",0);
	
    (*value)->refcount--;
    
    clear_pager_param_value(value);
}

struct pager_param_value * get_pager_param_serialization_from_file(fh)
     FILE *fh;
{
    int ch = fgetc(fh);
    struct pager_param_value * ret = NULL;

    enum pager_param_opcode opcode = pp_opcode_ANY; /* bad opcode */
    union pager_param_uvalue value;
    
    unsigned char debug_buffer[100];
    int debug_len = 0;

    ADD_DBUG(ch);
    
    if ('P' != ch) {
	if (EOF != ch) 
	    ungetc(ch,fh);

	DPRINT(DebugIO,30,(&DebugIO,
			   "get_pager_param_serialization_from_file: Not pager range param: %c\n",
			   ch));

	return NULL;
    }

    ch = fgetc(fh);
    ADD_DBUG(ch);

    if (ch >= '0' && ch <= '0') {
	int val = 0;
	
	/* opcode must be postive
	   except  pp_opcode_ANY which is missing value
	*/

	while (ch >= '0' && ch <= '9') {
	    val = val * 10 + ch - '0';
	    ch = fgetc(fh);
	    ADD_DBUG(ch);
	}

	opcode = val;
    }

    /* if value is NULL, it is missing value */

    value.magic = NULL;
    switch(ch) {
    case 't':

	value.text = get_string_serialization_from_file(fh);

	if (!value.text) {
	    panic("PAGER RANGE PANIC",__FILE__,__LINE__,
		  "get_pager_param_serialization_from_file",
		  "Bad data read from file",0);
	}

	ch = fgetc(fh);
	
	break;
    }

     if ('\n' != ch) {
	
	DPRINT(DebugIO,10,(&DebugIO,
			   "get_pager_param_serialization_from_file: Bad data %d (expected LF) buffer=",
			   ch));
	DEBUG_PRINT_BUFFER(DebugIO,10,debug_len, debug_buffer);
	DPRINT(DebugIO,10,(&DebugIO,"\n"));

	 panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	       "get_pager_param_serialization_from_file",
	       "Bad data read from file",0);
     }

     ret = malloc_pager_param_value(opcode,value);
     
     return ret;    
}

void put_pager_param_serialization_to_file(fh,param_value)
     FILE *fh;
     struct pager_param_value *param_value;
{
    enum pg_param_value_type value_type = 0;
    int ok = 0;
    
    if (PAGER_PARAM_VALUE_magic != param_value->magic)
    	panic("PAGER RANGE PANIC",__FILE__,__LINE__, "put_pager_param_serialization_to_file",
	      "Bad magic number (pager_param_value)",0);


    DPRINT(DebugIO,14,(&DebugIO,"put_pager_param_serialization_to_file: opcode=%d",
		       param_value->opcode));

    switch (param_value->opcode) {
    case pp_opcode_ANY:  /* Not really opcode */
	DPRINT(DebugIO,14,(&DebugIO," pp_opcode_ANY"));
	break;
    case pp_opcode_builtin:
	DPRINT(DebugIO,14,(&DebugIO," pp_opcode_builtin"));
	break;
    }

    if (param_value->value.magic) {
	value_type = * (param_value->value.magic);
	DPRINT(DebugIO,14,(&DebugIO," value type = %d",value_type));
	switch (value_type) {
	case  pg_pv_text:
	    DPRINT(DebugIO,14,(&DebugIO," pg_pv_text"));
	    ok = 1;
	    break;
	}
    } else {
	ok = 1;
	DPRINT(DebugIO,14,(&DebugIO," no value"));
    }
    DPRINT(DebugIO,14,(&DebugIO,"\n"));

    if (!ok)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,"put_pager_param_serialization_to_file",
	      "Bad magic number (pg_param_value)",0);
    
    fputc('P', fh);

    if (param_value->opcode >= 0) {
	/* pp_opcode_ANY is negative - treat as missing value */
	
	fprintf(fh,"%d",
		param_value->opcode);
    }

    if (param_value->value.magic) {
	switch (value_type) {
	case  pg_pv_text:
	    fputc('t', fh);
	    
	    put_string_serialization_to_file(fh,param_value->value.text);
	    break;
	}
    }
    
    fputc('\n',fh);    
}

int match_pager_param_value(param_value,opcode)
     struct pager_param_value *param_value;
     enum pager_param_opcode   opcode;
{

    if (PAGER_PARAM_VALUE_magic != param_value->magic)
    	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "match_pager_param_value",
	      "Bad magic number (pager_param_value)",0);
    
    if (pp_opcode_ANY == opcode ||
	/* XXXX is this correct ? */
	pp_opcode_ANY == param_value->opcode ||
	opcode == param_value->opcode)
	return 1;

    return 0;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
