static char rcsid[] = "@(#)$Id: mime_types.c,v 2.17 2021/10/23 15:05:08 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.17 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/


#include "def_misc.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"mime");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

struct mime_types_magic {
    unsigned            offset;
    size_t              length;
    unsigned char     * bytes;
};


struct mime_types_item {
    char                * extension;
    struct media_type   * type;

    char                * params;
    charset_t           params_cs;

    /* magic */

    struct mime_types_magic * magic;
    size_t                    magic_count;

    unsigned int         valid:1;
    unsigned int         changed:1;    
};

/* -------------------------------------------------------------------- */

#define U_(x) (unsigned char *)(x)

static struct mime_types_magic POSTSCRIPT[] = {    
    { 0,4, U_("%!PS") }
};

static struct mime_types_magic GIF[] = {
    { 0,3, U_("GIF") }
};

/* https://www.garykessler.net/library/file_sigs.html 

   FF D8                    Generic JPEGimage file

   FF D8 FF E0 xx xx 4A 46 49 46 00
                            JPEG/JFIF graphics file
   
   FF D8 FF E1 xx xx 45 78 69 66 00
                            Digital camera JPG using Exchangeable 
                            Image File Format (EXIF)

   http://www.faqs.org/faqs/jpeg-faq/part1/section-15.html
*/


static struct mime_types_magic JPEG_JFIF[] = {
    { 0,4, U_("\xff\xd8\xff\xe0") },
    
    /* magic includes \0  */
    { 6,5, U_("JFIF") }
};

static struct mime_types_magic JPEG_EXIF[] = {
    { 0,4, U_("\xff\xd8\xff\xe1") },

    /* magic includes \0  */
    { 6,5, U_("Exif") }
};

static struct mime_types_magic TIFF_little_endian[] = { /* Intel */
    { 0,4, U_("II\x2a\x00") },
};
static struct mime_types_magic TIFF_big_endian[] = { /* Motorola */
    { 0,4, U_("MM\x00\x2a") }
};


#define MIME_MAGIC(X) &(X[0]), sizeof (X) / sizeof(X[0])

static struct mime_types_item builtin_mimetypes_map_0[] = {

    /* text/plain -- no magic bytes */
    { "txt", &text_plain,            NULL, NULL,   NULL, 0,
      1,0
    },
    
    /* application/postscript */
    { "ps", &application_postscript, NULL, NULL,   MIME_MAGIC(POSTSCRIPT),
      1,0
    },

    /* image/gif */
    { "gif", &image_gif,             NULL, NULL,   MIME_MAGIC(GIF),
      1,0 },

    /* image/jpeg */

    /* RFC 2046 says: "An initial subtype is "jpeg" for the JPEG format
       using JFIF encoding [JPEG]."
    */

    { "jpeg",  &image_jpeg,          NULL, NULL,   MIME_MAGIC(JPEG_JFIF),
      1,0
    },
    { "jfif",  &image_jpeg,          NULL, NULL,   MIME_MAGIC(JPEG_JFIF),
      1,0
    },
    { "jpg",  &image_jpeg,           NULL, NULL,   MIME_MAGIC(JPEG_JFIF),
      1,0
    },

    /* But treat JPEG format with EXIF also as image/jpeg
     */
    
    { "jpg",  &image_jpeg,           NULL, NULL,   MIME_MAGIC(JPEG_EXIF),
      1,0
    },
    
    /* image/tiff */
    { "tiff",  &image_tiff,          NULL, NULL,   MIME_MAGIC(TIFF_big_endian),
      1,0
    },
    { "tiff",  &image_tiff,          NULL, NULL,   MIME_MAGIC(TIFF_little_endian),
      1,0
    },
    { "tif",   &image_tiff,          NULL, NULL,   MIME_MAGIC(TIFF_big_endian),
      1, 0
    },
    { "tif",   &image_tiff,          NULL, NULL,   MIME_MAGIC(TIFF_little_endian),
      1, 0
    },

    { NULL, NULL,                    NULL, NULL,   NULL, 0,
      0, 0
    },
};

struct mime_types_item  * builtin_mimetypes_map = builtin_mimetypes_map_0;


static int ok_byte P_((int byte));

#if __GNUC__ 
__inline__
#endif
static int ok_byte(byte)
     int byte;
{
    if (!isascii(byte))
	return 0;
    if (iscntrl(byte))
	return 0;
    if (!isprint(byte))
	return 0;
    if ('"' == byte)
	return 0;
    return 1;
}

static void dump_magic P_((FILE *f, unsigned offset,
			   size_t length,
			   unsigned char * bytes));
static void dump_magic(f,offset,length,bytes)
     FILE * f;
     unsigned            offset;
     size_t              length;
     unsigned char     * bytes;
{
    int q = 0;
    size_t i;

    fprintf(f,">%u,",offset);
    
    for (i = 0; i < length; i++) {

	if (!q && i < length-1 &&
	    ok_byte(bytes[i]) &&
	    ok_byte(bytes[i+1])) {

	    fputc(' ',f);
	    fputc('"',f);
	    q = 1;
	}

	if (!q) 
	    fputc(' ',f);
	else {
	    if (! ok_byte(bytes[i])) {
		fputc('"',f);
		q = 0;	    
		fputc(' ',f);			    
	    }
	}

	if (q)
	    fputc(bytes[i],f);
	else
	    fprintf(f,"%u",bytes[i]);
    }

    if (q) 
	fputc('"',f);

    fputc('\n',f);
}


static int parse_magic P_((char       * buf,
			   int          l1,
			   size_t     * reslen,  
			   char      ** resbuffer,
			   const char * filename,
			   int lineno,
			   int **errors));
static int parse_magic(buf,l1,reslen,resbuffer,
		       filename,
		       lineno,
		       errors)
     char       * buf; 
     int          l1; 
     size_t     * reslen;  
     char      ** resbuffer;
     int          lineno;
     const char * filename;
     int       ** errors;
{
    char * buffer = NULL;
    size_t    len = 0;
    char *      p = buf;

    while (p < buf +l1) {
   
	if (' ' == *p) {
	    p++;
	} else if ('"' == *p) {
	    char * start = ++p;
	    size_t len1;

	    while (p < buf +l1) {
		if ('"' == *p) 
		    break;
		p++;
	    }
	    if ('"' != *p) {
		DPRINT(Debug,2,(&Debug,"parse_magic: fail on: %s\n",p));
		goto fail;
	    }
	    len1 = p-start;
	    p++;

	    if (0 == len1) {
		DPRINT(Debug,2,(&Debug,"parse_magic: fail on: %s\n",p));
		goto fail;
	    }

	    safe_stralloc_append(&buffer,&len,start,len1);
	} else {
	    char * x;
	    char *p1 = p;
	    
	    long L = strtol(p,&x,0);

	    if ('\0' != *x && 
		' ' != *x) {
		DPRINT(Debug,2,(&Debug,"parse_magic: fail on: %s\n",p));
		goto fail;
	    }

	    if (L < 0) {
		DPRINT(Debug,2,(&Debug,"parse_magic: fail on: %s\n",p));
		goto fail;
	    }

	    p = x;

	    if (L < 256) {
		char X = L;

		safe_stralloc_append(&buffer,&len,&X,1);
	    } else {		
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeBadMagicValue,
				  "%s: %d: Magic value (%ld) unsupported: %.*s"),
			  filename,lineno,L,x-p1,p1);

		if (errors)
		    (*errors) ++;
				  
		if (L < 0x10000 &&
		    sizeof (uint16) == 2) {
		    /* Not well defined -- host byte order */
		    uint16 s = L;

		    safe_stralloc_append(&buffer,&len,(char *)&s,2);
		    
		} else {   /* Undefined :-) */
		    safe_stralloc_append(&buffer,&len,(char *)&L,sizeof L);
		}
	    }
	}
    }

    if (len > 0) {       
	*reslen    = len;
	*resbuffer = buffer;
	return 1;
    }
    DPRINT(Debug,2,(&Debug,"parse_magic: fail -- nothing collected: %s\n",
		    buf));

 fail:
    if (buffer)
	free(buffer);

    *reslen    = 0;
    *resbuffer = NULL;
    return 0;   
}

static const unsigned MAX_offset = 1024;

struct mime_types_item  * load_mime_types_map(filename,errors)
     const char *filename;
     int *errors;
{
    struct mime_types_item * result;
    charset_t  cs  = system_charset;    /* Charset of file */

    int result_len = 0;
    FILE * f;
    int max_result = 0;
    int c;
    char * buf = NULL;
    int lineno = 0;

    int last_c = 0;
    int err = can_open(filename,"r");

    if (err) {
	DPRINT(Debug,2,(&Debug,"load_mime_types_map: %s: %s (can_open)\n",
			filename,strerror(err)));
	return NULL;
    }

    f = fopen(filename,"r");
    if (!f) {
 	int err UNUSED_VAROK = errno;
	DPRINT(Debug,2,(&Debug,"load_mime_types_map: %s: %s\n",
			filename,strerror(err)));
	return NULL;
    }

    while(EOF != (c = fgetc(f))) {
	if ('\n' == c)
	    max_result++;
	last_c = c;
    }

    if (last_c && '\n' != last_c) {
	DPRINT(Debug,9,(&Debug, 
			"load_mime_types_map: %s, no newline on end of file\n",
			filename));
	max_result++;	
    }

    DPRINT(Debug,10,(&Debug,"load_mime_types_map: %s, max_result=%d\n",
		     filename,max_result));
    
    if (!max_result) {
	fclose(f);
	return NULL;
    }
    rewind(f);
    
    result = safe_calloc((max_result +1), sizeof (result[0]));
    
    /* malloc_gets reallocates buf and do not return \n.
       It returns -1 if line is longer than given limit (32000)
       and -2 is returned on error  (buffer may still be alloced)
    */

    while (result_len < max_result &&
	   !feof(f) && !ferror(f)) {
	int l1 = malloc_gets(&buf,32000,f);
	char * c = buf;
	char * extension = NULL;
	char * mime_type = NULL;
	char * subtype   = NULL;
	char * params    = NULL;
	char * c1;
	media_type_t TYPE;

	int ch;

	if (-1 == l1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLineNo,
			      "%s: %d: Too long line: %.30s..."),
		      filename,lineno+1,buf);
	    (*errors) ++;
	    goto OUT;

	} else if (-2 == l1) {
	    int err = errno;
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeReadError,
			  "%.30s: Reading error: %s"),
		      filename,strerror(err));
	    (*errors) ++;	    

	    goto OUT;
	}

	lineno++;
	
	if (0 == l1)
	    continue;
	
	while (l1-- > 0 && whitespace(buf[l1]))
	    buf[l1] = '\0';

	
	while (*c && whitespace ( *c)) /* skip leading whitespace */
	    c++;
	if (*c == '#') /* Skip comments */
	    continue;
	if (! *c)
	    continue;

	/* magic on wrong place */
	if ('>' == *c) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
		      filename,lineno,buf);
            (*errors) ++;
            break;
	}

	c1 = strpbrk(c," \t");
        if (!c1) {
           lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
		     filename,lineno,buf);
            (*errors) ++;
            break;
        }
        *c1 = '\0';

	if (0 == strcmp(c,"-"))
	    extension = NULL;
	else
	    extension = c;

	c1++;

	while (*c1 && whitespace (*c1)) /* skip leading whitespace */
            c1++;
        if (!*c1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
		      filename,lineno,buf);	    
            (*errors) ++;
            break;
        }
	
	mime_type = c1;

	while (*c1) {
	    if ('/' == *c1) {
		subtype = c1+1;
		*c1 = '\0';
	    }
	    if (';' == *c1)
		break;
	    if (whitespace (*c1)) {
		 lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
			   filename,lineno,buf);
		
		(*errors) ++;
		goto OUT;
	    }
	    c1++;
	}

	if (';' == *c1) {
	    *c1 = '\0';

	    c1++;
	    
	    while (*c1 && whitespace (*c1)) /* skip leading whitespace */
		c1++;

	    if (*c1)
		params = c1;

	} else if ('\0' != *c1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
		      filename,lineno,buf);	    
	    (*errors) ++;
	    goto OUT;	    
	}

	if (! mime_type[0] || !subtype || !subtype[0]) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
			      "%s: %d: Bad line: %.30s..."),
		      filename,lineno,buf);
	    (*errors) ++;
	    goto OUT;
	}

	TYPE = give_media_type(mime_type,subtype,1);
	if (!TYPE) {
	    DPRINT(Debug,1,(&Debug,
			    "load_mime_types_map: %s: FAILED to get type %s/%s\n",
			    filename,mime_type,subtype));
	    continue;
	}

	result[result_len].extension   = extension ?
	    safe_strdup(extension) : NULL;
	result[result_len].type        = TYPE;
	result[result_len].params      = params ?
	    safe_strdup(params) : NULL;
	result[result_len].params_cs    = cs;

	result[result_len].magic       = NULL;
	result[result_len].magic_count = 0;
	result[result_len].valid       = 1;
	result[result_len].changed     = 0;

	/* magic */
	ch = fgetc(f);

	if (ch == '>') {

	    struct mime_types_magic * magic        = NULL;
	    size_t                    magic_count  = 0;
	    unsigned                  min_offset   = 0;
	    
	    while (ch == '>') {
	    
		l1 = malloc_gets(&buf,32000,f);

		if (-1 == l1) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLineNo,
				      "%s: %d: Too long line: %.30s..."),
			      filename,lineno+1,buf);		
		    (*errors) ++;
		    
		    goto CLEAR1;
		
		} else if (-2 == l1) {
		    int err = errno;
		    
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeReadError,
				      "%s: Reading error: %s"),
			      filename,strerror(err));
		    (*errors) ++;	    
		    
		    goto CLEAR1;
		    
		}
		
		lineno++;
	    
		if (0 == l1 && 0 == magic_count)
		    goto nomagic;

		if (l1 > 0) {
		    unsigned offset = min_offset;
		    char * p;
		    char * buffer = NULL;
		    size_t    len    = 0;
		    		
		    p = buf;
		    if ('"' == *buf) {

			if (offset != 0) {
			    lib_error(CATGETS(elm_msg_cat, MeSet,
					      MeTreatingOffsetNonzero,
					      "%s: %d: Treating offset as %u: %.*s"),
				      filename,lineno,offset,
				      l1,buf);
			    (*errors) ++;					      
			}
			
			/* May also increment  (*errors) and return
			   1 on warning case
			*/
			if (! parse_magic(buf,l1,&len,&buffer,
					  filename,lineno,&errors)) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadMagicLineNo,
					      "%s: %d: Bad magic: %.30s..."),
				      filename,lineno,buf);

			    (*errors) ++;	    
			    goto CLEAR1;
			}
	    
		    } else {
			long L = strtol(buf,&p,10);
			
			if (L < min_offset || L > MAX_offset || *p != ',') {
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeMagicOffsetLineNo,
					      "%s: %d: Bad magic offset: %.30s..."),
				      filename,lineno,buf);
			    
			    (*errors) ++;	    		
			    goto CLEAR1;
			}
			    
			offset = L;
			p++;

			/* May also increment  (*errors) and return
			   1 on warning case
			*/
			if (! parse_magic(p,l1 - (p-buf),&len,&buffer,
					  filename,lineno,&errors)) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadMagicLineNo,
					      "%s: %d: Bad magic: %.30s..."),
				      filename,lineno,buf);
			    (*errors) ++;	    
			    goto CLEAR1;
			}

		    }

		    magic = 
			safe_array_realloc(magic,
					   magic_count+1,
					   sizeof (magic[0]));
		    
		    magic[magic_count].offset = offset;
		    magic[magic_count].length = len;
		    magic[magic_count].bytes  = s2us(buffer);

		    min_offset = offset + len;
		    
		    magic_count++;
		}

		ch = fgetc(f);
	    }
	    if (EOF != ch) 
		ungetc(ch,f);
	    
	    if (0) {
	    CLEAR1:

		if (magic) {
		    size_t z;

		    for (z = 0; z < magic_count; z++) {
			if (magic[z].bytes) {
			    free(magic[z].bytes);
			}
			magic[z].bytes  = NULL;
			magic[z].length = 0;
			magic[z].offset = 0;
		    }

		    free(magic);
		    magic = NULL;
		}
		magic_count = 0;		
	    }
	    

	    if (magic) {
		result[result_len].magic       = magic;
		result[result_len].magic_count = magic_count;
	    } else
		goto CLEAR;
	    	    
	} else if (EOF != ch) {
	    ungetc(ch,f);

	nomagic:
	    if (!extension) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMagicExtensionLineNo,
				  "%.30s: %d: Magic or extension required: %.30s..."),
			  filename,lineno,buf);
		(*errors) ++;	    		

	    CLEAR:
		if (result[result_len].extension) {
		    free(result[result_len].extension);
		    result[result_len].extension = NULL;
		}
		if (result[result_len].params) {
		    free(result[result_len].params);
		    result[result_len].params = NULL;
		}
		goto OUT;
	    }		
	}

	result_len++;
    }

    if (ferror(f)) {
	int err = errno;

	lib_error(CATGETS(elm_msg_cat, MeSet, MeReadError,
			  "%s: Reading error: %s"),
		  filename,strerror(err));
	(*errors) ++;
    }
    

 OUT:
    if (buf) {
	free(buf);
    }

    result[result_len].extension   = NULL;
    result[result_len].type        = NULL;
    result[result_len].params      = NULL;
    result[result_len].params_cs   = NULL;
    result[result_len].magic       = NULL;
    result[result_len].magic_count = 0;
    
    if (!feof(f)) {
	DPRINT(Debug,3,(&Debug,
			"load_mime_types_map: %s, All results not readed\n",
			filename));
    }

    fclose(f);

    
    DPRINT(Debug,10,(&Debug,
		     "load_mime_types_map: %s, result_len=%d\n",
		     filename,result_len));
    
    return result;
}

    
struct mime_types_item * loc_mime_type(list,extension)
     struct mime_types_item *list;
     const char *extension;
{
    struct mime_types_item *ptr;

    for (ptr = list; ptr->type; ptr++) {
	if (ptr->valid &&
	    ptr->extension &&
	    0 == istrcmp(ptr->extension,extension) &&
	    0 == ptr->magic_count)
	    return ptr;
    }

    return NULL;
}

media_type_t mime_type_to_media_type(item)
     const struct mime_types_item *item;
{
    return item->type; 
}

const char * mime_type_to_params(item,cs)
     const struct mime_types_item *item;
     charset_t *cs;
{
    if (cs)
	*cs = item->params_cs;

    return item->params;
}

const char * mime_type_to_extension(item)
     struct mime_types_item *item;
{
    return item->extension;
}

void dump_mime_types_map(f,map,commentfile,actor,version_buff)
     FILE *f; 
     struct mime_types_item  *map;
     FILE *commentfile; 
     const char *actor;
     char *version_buff;
{
    struct mime_types_item  *ptr;

    if (!map)
	return;

    insert_commentfile(f,ELMMIMETYPES_INFO,commentfile,actor,version_buff);
    
    for (ptr = map; ptr && ptr->type; ptr++) {

	if (!ptr->valid)
	    continue;
	
	if (ptr->extension)	    
	    fputs(ptr->extension,f);
	else
	    fputc('-',f);
	fputc('\t',f);
	fputs(get_major_type_name(ptr->type),f);
    	fputc('/',f);
	fputs(get_subtype_name(ptr->type),f);
	
	if (ptr->params) {
	    fputc(';',f);
	    fputc(' ',f);
	    fputs(ptr->params,f);
	}
    	fputc('\n',f);

	if (ptr->magic) {
	    size_t z;

	    for (z = 0; z < ptr->magic_count; z++) {		
		dump_magic(f,
			   ptr->magic[z].offset,
			   ptr->magic[z].length,
			   ptr->magic[z].bytes);
	    }
	}
    }
}
 
static int match_item P_((struct mime_types_item *p1,
			  struct mime_types_item *p2));
static int match_item(p1,p2)
     struct mime_types_item *p1;
     struct mime_types_item *p2;
{
    /* Do not check valid flag !! */
    
    if (p1->extension && p2->extension) {
	if (0 != strcmp(p1->extension,p2->extension))
	    return 0;
    } else if (p1->extension || p2->extension)
	return 0;

    if (p1->magic_count != p2->magic_count)
	return 0;

    if (p1->magic && p2->magic) {

	size_t z;

	for (z = 0; z < p1->magic_count; z++) {		
	    
	    if (p1->magic[z].length != p2->magic[z].length)
		return 0;
	    if (p1->magic[z].offset != p2->magic[z].offset)
		return 0;

	    if (p1->magic[z].bytes && p2->magic[z].bytes) {
		if (0 != memcmp(p1->magic[z].bytes,
				p2->magic[z].bytes,
				p1->magic[z].length))
		    return 0;
	    } else if (p1->magic[z].bytes || p2->magic[z].bytes) 
		return 0;

	}
	
    } else if (p1->magic || p2->magic)
	return 0;
    
    return 1;
}

/* Returns count of magic_numbers */
static size_t subset_match_item P_((struct mime_types_item *subset,
				    struct mime_types_item *newptr));
static size_t subset_match_item(subset,newptr)
     struct mime_types_item *subset;
     struct mime_types_item *newptr;
{
    /* Do not check valid flag !! */
    
    if (subset->extension && newptr->extension) {
	if (0 != strcmp(subset->extension,newptr->extension))
	    return 0;
    } else if (subset->extension || newptr->extension)
	return 0;
	
    if (subset->magic_count > newptr->magic_count)
	return 0;
    
    if (subset->magic && newptr->magic) {
	
	size_t z;
	
	for (z = 0; z < subset->magic_count; z++) {		
	    
	    if (subset->magic[z].length != newptr->magic[z].length)
		return 0;
	    if (subset->magic[z].offset != newptr->magic[z].offset)
		return 0;
	    
	    if (subset->magic[z].bytes && newptr->magic[z].bytes) {
		if (0 != memcmp(subset->magic[z].bytes,
				newptr->magic[z].bytes,
				subset->magic[z].length))
		    return 0;
	    } else if (subset->magic[z].bytes || newptr->magic[z].bytes) 
		return 0;	    
	}
	
    } else if (subset->magic || newptr->magic)
	return 0;
    
    return subset->magic_count;
}

void change_mime_types_map(map,new) 
     struct mime_types_item **map;
     struct mime_types_item *new;
{
    size_t count = 0;

    struct mime_types_item *ptr;

    for (ptr = *map; ptr && ptr->type; ptr++) 
	count++;
    
    for (ptr = new; ptr && ptr->type; ptr++)
	count++;

    if (!count)
	return;

    if (!*map) {
	*map = safe_calloc((count+1), sizeof ((*map)[0]));
	(*map)->type   = NULL;
	
    } else
	*map = safe_array_realloc(*map, (count+1), sizeof ((*map)[0]));


   for (ptr = new; ptr && ptr->type; ptr++) {
	struct mime_types_item *ptr2;

	struct mime_types_item *found     = NULL;
	size_t                  found_len = 0;
	
	
	if (!ptr->valid)
	    continue;
	
	
	for (ptr2 = *map; ptr2->type; ptr2++) {
	    if (match_item(ptr,ptr2)) {
		goto set_it;
	    }
	}

	for (ptr2 = *map; ptr2->type; ptr2++) {
	    size_t z = subset_match_item(ptr2,ptr);

	    if (z > found_len) {
		found     = ptr2;
		found_len = z;		
	    }
	}
	
	if (ptr2 > (*map) + count)
	    panic("CHARSET PANIC",__FILE__,__LINE__,"change_mime_types_map",
		  "Overflow",0);

	/* Remove subset from list */
	if (found && !found->changed)
	    found->valid = 0;

	ptr2->valid     = 0;
	ptr2->changed   = 0;
	ptr2->extension = NULL;
	if (ptr->extension)
	    ptr2->extension = safe_strdup(ptr->extension);

	ptr2->magic_count = ptr->magic_count;
	ptr2->magic = NULL;

	if (ptr->magic && ptr->magic_count) {
	    size_t z;
	    
	    ptr2->magic = safe_calloc(ptr->magic_count,
				      sizeof(ptr->magic[0]));

	    for (z = 0; z < ptr->magic_count; z++) {
	    
		ptr2->magic[z].length = ptr->magic[z].length;
		ptr2->magic[z].offset = ptr->magic[z].offset;

		ptr2->magic[z].bytes = NULL;
		if (ptr->magic[z].bytes && ptr->magic[z].length) {
		    ptr2->magic[z].bytes =
			safe_malloc(ptr->magic[z].length);
		    memcpy(ptr2->magic[z].bytes,
			   ptr->magic[z].bytes,ptr->
			   magic[z].length);
		}
	    }
	}
	
	ptr2->params = NULL;
	(ptr2+1)->type   = NULL;

   set_it:
	ptr2->valid     = 1;
	ptr2->changed   = 1;
	ptr2->type      = ptr->type;
	ptr2->params_cs = ptr->params_cs;
	
	if (ptr->params) {
	    ptr2->params = strmcpy(ptr2->params,ptr->params);

	} else if (ptr2->params) {
	    free(ptr2->params);
	    ptr2->params = NULL;
	}
	
   }
}

void ignore_mime_types_map(map,ignore)
     struct mime_types_item * map;
     struct mime_types_item * ignore;
{

    struct mime_types_item *ptr;

    for (ptr = ignore; ptr && ptr->type; ptr++) {
	struct mime_types_item *ptr2;

	for (ptr2 = map; ptr2->type; ptr2++) {
	    if (match_item(ptr,ptr2)) {

		ptr2->valid = 0;
	    }
	}
    }    
}

void free_mime_types_map(map)
     struct mime_types_item **map;
{
    struct mime_types_item *ptr;

    for (ptr = *map; ptr && ptr->type; ptr++) {

	if (ptr->extension) {
	    free(ptr->extension);
	    ptr->extension = NULL;
	}

	if (ptr->params) {
	    free(ptr->params);
	    ptr->params = NULL;
	}

	if (ptr->magic) {
	    size_t z;

	    for (z = 0; z < ptr->magic_count; z++) {
		
		if (ptr->magic[z].bytes) {
		    free(ptr->magic[z].bytes);
		    ptr->magic[z].bytes = NULL;
		}
		ptr->magic[z].length = 0;
	    }

	    free(ptr->magic);
	    ptr->magic = NULL;
	}
	ptr->magic_count = 0;
    }

    free(*map);
    *map = NULL;
}



/* -------------------------------------------------------------------- */

#define SCAN_LIST_magic		0xFC05

struct scan_list {
    unsigned short         magic;  /* SCAN_LIST_magic */

    int current_ptr;
    
    struct scan_list_item {
	int matches;     /* -1 == fail
			    number of matched magic so far

			    match if matches == magic_count
			 */

	struct mime_types_item *item;
    } * list;
    int list_len;    
};

struct scan_list * get_scan_list(list,extension)
     struct mime_types_item *list;
     const char *extension;
{
    struct mime_types_item *ptr;

    struct scan_list *ret = safe_zero_alloc(sizeof (*ret));
    
    ret->magic         = SCAN_LIST_magic;
    ret->current_ptr   = 0;

    ret->list      = NULL;
    ret->list_len  = 0;

    for (ptr = list; ptr->type; ptr++) {
	if (!ptr->valid)
	    continue;
       
	if (0 == ptr->magic_count)
	    continue;

	if (! ptr->extension ||
	    (extension && ptr->extension &&
	     0 == istrcmp(ptr->extension,extension))) {

	    ret->list = safe_array_realloc(ret->list,
					   (ret->list_len +1),
					   sizeof (ret->list[0]));

	    ret->list[ret->list_len].matches = 0;
	    ret->list[ret->list_len].item    = ptr;

	    ret->list_len++;
	}
    }   

    DPRINT(Debug,3,(&Debug,"get_scan_list: %d match (extension %s)\n",
		    ret->list_len,extension ? extension : "<none>"));
    
    return ret;
}

/* If there is type without magic, it means that any content
   is accepted for type */
struct scan_list * get_scan_list_by_type(list,mime_type)
     struct mime_types_item *list;
     media_type_t mime_type;
{
    struct mime_types_item *ptr;

    struct scan_list *ret = safe_zero_alloc(sizeof (*ret));
    
    ret->magic         = SCAN_LIST_magic;
    ret->current_ptr   = 0;

    ret->list      = NULL;
    ret->list_len  = 0;

    for (ptr = list; ptr->type; ptr++) {

	if (ptr->type == mime_type) {
	    ret->list = safe_array_realloc(ret->list,
					   sizeof (ret->list[0]),
					   (ret->list_len +1));

	    ret->list[ret->list_len].matches = 0;
	    ret->list[ret->list_len].item    = ptr;

	    ret->list_len++;

	}
    }

    DPRINT(Debug,3,(&Debug,"get_scan_list_by_type: %d match (type %s/%s)\n",
		    ret->list_len,
		    get_major_type_name(mime_type),
		    get_subtype_name(mime_type)));
	   
    return ret;
}



void free_scan_list(list)
     struct scan_list **list;
{

    if (SCAN_LIST_magic != (*list)->magic)
	panic("MIME TYPES PANIC",__FILE__,__LINE__,"free_scan_list",
	      "Bad magic",0);

    if ((*list)->list) {
	free((*list)->list);
	(*list)->list = NULL;
    }
    (*list)->list_len = 0;

    (*list)->magic = 0;  /* Invalidate */

    free(*list);
    *list = NULL;
}

void append_scanlist(list,other)
     struct scan_list *list;
     const struct scan_list *other;
{
    int newlen,i;

    if (SCAN_LIST_magic != list->magic)
	panic("MIME TYPES PANIC",__FILE__,__LINE__,"append_scanlist",
	      "Bad magic",0);

    if (SCAN_LIST_magic != other->magic)
	panic("MIME TYPES PANIC",__FILE__,__LINE__,"append_scanlist",
	      "Bad magic",0);

    if (list->current_ptr)
	panic("MIME TYPES PANIC",__FILE__,__LINE__,"append_scanlist",
	      "Called on middle on scan",0);

    /* Avoid allocing 0 bytes */
    if (other->list_len < 1)
	return;
	

    newlen = list->list_len + other->list_len;
    
    list->list = safe_array_realloc(list->list,
				    newlen,
				    sizeof (list->list[0]));
			     
    for (i = 0; i < other->list_len; i++) {

	if (list->list_len >= newlen)
	    panic("MIME TYPES PANIC",__FILE__,__LINE__,"append_scanlist",
		  "Overflow",0);

	list->list[list->list_len].matches = 0;
	list->list[list->list_len].item    = other->list[i].item;
	
	list->list_len++;
    }
}


void filter_scanlist(list,att,filter)
     struct scan_list *list; 
     const struct mimeinfo * att;
     filter_scanlist_OK_f filter;
{
    int i,p;
    int old_len;
    
    if (SCAN_LIST_magic != list->magic)
	panic("MIME TYPES PANIC",__FILE__,__LINE__,"filter_scanlist",
	      "Bad magic",0);

    if (list->current_ptr)
	panic("MIME TYPES PANIC",__FILE__,__LINE__,"filter_scanlist",
	      "Called on middle on scan",0);

    old_len = list->list_len;
    
    for (i = 0, p = 0; i < list->list_len; i++) {

	if (filter(att,list->list[i].item)) {

	    if (p != i) {

		DPRINT(Debug,7,(&Debug,
				"filter_scanlist: Moving %d -> %d\n",
				i,p));

		
		list->list[p] = list->list[i];
		
		list->list[i].item = NULL;

	    }

	    p++;
	} else {

	    DPRINT(Debug,7,(&Debug,
			    "filter_scanlist: Filtering item %d\n",
			    i));

	    list->list[i].item = NULL;
	}
    }

    if (p != list->list_len) {

	DPRINT(Debug,7,(&Debug,
			"filter_scanlist: Shortening list %d => %d\n",
			list->list_len,p));

	list->list_len = p;
    }

    for (; p < old_len; p++) {

	/* Should not happen */

	if (list->list[p].item) {
	    DPRINT(Debug,7,(&Debug,
			    "filter_scanlist: Zeroing old item %d\n",
			    p));
    

	    list->list[p].item = NULL;
	}
    }

}


int scanlist_need_more(list,ch)
     struct scan_list *list; 
     int ch;
{
    int i;
    int needmore = 0;


    if (SCAN_LIST_magic != list->magic)
	panic("MIME TYPES PANIC",__FILE__,__LINE__,"scanlist_need_more",
	      "Bad magic",0);

    if (0 == list->current_ptr) {
	DPRINT(Debug,7,(&Debug,
			"scanlist_need_more: %d items on scan\n",
			list->list_len));
    }

    
    for (i = 0; i < list->list_len; i++) {

	struct mime_types_item *item = 
	    list->list[i].item ;

    rescan_item:
	
	if (list->list[i].matches >= 0 &&
	    list->list[i].matches < item->magic_count &&
	    item->magic) {

	    struct mime_types_magic * mitem =
		&(item->magic[list->list[i].matches]);
	    
	    int x;

	    if (list->current_ptr < mitem->offset) {
		needmore = 1;
		continue;
	    }

	    if (mitem->offset == list->current_ptr &&
		0 == mitem->length) {
		DPRINT(Debug,7,(&Debug,
				"scanlist_need_more: [%d] %d/%lu no magic -- matches to any content at offset %d\n",
				i,
				list->list[i].matches,
				(unsigned long)(item->magic_count),
				list->current_ptr));

		goto do_match;

	    } else if (list->current_ptr >= mitem->offset + mitem->length) 
		panic("MIME TYPES PANIC",__FILE__,__LINE__,
		      "scanlist_need_more",
		      "matches was not incremented",0);

	    x = list->current_ptr - mitem->offset;
	    
	    if (ch == mitem->bytes[x]) {
		/* Still matches */

		if (x == mitem->length-1) {

		    int j;

		    DPRINT(Debug,7,(&Debug,
				    "scanlist_need_more: [%d] %d/%lu matches on offset %d ch=%d\n",
				    i,
				    list->list[i].matches,
				    (unsigned long)(item->magic_count),				
				    list->current_ptr,ch));
		    
		    DPRINT(Debug,7,(&Debug,
				    "scanlist_need_more: pattern %d,",
				    mitem->offset));

		    for (j = 0; j < mitem->length; j++) {
			DPRINT(Debug,7,(&Debug," %d",mitem->bytes[j]));
		    }
		    DPRINT(Debug,7,(&Debug,"\n"));


		do_match:
		    list->list[i].matches++;

		    if (list->list[i].matches == item->magic_count) {
			DPRINT(Debug,7,(&Debug,
					"scanlist_need_more: [%d] full match",
					i));
			if (item->type) {
			    DPRINT(Debug,7,(&Debug,", type %s/%s",
					    get_major_type_name(item->type),
					    get_subtype_name(item->type)));
			    if (item->params) {
				DPRINT(Debug,7,(&Debug,"; %s",
						item->params));
			    }
			    
			}
			DPRINT(Debug,7,(&Debug,"\n"));
			
			goto skip_scan;			
		    }

		    if (list->list[i].matches < item->magic_count &&
			list->current_ptr >
			item->magic[list->list[i].matches].offset) {
			DPRINT(Debug,7,(&Debug,
					"scanlist_need_more: [%d] %d/%lu start offset %lu passed already\n",
					i,
					list->list[i].matches,
					(unsigned long)(item->magic_count),
					item->magic[list->list[i].matches].offset));
			goto fail_this;
		    }
		    
		    goto rescan_item;
		}

	    } else {
		int j;

	    fail_this:
		DPRINT(Debug,7,(&Debug,
				"scanlist_need_more: [%d] %d/%lu fails on offset %d ch=%d\n",
				i,
				list->list[i].matches,
				(unsigned long)(item->magic_count),				
				list->current_ptr,ch));
		
		DPRINT(Debug,7,(&Debug,
				"scanlist_need_more: pattern %d,",
				mitem->offset));
		
		for (j = 0; j < mitem->length; j++) {
		    DPRINT(Debug,7,(&Debug," %d",mitem->bytes[j]));
		}
		DPRINT(Debug,7,(&Debug,"\n"));

		list->list[i].matches = -1;		
	    }
	}

    skip_scan:	
	if (list->list[i].matches >= 0 &&
	    list->list[i].matches <  item->magic_count)
	    needmore = 1;
    }

    list->current_ptr++;
    
    return needmore;
}

/* Returns 1 is list is not empty */
int have_scanlist(list)
     struct scan_list *list; 
{
    if (SCAN_LIST_magic != list->magic)
	panic("MIME TYPES PANIC",__FILE__,__LINE__,"have_scanlist",
	      "Bad magic",0);

    return list->list_len != 0;
}

struct mime_types_item * loc_mime_type_from_scan(list)
     struct scan_list *list;
{
    struct mime_types_item *ret = NULL;
    size_t match_len  = 0;
    int    have_match = 0;
    int i;

    if (SCAN_LIST_magic != list->magic)
	panic("MIME TYPES PANIC",__FILE__,__LINE__,
	      "loc_mime_type_from_scan",
	      "Bad magic",0);

    /* Pick longest match */
    
    for (i = 0; i < list->list_len; i++) {

	size_t z;
	size_t len = 0;
	
	for (z = 0; z < list->list[i].item->magic_count; z++)
	    len += list->list[i].item->magic[z].length;
	
	if (list->list[i].item->magic_count == list->list[i].matches) {

	    DPRINT(Debug,7,(&Debug,
			    "loc_mime_type_from_scan: [%d] matches (match len %lu, %lu strings)",
			    i,(unsigned long)len,
			    (unsigned long)(list->list[i].item->magic_count)));

	    if (list->list[i].item->type) {
		DPRINT(Debug,7,(&Debug,", type %s/%s",
				get_major_type_name(list->list[i].item->type),
				get_subtype_name(list->list[i].item->type)));

		if (list->list[i].item->params) {
		    DPRINT(Debug,7,(&Debug,"; %s",
				    list->list[i].item->params));
		}
	    }
	    
	    DPRINT(Debug,7,(&Debug,"\n"));
	    
	    if (!have_match ||
		match_len < len) {
	    	    
		if (0 == list->list[i].item->magic_count) {
		    DPRINT(Debug,7,(&Debug,
				    "loc_mime_type_from_scan:  [%d] no magic -- matches to any content\n",
				    i));
		}
				
		ret = list->list[i].item;
		match_len = len;
		have_match = 1;
	    }
	} else if (list->list[i].matches >= 0) {

	    DPRINT(Debug,7,(&Debug,
			    "loc_mime_type_from_scan: [%d] incomplete match (match len %lu, %d/%lu strings)",
			    i,(unsigned long)len,
			    list->list[i].matches,
			    (unsigned long)(list->list[i].item->magic_count)));
	    
	}
    }

    DPRINT(Debug,7,(&Debug,
		    "loc_mime_type_from_scan: "));
    if (ret) {
	DPRINT(Debug,7,(&Debug,"found match len %lu",
			(unsigned long)match_len));

	if (ret->type) {
	    DPRINT(Debug,7,(&Debug,", type %s/%s",
			    get_major_type_name(ret->type),
			    get_subtype_name(ret->type)));
	    if (ret->params) {
		DPRINT(Debug,7,(&Debug,"; %s",
				ret->params));
	    }
	}
	
    } else {
	DPRINT(Debug,7,(&Debug,"not found"));
    }
    DPRINT(Debug,7,(&Debug,"\n"));
    
    return ret;
}

struct mime_types_item * loc_mime_type_from_scan_extension(list,extension)
     struct scan_list *list; 
     char *extension;
{
    struct mime_types_item *ret = NULL;
    int match_len  = -1;
    int i;

    if (SCAN_LIST_magic != list->magic)
	panic("MIME TYPES PANIC",__FILE__,__LINE__,"scanlist_need_more",
	      "Bad magic",0);

    /* Pick longest match -- first try matching extension */
    
    for (i = 0; i < list->list_len; i++) {

	size_t z;
	size_t len = 0;
	
	for (z = 0; z < list->list[i].item->magic_count; z++)
	    len += list->list[i].item->magic[z].length;
	
	if (list->list[i].item->magic_count == list->list[i].matches &&
	    match_len < len &&
	    list->list[i].item->extension &&
	    0 == istrcmp(list->list[i].item->extension,extension)
	    
	    ) {

	    if (0 == list->list[i].item->magic_count) {
		DPRINT(Debug,7,(&Debug,
				"loc_mime_type_from_scan_extension:  [%d] no magic -- matches to any content\n",
				i));
	    }

	    ret = list->list[i].item;
	    match_len = len;
	}		
    }

    if (!ret)
	ret = loc_mime_type_from_scan(list);

    return ret;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
