static char rcsid[] = "@(#)$Id: mbox.c,v 2.38 2024/06/23 07:38:36 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.38 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *       or Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************
 * Based on Elm 2.4 code ../src/newmbox.c and ../src/lock.c.
 * That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

#include "def_mbox.h"
#ifdef USE_DLOPEN
#include "shared_imp.h"
#endif
#include "s_me.h"
#include "s_elm.h"
#include "quota_imp.h"

DEBUG_VAR(Debug,__FILE__,"mbox");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

static const  unsigned char *cs2us P_((const char *str));
static const  unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif


/* Folder type dependent routines ------------------------------------------ */

S_(mbx_close_folder mbx_close_no_name)
static int mbx_close_no_name P_((struct folder_info *folder,
				  enum close_mode mode,
				  struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
							      on remote mailbox
							    */));
static int mbx_close_no_name(folder,mode,cd) 
     struct folder_info *folder;
     enum close_mode mode;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  on remote mailbox
			       */;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_close_no_name=1 (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return 1;
}

S_(mbx_lock_folder mbx_lock_no_name)
static int  mbx_lock_no_name  P_((enum lock_direction direction,struct folder_info *folder));
static int  mbx_lock_no_name(direction,folder)
     enum lock_direction direction;
     struct folder_info *folder;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_lock_no_name=0 (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return 0;
}

S_(mbx_unlock_folder mbx_unlock_no_name)
static int mbx_unlock_no_name P_((int interrupt, struct folder_info *folder));
static int mbx_unlock_no_name(interrupt,folder) 
     int interrupt;
     struct folder_info *folder;
{
    DPRINT(Debug,11,(&Debug,					 
		     "mbx_unlock_no_name=1 (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return 1;
}

S_(mbx_init_folder mbx_init_null)
static void mbx_init_null P_((struct folder_info *folder));
static void mbx_init_null(folder) 
     struct folder_info *folder;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_init_null: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    folder -> cur_tempfolder[0] = '\0';
    folder -> p = NULL;
}

S_(mbx_free_folder mbx_free_null)
static int mbx_free_null P_((struct folder_info *folder,
			      struct cancel_data  * cd
			      ));
static int mbx_free_null(folder,cd)
     struct folder_info *folder;
     struct cancel_data  * cd;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_free_null=1: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    return 1;
}

S_(mbx_sessionlock_folder  mbx_sessionlock_no_name)
static enum sessionlock_status mbx_sessionlock_no_name P_((struct folder_info *folder,
							   enum sessionlock_mode mode,
							   int *err /* errno */,
							   RECONNECT_MODE   * reconnect_mode_ptr
							   ));

static enum sessionlock_status mbx_sessionlock_no_name(folder,mode,err,reconnect_mode_ptr)
     struct folder_info *folder;
     enum sessionlock_mode mode;
     int *err /* errno */;
     RECONNECT_MODE   * reconnect_mode_ptr;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_sessionlock_no_name=sessionlock_open (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));	   
    return sessionlock_open;
}

S_(mbx_flush_folder  mbx_flush_null)
static int mbx_flush_null P_((struct folder_info *folder,
			      int *err_p  /* errno value */,
			      struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
							 on remote mailbox
						      */));
static int mbx_flush_null(folder,err_p,cd) 
     struct folder_info *folder;
     int *err_p  /* errno value */;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  on remote mailbox
			       */;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_flush_null=1: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    return 1; /* Always succeed */
}

S_(mbx_ferror_folder mbx_ferror_no_name)
static int mbx_ferror_no_name P_((struct folder_info *folder, int clean));
static int mbx_ferror_no_name(folder,clean) 
     struct folder_info *folder; 
     int clean; 
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_ferror_no_name=0 (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return 0;
}

static struct read_folder_state *
  malloc_read_folder_state P_((void));
static  struct read_folder_state * malloc_read_folder_state()
{
    struct read_folder_state * ret = safe_zero_alloc(sizeof (*ret));

    ret -> magic         = RF_magic;
    ret -> fbytes        = 0;
    ret -> linecounter   = 0;
    
    ret -> fbytes_body   = 0;
    ret -> skipping      = sm_EOM; /*  0 == reading, 1 = skipping, 
					  -1 == end of message */
    ret -> skip_count    = 0;
    ret -> skip_bytes    = 0;

    ret->reconnect_copy_previous = NULL;
    
    DPRINT(Debug,11,(&Debug,
		     "malloc_read_folder_state=%p\n",
		     ret));

    return ret;
}



static void free_read_folder_state P_((struct read_folder_state **ptr));
static void free_read_folder_state(ptr)
     struct read_folder_state **ptr;
{
    DPRINT(Debug,11,(&Debug,
		     "free_read_folder_state: (*ptr)=%p\n",
		     *ptr));

    if (RF_magic != (*ptr) -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_read_folder_state",
	      "Bad magic number (read state)",0);

    
    
    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)*ptr,sizeof (struct read_folder_state));

    free(*ptr);
    *ptr = NULL;
}

struct reconnect_mode * malloc_reconnect_mode()
{
    struct reconnect_mode *ret = safe_zero_alloc(sizeof (*ret));

    ret->magic           = RECONNECT_MODE_magic;
    ret->tempfolder_fh   = NULL;
    ret->tempfolder_name = NULL;

    DPRINT(Debug,11,(&Debug,
		     "malloc_reconnect_mode=%p\n",
		     ret));
    
    return ret;
}

void free_reconnect_mode(ptr)
     struct reconnect_mode **ptr;
{
    DPRINT(Debug,11,(&Debug,
		     "free_reconnect_mode: (*ptr)=%p\n",
		     *ptr));

    
    if (RECONNECT_MODE_magic !=  (*ptr) -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_reconnect_mode",
	      "Bad magic number (reconnect_mode)",0);

    if ((*ptr)->tempfolder_fh) {
	if (EOF == fclose((*ptr)->tempfolder_fh)) {
	    int err UNUSED_VAROK  = errno;
	    
	    DPRINT(Debug,11,(&Debug,
			     "free_reconnect_mode: %s: fclose failed: %s (errno %d)",
			     (*ptr)->tempfolder_name ? (*ptr)->tempfolder_name : "<no name>",
			     strerror(err),err));
	}
    }

    if ((*ptr)->tempfolder_name) {
	int r = unlink((*ptr)->tempfolder_name);

	if (0 == r) {
	    DPRINT(Debug,11,(&Debug,
			     "free_reconnect_mode: %s unlinked\n",
			     (*ptr)->tempfolder_name));
	} else if (-1 == r) {
	    int err UNUSED_VAROK  = errno;

	    DPRINT(Debug,11,(&Debug,
			     "free_reconnect_mode: %s: unlink failed: %s (errno %d)",
			     (*ptr)->tempfolder_name,strerror(err),err));
	}

	free((*ptr)->tempfolder_name);
	(*ptr)->tempfolder_name = NULL;
    }

    (*ptr) -> magic = 0;  /* Invalidate */
    free (*ptr);
    *ptr = NULL;
}

S_(mbx_prepare_read_folder mbx_prepare_read_no_name)
static enum prepare_result mbx_prepare_read_no_name P_((struct folder_info *folder,
					enum prepare_mode mode,
					READ_STATE read_state_ptr,
					RECONNECT_MODE reconnect_ptr
					));
static enum prepare_result mbx_prepare_read_no_name(folder,mode,read_state_ptr,
						    reconnect_ptr)
     struct folder_info *folder;
     enum prepare_mode mode;
     READ_STATE read_state_ptr;
     RECONNECT_MODE reconnect_ptr;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_prepare_read_no_name: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                        : read_state_ptr=%p\n",
		     read_state_ptr));
    DPRINT(Debug,11,(&Debug,
		     "mbx_prepare_read_no_name=prepare_done\n"));
    return prepare_done;
}

S_(mbx_end_read_folder  mbx_end_read_null)
static int mbx_end_read_null P_((struct folder_info *folder,
				 READ_STATE read_state_ptr,
				 RECONNECT_MODE reconnect_ptr,
				 int silent));
static int mbx_end_read_null(folder,read_state_ptr,reconnect_ptr,silent)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     RECONNECT_MODE reconnect_ptr;
     int silent;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_read_null: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                 : read_state_ptr=%p\n",
		     read_state_ptr));
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_read_null=1\n"));
    return 1;
}

S_(mbx_copy_envelope_folder mbx_copy_envelope_no_name)
static enum copy_env_status mbx_copy_envelope_no_name P_((struct folder_info *folder,
							  READ_STATE read_state_ptr,
							  struct header_rec *entry,
							  int force));
static enum copy_env_status mbx_copy_envelope_no_name(folder,read_state_ptr,entry,force)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     struct header_rec *entry;
     int force;
{
    DPRINT(Debug,11,(&Debug,					 
		     "mbx_copy_envelope_no_name: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                         : read_state_ptr=%p\n",
		     read_state_ptr));

    entry->offset = 0;
    entry->status = VISIBLE;

    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_envelope_no_name=0\n"));
    return copy_env_eof;
}

S_(mbx_is_forwarded_folder mbx_is_forwarded_null)
static const char * mbx_is_forwarded_null P_((struct folder_info *folder,
					      READ_STATE read_state_ptr));
static const char * mbx_is_forwarded_null(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_is_forwarded_null: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                     : read_state_ptr=%p\n",
		     read_state_ptr));   
    DPRINT(Debug,11,(&Debug,
		     "mbx_is_forwarded_null=NULL\n"));
    return NULL;
}

S_(mbx_copy_header_folder mbx_copy_header_no_name)
static int mbx_copy_header_no_name P_((struct folder_info *folder,
				       READ_STATE read_state_ptr,
				       char **buffer, int *len));
static int mbx_copy_header_no_name(folder,read_state_ptr,buffer,len)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_header_no_name: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                       : read_state_ptr=%p\n",
		     read_state_ptr));
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_header_no_name=0\n"));
    return 0;
}

S_(mbx_copy_body_folder mbx_copy_body_no_name)
static int mbx_copy_body_no_name P_((struct folder_info *folder,
				     READ_STATE read_state_ptr,
				     char **buffer, int *len,
				     long *content_remaining));

static int mbx_copy_body_no_name(folder,read_state_ptr,buffer,len,
				 content_remaining)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
     long *content_remaining;

{ 
    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_body_no_name: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                     : read_state_ptr=%p\n",
		     read_state_ptr));

    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_body_no_name=0\n"));
    return 0;
}

S_(mbx_copy_envelope_end_folder mbx_copy_envelope_end_no_name)
static enum copy_env_end_status mbx_copy_envelope_end_no_name P_((struct folder_info *folder,
					     READ_STATE read_state_ptr,
					     long *newbytes,
					     int  *newmails
					     ));

static enum copy_env_end_status mbx_copy_envelope_end_no_name(folder,read_state_ptr,
					 newbytes,newmails)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     long *newbytes;
     int  *newmails;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_envelope_end_no_name: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                             : read_state_ptr=%p\n",
		     read_state_ptr));
    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_envelope_end_no_name=copy_env_end_match\n"));
    return copy_env_end_match; /* Always end of message */
}

S_(mbx_copy_envelope_reset_body mbx_copy_envelope_reset_body_no_name)
static int mbx_copy_envelope_reset_body_no_name P_((struct folder_info *folder,
						    READ_STATE
						    read_state_ptr));
static int mbx_copy_envelope_reset_body_no_name(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_envelope_reset_body_no_name: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                                    : read_state_ptr=%p\n",
		     read_state_ptr));
    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_envelope_reset_body_no_name=0\n"));
    return 0;
}

S_(mbx_folder_to_fd mbx_no_name_to_fd)
static FILE * mbx_no_name_to_fd P_((struct folder_info *folder,long offset));
static FILE * mbx_no_name_to_fd(folder,offset)
     struct folder_info *folder;
     long offset;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_no_name_to_fd: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "mbx_no_name_to_fd=NULL\n"));
    return NULL;
}

S_(mbx_new_mail_on_folder mbx_new_mail_on_no_name)
static enum new_mail_stat mbx_new_mail_on_no_name P_((struct folder_info *folder,
						      long *bytes,int  *new_mailcount,
						      int *err_p  /* errno value */,
						      struct cancel_data  * cd
						      /* Allow cancelation (Ctrl-C)
							 check of remote mailbox
						      */
						      ));
static enum new_mail_stat mbx_new_mail_on_no_name(folder,bytes,new_mailcount,err_p,
						  cd)
     struct folder_info *folder;
     long *bytes;
     int  *new_mailcount /* -1 for local mailbox */;
     int *err_p  /* errno value */;
     struct cancel_data  * cd
     /* Allow cancelation (Ctrl-C)
	check of remote mailbox
     */;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_new_mail_on_no_name: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "mbx_new_mail_on_no_name=no_new_mail\n"));

    if (bytes)
	*bytes = 0;
    if (new_mailcount)
	*new_mailcount = 0;
    if (err_p)
	*err_p =0;
    
    return no_new_mail;
}

S_(mbx_consider_remove_folder mbx_consider_remove_null)
static int mbx_consider_remove_null P_((struct folder_info *folder));
static int mbx_consider_remove_null(folder)
     struct folder_info *folder;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_consider_remove_null=0 (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return 0;
}

static void malloc_keep_folder_state P_((struct keep_folder_state **ptr));
static void malloc_keep_folder_state(ptr)
     struct keep_folder_state **ptr;
{
    *ptr = safe_zero_alloc(sizeof (struct keep_folder_state));

    (*ptr)->magic = KS_magic;
}

static void free_keep_folder_state P_((struct keep_folder_state **ptr));
static void free_keep_folder_state(ptr)
     struct keep_folder_state **ptr;
{
    bzero((void *)*ptr,sizeof (struct keep_folder_state));
    free(*ptr);
    *ptr = NULL;
}

S_(mbx_prepare_keep_folder mbx_prepare_keep_no_name)
static int mbx_prepare_keep_no_name P_((struct folder_info *folder,
				       KEEP_STATE keep_state_ptr));
static int mbx_prepare_keep_no_name(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_prepare_keep_no_name=0 (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return 0;
}

S_(mbx_end_keep_folder mbx_end_keep_no_name)
static int mbx_end_keep_no_name P_((struct folder_info *folder,
				    KEEP_STATE keep_state_ptr));
static int mbx_end_keep_no_name(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_keep_no_name=1 (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return 1;
}

S_(mbx_mark_keep_folder mbx_mark_keep_no_name)
static int mbx_mark_keep_no_name P_((struct folder_info *folder,
				      KEEP_STATE keep_state_ptr,
				      struct header_rec *entry,
				      int keep));
static int mbx_mark_keep_no_name(folder,keep_state_ptr,entry,keep)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
     struct header_rec *entry;
     int keep;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_mark_keep_no_name (dummy)=0: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    return 0;
}

S_(mbx_folder_type mbx_no_name_type)
const char * mbx_no_name_type P_((struct folder_info *folder));
const char * mbx_no_name_type(folder)
     struct folder_info *folder;
{
    static char * status = NULL;

    if (!status)
	status = catgets(elm_msg_cat, ElmSet, ElmNoFolder, 
			 "No folder");

    DPRINT(Debug,11,(&Debug,
		     "mbx_no_name_type: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,		       		 
		     "mbx_no_name_type=%s\n",
		     status));
    return status;
}

S_(mbx_start_edit_folder mbx_start_edit_no_name)
static int mbx_start_edit_no_name P_((struct folder_info *folder, 
				     const char **buffer));
static int mbx_start_edit_no_name(folder,buffer)
     struct folder_info *folder;
     const char **buffer;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_start_edit_no_name=0 (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return 0;
}

S_(mbx_end_edit_folder mbx_end_edit_no_name)
static enum end_edit_fol_status mbx_end_edit_no_name P_((struct folder_info *folder));
static enum end_edit_fol_status mbx_end_edit_no_name(folder)
     struct folder_info *folder;
{
    DPRINT(Debug,11,(&Debug,					 
		     "mbx_end_edit_no_name=end_edit_none (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return end_edit_none;
}

S_(mbx_get_folder_mode mbx_get_no_name_mode)
static int mbx_get_no_name_mode P_((struct folder_info *folder));
static int mbx_get_no_name_mode(folder)
     struct folder_info *folder;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_get_no_name_mode=0 (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return 0;
}


S_(mbx_zero_rs_fields_folder mbx_zero_rs_fields_null)
static void mbx_zero_rs_fields_null P_((struct read_folder_state *rs));
static void mbx_zero_rs_fields_null(rs)
     struct read_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_zero_rs_fields_null (dummy): rs=%p\n",
		     rs));
}

S_(mbx_free_rs_fields_folder mbx_free_rs_fields_null)
static void mbx_free_rs_fields_null P_((struct read_folder_state *rs));
static void mbx_free_rs_fields_null(rs)
     struct read_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_free_rs_fields_null (dummy): rs=%p\n",
		     rs));
}

S_(mbx_zero_ks_fields_folder mbx_zero_ks_fields_null)
static void mbx_zero_ks_fields_null P_((struct keep_folder_state *rs));
static void mbx_zero_ks_fields_null(rs)
     struct keep_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_zero_ks_fields_null (dummy): rs=%p\n",
		     rs));

}

S_(mbx_free_ks_fields_folder mbx_free_ks_fields_null)
static void mbx_free_ks_fields_null P_((struct keep_folder_state *rs));
static void mbx_free_ks_fields_null(rs)
     struct keep_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_free_ks_fields_null (dummy): rs=%p\n",
		     rs));

}

static struct folder_type no_name   = {
                                        FOLDER_TYPE_magic, "(none)",
                                        mbx_close_no_name,   
					mbx_lock_no_name,
                                        mbx_init_null,
					mbx_sessionlock_no_name,
					mbx_unlock_no_name,
					mbx_flush_null,
					mbx_ferror_no_name,
					mbx_prepare_read_no_name,
					mbx_end_read_null,
					mbx_copy_envelope_no_name,
					mbx_is_forwarded_null,
					mbx_copy_header_no_name,
					mbx_copy_body_no_name,
					mbx_copy_envelope_end_no_name,
					mbx_copy_envelope_reset_body_no_name,
					mbx_no_name_to_fd,
					mbx_new_mail_on_no_name,
					mbx_consider_remove_null,
					mbx_prepare_keep_no_name,
					mbx_end_keep_no_name,
					mbx_mark_keep_no_name,
					mbx_no_name_type,
					mbx_start_edit_no_name,
					mbx_end_edit_no_name,
					mbx_free_null,
					mbx_zero_rs_fields_null,
					mbx_free_rs_fields_null,
					mbx_zero_ks_fields_null,
					mbx_free_ks_fields_null,
					mbx_get_no_name_mode,
					mbx_give_normal_server,
					mbx_comp_prev_hdr_default,
					mbx_update_prev_hdr_default,
					mbx_have_default_quota,
					mbx_give_default_quotar_list,
					mbx_give_message_msize_default,
					mbx_give_message_count_default
};

const folder_type_p NO_NAME   = &no_name;
const folder_type_p NON_SPOOL = &non_spool;	/* mailfile -- not mailbox */
const folder_type_p SPOOL     = &spool;		/* normal mailbox */
const folder_type_p READ_ONLY = &read_only;	/* read only mailfile */
#ifdef REMOTE_MBX
const folder_type_p POP_MBX   = &pop_mbx;	/* POP mailbox */
const folder_type_p IMAP_MBX  = &imap_mbx;	/* IMAP mailbox */
#endif

/* General routines -------------------------------------------------------- */

folder_type_p get_folder_type(filename, in_mail,treat_as_spooled)
     const char *filename;
     enum folder_place *in_mail;
     int treat_as_spooled;
{
    /** returns the type of mailfile filename is
	NO_NAME = no name
	SPOOL = consisting only of mailhome plus base file name
	(no intervening directory name)
	NON_SPOOL = a name that is not SPOOL type above
	READ_ONLY = open this read-only
    **/

    struct stat    bufX, *buf = NULL;		/* stat command  */
    const char *mbx_dir = NULL;
    const char ** inf = NULL;
    const char * mailhome_val = NULL;
    enum syscall_status  r1;
    
    if (in_mail)
	*in_mail = in_none;

    /* if filename is null or is of zero length */
    if((filename == NULL) || (*filename == '\0')) {
	DPRINT(Debug,8,(&Debug, 
			"get_folder_type=NO_NAME: no filename\n"));
	return(NO_NAME);
    }
    
    DPRINT(Debug,8,(&Debug,
		    "get_folder_type: filename=%s\n",filename));

    r1 = stat(filename, &bufX);
    switch (r1) {
    case syscall_error /* -1 */: {
	int err UNUSED_VAROK = errno;
	DPRINT(Debug,8,(&Debug,
			"get_folder_type: errno %d (%s) attempting to stat file %s\n", 
			err, strerror(err), filename));
    }
	break;
    case syscall_success /* 0 */:
	buf = &bufX;
	break;
    }

    /* give_dt_estr_as_str adds / to end */
    mailhome_val = give_dt_estr_as_str(&mailhome_dir_e,
				       "mailhome-dir",
				       NULL,NULL);    
    if (mailhome_val &&
	in_directory(buf,filename,mailhome_val)) {
	DPRINT(Debug,8,(&Debug,
			"get_folder_type=SPOOL\n"));
	return(SPOOL);
    }
    
    mbx_dir = give_dt_estr_as_str(&extra_mailbox_dir_e,"extra-mailbox-dir",
				  NULL,NULL);
    if (mbx_dir) {
	if (in_directory(buf,filename,mbx_dir)) {
	    DPRINT(Debug,8,(&Debug, 
			    "get_folder_type=SPOOL (is on %s)\n",
			    mbx_dir));
	    if (in_mail)
		*in_mail = in_extra_dir;
	    return(SPOOL);
	}
    }

    if (in_mail) {
	/* give_dt_estr_as_str adds / to end */
	const char * str = give_dt_estr_as_str(&folders_e,"maildir",NULL,NULL);

	if (str) {
	    if (in_directory(buf,filename,str)) {
		DPRINT(Debug,8,(&Debug, 
				"get_folder_type: On folders directory: %s\n",
				str));
		*in_mail = in_folders;
	    }
	}
	if (home[0]) {
	    if (in_directory(buf,filename,home)) {
		DPRINT(Debug,8,(&Debug,
				"get_folder_type: On home directory: %s\n",
				home));
		*in_mail = in_home;
	    }
	}
    }

    inf = give_dt_path_as_elems(&incomingfolders,"incomingfolders");
    if (inf) {
	int i;

	for (i = 0; inf[i]; i++) {
	    if (strcmp(filename,inf[i]) == 0) {

		DPRINT(Debug,8,(&Debug,
				"get_folder_type=SPOOL (on incomingfolders list)\n"));
		return(SPOOL);

	    }
	}
    }


    /* if file name == default mailbox, its a spool file also
     * even if its not in the spool directory. (SVR4)
     */


    {
	const char * default_val = give_dt_estr_as_str(&defaultfile_e,
						       "incoming-mailbox",
						       NULL, NULL); /* XXXX */
	
	if (default_val &&
	    strcmp(filename, default_val) == 0) {
	    DPRINT(Debug,8,(&Debug,
			"get_folder_type=SPOOL\n"));
	    return(SPOOL);
	}
    }

    
    if (buf) {
	if (buf->st_mode & 07000) { 
	    /* is 'SPOOL' file if special modes set */
	    DPRINT(Debug,8,(&Debug,
			    "get_folder_type=SPOOL; mode=%05o\n",
			    buf->st_mode));
	    return(SPOOL);
	}
	/* If owner do not have write permission, open it read-only */
	if (0 == (buf->st_mode & 00200)) {
	    DPRINT(Debug,8,(&Debug,
			    "get_folder_type=READ_ONLY; mode=%05o\n",
			    buf->st_mode));
	    return(READ_ONLY);
	}
    }
    
    /* Open read only if we have no write access ... 
     * This also detects (at least on Linux) cases where filesystem is
     * mounted read-only
     */
    if (0 != access(filename,WRITE_ACCESS)) {
	int err UNUSED_VAROK = errno;

	DPRINT(Debug,8,(&Debug, 
			"get_folder_type=READ_ONLY; no write access: %s (errno %d)\n",
			strerror(err),err));
	return(READ_ONLY);
    }

    if (treat_as_spooled) {
	DPRINT(Debug,8,(&Debug,
			"get_folder_type=SPOOL; treat_as_spooled=%d\n",
			treat_as_spooled));
	return(SPOOL);
    }

    DPRINT(Debug,8,(&Debug,
		    "get_folder_type=NON_SPOOL\n"));
    return(NON_SPOOL);
}

int in_directory(buf1,name,dir)
     struct stat *buf1;
     const char *name;
     const char * dir;
{
    const char * ptr = strrchr(name,'/');
    const char * X = name;
    int l1   =  strlen(dir);    

    if (buf1) {
	DPRINT(Debug,8,(&Debug,
			"in_directory: buf1=%p ",buf1));
    } else {
	DPRINT(Debug,8,(&Debug,
			"in_directory: buf1=NULL "));
    }
    DPRINT(Debug,8,(&Debug,
		    "name=%s dir=%s\n",name,dir));

    if (l1 > 0 && '/' == dir[l1-1]) {
	l1--;
	DPRINT(Debug,8,(&Debug,
			"in_directory: Last char / on dir=%s (using len=%d)\n",
			dir,l1));
    }

    if (ptr) {
	int L   =  ptr-X;

	X = ptr+1;
	
	if (l1 == L &&
	    0 == strncmp(dir,name,L)) {
	    DPRINT(Debug,8,(&Debug,
			    "in_directory=1   same base=%.*s\n",
			    L,dir));
	    return 1;
	}
    }

    if (buf1) {
	char * cmp = elm_message(FRM("%.*s/%s"),l1,dir,X);
	struct stat buf2;

	enum syscall_status  r = stat(cmp, &buf2);

	switch (r) {
	case syscall_error /* -1 */: {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,8,(&Debug,
			    "in_directory=0: %s (errno=%d) attempting to stat file %s\n", 
			    strerror(err), err, cmp));
	    free(cmp);
	    return 0;
	}
	case syscall_success /* 0 */:
	    if (buf1->st_ino == buf2.st_ino && buf1->st_dev == buf2.st_dev) {
		DPRINT(Debug,8,(&Debug,
				"in_directory=1  (same dev=%lu and ino=%lu as %s)\n",
				(unsigned long)buf1->st_dev,
				(unsigned long)buf1->st_ino,
				cmp)); 
		free(cmp);
		return 1;
		
	    }
	    break;
	}
	free(cmp);
    }
	
    DPRINT(Debug,8,(&Debug,
		    "in_directory=0\n"));
    return 0;
}

int same_file(name1,name2)
     const char *name1, *name2;
{
    struct stat buf1, buf2;
    enum syscall_status  r1,r2;
    
    if (0 == strcmp(name1,name2)) {
	DPRINT(Debug,8,(&Debug,
			"same_file(%s,%s)=TRUE\n",name1,name2)); 
	return TRUE;
    }

    r1 = stat(name1, &buf1);
    switch (r1) {
    case syscall_error /* -1 */: {
	int err UNUSED_VAROK = errno;
	DPRINT(Debug,8,(&Debug, 
			"same_file(%s,%s)=FALSE: %s (errno=%d) attempting to stat file %s\n", 
			name1,name2,strerror(err), err, name1));
	return FALSE;
    }
	break;
    case syscall_success /* 0 */:
	DPRINT(Debug,10,(&Debug, 
			 "same_file(%s,%s): stat %s succeed\n",
			 name1,name2,name1));
	break;
    }

    r2 = stat(name2, &buf2);
    switch (r2) {
    case syscall_error /* -1 */: {
	int err UNUSED_VAROK = errno;
	DPRINT(Debug,8,(&Debug,
			"same_file(%s,%s)=FALSE: %s (errno=%d) attempting to stat file %s\n", 
			name1,name2,strerror(err), err, name2));
	return FALSE;
    }
    case syscall_success /* 0 */:
	DPRINT(Debug,10,(&Debug, 
			 "same_file(%s,%s): stat %s succeed\n",
			 name1,name2,name2));
	break;
    }
    
    if (syscall_success == r1 && syscall_success == r2) {
	if (buf1.st_ino == buf2.st_ino && buf1.st_dev == buf2.st_dev) {
	    DPRINT(Debug,8,(&Debug,
			    "same_file(%s,%s)=TRUE\n",name1,name2)); 
	    return TRUE;
	}
    }
    
    DPRINT(Debug,8,(&Debug, 
		    "same_file(%s,%s)=FALSE\n",name1,name2)); 
    return FALSE;
}

int close_folder(folder,mode,cd) 
     struct folder_info *folder;
     enum close_mode mode;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  on remote mailbox
			       */;
{
    int ret = 0;
    
    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"close_folder",
	      "Bad magic number (folder_info)",0);

    DPRINT(Debug,10,(&Debug,
		     "close_folder: folder=%p (%s), type=%p\n",
		     folder,folder->cur_folder_sys,folder -> folder_type));
    
    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"close_folder",
	      "Bad magic number (folder_type)",0);

    ret = folder -> folder_type->close_it(folder,mode,cd);

    if (folder -> p) {
	if (folder -> p->fh_folder != NULL)
	    fclose(folder -> p->fh_folder); 
	folder -> p->fh_folder = NULL;

	if (folder -> p->fh_temp != NULL)
	    fclose(folder -> p->fh_temp); 
	folder -> p->fh_temp = NULL;
    }

    DPRINT(Debug,10,(&Debug,
		     "close_folder=%d\n",
		     ret));
    
    return ret;
}

struct folder_info *mbx_new_folder() 
{
    struct folder_info *new_folder = 
	safe_zero_alloc(sizeof (struct folder_info));
        
    new_folder -> p = NULL;
    new_folder -> cur_folder_sys  = NULL;
    new_folder -> cur_folder_disp = NULL;
    new_folder -> cur_tempfolder[0] = '\0';
    new_folder -> mailfile_size     = 0;
    new_folder -> folder_type       = NULL;

    new_folder -> was_reconnected = 0;
    new_folder -> got_skipped_mail = 0;
    
    new_folder -> magic = FOLDER_INFO_magic;
    return new_folder;
}

struct folder_info *enter_new_folder(new_file, treat_as_spooled,
				     defaultfile_keyword, 
				     defaultfile_cmask) 
     const char *new_file;
     int treat_as_spooled;
     unsigned int defaultfile_keyword;
     unsigned int defaultfile_cmask;  
{
    struct folder_info *new_folder = NULL;

    DPRINT(Debug,10,(&Debug,
		     "enter_new_folder(%s%s)\n",
		     new_file ? new_file : "<NULL>",
		     treat_as_spooled ? ", treat as spooled" : ""));

    new_folder = mbx_new_folder();
    if (new_file) {
	new_folder -> cur_folder_sys  = safe_strdup(new_file);
	new_folder -> cur_folder_disp = new_string2(display_charset,
						    cs2us(new_file));
    } else {
	new_folder -> cur_folder_sys  = safe_strdup("/dev/null");
	new_folder -> cur_folder_disp = new_string(display_charset);
    }

    /* determine type of new mailfile and calculate temp file name
       remote_folder_type calls folder's init_it routine if
       succees
    */

    /* Explicite disallow $MAIL to be interpreted as remote mailbox */
    if (0 != (defaultfile_cmask & FOLDER_is_local_file)) {

	DPRINT(Debug,10,(&Debug,
			 "enter_new_folder: FOLDER_is_local_file is set for value\n"));

	new_folder -> folder_type = get_folder_type(new_file,NULL,
						    treat_as_spooled);
	new_folder -> folder_type->init_it(new_folder);
    } else if (!remote_folder_type(new_folder)) {
	new_folder -> folder_type = get_folder_type(new_file,NULL,
						    treat_as_spooled);
	new_folder -> folder_type->init_it(new_folder);
    }

    /* HACK */
    if (new_folder->p) {	
	if (PRIVATE_DATA_magic != new_folder->p->magic) 
	    panic("MBX PANIC",__FILE__,__LINE__,"enter_new_folder",
		  "Bad magic number (private_data)",0);
	
	new_folder->p->flags1 = 0;

    }

#if 0
    DPRINT(Debug,1,(&Debug,
		    "New folder %s type %s temp file %s (%s)\n", 
		    current_folder->cur_folder_sys, 
		    (current_folder -> folder_type == SPOOL ? 
		     "spool" : "non-spool"),
		    (*current_folder-> cur_tempfolder ? 
		     current_folder -> cur_tempfolder : "none"), "newmbox"));
#endif

    DPRINT(Debug,10,(&Debug,
		     "enter_new_folder=%p (%s), type=%p (%s)\n",
		     new_folder,new_folder->cur_folder_sys,
		     new_folder->folder_type,
		     new_folder->folder_type->type_name));

    return new_folder;
}

int leave_old_folder(folder,mode,cd) 
     struct folder_info **folder;
     enum close_mode mode;
     struct cancel_data  * cd /* Allow cancelation (Ctrl-C)
				 on remote mailbox
			      */;
{
    int ret = 1;
    int r;
    
    if (FOLDER_INFO_magic != (*folder)->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"leave_old_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != (*folder)->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"leave_old_folder",
	      "Bad magic number (folder_type)",0);


    DPRINT(Debug,10,(&Debug,
		     "leave_old_folder: *folder=%p (%s), type=%p (%s)\n",
		     folder,
		     (*folder)->cur_folder_sys,
		     (*folder)->folder_type,
		     (*folder)->folder_type->type_name));

    r = close_folder(*folder, mode,cd);
    if (!r) {
	DPRINT(Debug,10,(&Debug,
			 "leave_old_folder: close_folder failed\n"));
	ret = 0;
    }
    
    r = (*folder)->folder_type->free_it(*folder,cd);
    if (!r) {
	DPRINT(Debug,10,(&Debug,
			 "leave_old_folder: ...->free_it failed\n"));
	ret = 0;
    }

    if ((*folder) -> cur_folder_sys) {
	free((*folder) -> cur_folder_sys);
	(*folder) -> cur_folder_sys = NULL;
    }
    if ((*folder) -> cur_folder_disp)
	free_string(&((*folder) -> cur_folder_disp));

    free(*folder);
    *folder = NULL;

    DPRINT(Debug,10,(&Debug,
		     "leave_old_folder=%d\n",
		     ret));
    return ret;
}

int lock_folder(direction, folder)
     enum lock_direction direction;
     struct folder_info *folder;
{   
    int ret;

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"lock_folder",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "lock_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));
    
    ret = folder -> folder_type->lock_it(direction,folder);

    DPRINT(Debug,10,(&Debug,
		     "lock_folder=%d\n",
		     ret));

    return ret;
}

int unlock_folder(interrupt, folder)
     int interrupt;
     struct folder_info *folder;
{   
    int ret;
    int flush_ret = 1;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"unlock_folder",
	      "Bad magic number (folder_info)",interrupt);

    /* If interrupt is set, avoid calling no-reentrant routines ... */
    
    /** Flush the mailfile buffer if necessary before removing the lock.
     **/

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"unlock_folder",
	      "Bad magic number (folder type)",interrupt);

    SIGDPRINT(Debug,10,(&Debug,
		     "unlock_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    if (!interrupt) 
	flush_ret = flush_folder(folder,NULL,NULL);

    if (!flush_ret) {
	SIGDPRINT(Debug,10,(&Debug,
			 "unlock_folder: flush failed\n"));
    }
    

    ret = folder -> folder_type->unlock_it(interrupt,folder);

    if (ret && !flush_ret) {
	SIGDPRINT(Debug,10,(&Debug,
			 "unlock_folder: returning error because flush failed (unlock succeed)\n"));
	ret = 0;
    }
    
    SIGDPRINT(Debug,10,(&Debug,
			"unlock_folder=%d\n",
			ret));
    return ret;
}


/* Return 1 on succeed */
enum sessionlock_status sessionlock_folder(folder,mode,err,reconnect_mode_ptr)
     struct folder_info *folder;
     enum sessionlock_mode mode;
     int *err /* errno */;
     RECONNECT_MODE         * reconnect_mode_ptr  /* Hack for SESSIONLOCK_TRUNCATE */;
 {
    enum sessionlock_status ret;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"sessionlock_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"sessionlock_folder",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "sessionlock_folder: folder=%p (%s), type=%p (%s), mode=%d",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name,mode));
    switch(mode) {
    case SESSIONLOCK_NORMAL:    DPRINT(Debug,10,(&Debug, " SESSIONLOCK_NORMAL")); break;
    case SESSIONLOCK_REOPEN:    DPRINT(Debug,10,(&Debug, " SESSIONLOCK_REOPEN")); break;
    case SESSIONLOCK_TRUNCATE:  DPRINT(Debug,10,(&Debug, " SESSIONLOCK_TRUNCATE")); break;
    case SESSIONLOCK_CHECK:     DPRINT(Debug,10,(&Debug, " SESSIONLOCK_CHECK")); break;
    case SESSIONLOCK_NONE:      DPRINT(Debug,10,(&Debug, " SESSIONLOCK_NONE")); break;
    case SESSIONLOCK_RECONNECT: DPRINT(Debug,10,(&Debug, " SESSIONLOCK_RECONNECT")); break;
    case SESSIONLOCK_NONE_CHECKNEW: DPRINT(Debug,10,(&Debug, " SESSIONLOCK_NONE_CHECKNEW")); break;
    }
    DPRINT(Debug,10,(&Debug, "\n"));

    if (err)
	*err = 0;

    if (reconnect_mode_ptr) {
	if (*reconnect_mode_ptr) {
	    if (RECONNECT_MODE_magic != (*reconnect_mode_ptr)->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"sessionlock_folder",
		      "Bad magic number (reconnect_mode)",0);
	}
    }

    /* HACK */

    if (folder->p) {	
	if (PRIVATE_DATA_magic != folder->p->magic) 
	    panic("MBX PANIC",__FILE__,__LINE__,"sessionlock_folder",
		  "Bad magic number (private_data)",0);

	clearit(folder->p->flags1,FLAG1_CHECKNEW);
    }

    switch (mode) {
    case SESSIONLOCK_NONE_CHECKNEW:
    
	if (folder->p)	
	    setit (folder->p->flags1,FLAG1_CHECKNEW);
	else {
	    DPRINT(Debug,10,(&Debug,
			     "sessionlock_folder: SESSIONLOCK_NONE_CHECKNEW unsupported\n"));
	}

	mode = SESSIONLOCK_NONE;
	break;
    case SESSIONLOCK_TRUNCATE:
	if (reconnect_mode_ptr && folder->was_reconnected) {
	    DPRINT(Debug,10,(&Debug,
			     "sessionlock_folder: SESSIONLOCK_TRUNCATE: ->was_reconnected uses reconnect_mode_ptr\n"));
	} else if (folder->was_reconnected) {
	    folder->was_reconnected = 0;
	    DPRINT(Debug,10,(&Debug,
			     "sessionlock_folder: SESSIONLOCK_TRUNCATE: clearing ->was_reconnected\n"));
	}
	break;
    case SESSIONLOCK_REOPEN:
	if (folder->was_reconnected) {
	    folder->was_reconnected = 0;
	    DPRINT(Debug,10,(&Debug,
			     "sessionlock_folder: SESSIONLOCK_REOPEN: clearing ->was_reconnected\n"));
	}
	break;
    default:
	break;
    }

    ret = folder->folder_type->sessionlock_it(folder,mode,err,reconnect_mode_ptr);

    DPRINT(Debug,10,(&Debug,
		     "sessionlock_folder=%d",
		     ret));
    switch (ret) {
    case sessionlock_fail: DPRINT(Debug,10,(&Debug, " sessionlock_fail"));
	break;
    case sessionlock_open: DPRINT(Debug,10,(&Debug, " sessionlock_open"));
	break;
    case sessionlock_reconnect:
	DPRINT(Debug,10,(&Debug, " sessionlock_reconnect")); 
	if (! folder->was_reconnected) {
	    folder->was_reconnected = 1;
	    DPRINT(Debug,10,(&Debug, " (setting ->was_reconnected)"));
	}
	break;
    }
    if (err) {
	DPRINT(Debug,10,(&Debug, " *err=%d",
			 *err));
	if (*err) {
	    DPRINT(Debug,10,(&Debug, " (%s)",
			     strerror(*err)));
	}	    
    }
    if (reconnect_mode_ptr && *reconnect_mode_ptr) {
	DPRINT(Debug,10,(&Debug," ; *reconnect_mode_ptr=%p",
			 *reconnect_mode_ptr));	 
    }    
    DPRINT(Debug,10,(&Debug, "\n"));
    
    return ret;
}

/* Flush may fail if canceled */
int flush_folder(folder,err_p,cd)
     struct folder_info *folder;
     int *err_p  /* errno value */;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  on remote mailbox
			       */;
{
    int ret = 0;
    
    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"flush_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"flush_folder",
	      "Bad magic number (folder_type)",0);

    if (err_p)
	*err_p = 0;
    
    DPRINT(Debug,10,(&Debug,
		     "flush_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    ret = folder->folder_type->flush_it(folder,err_p,cd);

    DPRINT(Debug,10,(&Debug,
		     "flush_folder=%d",
		     ret));
    if (err_p) {
	DPRINT(Debug,10,(&Debug,
			 "; *err_p=%d",
			 *err_p));
	if (*err_p) {
	    DPRINT(Debug,10,(&Debug, " (%s)",
			     strerror(*err_p)));
	}
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return ret;
}

int ferror_folder(folder,clean)
     struct folder_info *folder; 
     int clean;
{
    int ret;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"ferror_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"ferror_folder",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		"ferror_folder: folder=%p (%s), type=%p (%s)\n",
		folder,folder->cur_folder_sys,folder -> folder_type,
		folder->folder_type->type_name));

    ret = folder->folder_type->ferror_it(folder,clean);

    DPRINT(Debug,10,(&Debug,
		     "ferror_folder=%d\n",
		     ret));
    return ret;
}

/* Returns:   0  == fail
              1  == succeed
              2  == reconnected
*/
enum prepare_result prepare_read_folder(folder,mode,read_state_ptr,
					reconnect_mode_ptr)
     struct folder_info *folder;
     enum prepare_mode mode;
     READ_STATE * read_state_ptr;
     RECONNECT_MODE  * reconnect_mode_ptr;
{
    enum prepare_result status = prepare_fail;
    READ_STATE ptr = malloc_read_folder_state();
    RECONNECT_MODE reconnect_ptr = NULL;
        
    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"prepare_read_folder",
	      "Bad magic number (folder_info)",0);
    
    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"prepare_read_folder",
	      "Bad magic number (folder_type)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "prepare_read_folder: folder=%p (%s), type=%p (%s), mode=%d",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name, mode));

    switch (mode) {
    case PREPARE_NORMAL:    DPRINT(Debug,10,(&Debug," PREPARE_NORMAL"));    break;
    case PREPARE_NEW_ONLY:  DPRINT(Debug,10,(&Debug," PREPARE_NEW_ONLY"));  break;
    case PREPARE_NOLOCK:    DPRINT(Debug,10,(&Debug," PREPARE_NOLOCK"));    break;
    case PREPARE_NEW_ONLY_NOLOCK: DPRINT(Debug,10,(&Debug," PREPARE_NEW_ONLY_NOLOCK")); break;
    case PREPARE_ACCESS:    DPRINT(Debug,10,(&Debug," PREPARE_ACCESS"));    break;
    case PREPARE_RECONNECT: DPRINT(Debug,10,(&Debug," PREPARE_RECONNECT")); break;
    }
    DPRINT(Debug,10,(&Debug,"\n"));

    folder->folder_type->zero_rs_fields_it(ptr);
    ptr->mode = mode;
    
    if (reconnect_mode_ptr) {
	if (*reconnect_mode_ptr) {
	    if (RECONNECT_MODE_magic != (*reconnect_mode_ptr)->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"prepare_read_folder",
		      "Bad magic number (reconnect_mode)",0);
	    
	    DPRINT(Debug,10,(&Debug,
			     "prepare_read_folder: *reconnect_mode_ptr is set\n"));

	    reconnect_ptr = *reconnect_mode_ptr;
	    
	} else if (folder->was_reconnected) {
	    DPRINT(Debug,10,(&Debug,
			     "prepare_read_folder: ->was_reconnected  is set\n"));
	    reconnect_ptr = malloc_reconnect_mode();
		    
	} else  if (PREPARE_RECONNECT == mode) {
	    DPRINT(Debug,10,(&Debug,
			     "prepare_read_folder: PREPARE_RECONNECT given\n"));
	    reconnect_ptr = malloc_reconnect_mode();
	}
	
    } else if (PREPARE_RECONNECT == mode) {
	DPRINT(Debug,10,(&Debug,
			 "prepare_read_folder: PREPARE_RECONNECT requires reconnect_mode_ptr\n"));
	goto fail;
    }
               
    status = folder->folder_type->prepare_read_it(folder,mode,ptr,reconnect_ptr);

    if (reconnect_mode_ptr &&
	*reconnect_mode_ptr == reconnect_ptr && reconnect_ptr) {

	DPRINT(Debug,10,(&Debug,
			 "prepare_read_folder: reconnect mode *reconnect_mode_ptr=%p was set by caller\n",
			 *reconnect_mode_ptr));
	reconnect_ptr = NULL;
	
    } else if (prepare_reconnected != status && reconnect_ptr) {
	free_reconnect_mode(&reconnect_ptr);

	DPRINT(Debug,10,(&Debug,
			 "prepare_read_folder: reconnect mode discarded\n"));
	
    } else if (reconnect_mode_ptr) {
	*reconnect_mode_ptr = reconnect_ptr;
	reconnect_ptr = NULL;
    } else if (reconnect_ptr)
	free_reconnect_mode(&reconnect_ptr);

 fail:
    if (prepare_fail == status) {
	folder->folder_type->free_rs_fields_it(ptr);
	free_read_folder_state(&ptr);
    }
    *read_state_ptr = ptr;

    DPRINT(Debug,10,(&Debug,
		     "prepare_read_folder=%d",
		     status));  
    switch (status) {
    case prepare_fail:  DPRINT(Debug,10,(&Debug," prepare_fail")); break;
    case prepare_done:  DPRINT(Debug,10,(&Debug," prepare_done")); break;
    case prepare_reconnected: DPRINT(Debug,10,(&Debug," prepare_reconnected")); break;
    }
    if (*read_state_ptr) {
	DPRINT(Debug,10,(&Debug," ; *read_state_ptr=%p",
			 *read_state_ptr));	 
    }
    if (reconnect_mode_ptr && *reconnect_mode_ptr) {
	DPRINT(Debug,10,(&Debug," ; *reconnect_mode_ptr=%p",
			 *reconnect_mode_ptr));	 
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return status;
}

int end_read_folder(folder,read_state_ptr,reconnect_mode_ptr,silent)
     struct folder_info *folder;
     READ_STATE * read_state_ptr;
     RECONNECT_MODE  * reconnect_mode_ptr;
     int silent;
{
    int status;
    READ_STATE ptr = *read_state_ptr;
    RECONNECT_MODE reconnect_ptr = NULL;
    long   temp_size;
    long   total;
    int err = 0;

    long previous_mailfile_size;
    

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"end_read_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"end_read_folder",
	      "Bad magic number (folder_type)",0);

    if (RF_magic != ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"end_read_folder",
	      "Bad magic number (read state)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "end_read_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));
    if (reconnect_mode_ptr) 
	reconnect_ptr = *reconnect_mode_ptr;

    previous_mailfile_size = folder -> mailfile_size;
    
    status = folder->folder_type->end_read_it(folder,ptr,reconnect_ptr,silent);

    if (status) {
	DPRINT(Debug,10,(&Debug,
			 "end_read_folder: mode=%d",
			 ptr->mode));
	switch (ptr->mode) {
	case PREPARE_NORMAL:    DPRINT(Debug,10,(&Debug," PREPARE_NORMAL"));
	    if (folder->was_reconnected) {
		/* was reconnected is set also on initial opening */
		DPRINT(Debug,10,(&Debug," (clearing ->was_reconnected)"));
		folder->was_reconnected =0;
	    }
	    if (folder->got_skipped_mail) {
		DPRINT(Debug,10,(&Debug," (clearing ->got_skipped_mail)"));
		folder->got_skipped_mail = 0;
	    }	    
	    break;
	case PREPARE_NOLOCK:    DPRINT(Debug,10,(&Debug," PREPARE_NOLOCK"));
	    if (folder->was_reconnected) {
		/* was reconnected is set also on initial opening */
		DPRINT(Debug,10,(&Debug," (clearing ->was_reconnected)"));
		folder->was_reconnected =0;
	    }
	    if (folder->got_skipped_mail) {
		DPRINT(Debug,10,(&Debug," (clearing ->got_skipped_mail)"));
		folder->got_skipped_mail = 0;
	    }
	    break;
	case PREPARE_RECONNECT: DPRINT(Debug,10,(&Debug," PREPARE_RECONNECT"));
	    if (folder->was_reconnected) {
		/* was reconnected is set also on initial opening */
		DPRINT(Debug,10,(&Debug," (clearing ->was_reconnected)"));
		folder->was_reconnected =0;
	    }
	    if (folder->got_skipped_mail) {
		DPRINT(Debug,10,(&Debug," (clearing ->got_skipped_mail)"));
		folder->got_skipped_mail = 0;
	    }
	    break;	    

	case PREPARE_NEW_ONLY:  DPRINT(Debug,10,(&Debug," PREPARE_NEW_ONLY"));  break;
	case PREPARE_NEW_ONLY_NOLOCK: DPRINT(Debug,10,(&Debug," PREPARE_NEW_ONLY_NOLOCK")); break;
	case PREPARE_ACCESS:    DPRINT(Debug,10,(&Debug," PREPARE_ACCESS"));    break;
	}
	DPRINT(Debug,10,(&Debug,"\n"));
    }

    
    DPRINT(Debug,10,(&Debug,"end_read_folder: ----------------------------------------\n"));
    DPRINT(Debug,10,(&Debug,"                 fbytes:      %7ld\n",ptr->fbytes));

    temp_size = file_bytes(folder-> cur_tempfolder,&err);
    if (-1L == temp_size) {
	DPRINT(Debug,10,(&Debug,"       tempfolder error: %s (errno=%d)\n",
			 strerror(err),err));
    } else {
	DPRINT(Debug,10,(&Debug,"        tempfolder size:      %7ld\n",temp_size));

	if (temp_size != ptr->fbytes) {
	    DPRINT(Debug,10,(&Debug,"                  ERROR:      %7ld\n",
			     ptr->fbytes - temp_size));
	}
    }
    DPRINT(Debug,10,(&Debug,"        tempfolder name: %s\n",folder-> cur_tempfolder));

    DPRINT(Debug,10,(&Debug,"             skip_bytes:      %7ld\n",ptr->skip_bytes));

    total = ptr->fbytes + ptr->skip_bytes;
    if (ptr->skip_bytes) {
	DPRINT(Debug,10,(&Debug,"                  TOTAL:      %7ld\n",total));
    }

    DPRINT(Debug,10,(&Debug," previous mailfile_size:      %7ld\n",previous_mailfile_size));
    
    if (previous_mailfile_size != total) {
	DPRINT(Debug,10,(&Debug,"                  ERROR:      %7ld\n",
			 previous_mailfile_size - total));
    }

    if (previous_mailfile_size != folder -> mailfile_size) {    
	DPRINT(Debug,10,(&Debug,"    final mailfile_size:      %7ld\n",folder -> mailfile_size));
    }
    
    DPRINT(Debug,10,(&Debug,"             skip_count:      %7d\n",ptr->skip_count));        
    DPRINT(Debug,10,(&Debug,"end_read_folder: ----------------------------------------\n"));
    
    folder->folder_type->free_rs_fields_it(ptr);
    
    free_read_folder_state(&ptr);
    *read_state_ptr = ptr;
    reconnect_ptr = NULL;

    if (reconnect_mode_ptr && *reconnect_mode_ptr) {
	free_reconnect_mode(reconnect_mode_ptr);

	if (folder->was_reconnected) {
	    DPRINT(Debug,10,(&Debug,"Clearing ->was_reconnected\n"));
	    folder->was_reconnected =0;
	}
    }
    
    DPRINT(Debug,10,(&Debug,
		     "end_read_folder=%d\n",
		     status));

    return status;
}

void copy_skipcount_folder(folder,read_state_ptr,skipcount,skipbytes)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     int *skipcount;
     long *skipbytes;
{
    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_skipcount_folder",
	      "Bad magic number (folder_info)",0);

    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"copy_skipcount_folder",
	      "Bad magic number (read state)",0);

    *skipcount = read_state_ptr -> skip_count;
    *skipbytes = read_state_ptr -> skip_bytes;
}

/* Returns:  0 = End of folder
             1 = OK
	    -1 = format error
	    -2 =  not found (skipped)
*/
enum copy_env_status copy_envelope_folder(folder,read_state_ptr,entry,
					  parse_header,parse_body,
			 counter)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     struct header_rec *entry;
     parse_header_callback *parse_header;
     parse_body_callback *parse_body;
     struct counter_data *counter;
{
    enum copy_env_status status;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_envelope_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_envelope_folder",
	      "Bad magic number (folder type)",0);
    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"copy_envelope_folder",
	      "Bad magic number (read state)",0);

    DPRINT(Debug,10,(&Debug,
		     "copy_envelope_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));
    DPRINT(Debug,10,(&Debug,
		     "                     : entry=%p",entry));
    if (!entry->mbx_info) {
	DPRINT(Debug,10,(&Debug,
		    ", { mbx_info=NULL }"));
    } else {
	DPRINT(Debug,10,(&Debug,
			 ", { mbx_info=%p, type=%p }",
			 entry->mbx_info,entry->mbx_info->type_code));
    }    
    DPRINT(Debug,10,(&Debug,
		     ", hdr index #%d\n",		      
		     entry->index_number_X));
        
    read_state_ptr ->skipping     = sm_reading; /*  0 == reading, 
						    1 == skipping, 
						   -1 == end of message */
    entry->body_parsed            = 0;
    entry->mime_parsed            = 0;

    status = folder->folder_type->copy_envelope_it(folder,read_state_ptr,
						   entry,0);
    read_state_ptr -> fbytes_body = read_state_ptr -> fbytes;
    entry->mime_rec.begin_offset  = read_state_ptr -> fbytes;;

    DPRINT(Debug,12,(&Debug,
		     "copy_envelope_folder: {mime} begin_offset = %ld\n",
		     entry->mime_rec.begin_offset));

    entry->header_charset         = display_charset; 
    /* read_folder_headers() will re-set header_charset
       if it sees X-ELM-OSV header with parameter
       hdr-charset
    */
    
    entry->content_length   = -1; /* not found yet */
    /* read_folder_headers() will re-set content_length
       if it sees Content-Length header 
    */

    /* TEMPORARY? */
    if (status >= copy_env_ok) {
	int s;
	header_list_ptr parsed_headers;

	DPRINT(Debug,12,(&Debug,
			 "copy_envelope_folder: hdr %ld body %ld\n",
			 entry->offset,read_state_ptr -> fbytes_body));

	entry->binary                 = TRUE;
	/* read_folder_headers() -- actually unfold_header()
	   resets binary flags if LF is seend without CR
	*/

	parsed_headers = 
	    read_folder_headers(read_state_ptr,folder,entry);

	entry->mime_rec.offset = read_state_ptr -> fbytes;
	DPRINT(Debug,12,(&Debug,
			 "copy_envelope_folder: {mime} offset = %ld\n",
			 entry->mime_rec.offset));

	if (entry->binary) {
	    DPRINT(Debug,10,(&Debug,
			     "copy_envelope_folder: binary flag was preserved\n"));
	}

	if (entry->header_charset != display_charset) {
	    const char * MIME_name UNUSED_VAROK = 
		get_charset_MIME_name(entry->header_charset);

	    DPRINT(Debug,10,(&Debug,
			     "copy_envelope_folder: header charset was modified: %s\n",
			     MIME_name ? MIME_name : "<no MIME name>"
			     ));
	    
	}

	if (-1 != entry->content_length) {
	    DPRINT(Debug,10,(&Debug,
			     "copy_envelope_folder: content_length set: %ld\n",
			     entry->content_length
			     ));
	}

	s = parse_header(folder,read_state_ptr,entry,parsed_headers);

	if (s <= 0) {
	    DPRINT(Debug,10,(&Debug,
			     "copy_envelope_folder: parse_header callback failed (%d)\n",
			     s));
	    status = copy_env_format;
	} else {

	    switch (read_state_ptr ->skipping) {
	    case sm_EOM:
		DPRINT(Debug,10,(&Debug,
				 "copy_envelope_folder: End of message, body skipped\n"));
		break;
	    case sm_skipping:
		DPRINT(Debug,10,(&Debug,
				 "copy_envelope_folder: Body skipped\n"));
		break;
	    case sm_reading:
		s = parse_body(folder,read_state_ptr,entry,parsed_headers,counter);
		if (s <= 0) {
		    DPRINT(Debug,10,(&Debug,
				     "copy_envelope_folder: parse_body callback failed (%d)\n",
				     s));
		    status = copy_env_format;
		}
		break;
	    }
	    delete_headers(&parsed_headers);
	}	
    }

    if (sm_EOM != read_state_ptr ->skipping &&
	( status >= copy_env_ok || 
	  copy_env_no_data == status)) {
	enum copy_env_end_status s;
	DPRINT(Debug,10,(&Debug,
			 "copy_envelope_folder: copy_envelope_end_folder not called or succeed (skipping=%d, status=%d)\n",
			 read_state_ptr ->skipping, status));

	s = folder->folder_type->copy_envelope_end_it(folder,read_state_ptr,
						      NULL,NULL);
	if (s <= copy_env_end_mismatch) {
	    DPRINT(Debug,10,(&Debug,
			     "copy_envelope_folder: copy_envelope_end_it failed (%d)\n",
			     s));
	    status = copy_env_format;
	} else
	    read_state_ptr ->skipping = sm_EOM;
    }

    DPRINT(Debug,10,(&Debug,
		     "copy_envelope_folder=%d",
		     status));

    switch (status) {
    case copy_env_no_data: DPRINT(Debug,10,(&Debug," copy_env_no_data")); break;
    case copy_env_format:  DPRINT(Debug,10,(&Debug," copy_env_format"));  break;
    case copy_env_eof:     DPRINT(Debug,10,(&Debug," copy_env_eof"));     break;
    case copy_env_ok:      DPRINT(Debug,10,(&Debug," copy_env_ok"));      break;
    }

    DPRINT(Debug,10,(&Debug,"\n"));
    return status;
}

static int match_header_rec P_((struct header_rec  * entry,
				struct header_rec  * prev_entry));
static int match_header_rec(entry,prev_entry)
     struct header_rec  * entry;
     struct header_rec  * prev_entry;
{
    int ok = 1;
    int match = 0;
    int ret;
    
    DPRINT(Debug,12,(&Debug,
		     "match_header_rec: entry=%p (#%d), prev_entry=%p (#%d)\n",
		     entry,entry->index_number_X,
		     prev_entry,prev_entry->index_number_X));

    if (entry->lines == prev_entry->lines) {
	if (entry->lines > 0) {
	    DPRINT(Debug,12,(&Debug,
			     "match_header_rec: lines %d matches\n",
			     entry->lines));
	}
	
    } else if (-1 != entry->lines &&
	       -1 != prev_entry->lines) {   /* -1 == unknown */
	DPRINT(Debug,12,(&Debug,
			 "match_header_rec: entry lines %d != prev_entry lines %d\n",
			 entry->lines,prev_entry->lines));
	ok = 0;
    }

    if (entry->content_length == prev_entry->content_length) {
	if (entry->content_length > 0) {
	    DPRINT(Debug,12,(&Debug,
			     "match_header_rec: content_length %ld matches\n",
			     entry->content_length));
	}
    } else if (-1L != entry->content_length &&
	       -1L != prev_entry->content_length) {      /* -1 == unknown */
	DPRINT(Debug,12,(&Debug,
			 "match_header_rec: entry content_length %ld != prev_entry content_length %ld\n",
			 entry->content_length,prev_entry->content_length));
	ok = 0;
    }

    if (0 == strcmp(entry->env_from, prev_entry->env_from)) {
	if (entry->env_from[0]) {
	    DPRINT(Debug,12,(&Debug,
			     "match_header_rec: env_from %s matches\n",
			     entry->env_from));
	}

    } else {
	DPRINT(Debug,12,(&Debug,
			 "match_header_rec: entry env_from %s != prev_entry env_from %s\n",
			 entry->env_from,prev_entry->env_from));
	ok = 0;
    }

    if (entry->from && prev_entry->from) {
	struct string * entry_from =
	    addr_list_to_string(entry->from);
	
	if (addr_list_is_same(entry->from,prev_entry->from)) {
	    if (entry_from) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: from %S matches\n",
				 entry_from));
	    }
	    
	} else {
	    struct string * prev_entry_from =
		addr_list_to_string(prev_entry->from);

	    if (prev_entry_from) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: entry from %S != prev_entry from %S\n",
				 entry_from,prev_entry_from));
		
		free_string(&prev_entry_from);
	    }
	    
	    ok = 0;

	}

	if (entry_from)
	    free_string(&entry_from);
    } else {

	if (entry->from) {
	    struct string * entry_from  =
		addr_list_to_string(entry->from);

	    if (entry_from) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: only entry from %S exists\n",
				 entry_from));
		free_string (& entry_from);
	    }
	    ok = 0;

	}

	if (prev_entry->from) {
	    struct string * prev_entry_from  =
		addr_list_to_string(prev_entry->from);

	    if (prev_entry_from) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: only prev_entry from %S exists\n",
				 prev_entry_from));
		free_string (& prev_entry_from);
	    }
	    ok = 0;
	}
    }

    if (entry->to && prev_entry->to) {
	struct string * entry_to =
	    addr_list_to_string(entry->to);
	
	if (addr_list_is_same(entry->to,prev_entry->to)) {
	    if (entry_to) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: to %S matches\n",
				 entry_to));
	    }
	    
	} else {
	    struct string * prev_entry_to =
		addr_list_to_string(prev_entry->to);

	    if (prev_entry_to) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: entry to %S != prev_entry to %S\n",
				 entry_to,prev_entry_to));
		
		free_string(&prev_entry_to);
	    }
	    
	    ok = 0;

	}

	if (entry_to)
	    free_string(&entry_to);
    } else {

	if (entry->to) {
	    struct string * entry_to  =
		addr_list_to_string(entry->to);

	    if (entry_to) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: only entry to %S exists\n",
				 entry_to));
		free_string (& entry_to);
	    }
	    ok = 0;

	}

	if (prev_entry->to) {
	    struct string * prev_entry_to  =
		addr_list_to_string(prev_entry->to);

	    if (prev_entry_to) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: only prev_entry to %S exists\n",
				 prev_entry_to));
		free_string (& prev_entry_to);
	    }
	    ok = 0;
	}
    }

    if (entry->cc && prev_entry->cc) {
	struct string * entry_cc =
	    addr_list_to_string(entry->cc);
	
	if (addr_list_is_same(entry->cc,prev_entry->cc)) {
	    if (entry_cc) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: cc %S matches\n",
				 entry_cc));
	    }
	    
	} else {
	    struct string * prev_entry_cc =
		addr_list_to_string(prev_entry->cc);

	    if (prev_entry_cc) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: entry cc %S != prev_entry cc %S\n",
				 entry_cc,prev_entry_cc));
		
		free_string(&prev_entry_cc);
	    }
	    
	    ok = 0;

	}

	if (entry_cc)
	    free_string(&entry_cc);
    } else {

	if (entry->cc) {
	    struct string * entry_cc  =
		addr_list_to_string(entry->cc);

	    if (entry_cc) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: only entry cc %S exists\n",
				 entry_cc));
		free_string (& entry_cc);
	    }
	    ok = 0;

	}

	if (prev_entry->cc) {
	    struct string * prev_entry_cc  =
		addr_list_to_string(prev_entry->cc);

	    if (prev_entry_cc) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: only prev_entry cc %S exists\n",
				 prev_entry_cc));
		free_string (& prev_entry_cc);
	    }
	    ok = 0;
	}
    }

    if (entry->reply_to && prev_entry->reply_to) {
	struct string * entry_reply_to =
	    addr_list_to_string(entry->reply_to);
	
	if (addr_list_is_same(entry->reply_to,prev_entry->reply_to)) {
	    if (entry_reply_to) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: reply_to %S matches\n",
				 entry_reply_to));
	    }
	    
	} else {
	    struct string * prev_entry_reply_to =
		addr_list_to_string(prev_entry->reply_to);

	    if (prev_entry_reply_to) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: entry reply_to %S != prev_entry reply_to %S\n",
				 entry_reply_to,prev_entry_reply_to));
		
		free_string(&prev_entry_reply_to);
	    }
	    
	    ok = 0;

	}

	if (entry_reply_to)
	    free_string(&entry_reply_to); 
    } else {

	if (entry->reply_to) {
	    struct string * entry_reply_to  =
		addr_list_to_string(entry->reply_to);

	    if (entry_reply_to) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: only entry reply_to %S exists\n",
				 entry_reply_to));
		free_string (& entry_reply_to);
	    }
	    ok = 0;

	}

	if (prev_entry->reply_to) {
	    struct string * prev_entry_reply_to  =
		addr_list_to_string(prev_entry->reply_to);

	    if (prev_entry_reply_to) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: only prev_entry reply_to %S exists\n",
				 prev_entry_reply_to));
		free_string (& prev_entry_reply_to);
	    }
	    ok = 0;
	}
    }

    if (entry->message_id && prev_entry->message_id) {
	struct string * entry_message_id =
	    message_id_to_string(entry->message_id);

	if (same_message_id(entry->message_id,prev_entry->message_id,
			    0 /* do not ignore comment */)) {

	    if (entry_message_id) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: message_id %S matches\n",
				 entry_message_id));
	    }
	    
	    match = 1;   /* message id's are really used for match */
	    
	} else {
	    struct string * prev_entry_message_id =
		message_id_to_string(prev_entry->message_id);	    

	    if (prev_entry_message_id) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: entry message_id %S != prev_entry message_id %S\n",
				 entry_message_id,entry_message_id));
		
		free_string(& prev_entry_message_id);
	    }

	    ok = 0;
	}

	if (entry_message_id)
	    free_string(& entry_message_id);
    } else {
	if (entry->message_id) {
	    struct string * entry_message_id =
		message_id_to_string(entry->message_id);
	    
	    if (entry_message_id) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: only entry message_id %S exists\n",
				 entry_message_id));
		free_string(& entry_message_id);
	    }
	    ok = 0;
	}
	
	if (prev_entry->message_id) {
	    struct string * prev_entry_message_id =
		message_id_to_string(prev_entry->message_id);	    
	    
	    if (prev_entry_message_id) {
		DPRINT(Debug,12,(&Debug,
				 "match_header_rec: only prev_entry message_id %S exists\n",
				 prev_entry_message_id));
		free_string(& prev_entry_message_id);
	    }
	    ok = 0;
	}
    }

    if (entry->in_reply_to && prev_entry->in_reply_to) {
	 struct string * entry_in_reply_to =
	     references_to_string(entry->in_reply_to);

	 if (same_references(entry->in_reply_to,prev_entry->in_reply_to)) {

	     if (entry_in_reply_to) {
		 DPRINT(Debug,12,(&Debug,
				  "match_header_rec: in_reply_to %S matches\n",
				  entry_in_reply_to));
	     }
	    
	 } else {
	     struct string * prev_entry_in_reply_to =
		 references_to_string(prev_entry->in_reply_to);

	     if (prev_entry_in_reply_to) {
		 
		 DPRINT(Debug,12,(&Debug,
				  "match_header_rec: entry in_reply_to %S != prev_entry in_reply_to %S\n",
				  entry_in_reply_to,prev_entry_in_reply_to));

		 free_string(& prev_entry_in_reply_to);
	     }
	     
	     ok = 0;
	 }
	     
	 if (entry_in_reply_to)
	     free_string(& entry_in_reply_to);

     }  else {

	 if (entry->in_reply_to) {
	     struct string * entry_in_reply_to =
		 references_to_string(entry->in_reply_to);

	     if (entry_in_reply_to) {
		 
		 DPRINT(Debug,12,(&Debug,
				  "match_header_rec: only entry in_reply_to %S exists\n",
				  entry_in_reply_to));
		 
		 free_string(& entry_in_reply_to);
	     }
	     
	     ok = 0;
	 }
	 
	 if (prev_entry->in_reply_to) {
	     struct string * prev_entry_in_reply_to =
		 references_to_string(prev_entry->in_reply_to);
	     
	     if (prev_entry_in_reply_to) {
		 
		 DPRINT(Debug,12,(&Debug,
				  "match_header_rec: only prev_entry in_reply_to %S exists\n",
				  prev_entry_in_reply_to));

		 free_string(& prev_entry_in_reply_to);
	     }	 
	     
	     ok = 0;
	 }
     }

    if (entry->references && prev_entry->references) {
	 struct string * entry_references =
	     references_to_string(entry->references);

	 if (same_references(entry->references,prev_entry->references)) {

	     if (entry_references) {
		 DPRINT(Debug,12,(&Debug,
				  "match_header_rec: references %S matches\n",
				  entry_references));
	     }
	    
	 } else {
	     struct string * prev_entry_references =
		 references_to_string(prev_entry->references);

	     if (prev_entry_references) {
		 
		 DPRINT(Debug,12,(&Debug,
				  "match_header_rec: entry references %S != prev_entry references %S\n",
				  entry_references,prev_entry_references));

		 free_string(& prev_entry_references);
	     }
	     
	     ok = 0;
	 }
	     
	 if (entry_references)
	     free_string(& entry_references);

     }  else {

	 if (entry->references) {
	     struct string * entry_references =
		 references_to_string(entry->references);

	     if (entry_references) {
		 
		 DPRINT(Debug,12,(&Debug,
				  "match_header_rec: only entry references %S exists\n",
				  entry_references));
		 
		 free_string(& entry_references);
	     }
	     
	     ok = 0;
	 }
	 
	 if (prev_entry->references) {
	     struct string * prev_entry_references =
		 references_to_string(prev_entry->references);
	     
	     if (prev_entry_references) {
		 
		 DPRINT(Debug,12,(&Debug,
				  "match_header_rec: only prev_entry references %S exists\n",
				  prev_entry_references));

		 free_string(& prev_entry_references);
	     }	 
	     
	     ok = 0;
	 }
    }
    
    if (0 == strcmp(entry->time_zone, prev_entry->time_zone)) {
	if (entry->time_zone[0]) {
	    DPRINT(Debug,12,(&Debug,
			     "match_header_rec: time_zone %s matches\n",
			     entry->time_zone));
	}

    } else {
	DPRINT(Debug,12,(&Debug,
			 "match_header_rec: entry time_zone %s != prev_entry time_zone %s\n",
			 entry->time_zone,prev_entry->time_zone));
	ok = 0;
    }

    if (entry->time_sent ==  prev_entry->time_sent) {

	if (entry->time_sent > 0 && entry->time_sent < time_MAX) {
	    char * s = ctime(&(entry->time_sent));

	    DPRINT(Debug,12,(&Debug,
			     "match_header_rec: time_sent %ld matches: %s",
			     (long)(entry->time_sent),
			     s ? s : "(no time)\n"));
		   
	}
    } else {
	DPRINT(Debug,12,(&Debug,
			 "match_header_rec: entry time_sent %ld != prev_entry time_sent %ld\n",
			 (long)(entry->time_sent),
			 (long)(prev_entry->time_sent)));
	ok = 0;

    }

    if (entry->tz_offset ==  prev_entry->tz_offset) {

	 DPRINT(Debug,12,(&Debug,
			  "match_header_rec: tz_offset %ld matches\n",
			  (long)(entry->tz_offset)));
		
    } else {
	DPRINT(Debug,12,(&Debug,
			 "match_header_rec: entry tz_offset %ld != prev_entry tz_offset %ld\n",
			 (long)(entry->tz_offset),
			 (long)(prev_entry->tz_offset)));
	ok = 0;
    }

    if (entry->subject && prev_entry->subject) {
#define FAIL_COMPARE -1000
	int r = string_cmp(entry->subject,prev_entry->subject,
			   FAIL_COMPARE);

	if (0 == r) {
	    DPRINT(Debug,12,(&Debug,
			     "match_header_rec: subject %S matches\n",
			     entry->subject));
	} else if (FAIL_COMPARE != r) {
	    DPRINT(Debug,12,(&Debug,
			     "match_header_rec: entry subject %S != prev_entry subject %S\n",
			     entry->subject,
			     prev_entry->subject));
	    
	    ok = 0;
	}
#undef  FAIL_COMPARE
    } else {
	if (entry->subject) {

	    DPRINT(Debug,12,(&Debug,
			     "match_header_rec: only entry subject %S exists\n",
			     entry->subject));
	    
	    ok = 0;
	}

	if (prev_entry->subject) {

	    DPRINT(Debug,12,(&Debug,
			     "match_header_rec: only prev_entry subject %S exists\n",
			     prev_entry->subject));
	    	    
	    ok = 0;
	}
    }

    ret = ok && match;

    DPRINT(Debug,12,(&Debug,
		     "match_header_rec=%d (%s), %s%s\n",
		     ret,
		     ret   ? "matched"         : "not found",
		     ok    ? "ok"              : "no match",
		     match ? ", message-id"    : ""));
    
    return ret;
}

struct previous_data * search_previous_header(folder,entry,
					      reconnect_mode_ptr,
					      previous_headers,
					      previous_count,
					      result_index)
     struct folder_info *folder;
     struct header_rec *entry;
     RECONNECT_MODE    reconnect_mode_ptr;
     struct previous_data * previous_headers;
     size_t                 previous_count;
     size_t                *result_index;
{
    struct previous_data * ret = NULL;
    size_t       i;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"search_previous_header",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"search_previous_header",
	      "Bad magic number (folder type)",0);

     if (RECONNECT_MODE_magic != reconnect_mode_ptr->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"search_previous_header",
		      "Bad magic number (reconnect_mode)",0);

    DPRINT(Debug,10,(&Debug,
		     "search_previous_header: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));
    DPRINT(Debug,10,(&Debug,
		     "                     : entry=%p",entry));
    if (!entry->mbx_info) {
	DPRINT(Debug,10,(&Debug,
		    ", { mbx_info=NULL }"));
    } else {
	DPRINT(Debug,10,(&Debug,
			 ", { mbx_info=%p, type=%p }",
			 entry->mbx_info,entry->mbx_info->type_code));
    }
    DPRINT(Debug,10,(&Debug,
		     ", hdr index #%d\n",		      
		     entry->index_number_X));
    DPRINT(Debug,10,(&Debug,
		     "                     : previous_headers=%p, previous_count=%lu, result_index=%p\n",
		     previous_headers,(unsigned long)previous_count,result_index));

     
     *result_index = 0L;

     for (i = 0; i < previous_count && !ret; i++) {
	 if ( -1 == previous_headers[i].found_index &&
	      previous_headers[i].rec) {
	     enum comp_prev_hdr_result r =
		 folder->folder_type->
		 comp_prev_hdr_it(folder,entry,
				  previous_headers[i].rec,
				  reconnect_mode_ptr);

	     switch (r) {
	     case comp_prev_hdr_miss:   break;
	     case comp_prev_hdr_pass:
		 if (!  match_header_rec(entry,
					 previous_headers[i].rec))
		     break;
		 /* FALL THROUGH */
	     case comp_prev_hdr_found:
		 *result_index = i;
		 ret = &( previous_headers[i]);
		 break;
	     }	       				       
	 }
     }
     

     if (ret) {
	  DPRINT(Debug,10,(&Debug,
			   "search_previous_header=%p: #%d, *result_index=%lu\n",
			   ret,
			   entry->index_number_X,
			   (unsigned long)*result_index
			   ));
     } else {
	  DPRINT(Debug,10,(&Debug,
			   "search_previous_header=NULL: #%d\n",
			   entry->index_number_X));	 
     }

    return ret;
}


static int cpprev_read_buffered_line P_((RECONNECT_MODE reconnect_mode_ptr,
					 struct FILE_rs *rs,
					 char **buffer, int *len));
static int cpprev_read_buffered_line(reconnect_mode_ptr,rs,
				     buffer,len)
     RECONNECT_MODE          reconnect_mode_ptr;
     struct FILE_rs        * rs;
     char                 ** buffer;
     int                   * len;
{
    long f = -1L;
    
    if (RECONNECT_MODE_magic != reconnect_mode_ptr->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"cpprev_read_buffered_line",
	      "Bad magic number (reconnect_mode)",0);

    if(rs->next_line) {
	*buffer = rs->next_line;
	*len    = rs->next_line_len;
	
    } else {
	f = ftell(reconnect_mode_ptr->tempfolder_fh);

	if (!mbx_read_line(reconnect_mode_ptr->tempfolder_fh,
			   buffer,len,
			   mbx_max_line_read(reconnect_mode_ptr -> tempfolder_size,f)))
	    return 0;

		       
	rs->next_line      = *buffer;
	rs->next_line_len  = *len;

    }

    DPRINT(Debug,60,(&Debug, 
		     "cpprev_read_buffered_line: "));
    if (f >= 0L) {
	DPRINT(Debug,60,(&Debug, "offset=%ld, ",
			 f));
    }

    DPRINT(Debug,60,(&Debug, "len=%d, buffer=",
		     *len));
    DEBUG_PRINT_BUFFER(Debug,60,*len,s2us(*buffer));
    if (*len < 1 || !(*buffer) || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,60,(&Debug, 
			 "\ncpprev_read_buffered_line: NO NEWLINE\n"));
    } 

    return 1;
}

static void cpprev_accept_buffered_line P_((RECONNECT_MODE reconnect_mode_ptr,
					   struct FILE_rs *rs,
					   READ_STATE read_state_ptr,
					   char **buffer, int *len));
static void cpprev_accept_buffered_line(reconnect_mode_ptr,rs,
				       read_state_ptr,buffer,len)
     RECONNECT_MODE reconnect_mode_ptr;
     struct FILE_rs *rs;
     READ_STATE read_state_ptr;
     char **buffer;
     int *len;
{
    if (RECONNECT_MODE_magic != reconnect_mode_ptr->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"cpprev_accept_buffered_line",
	      "Bad magic number (reconnect_mode)",0);
    
    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"cpprev_accept_buffered_line",
	      "Bad magic number (read state)",0);
    
    if (*buffer != rs->next_line)
       	panic("MBX PANIC",__FILE__,__LINE__,"cpprev_accept_buffered_line",
	      "Bad buffer!",0);

    if (!*buffer)
	return;

    if (*len < 1 || (*buffer)[*len -1] != '\n') {
    } else {
	read_state_ptr -> linecounter++;
    }

    rs->next_line     = NULL;
    read_state_ptr -> fbytes       += rs->next_line_len;
    rs->next_line_len = 0;

    free(*buffer);
    *buffer = NULL;
    *len    = 0;
}

#define CPPREV_RS_magic		0xFA07

/* Hook for copy_previous_mail() */
struct copy_previous_rs {
    unsigned short magic;     /* CPPREV_RS_magic */
    
    RECONNECT_MODE         reconnect_mode_ptr;
    struct FILE_rs         RS;

    long                   fbytes_body;
};


enum copy_env_status copy_previous_mail(folder,read_state_ptr,
					entry,parse_header,parse_body,
					counter,reconnect_mode_ptr,
					previous_header)
     struct folder_info *folder;
     READ_STATE          read_state_ptr;
     struct header_rec * entry;         /* May be replaced entry */
     parse_header_callback * parse_header;
     parse_body_callback   * parse_body;
     struct counter_data   * counter;
     RECONNECT_MODE         reconnect_mode_ptr;
     struct previous_data * previous_header;
{
    enum copy_env_status status = copy_env_no_data;
    
    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_previous_mail",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_previous_mail",
	      "Bad magic number (folder type)",0);

    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"copy_previous_mail",
	      "Bad magic number (read state)",0);
    
     if (RECONNECT_MODE_magic != reconnect_mode_ptr->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"copy_previous_mail",
		      "Bad magic number (reconnect_mode)",0);

    DPRINT(Debug,10,(&Debug,
		     "copy_previous_mail: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));
    DPRINT(Debug,10,(&Debug,
		     "                     : entry=%p",entry));
    if (!entry->mbx_info) {
	DPRINT(Debug,10,(&Debug,
		    ", { mbx_info=NULL }"));
    } else {
	DPRINT(Debug,10,(&Debug,
			 ", { mbx_info=%p, type=%p }",
			 entry->mbx_info,entry->mbx_info->type_code));
    }
    DPRINT(Debug,10,(&Debug,
		     ", hdr index #%d\n",		      
		     entry->index_number_X));
         
     read_state_ptr ->skipping     = sm_reading; /*  0 == reading, 
						     1 == skipping, 
						     -1 == end of message */
     entry->body_parsed            = 0;
     entry->mime_parsed            = 0;
     read_state_ptr -> linecounter   = 0;      /* Linecounter of current
						  message */

     
     if (previous_header->rec) {

	 if (previous_header->rec->body_parsed) {
	     struct copy_previous_rs  RSHOOK;
	     char * buffer = NULL;
	     int  len;
	     int r;
	     
	     /* bzero is defined hdrs/elm_defs.h */
	     bzero((void *)&RSHOOK,sizeof RSHOOK);
	     
	     RSHOOK.magic               = CPPREV_RS_magic;
	     RSHOOK.reconnect_mode_ptr  = reconnect_mode_ptr;
	     
	     zero_FILE_rs(& (RSHOOK.RS));
	     
	     RSHOOK.fbytes_body = previous_header->rec->offset;
	     
	     read_state_ptr -> reconnect_copy_previous = &RSHOOK;
	     
	     entry->status = previous_header->rec->status;
	     
	     entry->offset = read_state_ptr -> fbytes; 
	     /* Offset of current message */
	     
	     status = copy_env_eof;
	     
	     if (PREPARE_ACCESS == read_state_ptr->mode) 
		 panic("MBX PANIC",__FILE__,__LINE__,"copy_previous_mail",
		       "PREPARE_ACCESS not valid when copying previous mail",0);
	     
	     if (PRIVATE_DATA_magic != folder->p->magic) 
		 panic("MBX PANIC",__FILE__,__LINE__,"copy_previous_mail",
		       "Bad magic number (private_data)",0);

	     DPRINT(Debug,12,(&Debug,
			      "copy_previous_mail: entry offset = %ld (previous mail)\n",
			      previous_header->rec->offset));
	     
	     r = fseek(reconnect_mode_ptr->tempfolder_fh,
		       previous_header->rec->offset,
		       SEEK_SET);
	     if (-1 == r) {
		 int err = errno;
		 
		 lib_error(CATGETS(elm_msg_cat, ElmSet, 
				   ElmCouldntSeekBytesIntoFolder,
				   "\nCouldn't seek %ld bytes into folder.\n"),
			   previous_header->rec->offset);		
		 lib_error(FRM("** %s. **\n"), strerror(err));
		 
		 DPRINT(Debug,12,(&Debug,
				  "copy_previous_mail: Failed to seek %s to %ld: %s (errno=%d)\n",
				  reconnect_mode_ptr->tempfolder_name,
				  previous_header->rec->offset,
				  strerror(err),err));
		 status = copy_env_eof;
		 goto clean;
	     } else if (0 == r) {
		 DPRINT(Debug,12,(&Debug,
				  "copy_previous_mail: Seeked %s to %ld\n",
				  reconnect_mode_ptr->tempfolder_name,
				  previous_header->rec->offset));
	     }

	     entry->offset = read_state_ptr -> fbytes; /* Offset of current message */

	     DPRINT(Debug,12,(&Debug,
			      "copy_previous_mail: entry offset = %ld\n",
			      entry->offset));
	     
	     if (!cpprev_read_buffered_line(reconnect_mode_ptr,&(RSHOOK.RS),&buffer,&len)) {
		 status = copy_env_eof;
		 goto clean;
	     }
	     if (0 == len) {
		 status = copy_env_eof;
		 goto clean;
	     }
	     
	     if (real_from(buffer,entry)) {
		 long f;
		 header_list_ptr parsed_headers = NULL;
		 int s;

		 DPRINT(Debug,12,(&Debug, "copy_previous_mail: "));
		 if (entry->env_from[0]) {
		     DPRINT(Debug,12,(&Debug," env from %s",
				      entry->env_from));
		 } else {
		     DPRINT(Debug,12,(&Debug,", no env from"));
		 }
		 DPRINT(Debug,12,(&Debug, "\n"));
		 
		 if (!mbx_copy_line_to_temp(folder,buffer,len)) {
		     status = copy_env_eof;
		     goto clean;
		 }
		 
		 cpprev_accept_buffered_line(reconnect_mode_ptr,&(RSHOOK.RS),
					     read_state_ptr,&buffer,&len);
		 
		 f = ftell(reconnect_mode_ptr->tempfolder_fh);
		 
		 status = copy_env_ok;
		 read_state_ptr -> fbytes_body = read_state_ptr -> fbytes;
		 entry->mime_rec.begin_offset  = read_state_ptr -> fbytes;;
		 RSHOOK.fbytes_body         = f;
		 
		 DPRINT(Debug,12,(&Debug,
				  "copy_previous_mail: {mime} begin_offset = %ld\n",
				  entry->mime_rec.begin_offset));
		 DPRINT(Debug,12,(&Debug,
				  "copy_previous_mail: {mime} begin_offset = %ld (previous mail)\n",
				  RSHOOK.fbytes_body));
		 
		 entry->header_charset         = previous_header->rec->header_charset;
		 entry->content_length         = previous_header->rec->content_length;
		 
		 DPRINT(Debug,12,(&Debug,
				  "copy_previous_mail: hdr %ld body %ld content_length %ld%s\n",
				  entry->offset,read_state_ptr -> fbytes_body,
				  entry->content_length,
				  -1 == entry->content_length ? " (unset)" : ""));
		 
		 parsed_headers = 
		     read_folder_headers(read_state_ptr,folder,entry);
		 
		 entry->mime_rec.offset = read_state_ptr -> fbytes;
		 DPRINT(Debug,12,(&Debug,
				  "copy_previous_mail: {mime} offset = %ld\n",
				  entry->mime_rec.offset));
		 if (entry->binary) {
		     DPRINT(Debug,10,(&Debug,
				      "copy_previous_mail: have binary flag\n"));
		 }
		 
		 if (entry->header_charset != display_charset) {
		     const char * MIME_name UNUSED_VAROK = 
			 get_charset_MIME_name(entry->header_charset);
		     
		     DPRINT(Debug,10,(&Debug,
				      "copy_previous_mail: header charset is: %s\n",
				      MIME_name ? MIME_name : "<no MIME name>"
				      ));
		     
		 }
		 
		 if (-1 != entry->content_length) {
		     DPRINT(Debug,10,(&Debug,
				      "copy_previous_mail: content_length is: %ld\n",
				      entry->content_length
				      ));
		 }
		 
		 s = parse_header(folder,read_state_ptr,entry,parsed_headers);
		 
		 if (s <= 0) {
		     DPRINT(Debug,10,(&Debug,
				      "copy_previous_mail: parse_header callback failed (%d)\n",
				      s));
		     status = copy_env_format;
		 } else {
		     switch (read_state_ptr ->skipping) {
		     case sm_EOM:
			 DPRINT(Debug,10,(&Debug,
					  "copy_previous_mail: End of message, body skipped\n"));
			 break;
		     case sm_skipping:
			 DPRINT(Debug,10,(&Debug,
					  "copy_previous_mail: Body skipped\n"));
			 break;
		     case sm_reading:
			 s = parse_body(folder,read_state_ptr,entry,parsed_headers,counter);
			 if (s <= 0) {
			     DPRINT(Debug,10,(&Debug,
					      "copy_previous_mail: parse_body callback failed (%d)\n",
					      s));
			     status = copy_env_format;
			 }
			 break;
		     }		 		 
		 }
		 delete_headers(&parsed_headers);

		 if (sm_EOM != read_state_ptr ->skipping &&
		     ( status >= copy_env_ok || 
		       copy_env_no_data == status)) {
		     DPRINT(Debug,10,(&Debug,
				      "copy_previous_mail: copy_envelope_end_folder not called or succeed (skipping=%d, status=%d)\n",
				      read_state_ptr ->skipping, status));		 
		 }
		 
	     } else {
		 lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFolderCorrupt,
				   "Folder is corrupt!!  I can't read it!!"));
		 status = copy_env_format;
	     }
	     
	 clean:
	     if (buffer) {
		 if (buffer == RSHOOK.RS.next_line)
		     RSHOOK.RS.next_line = NULL;
		 free(buffer);
		 buffer = NULL;
	     }
	     
	     clear_FILE_rs(& (RSHOOK.RS));
	     
	     read_state_ptr -> reconnect_copy_previous = NULL;
	 } else {
	     DPRINT(Debug,10,(&Debug,
			      "copy_previous_mail: Previous mail was not parsed.\n"));
	     status = copy_env_no_data;
	 }

	 if (previous_header->rec->status_chgd) {
	     DPRINT(Debug,10,(&Debug,
			      "copy_previous_mail: message #%d: previous (#%d) was changed status",
			      entry->index_number_X,previous_header->rec->index_number_X
			      ));
	     
#define X(field,F,T) if (ison(previous_header->rec->field,F) && isoff(entry->field,F)) { \
			setit(entry->field,F);		\
			DPRINT(Debug,10,(&Debug,"; set %s",T)); \
			entry->status_chgd = 1;            \
		      }					\
		      else if (isoff(previous_header->rec->status,F) && ison(entry->status,F)) { \
			clearit(entry->status,F);		\
			DPRINT(Debug,10,(&Debug,"; clear %s",T)); \
			entry->status_chgd = 1;            \
		      }
				    
	     X(status,UNREAD,"UNREAD");
	     X(status,DELETED,"DELETED");
	     X(status,NEW,"NEW");
	     X(status,TAGGED,"TAGGED");
	     X(status,VISIBLE,"VISIBLE");
	     X(status,REPLIED_TO, "REPLIED_TO");
	     X(status1,S1_FLAGGED,"S1_FLAGGED");
	     
	     if (! entry->status_chgd) {
		 DPRINT(Debug,10,(&Debug, ", no changes to current"));
	     }
	     
	     DPRINT(Debug,10,(&Debug, "\n"));
#undef X
	 }

	 /* This usually does nothing */
	 
	 folder->folder_type->
	     update_prev_hdr_it(folder,entry,
				previous_header->rec,
				reconnect_mode_ptr,
				read_state_ptr);
			 
     }

     DPRINT(Debug,10,(&Debug,
		      "copy_previous_mail=%d",
		      status));
     switch (status) {
     case copy_env_no_data: DPRINT(Debug,10,(&Debug," copy_env_no_data")); break;
     case copy_env_format:  DPRINT(Debug,10,(&Debug," copy_env_format"));  break;
     case copy_env_eof:     DPRINT(Debug,10,(&Debug," copy_env_eof"));     break;
     case copy_env_ok:      DPRINT(Debug,10,(&Debug," copy_env_ok"));      break;
     }
     DPRINT(Debug,10,(&Debug,"\n"));
     
     return status;
}

#if ANSI_C
extern parse_mime_callback NO_mime_parse;
#endif
int NO_mime_parse(folder,entry,fp)
     struct folder_info *folder;
     struct header_rec *entry;
     FILE *fp;
{
    return 1;  /* OK, but not set parsed flag */
}

int prepare_message_access(folder,entry,parse_header,
			   parse_body,counter,parse_mime)
     struct folder_info *folder;
     struct header_rec *entry;
     parse_header_callback *parse_header;
     parse_body_callback *parse_body;
     struct counter_data *counter;
     parse_mime_callback *parse_mime;
{
    int status = 1;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"prepare_message_access",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"prepare_message_access",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "prepare_message_access: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));
    DPRINT(Debug,10,(&Debug,
		     "                     : entry=%p",entry));
    if (!entry->mbx_info) {
	DPRINT(Debug,10,(&Debug,
		    ", { mbx_info=NULL }"));
    } else {
	DPRINT(Debug,10,(&Debug,
			 ", { mbx_info=%p, type=%p }",
			 entry->mbx_info,entry->mbx_info->type_code));
    }
    DPRINT(Debug,10,(&Debug,
		     ", hdr index #%d",		      
		     entry->index_number_X));
    if (entry->message_id) {
	struct string * entry_message_id =
	    message_id_to_string(entry->message_id);
	if (entry_message_id) {
	    DPRINT(Debug,10,(&Debug,", message-id=%S",
			     entry_message_id));
	    free_string(& entry_message_id);
	}
    }
    if (entry->size_lines_s) {
	DPRINT(Debug,10,(&Debug,", clearing size_lines_s=%S",
			 entry->size_lines_s));
	free_string(& entry->size_lines_s);
    }    
    DPRINT(Debug,10,(&Debug,"\n"));

    
    if (! entry->body_parsed) {
	READ_STATE read_state_ptr = malloc_read_folder_state();
	enum copy_env_status status1 = copy_env_eof;

	DPRINT(Debug,10,(&Debug,
			"prepare_message_access: Need read message\n",
			 folder,folder->cur_folder_sys));
	
	folder->folder_type->zero_rs_fields_it(read_state_ptr);

	if (PRIVATE_DATA_magic != folder->p->magic) 
	    panic("MBX PANIC",__FILE__,__LINE__,"prepare_message_access",
		  "Bad magic number (private_data)",0);

	if (!folder->p->fh_temp) {
	    panic("MBX PANIC",__FILE__,__LINE__,"prepare_message_access",
		  "Internal error -- no temp file",0);	   
	}
       
	read_state_ptr->mode = PREPARE_ACCESS;   
	status = folder->folder_type->prepare_read_it(folder,PREPARE_ACCESS,
						      read_state_ptr,NULL);
	if (!status) 
	    goto fail2;

	/*  0 == reading, 1 = skipping, -1 == end of message */
	read_state_ptr ->skipping     = sm_reading;

	status1 = folder->folder_type->copy_envelope_it(folder,read_state_ptr,
						       entry,1 /*force */);
	read_state_ptr -> fbytes_body = read_state_ptr -> fbytes;
	entry->mime_rec.begin_offset  = read_state_ptr -> fbytes;
	
	DPRINT(Debug,12,(&Debug,
			 "prepare_message_access: {mime} begin_offset = %ld\n",
			 entry->mime_rec.begin_offset));

	if (status1 >= copy_env_ok) {
	    header_list_ptr parsed_headers = NULL;

	    folder->got_skipped_mail = 1;
	    
	    DPRINT(Debug,12,(&Debug,
			     "prepare_message_access: hdr %ld body %ld\n",
			     entry->offset,read_state_ptr -> fbytes_body));

	    parsed_headers = 
		read_folder_headers(read_state_ptr,folder,entry);
	    
	    status = parse_header(folder,read_state_ptr,entry,parsed_headers);
	    if (status <= 0) {
		DPRINT(Debug,10,(&Debug,
				 "prepare_message_access: parse_header callback failed (%d)\n",
				 status));
		goto fail3;
	    }

	    status = parse_body(folder,read_state_ptr,entry,parsed_headers,counter);
	    if (status <= 0) {
		DPRINT(Debug,10,(&Debug,
				 "prepare_message_access: parse_body callback failed (%d)\n",
				 status));
	    }
	fail3:
	    if (parsed_headers) {
		delete_headers(&parsed_headers);
	    }
	} else {
	    DPRINT(Debug,10,(&Debug,
			     "prepare_message_access: copy_envelope=%d\n",status1));
	    status = 0;
	}
    fail2:

	folder->folder_type->end_read_it(folder,read_state_ptr,NULL,1);	
	folder->folder_type->free_rs_fields_it(read_state_ptr);
	free_read_folder_state(&read_state_ptr);
		
	if (!status)
	    goto fail;
	
    }

    if (! entry->mime_parsed && parse_mime == NO_mime_parse) {
	DPRINT(Debug,10,(&Debug,
			"prepare_message_access: mime structure not needed\n"));

    } else if (! entry->mime_parsed) {
	FILE * F;

	DPRINT(Debug,10,(&Debug,
			 "prepare_message_access: Need parsing of mime structure (offset %ld)\n",
			 entry->offset
			 ));

	/* Offset of current message */
	F = folder->folder_type->xxx_to_fd(folder,entry->offset);

	if (!F) {
	    status = 0;
	    goto fail;
	}

	status = parse_mime(folder,entry,F);
	if (status <= 0) {
	    DPRINT(Debug,10,(&Debug,
			     "prepare_message_access: parse_mime callback failed (%d)\n",
			     status));
	}
    }

 fail:

    DPRINT(Debug,10,(&Debug,
		     "prepare_message_access=%d\n",status));

    return status;
 }

/* Return 1 on success, 0 on failure */

int folder_give_message_msize(folder,entry,ret_size)
     struct folder_info * folder;
     struct header_rec  * entry;
     unsigned long      * ret_size;
{
    int ret = 0;
    
    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"folder_give_message_msize",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"folder_give_message_msize",
	      "Bad magic number (folder_type)",0);

    if (ret_size)
	*ret_size = 0;

    DPRINT(Debug,10,(&Debug,
		     "folder_give_message_msize: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));
    DPRINT(Debug,10,(&Debug,
		     "                     : entry=%p",entry));
    if (!entry->mbx_info) {
	DPRINT(Debug,10,(&Debug,
		    ", { mbx_info=NULL }"));
    } else {
	DPRINT(Debug,10,(&Debug,
			 ", { mbx_info=%p, type=%p }",
			 entry->mbx_info,entry->mbx_info->type_code));
    }
    DPRINT(Debug,10,(&Debug,
		     ", hdr index #%d",		      
		     entry->index_number_X));
    if (entry->message_id) {
	struct string * entry_message_id =
	    message_id_to_string(entry->message_id);
	if (entry_message_id) {
	    DPRINT(Debug,10,(&Debug,", message-id=%S",
			     entry_message_id));
	    free_string(& entry_message_id);
	}
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    ret = folder->folder_type->
	mbx_give_message_msize_it(folder,entry,ret_size);

    
    DPRINT(Debug,10,(&Debug,
		     "folder_give_message_msize=%d",ret));
    if (ret_size) {
	DPRINT(Debug,10,(&Debug,"; *ret_size=%lu",*ret_size));
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return ret;
}

const char * is_forwarded_folder(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    const char *ret;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"is_forwarded_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"is_forwarded_folder",
	      "Bad magic number (folder_type)",0);
    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"is_forwarded_folder",
	      "Bad magic number (read state)",0);

    DPRINT(Debug,10,(&Debug,
		     "is_forwarded_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    ret=folder->folder_type->is_forwarded_it(folder,read_state_ptr);

    DPRINT(Debug,10,(&Debug,
		     "is_forwarded_folder=%s\n",
		     ret ? ret : "<NULL>"));
    return ret;
}

long copy_fbytes_folder(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_fvytes_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_fbytes_folder",
	      "Bad magic number (folder_type)",0);
    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"copy_fbytes_folder",
	      "Bad magic number (read state)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "copy_fbytes_folder=%ld: folder=%p (%s), type=%p (%s)\n",
		     read_state_ptr->fbytes,
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    return read_state_ptr->fbytes;
}



int copy_lines_folder(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_lines_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_lines_folder",
	      "Bad magic number (folder_type)",0);
    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"copy_lines_folder",
	      "Bad magic number (read state)",0);

    DPRINT(Debug,10,(&Debug,
		     "copy_lines_folder=%d: folder=%p (%s), type=%p (%s)\n",
		     read_state_ptr -> linecounter,
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));
    
    return read_state_ptr -> linecounter;
}

/* Gives header including continuation lines */
int copy_header_folder(folder,read_state_ptr,buffer,len)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
{
    int status;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_header_folder",
	      "Bad magic number (folder_info)",0);

    *buffer = NULL;
    *len = 0;

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_header_folder",
	      "Bad magic number (folder_type)",0);
    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"copy_header_folder",
	      "Bad magic number (read state)",0);

    DPRINT(Debug,10,(&Debug,
		     "copy_header_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    if (read_state_ptr -> reconnect_copy_previous) {
	RECONNECT_MODE         reconnect_mode_ptr;
	struct FILE_rs       * rs;

	char *buffer1 = NULL;
	int len1;

	DPRINT(Debug,50,(&Debug, 
			 "copy_header_folder: Using previous temp for copy_previous_mail()\n"));
	
	if (CPPREV_RS_magic !=
	    read_state_ptr -> reconnect_copy_previous -> magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"copy_header_folder",
		  "Bad magic number (copy_previous_rs)",0);

	reconnect_mode_ptr =
	    read_state_ptr -> reconnect_copy_previous -> reconnect_mode_ptr;
	rs = &(read_state_ptr -> reconnect_copy_previous -> RS);
	
	if (!cpprev_read_buffered_line(reconnect_mode_ptr,rs,
				       &buffer1,&len1)) {
	    status = 0;
	    goto clean;
	}

	if ((len1 == 1 && 0 == memcmp(buffer1,"\n",len1)) ||
	    (len1 == 2 && 0 == memcmp(buffer1,"\r\n",len1))) {
	    /* End of headers */
	    
	    if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
		status = 0;
		goto clean;
	    }

	    cpprev_accept_buffered_line(reconnect_mode_ptr,rs,
					read_state_ptr,&buffer1,&len1);
	    status = 0;
	    goto clean;
	}

	if (NULL == memchr(buffer1,':',len1) ||
	    (0 == memcmp(buffer1,"From ",5) && real_from(buffer1,NULL))) {
	    /* End of headers -- bad header */

	    
	    status = 0;
	    goto clean;
	}

	do {
	    append_buffer(buffer,len,buffer1,len1);

	    if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
		status = copy_env_eof;
		goto clean;
	    }
	    
	    cpprev_accept_buffered_line(reconnect_mode_ptr,rs,
					read_state_ptr,&buffer1,&len1);
	    
	    if (!cpprev_read_buffered_line(reconnect_mode_ptr,rs,
					   &buffer1,&len1)) {
		break;
	    }

	} while (len1 > 0 && whitespace(buffer1[0]));
	status = 1;
	
    clean:
	/* Not need to free() -- buffer1 is same than next_line -buffer */

	if (!status) {
	    if (*buffer) {
		free(*buffer);
		*buffer = NULL;
	    }
	}

	DPRINT(Debug,50,(&Debug, 
			 "copy_header_folder | len=%d, buffer=",
			 *len));
	DEBUG_PRINT_BUFFER(Debug,50,*len,s2us(*buffer));
	if (*len < 1 || !*buffer || (*buffer)[*len -1] != '\n') {
	    DPRINT(Debug,50,(&Debug, 
			     "\ncopy_header_folder <- NO NEWLINE\n"));
	}	

	
    } else {
	DPRINT(Debug,50,(&Debug, 
			 "copy_header_folder: Using folder\n"));
	
	status = folder->folder_type->copy_header_it(folder,read_state_ptr,
						     buffer,len);
    }
    
    read_state_ptr -> fbytes_body = read_state_ptr -> fbytes;
    
    DPRINT(Debug,10,(&Debug,
		     "copy_header_folder=%d\n",
		     status));
    return status;
}

int copy_body_folder(folder,read_state_ptr,buffer,len,content_remaining)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
     long *content_remaining;
{
    int status;
    *buffer = NULL;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_body_folder",
	      "Bad magic number (folder_info)",0);

    *len = 0;

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_body_folder",
	      "Bad magic number (folder_type)",0);
    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"copy_body_folder",
	      "Bad magic number (read state)",0);

    DPRINT(Debug,10,(&Debug,
		     "copy_body_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    if (*content_remaining == 0L) {
	/* If content remining has gone 0, let
	   let routine copy_envelope_end_folder() 
	   check end of message
	   -1 indicates no content-length given;
	*/
	DPRINT(Debug,10,(&Debug,
			 "copy_body_folder=0 (*content_remaining == 0)\n"));

	return 0;
    }

    if (read_state_ptr -> reconnect_copy_previous) {
	RECONNECT_MODE         reconnect_mode_ptr;
	struct FILE_rs       * rs;

	char *buffer1 = NULL;
	int len1;

	DPRINT(Debug,50,(&Debug, 
			 "copy_body_folder: Using previous temp for copy_previous_mail()\n"));

	reconnect_mode_ptr =
	    read_state_ptr -> reconnect_copy_previous -> reconnect_mode_ptr;
	rs = &(read_state_ptr -> reconnect_copy_previous -> RS);

	if (!cpprev_read_buffered_line(reconnect_mode_ptr,rs,
				       &buffer1,&len1)) {
	    status = 0;
	    goto clean;
	}

	if (*content_remaining < 1 &&
	    len1 > 5 && 'F' == buffer1[0] &&
	    0 == memcmp(buffer1,"From ",5) && real_from(buffer1,NULL)) {
	    /* End of body */
	    status = 0;
	    goto clean;	    
	}

	append_buffer(buffer,len,buffer1,len1);

	if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
	    status = 0;
	    goto clean;
	}
	adjust_content_remaining(content_remaining,len1);
	cpprev_accept_buffered_line(reconnect_mode_ptr,rs,
				    read_state_ptr,&buffer1,&len1);
	status = 1;
	
    clean:
	/* buffer1 is same then next_line -buffer -- not need free() */
	
	if (!status) {
	    if (*buffer) {
		free(*buffer);
		*buffer = NULL;
	    }
	}

	DPRINT(Debug,50,(&Debug, 
			 "copy_body_folder | remaining %ld, len=%d, buffer=",
			 *content_remaining,*len));
	DEBUG_PRINT_BUFFER(Debug,50,*len,s2us(*buffer));
	
	if (*len < 1 || !*buffer || (*buffer)[*len -1] != '\n') {
	    DPRINT(Debug,50,(&Debug, 
			     "\ncopy_body_folder <- NO NEWLINE\n"));
	}

	
    } else {
	DPRINT(Debug,50,(&Debug, 
			 "copy_body_folder: Using folder\n"));
	
	
	status=folder->folder_type->copy_body_it(folder,read_state_ptr,
					      buffer,len, 
					      content_remaining);

    }
    
    DPRINT(Debug,10,(&Debug,
		     "copy_body_folder=%d\n",
		     status));
    return status;
}

/* Returns 0 if not end of message (needs resync without content-length) */
enum copy_env_end_status  copy_envelope_end_folder(folder,read_state_ptr,
						   newbytes,newmails)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     long *newbytes;
     int  *newmails;
{
    enum copy_env_end_status  status;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_envelope_end",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_envelope_end_folder",
	      "Bad magic number (folder_type)",0);
    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"copy_envelope_end_folder",
	      "Bad magic number (read state)",0);

    if (newbytes)
	*newbytes = 0;
    if (newmails)
	*newmails = 0;
    
    DPRINT(Debug,10,(&Debug,
		     "copy_envelope_end_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    if (read_state_ptr -> reconnect_copy_previous) {
	RECONNECT_MODE         reconnect_mode;
	struct FILE_rs       * rs;

	char *buffer1 = NULL;
	int len1;

	DPRINT(Debug,50,(&Debug, 
			 "copy_envelope_end_folder: Using previous temp for copy_previous_mail()\n"));
	
	if (CPPREV_RS_magic !=
	    read_state_ptr -> reconnect_copy_previous -> magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"copy_envelope_end_folder",
		  "Bad magic number (copy_previous_rs)",0);

	reconnect_mode =
	    read_state_ptr -> reconnect_copy_previous -> reconnect_mode_ptr;
	rs = &(read_state_ptr -> reconnect_copy_previous -> RS);


	if (!cpprev_read_buffered_line(reconnect_mode,rs,
				       &buffer1,&len1)) {
	    status = copy_env_end_match;
	    goto clean;
	}
	
	if ((len1 == 1 && 0 == memcmp(buffer1,"\n",len1)) ||
	    (len1 == 2 && 0 == memcmp(buffer1,"\r\n",len1))) {

	    /* Allow empty line after mail */
	    
	    if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
		status = copy_env_end_failure;
		goto clean;
	    }

	    cpprev_accept_buffered_line(reconnect_mode,rs,
					read_state_ptr,&buffer1,&len1);	    
	}

	/* Will be re-handled on copy_envelope_folder() */
	if (!cpprev_read_buffered_line(reconnect_mode,rs,
				       &buffer1,&len1)) {
	    status = copy_env_end_match;
	    goto clean;
	}
	
	if (real_from(buffer1,NULL))
	    status = copy_env_end_match;
	else
	    status = copy_env_end_mismatch;
	
	/* buffer1 is same then next_line -buffer -- not need free() */

    } else {
	DPRINT(Debug,50,(&Debug, 
			 "copy_envelope_end_folder: Using folder\n"));
	
	status=folder->folder_type->copy_envelope_end_it(folder,read_state_ptr,
							 newbytes,newmails);

    }

 clean:
    if (status > copy_env_end_mismatch) {
	/*  0 == reading, 1 = skipping, -1 == end of message */
	read_state_ptr ->skipping = sm_EOM;
    }

    DPRINT(Debug,10,(&Debug,
		     "copy_envelope_end_folder=%d",
		     status));
    switch (status) {
    case copy_env_end_failure:  DPRINT(Debug,10,(&Debug," copy_env_end_failure"));  break;
    case copy_env_end_mismatch: DPRINT(Debug,10,(&Debug," copy_env_end_mismatch")); break;
    case copy_env_end_match:    DPRINT(Debug,10,(&Debug," copy_env_end_match"));    break;
    case copy_env_end_newmail:  DPRINT(Debug,10,(&Debug," copy_env_end_newmail"));  break;
    }
    if (newbytes) {
	DPRINT(Debug,10,(&Debug," *newbytes=%ld",
			 *newbytes));
    }
    if (newmails) {
	DPRINT(Debug,10,(&Debug," *newmails=%d",
			 *newmails));    }
    
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return status;
}

int copy_envelope_reset_body(folder,read_state_ptr,content_remaining)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     long *content_remaining;
{
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_envelope_reset_body",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_envelope_reset_body",
	      "Bad magic number (folder_type)",0);
    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"copy_envelope_reset_body",
	      "Bad magic number (read state)",0);

    DPRINT(Debug,10,(&Debug,
		     "copy_envelope_reset_body: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    if (*content_remaining < 0L) {
       	panic("MBX PANIC",__FILE__,__LINE__,"copy_envelope_reset_body",
	      "*content_remaining < 0",0);

    }
    *content_remaining = -1L;

    if (read_state_ptr -> reconnect_copy_previous) {

	RECONNECT_MODE         reconnect_mode;
	struct FILE_rs       * rs;

	long fbytes;
	int r;
	
	DPRINT(Debug,50,(&Debug, 
			 "copy_envelope_reset_body: Using previous temp for copy_previous_mail()\n"));
	
	if (CPPREV_RS_magic !=
	    read_state_ptr -> reconnect_copy_previous -> magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"copy_envelope_reset_body",
		  "Bad magic number (copy_previous_rs)",0);

	reconnect_mode =
	    read_state_ptr -> reconnect_copy_previous -> reconnect_mode_ptr;
	rs = &(read_state_ptr -> reconnect_copy_previous -> RS);
	fbytes = read_state_ptr -> reconnect_copy_previous -> fbytes_body;
	
	 if (!folder->p->fh_temp) {
	     DPRINT(Debug,11,(&Debug, 
			      "copy_envelope_reset_body -- NO temp file %s\n",
			      folder->cur_tempfolder));
	 } else {
	     int r1 = fseek(folder->p->fh_temp, read_state_ptr -> fbytes_body , 
			    SEEK_SET);
	     
	     if (r1 == -1) {
		 int err = errno;
		 
		 lib_error(CATGETS(elm_msg_cat, ElmSet, 
				   ElmCouldntSeekBytesIntoFolder,
				   "\nCouldn't seek %ld bytes into folder.\n"),
			   read_state_ptr -> fbytes_body);		
		 lib_error(FRM("** %s. **\n"), strerror(err));

		  DPRINT(Debug,12,(&Debug,
				   "copy_envelope_reset_body: Failed to seek %s to %ld: %s (errno=%d)\n",
				   folder->cur_tempfolder,
				   read_state_ptr -> fbytes_body,
				   strerror(err),
				   err));
				   
		 status = 0;
		 
		 goto clean;
	     } else if (0 == r1) {
		DPRINT(Debug,12,(&Debug,
				 "copy_envelope_reset_body:  Seeked %s to %ld\n",
				 folder->cur_tempfolder,
				 read_state_ptr -> fbytes_body));
	     }
	 }
	     
	 r = fseek(reconnect_mode->tempfolder_fh,fbytes,		    
		   SEEK_SET);
	 if (-1 == r) {
	     int err = errno;
	     
	     lib_error(CATGETS(elm_msg_cat, ElmSet, 
			       ElmCouldntSeekBytesIntoFolder,
			       "\nCouldn't seek %ld bytes into folder.\n"),
		       fbytes);		
	     lib_error(FRM("** %s. **\n"), strerror(err));
	     
	      DPRINT(Debug,12,(&Debug,
			       "copy_envelope_reset_body: Failed to seek %s to %ld: %s (errno=%d)\n",
			      reconnect_mode->tempfolder_name,
			      fbytes,
			      strerror(err),err));
	     status = 0;
	     
	     goto clean;
	 } else if (0 == r) {
	      DPRINT(Debug,12,(&Debug,
			       "copy_envelope_reset_body: Seeked %s to %ld\n",
			       reconnect_mode->tempfolder_name,
			       fbytes));
	      
	      status = 1;
	 }

	 clear_FILE_rs(rs);
	 
    } else {
	DPRINT(Debug,50,(&Debug, 
			 "copy_envelope_reset_body: Using folder\n"));
	
	
	status=folder->folder_type->copy_envelope_reset_it(folder,read_state_ptr);

    }

 clean:
    DPRINT(Debug,10,(&Debug,
		     "copy_envelope_reset_body=%d\n",
		     status));
    return status;
}

FILE * folder_to_fd(folder,offset)
     struct folder_info *folder;
     long offset;  
{
    FILE *ret;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"folder_to_fd",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"folder_to_fd",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "folder_to_fd: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));
	
    ret=folder->folder_type->xxx_to_fd(folder,offset);

    DPRINT(Debug,10,(&Debug,
		     "folder_to_fd=%s (%p)\n",
		     ret ? "NON-NULL" : "NULL",
		ret));
    return ret;
}

enum new_mail_stat new_mail_on_folder(folder,bytes,
				      new_mailcount,
				      err_p,cd)
     struct folder_info *folder;
     long *bytes;
     int  *new_mailcount /* -1 for local mailbox */;
     int *err_p  /* errno value */;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  check of remeote mailbox
			       */;
			 
{
    enum new_mail_stat ret;
    int was_canceled = 0;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"new_mail_on_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"new_mail_on_folder",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "new_mail_on_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    if (bytes)
	*bytes = 0;
    if (new_mailcount)
	*new_mailcount = 0;
    if (err_p)
	*err_p =0;
    
    if (folder->was_reconnected) {
	DPRINT(Debug,10,(&Debug,
			 "new_mail_on_folder: have ->was_reconnected, returning new_mail_reconnect\n"));

	ret = new_mail_reconnect;
    } else {
	ret=folder->folder_type->new_mail_on_it(folder,bytes,new_mailcount,
						err_p,cd);
    }

    was_canceled = (cd && is_canceled(cd));
    
    DPRINT(Debug,10,(&Debug,
		     "new_mail_on_folder=%d",
		     ret));

    switch (ret) {
    case failed_mbx_ign_new_mail:
	                        DPRINT(Debug,11,(&Debug," failed_mbx_ign_new_mail"));
	break;
    case new_mail_check_failed: DPRINT(Debug,11,(&Debug," new_mail_check_failed")); break;
    case no_new_mail:           DPRINT(Debug,10,(&Debug," no_new_mail")); break;
    case have_new_mail:         DPRINT(Debug,10,(&Debug," have_new_mail")); break;
    case new_mail_reconnect:    DPRINT(Debug,10,(&Debug," new_mail_reconnect")); break;
    }
    if (bytes) {
	DPRINT(Debug,10,(&Debug," *bytes=%ld",
			 *bytes));
    }
    if (new_mailcount) {
	DPRINT(Debug,10,(&Debug, " *new_mailcount=%d",
			 *new_mailcount));
    }
    if (err_p) {
	DPRINT(Debug,10,(&Debug, " *err_p=%d",
			 *err_p));
	if (*err_p) {
	    DPRINT(Debug,10,(&Debug, " (%s)",
			     strerror(*err_p)));
	}
    }
    if (was_canceled) {
	DPRINT(Debug,10,(&Debug, ", check canceled"));
    }    
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return ret;
}

int consider_remove_folder(folder)
     struct folder_info *folder; 
{    
    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"consider_remove_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"consider_remove_folder",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "consider_remove_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));
    
    if (!keep_empty_files) {
	if(folder->folder_type->consider_remove_it(folder)) {

	    int r = close_folder(folder,CLOSE_NORMAL,NULL);
	    
	    DPRINT(Debug,10,(&Debug,
			     "consider_remove_folder=%d%s\n",
			     r,
			     r ? "" : " (close_folder failed)"));
	    return r;
	}
    }
    DPRINT(Debug,10,(&Debug,
		     "consider_remove_folder=0 (keep_empty_files=%d)\n",
		     keep_empty_files));
    return 0;
}

int prepare_keep_folder(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE * keep_state_ptr;
{
    int status;
    KEEP_STATE ptr = NULL;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"prepare_keep_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"prepare_keep_folder",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "prepare_keep_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    malloc_keep_folder_state(&ptr);
    folder->folder_type->zero_ks_fields_it(ptr);

    status = folder->folder_type->prepare_keep_it(folder,ptr);
   
    if (!status) {
	folder->folder_type->free_ks_fields_it(ptr);
	free_keep_folder_state(&ptr);
    }

    *keep_state_ptr = ptr;

    DPRINT(Debug,10,(&Debug,
		     "prepare_keep_folder=%d\n",
		     status));
    return status;
}

void end_keep_folder_failure(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE * keep_state_ptr;
{
    KEEP_STATE ptr = *keep_state_ptr;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"end_keep_folder_failure",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"end_keep_folder_failure",
	      "Bad magic number (folder_type)",0);
    if (KS_magic != ptr->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"end_keep_folder_failure",
	      "Bad magic number (keep state)",0);

    DPRINT(Debug,10,(&Debug,
		     "end_keep_folder_failure: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));	


    folder->folder_type->free_ks_fields_it(ptr);
    free_keep_folder_state(&ptr);
    *keep_state_ptr = ptr;
}

int end_keep_folder(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE * keep_state_ptr;
{
    int status = 0;
    KEEP_STATE ptr = *keep_state_ptr;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"end_keep_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"end_keep_folder",
	      "Bad magic number (folder_type)",0);
    if (KS_magic != ptr->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"end_keep_folder",
	      "Bad magic number (keep state)",0);

    DPRINT(Debug,10,(&Debug,
		     "end_keep_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));	

    status = folder->folder_type->end_keep_it(folder,ptr);
    
    folder->folder_type->free_ks_fields_it(ptr);
    free_keep_folder_state(&ptr);
    *keep_state_ptr = ptr;

    DPRINT(Debug,10,(&Debug,
		     "end_keep_folder=%d\n",
		     status));
    return status;
}

int mark_keep_folder(folder,keep_state_ptr,entry,keep)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
     struct header_rec *entry;
     int keep;
{
    int ret = 0;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mark_keep_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mark_keep_folder",
	      "Bad magic number (folder_type)",0);
    if (KS_magic != keep_state_ptr->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mark_keep_folder",
	      "Bad magic number (keep state)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "mark_keep_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));
    DPRINT(Debug,10,(&Debug,
		     "                : entry=%p,keep=%d",entry,keep));
    if (!entry->mbx_info) {
	DPRINT(Debug,10,(&Debug,
			 ", { mbx_info=NULL }"));
    } else {
	DPRINT(Debug,10,(&Debug,
			 ", { mbx_info=%p, type=%p }",
			 entry->mbx_info,entry->mbx_info->type_code));
    }
    DPRINT(Debug,10,(&Debug,
		     ", hdr index #%d",		      
		     entry->index_number_X));
    if (entry->message_id) {
	struct string * entry_message_id =
	    message_id_to_string(entry->message_id);
	if (entry_message_id) {
	    DPRINT(Debug,10,(&Debug,", message-id=%S",
			     entry_message_id));
	    free_string(& entry_message_id);
	}
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    ret = folder->folder_type->mark_keep_it(folder,keep_state_ptr,entry,keep);

    DPRINT(Debug,10,(&Debug,
		     "mark_keep_folder=%d\n",ret));

    return ret;
}

const char * folder_type(folder)
     struct folder_info *folder;
{
    const char *ret;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"folder_type",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"folder_type",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "folder_type: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    ret=folder->folder_type->type(folder);

    DPRINT(Debug,10,(&Debug,
		     "folder_type=%s\n",
		     ret ? ret : "<NULL>"));
    return ret;
}

/* -1 if not available (local mail) */
int copy_num_messages_folder(folder) 
    struct folder_info *folder;
{
    int ret = -1;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_num_messages_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"copy_num_messages_folder",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "copy_num_messages_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    ret = folder->folder_type->mbx_give_message_count_it(folder);

    DPRINT(Debug,10,(&Debug,
		     "copy_num_messages_folder=%d\n",ret));
    
    return ret;
}

int get_folder_mode(folder) 
     struct folder_info *folder;
{
    int ret = 0;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"get_folder_mode",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"get_folder_mode",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "get_folder_mode: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));
    
    ret = folder->folder_type->get_it_mode(folder);
    
    DPRINT(Debug,10,(&Debug,
		     "get_folder_mode=%d",ret));
    if (ison(ret,FOLDER_MBOX))            { DPRINT(Debug,10,(&Debug," FOLDER_MBOX")); }
    if (ison(ret,FOLDER_RDONLY))          { DPRINT(Debug,10,(&Debug," FOLDER_RDONLY")); }
    if (ison(ret,FOLDER_FILE))            { DPRINT(Debug,10,(&Debug," FOLDER_FILE")); }
    if (ison(ret,FOLDER_DISCONNECTED))    { DPRINT(Debug,10,(&Debug," FOLDER_DISCONNECTED")); }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return ret;
}

/* Gives username@server handle */
const struct remote_server * give_folder_server(folder)
     struct folder_info *folder;
{

    const struct remote_server * ret = NULL;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"give_folder_server",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"give_folder_server",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "give_folder_server: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    ret = folder->folder_type->get_it_server(folder);

    DPRINT(Debug,10,(&Debug,
		     "give_folder_server=%p\n",ret));

    return ret;
}

/* Increments refcount */
struct mail_quota *have_folder_quota(folder)
     struct folder_info *folder;
{
    struct mail_quota *ret = NULL;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"have_folder_quota",
	      "Bad magic number (folder_info)",0);
    
    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"have_folder_quota",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "have_folder_quota: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    if (folder->p) {	
	if (PRIVATE_DATA_magic != folder->p->magic) 
	    panic("MBX PANIC",__FILE__,__LINE__,"have_folder_quota",
		  "Bad magic number (private_data)",0);

	if (folder->p->quota) {
	    DPRINT(Debug,11,(&Debug,
			     "have_folder_quota: quota already initialized for folder\n"));
	    ret = folder->p->quota;
	    inc_mail_quota_refcount(ret);
	}
    } else {
	DPRINT(Debug,11,(&Debug,
			 "have_folder_quota: No private data\n"));
    }

    if (!ret) {
	DPRINT(Debug,11,(&Debug,
			 "have_folder_quota: Need call ...->have_it_quota()\n"));
	ret = folder->folder_type->have_it_quota(folder);
    }
    
    DPRINT(Debug,10,(&Debug,"have_folder_quota="));
    if (ret) {
	DPRINT(Debug,10,(&Debug,"%p",ret));
    } else {
	DPRINT(Debug,10,(&Debug,"NULL"));
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return ret;
}

/* Increments refcount */
struct mail_quotaroot_list * give_folder_quotaroot_l(folder,cd,refresh)
     struct folder_info * folder;     
     struct cancel_data * cd /* Allow cancelation (Ctrl-C) on remote mailbox */;
     enum quota_mode      refresh;
{
    struct mail_quotaroot_list * ret = NULL;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"give_folder_quotaroot_l",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"give_folder_quotaroot_l",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "give_folder_quotaroot_l: folder=%p (%s), type=%p (%s)",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    DPRINT(Debug,10,(&Debug,
		     "; refresh=%d",refresh));
    switch (refresh) {
    case quotam_normal:     DPRINT(Debug,10,(&Debug," quotam_normal"));  break;
    case quotam_refresh:    DPRINT(Debug,10,(&Debug," quotam_refresh")); break;
    }
    DPRINT(Debug,10,(&Debug,"\n"));

    if (folder->p) {	
    	if (PRIVATE_DATA_magic != folder->p->magic) 
	    panic("MBX PANIC",__FILE__,__LINE__,"give_folder_quotaroot_l",
		  "Bad magic number (private_data)",0);
		
	if (folder->p->quota) {
	    DPRINT(Debug,11,(&Debug,
			     "give_folder_quotaroot_l: have folder quota\n"));
	    
	    ret = folder->folder_type->mbx_give_it_quotar_list(folder,folder->p->quota,cd,
							       refresh);
	    
	} else {
	    
	    struct mail_quota *quota_temp = NULL;
	    
	    DPRINT(Debug,11,(&Debug,
			     "give_folder_quotaroot_l: Need folder quota, calling  ...->have_it_quota()\n"));
	    
	    quota_temp = folder->folder_type->have_it_quota(folder);
	    if (quota_temp) {
		ret = folder->folder_type->mbx_give_it_quotar_list(folder,quota_temp,cd,
								   refresh);
		
		free_mail_quota(&quota_temp);
	    } else {
		DPRINT(Debug,11,(&Debug,
				 "give_folder_quotaroot_l: Folder quota not available\n"));
	    }
	}

	if (ret) {
	    quotaroot_list_set_refresh(ret,refresh);
	}

	
    } else {
	DPRINT(Debug,11,(&Debug,
			 "give_folder_quotaroot_l: No private data\n"));
    }

	
	
    DPRINT(Debug,10,(&Debug,"give_folder_quotaroot_l="));
    if (ret) {
	DPRINT(Debug,10,(&Debug,"%p",ret));
    } else {
	DPRINT(Debug,10,(&Debug,"NULL"));
    }
    DPRINT(Debug,10,(&Debug,"\n"));

    return ret;
}


int start_edit_folder(folder,buffer)
     struct folder_info *folder; 
     const char **buffer;
{
    int ret;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"start_edit_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"start_edit_folder",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "start_edit_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    ret=folder->folder_type->start_edit_it(folder,buffer);

    DPRINT(Debug,10,(&Debug,
		"start_edit_folder=%d\n",
		ret));
    return ret;
}

enum end_edit_fol_status  end_edit_folder(folder)
     struct folder_info *folder;
{
    enum end_edit_fol_status  ret = end_edit_none;

    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"end_edit_folder",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"end_edit_folder",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "end_edit_folder: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    ret=folder->folder_type->end_edit_it(folder);

    DPRINT(Debug,10,(&Debug,
		     "end_edit_folder=%d",
		     ret));
    switch (ret) {
    case end_edit_fail_abort:  DPRINT(Debug,10,(&Debug," end_edit_fail_abort")); break;
    case end_edit_none:        DPRINT(Debug,10,(&Debug," end_edit_none"));       break;
    case end_edit_open:        DPRINT(Debug,10,(&Debug," end_edit_open"));       break;
    case end_edit_reconnect:   DPRINT(Debug,10,(&Debug," end_edit_reconnect"));  break;
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return ret;
}

void write_folder_info(F,folder)
     FILE *F; 
     struct folder_info *folder;
{
    if (FOLDER_INFO_magic != folder->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"write_folder_info",
	      "Bad magic number (folder_info)",0);

    if (FOLDER_TYPE_magic != folder->folder_type->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"write_folder_info",
	      "Bad magic number (folder_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "write_folder_info: folder=%p (%s), type=%p (%s)\n",
		     folder,folder->cur_folder_sys,folder -> folder_type,
		     folder->folder_type->type_name));

    fprintf(F, "F%s\n",
	    (folder->folder_type == NON_SPOOL ? 
	     folder->cur_folder_sys : 
	     folder->cur_tempfolder));
}

void change_rec_mbx_info(entry, t)
     struct header_rec *entry;
     info_type_t t;
{
    DPRINT(Debug,14,(&Debug,
		     "change_rec_mbx_info: entry=%p, (new)t=%p\n",
		     entry,t));
    /* Make new mbx_info */
    if (entry->mbx_info) {
	DPRINT(Debug,14,(&Debug,
			 "                 : mbx_info=%p, type=%p\n",
			 entry->mbx_info,entry->mbx_info->type_code));	
	DPRINT(Debug,14,(&Debug,
			 "change_rec_mbx_info: resetting current mbx_info...\n"));
	if (INFO_TYPE_magic != entry->mbx_info->type_code->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"change_rec_mbx_info",
		  "Bad magic number (info_type)",0);
	
	entry->mbx_info->type_code->free_it(entry->mbx_info);
    } else {
	DPRINT(Debug,14,(&Debug,
			 "                 : mbx_info=NULL\n"));
    }

    if (INFO_TYPE_magic != t->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"change_rec_mbx_info",
	      "Bad magic number (info_type, t)",0);


    /* ? ? ? */
    entry->mbx_info = safe_realloc(entry->mbx_info, 
				   sizeof (struct mbx_hdr_info));
    bzero((void *)entry->mbx_info,sizeof (struct mbx_hdr_info));

    entry->mbx_info->type_code = t;
    entry->mbx_info->type_code->zero_it(entry->mbx_info);
}

void mbx_status_hook(hdr,buffer)
     struct header_rec *hdr; 
     char *buffer;
{
    /* buffer[0]     interim status of the message 
       buffer[1]     permanent attributes of the message
    */
    if (hdr->mbx_info) {

	if (INFO_TYPE_magic != hdr->mbx_info->type_code->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"mbx_status_hook",
		  "Bad magic number (info_type)",0);

	hdr->mbx_info->type_code->it_status(hdr->mbx_info,
					    buffer);

    }
}



void free_rec_mbx_info(entry)
     struct header_rec *entry;
{
    DPRINT(Debug,10,(&Debug,
		     "free_rec_mbx_info: entry=%p\n",
		     entry));

    if (!entry->mbx_info) {
	DPRINT(Debug,10,(&Debug,
			 "                 : mbx_info=NULL\n"));
	return;
    }
    if (entry->mbx_info) {
	DPRINT(Debug,10,(&Debug,
			 "                 : mbx_info=%p, type=%p\n",
			 entry->mbx_info,entry->mbx_info->type_code));

	if (INFO_TYPE_magic != entry->mbx_info->type_code->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"free_rec_mbx_info",
		  "Bad magic number (info_type)",0);

	entry->mbx_info->type_code->free_it(entry->mbx_info);
	free(entry->mbx_info);
	entry->mbx_info = NULL;
    }
}

void unfold_header(buffer,len,current_header)
     char *buffer; 
     int *len; 
     struct header_rec *current_header;
{
    char *c, *p;
    enum tag { t_NORMAL, t_CR, t_LF } tag = t_NORMAL;
    
    DPRINT(Debug,65,(&Debug,
		     "fold_header- len=%d,buffer=%.*s",*len,*len,buffer));
    if (*len < 1 || !(buffer) || (buffer)[*len -1] != '\n') {
	DPRINT(Debug,65,(&Debug,
			 "\nfold_header- NO NEWLINE\n"));
    } 

    /* Unfold and remove newline ... */
    for (c = buffer, p = buffer; c < buffer + *len; c++) {
	switch(*c) {
	case '\n':
	    if (tag != t_CR && current_header && current_header->binary) {
		current_header->binary = 0;
		DPRINT(Debug,12,(&Debug,
				 "-- Not a binary message\n"));
	    }			
	    tag = t_LF;
	    break;
	case '\r':
	    tag = t_CR;
	    break;
	case ' ':
	case '\t':
	    if (tag == t_LF) {
		*(p++) = ' ';
		tag = t_NORMAL;
		break;
	    }
	default:
	    *(p++) = *c;
	case '\0':
	    tag = t_NORMAL;
	}
    }
    *p = '\0';
    
    *len = p-buffer;
    DPRINT(Debug,65,(&Debug,
		     "fold_header: len=%d,buffer=%.*s\n",*len,*len,buffer));

}

/* result MALLOCED */
char * return_path_to_env_from_1(value)
     const char *value;
{
    char * temp = NULL;

    char ** tokens = rfc822_tokenize(value);

    remove_space_tokenized(tokens);

    if (tokens[0] && 0 == strcmp(tokens[0],"<")) {
	int start = 1,i;
	
	/* Strip source path */
	for (i = 1; tokens[i]; i++) {
	    if (0 == strcmp(":",tokens[i]))
		start = i+1;
	    if (0 == strcmp(">",tokens[i]))
		break;
	}

	if (!tokens[i]) {
	    DPRINT(Debug,20,(&Debug,
			     "Bad Return-Path: %s\n",value));
	} else {
	    int j;

	    temp = safe_strdup("");
		
	    if (0 != strcmp(tokens[1],"@"))
		start = 1;
	    else if (start > 1) {
		DPRINT(Debug,20,(&Debug,
				 "  (stripping source path)\n"));
	    }
	    
	    for (j = start; j < i; j++)
		temp = strmcat(temp,tokens[j]);
	}    
    } else {
	DPRINT(Debug,20,(&Debug,
			 "Bad Return-Path: %s\n",value));
    }
    free_rfc822tokenized(tokens);
    
    return temp;
}


void return_path_to_env_from(entry,value)
     struct header_rec *entry;
     const char *value;
{
    enum env_from_source_v ef = 
	give_dt_enumerate_as_int(&env_from_source);

    DPRINT(Debug,20,(&Debug,
		     "return_path_to_env_from- value=%s\n",value));
    DPRINT(Debug,20,(&Debug,
		     "env_from_source=%d\n",ef));

    /* env_from_source:      
       0 == forward-from,
       1 == from,
       2 == return-path
    */
    
    if (ef <= env_source_return_path) {
	char * temp = return_path_to_env_from_1(value);

	if (temp) {
	    strfcpy(entry->env_from, temp, sizeof(entry->env_from));
	    DPRINT(Debug,20,(&Debug,
			     "  user=%s\n", temp));
	    free(temp);
	}
    }    
}

/* ----------------------------------------------------------------------- */

struct private_data * malloc_mbx_private_data(lock_info)
     struct dt_flags_info  * lock_info;
{
    struct private_data *p = 
	safe_zero_alloc(sizeof (struct private_data));

    p->fh_temp     = NULL;
    p->fh_folder   = NULL;
    p->flags1      = 0;

    /* p->ostat */
    
    p->lock_info   = lock_info;
    p->magic       = PRIVATE_DATA_magic;
    p->a.dummy     = 0;
    p->quota       = NULL;

    return p;
}

void free_mbx_private_data(p)
     struct private_data **p;
{
    if (PRIVATE_DATA_magic != (*p)->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"free_mbx_private_data",
	      "Bad magic number (private_data)",0);

    if ((*p)->fh_temp) {
	fclose((*p)->fh_temp);
	(*p)->fh_temp = NULL;
    }
    
    if ((*p)->fh_folder) {
	fclose((*p)->fh_folder);	
	(*p)->fh_folder = NULL;
    }

    if ((*p)->quota)
	free_mail_quota(&  ((*p)->quota));
    free(*p);
    *p = NULL;
} 

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
