static char rcsid[] = "@(#)$Id: localmbx.c,v 2.79 2024/06/13 17:36:55 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.79 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************
 * Based on Elm 2.4 code ../src/newmbox.c, ../src/lock.c and ../src/string2.c. 
 * That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/


#include "def_mbox.h"
#include "mbxlocal_imp.h"
#include "s_me.h"
#include "s_elm.h"

#include "fbrowser.h"

/* For INT_MAX */
#include <limits.h>

#include "filelock.h"

DEBUG_VAR(Debug,__FILE__,"file");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}


/* Locking primites ------------------------------------------------------- */


#ifndef ANSI_C
extern int errno;
#endif

enum FLOCKING_status GrabRead_the_file(flock_fd,flags)
     int flock_fd;
     struct dt_flags_info *flags;
{

    int err = 0;
    enum FLOCKING_status ret;
    
    if (&folder_locking == flags) {
	DPRINT(Debug,13,(&Debug, "GrabRead_the_file: using folder-locking\n"));
    }
    if (&mailbox_locking == flags) {
	DPRINT(Debug,13,(&Debug, "GrabRead_the_file: using mailbox-locking\n"));
    }
    if (&mailhome_locking == flags) {
	DPRINT(Debug,13,(&Debug, "GrabRead_the_file: using mailhome-locking\n"));
    }

    ret = filelock_fd(flock_fd,FLOCKING_shared,
		      flags,NULL /* no filename */,FLOCKING_non_block,
		      &err);

    DPRINT(Debug,12,(&Debug,
		     "GrabRead_the_file=%d",ret));
    switch (ret) {
    case FLOCKING_FAIL:   DPRINT(Debug,12,(&Debug," FLOCKING_FAIL")); break;
    case FLOCKING_OK:     DPRINT(Debug,12,(&Debug," FLOCKING_OK"));   break;
    case FLOCKING_RETRY:  DPRINT(Debug,12,(&Debug," FLOCKING_RETRY")); break;
    case FLOCKING_UNSUPPORTED: DPRINT(Debug,12,(&Debug," FLOCKING_UNSUPPORTED")); break;
    }
    if (err) {
	DPRINT(Debug,12,(&Debug,", error %s (errno %d)",
			 strerror(err),err));
	errno = err;
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return ret;
        
}

enum FLOCKING_status Grab_the_file(flock_fd, flags)
     int flock_fd;
     struct dt_flags_info *flags;
{
    int err = 0;
    enum FLOCKING_status ret;

    if (&folder_locking == flags) {
	DPRINT(Debug,13,(&Debug, "Grab_the_file: using folder-locking\n"));
    }
    if (&mailbox_locking == flags) {
	DPRINT(Debug,13,(&Debug, "Grab_the_file: using mailbox-locking\n"));
    }
    if (&mailhome_locking == flags) {
	DPRINT(Debug,13,(&Debug, "Grab_the_file: using mailhome-locking\n"));
    }

    ret = filelock_fd(flock_fd,FLOCKING_exclusive,
		      flags,NULL /* no filename */,FLOCKING_non_block,
		      &err);


    DPRINT(Debug,12,(&Debug,
		     "Grab_the_file=%d",ret));
    switch (ret) {
    case FLOCKING_FAIL:   DPRINT(Debug,12,(&Debug," FLOCKING_FAIL")); break;
    case FLOCKING_OK:     DPRINT(Debug,12,(&Debug," FLOCKING_OK"));   break;
    case FLOCKING_RETRY:  DPRINT(Debug,12,(&Debug," FLOCKING_RETRY")); break;
    case FLOCKING_UNSUPPORTED: DPRINT(Debug,12,(&Debug," FLOCKING_UNSUPPORTED")); break;
    }
    if (err) {
	DPRINT(Debug,12,(&Debug,", error %s (errno %d)",
			 strerror(err),err));
	errno = err;
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return ret;
}

enum FLOCKING_status Release_the_file(flock_fd,flags)
    int flock_fd;
    struct dt_flags_info *flags;
{
    int err = 0;
    enum FLOCKING_status ret;
    
    if (&folder_locking == flags) {
	DPRINT(Debug,13,(&Debug, "Release_the_file: using folder-locking\n"));
    }
    if (&mailbox_locking == flags) {
	DPRINT(Debug,13,(&Debug, "Release_the_file: using mailbox-locking\n"));
    }
    if (&mailhome_locking == flags) {
	DPRINT(Debug,13,(&Debug, "Release_the_file: using mailhome-locking\n"));
    }

    ret = filelock_fd(flock_fd,FLOCKING_release,
		      flags,NULL /* no filename */,FLOCKING_non_block,
		      &err);
		      
    DPRINT(Debug,12,(&Debug,
		     "Release_the_file=%d",ret));
    switch (ret) {
    case FLOCKING_FAIL:   DPRINT(Debug,12,(&Debug," FLOCKING_FAIL")); break;
    case FLOCKING_OK:     DPRINT(Debug,12,(&Debug," FLOCKING_OK"));   break;
    case FLOCKING_RETRY:  DPRINT(Debug,12,(&Debug," FLOCKING_RETRY")); break;
    case FLOCKING_UNSUPPORTED: DPRINT(Debug,12,(&Debug," FLOCKING_UNSUPPORTED")); break;
    }
    if (err) {
	DPRINT(Debug,12,(&Debug,", error %s (errno %d)",
			 strerror(err),err));
	errno = err;
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return ret;
}

#if 0
/* XXXX BAD for flock() !!!  */
static enum FLOCKING_query_status {
    FLOCKING_bad     = -1,        /* Failed to get status */
    FLOCKING_no      = 0,         /* No lock (or not supported) */
    FLOCKING_lock_ro = 1,
    FLOCKING_lock    = 2
} GrabQuery_the_file P_((int flock_fd,struct dt_flags_info *flags));
static enum FLOCKING_query_status GrabQuery_the_file(flock_fd,flags)
     int flock_fd;
     struct dt_flags_info *flags
{
#ifdef COMPILE_FCNTL_LOCKING

    if (dt_flag_is_set(flags,lock_fcntl_flag)) {
	DPRINT(Debug,12,(&Debug, 
			 "GrabQuery_the_file: Use fcntl locking\n"));

	struct flock lock_info;
	
	lock_info.l_type = F_WRLCK;
	lock_info.l_whence = 0;
	lock_info.l_start = 0;
	lock_info.l_len = 0;
	
	if (fcntl(flock_fd, F_GETLK, &lock_info) != 0) {
	    int err = errno;
	    DPRINT(Debug,12,(&Debug,
			     "GrabQuery_the_file: fcntl: %s (errno %d)\n",
			     strerror(err),err));
	    return FLOCKING_bad;	
	}
	
	switch (lock_info.l_type) {
	case F_UNLCK:
	    
	    return FLOCKING_no;
	case F_WRLCK:
	    return FLOCKING_lock;
	    
	case F_RDLCK:
	    return FLOCKING_lock_ro;
	}
	
	DPRINT(Debug,12,(&Debug,
			 "GrabQuery_the_file: Bad lock type\n"));
	
	return FLOCKING_bad;	
    } 
#endif
    return FLOCKING_no;
}

#endif

/* ------------------------------------------------------------------------  */

/* LOCAL browser */

#ifdef DIROPS

#define LOCAL_BROWSER_magic	0xF50F

struct LOCAL_BROWSER {
    unsigned short            magic;     /* LOCAL_BROWSER_magic */

    int     dotmode;                     /* Include .dotfiles on listing */
    DIR *   handle;

    int max_namelen;                     /* Max length of file name */

    /* last read file for this directory */
    struct last_read_cache * this_last_read;
};


#define NAME_LIMIT 2048

static struct LOCAL_BROWSER * malloc_LOCAL_BROWSER P_((void));
static struct LOCAL_BROWSER * malloc_LOCAL_BROWSER()
{
    struct LOCAL_BROWSER * ret = safe_zero_alloc(sizeof (*ret));

    ret->magic            = LOCAL_BROWSER_magic;

    ret->dotmode          = 0;
    ret->handle           = NULL;

#ifdef _POSIX_NAME_MAX
    ret->max_namelen  = _POSIX_NAME_MAX;
#else
    ret->max_namelen      = 256;
#endif

    if (ret->max_namelen > NAME_LIMIT) {
	DPRINT(Debug,12,(&Debug,
			 "malloc_LOCAL_BROWSER: limiting max_namelen=%d (was %d)\n",
			 NAME_LIMIT,ret->max_namelen));

	ret->max_namelen = NAME_LIMIT;
    }


    ret->this_last_read  = NULL;
    
    return ret;
}

static void free_LOCAL_BROWSER P_((struct LOCAL_BROWSER **ptr));
static void free_LOCAL_BROWSER(ptr)
     struct LOCAL_BROWSER **ptr;
{
    if (LOCAL_BROWSER_magic != (*ptr)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_LOCAL_BROWSER",
	      "Bad magic type",0);

    if ((*ptr)->handle) {
	closedir((*ptr)->handle);
	(*ptr)->handle = NULL;
    }


    if ((*ptr)->this_last_read) {
	flush_last_read((*ptr)->this_last_read);

	/* Decrements refcount */
	free_last_read_cache(& ((*ptr)->this_last_read));
    }
	
    (*ptr)->magic   = 0;        /* Invalidate */

    free(*ptr);
    (*ptr)          = NULL;
}

/* ------------------------- */

static struct fbentry * malloc_fbentry P_((void));
static struct fbentry * malloc_fbentry()
{
    struct fbentry * ret = safe_malloc(sizeof (*ret));

    ret->magic           = FBENTRY_magic;

    ret->name            = NULL;
    ret->display_name    = NULL;

    ret->mode            = 0;
    ret->uid             = 0;
    ret->size            = 0;
    ret->mtime           = 0;

    return ret;
}


static void free_fbentry P_((struct fbentry **ptr));
static void free_fbentry(ptr)
     struct fbentry **ptr;
{
    if (FBENTRY_magic != (*ptr)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_fbentry",
	      "Bad magic type",0);

    if ((*ptr)->name)
	free((*ptr)->name);
    (*ptr)->name = NULL;

    if ((*ptr)->display_name)
	free_string(& ((*ptr)->display_name));
    
    (*ptr)->magic     = 0;    /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}

/* ------------------------- */

#define LOCAL_BR_entry_magic	0xF511

struct LOCAL_BROWSER_entry {
    unsigned short            magic;     /* LOCAL_BR_entry_magic */

    struct fbentry   * fbentry;
    struct string    * line_name;
    struct string    * comment1;

    char   * sessionlock_file;          /* Sessionlock file for this mailbox */
};

static struct LOCAL_BROWSER_entry * malloc_LOCAL_BROWSER_entry P_((void));
static struct LOCAL_BROWSER_entry * malloc_LOCAL_BROWSER_entry() 
{
    struct LOCAL_BROWSER_entry * ret = safe_zero_alloc(sizeof (*ret));

    ret->magic            = LOCAL_BR_entry_magic;

    ret->fbentry          = NULL;
    ret->line_name        = NULL;
    ret->comment1         = NULL;
    ret->sessionlock_file = NULL;
           
    return ret;
}

static void free_LOCAL_BROWSER_entry P_((struct LOCAL_BROWSER_entry **entry));
static void free_LOCAL_BROWSER_entry(entry)
     struct LOCAL_BROWSER_entry **entry;
{
    if (LOCAL_BR_entry_magic != (*entry)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_LOCAL_BROWSER_entry",
	      "Bad magic type",0);


    if ((*entry)->fbentry)
	free_fbentry(& ((*entry)->fbentry));
    
    if ((*entry)->line_name)
	free_string(& ((*entry)->line_name));

    if ((*entry)->comment1)
	free_string(& ((*entry)->comment1));

    if ((*entry)->sessionlock_file) {
	free((*entry)->sessionlock_file);
	(*entry)->sessionlock_file = NULL;
    }

    (*entry)->magic      = 0;  /* Invalidate */
    free(*entry);
    *entry = NULL;
}

/* ------------------------- */

static void browser_zero_local P_((struct folder_browser *dir));
static void browser_zero_local(dir)
     struct folder_browser *dir;
{
    DPRINT(Debug,11,(&Debug,"browser_zero_local: dir=%p\n", dir));

    dir->a.local_browser = malloc_LOCAL_BROWSER();

}

static void browser_free_local P_((struct folder_browser *dir));

static void browser_free_local(dir)
     struct folder_browser *dir;
{
    DPRINT(Debug,11,(&Debug,
		     "browser_free_local: dir=%p (%s)\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>"));

    free_LOCAL_BROWSER( & (dir->a.local_browser));
}


/* result is malloced */
char * cat_dir_entry(sys_dir,entryname)
     const char *sys_dir;
     const char *entryname;
{
    size_t l1 = 0;
    char * pathname = NULL;
    
    if (sys_dir) {
	l1 = strlen(sys_dir);
	pathname = safe_strdup(sys_dir);
	if (l1 > 0 && '/' != sys_dir[l1-1])
	    pathname = strmcat(pathname,"/");
    }
    
    pathname = strmcat(pathname,entryname);

    DPRINT(Debug,20,(&Debug,
		     "cat_dir_entry=%s; sys_dir=%s (len %zu), entryname=%s\n",
		     pathname,
		     sys_dir ? sys_dir : "<NULL>",l1,entryname));

    return pathname;
}

/* Result is malloced */
static char * give_sessionlock_fname P_((char *mbox,struct stat *mbox_stat));
				 

struct last_read_cache *browser_local_last_read(dir)
     struct folder_browser *dir;
{

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_local_last_read",
	      "Bad magic type (folder_browser)",0);

    if (&local_browser == dir->type &&
	dir->a.local_browser) {


	if (LOCAL_BROWSER_magic != dir->a.local_browser->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"browser_local_last_read",
		  "Bad magic type",0);

        return dir->a.local_browser->this_last_read;
    }
    
    return NULL;
}


static void browser_local_do_stat_2 P_((struct folder_browser *dir,
					struct name_vector *entry,
					int add_fbentry));
static void browser_local_do_stat_2(dir,entry,add_fbentry)
     struct folder_browser *dir;
     struct name_vector *entry;
     int add_fbentry;
{
    char * pathname;
    struct stat buf, *Xbuf = NULL;
    int is_mailbox = 0;

    char * entryname;
    int flags;
    int have_stat = 0;
    enum syscall_status  r;
    
#ifdef SYMLINK
    int       have_lbuf = 0;
    struct stat    lbuf;		/* lstat command  */
#endif

    int preserve_flags = entry->flags & BROWSER_HAVELOCKFILE;

    entryname = entry->sys_name;

    pathname =  cat_dir_entry(dir->sys_dir,entryname);
    
    DPRINT(Debug,20,(&Debug,
		     "browser_local_do_stat_2: %s -> %s\n", 
		     entryname,pathname));

    if (LOCAL_BROWSER_magic != dir->a.local_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_local_do_stat_2",
	      "Bad magic number (LOCAL_BROWSER)",0);
  
#ifdef SYMLINK
    r = lstat(pathname, &lbuf);

    switch(r) {
    case syscall_success /* 0 */: {
	
	have_lbuf = 1;
	
	if (
#ifdef S_ISLNK
	    S_ISLNK(lbuf.st_mode)
#else
	    (lbuf.st_mode & S_IFMT) == S_IFLNK
#endif
	    ) {
	    int    maxlen = dir->a.local_browser->max_namelen;
	    char * buffer = safe_malloc(maxlen+4);
	    int len;
	    
	    DPRINT(Debug,13,(&Debug,
			     "browser_local_do_stat_2: %s is symbolic link\n",pathname));


	    len = readlink(pathname,buffer,maxlen);
	    
	    if (-1 == len) {

		int err UNUSED_VAROK = errno;
		
		DPRINT(Debug,13,(&Debug,
				 "browser_local_do_stat_2: readlink %s: (errno=%d) %s\n",
				 pathname,err,strerror(err)));

	    } else if (len == maxlen) {
		DPRINT(Debug,13,(&Debug,
				 "browser_local_do_stat_2: Overflow? (maxlen=%d)\n",
				 maxlen));

		strcpy(buffer+maxlen,"...");
	    } else if (len > 0) {
		buffer[len] = '\0';
	    }
	    
	    /* Ignore empty symlinks */

	    if (len > 0) {

		struct string * link_name = new_string2(local_fs_charset,
							s2us(buffer));

		DPRINT(Debug,13,(&Debug,
				 "browser_local_do_stat_2: %s -> %s\n",
				 pathname,buffer));

		if (!entry->a.local_browser) 
		    entry->a.local_browser = 
			malloc_LOCAL_BROWSER_entry();
		else if (LOCAL_BR_entry_magic != entry->a.local_browser->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"browser_local_do_stat_2",
			  "Bad magic number on local_browser",0);

		if (entry->a.local_browser->line_name)
		    free_string(& (entry->a.local_browser->line_name));

		entry->a.local_browser->line_name = 
		    format_string(FRM("%S -> %S"),
				  entry->disp_name,
				  link_name);

		free_string(&link_name);
	    }

	    free(buffer); buffer = NULL;
	}

    }
	break;

    case syscall_error /* -1 */: {
	int err UNUSED_VAROK = errno;
	
	DPRINT(Debug,13,(&Debug,
			 "browser_local_do_stat_2: lstat %s: (errno=%d) %s\n",
			 pathname,err,strerror(err)));
	
	
    }
	break;
    }
#endif

    r = stat(pathname,&buf);

    switch(r) {
    case syscall_success /* 0 */:	
	have_stat = 1;
	break;

    case syscall_error /* -1 */: {
	int err UNUSED_VAROK = errno;
	DPRINT(Debug,13,(&Debug,
			 "browser_local_do_stat_2: stat %s: (errno=%d) %s\n",
			 pathname,err,strerror(err)));
	flags = BROWSER_NOFOLDER|BROWSER_NODIR;
	
#ifdef SYMLINK
	if (have_lbuf) {
	    buf = lbuf;
	    have_stat = 1;
	    
	    DPRINT(Debug,13,(&Debug,
			     "browser_local_do_stat_2: Using lstat result instead\n"));
	}
#endif	
    }
	break;
    }

    if (have_stat) {
	Xbuf = &buf;
	entry->mtime   = buf.st_mtime;

	
	if (add_fbentry) {

	    if (!entry->a.local_browser) 
		entry->a.local_browser = 
		    malloc_LOCAL_BROWSER_entry();
	    else if (LOCAL_BR_entry_magic != entry->a.local_browser->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"browser_local_do_stat_2",
		      "Bad magic number on local_browser",0);

	    if (!entry->a.local_browser->fbentry) { 
		struct fbentry *fb  = malloc_fbentry();

		entry->a.local_browser->fbentry = fb;


		fb->name = safe_strdup(entryname);
		fb->display_name = dup_string(entry->disp_name);   

		DPRINT(Debug,12,(&Debug," browser_local_do_stat_2:  name         =%s\n",
				 fb->name));
		DPRINT(Debug,12,(&Debug,"                           display_name =%S\n",
				 fb->display_name));
	    }

	}

	flags = local_stat_flags(pathname,&buf,
				 /* Not correct if this is symlink entry */
				 dir->a.local_browser->this_last_read);

	if (entry->a.local_browser) {


	    if (LOCAL_BR_entry_magic != entry->a.local_browser->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"browser_local_do_stat_2",
		      "Bad magic number on local_browser",0);


	    
	    if (entry->a.local_browser->fbentry) {

		struct fbentry *fb = entry->a.local_browser->fbentry;
		
		if (FBENTRY_magic != fb->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"browser_local_do_stat_2",
			  "Bad magic type",0);
				
		fb->mode  = buf.st_mode;
		fb->uid   = buf.st_uid;
		fb->size  = buf.st_size;
		fb->mtime = buf.st_mtime;
		
		DPRINT(Debug,13,(&Debug," browser_local_do_stat_2:  mode         =0%03o\n",
				 fb->mode));
		DPRINT(Debug,13,(&Debug,"                           uid          =%d\n",
				 fb->uid));
		DPRINT(Debug,13,(&Debug,"                           size         =%ld\n",
				 (long) fb->size));
		DPRINT(Debug,13,(&Debug,"                           mtime        =%ld\n",
				 (long) fb->mtime));
	    }
	    
	} 

	
    }


    {
	const char * mailhome_val = give_dt_estr_as_str(&mailhome_dir_e,
							"mailhome-dir",
							NULL,NULL);
	
	/* sessionlock filename  -- check that this is mailbox 
	   (not folder) 
	*/
	is_mailbox = mailhome_val &&
	    in_directory(Xbuf,pathname,mailhome_val);
    }

    if (! is_mailbox) {
	const char * default_val =
	    give_dt_estr_as_str(&defaultfile_e,
				"incoming-mailbox",
				NULL, NULL); /* XXXX */
	
	if (default_val) {
	    is_mailbox = 0 == strcmp(pathname, default_val);
	}	    
    }

    if (! is_mailbox) {
	const char * mbx_dir = give_dt_estr_as_str(&extra_mailbox_dir_e,
					     "extra-mailbox-dir",
				      NULL,NULL);

	if (mbx_dir) 
	    is_mailbox = in_directory(Xbuf,pathname,mbx_dir);	
    }
    
    if (! is_mailbox) {
	const char ** inf = give_dt_path_as_elems(&incomingfolders,
						  "incomingfolders");
	
	if (inf) {
	    int i;

	    for (i = 0; inf[i]; i++) {
		if (strcmp(pathname,inf[i]) == 0) {
		    is_mailbox = 1;
		    break;
		}
	    }
	}
    }

    if (! is_mailbox && have_stat)
	is_mailbox = (buf.st_mode & 07000);

    DPRINT(Debug,13,(&Debug,
		     "browser_local_do_stat_2: %s: is_mailbox=%d\n",
		     pathname,is_mailbox));
    if (is_mailbox) {
	if (!entry->a.local_browser) 
	    entry->a.local_browser = 
		malloc_LOCAL_BROWSER_entry();
	else if (LOCAL_BR_entry_magic != entry->a.local_browser->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"browser_local_do_stat_2",
		  "Bad magic number on local_browser",0);

	if (! entry->a.local_browser->sessionlock_file)
	    entry->a.local_browser->sessionlock_file =
		give_sessionlock_fname(pathname,Xbuf);

	if (entry->a.local_browser->sessionlock_file &&
	    0 == access(entry->a.local_browser->sessionlock_file, 
			ACCESS_EXISTS)) {
		
	    flags |= BROWSER_SESSIONLOCK;
		
	    DPRINT(Debug,13,(&Debug,
			     "browser_local_do_stat_2: Sesssionlock file %s exists.\n",
			     entry->a.local_browser->sessionlock_file));
	}       
    }

    flags |= preserve_flags;

    DPRINT(Debug,13,(&Debug,
		     "browser_local_do_stat_2: changing flags %X -> %X\n",
		     entry->flags,flags));

    entry->flags = flags;


    DPRINT(Debug,13, (&Debug,
		      "                        %S have flags:%s%s%s%s%s%s%s%s%s%s\n",
		      entry->disp_name,
		      flags & BROWSER_NODIR    ?   " NODIR":    "",
		      flags & BROWSER_NOFOLDER ?   " NOFOLDER": "",
		      flags & BROWSER_MARKED   ?   " MARKED":   "",
		      flags & BROWSER_HAVELOCKFILE  ?   " HAVELOCKFILE":   "",
		      flags & BROWSER_SESSIONLOCK   ?   " SESSIONLOCK":   "",
		      
		      flags & BROWSER_MAILFILE ?   " MAILFILE": "",
		      flags & BROWSER_SELECTED ?   " SELECTED": "",
		      flags & BROWSER_EXIST    ?   " EXIST"   : "",
		      flags & BROWSER_DIRPREFIX ?   " DIRPREFIX"   : "",
		      !flags                   ?   " none"    : ""));

    free(pathname);		    
}


static void browser_local_do_stat_1 P_((struct folder_browser *dir,
				      int idx, int add_fbentry));
static void browser_local_do_stat_1(dir,idx,add_fbentry)
     struct folder_browser *dir;
     int idx;
     int add_fbentry;
{
    DPRINT(Debug,13,(&Debug,
		     "browser_local_do_stat_1: dir=%p, idx=%d, add_fbentry=%d\n", 
		     dir,idx,add_fbentry));

    if (idx < 0 || idx >= dir->vector_len)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_local_do_stat_1",
	      "Bad index",0);

    browser_local_do_stat_2(dir,& (dir->vector[idx]), add_fbentry);
}

static void browser_local_do_stat P_((struct folder_browser *dir,
				      int idx));
static void browser_local_do_stat(dir,idx)
     struct folder_browser *dir;
     int idx;
{
    browser_local_do_stat_1(dir,idx,0);
}

static void browser_fill_local P_((struct folder_browser *dir));
static void browser_fill_local(dir)
     struct folder_browser *dir;
{
#if DIROPS == USE_DIRENT
    struct dirent * Xptr;
#endif /* DIROPS == USE_DIRENT */
#if DIROPS == USE_SYSDIR
    struct direct * Xptr;
#endif /* DIROPS == USE_SYSDIR */

    int added = 0;
    int skipped = 0;
    int lockskipped = 0;

    int L_marked = 0;
    int L_added  = 0;
    int L_skipped = 0;

    struct lockname_buffer {
	char * lockname;
	char * basename;

	struct lockname_buffer * next;
    } * lockname_buffer = NULL;

    DPRINT(Debug,13,(&Debug,"browser_fill_local: dir=%p\n", dir));
    
    clear_dir_vector(dir);

    if (LOCAL_BROWSER_magic != dir->a.local_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_fill_local",
	      "Bad magic type",0);
 
#ifdef _PC_NAME_MAX
    {
	int L2;

	errno = 0;  /* If the system does not have a limit for  the  requested  resource,  
		       -1  is  returned,  and  errno is unchanged.
		    */

	L2 = pathconf(dir->sys_dir,_PC_NAME_MAX);

	if (-1 == L2) {
	    int err = errno;

	    if (0 == err) {
		DPRINT(Debug,12,(&Debug,
				 "browser_fill_local: pathconf _PC_NAME_MAX: %s: No limit\n",
				 dir->sys_dir));
	    } else {
		DPRINT(Debug,12,(&Debug,
				 "browser_fill_local: pathconf _PC_NAME_MAX: %s: (errno=%d) %s\n",
				 dir->sys_dir,err,strerror(err)));

	    }
	} else if (L2 > 0) {
	    dir->a.local_browser->max_namelen = L2;
	    
	    if ((long) dir->a.local_browser->max_namelen == L2) {
		DPRINT(Debug,12,(&Debug,
				 "browser_fill_local: pathconf _PC_NAME_MAX: %s: setting max_namelen=%d\n",
				 dir->sys_dir,dir->a.local_browser->max_namelen));
	    } else {
		dir->a.local_browser->max_namelen = NAME_LIMIT;
		DPRINT(Debug,12,(&Debug,
				 "browser_fill_local: pathconf _PC_NAME_MAX: %s: setting max_namelen=%d (really %ld)\n",
				 
				 dir->sys_dir,dir->a.local_browser->max_namelen,L2));

	    }

	    if (dir->a.local_browser->max_namelen > NAME_LIMIT) {
		DPRINT(Debug,12,(&Debug,
				 "browser_fill_local: limiting max_namelen=%d (was %d)\n",
				 NAME_LIMIT,dir->a.local_browser->max_namelen));

		dir->a.local_browser->max_namelen = NAME_LIMIT;
	    }
	}
    }
#endif
       
    if (update_last_read(& (dir->a.local_browser->this_last_read),
			 dir->sys_dir)) {

	DPRINT(Debug,15,(&Debug,
			 "browser_fill_local: dir=%p this_last_read=%p for %s\n",
			 dir,
			 dir->a.local_browser->this_last_read,
			 dir->sys_dir));
			 
    } else {
	DPRINT(Debug,15,(&Debug,
			 "browser_fill_local: dir=%p No last read for %s\n",
			 dir,
			 dir->sys_dir));
    }
    
    rewinddir(dir->a.local_browser->handle);
    
    
   
    while (NULL != (Xptr = readdir(dir->a.local_browser->handle))) {
	char * lock_basename = NULL;

#if DIROPS == USE_DIRENT
	char * entryname = safe_strdup(Xptr->d_name);
#endif /* DIROPS == USE_DIRENT */
#if DIROPS == USE_SYSDIR
	char * entryname = safe_malloc(Xptr->d_namlen+1);
	
	strncpy(entryname,Xptr->d_name,Xptr->d_namlen);
	entryname[Xptr->d_namlen] = '\0';
#endif /* DIROPS == USE_SYSDIR */

	/* Returns static pointer */
	lock_basename = is_lockname(entryname);
	if (lock_basename) {
	    struct lockname_buffer * lbuffer =
		safe_zero_alloc(sizeof (struct lockname_buffer));

	    lbuffer->next      = lockname_buffer;
	    lbuffer->lockname  = entryname;    entryname = NULL;
	    lbuffer->basename  = safe_strdup(lock_basename);
	    lockname_buffer = lbuffer;
	    
	    lockskipped++;
	} else 	if ('.' != Xptr->d_name[0] ||
	    dir->a.local_browser->dotmode) {


	    if (dir->a.local_browser->this_last_read)
		flag_last_read(dir->a.local_browser->this_last_read,
			       dir->sys_dir,
			       entryname);
	    
	    /* WARNING: add_dir_vector does not allocate strings -- 
	     *          it just assign pointers!	    
	     */
	    add_dir_vector(dir,
                           entryname,new_string2(local_fs_charset,
						 s2us(entryname)),
			   BROWSER_NEEDSTAT);

	    /* Do not free 'entryname' --  pointer is stored by add_dir_vector 
	     */
	    entryname = NULL;

	    added++;
	} else
	    skipped++;


	if (entryname)
	    free(entryname);
    }

    while(lockname_buffer) {
	struct lockname_buffer * X = lockname_buffer;

	int found  = 0;
	int i;

	for (i = 0; i < dir -> vector_len; i++) {
	    if (0 == strcmp(dir->vector[i].sys_name,
			    X->basename)) {

		dir->vector[i].flags |= BROWSER_HAVELOCKFILE;

		found++;
	    }
	}

	if (!found) {

	    if ('.' != X->lockname[0] ||
		dir->a.local_browser->dotmode) {
	    
		if (dir->a.local_browser->this_last_read)
		    flag_last_read(dir->a.local_browser->this_last_read,
				   dir->sys_dir,
				   X->lockname);

		/* WARNING: add_dir_vector does not allocate strings -- 
		 *          it just assign pointers!	    
		 */
		add_dir_vector(dir,
			       X->lockname,new_string2(local_fs_charset,
						       s2us(X->lockname)),
			       BROWSER_NEEDSTAT);
		
		/* Do not free 'lockname' --  pointer is stored by add_dir_vector 
		 */
		X->lockname = NULL;
		
		L_added++;
	    } else
		L_skipped++;

	} else
	    L_marked++;

	if (X->lockname) {
	    free(X->lockname);
	    X->lockname = NULL;
	}

	free(X->basename);
	X->basename = NULL;

	free(X);

	lockname_buffer = lockname_buffer->next;
    }


    if (dir->a.local_browser->this_last_read) {
	purge_last_read(dir->a.local_browser->this_last_read,
			dir->sys_dir);
	flush_last_read(dir->a.local_browser->this_last_read);
    }

    DPRINT(Debug,13,(&Debug,
		     "browser_fill_local: added=%d skipped=%d lockskipped=%d (marked=%d added=%d skipped=%d)\n",
		     added,skipped,lockskipped,
		     L_marked,L_added,L_skipped));

}

static void names_from_listing P_((struct folder_browser *dir,
				   struct name_vector *item,
				   struct string ** Lstr,
				   char ** str));
static void names_from_listing(dir,item,Lstr,str)
     struct folder_browser *dir;
     struct name_vector    * item;
     struct string ** Lstr;
     char ** str;
{
    
    int l1 = strlen(dir->sys_dir);
    int add_slash = 0;
    
    if (l1 > 0 && '/' != dir->sys_dir[l1-1]) {
	add_slash = 1;
    }
        
    *str = safe_strdup(dir->sys_dir);
    *Lstr = dup_string(dir->dirname);

    if (add_slash) {
	*str = strmcat(*str,"/");
	fill_ascii_to_string(*Lstr,1,'/');	
    }

    *str = strmcat(*str,item->sys_name);
    append_string(Lstr,item->disp_name,0);

    DPRINT(Debug,14,(&Debug,
		     "names_from_listing: sys  | %s and %s => %s\n",
		     dir->sys_dir,item->sys_name,*str));
    DPRINT(Debug,14,(&Debug,
		     "names_from_listing: disp | %S and %S => %S\n",
		     dir->dirname,item->disp_name,*Lstr));
}

/* Returns name relative to directory of browser */
static struct string * browser_descend_local P_((struct folder_browser *dir,
						 const struct string *rel_name1));
static struct string * browser_descend_local(dir,rel_name1)
     struct folder_browser *dir;
     const struct string *rel_name1;
{
    int L, idx;
    static struct string * ret = NULL;
    struct string *rel_name = NULL;
    int is_ascii = 0;
    int restarting = 0;
    const char * rel_name_MIME_name = NULL;

    int last_idx_ok = -1;

    DPRINT(Debug,12,(&Debug,"browser_descend_local: dir=%p\n", dir));

    if (!rel_name1) {
	DPRINT(Debug,12,(&Debug,"browser_descend_local=NULL\n"));
	return NULL;
    }

    rel_name = ascify_string(rel_name1);
    DPRINT(Debug,12,(&Debug,"browser_descend_local: rel_name=%S\n",rel_name));

    L = string_len(rel_name);

    if (LOCAL_BROWSER_magic != dir->a.local_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_descend_local",
	      "Bad magic type",0);

    if ((rel_name_MIME_name = get_string_MIME_name(rel_name)) &&
	0 == istrcmp(rel_name_MIME_name,"US-ASCII")) {
	is_ascii++;

	DPRINT(Debug,12,(&Debug,
			 "browser_descend_local: relname is ASCII only...\n"));
    }

 restart:

    for (idx = 0; idx < L ; idx++) {
	uint16 code = give_unicode_from_string(rel_name,idx);
       
	if (0x002F /* '/' */ == code) {
	    int X1 = 0;
	    struct string * A1 = NULL;

	    if (!restarting &&
		       (local_fast_lookup || is_ascii)) {
		int idx2;
		DPRINT(Debug,12,(&Debug,
				 "browser_descend_local:%s%s: Going to last directory on path\n",
				 local_fast_lookup ? " local_fast_lookup" : "",
				 is_ascii ? " is_ascii" : ""));
		
		for (idx2 = idx+1; idx2 < L; idx2++) {
		    uint16 code2 = give_unicode_from_string(rel_name,idx2);
		    if (0x002F /* '/' */ == code2)
			idx = idx2;
		}
		DPRINT(Debug,12,(&Debug,
				 "browser_descend_local: last found idx=%d\n",idx));
	    } 
		
	    /* 1) First check is current path have same prefix ... */
	    A1 = clip_from_string(rel_name,&X1,idx+1);	    
	    if (dir->dirname) {
		int X2 = 0;
		struct string * A2 = clip_from_string(dir->dirname,&X2,idx+1); 

		if (X1 == X2 &&
		    0 == string_cmp(A1,A2,-99 /* Unknown indicator */ )) {

		    DPRINT(Debug,12,(&Debug,
				     "browser_descend_local: sep idx=%d -- so far ok\n",
				     idx));

		    last_idx_ok = idx;
		    free_string(&A1);
		    free_string(&A2);
		    continue;
		}
		
		free_string(&A2);
	    }

	    /* 2) Then check if current path have prefix of wanted */
	    if (!dir->dirname || 
		0 != string_cmp(A1,dir->dirname,
				-99 /* unknown indicator */)) {
		struct string * Lstr =  NULL;
		char * str           =  NULL;
		struct stat buf;

		DPRINT(Debug,12,(&Debug,
				 "browser_descend_local: sep idx=%d -- need change of directory\n",
				 idx));
		
		if (dir->dirname && 
		    string_len(dir->dirname) == last_idx_ok+1) {
		    int Z1 = last_idx_ok+1;

		    /* NOTE: A1 includes terminal /  --
		             however we do NOT want it when comparing
			     path component agaist directory listing
		    */

		    int  wanted_len = string_len(A1);

		    struct string * Z2;

		    int i;

		    if (wanted_len > 0  &&
			0x002F /* '/' */ == 
			give_unicode_from_string(A1,wanted_len-1)) {
			wanted_len--;
			wanted_len -= Z1;
		    } else {
			DPRINT(Debug,12,(&Debug,
					 "browser_descend_local: OOPS -- Not removing / from end of %S\n",
					 A1));
		    }

		    if (wanted_len < 1)
			goto no_way;

		    Z2 = clip_from_string(A1,&Z1,wanted_len);

		    /* Update cache ... */
		    browser_vector_len(dir);

		    for (i = 0; i < dir->vector_len; i++)
			if (0 == (string_cmp(Z2,
					     dir->vector[i].disp_name,
					     -99 /* Unknown indicator */)))
			    break;

		    if (i >= dir->vector_len) {
			DPRINT(Debug,1,(&Debug,
					"browser_descend_local: Name %S do NOT found from directory listing of %S\n",
					Z2,dir->dirname));
		    }
		    
		    free_string(&Z2);

		    if (i < dir->vector_len) {

			names_from_listing(dir, & dir->vector[i], &Lstr, &str);

			DPRINT(Debug,12,(&Debug,
					 "browser_descend_local: Using name %s (%S) from listing\n",
					 str,Lstr));

		    } else
			goto no_way;

		} else {
		no_way:
		    Lstr = convert_string(local_fs_charset,A1,0);
		    str  =  us2s(stream_from_string(Lstr,0,NULL));
		    DPRINT(Debug,12,(&Debug,
				     "browser_descend_local: Using name %s (%S)\n",
				     str,Lstr));
		}

		if (0 != stat(str,&buf) 
		    || 0 != access(str,READ_ACCESS)) {
		    int err UNUSED_VAROK = errno;

		    if (!restarting &&
			(local_fast_lookup || is_ascii)) {
			
			DPRINT(Debug,12,(&Debug,
					 "browser_descend_local: %s: %s -- restarting\n",
					 str, strerror(err)));
		 
			restarting ++;

			free_string(&A1);
			free_string(&Lstr);
			free(str);

			goto restart;
		    }

		    DPRINT(Debug,12,(&Debug,
				     "browser_descend_local: %s: %s -- bailing out\n",
				     str, strerror(err)));
		 
		fail_stat:
		    free_string(&A1);
		    free_string(&Lstr);
		    free(str);
		    break;
		} 

#ifdef S_ISDIR
		if (!S_ISDIR(buf.st_mode)) {

		    DPRINT(Debug,12,(&Debug,
				     "browser_descend_local: %s not a directory -- bailing out\n",
				     str));
		    
		    goto fail_stat;
		}		    
#endif

		if (dir->a.local_browser->handle) 
		    closedir(dir->a.local_browser->handle);
		dir->a.local_browser->handle = NULL;

		/* Make empty selection folder */
		clear_dir_vector(dir);

		if (dir->dirname)
		    free_string(&dir->dirname);
		dir->dirname = Lstr;
		Lstr = NULL;
		
		if (dir->sys_dir) {
		    free(dir->sys_dir);
		    dir->sys_dir = NULL;
		}
		dir->sys_dir = str;
		str = NULL;

		dir->a.local_browser->handle = opendir(dir->sys_dir);

		if (dir->a.local_browser->handle) {
		    clear_dir_vector(dir);
		    DPRINT(Debug,12,(&Debug,
				     "browser_descend_local: Delaying reading of directory\n"));
		    dir->vector_len = -1;
		} else {
		    DPRINT(Debug,12,(&Debug, 
				     "browser_descend_local: No dir listing for %s (%S)\n",
				     dir->sys_dir,dir->dirname));
		}
	    }

	    DPRINT(Debug,12,(&Debug,
			     "browser_descend_local: sep idx=%d -- ok\n",
			idx));
	    last_idx_ok = idx;		
	    free_string(&A1);
	}
    }

    if (dir->dirname &&
	last_idx_ok < string_len(dir->dirname)) {

	DPRINT(Debug,12,(&Debug,
			 "browser_descend_local:  last_idx_ok = %d < len of dirname %S\n",
			 last_idx_ok,
			 dir->dirname));
			 
	if (last_idx_ok >= 0) {	    
	    struct string * Lstr =  NULL;
	    unsigned char * str  =  NULL;
	    int X = 0;
	    struct string * A1   = NULL;

	    DPRINT(Debug,12,(&Debug,
			     "browser_descend_local:  Target directory is substring of current!\n"));

	    A1 = 
		clip_from_string(dir->dirname,&X,
				 last_idx_ok > 0 ? last_idx_ok : 1);
	    	    
	    Lstr = convert_string(local_fs_charset,A1,0);
	    str  =  stream_from_string(Lstr,0,NULL);
	    
	    /* We assume that directory name is OK without checking ... */
	    
	    DPRINT(Debug,12,(&Debug,
			     "... using directory name %s (%S)\n",
			     str,Lstr));
	    
	    if (dir->a.local_browser->handle) 
		closedir(dir->a.local_browser->handle);
	    dir->a.local_browser->handle = NULL;

	    /* Make empty selection folder */
	    clear_dir_vector(dir);
	    
	    if (dir->dirname)
		free_string(&dir->dirname);
	    dir->dirname = Lstr;
	    Lstr = NULL;
	    
	    if (dir->sys_dir) {
		free(dir->sys_dir);
		dir->sys_dir = NULL;
	    }
	    dir->sys_dir = us2s(str);
	    str = NULL;
	    if (0 != access(dir->sys_dir,READ_ACCESS)) {
		DPRINT(Debug,12,(&Debug,
				 " -- No read access for %s\n",
				 dir->sys_dir));
		dir->a.local_browser->handle = NULL;
	    } else
		dir->a.local_browser->handle = opendir(dir->sys_dir);
	    
	    if (dir->a.local_browser->handle) {
		clear_dir_vector(dir);
		DPRINT(Debug,12,(&Debug,
				 "browser_descend_local: Delaying reading of directory\n"));
		dir->vector_len = -1;
	    } else {
		DPRINT(Debug,12,(&Debug,
				 "browser_descend_local: No dir listing for %s\n",
				 dir->sys_dir));
	    }
	    
	    free_string(&A1);
	} else {
	    DPRINT(Debug,12,(&Debug,
			     "browser_descend_local: No directory name on rel_name: %S\n",
			     rel_name));

	    if (dir->a.local_browser->handle) 
		closedir(dir->a.local_browser->handle);
	    dir->a.local_browser->handle = NULL;

	    /* Make empty selection folder */
	    clear_dir_vector(dir);

	    if (dir->dirname)
		free_string(&dir->dirname);

	    if (dir->sys_dir) {
		free(dir->sys_dir);
		dir->sys_dir = NULL;
	    }
	}
    } 

    DPRINT(Debug,12,(&Debug,
		     "browser_descend_local: Up to %d OK\n",last_idx_ok));

    last_idx_ok++;

    ret = clip_from_string(rel_name,&last_idx_ok,string_len(rel_name));

    if (dir->dirname)
	DPRINT(Debug,12,(&Debug,"*** %S as relative to %S is %S\n",
			 rel_name,dir->dirname,ret));

    free_string(&rel_name);
    
    DPRINT(Debug,12,(&Debug,
		"browser_descend_local=%S   -- consumed to %d\n",
		ret,last_idx_ok));
    return ret;
}

static void recover_old_dir P_((struct folder_browser *dir,
				struct string * old_dirname,
				char          * old_sys_dir));
static void recover_old_dir(dir,old_dirname,old_sys_dir)
     struct folder_browser *dir;
     struct string * old_dirname;
     char          * old_sys_dir;
{
    dir->sys_dir = strmcpy(dir->sys_dir,old_sys_dir);

    if (dir->dirname)
	free_string(&dir->dirname);
    if (old_dirname)
	dir->dirname = dup_string(old_dirname);
    else
	dir->dirname = new_string2(local_fs_charset,s2us(dir->sys_dir));
    
    DPRINT(Debug,12,(&Debug,
		     "recover_old_dir: Directory is %S, sys_dir=%s\n",
		     dir->dirname,dir->sys_dir));

    if (dir->a.local_browser->handle) 
	closedir(dir->a.local_browser->handle);

    dir->a.local_browser->handle = NULL;
    clear_dir_vector(dir);

    if (0 != access(dir->sys_dir,READ_ACCESS)) {
	dir->a.local_browser->handle = NULL;
    } else
	dir->a.local_browser->handle = opendir(dir->sys_dir);

    if (dir->a.local_browser->handle) {
	DPRINT(Debug,12,(&Debug,
			 "recover_old_dir: Delaying reading of directory\n"));
	dir->vector_len = -1;
    } else {
	int err UNUSED_VAROK = errno;

	DPRINT(Debug,12,(&Debug,
			 "recover_old_dir: %s: %s (errno %d)\n",
			 dir->sys_dir,strerror(err),err));
    }
}
				 
/* rel_dirname is relative to type -- (not include user@hostname) */
static int browser_change_local P_((struct folder_browser *dir,
				    const struct string *rel_dirname,
				    struct string **dispname));
static int browser_change_local(dir,rel_dirname,dispname)
     struct folder_browser *dir;
     const struct string *rel_dirname;
     struct string **dispname;
{
    int ret = 0;
    struct string * relative = NULL;
    struct string * Lstr = NULL;
    char          * str  = NULL;
    
    DIR           * new_dir = NULL;

    struct string * old_dirname = NULL;  
    char          * old_sys_dir = dir->sys_dir ? safe_strdup(dir->sys_dir) : NULL;

    DPRINT(Debug,11,(&Debug,"browser_change_local: dir=%p\n", dir));


    
    if (dir->dirname)
	old_dirname =  dup_string(dir->dirname);


    
    relative = browser_descend_local(dir,rel_dirname);

    /* browser_select_generic uses 'rel_dirname' if
       'relative' is null 
    */
    browser_select_generic(dir,dir->dirname,
			   rel_dirname,relative,&Lstr,&str,'/',
			   local_fs_charset);	

    if (LOCAL_BROWSER_magic != dir->a.local_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_change_local",
	      "Bad magic type",0);

    if (0 != access(str,READ_ACCESS)) {
	new_dir = NULL;
    } else
	new_dir = opendir(str);

    if (!new_dir) {
	int err = errno;
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDirError,
			  "Directory %S: %s"),
		  Lstr, strerror(err));
      
	goto fail;
    }

    
    
    if (dir->a.local_browser->handle) 
	closedir(dir->a.local_browser->handle);
    dir->a.local_browser->handle = new_dir;

    if (dir->a.local_browser->this_last_read) {
	flush_last_read(dir->a.local_browser->this_last_read);
	
	free_last_read_cache(& (dir->a.local_browser->this_last_read));
    }
	
    ret = 1;
		
    if (dir->dirname)
	free_string(&dir->dirname);
    /* Copy just from edit buffer */
    dir->dirname = dup_string(*dispname);
    
    if (dir->sys_dir) {
	free(dir->sys_dir);
	dir->sys_dir = NULL;
    }

    dir->sys_dir = str;
    str = NULL;
    
    clear_dir_vector(dir);
    DPRINT(Debug,12,(&Debug,
		     "browser_change_local: Delaying reading of directory\n"));
    dir->vector_len = -1;
       
 fail:
    if (!ret) {	
	if (old_sys_dir && !dir->sys_dir)
	    recover_old_dir(dir,old_dirname,old_sys_dir);
    } 
	
    if (old_dirname)
	free_string(& old_dirname);
    if (old_sys_dir)
	free(old_sys_dir);
        
    if (Lstr)
	free_string(&Lstr);    
    if (str)
	free(str);

    if (relative)
	free_string(&relative);

    DPRINT(Debug,11,(&Debug,"browser_change_local=%d\n", ret));
    return ret;
}

static int browser_change_v_local P_((struct folder_browser *dir,
				      struct name_vector *X,
				      struct string **dispname));
static int browser_change_v_local(dir,X,dispname)
     struct folder_browser *dir;
     struct name_vector *X;
     struct string **dispname;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_change_v_local: dir=%p\n", dir));

    if (LOCAL_BROWSER_magic != dir->a.local_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_change_v_local",
	      "Bad magic type",0);

    if (dir->dirname && dir->sys_dir) {
	struct string * Lstr = NULL;
	char          * str  = NULL;
	struct stat buf1;
	struct stat buf3;

	names_from_listing(dir,X,&Lstr,&str);
	
	DPRINT(Debug,12,(&Debug,
			 "browser_change_v_local: Using name %s (%S) from listing\n",
			 str,Lstr));

	if (stat(str, &buf1) != 0) {
	    int err = errno;
	    DPRINT(Debug,14,(&Debug, 
			     "browser_change_v_local: %s (errno=%d) attempting to stat %s\n", 
			     strerror(err), err, str));
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDirError,
			      "Directory %S: %s"),
		  Lstr, strerror(err));
	    
	    goto fail;
	}

	/* Ignore change is it leads to same directory */

	if (stat(dir->sys_dir, &buf3) != 0) {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,14,(&Debug, 
			     "browser_change_v_local: %s (errno=%d) attempting to stat %s\n", 
			     strerror(err), err, dir->sys_dir));
	    goto FAIL2;
	}

	if (buf1.st_ino == buf3.st_ino && buf1.st_dev == buf3.st_dev) {
	    DPRINT(Debug,14,(&Debug,
			     "browser_change_v_local: same directory %s and %s -- ignoring change\n",
			     str,dir->sys_dir));

	    ret = 1;
	    goto OUT2;	    
	} else {
	    DPRINT(Debug,14,(&Debug, 
			     "browser_change_v_local: sys_name=%s 	is not %s\n",
			     X->sys_name,dir->sys_dir));
	}

    FAIL2:      
	/* Handle changing to .. */
	
	if (0 == strcmp(X->sys_name,"..")) {    
	    int slen = strlen(dir->sys_dir);
	    int len = -1;
	    char * temp = NULL;
	    struct stat buf2;

	    /* Ignore seperator if it is on last character ... */
	    int i;

	    for (i = 0; i < slen-1; i++) {
		if ('/' == dir->sys_dir[i])
		    len = i;
	    }
	    
	    DPRINT(Debug,14,(&Debug, 
			     "browser_change_v_local: sys_dir=%s len=%d (last /)\n",
			     dir->sys_dir,len));

	    if (len < 1) {

		if (0 == len ||
		    0 == strcmp(dir->sys_dir,".")) {
		    DPRINT(Debug,14,(&Debug, 
				     "browser_change_v_local: sys_name=%s testing root directory\n",
				     X->sys_name));
		    
		    temp = safe_strdup("/");
		} else {
		    DPRINT(Debug,14,(&Debug, 
				     "browser_change_v_local: sys_name=%s testing current directory\n",
				     X->sys_name));
		    
		    temp = safe_strdup(".");
		}
	    } else {
		temp = safe_malloc(len+1);
		memcpy(temp,dir->sys_dir,len);
		temp[len] = '\0';
	    }	    

	    DPRINT(Debug,14,(&Debug, 
			     "browser_change_v_local: sys_name=%s possible previous directory=%s\n",
			     X->sys_name,temp));

	    if (stat(temp, &buf2) != 0) {
		int err UNUSED_VAROK = errno;
		DPRINT(Debug,14,(&Debug, 
				 "browser_change_v_local: %s (errno=%d) attempting to stat %s\n", 
				 strerror(err), err, temp));
		goto FAIL1;
	    }

	    if (buf1.st_ino == buf2.st_ino && buf1.st_dev == buf2.st_dev) {
		DPRINT(Debug,14,(&Debug,
				"browser_change_v_local: same directory %s and %s\n",
				str,temp));

		free(str); str = temp; temp = NULL;
		free_string(&Lstr);
		Lstr = new_string2(local_fs_charset,s2us(str));

		DPRINT(Debug,12,(&Debug,
			 "browser_change_v_local: Using name %s (%S) from path\n",
			 str,Lstr));

	    } else {
		DPRINT(Debug,14,(&Debug, 
				 "browser_change_v_local: sys_name=%s 	is not %s\n",
				 X->sys_name,temp));
	    }
			   
	FAIL1:

	    if (temp)
		free(temp);
	}

	if (dir->a.local_browser->handle) 
	    closedir(dir->a.local_browser->handle);

	
	if (0 != access(str,READ_ACCESS)) {
	    dir->a.local_browser->handle = NULL;
	} else
	    dir->a.local_browser->handle = opendir(str);
    
	if (!dir->a.local_browser->handle) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDirError,
			      "Directory %S: %s"),
		      Lstr, strerror(err));
	    goto fail;
	}

	if (dir->a.local_browser->this_last_read) {
	    flush_last_read(dir->a.local_browser->this_last_read);
	    
	    free_last_read_cache(& (dir->a.local_browser->this_last_read));
	}
	ret = 1;
	
	
	if (dir->dirname)
	    free_string(&dir->dirname);
	dir->dirname = Lstr;
	Lstr = NULL;
	
	if (dir->sys_dir) {
	    free(dir->sys_dir);
	    dir->sys_dir = NULL;
	}
	dir->sys_dir = str;
	str = NULL;
	

    OUT2:
    fail:
	if (Lstr)
	    free_string(&Lstr);    
	if (str)
	    free(str);

    } else if (!dir->dirname && !dir->sys_dir) {
	struct stat buf1;
	/* fbrowser seems go to here */
	
	if (stat(X->sys_name, &buf1) != 0) {
	    int err = errno;
	    DPRINT(Debug,14,(&Debug, 
			     "browser_change_v_local: %s (errno=%d) attempting to stat %s\n", 
			     strerror(err), err, X->sys_name));
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDirError,
			      "Directory %S: %s"),
		  X->disp_name, strerror(err));
	    
	    goto fail_empty;
	}

	if (dir->a.local_browser->handle) 
	    closedir(dir->a.local_browser->handle);

	
	if (0 != access(X->sys_name,READ_ACCESS)) {
	    dir->a.local_browser->handle = NULL;
	} else
	    dir->a.local_browser->handle = opendir(X->sys_name);

	if (!dir->a.local_browser->handle) {
	    int err = errno;

	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDirError,
			      "Directory %S: %s"),
		      X->disp_name, strerror(err));
	    
	    goto fail_empty;
	}

	if (dir->a.local_browser->this_last_read) {
	    flush_last_read(dir->a.local_browser->this_last_read);
	    
	    free_last_read_cache(& (dir->a.local_browser->this_last_read));
	}
	
	ret = 1;
	
	dir->dirname = dup_string(X->disp_name);
	dir->sys_dir = safe_strdup(X->sys_name);
	
    } else {
    fail_empty:
	DPRINT(Debug,12,(&Debug,
			 "browser_change_v_local:%s%s\n",
			 dir->dirname ? "" : " no dirname",
			 dir->sys_dir ? "" : " no sys_dir"));
    }

    if (ret) {
	if (*dispname)
	    free_string(dispname);
	/* Copy name to edit buffer */
	*dispname = dup_string(dir->dirname);

	clear_dir_vector(dir);
	DPRINT(Debug,12,(&Debug,
			 "browser_change_v_local: Delaying reading of directory\n"));
	dir->vector_len = -1;
    }
    
    DPRINT(Debug,11,(&Debug,"browser_change_v_local=%d\n", ret));

    return ret;   
}


static int browser_change_up_local P_((struct folder_browser *dir,
				       struct string **dispname,
				       struct string ** disp_tail));
static int browser_change_up_local(dir,dispname,disp_tail)
     struct folder_browser *dir;
     struct string **dispname;
     struct string ** disp_tail;
{
    int ret = 0;
    
    int L1;
    
    DPRINT(Debug,11,(&Debug,"browser_change_up_local: dir=%p\n", dir));

    if (! dir->dirname) {

	DPRINT(Debug,11,(&Debug,
			 "browser_change_up_local: Unknown local directory\n"));


	ret = -1; /* Signals caller to generate default menu */
	goto FAIL;
    }


    if (dir->a.local_browser->this_last_read) {
	flush_last_read(dir->a.local_browser->this_last_read);
	
	free_last_read_cache(& (dir->a.local_browser->this_last_read));
    }
    
    L1 =  browser_change_up_helper(dir->dirname,0x002F /* '/' */);

    if (L1 > 0) {
	int X = 0;
	struct string * A1 = clip_from_string(dir->dirname,&X,L1);

	if (disp_tail) 
	    *disp_tail = browser_disp_tail_helper(dir->dirname,X,0x002F /* '/' */);

	/* Need not generate actual directory listing */
	
	if (*dispname)
	    free_string(dispname);	    
	*dispname = A1;
	
	ret = 1;
    } else
	ret = -1;   /* Signals caller to generate default menu */

 FAIL:
    DPRINT(Debug,11,(&Debug,
		     "browser_change_up_local=%d\n", 
		     ret));
    return ret;   
}



/* rel_itemname is relative to type -- (not include user@hostname) */
static int browser_select_local P_((struct folder_browser *dir,
				    const struct string *rel_itemname,
				    struct string **dispname,
				    int *newpos));
static int browser_select_local(dir,rel_itemname,dispname,newpos)
     struct folder_browser *dir;
     const struct string *rel_itemname;
     struct string **dispname;
     int *newpos;
{
    int ret = 0;

    struct string * relative = NULL;
    struct string * old_dirname = NULL;  
    char          * old_sys_dir = dir->sys_dir ? safe_strdup(dir->sys_dir) : NULL;
    
    DPRINT(Debug,11,(&Debug,"browser_select_local: dir=%p\n", dir));

    if (dir->dirname)
	old_dirname =  dup_string(dir->dirname);


    if (rel_itemname) {
	DPRINT(Debug,11,(&Debug,"browser_select_local: rel_itemname=%S\n", 
		    rel_itemname));
    } else {
	DPRINT(Debug,11,(&Debug,"browser_select_local: rel_itemname=NULL\n"));
    }

    relative = browser_descend_local(dir,rel_itemname);

    if (relative) {
	ret = real_select_local(dir,rel_itemname,relative,newpos);
	free_string(&relative);
    }

    if (!ret) {
	if (old_sys_dir && !dir->sys_dir)
	    recover_old_dir(dir,old_dirname,old_sys_dir);
    } 
     
    if (old_dirname)
	free_string(& old_dirname);
    if (old_sys_dir)
	free(old_sys_dir);


    DPRINT(Debug,11,(&Debug,"browser_select_local=%d\n", ret));
    return ret;
}


S_(browser_select_idx_dir browser_select_idx_local)
static int browser_select_idx_local P_((struct folder_browser *dir,
					struct name_vector *entry,
					struct string **buffer));

static int browser_select_idx_local(dir,entry,buffer)
     struct folder_browser *dir;
     struct name_vector *entry;
     struct string **buffer;
{
    int F = 0;
    
    char * str = cat_dir_entry(dir->sys_dir,entry->sys_name);

    if (LOCAL_BROWSER_magic != dir->a.local_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_select_idx_local",
	      "Bad magic number (LOCAL_BROWSER)",0);

    
    F = local_selection_stat(str,
			     dir->a.local_browser->this_last_read);

    /* WARNING: set_dir_selection does not allocate strings -- 
     *          it just assign pointers!
     */
    set_dir_selection(dir,
		      safe_strdup(entry->sys_name),
		      dup_string(entry->disp_name),
		      entry->flags|F);

    free(str);

    return 1;
}


static struct string * browser_give_title_local P_((struct folder_browser *dir));
static struct string * browser_give_title_local(dir)
     struct folder_browser *dir;
{
    static struct string *ret;
    
    DPRINT(Debug,11,(&Debug,"browser_give_title_local: dir=%p\n", dir));
    
    if (!dir->sys_dir)
	dir->sys_dir = safe_strdup("");
      
    if (!dir->dirname)
	dir->dirname = new_string2(local_fs_charset,s2us(dir->sys_dir));

    if (dir->filter)
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeLocalDirFilter,
				    "Local directory %S with filter %S"),
			    dir->dirname,dir->filter);
    else
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeLocalDir,
				    "Local directory %S"),
			    dir->dirname);
        
    DPRINT(Debug,11,(&Debug,"browser_give_title_local=%S\n", ret));

    return ret;
}

static char browser_separator_local P_((struct folder_browser *dir));
static char browser_separator_local(dir)
     struct folder_browser *dir;
{
    return '/';
}

static struct string * browser_name_local P_((struct folder_browser *dir));
static struct string * browser_name_local(dir)
     struct folder_browser *dir;
{
    static struct string *ret;

    DPRINT(Debug,11,(&Debug,
		     "browser_name_local: dir=%p\n", 
		     dir));

    if (dir->sys_dir) {
    
	if (!dir->dirname)
	    dir->dirname = new_string2(local_fs_charset,s2us(dir->sys_dir));
	
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeLocalDir,
				    "Local directory %S"),
			    dir->dirname);
    } else
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeLocalDirUnknown,
				    "Unknown local directory"));

    DPRINT(Debug,11,(&Debug,"browser_name_local=%S\n", ret));

    return ret;
}

static struct string * browser_cat_local P_((struct folder_browser *dir,
					     const struct string * item));
static struct string * browser_cat_local(dir,item)
     struct folder_browser *dir;
     const struct string * item;
{
    struct string * ret = NULL;
    int L = 0;

    DPRINT(Debug,11,(&Debug,"browser_cat_local: dir=%p\n", 
		     dir));
   
    if (dir->dirname &&
	(L = string_len(dir->dirname)) > 0) {
	
	ret = dup_string(dir->dirname);
	if (L > 0 && 
	    0x002F /* '/' */  !=  give_unicode_from_string(dir->dirname,
							   L-1))
	    fill_ascii_to_string(ret,1,'/');
	
	/* Some local directory! */
	if (item) {
	    append_string(&ret, item, 0);
	}
    } else {
	/* Default MENU  -- also default directory ! */

	if (item)
	    ret = dup_string(item);
	else
	    ret = new_string(system_charset);
    }
    
    DPRINT(Debug,11,(&Debug,"browser_cat_local=%S\n",ret));
    return ret;
}

static struct folder_info * 
browser_folder_from_local P_((struct folder_browser *dir,
			      int treat_as_spooled));

static struct folder_info * browser_folder_from_local(dir,treat_as_spooled)
     struct folder_browser *dir;
     int treat_as_spooled;
{
    struct folder_info * res = NULL;

    DPRINT(Debug,11,(&Debug,"browser_folder_from_local: dir=%p\n", 
		     dir));

    res = real_folder_from_local(dir,treat_as_spooled,0);

    if (res) {
	DPRINT(Debug,11,(&Debug,"browser_folder_from_local=%p\n",res));
    } else {
	DPRINT(Debug,11,(&Debug,"browser_folder_from_local=NULL\n"));
    }
    return res;
}

static int browser_create_selection_local P_((struct folder_browser *dir));

static int browser_create_selection_local(dir)
     struct folder_browser *dir;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_create_selection_local: dir=%p\n", 
		     dir));

    ret = real_create_selection_local(dir,0);

    DPRINT(Debug,11,(&Debug,"browser_create_selection_local=%d\n", ret));
    
    return ret;
}


static int browser_prepare_write_local P_((struct folder_browser *dir,
				  WRITE_STATE ptr));
static int browser_prepare_write_local(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;
    char * name = NULL;

    DPRINT(Debug,11,(&Debug,"browser_prepare_write_local: dir=%p\n", dir));

    name =  resolve_selection_local(dir,0);
    if (name) {
	ret = real_prepare_write_local(dir,ptr,name);
	
	free(name);
    } else {
	DPRINT(Debug,11,(&Debug,
			 "browser_prepare_write_local: No selection name\n"));
    }

    DPRINT(Debug,11,(&Debug,"browser_prepare_write_local=%d\n", ret));

    return ret;
}

static int browser_sync_write_local P_((struct folder_browser *dir,
					WRITE_STATE ptr));
static int browser_sync_write_local(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;
    char * name = NULL;

    DPRINT(Debug,11,(&Debug,"browser_sync_write_local: dir=%p\n", dir));

    name =  resolve_selection_local(dir,0);
    if (name) {
	ret = real_sync_write_local(dir,ptr,name);
	
	free(name);
    } else {
	DPRINT(Debug,11,(&Debug,
			 "browser_sync_write_local: No selection mame\n"));
    }
    
    DPRINT(Debug,11,(&Debug,"browser_sync_write_local=%d\n", ret));
    
    return ret;
}


static int browser_end_write_local P_((struct folder_browser *dir,
			      WRITE_STATE ptr));
static int browser_end_write_local(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;
    char * name = NULL;

    DPRINT(Debug,11,(&Debug,"browser_end_write_local: dir=%p\n", dir));

    name =  resolve_selection_local(dir,0);
    if (name) {
	ret = real_end_write_local(dir,ptr,name);
    
	free(name);
    } else {
	DPRINT(Debug,11,(&Debug,
			 "browser_end_write_local: No selection name\n"));
    }
    
    DPRINT(Debug,11,(&Debug,"browser_end_write_local=%d\n", ret));
    
    return ret;
}

static long browser_tell_local_ws P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr));
static long browser_tell_local_ws(dir,write_state_ptr)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
{
    long ret = -1L;

    DPRINT(Debug,11,(&Debug,"browser_tell_local_ws: dir=%p\n", 
		     dir));

    ret = real_browser_tell_ws(dir,write_state_ptr);
    
    DPRINT(Debug,11,(&Debug,"browser_tell_local_ws=%ld\n",ret));
    return ret;
}

/* Returns 0 on failure */
static int browser_seek_local_ws P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr,
				      long pos));
static int browser_seek_local_ws(dir,write_state_ptr,pos)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     long pos;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_seek_local_ws: dir=%p\n", 
		     dir));

    ret = real_browser_seek_ws(dir,write_state_ptr,pos);
    
    DPRINT(Debug,11,(&Debug,"browser_seek_local_ws=%ld\n",ret));
    return ret;
}

static int browser_write_local_ws P_((struct folder_browser *dir,
				      WRITE_STATE ptr,
				      int l, const char *buffer));
static int browser_write_local_ws(dir,ptr,l,buffer)
     struct folder_browser *dir;
     WRITE_STATE ptr;
     int l; 
     const char *buffer;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_write_local_ws: dir=%p\n", 
		     dir));

    ret = real_browser_write_ws(dir,ptr,l,buffer);
    
    DPRINT(Debug,11,(&Debug,"browser_write_local_ws=%d\n",ret));
    return ret;
}


static int browser_start_we_local P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr,
				      int write_envelope,
				      struct header_rec *current_header,
				      int *env_flags));
static int browser_start_we_local(dir,write_state_ptr,
				  write_envelope,current_header,
				  env_flags)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
     int *env_flags;
{
    int ret = 1;

    DPRINT(Debug,11,(&Debug,"browser_start_we_local: dir=%p\n", 
		     dir));

    *env_flags = 0;
    if (write_envelope)
	ret = real_start_we_local(dir,write_state_ptr,current_header,
				  env_flags);
	    
    DPRINT(Debug,11,(&Debug,"browser_start_we_local=%d\n",ret));
    return ret;
}


static int browser_end_we_local P_((struct folder_browser *dir,
				  WRITE_STATE write_state_ptr,
				  int write_envelope,
				  struct header_rec *current_header));
static int browser_end_we_local(dir,write_state_ptr,
			      write_envelope,current_header)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
{
    int ret = 1;
    DPRINT(Debug,11,(&Debug,"browser_end_we_local: dir=%p\n", 
		     dir));

    if (write_envelope)
	ret = real_end_we_local(dir,write_state_ptr,current_header);

    DPRINT(Debug,11,(&Debug,"browser_end_we_local=%d\n",ret));
    return ret;
}

static int browser_selection_is_local P_((struct folder_browser *dir,
					  struct folder_info *folder));
static int browser_selection_is_local(dir,folder)
     struct folder_browser *dir;
     struct folder_info *folder;
{
    int ret = 0;
    DPRINT(Debug,11,(&Debug,"browser_selection_is_local: dir=%p\n", 
		     dir));
    
    ret = real_selection_is_local(dir,folder,0);
    
    DPRINT(Debug,11,(&Debug,"browser_selection_is_local=%d\n",ret));
    return ret;
}

static int browser_make_ref_local P_((struct folder_browser *dir,
				      char **refname, int *iscopy,
				      int is_text));
static int browser_make_ref_local(dir,refname,iscopy,is_text)
     struct folder_browser *dir;
     char **refname; 
     int *iscopy;
     int is_text;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_make_ref_local: dir=%p\n", 
		     dir));

    ret = real_make_ref_local(dir,refname,iscopy,is_text); 

    DPRINT(Debug,11,(&Debug,"browser_make_ref_local=%d\n",ret));
    
    return ret;
}

static void browser_update_local P_((struct folder_browser *dir));
static void browser_update_local(dir)
     struct folder_browser *dir;
{
    clear_dir_vector(dir);

    if (LOCAL_BROWSER_magic != dir->a.local_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_update_local",
	      "Bad magic type",0);
    
    if (dir->a.local_browser->handle) {
	DPRINT(Debug,10,(&Debug, 
			 "browser_fill_local: Reading directory of %s\n",
			 dir->sys_dir ? dir->sys_dir : "<NULL>"));

	browser_fill_local(dir);
    } else {
	DPRINT(Debug,11,(&Debug, 
			 "browser_fill_local: NO directory handle\n"));
    }
}

static int sort_by_name P_((const void *A, const void *B));
static int sort_by_name(A,B)
     const void *A; 
     const void *B;
{
    const struct name_vector * A1 = A;
    const struct name_vector * B1 = B;

    int r = string_cmp(A1->disp_name,
		       B1->disp_name,
		       0  /* compare failure */);

    if (0 == r)
	r = strcmp(A1->sys_name,B1->sys_name);
   
    return r;
}

static int sort_by_revname P_((const void *A, const void *B));
static int sort_by_revname(A,B)
     const void *A; 
     const void *B;
{
    return -sort_by_name(A,B);
}

static int sort_by_mtime P_((const void *A, const void *B));
static int sort_by_mtime(A,B)
     const void *A; 
     const void *B;
{
    const struct name_vector * A1 = A;
    const struct name_vector * B1 = B;

    int r = A1->mtime - B1->mtime;

    if (0 == r)
	r = strcmp(A1->sys_name,B1->sys_name);
    
    return r;

}
static int sort_by_revmtime P_((const void *A, const void *B));
static int sort_by_revmtime(A,B)
     const void *A; 
     const void *B;
{
    return -sort_by_mtime(A,B);
}


S_(browser_folder_sort_dir browser_folder_sort_local)
static void browser_folder_sort_local P_((struct folder_browser *dir,
					  print_sort_message * print));
static void browser_folder_sort_local(dir,print)
     struct folder_browser *dir;
     print_sort_message * print;
{
    struct string * msg = NULL;
    int i;

    DPRINT(Debug,12,(&Debug,
		     "browser_folder_sort_local: dir=%p; vector_len=%d\n",
		     dir,dir->vector_len));

    if (dir->vector_len < 2)
	return;

    switch (give_dt_sort_as_int(&local_dir_sortby)) {
    case LD_NONE:                 return;
    case LD_NAME_SORT:
	
	if (dir->vector_len > 10 &&
	    dir->dirname) {
	    msg = format_string(CATGETS(elm_msg_cat, MeSet, MeSortingDirByName, 
					"Sorting directory %S by Name"),
				dir->dirname);
	    print(msg);
	}
	qsort(dir->vector,dir->vector_len,sizeof (dir->vector[0]),
	      sort_by_name);
	break;
    case REVERSE LD_NAME_SORT:
	
	if (dir->vector_len > 10 &&
	    dir->dirname) {
	    msg = format_string(CATGETS(elm_msg_cat, MeSet, MeSortingDirByRevName, 
					"Sorting directory %S by Reverse Name"),
				dir->dirname);
	    print(msg);
	}
	qsort(dir->vector,dir->vector_len,sizeof (dir->vector[0]),
	      sort_by_revname);
	break;
    case LD_MTIME_SORT:

	if (dir->vector_len > 10 &&
	    dir->dirname) {
	    msg = format_string(CATGETS(elm_msg_cat, MeSet, MeSortingDirByMtime, 
					"Sorting directory %S by Modify Time"),
				dir->dirname);
	    print(msg);
	}
	for (i = 0; i < dir->vector_len; i++)
	    if (dir->vector[i].flags & BROWSER_NEEDSTAT)
		browser_local_do_stat(dir,i);


	qsort(dir->vector,dir->vector_len,sizeof (dir->vector[0]),
	      sort_by_mtime);
	break;
    case REVERSE LD_MTIME_SORT:

	if (dir->vector_len > 10) {
	    msg = format_string(CATGETS(elm_msg_cat, MeSet, MeSortingDirByRevMtime, 
					"Sorting directory %S by Reverse Modify Time"),
				dir->dirname);
	    print(msg);
	}

	for (i = 0; i < dir->vector_len; i++)
	    if (dir->vector[i].flags & BROWSER_NEEDSTAT)
		browser_local_do_stat(dir,i);

	qsort(dir->vector,dir->vector_len,sizeof (dir->vector[0]),
	      sort_by_revmtime);
	break;

    }

    if (msg) {
	print(NULL);
	free_string(&msg);
    }
}



S_(browser_free_name_vector_dir browser_free_name_vector_local)
static void browser_free_name_vector_local P_((struct folder_browser *dir,
					       struct name_vector *entry));
static void browser_free_name_vector_local(dir,entry)
     struct folder_browser *dir;
     struct name_vector *entry;
{    

    DPRINT(Debug,12,(&Debug,"browser_free_name_vector_local: dir=%p\n", dir));

    if (entry->a.local_browser)
	free_LOCAL_BROWSER_entry(& (entry->a.local_browser));

}

/* Increments refcount */
S_(browser_gethm_name_vector_dir browser_gethm_name_vector_local)
static struct hashmark_item *
    browser_gethm_name_vector_local P_((struct folder_browser * dir,
					struct name_vector    * entry));
static struct hashmark_item * browser_gethm_name_vector_local(dir,entry)
     struct folder_browser * dir;
     struct name_vector    * entry;
{
    DPRINT(Debug,12,(&Debug,"browser_gethm_name_vector_local: dir=%p\n", dir));

    if (entry->a.local_browser) {
	if (LOCAL_BR_entry_magic != entry->a.local_browser->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"browser_gethm_name_vector_local",
		  "Bad magic type",0);

	/* Entry do not have hashmark */

    }

    DPRINT(Debug,12,(&Debug,"browser_sethm_name_vector_local=NULL\n", dir));
    return NULL;
}

/* Returns 1 if hashmark saved
   Increments refcount, if hashmark saved */
S_(browser_sethm_name_vector_dir browser_sethm_name_vector_local)
static int browser_sethm_name_vector_local P_((struct folder_browser * dir,
					      struct name_vector    * entry,
					      struct hashmark_item  * hm
					      ));
static int browser_sethm_name_vector_local(dir,entry,hm)
     struct folder_browser * dir;
     struct name_vector    * entry;
     struct hashmark_item  * hm;
{
    DPRINT(Debug,12,(&Debug,"browser_sethm_name_vector_local: dir=%p\n", dir));
    
    if (entry->a.local_browser) {
	if (LOCAL_BR_entry_magic != entry->a.local_browser->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"browser_sethm_name_vector_local",
		  "Bad magic type",0);

	/* Entry do not have hashmark */

    }

    DPRINT(Debug,12,(&Debug,"browser_sethm_name_vector_local=0\n", dir));
    return 0;
}

/* Returns 1 if changed,
   -1 error
 */

S_(browser_reset_filter_dir browser_reset_filter_local)
static int browser_reset_filter_local P_((struct folder_browser *dir));
static int browser_reset_filter_local(dir)
     struct folder_browser *dir;
{    

    clear_dir_vector(dir);
    free_string(&(dir->filter));
	
    DPRINT(Debug,12,(&Debug,"browser_reset_filter_local: dir=%p\n", dir));


    if (LOCAL_BROWSER_magic != dir->a.local_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_reset_filter_local",
	      "Bad magic type",0);

    if (dir->a.local_browser->handle) {

	DPRINT(Debug,12,(&Debug,
			 "browser_reset_filter_local: Delaying reading of directory\n"));
	
	dir->vector_len = -1;
    }

    DPRINT(Debug,12,(&Debug,"browser_reset_filter_local=1\n"));
	
    return 1;
}


S_(browser_line_idx_dir browser_line_idx_local)
static const struct string * browser_line_idx_local 
                      P_((struct folder_browser *dir,
			  struct name_vector *entry,
			  const struct string  **comment_p));
static const struct string * browser_line_idx_local(dir,entry,comment_p)
     struct folder_browser *dir;
     struct name_vector *entry;
     const struct string  **comment_p;
{

    DPRINT(Debug,12,(&Debug,"browser_line_idx_local: dir=%p\n", dir));

    if (LOCAL_BROWSER_magic != dir->a.local_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_line_idx_local",
	      "Bad magic type",0);
        
    if (entry->a.local_browser) {
	
	if (LOCAL_BR_entry_magic != entry->a.local_browser->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"browser_line_idx_local",
		  "Bad magic number on local_browser",0);

	if (comment_p &&
	    entry->a.local_browser->comment1) {
	    
	    *comment_p = entry->a.local_browser->comment1;

	    DPRINT(Debug,12,(&Debug,
			     "browser_line_idx_local: comment=%S\n",
			     *comment_p));
	}

	if (entry->a.local_browser->line_name) {

	    DPRINT(Debug,12,(&Debug,
			     "browser_line_idx_local=%S\n",
			     entry->a.local_browser->line_name));

	    return entry->a.local_browser->line_name;

	}

    }


    /* Calculate possible name of symbolic link */

    if (entry->flags & BROWSER_NEEDSTAT) {
	browser_local_do_stat_2(dir,entry,0);

	if (entry->a.local_browser) {
	
	    if (LOCAL_BR_entry_magic != entry->a.local_browser->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"browser_line_idx_local",
		      "Bad magic number on local_browser",0);


	    if (entry->a.local_browser->line_name) {
	    
		DPRINT(Debug,12,(&Debug,
				 "browser_line_idx_local=%S\n",
				 entry->a.local_browser->line_name));
		
		return entry->a.local_browser->line_name;
	    }
	}
    }

    if ('.' != entry->sys_name[0]) {

	DPRINT(Debug,12,(&Debug,
			 "browser_line_idx_local=NULL\n"));

	return NULL;
    }

    if (0 != strcmp(entry->sys_name,".") &&
	0 != strcmp(entry->sys_name,"..")) {

	DPRINT(Debug,12,(&Debug,
			 "browser_line_idx_local=NULL\n"));

	return NULL;
    }

    if (!entry->a.local_browser) 
	entry->a.local_browser = 
	    malloc_LOCAL_BROWSER_entry();

    if (0 == strcmp(entry->sys_name,".")) {
	entry->a.local_browser->line_name = format_string(FRM("."));
	entry->a.local_browser->comment1 = 
	    format_string(CATGETS(elm_msg_cat, MeSet, MeLocalEntryCurrentDir,
				  "[current directory]"));
    } else if (0 == strcmp(entry->sys_name,"..")) {
	entry->a.local_browser->line_name = format_string(FRM(".."));	    
	entry->a.local_browser->comment1 = 
	    format_string(CATGETS(elm_msg_cat, MeSet, MeLocalEntryParentDir,
				  "[parent directory]"));	
    }

    if (comment_p &&
	entry->a.local_browser->comment1) {
	
	*comment_p = entry->a.local_browser->comment1;
	
	DPRINT(Debug,12,(&Debug,
			 "browser_line_idx_local: comment=%S\n",
			 *comment_p));
    }

    if (entry->a.local_browser->line_name) {
	DPRINT(Debug,12,(&Debug,
			 "browser_line_idx_local=%S\n",
			 entry->a.local_browser->line_name));
    } else {
	DPRINT(Debug,12,(&Debug,
			 "browser_line_idx_local=NULL\n"));
    }

    return entry->a.local_browser->line_name;
}

S_(browser_reload_dir browser_reload_local)
static int browser_reload_local P_((struct folder_browser *dir));
static int browser_reload_local(dir)
     struct folder_browser *dir;
{
    int ret = 0;

    DPRINT(Debug,12,(&Debug,"browser_reload_local: dir=%p\n", dir));

    if (LOCAL_BROWSER_magic != dir->a.local_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_line_idx_local",
	      "Bad magic type",0);

    if (! dir->sys_dir) {	
	DPRINT(Debug,12,(&Debug,"browser_reload_local: No directory name\n"));
	goto fail;
    }

    if (! dir->dirname) 
	dir->dirname = new_string2(local_fs_charset,s2us(dir->sys_dir));
	

    if (dir->a.local_browser->handle)
	rewinddir(dir->a.local_browser->handle);
    else  if (0 != access(dir->sys_dir,READ_ACCESS)) {
	DPRINT(Debug,12,(&Debug,
			 " -- No read access for %s\n",
			 dir->sys_dir));
	
    } else 
	dir->a.local_browser->handle = opendir(dir->sys_dir);
    
    if (dir->a.local_browser->handle) {
	ret = 1;

	clear_dir_vector(dir);
	DPRINT(Debug,12,(&Debug,
			 "browser_reload_local: Delaying reading of directory\n"));
	dir->vector_len = -1;
    } 

    if (!dir->a.local_browser->handle) {
	int err = errno;
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDirError,
			  "Directory %S: %s"),
		  dir->dirname, strerror(err));
    }

 fail:
    DPRINT(Debug,12,(&Debug,"browser_reload_local=%d\n", ret));

    return ret;
}

struct browser_type local_browser = {
				       BROWSER_TYPE_magic,  
				       browser_zero_local,
				       browser_free_local,
				       browser_change_local,
				       browser_give_title_local,
				       browser_separator_local,
				       browser_name_local,
				       browser_cat_local,
				       browser_select_local,
				       browser_folder_from_local,
				       browser_change_v_local,
				       browser_change_up_local,
				       browser_create_selection_local,
				       zero_ws_fields_local,
				       free_ws_fields_local,
				       browser_prepare_write_local,
				       browser_end_write_local,
				       browser_tell_local_ws,
				       browser_seek_local_ws,
				       browser_write_local_ws,
				       browser_start_we_local,
				       browser_end_we_local,
				       browser_selection_is_local,
				       browser_make_ref_local,
				       browser_update_local,
				       browser_local_do_stat,
				       browser_sync_write_local,
				       browser_folder_sort_local,
				       browser_free_name_vector_local,
				       browser_reset_filter_local,
				       browser_select_idx_local,
				       browser_line_idx_local,
				       browser_reload_local,
				       browser_gethm_name_vector_local,
				       browser_sethm_name_vector_local

};

#endif  /* DIROPS */


/* ------------------------------------------------------------------------ */

int fbrowser_supported_on_dir(dir)
     struct folder_browser *dir;
{

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_supported_on_dir",
	      "Bad magic (browser_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "fbrowser_supported_on_dir: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));


    if (dir->type != &local_browser) {
	DPRINT(Debug,10,(&Debug,"fbrowser_supported_on_dir=0: dir=%p not a local browser\n", 
			 dir));

	return 0;
    }

    DPRINT(Debug,10,(&Debug,"fbrowser_supported_on_dir=1\n", 
		     dir));
    return 1;
}

const struct fbentry * fbrowser_give_entry(dir,idx)
     struct folder_browser *dir;
     int idx;
{
    const struct fbentry * ret = NULL;

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_give_entry",
	      "Bad magic (browser_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "fbrowser_give_entry: dir=%p (%s); type=%p, idx=%d\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type,idx));

    if (idx < 0 || idx >= dir->vector_len)
	panic("MBX PANIC",__FILE__,__LINE__,"fbrowser_give_entry",
	      "Bad index",0);

    if (dir->type != &local_browser) {
	DPRINT(Debug,10,(&Debug,"fbrowser_give_entry=NULL: dir=%p not a local browser\n", 
			 dir));

	return NULL;
    }
    
    if (!dir->vector[idx].a.local_browser) 
	dir->vector[idx].a.local_browser =  malloc_LOCAL_BROWSER_entry();

    else if (LOCAL_BR_entry_magic != dir->vector[idx].a.local_browser->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"fbrowser_give_entry",
	      "Bad magic type on local_browser",0);

    if ((dir->vector[idx].flags & BROWSER_NEEDSTAT) ||
	! dir->vector[idx].a.local_browser->fbentry) 
	browser_local_do_stat_1(dir,idx,1);
    
    if (! dir->vector[idx].a.local_browser->fbentry) {
	DPRINT(Debug,10,(&Debug,"fbrowser_give_entry=NULL: Stat failed\n"));
	
	return NULL;
    }

    ret = dir->vector[idx].a.local_browser->fbentry;

    DPRINT(Debug,10,(&Debug,"fbrowser_give_entry=%p\n",ret));	     

    if (FBENTRY_magic != ret->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"fbrowser_give_entry",
	      "Bad magic type on fbentry",0);

    DPRINT(Debug,12,(&Debug,"                    name         =%s\n",ret->name));
    DPRINT(Debug,12,(&Debug,"                    display_name =%S\n",ret->display_name));
    DPRINT(Debug,12,(&Debug,"                    mode         =0%3o\n",ret->mode));
    DPRINT(Debug,12,(&Debug,"                    uid          =%d\n",ret->uid));
    DPRINT(Debug,12,(&Debug,"                    size         =%ld\n",(long)ret->size));
    DPRINT(Debug,12,(&Debug,"                    mtime        =%ld\n",(long)ret->mtime));

    return ret;
}

static int local_make_selection_str P_((struct folder_browser *dir,
					 char **strp,
					 struct string **Strp));
static int local_make_selection_str(dir,strp,Strp)
     struct folder_browser *dir;
     char **strp;
     struct string **Strp;
{
    int add_sep   = 0;

    char *str            = *strp;
    struct string * Str  = *Strp;

    if (!dir->selection->sys_name ||
	!dir->selection->disp_name) {

	DPRINT(Debug,12,(&Debug, 
			 "local_make_selection_str=1: No selection\n"));
	return 0;
    }


    if (dir->sys_dir) {
	int l;

	str = safe_strdup(dir->sys_dir);

	l = strlen(dir->sys_dir);
	if (l > 0 && '/' != str[l-1]) {
	    char buf[2];
            buf[0] = '/';
            buf[1] = '\0';
            str = strmcat(str,buf);

	    add_sep = 1;
	}
    }

    if (dir->dirname) {
	Str = dup_string(dir->dirname);
	
	if (add_sep) 
	    fill_ascii_to_string(Str,1,'/');
    }

    str = strmcat(str,dir->selection->sys_name);
    append_string(&Str,dir->selection->disp_name,0);

    *strp = str;
    *Strp = Str;

    DPRINT(Debug,12,(&Debug, 
		     "local_make_selection_str=1: add_sep=%d, str=%s, Str=%S\n",
		     add_sep,str,Str));

    return 1;
}

int fbrowser_selection_have_access(dir,acc_mode,print_error)
     struct folder_browser *dir;
     int acc_mode;
     int print_error;
{
    int ret       = 0;
    char          * str = NULL;
    struct string * Str = NULL;

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "fbrowser_selection_have_access",
	      "Bad magic (browser_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "fbrowser_selection_have_access: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (!dir->selection) 
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_selection_have_access",
	      "No selection",0);

    if (0 == (dir->selection->flags & BROWSER_EXIST)) {

	if (print_error) {
	    if (dir->dirname)
		lib_error(CATGETS(elm_msg_cat, MeSet, MeSelectionNotExists,
				  "Selection \"%S\" does not exist on \"%S\"."),
			  dir->selection->disp_name,
			  dir->dirname);
	    else
		lib_error(CATGETS(elm_msg_cat, MeSet, MeSelectionNotExists0,
				  "Selection \"%S\" does not exist."),
			  dir->selection->disp_name);
	    
	}

	DPRINT(Debug,10,(&Debug, 
			 "fbrowser_selection_have_access=0: No BROWSER_EXIST; selection flags=%d\n",
			 dir->selection->flags));

	return 0;
    } 

    if (dir->type != &local_browser) {
	DPRINT(Debug,10,(&Debug,"fbrowser_selection_have_access=1: dir=%p not a local browser\n", 
			 dir));

	return 1;
    }

    if (local_make_selection_str(dir,&str,&Str)) {

	if (0 != access(str,acc_mode)) {
	    int err = errno;
	    
	    if (print_error) 
		lib_error(CATGETS(elm_msg_cat, MeSet, MeSelectionNoAccess,
				  "Selection \"%S\": %s"),
			  Str,strerror(err));
	    
	    DPRINT(Debug,10,(&Debug,
			     "fbrowser_selection_have_access=0: Access %s (%S) failed (mode %d): %s (errno %d)\n",
			     str,Str,acc_mode,strerror(err),err));
	    
	    ret = 0;
	    goto FAIL;
	
	}
   
	DPRINT(Debug,10,(&Debug,
			 "fbrowser_selection_have_access=1: Access %s (%S) OK (mode %d)\n",
			 str,Str,acc_mode));
	
	ret = 1;

    } else {
	DPRINT(Debug,10,(&Debug,
			 "fbrowser_selection_have_access=0: Selection not OK\n"));
	ret = 0;
    }

 FAIL:

    if (str)
	free(str);
    if (Str)
	free_string(&Str);

    return ret;
}

static int mbx_copy_envelope_check_MMDF P_((char *buffer,int len));

int fbrowser_selection_is_mbox(dir,print_error)
     struct folder_browser *dir;
     int print_error;
{
    int ret       = 0;
    char          * str = NULL;
    struct string * Str = NULL;

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_selection_is_mbox",
	      "Bad magic (browser_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "fbrowser_selection_is_mbox: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (!dir->selection) 
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_selection_is_mbox",
	      "No selection",0);

    if (dir->type != &local_browser) {
	DPRINT(Debug,10,(&Debug,"fbrowser_selection_is_mbox=1: dir=%p not a local browser\n", 
			 dir));

	return 1;
    }


    if (local_make_selection_str(dir,&str,&Str)) {
	FILE          * F = NULL;
	
	struct stat    buf;		/* stat command  */
	int have_stat = 0;
	
	char *buffer = NULL;
	int      len = 0;
	enum syscall_status  r = stat(str,&buf);
	int err;
	
	switch (r) {
	case syscall_success /* 0 */:
	    have_stat = 1;
	    break;

	case syscall_error /* -1 */: {
	    err = errno;
	    
	    if (print_error) 
		lib_error(CATGETS(elm_msg_cat, MeSet, MeSelectionNoAccess,
				  "Selection \"%S\": %s"),
			  Str,strerror(err));	
	    
	    DPRINT(Debug,10,(&Debug,
			     "fbrowser_selection_is_mbox=0: Stat %s (%S) failed: %s (errno %d)\n",
			     str,Str,strerror(err),err));
	    
	    ret = 0;
	    goto FAIL;       
	}
	    break;
	}

	err = can_open(str,"r");
	if (err) {

	    if (print_error) 
		lib_error(CATGETS(elm_msg_cat, MeSet, MeSelectionNoAccess,
				  "Selection \"%S\": %s"),
			  Str,strerror(err));	
	    
	    
	    DPRINT(Debug,10,(&Debug,
			     "fbrowser_selection_is_mbox=0: can_open %s (%S) failed: %s (errno %d)\n",
			     str,Str,strerror(err),err));

	    ret = 0;
	    goto FAIL;	    
	}

	
	F = fopen(str,"r");
	
	if (!F) {
	    err = errno;
	    
	    if (print_error) 
		lib_error(CATGETS(elm_msg_cat, MeSet, MeSelectionNoAccess,
				  "Selection \"%S\": %s"),
			  Str,strerror(err));	
	    
	    
	    DPRINT(Debug,10,(&Debug,
			     "fbrowser_selection_is_mbox=0: Open %s (%S) failed: %s (errno %d)\n",
			     str,Str,strerror(err),err));
	    
	    ret = 0;
	    goto FAIL;

	}
	
	if (!mbx_read_line(F,&buffer,&len,1000)) {
	    DPRINT(Debug,10,(&Debug,
			     "fbrowser_selection_is_mbox=1: Empty file\n"));
	    
	    ret = 1;
	    goto FAIL;
	}

	if (have_MMDF &&
	    mbx_copy_envelope_check_MMDF(buffer,len)) {
	    
	    DPRINT(Debug,10,(&Debug,
			     "fbrowser_selection_is_mbox: MMDF envelope seen\n"));
	    
	    free(buffer); buffer = NULL;
	    
	    if (!mbx_read_line(F,&buffer,&len,1000)) {
		DPRINT(Debug,10,(&Debug,
				 "fbrowser_selection_is_mbox: EOF after MMDF envelope\n"));
		
		goto FAIL;
	    }
	    
	    ret = 1;
	}

	if (real_from(buffer,NULL)) {
	
	    DPRINT(Debug,10,(&Debug,
			     "fbrowser_selection_is_mbox: From seen\n"));
	    ret = 1;
	}
	
	if (!ret) {
	    if (print_error) 
		lib_error(CATGETS(elm_msg_cat, MeSet, MeSelectionNotLocalMailbox,
				  "Selection \"%S\" is not valid local mailbox."),
			  Str);
	}
	
	DPRINT(Debug,10,(&Debug,
			 "fbrowser_selection_is_mbox=%d\n",
			 ret));

 FAIL:

	if (F)
	    fclose(F);
	
	if (have_stat) {
	    int err = 0;

	    if (change_file_utime(str,
				    &buf,
				    &err)) {
		DPRINT(Debug,10,(&Debug,
				 "fbrowser_selection_is_mbox: %s: utime succeed\n",
				 str));
	    } else if (err) {
		DPRINT(Debug,10,(&Debug,
				 "fbrowser_selection_is_mbox: %s: encountered error doing utime:: %s (errno %d  -- owner = %d (my uid = %d, euid = %d))\n",
				 str,
				 strerror(err),err,
				 buf.st_uid,userid,geteuid()));
		
		
		/* Ignore error if we are not owner of file */
		
		if (print_error &&
		    (buf.st_uid == userid ||       /* should be uid == geteuid() */
		     buf.st_uid == geteuid())) 
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveChangingAccessTime,
				      "Error %s trying to change file %S access time."), 
			      strerror(err), 
			      Str);
	    }
	}
	
	if (buffer)
	    free(buffer);

    } else {
	DPRINT(Debug,10,(&Debug,
			 "fbrowser_selection_is_mbox=0: Selection not OK\n"));
	ret = 0;
    }
    
    if (str)
	free(str);
    if (Str)
	free_string(&Str);

    return ret;
}

void fbrowser_set_dotmode(dir,dotmode)
     struct folder_browser *dir;
     int dotmode;
{
    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_set_dotmode",
	      "Bad magic (browser_type)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "fbrowser_set_dotmode: dir=%p (%s); type=%p, dotmode=%d\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type,
		     dotmode));


    if (dir->type != &local_browser) {
	DPRINT(Debug,10,(&Debug,"fbrowser_set_dotmode: dir=%p not a local browser\n", 
			 dir));

	return;
    }

    if (LOCAL_BROWSER_magic != dir->a.local_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"fbrowser_set_dotmode",
	      "Bad magic type",0);

    clear_dir_vector(dir);
    
    if (dir->a.local_browser->handle) {

	DPRINT(Debug,12,(&Debug,
			 "fbrowser_set_dotmode: Delaying reading of directory\n"));
	dir->vector_len = -1;
    }

    dir->a.local_browser->dotmode = dotmode;
}

/* Returns pid */

static int fork_mkdir P_((char * long_name));
static int fork_mkdir(long_name)
     char * long_name;
{
     int pid = -1;

#ifdef VFORK
    pid = vfork();
#else
    pid = fork();
#endif

    if (0 == pid) {

	if (-1 == setgid(groupid)) {	    
	    int err = errno;
	    fprintf(stderr,"fbrowser_make_directory: setgid(%d) FAILED: %s\n",
		    groupid,strerror(err));
	    fflush(stderr);
	    _exit(err != 0? err : 1);	/* never return zero! */
	}

	if (-1 == setuid(userid)) {		/** back to normal userid **/
	    int err = errno;
	    fprintf(stderr,"fbrowser_make_directory: setuid(%d) FAILED: %s\n",
		    userid,strerror(err));
	    fflush(stderr);
	    _exit(err != 0? err : 1);	/* never return zero! */
	}
	
	/* mode is affected by umask ! */
	if (0 == mkdir(long_name,0777)) {
	    _exit(0);
	}
	_exit(errno != 0 ? errno : 1);   

    } 

    return pid;  /* pid or -1 */    
}

int fbrowser_make_directory(dir,name,pos)
     struct folder_browser *dir;
     const struct string *name;
     int *pos;
{
    char          * long_name  = NULL;
    struct string *Lstr = NULL;
    char *         str = NULL;

    int l1 = 0;
    int add_slash = 0;
    int ret  = 0;

    int pid;

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"fbrowser_make_directory",
	      "Bad magic (browser_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "fbrowser_make_directory: dir=%p (%s); type=%p, name=%S\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type,
		     name));


    if (dir->type != &local_browser) {
	DPRINT(Debug,10,(&Debug,"fbrowser_make_directory=0: dir=%p not a local browser\n", 
			 dir));

	return 0;
    }

    if (!dir->sys_dir || !dir->dirname) {
	DPRINT(Debug,10,(&Debug,"fbrowser_make_directory=0: no dir\n"));
	
	return 0;
    }

    l1 = strlen(dir->sys_dir);
    if (l1 > 0 && '/' != dir->sys_dir[l1-1]) {
	add_slash = 1;
    }
    
    long_name  = safe_strdup(dir->sys_dir);

    if (add_slash) {
	long_name = strmcat(long_name,"/");
    }

    Lstr = convert_string(local_fs_charset,name,0);
    str  =  us2s(stream_from_string(Lstr,0,NULL));

    DPRINT(Debug,12,(&Debug,
		     "... using name %s (%S)\n",str,Lstr));

    long_name = strmcat(long_name,str);

    DPRINT(Debug,12,(&Debug,
		     "... long name %s\n",long_name));


    /* Do not use setgid privileges on creation of directory */

    pid = fork_mkdir(long_name);
    if (-1 == pid) {

	int err = errno;

	DPRINT(Debug,10,(&Debug,
			 "browser_reload_local: fork failed: %s (errno %d)\n",
			 strerror(err),err));
	
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDirCreateError,
			  "Failed to create %S to %S: %s"),
		  name,dir->dirname, strerror(err));

    } else {
	S__ status;
	int w;


	DPRINT(Debug,12,(&Debug,"browser_reload_local: worked pid = %d\n",pid)); 

	while ((w = my_wait(pid,&status)) != pid && 
	       (w != -1 || EINTR == errno))
	  ;

	if (-1 == w) {
	    int err = errno;
	    
	    DPRINT(Debug,10,(&Debug,
			     "browser_reload_local: wait failed: %s (errno %d)\n",
			     strerror(err),err));

	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDirCreateError,
			      "Failed to create %S to %S: %s"),
		      name,dir->dirname, strerror(err));

	} else {
	    int exit_status;
	    int sig = convert_status(status,&exit_status);

	    if (sig) {
		DPRINT(Debug,10,(&Debug,"browser_reload_local: Process %d died on signal %d\n",
				 pid,sig));
		
		lib_error(CATGETS(elm_msg_cat, MeSet, MeDirCreateDied,
				  "Failed to create %S to %S: Died on signal %d"),
			  name,dir->dirname, sig);
		

	    } else if (exit_status) {

		DPRINT(Debug,10,(&Debug,
				 "browser_reload_local: Process %d exit status %d: %s: %s\n",
				 pid,exit_status,
				 long_name, strerror(exit_status)));

		lib_error(CATGETS(elm_msg_cat, MeSet, MeDirCreateError,
				  "Failed to create %S to %S: %s"),
			  name,dir->dirname, strerror(exit_status));

	    } else {

		*pos = -1;
		
		if (NULL != strchr(str,'/')) {
		    DPRINT(Debug,12,(&Debug,
				     "... name %s includes '/' -- not in this directory %S\n",
				     str,dir->dirname));
		    
		} else if (dir->vector_len >= 0) {
		    
		    /* WARNING: add_dir_vector does not allocate strings -- 
		     *          it just assign pointers!	    
		     */
		    
		    add_dir_vector(dir,str,Lstr,BROWSER_NOFOLDER);
		    
		    if (pos)
			*pos = dir->vector_len-1;
		    
		    str  = NULL;
		    Lstr = NULL;
		}
		
		ret = 1;
	    }
	}
    }
	
    if (str)
	free(str);
    if (Lstr)
	free_string(&Lstr);

    free(long_name);


    DPRINT(Debug,10,(&Debug,"fbrowser_make_directory=%d\n",ret));
    
    return ret;

}


/* ------------------------------------------------------------------------- */

/* LOCAL mailbox */

S_(mbx_close_folder mbx_close_non_spool)
static int mbx_close_non_spool P_((struct folder_info *folder,
				    enum close_mode mode,
				    struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
								 on remote mailbox
							      */));
static int mbx_close_non_spool(folder,mode,cd) 
     struct folder_info *folder;
     enum close_mode mode;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  on remote mailbox
			       */;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_close_non_spool",
	      "Bad magic number (folder_info)",0);
        
    DPRINT(Debug,11,(&Debug, 					 
		     "mbx_close_non_spool: folder=%X (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_close_non_spool",
	      "Bad magic number (private_data)",0);

    /* normal folder */

    if (folder->p->fh_folder) {
	switch (mode) {
	case CLOSE_NORMAL:
	    if (lockfolders)
		Release_the_file(fileno(folder->p->fh_folder),
				 folder->p->lock_info);      
	    break;
	case CLOSE_LEAVE_LOCKED:
	    break;
	}
    }

    DPRINT(Debug,11,(&Debug, 					 
		     "mbx_close_non_spool=1\n"));
    return 1;
}

S_(mbx_close_folder mbx_close_spool)
static int mbx_close_spool P_((struct folder_info *folder,
				enum close_mode mode,
				struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
							      on remote mailbox
							  */));
static int mbx_close_spool(folder,mode,cd) 
     struct folder_info *folder;
     enum close_mode mode;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  on remote mailbox
			       */;
{
    int ret = 1;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_close_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_close_spool: folder=%X (%s)\n",
		     folder,folder->cur_folder_sys));


    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_close_spool",
	      "Bad magic number (private_data)",0);

    switch (mode) {
    case CLOSE_NORMAL:

	if (folder->p->a.spool.lock_state == ON)
	    unlock_folder(0,folder);
    
	/* falltru */
    case CLOSE_LEAVE_LOCKED:
	if (!folder->p->fh_temp) {
	    DPRINT(Debug,1,(&Debug, 
			    "mbx_close_spool: tempfile (%s) not open -- will not unlink\n",
			    folder -> cur_tempfolder));
	} else {
	    int r = unlink(folder -> cur_tempfolder);
	    if (-1 == r) {
		int err = errno;
		
		if (err != ENOENT) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSorryCantUnlinkTemp,
				      "Sorry, can't unlink the temp file %s [%s]!\n\r"),
			  folder -> cur_tempfolder, strerror(err));

		    ret = 0;
		}
	    } else if (0 == r) {
		DPRINT(Debug,1,(&Debug, 
				"close_folder: Unlinking tempfile (%s).\n",
				folder -> cur_tempfolder));
	    }      
	}
	break;
    }

    DPRINT(Debug,11,(&Debug, 
		     "mbx_close_spool=%d\n",
		     ret));
    return ret;
}


S_(mbx_lock_folder mbx_lock_non_spool)
static int  mbx_lock_non_spool P_((enum lock_direction direction,
				    struct folder_info *folder));
static int  mbx_lock_non_spool(direction,folder)
     enum lock_direction direction;
     struct folder_info *folder;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_lock_non_spool",
	      "Bad magic number (folder_info)",0);

    DPRINT(Debug,11,(&Debug, 
		     "mbx_lock_non_spool=1 (dummy): folder=%X (%s)\n",
		     folder,folder->cur_folder_sys));    	     
    return 1;
}


static int  mbx_dotlock_file P_((enum lock_direction direction,
				 struct folder_info *folder));
static int  mbx_dotlock_file(direction,folder)
     enum lock_direction direction;
     struct folder_info *folder;
{
    int status = 0;
    int create_iteration = 0;
    char pid_buffer[STRING];

    struct stat    buf;		/* stat command  */
    enum syscall_status stat_code = -1;

    int err       = 0;
    int temp_fd = -1;
    int create_fd = -1;	/* file descriptor for creating lock file */

    gid_t oldgid = getegid();
    int old_dotfile = 0;
    int need_wait = 0;

    enum lockfile_pidcheck_v pidcheck_mode =
#ifdef PIDCHECK
	give_dt_enumerate_as_int(&local_lockfile_pidcheck)

#else
	pidcheck_no
#endif
	;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_dotlock_file",
	      "Bad magic number (folder_info)",0);
    
    if (have_saved_ids) {
	/* reset id so that we can get at lock file */
	if (setgid(mailgroupid) < 0) {
	    int err = errno;
	    lib_error(FRM("Lock: setgid(%d) FAILED: %s"),
		      mailgroupid,strerror(err));	    
	}
    }

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_dotlock_file",
	      "Bad magic number (private_data)",0);

    if(folder->p->a.spool.lock_state == ON) {
	DPRINT(Debug,1,(&Debug, 
			"mbx_dotlock_file: Folder %s already locked.\n", 
			folder->cur_folder_sys));
	if (direction == LOCK_INCOMING)
	    DPRINT(Debug,1,(&Debug, 
			    "mbx_dotlock_file: ERROR: ... And direction INCOMING! Software error!\n"));
	else {
	    status = 1;
	    goto clean;
	}		   
    }
    

    /** first, try to read the lock file, and if possible, check the pid.
	If we can validate that the pid is no longer active, then remove
	the lock file.
    **/
    	
    if((temp_fd=open(folder->p->a.spool.lockfile,O_RDONLY)) != -1) {
	int n;
	struct stat oX;

	enum syscall_status   ocode = fstat(temp_fd,&oX);

	switch (ocode) {
	case syscall_success /* 0 */: {

	    DPRINT(Debug,5,(&Debug, 
			    "mbx_dotlock_file: atime (before read) = %d \n",
			    oX.st_atime)); 
	    
	}
	    break;
	case syscall_error /* -1 */: {
	    int err = errno;
	    
	    DPRINT(Debug,5,(&Debug, 
			    "mbx_dotlock_file: fstat %d failed: (errno=%d) %s\n",
			    temp_fd,err,strerror(err)));
	    
	    lib_error(FRM("%s: %s"), 
		      folder->p->a.spool.lockfile,strerror(err));	    	    
	}
	    break;
	}
	
	n = read(temp_fd, pid_buffer, sizeof pid_buffer);
	if (n < 0) {
	    int err = errno;
	    
	    DPRINT(Debug,5,(&Debug, 
			    "mbx_dotlock_file: %s read failed; errno=%d\n",
			    folder->p->a.spool.lockfile,err));

	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmReadLockfileE,
			      "Couldn't read the lock file %s: %s"),
		      folder->p->a.spool.lockfile,
		      strerror(err));

	} else {

	    struct stat X;
	    enum syscall_status r = fstat(temp_fd,&X);
	    
	    if (n > 0 && '\n' == pid_buffer[n-1])
		n--;

	    switch (r) {
	    case syscall_error /* -1 */: {
		int err UNUSED_VAROK = errno;

		DPRINT(Debug,5,(&Debug, 
				"mbx_dotlock_file: %s fstat failed; errno=%d\n",
				folder->p->a.spool.lockfile,err));
		
		lib_error(FRM("%s: %s"), 
			  folder->p->a.spool.lockfile,strerror(err));

	    }
		break;
	    case syscall_success /* 0 */:
		if (ocode == syscall_success /* 0 */) {
		    DPRINT(Debug,5,(&Debug, 
				    "mbx_dotlock_file: atime (now) = %d \n",
				    X.st_atime)); 
		    DPRINT(Debug,5,(&Debug, 
				    "                  mtime       = %d \n",
				    X.st_mtime)); 
		    
		    if (oX.st_atime == X.st_atime) {
			time_t now = time(NULL);
			
			DPRINT(Debug,5,(&Debug, 
					"mbx_dotlock_file: now         = %d \n",
					now)); 
			
			
			DPRINT(Debug,5,(&Debug, "File read does not change access time!\n"));
			old_dotfile = X.st_mtime < now - 5 * 60;
			
			DPRINT(Debug,5,(&Debug, 
					"                  (modified %1.1f minutes ago)\n",
					(double)(now - X.st_mtime) / 60.0));
			
			
		    } else {
			old_dotfile = X.st_mtime < X.st_atime - 5 * 60;
			
			DPRINT(Debug,5,(&Debug, 
					"                  (modified %1.1f minutes ago from read)\n",
					(double)(X.st_atime - X.st_mtime) / 60.0));
			
		    }
		    
		    DPRINT(Debug,1,(&Debug, 
				    "mbx_dotlock_file: old_dotfile = %d\n",
				    old_dotfile));
		}
		break;
	    }
	}

	
	if (pidcheck_mode > pidcheck_no) {	    
	    if (n > 0) {
		int L;
		char * ptr = NULL;
		int pid_value = 0;

		pid_buffer[n] = '\0';

		DPRINT(Debug,7,(&Debug, 
				"mbx_dotlock_file: dotfile value: %s\n",
				pid_buffer));

		errno = 0;  /* To see is errno set */
		pid_value = L = strtol(pid_buffer,&ptr,10);

		if ((ptr && *ptr && *ptr != '@') || 
		    errno || pid_value < 0) {
		    int err = errno;

		    if (err)
			lib_error(CATGETS(elm_msg_cat, ElmSet,
					  ElmParseLockfileE,
					  "Couldn't parse the lock file %s: %s"),
				  folder->p->a.spool.lockfile,
				  strerror(err));
		    else
			lib_error(CATGETS(elm_msg_cat, ElmSet,
					  ElmParseLockfile,
					  "Couldn't parse the lock file %s"),
				  folder->p->a.spool.lockfile);

		} else if (pid_value <= 1) {
		    /* Dummy value -- no pid set */
		    
		    goto no_pid_seen;

		} else if (ptr && *ptr == '@' &&
			   0 != strcmp(ptr+1,hostname) &&
			   0 != strcmp(ptr+1,hostfullname) &&
			   pidcheck_ignore_hostname != pidcheck_mode) {

		    lib_error(CATGETS(elm_msg_cat, ElmSet,
				      ElmMailboxLockedonHost,
				      "Mailbox %S is locked by pid %ld on host %s"),				      		   		
			      folder->cur_folder_disp,
			      L,ptr+1);
		} else if ((!ptr || *ptr != '@') &&
			   pidcheck_require_hostname == pidcheck_mode) {
		    
		    lib_error(CATGETS(elm_msg_cat, ElmSet,
				      ElmMailboxLockedonSome,
				      "Mailbox %S is locked by pid %ld on some host"),				      		   		
			      folder->cur_folder_disp, L);
		} else if (pid_value == L) {

		    enum syscall_status kill_status = kill(pid_value, 0);

		    switch (kill_status) {
			int err;
		    case syscall_error /* -1 */:
			err = errno;
			
			if (EPERM == err)  {
			case syscall_success /* 0 */:
			    lib_error(CATGETS(elm_msg_cat, ElmSet,
					      ElmMailboxLockedPid,
					      "Mailbox %S is locked by pid %ld"),
				      folder->cur_folder_disp,L);
			    break;
			} else {
			    enum syscall_status unlink_status =
				unlink(folder->p->a.spool.lockfile);

			    switch (unlink_status) {
			    case syscall_error /* -1 */: {
				err = errno;
				
				DPRINT(Debug,1,(&Debug, 
						"Error %s\n\ttrying to unlink file %s (%s)\n", 
						strerror(errno), 
						folder->p->a.spool.lockfile, "lock"));
				lib_error(CATGETS(elm_msg_cat, ElmSet, 
						  ElmLeaveCouldntRemoveLock,
						  "Couldn't remove the current lock file %s: %s"), 
					  folder->p->a.spool.lockfile,
					  strerror(err));
				
				close(temp_fd);
				status = 0;
				goto clean;
			    }
				break;
				
			    case syscall_success /* 0 */: {
				DPRINT(Debug,9,(&Debug,
						"mbx_dotlock_file: Folder %s: Removed stale lockfile %s: No process %d\n",
						folder->cur_folder_sys,
						folder->p->a.spool.lockfile,
						pid_value));
			    }
				break;
			    }
			}
			break;
		    }
		    
		}

	    }
	no_pid_seen: ;
		
	}

	close(temp_fd);
    } else {	    
	int err = errno;

	DPRINT(Debug,5,(&Debug, 
			"mbx_dotlock_file: %s open failed; errno=%d\n",
			folder->p->a.spool.lockfile,err));
	
	if (ENOENT != err && pidcheck_mode > pidcheck_no) {

	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmReadLockfileE,
			      "Couldn't read the lock file %s: %s"),
		      folder->p->a.spool.lockfile,
		      strerror(err));
	    
	}
    }

    /* try to assert create lock file MAX_ATTEMPTS times */
    do {
	err = 0;
	if((create_fd=open(folder->p->a.spool.lockfile,
				   O_WRONLY | O_CREAT | 
				   O_EXCL,0444)) != -1)
	    break;
	else {
	    err = errno;

	    DPRINT(Debug,9,(&Debug,
			    "open (creating) %s: %s (errno %d) --  try %d\n",
			    folder->p->a.spool.lockfile,
			    strerror(err),err,
			    create_iteration));

	    stat_code = stat(folder->p->a.spool.lockfile,&buf);

	    switch (stat_code) {
	    case syscall_success /* 0 */:
		DPRINT(Debug,9,(&Debug,
				"stat %s succeed!\n",
				folder->p->a.spool.lockfile));				
		break;

	    /* Creation of lock failed NOT because it already exists!!! */
		
	    case syscall_error /* -1 */:
		
		if (err != EEXIST) {		    

		    const char * mailhome_val = 
			give_dt_estr_as_str(&mailhome_dir_e,
					    "mailhome-dir",
					    NULL,NULL);
		    
		    
		    DPRINT(Debug,1,(&Debug, 
				    "Error encountered attempting to create lock %s\n", 
				    folder->p->a.spool.lockfile));
		    DPRINT(Debug,1,(&Debug, 
				    "** %s **\n", strerror(err)));
		    
		    /* If /var/mail nfs mounted on Solaris 2.3 at least you can */
		    /* get EACCES.  Treat it like EEXIST.                       */
		
		    if (err != EACCES ||
			! mailhome_val ||
			/* Stat buffer argument is NULL because stat failed */
			! in_directory(NULL,folder->p->a.spool.lockfile,
				       mailhome_val)) {

			lib_error(CATGETS(elm_msg_cat, ElmSet, 
					  ElmLeaveErrorCreatingLock,
					  "\nError encountered while attempting to create lock file %s;\n"),
				  folder->p->a.spool.lockfile);
			lib_error(CATGETS(elm_msg_cat, ElmSet, 
					  ElmErrorAccess,
					  "%.45s: %.33s"),
				  folder->p->a.spool.lockfile,
				  strerror(err));
		    } else {	
			lib_error(CATGETS(elm_msg_cat, ElmSet, 
					  ElmLeaveCantCreateLock,
					  "Can't create lock file! Need write permission in \"%s\".\n"),
				  mailhome_val);
		    }
		    status = 0;
		    goto clean;
		}
		break;
	    }
	}
	DPRINT(Debug,2,(&Debug, 
			"File '%s' already exists!  Waiting...(lock)\n", 
			folder->p->a.spool.lockfile));
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveWaitingToRead,
			  "Waiting to read mailbox while mail is being received: attempt #%d"),
		  create_iteration);
	
	if (POLL_method)
	    wait_for_timeout(5);
	else
	    sleep(5);

    } while (create_iteration++ < MAX_ATTEMPTS);
    
    if(create_fd == -1) {	
	struct stat    buf2;	/* stat command  */
	enum syscall_status  stat2_code = stat(folder->cur_folder_sys,&buf2);
	time_t now = time(NULL);
	int can_remove = 0;

	DPRINT(Debug,2,(&Debug, 
			"mbx_dotlock_file: now       = %d \n",
			now));
	
	switch (stat_code) {
	case syscall_success /* 0 */:
	    DPRINT(Debug,1,(&Debug, 
			    "mbx_dotlock_file: lock time = %d (%s)\n",
			    buf.st_mtime,folder->p->a.spool.lockfile));
	    can_remove = buf.st_mtime < now - 10 * 60;
	    DPRINT(Debug,1,(&Debug, 
			    "                  (modified %1.1f minutes ago)\n",
			    (double)(now - buf.st_mtime) / 60.0));
	    break;
	case syscall_error /* -1 */:
	     DPRINT(Debug,10,(&Debug, 
			      "mbx_dotlock_file: %s: lock time not available\n",
			      folder->p->a.spool.lockfile));
	     break;
	}
	
	switch (stat2_code) {
	case syscall_success /* 0 */:
	    
	    DPRINT(Debug,1,(&Debug, 
			    "mbx_dotlock_file: read time = %d (%s)\n",
			    buf2.st_atime,folder->cur_folder_sys));
	    if (buf2.st_atime > now - 10 * 60)
		can_remove = 0;
	    DPRINT(Debug,1,(&Debug, 
			    "                  (read %1.1f minutes ago)\n",
			    (double)(now - buf2.st_atime) / 60.0));
	    break;
	case syscall_error /* -1 */:
	    DPRINT(Debug,10,(&Debug,
			     "mbx_dotlock_file: %s: rwad time not available\n",
			     folder->cur_folder_sys));
	    break;	    
	}
	
	DPRINT(Debug,2,(&Debug, 
			"mbx_dotlock_file: can remove = %d (old dotfile = %d)\n",
			can_remove,old_dotfile)); 

	/* we weren't able to create the lock file */
	
	if (can_remove && old_dotfile) {
	    enum syscall_status  r;

	    /** time to waste the lock file!  Must be there in error! **/
	    DPRINT(Debug,2,(&Debug, 
			    "Warning: I'm giving up waiting - removing lock file(lock)\n"));
	    if (direction == LOCK_INCOMING)
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmLeaveTimedOutRemoving,
				  "\nTimed out - removing current lock file..."));
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveThrowingAwayLock,
				  "Throwing away the current lock file!"));


	    r = unlink(folder->p->a.spool.lockfile);
	    switch (r) {
	    case syscall_error /* -1 */: {
		int err = errno;
		
		DPRINT(Debug,1,(&Debug, 
				"Error %s\n\ttrying to unlink file %s (%s)\n", 
				strerror(errno), folder->p->a.spool.lockfile, "lock"));
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmLeaveCouldntRemoveLock,
				  "Couldn't remove the current lock file %s: %s"), 
			  folder->p->a.spool.lockfile,
			  strerror(err));
		
		status = 0;
		goto clean;
	    }
		break;
	    case syscall_success /* 0 */:
		DPRINT(Debug,10,(&Debug, 
				 "mbx_dotlock_file: Removed %s\n",
				 folder->p->a.spool.lockfile));
		break;
	    }
	
	    /* we've removed the bad lock, let's try to assert lock once more */
	    if((create_fd=open(folder->p->a.spool.lockfile,
			       O_WRONLY | O_CREAT | O_EXCL,0444)) == -1){
		int err=errno;

		/* still can't lock it - just give up */
		DPRINT(Debug,1,(&Debug, 
			   "Error encountered attempting to create lock %s\n", 
			   folder->p->a.spool.lockfile));
		DPRINT(Debug,1,(&Debug, "%s (errno %d)\n",
				strerror(err), err));

		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveErrorCreatingLock,
				  "\nError encountered while attempting to create lock file %s;\n"),
			  folder->p->a.spool.lockfile);	    
		lib_error(FRM("** %s. **\n\n"), 
			  strerror(err));
		status = 0;
		goto clean;

	    } else 
		/* wait sleepmsg seconds so that
		        Timed out - removing current lock file...
		   is visible
		*/
		need_wait = 1;

	} else {

	    /* Okay...we die and leave, not updating the mailfile mbox or
	       any of those! */

	    if (direction == LOCK_INCOMING) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveGivingUp,
				  "\n\nGiving up after %d iterations.\n"), 
			  create_iteration);		
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeavePleaseTryAgain,
				  "\nPlease try to read your mail again in a few minutes.\n\n"));
		DPRINT(Debug,1,(&Debug, 
			   "Warning: bailing out after %d iterations...(lock)\n",
			   create_iteration));
		status = 0;
		goto clean;
		
	    } else {
		DPRINT(Debug,1,(&Debug, 
				"Warning: after %d iterations, timed out! (lock)\n",
				create_iteration));
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveErrorTimedOutLock,
				  "Timed out on locking mailbox.  Leaving program.\n"));
		status = 0;
		goto clean;
	    }
	}
    }

    status = 1;
    /* If we're here we successfully created the lock file */
    DPRINT(Debug,55,(&Debug, 
		     "Lock %s %s for file %s on.\n", 
		     folder->p->a.spool.lockfile,
		     (direction == LOCK_INCOMING ? "incoming" : "outgoing"), 
		     folder->cur_folder_sys));
    
    /* Place the pid of Elm into the lock file for SVR3.2 and its ilk */

    switch(pidcheck_mode) {
    case pidcheck_hostname:
    case pidcheck_require_hostname:
	elm_sfprintf(pid_buffer, sizeof pid_buffer,
		     FRM("%d@%s"), 
		     getpid(),hostname);
	break;
	
    default:	
	elm_sfprintf(pid_buffer, sizeof pid_buffer,
		     FRM("%d"), 
		     getpid());
	break;
    }

    if (-1 == write(create_fd, pid_buffer, strlen(pid_buffer))) {	
	DPRINT(Debug,2,(&Debug, 
			"Failed to write pid to file %s: &s\n",
			folder->p->a.spool.lockfile));	
	status = 0;
    }
    
 clean:
    if (create_fd != -1)
	close(create_fd);

    if (have_saved_ids) {
	if (setgid(oldgid) <0) {
    	    int err = errno;
	    lib_error(FRM("Lock: setgid(%d) FAILED: %s"),
		      oldgid,strerror(err));
	}
    }

    if (need_wait && sleepmsg > 0) {  /* HACK */
	/* wait sleepmsg seconds so that
	        Timed out - removing current lock file...
	   is visible
	*/

	if (POLL_method)
	    wait_for_timeout(sleepmsg);
	else
	    sleep(sleepmsg);
    }
    
    DPRINT(Debug,2,(&Debug, 
		    "mbx_dotlock_file = %d \n",
		    status));

    return status;
}

#ifdef SYSCALL_LOCKING
static int  mbx_syscall_lock_file P_((enum lock_direction direction,
				      struct folder_info *folder));
static int  mbx_syscall_lock_file(direction,folder)
     enum lock_direction direction;
     struct folder_info *folder;
{
    int status = 0;
     enum FLOCKING_status stat = FLOCKING_FAIL;
    int flock_iteration = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_syscall_lock_file",
	      "Bad magic number (folder_info)",0);
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_syscall_lock_file",
	      "Bad magic number (private_data)",0);

    /* try to assert lock MAX_ATTEMPTS times */
    do {
	stat = Grab_the_file (fileno(folder->p->fh_folder),
			      folder->p->lock_info
			      );

	switch (stat) {
	case FLOCKING_OK:
	    goto lock_ok;
	case FLOCKING_UNSUPPORTED:
	    DPRINT(Debug,1,(&Debug,  
			    "Mailbox not locked (%s) -- unsupported.\n",
			    folder->cur_folder_sys));
	    goto lock_ok;
	    
	case FLOCKING_FAIL:
	    /*
	     *	Creation of lock failed
	     *	NOT because it already exists!!!
	     */
	    
	    DPRINT(Debug,1,(&Debug, 
			    "Error encountered attempting to lock %s\n",
			    folder->cur_folder_sys));
	    DPRINT(Debug,1,(&Debug, 
			    "** %s **\n", 
			    strerror(errno)));
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveErrorFlockMailbox,
			      "Error encountered while attempting to flock mailbox %S: %s"), 
		      folder->cur_folder_disp,
		      strerror(errno));
	    
	    status = 0;
	    goto clean;

	case	FLOCKING_RETRY:
	    DPRINT(Debug,2,(&Debug, 
			    "Mailbox '%s' already locked!  Waiting...(lock)\n", 
			    folder->cur_folder_sys));
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveWaitingToRead,
			      "Waiting to read mailbox while mail is being received: attempt #%d"),
		      flock_iteration);

	    if (POLL_method)
		wait_for_timeout(5);
	    else
		sleep(5);
	}
      } while (flock_iteration++ < MAX_ATTEMPTS);

 lock_ok:
    
      lib_transient(FRM(""));

      if (stat != FLOCKING_OK &&
	  stat != FLOCKING_UNSUPPORTED) {
	  /* We couldn't lock the file. We die and leave not updating
	   * the mailfile mbox or any of those! */
	  
	  if (direction == LOCK_INCOMING) {
	      lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveGivingUp,
				"\n\nGiving up after %d iterations.\n"), 
			flock_iteration);	    
	      lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeavePleaseTryAgain,
				"\nPlease try to read your mail again in a few minutes.\n\n"));
	      DPRINT(Debug,1,(&Debug, 
			      "Warning: bailing out after %d iterations...(lock)\n",
			      flock_iteration));
	  } else {
	      DPRINT(Debug,1,(&Debug, 
			      "Warning: after %d iterations, timed out! (lock)\n",
			      flock_iteration));
	  }
	  lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveErrorTimedOutLock,
			    "Timed out on locking mailbox.  Leaving program.\n"));
	  status = 0;
	  goto clean;
      }

      /* We locked the file */
      DPRINT(Debug,5,(&Debug, 					 
		      "Lock %s on file %s on.\n",
		      (direction == LOCK_INCOMING ? "incoming" : "outgoing"), 
		      folder->cur_folder_sys));
      status = 1;

 clean:
      DPRINT(Debug,2,(&Debug, 
		      "mbx_syscall_lock_file = %d \n",
		      status));
      return status;
}
#endif

S_(mbx_lock_folder mbx_lock_spool)
static int  mbx_lock_spool P_((enum lock_direction direction,
			       struct folder_info *folder));
static int  mbx_lock_spool(direction,folder)
     enum lock_direction direction;
     struct folder_info *folder;
{
     /** Create lock file to ensure that we don't get any mail 
	 while altering the folder contents!
	 If it already exists sit and spin until 
	 either the lock file is removed...indicating new mail
	 or
	 we have iterated MAX_ATTEMPTS times, in which case we
	 either fail or remove it and make our own (determined
	 by if REMOVE_AT_LAST is defined in header file
	 If direction == INCOMING then DON'T remove the lock file
	 on the way out!  (It'd mess up whatever created it!).
	 
	 But if that succeeds and if we are also locking by flock(),
	 follow a similar algorithm. Now if we can't lock by flock(),
	 we DO need to remove the lock file, since if we got this far,
	 we DID create it, not another process.
      **/
    
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_lock_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_lock_spool: folder=%X (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__," mbx_lock_spool",
	      "Bad magic number (private_data)",0);

    if (dt_flag_is_set(folder->p->lock_info,lock_lockfile_flag)) {	
	if (!mbx_dotlock_file(direction,folder)) {
	    status = 0;
	    goto clean;
	}
    }

#ifdef SYSCALL_LOCKING
      /* Now we also need to lock the file with flock(2) */

    if(!folder->p->fh_folder) {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_lock_spool: non-existent -- no syscall locking.\n"));
    } else if (!mbx_syscall_lock_file(direction,folder)) {

	if (dt_flag_is_set(folder->p->lock_info,lock_lockfile_flag)) {	
	    (void)unlink(folder->p->a.spool.lockfile);
	}

	status = 0;
	goto clean;
    }
#endif

    DPRINT(Debug,5,(&Debug, 
		    "Lock %s for file %s on successfully.\n",
		    (direction == LOCK_INCOMING ? "incoming" : "outgoing"), 
		    folder->cur_folder_sys));

    folder->p->a.spool.lock_state = ON;
    status = 1;
    
 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_lock_spool=%d\n", status));

    return status;
}

S_(mbx_unlock_folder mbx_unlock_non_spool)
static int mbx_unlock_non_spool P_((int interrupt, 
				    struct folder_info *folder));
static int mbx_unlock_non_spool(interrupt,folder) 
     int interrupt;
     struct folder_info *folder;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_unlock_non_spool",
	      "Bad magic number (folder_info)",interrupt);
    
    SIGDPRINT(Debug,11,(&Debug, 
		     "mbx_unlock_non_spool=1 (dummy): folder=%X (%s)\n",
		     folder,folder->cur_folder_sys));
    return 1;
}

#ifdef SYSCALL_LOCKING

static int mbx_syscall_unlock_file P_((int interrupt, 
				       struct folder_info *folder));

static int mbx_syscall_unlock_file(interrupt,folder)
     int interrupt; 
     struct folder_info *folder;
{
    int status = 0;
    enum FLOCKING_status r;
    int err = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_syscall_unlock_file",
	      "Bad magic number (folder_info)",interrupt);
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_syscall_unlock_file",
	      "Bad magic number (private_data)",interrupt);



    r =  filelock_fd (fileno(folder->p->fh_folder),
		      FLOCKING_release,
		      folder->p->lock_info,
		      folder->cur_folder_sys,
		      interrupt ? FLOCKING_interrupt : FLOCKING_non_block,
		      &err);
		      
    switch (r) {
    case FLOCKING_FAIL:
	SIGDPRINT(Debug,1,(&Debug, 
			   "Error %s\n\ttrying to unlock file %s (%s)\n", 
			   strerror(errno), 
			   folder->cur_folder_sys, "unlock"));
	break;
    case FLOCKING_OK:
	status = 1;
	break;
    case FLOCKING_RETRY:
    case FLOCKING_UNSUPPORTED:
	break;
    }

    return status;
}
#endif

static int mbx_dotunlock_file P_((int interrupt, 
				  struct folder_info *folder));

static int mbx_dotunlock_file(interrupt,folder)
     int interrupt; 
     struct folder_info *folder;
{
    int status = 0;
    gid_t oldgid = getegid();

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_dotunlock_file",
	      "Bad magic number (folder_info)",interrupt);

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_dotunlock_file",
	      "Bad magic number (private_data)",interrupt);

    if (have_saved_ids) {
	/* reset id so that we can get at lock file */
	
	if (setgid(mailgroupid) < 0) {
    	    int err = errno;
	    SIGDPRINT(Debug,1,(&Debug,
			       "mbx_dotunlock_file: setgid(%d) FAILED: %s\n",
			       mailgroupid,strerror(err)));
	    if (!interrupt)
		lib_error(FRM("Unlock: setgid(%d) FAILED: %s"),
			  mailgroupid,strerror(err));
	}
    }

    if(unlink(folder->p->a.spool.lockfile) == 0) {	/* remove lock file */
	status = 1;
    } else {
	SIGDPRINT(Debug,1,(&Debug, 
			"Error %s\n\ttrying to unlink file %s (%s)\n", 
			strerror(errno), 
			folder->p->a.spool.lockfile,
			"unlock"));
	if (!interrupt)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmLeaveCouldntRemoveOwnLock,
			      "Couldn't remove my own lock file %s!"), 
		      folder->p->a.spool.lockfile);
    }

    if (have_saved_ids) {
	if (setgid(oldgid) < 0) {
    	    int err = errno;
	    SIGDPRINT(Debug,1,(&Debug, 
			       "mbx_dotunlock_file: setgid(%d) FAILED: %s\n",
			       oldgid,strerror(err)));
	    if (!interrupt)
		lib_error(FRM("Unlock: setgid(%d) FAILED: %s"),
			  oldgid,strerror(err));
	}
    }

    return status;
}

S_(mbx_unlock_folder mbx_unlock_spool)
static int mbx_unlock_spool P_((int interrupt, struct folder_info *folder));

static int mbx_unlock_spool(interrupt,folder)
     int interrupt; 
     struct folder_info *folder;
{
    /** Remove the lock file!    This must be part of the interrupt
	processing routine to ensure that the lock file is NEVER
	left sitting in the mailhome directory!
	
	If also using flock(), remove the file lock as well.
    **/

    int status = 1;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_unlock_spool",
	      "Bad magic number (folder_info)",interrupt);
    
    SIGDPRINT(Debug,11,(&Debug, 
		     "mbx_unlock_spool: folder=%X (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__," mbx_unlock_spool",
	      "Bad magic number (private_data)",interrupt);

    if (!dt_flag_is_set(folder->p->lock_info,lock_lockfile_flag)) {	

	SIGDPRINT(Debug,5,(&Debug, 
			   "Lock (no .lock) for file %s %s off.\n",
			   folder->cur_folder_sys, 
			   (folder-> p->a.spool.lock_state == ON ? 
			    "going" : "already")));
    } else {

	SIGDPRINT(Debug,5,(&Debug, 
			   "Lock %s for file %s %s off.\n",
			   (folder->p->a.spool.lockfile ? 
			    folder->p->a.spool.lockfile : "none"),
			   folder->cur_folder_sys,
			   (folder->p->a.spool.lock_state == ON ? 
			    "going" : "already")));
    }

    if(folder->p->a.spool.lock_state == ON) {
#ifdef SYSCALL_LOCKING
	if(!folder->p->fh_folder) {
	    SIGDPRINT(Debug,11,(&Debug, 
			"mbx_unlock_spool: non-existent -- no syscall locking.\n"));
	} else  if (!mbx_syscall_unlock_file(interrupt,folder))
	    status = 0;
#endif

	if (dt_flag_is_set(folder->p->lock_info,lock_lockfile_flag)) {	
	    if (!mbx_dotunlock_file(interrupt,folder))
		status = 0;    
	}
    
	if (status)
	    folder->p->a.spool.lock_state = OFF; /* indicate we don't have a lock on */
    
    }
    SIGDPRINT(Debug,11,(&Debug, 
			"mbx_unlock_non_spool=%d\n",
			status));
    return status;
}

static void mk_temp_mail_fn P_((char *tempfn, char *mbox,
				int tempfn_size));

S_(mbx_init_folder mbx_init_non_spool)
static void mbx_init_non_spool P_((struct folder_info *folder));
static void mbx_init_non_spool(folder) 
     struct folder_info *folder;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_init_non_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_init_non_spool: folder=%X (%s)\n",
		     folder,folder->cur_folder_sys));

    folder->p = malloc_mbx_private_data(&folder_locking);

}

S_(mbx_init_folder mbx_init_spool)
static void mbx_init_spool P_((struct folder_info *folder));
static void mbx_init_spool(folder) 
     struct folder_info *folder;
{	
    struct stat buf, *Xbuf = NULL;
    const char * mailhome_val = give_dt_estr_as_str(&mailhome_dir_e,
						    "mailhome-dir",
						    NULL,NULL);
    enum syscall_status  r;
    
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_init_spool",
	      "Bad magic number (folder_info)",0);

    DPRINT(Debug,11,(&Debug, 
		     "mbx_init_spool: folder=%X (%s)\n",
		     folder,folder->cur_folder_sys));

    mk_temp_mail_fn(folder-> cur_tempfolder, 
		    folder->cur_folder_sys, 
		    sizeof folder -> cur_tempfolder);

    folder->p = malloc_mbx_private_data(NULL);

    r = stat(folder->cur_folder_sys,&buf);

    switch(r) {
    case syscall_success /* 0 */: {
	Xbuf = &buf;
    }
	break;
    case syscall_error /* -1 */: {
	int err UNUSED_VAROK = errno;
	DPRINT(Debug,11,(&Debug,
			 "... stat %s: (errno=%d) %s\n",
			 folder->cur_folder_sys,err,strerror(err)));
    }
	break;
    }
	
    if (mailhome_val &&
	in_directory(Xbuf,folder->cur_folder_sys,mailhome_val)) {
	DPRINT(Debug,11,(&Debug," .. in mailhome\n"));
	folder ->p->lock_info   = &mailhome_locking;
    } else
	folder ->p->lock_info   = &mailbox_locking;

    folder->p->a.spool.lock_state = OFF;

    if (dt_flag_is_set(folder->p->lock_info,lock_lockfile_flag)) {	
	folder->p->a.spool.lockfile=
	    safe_strdup(mk_lockname(folder->cur_folder_sys));
    } else
	folder -> p -> a.spool.lockfile = NULL;
}


/* Result is malloced */
static char * give_sessionlock_fname(mbox,mbox_stat)
     char *mbox;
     struct stat *mbox_stat;  /* May be NULL */
{

    char * ret = NULL;
    char * ptr = NULL;

    const char * tmp = 
	give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    enum local_seslck_use_home_v ls  = 
	give_dt_enumerate_as_int(&local_sessionlock_use_home);
    const char * mailhome_val = give_dt_estr_as_str(&mailhome_dir_e,
						    "mailhome-dir",
						    NULL,NULL);

    const char *Y = NULL, *Y1 = "";
    int L;

    char * prefix = "mbox.";
    char *cp = NULL;

    int in_spool = mailhome_val &&
	in_directory(mbox_stat,mbox,mailhome_val);

    DPRINT(Debug,11,(&Debug, 
		     "give_sessionlock_fname: in_spool=%d, mbox=%s, local_sessionlock_use_home=%d\n",
		     in_spool,mbox,ls));

    /* We need not use default_temp or local_sessionlock_dir
       if file is not on spool area 
    */
    
    switch (ls) {
    case local_seslck_use_home_never: /* never */
    default:
	Y = in_spool || !tmp ? local_sessionlock_dir : tmp;
	break;
    case local_seslck_use_home_spool: /* spool */
	Y       = in_spool || !tmp ? home     : tmp;
	prefix  = in_spool || !tmp ? ".mbox." : "mbox.";
	break;
    case local_seslck_use_home_always: /* always */
	Y      = home;
	prefix = ".mbox.";
	break;
    case local_seslck_use_home_non_spool: /* non-spool */
	Y      = in_spool ? local_sessionlock_dir : home;
	prefix = in_spool ? "mbox."               : ".mbox.";
	break;
    }

    DPRINT(Debug,11,(&Debug, 
		     "     => temp dir=%s prefix=%s\n",Y,prefix));

    L = strlen(Y);
    
    if (L < 1) {
	DPRINT(Debug,1,(&Debug,
			"give_sessionlock_fname: Dir empty!!\n"));

	ret = safe_strdup("TEMP_MBOX");
	goto done;
	
    }

    if (Y[L-1] != '/')
	Y1 = "/";

    if ((cp = strrchr(mbox, '/')) != NULL) {    
	cp++;
	if (strcmp(cp, "mbox") == 0 || strcmp(cp, "mailbox") == 0 ||
	    strcmp(cp, "inbox") == 0 || *cp == '.') {
	    ptr = username;
	}
	else {
	    ptr = cp;
	}
    } else {
	ptr = mbox;
    }

    if (!in_spool)
	ret = elm_message(FRM("%s%s%s%s-%s"), 
			   Y, Y1, prefix, ptr, username);

    else
	ret = elm_message(FRM("%s%s%s%s"), 
			   Y, Y1, prefix, ptr);


 done:

    DPRINT(Debug,8,(&Debug, 
	      "give_sessionlock_fname=%s\n",ret));

    return ret;
}


static void mk_temp_mail_fn(tempfn, mbox, tempfn_size)
     char *tempfn, *mbox;
     int tempfn_size;
{
    /** create in tempfn the name of the temp file corresponding to
	mailfile mbox. Used to session lock also.
    **/

    struct stat    buf;		/* stat command  */
    struct stat  * Xbuf = NULL;
    char * result_name = NULL;
    int L;
    enum syscall_status  r;

    r = stat(mbox,&buf);

    switch(r) {
    case syscall_error /* -1 */: {
	int err UNUSED_VAROK = errno;
	
	DPRINT(Debug,8,(&Debug, 
			"mk_temp_mail_fn: stat %s failed:  (errno=%d) %s\n",
			mbox,err,strerror(err)));
	
    }
	break;
    case syscall_success /* 0 */:
	Xbuf = &buf;
	break;
    }

    result_name = give_sessionlock_fname(mbox,Xbuf);

    if ((L = strlen(result_name)) > tempfn_size-2) {
	DPRINT(Debug,1,(&Debug,
			"mk_temp_mail_fn: Too long path! (len=%d)\n",
			L));
	strfcpy(tempfn,"TEMP_MBOX", tempfn_size);
	return;
    }
  
    strfcpy(tempfn,result_name,tempfn_size);

    free(result_name);
    
    DPRINT(Debug,8,(&Debug, 
		    "mk_temp_mail_fn: tempfname=%s\n",tempfn));
}

S_(mbx_free_folder mbx_free_non_spool)
static int mbx_free_non_spool P_((struct folder_info *folder,
				   struct cancel_data  * cd));
static int mbx_free_non_spool(folder,cd)
     struct folder_info *folder;
     struct cancel_data  * cd;
{
    int ret =1;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_free_non_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		"mbx_free_non_spool: folder=%X (%s)\n",
		folder,folder->cur_folder_sys));    
    
    free_mbx_private_data(&(folder->p));

    DPRINT(Debug,11,(&Debug, 
		     "mbx_free_non_spool=%d\n",ret));
    
    return ret;
}

S_(mbx_free_folder mbx_free_spool)
static int mbx_free_spool P_((struct folder_info *folder,
			       struct cancel_data  * cd
			       ));
static int mbx_free_spool(folder,cd)
     struct folder_info *folder;
     struct cancel_data  * cd;
{
    int ret = 1;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_free_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_free_spool: folder=%X (%s)\n",
		     folder,folder->cur_folder_sys));    

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_free_spool",
	      "Bad magic number (private_data)",0);

    
    if (folder->p->a.spool.lockfile) {
	free(folder->p->a.spool.lockfile);
	folder->p->a.spool.lockfile = NULL;
    }
    
    free_mbx_private_data(& (folder->p));

    DPRINT(Debug,11,(&Debug,
		     "mbx_free_spool=%d\n",ret));
    
    return ret;
}


static int mbx_set_ostat P_((struct folder_info *folder,
			     int *errcode /* errno */));
static int mbx_set_ostat(folder,errcode)
     struct folder_info *folder;
     int *errcode /* errno */;
{
    int ret = 1;

    enum use_lastread_v use_lastread =
	give_dt_enumerate_as_int(&use_last_read_file);

    enum syscall_status  ofolder_code;
    
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_set_ostat",
	      "Bad magic number (folder_info)",0);

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_set_ostat",
	      "Bad magic number (private_data)",0);
	      
    if (!folder->p->fh_folder &&
	use_lastread > use_lastread_no) {

	clearit(folder->p->flags1,FLAG1_have_ostat);
	
	ofolder_code = stat(folder->cur_folder_sys,
			    &(folder->p->ostat));
	
	switch (ofolder_code) {
	case syscall_success /* 0 */: {
	    char * X;
	    
	    setit(folder->p->flags1,FLAG1_have_ostat);

	    DPRINT(Debug,11,(&Debug, 
			     "mbx_set_ostat: folder=%p %s ostat dev/ino=%lu/%lu atime=%ld",
			     folder,
			     folder -> cur_folder_sys,
			     (unsigned long)(folder->p->ostat.st_dev),
			     (unsigned long)(folder->p->ostat.st_ino),
			     (long)folder->p->ostat.st_atime));

	    X = ctime(&(folder->p->ostat.st_atime));

	    /* ctime() includes newline \n to result */
	    if (X) {
		DPRINT(Debug,14,(&Debug," %s",
				 X));
	    } else {
		DPRINT(Debug,14,(&Debug,"\n"));
	    }
	    
	}
	    break;
	case syscall_error /* -1 */: {
	    int err = errno;

	    DPRINT(Debug,11,(&Debug, 
			     "mbx_set_ostat: folder=%p stat %s failed: %s\n",
			     folder,
			     folder -> cur_folder_sys,
			     strerror(err)));
	    
	    if (err != ENOENT) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantOpenFolderRead,
				  "Can't open folder '%S' for reading: %s"), 
			  folder->cur_folder_disp,		  
			  strerror(err));
		
		if (errcode)
		    *errcode = err;
		
		ret = 0;
	    }
	}
	    break;
	}
    } else if (folder->p->fh_folder) {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_set_ostat: folder=%p %s already open",
			 folder,
			 folder -> cur_folder_sys));

	if (ison(folder->p->flags1,FLAG1_have_ostat)) {
	    char * X;
	    DPRINT(Debug,13,(&Debug," ostat dev/ino=%lu/%lu atime=%ld",
			     (unsigned long)(folder->p->ostat.st_dev),
			     (unsigned long)(folder->p->ostat.st_ino),
			     (long)folder->p->ostat.st_atime));
	    X = ctime(&(folder->p->ostat.st_atime));
	    
	    /* ctime() includes newline \n to result */
	    if (X) {
		DPRINT(Debug,14,(&Debug," %s",
				 X));
	    } else {
		DPRINT(Debug,14,(&Debug,"\n"));
	    }
	    
	} else {
	    DPRINT(Debug,14,(&Debug," no ostat\n"));
	}	
    }

    DPRINT(Debug,11,(&Debug, 
		     "mbx_set_ostat=%d folder=%s",
		     ret,folder -> cur_folder_sys));
    if (!ret && errcode && *errcode) {
	DPRINT(Debug,11,(&Debug, " *errcode=%d %s",
			 *errcode,strerror(*errcode)));
    }
    DPRINT(Debug,11,(&Debug, "\n"));
    
    return ret;
}

static int mbx_set_read P_((struct folder_info *folder));
static int mbx_set_read(folder)
     struct folder_info *folder;
{
    int ret = 1;

    enum use_lastread_v use_lastread =
	give_dt_enumerate_as_int(&use_last_read_file);
    
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_set_read",
	      "Bad magic number (folder_info)",0);
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_set_read",
	      "Bad magic number (private_data)",0);
    
    if (folder->p->fh_folder &&
	use_lastread > use_lastread_no) {
	struct stat          folder_stat;
	enum syscall_status  folder_code;
	enum FLAG1_bits x1;

	DPRINT(Debug,13,(&Debug,"mbx_set_read: folder=%p %s",
			 folder,
			 folder->cur_folder_sys ?
			 folder->cur_folder_sys : "(no name)"));

	DPRINT(Debug,13,(&Debug," flags1=%d",
			 folder->p->flags1));
	for (x1 = 0; x1 < FLAG1_bit_count; x1++) {
	    if (ison(folder->p->flags1,x1)) {
		switch (x1) {
		case FLAG1_bit_CHECKNEW:   DPRINT(Debug,13,(&Debug," FLAG1_CHECKNEW"));   break;
		case FLAG1_bit_have_ostat: DPRINT(Debug,13,(&Debug," FLAG1_have_ostat")); break;
		case FLAG1_bit_count:                                                     break;
		}
	    }
	}
	if (ison(folder->p->flags1,FLAG1_have_ostat)) {
	    char * X;
	    DPRINT(Debug,13,(&Debug," ostat dev/ino=%lu/%lu atime=%ld",
			     (unsigned long)(folder->p->ostat.st_dev),
			     (unsigned long)(folder->p->ostat.st_ino),
			     (long)folder->p->ostat.st_atime));
	    X = ctime(&(folder->p->ostat.st_atime));
	    
	    /* ctime() includes newline \n to result */
	    if (X) {
		DPRINT(Debug,14,(&Debug," %s",
				 X));
	    } else {
		DPRINT(Debug,14,(&Debug,"\n"));
	    }

	} else {
	    DPRINT(Debug,14,(&Debug," no ostat\n"));
	}

	
	folder_code = fstat(fileno(folder -> p->fh_folder),
			    &folder_stat);
      	
	switch (folder_code) {
	case syscall_success /* 0 */: {
	    struct last_read_cache * cache =
		last_read_enabled(use_lastread,
				  folder -> cur_folder_sys,
				  ison(folder->p->flags1,FLAG1_have_ostat) ?
				  &(folder->p->ostat) : NULL,
				  &folder_stat);

	    if (cache) {
		mark_last_read(cache,folder -> cur_folder_sys,
			       &folder_stat);
	    
		free_last_read_cache(& cache);
	    }
	}
	    break;
	case syscall_error /* -1 */: {
	    int err = errno;
	    
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_set_read: fstat %d (%s) failed: %s\n",
			     fileno(folder -> p->fh_folder),
			     folder -> cur_folder_sys));
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFolderStatError,
			      "Error encountered while attempting to stat folder %S: %s"), 
		      folder->cur_folder_disp,
		      strerror(err));
	}
	    ret = 0;
	    goto clean;
	}
    }


 clean:

    DPRINT(Debug,11,(&Debug, 
		     "mbx_set_read=%d folder=%s\n",
		     ret,folder -> cur_folder_sys));

    
    return ret;
}

static int mbx_sessionlock_filehandle P_((struct folder_info *folder,
					  enum sessionlock_mode mode,
					  int *errcode /* errno */
					  ));

static int mbx_sessionlock_filehandle(folder,mode,errcode)
     struct folder_info *folder;
     enum sessionlock_mode mode;
     int *errcode /* errno */;
{
    int status = 0;
        
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_filehandle",
	      "Bad magic number (folder_info)",0);

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_filehandle",
	      "Bad magic number (private_data)",0);
    
    if (mode == SESSIONLOCK_CHECK) {
	int code = 0;
	if ((code = can_open(folder->cur_folder_sys, "r+")) != 0 && 
	    code != ENOENT) {
	    DPRINT(Debug,1,(&Debug, 
			    "Error: given file %s as folder - unreadable (%s)!\n", 
			    folder->cur_folder_sys, 
			    strerror(code)));
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantOpenFolderRead,
			      "Can't open folder '%S' for reading: %s"), 
		      folder->cur_folder_disp,
		      strerror(code));
	    status = 0;
	    
	    if (errcode)
		*errcode = code;

	    goto clean;
	}
	DPRINT(Debug,12,(&Debug, 
			 "mbx_sessionlock_filehandle: %s SESSIONLOCK_CHECK OK %s\n",
			 folder->cur_folder_sys,
 			 code ? strerror(code) : ""));
    }

    if (mode == SESSIONLOCK_REOPEN && folder->p->fh_folder) {
	fclose(folder->p->fh_folder);
	folder->p->fh_folder = NULL;
    }

    status = 1;

 clean:
    return status;
}     

static void set_closeonexec_folder P_((struct folder_info *folder));

S_(mbx_sessionlock_folder mbx_sessionlock_read_only)
static enum sessionlock_status mbx_sessionlock_read_only
    P_((struct folder_info *folder,
	enum sessionlock_mode mode,
	int *errcode /* errno */,
	RECONNECT_MODE * reconnect_ptr
	));

static enum sessionlock_status mbx_sessionlock_read_only(folder,mode,errcode,reconnect_ptr)
     struct folder_info *folder;
     enum sessionlock_mode mode;
     int *errcode /* errno */;
     RECONNECT_MODE * reconnect_ptr;
{
    enum sessionlock_status status = sessionlock_fail;
	
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_read_only",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_sessionlock_read_only: folder=%p (%s), mode=%d",
		     folder,folder->cur_folder_sys,mode));
    switch(mode) {
    case SESSIONLOCK_NORMAL:   DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NORMAL")); break;
    case SESSIONLOCK_REOPEN:   DPRINT(Debug,11,(&Debug, " SESSIONLOCK_REOPEN")); break;
    case SESSIONLOCK_TRUNCATE: DPRINT(Debug,11,(&Debug, " SESSIONLOCK_TRUNCATE")); break;
    case SESSIONLOCK_CHECK:    DPRINT(Debug,11,(&Debug, " SESSIONLOCK_CHECK")); break;
    case SESSIONLOCK_NONE:     DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NONE")); break;
    case SESSIONLOCK_RECONNECT: DPRINT(Debug,11,(&Debug, " SESSIONLOCK_RECONNECT")); break;
    case SESSIONLOCK_NONE_CHECKNEW: DPRINT(Debug,11,(&Debug,
						     " SESSIONLOCK_NONE_CHECKNEW")); break;
    }
    DPRINT(Debug,11,(&Debug, "\n"));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_read_only",
	      "Bad magic number (private_data)",0);

    
    if (! mbx_set_ostat(folder,errcode)) {
	status = sessionlock_fail;
	goto clean;
    }
        
    if (mode == SESSIONLOCK_CHECK) {
	if ((errno = can_open(folder->cur_folder_sys, 
			      "r")) != 0) {
	    int err1 = errno;
	    DPRINT(Debug,1,(&Debug, 
			    "Error: given file %s as folder - unreadable (%s)!\n", 
			    folder->cur_folder_sys, 
			    strerror(err1)));
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantOpenFolderRead,
			      "Can't open folder '%S' for reading: %s"), 
		      folder->cur_folder_disp,
		      strerror(err1));
	    if (errcode)
		*errcode = err1;
	    status = sessionlock_fail;
	    goto clean;
	}
    }

    
    if (mode == SESSIONLOCK_REOPEN && folder->p->fh_folder) {
	fclose(folder->p->fh_folder);
	folder->p->fh_folder = NULL;
    }

    if (mode == SESSIONLOCK_TRUNCATE) {
	DPRINT(Debug,10,(&Debug,  
			 "mbx_sessionlock_read_only: mode == SESSIONLOCK_TRUNCATE (ignored) : %s\n",
			 folder->cur_folder_sys));
    }

    if (!folder->p->fh_folder) {	
	folder->p->fh_folder = fopen(folder->cur_folder_sys,"r");
	
	if (!folder->p->fh_folder) {
	    int err = errno;
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailOnOpenNewmbox,
			      "Mailbox %S open failed: %s"),
		      folder->cur_folder_disp,
		      strerror(err));
	    DPRINT(Debug,1,(&Debug, 
			    "fail on open in mbx_sessionlock_read_only, file %s!!\n",
			    folder->cur_folder_sys));
	    status = sessionlock_fail;
	    if (errcode)
		*errcode = err;
	    goto clean;
	}
	DPRINT(Debug,10,(&Debug, 
			 "Mailfile (folder) opened: %s\n",
			 folder->cur_folder_sys));
	set_closeonexec_folder(folder);
    } else
	rewind(folder->p->fh_folder);

    if (mode == SESSIONLOCK_RECONNECT) {
	DPRINT(Debug,10,(&Debug,  
			 "mbx_sessionlock_read_only: mode == SESSIONLOCK_RECONNECT (ignored) : %s\n",
			 folder->cur_folder_sys));
	status = sessionlock_open;
	goto clean;
    } else if (mode == SESSIONLOCK_NONE) {
	status = sessionlock_open;
	goto clean;
    }

    if (lockfolders) {
	switch (GrabRead_the_file(fileno(folder -> p->fh_folder),
				  folder->p->lock_info)) {
	case FLOCKING_OK:
	    DPRINT(Debug,1,(&Debug,  
			    "Folder locked (%s).\n",
			    folder -> cur_folder_sys));
	    break;
	case FLOCKING_RETRY:
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmAlreadyRW,
			      "You seem to have ELM already reading (read-write) this mail!"));	
	    status =  sessionlock_fail;;
	    goto clean;	    

	case	FLOCKING_FAIL:
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveErrorLockFolder,
			      "Error encountered while attempting to lock folder %S: %s"), 
		      folder->cur_folder_disp,
		      strerror(errno));

	    status =  sessionlock_fail;
	    goto clean;
	case FLOCKING_UNSUPPORTED:
	    DPRINT(Debug,1,(&Debug,  
			    "Folder not locked (%s) -- unsupported.\n",
			    folder -> cur_folder_sys));
	    break;
	}
    }

    
    status = sessionlock_open;

 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_sessionlock_read_only=%d",status));
    switch (status) {
    case sessionlock_fail: DPRINT(Debug,11,(&Debug, " sessionlock_fail")); break;
    case sessionlock_open: DPRINT(Debug,11,(&Debug, " sessionlock_open")); break;
    case sessionlock_reconnect: DPRINT(Debug,11,(&Debug, " sessionlock_reconnect")); break;
    }
    DPRINT(Debug,11,(&Debug, "\n"));
    
    return status;
}

static void set_closeonexec_folder(folder)
     struct folder_info *folder;
{
    int r;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"set_closeonexec_folder",
	      "Bad magic number (folder_info)",0);
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"set_closeonexec_folder",
	      "Bad magic number (private_data)",0);

#ifdef FD_CLOEXEC
    r = fcntl(fileno(folder->p->fh_folder), F_SETFD, FD_CLOEXEC);
#else
    r = fcntl(fileno(folder->p->fh_folder), F_SETFD, 1);
#endif    

    if (-1 == r) {
	int err UNUSED_VAROK = errno;

	DPRINT(Debug,1,(&Debug, 
			"Failed to set close-on-exec flag for %s: %s\n",
			folder->cur_folder_sys,
			strerror(err)));
    } else if (0 == r) {
	DPRINT(Debug,11,(&Debug, 
			"Set close-on-exec flag for %s\n",
			 folder->cur_folder_sys));
    }
}

S_(mbx_sessionlock_folder mbx_sessionlock_non_spool)
static enum sessionlock_status mbx_sessionlock_non_spool
   P_((struct folder_info *folder,
       enum sessionlock_mode mode,
       int *errcode /* errno */,
       RECONNECT_MODE * reconnect_ptr));

static enum sessionlock_status mbx_sessionlock_non_spool(folder,mode,errcode,reconnect_ptr)
     struct folder_info *folder;
     enum sessionlock_mode mode;
     int *errcode /* errno */;
     RECONNECT_MODE * reconnect_ptr;
{
    enum sessionlock_status status = sessionlock_fail;
        
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_non_spool",
	      "Bad magic number (folder_info)",0);
	    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_sessionlock_non_spool: folder=%p (%s), mode=%d",
		     folder,folder->cur_folder_sys,mode));
    switch(mode) {
    case SESSIONLOCK_NORMAL:   DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NORMAL")); break;
    case SESSIONLOCK_REOPEN:   DPRINT(Debug,11,(&Debug, " SESSIONLOCK_REOPEN")); break;
    case SESSIONLOCK_TRUNCATE: DPRINT(Debug,11,(&Debug, " SESSIONLOCK_TRUNCATE")); break;
    case SESSIONLOCK_CHECK:    DPRINT(Debug,11,(&Debug, " SESSIONLOCK_CHECK")); break;
    case SESSIONLOCK_NONE:     DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NONE")); break;
    case SESSIONLOCK_RECONNECT: DPRINT(Debug,11,(&Debug, " SESSIONLOCK_RECONNECT")); break;
    case SESSIONLOCK_NONE_CHECKNEW: DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NONE_CHECKNEW")); break;
    }
    DPRINT(Debug,11,(&Debug, "\n"));

    if (! mbx_set_ostat(folder,errcode)) {
	status = sessionlock_fail;
	goto clean;
    }
      
    if (!mbx_sessionlock_filehandle(folder,mode,errcode)) {
	status =  sessionlock_fail;
	goto clean;
    }

    if (mode == SESSIONLOCK_TRUNCATE) {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_sessionlock_non_spool: mode == SESSIONLOCK_TRUNCATE (ignored) : %s\n",
			 folder->cur_folder_sys));
    }

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_non_spool",
	      "Bad magic number (private_data)",0);

    if (!folder->p->fh_folder) {	
	folder->p->fh_folder = fopen(folder->cur_folder_sys,"r+");

	if (!folder->p->fh_folder) {
	    int err = errno;
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailOnOpenNewmbox,
			      "Mailbox %S open failed: %s"),
		      folder->cur_folder_disp,
		      strerror(err));
	    DPRINT(Debug,1,(&Debug, 
			    "fail on open in mbx_sessionlock_non_spool, file %s!!\n",
			    folder->cur_folder_sys));
	    status =  sessionlock_fail;
	    
	    if (errcode)
		*errcode = err;
	    goto clean;
	}
	DPRINT(Debug,10,(&Debug, 
			 "Mailfile (folder) opened: %s\n",
			 folder->cur_folder_sys));
	set_closeonexec_folder(folder);
    } else
	rewind(folder->p->fh_folder);
    
    if (mode == SESSIONLOCK_RECONNECT) {
	DPRINT(Debug,10,(&Debug,  
			 "mbx_sessionlock_non_spool: mode == SESSIONLOCK_RECONNECT (ignored) : %s\n",
			 folder->cur_folder_sys));
	status = sessionlock_open;
	goto clean;
     } else if (mode == SESSIONLOCK_NONE) {
	status = sessionlock_open;
	goto clean;
    }

    if (lockfolders) {
	switch (Grab_the_file(fileno(folder->p->fh_folder),
			      folder->p->lock_info)) {
	case FLOCKING_OK:
	    DPRINT(Debug,1,(&Debug, 
			    "Folder locked (%s).\n",
			    folder -> cur_folder_sys));
	    break;
	case FLOCKING_RETRY:
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmAlreadyRunning1,
			      "You seem to have ELM already reading this mail!"));	
	    status = sessionlock_fail;
	    goto clean;	    

	case	FLOCKING_FAIL:
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveErrorLockFolder,
			      "Error encountered while attempting to lock folder %S: %s"), 
		      folder->cur_folder_sys,
		      strerror(errno));

	    status = sessionlock_fail;
	    goto clean;
	case FLOCKING_UNSUPPORTED:
	    DPRINT(Debug,1,(&Debug, 
			    "Folder not locked (%s) -- unsupported.\n",
			    folder -> cur_folder_sys));
	    break;
	}
    }

   
    status = sessionlock_open;

 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_sessionlock_non_spool=%d\n",status));
    return status;
}

S_(mbx_sessionlock_folder mbx_sessionlock_spool)
static enum sessionlock_status mbx_sessionlock_spool
    P_((struct folder_info *folder,
	enum sessionlock_mode mode,
	int *errcode /* errno */,
	RECONNECT_MODE * reconnect_ptr
	));

static enum sessionlock_status mbx_sessionlock_spool(folder,mode,errcode,reconnect_ptr)
     struct folder_info *folder;
     enum sessionlock_mode mode;
     int *errcode;
     RECONNECT_MODE * reconnect_ptr;
{
    enum sessionlock_status status = sessionlock_fail;
    
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_sessionlock_spool: folder=%p (%s), mode=%d",
		     folder,folder->cur_folder_sys,mode));
    switch(mode) {
    case SESSIONLOCK_NORMAL:   DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NORMAL")); break;
    case SESSIONLOCK_REOPEN:   DPRINT(Debug,11,(&Debug, " SESSIONLOCK_REOPEN")); break;
    case SESSIONLOCK_TRUNCATE: DPRINT(Debug,11,(&Debug, " SESSIONLOCK_TRUNCATE")); break;
    case SESSIONLOCK_CHECK:    DPRINT(Debug,11,(&Debug, " SESSIONLOCK_CHECK")); break;
    case SESSIONLOCK_NONE:     DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NONE")); break;
    case SESSIONLOCK_RECONNECT: DPRINT(Debug,11,(&Debug, " SESSIONLOCK_RECONNECT")); break;
    case SESSIONLOCK_NONE_CHECKNEW: DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NONE_CHECKNEW")); break;
    }
    DPRINT(Debug,11,(&Debug, "\n"));

    if (! mbx_set_ostat(folder,errcode)) {
	status = sessionlock_fail;
	goto clean;
    }
    
    if (!mbx_sessionlock_filehandle(folder,mode,errcode)) {
	status = sessionlock_fail;
	goto clean;
    }

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_spool",
	      "Bad magic number (private_data)",0);

    if (!folder->p->fh_folder) {
	folder->p->fh_folder = fopen(folder->cur_folder_sys,"r+");
	if (!folder->p->fh_folder) {
	    if (errno != ENOENT) {
		int err = errno;
	    
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailOnOpenNewmbox,
				  "Mailbox %S open failed: %s"),
			  folder->cur_folder_disp,
			  strerror(err));
		DPRINT(Debug,1,(&Debug, 
				"fail on open in mbx_sessionlock_spool, file %s!!\n",
				folder->cur_folder_sys));
		status = sessionlock_fail;
		if (errcode)
		    *errcode = err;
		goto clean;
	    } else {
		folder->mailfile_size = 0;    
		/* must be non-existant folder */
	    }
	}
	DPRINT(Debug,10,(&Debug, 
			 "Mailfile (folder) opened: %s %s\n",
			 folder->cur_folder_sys,
			 folder->p->fh_folder ? "" : " (non existant)"));
	if (folder->p->fh_folder)
	    set_closeonexec_folder(folder);
    } else {
	rewind(folder->p->fh_folder);
    }

    switch(mode) {
	int need_reopen;

    case SESSIONLOCK_RECONNECT:
	DPRINT(Debug,10,(&Debug,  
			 "mbx_sessionlock_spool: mode == SESSIONLOCK_RECONNECT (ignored) : %s\n",
			 folder->cur_folder_sys));
	/* FALLTHRU */
    case SESSIONLOCK_NONE:
    case SESSIONLOCK_NONE_CHECKNEW:
	
	status = sessionlock_open;
	goto clean;
    case SESSIONLOCK_TRUNCATE:
	need_reopen = 1;
#ifdef FTRUNCATE  
	if (folder->p->fh_temp) {
	    int r;
	    
	    rewind(folder->p->fh_temp);
	    
	    r = ftruncate(fileno(folder->p->fh_temp),0);
	    if (0 == r) {
		need_reopen = 0;
		DPRINT(Debug,10,(&Debug, 
				 "truncate_tempfolder: tempfile %s truncated (not recreated)\n",
				 folder->cur_tempfolder));
	    } else if (-1 == r) {
		int err = errno;
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmSorryCantTruncateTemp,
				  "Sorry, can't truncate the temp file %s [%s]!"),
			  folder -> cur_tempfolder, strerror(err));
	    }
	    /* IF FAIL REOPEN IT INSTEAD */
	}
#endif
	if (!need_reopen)
	    break;  /* DONE */
	if (folder->p->fh_temp) {
	    fclose (folder->p->fh_temp);
	    folder->p->fh_temp = NULL;
	}

	if (0 != unlink(folder -> cur_tempfolder)) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSorryCantTruncateTemp,
			      "Sorry, can't truncate the temp file %s [%s]!"),
		      folder -> cur_tempfolder, strerror(err));    
	    status =  sessionlock_fail;
	    if (errcode)
		*errcode = err;
	    goto clean;
	}
	goto create_it;
    case   SESSIONLOCK_REOPEN:
	if (folder -> p->fh_temp) {
	    int temp_handle;
	    fclose(folder -> p->fh_temp);
	    folder -> p->fh_temp = NULL;

	    if ( (temp_handle = open(folder-> cur_tempfolder, 
				     O_RDWR,
				     0600)) == -1 ||
		 NULL == (folder -> p->fh_temp = fdopen(temp_handle,"w+"))) {
		int err = errno;
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSorryCantReopenTemp,
				  "Sorry, can't reopen the temp file %s [%s]!"),
			  folder -> cur_tempfolder, strerror(err));  
		
		if (-1 != temp_handle)
		    close(temp_handle);
  
		status =  sessionlock_fail;
		if (errcode)
		    *errcode = err;
		goto clean;
	    } 
	    break;
	}
	/* FALLTHRU */
    case SESSIONLOCK_NORMAL:
    case SESSIONLOCK_CHECK:
    create_it:
	if (!folder -> p->fh_temp) {	    
	    int err = 0;
	    
	    /* If we are changing files and we are changing to a spool file,
	     * make sure there isn't a temp file for it, because if
	     * there is, someone else is using ELM to read the new file,
	     * and we don't want to be reading it at the same time.
	     */

	    if (sessionlock_create_tempfolder(folder,&err)) {

		DPRINT(Debug,1,(&Debug, 
				"Creating tempfile (%s).\n",
				folder -> cur_tempfolder));
		
		elm_chown(folder->cur_tempfolder, userid, groupid,
			  NULL);
		chmod(folder -> cur_tempfolder, 0600);	
		/* shut off file for other people! */	    
		
	    } else {
		status =  sessionlock_fail;
		if (errcode)
		    *errcode = err;
		
		if (err == EEXIST) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmAlreadyRunningX,
				      "You seem to have ELM already reading this mail! You may not have two copies of\n\
ELM running simultaneously. If this is in error, then you'll need to remove \n\
the following file: %s"),
			      folder -> cur_tempfolder);

		    if (POLL_method)
			wait_for_timeout(2 + sleepmsg);
		    else
			sleep(2 + sleepmsg);

		    goto clean;	    
		}

		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedCreateTmpFolder,
				  "Failed to create %.50s: %.60s"),
			  folder -> cur_tempfolder,
			  strerror(err));
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIGiveUp,
				  "Ahhhh... I give up."));
		
		goto clean;	    
	    }
	    
	}
	break;
    default:
	break;
    }

    
    status = sessionlock_open;

 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_sessionlock_spool=%d",status));
    switch (status) {
    case sessionlock_fail: DPRINT(Debug,11,(&Debug, " sessionlock_fail")); break;
    case sessionlock_open: DPRINT(Debug,11,(&Debug, " sessionlock_open")); break;
    case sessionlock_reconnect: DPRINT(Debug,11,(&Debug, " sessionlock_reconnect")); break;
    }
    DPRINT(Debug,11,(&Debug, "\n"));
    
    return status;
}

int mbx_flush_temp(folder,err_p) 
     struct folder_info *folder;
     int *err_p  /* errno value */;
{
    int ret = 0;
    int r;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_flush_temp",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,12,(&Debug, 
		     "mbx_flush_temp: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_flush_temp",
	      "Bad magic number (private_data)",0);

    if (!folder->p->fh_temp) {
	ret = 1;
	
	goto cleanup;
    }
    
#ifdef ultrix
    /** Ultrix returns an error if fflush is called on a file opened
	for read only, so we have to account for this.
    **/
    
    if (folder->p->fh_temp->_flag & (_IOREAD|_IORW) == _IOREAD) {
	ret = 1;
	goto cleanup;
    }
#endif
    r = fflush(folder->p->fh_temp);
    
    if (0 == r) {
	
	ret = 1;
	
    } else if (EOF == r) {
	int err = errno;

	DPRINT(Debug,12,(&Debug, 
			 "mbx_flush_temp:  failed to flush %s: %s (errno=%d)\n",
			 folder->cur_tempfolder,
			  strerror(err),
			 err));
	ret = 0;

	if (err_p)
	    *err_p = err;
    }
    
 cleanup:
     DPRINT(Debug,12,(&Debug, 
		      "mbx_flush_temp=%d\n",
		      ret));
     return ret;    
}

S_(mbx_flush_folder mbx_flush_read_only)
static int mbx_flush_read_only P_((struct folder_info *folder,
				   int *err_p  /* errno value */,
				   struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
								on remote mailbox
							     */));
static int mbx_flush_read_only(folder,err_p,cd) 
     struct folder_info *folder;
     int *err_p  /* errno value */;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  on remote mailbox
			       */;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_flush_read_only",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_flush_read_only=1 (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    return 1;
}

S_(mbx_flush_folder mbx_flush_non_spool)
static int mbx_flush_non_spool P_((struct folder_info *folder,
				   int *err_p  /* errno value */,
				   struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
								on remote mailbox
							     */));
static int mbx_flush_non_spool(folder,err_p,cd) 
     struct folder_info *folder;
     int *err_p  /* errno value */;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  on remote mailbox
			       */;
{
    int ret = 0;
    int r;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_flush_non_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_flush_non_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (!folder->p->fh_folder) {
	ret =  1;
	goto cleanup;
    }

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_flush_non_spool",
	      "Bad magic number (private_data)",0);

#ifdef ultrix
	/** Ultrix returns an error if fflush is called on a file opened
	    for read only, so we have to account for this.
	**/

    if (folder->p->fh_folder->_flag & (_IOREAD|_IORW) == _IOREAD) {
	ret = 1;
	goto cleanup;
    }
#endif
    
    r = fflush(folder->p->fh_folder);

    if (0 == r) {
	ret = 1;
    } else if (EOF == r) {
	int err = errno;

	DPRINT(Debug,11,(&Debug, 
			 "mbx_flush_non_spool: failed to flush %s: %s (errno=%d)\n",
			 folder -> cur_folder_sys,
			 strerror(err),
			 err));
	ret = 0;

	if (err_p)
	    *err_p = err;
    }

 cleanup:
	DPRINT(Debug,11,(&Debug, 
			 "mbx_flush_non_spool=%d\n",
			 ret));
	return ret;
}

S_(mbx_flush_folder mbx_flush_spool)
static int mbx_flush_spool P_((struct folder_info *folder,
			       int *err_p  /* errno value */,
			       struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
							    on remote mailbox
							 */));
static int mbx_flush_spool(folder,err_p,cd) 
     struct folder_info *folder;
     int *err_p  /* errno value */;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  on remote mailbox
			       */;
{
    int ret1, ret2, ret;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_flush_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_flush_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    ret1 = mbx_flush_temp(folder,err_p);
    ret2 = mbx_flush_non_spool(folder,err_p,cd);

    ret = ret1 && ret2 ;

    DPRINT(Debug,11,(&Debug, 
		     "mbx_flush_spool=%d\n",
		     ret));
    return ret;
}

S_(mbx_ferror_folder mbx_ferror_non_spool)
static int mbx_ferror_non_spool P_((struct folder_info *folder, int clean));
static int mbx_ferror_non_spool(folder,clean) 
     struct folder_info *folder; 
     int clean; 
{
    int status;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_ferror_non_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_ferror_non_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_ferror_non_spool",
	      "Bad magic number (private_data)",0);
    
    status = ferror(folder->p->fh_folder);
    if (clean)
	clearerr(folder->p->fh_folder);

    DPRINT(Debug,11,(&Debug, 
		     "mbx_ferror_non_spool=%d\n",status));
    return status;
}

S_(mbx_ferror_folder mbx_ferror_spool)
static int mbx_ferror_spool P_((struct folder_info *folder, int clean));
static int mbx_ferror_spool(folder,clean) 
     struct folder_info *folder; 
     int clean; 
{
    int status;

    DPRINT(Debug,11,(&Debug, 
		     "mbx_ferror_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_ferror_spool",
	      "Bad magic number (private_data)",0);

    if(!folder->p->fh_folder) {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_ferror_spool: non-existent -- no error.\n"));
	status = 0;
    } else {
	status = ferror(folder->p->fh_folder);
	if (clean)
	    clearerr(folder->p->fh_folder);
    }

    if (folder->p->fh_temp) {
	if (ferror(folder->p->fh_temp))
	    status = 1;
	if (clean)
	    clearerr(folder->p->fh_temp);
    }

    DPRINT(Debug,11,(&Debug, 
		     "mbx_ferror_spool=%d\n",status));
    return status;
}


void zero_FILE_rs(rs)
     struct FILE_rs *rs;
{
    DPRINT(Debug,12,(&Debug, "zero_FILE_rs: rs=%p\n",
		     rs));

    rs -> next_line     = NULL;
    rs -> next_line_len = 0;    
}

S_(mbx_zero_rs_fields_folder mbx_zero_rs_fields_file)
static void mbx_zero_rs_fields_file P_((struct read_folder_state *rs));
static void mbx_zero_rs_fields_file(rs)
     struct read_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug, 
		     "mbx_zero_rs_fields_file: rs=%p\n",
		     rs));

    zero_FILE_rs(& (rs -> a.file));
}

void clear_FILE_rs(rs)
     struct FILE_rs *rs;
{
    DPRINT(Debug,12,(&Debug, "clear_FILE_rs: rs=%p\n",
		     rs));

    if (rs -> next_line) {
	DPRINT(Debug,50,(&Debug, "clear_FILE_rs: next_line="));

	DEBUG_PRINT_BUFFER(Debug,50,rs -> next_line_len,
			   s2us(rs -> next_line));

	if (rs -> next_line_len < 1 || ! (rs -> next_line) ||
	    (rs -> next_line)[rs -> next_line_len -1] != '\n') {
	    DPRINT(Debug,50,(&Debug, "\nclear_FILE_rs: NO NEWLINE\n"));
	}
	
	free(rs -> next_line);
	rs -> next_line = NULL;
    }
    rs -> next_line_len = 0;        
}

S_(mbx_free_rs_fields_folder mbx_free_rs_fields_file)
static void mbx_free_rs_fields_file P_((struct read_folder_state *rs));
static void mbx_free_rs_fields_file(rs)
     struct read_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug, 
		     "mbx_free_rs_fields_file: rs=%p\n",
		     rs));

    clear_FILE_rs(& (rs -> a.file));    
}

static int mbx_prepare_seek_folder P_((struct folder_info *folder,
				       int add_new_only,
				       READ_STATE read_state_ptr));
static int mbx_prepare_seek_folder(folder,add_new_only,read_state_ptr)
     struct folder_info *folder;
     int add_new_only;
     READ_STATE read_state_ptr;
{
    int status = 0;
    long result;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_seek_folder",
	      "Bad magic number (folder_info)",0);
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_seek_folder",
	      "Bad magic number (private_data)",0);

    if (add_new_only) {
	enum syscall_status r =
	    fseek(folder->p->fh_folder, 
		  folder->mailfile_size, SEEK_SET);

	switch(r) {
	case syscall_error /* -1 */: {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldntSeekEndFolder,
			      "Couldn't seek to %ld (end of folder) in %S: %s"),
		      folder-> mailfile_size, 
		      folder-> cur_folder_disp,   
		      strerror(err));

	     DPRINT(Debug,12,(&Debug,
			      "mbx_prepare_seek_folder: Failed to seek %s to end of file, offset %ld: %s (errno=%d)\n",
			      folder -> cur_folder_sys,
			      folder->mailfile_size,
			      strerror(err),
			      err));
	}
	    status = 0;
	    goto clean;
	case syscall_success /* 0 */: {
	    DPRINT(Debug,12,(&Debug,
			     "mbx_prepare_seek_folder: Seeked %s to end of file, offset %ld\n",
			     folder -> cur_folder_sys,
			     folder->mailfile_size));
	}
	    break;
	}
	
	/* read_state_ptr->count  = message_count; */	  /* next available  */
	read_state_ptr->fbytes = folder-> mailfile_size;  /* start correctly */
    }

    result = file_bytes(folder->cur_folder_sys,NULL);
    if (-1L == result) {
	status = 0;
	goto clean;
    }
    folder -> mailfile_size = result;

    status = 1;

 clean:
     DPRINT(Debug,12,(&Debug,
		      "mbx_prepare_seek_folder=%d\n",
		      status));
    return status;
}

S_(mbx_prepare_read_folder mbx_prepare_read_non_spool)
static enum prepare_result mbx_prepare_read_non_spool P_((struct folder_info *folder,
					  enum prepare_mode mode,
					  READ_STATE read_state_ptr,
					  RECONNECT_MODE reconnect_ptr
					  ));
static enum prepare_result mbx_prepare_read_non_spool(folder,mode,read_state_ptr,reconnect_ptr)
     struct folder_info *folder;
     enum prepare_mode mode;
     READ_STATE read_state_ptr;
     RECONNECT_MODE reconnect_ptr;
{
    enum prepare_result status = prepare_fail;
    int add_new_only = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_read_non_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_prepare_read_non_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                          : read_state_ptr=%p\n",
		     read_state_ptr));

    switch (mode) {       	
    case PREPARE_NEW_ONLY:
    case PREPARE_NEW_ONLY_NOLOCK:
    case PREPARE_ACCESS:
	add_new_only = 1;
	DPRINT(Debug,11,(&Debug,
			 "mbx_prepare_read_non_spool: Add new only\n"));
	break;
    case PREPARE_RECONNECT:	
    case PREPARE_NORMAL:
    case PREPARE_NOLOCK:
    	if (reconnect_ptr) {
	    if (RECONNECT_MODE_magic != reconnect_ptr->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_read_non_spool",
		      "Bad magic number (reconnect_mode)",0);

	    DPRINT(Debug,11,(&Debug,
			     "mbx_prepare_read_non_spool: Ignoring reconnect mode\n"));
	    
	}
	break;
    }
    
    if (!mbx_prepare_seek_folder(folder, add_new_only, read_state_ptr)) {
	status = prepare_fail;
	goto clean;
    }
    status = prepare_done;

 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_prepare_read_non_spool=%d",status));
    switch (status) {
    case prepare_fail:  DPRINT(Debug,11,(&Debug, " prepare_fail")); break;
    case prepare_done:  DPRINT(Debug,11,(&Debug, " prepare_done")); break;
    case prepare_reconnected: DPRINT(Debug,11,(&Debug, " prepare_reconnected")); break;
    }   
    DPRINT(Debug,11,(&Debug, "\n"));
    
    return status;
}

S_(mbx_prepare_read_folder mbx_prepare_read_spool)
static enum prepare_result mbx_prepare_read_spool P_((struct folder_info *folder,
						      enum prepare_mode mode,
						      READ_STATE read_state_ptr,
						      RECONNECT_MODE reconnect_ptr
						      ));
static enum prepare_result mbx_prepare_read_spool(folder,mode,read_state_ptr,reconnect_ptr)
     struct folder_info *folder;
     enum prepare_mode mode;
     READ_STATE read_state_ptr;
     RECONNECT_MODE reconnect_ptr;
{
    enum prepare_result status = prepare_fail;
    int add_new_only = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_read_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_prepare_read_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                      : read_state_ptr=%p\n",
		     read_state_ptr));

    switch (mode) {       	
    case PREPARE_NEW_ONLY:
    case PREPARE_NEW_ONLY_NOLOCK:
    case PREPARE_ACCESS:
	add_new_only = 1;
	DPRINT(Debug,11,(&Debug,
			 "mbx_prepare_read_spool: Add new only\n"));
	break;
    case PREPARE_RECONNECT:	
    case PREPARE_NORMAL:
    case PREPARE_NOLOCK:
    	if (reconnect_ptr) {
	    if (RECONNECT_MODE_magic != reconnect_ptr->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_read_spool",
		      "Bad magic number (reconnect_mode)",0);

	    DPRINT(Debug,11,(&Debug,
			     "mbx_prepare_read_spool: Ignoring reconnect mode\n"));
	    
	}
	break;
    }

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_read_spool",
	      "Bad magic number (private_data)",0);
        
    if (folder -> p->a.spool.lock_state == ON) {
	DPRINT(Debug,1,(&Debug, 
			"mbx_prepare_read_spool: %s already locked.\n",
			folder->cur_folder_sys));
    } else {
	switch (mode) {
	case PREPARE_NOLOCK:
	case PREPARE_NEW_ONLY_NOLOCK:
	case PREPARE_ACCESS:
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_prepare_read_spool: mode == PREPARE_NOLOCK (or PREPARE_ACCESS)\n"));
	    break;
	default:
	    /* ensure no mail arrives while we do this! */
	    if (!lock_folder(LOCK_INCOMING, folder)) {
		status = prepare_fail;
		goto clean;
	    }
	    break;
	}
    }


    if (add_new_only) {
	if (!folder->p->fh_temp) {
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_prepare_read_spool -- NO temp file %s\n",
			     folder->cur_tempfolder));
	} else {
	    enum syscall_status  r =
		fseek(folder -> p->fh_temp, 
		      folder -> mailfile_size, SEEK_SET);

	    switch(r) {
	    case syscall_error /* -1 */: {
		int err = errno;
		
		unlock_folder(0,folder);	/* remove lock file! */
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldntSeekTempEnd,
				  "\nCouldn't fseek to end of temp mbox.\n"));
		lib_error(FRM("** %s. **\n"), strerror(err));

		DPRINT(Debug,11,(&Debug,
				 "mbx_prepare_read_spool: Failed to seek temp %s to end of file, offset %ld: %s (errno=%d)\n",
				 folder->cur_tempfolder,
				 folder -> mailfile_size,
				 strerror(err),
				 err));
	    }
		status = prepare_fail;
		goto clean;
	    case syscall_success /* 0 */:  {
		DPRINT(Debug,11,(&Debug,
				 "mbx_prepare_read_spool: Seeked temp %s to end of file, offset %ld\n",
				 folder->cur_tempfolder,
				 folder -> mailfile_size));
	    }
		break;
	    }
	}
    }

    if (!mbx_prepare_seek_folder(folder,add_new_only,read_state_ptr)) {
	status = prepare_fail;
	goto clean;
    }
    status = prepare_done;

 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_prepare_read_spool=%d",status));
    switch (status) {
    case prepare_fail:  DPRINT(Debug,11,(&Debug, " prepare_fail")); break;
    case prepare_done:  DPRINT(Debug,11,(&Debug, " prepare_done")); break;
    case prepare_reconnected: DPRINT(Debug,11,(&Debug, " prepare_reconnected")); break;
    }   
    DPRINT(Debug,11,(&Debug, "\n"));
    return status;
}

S_(mbx_end_read_folder mbx_end_read_non_spool)
static int mbx_end_read_non_spool P_((struct folder_info *folder,
				      READ_STATE read_state_ptr,
				      RECONNECT_MODE reconnect_ptr,
				      int silent));

static int mbx_end_read_non_spool(folder,read_state_ptr,reconnect_ptr,silent)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     RECONNECT_MODE reconnect_ptr;
     int silent;
{
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_read_non_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_end_read_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                  : read_state_ptr=%p\n",
		     read_state_ptr));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_read_non_spool",
	      "Bad magic number (private_data)",0);

    rewind(folder->p->fh_folder);

    status = mbx_set_read(folder);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_end_read_non_spool=%d\n",status));
    return status;
}

S_(mbx_end_read_folder mbx_end_read_spool)
static int mbx_end_read_spool P_((struct folder_info *folder,
				  READ_STATE read_state_ptr,
				  RECONNECT_MODE reconnect_ptr,
				  int silent));

static int mbx_end_read_spool(folder,read_state_ptr,reconnect_ptr,silent)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     RECONNECT_MODE reconnect_ptr;
     int silent;
{
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_read_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_end_read_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                  : read_state_ptr=%p\n",
		     read_state_ptr));

    if (PREPARE_NOLOCK == read_state_ptr->mode ||
	PREPARE_NEW_ONLY_NOLOCK == read_state_ptr->mode ||
	PREPARE_ACCESS == read_state_ptr->mode) {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_end_read_spool: PREPARE_NOLOCK == mode (or PREPARE_ACCESS)\n"));
    } else 
	unlock_folder(0, folder);

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_read_spool",
	      "Bad magic number (private_data)",0);

    if (!folder->p->fh_temp) {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_end_read_spool=1 -- NO temp file %s\n",
			 folder->cur_tempfolder));
	return 1;
    }

    if ((ferror(folder->p->fh_temp)) || 
	(fflush(folder->p->fh_temp) == EOF)) {
	if (!silent) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFlushOnTempFailed,
			      "Flush on temp file %s failed: %s"), 
		      folder -> cur_tempfolder,
		      strerror(err));
	}
	DPRINT(Debug,1,(&Debug, 
		   "Can't flush on temp file %s!!\n",
		   folder -> cur_tempfolder));
	status = 0;
	goto clean;
    }
	 
    /* sanity check on append - is resulting temp file longer??? */
    if (file_bytes(folder-> cur_tempfolder,NULL) != 
	folder-> mailfile_size) {
	if (!silent) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLengthNESpool,
			      "\nnewmbox - length of mbox. != spool mailbox length!!\n"));
	}
	DPRINT(Debug,1,(&Debug,  
			"newmbox - mbox. != spool mail length"));
	
    }
	  
    rewind(folder->p->fh_temp);
    status = mbx_set_read(folder);
    
 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_end_read_spool=%d\n",status));
    return status;
}



/* Calculate maximun line length to be read 
 *
 *  Actually 1000 (+2 for CRLF) is limit for mail format lines 
 */
int mbx_max_line_read(size,offset)
     long size;   /* or end of block ofset */
     long offset;
{
#define MBX_MAX_READ (10 * 1024 * 1024)

    int result = MBX_MAX_READ;          /*  must be < INT_MAX, but to keep
					 *  memory allocation on mbx_read_line
					 * limited, use  10 MB as tail value
					 * (10485760, INT_MAX is assumed to be
					 *  2147483647 at least)
					 */
    
    static long limit_message = -1L;

    if (offset > size) {
	result = 0;
	DPRINT(Debug,10,(&Debug, "mbx_max_line_read=%d: offset=%ld > size=%ld !!!\n",
			 result,offset,size));
	
	limit_message = size;
	
    } else if (size > MBX_MAX_READ) {
	
	if (size - MBX_MAX_READ > offset) {
	    result = MBX_MAX_READ;

	    if (limit_message != size) {

		DPRINT(Debug,15,(&Debug, "mbx_max_line_read=%d limiting: offset=%ld, size=%ld\n",
				 result,offset,size));
		limit_message = size;
	    }

	} else  {
	    result = size - offset;
	    limit_message = -1L;
	}

    } else  {
	result = size - offset;
	limit_message = -1L;
    }
    
    return result;
}

int mbx_read_line(fh,buffer,len,max_read)
     FILE *fh;
     char **buffer; 
     int *len; 
     int max_read;
{
    int       LEN = 0;
    char * BUFFER = NULL;
    int alloced = 1024;
    int ch;
    int ret = 1;

    *buffer = NULL;
    *len    = 0;

    if (max_read < 1)
	return 0;

    if (alloced > max_read+1) 
	alloced = max_read+1;

    BUFFER = safe_malloc(alloced);
    
    while (LEN < max_read) {
	ch = getc(fh); /* Macro, faster than  fgetc() ! */

	if (ch == EOF) {
	    if (0 == LEN) {
		ret = 0;
		DPRINT(Debug,65,(&Debug, "mbx_read_line: Got EOF !!!\n"));

		free(BUFFER);
		BUFFER = NULL;
		goto out;
	    }
	    break;
	}

	if (LEN+1 > alloced) {
	    int n = alloced + 1024;

	    if (n > max_read+1)
		n = max_read+1;

	    BUFFER = safe_realloc(BUFFER,n);
	    alloced = n;
	}
	BUFFER[LEN++] = ch;

	if (ch == '\n')
	    break;
    }
   
    if (LEN == max_read) {
	DPRINT(Debug,65,(&Debug, "mbx_read_line: max_read=%d (readed)\n",max_read));
    }

    BUFFER = safe_realloc(BUFFER,LEN+1);
    BUFFER[LEN] = '\0';
    
 out:

    *len    = LEN;
    *buffer = BUFFER;


    DPRINT(Debug,65,(&Debug, 
		     "mbx_read_line=%d: len=%d, buffer=%.*s",
		     ret,
		     *len,*len,(*buffer) ? (*buffer) : ""));
    if (*len < 1 || !(*buffer) || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,65,(&Debug, 
			 "\nmbx_read_line: NO NEWLINE\n"));
    } 
    return ret;
}

static int mbx_read_buffered_line P_((struct folder_info *folder,
				      READ_STATE read_state_ptr,
				      char **buffer, int *len));
static int mbx_read_buffered_line(folder,read_state_ptr,buffer,len)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
{
    long f = -1L;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_read_buffered_line",
	      "Bad magic number (folder_info)",0);
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_read_buffered_line",
	      "Bad magic number (private_data)",0);

    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_read_buffered_line",
	      "Bad magic number (read state)",0);
    
    if(read_state_ptr ->a.file.next_line) {
	*buffer = read_state_ptr -> a.file.next_line;
	*len    = read_state_ptr -> a.file.next_line_len;
    } else {
	f = ftell(folder->p->fh_folder);
		  
	if (!mbx_read_line(folder->p->fh_folder,buffer,len,
			   mbx_max_line_read(folder->mailfile_size,
					     read_state_ptr->fbytes)))
	    return 0;

	read_state_ptr -> a.file.next_line      = *buffer;
	read_state_ptr -> a.file.next_line_len  = *len;
	
    }
    
    DPRINT(Debug,60,(&Debug, 
		     "mbx_read_buffered_line: "));
    if (f >= 0L) {
	DPRINT(Debug,60,(&Debug, "offset=%ld, ",
			 f));
	if (f != read_state_ptr->fbytes) {
	    DPRINT(Debug,60,(&Debug, "BUT fbytes=%ld, ",
			     read_state_ptr->fbytes));
	}
    }
    DPRINT(Debug,60,(&Debug, "len=%d, buffer=",
		     *len));
    DEBUG_PRINT_BUFFER(Debug,60,*len,s2us(*buffer));
    if (*len < 1 || !(*buffer) || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,60,(&Debug, 
			 "\nmbx_read_buffered_line: NO NEWLINE\n"));
    } 

    return 1;
}

static void mbx_accept_buffered_line P_((struct folder_info *folder,
					 READ_STATE read_state_ptr,
					 char **buffer, int *len));
static void mbx_accept_buffered_line(folder,read_state_ptr,buffer,len)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_accept_buffered_line",
	      "Bad magic number (folder_info)",0);

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_accept_buffered_line",
	      "Bad magic number (private_data)",0);

    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_accept_buffered_line",
	      "Bad magic number (read state)",0);

    if (*buffer != read_state_ptr->a.file.next_line)
       	panic("MBX PANIC",__FILE__,__LINE__,"mbx_accept_buffered_line",
	      "Bad buffer!",0);

    if (!*buffer)
	return;

    if (*len < 1 || (*buffer)[*len -1] != '\n') {
    } else {
	read_state_ptr -> linecounter++;
    }
    read_state_ptr -> a.file.next_line     = NULL;
    read_state_ptr -> fbytes       += read_state_ptr -> a.file.next_line_len;
    read_state_ptr -> a.file.next_line_len = 0;

    free(*buffer);
    *buffer = NULL;
    *len    = 0;
}

static int mbx_copy_envelope_check_MMDF(buffer,len)
     char *buffer;
     int len; 
{
    if (strcmp(buffer, MSG_SEPARATOR) == 0)
	return 1;
    return 0;
}

S_(mbx_copy_envelope_folder mbx_copy_envelope_non_spool)
static enum copy_env_status mbx_copy_envelope_non_spool 
    P_((struct folder_info *folder,
	READ_STATE read_state_ptr,
	struct header_rec *entry,
	int force));
static enum copy_env_status mbx_copy_envelope_non_spool(folder,read_state_ptr,
							entry,force)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     struct header_rec *entry;
     int force;
{
    char * buffer = NULL;
    int  len;

    /* NO COPY */
    enum copy_env_status status = copy_env_eof;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_non_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_non_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                           : read_state_ptr=%p\n",
		     read_state_ptr));

    /* !!! */
    if (PREPARE_ACCESS == read_state_ptr->mode) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_non_spool",
	      "Skipping not supported!!",0);


    read_state_ptr -> linecounter   = 0;      /* Linecounter of current
					         message */
    entry->offset = read_state_ptr -> fbytes; /* Offset of current message */

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_non_spool",
	      "Bad magic number (private_data)",0);

    if (ison(folder->p->flags1,FLAG1_CHECKNEW)) {
	/* Really treat as mailbox and not as folder ... */
	entry->status  = VISIBLE | NEW | UNREAD; /* Default status of message    */

    } else {
	enum folder_status_v de = 
	    give_dt_enumerate_as_int(&def_folder_status);

	entry->status = VISIBLE | UNREAD;         /* Default status of message */

	switch(de) {
	case FOLDER_STATUS_READ: 
	    entry->status = VISIBLE; 
	    break;
	case FOLDER_STATUS_OLD:
	    entry->status = VISIBLE | UNREAD;
	    break;
	case FOLDER_STATUS_NEW:
	    entry->status = VISIBLE | NEW | UNREAD;
	    break;
	default:
	    DPRINT(Debug,1,(&Debug,
			    "default-folder-status=%d BAD\n",
			    de));
	}
	DPRINT(Debug,11,(&Debug, "           status=%x -- def_folder_status=%d\n",
			 entry->status,entry->status));
    }

    if (!mbx_read_buffered_line(folder,read_state_ptr,
				     &buffer,&len)) {
	status = copy_env_eof;
	goto clean;
    }
    if (0 == len) {
	status = copy_env_eof;
	goto clean;
    }

    if (have_MMDF) {
	if (mbx_copy_envelope_check_MMDF(buffer,len)) {
	    mbx_accept_buffered_line(folder,read_state_ptr,
				     &buffer,&len);
	    if (!mbx_read_buffered_line(folder,read_state_ptr,
					&buffer,&len)) {
		status = copy_env_eof;
		goto clean;
	    }	
	    status = copy_env_ok;
	} else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFolderCorrupt,
			      "Folder is corrupt!!  I can't read it!!"));
	    status = copy_env_format;
	    goto clean;
	}
    }

    if (real_from(buffer,entry)) {

	DPRINT(Debug,12,(&Debug, "mbx_copy_envelope_non_spool: "));
	if (entry->env_from[0]) {
	    DPRINT(Debug,12,(&Debug," env from %s",
			     entry->env_from));
	} else {
	    DPRINT(Debug,12,(&Debug,", no env from"));
	}
	DPRINT(Debug,12,(&Debug, "\n"));
	
	mbx_accept_buffered_line(folder,read_state_ptr,
				      &buffer,&len);
	status = copy_env_ok;

	while (mbx_read_buffered_line(folder,read_state_ptr,
					   &buffer,&len) &&
	       first_word_nc(buffer,">From ")) {
	    forwarded(buffer+6, entry); /* return address */
	    mbx_accept_buffered_line(folder,read_state_ptr,
					  &buffer,&len);	    
	}

    } else if (copy_env_eof == status) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFolderCorrupt,
			  "Folder is corrupt!!  I can't read it!!"));
	status = copy_env_format;
    }

 clean:

    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_non_spool=%d\n",
		     status));

#if 0
    if (RF_magic != read_state_ptr -> magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_non_spool",
	      "Bad magic number (read state)",0);
#endif

    return status;
}


int mbx_copy_line_to_temp(folder,buffer,len)
     struct folder_info *folder;
     char *buffer; 
     int len;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_line_to_temp",
	      "Bad magic number (folder_info)",0);

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_line_to_temp",
	      "Bad magic number (private_data)",0);

    if (!folder->p->fh_temp) {
	DPRINT(Debug,15,(&Debug, 
			 "mbx_copy_line_to_temp=1 -- NO temp file %s\n",
			 folder->cur_tempfolder));
	return 1;
    }

    if (fwrite(buffer, 1, len, folder->p->fh_temp) != len) {
	int err = errno;

	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmWriteToTempFailed,
			  "\nWrite to temp file %s failed!!\n"),
		  folder->cur_tempfolder);       
	lib_error(FRM("** %s. **\n"), strerror(err));

	DPRINT(Debug,1,(&Debug, 
		   "Can't write to temp file %s!!\n",
		   folder->cur_tempfolder));
	return 0;
    }
    DPRINT(Debug,15,(&Debug, 
		     "mbx_copy_line_to_temp=1 -> %.*s",
		     len,buffer));
    if (len < 1 || buffer[len-1] != '\n') {
	DPRINT(Debug,15,(&Debug, 
			 "\nmbx_copy_line_to_temp <-- NO NEWLINE\n"));
    }

    return 1;
}

S_(mbx_copy_envelope_folder mbx_copy_envelope_spool)
static enum copy_env_status mbx_copy_envelope_spool 
    P_((struct folder_info *folder,
	READ_STATE read_state_ptr,
	struct header_rec *entry,
	int force));
static enum copy_env_status mbx_copy_envelope_spool(folder,read_state_ptr,
						    entry,force)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     struct header_rec *entry;
     int force;
{
    char * buffer = NULL;
    int  len;

    /* COPY */
    enum copy_env_status status = copy_env_eof;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                       : read_state_ptr=%p\n",
		     read_state_ptr));

    /* !!! */
    if (PREPARE_ACCESS == read_state_ptr->mode) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_spool",
	      "Skipping not supported!!",0);

    read_state_ptr -> linecounter   = 0;   /* Linecounter of current message */
    entry->status  = VISIBLE | NEW | UNREAD; /* Default status of message    */

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_spool",
	      "Bad magic number (private_data)",0);
    
    if (!folder->p->fh_folder) {
	DPRINT(Debug,11,(&Debug, 
			 "                      : folder not open...\n"));
	status = copy_env_eof;
	goto clean;
    } 

    do {
	entry->offset = read_state_ptr -> fbytes; 
	/* Offset of current message */

	if (!mbx_read_buffered_line(folder,read_state_ptr,
					 &buffer,&len)) {
	    status = copy_env_eof;
	    goto clean;
	}
	if (0 == len) {
	    status = copy_env_eof;
	    goto clean;
	}
		
	if ((len == 1 && 0 == memcmp(buffer,"\n",len)) ||
	    (len == 2 && 0 == memcmp(buffer,"\r\n",len))) {
	    if (!mbx_copy_line_to_temp(folder,buffer,len)) {
		status = copy_env_eof;
		goto clean;
	    }

	    mbx_accept_buffered_line(folder,read_state_ptr,
					  &buffer,&len);
	} else
	    break;
    } while (1); 

    if (have_MMDF) {
	if (mbx_copy_envelope_check_MMDF(buffer,len)) {
	    if (!mbx_copy_line_to_temp(folder,buffer,len)) {
		status = copy_env_eof;
		goto clean;
	    }
	    mbx_accept_buffered_line(folder,read_state_ptr,
				     &buffer,&len);
	    if (!mbx_read_buffered_line(folder,read_state_ptr,
					&buffer,&len)) {
		status = copy_env_eof;
		goto clean;
	    }	
	    status = copy_env_ok;
	} else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFolderCorrupt,
			      "Folder is corrupt!!  I can't read it!!"));
	    status = copy_env_format;
	    goto clean;
	}
    }

    if (real_from(buffer,entry)) {

	DPRINT(Debug,12,(&Debug, "mbx_copy_envelope_spool: "));
	if (entry->env_from[0]) {
	    DPRINT(Debug,12,(&Debug," env from %s",
			     entry->env_from));
	} else {
	    DPRINT(Debug,12,(&Debug,", no env from"));
	}
	DPRINT(Debug,12,(&Debug, "\n"));

	
	if (!mbx_copy_line_to_temp(folder,buffer,len)) {
	    status = copy_env_eof;
	    goto clean;
	}

	mbx_accept_buffered_line(folder,read_state_ptr,
				      &buffer,&len);
	status = copy_env_ok;

	while (mbx_read_buffered_line(folder,read_state_ptr,
					   &buffer,&len) &&
	       first_word_nc(buffer,">From ")) {
	    forwarded(buffer+6, entry); /* return address */

	    if (!mbx_copy_line_to_temp(folder,buffer,len)) {
		status = 0;
		goto clean;
	    }
	    mbx_accept_buffered_line(folder,read_state_ptr,
					  &buffer,&len);	    
	}

    } else if (copy_env_eof == status) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFolderCorrupt,
			  "Folder is corrupt!!  I can't read it!!"));
	status = copy_env_format;
    }


 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_spool=%d\n",
		     status));
    return status;
}

S_(mbx_is_forwarded_folder mbx_is_forwarded_spool)
static const char * mbx_is_forwarded_spool P_((struct folder_info *folder,
					      READ_STATE read_state_ptr));
static const char * mbx_is_forwarded_spool(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    char * buffer = NULL;
    const char * status = NULL; 
    int  len;

    DPRINT(Debug,11,(&Debug, 
		     "mbx_is_forwarded_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                      : read_state_ptr=%p\n",
		     read_state_ptr));

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_is_forwarded_spool",
	      "Bad magic number (folder_info)",0);
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_is_forwarded_spool",
	      "Bad magic number (private_data)",0);

    if (!folder->p->fh_folder) {
	DPRINT(Debug,11,(&Debug, 
			 "                      : folder not open...\n"));
    } else {
	if (!mbx_read_buffered_line(folder,read_state_ptr,
				    &buffer,&len)) {
	    status = NULL;
	    goto clean;
	}

	if (len > 11 && first_word_nc(buffer, "forward to "))
	    status = buffer + 11;
    }

 clean:

    DPRINT(Debug,11,(&Debug, 
		     "mbx_is_forwarded_spool=%s\n",
		     status ? status : "NULL"));
    return status;
}

void append_buffer(buffer,len,buffer1,len1)
     char **buffer; 
     int *len;
     char *buffer1; 
     int len1;
{
    int res = *len + len1 + 1;

    *buffer = safe_realloc(*buffer,res);

    /* bcopy is defined on hdrs/defs.h */
    bcopy(buffer1,(*buffer)+(*len),len1); 
    (*buffer)[(*len) + len1] = '\0';
    (*len) += len1;
}

S_(mbx_copy_header_folder mbx_copy_header_non_spool)
static int mbx_copy_header_non_spool P_((struct folder_info *folder,
				       READ_STATE read_state_ptr,
				       char **buffer, int *len));
static int mbx_copy_header_non_spool(folder,read_state_ptr,buffer,len)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
{
    char *buffer1 = NULL;
    int len1;
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_header_non_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_header_non_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                         : read_state_ptr=%p\n",
		     read_state_ptr));

    if (!mbx_read_buffered_line(folder,read_state_ptr,
				     &buffer1,&len1)) {
	status = 0;
	goto clean;
    }

    if ((len1 == 1 && 0 == memcmp(buffer1,"\n",len1)) ||
	(len1 == 2 && 0 == memcmp(buffer1,"\r\n",len1))) {
	/* End of headers */
	mbx_accept_buffered_line(folder,read_state_ptr,
				      &buffer1,&len1);
	status = 0;
	goto clean;
    }
    
    if (NULL == memchr(buffer1,':',len1) ||
	( have_MMDF &&
	  mbx_copy_envelope_check_MMDF(buffer1,len1) 
	  ) ||
	(0 == memcmp(buffer1,"From ",5) && real_from(buffer1,NULL))) {
	/* End of headers -- bad header */
	status = 0;
	goto clean;
    }

    do {
	append_buffer(buffer,len,buffer1,len1);
	mbx_accept_buffered_line(folder,read_state_ptr,
				 &buffer1,&len1);
	if (!mbx_read_buffered_line(folder,read_state_ptr,
					 &buffer1,&len1)) {
	    break;
	}
    } while (len1 > 0 && whitespace(buffer1[0]));
    status = 1;

 clean:
    /* Not need to free() -- buffer1 is same than next_line -buffer */

    if (!status) {
	if (*buffer) {
	    free(*buffer);
	    *buffer = NULL;
	}
    }

    DPRINT(Debug,50,(&Debug, 
		     "mbx_copy_header_non_spool=%d | len=%d, buffer=",
		     status,*len));
    DEBUG_PRINT_BUFFER(Debug,50,*len,s2us(*buffer));
    if (*len < 1 || !*buffer || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,50,(&Debug, 
		    "\nmbx_copy_header_non_spool <- NO NEWLINE\n"));
    }
    return status;
}

S_(mbx_copy_header_folder mbx_copy_header_spool)
static int mbx_copy_header_spool P_((struct folder_info *folder,
				     READ_STATE read_state_ptr,
				     char **buffer, int *len));
static int mbx_copy_header_spool(folder,read_state_ptr,buffer,len)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
{
    char *buffer1 = NULL;
    int len1;

    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_header_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_header_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                     : read_state_ptr=%p\n",
		     read_state_ptr));

    if (!mbx_read_buffered_line(folder,read_state_ptr,
				     &buffer1,&len1)) {
	status = 0;
	goto clean;
    }

    if ((len1 == 1 && 0 == memcmp(buffer1,"\n",len1)) ||
	(len1 == 2 && 0 == memcmp(buffer1,"\r\n",len1))) {
	/* End of headers */
	if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
	    status = 0;
	    goto clean;
	}
	mbx_accept_buffered_line(folder,read_state_ptr,
				      &buffer1,&len1);
	status = 0;
	goto clean;
    }
    
    if (NULL == memchr(buffer1,':',len1) ||
	( have_MMDF &&
	  mbx_copy_envelope_check_MMDF(buffer1,len1) 
	  ) ||
	(0 == memcmp(buffer1,"From ",5) && real_from(buffer1,NULL))) {
	/* End of headers -- bad header */
	status = 0;
	goto clean;
    }

    do {
	append_buffer(buffer,len,buffer1,len1);
	if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
	    status = 0;
	    goto clean;
	}
	mbx_accept_buffered_line(folder,read_state_ptr,
				      &buffer1,&len1);
	if (!mbx_read_buffered_line(folder,read_state_ptr,
					 &buffer1,&len1)) {
	    break;
	}
    } while (len1 > 0 && whitespace(buffer1[0]));
    status = 1;

 clean:
    /* Not need to free() -- buffer1 is same than next_line -buffer */

    if (!status) {
	if (*buffer) {
	    free(*buffer);
	    *buffer = NULL;
	}
    }

    DPRINT(Debug,50,(&Debug, 
		     "mbx_copy_header_spool=%d | len=%d, buffer=",
		     status,*len));
    DEBUG_PRINT_BUFFER(Debug,50,*len,s2us(*buffer));
    if (*len < 1 || !*buffer || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,50,(&Debug, 
			 "\nmbx_copy_header_spool <- NO NEWLINE\n"));
    }
    return status;
}

static int copy_body_common P_((struct folder_info *folder,
				READ_STATE read_state_ptr,
				char **buf, int *len,
				long *content_remaining));
static int copy_body_common(folder,read_state_ptr,buf,len,
			    content_remaining)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buf; 
     int *len;
     long *content_remaining;
{
    char *buffer1 = *buf;
    int len1 = *len;
    int r = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"copy_body_common",
	      "Bad magic number (folder_info)",0);
    
    if (!mbx_read_buffered_line(folder,read_state_ptr,
				     &buffer1,&len1)) {
	r = 0;
	goto clean;
    }

    if (*content_remaining < 1 && 
	(
	 ( have_MMDF &&
	   mbx_copy_envelope_check_MMDF(buffer1,len1)
	   )
	 ||
	 ( !have_MMDF &&
	   len1 > 5 && 'F' == buffer1[0] &&
	   0 == memcmp(buffer1,"From ",5) &&
	   real_from(buffer1,NULL)
	   )
	 )
	) {
	/* End of body */
	r = 0;
	goto clean;
    }

    r = 1;

 clean:
    *buf = buffer1;
    *len = len1;
    return r;
}

void adjust_content_remaining(content_remaining,len)
     long *content_remaining;
     int len;
{
    if (*content_remaining > 0 ) {
	long  L = *content_remaining;

	if (L > (long)len)
	    L -= (long)len;
	else if (L > 0) {
	    DPRINT(Debug,11,(&Debug, 
			     "adjust_content_remaining: End of content"));

	    if ((long)len > L) {
		DPRINT(Debug,11,(&Debug, ", overflow %ld bytes",
				 (long)len-L));
	    } else if ((long)len == L) {
		DPRINT(Debug,11,(&Debug, ", correct content-length"));
	    }
	    DPRINT(Debug,11,(&Debug, "\n"));

	    L = 0;
	} else {
	    DPRINT(Debug,11,(&Debug, 
			     "adjust_content_remaining: overflow %d bytes\n",
			     len));
	}

	*content_remaining = L;
    }    
}

S_(mbx_copy_body_folder mbx_copy_body_non_spool)
static int mbx_copy_body_non_spool P_((struct folder_info *folder,
				       READ_STATE read_state_ptr,
				       char **buffer, int *len,
				       long *content_remaining));


static int mbx_copy_body_non_spool(folder,read_state_ptr,buffer,len,
				   content_remaining)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
     long *content_remaining;
{
    char *buffer1;
    int len1;

    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_body_non_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_body_non_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                       : read_state_ptr=%p\n",
		     read_state_ptr));


    if (!copy_body_common(folder,read_state_ptr,
			  &buffer1,&len1,content_remaining)) {
	status = 0;
	goto clean;
    }


    append_buffer(buffer,len,buffer1,len1);
    adjust_content_remaining(content_remaining,len1);

    mbx_accept_buffered_line(folder,read_state_ptr,
			     &buffer1,&len1);
    
    status = 1;
    
 clean:
    /* buffer1 is same then next_line -buffer -- not need free() */

    if (!status) {
	if (*buffer) {
	    free(*buffer);
	    *buffer = NULL;
	}
    }

    DPRINT(Debug,50,(&Debug, 
		     "mbx_copy_body_non_spool=%d | remaining %ld, len=%d, buffer=",
		     status,*content_remaining,*len));
    DEBUG_PRINT_BUFFER(Debug,50,*len,s2us(*buffer));
    if (*len < 1 || !*buffer || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,50,(&Debug, 
			 "\nmbx_copy_body_non_spool <- NO NEWLINE\n"));
    }
    return status;
}

S_( mbx_copy_body_folder mbx_copy_body_spool)
static int mbx_copy_body_spool P_((struct folder_info *folder,
				   READ_STATE read_state_ptr,
				   char **buffer, int *len,
				   long *content_remaining));

static int mbx_copy_body_spool(folder,read_state_ptr,buffer,len,
				   content_remaining)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
     long *content_remaining;
{
    char *buffer1;
    int len1;
 
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_body_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_body_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                   : read_state_ptr=%p\n",
		     read_state_ptr));

    if (!copy_body_common(folder,read_state_ptr,
			  &buffer1,&len1,content_remaining)) {
	status = 0;
	goto clean;
    }

    append_buffer(buffer,len,buffer1,len1);
    if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
	status = 0;
	goto clean;
    }
    adjust_content_remaining(content_remaining,len1);
    mbx_accept_buffered_line(folder,read_state_ptr,
			     &buffer1,&len1);
    
    status = 1;

 clean:
    /* buffer1 is same then next_line -buffer -- not need free() */

    if (!status) {
	if (*buffer) {
	    free(*buffer);
	    *buffer = NULL;
	}
    }

    DPRINT(Debug,50,(&Debug, 
		     "mbx_copy_body_spool=%d | remaining %ld, len=%d, buffer=",
		     status,*content_remaining,*len));
    DEBUG_PRINT_BUFFER(Debug,50,*len,s2us(*buffer));

    if (*len < 1 || !*buffer || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,50,(&Debug, 
			 "\nmbx_copy_body_spool <- NO NEWLINE\n"));
    }
    return status;
}

static int mbx_copy_envelope_end_from P_((struct folder_info *folder,
					  READ_STATE read_state_ptr));

static int mbx_copy_envelope_end_from(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    char *buffer1;
    int len1;
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_end_from",
	      "Bad magic number (folder_info)",0);
    
    /* Will be re-handled on copy_envelope_folder() */
    if (!mbx_read_buffered_line(folder,read_state_ptr,
				     &buffer1,&len1)) {
	status = 1;
	goto clean;
    }
    if (real_from(buffer1,NULL))
	status = 1;
    else
	status = 0;
 clean:
    DPRINT(Debug,12,(&Debug, 
		     "mbx_copy_envelope_end_from=%d\n",
		     status));
    return status;
}

S_(mbx_copy_envelope_end_folder  mbx_copy_envelope_end_non_spool)
static enum copy_env_end_status mbx_copy_envelope_end_non_spool
      P_((struct folder_info *folder,
	  READ_STATE read_state_ptr,
	  long *newbytes,
	  int  *newmails));

static enum copy_env_end_status mbx_copy_envelope_end_non_spool(folder,read_state_ptr,
								newbytes,newmails)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     long *newbytes;
     int  *newmails;
{
    char *buffer1;
    int len1;

    enum copy_env_end_status status = copy_env_end_mismatch;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_end_non_spool",
	      "Bad magic number (folder_info)",0);

    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_end_non_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                               : read_state_ptr=%p\n",
		     read_state_ptr));


    if (!mbx_read_buffered_line(folder,read_state_ptr,
				     &buffer1,&len1)) {
	status = copy_env_end_match;
	goto clean;
    }

    if (
	( have_MMDF &&
	  mbx_copy_envelope_check_MMDF(buffer1,len1) 
	  )
	||
	( !have_MMDF &&
	  ( (len1 == 1 && 0 == memcmp(buffer1,"\n",len1)) ||
	    (len1 == 2 && 0 == memcmp(buffer1,"\r\n",len1))
	    )
	  )
	) {
	mbx_accept_buffered_line(folder,read_state_ptr,
				      &buffer1,&len1);
	status = copy_env_end_match;
    }

    if (!have_MMDF) {
	/* Will be re-handled on copy_envelope_folder() */
	if (mbx_copy_envelope_end_from(folder,read_state_ptr))
	    status = copy_env_end_match;
	else
	    status = copy_env_end_mismatch;
    }

 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_end_non_spool=%d",
		     status));
    switch (status) {
    case copy_env_end_failure:  DPRINT(Debug,10,(&Debug," copy_env_end_failure"));  break;
    case copy_env_end_mismatch: DPRINT(Debug,10,(&Debug," copy_env_end_mismatch")); break;
    case copy_env_end_match:    DPRINT(Debug,10,(&Debug," copy_env_end_match"));    break;
    case copy_env_end_newmail:  DPRINT(Debug,10,(&Debug," copy_env_end_newmail"));  break;
    }
    if (newbytes) {
	DPRINT(Debug,11,(&Debug," *newbytes=%ld",
			 *newbytes));
    }
    if (newmails) {
	DPRINT(Debug,11,(&Debug," *newmails=%d",
			 *newmails));    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return status; 
}

S_(mbx_copy_envelope_end_folder mbx_copy_envelope_end_spool)
static enum copy_env_end_status mbx_copy_envelope_end_spool P_((struct folder_info *folder,
					   READ_STATE read_state_ptr,
					   long *newbytes,
					   int  *newmails
					   ));

static enum copy_env_end_status mbx_copy_envelope_end_spool(folder,read_state_ptr,
							    newbytes,newmails)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     long *newbytes;
     int  *newmails;
{
    char *buffer1;
    int len1;

    enum copy_env_end_status status =  copy_env_end_mismatch;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_end_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		"mbx_copy_envelope_end_spool: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                           : read_state_ptr=%p\n",
		     read_state_ptr));


    if (!mbx_read_buffered_line(folder,read_state_ptr,
				     &buffer1,&len1)) {
	status = copy_env_end_match;
	goto clean;
    }

    if (
	( have_MMDF &&
	  mbx_copy_envelope_check_MMDF(buffer1,len1) 
	  )
	||
	( !have_MMDF &&
	  (
	   (len1 == 1 && 0 == memcmp(buffer1,"\n",len1)) ||
	   (len1 == 2 && 0 == memcmp(buffer1,"\r\n",len1))
	   )
	  )
	) {
       	
	if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
	    status = copy_env_end_failure;
	    goto clean;
	}
	mbx_accept_buffered_line(folder,read_state_ptr,
				      &buffer1,&len1);
	status = copy_env_end_match;
    }

    if (!have_MMDF) {
	/* Will be re-handled on copy_envelope_folder() */
	if (mbx_copy_envelope_end_from(folder,read_state_ptr))
	    status = copy_env_end_match;
	else
	    status = copy_env_end_mismatch;
    }

 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_end_spool=%d",
		     status));

    switch (status) {
    case copy_env_end_failure:  DPRINT(Debug,10,(&Debug," copy_env_end_failure"));  break;
    case copy_env_end_mismatch: DPRINT(Debug,10,(&Debug," copy_env_end_mismatch")); break;
    case copy_env_end_match:    DPRINT(Debug,10,(&Debug," copy_env_end_match"));    break;
    case copy_env_end_newmail:  DPRINT(Debug,10,(&Debug," copy_env_end_newmail"));  break;
    }
    if (newbytes) {
	DPRINT(Debug,11,(&Debug," *newbytes=%ld",
			 *newbytes));
    }
    if (newmails) {
	DPRINT(Debug,11,(&Debug," *newmails=%d",
			 *newmails));    }
    DPRINT(Debug,10,(&Debug,"\n"));

    return status; 
}


static int mbx_reset_folder P_((struct folder_info *folder,
				READ_STATE read_state_ptr));

static int mbx_reset_folder(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    int status = 0;
    enum syscall_status  r;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_reset_folder",
	      "Bad magic number (folder_info)",0);

    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_reset_folder",
	      "Bad magic number (private_data)",0);

    r = fseek(folder->p->fh_folder, read_state_ptr -> fbytes_body, 
	      SEEK_SET);

    switch(r) {
    case syscall_error /* -1 */: {
	int err = errno;

	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmCouldntSeekBytesIntoFolder,
			  "\nCouldn't seek %ld bytes into folder.\n"),
		  read_state_ptr -> fbytes_body);		
	lib_error(FRM("** %s. **\n"), strerror(err));

	DPRINT(Debug,12,(&Debug,
			 "mbx_reset_folder: Failed to seek %s to %ld: %s (errno=%d)\n",
			 folder->cur_folder_sys,
			 read_state_ptr -> fbytes_body,
			 strerror(err),err));
    }
	status = 0;
	
	goto clean;
    case syscall_success /* 0 */: {
	DPRINT(Debug,12,(&Debug,
			 "mbx_reset_folder: Seeked %s to %ld: %s\n",
			 folder->cur_folder_sys,
			 read_state_ptr -> fbytes_body));
    }
	break;
    }
    
    read_state_ptr -> fbytes = read_state_ptr -> fbytes_body;
    mbx_free_rs_fields_file(read_state_ptr);
    status = 1;

 clean:
    DPRINT(Debug,12,(&Debug, 
		     "mbx_reset_folder=%d\n",
		     status));
    return status;
}

S_(mbx_copy_envelope_reset_body mbx_copy_envelope_reset_body_non_spool)
static int mbx_copy_envelope_reset_body_non_spool P_((struct folder_info 
						      *folder,
						      READ_STATE
						      read_state_ptr));
static int mbx_copy_envelope_reset_body_non_spool(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_reset_body_non_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_reset_body_non_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                                      : read_state_ptr=%p\n",
		     read_state_ptr));

    status = mbx_reset_folder(folder,read_state_ptr);

    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_reset_body_non_spool=%d\n",status));
    return status;
}

S_(mbx_copy_envelope_reset_body mbx_copy_envelope_reset_body_spool)
static int mbx_copy_envelope_reset_body_spool P_((struct folder_info 
						  *folder,
						  READ_STATE
						  read_state_ptr));
static int mbx_copy_envelope_reset_body_spool(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_reset_body_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_reset_body_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                                  : read_state_ptr=%p\n",
		     read_state_ptr));


    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "mbx_copy_envelope_reset_body_spool",
	      "Bad magic number (private_data)",0);
    
    if (!folder->p->fh_temp) {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_copy_envelope_reset_body_spool -- NO temp file %s\n",
			 folder->cur_tempfolder));
    } else {
	enum syscall_status  r =
	    fseek(folder->p->fh_temp, read_state_ptr -> fbytes_body , 
		  SEEK_SET);

	switch(r) {
	case syscall_error /* -1 */: {
	    int err = errno;
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmCouldntSeekBytesIntoFolder,
			      "\nCouldn't seek %ld bytes into folder.\n"),
		      read_state_ptr -> fbytes_body);		
	    lib_error(FRM("** %s. **\n"), strerror(err));

	    DPRINT(Debug,11,(&Debug,
			     "mbx_copy_envelope_reset_body_spool: Failed to seek %s to %ld: %s (errno=%d)\n",
			     folder->cur_tempfolder,
			     read_state_ptr -> fbytes_body,
			     strerror(err),
			     err));
	}
	    status = 0;

	    goto clean;
	case syscall_success /* 0 */: {
	    DPRINT(Debug,11,(&Debug,
			     "mbx_copy_envelope_reset_body_spool: Seeked %s to %ld\n",
			     folder->cur_tempfolder,
			     read_state_ptr -> fbytes_body));
	}
	    break;
	}
    }

    status = mbx_reset_folder(folder,read_state_ptr);

 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_reset_body_spool=%d\n",status));
    return status;
}

S_(mbx_folder_to_fd mbx_non_spool_to_fd)
static FILE * mbx_non_spool_to_fd P_((struct folder_info *folder,long offset));
static FILE * mbx_non_spool_to_fd(folder,offset)
     struct folder_info *folder;
     long offset;
{
    FILE * status = NULL;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_non_spool_to_fd",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_non_spool_to_fd: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "mbx_non_spool_to_fd",
	      "Bad magic number (private_data)",0);

    if (offset != -1L) {
	enum syscall_status  r =
	    fseek(folder->p->fh_folder, offset , 
		  SEEK_SET);

	switch(r) {
	case syscall_error /* -1 */: {
	    int err = errno;
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmCouldntSeekBytesIntoFolder,
			      "\nCouldn't seek %ld bytes into folder.\n"),
		      offset);		
	    lib_error(FRM("** %s. **\n"), strerror(err));

	     DPRINT(Debug,11,(&Debug,
			      "mbx_non_spool_to_fd: Failed to seek %s to %ld: %s (errno=%d)\n",
			      folder -> cur_folder_sys,
			      offset,
			      strerror(err),
			      err));
	}
	    status = NULL;
	    goto clean;
	case syscall_success /* 0 */: {
	    DPRINT(Debug,11,(&Debug,
			     "mbx_non_spool_to_fd: Seeked %s to %ld\n",
			     folder -> cur_folder_sys,
			     offset));
	}
	    break;
	}
    }
	
    status = folder->p->fh_folder;

 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_non_spool_to_fd=%s (%p)\n",
		     status ? "NON-NULL" : "NULL",
		     status));
    return status;
}

S_(mbx_folder_to_fd mbx_spool_to_fd)
static FILE * mbx_spool_to_fd P_((struct folder_info *folder,long offset));
static FILE * mbx_spool_to_fd(folder,offset)
     struct folder_info *folder;
     long offset;
{
    FILE * status = NULL;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_spool_to_fd",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_spool_to_fd: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "mbx_spool_to_fd",
	      "Bad magic number (private_data)",0);

    if (offset != -1L) {
	enum syscall_status  r = 
	    fseek(folder->p->fh_temp, offset , 
		  SEEK_SET);

	switch(r) {
	case syscall_error /* -1 */: {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmCouldntSeekBytesIntoFolder,
			      "\nCouldn't seek %ld bytes into folder.\n"),
		      offset);		
	    lib_error(FRM("** %s. **\n"), strerror(err));

	    DPRINT(Debug,11,(&Debug,
			     "mbx_spool_to_fd: Failed to seek %s to %ld: %s (errno=%d)\n",
			     folder->cur_tempfolder,
			     offset,
			     strerror(err),
			     err));
	}
	    status = NULL;
	    goto clean;
	case syscall_success /* 0 */: {
	    DPRINT(Debug,11,(&Debug,
			     "mbx_spool_to_fd: Seeked %s to %ld\n",
			     folder->cur_tempfolder,
			     offset));
	}
	    break;
	}
    }
	
    status = folder->p->fh_temp;

 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_spool_to_fd=%s (%p)\n",
		     status ? "NON-NULL" : "NULL",
		     status));
    return status;
}

S_(mbx_new_mail_on_folder mbx_new_mail_on_normal)
static enum new_mail_stat mbx_new_mail_on_normal P_((struct folder_info *folder,
				      long *bytes,int *new_mailcount,
				      int *err_p  /* errno value */,
				      struct cancel_data  * cd
				      /* Allow cancelation (Ctrl-C)
					 check of remeote mailbox
				      */
				      ));
static enum new_mail_stat mbx_new_mail_on_normal(folder,bytes,new_mailcount,err_p,cd)
     struct folder_info *folder;
     long *bytes;
     int  *new_mailcount /* -1 for local mailbox */;
     int *err_p  /* errno value */;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  check of remeote mailbox
			       */;
{
    enum new_mail_stat status = no_new_mail;
    long size = 0L;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_new_mail_on_normal",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_new_mail_on_normal: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

     if (err_p)
	 *err_p = 0;
     if (bytes)
	 *bytes = 0;
     if (new_mailcount)
	 *new_mailcount = 0;
    
     size = file_bytes(folder->cur_folder_sys,err_p);
     
     if (-1 == size)
	 status = new_mail_check_failed;
     else {
	 if (size > folder->mailfile_size)
	     status = have_new_mail;
	 if (bytes) {
	     *bytes = size - folder->mailfile_size;
	 }
	 if (new_mailcount) {
	     *new_mailcount = -1;
	 }
     }
     
     DPRINT(Debug,11,(&Debug,
		      "mbx_new_mail_on_normal=%d",
		      status));
     switch (status) {
     case failed_mbx_ign_new_mail: DPRINT(Debug,11,(&Debug," failed_mbx_ign_new_mail")); break;
     case new_mail_check_failed:   DPRINT(Debug,11,(&Debug," new_mail_check_failed"));   break;
     case no_new_mail:             DPRINT(Debug,11,(&Debug," no_new_mail"));             break;
     case have_new_mail:           DPRINT(Debug,11,(&Debug," have_new_mail"));           break;
     case new_mail_reconnect:      DPRINT(Debug,11,(&Debug," new_mail_reconnect"));      break;
     }
     if (bytes) {
	 DPRINT(Debug,11,(&Debug, " *bytes=%ld",
			  *bytes));
     }
     if (new_mailcount) {
	 DPRINT(Debug,11,(&Debug, " *new_mailcount=%d",
			  *new_mailcount));
     }
     if (err_p) {
	 DPRINT(Debug,11,(&Debug, " *err_p=%d",
			  *err_p));
	 if (*err_p) {
	     DPRINT(Debug,11,(&Debug, " (%s)",
			      strerror(*err_p)));
	 }
     }
     DPRINT(Debug,11,(&Debug, "\n"));
     
     return status;
}

S_(mbx_consider_remove_folder mbx_consider_remove_non_spool)
static int mbx_consider_remove_non_spool P_((struct folder_info *folder));
static int mbx_consider_remove_non_spool(folder)
     struct folder_info *folder;
{
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_consider_remove_non_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_consider_remove_non_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (0L == file_bytes(folder->cur_folder_sys,NULL)) {
	int r = unlink(folder->cur_folder_sys);
		
	if (0 == r)
	    status = 1;
	else if (-1 == r) {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,11,(&Debug,
			     "mbx_consider_remove_non_spool: unlink(%s) failed: %s (errno=%d)\n",
			     folder->cur_folder_sys,
			     strerror(err),err));
	}
	folder->mailfile_size = 0;
    }
    DPRINT(Debug,11,(&Debug,
		     "mbx_consider_remove_non_spool=%d\n",
		     status));
    return status;
}

S_(mbx_zero_ks_fields_folder mbx_zero_ks_fields_file)
static void mbx_zero_ks_fields_file P_((struct keep_folder_state *rs));
static void mbx_zero_ks_fields_file(rs)
     struct keep_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_zero_ks_fields_file: rs=%p\n",
		     rs));

    rs -> a.file.temp_keep_file       = NULL;
    rs -> a.file.keep_file            = NULL;
}

S_(mbx_free_ks_fields_folder mbx_free_ks_fields_file)
static void mbx_free_ks_fields_file P_((struct keep_folder_state *rs));
static void mbx_free_ks_fields_file(rs)
     struct keep_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_free_ks_fields_file: rs=%p\n",
		     rs));

    if (rs -> a.file.temp_keep_file) {
	free(rs -> a.file.temp_keep_file);
	rs -> a.file.temp_keep_file = NULL;
    }

    if (rs -> a.file.keep_file) {
	fclose(rs -> a.file.keep_file);
	rs -> a.file.keep_file = NULL;
    }
}


static int mbx_open_temp_file P_((struct folder_info *folder,
				  KEEP_STATE keep_state_ptr));
static int mbx_open_temp_file(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
{
    int status = 0,err;
    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    
    if (!tmp)
	tmp = "/tmp/";

    keep_state_ptr -> a.file.temp_keep_file = 
	elm_message(FRM("%s%s%d"),tmp, temp_file, getpid());

    if ((err = can_open(keep_state_ptr -> a.file.temp_keep_file, "sw"))) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveTempFileDenied,
			  "Permission to create temp file %s for writing denied! Leaving folder intact."),
		  keep_state_ptr -> a.file.temp_keep_file);

	DPRINT(Debug,1,(&Debug,
			"Error: Permission to create temp file %s denied!! (%s)\n",
			keep_state_ptr -> a.file.temp_keep_file, 
			"mbx_open_temp_file"));
	DPRINT(Debug,1,(&Debug,
			"** %s ** (errno %d)\n", 
			strerror(err),err));
	status = 0;
	goto clean;
    }

    if ((keep_state_ptr -> a.file.keep_file = 
	 fopen(keep_state_ptr -> a.file.temp_keep_file,"w+")) == NULL) {
	err = errno;
	
	DPRINT(Debug,1,(&Debug,
			"Error: could not create file %s\n", 
			keep_state_ptr ->a.file.temp_keep_file));
	DPRINT(Debug,1,(&Debug,
			"** %s **\n", strerror(err)));
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveCouldNotCreate,
			  "Could not create temp file %s!"), 
		  keep_state_ptr -> a.file.temp_keep_file);

	status = 0;
	goto clean;
    }
    status = 1;

 clean:	
    return status;
}

S_(mbx_prepare_keep_folder mbx_prepare_keep_read_only)
static int mbx_prepare_keep_read_only P_((struct folder_info *folder,
				       KEEP_STATE keep_state_ptr));
static int mbx_prepare_keep_read_only(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
{

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_keep_read_only",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_prepare_keep_read_only=0 (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return 0;
}

S_(mbx_prepare_keep_folder mbx_prepare_keep_normal)
static int mbx_prepare_keep_normal P_((struct folder_info *folder,
				       KEEP_STATE keep_state_ptr));
static int mbx_prepare_keep_normal(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
{
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_keep_normal",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_prepare_keep_normal: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (!mbx_open_temp_file(folder,keep_state_ptr)) {
	status = 0;
	goto clean;
    }
    status = 1;

 clean:
    DPRINT(Debug,11,(&Debug,
		     "mbx_prepare_keep_normal=%d\n",
		     status));
    return status;
}


static int copy_to_folder P_((FILE * from, struct folder_info *to));
static int copy_to_folder(from, to) 
     FILE * from;
     struct folder_info *to;
{
    int need_reopen = 1;
    int truncate_fh = -1;
    int code;

    if (FOLDER_INFO_magic != to->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"copy_to_folder",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "copy_to_folder: folder=%p (%s)\n",
		     to,to->cur_folder_sys));

    if (PRIVATE_DATA_magic != to->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "copy_to_folder",
	      "Bad magic number (private_data)",0);
    
    rewind(to->p->fh_folder);
#ifdef FTRUNCATE
    {
	int r = ftruncate(fileno(to->p->fh_folder),0);
	if (0 == r) {
	    DPRINT(Debug,10,(&Debug,
			 "copy_to_folder: truncated (ftruncate)\n"));
	    need_reopen = 0;
	} else if (-1 == r) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSorryCantTruncateFolder,
			      "Sorry, can't truncate the folder %S [%s]!"),
		      to -> cur_folder_disp, strerror(err));    
	}
    }
#endif
  if (need_reopen) {
      truncate_fh = open(to->cur_folder_sys,O_RDWR|O_TRUNC,0600);
      if (truncate_fh == -1) {
	  int err = errno;
	  
	  lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSorryCantTruncateFolder,
			    "Sorry, can't truncate the folder %S [%s]!"),
		    to -> cur_folder_disp, strerror(err));    
	  return 1;
      }
      DPRINT(Debug,10,(&Debug,
		       "copy_to_folder- truncated (reopened)\n"));
  }
  code = copy_to_fh(from,to->p->fh_folder);
  
  /* In this point we may lost locks,
     but copy is already done */
  if (truncate_fh != -1)
      close(truncate_fh);

  DPRINT(Debug,10,(&Debug,
		   "copy_to_folder=%d\n",code));
  return code;
}


static int mbx_move_temp_file P_((struct folder_info *folder,
				  KEEP_STATE keep_state_ptr,
				  int need_to_copy));
static int mbx_move_temp_file(folder,keep_state_ptr,need_to_copy)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
     int need_to_copy;
{
    int status = 0;
    int was_symlink = FALSE;

    struct stat    buf;		/* stat command  */
    int   have_stat = 0;
    enum syscall_status  r;
    
#ifdef SYMLINK
    struct stat    lbuf;		/* lstat command  */
    int         have_lbuf = 0;
#endif

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_move_temp_file",
	      "Bad magic number (folder_info)",0);
    
    if (fflush(keep_state_ptr->a.file.keep_file) == EOF ) {
	int err = errno;
	
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveFlushFailedTemp,
			  "Flush failed in leavembox: %s: %s"),
		  keep_state_ptr->a.file.temp_keep_file,
		  strerror(err));
	DPRINT(Debug,2,(&Debug,
			"\n\rfflush err on temp_keep_file - mbx_move_temp_file\n\r"));
	unlink(keep_state_ptr->a.file.temp_keep_file);
	status = 0;
	goto clean;
    }

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "mbx_move_temp_file",
	      "Bad magic number (private_data)",0);

    if (ferror(folder->p->fh_folder)) {
	DPRINT(Debug,1,(&Debug,
			"Failed to read mailfile!"));

	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorReadMailfile,
			  "Error when reading mailfile %S"),
		  folder->cur_folder_disp);

	clearerr(folder->p->fh_folder);
	unlink(keep_state_ptr->a.file.temp_keep_file);
	status = 0;
	goto clean;
    }
    
    DPRINT(Debug,10,(&Debug,
		     "!! Need copy (or link) from %s to %s\n",
		     keep_state_ptr->a.file.temp_keep_file, 
		     folder->cur_folder_sys));	    

    /* Get original permissions and access time of the original
     * mail folder before we remove it.
     */
    r = save_file_stats(folder->cur_folder_sys,&buf);
    switch (r) {
    case syscall_error /* -1 */: {
	int err UNUSED_VAROK = errno;
	DPRINT(Debug,1,(&Debug,
			"Error: errno %d (%s) attempting to stat file %s\n", 
			err,strerror(err), 
			folder->cur_folder_sys));

	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmPermFolder,
			  "Problems saving permissions of folder %S!"), 
		  folder->cur_folder_disp);
	
    }
	break;
    case syscall_success /* 0 */:
	have_stat = 1;
	break;
    }
	
#ifdef SYMLINK
    r = lstat(folder->cur_folder_sys, &lbuf);

    switch(r) {
    case syscall_success /* 0 */:
	have_lbuf = 1;
	break;
    case syscall_error /* -1 */: {
	int err = errno;
	DPRINT(Debug,1,(&Debug,
			"Error: errno %d (%s) attempting to stat file %s\n", 
			err,strerror(err), 
			folder->cur_folder_sys));
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveErrorStat,
			  "Error stat %S: %s"), 
		  folder->cur_folder_disp,
		  strerror(err));
    }
	break;
    }
#endif

    if (!flush_folder(folder,NULL,NULL)) {
	DPRINT(Debug,12,(&Debug,
			 "mbx_move_temp_file: Failed to flush folder\n"));
	
	status = 0;
	goto clean;
    }

    if (have_stat) {

	if (buf.st_nlink > 1) {
	    DPRINT(Debug,10,(&Debug,
			     "* several links\n"));
	    need_to_copy = TRUE;
	}
	if (buf.st_mode & 07000) { /* copy if special modes set */
	    DPRINT(Debug,10,(&Debug,
			     "* special modes\n"));
	    need_to_copy = TRUE;    /* such as enforcement lock */
	}
    }
#ifdef SYMLINK
    if ( have_lbuf &&

#ifdef S_ISLNK
	S_ISLNK(lbuf.st_mode)
#else
	(lbuf.st_mode & S_IFMT) == S_IFLNK
#endif
	) {
	DPRINT(Debug,10,(&Debug,
			 "* symlink\n"));
	need_to_copy = TRUE;
	was_symlink = TRUE;
    }
#endif

#ifdef _PC_CHOWN_RESTRICTED

    if (!need_to_copy) {
/*
 * Chown may or may not be restricted to root in SVR4, if it is,
 *	then need to copy must be true, and no restore of permissions
 *	should be performed.
 */
	if (0 != pathconf(folder->cur_folder_sys, 
			  _PC_CHOWN_RESTRICTED)) {
	    DPRINT(Debug,10,(&Debug,
			     "* _PC_CHOWN_RESTRICTED\n"));
	    
	    need_to_copy = TRUE;
	}
    }
#endif  /* _PC_CHOWN_RESTRICTED */

    DPRINT(Debug,10,(&Debug,
		     "-- need_to_copy=%d\n",
		     need_to_copy));

    if(!need_to_copy) {
	r = unlink(folder->cur_folder_sys);
	switch (r) {
	case syscall_success /* 0 */:
	    DPRINT(Debug,10,(&Debug,
			     "!! unlinked folder: %s\n",
			     folder->cur_folder_sys));
	    break;
	case syscall_error /* -1 */:  {
	    int err UNUSED_VAROK = errno;

	    DPRINT(Debug,10,(&Debug,
			     "mbx_move_temp_file: unlink %s: (errno=%d) %s\n",
			     folder->cur_folder_sys, err,strerror(err)));
	}
	    break;
	}
	    	    
	r = link(keep_state_ptr->a.file.temp_keep_file, 
		 folder->cur_folder_sys);
	
	switch(r) {
	case syscall_error /* -1 */: {	    
	    int err = errno;
	    
	    if(err == EXDEV || err == EEXIST) {

		/* oops - can't link across file systems - 
		   use copy instead */
		
		DPRINT(Debug,10,(&Debug,
				 "link(%s, %s) failed: %s (errno %d)\n", 
				 keep_state_ptr->a.file.temp_keep_file, 
				 folder->cur_folder_sys,
				 strerror(err),err));		
		need_to_copy = TRUE;
	    } else {

		DPRINT(Debug,1,(&Debug,
				"link(%s, %s) failed (leavembox)\n", 
				keep_state_ptr->a.file.temp_keep_file, 
				folder->cur_folder_sys));
		DPRINT(Debug,1,(&Debug, 
				"** %s ** (errno %d)\n", 
				strerror(err),err));
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveLinkFailed,
				  "Link failed! %s.\n"), 
			  strerror(err));
		need_to_copy = TRUE;
	    }
	}
	    break;
	case syscall_success /* 0 */: {
	    DPRINT(Debug,19,(&Debug,
			     "!! linked %s to %s\n", 
			     keep_state_ptr->a.file.temp_keep_file, 
			     folder->cur_folder_sys));
	    status = 1;
	}
	    break;
	}

	if (folder->p->fh_folder)
	    fclose(folder->p->fh_folder);	
	folder->p->fh_folder = NULL;
	
	folder->p->fh_folder = open_or_create(folder->cur_folder_sys);
	    
	if (NULL == folder->p->fh_folder) {
	    goto PANIC_copy_failed;
	}
	DPRINT(Debug,10,(&Debug,
			 "!! reopened (created) current folder after unlink: %s\n",
			 folder->cur_folder_sys));	        
    }

    if(need_to_copy) {
	char new_cur_folder[100];

	rewind(keep_state_ptr->a.file.keep_file);
	if (copy_to_folder(keep_state_ptr->a.file.keep_file, folder) != 0) {
	    /* copy to cur_folder failed - try to copy to special file */
	    {
		int err UNUSED_VAROK = errno;
		DPRINT(Debug,1,(&Debug,
				"leavembox: copy(%s, %s) failed;",
				keep_state_ptr->a.file.temp_keep_file, 
				folder->cur_folder_sys));
		DPRINT(Debug,1,(&Debug,
			   "** %s ** (errno %d)\n", 
				strerror(err),err));
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmLeaveCouldntModifyFolder,
				  "Couldn't modify folder!"));
	    }
	PANIC_copy_failed:

	    if (sleepmsg > 0) {
		if (POLL_method)
		    wait_for_timeout((sleepmsg + 1) / 2);
		else
		    sleep((sleepmsg + 1) / 2);
	    }

	    elm_sfprintf(new_cur_folder,sizeof new_cur_folder,
			 FRM("%s/%s"), home, unedited_mail);

	    if (copy1(keep_state_ptr->a.file.keep_file, 
		      new_cur_folder,FALSE) != 0) {

		/* couldn't copy to special file either */
		int err UNUSED_VAROK = errno;
		DPRINT(Debug,1,(&Debug,
				"leavembox: couldn't copy to %s either!!  Help;", 
				new_cur_folder));
		DPRINT(Debug,1,(&Debug,
				"** %s ** (errno %d)\n", 
				strerror(err), err));
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveCantCopyMailbox,
				  "Can't copy folder, system trouble : contents preserved in %s\n"),
			  keep_state_ptr->a.file.temp_keep_file);
		
		fclose(keep_state_ptr->a.file.keep_file);
		keep_state_ptr->a.file.keep_file = NULL;
		status = 0;
		goto clean;
	    } else {
		DPRINT(Debug,1,(&Debug,
				"\nWoah! Confused - Saved mail in %s (leavembox)\n", 
				new_cur_folder));
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveSavedMailIn,
				  "Saved mail in %s."), 
			  new_cur_folder);
	    }
	} else
	    status = 1;
    }

    /* link or copy complete - remove temp keep file */
    unlink(keep_state_ptr->a.file.temp_keep_file);
    fclose(keep_state_ptr->a.file.keep_file);
    keep_state_ptr->a.file.keep_file = NULL;

    /*
     * restore permissions and access times of folder only if not
     * a symlink, as symlinks have no permissions, and not worth
     * tracking down what it points to.
     */

    if (!was_symlink) {
	if(restore_file_stats(folder->cur_folder_sys) != 1) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveProblemsRestoringPerms,
			      "Problems restoring permissions of folder %S"),
		      folder->cur_folder_disp);
	}
    }

    if (have_stat) {
	int err = 0;
	
	if (change_file_utime(folder->cur_folder_sys,
			      &buf,&err)) {
	    DPRINT(Debug,12,(&Debug,
			     "mbx_move_temp_file: utime for %s succeed\n",
			     folder->cur_folder_sys));

	} else if (err) {
	
	    DPRINT(Debug,1,(&Debug,
			    "Error: encountered error doing utime (mbx_move_temp_file) for %s\n",
			    folder->cur_folder_sys
			    ));
	    DPRINT(Debug,1,(&Debug,
			    "** %s ** (errno %d)\n", strerror(err), err));
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLeaveChangingAccessTime,
			      "Error %s trying to change file %S access time."), 
		      strerror(err), 
		      folder->cur_folder_disp);
	} 
    }
    
 clean:
    DPRINT(Debug,12,(&Debug,
		     "mbx_move_temp_file=%d\n",
		     status));
    return status;
}
    
S_(mbx_end_keep_folder mbx_end_keep_read_only)
static int mbx_end_keep_read_only P_((struct folder_info *folder,
				    KEEP_STATE keep_state_ptr));
static int mbx_end_keep_read_only(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_keep_read_only",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_keep_read_only=1 (dummy): folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (KS_magic !=  keep_state_ptr->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_keep_read_only",
	      "Bad magic number (keep_folder_state)",0);

    return 1;
}

S_(mbx_end_keep_folder mbx_end_keep_non_spool)
static int mbx_end_keep_non_spool P_((struct folder_info *folder,
				    KEEP_STATE keep_state_ptr));
static int mbx_end_keep_non_spool(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
{
    int need_to_copy = FALSE;
    int status = 0;
    long size;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_keep_non_spool",
	      "Bad magic number (folder_info)",0);
        
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_keep_non_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (lockfolders) {
	DPRINT(Debug,10,(&Debug,
			 "* lockfolders\n"));
	need_to_copy = TRUE;
    }

    if (!mbx_move_temp_file(folder,keep_state_ptr,need_to_copy)) {
	status = 0;
	goto clean;
    }
    size = file_bytes(folder->cur_folder_sys,NULL);
    if (-1L == size) {
	status = 0;
	goto clean;
    }
    folder->mailfile_size = size;
    status = 1;

 clean:
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_keep_non_spool=%d\n",
		     status));
    return status;
}

S_(mbx_end_keep_folder mbx_end_keep_spool)
static int mbx_end_keep_spool P_((struct folder_info *folder,
				    KEEP_STATE keep_state_ptr));
static int mbx_end_keep_spool(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
{
    int status = 0;
    int need_to_copy = FALSE;

    gid_t oldgid = getegid();
    long size;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_keep_spool",
	      "Bad magic number (folder_info)",0);

    DPRINT(Debug,11,(&Debug,
		     "mbx_end_keep_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    
    if (have_saved_ids) {
	enum syscall_status  r = setgid(mailgroupid);

	switch(r) {
	case syscall_success /* 0 */: {
	    DPRINT(Debug,11,(&Debug,
			     "mbx_end_keep_spool: setgid(%d) succeed\n",
			     mailgroupid));
	}
	    break;
	case syscall_error /* -1 */: {
	    int err = errno;
	    lib_error(FRM("Mailbox: setgid(%d) FAILED: %s"),
		      mailgroupid,strerror(err));
	}
	    break;
	}
    }

#ifdef SYSCALL_LOCKING
    DPRINT(Debug,10,(&Debug,
		     "* SYSCALL_LOCKING (SPOOL)\n"));
    need_to_copy = TRUE;
#endif /* SYSCALL_LOCKING */

    if (!mbx_move_temp_file(folder,keep_state_ptr,need_to_copy)) {
	status = 0;
	goto clean;
    }
    size = file_bytes(folder->cur_folder_sys,NULL);
    if (-1L == size) {
	status = 0;
	goto clean;
    }
    folder->mailfile_size = size;
    status = 1;

 clean:

    if (have_saved_ids) {
	enum syscall_status  r = setgid(oldgid);

	switch(r) {
	case syscall_success /* 0 */: {
	    DPRINT(Debug,11,(&Debug,
			     "mbx_end_keep_spool: setgid(%d) succeed\n",
			     oldgid));
	}
	    break;	    
	case syscall_error /* -1 */: {
    	    int err = errno;
	    lib_error(FRM("Mailbox: setgid(%d) FAILED: %s"),
		      oldgid,strerror(err));
	}
	    break;
	}
    }

    DPRINT(Debug,11,(&Debug,
		     "mbx_end_keep_spool=%d\n",
		     status));
    return status;
}

S_(mbx_mark_keep_folder mbx_mark_keep_read_only)
static int mbx_mark_keep_read_only P_((struct folder_info *folder,
				      KEEP_STATE keep_state_ptr,
				      struct header_rec *entry,
				      int keep));
static int mbx_mark_keep_read_only(folder,keep_state_ptr,entry,keep)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
     struct header_rec *entry;
     int keep;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_mark_keep_read_only=1: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    return 1;
}


S_(mbx_mark_keep_folder mbx_mark_keep_normal)
static int mbx_mark_keep_normal P_((struct folder_info *folder,
				     KEEP_STATE keep_state_ptr,
				     struct header_rec *entry,
				     int keep));
static int mbx_mark_keep_normal(folder,keep_state_ptr,entry,keep)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
     struct header_rec *entry;
     int keep;
{
    int count;
    FILE * fh;
    int status_written = 0;
    int ret = 1;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_mark_keep_normal",
	      "Bad magic number (folder_info)",0);
    
    /*  FIXME:
     *
     *  This should probably use real_start_we_local()
     *                           real_end_we_local()
     *   on savefolder.c
     *
     *  there is too many similar routines
     */

    DPRINT(Debug,11,(&Debug,
		     "mbx_mark_keep_normal: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (KS_magic !=  keep_state_ptr->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_mark_keep_normal",
	      "Bad magic number (keep_folder_state)",0);

    
    if (keep) {
	int ch = '\0';
	long pos;

	fh = folder_to_fd(folder,entry->offset);
	if (!fh) {
	    ret = 0;
	    goto FAIL2;
	}
    
	DPRINT(Debug,11,(&Debug,
			 "mbx_mark_keep_normal: envelope + header %ld - %ld\n",
			 entry->offset,entry->mime_rec.offset));

	count = 0;
	while ((pos = ftell(fh)) < entry->mime_rec.offset) {
	    char * buffer = NULL;
	    int len,i;

	    if (-1 == pos) {
		DPRINT(Debug,11,(&Debug,
				 "mbx_mark_keep_normal: ftell failed\n"));
		ret = 0;
		goto FAIL;		 
	    }
	    
	    if (!mbx_read_line(fh,&buffer,&len,
			       mbx_max_line_read(entry->mime_rec.offset /* end of headers == start of body */,
						 pos))) {
		DPRINT(Debug,11,(&Debug,
				 "mbx_mark_keep_normal: mbx_read_line failed at %ld  (from %ld) -- EOF\n",
				 ftell(fh), pos));

		
		ret = 0;
		goto FAIL;
		
	    }
	    
	    /* End of headers ? */
	    if ((len == 1 && 0 == memcmp(buffer,"\n",len)) ||
		(len == 2 && 0 == memcmp(buffer,"\r\n",len))) {
		if (!status_written) {
		    char buffer1[WLEN+10];   /* Enough space for status letters */
		    
		    int A = fprintf(keep_state_ptr->a.file.keep_file,
				     "Status: ");
		    
		    if (A < 0) {
			DPRINT(Debug,11,(&Debug,
					 "mbx_mark_keep_normal: Got error when writing\n"));
			ret = 0;
			goto fail_status;
		    }		    
		    count += A;

		    
 		    if (status_2_mailbox(entry,buffer1,sizeof buffer1) > 0) {
			int B = fprintf(keep_state_ptr->a.file.keep_file,
					"%s",buffer1);
			
			if (B < 0) {
			    DPRINT(Debug,11,(&Debug,
					     "mbx_mark_keep_normal: Got error when writing\n"));
			    ret = 0;
			    goto fail_status;
			}					   
			count += B;
		    }

		    status_written = 1;
		    
		fail_status:
		    /* Write EOLN */
		    for (i = 0; i < len; i++) {
			if (EOF == putc(buffer[i],keep_state_ptr->a.file.keep_file)) {
			    DPRINT(Debug,11,(&Debug,
					     "mbx_mark_keep_normal: Got EOF (error) when writing\n"));
			    ret = 0;
			    break;
			    
			}
			count++;
		    }

		}
	    }

	    if (NULL != header_cmp(buffer, "Status", NULL)) {
		/* SKIP */
	    } else {
		for (i = 0; i < len; i++) {
		    putc(buffer[i],keep_state_ptr->a.file.keep_file);
		    count++;
		}
	    }
	    free(buffer);
	}
	if (!status_written) {
	    DPRINT(Debug,1,(&Debug,
			    "mbx_mark_keep_normal: ERROR : Status: not written!\n"));
	    ret = 0;
	}

	pos = ftell(fh);
	if (pos != entry->mime_rec.offset) {
	    DPRINT(Debug,1,(&Debug,
			    "mbx_mark_keep_normal: ERROR : Bad position %ld (should be %ld)\n",
			    pos,entry->mime_rec.offset));
	    ret = 0;
	}

	DPRINT(Debug,11,(&Debug,
			 "mbx_mark_keep_normal: wrote %d bytes of header\n",
			 count));
	DPRINT(Debug,11,(&Debug,
			 "mbx_mark_keep_normal: content length %ld\n",
			 entry->content_length));
	
	count = 0;
	while (count < entry->content_length) {
	    ch = getc(fh);
	    
	    if (EOF == ch) {
		DPRINT(Debug,11,(&Debug,
				 "mbx_mark_keep_normal: Got EOF when reading\n"));
		ret = 0;
		break;
	    }

	    if (EOF == putc(ch,keep_state_ptr->a.file.keep_file)) {
		DPRINT(Debug,11,(&Debug,
				 "mbx_mark_keep_normal: Got EOF (error) when writing\n"));
		ret = 0;
		break;
	    }
	    count++;
	}
	DPRINT(Debug,11,(&Debug,
			 "mbx_mark_keep_normal: wrote %d bytes of body\n",
			 count));
	

    FAIL:
	/*
	 * parse_body_routine() excludes ending empty line from
	 * content_length, so add empty line always to mail 
	 */

	if (ch != '\n') {
	    DPRINT(Debug,1,(&Debug,
			     "mbx_mark_keep_normal: Last character was not NL !!\n"));
	    if (EOF != putc('\n',keep_state_ptr->a.file.keep_file)) {
		DPRINT(Debug,11,(&Debug,
				 "mbx_mark_keep_normal: Added extra NL\n"));
	    } else {
		ret = 0;
	    }
	}

	if (!have_MMDF) {

	    /* blank line to keep mailx happy *sigh* */      
	    if (fprintf(keep_state_ptr->a.file.keep_file, "\n") == EOF) {
		DPRINT(Debug,5,(&Debug,
				"mbx_mark_keep_normal: Failed to add second NL to end of mail\n"));
		ret = 0;
	    } else {
		DPRINT(Debug,11,(&Debug,
				 "mbx_mark_keep_normal: Added NL for empty line.\n"));
	    }

	} else {
	    /* FIXME --- What is correct? */

	}
	
    }

 FAIL2:

    DPRINT(Debug,11,(&Debug, "mbx_mark_keep_normal=%d\n",ret));

    return ret;
}

S_(mbx_folder_type mbx_non_spool_type)
const char * mbx_non_spool_type P_((struct folder_info *folder));
const char * mbx_non_spool_type(folder)
     struct folder_info *folder;
{
    static char *FOLDER = NULL;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_non_spool_type",
	      "Bad magic number (folder_info)",0);
    
    if (!FOLDER)
	FOLDER = catgets(elm_msg_cat, ElmSet, ElmFolder, "Folder");

    DPRINT(Debug,11,(&Debug,
		     "mbx_non_spool_type: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "mbx_non_spool_type=%s\n",
		     FOLDER));
    return FOLDER;
}

S_(mbx_folder_type mbx_read_only_type)
const char * mbx_read_only_type P_((struct folder_info *folder));
const char * mbx_read_only_type(folder)
     struct folder_info *folder;
{
    static char *FOLDER = NULL;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_read_only_type",
	      "Bad magic number (folder_info)",0);
    
    if (!FOLDER)
	FOLDER = catgets(elm_msg_cat, ElmSet, ElmReadOnlyFolder, 
			 "Read-only folder");

    DPRINT(Debug,11,(&Debug,
		     "mbx_read_only_type: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "mbx_read_only_type=%s\n",
		     FOLDER));
    return FOLDER;
}

S_(mbx_folder_type mbx_spool_type)
const char * mbx_spool_type P_((struct folder_info *folder));
const char * mbx_spool_type(folder)
     struct folder_info *folder;
{
    static char *mailbox = NULL;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_spool_type",
	      "Bad magic number (folder_info)",0);
    
    if (!mailbox)
	mailbox = catgets(elm_msg_cat, ElmSet, ElmMailbox, "Mailbox");

    DPRINT(Debug,11,(&Debug,
		     "mbx_spool_type: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "mbx_spool_type=%s\n",
		     mailbox));
    return mailbox;
}

S_(mbx_start_edit_folder mbx_start_edit_non_spool)
static int mbx_start_edit_non_spool P_((struct folder_info *folder, 
					const char **buffer));
static int mbx_start_edit_non_spool(folder,buffer)
     struct folder_info *folder;
     const char **buffer;
{
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_start_edit_non_spool",
	      "Bad magic number (folder_info)",0);

    
    DPRINT(Debug,11,(&Debug,
		     "mbx_start_edit_non_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    *buffer = folder->cur_folder_sys;
    status = 1;

    DPRINT(Debug,50,(&Debug,
		     "mbx_start_edit_non_spool=%d | *buffer=%s\n",
		     status,
		     *buffer ? *buffer : "NULL"));
    return status;
}

S_(mbx_start_edit_folder mbx_start_edit_spool)
static int mbx_start_edit_spool P_((struct folder_info *folder, 
				     const char **buffer));
static int mbx_start_edit_spool(folder,buffer)
     struct folder_info *folder;
     const char **buffer;
{
    int status = 0;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_start_edit_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_start_edit_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    *buffer = NULL;

    if(save_file_stats(folder->cur_folder_sys,NULL) != 0) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmPermFolder,
			  "Problems saving permissions of folder %S!"), 
		  folder->cur_folder_disp);
	status = 0;
	goto clean;
    }

    *buffer = folder->cur_tempfolder;
    status = 1;

 clean:
    DPRINT(Debug,50,(&Debug,
		     "mbx_start_edit_spool=%d | *buffer=%s\n",
		     status,
		     *buffer ? *buffer : "NULL"));
    return status;
}

S_(mbx_end_edit_folder mbx_end_edit_non_spool)
static enum end_edit_fol_status mbx_end_edit_non_spool P_((struct folder_info *folder));
static enum end_edit_fol_status mbx_end_edit_non_spool(folder)
     struct folder_info *folder;
{
    enum end_edit_fol_status status = end_edit_none;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_edit_non_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_edit_non_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    switch (sessionlock_folder(folder,SESSIONLOCK_REOPEN,NULL,NULL)) {
    case sessionlock_fail:	
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldntOpenFolder,
			  "Couldn't open %s for reading!  Edit LOST!"), 
		  folder->cur_folder_sys);
	status = end_edit_none;       /* Is this correct? */
	goto clean;
    case sessionlock_open:
	status = end_edit_open;
	if (0) {
	case sessionlock_reconnect:   /* Not used with local mailbox */
	    status = end_edit_reconnect;
	}
	lib_transient(CATGETS(elm_msg_cat, ElmSet, ElmResyncingNewVersion,
			      "Resynchronizing with new version of folder..."));
	break;
    }

 clean:
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_edit_non_spool=%d",		     
		     status));
    switch (status) {
    case end_edit_fail_abort:  DPRINT(Debug,11,(&Debug," end_edit_fail_abort")); break;
    case end_edit_none:        DPRINT(Debug,11,(&Debug," end_edit_none"));       break;
    case end_edit_open:        DPRINT(Debug,11,(&Debug," end_edit_open"));       break;
    case end_edit_reconnect:   DPRINT(Debug,11,(&Debug," end_edit_reconnect"));  break;
    }
    DPRINT(Debug,11,(&Debug,"\n"));
    
    return status;
}

S_(mbx_end_edit_folder  mbx_end_edit_spool)
static enum end_edit_fol_status mbx_end_edit_spool P_((struct folder_info *folder));
static enum end_edit_fol_status mbx_end_edit_spool(folder)
     struct folder_info *folder;
{
    enum end_edit_fol_status status =  end_edit_none;
    gid_t oldgid = getegid();
    long size;

    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_edit_spool",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_edit_spool: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    
    switch (sessionlock_folder(folder,SESSIONLOCK_REOPEN,NULL,NULL)) {
    case sessionlock_fail:
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldntOpenFolder,
			  "Couldn't open %s for reading!  Edit LOST!"), 
		  folder->cur_folder_sys);
	status = end_edit_none;
	goto clean;

    case sessionlock_open:
	status = end_edit_open;
	if (0) {
	case sessionlock_reconnect:  /* Not used on local mailer */
	    status = end_edit_reconnect;
	}
    
	if (!lock_folder(LOCK_OUTGOING,folder)) {
	    status = end_edit_none;
	    goto clean;
	}
	
	if (PRIVATE_DATA_magic != folder->p->magic) 
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "mbx_end_edit_spool",
		  "Bad magic number (private_data)",0);
	
	size = file_bytes(folder->cur_folder_sys,NULL);
	if (-1L == size) {
	    status = end_edit_none;
	    unlock_folder(0,folder);
	    goto clean;
	}
	
	if ( size !=  folder->mailfile_size) {
	    char buffer[1000];
	    int len;
	    enum syscall_status  r;
	    
	    /* SIGH.  We've received mail since we invoked the editor
	       on the folder.  We'll have to do some strange stuff to
	       remedy the problem... */
	    
	    lib_transient(CATGETS(elm_msg_cat, ElmSet, 
				  ElmWarnNewMailRecv,
				  "Warning: new mail received..."));

	    
	    r = fseek(folder->p->fh_temp,0,SEEK_END);
	    switch(r) {
	    case syscall_success /* 0 */: 
		DPRINT(Debug,11,(&Debug,
				 "mbx_end_edit_spool: Seek end of temp file\n"));
		break;		    	    
	    case syscall_error /* -1 */: {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldntSeekTemp,
				  "Couldn't seek to end of temp file. Edit LOST!"));
		status = end_edit_none;
		unlock_folder(0,folder);
		goto clean;
	    }
		break;
	    }
		
	    r = fseek(folder->p->fh_folder, 
		      folder->mailfile_size, 0);
	    switch(r) {
	    case syscall_success /* 0 */: 
		DPRINT(Debug,11,(&Debug,
				 "mbx_end_edit_spool: Seek end of cur_folder (offset %ld)\n",
				 folder->mailfile_size));
		break;
	    case syscall_error /* -1 */: {
		DPRINT(Debug,1,(&Debug,
				"Couldn't seek to end of cur_folder (offset %ld) (%s)\n",
				folder->mailfile_size, "mbx_end_edit_non_spool"));
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldntSeekEnd,
				  "Couldn't seek to end of folder.  Edit LOST!"));
		
		unlock_folder(0,folder);
		status = end_edit_none;
		goto clean;
	    }
		break;
	    }
	    
	    /** Now we can finally stream the new mail into the tempfile **/
	    
	    while ((len = mail_gets(buffer, SLEN, 
				    folder ->p->fh_folder)) != 0)
		if (fwrite(buffer, 1, len, folder ->p->fh_temp) != len) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldntCopyMailfile,
				      "Couldn't copy %S to mail file %s!"),
			      folder->cur_folder_disp, 
			      folder->cur_tempfolder);
		    unlock_folder(0,folder);
		    status = end_edit_fail_abort;
		    goto clean;
		}
	    
	    if (0 != fflush(folder->p->fh_temp)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldntCopyMailfile,
				  "Couldn't copy %S to mail file %s!"),
			  folder->cur_folder_disp, 
			  folder->cur_tempfolder);
		unlock_folder(0,folder);
		status = end_edit_fail_abort;
		goto clean;
	    }
	    

	    lib_transient(CATGETS(elm_msg_cat, ElmSet, ElmChangesIncorporated,
			      "Changes incorporated into new mail..."));
	} else 
	    lib_transient(CATGETS(elm_msg_cat, ElmSet, ElmResyncingNewVersion,
				  "Resynchronizing with new version of folder..."));
		
	if (have_saved_ids) {
	    enum syscall_status  r = setgid(mailgroupid);
	    switch(r) {
	    case syscall_success /* 0 */: {
		DPRINT(Debug,11,(&Debug,
				 "mbx_end_edit_spool: setgid(%d) succeed\n",
				 mailgroupid));
	    }
		break;	    
	    case syscall_error /* -1 */: {
		int err = errno;
		lib_error(FRM("Mailbox: setgid(%d) FAILED: %s"),
			  mailgroupid,strerror(err));
	    }
		break;
	    }
	}

	/* remove real mail_file and then
	 * link or copy the edited mailfile to real mail_file */
	
	if (copy_to_folder(folder->p->fh_temp,folder) != 0) {
	    unlock_folder(0,folder);
	    status = end_edit_fail_abort;
	    goto clean;
	}
	
	/* restore file permissions before removing lock */
	
	if(restore_file_stats(folder->cur_folder_sys) != 1) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmProblemsRestoringPerms,
			      "Problems restoring permissions of folder %S"),
		      folder->cur_folder_disp);
	    lib_error(FRM(""));
	}
	
	unlock_folder(0,folder);
	break;
    }
    
 clean:
    
    if (have_saved_ids) {
	enum syscall_status  r = setgid(oldgid);

	switch(r) {
	case syscall_success /* 0 */: {
	    DPRINT(Debug,11,(&Debug,
			     "mbx_end_edit_spool: setgid(%d) succeed\n",
			     oldgid));
	}
	    break;	    
	case syscall_error /* -1 */: {
	    int err = errno;
	    lib_error(FRM("Mailbox: setgid(%d) FAILED: %s"),
		      oldgid,strerror(err));
	}
	    break;
	}
    }

    DPRINT(Debug,11,(&Debug,
		     "mbx_end_edit_spool=%d",
		     status));
    switch (status) {
    case end_edit_fail_abort:  DPRINT(Debug,11,(&Debug," end_edit_fail_abort")); break;
    case end_edit_none:        DPRINT(Debug,11,(&Debug," end_edit_none"));       break;
    case end_edit_open:        DPRINT(Debug,11,(&Debug," end_edit_open"));       break;
    case end_edit_reconnect:   DPRINT(Debug,11,(&Debug," end_edit_reconnect"));  break;
    }
    DPRINT(Debug,11,(&Debug,"\n"));
    
    return status;
}

S_(mbx_get_folder_mode mbx_get_spool_mode)
static int mbx_get_spool_mode P_((struct folder_info *folder));
static int mbx_get_spool_mode(folder)
     struct folder_info *folder;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_get_spool_mode",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_get_spool_mode=FOLDER_MBOX|FOLDER_FILE: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return FOLDER_MBOX | FOLDER_FILE;
}

S_(mbx_get_folder_mode mbx_get_non_spool_mode)
static int mbx_get_non_spool_mode P_((struct folder_info *folder));
static int mbx_get_non_spool_mode(folder)
     struct folder_info *folder;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_get_non_spool_mode",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_get_spool_mode=FOLDER_FILE: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return FOLDER_FILE;
}

S_(mbx_get_folder_mode mbx_get_read_only_mode)
static int mbx_get_read_only_mode P_((struct folder_info *folder));
static int mbx_get_read_only_mode(folder)
     struct folder_info *folder;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_get_read_only_mode",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_get_read_only_mode=FOLDER_RDONLY|FOLDER_FILE: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    return FOLDER_RDONLY|FOLDER_FILE;
}

S_(mbx_is_forwarded_folder mbx_is_forwarded_normal)
static const char * mbx_is_forwarded_normal P_((struct folder_info *folder,
					      READ_STATE read_state_ptr));
static const char * mbx_is_forwarded_normal(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_is_forwarded_normal",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_is_forwarded_normal: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                     : read_state_ptr=%p\n",
		     read_state_ptr));

    DPRINT(Debug,11,(&Debug,
		     "mbx_is_forwarded_normal=NULL\n"));
    return NULL;
}

S_(mbx_consider_remove_folder mbx_consider_remove_normal)
static int mbx_consider_remove_normal P_((struct folder_info *folder));
static int mbx_consider_remove_normal(folder)
     struct folder_info *folder;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_consider_remove_normal",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_consider_remove_normal: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "mbx_consider_remove_normal=0\n"));
    return 0;
}

S_(mbx_start_edit_folder mbx_start_edit_read_only)
static int mbx_start_edit_read_only P_((struct folder_info *folder, 
				     const char **buffer));
static int mbx_start_edit_read_only(folder,buffer)
     struct folder_info *folder;
     const char **buffer;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_start_edit_read_only",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_start_edit_read_only: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "mbx_start_edit_read_only=0\n"));
    return 0;
}

S_( mbx_end_edit_folder mbx_end_edit_read_only)
static enum end_edit_fol_status mbx_end_edit_read_only P_((struct folder_info *folder));
static enum end_edit_fol_status mbx_end_edit_read_only(folder)
     struct folder_info *folder;
{
    if (FOLDER_INFO_magic != folder->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_edit_read_only",
	      "Bad magic number (folder_info)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_edit_read_only: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_edit_read_only=end_edit_none\n"));
    return end_edit_none ;
}

E_(mbx_give_folder_server mbx_give_normal_server)
const struct remote_server * mbx_give_normal_server(folder)
     struct folder_info *folder;
{
    return LOCAL_SERVER;
}

struct folder_type read_only = {
                                 FOLDER_TYPE_magic, "FOLDER",
                                 mbx_close_non_spool,
				 mbx_lock_non_spool,
				 mbx_init_non_spool,
				 mbx_sessionlock_read_only,
				 mbx_unlock_non_spool,
				 mbx_flush_read_only,
				 mbx_ferror_non_spool,
				 mbx_prepare_read_non_spool,
				 mbx_end_read_non_spool,
				 mbx_copy_envelope_non_spool,
				 mbx_is_forwarded_normal,
				 mbx_copy_header_non_spool,
				 mbx_copy_body_non_spool,
				 mbx_copy_envelope_end_non_spool,
				 mbx_copy_envelope_reset_body_non_spool,
				 mbx_non_spool_to_fd,
				 mbx_new_mail_on_normal,
				 mbx_consider_remove_normal,
				 mbx_prepare_keep_read_only,
				 mbx_end_keep_read_only,
				 mbx_mark_keep_read_only,
				 mbx_read_only_type,
				 mbx_start_edit_read_only,
				 mbx_end_edit_read_only,
				 mbx_free_non_spool,
				 mbx_zero_rs_fields_file,
				 mbx_free_rs_fields_file,
				 mbx_zero_ks_fields_file,
				 mbx_free_ks_fields_file,
				 mbx_get_read_only_mode,
				 mbx_give_normal_server,
				 mbx_comp_prev_hdr_default,
				 mbx_update_prev_hdr_default,
				 mbx_have_default_quota,
				 mbx_give_default_quotar_list,
				 mbx_give_message_msize_default,
				 mbx_give_message_count_default

};

struct folder_type non_spool = {
                                 FOLDER_TYPE_magic, "FOLDER",
                                 mbx_close_non_spool,
				 mbx_lock_non_spool,
				 mbx_init_non_spool,
				 mbx_sessionlock_non_spool,
				 mbx_unlock_non_spool,
				 mbx_flush_non_spool,
				 mbx_ferror_non_spool,
				 mbx_prepare_read_non_spool,
				 mbx_end_read_non_spool,
				 mbx_copy_envelope_non_spool,
				 mbx_is_forwarded_normal,
				 mbx_copy_header_non_spool,
				 mbx_copy_body_non_spool,
				 mbx_copy_envelope_end_non_spool,
				 mbx_copy_envelope_reset_body_non_spool,
				 mbx_non_spool_to_fd,
				 mbx_new_mail_on_normal,
				 mbx_consider_remove_non_spool,
				 mbx_prepare_keep_normal,
				 mbx_end_keep_non_spool,
				 mbx_mark_keep_normal,
				 mbx_non_spool_type,
				 mbx_start_edit_non_spool,
				 mbx_end_edit_non_spool,
				 mbx_free_non_spool,
				 mbx_zero_rs_fields_file,
				 mbx_free_rs_fields_file,
				 mbx_zero_ks_fields_file,
				 mbx_free_ks_fields_file,
				 mbx_get_non_spool_mode,
				 mbx_give_normal_server,
				 mbx_comp_prev_hdr_default,
				 mbx_update_prev_hdr_default,
				 mbx_have_default_quota,
				 mbx_give_default_quotar_list,
				 mbx_give_message_msize_default,
				 mbx_give_message_count_default

};

struct folder_type spool     = {
                                 FOLDER_TYPE_magic, "MAILBOX",
                                 mbx_close_spool,
				 mbx_lock_spool,
				 mbx_init_spool,
				 mbx_sessionlock_spool,
				 mbx_unlock_spool,
				 mbx_flush_spool,
				 mbx_ferror_spool,
				 mbx_prepare_read_spool,
				 mbx_end_read_spool,
				 mbx_copy_envelope_spool,
				 mbx_is_forwarded_spool,
				 mbx_copy_header_spool,
				 mbx_copy_body_spool,
				 mbx_copy_envelope_end_spool,
				 mbx_copy_envelope_reset_body_spool,
				 mbx_spool_to_fd,
				 mbx_new_mail_on_normal,
				 mbx_consider_remove_normal,
				 mbx_prepare_keep_normal,
				 mbx_end_keep_spool,
				 mbx_mark_keep_normal,
				 mbx_spool_type,
				 mbx_start_edit_spool,
				 mbx_end_edit_spool,
				 mbx_free_spool,
				 mbx_zero_rs_fields_file,
				 mbx_free_rs_fields_file,
				 mbx_zero_ks_fields_file,
				 mbx_free_ks_fields_file,
				 mbx_get_spool_mode,
				 mbx_give_normal_server,
				 mbx_comp_prev_hdr_default,
				 mbx_update_prev_hdr_default,
				 mbx_have_default_quota,
				 mbx_give_default_quotar_list,
				 mbx_give_message_msize_default,
				 mbx_give_message_count_default
};

void mailbox_2_status(status,entry)
     char * status; 
     struct header_rec *entry;
{

    strfcpy(entry->mailx_status, status, WLEN);

    remove_possible_trailing_spaces(entry->mailx_status);
	
    /* Okay readjust the status. If there's an 'R', message
     * is read; if there is no 'R' but there is an 'O', message
     * is unread. In any case it isn't new because a new 
     * message wouldn't have a Status: header. 
     * -- if message is re$ynced there may now be 
     * Status: -header
     */
    if (index(status, 'R') != NULL)
	entry->status &= ~(NEW | UNREAD);

    else if (index(status,'O') != NULL) {
	entry->status &= ~NEW;
	entry->status |= UNREAD;
    }

    if (index(status, 'r') != NULL)
	entry->status  |= REPLIED_TO;
    if (index(status, 'F') != NULL)
	entry->status1 |= S1_FLAGGED;

}

int status_2_mailbox(entry,buffer,size)
     struct header_rec *entry;
     char *buffer;
     int size;
{
    int count = 0;
    int i;

    size--;    /* For trailing \0' */

#define ADD(x)   do { if (count < size) { buffer[count++] = (x); } else { goto fail; } } while(0)

    if (!ison(entry->status, UNREAD)) 
	ADD('R');
	    
    if (ison(entry->status, NEW)) 
	ADD('N');
    else 
	ADD('O');

    if (ison(entry->status, REPLIED_TO)) 
	ADD('r');


    if (ison(entry->status1, S1_FLAGGED)) 
	ADD('F');

    /* Write unknown flags */
    for (i=0; entry->mailx_status[i] != '\0'; i++)
	switch(entry->mailx_status[i]) {
	case 'R':
	case 'O':
	case 'r':
	case 'N':
	case 'F':
	    break;
	default:
	    ADD(entry->mailx_status[i]);
	}

 fail:
    buffer[count] = '\0';

#undef ADD

    return count;
}

void remove_possible_trailing_spaces(string)
     char *string;
{
	/** an incredibly simple routine that will read backwards through
	    a string and remove all trailing whitespace.
	**/

	int i, j;

	for ( j = i = strlen(string); --i >= 0 && whitespace(string[i]); )
		/** spin backwards, semicolon intented **/ ;

	if (i > 0 && string[i-1] == '\\') /* allow for line to end with \blank  */
	  i++;

	if (i < j)
	  string[i+1] = '\0';	/* note that even in the worst case when there
				   are no trailing spaces at all, we'll simply
				   end up replacing the existing '\0' with
				   another one!  No worries, as M.G. would say
				*/
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

