static char rcsid[] = "@(#)$Id: imap.c,v 2.52 2024/06/13 17:36:55 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.52 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mbox.h"
#include "ss_imp.h"
#include "s_me.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"imap");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

#ifdef REMOTE_MBX
#include "mbximap_imp.h"

static struct IMAP_BROWSER * malloc_IMAP_BROWSER P_((void));
static struct IMAP_BROWSER * malloc_IMAP_BROWSER()
{
    struct IMAP_BROWSER * ret = safe_malloc(sizeof (*ret));

    ret->magic            = IMAP_BROWSER_magic;

    ret->Ch               = NULL;
    ret->dir_name_cache   = NULL;
    ret->cur_sep          = '\0';
    ret->dir_entries      = NULL;
    ret->dir_entry_count  = 0;
    ret->passhm           = NULL;

    return ret;
}


static void free_dir_entries P_((struct IMAP_BROWSER *ptr));
static void free_dir_entries(ptr)
     struct IMAP_BROWSER *ptr;
{
    if (IMAP_BROWSER_magic != ptr->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_dir_entries",
	      "Bad magic type",0);


    if (ptr->dir_entries) {
	int i;
	
	for (i = 0; i < ptr->dir_entry_count; i++) {
	    if (ptr->dir_entries[i].imap_name) {
		free(ptr->dir_entries[i].imap_name);
		ptr->dir_entries[i].imap_name = NULL;
	    }

	    if (ptr->dir_entries[i].translated_name) 
		free_string(&(ptr->dir_entries[i].translated_name));
	    ptr->dir_entries[i].sep   = '\0';
	    ptr->dir_entries[i].flags = 0;
	}
	free(ptr->dir_entries);
	ptr->dir_entries     = NULL;
	ptr->dir_entry_count = 0;
    }

} 

static void free_IMAP_BROWSER P_((struct IMAP_BROWSER **ptr));
static void free_IMAP_BROWSER(ptr) 
     struct IMAP_BROWSER **ptr;
{
    if (IMAP_BROWSER_magic != (*ptr)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_free_imap",
	      "Bad magic type",0);

    if ((*ptr)->Ch) {
	int r = free_connection( &((*ptr)->Ch), NULL);

	if (!r) {
	    DPRINT(Debug,11,(&Debug, "browser_free_imap: free_connection failed\n"));
	}
    }
	
    if ((*ptr)->dir_name_cache)
	free_string(&((*ptr)->dir_name_cache));

    free_dir_entries(*ptr);

    if ((*ptr)->passhm)
	free_browser_passhm(& ((*ptr)->passhm));

    (*ptr)->magic = 0;           /* Invalidate */

    free(*ptr);
    *ptr = NULL;
}


/* Seems that h_errno is macro on AIX */
#ifndef h_errno
extern int h_errno;
#endif


static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif



/* IMAP connection -------------------------------------------------------- */

static void increment_uid_number P_((struct IMAP_MBX *M, 
				     struct imap_uid_ref *UID));
static void increment_uid_number (M,UID)
     struct IMAP_MBX *M;
     struct imap_uid_ref *UID;
{
    int val = UID->uid_number;
    
    /*
      M->reference_count == number of messages + 1
      M->references [0]     is not used
    */

    if (UID->last_mbx_index > 0 && UID->last_mbx_index < M->reference_count) {

	/* 1) IF mails are deleted by server meanwhile, 
	      current UID->last_mbx_index may point to next message 
	*/

	if (M->references[UID->last_mbx_index].uid_number > UID->uid_number) {
	    
	    UID->uid_number = M->references[UID->last_mbx_index].uid_number;

	    if (UID->uid_number != val+1) {
		DPRINT(Debug,4,(&Debug,
				"increment_uid_number: Jumping UID from %d to %d (index %d) -- mail deleted?\n",
				val,UID->uid_number,UID->last_mbx_index));
	    }

	    return;

	}
	
	/* 2) UID->last_mbx_index was refering to last uid  */
	if (M->references[UID->last_mbx_index].uid_number == UID->uid_number &&
	    UID->last_mbx_index+1 < M->reference_count) {
	

	    if (M->references[UID->last_mbx_index+1].uid_number < UID->uid_number) {
		
		DPRINT(Debug,1,(&Debug,"increment_uid_number: DANGER, UIDs not asceding: [%d]=%d and [%d]=%d\n",
				UID->last_mbx_index,   M->references[UID->last_mbx_index].uid_number,
				UID->last_mbx_index+1, M->references[UID->last_mbx_index+1].uid_number));

		/* Use default code */
	    } else {

		UID->last_mbx_index++;
	    
		UID->uid_number = M->references[UID->last_mbx_index].uid_number;
		
		if (UID->uid_number != val+1) {
		    DPRINT(Debug,4,(&Debug,"increment_uid_number: Jumping UID from %d to %d (index %d) -- whole on UID range\n",
				    val,UID->uid_number,UID->last_mbx_index));
		}

		return;
	    }
	}
	
    }


    /* 3) UID->last_mbx_index is invalid and we are beginning of mail */


    if (-1 == UID->last_mbx_index && M->reference_count > 1 &&
	UID->uid_number < M->references[1].uid_number) {

	UID->last_mbx_index = 1;
	
	UID->uid_number = M->references[UID->last_mbx_index].uid_number;
		
	if (UID->uid_number != val+1) {
	    DPRINT(Debug,4,(&Debug,"increment_uid_number: Jumping UID from %d to %d (index %d) -- beginning of mail\n",
			    val,UID->uid_number,UID->last_mbx_index));
	}

	return;
    }


    UID->uid_number++;
}

static struct imap_reference  *find_reference P_((struct IMAP_MBX *M, 
						  struct imap_uid_ref *UID));
static struct imap_reference  *find_reference(M,UID)
     struct IMAP_MBX *M;
     struct imap_uid_ref *UID;
{
    int i;
    if (UID->last_mbx_index > 0 && UID->last_mbx_index < M->reference_count &&
	M->references[UID->last_mbx_index].uid_number ==
	UID->uid_number) {

	/* Found via cache */
	return &(M->references[UID->last_mbx_index]);
    }

    for (i = 1; i < M->reference_count; i++)
	if (M->references[i].uid_number == UID->uid_number) {
	    UID->last_mbx_index = i;  /* Update cache */
	    return &(M->references[UID->last_mbx_index]);
	}

    DPRINT(Debug,1,(&Debug, "find_reference: No mail with IMAP uid %d\n",
		    UID->uid_number));

    UID->last_mbx_index = -1;  /* Invalidate cache */
    return NULL;
}

static char * conv_to_imap_name P_((struct string *name));
static char * conv_to_imap_name(name)
     struct string *name;
{
    struct string * ConvRelative = 
	convert_string(IMAP_name_convention ? IMAP_ENCODING : imap_charset,
		       name,0);

    char * ret = us2s(stream_from_string(ConvRelative,0,NULL));

    free_string(&ConvRelative);

    DPRINT(Debug,12,(&Debug, 
		     "conv_to_imap_name=%s    (name=%S, relative=%S)\n",
		     ret,name,ConvRelative));    
    return ret;
}


static void imap_idle_wait P_((struct folder_info *folder));
static void imap_idle_wait(folder)
     struct folder_info *folder;
{
    struct IMAP_MBX * M;

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"imap_idle_wait",
	      "Bad magic number (private_data)",0);

    M = & (folder->p->a.imap_mbx);
 
    imapcon_idle_wait(M->Ch);
}

/* ----------------------------------------------------------------------- */

/* IMAP browser */

static void browser_zero_imap P_((struct folder_browser *dir));
static void browser_zero_imap(dir)
     struct folder_browser *dir;
{
    DPRINT(Debug,11,(&Debug, "browser_zero_imap: dir=%p\n", dir));

    dir->a.imap_browser = malloc_IMAP_BROWSER();

    dir->a.imap_browser->Ch              = create_connection(&IMAP_connection);
    dir->a.imap_browser->Ch->d  = dir;   /* Back link */
    dir->a.imap_browser->dir_name_cache  = NULL;
    dir->a.imap_browser->cur_sep         = '\0';
    dir->a.imap_browser->dir_entries     = NULL;
    dir->a.imap_browser->dir_entry_count = 0;
}

static void imap_free_dirlist P_((struct folder_browser *dir));

static void imap_free_dirlist(dir)
     struct folder_browser *dir;
{
    DPRINT(Debug,12,(&Debug, 
		     "imap_free_dirlist: dir=%p\n", 
		     dir));

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"imap_free_dirlist",
	      "Bad magic type",0);
    
    free_dir_entries(dir->a.imap_browser);
}

static void browser_free_imap P_((struct folder_browser *dir));
static void browser_free_imap(dir)
     struct folder_browser *dir;
{
    DPRINT(Debug,11,(&Debug, 
		     "browser_free_imap: dir=%p (%s)\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>"));


    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_free_imap",
	      "Bad magic type",0);

    if (dir->a.imap_browser->Ch) {
	const enum connection_state S = 
	    get_connection_state(dir->a.imap_browser->Ch);

	DPRINT(Debug,12,(&Debug, " ... caching %s%s\n",
			 IMAP_connection_cache ? "enabled" : "disabled",
			 CON_error == S ?
			 ", on error state" : "" ));
	
	if (S != CON_error && IMAP_connection_cache) {
	    /* Give connection back to connection cache */
	    cache_connection(dir->a.imap_browser->Ch);
	    dir->a.imap_browser->Ch = NULL;
	} else {
	    DPRINT(Debug,12,(&Debug, " ... connection not cached.\n"));
	}
    }

    dir->a.imap_browser->cur_sep         = '\0';

    imap_free_dirlist(dir);

    free_IMAP_BROWSER(& dir->a.imap_browser);
}
    
/* Does actual 'change' of directory */
static int browser_update_imaplisting P_((struct folder_browser *
					  dir,
					  struct string *dirname,
					  char *sysname,
					  int sep));
static int browser_update_imaplisting(dir,dirname,sysname,sep)
     struct folder_browser *dir;
     struct string *dirname;
     char *sysname;
     int sep;
{
    imap_states ret1;
    int ret = 0;
    char           * newsysdir  = NULL;
    struct string  * newdirname = NULL;
    enum connection_state S;

    DPRINT(Debug,12,(&Debug, 
		     "browser_update_imaplisting: dir=%p, sysname=%s\n", 
		     dir,sysname ? sysname : "<NULL>" ));

    if (sysname && sysname[0] && !sep) {
	DPRINT(Debug,12,(&Debug, 
			 "browser_update_imaplisting -- no separator character ...\n"));
	goto fail;
    }

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_update_imaplisting",
	      "Bad magic type",0);

    /* browser_folder_from_imap may close connection ... */
    if ((S = get_connection_state(dir->a.imap_browser->Ch)) != CON_logged) {

	DPRINT(Debug,12,(&Debug, 
			 "browser_update_imaplisting: con=%p state=%d\n",
			 dir->a.imap_browser->Ch,S));

	if (!login_connection_hm(dir->a.imap_browser->Ch,
				 dir->a.imap_browser->passhm))
	    goto fail;
    }

    imap_free_dirlist(dir);
    if (!start_imap_command(dir->a.imap_browser->Ch,"LIST"))
	goto fail;

    /* Reference name -- currently empty */
    imap_command_push_string(dir->a.imap_browser->Ch,"");

    /* Listing pattern -- % match one component on hierarchy */
    if (!sysname || !sysname[0]) {
        imap_command_push_astring(dir->a.imap_browser->Ch,"%");
	newsysdir  = safe_strdup("");
	newdirname = new_string(ASCII_SET);
    } else {
	int l1;
	char * pattern = NULL;
	char buf[2];

	newsysdir = safe_strdup(sysname);
	l1 = strlen(newsysdir);
	if (l1 > 0 && sep == newsysdir[l1-1]) {
	    DPRINT(Debug,12,(&Debug, 
			     "browser_update_imaplisting: Removing %c from %s\n",
			     sep,sysname));
	    newsysdir[l1-1] = '\0';
	}
	
	/* IMAP character set is assumed to be ASCII compatible on
	   here! */
	l1 = string_len(dirname);
	if (l1 > 0 && sep == give_unicode_from_string(dirname,l1-1)) {
	    int X = 0;
	    DPRINT(Debug,12,(&Debug,  
			     "browser_update_imaplisting: Removing %04X from %S\n",
			     sep,dirname));	       
	    newdirname = clip_from_string(dirname,&X,l1-1);
	} else
	    newdirname = dup_string(dirname);
	
	pattern = safe_strdup(newsysdir);
	buf[0] = sep;
	buf[1] = '\0';
	pattern = strmcat(pattern,buf);
	pattern = strmcat(pattern,"%");

	imap_command_push_astring(dir->a.imap_browser->Ch,pattern);
	free(pattern);
    }
    end_imap_command(dir->a.imap_browser->Ch);

    if (!imap_command_ok(dir->a.imap_browser->Ch,&ret1,NULL))
	goto fail;

    if (dir->sys_dir)
	free(dir->sys_dir);
    dir->sys_dir = newsysdir; newsysdir = NULL;

    if (dir->a.imap_browser->dir_name_cache)
	free_string(&(dir->a.imap_browser->dir_name_cache));    
    dir->a.imap_browser->dir_name_cache = dup_string(newdirname);

    if (dir->dirname)
	free_string(&(dir->dirname));

    {
	const struct remote_account * C =
	    give_con_remote_account(dir->a.imap_browser->Ch);

	dir->dirname = format_string(FRM("%s@%s%s%S"),
				     C->username ? C->username : "?",
				     C->host     ? C->host     : "?",
				     dir->sys_dir[0] != '/'  &&
				     dir->sys_dir[0] != '\0' ? ":" : "",
				     newdirname);
    }

    /* Update current separator */
    dir->a.imap_browser->cur_sep         = sep;

    /* Make empty selection folder */
    clear_dir_vector(dir);

    if (!(dir->sys_dir[0])) {     /* Top level directory */	
	int i;
	for (i = 0; i < dir->a.imap_browser->dir_entry_count; i++) {
	    /* WARNING: add_dir_vector does not allocate strings -- 
	     *          it just assign pointers!
	     */
	    add_dir_vector(dir,
			   safe_strdup(dir->a.imap_browser->
				       dir_entries[i].imap_name),
			   dup_string(dir->a.imap_browser->
				      dir_entries[i].translated_name),
			   dir->a.imap_browser->dir_entries[i].flags
			   );
	}
    } else {
	/* newdirname and dir->sys_dir does NOT include separator 
	   character on end 
	*/
	int L  = string_len(newdirname);
	int l1 = strlen(dir->sys_dir);  
	int m = 0,i;

	for (i = 0; i < dir->a.imap_browser->dir_entry_count; i++) {
	    struct imap_dir_entry * E = &(dir->a.imap_browser->dir_entries[i]);
	    int L1 = string_len(E->translated_name);
	    int l2 = strlen(E->imap_name);

	    if (L < L1 && l1 < l2) {	
		int X = 0;
		struct string * Z1 = clip_from_string(E->translated_name,&X,L);
		
		/* IMAP character set is assumed to be ASCII compatible on
		   here! */
		if (0 == string_cmp(newdirname,
				    Z1,-99 /* Unknown indicator */)  &&
		    X < L1 && E->sep == sep &&
		    give_unicode_from_string(E->translated_name,X) == sep &&

		    0 == strncmp(dir->sys_dir,E->imap_name,l1) &&
		    E->imap_name[l1] == sep) {
		    X++;
				    
		    if (X < L1) { /* No just directory name */

			/* WARNING: add_dir_vector does not allocate strings --
			 *          it just assign pointers!
			 */
			
			
			struct string * Z2 = NULL;
			
			Z2 = clip_from_string(E->translated_name,&X,L1);
			
			add_dir_vector(dir,safe_strdup(&(E->imap_name[l1+1])),
				       Z2,
				       dir->a.imap_browser->dir_entries[i].flags
				       );
			m++;
		    }
		}
	    }
	}
	DPRINT(Debug,12,(&Debug, 
			 "browser_update_imaplisting -- %d entries of %d matches to dirname\n",
			 m,dir->a.imap_browser->dir_entry_count));
    }
    
    ret = 1;
    
 fail:
    imap_clear_command(dir->a.imap_browser->Ch);

    if (newsysdir)
	free(newsysdir);
    if (newdirname)
	free_string(&newdirname);

    DPRINT(Debug,12,(&Debug, "browser_update_imaplisting=%d\n",ret));
    return ret;
}

/* Returns -1 on failure */
static int match_relname_to_imaplisting P_((struct folder_browser *dir,
					    const struct string *rel_name,
					    int *is_full));
static int match_relname_to_imaplisting(dir,rel_name,is_full)
     struct folder_browser *dir;
     const struct string *rel_name;
     int *is_full;
{
    int match_len = 0;
    int ret = -1;
    int L,i;
    
    DPRINT(Debug,12,(&Debug, 
		"match_relname_to_imaplisting: dir=%p\n", 
		dir));

    DPRINT(Debug,12,(&Debug, 
		     "match_relname_to_imaplisting: relname=%S\n",
		     rel_name));


    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"match_relname_to_imaplisting",
	      "Bad magic type",0);

    L = string_len(rel_name);
    *is_full = 0;

    for (i = 0; i < dir->a.imap_browser->dir_entry_count && !*is_full; i++) {
	int L1 = string_len(dir->a.imap_browser->dir_entries[i].
			    translated_name);
	
	if (L1 <= L) {	
	    int X = 0;
	    struct string * Z1 = clip_from_string(rel_name,&X,L1);

	    if (0 == string_cmp(dir->a.imap_browser->dir_entries[i].
				translated_name,
				Z1,-99 /* Unknown indicator */)) {

		/* IMAP character set is assumed to be ASCII compatible 
		   on here! */
		
		if (X < L &&		    
		    dir->a.imap_browser->dir_entries[i].sep ==
		    give_unicode_from_string(rel_name,X)) {
		    
		    DPRINT(Debug,12,(&Debug, 
				     "match_relname_to_imaplisting: %d: sep idx=%d -- found matching prefix\n",
				i,X));
		    if (X > match_len) {
			match_len = X;
			ret = i;
		    } 
		}
		
		if (X == L) {
		    DPRINT(Debug,12,(&Debug, 
				     "match_relname_to_imaplisting: %d: len = %d -- found matching name\n",
				i,X));
		    match_len = X;
		    ret = i;
		    *is_full = 1;
		}
	    }
	    free_string(&Z1);
	}
    }

    DPRINT(Debug,12,(&Debug, 
		     "match_relname_to_imaplisting=%d%s, match len = %d, *is_full=%d\n",
		     ret, ret == -1 ? " (no match)" : "",
		     match_len,
		     *is_full
		     ));
    return ret;
}

/* Returns name relative to directory of browser */
static struct string * browser_descend_imap P_((struct folder_browser *dir,
						const struct string *rel_name,
						struct string **disp_path,
						char *retsep));
static struct string * browser_descend_imap(dir,rel_name,disp_path, retsep)
     struct folder_browser *dir;
     const struct string *rel_name;
     struct string **disp_path;
     char *retsep;
{
    struct string * ret = NULL;

    char used_sep = '\0';
    int L,idx,is_full;
    int last_idx = 0;
    struct string * dirname_disp     = NULL;
    char *          dirname_sys      = NULL;

    DPRINT(Debug,12,(&Debug, "browser_descend_imap: dir=%p\n", dir));

    *retsep = '\0';

    *disp_path = NULL;

    if (!rel_name) {
	DPRINT(Debug,12,(&Debug, "browser_descend_imap=NULL\n"));
	return NULL;
    }
       
    L = string_len(rel_name);

    idx = match_relname_to_imaplisting(dir,rel_name,&is_full);
    if (idx < 0) {
	/* Start from root */
	DPRINT(Debug,12,(&Debug, 
			"browser_descend_imap: Restarting search from root\n"));

	if (!browser_update_imaplisting(dir,NULL,NULL,'\0'))
	    goto fail;
	idx = match_relname_to_imaplisting(dir,rel_name,&is_full);
    }


    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_descend_imap",
	      "Bad magic type",0);

    if (idx >= 0 && !is_full && imap_fast_lookup &&
	'\0' != (used_sep = dir->a.imap_browser->dir_entries[idx].sep)) {

	int idxN;
	int found = -1;

	DPRINT(Debug,12,(&Debug, 
			 "browser_descend_imap: imap_fast_lookup: Going to last directory on path, sep=%c\n",used_sep));
	
	for (idxN = 0; idxN < L; idxN++) {
	    uint16 code2 = give_unicode_from_string(rel_name,idxN);
	    /* IMAP character set is assumed to be ASCII compatible on
	       here! */

	    if (used_sep == code2)
		found = idxN;
	}
	DPRINT(Debug,12,(&Debug, 
			 "... last found idx=%d\n",found));

	if (found > 0) {
	    int X = 0;
	    struct string * dir_name = clip_from_string(rel_name,&X,found);
	    char * sys_name  = conv_to_imap_name(dir_name);

	    DPRINT(Debug,12,(&Debug, 
			     "... Using generated name %s (%S)\n",
			     sys_name,dir_name));

	    if (!browser_update_imaplisting(dir,dir_name,sys_name,
					    dir->a.imap_browser->
					    dir_entries[idx].sep)) {
		free_string(&dir_name);
		free(sys_name);
		goto fast_fail;
	    }
	    idx = match_relname_to_imaplisting(dir,rel_name,&is_full);

	    free_string(&dir_name);
	    free(sys_name);			
	}
	
    }

fast_fail:
    /* Descend to rigth directory */
    while (idx >= 0 && !is_full) {
	struct string * dir_name = dup_string(dir->a.imap_browser->
					      dir_entries[idx].
					      translated_name);
	char * sys_name          = safe_strdup(dir->a.imap_browser->
					       dir_entries[idx].imap_name);

	if (!browser_update_imaplisting(dir,dir_name,sys_name,
					dir->a.imap_browser->
					dir_entries[idx].sep)) {
	    free_string(&dir_name);
	    free(sys_name);
	    goto fail;
	}
	idx = match_relname_to_imaplisting(dir,rel_name,&is_full);



	free_string(&dir_name);
	free(sys_name);
    }

    if (is_full) {
	int x1 = -1;
	int x2 = -1;
	int i;
	int Llen = string_len(dir->a.imap_browser->dir_entries[idx].
			      translated_name);
	
	used_sep = dir->a.imap_browser->dir_entries[idx].sep;

	DPRINT(Debug,12,(&Debug, 
			 "browser_descend_imap: Using entry %d from listing: %s\n",
			 idx,dir->a.imap_browser->dir_entries[idx].imap_name));

	if ('\0' != used_sep) {
	    for (i = 0; dir->a.imap_browser->dir_entries[idx].imap_name[i]; 
		 i++) {
		if (used_sep == dir->a.imap_browser->dir_entries[idx].imap_name[i])
		    x1 = i;
	    }
	    
	    for (i = 0; i < Llen; i++) {
		/* IMAP character set is assumed to be ASCII compatible on
		   here! */
		if (used_sep == 
		    give_unicode_from_string(dir->a.imap_browser->dir_entries[idx].
					     translated_name,i))
		    x2 = i;
	    }
	}
	last_idx = x2+1;
	
	if (x2 > 0) {
	    int X = 0;
	    dirname_disp = clip_from_string(dir->a.imap_browser->dir_entries[idx].
					    translated_name,&X,x2);
	    dirname_sys = safe_malloc(x1+1);
	    strncpy(dirname_sys,dir->a.imap_browser->dir_entries[idx].imap_name,x1);
	    dirname_sys[x1] = '\0';
	} else {
	    dirname_disp = new_string(imap_charset);
	    dirname_sys  = safe_strdup("");
	}
	
    } else {
	int x2 = -1;

	if (dir->a.imap_browser->cur_sep) {
	    int i;

	    used_sep = dir->a.imap_browser->cur_sep;
	    for (i = 0; i < L; i++) {
		
		/* IMAP character set is assumed to be ASCII compatible on
		   here! */
		if (used_sep == give_unicode_from_string(rel_name,i))
		    x2 =i; 
	    }
	}
	last_idx = x2 +1; 

	if (x2 > 0) {
	    int X = 0;
	    dirname_disp = clip_from_string(rel_name,&X,x2);
	    dirname_sys  = conv_to_imap_name(dirname_disp);
	} else {
	    dirname_disp = new_string(imap_charset);
	    dirname_sys  = safe_strdup("");
	}
    }

    if (dir->dirname) {
	DPRINT(Debug,12,(&Debug, 
			 "browser_descend_imap: index %d, directory=%S sysname=%s\n",
			 last_idx,dirname_disp,dirname_sys));
    }
    if (used_sep) {
	DPRINT(Debug,12,(&Debug, 
			 "browser_descend_imap: Directory separator %c\n",
			 used_sep));
	*retsep = used_sep;
    }

    ret = clip_from_string(rel_name,&last_idx,string_len(rel_name));

    /* Check that current directory is right
       if not change to it ... */
	
    if ((dirname_sys[0] && !dir->sys_dir) ||
	(dir->sys_dir && 0 != strcmp(dirname_sys,dir->sys_dir))) {
	
	DPRINT(Debug,12,(&Debug, 
			 "browser_descend_imap: Match have found, but trying update directory %s -> %s\n",
			 dir->sys_dir ? dir->sys_dir : "<NULL>",
			 dirname_sys));
	    
	if (!browser_update_imaplisting(dir,dirname_disp,dirname_sys,
					used_sep)) {
	    DPRINT(Debug,12,(&Debug, 
			     "browser_descend_imap: OOPS! Can't update directory\n"));
	    free_string(&ret);   /* Delete answer */
	    goto fail;
	}
    } 

    if (dir->dirname) {
	DPRINT(Debug,12,(&Debug, 
			 "*** %S as relative to %S is %S\n",
			 rel_name,dir->dirname,ret));
    }
    if (dirname_disp) {
	DPRINT(Debug,12,(&Debug, 
			 "*** on directyry %S\n",		    
			 dirname_disp));
    }
        
fail:
    if (ret) {
	*disp_path = dirname_disp;
    } else {
	if (dirname_disp)
	    free_string(&dirname_disp);
    }
    if (dirname_sys)
	free(dirname_sys);

    if (!ret) {
	DPRINT(Debug,12,(&Debug, "browser_descend_imap=NULL\n"));
    } else {
	DPRINT(Debug,12,(&Debug, "browser_descend_imap=%S\n",ret));
    }
    return ret;
}

/* rel_dirname is relative to type -- not include user@hostname */
static int browser_change_imap P_((struct folder_browser *dir,
				   const struct string *rel_dirname,
				   struct string **dispname));
static int browser_change_imap(dir,rel_dirname,dispname)
     struct folder_browser *dir;
     const struct string *rel_dirname;
     struct string **dispname;
{
    int ret = 0;
    
    DPRINT(Debug,11,(&Debug, "browser_change_imap: dir=%p\n", dir));
    
    if (NULL ==  rel_dirname) {
	DPRINT(Debug,11,(&Debug, 
			 "browser_change_imap:  To default directory\n"));
	ret = browser_update_imaplisting(dir,NULL,NULL,'\0');
    } else {
	char sep = '\0';
	struct string * relative1 = NULL;
	struct string * relative  = browser_descend_imap(dir,
							 rel_dirname,
							 &relative1,
							 &sep);
	struct string * Lstr = NULL;
        char * str         = NULL;

	/* browser_select_generic uses 'rel_dirname' if
	   'relative' is null 
	*/
	browser_select_generic(dir,relative1,
			       rel_dirname,relative,&Lstr,&str,
			       sep,imap_charset);
	ret = browser_update_imaplisting(dir,Lstr,str,
					 sep);

	if (!ret) {
	    const struct remote_account * C =
		give_con_remote_account(dir->a.imap_browser->Ch);

	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDirErrorUnreadable,
			      "Directory %S unreadable on %s@%s"),
		      Lstr,
		      C->username ? C->username : "?",
		      C->host     ? C->host     : "?");
	}
	
	if (Lstr)
	    free_string(&Lstr);    
	if (str)
	    free(str);
	
	if (relative)
	    free_string(&relative);
	
	if (relative1)
	    free_string(&relative1);
    }

    DPRINT(Debug,11,(&Debug, "browser_change_imap=%d\n", ret));
    return ret;
}




static int browser_change_v_imap P_((struct folder_browser *dir,
				     struct name_vector *X,
				     struct string **dispname));
static int browser_change_v_imap(dir,X,dispname)
     struct folder_browser *dir;
     struct name_vector *X;
     struct string **dispname;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug, "browser_change_v_imap: dir=%p\n", dir));


    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_change_v_imap",
	      "Bad magic type",0);

    if (dir->dirname && dir->sys_dir) {

	struct string * relative1 = NULL;
	    

	struct string * Lstr = NULL;
	char          * str  = NULL;
	int l1 = strlen(dir->sys_dir);
	int add_sep = 0;
	int idx    = -1;
	int is_full;          /* Not used */

	if (l1 > 0 && dir->a.imap_browser->cur_sep == '\0') {
	    DPRINT(Debug,11,(&Debug, "browser_change_v_imap: No separator"));
	    goto fail;
	}
	
	if (! dir->a.imap_browser->dir_name_cache) {
	    DPRINT(Debug,11,(&Debug, "browser_change_v_imap: No dir_name_cache"));
	    goto fail;

	}

	relative1 = dup_string(dir->a.imap_browser->dir_name_cache);

	if (l1 > 0 && dir->a.imap_browser->cur_sep != dir->sys_dir[l1-1]) {
	    fill_ascii_to_string(relative1,1,dir->a.imap_browser->cur_sep);
	    add_sep = 1;
	}

	Lstr = cat_strings(relative1,
			   X->disp_name,
			   0);
	
	str = safe_strdup(dir->sys_dir);
	if (add_sep) {
	    char buf[2];
	    buf[0] = dir->a.imap_browser->cur_sep;
	    buf[1] = '\0';

	    str = strmcat(str,buf);
	}
	str = strmcat(str,X->sys_name);

	if (dir->a.imap_browser->cur_sep == '\0' &&
	    (idx = match_relname_to_imaplisting(dir,Lstr,&is_full)) < 0) {
	    
	    DPRINT(Debug,12,(&Debug, 
			     "browser_change_v_imap: Not found from IMAP listing -- trying update...\n"));

	    if (browser_update_imaplisting(dir,relative1,dir->sys_dir,
					   dir->a.imap_browser->cur_sep))
		idx = match_relname_to_imaplisting(dir,Lstr,&is_full);
	}

	if (dir->a.imap_browser->cur_sep == '\0' &&
	    idx >= 0) {
	    DPRINT(Debug,12,(&Debug, 
			     "browser_change_v_imap: Using name %s from listing, with idx=%d imap listing, separator=%c\n",
			str,idx,dir->a.imap_browser->dir_entries[idx].sep));

	    ret = browser_update_imaplisting(dir,Lstr,str,
					     dir->a.imap_browser->
					     dir_entries[idx].sep);
	    
	} else {	
	    DPRINT(Debug,12,(&Debug, 
			     "browser_change_v_imap: Using name %s from listing, with separator=%c\n",
			str,dir->a.imap_browser->cur_sep));

	    ret = browser_update_imaplisting(dir,Lstr,str,
					     dir->a.imap_browser->cur_sep);
	}
	if (!ret) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDirErrorUnreadable,
			      "Directory %S unreadable on %s@%s"),
		      Lstr,
		      dir->a.imap_browser->Ch->C.username ?
		      dir->a.imap_browser->Ch->C.username :
		      "?",
		      dir->a.imap_browser->Ch->C.host ?
		      dir->a.imap_browser->Ch->C.host :
		      "?");
	}

    fail:
	if (relative1)
	    free_string(&relative1);
    }

    DPRINT(Debug,11,(&Debug, "browser_change_v_imap=%d\n", ret));
    return ret;
}

static int browser_change_up_imap P_((struct folder_browser *dir,
				      struct string **dispname,
				      struct string ** disp_tail));
static int browser_change_up_imap(dir,dispname,disp_tail)
     struct folder_browser *dir;
     struct string **dispname;
     struct string ** disp_tail;
{
    int ret = 0;
          
    DPRINT(Debug,11,(&Debug, "browser_change_up_imap: dir=%p\n", dir));

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_change_up_imap",
	      "Bad magic type",0);


    if (dir->a.imap_browser->cur_sep != '\0' &&
	dir->a.imap_browser->dir_name_cache) {

	const struct string * relative1 = 
	    dir->a.imap_browser->dir_name_cache;

	int L         = string_len(relative1);

	if (L > 0) {   /* In case of L1 == 0 returns default imap directory */

	    /* ASCII assumed on separator */
	    int L1 = browser_change_up_helper(relative1,
					  dir->a.imap_browser->cur_sep);

	    int X = 0;
	    struct string * A1 = clip_from_string(relative1,&X,L1);
	    
	    if (disp_tail) 
		*disp_tail = browser_disp_tail_helper(relative1,X,dir->a.imap_browser->cur_sep);

	    /* Need not generate actual directory listing */
	    
	    if (*dispname)
		free_string(dispname);	    
	    *dispname = format_string(FRM("%s@%s%s%S"),
				      dir->a.imap_browser->Ch->C.username ?
				      dir->a.imap_browser->Ch->C.username :
				      "?",
				      dir->a.imap_browser->Ch->C.host ?
				      dir->a.imap_browser->Ch->C.host :
				      "?",
				      dir->sys_dir[0] != '/'  &&
				      dir->sys_dir[0] != '\0' ? ":" : "",
				      A1);
	    free_string(&A1);
	    
	    ret = 1;
	} else
	    ret = -1;   /* Signals caller to generate default menu */

    } else
	ret = -1;   /* Signals caller to generate default menu */

    DPRINT(Debug,11,(&Debug, "browser_change_up_imap=%d\n", ret));
    return ret;
}

/* rel_dirname is relative to type -- not include user@hostname */
static int browser_select_imap P_((struct folder_browser *dir,
				   const struct string *rel_itemname,
				   struct string **dispname,
				   int *newpos));
static int browser_select_imap(dir,rel_itemname,dispname,newpos)
     struct folder_browser *dir;
     const struct string *rel_itemname;
     struct string **dispname;
     int *newpos;
{
    int ret = 0;
    char sep = '\0';
    struct string * relative1 = NULL;
    struct string * relative = NULL;

    DPRINT(Debug,11,(&Debug, "browser_select_imap: dir=%p\n", dir));
    if (rel_itemname) {
	DPRINT(Debug,11,(&Debug, "browser_select_imap: rel_itemname=%S\n", 
			 rel_itemname));
    } else {
	DPRINT(Debug,11,(&Debug, "browser_select_imap: rel_itemname=NULL\n"));
    }

    relative = browser_descend_imap(dir,rel_itemname,&relative1,&sep);

    if (relative) {
	struct string * Lstr =  NULL;
	char          * str  = NULL;
	int idx = browser_select_generic(dir,relative1,
					 rel_itemname,relative,
					 &Lstr,&str,
					 sep,imap_charset);
	if (idx >= 0) {
	    DPRINT(Debug,11,(&Debug, 
			     "browser_select_imap: Using index %d: %s\n",
			     idx,dir->vector[idx].sys_name));

	    /* WARNING: set_dir_selection does not allocate strings -- 
	     *          it just assign pointers!
	     */
	    set_dir_selection(dir,
			      safe_strdup(dir->vector[idx].sys_name),
			      dup_string(dir->vector[idx].disp_name),
			      dir->vector[idx].flags
			      |BROWSER_MAILFILE   /* Assume folder    */
			      |BROWSER_EXIST   /* Exists because on listing */
			      );

	    if (newpos)
		*newpos = idx;
	    
	} else {
	    char * rel_str = conv_to_imap_name(relative);
	    int F = 0;

	    DPRINT(Debug,11,(&Debug, 
			"browser_select_imap: Using given name: %s\n",
			rel_str));


	    if (-2 == idx) {
		DPRINT(Debug,11,(&Debug,
				 "browser_select_imap: Directory prefix was given.\n"));
		F |= BROWSER_DIRPREFIX;
	    } else
		F |= BROWSER_MAILFILE;   /* Assume folder    */


	    /* WARNING: set_dir_selection does not allocate strings -- 
	     *          it just assign pointers!
	     */
	    set_dir_selection(dir,rel_str,dup_string(relative),
			      F);	    
	}
	ret = 1;

	free_string(&relative);
	free_string(&Lstr);
	free(str);
    }

    if (relative1)
	free_string(&relative1);

    DPRINT(Debug,11,(&Debug, "browser_select_imap=%d\n", ret));
    return ret;
}

static struct string * browser_imap_N P_((struct folder_browser *dir));
static struct string * browser_imap_N(dir)
     struct folder_browser *dir;
{
    static struct string *ret;
    
    if (!dir->sys_dir)
	dir->sys_dir = safe_strdup("");

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__," browser_imap_N",
	      "Bad magic type",0);

    /* OOPS */
    if (!dir->dirname) {	
	dir->dirname = format_string(FRM("%s@%s%s%s"),
				     dir->a.imap_browser->Ch->C.username ?
				     dir->a.imap_browser->Ch->C.username :
				     "?",
				     dir->a.imap_browser->Ch->C.host ?
                                     dir->a.imap_browser->Ch->C.host :
                                     "?",
				     dir->sys_dir[0] != '/'  &&
				     dir->sys_dir[0] != '\0' ? ":" : "",
				     dir->sys_dir);
    }

    if (dir->filter)
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeImapDirFilter,
				    "IMAP directory %S with filter %S"),
			    dir->dirname,dir->filter);
    else
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeImapDir,
				    "IMAP directory %S"),
			    dir->dirname);

    return ret;
}

static struct string * browser_give_title_imap P_((struct folder_browser *dir));
static struct string * browser_give_title_imap(dir)
     struct folder_browser *dir;
{
    static struct string *ret;
    
    DPRINT(Debug,11,(&Debug, "browser_give_title_imap: dir=%p\n", dir));
    
    ret = browser_imap_N(dir);
    
    DPRINT(Debug,12,(&Debug, "browser_give_title_imap=%p\n", ret));

    return ret;
}


static char browser_separator_imap P_((struct folder_browser *dir));
static char browser_separator_imap(dir)
     struct folder_browser *dir;
{
    char ret;

    DPRINT(Debug,11,(&Debug, "browser_separator_imap: dir=%p\n", dir));

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_separator_imap ",
	      "Bad magic type",0);

    ret = dir->a.imap_browser->cur_sep;
    
    if ('\0' == ret) {
	DPRINT(Debug,11,(&Debug, "browser_separator_imap=NUL"));
    } else {
	DPRINT(Debug,11,(&Debug, "browser_separator_imap=%c\n", ret));
    }
    return ret;
}

static struct string * browser_name_imap P_((struct folder_browser *dir));
static struct string * browser_name_imap(dir)
     struct folder_browser *dir;
{
    static struct string *ret;

    DPRINT(Debug,11,(&Debug, 
		     "browser_name_imap: dir=%p\n",dir));

    ret = browser_imap_N(dir);

    DPRINT(Debug,11,(&Debug, "browser_name_imap=%p\n", ret));

    return ret;
}


static struct string * browser_cat_imap P_((struct folder_browser *dir,
					    const struct string * item));
static struct string * browser_cat_imap(dir,item)
     struct folder_browser *dir;
     const struct string * item;
{
    struct string * ret = NULL;

    DPRINT(Debug,11,(&Debug, "browser_cat_imap: dir=%p\n", 
		dir));


    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_cat_imap",
	      "Bad magic type",0);

    if (dir->dirname) {

	struct string * relative1 = NULL;
	    
	int L;

	if (! dir->a.imap_browser->dir_name_cache) {
	    DPRINT(Debug,11,(&Debug, "browser_cat_imap: No dir_name_cache!\n"));
	    goto fail;
	}
	relative1 = dup_string(dir->a.imap_browser->dir_name_cache);

	L = string_len(relative1);

	if (L > 0) {
	    if (dir->a.imap_browser->cur_sep == '\0') {
		DPRINT(Debug,11,(&Debug, "browser_cat_imap: No separator!\n"));
		goto fail;

	    } else if ( dir->a.imap_browser->cur_sep !=   /* ASCII assumed! */
		       give_unicode_from_string(dir->dirname,L-1))
		fill_ascii_to_string(relative1,1,dir->a.imap_browser->cur_sep);
	}

	ret = format_string(FRM("%s@%s%s%S"),
			    dir->a.imap_browser->Ch->C.username ?
			    dir->a.imap_browser->Ch->C.username :
			    "?",
			    dir->a.imap_browser->Ch->C.host ?
			    dir->a.imap_browser->Ch->C.host :
			    "?",
			    dir->sys_dir[0] != '/'  &&
			    (L > 0 || item) ? ":" : "",
			    relative1);

	/* Some IMAP directory! */
	if (item) {
	    append_string(&ret, item, 0);
	}

    fail:
	if (relative1)
	    free_string(&relative1);

    } else {        /* !!!! XXXXX Should not happen */
	ret = format_string(FRM("%s@%s%s"),
			    dir->a.imap_browser->Ch->C.username ?
			    dir->a.imap_browser->Ch->C.username :
			    "?",
			    dir->a.imap_browser->Ch->C.host ?
			    dir->a.imap_browser->Ch->C.host :
			    "?",
			    item ? ":" : "");
	if (item) {
	    append_string(&ret, item, 0);
	}
    } 
	
    if (ret) {
	DPRINT(Debug,11,(&Debug, "browser_cat_imap=%S\n",ret));
    } else {
	DPRINT(Debug,11,(&Debug, "browser_cat_imap=NULL\n"));
    }
    return ret;
}

static int fullname_from_imap_selection P_((struct folder_browser *dir,
					     struct string ** Lstr,
					     char ** str));
static int fullname_from_imap_selection(dir,Lstr,str)
     struct folder_browser *dir;
     struct string ** Lstr;
     char ** str;
{
    int ret = 0;

    *str = NULL;
    *Lstr = NULL;

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"fullname_from_imap_selection",
	      "Bad magic type",0);

    if (!dir->sys_dir) {
	
	DPRINT(Debug,12,(&Debug, 
		    "fullname_from_imap_selection: On default directory\n"));

	if (!dir->selection->sys_name) {
	    if (dir->selection->disp_name)
		panic("MBX PANIC",__FILE__,__LINE__,
		      "fullname_from_imap_selection",
		      "selection sys_name not given but disp_name exists",0);

	    DPRINT(Debug,12,(&Debug, 
			     "fullname_from_imap_selection: Assuming INBOX\n"));

	    *str = safe_strdup("INBOX");
	    *Lstr = new_string2(display_charset,s2us(*str));

	} else {
	       
	    *str  = safe_strdup(dir->selection->sys_name);
	    *Lstr = dup_string(dir->selection->disp_name);
	}
	ret = 1;

    } else if (dir->selection->sys_name &&
	       dir->selection->disp_name) {
	int add_sep = 0;

	struct string * relative1 = NULL;
	    
	if (! dir->a.imap_browser->dir_name_cache) {
	    DPRINT(Debug,12,(&Debug, 
			     "fullname_from_imap_selection: No dir_name_cache\n"));
	    goto fail;
	}
	relative1 = dup_string(dir->a.imap_browser->dir_name_cache);

	if (dir->sys_dir[0]) {
	    int l1 = strlen(dir->sys_dir);
	    
	    if ('\0' == dir->a.imap_browser->cur_sep) {
		DPRINT(Debug,12,(&Debug, 
			    "fullname_from_imap_selection: No separator\n"));

		free_string(&relative1);
		goto fail;
	    }
	    
	    if (l1 > 0 && dir->a.imap_browser->cur_sep != dir->sys_dir[l1-1]) {
		fill_ascii_to_string(relative1,1,dir->a.imap_browser->cur_sep);
		add_sep = 1;
	    }
	}
	
	*Lstr = cat_strings(relative1,
			    dir->selection->disp_name,
			    0);
	
	*str = safe_strdup(dir->sys_dir);
	if (add_sep) {
	    char buf[2];
	    buf[0] = dir->a.imap_browser->cur_sep;
	    buf[1] = '\0';
	    
	    *str = strmcat(*str,buf);
	}
	*str = strmcat(*str,dir->selection->sys_name);
	ret = 1;

	free_string(&relative1);

    } else {
	DPRINT(Debug,12,(&Debug, 
			 "fullname_from_imap_selection: No selection name\n"));
	goto fail;

    }

    DPRINT(Debug,12,(&Debug, 
		     "fullname_from_imap_selection: selection %S, sysname %s\n",
		     *Lstr,*str));

 fail:
    
    DPRINT(Debug,11,(&Debug, "fullname_from_imap_selection=%d\n", 
		     ret));

    return ret;
}

static struct folder_info * 
browser_folder_from_imap P_((struct folder_browser *dir,
			     int treat_as_spooled));

static struct folder_info * browser_folder_from_imap(dir,treat_as_spooled)
     struct folder_browser *dir;
     int treat_as_spooled;
{
    struct folder_info * res = NULL;
    struct string * Lstr = NULL;
    char *          str  = NULL;
    char *          str1  = NULL;
    /* Make backup */
    char * username = NULL;
    char * host     = NULL;
    enum connection_state S;

    DPRINT(Debug,11,(&Debug, "browser_folder_from_imap: dir=%p\n", 
		     dir));

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_folder_from_imap",
	      "Bad magic type",0);

    username = safe_strdup(dir->a.imap_browser->Ch->C.username);
    host     = safe_strdup(dir->a.imap_browser->Ch->C.host);

    if (!fullname_from_imap_selection(dir,&Lstr,&str))
	goto fail;
    
    DPRINT(Debug,12,(&Debug, 
		     "browser_folder_from_imap: Using name %s from selection\n",
		     str));
    
    res = mbx_new_folder();
    
    if (str[0] != '/' && str[0] != '\0')
	str1 = safe_strdup(":");
    str1 = strmcat(str1,str);

    res -> cur_folder_sys = 
	elm_message(FRM("%s@%s%s"),username,host,
		    str1);
    
    res -> cur_folder_disp =
	format_string(FRM("%s@%s%s%S"),username,host,
		      str[0] != '/'  &&
		      str[0] != '\0' ? ":" : "",
		      Lstr);
    
    res -> folder_type = IMAP_MBX;
    res->folder_type->init_it(res);

    res->p->a.imap_mbx.folder            = safe_strdup(str);    
    res->p->a.imap_mbx.folder_name_cache = dup_string(Lstr);


    /* IMAP temporary mailboxs are always put to
       temp_dir */
    set_remote_tempmbox(res,
			& (dir->a.imap_browser->Ch->C));
    
    if ((S = get_connection_state(dir->a.imap_browser->Ch)) == CON_logged) {
	
	DPRINT(Debug,12,(&Debug, 
			 "browser_folder_from_imap: Transfering connection to folder...\n"));
	
	/* Transfer connection from browser to folder */
	if (!join_connection_hm(res -> p->a.imap_mbx.Ch,
				&(dir->a.imap_browser->Ch->C),
				S,dir->a.imap_browser->passhm))
	    res->p->a.imap_mbx.Ch->a.imap_con->imap_state = IMAP_error;
	else {
	    res->p->a.imap_mbx.Ch->a.imap_con->imap_state = IMAP_idle;
	}
	
	if (res -> p->a.imap_mbx.passhm)
	    free_browser_passhm(& (res -> p->a.imap_mbx.passhm));
	res -> p->a.imap_mbx.passhm =
	    dir->a.imap_browser->passhm;
	if (res -> p->a.imap_mbx.passhm)
	    inc_browser_passhm_refcount(res -> p->a.imap_mbx.passhm);

	/* Signal to browser_update_imaplisting() for reopening ...*/
	if (NULL == dir->a.imap_browser->Ch->C.stream)
	    set_connection_state(dir->a.imap_browser->Ch,CON_error);
	
	/* return from backup is needed */
	if (!dir->a.imap_browser->Ch->C.username) {
	    dir->a.imap_browser->Ch->C.username = username;
	    username = NULL;
	}
	if (!dir->a.imap_browser->Ch->C.host) {
	    dir->a.imap_browser->Ch->C.host = host;
	    host = NULL;
	}
    } else {
	/* Glue */
	if (!res -> p->a.imap_mbx.Ch->C.username) {
	    res -> p->a.imap_mbx.Ch->C.username = username;
	    username = NULL;
	}
	
	if (!res -> p->a.imap_mbx.Ch->C.host) {
	    res -> p->a.imap_mbx.Ch->C.host = host;
	    host = NULL;
	}
    }

 fail:
    if (username)
	free(username);
    if (host)
	free(host);
    if (str)
	free(str);
    if (str1)
	free(str1);
    if (Lstr)
	free_string(&Lstr);

    if (res) {
	DPRINT(Debug,11,(&Debug, "browser_folder_from_imap=%p\n",res));
    } else {
	DPRINT(Debug,11,(&Debug, "browser_folder_from_imap=NULL\n"));
    }
    return res;
}

static int browser_create_selection_imap P_((struct folder_browser *dir));
static int browser_create_selection_imap(dir)
     struct folder_browser *dir;
{
    int ret = 0;
    struct string * Lstr = NULL;
    char *          str  = NULL;
    imap_states     ret1;

    DPRINT(Debug,11,(&Debug, "browser_create_selection_imap: dir=%p\n", 
		     dir));

    if (!fullname_from_imap_selection(dir,&Lstr,&str))
	goto fail;
    
    DPRINT(Debug,12,(&Debug, 
		     "browser_create_selection_imap: Using name %s (%S) from selection\n",
		     str,Lstr));
    
    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__," browser_create_selection_imap",
	      "Bad magic type",0);

    /* browser_folder_from_imap may close connection ... */
    if (get_connection_state(dir->a.imap_browser->Ch) != CON_logged) {

	DPRINT(Debug,12,(&Debug, 
			 "browser_create_selection_imap: con=%p state=%d\n",
			 dir->a.imap_browser->Ch,
			 get_connection_state(dir->a.imap_browser->Ch)));


	if (!login_connection_hm(dir->a.imap_browser->Ch,
				 dir->a.imap_browser->passhm))
	    goto fail;
    }
    
    if (!start_imap_command(dir->a.imap_browser->Ch,"CREATE"))
	goto fail;
    
    /* Mailbox name */
    imap_command_push_astring(dir->a.imap_browser->Ch,str);
    
    end_imap_command(dir->a.imap_browser->Ch);
	
    if (!imap_command_ok(dir->a.imap_browser->Ch,&ret1,NULL))
	goto fail;
    
    ret = 1;
 fail:
    imap_clear_command(dir->a.imap_browser->Ch);
    
    if (str)
	free(str);
    if (Lstr)
	free_string(&Lstr);

    DPRINT(Debug,11,(&Debug, "browser_create_selection_imap=%d\n", ret));
    
    return ret;
}

static void zero_ws_fields_imap P_((WRITE_STATE ptr));
static void zero_ws_fields_imap(ptr)
     WRITE_STATE ptr;
{
    DPRINT(Debug,12,(&Debug, 
		     "zero_ws_fields_imap; ptr=%p\n",ptr));

    ptr->a.imap.literal_len = 0;
    ptr->a.imap.literal     = NULL;
}

static void free_ws_fields_imap P_((WRITE_STATE ptr));
static void free_ws_fields_imap(ptr)
     WRITE_STATE ptr;
{
    DPRINT(Debug,12,(&Debug, 
		"free_ws_fields_imap; ptr=%p\n",ptr));
    if (ptr->a.imap.literal) {
	free(ptr->a.imap.literal);
	ptr->a.imap.literal_len = 0;
	ptr->a.imap.literal     = NULL;
    }
}

static int browser_prepare_write_imap P_((struct folder_browser *dir,
					  WRITE_STATE ptr));
static int browser_prepare_write_imap(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug, 
		"browser_prepare_write_imap: dir=%p, ws=%p\n", 
		dir,ptr));

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_prepare_write_imap",
	      "Bad magic type",0);

    /* browser_folder_from_imap may close connection ... */
    if (get_connection_state(dir->a.imap_browser->Ch) != CON_logged) {

	DPRINT(Debug,12,(&Debug, 
			 "browser_prepare_write_imap: con=%p state=%d\n",
			 dir->a.imap_browser->Ch,
			 get_connection_state(dir->a.imap_browser->Ch)));

	if (!login_connection_hm(dir->a.imap_browser->Ch,
				 dir->a.imap_browser->passhm))
	    goto fail;
    }

    /* Should we check on here that mailbox exists? */
    ret = 1;

 fail:
    DPRINT(Debug,11,(&Debug, 
		"browser_prepare_write_imap=%d\n",ret));
    return ret;
}

static int browser_sync_write_imap P_((struct folder_browser *dir,
					WRITE_STATE ptr));
static int browser_sync_write_imap(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    DPRINT(Debug,11,(&Debug, 
		     "browser_sync_write_imap=1:    dir=%p, ws=%p\n", 
		     dir,ptr));
    
    return 1;
}


static int browser_end_write_imap P_((struct folder_browser *dir,
			     WRITE_STATE ptr));
static int browser_end_write_imap(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    DPRINT(Debug,11,(&Debug, 
		"browser_end_write_imap=1:    dir=%p, ws=%p\n", 
		dir,ptr));
    
    return 1;
}

static long browser_tell_imap_ws P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr));
static long browser_tell_imap_ws(dir,write_state_ptr)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
{
    DPRINT(Debug,11,(&Debug, "browser_tell_dummy_ws=-1: dir=%p\n", 
		     dir));   
    return -1;
}

static int browser_seek_imap_ws P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr,
				      long pos));
static int browser_seek_imap_ws(dir,write_state_ptr,pos)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     long pos;
{
    DPRINT(Debug,11,(&Debug, "browser_seek_imap_ws=0: dir=%p\n", 
		dir));

    return 0;
}

static int browser_write_imap_ws P_((struct folder_browser *dir,
				      WRITE_STATE ptr,
				      int l, const char *buffer));
static int browser_write_imap_ws(dir,ptr,l,buffer)
     struct folder_browser *dir;
     WRITE_STATE ptr;
     int l; 
     const char *buffer;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug, "browser_write_imap_ws: dir=%p, ptr=%p\n", 
		     dir,ptr));

    if (l < 0) {
	panic("MBX PANIC",__FILE__,__LINE__,
	      "browser_write_imap_ws",
	      "Negative write len",0);

    }
    
    ptr->a.imap.literal = 
	safe_realloc(ptr->a.imap.literal,
		     ptr->a.imap.literal_len + l + 1);

    memcpy(ptr->a.imap.literal + ptr->a.imap.literal_len, buffer,l);

    ptr->a.imap.literal_len += l;
    ptr->a.imap.literal[ptr->a.imap.literal_len] = '\0';
    ret = 1;
    
    DPRINT(Debug,11,(&Debug, "browser_write_imap_ws=%d\n",ret));
    return ret;
}

static int browser_start_we_imap P_((struct folder_browser *dir,
				     WRITE_STATE write_state_ptr,
				     int write_envelope,
				     struct header_rec *current_header,
				     int *env_flags));
static int browser_start_we_imap(dir,write_state_ptr,
				 write_envelope,current_header,
				 env_flags)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
     int *env_flags;
{
    int ret = 0;
    struct string * Lstr  = NULL;
    char *          str   = NULL;
    long            flags = 0;
    const char ** LIST = safe_calloc(( IMAP_flags_count +1 ), sizeof(LIST[0]));
    int j;
    int X;
    char *dtime = NULL;
    int len;

    struct tm tm_val;		/* Time structure, see CTIME(3C) */
    long      tzmin;		/* number of minutes off gmt 	 */
    int	  tzsign;		/* + or - gmt 			 */
    struct tm * tm1;
    
    DPRINT(Debug,11,(&Debug, 
		     "browser_start_we_imap: dir=%p, ws=%p, header=%p\n", 
		     dir,write_state_ptr,current_header));

    *env_flags = WE_ADD_RETURN_PATH | WE_USE_CRLF;

    free_ws_fields_imap(write_state_ptr);
    
    if (!fullname_from_imap_selection(dir,&Lstr,&str))
	goto fail;
    
    DPRINT(Debug,12,(&Debug, 
		     "browser_start_we_imap: Using name %s from selection\n",
		     str));
        
    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_start_we_imap",
	      "Bad magic type",0);


    if (!start_imap_command(dir->a.imap_browser->Ch,"APPEND"))
	goto fail;

    /* Mailbox name */
    imap_command_push_astring(dir->a.imap_browser->Ch,str);
    

    if (0 == (current_header->status & UNREAD))
	flags |= IMAP_Seen;

    for (j = 0; j < IMAP_flags_count; j++) {

	if (0 != (IMAP_flags[j].mask_elm & current_header->status))
	    flags |= IMAP_flags[j].mask_imap;		 

	if (0 != (IMAP_flags[j].mask_elm1 & current_header->status1))
	    flags |= IMAP_flags[j].mask_imap;		 
    }
    
    DPRINT(Debug,9,(&Debug, 
		     "browser_start_we_imap: Flags"));
    
    X = 0;
    for (j = 0; j < IMAP_flags_count; j++) {
	if (0 != (flags & IMAP_flags[j].mask_imap) &&
	    0 == (IMAP_Recent     & IMAP_flags[j].mask_imap)) {
	    LIST[X++] = IMAP_flags[j].flag;
	    DPRINT(Debug,9,(&Debug, " %s",IMAP_flags[j].flag));
	}
    }	    
    LIST[X] = NULL;

    DPRINT(Debug,9,(&Debug, "\n"));

    /* Flag list */
    imap_command_push_list(dir->a.imap_browser->Ch,LIST);

    tm1 = localtime(& current_header->received_time);
    if (tm1) {
	/*
	 * The get_tz_mins() routine steps on the static data returned
	 * by localtime(), so we need to save off the value obtained here.
	 */
	tm_val = *tm1;
	
	if ((tzmin = -get_tz_mins(current_header->received_time)) >= 0) {
	    tzsign = '+';
	} else {
	    tzsign = '-';
	    tzmin = -tzmin;
	}
	

	dtime = elm_message(FRM("%02d-%s-%04d %02d:%02d:%02d %c%02d%02d"),
			    tm_val.tm_mday,arpa_monname[tm_val.tm_mon],
			    tm_val.tm_year + 1900,
			    tm_val.tm_hour,tm_val.tm_min,tm_val.tm_sec,
			    tzsign, tzmin / 60, tzmin % 60);
	
	if ((len = strlen(dtime)) != 26) {
	    DPRINT(Debug,1,(&Debug, 
			    "browser_start_we_imap: generated date 26 char string expected, got %d chars: %s\n",
			    len,dtime));
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "browser_start_we_imap",
		  "generated date-time have wrong length",0);
	}

	/* date-time */
	imap_command_push_string(dir->a.imap_browser->Ch,dtime);
	
	ret = 1;
    } else {
	ret = 0;
	DPRINT(Debug,12,(&Debug, 
			 "browser_start_we_imap: received_time=%ld -- localtime failed\n",
			 (long)(current_header->received_time)));
    }
	
 fail:
    /* Clear command only if we fail ... */
    if (!ret)
	imap_clear_command(dir->a.imap_browser->Ch);    
    if (str)
	free(str);
    if (Lstr)
	free_string(&Lstr);
    if (dtime)
	free(dtime);
    free(LIST);

    DPRINT(Debug,11,(&Debug, "browser_start_we_imap=%d\n", ret));

    return ret;
}


static int browser_end_we_imap P_((struct folder_browser *dir,
				  WRITE_STATE write_state_ptr,
				  int write_envelope,
				  struct header_rec *current_header));
static int browser_end_we_imap(dir,write_state_ptr,
			      write_envelope,current_header)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
{
    imap_states ret1;
    int ret = 0;

    DPRINT(Debug,11,(&Debug, 
		     "browser_end_we_imap: dir=%p, ws=%p, header=%p\n", 
		     dir,write_state_ptr,current_header));


    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__," browser_end_we_imap",
	      "Bad magic type",0);

    if (0 == write_state_ptr->a.imap.literal_len)
	imap_command_push_literal(dir->a.imap_browser->Ch,
				  write_state_ptr->a.imap.literal_len,
				  "");

    else
	imap_command_push_literal(dir->a.imap_browser->Ch,
				  write_state_ptr->a.imap.literal_len,
				  write_state_ptr->a.imap.literal);
    
    end_imap_command(dir->a.imap_browser->Ch);

    if (!imap_command_ok(dir->a.imap_browser->Ch,&ret1,NULL))
	goto fail;
    
    ret = 1;
 fail:
    imap_clear_command(dir->a.imap_browser->Ch);
    free_ws_fields_imap(write_state_ptr);

    DPRINT(Debug,11,(&Debug, "browser_end_we_imap=%d\n", ret));
    return ret;
}

int imap_same_host_user_nopath P_((const struct folder_info *fh,
				   const struct remote_account *ra2));
int imap_same_host_user_nopath(fh,ra2)
     const struct folder_info *fh;
     const struct remote_account *ra2;
{
    int ret = 0;

    if (fh->folder_type != &imap_mbx)
	panic("MBX PANIC",__FILE__,__LINE__,"imap_same_host_user_nopath",
	      "Bad folder type",0);

    if (PRIVATE_DATA_magic != fh->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"imap_same_host_user_nopath",
	      "Bad magic number (private_data)",0);

    if (same_remote_account_host_user(& (fh->p->a.imap_mbx.Ch->C),
				      ra2)) {

	if (! fh->p->a.imap_mbx.folder[0] ||
	    0 == strcmp("INBOX",fh->p->a.imap_mbx.folder)) {
	    ret = 1;
	} else {
	    DPRINT(Debug,11,(&Debug, 
			     "imap_same_host_user_nopath: folder %s\n",
			     fh->p->a.imap_mbx.folder));
	}

    } else {
	DPRINT(Debug,11,(&Debug, 
			 "imap_same_host_user_nopath: Host or username does not match\n"));
	 ret = 0;
    }

    DPRINT(Debug,11,(&Debug, "imap_same_host_user_nopath=%d -- not supported?\n", ret));

    return ret;
}


static int browser_selection_is_imap P_((struct folder_browser *dir,
					 struct folder_info *folder));
static int browser_selection_is_imap(dir,folder)
     struct folder_browser *dir;
     struct folder_info *folder;
{
    int ret = 0;
    DPRINT(Debug,11,(&Debug, "browser_selection_is_imap: dir=%p\n", 
		     dir));
    

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_selection_is_imap",
	      "Bad magic type",0);

    if (folder->folder_type != &imap_mbx) {
	DPRINT(Debug,11,(&Debug, 
			 "browser_selection_is_imap: Not a imap folder\n"));
	ret = 0;
    } else {
	struct connection_cache * ch1 = dir->a.imap_browser->Ch;
	struct connection_cache * ch2;
	
	if (PRIVATE_DATA_magic != folder->p->magic) 
	    panic("MBX PANIC",__FILE__,__LINE__,"browser_selection_is_imap",
		  "Bad magic number (private_data)",0);
	
	ch2 = folder->p->a.imap_mbx.Ch;

	if (same_remote_account_host_user(& (ch1->C),
					  & (ch2->C))) {
	    struct string * Lstr = NULL;
	    char *          str  = NULL;
	    
	    if (!fullname_from_imap_selection(dir,&Lstr,&str))
		goto fail;
	    
	    ret = 0 == strcmp(str,folder->p->a.imap_mbx.folder);

	    if (!ret) {
		DPRINT(Debug,11,(&Debug, 
				 "browser_selection_is_imap: %s <> %s\n",
				 str,folder->p->a.imap_mbx.folder));
	    }

	fail:
	    if (str)
		free(str);
	    if (Lstr)
		free_string(&Lstr);
	} else {
	    DPRINT(Debug,11,(&Debug, "browser_selection_is_imap: Host or username does not match\n"));
	    ret = 0;
	}
    }

    DPRINT(Debug,11,(&Debug, "browser_selection_is_imap=%d\n",ret));
    return ret;
}


static int browser_make_ref_imap P_((struct folder_browser *dir,
				     char **refname, int *iscopy,
				     int is_text));
static int browser_make_ref_imap(dir,refname,iscopy,is_text)
     struct folder_browser *dir;
     char **refname; 
     int *iscopy;
     int is_text;
{
    
    DPRINT(Debug,11,(&Debug, "browser_make_ref_imap=0: dir=%p\n", 
		     dir));
    return 0;
}

static void browser_update_imap P_((struct folder_browser *dir));
static void browser_update_imap(dir)
     struct folder_browser *dir;
{
    panic("BROWSER PANIC",__FILE__,__LINE__,"browser_update_imap",
	  "browser_update_imap() called",0);	
}

static void browser_imap_do_stat P_((struct folder_browser *dir,
				      int idx));
static void browser_imap_do_stat(dir,idx)
     struct folder_browser *dir;
     int idx;
{
    panic("BROWSER PANIC",__FILE__,__LINE__,"browser_update_imap",
	  "browser_update_imap() called",0);	
}

static int sort_by_name P_((const void *A, const void *B));
static int sort_by_name(A,B)
     const void *A; 
     const void *B;
{
    const struct name_vector * A1 = A;
    const struct name_vector * B1 = B;

    int r = string_cmp(A1->disp_name,
		       B1->disp_name,
		       0  /* compare fialure */);

    if (0 == r)
	r = strcmp(A1->sys_name,B1->sys_name);
   
    return r;
}

static int sort_by_revname P_((const void *A, const void *B));
static int sort_by_revname(A,B)
     const void *A; 
     const void *B;
{
    return -sort_by_name(A,B);
}

S_(browser_folder_sort_dir browser_folder_sort_imap)
static void browser_folder_sort_imap P_((struct folder_browser *dir,
					  print_sort_message * print));
static void browser_folder_sort_imap(dir,print)
     struct folder_browser *dir;
     print_sort_message * print;
{
    struct string * msg = NULL;

    if (dir->vector_len < 2)
	return;

    switch (give_dt_sort_as_int(&imap_dir_sortby)) {
    case ID_NONE:                 return;
    case ID_NAME_SORT:
	
	if (dir->vector_len > 10) {
	    msg = format_string(CATGETS(elm_msg_cat, MeSet, MeSortingDirByName, 
					"Sorting directory %S by Name"),
				dir->dirname);
	    print(msg);
	}
	qsort(dir->vector,dir->vector_len,sizeof (dir->vector[0]),
	      sort_by_name);
	break;
    case REVERSE ID_NAME_SORT:
	
	if (dir->vector_len > 10) {
	    msg = format_string(CATGETS(elm_msg_cat, MeSet, MeSortingDirByRevName, 
					"Sorting directory %S by Reverse Name"),
				dir->dirname);
	    print(msg);
	}
	qsort(dir->vector,dir->vector_len,sizeof (dir->vector[0]),
	      sort_by_revname);
	break;

    }

    if (msg) {
	print(NULL);
	free_string(&msg);
    }
}

S_(browser_free_name_vector_dir browser_free_name_vector_imap)
static void browser_free_name_vector_imap P_((struct folder_browser *dir,
					       struct name_vector *entry));
static void browser_free_name_vector_imap(dir,entry)
     struct folder_browser *dir;
     struct name_vector *entry;
{
    entry->a.dummy = NULL;
}


S_(browser_line_idx_dir browser_line_idx_imap)
static const struct string * browser_line_idx_imap 
                      P_((struct folder_browser *dir,
			  struct name_vector *entry,
			  const struct string  **comment_p));
static const struct string * browser_line_idx_imap(dir,entry,comment_p)
     struct folder_browser *dir;
     struct name_vector *entry;
     const struct string  **comment_p;
{

    DPRINT(Debug,11,(&Debug,"browser_line_idx_imap=NULL\n"));

    return NULL;
}

S_(browser_reload_dir browser_reload_imap)
static int browser_reload_imap P_((struct folder_browser *dir));
static int browser_reload_imap(dir)
     struct folder_browser *dir;
{
    int ret = 0;
    char *          sysname = NULL;
    struct string * dirname = NULL;
    int sep                 = '\0';

    DPRINT(Debug,11,(&Debug, "browser_reload_imap: dir=%p\n", dir));

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_reload_imap",
	      "Bad magic type",0);

    if (dir->sys_dir) {
	sysname = safe_strdup(dir->sys_dir);
	DPRINT(Debug,11,(&Debug, "browser_reload_imap: sysname=%s\n", 
			 sysname));
    }

    if (dir->a.imap_browser->dir_name_cache) {
	dirname = dup_string(dir->a.imap_browser->dir_name_cache);
	DPRINT(Debug,11,(&Debug, "browser_reload_imap: dirname=%S\n", 
			 dirname));
    }
	
    if (dir->a.imap_browser->cur_sep) {
	sep = dir->a.imap_browser->cur_sep;
	DPRINT(Debug,11,(&Debug, "browser_reload_imap: sep=%c\n", 
			 sep));
    }


    ret = browser_update_imaplisting(dir,dirname,sysname,sep);

    if (dirname)
	free_string(&dirname);
    if (sysname)
	free(sysname);

    DPRINT(Debug,11,(&Debug, "browser_reload_imap=%d\n", ret));

    return ret;
}


struct browser_type imap_browser = {
				     BROWSER_TYPE_magic, 
				     browser_zero_imap,
				     browser_free_imap,
				     browser_change_imap,
				     browser_give_title_imap,
				     browser_separator_imap,
				     browser_name_imap,
				     browser_cat_imap,
				     browser_select_imap,
				     browser_folder_from_imap,
				     browser_change_v_imap,
				     browser_change_up_imap,
				     browser_create_selection_imap,
				     zero_ws_fields_imap,
				     free_ws_fields_imap,
				     browser_prepare_write_imap,
				     browser_end_write_imap,
				     browser_tell_imap_ws,
				     browser_seek_imap_ws,
				     browser_write_imap_ws,
				     browser_start_we_imap,
				     browser_end_we_imap,
				     browser_selection_is_imap,
				     browser_make_ref_imap,
				     browser_update_imap,
				     browser_imap_do_stat,
				     browser_sync_write_imap,
				     browser_folder_sort_imap,
				     browser_free_name_vector_imap,
				     browser_reset_filter_generic,
		/* Not called, select_dir_item_by_idx() is used
		   only on fbrowser (Elm 2.5 stype file browser) */
				     browser_select_idx_generic,
				     browser_line_idx_imap,
				     browser_reload_imap,
				     browser_gethm_name_vector_null,
				     browser_sethm_name_vector_null
				     
};

/* ---------------------------------------------------------------------- */

/* increments refcount of passhm */
void imap_browser_set_passhm(dir,passhm)
     struct folder_browser *dir;
     struct browser_passhm  * passhm;
{

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"imap_browser_set_passhm",
	      "Bad magic (browser_type)",0);

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"imap_browser_set_passhm",
	      "Bad magic  (IMAP_BROWSER)",0);

    if (dir->a.imap_browser->passhm)
	free_browser_passhm(& (dir->a.imap_browser->passhm));

    if (passhm && 
	browser_passhm_check_type(passhm, hmcon_imap)) {

	dir->a.imap_browser->passhm = passhm;	
	inc_browser_passhm_refcount(dir->a.imap_browser->passhm);
    } else if (passhm) {
	DPRINT(Debug,10,(&Debug,"imap_browser_set_passhm: browser_passhm discarded\n"));
    }

}



/* ----------------------------------------------------------------------- */

/* IMAP mailbox */

S_(mbx_close_folder mbx_close_imap)
static int mbx_close_imap P_((struct folder_info *folder,
			      enum close_mode mode,
			      struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
							   on remote mailbox
							*/
			      ));
static int mbx_close_imap(folder,mode,cd) 
     struct folder_info *folder;
     enum close_mode mode;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  on remote mailbox
			       */;
{
    int ret = 1;
    struct connection_cache *con;

    DPRINT(Debug,11,(&Debug, 
		     "mbx_close_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_close_imap",
	      "Bad magic number (private_data)",0);
    
    con = folder->p->a.imap_mbx.Ch;
    if (con) {

	struct IMAP_CON * Ma;
	
	if (con->type != &IMAP_connection) {
	    panic("MBX PANIC",__FILE__,__LINE__,"mbx_close_imap",
		  "Wrong type connection attached to folder",0);
	}
	
	DPRINT(Debug,12,(&Debug, " ... caching %s\n",
		    IMAP_connection_cache ? "enabled" : "disabled"));

	Ma = con->a.imap_con;
	
	if (!Ma)
	    panic("MBX PANIC",__FILE__,__LINE__,"mbx_close_imap",
		  "No imap connection",0);

	DPRINT(Debug,12,(&Debug, " ... capability%s%s%s\n",
			 (Ma->capability_bits & CAPA_IMAP4rev1) ? 
			 " IMAP4rev1" : "",
			 (Ma->capability_bits & CAPA_UNSELECT) ? 
			 " UNSELECT" : "",
			 Ma->capability_bits ? "" : " none"));

	ret = 1;
	
	if ( 0 != (folder->p->a.imap_mbx.folder_status & IMAP_can_CLOSE) &&
	     IMAP_connection_cache) {
	    int OK = 0;
	    int failed_connection = 0;
	    
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_close_imap: IMAP_can_CLOSE set: Giving connection back to cache\n"));

	    if (start_imap_command_c(con,"CLOSE",cd)) {
		imap_states res;
		int r;
		
		end_imap_command(con);
		OK = imap_command_ok_c(con,&res,NULL,cd);
		r = imap_clear_command_c(con,cd);

		if (!r) {
		    DPRINT(Debug,11,(&Debug, 
				     "mbx_close_imap: clear command failed\n"));

		    failed_connection = 1;
		    ret = 0;
		}
	    }
	    
	    if (OK) {	   	    
		cache_connection(con);
		folder->p->a.imap_mbx.Ch = NULL;
		
	    } else {
		int r;
		
		DPRINT(Debug,11,(&Debug, 
				 "mbx_close_imap: CLOSE failed .. closing instead\n"));
		r = close_connection(con,failed_connection,cd);
		if (!r) {
		    DPRINT(Debug,11,(&Debug, 
				     "mbx_close_imap: close_connection failed\n"));
		    ret = 0;
		}
	    }

	} else if (0 != (Ma->capability_bits & CAPA_UNSELECT) &&
		   IMAP_connection_cache) {
	    int OK = 0;
	    int failed_connection = 0;
	    
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_close_imap: Can UNSELECT: Giving connection back to cache\n"));
	    
	    if (start_imap_command(con,"UNSELECT")) {
		imap_states res;
		int r;
		
		end_imap_command(con);
		OK = imap_command_ok_c(con,&res,NULL,cd);
		r = imap_clear_command_c(con,cd);
		if (!r) {
		    DPRINT(Debug,11,(&Debug, 
				     "mbx_close_imap: clear command failed\n"));

		    failed_connection = 1;
		    ret = 0;
		}
	    }
	    
	    if (OK) {	   	    
		cache_connection(con);
		folder->p->a.imap_mbx.Ch = NULL;
				
	    } else {
		int r;
		
		DPRINT(Debug,11,(&Debug, 
				 "mbx_close_imap: UNSELECT failed .. closing instead\n"));
		r = close_connection(con,failed_connection,cd);
		if (!r) {
		    DPRINT(Debug,11,(&Debug, 
				     "mbx_close_imap: close_connection failed\n"));
		    ret = 0;
		}
	    }
	    
	    
	} else {
	    int r;
	    
	    DPRINT(Debug,12,(&Debug, " ... connection not cached.\n"))	;
	    
	    r = close_connection(con,0,cd);
	    if (!r) {
		DPRINT(Debug,11,(&Debug, 
				 "mbx_close_imap: close_connection failed\n"));
		ret = 0;
	    }
	}
	
    } else {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_close_imap: Connection already given out -- duplicate close?\n"));
	ret = 1;
    }

    switch (mode) {
    case CLOSE_NORMAL:
	
	/* falltru */
    case CLOSE_LEAVE_LOCKED:
	if (!folder -> p->fh_temp) {
	    DPRINT(Debug,1,(&Debug, 
			    "mbx_close_imap: tempfile (%s) not open -- will not unlink\n",
			    folder -> cur_tempfolder));
	} else {
	    int r = unlink(folder -> cur_tempfolder);
	    if (-1 == r) {
		int err = errno;
		if (err != ENOENT) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSorryCantUnlinkTemp,
				      "Sorry, can't unlink the temp file %s [%s]!\n\r"),
			      folder -> cur_tempfolder, strerror(err));
		    ret = 0;
		}
	    } else if (0 == r) {
		DPRINT(Debug,1,(&Debug, 
				"close_folder: Unlinking tempfile (%s).\n",
				folder -> cur_tempfolder));
	    }
	}
    }

    DPRINT(Debug,11,(&Debug, 
		     "mbx_close_imap=%d\n",
		     ret));
    
    return ret;    
}

S_(mbx_lock_folder mbx_lock_imap)
static int  mbx_lock_imap  P_((enum lock_direction direction,
			       struct folder_info *folder));
static int  mbx_lock_imap(direction,folder)
     enum lock_direction direction;
     struct folder_info *folder;
{
    int status;
    DPRINT(Debug,11,(&Debug, 
		     "mbx_lock_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_lock_imap",
	      "Bad magic number (private_data)",0);

    status = 1;
    folder->p->a.imap_mbx.folder_status &= ~IMAP_can_CLOSE;

    DPRINT(Debug,11,(&Debug, 
		     "mbx_lock_imap=%d\n",
		     status));
    return status;
}

S_(mbx_init_folder mbx_init_imap)
static void mbx_init_imap P_((struct folder_info *folder));
static void mbx_init_imap(folder) 
     struct folder_info *folder;
{
    DPRINT(Debug,11,(&Debug, 
		     "mbx_init_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    folder -> cur_tempfolder[0] = '\0';
    /* remote_mbox.c sets temporary folder name ... */

    folder->p = malloc_mbx_private_data(NULL);

    folder->p->a.imap_mbx.Ch = create_connection(&IMAP_connection);
    folder->p->a.imap_mbx.Ch->f = folder;   /* Back link */
    
    folder->p->a.imap_mbx.passhm      = NULL;

    folder->p->a.imap_mbx.folder       = NULL; /* Set by remote_mbox.c */
    folder->p->a.imap_mbx.folder_name_cache = NULL;

    folder->p->a.imap_mbx.folder_status   = 0;

    folder->p->a.imap_mbx.flag_bits       = 0;
    folder->p->a.imap_mbx.num_recent      = 0;
    folder->p->a.imap_mbx.uidvalidity     = -1;
    folder->p->a.imap_mbx.last_uid_number = 0;

    folder->p->a.imap_mbx.references      = NULL;
    folder->p->a.imap_mbx.reference_count = 0;

    folder->p->a.imap_mbx.got_EXPUNGE     = 0;
}

static void imap_clear_buffers P_((struct folder_info *folder));
static void imap_clear_buffers(folder)
     struct folder_info *folder;
{
    DPRINT(Debug,12,(&Debug, 
		     "imap_clear_buffers: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));    

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"imap_clear_buffers",
	      "Bad magic number (private_data)",0);

    folder->p->a.imap_mbx.folder_status   = 0;

    folder->p->a.imap_mbx.flag_bits       = 0;
    folder->p->a.imap_mbx.last_uid_number = 0;

    folder->p->a.imap_mbx.num_recent      = 0;
    folder->p->a.imap_mbx.uidvalidity     = -1;
    
    if (folder->p->a.imap_mbx.references) {
	int i;
    
	for (i = 0; i < folder->p->a.imap_mbx.reference_count; i++) {
	    imap_free_reference(&(folder->p->a.imap_mbx.references[i]));
	}
	free(folder->p->a.imap_mbx.references);
	folder->p->a.imap_mbx.references = NULL;
    }
    folder->p->a.imap_mbx.reference_count = 0;
    
}

static int imap_open_connection P_((struct folder_info *folder));
static int imap_open_connection(folder)
     struct folder_info *folder;
{
    struct connection_cache *con;
    const struct string  *XXX;

    int status = 0;


    DPRINT(Debug,12,(&Debug, 
		"imap_open_connection: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"imap_open_connection",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__,"imap_open_connection",
	      "Wrong type connection attached to folder",0);
    }
    
    /* Clear buffer from old data ... */
    imap_clear_buffers(folder);

    DPRINT(Debug,12,(&Debug, 
		     "imap_open_connection: con=%p state=%d\n",
		     con,
		     con->state));

    /* folder -> p->a.imap_mbx.passhm may be NULL */
    if (!login_connection_hm(con,folder -> p->a.imap_mbx.passhm)) {
	status = 0;
	goto clean;
    }
    status = 1;
    
    /* canonify folder name .... */
    free(folder->cur_folder_sys);
    folder->cur_folder_sys = elm_message(FRM("%s@%s%s%s"),
					 con->C.username,
					 con->C.host,
					 folder->p->a.imap_mbx.folder[0] == '/'
					 ? "" : ":",
					 folder->p->a.imap_mbx.folder);


    XXX = folder->p->a.imap_mbx.folder_name_cache;
    free_string(&(folder->cur_folder_disp));

    if (!XXX || 0 == string_len(XXX)) {
	struct string * X1 = conv_from_imap_name(folder->p->a.imap_mbx.folder);
	
	DPRINT(Debug,12,(&Debug, 
		    "imap_open_connection: No printable folder name available\n"));
	DPRINT(Debug,12,(&Debug, 
			 "imap_open_connection: using raw folder name: %s\n",
			 folder->p->a.imap_mbx.folder));
	DPRINT(Debug,12,(&Debug, 
			 "imap_open_connection: Gives %S\n",
			 X1));
	
	folder->cur_folder_disp = format_string(FRM("%s@%s%s%S"),
						con->C.username,
						con->C.host,
						folder->p->a.imap_mbx.folder[0] == '/'
						? "" : ":",
						X1);

	free_string(&X1);

    } else {
	folder->cur_folder_disp = format_string(FRM("%s@%s%s%S"),
						con->C.username,
						con->C.host,
						folder->p->a.imap_mbx.folder[0] 
						== '/' ? "" : ":",
						XXX);
    }
        
 clean:

    DPRINT(Debug,12,(&Debug, 
		"imap_open_connection=%d\n",
		status));
    return status;
}

S_(mbx_sessionlock_folder mbx_sessionlock_imap)
static enum sessionlock_status mbx_sessionlock_imap
      P_((struct folder_info *folder,
	  enum sessionlock_mode mode,
	  int *errcode /* errno */,
	  RECONNECT_MODE * reconnect_ptr
	  ));
static enum sessionlock_status mbx_sessionlock_imap(folder,mode,errcode,reconnect_ptr)
     struct folder_info *folder;
     enum sessionlock_mode mode;
     int *errcode /* errno */;
     RECONNECT_MODE * reconnect_ptr;
{
    struct connection_cache *con;
    
    imap_states res;
    enum sessionlock_status status  = sessionlock_fail;
    enum sessionlock_status succeed = sessionlock_open;
    
    DPRINT(Debug,11,(&Debug, 
		"mbx_sessionlock_imap: folder=%p (%s), mode=%d",
		     folder,folder->cur_folder_sys,mode));
    switch(mode) {
    case SESSIONLOCK_NORMAL:    DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NORMAL")); break;
    case SESSIONLOCK_REOPEN:    DPRINT(Debug,11,(&Debug, " SESSIONLOCK_REOPEN")); break;
    case SESSIONLOCK_TRUNCATE:  DPRINT(Debug,11,(&Debug, " SESSIONLOCK_TRUNCATE")); break;
    case SESSIONLOCK_CHECK:     DPRINT(Debug,11,(&Debug, " SESSIONLOCK_CHECK")); break;
    case SESSIONLOCK_NONE:      DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NONE")); break;
    case SESSIONLOCK_RECONNECT: DPRINT(Debug,11,(&Debug, " SESSIONLOCK_RECONNECT")); break;
    case SESSIONLOCK_NONE_CHECKNEW: DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NONE_CHECKNEW")); break;
    }
    DPRINT(Debug,11,(&Debug, "\n"));
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_imap",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_imap",
	      "Wrong type connection attached to folder",0);
    }

    if (NULL == con->C.stream ||
	con->state  != CON_logged ) {

	if (reconnect_ptr && !*reconnect_ptr &&
	    SESSIONLOCK_REOPEN != mode) {
	    *reconnect_ptr = malloc_reconnect_mode();
	}
	
	if (!imap_open_connection(folder)) {
	    status = sessionlock_fail;
	    goto clean;
	}
	folder->p->a.imap_mbx.folder_status = 0;

	if (SESSIONLOCK_REOPEN == mode) {
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_sessionlock_imap: SESSIONLOCK_REOPEN: No reconnect handling\n"));
	} else {	
	    succeed = sessionlock_reconnect;
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_sessionlock_imap: succeed = sessionlock_reconnect\n"));

	    if (reconnect_ptr && *reconnect_ptr) {

		if (RECONNECT_MODE_magic != (*reconnect_ptr)->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_imap",
			  "Bad magic number (reconnect_mode)",0);

		switch (mbx_prepare_read_reconnect(folder,*reconnect_ptr,
						   & (con->C))) {
		case prepare_reconnect_fail:
		    DPRINT(Debug,11,(&Debug, 
				     "mbx_sessionlock_imap: Failing on mbx_prepare_read_reconnect\n"));
		    status = sessionlock_fail;
		    goto clean;
		case prepare_reconnect_ok:
		    DPRINT(Debug,11,(&Debug, 
				     "mbx_sessionlock_imap: OK mbx_prepare_read_reconnect\n"));
		    break;
		case prepare_reconnect_none:
		    break;
		}
	    }	    
	}


    } else if (SESSIONLOCK_RECONNECT == mode && reconnect_ptr && folder->was_reconnected) {
	
	DPRINT(Debug,11,(&Debug, 
			 "mbx_sessionlock_imap: SESSIONLOCK_RECONNECT set, have reconnect_mode_ptr and folder is reconnected. Will move temp folder to reconnect_mode\n"));

	goto set_reconnect_mode;
	
    } else if (SESSIONLOCK_TRUNCATE == mode && reconnect_ptr && folder->p->a.imap_mbx.got_EXPUNGE) {
	
	DPRINT(Debug,11,(&Debug, 
			 "mbx_sessionlock_imap: SESSIONLOCK_TRUNCATE set, have reconnect_mode_ptr and some mail is EXPUNGEd. Will move temp folder to reconnect_mode\n"));

	goto set_reconnect_mode;

    } else if (SESSIONLOCK_TRUNCATE == mode && reconnect_ptr && folder->got_skipped_mail) {
	
	DPRINT(Debug,11,(&Debug, 
			 "mbx_sessionlock_imap: SESSIONLOCK_TRUNCATE set, have reconnect_mode_ptr and skipped mail is loaded. Will move temp folder to reconnect_mode\n"));

	goto set_reconnect_mode;
		
    } else if (SESSIONLOCK_TRUNCATE == mode && reconnect_ptr && folder->was_reconnected) {
	
	DPRINT(Debug,11,(&Debug, 
			 "mbx_sessionlock_imap: SESSIONLOCK_TRUNCATE set, have reconnect_mode_ptr and folder is reconnected. Will move temp folder to reconnect_mode\n"));

    set_reconnect_mode:
	
	if (!*reconnect_ptr) {
	    *reconnect_ptr = malloc_reconnect_mode();
	}
	
	if (RECONNECT_MODE_magic != (*reconnect_ptr)->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_imap",
		  "Bad magic number (reconnect_mode)",0);
	
	switch (mbx_prepare_read_reconnect(folder,*reconnect_ptr,
					   & (con->C))) {
	case prepare_reconnect_fail:
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_sessionlock_imap: Failing on mbx_prepare_read_reconnect\n"));
	    status = sessionlock_fail;
	    goto clean;
	case prepare_reconnect_ok:
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_sessionlock_imap: OK mbx_prepare_read_reconnect\n"));
	    break;
	case prepare_reconnect_none:
	    break;
	}	    
    }

    if (0 == (folder->p->a.imap_mbx.folder_status & IMAP_folder_open)) {

	switch(mode) {
	case SESSIONLOCK_NONE:
	    if (IMAP_use_examine) {
		if (!start_imap_command(con,"EXAMINE")) 
		    goto clean;
		break;	
	    }
	    /* FALLTHRU */
	default:

	    DPRINT(Debug,5,(&Debug, 
			     "mbx_sessionlock_imap: Assuming writable folder (for while)\n"));
	    folder->p->a.imap_mbx.folder_status  |= IMAP_writable;

	    if (!start_imap_command(con,"SELECT")) 
		goto clean;

	    break;
	}
	
	imap_command_push_astring(con,folder -> p->a.imap_mbx.folder);
	end_imap_command(con);
	
	if (!imap_command_ok(con,&res,NULL))
	    goto clean;
	folder->p->a.imap_mbx.folder_status |= IMAP_folder_open;
    }

    switch(mode) {
	int need_reopen;
    case SESSIONLOCK_RECONNECT:
	if (!folder -> p->fh_temp) {
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_sessionlock_imap: SESSIONLOCK_RECONNECT need new temp file\n"));
	    goto create_it;
	}
	/* FALLTHRU */
    case SESSIONLOCK_NONE:
    case SESSIONLOCK_NONE_CHECKNEW:
	status = succeed;
	DPRINT(Debug,11,(&Debug, 
			 "mbx_sessionlock_imap: succeed %d\n",
			 succeed));
	
	goto clean;
    case SESSIONLOCK_TRUNCATE:
	need_reopen = 1;
#ifdef FTRUNCATE  
	if (folder->p->fh_temp) {
	    int r;
	    rewind(folder->p->fh_temp);

	    r = ftruncate(fileno(folder->p->fh_temp),0);
	    if (0 == r) {
		need_reopen = 0;

		DPRINT(Debug,10,(&Debug, 
			    "truncate_tempfolder: tempfile %s truncated (not recreated)\n",
			    folder->cur_tempfolder));
	    } else if (-1 == r) {
		int err = errno;
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmSorryCantTruncateTemp,
				  "Sorry, can't truncate the temp file %s [%s]!"),
			  folder->cur_tempfolder, strerror(err));
	    }
	    /* IF FAIL REOPEN IT INSTEAD */
	}
#endif
	if (!need_reopen)
	    break;  /* DONE */
	if (folder->p->fh_temp) {
	    fclose (folder->p->fh_temp);
	    folder->p->fh_temp = NULL;
	}
	
	if (0 != unlink(folder->cur_tempfolder)) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSorryCantTruncateTemp,
			      "Sorry, can't truncate the temp file %s [%s]!"),
		      folder->cur_tempfolder, strerror(err));    
	    status = sessionlock_fail;
	    if (errcode)
		*errcode = err;
	    goto clean;
	}
	goto create_it;
    case   SESSIONLOCK_REOPEN:
	if (folder -> p->fh_temp) {
	    int temp_handle;
	    fclose(folder -> p->fh_temp);
	    folder -> p->fh_temp = NULL;
	    
	    if ( (temp_handle = open(folder-> cur_tempfolder, 
				     O_RDWR,
				     0600)) == -1 ||
		 NULL == (folder -> p->fh_temp = fdopen(temp_handle,"w+"))) {
		int err = errno;
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSorryCantReopenTemp,
				  "Sorry, can't reopen the temp file %s [%s]!"),
			  folder -> cur_tempfolder, strerror(err));  
		
		if (-1 != temp_handle)
		    close(temp_handle);
		
		status = sessionlock_fail;
		if (errcode)
		    *errcode = err;
		goto clean;
	    }
	    
	    break;
	}
	/* FALLTHRU */
    case SESSIONLOCK_NORMAL:
    case SESSIONLOCK_CHECK:
    create_it:
        if (!folder -> p->fh_temp) {
	    int err = 0;
	    
	    /* If we are changing files and we are changing to a spool file,
	     * make sure there isn't a temp file for it, because if
	     * there is, someone else is using ELM to read the new file,
	     * and we don't want to be reading it at the same time.
	     */

	    if (sessionlock_create_tempfolder(folder,&err)) {


		DPRINT(Debug,1,(&Debug, 
				"Creating tempfile (%s).\n",
				folder -> cur_tempfolder));
		
		elm_chown(folder->cur_tempfolder, userid, groupid,
			  NULL);
		chmod(folder -> cur_tempfolder, 0600);	
		/* shut off file for other people! */	    
	       		
	    } else {

		status = sessionlock_fail;
		if (errcode)
		    *errcode = err;
		
		if (err == EEXIST) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmAlreadyRunningX,
				      "You seem to have ELM already reading this mail! You may not have two copies of\n\
ELM running simultaneously. If this is in error, then you'll need to remove \n\
the following file: %s"),
			      folder -> cur_tempfolder);
		    wait_for_timeout(5 + 6 * sleepmsg);
		    
		    goto clean;	    
		}

		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedCreateTmpFolder,
				  "Failed to create %.50s: %.60s"),
			  folder -> cur_tempfolder,
			  strerror(err));
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIGiveUp,
				  "Ahhhh... I give up."));
				
		goto clean;	    
	    }
	    
	}
	break;
    default:
	break;
    }
    status = succeed;

 clean:
    imap_clear_command(con);

    DPRINT(Debug,11,(&Debug, 
		     "mbx_sessionlock_imap=%d",
		     status));
    switch (status) {
    case sessionlock_fail: DPRINT(Debug,11,(&Debug, " sessionlock_fail")); break;
    case sessionlock_open: DPRINT(Debug,11,(&Debug, " sessionlock_open")); break;
    case sessionlock_reconnect: DPRINT(Debug,11,(&Debug, " sessionlock_reconnect")); break;
    }
    DPRINT(Debug,11,(&Debug, "\n"));
    
    return status;
}

S_(mbx_unlock_folder mbx_unlock_imap)
static int mbx_unlock_imap P_((int interrupt, struct folder_info *folder));
static int mbx_unlock_imap(interrupt,folder) 
     int interrupt;
     struct folder_info *folder;
{
    int status = 0;

    DPRINT(Debug,11,(&Debug, 
		     "mbx_unlock_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    status = 1;

    DPRINT(Debug,11,(&Debug, 
		     "mbx_unlock_imap=%d\n",
		     status));
    return status;
}

S_(mbx_flush_folder mbx_flush_imap)
static int mbx_flush_imap P_((struct folder_info *folder,
			      int *err_p  /* errno value */,
			      struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
							   on remote mailbox
							*/));
static int mbx_flush_imap(folder,err_p,cd) 
     struct folder_info *folder;
     int *err_p  /* errno value */;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  on remote mailbox
			       */;
{
    struct connection_cache *con;
    int ret = 1;
        
    DPRINT(Debug,11,(&Debug, 
		     "mbx_flush_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_flush_imap",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con) {
	int ret1 = 0;

#if 0
	imap_states res;
#endif

	if (con->type != &IMAP_connection) {
	    panic("MBX PANIC",__FILE__,__LINE__,"mbx_flush_imap",
		  "Wrong type connection attached to folder",0);
	}
	
	ret1 = mbx_flush_temp(folder,err_p);

#if 0         
	/* It does not make sense on here to force IMAP server to flush
	 *  mailbox -- that flushing is needed only in
	 *  mbx_end_keep_imap()
	 */
	if (!start_imap_command(con,"CHECK")) {
	    DPRINT(Debug,11,(&Debug, 
			"mbx_flush_imap: Setting IMAP_flusherr\n"));
	    folder->p->a.imap_mbx.folder_status |= IMAP_flusherr;
	    goto clean;
	}
	end_imap_command(con);
	
	if (!imap_command_ok(con,&res,NULL)) {
	    folder->p->a.imap_mbx.folder_status |= IMAP_flusherr;
	    DPRINT(Debug,11,(&Debug, 
			"mbx_flush_imap: Setting IMAP_flusherr\n"));
	}
    
    clean: 
#endif
	ret = imap_clear_command_c(con,cd);
	if (ret) {
	    ret = ret1;
	} else {
	    folder->p->a.imap_mbx.folder_status |= IMAP_flusherr;
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_flush_imap: Setting IMAP_flusherr\n"));
	}
	
    } else {
	DPRINT(Debug,11,(&Debug, 
		    "mbx_flush_imap: No connection structure left...\n"));
    }

    DPRINT(Debug,11,(&Debug, 
		     "mbx_flush_imap=%d\n",
		     ret));
    return ret;
}

S_(mbx_ferror_folder mbx_ferror_imap) 
static int mbx_ferror_imap P_((struct folder_info *folder, int clean));
static int mbx_ferror_imap(folder,clean) 
     struct folder_info *folder; 
     int clean; 
{
    int status = 0;
    DPRINT(Debug,11,(&Debug, 
		"mbx_ferror_imap: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_ferror_imap",
	      "Bad magic number (private_data)",0);

    if (0 != (folder->p->a.imap_mbx.folder_status & IMAP_flusherr)) {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_ferror_imap: folder_status ~ IMAP_flusherr is set\n"));
	status = 1;
    }
    if (clean)
	folder->p->a.imap_mbx.folder_status &= ~IMAP_flusherr;

    if (folder->p->fh_temp) {
	if (ferror(folder->p->fh_temp))
	    status = 1;
	if (clean)
	    clearerr(folder->p->fh_temp);
    }

    DPRINT(Debug,11,(&Debug, 
		     "mbx_ferror_imap=%d\n",status));
    return status;
}

S_(mbx_prepare_read_folder mbx_prepare_read_imap)
static enum prepare_result mbx_prepare_read_imap P_((struct folder_info *folder,
						     enum prepare_mode mode,
						     READ_STATE read_state_ptr,
						     RECONNECT_MODE reconnect_ptr
						     ));
static enum prepare_result mbx_prepare_read_imap(folder,mode,read_state_ptr,reconnect_ptr)
     struct folder_info *folder;
     enum prepare_mode mode;
     READ_STATE read_state_ptr;
     RECONNECT_MODE reconnect_ptr; 
{
    struct connection_cache *con;
    enum prepare_result status  = prepare_fail;
    enum prepare_result success = prepare_done;
    int i,start;
    int add_new_only = 0;
    
    DPRINT(Debug,11,(&Debug, 
		"mbx_prepare_read_imap: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		"                    : read_state_ptr=%p\n",
		read_state_ptr));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_read_imap",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_read_imap",
	      "Wrong type connection attached to folder",0);
    }

    /* Wait that IDLE timer command is completed so that there
     * is no EXPUNGE coming...
     */
    imap_idle_wait(folder);

    switch (mode) {       	
    case PREPARE_NEW_ONLY:
    case PREPARE_NEW_ONLY_NOLOCK:
    case PREPARE_ACCESS:
	add_new_only = 1;
	DPRINT(Debug,11,(&Debug, 
			 "mbx_prepare_read_imap: Add new only\n"));
	break;
    case PREPARE_RECONNECT:	
    case PREPARE_NORMAL:
    case PREPARE_NOLOCK:
	if (reconnect_ptr) {
	    if (RECONNECT_MODE_magic != reconnect_ptr->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_read_imap",
		      "Bad magic number (reconnect_mode)",0);

	    if (folder->p->a.imap_mbx.got_EXPUNGE) {
		DPRINT(Debug,11,(&Debug, 
				 "mbx_prepare_read_imap: clearing got_EXPUNGE\n"));
		
		folder->p->a.imap_mbx.got_EXPUNGE = 0;
	    }
	    
	    switch (mbx_prepare_read_reconnect(folder,reconnect_ptr,
					       & (con->C))) {
	    case prepare_reconnect_fail:
		DPRINT(Debug,11,(&Debug, 
				 "mbx_prepare_read_imap: Failing on mbx_prepare_read_reconnect\n"));
		status = prepare_fail;
		goto clean;	
	    case prepare_reconnect_none:
		DPRINT(Debug,11,(&Debug, 
				 "mbx_prepare_read_imap: No mbx_prepare_read_reconnect\n"));
		success  = prepare_done;
		break;
	    case prepare_reconnect_ok:
		DPRINT(Debug,11,(&Debug, 
				 "mbx_prepare_read_imap: OK mbx_prepare_read_reconnect\n"));
		success = prepare_reconnected;
		break;
	    }	    	    	      	    
	}	
	break;
    }
    
    if (add_new_only) {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_prepare_read_imap: add_new_only=%d -- start from unread messages\n",
			 add_new_only));
	/* start from unread messages */

	read_state_ptr->a.imap_mbx.current_message.uid_number =
	    folder -> p->a.imap_mbx.last_uid_number+1;

	start = 1;
	while (start < folder -> p->a.imap_mbx.reference_count &&
	       folder -> p->a.imap_mbx.references[start].uid_number != -1 &&
	       folder -> p->a.imap_mbx.references[start].uid_number <
	       read_state_ptr->a.imap_mbx.current_message.uid_number)
	    start++;

	if (!folder->p->fh_temp) {
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_prepare_read_imap -- NO temp file %s\n",
			     folder->cur_tempfolder));
	} else {
	    int r1 = fseek(folder -> p->fh_temp, 
			  folder -> mailfile_size, SEEK_SET);
	    if (-1 == r1) {
		int err = errno;
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldntSeekTempEnd,
				  "\nCouldn't fseek to end of temp mbox.\n"));
		lib_error(FRM("** %s. **\n"), strerror(err));
		
		DPRINT(Debug,11,(&Debug,
				 "mbx_prepare_read_imap: Failed to seek temp %s to end of file, offset %ld: %s (errno=%d)\n",
				 folder->cur_tempfolder,
				 folder -> mailfile_size,
				 strerror(err),
				 err));		
		
		status = prepare_fail;
		goto clean;	    
	    }  else if (0 == r1) {
		DPRINT(Debug,11,(&Debug,
				 "mbx_prepare_read_imap: Seeked temp %s to end of file, offset %ld\n",
				 folder->cur_tempfolder,
				 folder -> mailfile_size));
	    }
	}
		
	read_state_ptr->fbytes = folder-> mailfile_size;  /* start correctly */
    } else {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_prepare_read_imap: add_new_only=%d -- start from beginning\n",
			 add_new_only));
	
	/* Start from beginning */
	read_state_ptr->a.imap_mbx.current_message.uid_number = 1;
	start = 1;
    }

    DPRINT(Debug,11,(&Debug, 
		"mbx_prepare_read_imap: UID start=%d  start=%d refcount=%d\n",
		read_state_ptr->a.imap_mbx.current_message.uid_number,start,
		folder -> p->a.imap_mbx.reference_count));
	
    if (start <= folder -> p->a.imap_mbx.reference_count-1) {	    

	const char * L[] = { "RFC822.SIZE", "UID", NULL };
	imap_states res;

	DPRINT(Debug,12,(&Debug,
			 "mbx_prepare_read_imap: need FETCH %d - %d\n",
			 start,
			 folder -> p->a.imap_mbx.reference_count-1));
	
	if (!start_imap_command(con,"FETCH")) {
	    status = prepare_fail;
	    goto clean;
	}
    
	imap_command_push_range(con,
				start,
				folder -> p->a.imap_mbx.reference_count-1);

	imap_command_push_list(con,L);
	end_imap_command(con);

	if (!imap_command_ok(con,&res,NULL))
	    goto clean;
    }

    /* Estimate new size for percent display */
    for ( i = start; i < folder -> p->a.imap_mbx.reference_count; i++) {
	folder -> mailfile_size += 
	    folder-> p->a.imap_mbx.references[i].rfc822_size +
	    folder-> p->a.imap_mbx.references[i].From_size;
	folder -> p->a.imap_mbx.last_uid_number = 
	    folder-> p->a.imap_mbx.references[i].uid_number;
    }
    read_state_ptr->a.imap_mbx.data_idx = 0;
    status = success;

    DPRINT(Debug,11,(&Debug, 
		     "                    new last UID %d\n",
		     folder -> p->a.imap_mbx.last_uid_number));
 clean:
    imap_clear_command(con);
    DPRINT(Debug,11,(&Debug, 
		     "mbx_prepare_read_imap=%d",status));
    switch (status) {
    case prepare_fail:  DPRINT(Debug,11,(&Debug, " prepare_fail")); break;
    case prepare_done:  DPRINT(Debug,11,(&Debug, " prepare_done"));
	if (reconnect_ptr) {
	    DPRINT(Debug,11,(&Debug, " (reconnect state not available)"));
	}

	break;
    case prepare_reconnected: DPRINT(Debug,11,(&Debug, " prepare_reconnected")); break;
    }   
    DPRINT(Debug,11,(&Debug, "\n"));
    
    return status;
}

S_(mbx_end_read_folder mbx_end_read_imap)
static int mbx_end_read_imap P_((struct folder_info *folder,
				 READ_STATE read_state_ptr,
				 RECONNECT_MODE reconnect_ptr,
				int silent));
static int mbx_end_read_imap(folder,read_state_ptr,
			     reconnect_ptr,
			     silent)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     RECONNECT_MODE reconnect_ptr;   
     int silent;
{
    int status = 0;
    long size;

    DPRINT(Debug,11,(&Debug, 
		"mbx_end_read_imap: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		"                : read_state_ptr=%p\n",
		read_state_ptr));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_read_imap",
	      "Bad magic number (private_data)",0);
    
    if (PREPARE_ACCESS != read_state_ptr->mode) {
	if (folder->p->a.imap_mbx.reference_count > 1 &&
	    folder->p->a.imap_mbx.references[folder-> 
					     p->a.imap_mbx.reference_count-1].
	    uid_number > read_state_ptr->a.imap_mbx.current_message.
	    uid_number) {
	    if (!silent) {
		lib_error(CATGETS(elm_msg_cat, MeSet,MeMessagesNotReadImap,
				  "Messages not read; current UID=%d, last UID=%d"),
			  read_state_ptr->a.imap_mbx.current_message.uid_number,
			  folder-> p->a.imap_mbx.references[folder -> 
							   p->a.imap_mbx.
							   reference_count-1].
			  uid_number);
	    }
	    status = 0;
	    goto clean;	
	}
    }
    
    if (!folder->p->fh_temp) {
	DPRINT(Debug,11,(&Debug, 
		    "mbx_end_read_imap=1 -- NO temp file %s\n",
		    folder->cur_tempfolder));
	return 1;
    }
    
    if ((ferror(folder->p->fh_temp)) || 
	(fflush(folder->p->fh_temp) == EOF)) {
	if (!silent) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFlushOnTempFailed,
			      "Flush on temp file %s failed: %s"), 
		      folder -> cur_tempfolder,
		      strerror(err));
	}
	DPRINT(Debug,1,(&Debug,  "Can't flush on temp file %s!!\n",
			folder -> cur_tempfolder));
	status = 0;
	goto clean;
    }

    size = file_bytes(folder-> cur_tempfolder,NULL);
    if (-1L == size) {
	status = 0;
	goto clean;
    }

    if (size != folder-> mailfile_size) {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_end_read_imap: Adjusting mailfile_size %ld => %ld",
			 (long)(folder-> mailfile_size),(long)(size)));

	if (read_state_ptr->skip_bytes) {
	    DPRINT(Debug,11,(&Debug, ", skip_bytes %ld",
			     (long)(read_state_ptr->skip_bytes)));
	}
	
	DPRINT(Debug,11,(&Debug, "\n"));
    }
    
    /* Use tempfile size as folder size */
    folder-> mailfile_size = size;
    
    rewind(folder->p->fh_temp);
    status = 1;
    
 clean:
    DPRINT(Debug,11,(&Debug, 
		     "mbx_end_read_imap=%d\n",status));
    return status;
}

S_(info_zero_routine info_zero_imap_mbx)
static void info_zero_imap_mbx P_((struct mbx_hdr_info *info));
static void info_zero_imap_mbx(info)
     struct mbx_hdr_info *info;
{
    info->a.imap_mbx.ref.last_mbx_index = -1;
    info->a.imap_mbx.ref.uid_number = 0;
    info->a.imap_mbx.uidvalidity = -1;
}

S_(info_free_routine info_free_imap_mbx)
static void info_free_imap_mbx P_((struct mbx_hdr_info *info));
static void info_free_imap_mbx(info)
     struct mbx_hdr_info *info;
{
    info->a.imap_mbx.ref.last_mbx_index = -1;
    info->a.imap_mbx.ref.uid_number = 0;
    info->a.imap_mbx.uidvalidity = -1;
}

S_(info_status_routine info_imap_mbx_status)
static void info_imap_mbx_status P_((struct mbx_hdr_info *info,
				    char status_buffer[3]));
static void info_imap_mbx_status(info,status_buffer)
     struct mbx_hdr_info *info;
     char status_buffer[3];
{
    if (-1 == info->a.imap_mbx.ref.last_mbx_index)
	status_buffer[0] = 'X';

}


static struct info_type imap_mbx_info = {
					  INFO_TYPE_magic, info_zero_imap_mbx, 
					  info_free_imap_mbx,
					  info_imap_mbx_status
};


S_(mbx_copy_envelope_folder mbx_copy_envelope_imap)
static enum copy_env_status mbx_copy_envelope_imap 
    P_((struct folder_info *folder,
	READ_STATE read_state_ptr,
	struct header_rec *entry,
	int force));
static enum copy_env_status mbx_copy_envelope_imap(folder,read_state_ptr,
						   entry,force)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     struct header_rec *entry;
     int force;
{
    struct connection_cache *con;
    enum copy_env_status status = copy_env_eof;
    imap_states res;
    struct imap_reference  *ref = NULL;
    
    int j;

    const char * L[] = { "FLAGS", "INTERNALDATE", "BODY.PEEK[HEADER]", NULL };

    DPRINT(Debug,11,(&Debug, 
		"mbx_copy_envelope_imap: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		"                     : read_state_ptr=%p\n",
		read_state_ptr));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_imap",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_imap",
	      "Wrong type connection attached to folder",0);
    }

    /* Wait that IDLE timer command is completed so that there
     * is no EXPUNGE coming...
     */
    imap_idle_wait(folder);

    if (PREPARE_ACCESS == read_state_ptr->mode) {
	if(entry->mbx_info &&
	   &imap_mbx_info == entry->mbx_info->type_code) {

	    if (folder->p->a.imap_mbx.uidvalidity ==
		entry->mbx_info->a.imap_mbx.uidvalidity) {
	    
		/* Hack it */
		read_state_ptr->a.imap_mbx.current_message = 
		    entry->mbx_info->a.imap_mbx.ref;
		ref = 
		    find_reference(& (folder->p->a.imap_mbx),
				   & (entry->mbx_info->a.imap_mbx.ref));
	    } else {
		DPRINT(Debug,1,(&Debug,
				"mbx_copy_envelope_imap: hdr index #%d: Bad uidvalidity %d != %d\n",
				entry->index_number_X,
				entry->mbx_info->a.imap_mbx.uidvalidity,
				folder->p->a.imap_mbx.uidvalidity));
	    }
	}
	if (!ref) {
	    DPRINT(Debug,11,(&Debug, 
			     "                     : Unable to access message\n"));

	    status                    = copy_env_eof;

	    goto clean;	    
	}
       	
    } else {
	
	do {	    	    
	    ref = find_reference(& (folder -> p->a.imap_mbx),
				 & (read_state_ptr->a.imap_mbx.
				    current_message));
	    if (!ref) {
		const int count = folder -> p->a.imap_mbx.reference_count;
		if (count  <= 1 ||
		    folder-> p->a.imap_mbx.references[count-1].uid_number < 
		    read_state_ptr->a.imap_mbx.current_message.uid_number) {
		    DPRINT(Debug,11,(&Debug, 
				     "                     : End of messages\n"));
		    status = copy_env_eof;
		    goto clean;
		}

		increment_uid_number(& (folder -> p->a.imap_mbx),
				     & (read_state_ptr->a.imap_mbx.
					current_message));
	    }
	} while (!ref);

	/* Do not reset status if PREPARE_ACCESS */
	entry->status = VISIBLE;
    }
    read_state_ptr -> linecounter   = 0;      /* Linecounter of current
					         message */
    read_state_ptr -> skipping      = sm_reading;      /* not skipping */
    entry->offset = read_state_ptr -> fbytes; /* Offset of current message */

    DPRINT(Debug,12,(&Debug,
		     "mbx_copy_envelope_imap: UID %d IMAP message number %d\n",
		     ref->uid_number,
		     read_state_ptr->a.imap_mbx.current_message.last_mbx_index));
   
    DPRINT(Debug,12,(&Debug,
		     "                     : temp file offset %ld\n",
		     entry->offset));


    if (!start_imap_command(con,"FETCH")) 
	goto clean;

    imap_command_push_number(con,
			     read_state_ptr->
			     a.imap_mbx.current_message.last_mbx_index);
    imap_command_push_list(con,L);
    end_imap_command(con);
    
    if (!imap_command_ok(con,&res,NULL))
	goto clean;

    /* Check that message is not deleted meanwhile --
       Should not happen: EXPUNGE response is not allowed during of FETCH
     */
    ref = find_reference(& (folder -> p->a.imap_mbx),
			 & (read_state_ptr->a.imap_mbx.current_message));
    if (!ref) {
	DPRINT(Debug,1,(&Debug, 
		   "mbx_copy_envelope_imap: EXPUNGE middle of FETCH ???\n"));

	status                    = copy_env_no_data;

	goto no_data;
    }

    DPRINT(Debug,14,(&Debug, 
		     "mbx_copy_envelope_imap: Flags"));

    for (j = 0; j < IMAP_flags_count; j++) {
	if (0 != (ref->imap_flags & IMAP_flags[j].mask_imap)) {
	    entry->status  |= IMAP_flags[j].mask_elm; 
	    entry->status1 |= IMAP_flags[j].mask_elm1; 
	    DPRINT(Debug,14,(&Debug, " %s", IMAP_flags[j].flag));
	}
    }

    DPRINT(Debug,14,(&Debug, "\n"));	   

    if (0 == (ref->imap_flags & IMAP_Seen))
	entry->status |= UNREAD;
    else if (0 != (entry->status & NEW)) {
	/* If both \Recent and \Seen is set then we have re$syncroning
	 * with new mail for this session (which however is already
	 * read), so ignore NEW flag
	 */
	DPRINT(Debug,14,(&Debug, 
			 "mbx_copy_envelope_imap: Both \\Seen and \\Recent set -- ignore \\Recent (NEW)\n"));
	clearit(entry->status, NEW);   /* it's been read now! */
    }


    if (!ref->header) {
	DPRINT(Debug,1,(&Debug, 
		   "mbx_copy_envelope_imap: BODY[HEADER] not got\n"));

	goto no_data;
    }
    if (!ref->internaldate) {
	DPRINT(Debug,1,(&Debug, 
			"mbx_copy_envelope_imap: INTERNADATE not got\n"));
	goto no_data;
    }

    if (! ref->From_separator) {
        char * from_separator   = NULL;
	char * return_path_data = NULL;
	char mon[4];
	
	int day_d   = 0;
	int month_d = 0;
	int year_d  = 0;
	int dateok = 1;
	char * P = NULL;
	const char * wday_s = "XXX";
	long num;
		
	/* Very lazy method to get something to put "From " -separator */
	if (0 == memcmp("Return-Path: ",ref->header,13)) {
	    int i;
	    char * XXX;
	    
	    for (i=13; i < ref->header_len; i++) {
		if ('\r' == ref->header[i] ||
		    '\n' == ref->header[i])
		    break;
	    }
	    XXX = safe_malloc(i-12);
	    memcpy(XXX,ref->header+13,i-13);
	    XXX[i-13]='\0';
	    
	    return_path_data = return_path_to_env_from_1(XXX);  /* result MALLOCED */

	    free(XXX);
	    if (! return_path_data)
		return_path_data = safe_strdup("nobody@localhost");		
	} else
	    return_path_data = safe_strdup("nobody@localhost");

	strncpy(mon,ref->internaldate+3,3);
	mon[3] = '\0';

	num = strtol(ref->internaldate,&P,10);
	if (P == ref->internaldate+2) {
	    day_d = num;
	    DPRINT(Debug,14,(&Debug, 
			     "mbx_copy_envelope_imap: %2.2s => day number = %d\n",
			     ref->internaldate,day_d));
	} else
	    dateok = 0;

	/* Convert a month name to number (Jan = 1) */
	if (cvt_monthname_to_monthnum(mon,&month_d)) {
	    DPRINT(Debug,14,(&Debug, 
			     "mbx_copy_envelope_imap: %s => month number = %d\n",
			     mon,month_d));
	    
	} else
	    dateok = 0;
	
	num = strtol(ref->internaldate+7,&P,10);
	if (P == ref->internaldate+11) {
	    year_d = num;
	    
	    DPRINT(Debug,14,(&Debug, 
			     "mbx_copy_envelope_imap: %4.4s => year number = %d\n",
			     ref->internaldate+7,year_d));			 
	} else
	    dateok = 0;
	
	if (dateok) {
	    int wday;
	    
	    static const char * dayname[] =
		{ "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
	    
	    /*  wday_from_year_month_day:
		
		return (0 = Sunday, 1 = Monday, ... 6 = Saturday),
		input: month (m= 1 to 12), day (d= 1 to 31), and year
	    */
	    
	    wday = wday_from_year_month_day(year_d,month_d,day_d);
	    DPRINT(Debug,14,(&Debug, 
			     "mbx_copy_envelope_imap: wday number = %d (0 = Sunday, 1 = Monday, ... 6 = Saturday)\n",
			     wday));
	    
	    if (wday >= 0 && wday <= (sizeof dayname) / sizeof dayname[0]) {
		wday_s = dayname[wday];
		
		DPRINT(Debug,14,(&Debug, 
				 "mbx_copy_envelope_imap: wday name = %s\n",
				 wday_s));
	    }
	}
    
	/* real_from() does not care about weekday */
	from_separator = elm_message(FRM("From %s %s %3.3s %2.2s %8.8s %5.5s %4.4s\n"),
				     return_path_data,
				     wday_s,
				     ref->internaldate+3,
				     ref->internaldate,
				     ref->internaldate+12,
				     ref->internaldate+21,
				     ref->internaldate+7);

	DPRINT(Debug,14,(&Debug, 
			 "mbx_copy_envelope_imap: generated %s",
			 from_separator));

	ref->From_separator = from_separator;
	ref->From_size      = strlen(ref->From_separator);
	from_separator = NULL;
	free(return_path_data); return_path_data = NULL;

	folder -> mailfile_size += ref->From_size;
	
    } else {
	DPRINT(Debug,14,(&Debug, 
			 "mbx_copy_envelope_imap: stored From_separator %s\n",
			 ref->From_separator));
    }

    if (!real_from(ref->From_separator,entry)) {
	DPRINT(Debug,1,(&Debug, 
			"mbx_copy_envelope_imap: real_from() failed with generated separator line: %s",
			ref->From_separator));

	goto clean;
    }
    
    if (imap_max_dl_size >= 0 &&
	ref->rfc822_size > imap_max_dl_size &&
	!force) {
	/* Need skip this */
	
	read_state_ptr-> skipping = sm_skipping;
	entry->offset             = -1;
	DPRINT(Debug,8,(&Debug,
			"mbx_copy_envelope_imap: Size %ld -- skipping this message (only read headers)\n",
			ref->rfc822_size));
	
	read_state_ptr->skip_count++;
	read_state_ptr->skip_bytes += ref->rfc822_size;
	read_state_ptr->skip_bytes += ref->From_size;

	/* Hack -- no line numbers available */
	entry->lines = -1;
    }

    
    
    switch (read_state_ptr -> skipping) {

    case sm_EOM:
	DPRINT(Debug,20,(&Debug,
                        "mbx_copy_envelope_imap: end of message, skipping body\n"));
	break;

    case sm_reading: {
	
	entry->env_from[0] = '\0';         /* Env from is bogus ... */
	
	/* notice that on mailbox separator line we don't use \r\n
	 * that even when on 'binary' mail message lines are terminated with 
	 * \r\n
	 */
	if (!mbx_copy_line_to_temp(folder,
				   ref->From_separator,
				   ref->From_size)) {
	    status = 0;

	    goto clean;
	}
	read_state_ptr -> fbytes += ref->From_size;
	read_state_ptr -> linecounter++;

    }
	break;

    case sm_skipping:
	DPRINT(Debug,20,(&Debug,
			 "mbx_copy_envelope_imap: skipping body\n"));
	break;
    }

    read_state_ptr->a.imap_mbx.data_idx = 0;
    change_rec_mbx_info(entry,&imap_mbx_info);
    entry->mbx_info->a.imap_mbx.ref = read_state_ptr->a.imap_mbx.current_message;
    entry->mbx_info->a.imap_mbx.uidvalidity = folder->p->a.imap_mbx.uidvalidity;

    DPRINT(Debug,20,(&Debug,
		     "mbx_copy_envelope_imap: entry last_mbx_index=%d uid_number=%d uidvalidity=%d\n",
		     entry->mbx_info->a.imap_mbx.ref.last_mbx_index,
		     entry->mbx_info->a.imap_mbx.ref.uid_number,
		     entry->mbx_info->a.imap_mbx.uidvalidity));
    
    status = copy_env_ok;
 clean:
    
    imap_clear_command(con);
    DPRINT(Debug,11,(&Debug, 
		"mbx_copy_envelope_imap=%d\n",
		status));
    return status;
 no_data:

    read_state_ptr-> skipping = sm_skipping;
    entry->offset             = -1;
    entry->lines              = -1;
    read_state_ptr->skip_count++;
    if (ref) {
	read_state_ptr->skip_bytes += ref->rfc822_size;
	read_state_ptr->skip_bytes += ref->From_size;
    }
    
    change_rec_mbx_info(entry,&imap_mbx_info);
    entry->mbx_info->a.imap_mbx.ref = read_state_ptr->a.imap_mbx.current_message;
    entry->mbx_info->a.imap_mbx.uidvalidity = folder->p->a.imap_mbx.uidvalidity;

    DPRINT(Debug,20,(&Debug,
		     "mbx_copy_envelope_imap: entry last_mbx_index=%d uid_number=%d uidvalidity=%d\n",
		     entry->mbx_info->a.imap_mbx.ref.last_mbx_index,
		     entry->mbx_info->a.imap_mbx.ref.uid_number,
		     entry->mbx_info->a.imap_mbx.uidvalidity));
    
    status = copy_env_no_data;

    imap_clear_command(con);
    DPRINT(Debug,11,(&Debug, 
		"mbx_copy_envelope_imap=%d (no data)\n",
		status));
    return status;
}

S_(mbx_is_forwarded_folder mbx_is_forwarded_imap)
static const char * mbx_is_forwarded_imap P_((struct folder_info *folder,
					      READ_STATE read_state_ptr));
static const char * mbx_is_forwarded_imap(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    DPRINT(Debug,11,(&Debug, 
		"mbx_is_forwarded_imap: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		"                     : read_state_ptr=%p\n",
		read_state_ptr));

    DPRINT(Debug,11,(&Debug, 
		"mbx_is_forwarded_imap=NULL\n"));
    return NULL;
}

static int imap_got_header_line P_((struct folder_info *folder,
				    READ_STATE read_state_ptr,
				    char **buffer,
				    int *len,
				    struct imap_reference  *ref));
static int imap_got_header_line(folder,read_state_ptr,buffer,len,ref)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
     struct imap_reference  *ref;
{
    int i;

    if (read_state_ptr->a.imap_mbx.data_idx >= ref->header_len) {
	DPRINT(Debug,65,(&Debug, 
		    "imap_got_header_line=0\n"));
	return 0;
    }

    for (i = read_state_ptr->a.imap_mbx.data_idx;
	 i < ref->header_len-1; i++) {
	if ('\r' == ref->header[i] &&
	    '\n' == ref->header[i+1]) {
	    i += 2;
	    goto found;
	}
    }
    i =  ref->header_len;
    DPRINT(Debug,10,(&Debug,"imap_got_header_line: EOLN NOT FOUND\n"));

 found:
    *len = i - read_state_ptr->a.imap_mbx.data_idx;
    *buffer = safe_malloc(*len+1);
    memcpy(*buffer,
	   ref->header + read_state_ptr->a.imap_mbx.data_idx,
	   *len);
    (*buffer)[*len] = '\0';

    DPRINT(Debug,65,(&Debug, 
		     "imap_got_header_line=1: len=%d, buffer=",
		     *len));
    DEBUG_PRINT_BUFFER(Debug,65,*len,s2us(*buffer));
    if (*len < 1 || !(*buffer) || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,65,(&Debug, 
		    "\nimap_got_header_line: NO NEWLINE\n"));
    } 
    return 1;    
}


static int imap_got_body_line P_((struct folder_info *folder,
				    READ_STATE read_state_ptr,
				    char **buffer,
				    int *len,
				    struct imap_reference  *ref));
static int imap_got_body_line(folder,read_state_ptr,buffer,len,ref)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
     struct imap_reference  *ref;
{
    int i;

    if (read_state_ptr->a.imap_mbx.data_idx >= ref->body_len) {
	DPRINT(Debug,65,(&Debug, 
		    "imap_got_body_line=0\n"));
	return 0;
    }

    for (i = read_state_ptr->a.imap_mbx.data_idx;
	 i < ref->body_len-1; i++) {
	if ('\r' == ref->body[i] &&
	    '\n' == ref->body[i+1]) {
	    i += 2;
	    goto found;
	}
    }
    i = ref->body_len;
    DPRINT(Debug,10,(&Debug,"imap_got_body_line: EOLN NOT FOUND\n"));

 found:
    *len = i - read_state_ptr->a.imap_mbx.data_idx;
    *buffer = safe_malloc(*len+1);
    memcpy(*buffer,
	   ref->body + read_state_ptr->a.imap_mbx.data_idx,
	   *len);

    DPRINT(Debug,65,(&Debug, 
		     "imap_got_body_line=1: len=%d, buffer=",
		     *len));
    DEBUG_PRINT_BUFFER(Debug,65,*len,s2us(*buffer));
    if (*len < 1 || !(*buffer) || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,65,(&Debug, 
			 "\nimap_got_body_line: NO NEWLINE\n"));
    } 
    return 1;    
}


static void imap_accept_line P_((struct folder_info *folder,
				 READ_STATE read_state_ptr,
				 char **buffer, int *len));
static void imap_accept_line(folder,read_state_ptr,buffer,len)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
{
    if (!*buffer)
	return;

    if (*len < 1 || (*buffer)[*len -1] != '\n') {
    } else {
	read_state_ptr -> linecounter++;
    }
    
    switch (read_state_ptr -> skipping) {
    case sm_EOM:
	DPRINT(Debug,20,(&Debug, 
			 "imap_accept_line: end of message, skipping body\n"));
	break;

    case sm_reading:
	read_state_ptr -> fbytes              += *len;
	break;

    case sm_skipping:
	DPRINT(Debug,20,(&Debug, 
			 "imap_accept_line: skipping body\n"));
	break;
    }
    read_state_ptr->a.imap_mbx.data_idx   += *len;

    free(*buffer);
    *buffer = NULL;
    *len    = 0;
}

S_(mbx_copy_header_folder mbx_copy_header_imap)
static int mbx_copy_header_imap P_((struct folder_info *folder,
				   READ_STATE read_state_ptr,
				   char **buffer, int *len));
static int mbx_copy_header_imap(folder,read_state_ptr,buffer,len)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
{
    struct connection_cache *con;
    int status = 0;
    char *buffer1 = NULL;
    int len1;
    struct imap_reference  *ref = NULL;

    const char * L[] = { "BODY.PEEK[TEXT]", NULL };

    DPRINT(Debug,11,(&Debug, 
		"mbx_copy_header_imap: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		"                   : read_state_ptr=%p\n",
		read_state_ptr));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_header_imap",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_header_imap",
	      "Wrong type connection attached to folder",0);
    }

    /* Wait that IDLE timer command is completed so that there
     * is no EXPUNGE coming...
     */
    imap_idle_wait(folder);

    /* Check that message is not deleted meanwhile --
       Should not happen: EXPUNGE response is not allowed during of FETCH
       */
    ref = find_reference(& (folder -> p->a.imap_mbx),
			 & (read_state_ptr->a.imap_mbx.current_message));
    if (!ref) {
	DPRINT(Debug,1,(&Debug, 
		   "mbx_copy_header_imap=0: EXPUNGE middle of FETCH ???\n"));
	return 0;
    }

hide:
    if (buffer1) {
	DPRINT(Debug,11,(&Debug, " ... Freeing buffer1=%s\n",buffer1));
	free(buffer1);
	buffer1 = NULL;
    }

    if (!imap_got_header_line(folder,read_state_ptr,&buffer1,&len1,ref)) {
	status = 0;
	goto clean;
    }

    if ((len1 == 1 && 0 == memcmp(buffer1,"\n",len1)) ||
	(len1 == 2 && 0 == memcmp(buffer1,"\r\n",len1))) {
	imap_states res;

	switch (read_state_ptr -> skipping) {
	case sm_EOM:
	    DPRINT(Debug,20,(&Debug, 
			     "mbx_copy_header_imap: end of message, skipping body\n"));
	    break;
	case sm_reading: {
	    /* End of headers */
	    if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
		status = 0;
		goto clean;
	    }
	}
	    break;
	case sm_skipping:
	    DPRINT(Debug,20,(&Debug, 
			     "mbx_copy_header_imap: skipping body\n"));
	    break;	    
	}

	imap_accept_line(folder,read_state_ptr,
			&buffer1,&len1);
	status = 0;

    read_body:

	/* Free used data */
	free(ref->header);
	ref->header     = NULL;
	ref->header_len = 0;

	switch (read_state_ptr -> skipping) {
	case sm_EOM:
	    DPRINT(Debug,20,(&Debug, 
			     "mbx_copy_header_imap: end of message, skipping body\n"));
	    break;

	case sm_reading:  {
	    /* FETCH body */
	    if (!start_imap_command(con,"FETCH")) 
		goto clean;
	    
	    imap_command_push_number(con,
				     read_state_ptr->
				     a.imap_mbx.current_message.last_mbx_index);
	    imap_command_push_list(con,L);
	    end_imap_command(con);

	    imap_command_ok(con,&res,NULL);
	}
	    break;
	case sm_skipping:
	    DPRINT(Debug,20,(&Debug, 
			     "mbx_copy_header_imap: skipping body\n"));
	    break;	    
	}

	goto clean;
    }
    if (NULL == memchr(buffer1,':',len1)) {
	/* End of headers -- bad header */
	DPRINT(Debug,1,(&Debug, 
		   "mbx_copy_header_imap: IMAP server is returning bad header...\n"));
	status = 0;
	goto read_body;
    }

    do {
	append_buffer(buffer,len,buffer1,len1);
	switch (read_state_ptr -> skipping) {
	case sm_reading: {
	    if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
		status = 0;
		goto clean;
	    }
	}
	    break;
	case sm_EOM:
	    DPRINT(Debug,20,(&Debug, 
			     "mbx_copy_header_imap: end of message, skipping body\n"));
	    break;
	case sm_skipping:
	    DPRINT(Debug,20,(&Debug, 
			     "mbx_copy_header_imap: skipping body\n"));
	    break;
	}

	imap_accept_line(folder,read_state_ptr,
			 &buffer1,&len1);
	if (!imap_got_header_line(folder,read_state_ptr,
				  &buffer1,&len1,ref)) {
	    break;
	}
    } while (len1 > 0 && whitespace(buffer1[0]));
    
    if (*buffer && *len > 6 &&
	header_cmp(*buffer, "Status", NULL)) {
	DPRINT(Debug,11,(&Debug, 
			 "                   : Hiding header %s",*buffer));
	free(*buffer);
	*buffer = NULL;
	*len = 0;
	goto hide;
    }
	
    status = 1;

clean:
    if (buffer1) {
	free(buffer1);
	buffer1 = NULL;
    }

    if (!status) {
	/* So that reading of body starts correctly */
	read_state_ptr->a.imap_mbx.data_idx = 0;
	if (*buffer) {
	    free(*buffer);
	    *buffer = NULL;
	}
    }

    DPRINT(Debug,40,(&Debug, 
		     "mbx_copy_header_imap=%d | len=%d, buffer=",
		     status,*len));
    DEBUG_PRINT_BUFFER(Debug,40,*len,s2us(*buffer));
    if (*len < 1 || !*buffer || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,40,(&Debug, 
			 "\nmbx_copy_header_imap <- NO NEWLINE\n"));
    }
    return status;
}

S_(mbx_copy_body_folder mbx_copy_body_imap)
static int mbx_copy_body_imap P_((struct folder_info *folder,
				     READ_STATE read_state_ptr,
				     char **buffer, int *len,
				     long *content_remaining));

static int mbx_copy_body_imap(folder,read_state_ptr,buffer,len,
			      content_remaining)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
     long *content_remaining;
{ 
    char *buffer1 = NULL;
    int len1;
    int status = 0;
    struct imap_reference  *ref = NULL;

    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_body_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                 : read_state_ptr=%p\n",
		     read_state_ptr));

    switch (read_state_ptr -> skipping) {
    case sm_EOM: 
	DPRINT(Debug,20,(&Debug, 
			 "mbx_copy_body_imap = 0 (end of message, skipping body)\n"));
	    return 0;
    case sm_reading:
	DPRINT(Debug,20,(&Debug, 
			 "mbx_copy_body_imap: Reading body\n"));
	break;
    case sm_skipping:
	DPRINT(Debug,20,(&Debug, 
			 "mbx_copy_body_imap = 0 (skipping body)\n"));
	return 0;
    }

    /* Check that message is not deleted meanwhile --
       Should not happen: EXPUNGE response is not allowed during of FETCH
       */
    ref = find_reference(& (folder -> p->a.imap_mbx),
			 & (read_state_ptr->a.imap_mbx.current_message));
    if (!ref) {
	DPRINT(Debug,1,(&Debug, 
			"mbx_copy_body_imap=0: EXPUNGE middle of FETCH ???\n"));
	return 0;
    }

    if (!imap_got_body_line(folder,read_state_ptr,
			    &buffer1,&len1,ref)) {
	status = 0;
	goto clean;
    }
    
    append_buffer(buffer,len,buffer1,len1);
    if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
	status = 0;
	goto clean;
    }
    imap_accept_line(folder,read_state_ptr,
		     &buffer1,&len1);
    status = 1;

 clean:
    if (buffer1) {
	free(buffer1);
	buffer1 = NULL;
    }

    if (!status) {
	if (*buffer) {
	    free(*buffer);
	    *buffer = NULL;
	}
    }

    DPRINT(Debug,40,(&Debug, 
		"mbx_copy_body_imap=%d | len=%d, buffer=",
		     status,*len));
    DEBUG_PRINT_BUFFER(Debug,40,*len,s2us(*buffer));
    if (*len < 1 || !*buffer || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,40,(&Debug, 
			 "\nmbx_copy_body_imap <- NO NEWLINE\n"));
    }
    return status;
}

S_(mbx_copy_envelope_end_folder mbx_copy_envelope_end_imap)
static enum copy_env_end_status mbx_copy_envelope_end_imap
     P_((struct folder_info *folder,
	 READ_STATE read_state_ptr,
	 long *newbytes,
	 int  *newmails
	 ));
static enum copy_env_end_status mbx_copy_envelope_end_imap(folder,read_state_ptr,
							   newbytes,newmails)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     long *newbytes;
     int  *newmails;
{
    struct connection_cache *con;
    enum copy_env_end_status status = copy_env_end_mismatch;
    struct imap_reference  *ref = NULL;
    int havenew = 0;

    DPRINT(Debug,11,(&Debug, 
		"mbx_copy_envelope_end_imap: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		"                         : read_state_ptr=%p\n",
		read_state_ptr));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_end_imap",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_end_imap",
	      "Wrong type connection attached to folder",0);
    }

    /* Check that message is not deleted meanwhile --
       Should not happen: EXPUNGE response is not allowed during of FETCH
       */
    ref = find_reference(& (folder -> p->a.imap_mbx),
			 & (read_state_ptr->a.imap_mbx.current_message));
    if (ref) {
	int idx;
	
	if (ref->header) {
	    /* Free used data */
	    free(ref->header);
	    ref->header     = NULL;
	    ref->header_len = 0;
	}
	if (ref->body) {
	    /* Free used data */
	    free(ref->body);
	    ref->body     = NULL;
	    ref->body_len = 0;
	}


	idx = ref - folder -> p->a.imap_mbx.references + 1;
	
	/* Got next UID */
	if (idx <  folder -> p->a.imap_mbx.reference_count) {
	    	    
	    if (-1 == folder -> p->a.imap_mbx.references[idx].uid_number) {    /* Pick new messages */
		
		DPRINT(Debug,11,(&Debug, "mbx_copy_envelope_end_imap: No UID for index %d\n",
				 idx));

		if (start_imap_command(con,"FETCH")) {
		    int start = idx;
		    
		    const char * L[] = { "RFC822.SIZE", "UID", NULL };
		    imap_states res;
		    int oldlast = folder -> p->a.imap_mbx.last_uid_number;
		    int i;
		    
		    imap_command_push_range(con,
					    start,
					    folder -> p->a.imap_mbx.reference_count-1);
		    
		    imap_command_push_list(con,L);
		    end_imap_command(con);

		    if (!imap_command_ok(con,&res,NULL)) {
			goto clean1;
		    }

		    if (newbytes)
			*newbytes = 0;
		    if (newmails)
			*newmails = 0;
		    
		    /* Estimate new size for percent display */
		    for (i = start; i < folder -> p->a.imap_mbx.reference_count; i++) {

			if (folder -> p->a.imap_mbx.references[i].uid_number >
			    oldlast) {			    
			    havenew = 1;
			    
			    folder -> mailfile_size += 
				folder-> p->a.imap_mbx.references[i].rfc822_size +
				folder-> p->a.imap_mbx.references[i].From_size;
			    if (newbytes)
				*newbytes += 
				    folder-> p->a.imap_mbx.references[i].rfc822_size;
			    if (newmails)
				*newmails += 1;

			    folder -> p->a.imap_mbx.last_uid_number = 
				folder-> p->a.imap_mbx.references[i].uid_number;
			}
		    }

		    if (oldlast != folder -> p->a.imap_mbx.last_uid_number) {

			DPRINT(Debug,11,(&Debug,
					 "mbx_copy_envelope_end_imap: Updated last UID %d => %d\n",
					 oldlast, folder -> p->a.imap_mbx.last_uid_number));
		    }
		    		    
		} else {
		clean1:
		    DPRINT(Debug,11,(&Debug,
				     "mbx_copy_envelope_end_imap: Can't update listing\n"));
		}

		increment_uid_number(& (folder -> p->a.imap_mbx),
				     & (read_state_ptr->a.imap_mbx.current_message));
				
	    } else {
		
		read_state_ptr->a.imap_mbx.current_message.uid_number =
		    folder -> p->a.imap_mbx.references[idx].uid_number;
		read_state_ptr->a.imap_mbx.current_message.last_mbx_index =
		    idx;
	    }
	    
	} else {
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_copy_envelope_end_imap: End of mailbox\n"));
	    read_state_ptr->a.imap_mbx.current_message.uid_number++;
	}

    } else {
	read_state_ptr->a.imap_mbx.current_message.uid_number++;

	DPRINT(Debug,1,(&Debug, 
			"mbx_copy_envelope_end_imap: EXPUNGE middle of FETCH ???\n"));
    } 

    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_end_imap: Next UID %d -- assumed index %d\n",
		     read_state_ptr->a.imap_mbx.current_message.uid_number,
		     read_state_ptr->a.imap_mbx.current_message.last_mbx_index));
    
    switch (read_state_ptr -> skipping) {
    case sm_EOM:
	DPRINT(Debug,20,(&Debug, 
			 "mbx_copy_envelope_end_imap: end of message, skipping body\n"));
	break;
    case  sm_reading: {
	/* On mailbox format there is empty line (\n\n)
	 * on end of mail...
	 */
	
	if (!mbx_copy_line_to_temp(folder,"\n",1)) {
	    status = copy_env_end_failure;
	    goto clean;
	}
	read_state_ptr -> fbytes ++;    /* Increment position */
    }
	break;
    case sm_skipping:
	DPRINT(Debug,20,(&Debug, 
			 "mbx_copy_envelope_end_imap: skipping body\n"));
	break;

    }
    status = havenew ? copy_env_end_newmail : copy_env_end_match;

clean:
    read_state_ptr -> skipping = sm_reading;
    imap_clear_command(con);

    DPRINT(Debug,11,(&Debug, 
		"mbx_copy_envelope_end_imap=%d",
		status));
    switch (status) {
    case copy_env_end_failure:  DPRINT(Debug,11,(&Debug," copy_env_end_failure"));  break;
    case copy_env_end_mismatch: DPRINT(Debug,11,(&Debug," copy_env_end_mismatch")); break;
    case copy_env_end_match:    DPRINT(Debug,11,(&Debug," copy_env_end_match"));    break;
    case copy_env_end_newmail:  DPRINT(Debug,11,(&Debug," copy_env_end_newmail"));  break;
    }
    if (newbytes) {
	DPRINT(Debug,11,(&Debug," *newbytes=%ld",
			 *newbytes));
    }
    if (newmails) {
	DPRINT(Debug,11,(&Debug," *newmails=%d",
			 *newmails));    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return status; 
}

S_(mbx_copy_envelope_reset_body mbx_copy_envelope_reset_body_imap)
static int mbx_copy_envelope_reset_body_imap P_((struct folder_info *folder,
						READ_STATE read_state_ptr));
static int mbx_copy_envelope_reset_body_imap(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_reset_body_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "                                : read_state_ptr=%p\n",
		     read_state_ptr));
    DPRINT(Debug,11,(&Debug, 
		     "mbx_copy_envelope_reset_body_imap=0\n"));
    return 0;
}

S_(mbx_folder_to_fd mbx_imap_to_fd)
static FILE * mbx_imap_to_fd P_((struct folder_info *folder,long offset));
static FILE * mbx_imap_to_fd(folder,offset)
     struct folder_info *folder;
     long offset;
{
    FILE * status = NULL;

    DPRINT(Debug,11,(&Debug, 
		"mbx_imap_to_fd: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_imap_to_fd",
	      "Bad magic number (private_data)",0);

    if (offset != -1L &&
	fseek(folder->p->fh_temp, offset , 
	      SEEK_SET) == -1) {
	int err = errno;
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmCouldntSeekBytesIntoFolder,
			  "\nCouldn't seek %ld bytes into folder.\n"),
		  offset);		
	lib_error(FRM("** %s. **\n"), strerror(err));

	DPRINT(Debug,1,(&Debug, 
		   "Error: Couldn't seek folder %s: (offset %ld) Errno %s (%s)\n",
		   folder->cur_tempfolder, offset, 
		   strerror(err), "mbx_imap_to_fd"));

	status = NULL;
	goto clean;
    }
    status = folder->p->fh_temp;

 clean:
    DPRINT(Debug,11,(&Debug, 
		"mbx_imap_to_fd=%s (%p)\n",
		status ? "NON-NULL" : "NULL",
		status));
    return status;
}



S_(mbx_new_mail_on_folder mbx_new_mail_on_imap)
static enum new_mail_stat mbx_new_mail_on_imap P_((struct folder_info *folder,
				    long *bytes, int *new_mailcount,
				    int *err_p  /* errno value */,
				    struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
								 check of remote mailbox
							      */
				    ));
static enum new_mail_stat mbx_new_mail_on_imap(folder,bytes,new_mailcount,err_p,
				cd)
     struct folder_info *folder;
     long *bytes;
     int  *new_mailcount /* -1 for local mailbox */;
     int *err_p  /* errno value */;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  check of remote mailbox
			       */;
{
    struct connection_cache *con;
    enum new_mail_stat status = no_new_mail;
    int i;
    int start = 1;
    imap_states res;
	
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_new_mail_on_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_new_mail_on_imap",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_new_mail_on_imap",
	      "Wrong type connection attached to folder",0);
    }

    if (bytes)
	*bytes = 0;
    if (new_mailcount)
	*new_mailcount = 0;
    if (err_p)
	*err_p =0;
    
    if (!start_imap_command_c(con,"NOOP",cd))  {
	status = new_mail_check_failed;
	goto clean;
    }
    end_imap_command(con);
    if (!imap_command_ok_c(con,&res,NULL,cd))  {
	status = new_mail_check_failed;
	goto clean;
    }

    /* imap_clear_command_c can be interrupted, and therefore 
       fail 
    */
    if (!imap_clear_command_c(con,cd)) {
	status = new_mail_check_failed;
	goto clean;
    }

    /* Messages which do not have UID have unknown ... */
    
    start = 1;
    while (start < folder -> p->a.imap_mbx.reference_count &&
	   folder -> p->a.imap_mbx.references[start].uid_number != -1) {
	start++;
    }

    DPRINT(Debug,11,(&Debug, 
		     "                    : start=%d refcount=%d last uid number=%d\n",
		     start,
		     folder -> p->a.imap_mbx.reference_count,
		     folder -> p->a.imap_mbx.last_uid_number));
    
    if (start <= folder -> p->a.imap_mbx.reference_count-1) {	    
	
	const char * L[] = { "RFC822.SIZE", "UID", NULL };
	
	if (!start_imap_command_c(con,"FETCH",cd)) {
	    status = new_mail_check_failed;
	    goto clean;
	}
	
	imap_command_push_range(con,
				start,
				folder -> p->a.imap_mbx.reference_count-1);
	
	imap_command_push_list(con,L);
	end_imap_command(con);
	
	if (!imap_command_ok_c(con,&res,NULL,cd)) {
	    status = new_mail_check_failed;
	    goto clean;
	}
    }

    for (i = 1; i < folder -> p->a.imap_mbx.reference_count; i++) {
	if (folder -> p->a.imap_mbx.references[i].uid_number >
	    folder -> p->a.imap_mbx.last_uid_number) {
	    status = have_new_mail;
	    if (bytes) {
		*bytes +=
		    folder-> p->a.imap_mbx.references[i].rfc822_size +
		    folder-> p->a.imap_mbx.references[i].From_size;
	    }
	    if (new_mailcount) {
		*new_mailcount += 1;
	    }	    
	}
    }
    
 clean:
    if (!imap_clear_command_c(con,cd)) {
	DPRINT(Debug,11,(&Debug, 
			 "mbx_new_mail_on_imap: OOPS failure on clear command (canceled?)\n"));
	status = new_mail_check_failed;
    }

    DPRINT(Debug,11,(&Debug, 
		     "mbx_new_mail_on_imap=%d",
		     status));
    switch (status) {
    case failed_mbx_ign_new_mail: DPRINT(Debug,11,(&Debug," failed_mbx_ign_new_mail")); break;
    case new_mail_check_failed:   DPRINT(Debug,11,(&Debug," new_mail_check_failed"));   break;
    case no_new_mail:             DPRINT(Debug,11,(&Debug," no_new_mail"));             break;
    case have_new_mail:           DPRINT(Debug,11,(&Debug," have_new_mail"));           break;
    case new_mail_reconnect:      DPRINT(Debug,11,(&Debug," new_mail_reconnect"));      break;
    }
    if (bytes) {
	DPRINT(Debug,11,(&Debug, " *bytes=%ld",
			 *bytes));
    }
    if (new_mailcount) {
	DPRINT(Debug,11,(&Debug, " *new_mailcount=%d",
			 *new_mailcount));
    }
    if (err_p) {
	DPRINT(Debug,11,(&Debug, " *err_p=%d",
			 *err_p));
	if (*err_p) {
	    DPRINT(Debug,11,(&Debug, " (%s)",
			     strerror(*err_p)));
	}
    }
    DPRINT(Debug,11,(&Debug, "\n"));
    
    return status;
}

S_(mbx_consider_remove_folder mbx_consider_remove_imap)
static int mbx_consider_remove_imap P_((struct folder_info *folder));
static int mbx_consider_remove_imap(folder)
     struct folder_info *folder;
{
    DPRINT(Debug,11,(&Debug, 
		     "mbx_consider_remove_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug, 
		     "mbx_consider_remove_imap=0\n"));
    return 0;
}

S_(mbx_prepare_keep_folder mbx_prepare_keep_imap)
static int mbx_prepare_keep_imap P_((struct folder_info *folder,
				    KEEP_STATE keep_state_ptr));
static int mbx_prepare_keep_imap(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
{
    int status = 0; 

    DPRINT(Debug,11,(&Debug, 
		     "mbx_prepare_keep_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_keep_imap",
	      "Bad magic number (private_data)",0);

    folder->p->a.imap_mbx.folder_status &= ~IMAP_can_CLOSE;
    status = 1;   
    DPRINT(Debug,11,(&Debug, 
		     "mbx_prepare_keep_imap=%d\n",
		     status));
    return status;
}

S_(mbx_end_keep_folder mbx_end_keep_imap)
static int mbx_end_keep_imap P_((struct folder_info *folder,
				KEEP_STATE keep_state_ptr));
static int mbx_end_keep_imap(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
{
    struct connection_cache *con;
    int status = 0;
    imap_states res;

    DPRINT(Debug,11,(&Debug, 
		"mbx_end_keep_imap: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_keep_imap",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_keep_imap",
	      "Wrong type connection attached to folder",0);
    }

    folder->p->a.imap_mbx.folder_status &= ~IMAP_can_CLOSE;
    if (0 != (folder->p->a.imap_mbx.folder_status & IMAP_flusherr))
	goto clean;

    if (!start_imap_command(con,"EXPUNGE"))
	goto clean;
    end_imap_command(con);
    
    if (!imap_command_ok(con,&res,NULL))
	goto clean;
    imap_clear_command(con);

    if (!start_imap_command(con,"CHECK"))
	goto clean;
    end_imap_command(con);
    
    if (!imap_command_ok(con,&res,NULL))
	goto clean;
        
    status = 1;
    folder->p->a.imap_mbx.folder_status |= IMAP_can_CLOSE;

 clean:
    folder->p->a.imap_mbx.folder_status &= ~IMAP_flusherr;
    imap_clear_command(con);

    DPRINT(Debug,11,(&Debug, 
		     "mbx_end_keep_imap=%d\n",
		     status));
    return status;
}

S_(mbx_mark_keep_folder  mbx_mark_keep_imap)
static int mbx_mark_keep_imap P_((struct folder_info *folder,
				      KEEP_STATE keep_state_ptr,
				      struct header_rec *entry,
				      int keep));
static int mbx_mark_keep_imap(folder,keep_state_ptr,entry,keep)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
     struct header_rec *entry;
     int keep;
{
    int ret = 1;

    struct connection_cache *con;
    DPRINT(Debug,11,(&Debug, 
		     "mbx_mark_keep_imap: folder=%p (%s), ks=%p, header=%p\n",
		     folder,folder->cur_folder_sys,keep_state_ptr,entry));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_mark_keep_imap",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_mark_keep_imap",
	      "Wrong type connection attached to folder",0);
    }

    /* Wait that IDLE timer command is completetd so that there
     * is no EXPUNGE coming...
     */
    imap_idle_wait(folder);

    if(entry->mbx_info &&
       &imap_mbx_info == entry->mbx_info->type_code) {

	struct imap_reference  *ref =  NULL;

	if (folder->p->a.imap_mbx.uidvalidity ==
	    entry->mbx_info->a.imap_mbx.uidvalidity) {
	    ref = 
		find_reference(& (folder -> p->a.imap_mbx),
			       & (entry->mbx_info->a.imap_mbx.ref));
	} else {
	    DPRINT(Debug,1,(&Debug,
			    "mbx_mark_keep_imap: hdr index #%d: Bad uidvalidity %d != %d\n",
			    entry->index_number_X,
			    entry->mbx_info->a.imap_mbx.uidvalidity,
			    folder->p->a.imap_mbx.uidvalidity));
	}
	    
	if (ref) {
	    long old_bits = ref->imap_flags;
	    int minus_delta = 0;

	    int j;

	    const char ** LIST = safe_calloc(( IMAP_flags_count +1 ),
					     sizeof(LIST[0]));
	    imap_states res;

	    int X;

	    if (0 == (entry->status & UNREAD)) {
		ref->imap_flags |= IMAP_Seen;
	    } else {
		/* So that 'U' (unread) works: */
		ref->imap_flags &= ~IMAP_Seen;
	    }

	    for (j = 0; j < IMAP_flags_count; j++) {

		if (0 != (IMAP_flags[j].mask_elm  & entry->status) ||
		    0 != (IMAP_flags[j].mask_elm1 & entry->status1))
		    ref->imap_flags |= IMAP_flags[j].mask_imap;		 

		/* So that 't' (tag / untag) works: 
		 * Note that IMAP_flags[j].mask_elm == 0 for \Seen !!!
		 */
		else if (0 != IMAP_flags[j].mask_elm ||
			 0 != IMAP_flags[j].mask_elm1)
		    ref->imap_flags &= ~IMAP_flags[j].mask_imap;
	    }
	    if (keep)
		ref->imap_flags &= ~IMAP_Deleted;
	    else
		ref->imap_flags |= IMAP_Deleted;
	    

	    DPRINT(Debug,9,(&Debug, 
			    "mbx_mark_keep_imap: (mailbox index %d) Flags",
			    entry->mbx_info->a.imap_mbx.ref.last_mbx_index));

	    X = 0;
	    for (j = 0; j < IMAP_flags_count; j++) {
		if (0 != (ref->imap_flags & IMAP_flags[j].mask_imap) &&
		    0 == (IMAP_Recent     & IMAP_flags[j].mask_imap)) {
		    LIST[X++] = IMAP_flags[j].flag;
		    DPRINT(Debug,9,(&Debug, " %s",IMAP_flags[j].flag));
		}
	    }	    
	    LIST[X] = NULL;

	    DPRINT(Debug,9,(&Debug, "\n"));

	    /* Cyrys imapd 2.0.14 fails parse command
	           nnnn STORE xxx FLAGS ()
	       Therefore do not issue store commands if 
	       FLAGS is empty and status_chgd is not set
	    */

	    if (0 == X) {
		minus_delta = 1;

		DPRINT(Debug,9,(&Debug, 
				"mbx_mark_keep_imap: (mailbox index %d) -Flags",
				entry->mbx_info->a.imap_mbx.ref.last_mbx_index));


		X = 0;
		for (j = 0; j < IMAP_flags_count; j++) {
		    if (0 == (ref->imap_flags & IMAP_flags[j].mask_imap) &&
			0 != (old_bits & IMAP_flags[j].mask_imap) &&
			0 == (IMAP_Recent     & IMAP_flags[j].mask_imap)) {
			LIST[X++] = IMAP_flags[j].flag;
			DPRINT(Debug,9,(&Debug, " %s",IMAP_flags[j].flag));
		    }
		}	    
		LIST[X] = NULL;
				
		if (0 == X) {
		    DPRINT(Debug,9,(&Debug, " (empty)\n"));
		    goto LIST_clean;
		}
	    }

	    if (!start_imap_command(con,"STORE")) {
		folder->p->a.imap_mbx.folder_status |= IMAP_flusherr;

		ret = 0;
		goto LIST_clean;
	    }
	    imap_command_push_number(con,
				     entry->mbx_info->
				     a.imap_mbx.ref.last_mbx_index);


	    if (minus_delta)
		imap_command_push_atom(con,"-FLAGS");
	    else
		imap_command_push_atom(con,"FLAGS");
	    imap_command_push_list(con,LIST);
	    end_imap_command(con);


	    if (!imap_command_ok(con,&res,NULL)) {
		folder->p->a.imap_mbx.folder_status |= IMAP_flusherr;

		ret = 0;

	    LIST_clean:
		free(LIST);

		goto clean;
	    }

	    free(LIST);

	}
    } else {
	DPRINT(Debug,11,(&Debug, "mbx_mark_keep_imap: No valid mbx_info\n"));
    }
clean:
    imap_clear_command(con);

    DPRINT(Debug,11,(&Debug, "mbx_mark_keep_imap=%d\n",ret));

    return ret;
}

S_(mbx_folder_type mbx_imap_type)
const char * mbx_imap_type P_((struct folder_info *folder));
const char * mbx_imap_type(folder)
     struct folder_info *folder;
{
    char *result = NULL;

    DPRINT(Debug,11,(&Debug, 
		     "mbx_imap_type: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_imap_type",
	      "Bad magic number (private_data)",0);

    if (0 != (folder->p->a.imap_mbx.folder_status & IMAP_writable)) {

	if (0 == istrcmp(folder -> p->a.imap_mbx.folder,
			 "INBOX")) {
	    static char *mailbox = NULL;

	    if (!mailbox)
		mailbox = catgets(elm_msg_cat, MeSet, MeImapMailbox, 
				  "IMAP mailbox");
	    result = mailbox;
	} else {
	    static char *mailbox_normal = NULL;

	    if (!mailbox_normal)
		mailbox_normal = catgets(elm_msg_cat, MeSet, MeImapFolder, 
				  "IMAP folder");
	    result = mailbox_normal;
	}
    } else {
	static char *ro_mailbox = NULL;
	
	if (!ro_mailbox)
	    ro_mailbox = catgets(elm_msg_cat, MeSet, MeROImapFolder, 
				 "Read-only IMAP folder");	
	result = ro_mailbox;
    }

    DPRINT(Debug,11,(&Debug, 
		     "mbx_imap_type=%s\n",
		     result));
    return result;
}

S_(mbx_start_edit_folder mbx_start_edit_imap)
static int mbx_start_edit_imap P_((struct folder_info *folder, 
				  const char **buffer));
static int mbx_start_edit_imap(folder,buffer)
     struct folder_info *folder;
     const char **buffer;
{
    DPRINT(Debug,11,(&Debug, 
		     "mbx_start_edit_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    DPRINT(Debug,11,(&Debug, 
		     "mbx_start_edit_imap=0\n"));
    return 0;
}

S_(mbx_end_edit_folder  mbx_end_edit_imap)
static enum end_edit_fol_status mbx_end_edit_imap P_((struct folder_info *folder));
static enum end_edit_fol_status mbx_end_edit_imap(folder)
     struct folder_info *folder;
{
    DPRINT(Debug,11,(&Debug, 
		"mbx_end_edit_imap: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));

    DPRINT(Debug,11,(&Debug, 
		"mbx_end_edit_imap=end_edit_none\n"));
    return end_edit_none;
}

S_(mbx_free_folder mbx_free_imap)
static int mbx_free_imap P_((struct folder_info *folder,
			     struct cancel_data  * cd));
static int mbx_free_imap(folder,cd)
     struct folder_info *folder;
     struct cancel_data  * cd;
{
    int ret = 1;
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_free_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));    

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_free_imap",
	      "Bad magic number (private_data)",0);

    if (folder->p->a.imap_mbx.folder_name_cache)
	free_string(&(folder->p->a.imap_mbx.folder_name_cache));

    if (folder->p->a.imap_mbx.Ch) {
	int r = free_connection(&(folder->p->a.imap_mbx.Ch),cd);

	if (!r) {
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_free_imap: free_connecion failed\n"));
	    ret = 0;
	}
    }

    if (folder->p->a.imap_mbx.folder) {
	free(folder->p->a.imap_mbx.folder);
	folder->p->a.imap_mbx.folder = NULL;
    }

    if (folder->p->a.imap_mbx.passhm)
	free_browser_passhm(& (folder -> p->a.imap_mbx.passhm));

    imap_clear_buffers(folder);

    free_mbx_private_data(& (folder->p));

     DPRINT(Debug,11,(&Debug, 
		      "mbx_free_imap=%d\n",ret));
    return ret;
}

S_(mbx_zero_rs_fields_folder mbx_zero_rs_fields_imap)
static void mbx_zero_rs_fields_imap P_((struct read_folder_state *rs));
static void mbx_zero_rs_fields_imap(rs)
     struct read_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug, 
		"mbx_zero_rs_fields_imap: rs=%p\n",
		rs));

    rs->a.imap_mbx.current_message.last_mbx_index = -1;
    rs->a.imap_mbx.current_message.uid_number     = 0;
    rs->a.imap_mbx.data_idx = 0;
}

S_(mbx_free_rs_fields_folder  mbx_free_rs_fields_imap)
static void mbx_free_rs_fields_imap P_((struct read_folder_state *rs));
static void mbx_free_rs_fields_imap(rs)
     struct read_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug, 
		"mbx_free_rs_fields_imap: rs=%p\n",
		rs));

    rs->a.imap_mbx.current_message.last_mbx_index = -1;
    rs->a.imap_mbx.current_message.uid_number     = 0;
    rs->a.imap_mbx.data_idx = 0;    
}

S_(mbx_zero_ks_fields_folder mbx_zero_ks_fields_imap)
static void mbx_zero_ks_fields_imap P_((struct keep_folder_state *rs));
static void mbx_zero_ks_fields_imap(rs)
     struct keep_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug, 
		     "mbx_zero_ks_fields_imap: rs=%p\n",
		     rs));
}

S_(mbx_zero_ks_fields_folder mbx_free_ks_fields_imap)
static void mbx_free_ks_fields_imap P_((struct keep_folder_state *rs));
static void mbx_free_ks_fields_imap(rs)
     struct keep_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug, 
		     "mbx_free_ks_fields_imap: rs=%p\n",
		     rs));
}

S_(mbx_get_folder_mode mbx_get_imap_mode)
static int mbx_get_imap_mode P_((struct folder_info *folder));
static int mbx_get_imap_mode(folder)
     struct folder_info *folder;
{
    int ret = 0;
    struct connection_cache *con;
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_get_imap_mode: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_get_imap_mode",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_get_imap_mode",
	      "Wrong type connection attached to folder",0);
    }
    
    if (0 != (folder->p->a.imap_mbx.folder_status & IMAP_writable)) {
	if (0 == istrcmp(folder -> p->a.imap_mbx.folder,
			 "INBOX")) {	    
	    ret =  FOLDER_MBOX;
	} else {
	    ret = 0;
	}
    } else {

	ret = FOLDER_RDONLY;
    }

    if (NULL == con->C.stream ||
	con->state  != CON_logged )
	ret |= FOLDER_DISCONNECTED;
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_get_imap_mode=%d",
		     ret));
    if (ison(ret,FOLDER_MBOX))            { DPRINT(Debug,11,(&Debug," FOLDER_MBOX")); }
    if (ison(ret,FOLDER_RDONLY))          { DPRINT(Debug,11,(&Debug," FOLDER_RDONLY")); }
    if (ison(ret,FOLDER_FILE))            { DPRINT(Debug,11,(&Debug," FOLDER_FILE")); }
    if (ison(ret,FOLDER_DISCONNECTED))    { DPRINT(Debug,11,(&Debug," FOLDER_DISCONNECTED")); }
    DPRINT(Debug,11,(&Debug,"\n"));
    
    return ret;
}

S_(mbx_give_folder_server mbx_give_imap_server)
const struct remote_server * mbx_give_imap_server 
    P_((struct folder_info *folder));
const struct remote_server * mbx_give_imap_server(folder)
     struct folder_info *folder;
{
    struct connection_cache *con;
    struct remote_server * ret;

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_give_imap_server",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__," mbx_give_imap_server",
	      "Wrong type connection attached to folder",0);
    }

    if (con->this_server)
	return con->this_server;

    ret = give_remote_server_from_connection(con,IMAP_SERVER_TYPE,
					     folder->p->a.imap_mbx.passhm);

    con->this_server = ret;

    return ret;
}

S_(mbx_comp_prev_hdr_folder mbx_comp_prev_hdr_imap)
static enum comp_prev_hdr_result mbx_comp_prev_hdr_imap
    P_((struct folder_info * folder,
	struct header_rec  * entry,
	struct header_rec  * prev_entry,
	RECONNECT_MODE    reconnect_ptr));
enum comp_prev_hdr_result mbx_comp_prev_hdr_imap(folder,entry,prev_entry,
						    reconnect_ptr)
     struct folder_info * folder;
     struct header_rec  * entry;
     struct header_rec  * prev_entry;
     RECONNECT_MODE    reconnect_ptr;
{
    enum comp_prev_hdr_result ret = comp_prev_hdr_pass;
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_comp_prev_hdr_imap",
	      "Bad magic number (private_data)",0);
    if(entry->mbx_info &&
       &imap_mbx_info == entry->mbx_info->type_code &&
       prev_entry->mbx_info &&
       &imap_mbx_info == prev_entry->mbx_info->type_code) {
	
	if (entry->mbx_info->a.imap_mbx.uidvalidity >= 0 &&
	    entry->mbx_info->a.imap_mbx.uidvalidity ==
	    prev_entry->mbx_info->a.imap_mbx.uidvalidity) {

	    if (entry->mbx_info->a.imap_mbx.ref.uid_number ==
		prev_entry->mbx_info->a.imap_mbx.ref.uid_number &&
		entry->mbx_info->a.imap_mbx.ref.uid_number > 0) {
		DPRINT(Debug,12,(&Debug,
				 "mbx_comp_prev_hdr_imap: UID %d matches, UIDVALIDITY %d\n",
				 entry->mbx_info->a.imap_mbx.ref.uid_number,
				 entry->mbx_info->a.imap_mbx.uidvalidity
				 ));
		ret = comp_prev_hdr_found;
	    } else if (entry->mbx_info->a.imap_mbx.ref.uid_number > 0 &&
		       prev_entry->mbx_info->a.imap_mbx.ref.uid_number > 0) {
		DPRINT(Debug,12,(&Debug,
				 "mbx_comp_prev_hdr_imap: entry UID %d != prev_entry UID %d, UIDVALIDITY %d\n",
				 entry->mbx_info->a.imap_mbx.ref.uid_number,
				 prev_entry->mbx_info->a.imap_mbx.ref.uid_number,
				 entry->mbx_info->a.imap_mbx.uidvalidity
				 ));
		ret = comp_prev_hdr_miss;
	    }
	} else if (entry->mbx_info->a.imap_mbx.uidvalidity       >= 0 &&
		   prev_entry->mbx_info->a.imap_mbx.uidvalidity  >= 0) {

	    	DPRINT(Debug,12,(&Debug,
				 "mbx_comp_prev_hdr_imap: entry UIDVALIDITY %d != prev_entry UIDVALIDY %d\n",
				 entry->mbx_info->a.imap_mbx.uidvalidity,
				 prev_entry->mbx_info->a.imap_mbx.uidvalidity));
	}
    }
    
    DPRINT(Debug,12,(&Debug,"mbx_comp_prev_hdr_imap=%d",
		     ret));
    switch (ret) {
    case comp_prev_hdr_miss:  DPRINT(Debug,12,(&Debug," comp_prev_hdr_miss")); break;
    case comp_prev_hdr_pass:  DPRINT(Debug,12,(&Debug," comp_prev_hdr_pass")); break;
    case comp_prev_hdr_found: DPRINT(Debug,12,(&Debug," comp_prev_hdr_found")); break;
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return ret;
}

S_(mbx_update_prev_hdr_folder mbx_update_prev_hdr_imap)
static void mbx_update_prev_hdr_imap  P_((struct folder_info * folder,
					  struct header_rec  * entry,
					  struct header_rec  * prev_entry,
					  RECONNECT_MODE    reconnect_mode_ptr,
					  READ_STATE        read_state_ptr
					  ));
static void mbx_update_prev_hdr_imap(folder,entry,prev_entry,
				     reconnect_mode_ptr,
				     read_state_ptr)
     struct folder_info * folder;
     struct header_rec  * entry;
     struct header_rec  * prev_entry;
     RECONNECT_MODE    reconnect_mode_ptr;
     READ_STATE        read_state_ptr;
{
    int ok = 0;

    /* Not really needed */
    
    DPRINT(Debug,12,(&Debug,"mbx_update_prev_hdr_imap\n"));

    if (entry->mbx_info &&
	&imap_mbx_info == entry->mbx_info->type_code) {
	
	if (entry->mbx_info->a.imap_mbx.uidvalidity != folder->p->a.imap_mbx.uidvalidity &&
	    -1 != entry->mbx_info->a.imap_mbx.uidvalidity) {

	    DPRINT(Debug,1,(&Debug,
			    "mbx_update_prev_hdr_imap: hdr index #%d: Bad uidvalidity %d != %d\n",
			    entry->index_number_X,
			    entry->mbx_info->a.imap_mbx.uidvalidity,
			    folder->p->a.imap_mbx.uidvalidity));

	    entry->mbx_info->a.imap_mbx.ref.uid_number     = 0;
	    entry->mbx_info->a.imap_mbx.ref.last_mbx_index = -1;
	} else if (-1 != entry->mbx_info->a.imap_mbx.ref.uid_number &&
		   folder->p->a.imap_mbx.uidvalidity ==
		   prev_entry->mbx_info->a.imap_mbx.uidvalidity) {
	    DPRINT(Debug,12,(&Debug,
			     "mbx_update_prev_hdr_imap: hdr index #%d: Have UID %d mailbox index %d uidvalidity %d\n",
			     entry->index_number_X,
			     entry->mbx_info->a.imap_mbx.ref.uid_number,
			     entry->mbx_info->a.imap_mbx.ref.last_mbx_index,
			     entry->mbx_info->a.imap_mbx.uidvalidity));
	    ok = 1;
	}
    }
    
    if (prev_entry->mbx_info &&
	&imap_mbx_info == prev_entry->mbx_info->type_code &&
	prev_entry->mbx_info->a.imap_mbx.uidvalidity >= 0 &&
	folder->p->a.imap_mbx.uidvalidity ==
	prev_entry->mbx_info->a.imap_mbx.uidvalidity 
	) {
	struct imap_reference  * ref =
	    find_reference(& (folder->p->a.imap_mbx),
			   & (prev_entry->mbx_info->a.imap_mbx.ref));

	if (ref) {
	    int idx = ref - folder -> p->a.imap_mbx.references;
	    
	    change_rec_mbx_info(entry,&imap_mbx_info);

	    entry->mbx_info->a.imap_mbx.ref.uid_number     = ref->uid_number;
	    entry->mbx_info->a.imap_mbx.ref.last_mbx_index = idx;
	    entry->mbx_info->a.imap_mbx.uidvalidity        = folder->p->a.imap_mbx.uidvalidity;

	    DPRINT(Debug,12,(&Debug,
			     "mbx_update_prev_hdr_imap: hdr index #%d: Setting UID %d mailbox index %d uidvalidity %d\n",
			     entry->index_number_X,
			     entry->mbx_info->a.imap_mbx.ref.uid_number,
			     entry->mbx_info->a.imap_mbx.ref.last_mbx_index,
			     entry->mbx_info->a.imap_mbx.uidvalidity));
	} else {
	    DPRINT(Debug,12,(&Debug,
			     "mbx_update_prev_hdr_imap: hdr index #%d: prev hdr index #%d, UID %d mailbox index %d uidvalidity %d not found -- OK\n",
			     entry->index_number_X,
			     prev_entry->index_number_X,
			     prev_entry->mbx_info->a.imap_mbx.ref.uid_number,
			     prev_entry->mbx_info->a.imap_mbx.ref.last_mbx_index,
			     prev_entry->mbx_info->a.imap_mbx.uidvalidity));

	    if (!ok) {
		
		change_rec_mbx_info(entry,&imap_mbx_info);
		
		entry->mbx_info->a.imap_mbx.ref.uid_number     = prev_entry->mbx_info->a.imap_mbx.ref.uid_number;
		entry->mbx_info->a.imap_mbx.ref.last_mbx_index = -1;
		entry->mbx_info->a.imap_mbx.uidvalidity        = folder->p->a.imap_mbx.uidvalidity;
		
		DPRINT(Debug,12,(&Debug,
				 "mbx_update_prev_hdr_imap: hdr index #%d: Setting UID %d mailbox index %d (none) uidvalidity %d\n",
				 entry->index_number_X,
				 entry->mbx_info->a.imap_mbx.ref.uid_number,
				 entry->mbx_info->a.imap_mbx.ref.last_mbx_index,
				 entry->mbx_info->a.imap_mbx.uidvalidity));
	    }
	}
    }
}


S_(mbx_give_message_msize_folder  mbx_give_message_msize_imap)
static int mbx_give_message_msize_imap P_((struct folder_info * folder,
					  struct header_rec  * entry,
					  unsigned long      * ret_size));
static int mbx_give_message_msize_imap(folder,entry,ret_size)
     struct folder_info * folder;
     struct header_rec  * entry;
     unsigned long      * ret_size;
{
    int ret = 0;
    
    DPRINT(Debug,11,(&Debug, 
		     "mbx_give_message_msize_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_give_message_msize_imap",
	      "Bad magic number (private_data)",0);

    if(entry->mbx_info &&
       &imap_mbx_info == entry->mbx_info->type_code) {
	
	struct imap_reference  *ref =  NULL;

	if (folder->p->a.imap_mbx.uidvalidity ==
	    entry->mbx_info->a.imap_mbx.uidvalidity) {
	    ref = 
		find_reference(& (folder -> p->a.imap_mbx),
			       & (entry->mbx_info->a.imap_mbx.ref));
	} else {
	    DPRINT(Debug,1,(&Debug,
			    "mbx_give_message_msize_imap: hdr index #%d: Bad uidvalidity %d != %d\n",
			    entry->index_number_X,
			    entry->mbx_info->a.imap_mbx.uidvalidity,
			    folder->p->a.imap_mbx.uidvalidity));
	}

	if (ref) {
	    if (ret_size)
		*ret_size = ref->rfc822_size;
	    ret = 1;
	}
    }

    DPRINT(Debug,11,(&Debug, 
		     "mbx_give_message_msize_imap=%d\n",ret));
    return ret;
}

/* Return -1 if not available (local mail) */
S_(mbx_give_message_count_folder mbx_give_message_count_imap)
static int mbx_give_message_count_imap P_((struct folder_info * folder));
static int mbx_give_message_count_imap(folder)
     struct folder_info * folder;
{
    struct connection_cache *con;
    int ret = -1;
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_give_message_count_imap: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
        
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_give_message_count_imap",
	      "Bad magic number (private_data)",0);

    con = folder->p->a.imap_mbx.Ch;
    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_give_message_count_imap",
	      "Wrong type connection attached to folder",0);
    }

    ret = folder -> p->a.imap_mbx.reference_count-1;

    DPRINT(Debug,11,(&Debug,
		     "mbx_give_message_count_imap=%d\n",ret));

    return ret;
}


struct folder_type imap_mbx = {
                                FOLDER_TYPE_magic, "IMAP",
                                mbx_close_imap,
				mbx_lock_imap,
				mbx_init_imap,
				mbx_sessionlock_imap,
				mbx_unlock_imap,
				mbx_flush_imap,
				mbx_ferror_imap,
				mbx_prepare_read_imap,  
				mbx_end_read_imap,
				mbx_copy_envelope_imap,
				mbx_is_forwarded_imap,
				mbx_copy_header_imap,
				mbx_copy_body_imap,
				mbx_copy_envelope_end_imap,
				mbx_copy_envelope_reset_body_imap,
				mbx_imap_to_fd,
				mbx_new_mail_on_imap,
				mbx_consider_remove_imap,
				mbx_prepare_keep_imap,
				mbx_end_keep_imap,
				mbx_mark_keep_imap,
				mbx_imap_type,
				mbx_start_edit_imap,
				mbx_end_edit_imap,
				mbx_free_imap,
				mbx_zero_rs_fields_imap,
				mbx_free_rs_fields_imap,
				mbx_zero_ks_fields_imap,
				mbx_free_ks_fields_imap,
				mbx_get_imap_mode,
				mbx_give_imap_server,
				mbx_comp_prev_hdr_imap,
				mbx_update_prev_hdr_imap,
				mbx_have_imap_quota,
				mbx_give_imap_quotar_list,
				mbx_give_message_msize_imap,
				mbx_give_message_count_imap
};


/* --------------------------------------------------- */

static struct remote_account *set_imap_connection_1
    P_((struct folder_info *fh,
	struct remote_account *X,
	struct browser_passhm *passhm,
	enum connection_state st));

static struct remote_account *set_imap_connection_1(fh,X,passhm,st)
     struct folder_info *fh;
     struct remote_account *X;
     struct browser_passhm *passhm;
     enum connection_state st;
{
    if (PRIVATE_DATA_magic != fh->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"set_imap_connection_1",
	      "Bad magic number (private_data)",0);

    if (passhm &&
	browser_passhm_check_type(passhm, hmcon_imap)) {

	if (!join_connection_hm(fh->p->a.imap_mbx.Ch,X,st,passhm)) 
	    set_imap_connection_state(fh->p->a.imap_mbx.Ch,IMAP_error);
	else
	    set_imap_connection_state(fh->p->a.imap_mbx.Ch,IMAP_idle);

	if (fh -> p->a.imap_mbx.passhm)
	    free_browser_passhm(& (fh -> p->a.imap_mbx.passhm));
	fh -> p->a.imap_mbx.passhm = passhm;
	inc_browser_passhm_refcount(fh -> p->a.imap_mbx.passhm);

    } else {

	if (!join_connection(fh->p->a.imap_mbx.Ch,X,st)) 
	    set_imap_connection_state(fh->p->a.imap_mbx.Ch,IMAP_error);
	else
	    set_imap_connection_state(fh->p->a.imap_mbx.Ch,IMAP_idle);

	if (fh -> p->a.imap_mbx.passhm)
	    free_browser_passhm(& (fh -> p->a.imap_mbx.passhm));

	if (passhm) {
	    DPRINT(Debug,10,(&Debug,"set_imap_connection_1: browser_passhm discarded\n"));
	}
    }

    return &(fh->p->a.imap_mbx.Ch->C);
}

struct remote_account *set_imap_connection_C(fh,CX,passhm)
     struct folder_info       * fh;
     struct connection_cache  * CX;
     struct browser_passhm     *passhm;
{
    if (fh->folder_type != &imap_mbx)
	panic("MBX PANIC",__FILE__,__LINE__,"set_imap_connection_C",
	      "Bad folder type",0);


    return set_imap_connection_1(fh,&(CX->C),passhm,
				 get_connection_state (CX));
}

struct remote_account *set_imap_connection(fh,X,passhm)
     struct folder_info *fh;
     struct remote_account *X;
     struct browser_passhm *passhm;
{
    if (fh->folder_type != &imap_mbx)
	panic("MBX PANIC",__FILE__,__LINE__,"set_imap_connection",
	      "Bad folder type",0);

    return set_imap_connection_1(fh,X,passhm,CON_greeting);
}

void imap_mbx_set_passhm(fh,passhm)
     struct folder_info *fh;
     struct browser_passhm * passhm;
{
    if (fh->folder_type != &imap_mbx)
	panic("MBX PANIC",__FILE__,__LINE__,"imap_mbx_set_passhm",
	      "Bad folder type",0);

    if (PRIVATE_DATA_magic != fh->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"imap_mbx_set_passhm",
	      "Bad magic number (private_data)",0);
 
   if (fh->p->a.imap_mbx.passhm)
	free_browser_passhm(& (fh->p->a.imap_mbx.passhm));

    if (passhm &&
	browser_passhm_check_type(passhm, hmcon_imap)) {
	fh->p->a.imap_mbx.passhm = passhm;
	inc_browser_passhm_refcount(fh -> p->a.imap_mbx.passhm);
    } else if (passhm) {
	DPRINT(Debug,10,(&Debug,
			 "imap_mbx_set_passhm: browser_passhm discarded\n"));
    }
}

/* URL handling --------------------------------------- */


S_(url_default_port_f url_default_port_imap)
static uint16 url_default_port_imap P_((
	 const struct browser_url_method  *schema));

static uint16 url_default_port_imap(schema)
     const struct browser_url_method  *schema;
{

    return PORT_imap4;
}


static struct connection_cache * 
  url_make_connection_x P_((const struct browser_url_method  *schema,
			    const struct string    *user   /* May be NULL */,
			    const struct string    *password /* May be NULL */,
			    const char             *host   /* May be NULL */,
			    int                      port,   
			    enum initial_tls_flag    initial_tls));

static struct connection_cache * 
   url_make_connection_x(schema,user,password,host,port,initial_tls)     
     const struct browser_url_method  *schema;
     const struct string              *user   /* May be NULL */;
     const struct string              *password /* May be NULL */;
     const char                       *host   /* May be NULL */;
     int                               port;   
     enum initial_tls_flag             initial_tls;
{

    struct remote_account    ra;
    struct string            * user2;
    struct service_entry     * se;
    struct connection_cache  * CX;
    enum connection_state S;

    if (!host || !user) {
	DPRINT(Debug,7,(&Debug, 
			"url_make_connection_x=NULL: No host or user\n"));
	return NULL;            /*  Not supported currently */
    }

    se = give_service_entry2(host, STFLAG_is_imap,initial_tls);
    if (!se) {
	DPRINT(Debug,7,(&Debug, 
			"url_make_connection_x=NULL: No service entry\n"));
	return NULL;
    }
    
    zero_remote_account(&ra);

    /* Canonify host name ... */
    ra.host  = safe_strdup(se->official_name);

    if (! can_ascii_string(user))
	  lib_error(CATGETS(elm_msg_cat, MeSet,
			    MeNonAsciiUsernameUrl,
			    "Non-ascii usernames on url are not supported: %S"),
		      user);

    /* NOTE: Original string is kept, if it can not converted
       to US-ASCII -- then this produces garbage, but
       user is warned
    */
    user2 = ascify_string(user);

    ra.username  = us2s(stream_from_string(user2,0,NULL));

    free_string(&user2);

    CX = locate_from_cache(ra.username,ra.host,&IMAP_connection,port,
			   se);

    if (!CX) {
	int got;

	PORTS ports_imaponly[] = { PORT_imap4, PORT_end };
	PORTS force_port = PORT_end;
	enum itls_status have_initial_tls = itls_none;

	if (BROWSER_URL_method_magic != schema->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"url_make_connection_x",
		  "Bad magic type (browser_url_method)",0);

	ports_imaponly[0] = 
	    schema->url_default_port_it(schema);
	DPRINT(Debug,7,(&Debug, 
			"url_make_connection_x: default port is %d\n",
			ports_imaponly[0]));

	if (port)
	    force_port = port;
	       
	/* Make connection, if not available from cache */

	if (!connect_remote_account(&ra,&got,se,
				    ports_imaponly,
				    force_port,
				    NULL)) {
	    DPRINT(Debug,7,(&Debug, 
			    "url_make_connection_x: connect_remote_account failed\n"));

	    goto fail;
	}

	/* needed for imaps */
	switch ((have_initial_tls = remote_account_init_tls(&ra))) {

	case itls_unsupported:
        case itls_failed:
            
            goto fail;
	    
        case itls_none:
            break;
	    
        case itls_done:
            DPRINT(Debug,8,(&Debug,
                            "url_make_connection_x: Initial TLS done (no STARTTLS)\n"));
	    
	}

	CX =  create_connection(&IMAP_connection);
	CX->C = ra;
	zero_remote_account(&ra);     /* Avoid double free */
	CX->port = port;
	CX->state = CON_greeting;      /* wait greeting*/
	/* browser_from_connection() requires that connection is cached */
	cache_connection(CX);

	if (! imap_handle_greeting(CX)) {
	     CX = NULL;   /* Is on cache so this do not cause leak */
	     goto fail;
	}
    }

    /* If not logged, must be logged first */
    if ((S = get_connection_state (CX)) != CON_logged) {

	DPRINT(Debug,12,(&Debug, 
			 "url_make_connection_x: con=%p state=%d\n",
			 CX,
			 S));

	if (! login_connection(CX,password)) {

	    CX = NULL;   /* Is on cache so this do not cause leak */
	    goto fail;
	}	
    }


 fail:    
    clear_remote_account(&ra);

    if (CX) {
	DPRINT(Debug,7,(&Debug, 
			"url_make_connection_x=%p\n",CX));
    }

    free_service_entry(&se);
    return CX;
}




S_(url_make_connection_f url_make_connection_imap)


/* May not be 
      uin16 port
   because this is promoted to int
*/

static struct connection_cache * url_make_connection_imap P_((
	 const struct browser_url_method  *schema,
	 const struct string              *user   /* May be NULL */,
	 const struct string              *password /* May be NULL */,
	 const char                       *host   /* May be NULL */,
	 int                      port   /* May be NULL */));
static struct connection_cache * url_make_connection_imap(schema,user,
							  password,host,
							  port)
     const struct browser_url_method  *schema;
     const struct string              *user   /* May be NULL */;
     const struct string              *password /* May be NULL */;
     const char                       *host   /* May be NULL */;
     int                      port   /* May be NULL */;
{

    /* search imap but not imaps */
    return url_make_connection_x(schema,user,password,host,
				 port,itls_disable);
				   
}

/* Returns -1 on failure */
static int match_item_to_imaplisting P_((struct folder_browser *dir,
					 struct string * dirname_disp,
					 char *          dirname_sys,
					 const struct string * item));
static int match_item_to_imaplisting(dir,dirname_disp,
				     dirname_sys,item)
     struct folder_browser *dir;
     struct string * dirname_disp;
     char *          dirname_sys;
     const struct string * item;
{
    int L = 0;
    int match = -1;
    int i;

    if (dirname_sys)
	L = strlen(dirname_sys);

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"match_item_to_imaplisting",
	      "Bad magic type",0);


    for (i = 0; i < dir->a.imap_browser->dir_entry_count && -1 == match; i++) {
	char *tail = dir->a.imap_browser->dir_entries[i].imap_name;
	const charset_t item_cs     = get_string_type(item);

	if (dirname_sys) {
	    if (0 != strncmp(dirname_sys,tail,L) ||
		tail[L] != dir->a.imap_browser->dir_entries[i].sep) {

		DPRINT(Debug,12,(&Debug, 
				 "match_item_to_imaplisting: item %d '%s' do not match prefix '%s' separator '%c'\n",
				 i,tail,
				 dirname_sys,
				 dir->a.imap_browser->dir_entries[i].sep));
		continue;
	    }
	    
	    tail = tail + (L+1);
	}

	if (RAW_BUFFER != item_cs) {
	    char * buf;
	    int buflen;

	    bytestream_from_string(item,&buf,&buflen);

	    if (0 == strcmp(tail,buf) && strlen(tail) == buflen) {

		DPRINT(Debug,12,(&Debug, 
				 "match_item_to_imaplisting: raw item %d '%s' matches\n",
				 i,tail));
		match = i;
	    }	  

	    free(buf);
	} else {
	    struct string * buf2 = conv_from_imap_name(tail);

	    if (0 == string_cmp(item,
				buf2,-99 /* Unknown indicator */)) {

		DPRINT(Debug,12,(&Debug, 
				 "match_item_to_imaplisting: translated item %d '%S' matches\n",
				 i,buf2));
		match = i;	   
	    }

	    free_string(&buf2);
	}
    }


    return match;
}

/* As strncmp */
static int string_prefix_match P_((const struct string *str1,
				   const struct string *str2,
				   int L,int unknown_val));
static int string_prefix_match(str1,str2,L,unknown_val)
     const struct string *str1;
     const struct string *str2;
     int L;
     int unknown_val;
{
    int ret;

    int X1 = 0;
    struct string * Z1 = clip_from_string(str1,&X1,L);

    int X2 = 0;
    struct string * Z2 = clip_from_string(str1,&X2,L);

    ret = string_cmp(Z1,Z2,unknown_val);

    free_string(&Z2);
    free_string(&Z1);

    return ret;
}

static int translate_path_to_imap_selection P_((struct folder_browser *
						dir,
						struct string *dirname,
						char *sysname,
						int sep));
static int translate_path_to_imap_selection(dir,dirname,sysname,sep)
     struct folder_browser *dir;
     struct string *dirname;
     char *sysname;
     int sep;
{
    int ret = 0;

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"translate_path_to_imap_selection",
	      "Bad magic type",0);

    if (!dir->sys_dir) {
	DPRINT(Debug,12,(&Debug, 
			 "translate_path_to_imap_selection: On default directory\n"));

	/* WARNING: set_dir_selection does not allocate strings -- 
	 *          it just assign pointers!
	 */
	set_dir_selection(dir,
			  safe_strdup(sysname),
			  dup_string(dirname),
			  BROWSER_MAILFILE   /* Assume folder    */
			  | BROWSER_EXIST   /* Exists because on listing */
			  );

	ret = 1;

    } else {
	
	char * s1;
	struct string * s2;

	int L1 = strlen(dir->sys_dir);
	int L2;

	
	const struct string * relative1 = NULL;
	int X;

	if (0 != strncmp(dir->sys_dir,sysname,L1) ||
	    sep != sysname[L1]) {

	    DPRINT(Debug,12,(&Debug, 
			     "translate_path_to_imap_selection: Directory not match\n"));
	    goto fail;

	}

	relative1 = dir->a.imap_browser->dir_name_cache;

	if (!relative1) {
	    DPRINT(Debug,12,(&Debug, 
			     "translate_path_to_imap_selection: No dir_name_cache\n"));
	    goto fail;
	}

	L2 = string_len(relative1);


	/* IMAP character set is assumed to be ASCII compatible on
	   here! */
	if (L2 +1 > string_len(dirname) ||
	    0 != string_prefix_match(relative1,dirname,L2,
				     99 /* unknown val */) ||
	    sep != give_unicode_from_string(dirname,L2)) {

	    DPRINT(Debug,12,(&Debug, 
			     "translate_path_to_imap_selection: Directory not match\n"));

	    goto fail;
	}


	s1 = sysname + L1 + 1;
	
	X = L2+1;
	s2 = clip_from_string(dirname,&X,string_len(dirname));


	/* WARNING: set_dir_selection does not allocate strings -- 
	 *          it just assign pointers!
	 */
	set_dir_selection(dir,
			  safe_strdup(s1),
			  s2,
			  BROWSER_MAILFILE   /* Assume folder    */
			  | BROWSER_EXIST   /* Exists because on listing */
			  );

	ret = 1;
    }

 fail:

    DPRINT(Debug,12,(&Debug, 
		     "translate_path_to_imap_selection=%d\n",
		     ret));

    return ret;
}


/* This is always absolute path -- from root as defined for URL */
S_(url_select_item_from_URL_f url_select_item_from_imap_path)
static int url_select_item_from_imap_path P_((
	const struct browser_url_method  *schema,
	struct folder_browser *dir,
	int elems_count,
	const struct string **elems));
static int url_select_item_from_imap_path(schema,dir,elems_count,elems)
     const struct browser_url_method  *schema;
     struct folder_browser *dir;
     int elems_count;    /* 0 elems is root if absolute */
     const struct string **elems;
{
    char used_sep = '\0';
    struct string * dirname_disp     = NULL;
    char *          dirname_sys      = NULL;
    int ret = 0;

    if (!browser_update_imaplisting(dir,NULL,NULL,'\0'))
	goto fail;

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"url_select_item_from_imap_path",
	      "Bad magic type",0);
    
    /* Assume native separators */
    if (1 == elems_count) {
	const charset_t elems_0_cs     = get_string_type(elems[0]);

	if (RAW_BUFFER != elems_0_cs) {
	    struct string * dispname_not_used = NULL;
	    
	    ret = browser_select_imap(dir,elems[0],&dispname_not_used,
				      NULL);

	    if (dispname_not_used)
		free_string(&dispname_not_used);

	    DPRINT(Debug,12,(&Debug, 
			     "url_select_item_from_imap_path = %d (native result)\n",
			     ret));

	    goto done;
	} else {

	    DPRINT(Debug,12,(&Debug, 
			     "url_select_item_from_imap_path: Raw  buffer type currently not supported ...\n"));
	    
	    goto not_supported;
	}
	
    } else {
	/* / is used as separator */

	int i;

    not_supported:
	for (i = 0; i < elems_count; i++) {
	    	    
	    int match = match_item_to_imaplisting(dir,dirname_disp,
						  dirname_sys,elems[i]);
	    
	    
	    if (match >= 0) {
		
		if (match >=  dir->a.imap_browser->dir_entry_count) {
		    panic("MBX PANIC",__FILE__,__LINE__,
			  "url_descend_to_imap_directory",
			  "Bad match",0);
		}
		
		if (dirname_disp) 
		    free_string(&dirname_disp);
		dirname_disp = dup_string(dir->a.imap_browser->
					  dir_entries[match].
					  translated_name);
		if (dirname_sys)
		    free(dirname_sys);
		dirname_sys = safe_strdup(dir->a.imap_browser->
					  dir_entries[match].
					  imap_name);

		used_sep = dir->a.imap_browser->dir_entries[match].sep;			       		
		if (i < elems_count-1) {

		    DPRINT(Debug,12,(&Debug, 
				     "url_select_item_from_imap_path: %d .. going translated='%S' sysname='%s'\n",
				     i,dirname_disp,dirname_sys));
		    
		    
		    if (! browser_update_imaplisting(dir,dirname_disp,dirname_sys,
						     used_sep)) {
			DPRINT(Debug,12,(&Debug, 
					 "url_select_item_from_imap_path: %d ... failed\n"));
			goto fail;
		    }

		} else {

		    DPRINT(Debug,12,(&Debug, 
				     "url_select_item_from_imap_path: %d .. selecting translated='%S' sysname='%s'\n",
				     i,dirname_disp,dirname_sys));


		    ret = translate_path_to_imap_selection(dir,dirname_disp,dirname_sys,
							   used_sep);
		    
		}

	
		continue;
	    }
	    
	    
	    DPRINT(Debug,12,(&Debug, 
			     "url_select_item_from_imap_path: %d failed at %S\n",
			     i,elems[i]));

	    goto fail;
	    
	}

	DPRINT(Debug,13,(&Debug, 
			 "url_select_item_from_imap_path: loop ended, not selected item\n"));
    }

 fail:
 done:

    DPRINT(Debug,12,(&Debug, 
		     "url_select_item_from_imap_path=%d\n",
		     ret));


    return ret;
}

struct connection_cache *give_imap_connection(dir)
     struct folder_browser *dir;
{
    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"give_imap_connection",
	      "Bad magic type",0);


    return dir->a.imap_browser->Ch;
}


const struct browser_url_method  URL_imap = {
    BROWSER_URL_method_magic,
    & imap_browser,

    url_default_port_imap,
    url_make_connection_imap,
    url_select_item_from_imap_path
    
};

S_(url_default_port_f url_default_port_imaps)
static uint16 url_default_port_imaps P_((
	 const struct browser_url_method  *schema));

S_(url_make_connection_f url_make_connection_imaps)

static struct connection_cache * url_make_connection_imaps P_((
	 const struct browser_url_method  *schema,
	 const struct string              *user   /* May be NULL */,
	 const struct string              *password /* May be NULL */,
	 const char                       *host   /* May be NULL */,
	 int                      port   ));
static struct connection_cache * url_make_connection_imaps(schema,user,
							  password,host,
							  port)
     const struct browser_url_method  *schema;
     const struct string              *user   /* May be NULL */;
     const struct string              *password /* May be NULL */;
     const char                       *host   /* May be NULL */;
     int                      port   ;
{

    /* search imaps but not imap */
    return url_make_connection_x(schema,user,password,host,
				 port,itls_require);

}


static uint16 url_default_port_imaps(schema)
     const struct browser_url_method  *schema;
{

    return PORT_imaps;
}


const struct browser_url_method  URL_imaps = {
    BROWSER_URL_method_magic,
    & imap_browser,

    url_default_port_imaps,
    url_make_connection_imaps,
    url_select_item_from_imap_path
    
}; 

#endif


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
