static char rcsid[] = "@(#)$Id: hashmark_remote.c,v 2.12 2022/08/04 05:53:02 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.12 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *       or Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mbox.h"
#include "hashmark.h"
#include "hashmark_imp.h"
#include "s_me.h"
#include "s_elm.h"
#include "ss_imp.h"

#include "rc_imp.h"

DEBUG_VAR(Debug,__FILE__,"mbox");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

#include <errno.h> 
#ifndef ANSI_C
extern int errno;
#endif



S_(usermap_copy_callback null_usermap_copy_callback)
static FILE *null_usermap_copy_callback P_((const char *name, 
					    const char *pathname));
static FILE *null_usermap_copy_callback(name,pathname)
     const char *name;
     const char *pathname;
{
    return NULL;
}

static usermap_copy_callback  * copy_cb = null_usermap_copy_callback;
void set_usermap_copy_callback(cb)
     usermap_copy_callback *cb;
{
    copy_cb = cb;
}


#if defined(I_NETINET_IN) && defined(I_ARPA_INET) && defined(REMOTE_MBX)

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}


S_(hashtype_init_item_f hashtype_init_item_remote)
static void hashtype_init_item_remote P_((struct hashmark_item *item));
S_(hashtype_free_item_f  hashtype_free_item_remote)
static void hashtype_free_item_remote P_((struct hashmark_item *item));
S_(hashtype_parse_kw_value_f hashtype_parse_kw_value_remote)
static int hashtype_parse_kw_value_remote P_((const char *filename,int lineno,
					      struct string * rest_S,
					      struct hashmark_item *item,
					      enum hashkeyword_v kw,
					      struct string * value,
					      enum record_mode rc,
					      const struct schedule_timelimit * now));

S_(hashtype_parse_kw_f hashtype_parse_kw_remote)
static int hashtype_parse_kw_remote P_((const char *filename,int lineno,
					struct string * rest_S,
					struct hashmark_item *item,
					enum hashkeyword_v kw,
					enum record_mode rc,
					const struct schedule_timelimit * now));

S_(hashtype_parse_check_f hashtype_parse_check_remote);
static int hashtype_parse_check_remote P_((const char *filename,int lineno,
					   struct hashmark_item *item,
					   enum record_mode rc,
					   const struct schedule_timelimit * now));

S_(hashtype_dump_kw_f hashtype_dump_kw_remote)
static int hashtype_dump_kw_remote P_((FILE *F,
					  struct hashmark_item *item,
					  enum hashkeyword_v kw,
					  charset_t fileset,
					  const char * sysnam,
					  enum record_mode rc));


S_(hashtype_merge_f hashtype_merge_remote)
static void hashtype_merge_remote P_((struct hashmark_item *item,
				      const struct hashmark_item * base,
				      const struct hashmark_item * overwrite));

S_(hashtype_browser_flags_f hashtype_brflags_remote)
static int  hashtype_brflags_remote P_((const struct hashmark_item *item));

S_(hashtype_init_data_f hashtype_initd_remote)
static void hashtype_initd_remote P_((const struct hashmark_item *item,
				    union hashmark_data      * hashmark_data));

S_(hashtype_free_data_f hashtype_freed_remote)
static void hashtype_freed_remote P_((const struct hashmark_item *item,
				    union hashmark_data      * hashmark_data));


S_(hashtype_browser_changedir_f hashtype_brchdir_remote)
static int hashtype_brchdir_remote P_((struct hashmark_item       * item,
				       struct folder_browser      * dir,
				       union hashmark_data        * hashmark_data,
				       const struct string        * path_tail,
				       struct string             ** dispname));


S_(hashtype_passhm_verify_ra_con_f hashtype_verify_racon_remote)
static int hashtype_verify_racon_remote P_(( const struct hashmark_item * item,
					   struct browser_passhm *passhm,
					   struct remote_account *C));

S_(hashtype_passhm_open_ra_f hashtype_passhm_open_ra_remote)
static int hashtype_passhm_open_ra_remote P_((const struct hashmark_item * item,
					      struct browser_passhm      * passhm,
					      struct remote_account      * ra,
					      const PORTS                  default_portlist[],
					      int                          give_service_entry_flag,
					      PORTS                        force_port));

static int hashtype_passhm_open_ra2_remote P_((const struct hashmark_item * item,
					       struct browser_passhm      * passhm,
					       struct remote_account      * ra,
					       int                          give_service_entry_flag,
					       
					       /* For enumerate_service_type */
					       struct enum_service_type   * est
					       ));

S_(hashtype_selectbr_item_f hashtype_selectbr_item_remote)
static int hashtype_selectbr_item_remote P_((struct hashmark_item  * item,
					     struct folder_browser * dir,
					     union hashmark_data   * hashmark_data,
					     const struct string   * path_tail,
					     struct string        ** dispname,
					     int                   * newpos));


S_(hashtype_folder_from_browser_f hashtype_folder_from_remote)
static struct folder_info * hashtype_folder_from_remote
    P_((struct hashmark_item  * item,
	struct folder_browser * dir,
	union hashmark_data   * hashmark_data,
	int                     treat_as_spooled));

/* -1 == use fallback
    0 == lookup failed
    1 == ok

    caller must free *se, *username, *hostname, *addr
*/
S_(hashtype_remote_lookup_f hashtype_remote_lookup_remote)
static int hashtype_remote_lookup_remote P_((const struct hashmark_item * item,
					     struct service_entry      ** se,
					     int                          lookup_flags,
					     char                      ** username,
					     char                      ** hostname,
					     struct address            ** addr,
					     int                        * useraddr_flags,
					     struct cancel_data         ** cancel_p
					        /* May be NULL,
						   Used if dns lookup was
						   cancelable 
						*/

					     ));

S_(hashtype_selection_is_folder_f hashtype_selection_is_remote)
static int hashtype_selection_is_remote P_((struct hashmark_item  * item,
					 struct folder_browser * dir,
					 union hashmark_data   * hashmark_data,
					 struct folder_info    * folder));

S_(hashtype_valid_on_user_f hashtype_valid_on_user_remote)
static enum haskmark_valid_v hashtype_valid_on_user_remote 
    P_((const struct hashmark_item  * item));


S_(hashtype_prepare_write_item_f hashtype_prepare_write_remote)
static int hashtype_prepare_write_remote
    P_((struct hashmark_item  * item,
	struct folder_browser * dir,
	union hashmark_data   * hashmark_data,
	WRITE_STATE             ptr));

/* Return directory prefix and l_item,
   changes browser type
*/
S_(hashtype_cat_hashmark_item_f hashtype_cat_hashmark_remote)
static  struct string * hashtype_cat_hashmark_remote
    P_((struct hashmark_item  * h_item,
	struct folder_browser * dir,
	union hashmark_data   * hashmark_data,
	const struct string   * l_item));

struct hashtype_actions hashtype_remote_mbx_action = 
    { /* hashtype_remote_mbx */
	HASHTYPE_ACTIONS_magic,
	hashtype_init_item_remote,
	hashtype_free_item_remote,
	hashtype_parse_kw_value_remote,
	hashtype_parse_kw_remote,
	hashtype_parse_check_remote,
	hashtype_dump_kw_remote,
	hashtype_merge_remote,
	hashtype_brflags_remote,
	hashtype_initd_remote,
	hashtype_freed_remote,
	hashtype_brchdir_remote,	
	hashtype_verify_racon_remote,
	hashtype_passhm_open_ra_remote,
	hashtype_selectbr_item_remote,
	hashtype_folder_from_remote,
	hashtype_remote_lookup_remote,
	hashtype_selection_is_remote,
	hashtype_valid_on_user_remote,
	hashtype_prepare_write_remote,
	hashtype_cat_hashmark_remote,
	hashtype_passhm_open_ra2_remote
    };

#define HASHMARK_REMOTE_magic	0xF51E

struct hashmark_remote {
    unsigned short magic; 	/* HASHMARK_REMOTE_magic */

    struct service_entry        * service;       /* refcounted */
    char                        * hostname;      /* ASCII only */
    union SOCKADDR_ptr           ip_literal_address;
    size_t                       ip_literal_addrsize;
    const struct service_type  * servicetype;
    
    struct string   ** req_tls_peername_list;
    int                req_tls_peername_count;

    /* LOCAL_RC */

    char             * remote_userid;     /* currently ASCII only */
	                                 /* if NULL, use default remote-userid */
    struct  address * from_address;

    /* SYSTEM_RC */

    struct hashmark_usermap_entry {
	char * local_username;           /* currently ASCII only */
	char * remote_userid;            /* currently ASCII only */
	                                 /* if NULL, use default remote-userid */


	struct address * from_address;
	
    }  *         usermaps;
    int          usermap_count;
    int          usermap_alloc_count;

#define HASHMARK_USERMAP_FILE_magic	0xF51F

    struct hashmark_usermap_file {
	unsigned short magic;      /* HASHMARK_USERMAP_FILE_magic */

	int            refcount;
	char         * filename;  /* systen charset */

	struct hashmark_list_entry {
	    struct string * hashmark;

	    struct hashmark_usermap_entry * usermaps;
	    int                             usermap_count;
	    int                             usermap_alloc_count;
	} *  hashmarks;
	int  hashmark_count;


    }       *   usermap_file;
    struct string * usermap_file_value; /* unexpanded */

    /* flags */

    unsigned     verify_tls_certificate :1;
    unsigned     require_tls_peer_name  :1;  /* hostname or service->official_name */
    unsigned     self_cc :1;
    unsigned     self_bcc :1;
};

struct hashmark_usermap_entry * find_current_user P_((struct hashmark_usermap_entry
						      * usermaps,
						      int          usermap_count));
struct hashmark_usermap_entry * find_current_user (usermaps,usermap_count)
     struct hashmark_usermap_entry * usermaps;
     int                             usermap_count;
{
    int i;

    for (i = 0; i < usermap_count; i++) {
	if (0 == strcmp(username,usermaps[i].local_username))
	    return (& (usermaps[i]));
    }

    return NULL;
}



static struct hashmark_usermap_file * malloc_hashmark_usermap_file P_((const char *filename));
static struct hashmark_usermap_file * malloc_hashmark_usermap_file(filename)
     const char *filename;
{
    struct hashmark_usermap_file *ret = safe_zero_alloc(sizeof(*ret));

    ret->magic = HASHMARK_USERMAP_FILE_magic;
    ret->refcount = 1;
    ret->filename = safe_strdup(filename);
    
    ret->hashmarks = 0;
    ret->hashmark_count = 0;

    return ret;
}

static void free_hashmark_usermap P_((struct hashmark_usermap_entry ** usermaps,
				      int                            * usermap_count));

static void free_hashmark_usermap(usermaps,usermap_count)
     struct hashmark_usermap_entry ** usermaps;
     int                            * usermap_count;
{
    struct hashmark_usermap_entry * USERMAPS = *usermaps;
    int USERMAP_COUNT = *usermap_count;

    if (USERMAPS) {
	int i;

	for (i = 0; i < USERMAP_COUNT; i++) {
	    if (USERMAPS[i].local_username) {
		free(USERMAPS[i].local_username);
		USERMAPS[i].local_username = NULL;
	    }
	    if (USERMAPS[i].remote_userid) {
		free(USERMAPS[i].remote_userid);
		USERMAPS[i].remote_userid = NULL;
	    }

	    if (USERMAPS[i].from_address)
		free_address(& (USERMAPS[i].from_address));
	}

	free(USERMAPS);
	USERMAPS = NULL;
    }
    USERMAP_COUNT = 0;

    
    *usermap_count = USERMAP_COUNT;
    *usermaps = USERMAPS;    
}

static void free_hashmark_list P_((struct hashmark_list_entry **hasmarks,
				   int * hashmark_count));
static void free_hashmark_list(hashmarks,hashmark_count)
     struct hashmark_list_entry **hashmarks;
     int * hashmark_count;
{
    struct hashmark_list_entry *HASHMARKS = *hashmarks;
    int HASHMARK_COUNT                   = *hashmark_count;
    
    if (HASHMARKS) {
	int i;

	for (i = 0; i < HASHMARK_COUNT; i++) {
	    if (HASHMARKS[i].hashmark)
		free_string(& (HASHMARKS[i].hashmark));

	    free_hashmark_usermap( & (HASHMARKS[i].usermaps),
				   & (HASHMARKS[i].usermap_count));
	    
	    HASHMARKS[i].usermap_alloc_count = 0;

	}

	free(HASHMARKS);
	HASHMARKS = 0;
    }
    HASHMARK_COUNT = 0;

    *hashmarks      = HASHMARKS;
    *hashmark_count = HASHMARK_COUNT;
}

static void inc_hashmark_usermap_refcount P_((struct hashmark_usermap_file *usermap_file));
static void inc_hashmark_usermap_refcount(usermap_file)
     struct hashmark_usermap_file *usermap_file;
{
    if (HASHMARK_USERMAP_FILE_magic != usermap_file->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "inc_hashmark_usermap_refcount",
              "Bad magic number",0);

    usermap_file->refcount++;
}


static void free_hashmark_usermap_file P_((struct hashmark_usermap_file **usermap_file));
static void free_hashmark_usermap_file(usermap_file)
     struct hashmark_usermap_file **usermap_file;
{
    if (HASHMARK_USERMAP_FILE_magic != (*usermap_file)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_hashmark_usermap_file",
              "Bad magic number",0);

    if ((*usermap_file)->refcount < 1)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_hashmark_usermap_file",
	      "Bad recount",0);

    (*usermap_file)->refcount--;
    if ((*usermap_file)->refcount > 0) {  /* Do not free */

        (*usermap_file) = NULL;           /* Refefence count for this pointer is decrement, */
	                                  /* so this must have be reset                     */
        return;	
    }

    if ((*usermap_file)->filename) {
	free((*usermap_file)->filename);
	(*usermap_file)->filename = NULL;
    }

    free_hashmark_list(& ((*usermap_file)->hashmarks),
		       & ((*usermap_file)->hashmark_count));

    (*usermap_file)->magic = 0;
    free(*usermap_file);
    *usermap_file = NULL;
}

static struct hashmark_remote * malloc_hashmark_remote P_((void));
static struct hashmark_remote * malloc_hashmark_remote() 
{
    struct hashmark_remote * ret =  safe_zero_alloc(sizeof(* ret));

    ret->magic    = HASHMARK_REMOTE_magic;
    ret->service  = NULL;
    ret->hostname = NULL;
    ret->ip_literal_address.dummy = NULL;
    ret->ip_literal_addrsize      = 0;
    ret->servicetype = NULL;
    ret->req_tls_peername_list  = NULL;
    ret->req_tls_peername_count = 0;
    ret->remote_userid          = NULL;
    ret->from_address = NULL;
    ret->usermaps    = NULL;
    ret->usermap_count = 0;
    ret->usermap_alloc_count = 0;
    ret->usermap_file  = NULL;
    ret->usermap_file_value = NULL;
    ret->verify_tls_certificate = 0;
    ret->require_tls_peer_name  = 0;
    ret->self_cc                = 0;
    ret->self_bcc               = 0;
    
    return ret;
}

static void free_hashmark_remote P_((struct hashmark_remote **remote));
static void free_hashmark_remote(remote)
     struct hashmark_remote **remote;
{

    if (HASHMARK_REMOTE_magic != (*remote)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_hashmark_remote",
              "Bad magic number",0);

    if ((*remote)->service)
	free_service_entry(& ((*remote)->service)); /* Decrement refcount */
    if ((*remote)->hostname) {
	free((*remote)->hostname);
	(*remote)->hostname = NULL;
    }
    if ((*remote)->ip_literal_address.dummy) {
	free((*remote)->ip_literal_address.dummy);
	(*remote)->ip_literal_address.dummy = NULL;
    }
    (*remote)->ip_literal_addrsize = 0;
    (*remote)->servicetype = NULL;

    if ((*remote)->req_tls_peername_list) {
	int i;

	for (i = 0; i < (*remote)->req_tls_peername_count; i++) 
	    free_string(& ((*remote)->req_tls_peername_list[i]));
	
	free((*remote)->req_tls_peername_list);
	(*remote)->req_tls_peername_list = NULL;	
    }
    (*remote)->req_tls_peername_count = 0;

    if ((*remote)->remote_userid) {
	free((*remote)->remote_userid);
	(*remote)->remote_userid = NULL;
    }

    if ((*remote)->from_address)
	free_address(& ((*remote)->from_address));

    free_hashmark_usermap(& ((*remote)->usermaps),
			  & ((*remote)->usermap_count));
    (*remote)->usermap_alloc_count = 0;

    if ((*remote)->usermap_file)
	free_hashmark_usermap_file( & ((*remote)->usermap_file));

    if ((*remote)->usermap_file_value)
	free_string(&((*remote)->usermap_file_value));

    (*remote)->verify_tls_certificate = 0;
    (*remote)->require_tls_peer_name  = 0;
    (*remote)->self_cc                = 0;
    (*remote)->self_bcc               = 0;
    
    free(*remote);
    *remote = NULL;
}

static struct hashmark_usermap_file ** usermap_files = NULL;
static int                             usermap_files_count = 0;

/* Increments refcount */
static struct hashmark_usermap_file * locate_hashmark_usermap_file P_((const char *filename));
static struct hashmark_usermap_file * locate_hashmark_usermap_file(filename)
     const char *filename;
{
    int i;

    for (i = 0; i < usermap_files_count; i++) {
	if (HASHMARK_USERMAP_FILE_magic != usermap_files[i]->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "free_hashmark_usermap_file",
		  "locate_hashmark_usermap_file",0);

	if (0 != strcmp(usermap_files[i]->filename,
			filename)) {
	    inc_hashmark_usermap_refcount(usermap_files[i]);
	    return usermap_files[i];
	}
    }

    return NULL;
}

static void hashtype_init_item_remote(item)
     struct hashmark_item *item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_init_item_remote",
              "Bad magic number",0);

    item->u.remote =  malloc_hashmark_remote();
}

static void hashtype_free_item_remote(item)
     struct hashmark_item *item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_free_item_remote",
              "Bad magic number",0);

    if (item->u.remote) 
	free_hashmark_remote(& (item->u.remote)); 

}

/* mallocs value, NULL on error */
static char * hashtype_ascii_param_value1 P_((const char *filename,
					     int lineno,
					     struct string * value));
static char * hashtype_ascii_param_value1(filename,lineno,value)
     const char *filename;
     int lineno;
     struct string * value;
{
    const char *csn = get_string_MIME_name(value);
    const char * asciinam = get_charset_MIME_name(ASCII_SET);
    int ERRORS = 0;
    struct string * S2 = convert_string2(ASCII_SET,value,&ERRORS);
    unsigned char * res = NULL;	

    if (!asciinam)
	asciinam = "US-ASCII";

    if (ERRORS) {
	if (hashmark_is_printableln(filename,lineno,value))
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeFailedConvertValue,
			      "%s: %d: There is %d errors when converting value %S from charset %s to %s."),
		      filename,lineno,
		      ERRORS,
		      csn,asciinam);

	free_string(&S2);
	return NULL;
    }
    
    res = stream_from_string(S2,0,NULL);
    free_string(&S2);
    
    return us2s(res);	
}

/* mallocs value, NULL on error */
static char * hashtype_cs_param_value P_((const char *filename,
					  int lineno,
					  struct string * rest_S,
					  enum hashkeyword_v kw,
					  struct string * value,
					  charset_t cs,
					  const char * csname));

static char * hashtype_cs_param_value(filename,lineno,rest_S,kw,
				      value,cs,csname)
     const char *filename;
     int lineno;
     struct string * rest_S;
     enum hashkeyword_v kw;
     struct string * value;
     charset_t cs;
     const char * csname;
{
    const char *csn = get_string_MIME_name(value);
    const char * csnam = get_charset_MIME_name(cs);
    int ERRORS = 0;
    struct string * S2 = convert_string2(cs,value,&ERRORS);
    unsigned char * res = NULL;	

    if (!cs)
	csnam = csname;

    if (ERRORS) {
	if (hashmark_is_printableln(filename,lineno,rest_S))
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeFailedConvertKwTo,
			      "%s: %d: There is %d errors when converting keyword %s value %S from charset %s to %s on line: %S"),
		      filename,lineno,
		      ERRORS,
		      hashmark_Keyword(kw),value,
		      csn,csnam,rest_S);
	   
	 free_string(&S2);
	 return NULL;
    }

    res = stream_from_string(S2,0,NULL);
    free_string(&S2);

    return us2s(res);
}


/* mallocs value, NULL on error */
static char * hashtype_ascii_param_value P_((const char *filename,
					     int lineno,
					     struct string * rest_S,
					     enum hashkeyword_v kw,
					     struct string * value));
static char * hashtype_ascii_param_value(filename,lineno,rest_S,kw,value)
     const char *filename;
     int lineno;
     struct string * rest_S;
     enum hashkeyword_v kw;
     struct string * value;
{
    return hashtype_cs_param_value(filename,lineno,rest_S,kw,
				   value,ASCII_SET,"US-ASCII");
}

static struct string * hashmark_usermap_entry_string P_((struct hashmark_usermap_entry 
							 *entry,
							 charset_t cs));
static struct string * hashmark_usermap_entry_string(entry,cs)
     struct hashmark_usermap_entry *entry;
     charset_t cs;
{
    struct string * ret = new_string(cs);
    
    struct string * L = new_string2(ASCII_SET,s2us(entry->local_username));

    if (string_need_quote(L))
	append_quoted_to_string(&ret,L);
    else
	append_string(&ret,L,0);

    add_ascii_to_string(ret,s2us(" "));

    if (entry->remote_userid) {
	struct string * R = new_string2(ASCII_SET,s2us(entry->remote_userid));

	if (string_need_quote(R))
	    append_quoted_to_string(&ret,R);
	else
	    append_string(&ret,R,0);
       
	free_string(&R);

    } else
	add_ascii_to_string(ret,s2us(":"));


    if (entry->from_address) {
	struct string * from_addr = address_to_string(entry->from_address);

	if (from_addr) {
	    add_ascii_to_string(ret,s2us(" "));

	    append_string(&ret,from_addr,0);
	
	    free_string(&from_addr);
	}
    }

    free_string(&L);

    return ret;
}


static int hashmark_usermap_parse_rest P_((const char *filename, int lineno,
					   struct hashmark_usermap_entry ** usermaps,
					   int                            * usermap_count,
					   int                            * usermap_alloc_count,
					   struct string_token * tokenized,					   
					   int pos));
static int hashmark_usermap_parse_rest(filename,lineno,
				       usermaps,usermap_count,usermap_alloc_count,
				       tokenized,pos)
     const char *filename;
     int lineno;
     struct hashmark_usermap_entry ** usermaps;
     int                            * usermap_count;
     int                            * usermap_alloc_count;    
     struct string_token * tokenized;
     int pos;
{
    struct hashmark_usermap_entry * USERMAPS = *usermaps;
    int USERMAP_COUNT                        = *usermap_count;
    int ALLOC_COUNT                          = * usermap_alloc_count;

    int ret = 0;
    int parse_error = 0;    

    /*  local-userid    username   aaa bbbb <local-part@addr>   
	"local-userid"  "username" "phrase" <local-part@addr>
	local-userid    username
	"local-userid"  "username"
	local-userid    :
	local-userid    :          aaa bbbb <local-part@addr>
	"local-userid"  :          "phrase" <local-part@addr>
     */

    struct string    * local_userid = NULL;
    struct string    * remote_userid = NULL;
    struct address   * from_address  = NULL;



    if (0x0020 /* SPACE  */ == tokenized[pos].special) { 
	DPRINT(Debug,13,(&Debug, 
			 "hashmark_usermap_parse_rest: [%d] token=%S special=%04x skipping\n",
			 pos,tokenized[pos].token,tokenized[pos].special));
			 

	pos++;
    }
    
    if (! tokenized[pos].token) {
	lib_error(CATGETS(elm_msg_cat, MeSet, 
			  MeExpectedLocalUseridEOLN,
			  "%s: %d: Expected local-userid when got end of line."),
		  filename,lineno);
	ret = 0;
	goto fail;			  
    }

    /* local-userid */
    switch (tokenized[pos].special) {

    case 0: /* not special */
	local_userid = dup_string(tokenized[pos].token);

	DPRINT(Debug,13,(&Debug, 
			 "hashmark_usermap_parse_rest: [%d] token=%S special=%04x parsed as local userid: %S\n",
			 pos,tokenized[pos].token,tokenized[pos].special,
			 local_userid
			 ));
			 

	break;

    case 0x0022  /* " */:
	
	local_userid = unquote_string(tokenized[pos].token,
				      hashmark_is_printableln,
				      filename,lineno,
				      &parse_error);

	if (!local_userid || parse_error) {

	    if (hashmark_is_printableln(filename,lineno,tokenized[pos].token))
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeExpectedLocalQUidGotError,
				  "%s: %d: Expected quoted local-userid when got '%S'."),
			  filename,lineno,tokenized[pos].token);

	    ret = 0;
	    goto fail;
	}

	DPRINT(Debug,13,(&Debug, 
			 "hashmark_usermap_parse_rest: [%d] token=%S special=%04x parsed as quoted local userid: %S\n",
			 pos,tokenized[pos].token,tokenized[pos].special,
			 local_userid));

	break;

    default:
	if (hashmark_is_printableln(filename,lineno,tokenized[pos].token))
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeExpectedLocalUseridGotSpecial,
			      "%s: %d: Expected local-userid when got \"%S\"."),
		  filename,lineno,tokenized[pos].token);
	ret = 0;
	goto fail;
    }

    pos++;
    if (!  tokenized[pos].token) {
	lib_error(CATGETS(elm_msg_cat, MeSet, 
			  MeExpectedSpaceLocalUseridEOLN,
			  "%s: %d: Expected space after local-userid when got end of line."),
		  filename,lineno);
	ret = 0;
	goto fail;	
    }

    if ( 0x0020 /* SPACE  */ == tokenized[pos].special) {
	DPRINT(Debug,13,(&Debug, 
			 "hashmark_usermap_parse_rest: [%d] token=%S special=%04x skipping\n",
			 pos,tokenized[pos].token,tokenized[pos].special));
		    
	pos++;
	
    } else {
	if (hashmark_is_printableln(filename,lineno,tokenized[pos].token))
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeExpectedSpaceLocalUseridGotSpecial,
			      "%s: %d: Expected space after local-userid when got \"%S\"."),
		      filename,lineno);
	ret = 0;
	goto fail;	
    }

    if (! tokenized[pos].token) {
	lib_error(CATGETS(elm_msg_cat, MeSet, 
			  MeExpectedRemoteUseridEOLN,
			  "%s: %d: Expected remote-userid when got end of line."),
		  filename,lineno);
	ret = 0;
	goto fail;			  
    }

    /* remote-userid */
    switch (tokenized[pos].special) {

    case 0: /* not special */
	remote_userid = dup_string(tokenized[pos].token);

	DPRINT(Debug,13,(&Debug, 
			 "hashmark_usermap_parse_rest: [%d] token=%S special=%04x parsed as remote userid: %S\n",
			 pos,tokenized[pos].token,tokenized[pos].special,
			 remote_userid
			 ));
			 

	break;

    case 0x0022  /* " */:
	remote_userid = unquote_string(tokenized[pos].token,
				       hashmark_is_printableln,
				       filename,lineno,
				       &parse_error);

	if (!remote_userid  || parse_error) {

	    if (hashmark_is_printableln(filename,lineno,tokenized[pos].token))
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeExpectedRemoteQUidGotError,
				  "%s: %d: Expected quoted remote-userid when got '%S'."),
			  filename,lineno,tokenized[pos].token);

	    ret = 0;
	    goto fail;
	}

	DPRINT(Debug,13,(&Debug, 
			 "hashmark_usermap_parse_rest: [%d] token=%S special=%04x parsed as quoted remote userid: %S\n",
			 pos,tokenized[pos].token,tokenized[pos].special,
			 remote_userid));

	break;

    case  0x003A    /* ':' */:
	DPRINT(Debug,13,(&Debug, 
			 "hashmark_usermap_parse_rest: [%d] token=%S special=%04x parsed as default remote userid\n",
			 pos,tokenized[pos].token,tokenized[pos].special));
	break;

    default:
	if (hashmark_is_printableln(filename,lineno,tokenized[pos].token))
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeExpectedRemoteUseridGotSpecial,
			      "%s: %d: Expected remote-userid when got \"%S\"."),
		  filename,lineno,tokenized[pos].token);
	ret = 0;
	goto fail;
    }
    pos++;


    /* from address is optional */
    if (tokenized[pos].token) {
	
	if ( 0x0020 /* SPACE  */ == tokenized[pos].special) {
	    DPRINT(Debug,13,(&Debug, 
			     "hashmark_usermap_parse_rest: [%d] token=%S special=%04x skipping\n",
			     pos,tokenized[pos].token,tokenized[pos].special));
	    
	    pos++;
	    
	} else {
	    if (hashmark_is_printableln(filename,lineno,tokenized[pos].token))
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeExpectedSpaceRemoteUseridGotSpecial,
				  "%s: %d: Expected space after remote-userid when got \"%S\"."),
			  filename,lineno);
	    ret = 0;
	    goto fail;	
	}

	if (tokenized[pos].token) {
	    int last;

	    for (last = pos; tokenized[last].token; last++) {
		DPRINT(Debug,13,(&Debug, 
				 "hashmark_usermap_parse_rest: [%d] token=%S special=%04x part of from header\n",
				 pos,tokenized[last].token,tokenized[last].special));

	    }
	
	    from_address =  
		parse_one_tokenized_address(tokenized,pos,last,
					    &parse_error,
					    hashmark_is_printableln,
					    filename,lineno);

	    if (!from_address || parse_error) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeExpectedFromAddrAfterRemoteUserLine,
				  "%s: %d: Expected from address after remote-userid on line."),
			  filename,lineno);
		ret = 0;
		goto fail;		  
	    }	       
	}
    }

    if (ALLOC_COUNT < USERMAP_COUNT+1) {
	int new_count = USERMAP_COUNT + 10;

	USERMAPS = safe_array_realloc(USERMAPS,
				      new_count,
				      sizeof (USERMAPS[0]));

	if (ALLOC_COUNT < USERMAP_COUNT)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  " hashmark_usermap_parse_rest",
		  "Bad usermap_alloc_count",0);
	
	while (ALLOC_COUNT < new_count) {
	    USERMAPS[ALLOC_COUNT].local_username = NULL;
	    USERMAPS[ALLOC_COUNT].remote_userid  = NULL;
	    USERMAPS[ALLOC_COUNT].from_address   = NULL;
	    ALLOC_COUNT++;
	}
    }

    /* Currently ascii only */
    USERMAPS[USERMAP_COUNT].local_username =
	hashtype_ascii_param_value1(filename,lineno,
				    local_userid);
    if (! USERMAPS[USERMAP_COUNT].local_username)
	goto fail1;

    if (remote_userid) {
	USERMAPS[USERMAP_COUNT].remote_userid =
	    hashtype_ascii_param_value1(filename,lineno,
					remote_userid);
	if (! USERMAPS[USERMAP_COUNT].remote_userid)
	    goto fail1;;
    } else
	USERMAPS[USERMAP_COUNT].remote_userid = NULL;	

    USERMAPS[USERMAP_COUNT].from_address = from_address;
    from_address = NULL;
    USERMAP_COUNT++;
    ret = 1;

    if (!ret) {
    fail1:
	if (USERMAPS[USERMAP_COUNT].local_username) {
	    free(USERMAPS[USERMAP_COUNT].local_username);
	    USERMAPS[USERMAP_COUNT].local_username = NULL;
	}
	if (USERMAPS[USERMAP_COUNT].remote_userid) {
	    free(USERMAPS[USERMAP_COUNT].remote_userid);
	    USERMAPS[USERMAP_COUNT].remote_userid = NULL;
	}
	if (USERMAPS[USERMAP_COUNT].from_address) 
	    free_address(& (USERMAPS[USERMAP_COUNT].from_address));
    }

 fail:
    if (from_address)
	free_address(& from_address);
    if (remote_userid)
	free_string(&remote_userid);
    if (local_userid)
	free_string(&local_userid);

    *usermaps            =  USERMAPS;
    *usermap_count       =  USERMAP_COUNT;
    *usermap_alloc_count =  ALLOC_COUNT;

    if (!ret) {
	if (!  tokenized[pos].token) {
	    DPRINT(Debug,13,(&Debug, 
			     "hashmark_usermap_parse_rest: [%d] failed at end of tokens\n",
			     pos));
	} else {
	    DPRINT(Debug,13,(&Debug, 
			     "hashmark_usermap_parse_rest: [%d] token=%S special=%04x failed\n",
			     pos,tokenized[pos].token,tokenized[pos].special));
	}
    }

    return ret;
}


static struct hashmark_usermap_file * load_hashmark_usermap_file 
       P_((const char *conf_filename,
	   int conf_lineno,
	   struct string * rest_S,
	   enum hashkeyword_v kw,
	   struct string * value,
	   const char *usermap_filename,
	   const char *usermap_cb_filename
	   ));
static struct hashmark_usermap_file * load_hashmark_usermap_file(
     conf_filename,conf_lineno,
     rest_S,kw,value,
     usermap_filename,usermap_cb_filename)
     const char       * conf_filename;
     int                conf_lineno;
     struct string    * rest_S;
     enum hashkeyword_v kw;
     struct string    * value;
     const char       * usermap_filename;
     const char       * usermap_cb_filename;
{
    struct hashmark_usermap_file * ret = NULL;
    int err = can_open(usermap_filename,"r");
    FILE * F = NULL;
    int max_count = 0;
    charset_t cs = system_charset;
    const char * csn = get_charset_MIME_name(cs);
    int usermap_lineno = 0;
    int last_c = 0;
    int c;

#if USERMAP_PROPLINE
    enum editor_propline_v propline_mode = 
	give_dt_enumerate_as_int(&editor_usermap_propline);
#endif

    char * buf = NULL;

    if (ENOENT == err && usermap_cb_filename) {
	F = copy_cb(usermap_cb_filename,
		    usermap_filename);
	if (F) {
	    DPRINT(Debug,9,(&Debug, 
			    "load_hashmark_usermap_file: copied %s to %s\n",
			    usermap_cb_filename,usermap_filename));
	}
    }

    if (0 == err) {
	F = fopen(usermap_filename,"r");
	if (F) {
	    DPRINT(Debug,9,(&Debug, 
			    "load_hashmark_usermap_file: using %s\n",
			    usermap_filename));

	} else {
	    err = errno;
	}
    }
    
    if (!F) {
	if (hashmark_is_printableln(conf_filename,conf_lineno,rest_S))
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MePathKwValueNotReadable,
			      "%s: %d: Path \"%s\" is not readable (%s) from keyword %s value %S on line: %S"),
		      conf_filename,conf_lineno,
		      usermap_filename,
		      strerror(err),
		      hashmark_Keyword(kw),value,
		      rest_S);
	return NULL;
		      
    }

#if USERMAP_PROPLINE
    if (propline_mode != editor_propline_ignore) {
	struct editor_propline * temp_propline = NULL;
	int i;
	
	for (i = 0; i < 3 && !temp_propline; i++) {
	    int l1;
	    
	    c = fgetc(F);
	    
	    if (EOF == c)
		goto propline_failure;
	    if ('\n' == c)
		continue;

	    ungetc(c,F);

	    if ('#' != c)
		break;

	    l1 = malloc_gets(&buf,LONG_STRING,F);
	    if (l1 > 1) {

		if ('#' != buf[0])
		    goto propline_failure;

		temp_propline =
		    detect_editor_propline(propline_mode,
					   buf,l1,usermap_filename,i,&cs);

		csn = get_charset_MIME_name(cs);

	    } else { /* something wrong? */
	    propline_failure:
		
		rewind(F);
		break;
	    }
	}

	if (temp_propline)
	    free_editor_propline(&temp_propline);
    }
#endif

    while(EOF != (c = fgetc(F))) {
	if ('\n' == c)
	    max_count++;
	last_c = c;
    }

     if (last_c && '\n' != last_c) {
	DPRINT(Debug,9,(&Debug, 
			"load_hashmark_usermap_file: %s, no newline on end of file\n",
			usermap_filename));
	max_count++;	
    }

     DPRINT(Debug,9,(&Debug, 
		     "load_hashmark_usermap_file: %s, max_count=%d\n",
		     usermap_filename,max_count));

     ret = malloc_hashmark_usermap_file(usermap_filename);
     rewind(F);


     if (max_count) {
	 struct string * last_hashmark = NULL;
	 int             last_hashmark_index = -1;

	 ret->hashmarks = 
	     safe_calloc(max_count,sizeof (ret->hashmarks[0]));

	 while (ret->hashmark_count < max_count) {
	     int l1 = malloc_gets(&buf,LONG_STRING,F);
	     struct string_token * tokenized UNUSED_VAROK = NULL;
	     struct string * rest_S = NULL;
	     int ERRORS UNUSED_VAROK = 0;
	     int rs UNUSED_VAROK;

	     char *cx = buf;

	     if (-1 == l1) {
		 lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLineNo,
				   "%s: %d: Too long line: %.30s..."),
			   usermap_filename,usermap_lineno+1,buf);
		 break;
	     } else if (l1 < 0) {
		 DPRINT(Debug,9,(&Debug, 
				 "load_hashmark_usermap_file: %s: %d: read error or EOF\n",
				 usermap_filename,usermap_lineno+1));
		 break;
	     }
	     
	     usermap_lineno++;
	     
	     if (l1 == 0) {
		 if (last_hashmark)   /* forget last hashmark on empty line */
		     free_string(&last_hashmark);
		 last_hashmark_index = -1;

		 continue;
	     }
	     
	     l1 = trim_whitespace_from_end(buf,l1);

	     if (read_charset_tag(buf,l1,
				  usermap_filename,
				  usermap_lineno,NULL,&cs,&csn))
		 continue;


	     cx = buf;
	     while (*cx && whitespace (*cx)) /* skip leading whitespace */
		 cx++;
	     if ('#' == *cx) {
		 if (last_hashmark)   /* forget last hashmark on comment */
		     free_string(&last_hashmark);
		 last_hashmark_index = -1;

		 continue;
	     }

	     if (!*cx){

		 if (cx < buf + l1) {
		     lib_error(CATGETS(elm_msg_cat, MeSet, 
				       MeNULcharacterOnLine,
				       "%s: %d: NUL character on line."),
			       usermap_filename,
			       usermap_lineno);
		 }

		 if (last_hashmark)   /* forget last hashmark on comment */
		     free_string(&last_hashmark);
		 last_hashmark_index = -1;

		 continue;
	     }

	     /* hashmark must be first item, without
		leading space */

	     if (cx > buf) {

		 lib_error(CATGETS(elm_msg_cat, MeSet, 
				   MeHashMarkBeginngOfLine,
				   "%s: %d: Hasmark must be on beginning of line."),
			   usermap_filename,usermap_lineno+1);
				   
		 if (last_hashmark)   /* forget last hashmark on empty line */
		     free_string(&last_hashmark);
		 last_hashmark_index = -1;

		 continue;
	     }

	     rest_S = new_string(cs);
	     rs =  add_streambytes_to_string(rest_S,l1,
					     cs2us(buf),&ERRORS);
	     if (rs < l1) {
		 lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedAsCharset,
				   "Failed to parse line %d as charset %s in \"%s\" file"),
			   usermap_lineno, csn ? csn : "<no MIME name>",
			   usermap_filename);

		 goto fail;
	     }
	     
	     if (ERRORS) {
		 lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFileAsCharsetErrors,
				   "There is %d errors when parsing line %d as charset %s in \"%s\" file"),
			   ERRORS,usermap_lineno,
			   csn ? csn : "<no MIME name>",
			   usermap_filename);
		 
		 goto fail;
	     }	 
    

	     tokenized = string_tokenize(rest_S,TOK_mail);
	     
	     if (tokenized) {
		 
		 if (tokenized[0].token) {
		     const struct string * current_hashmark = NULL;  /* Shared pointer */
		     /*  First token is hashmark */

		     switch (tokenized[0].special) {
		     case 0x003A /* : */:

			 if (last_hashmark)
			     current_hashmark = last_hashmark;
			 else {
			      lib_error(CATGETS(elm_msg_cat, MeSet, 
						MeNeedHaskmarkNotPrevious,
						"%s: %d: Need hashmark's name, previous hashmark not available."),
					usermap_filename,
					usermap_lineno);
			      goto fail;
			 }
			 
			 break;
		     case 0x0000:

			 current_hashmark = tokenized[0].token;

			 break;
		     default:
			 if (hashmark_is_printableln(usermap_filename,
						     usermap_lineno,tokenized[0].token))
			     lib_error(CATGETS(elm_msg_cat, MeSet, 
					       MeExpectedHashMarkdGotSpecial,
					       "%s: %d: Expected hashmark's name when got \"%S\"."),
				       usermap_filename,
				       usermap_lineno,
				       tokenized[0].token);
			 goto fail;
		     }

		     if (last_hashmark_index < 0 ||
			 current_hashmark != last_hashmark) {
			 int i;

			 for (i = 0; i < ret->hashmark_count; i++) {
			     if ( 0 == string_cmp(current_hashmark,
						  ret->hashmarks[i].hashmark,
						  -99)) {
				 last_hashmark_index = i;

				 DPRINT(Debug,13,(&Debug, 
						  "load_hashmark_usermap_file: %s: %d: Using hashmark #%d = %S\n",
						  usermap_filename,usermap_lineno,
						  i,current_hashmark));
				 goto found_hashmark;
			     }
			 }

			 if (ret->hashmark_count >= max_count)
			     panic("CONNECTION PANIC",__FILE__,__LINE__,
				   "load_hashmark_usermap_file",
				   "Overflow",0);

			 ret->hashmarks[ret->hashmark_count].
			     hashmark = dup_string(current_hashmark);
			 ret->hashmarks[ret->hashmark_count].
			     usermaps = NULL;
			 ret->hashmarks[ret->hashmark_count].
			     usermap_count = 0;
			 ret->hashmarks[ret->hashmark_count].
			     usermap_alloc_count = 0;
		
			 DPRINT(Debug,13,(&Debug, 
					  "load_hashmark_usermap_file: %s: %d: Added hashmark #%d = %S\n",
					  usermap_filename,usermap_lineno,
					  ret->hashmark_count,current_hashmark));
			
			 last_hashmark_index = ret->hashmark_count++;
		     } 

		 found_hashmark:
		     if (last_hashmark_index >= ret->hashmark_count ||
			 !  ret->hashmarks[last_hashmark_index].hashmark ||
			 0 != string_cmp(current_hashmark,
					 ret->hashmarks[last_hashmark_index].hashmark,
					 -99))
			 panic("CONNECTION PANIC",__FILE__,__LINE__,
			       "load_hashmark_usermap_file",
			       "Bad last_hashmark_index",0);
				

		     hashmark_usermap_parse_rest(usermap_filename,
						 usermap_lineno,
						 & (ret->hashmarks[last_hashmark_index].
						    usermaps),
						 & (ret->hashmarks[last_hashmark_index].
						    usermap_count),
						 & (ret->hashmarks[last_hashmark_index].
						    usermap_alloc_count),
						 tokenized,
						 1);
						 		     
		     if (current_hashmark != last_hashmark) {
			 if (last_hashmark)
			     free_string(&last_hashmark);
			 
			 last_hashmark = dup_string(current_hashmark);
			 last_hashmark_index = -1;
		     }

		 } else if (last_hashmark) {  /* forget last hashmark on empty line */
		     free_string(&last_hashmark);
		     last_hashmark_index = -1;
		 }

		 free_string_tokenized(&tokenized);
	     }

	 fail:
	     free_string(&rest_S);
	 }
     
	 DPRINT(Debug,9,(&Debug, 		    
			 "load_hashmark_usermap_file: %s, %d parsed\n",
			  usermap_filename,ret->hashmark_count));
	 
	 if (last_hashmark)
	     free_string(&last_hashmark);

     }

     fclose(F);

    if (buf)
	free(buf);

    if (ret) {
	usermap_files =
	    safe_array_realloc(usermap_files,
			       usermap_files_count+1,
			       sizeof (usermap_files[0]));
	
	usermap_files[usermap_files_count] = ret;
	inc_hashmark_usermap_refcount(usermap_files[usermap_files_count]);
	usermap_files_count++;
    }

    return ret;
}

static int ht_is_on_mapdir P_((const struct string * value));
static int ht_is_on_mapdir(value)
     const struct string * value;
{
    int loc = string_have_ascii_sep(value,cs2us("/$\\;{}()"));

    if (loc >= 0) {
	DPRINT(Debug,13,(&Debug, "ht_is_on_mapdir=0: found separator at #%d on %S\n",
			 loc,value));
	return 0;				 
    }

    DPRINT(Debug,13,(&Debug, "ht_is_on_mapdir=1: no separator on %S\n",
		     value));

    return 1;
}

static int hashtype_parse_kw_value_remote(filename,lineno,
					  rest_S,item,kw,
					  value,rc,now)
     const char *filename;
     int lineno;
     struct string * rest_S;
     struct hashmark_item *item;
     enum hashkeyword_v kw;
     struct string * value;
     enum record_mode rc;
     const struct schedule_timelimit * now;
{
    int ret = 0;

    struct hashmark_remote * item_r;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_parse_kw_value_remote",
	      "Bad magic (hashmark_item)",0);	

    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_parse_kw_value_remote",
	      "Bad magic (hashmark_remote)",0);	
    item_r = item->u.remote;

    switch (kw) {
       
    case hashkeyword_hostname:
    case hashkeyword_ipaddr:
	if (item_r->ip_literal_address.dummy ||
	    item_r->ip_literal_addrsize ||
	    item_r->hostname) {
	    if (hashmark_is_printableln(filename,lineno,rest_S))
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeDuplicateKwOnLine,
				  "%s: %d: Duplicate keyword %s on line: %S"),
			  filename,lineno,
			  hashmark_Keyword(kw),rest_S);


	    ret = 0;

	} else {
	    struct string          * unquoted_value = NULL;
	    int                      value_error    = 0;

	    if (item_r->service)
		free_service_entry(& (item_r->service));

	    unquoted_value = 
		unquote_string(value,hashmark_is_printableln,
			       filename,lineno,&value_error);

	    if (value_error) {
		if (hashmark_is_printableln(filename,lineno,rest_S))
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeKwdBADvalueOnLine,
				      "%s: %d: Keyword %s have bad value on line: %S"),
			      filename,lineno,
			      hashmark_Keyword(kw),rest_S);
	    }

	    if (unquoted_value) {
		item_r->hostname =
		    hashtype_ascii_param_value(filename,lineno,rest_S,kw,
					       unquoted_value);
		if (item_r->hostname) {
		    SOCKADDR ip_literal;
		    
		    if (get_ip(&ip_literal,item_r->hostname)) {
			
			set_address_from_gen_helper(& (item_r->ip_literal_addrsize),
						    & (item_r->ip_literal_address),
						    & ip_literal);
			
			ret = !value_error;
		    } else if (hashkeyword_ipaddr == kw) {
			if (hashmark_is_printableln(filename,lineno,rest_S))
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeInvalidIPaddressKw,
					      "%s: %d: Invalid IP-address on keyword %s value %S on line: %S"),
				      filename,lineno,
				      hashmark_Keyword(kw),
				      unquoted_value,
				      rest_S);
			
			ret = 0;
		    } else 
			ret = !value_error;	    
		} else
		    ret = 0;
	    
		free_string(& unquoted_value);
	    } else
		ret = 0;
	}
	break;
	
    case hashkeyword_servicetype: 

	if (item_r->servicetype) {
	    if (hashmark_is_printableln(filename,lineno,rest_S))
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeDuplicateKwOnLine,
				  "%s: %d: Duplicate keyword %s on line: %S"),
			  filename,lineno,
			  hashmark_Keyword(kw),rest_S);


	    ret = 0;

	} else {
	    struct string          * unquoted_value = NULL;
	    int                      value_error    = 0;
	    
	    if (item_r->service)
		free_service_entry(& (item_r->service));

	    unquoted_value = 
		unquote_string(value,hashmark_is_printableln,
			       filename,lineno,&value_error);

	    if (value_error) {
		if (hashmark_is_printableln(filename,lineno,rest_S))
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeKwdBADvalueOnLine,
				      "%s: %d: Keyword %s have bad value on line: %S"),
			      filename,lineno,
			      hashmark_Keyword(kw),rest_S);
	    }

	    if (unquoted_value) {
	
		char * ascii_value = 
		    hashtype_ascii_param_value(filename,lineno,rest_S,kw,value);

		if (ascii_value) {

		    item_r->servicetype = 
			service_type_from_name(ascii_value);

		    if (item_r->servicetype) {

			if (service_type_flags(item_r->servicetype,STFLAG_mbox)) 
			    ret = !value_error;
			else {
			    
			    if (hashmark_is_printableln(filename,lineno,rest_S))
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeSerTypeHashTypeOnLine,
						  "%s: %d: Service type %s is not valid for hashmark type %s on keyword %s value %S on line: %S"),
					  filename,lineno,
					  service_type_name(item_r->servicetype),
					  hashmark_Hashtype(item->hashtype),
					  hashmark_Keyword(kw),
					  ascii_value,
					  rest_S);
			    			    
			    item_r->servicetype = NULL;
			    ret = 0;
			}
		    } else
			ret = 0;

		    free(ascii_value);
		} else
		    ret = 0;

		free_string(&unquoted_value);
		
	    } else
		ret = 0;
	}
	    
	break;

    case hashkeyword_req_tls_peername: {
	int                      value_error    = 0;
	struct string         *  unquoted_value = 
	    unquote_string(value,hashmark_is_printableln,
			   filename,lineno,&value_error);

	if (value_error) {
	    if (hashmark_is_printableln(filename,lineno,rest_S))
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeKwdBADvalueOnLine,
				  "%s: %d: Keyword %s have bad value on line: %S"),
			  filename,lineno,
			  hashmark_Keyword(kw),rest_S);
	}

	if (unquoted_value) {

	    if (item_r->req_tls_peername_list) {
		int i;
		
		for (i = 0; i < item_r->req_tls_peername_count; i++) {
		    if (0 == string_cmp(item_r->req_tls_peername_list[i],
					unquoted_value,
					99)) {
			if (hashmark_is_printableln(filename,lineno,rest_S))
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeDuplicateKwOnLine,
					      "%s: %d: Duplicate keyword %s on line: %S"),
				      filename,lineno,
				      hashmark_Keyword(kw),rest_S);
			ret = 0;
			free_string(& unquoted_value);
			goto fail;
		    }
		}
	    }
	    
	    item_r->req_tls_peername_list
		= safe_array_realloc(item_r->req_tls_peername_list,
				     item_r->req_tls_peername_count+1,
				     sizeof(item_r->req_tls_peername_list[0]));
	    
	    item_r->
		req_tls_peername_list[item_r->
				      req_tls_peername_count++] = 
		unquoted_value; unquoted_value = NULL;
	    ret = !value_error;

	} else
	    ret = 0;
    }
	break;
	
    case hashkeyword_userid:

	if (item_r->remote_userid) {
	    if (hashmark_is_printableln(filename,lineno,rest_S))
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeDuplicateKwOnLine,
				  "%s: %d: Duplicate keyword %s on line: %S"),
			  filename,lineno,
			  hashmark_Keyword(kw),rest_S);
	    
	    
	    ret = 0;
	    break;
	}

	switch (rc) {
		     
	case LOCAL_RC: {
	    int                      value_error    = 0;
	    struct string         *  unquoted_value = 
		unquote_string(value,hashmark_is_printableln,
			       filename,lineno,&value_error);

	    if (value_error) {
		if (hashmark_is_printableln(filename,lineno,rest_S))
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeKwdBADvalueOnLine,
				      "%s: %d: Keyword %s have bad value on line: %S"),
			      filename,lineno,
			      hashmark_Keyword(kw),rest_S);
	    }

	    if (unquoted_value) {
		item_r->remote_userid =
		    hashtype_ascii_param_value(filename,lineno,rest_S,kw,
					       unquoted_value);
		if (item_r->remote_userid)
		    ret = !value_error;
		else
		    ret = 0;
		free_string(& unquoted_value);
	    } else 
		ret = 0;
	}
	    
	    break;
	case SYSTEM_RC:
	case RC_MODE_COUNT:
	    goto not_local;
	}
	break;

    case hashkeyword_fromaddr:
	if (item_r->from_address) {
	    if (hashmark_is_printableln(filename,lineno,rest_S))
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeDuplicateKwOnLine,
				  "%s: %d: Duplicate keyword %s on line: %S"),
			  filename,lineno,
			  hashmark_Keyword(kw),rest_S);
	    	    
	    ret = 0;
	    break;
	}

	switch (rc) {
	    int parse_error;
	    
	case LOCAL_RC:
	    parse_error = 0;

	    /* Do not report about parse errors */
	    item_r->from_address = 
		parse_one_string_address(value,
					 hashmark_is_printableln,
					 filename,lineno,&parse_error);
	    if (parse_error) {
		if (hashmark_is_printableln(filename,lineno,rest_S))
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeParseErrAddrGivenKw,
				      "%s: %d: Failed to parse address given on keyword %s on line: %S"),
			      filename,lineno,
			      hashmark_Keyword(kw),rest_S);
		ret = 0;
	    } else if (item_r->from_address)
		ret = 1;
	    else {
		if (hashmark_is_printableln(filename,lineno,rest_S))
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeNoAddrGivenKw,
				      "%s: %d: No address given on keyword %s on line: %S"),
			      filename,lineno,
			      hashmark_Keyword(kw),rest_S);
		ret = 0;
	    }

	    break;
	    
	case SYSTEM_RC:
	case RC_MODE_COUNT:
	not_local:
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeHTypeSupUserFileKwOnLIne,
			      "%s: %d: Hashtype %s supports keyword %s on user's %s -file; not on line %S"),
		      filename,lineno,
		      hashmark_Hashtype(item->hashtype),
		      hashmark_Keyword(kw),
		      USER_HASHMARK_FILE,
		      rest_S); 
	    ret = 0;
	    break;
	}   
	break;

    case hashkeyword_usermap:

	switch (rc) {
	    struct string_token * tokenized;

	case SYSTEM_RC:
	    
	    tokenized = string_tokenize(value,TOK_mail);

	    if (tokenized) {
	    
		ret = hashmark_usermap_parse_rest(filename,lineno,
						  & (item_r->usermaps),
						  & (item_r->usermap_count),
						  & (item_r->usermap_alloc_count),
						  tokenized,0);
	    
		if (!ret) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeUserMapNotValidOnLine,
				      "%s: %d: User mapping not valid on keyword %s value %S on line: %S"),
			      filename,lineno,
			      hashmark_Keyword(kw),value,
			      rest_S);
		}

		free_string_tokenized(&tokenized);
				
	    } else {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeUserMapNotValidOnLine,
				  "%s: %d: User mapping not valid on keyword %s value %S on line: %S"),
			  filename,lineno,
			  hashmark_Keyword(kw),value,
			  rest_S);

		ret = 0;
	    }

	    break;
	    
	case LOCAL_RC:
	case RC_MODE_COUNT:
	    goto not_remote;
	}				  

	break;

    case hashkeyword_usermapfile:
	if (item_r->usermap_file ||
	    item_r->usermap_file_value) {
	    if (hashmark_is_printableln(filename,lineno,rest_S))
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeDuplicateKwOnLine,
				  "%s: %d: Duplicate keyword %s on line: %S"),
			  filename,lineno,
			  hashmark_Keyword(kw),rest_S);
	    
	    
	    ret = 0;
	    break;
	}

	switch (rc) {
	case SYSTEM_RC: {
	    int                      value_error    = 0;
	    struct string         *  unquoted_value = 
		unquote_string(value,hashmark_is_printableln,
			       filename,lineno,&value_error);

	    if (value_error) {
		if (hashmark_is_printableln(filename,lineno,rest_S))
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeKwdBADvalueOnLine,
				      "%s: %d: Keyword %s have bad value on line: %S"),
			      filename,lineno,
			      hashmark_Keyword(kw),rest_S);
	    }

	    if (unquoted_value) {	    

		if (ht_is_on_mapdir(unquoted_value)) {
		    /* give_dt_estr_as_str adds / to end */
		    const char * user_map_dir_val = 
			give_dt_estr_as_str(&user_map_dir_e,
					    "user-map-dir",
					    NULL,NULL);
		    
		    char * sysvalue =  
			hashtype_cs_param_value(filename,lineno,
						rest_S,kw,
						unquoted_value,
						system_charset,
						"system charset");
		    
		    if (user_map_dir_val && 
			0 != access(user_map_dir_val,ACCESS_EXISTS)) {
			int err = errno;
			
			if (hashmark_is_printableln(filename,lineno,unquoted_value))
			    lib_error(CATGETS(elm_msg_cat, MeSet,MeNoUserMapDir,
					      "Map %S: no user-map-dir exists: %s: %s"),
				      value,user_map_dir_val,strerror(err));
			
			ret = 0;
			
		    } else if (sysvalue && user_map_dir_val) {
			
			char * longname =
			    elm_message(FRM("%s%s"),user_map_dir_val,sysvalue);
			
			item_r->usermap_file = 
			    locate_hashmark_usermap_file(longname);
			
			if (! item_r->usermap_file) {
			    
			    item_r->usermap_file
				= load_hashmark_usermap_file(filename,lineno,
							     rest_S,kw,value,
							     longname,
							     sysvalue);
			}
			
			if (item_r->usermap_file) {
			    item_r->usermap_file_value = unquoted_value;
			    unquoted_value = NULL;
			    ret = !value_error;
			} else
			    ret = 0;
			
			free(longname);
		    } else {
			ret = 0;
		    }
		    
		    if (sysvalue)		    
			free(sysvalue);
		    
		} else { /* !ht_is_on_mapdir(unquoted_value) */
		    char * sysvalue =  hashtype_cs_param_value(filename,lineno,
							       rest_S,kw,unquoted_value,
							       system_charset,
							       "system charset");		   
		    if (sysvalue) {
			char buffer[STRING];
			
			/* -1 == error */
			if (0 <= expand_meta(buffer,sysvalue,sizeof buffer)) {
			    if ('/' == buffer[0]) {			    
				item_r->usermap_file = 
				    locate_hashmark_usermap_file(buffer);
				
				if (! item_r->usermap_file) {
				    item_r->usermap_file
					= load_hashmark_usermap_file(filename,lineno,
								     rest_S,kw,
								     unquoted_value,
								     buffer,
								     NULL);
				}
				
				if (item_r->usermap_file) {
				    item_r->usermap_file_value = unquoted_value;
				    unquoted_value = NULL;
				    ret = !value_error;
				} else
				    ret = 0;
				
			    } else {
				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MePathNotAbsExpKwValueOnLine,
						      "%s: %d: Path \"%s\" is not absolute expanded from keyword %s value %S on line: %S"),
					      filename,lineno,
					      buffer,
					      hashmark_Keyword(kw),
					      unquoted_value,
					      rest_S);
				ret = 0;
			    }
			    
			} else {
			    if (hashmark_is_printableln(filename,lineno,rest_S))
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeFailedExpandKwValueOnLine,
						  "%s: %d: Failed to expand keyword %s value %S on line: %S"),
					  filename,lineno,
					  hashmark_Keyword(kw),unquoted_value,
					  rest_S);
			    ret = 0;
			    
			}
			
			free(sysvalue);					
		    } else {
			ret = 0;
		    }
		
		}

		if (unquoted_value)
		    free_string(& unquoted_value);
	    }
	}
	
	    break;
	case LOCAL_RC:
	case RC_MODE_COUNT:
	not_remote:
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeHTypeSupSysFileKwOnLIne,
			      "%s: %d: Hashtype %s supports keyword %s on system's %s -file; not on line %S"),
		      filename,lineno,
		      hashmark_Hashtype(item->hashtype),
		      hashmark_Keyword(kw),
		      SYSTEM_HASHMARK_FILE,
		      rest_S); 
	    ret = 0;
	    break;
	}
	break;

    default:

	if (hashmark_is_printableln(filename,lineno,rest_S))
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeHTypeDontSupoKwOnLIne,
			      "%s: %d: Hashtype %s do not support keyword %s on line %S"),
		      filename,lineno,
		      hashmark_Hashtype(item->hashtype),
		      hashmark_Keyword(kw),
		      rest_S);
	ret = 0;
	break;
    }    

 fail:


    return ret;
}

static int hashtype_parse_kw_remote(filename,lineno,rest_S,item,kw,rc,now)
     const char *filename;
     int lineno;
     struct string * rest_S;
     struct hashmark_item *item;
     enum hashkeyword_v kw;
     enum record_mode rc;
     const struct schedule_timelimit * now;
{
    int ret = 0;
    struct hashmark_remote * item_r;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_parse_kw_remote",
	      "Bad magic (hashmark_item)",0);	
    
    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_parse_kw_remote",
	      "Bad magic (hashmark_remote)",0);	
    item_r = item->u.remote;
    

   switch (kw) {
   case hashkeyword_ver_tls_cert:

       if (item_r->verify_tls_certificate) {
	   if (hashmark_is_printableln(filename,lineno,rest_S))
	       lib_error(CATGETS(elm_msg_cat, MeSet, 
				 MeDuplicateKwOnLine,
				 "%s: %d: Duplicate keyword %s on line: %S"),
			 filename,lineno,
			 hashmark_Keyword(kw),rest_S);
	   ret = 0;
       } else {
	   item_r->verify_tls_certificate = 1;
	   ret = 1;
       }
       break;

   case hashkeyword_req_tls_peername:

       if (item_r->require_tls_peer_name) {
	   if (hashmark_is_printableln(filename,lineno,rest_S))
	       lib_error(CATGETS(elm_msg_cat, MeSet, 
				 MeDuplicateKwOnLine,
				 "%s: %d: Duplicate keyword %s on line: %S"),
			 filename,lineno,
			 hashmark_Keyword(kw),rest_S);

	   ret = 0;
       } else {
	   item_r->require_tls_peer_name = 1;
	   ret = 1;
       }
       break;
   case hashkeyword_self_cc:
       if (item_r->self_cc) {
	   if (hashmark_is_printableln(filename,lineno,rest_S))
	       lib_error(CATGETS(elm_msg_cat, MeSet, 
				 MeDuplicateKwOnLine,
				 "%s: %d: Duplicate keyword %s on line: %S"),
			 filename,lineno,
			 hashmark_Keyword(kw),rest_S);

	   ret = 0;
       } else {
	   item_r->self_cc = 1;
	   ret = 1;
       }
       break;

   case hashkeyword_self_bcc:
       if (item_r->self_bcc) {
	   if (hashmark_is_printableln(filename,lineno,rest_S))
	       lib_error(CATGETS(elm_msg_cat, MeSet, 
				 MeDuplicateKwOnLine,
				 "%s: %d: Duplicate keyword %s on line: %S"),
			 filename,lineno,
			 hashmark_Keyword(kw),rest_S);

	   ret = 0;
       } else {
	   item_r->self_bcc = 1;
	   ret = 1;
       }
       break;

       
   default:
       if (hashmark_is_printableln(filename,lineno,rest_S))
	   lib_error(CATGETS(elm_msg_cat, MeSet, 
			     MeHTypeDontSupoKwOnLIne,
			     "%s: %d: Hashtype %s do not support keyword %s on line %S"),
		     filename,lineno,
		     hashmark_Hashtype(item->hashtype),
		     hashmark_Keyword(kw),
		     rest_S);
       break;
   }

    return ret;
}

static void hashtype_add_tls_peername P_((struct hashmark_remote * item_r,
					  const struct string * peername));
static void hashtype_add_tls_peername(item_r,peername)
     struct hashmark_remote * item_r;
     const struct string * peername;
{
    int j;

    if (HASHMARK_REMOTE_magic != item_r->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_add_tls_peername",
	      "Bad magic (hashmark_remote)",0);	

    for (j = 0; j < item_r->req_tls_peername_count; j++)  {
	if (0 == string_cmp(item_r->req_tls_peername_list[j],
			    peername,
			    99))
	    return;
    }
    
    item_r->req_tls_peername_list =
	safe_array_realloc(item_r->req_tls_peername_list,
			   item_r->req_tls_peername_count+1,
			   sizeof (item_r->req_tls_peername_list[0]));

    item_r->req_tls_peername_list[item_r->req_tls_peername_count++] =
	dup_string(peername);
}
				     

static int hashtype_parse_check_remote(filename,lineno,item,rc,now)
     const char *filename;
     int lineno;
     struct hashmark_item *item;
     enum record_mode rc;
     const struct schedule_timelimit  * now;
{
    int ret = 1;
    struct hashmark_remote * item_r;

     if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_parse_check_remote",
	      "Bad magic (hashmark_item)",0);	

    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_parse_check_remote",
	      "Bad magic (hashmark_remote)",0);	
    item_r = item->u.remote;

    if (item_r->hostname) {

	if (item_r->service)
	    free_service_entry(& (item_r->service));

	switch (rc) {
		    
	case LOCAL_RC:
	    
	    if (user_mail_services_conf) {
			/* Increments refcount */
		item_r->service
		    = scan_mail_services_conf(user_mail_services_conf,
					      item_r->hostname,
					      STFLAG_mbox,itls_ignore,now,
					      item_r->ip_literal_address.
					      generic,
					      item_r->servicetype);
		
		if (item_r->service) 		    
		    break;
		
	    }
	    
	    /* FALLTHRU */
	case SYSTEM_RC:
	    
	    if (system_mail_services_conf) {
		/* Increments refcount */
		item_r->service
		    = scan_mail_services_conf(system_mail_services_conf,
					      item_r->hostname,
					      STFLAG_mbox,itls_ignore,now,
					      item_r->ip_literal_address.
					      generic,
					      item_r->servicetype);
		
		if (item_r->service) 
		    break;
	    }

	    /* FALLTHRU */
	case RC_MODE_COUNT:	    
	    break;
	}

    } else {

	item_r->hostname =
	    safe_strdup(hostfullname);

	lib_error(CATGETS(elm_msg_cat, MeSet, 
			  MeRequireAorBsetting,
			  "%s: %d: Hashtype %s requires keyword %s or %s; setting %s to %s."),
		  filename,lineno,
		  hashmark_Hashtype(item->hashtype),
		  hashmark_Keyword(hashkeyword_hostname),
		  hashmark_Keyword(hashkeyword_ipaddr),
		  hashmark_Keyword(hashkeyword_hostname),
		  item_r->hostname);
	
	ret = 0;		       
    }


    switch (rc) {
		    
    case LOCAL_RC:

	if (! item_r->remote_userid) {

	    item_r->remote_userid =
		safe_strdup(username);

	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeRequireAonUsers,
			      "%s: %d: Hashtype %s requires keyword %s on user's %s -file; setting %s to %s."),
		      filename,lineno,
		      hashmark_Hashtype(item->hashtype),
		      hashmark_Keyword(hashkeyword_userid),
		      USER_HASHMARK_FILE,
		      hashmark_Keyword(hashkeyword_userid),
		      item_r->remote_userid);
	    
	    ret = 0;
	}

	break;

    case SYSTEM_RC:
    case RC_MODE_COUNT:	    
	    break;
	}	

    return ret;
}



static void dump_fileset_kw_helper P_((FILE               * F,
				     enum hashkeyword_v   kw,
				     const char         * val));

static void dump_fileset_kw_helper(F,kw,val)
     FILE               * F;
     enum hashkeyword_v   kw;
     const char         * val;
{
    fputs(hashmark_Keyword(kw),F); putc(':', F); putc(' ', F);	    

    if (NULL != strpbrk(val,"\\\" \t")) {
	const char *x;
	putc('"', F);
	
	for (x = val; *x; x++) {
	    switch (*x) {
	    case '"':
	    case '\\':
		putc('\\', F);
		
		/* FALLTHRU */
	    default:
		putc(*x, F);
		break;
	    }
	}
	
	putc('"', F);
    } else
	fputs(val,F); 
    
    putc('\n', F);
}

static int hashtype_dump_kw_remote(F,item,kw,
				 fileset,sysnam,rc)
     FILE                 * F;
     struct hashmark_item * item;
     enum hashkeyword_v     kw;
     charset_t              fileset;
     const char           * sysnam;
     enum record_mode       rc;
{
    int fail = 0;
    struct hashmark_remote * item_r;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_dump_kw_remote",
              "Bad magic number",0);

    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_dump_kw_remote",
	      "Bad magic (hashmark_remote)",0);	
    item_r = item->u.remote;

    switch (kw) {
    case hashkeyword_ipaddr:
	if (item_r->ip_literal_address.dummy &&
	    item_r->ip_literal_addrsize) {
	    
	    SOCKADDR X;

	    if (set_SOCKADDR_from_ptr_helper(& X,
					     item_r->ip_literal_addrsize,
					     item_r->ip_literal_address)) {

		char mybuffer[256];
		const char *addr_string = 
		    give_addr_as_string(&X,mybuffer,sizeof mybuffer);

		if (addr_string) {

		    fputs(hashmark_Keyword(kw),F); putc(':', F); putc(' ', F);
		    fputs(addr_string,F); putc('\n', F);

		} else
		    goto failaddr;

	    } else {
	    failaddr:
		
		if (item_r->hostname) 
		    goto dumphostname;

	    }
	}
	break;

    case hashkeyword_hostname:

	if (item_r->hostname &&
	    ! item_r->ip_literal_address.dummy &&
	    ! item_r->ip_literal_addrsize) {
	    int err;
	    char *val;

	dumphostname:

	    err = 0;
	    val = convert_to_fileset(item_r->hostname,
				     fileset,&err);
	    
	    if (err) {
		DPRINT(Debug,8,(&Debug, 
				"hashtype_dump_kw_remote: Failed to convert hashmark keyword %s value \"%s\" to %s, %d errors\n",
				hashmark_Keyword(kw),
				item_r->hostname,
				sysnam ? sysnam : "<no NIME name>",
				err));   
		
		fail = 1;
	    }
	    
	    if (!val)
		break;

	    dump_fileset_kw_helper(F,kw,val);
	    
	    if (val != item_r->hostname)
		free(val);
	    
	}	
	break;

    case hashkeyword_servicetype:
	
	if (item_r->servicetype) 
	    dump_fileset_kw_helper(F,kw,service_type_name(item_r->servicetype));
	
	break;

    case hashkeyword_ver_tls_cert:

	if (item_r->verify_tls_certificate) {
	    fputs(hashmark_Keyword(kw),F);  putc('\n', F);
	}

	break;
	
    case hashkeyword_req_tls_peername: 
	
	if (item_r->require_tls_peer_name) {
	    fputs(hashmark_Keyword(kw),F);  putc('\n', F);
	}

	if (item_r->req_tls_peername_list) {
	    int i;

	    for (i = 0; i < item_r->req_tls_peername_count; i++) {
		if (! dump_string_quote_helper(F,fileset,sysnam,kw,
					       item_r->
					       req_tls_peername_list[i],-1))  {
		    fail = 1;
		}
	    }
	}
	
	break;
	   
    case hashkeyword_userid:

	if (item_r->remote_userid) {
	     int err;
	     char *val;

	     err = 0;  
	     val = convert_to_fileset(item_r->remote_userid,
				      fileset,&err);
	     
	     if (err) {
		 DPRINT(Debug,8,(&Debug, 
				 "hashtype_dump_kw_remote: Failed to convert hashmark keyword %s value \"%s\" to %s, %d errors\n",
				 hashmark_Keyword(kw),
				 item_r->remote_userid,
				 sysnam ? sysnam : "<no NIME name>",
				 err));   
		 
		 fail = 1;
	     }
	     
	     if (!val)
		 break;
	     
	     dump_fileset_kw_helper(F,kw,val);
	     
	     if (val != item_r->remote_userid)
		 free(val);
	     
	}

	break;

    case hashkeyword_fromaddr:

	if (item_r->from_address) {
	    struct string * from_address = 
		address_to_string(item_r->from_address);

	    if (from_address) {
		int err = 0;

		char *value = convert_to_fileset2(from_address,fileset,&err);
		
		if (err) {
		    DPRINT(Debug,8,(&Debug, 
				    "hashtype_dump_kw_remote: Failed to convert hashmark keyword %s value \"%S\" to %s, %d errors\n",
				    hashmark_Keyword(kw),
				    from_address,
				    sysnam ? sysnam : "<no NIME name>",
				    err));   
		    
		    fail = 1;
		}
		
		fputs(hashmark_Keyword(kw),F); putc(':', F); putc(' ', F);
		fputs(value,F); putc('\n', F);
		
		free(value);
		
		free_string(&from_address);
	    }
	}

	break;


    case hashkeyword_usermap:

	if (item_r->usermaps) {
	    int i;

	    for (i = 0; i < item_r->usermap_count; i++) {

		struct string * S = 
		    hashmark_usermap_entry_string(& (item_r->usermaps[i]),
						  fileset);

		int err = 0;
		char *value = convert_to_fileset2(S,fileset,&err);
		
		if (err) {
		    DPRINT(Debug,8,(&Debug, 
				    "hashtype_dump_kw_remote: Failed to convert hashmark keyword %s value \"%S\" to %s, %d errors\n",
				    hashmark_Keyword(kw),
				    S,
				    sysnam ? sysnam : "<no NIME name>",
				    err));   
		    
		    fail = 1;
		}

		fputs(hashmark_Keyword(kw),F); putc(':', F); putc(' ', F);
		fputs(value,F); putc('\n', F);
		
		free(value);
		
		free_string(&S);
	    }
	}

	break;

    case hashkeyword_usermapfile:
	if (item_r->usermap_file_value) {

	    if (! dump_string_quote_helper(F,fileset,sysnam,kw,
					   item_r->usermap_file_value,-1)) {
		fail = 1;
	    }
	}

	break;

    case  hashkeyword_self_cc:

	if (item_r->self_cc) {
	    fputs(hashmark_Keyword(kw),F);  putc('\n', F);
	}

	break;

    case  hashkeyword_self_bcc:

	if (item_r->self_bcc) {
	    fputs(hashmark_Keyword(kw),F);  putc('\n', F);
	}

	break;


    default:
	DPRINT(Debug,8,(&Debug, 
			"hashtype_dump_kw_remote: keyword %s not used.\n",
			hashmark_Keyword(kw)));
	

	break;
       
    }

    return !fail;
}


static void hashtype_merge_remote(item,base,overwrite)
     struct hashmark_item *item;
     const struct hashmark_item * base;
     const struct hashmark_item * overwrite;
{
    int from_base = 0;
    struct hashmark_remote * item_r;
    struct hashmark_remote * base_r;
    struct hashmark_remote * overwrite_r;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_remote",
              "Bad magic number (item)",0);

    if (HASHMARK_ITEM_magic != base->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_remote",
              "Bad magic number (base)",0);

    if (HASHMARK_ITEM_magic != overwrite->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_remote",
              "Bad magic number (overwrite)",0);

    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_remote",
	      "Bad magic (hashmark_remote, item)",0);
    item_r = item->u.remote;	

    if (HASHMARK_REMOTE_magic != base->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_remote",
	      "Bad magic (hashmark_remote, base)",0);	
    base_r = base->u.remote;

    if (HASHMARK_REMOTE_magic != overwrite->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_remote",
	      "Bad magic (hashmark_remote, overwrite)",0);	
    overwrite_r = overwrite->u.remote;
    
    if (overwrite_r->hostname) {

	item_r->hostname =
	    safe_strdup(overwrite_r->hostname);

	if (overwrite_r->ip_literal_address.dummy &&
	    overwrite_r->ip_literal_addrsize)
	    set_SOCKADDR_ptr_helper(& (item_r->ip_literal_addrsize),
				    & (item_r->ip_literal_address),
				    overwrite_r->ip_literal_addrsize,
				    overwrite_r->ip_literal_address);
	
	if (overwrite_r->service) {
	    item_r->service =
		overwrite_r->service;
	    inc_service_entry_refcount(item_r->service);
	}
		
    } else if (base_r->hostname) {

	item_r->hostname =
	    safe_strdup(base_r->hostname);

	if (base_r->ip_literal_address.dummy &&
	    base_r->ip_literal_addrsize)
	    set_SOCKADDR_ptr_helper(& (item_r->ip_literal_addrsize),
				    & (item_r->ip_literal_address),
				    base_r->ip_literal_addrsize,
				    base_r->ip_literal_address);


	from_base = 1;
		          	
    }

    if (overwrite_r->servicetype) {
	item_r->servicetype =
	    overwrite_r->servicetype;
	from_base = 0;

    } else if (from_base &&
	       base_r->servicetype)
	item_r->servicetype =
	    base_r->servicetype;

    if (from_base &&
	base_r->service) {
	item_r->service =
	    base_r->service;
	inc_service_entry_refcount(item_r->service);
    }


    if (base_r->req_tls_peername_count < 0 ||
	overwrite_r->req_tls_peername_count < 0)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_remote",
	      "Bad req_tls_peername_count",0); 
    
    if (base_r->req_tls_peername_count > 0 ||
	overwrite_r->req_tls_peername_count > 0) {
	
	int maxlen =
	    base_r->req_tls_peername_count +
	    overwrite_r->req_tls_peername_count;
	int idx = 0,i;

	int include_base = 
	    item_r->hostname &&
	    base_r->hostname &&
	    0 == istrcmp(item_r->hostname,
			 base_r->hostname);

	item_r->req_tls_peername_list
	    = safe_calloc(maxlen,
			  sizeof(item_r->req_tls_peername_list[0]));

	for (i = 0; i < maxlen; i++)
	    item_r->req_tls_peername_list[i] = NULL;
	
	if (include_base) {
	    
	    for (idx = 0; idx < base_r->req_tls_peername_count; idx++)
		item_r->req_tls_peername_list[idx] =
		    dup_string(base_r->req_tls_peername_list[idx]);
	}
		
	for (i = 0; 
	     i < overwrite_r->req_tls_peername_count &&
		 idx < maxlen; i++) {

	    if (include_base) {
		int j;

		for (j = 0; j < base_r->req_tls_peername_count; j++)  {
		    if (0 == string_cmp(item_r->req_tls_peername_list[j],
					overwrite_r->req_tls_peername_list[i],
					99))
			goto foundname;
		}
	    }		

	    item_r->req_tls_peername_list[idx++] =
		dup_string(overwrite_r->req_tls_peername_list[i]);
	    
	foundname:;
	}
	item_r->req_tls_peername_count = idx;
    }

    /* LOCAL_RC */

    if (overwrite_r->remote_userid) {
	item_r->remote_userid
	    = safe_strdup(overwrite_r->remote_userid);
    } else if (base_r->remote_userid) {
	item_r->remote_userid
	    = safe_strdup(overwrite_r->remote_userid);
    }	    

    if (overwrite_r->from_address) {
	item_r->from_address =
	    dup_address(overwrite_r->from_address);
    } else if (overwrite_r->from_address) {
	item_r->from_address =
	    dup_address(overwrite_r->from_address);
    }

    /* SYSTEM_RC */

    if (base_r->usermap_count < 0 ||
	overwrite_r->usermap_count < 0)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_remote",
	      "Bad usermap_count",0); 


    if (base_r->usermap_count > 0 ||
	overwrite_r->usermap_count > 0) {
	int maxlen = base_r->usermap_count +
	    overwrite_r->usermap_count;
	int idx = 0,i;

	item_r->usermaps = 
	    safe_calloc(maxlen,
			sizeof(item_r->usermaps[0]));

	for ( i = 0; i < maxlen; i++) {
	    item_r->usermaps[i].local_username = NULL;
	    item_r->usermaps[i].remote_userid  = NULL;
	    item_r->usermaps[i].from_address   = NULL;
	}
	item_r->usermap_alloc_count = maxlen;

	for (idx = 0; idx < base_r->usermap_count; idx++) {

	    item_r->usermaps[idx].local_username =
		safe_strdup(base_r->usermaps[idx].local_username);
	    
	    item_r->usermaps[idx].remote_userid =
		base_r->usermaps[idx].remote_userid ?
		safe_strdup(base_r->usermaps[idx].remote_userid) :
		NULL;
	    item_r->usermaps[idx].from_address =
		base_r->usermaps[idx].from_address ?
		dup_address(base_r->usermaps[idx].from_address) :
		NULL;
	}

	for (i = 0; 
	     i < overwrite_r->usermap_count &&
		 idx < maxlen; i++) {

	    int j;
	    struct hashmark_usermap_entry * found = NULL;

	    for (j = 0; j < base_r->usermap_count;j++) {
		if (0 == strcmp(item_r->usermaps[j].local_username,
				overwrite_r->usermaps[i].local_username)) {
		    found = & ( item_r->usermaps[j] );
		    goto founduser;
		}
	    }

	    item_r->usermaps[idx].local_username = 
		safe_strdup(overwrite_r->usermaps[i].local_username);
	    item_r->usermaps[idx].remote_userid = NULL;
	    item_r->usermaps[idx].from_address  = NULL;
	    found = & (item_r->usermaps[idx++] );

	founduser:
	    if (found->remote_userid)
		free(found->remote_userid);
	    found->remote_userid =
		overwrite_r->usermaps[i].remote_userid ?
		safe_strdup(overwrite_r->usermaps[i].remote_userid) :
		NULL;

	    if (found->from_address)
		free_address(& (found->from_address));
	    found->from_address =
		overwrite_r->usermaps[i].from_address ?
		dup_address(overwrite_r->usermaps[i].from_address) :
		NULL;		
	}

	item_r->usermap_count = idx;
    }

    if (overwrite_r->usermap_file_value) {

	item_r->usermap_file_value =
	    dup_string(overwrite_r->usermap_file_value);

	if (overwrite_r->usermap_file) {
	    item_r->usermap_file = overwrite_r->usermap_file;
	    inc_hashmark_usermap_refcount(item_r->usermap_file);
	}

    } else if (base_r->usermap_file_value) {

	item_r->usermap_file_value =
	    dup_string(base_r->usermap_file_value);

	if (base_r->usermap_file) {
	    item_r->usermap_file = base_r->usermap_file;
	    inc_hashmark_usermap_refcount(item_r->usermap_file);
	}

    }

    /* flags */

    if (overwrite_r->verify_tls_certificate ||
	base_r->verify_tls_certificate)
	item_r->verify_tls_certificate = 1;

    if (overwrite_r->require_tls_peer_name) {

	if (overwrite_r->hostname &&
	    item_r->hostname &&
	    0 == strcmp(overwrite_r->hostname,
			item_r->hostname))
	    item_r->require_tls_peer_name = 1;
	else if (overwrite_r->hostname) {
	    struct string * peer = new_string2(system_charset,
					       s2us(overwrite_r->hostname));

	    hashtype_add_tls_peername(item_r,peer);
	    free_string(&peer);

	} else
	    item_r->require_tls_peer_name = 1; 

	if (overwrite_r->service) {
	    if (SERVICE_ENTRY_magic != overwrite_r->service->magic)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "hashtype_merge_remote",
		      "Bad magic (service_entry, overwrite)",0);	

	    if (overwrite_r->service == item_r->service) 
		item_r->require_tls_peer_name = 1; 
	    else {
		struct string * peer = 
		    new_string2(system_charset,
				s2us(overwrite_r->service->official_name));

		hashtype_add_tls_peername(item_r,peer);
		free_string(&peer);
	    }
	}
    }

    if (base_r->require_tls_peer_name) {

	if (base_r->hostname &&
	    item_r->hostname &&
	    0 == strcmp(base_r->hostname,
			item_r->hostname))
	    item_r->require_tls_peer_name = 1;
	else if (base_r->hostname) {
	    struct string * peer = new_string2(system_charset,
					       s2us(base_r->hostname));

	    hashtype_add_tls_peername(item_r,peer);
	    free_string(&peer);

	} else
	    item_r->require_tls_peer_name = 1; 

	if (base_r->service) {
	    if (SERVICE_ENTRY_magic != base_r->service->magic)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "hashtype_merge_remote",
		      "Bad magic (service_entry, base)",0);	
	    
	    if (base_r->service == item_r->service) 
		item_r->require_tls_peer_name = 1; 
	    else {
		struct string * peer = 
		    new_string2(system_charset,
				s2us(base_r->service->official_name));
		
		hashtype_add_tls_peername(item_r,peer);
		free_string(&peer);
	    }
	}
    }

    if (overwrite_r->self_cc ||
	base_r->self_cc)
	item_r->self_cc = 1;

    if (overwrite_r->self_bcc ||
	base_r->self_bcc)
	item_r->self_bcc = 1;
    
}

static int  hashtype_brflags_remote(item) 
     const struct hashmark_item *item;
{
    int f = 0;

    struct hashmark_remote * item_r;
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_brflags_remote",
              "Bad magic number",0);

    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_brflags_remote",
	      "Bad magic (hashmark_remote, item)",0);
    item_r = item->u.remote;	

    if (item_r->servicetype) {
	int tflag =  service_type_flags(item_r->servicetype,0);
					
	if (0 == (tflag & STFLAG_mbox))
	    f |= BROWSER_NOFOLDER;

	if (0 == (tflag & STFLAG_browser))
	    f |= BROWSER_NODIR;
    }

    return f;
}

#define HASHMARK_RDATA_magic	0xF521

struct hashmark_rdata {
    unsigned short magic;       /* HASHMARK_RDATA_magic */

    int                     refcount;

    enum hmcon                have_con;  /* defined in mbx_imp.h */

    struct connection_cache * CX;
    struct remote_account   * ra;
    struct address          * from_address;

    enum hmcert               need_cert_check; /* defined in mbx_imp.h */
};

static void hashtype_initd_remote(item,hashmark_data)
     const struct hashmark_item *item;
     union hashmark_data      * hashmark_data;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_initd_remote",
              "Bad magic number (hashmark_item)",0);
    
    if (hashmark_data->dummy) {
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_initd_remote",
              "hassmark_data is set",0);
    }

    hashmark_data->remote =  safe_zero_alloc(sizeof(* (hashmark_data->remote)));
    hashmark_data->remote->magic = HASHMARK_RDATA_magic;
    hashmark_data->remote->refcount = 1;
    hashmark_data->remote->CX    = NULL;
    hashmark_data->remote->ra    = NULL;
    hashmark_data->remote->from_address = NULL;
    hashmark_data->remote->have_con = hmcon_none;
    hashmark_data->remote->need_cert_check = hmcert_needcheck;

}


static void inc_hashtype_rdata_refcount P_((struct hashmark_rdata  * rdata));
static void inc_hashtype_rdata_refcount(rdata)
     struct hashmark_rdata  * rdata;
{
    if (HASHMARK_RDATA_magic !=  rdata->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "inc_hashtype_rdata_refcount",
              "Bad magic number (hashmark_rdata)",0);

    rdata->refcount++;
}

static void  hashtype_free_rdata P_((struct hashmark_rdata  ** rdata));
static void  hashtype_free_rdata(rdata)
     struct hashmark_rdata  ** rdata;
{
    if (HASHMARK_RDATA_magic !=  (*rdata)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_free_rdata",
              "Bad magic number (hashmark_rdata)",0);

    (*rdata)->refcount--;

    if ((*rdata)->refcount > 0) {/* Just remove this reference */
        
        *rdata = NULL;

        return;
    }

    if ((*rdata)->CX) {
	int r = free_connection(& ((*rdata)->CX),NULL);

	if (!r) {
	    DPRINT(Debug,13,(&Debug, "hashtype_free_rdata: free_connection failed\n"));
	}	
    }
	
    if ((*rdata)->ra)
	free_remote_account(& ((*rdata)->ra));

    if ((*rdata)->from_address)
	free_address(&  ((*rdata)->from_address));

    (*rdata)->magic = 0;  /* invalidate */
    free(*rdata);
    *rdata = NULL;
}





static void hashtype_freed_remote(item,hashmark_data)
     const struct hashmark_item *item;
     union hashmark_data      * hashmark_data;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_freed_remote",
              "Bad magic number (hashmark_item)",0);


    hashtype_free_rdata(& (hashmark_data->remote));
}

/*  hmcert_needcheck (0) == not verified, not encrypted */
static enum hmcert hmremote_verify_cert_1 P_((const struct hashmark_remote * item_r,
					      struct remote_account        * ra,
					      int force  /* verify failure, if not encypted */,
					      const enum hmcon               have_con));
static enum hmcert hmremote_verify_cert_1(item_r,ra,force,have_con)
     const struct hashmark_remote * item_r;
     struct remote_account        * ra;
     int force  /* verify failure, if not encypted */;
     const enum hmcon               have_con;
{
    struct string * Server = NULL;
    enum hmcert ret = hmcert_needcheck;

    if (HASHMARK_REMOTE_magic != item_r->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hmremote_verify_cert_1",
	      "Bad magic (hashmark_remote",0);
        
    if (REMOTE_ACCOUNT_magic != ra->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hmremote_verify_cert",
	      "Bad magic number (remote_account)",0);


    switch(have_con) {
    case hmcon_imap: Server = format_string(CATGETS(elm_msg_cat, MeSet,MeIMAPserver,
						    "IMAP server"));
	break;
    case hmcon_pop: Server = format_string(CATGETS(elm_msg_cat, MeSet,MePOPserver,
						  "POP server"));
	break;
    default: Server = format_string(CATGETS(elm_msg_cat, MeSet,MeXXserver,
					    "Server"));
	break;
    }

    if (ra->stream) {
	if (item_r->verify_tls_certificate ||
	    item_r->require_tls_peer_name ||
	    item_r->req_tls_peername_count >0) {
	    
	    ret = hmcert_ok; /* Assume OK */

	    DPRINT(Debug,8,(&Debug,
			    "hmremote_verify_cert_1: Need verify%s%s%s\n",
			    item_r->verify_tls_certificate ?
			    " /certificate" : "",
			    item_r->require_tls_peer_name ?
			    " /peer name (as host)" : "",
			    item_r->req_tls_peername_count > 0 ?
			    " /peer name (from name list)" : ""));

	    if (item_r->verify_tls_certificate) {
		int r = 0;

		StreamInfo(ra->stream,SS_verify,&r,NULL,NULL);
		
		if (r < 0) {
		    
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeRAFailedVerifyCert,
				      "Failed to verify %S %s certificate (required)."),
			      Server,ra->host);
		    ret =  hmcert_fail;
		    goto failure;
		} else if (0 == r) {
	
		    int bits = 0;
		    
		    StreamInfo(ra->stream,SS_ssf,&bits,NULL,NULL);
	   
		    if (bits > 0) {
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeRANotVerifiedCert,  
					  "%S %s certificate not verified (required)."),
				  Server,ra->host);
			ret =  hmcert_fail;
		    } else if (force) {
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeRANotEncryptedCert,
					  "%S %s connection not encrypted (certificate required)."),
				  Server,ra->host);
			ret =  hmcert_fail;		    
		    } else {
			DPRINT(Debug,8,(&Debug,
					"hmremote_verify_cert_1: %s: certificate verify required, not encrypted connection\n",
					ra->hostname));
			
			ret = hmcert_needcheck;
		    }
		    goto failure;
		}		  	
	    }


	    if (item_r->require_tls_peer_name ||
		item_r->req_tls_peername_count >0) {

		struct string * returned_name = NULL;
		struct string * returned_host_name = NULL;

		struct string * official_name = NULL;
		struct string * hostname      = NULL;
		int            hostname_is_ip = 0;
		const struct string *require_name = NULL;
		int found = 0;
		
		int i;
		int start = 0;
		
		if (item_r->require_tls_peer_name) {
		    SOCKADDR ip_literal;
		    
		    start = -1;

		    hostname = new_string2(system_charset,
					   s2us(item_r->hostname));

		    if (get_ip(&ip_literal,item_r->hostname)) {
			DPRINT(Debug,8,(&Debug,
					"hmremote_verify_cert_1: %s is ip address\n",
					item_r->hostname));
			hostname_is_ip = 1;
		    }

		    if (ra->service) {
			
			if (SERVICE_ENTRY_magic != ra->service->magic)
			    panic("CONNECTION PANIC",__FILE__,__LINE__,
				  "remote_account_verify_peer",
				  "Bad magic number (service_entry)",0);
			
			start = -2;
			official_name = new_string2(system_charset,
						    s2us(ra->service->
							 official_name));
			
		    } else {
			DPRINT(Debug,8,(&Debug,
					"hmremote_verify_cert_1: %s: No service\n",
					ra->host));
		    }
		}		    

		for (i = start; i < item_r->req_tls_peername_count; i++) {
		    int r;
		    int is_ip_literal = 0;
		    
		    if (i < -1) {
			require_name = official_name;
			is_ip_literal = NULL != ra->service->ip_literal_address.dummy;
		    } else if (-1 == i) {
			require_name = hostname;
			is_ip_literal = hostname_is_ip;			
		    } else
			require_name = item_r->req_tls_peername_list[i];
			
		    DPRINT(Debug,8,(&Debug,
				    "hmremote_verify_cert_1: (%d) verifying name %S\n",
				    i,require_name));

		    r = StreamVerifyName(ra->stream,SS_peer_cn,require_name,
					 & returned_name);

		    DPRINT(Debug,8,(&Debug,
				    "hmremote_verify_cert_1: %d: StreamVerifyName SS_peer_cn require_name=%S",
				    i,require_name));
		    if (returned_name) {
			DPRINT(Debug,8,(&Debug, " returned_name=%S",
					returned_name));
		    }
		    
		    if (r) {
			DPRINT(Debug,8,(&Debug,
					" -- succeed (return %d) -- %S ok\n",
					r, require_name));
			found = 1;
			break;
		    }
		    DPRINT(Debug,8,(&Debug, " -- failed\n",r));

		    if (is_ip_literal) {
			DPRINT(Debug,8,(&Debug,
					"hmremote_verify_cert_1: %d: %S marked as ip literal -- no SS_check_host\n",
					i,require_name));
			
			/* XXXX Need SS_check_ip */
			
			
		    } else {
			r = StreamVerifyName(ra->stream,SS_check_host,require_name,
					  & returned_host_name);

		     DPRINT(Debug,8,(&Debug,
				     "hmremote_verify_cert_1: StreamVerifyName SS_check_host require_name=%S",
				     require_name));

		     if (returned_host_name) {
			 DPRINT(Debug,8,(&Debug, " returned_host_namename=%S",
					 returned_host_name));
		     }
		     
		     if (r) {
			 found = 1;
			 DPRINT(Debug,8,(&Debug, " -- succeed (return %d)\n",r));
		     } else {
			 DPRINT(Debug,8,(&Debug, " -- failed\n",r));
		     }
		    }
		    
		}
		    
		if (!found) {
		    ret =  hmcert_fail;
		    
		    DPRINT(Debug,8,(&Debug,
				    "hmremote_verify_cert_1: Name not found.\n"));
		    
		    if (start < ra->service->req_tls_peername_count -1 ||
			!require_name) {
			/* Several possible names */
			
			if (returned_name) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeRACertCNWrongOther,
					      "%S %s certificate CN is %S. Other certificate is required."),
				      Server,ra->host,returned_name);
			    
			} else {
			    
			    int bits = 0;
			    
			    StreamInfo(ra->stream,SS_ssf,&bits,NULL,NULL);
			    
			    if (bits > 0)
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeRACertCNWrongCert,
						  "%S %s certificate CN is wrong."),
					  Server,ra->host);
			    else if (force) {
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeRANotEncryptedCN,
						  "%S %s connection not encrypted, certificate required."),
					  Server,ra->host);
			    } else {
				DPRINT(Debug,8,(&Debug,
						"hmremote_verify_cert_1: %s: certificate name required, not encrypted connection\n",
						    ra->hostname));
				
				ret = hmcert_needcheck;
			    }
			}

		    } else {
			if (returned_name) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeRACertCNWrongRequired,
					      "%S %s certificate CN is %S, but %S required."),
					  Server,ra->host,returned_name,
				      require_name);
			    
			} else {
			    int bits = 0;
			    
			    StreamInfo(ra->stream,SS_ssf,&bits,NULL,NULL);
			    
			    if (bits > 0)
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeRACertCNRequired,
						  "%S %s certificate CN %S required."),
					  Server,ra->host,require_name);
			    else if (force) {
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeRANotEncryptedRequired,
						  "%S %s connection not encrypted, but certificate CN %S required."),
					  Server,ra->host,require_name);
				
			    } else {
				DPRINT(Debug,8,(&Debug,
						"hmremote_verify_cert_1: %s: certificate name %S required, not encrypted connection\n",
						ra->hostname,require_name));
				
				ret = hmcert_needcheck;
			    }
			}
		    }			
		}

		if (returned_host_name)
		    free_string(&returned_host_name);	 
		if (returned_name)			       
		    free_string(&returned_name);
		if (official_name)
		    free_string(&official_name);
		
		if (official_name)
		    free_string(&hostname);
	    }	   

	} else {
	    DPRINT(Debug,8,(&Debug,
			    "hmremote_verify_cert_1: %s: No verify required\n",
			    ra->host));
	    ret = hmcert_none;
	}
	
    } else {	
	DPRINT(Debug,8,(&Debug,
			"hmremote_verify_cert_1: %s: No connection\n",
			ra->host ? ra->host: "<no hostname>"));
	
	ret = force ? hmcert_fail : hmcert_needcheck;
	
    }

 failure:

    if (Server)
	free_string(&Server);

     DPRINT(Debug,15,(&Debug,
		    "hmremote_verify_cert_1=%d",ret));

    switch(ret) {
    case hmcert_needcheck: DPRINT(Debug,15,(&Debug," (hmcert_needcheck)")); break;
    case hmcert_ok:        DPRINT(Debug,15,(&Debug," (hmcert_ok)")); break;
    case hmcert_none:      DPRINT(Debug,15,(&Debug," (hmcert_none)")); break;
    case hmcert_fail:      DPRINT(Debug,15,(&Debug," (hmcert_fail)")); break;
    }
    
    DPRINT(Debug,15,(&Debug,
		    "\n"));

    return ret;
}

/*  hmcert_needcheck (0) == not verified, not encrypted */
static enum hmcert hmremote_verify_cert P_((const struct hashmark_remote * item_r,
					    struct hashmark_rdata        * rdata,
					    struct remote_account        * ra,
					    int force  /* verify failure, if not encypted */));
static enum hmcert hmremote_verify_cert(item_r,rdata,ra,force)
     const struct hashmark_remote * item_r;
     struct hashmark_rdata        * rdata;
     struct remote_account        * ra;
     int force  /* verify failure, if not encypted */;
{
    enum hmcert ret = hmcert_needcheck;

    if (HASHMARK_REMOTE_magic != item_r->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hmremote_verify_cert",
	      "Bad magic (hashmark_remote",0);
    
    if (HASHMARK_RDATA_magic !=  rdata->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hmremote_verify_cert",
              "Bad magic number (hashmark_rdata)",0);
    
    if (REMOTE_ACCOUNT_magic != ra->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hmremote_verify_cert",
	      "Bad magic number (remote_account)",0);

    ret =   hmremote_verify_cert_1(item_r,ra,force,
				   rdata->have_con);

    rdata->need_cert_check = ret;
    
    DPRINT(Debug,8,(&Debug,
		    "hmremote_verify_cert=%d",ret));
    
    switch(ret) {
    case hmcert_needcheck: DPRINT(Debug,8,(&Debug," (hmcert_needcheck)")); break;
    case hmcert_ok:        DPRINT(Debug,8,(&Debug," (hmcert_ok)")); break;
    case hmcert_none:      DPRINT(Debug,8,(&Debug," (hmcert_none)")); break;
    case hmcert_fail:      DPRINT(Debug,8,(&Debug," (hmcert_fail)")); break;
    }
    
    DPRINT(Debug,8,(&Debug,
		    "\n"));
        
    return ret;
}

/* increments refcount, caller must call free_service_entry() */

static  struct service_entry *
   htrem_give_service_entry P_((struct hashmark_remote * item_r,
				int give_service_entry_flag,
				struct cancel_data         ** cancel_p
				/* May be NULL, Used if dns lookup was
				   cancelable 
				*/
				));
static  struct service_entry *htrem_give_service_entry(item_r,
						       give_service_entry_flag,cancel_p)
     struct hashmark_remote * item_r;
     int give_service_entry_flag;
     struct cancel_data         ** cancel_p
     /* May be NULL, Used if dns lookup was cancelable */;  
{
    struct service_entry * se = NULL;


    struct schedule_timelimit  now = NO_schedule_timelimit;

    /* Should succeed -- error == ((time_t) -1) ignored */
    schedule_set_current_time(&now);
    
    if (HASHMARK_REMOTE_magic != item_r->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "htrem_give_service_entry",
	      "Bad magic (hashmark_remote, item)",0);

    if (item_r->service) {

	se = give_service_entryS(item_r->service,
				 give_service_entry_flag,
				 itls_ignore,
				 &now,cancel_p);
    } else {
	
	se = give_service_entry5(item_r->hostname,
				 item_r->ip_literal_address.
				 generic,
				 item_r->servicetype,
				 give_service_entry_flag,
				 itls_ignore,
				 &now,
				 cancel_p);
    }
	
    return se;
}

static const char * htrem_get_username P_((const struct hashmark_item * item,
					   struct address **addr));
static const char * htrem_get_username(item,addr)
     const struct hashmark_item * item;
     struct address **addr;

{
    const char * remote_user = NULL;
    const char * hostname UNUSED_VAROK = "<unknown>";
    struct hashmark_remote * item_r ;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "htrem_get_username",
              "Bad magic number (hashmark_item)",0);


    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "htrem_get_username",
	      "Bad magic (hashmark_remote, item)",0);
    item_r = item->u.remote;

    if (item_r->hostname)
	hostname = item_r->hostname;

    if (addr && *addr)
	free_address(addr);
   
    if (item_r->remote_userid) {
	remote_user = item_r->remote_userid;
	
	DPRINT(Debug,15,(&Debug,
			 "htrem_get_username: %s: LOCAL_RC: user %s\n",
			 hostname,remote_user));

	if (addr && item_r->from_address) {
	    const char * A;
	    *addr = dup_address(item_r->from_address);
	    A = address_get_ascii_addr(*addr);
	    
	    if (A) {
		DPRINT(Debug,15,(&Debug,
				 "htrem_get_username: %s: LOCAL_RC: Address is %s\n",
				 hostname,A));
	    }	
	}

    } else if (item_r->usermaps || item_r->usermap_file) {

	const struct hashmark_usermap_entry * U = NULL;

	if (item_r->usermaps)
	    U = find_current_user(item_r->usermaps,
				  item_r->usermap_count);

	if (!U && item_r->usermap_file) {
	    struct hashmark_usermap_file * item_f;

	    if (HASHMARK_USERMAP_FILE_magic != item_r->usermap_file->magic)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "htrem_get_username",
		      "Bad magic number (hashmark_usermap_file)",0);
	    item_f =  item_r->usermap_file;

	    if (! item->hashmark_name) {
		DPRINT(Debug,15,(&Debug,
				 "htrem_get_username: %s: No hashmark name\n",
				 hostname));
		goto fail;
	    }

	    if (item_f->hashmarks) {
		int j;

		for (j = 0; j < item_f->hashmark_count; j++) {
		    if (0 == string_cmp(item->hashmark_name,
					item_f->hashmarks[j].hashmark,
					99 /* unknown character */)) {
			
			DPRINT(Debug,15,(&Debug,
					 "htrem_get_username: %s: Hashmark %S found from %s\n",
					 hostname, item->hashmark_name,
					 item_f->filename ? item_f->filename : "<no name>"));
			U = find_current_user(item_f->hashmarks[j].usermaps,
					      item_f->hashmarks[j].usermap_count);
			if (U)
			    break;					      
		    }
		}
	    }
	} /* !U && item_r->usermap_file */

	if (U) {
	    if (U->remote_userid) {
		remote_user = U->remote_userid;
		
		DPRINT(Debug,15,(&Debug,
				 "htrem_get_username: %s: SYSTEM_RC: user %s (mapped from %s)\n",
				 hostname,remote_user,username));
	    } else {
		remote_user = username;
		DPRINT(Debug,15,(&Debug,
				 "htrem_get_username: %s: SYSTEM_RC: user %s (uses local username)\n",
				 hostname,remote_user));
	    }

	    if (addr && U->from_address) {
		const char * A;
		*addr = dup_address(U->from_address);
		A = address_get_ascii_addr(*addr);
		
		if (A) {
		    DPRINT(Debug,15,(&Debug,
				     "htrem_get_username: %s: SYSTEM_RC: Address is %s\n",
				     hostname,A));
		}		    
	    }

	} else {
	    DPRINT(Debug,15,(&Debug,
			     "htrem_get_username: %s: LOCAL_RC: No mapping for user %s\n",
			     hostname,username));
	}
		
    } else {
	DPRINT(Debug,15,(&Debug,
			 "htrem_get_username: %s: userid not set\n",
			 hostname));
    }

 fail:
    return remote_user;
}


/* Check that connection is valid */
static enum hmcon htrem_have_connection P_((const struct hashmark_item * item,
					    union hashmark_data        * hashmark_data,
					    int                          need_browser));
static enum hmcon htrem_have_connection(item,hashmark_data,need_browser)
     const struct hashmark_item * item;
     union hashmark_data        * hashmark_data;
     int                          need_browser;
{
    enum hmcon ret = hmcon_none;
    struct hashmark_remote * item_r ;
    struct hashmark_rdata  * rdata;
    
    struct service_entry   * se = NULL;
    struct service_entry   * result_se  = NULL;

    const char * remote_user = NULL;

    struct cancel_data *main_cancel = NULL /* Used if dns lookup was
					      cancelable */;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "htrem_have_connection",
              "Bad magic number (hashmark_item)",0);

    if (HASHMARK_RDATA_magic !=  hashmark_data->remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "htrem_have_connection",
              "Bad magic number (hashmark_rdata)",0);
    rdata = hashmark_data->remote;

    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "htrem_have_connection",
	      "Bad magic (hashmark_remote, item)",0);
    item_r = item->u.remote;

    if (!item_r->hostname) {
	DPRINT(Debug,15,(&Debug,
			 "htrem_have_connection: No hostname\n"));
	goto fail;
    }
    if (item_r->verify_tls_certificate ||
	item_r->require_tls_peer_name ||
	item_r->req_tls_peername_count > 0) {
	DPRINT(Debug,15,(&Debug,
			 "htrem_have_connection: %s: need certificate checks\n",
			 item_r->hostname));
	rdata->need_cert_check = hmcert_needcheck;
    } else {
	DPRINT(Debug,15,(&Debug,
			 "htrem_have_connection: %s: no hashmark specific certificate checks\n",
			 item_r->hostname));

	rdata->need_cert_check = hmcert_none ;
    }

    remote_user = htrem_get_username(item,&  (rdata->from_address));
    if (!remote_user) {
	DPRINT(Debug,15,(&Debug,
			 "htrem_have_connection: %s: userid not set\n",
			 item_r->hostname));
	goto fail;
    }

    switch (rdata->have_con) {
    case hmcon_imap:

	if (rdata->CX) {
	    int r;
	    
	    enum connection_state s = get_connection_state(rdata->CX);

	    if (connection_match_username_host(rdata->CX,
					       remote_user,item_r->hostname)) {
		
		if (s >= CON_open) {
		    DPRINT(Debug,15,(&Debug,
				     "htrem_have_connection: %s: assuming valid connection.\n",
				     item_r->hostname));
		    goto have_CX;
		}
	    } else {
		DPRINT(Debug,15,(&Debug,
				 "htrem_have_connection: %s: connection does not match.\n",
				  item_r->hostname));
	    }

	    DPRINT(Debug,15,(&Debug,
			     "htrem_have_connection: %s: connection state %d, freeing\n",
			     item_r->hostname,s));

	    r = free_connection(& (rdata->CX),
				main_cancel  /* NULL */);

	    if (!r) {
		DPRINT(Debug,15,(&Debug,
				 "htrem_have_connection: free_connection failed\n"));
	    }
	}

	/* FALLTHRU */
    case hmcon_pop:

	if (rdata->ra) {
	    if (remote_account_match(rdata->ra,remote_user,item_r->hostname)) {
		DPRINT(Debug,15,(&Debug,
				 "htrem_have_connection: %s: assuming valid connection (ra).\n",
				 item_r->hostname));
		
		if (!need_browser || rdata->have_con >= hmcon_imap) {


		    if (hmcert_needcheck == rdata->need_cert_check)
			hmremote_verify_cert(item_r,rdata,
					     rdata->ra,0);
		    if (hmcert_fail     == rdata->need_cert_check) {
			DPRINT(Debug,15,(&Debug,
					 "htrem_have_connection: %s: certificate verify failure\n",
					 item_r->hostname));
			ret = hmcon_none ;
			goto fail;
		    }

		    ret = rdata->have_con;
		    goto have_ra;
		}

	    } else {
		DPRINT(Debug,15,(&Debug,
				 "htrem_have_connection: %s: connection (ra) does not match.\n",
				 item_r->hostname));
	    }

	    DPRINT(Debug,15,(&Debug,
			     "htrem_have_connection: %s: connection type %d, clearing\n",
			     item_r->hostname,rdata->have_con));
	    clear_remote_account(rdata->ra);
	}

	/* FALLTHRU */
    case hmcon_none:
	DPRINT(Debug,15,(&Debug,
			 "htrem_have_connection: %s: need connection\n",
			 item_r->hostname));
	rdata->have_con = hmcon_none;
			
    }

    if (rdata->CX) {
	int r = free_connection(& (rdata->CX),
				main_cancel /* NULL */);

	if (!r) {
	    DPRINT(Debug,15,(&Debug,
			     "htrem_have_connection: free_connection failed\n"));
	}
    }


    /* increments refcount, caller must call free_service_entry() */    
    se = htrem_give_service_entry(item_r,
				  STFLAG_mbox |
				  (need_browser ? STFLAG_browser : 0),
				  &main_cancel
				  );

	
    if (!se) {
	if (main_cancel && is_canceled(main_cancel)) {
	    DPRINT(Debug,12,(&Debug,
			     "htrem_have_connection: DNS lookup canceled\n"));
	} else {
	    DPRINT(Debug,15,(&Debug,
			     "htrem_have_connection: %s not found%s\n",
			     item_r->hostname,
			     need_browser ? ", need browser" : ""));
	}	
	goto fail;
    }


    /* Only IMAP connections are cached */
    rdata->CX = locate_from_cache(remote_user,item_r->hostname,
			   &IMAP_connection,
			   0 /* Default port */,se);

    if (rdata->CX) {
	DPRINT(Debug,15,(&Debug,
			 "htrem_have_connection: %s: Found cached IMAP connection\n",
			 item_r->hostname));
	
    have_CX:

	if (CONNECTION_TYPE_magic != rdata->CX->type->magic)
	    panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
		  "htrem_have_connection",
		  "Bad connection type",0);
	
	rdata->have_con = hmcon_imap;
	ret             = hmcon_imap;

	if (hmcert_needcheck == rdata->need_cert_check)
	    hmremote_verify_cert(item_r,rdata,
				 &(rdata->CX->C),0);
	if (hmcert_fail     == rdata->need_cert_check) {
	    DPRINT(Debug,15,(&Debug,
			     "htrem_have_connection: %s: certificate verify failure\n",
			     item_r->hostname));
	    ret = hmcon_none ;
	    goto fail;
	}

    } else {
	
	static const int port_masks [] = { STFLAG_is_imap, STFLAG_is_pop, 0 };
	static const int port_imaponly_masks [] = { STFLAG_is_imap, 0 };

	const struct service_type  * got_type = NULL;
	int                          got_port = 0;
	
	struct enum_service_type ports_2 = NULL_enum_service_type;
	struct service_entry   * seX;
	
	init_enum_service_list(&ports_2,init_tls_default,
			       need_browser ? port_imaponly_masks :  port_masks,
			       STFLAG_mbox);
	
	if (rdata->ra) {
	    clear_remote_account(rdata->ra);
	    
	    rdata->ra->username = strmcpy(rdata->ra->username,remote_user);
	    rdata->ra->host = strmcpy(rdata->ra->host,item_r->hostname);
	    
	} else 
	    rdata->ra =   
		malloc_remote_account(remote_user,item_r->hostname);
	
	if (connect_remote_account_2(rdata->ra,se,main_cancel,
				     
				     &result_se,
				     &got_type,
				     &got_port,
				     
				     &ports_2)) {
	    
	    enum itls_status have_initial_tls = itls_none;
	    int stflag;
	    
	    DPRINT(Debug,15,(&Debug,
			     "htrem_have_connection: %s: connected to port %d",
			     item_r->hostname,got_port));
	    if (got_type) {
		DPRINT(Debug,15,(&Debug,", type=%p %s defport %d",
				 got_type,
				 service_type_name(got_type),
				 service_type_defport(got_type)));
	    }
	    DPRINT(Debug,15,(&Debug,"\n"));

	    switch((have_initial_tls = remote_account_init_tls((rdata->ra)))) {
		
	    case itls_unsupported:
	    case itls_failed:
		
		DPRINT(Debug,15,(&Debug,
				 "htrem_have_connection: %s: Initial TLS not OK\n",
				 item_r->hostname));
		
		goto fail;
		
	    case itls_none:
		break;
		
	    case itls_done:
		DPRINT(Debug,8,(&Debug,
				"htrem_have_connection: %s: Initial TLS done (no STARTTLS)\n",
				item_r->hostname));
		
		if (hmcert_fail == hmremote_verify_cert(item_r,rdata,rdata->ra,1)) {
		    DPRINT(Debug,15,(&Debug,
				     "htrem_have_connection: %s: certificate verify failure\n",
				     item_r->hostname));
		    ret = hmcon_none ;
		    goto fail;
		}		
	    }

	    if (result_se) {
		if (SERVICE_ENTRY_magic != result_se->magic)
		    panic("MAILER PANIC",__FILE__,__LINE__,
			  "htrem_have_connection",
			  "Bad magic (service_entry)",0);
	    
		DPRINT(Debug,8,(&Debug,
				"htrem_have_connection: Got service entry %p %s, type %p %s\n",
				result_se,result_se->official_name,
				result_se->service,service_type_name(result_se->service)));
		
		seX = result_se;
	    } else
		seX = se;

	    
	    switch((stflag = STFLAG_service_type(seX))) {
	    case STFLAG_is_imap:
		goto is_imap;
	    case STFLAG_is_pop:
		goto is_pop;
		
	    default:
		DPRINT(Debug,8,(&Debug,
				"htrem_have_connection: %s: unknown stflag=x%04x (service=%p %s, port=%d)\n",
				item_r->hostname,stflag,seX->service,
				service_type_name(seX->service), got_port));
	    }

	    switch (got_port) {
	    case PORT_pop3s:
		DPRINT(Debug,8,(&Debug,
				"htrem_have_connection: %s: pops service ... handled as pop\n",
				item_r->hostname));
		
	    case PORT_pop3:
	    is_pop:
		
		rdata->have_con = hmcon_pop;
		ret             = hmcon_pop;

		break;

	    case PORT_imaps:
		DPRINT(Debug,8,(&Debug,
				"htrem_have_connection: %s: imaps service ... handled as imap\n",
				item_r->hostname));
		
	    case PORT_imap4:
	    is_imap:

		rdata->have_con = hmcon_imap;
		ret             = hmcon_imap;
	
		break;
	    default:
		DPRINT(Debug,8,(&Debug,
				"htrem_have_connection: %s: Unknown mailbox type (service=%p %s, port =%d)\n",
				
				item_r->hostname,seX->service,
				service_type_name(seX->service), got_port));
		
		break;

	    }	    
	} else {
	    DPRINT(Debug,15,(&Debug,
			     "htrem_have_connection: %s: failed to connect\n",
			     item_r->hostname));
	}
    }

 have_ra:
 fail:

    
    if (ret != rdata->have_con) {
	DPRINT(Debug,15,(&Debug,
			 "htrem_have_connection: %s: reset\n",
			 item_r->hostname ? item_r->hostname : "(no hostname)"));

	if (hashmark_data->remote->CX) {
	    int r = free_connection(& (hashmark_data->remote->CX),
			    main_cancel /* ? ? ? */);

	    if (!r) {
		DPRINT(Debug,15,(&Debug,
				 "htrem_have_connection: free_connection failed\n"));
	    }
	}
	if (hashmark_data->remote->ra)
	    free_remote_account(& (hashmark_data->remote->ra));
	if (hashmark_data->remote->from_address)
	    free_address(&  (hashmark_data->remote->from_address));
	rdata->have_con = hmcon_none;       
	ret =  hmcon_none;
    }

    if (main_cancel)
	free_cancel(& (main_cancel));

    
    /* 'se' can be NULL */
    free_service_entry(&se);
    free_service_entry(&result_se);

    DPRINT(Debug,15,(&Debug,
		     "htrem_have_connection=%d",ret));

    switch (ret) {
    case hmcon_none:
	DPRINT(Debug,15,(&Debug," (hmcon_none)"));
	break;
    case hmcon_pop:
	DPRINT(Debug,15,(&Debug," (hmcon_pop)"));
	break;
    case  hmcon_imap:
    	DPRINT(Debug,15,(&Debug," (hmcon_imap)"));
	break;
    }
    DPRINT(Debug,15,(&Debug,"\n"));
	
    return ret;
}

/* Connects to remote server and changes dir type */
static int hashtype_br_change_imap P_((struct hashmark_item * item,
				       union hashmark_data        * hashmark_data,
				       struct folder_browser      * dir));

static int hashtype_br_change_imap(item,hashmark_data,dir)
     struct hashmark_item * item;
     union hashmark_data        * hashmark_data;
     struct folder_browser        * dir;
{
    int ret = 0;
    struct hashmark_remote * item_r ;
    struct hashmark_rdata  * rdata = NULL;
    struct browser_passhm  * passhm = NULL;


    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_br_change_imap",
              "Bad magic number (hashmark_item)",0);

    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_br_change_imap",
	      "Bad magic (hashmark_remote, item)",0);
    item_r = item->u.remote;
    
    if (HASHMARK_RDATA_magic !=  hashmark_data->remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_br_change_imap",
              "Bad magic number (hashmark_rdata)",0);
    rdata = hashmark_data->remote;
    inc_hashtype_rdata_refcount(rdata);  /* So that can be free'ed on end */

    /*   union hashmark_data   is not valid after change_folder_browser_type
	 is called !!!!
    */

    switch (rdata->have_con) {
    case hmcon_imap:
		
	if (change_folder_browser_type(dir,&imap_browser)) {
	
	    passhm = new_browser_passhm(item,rdata->from_address,
					rdata->need_cert_check,
					rdata->have_con);

	    if (rdata->CX) {
		DPRINT(Debug,15,(&Debug,
				 "hashtype_br_change_imap: %s: Have cached IMAP connection\n",
				 item_r->hostname));

		/*  browser_from_connectionX sets rdata->CX = NULL */
		browser_from_connectionX(&(rdata->CX),dir);
		ret = 1;
	    } else if (rdata->ra) {
		DPRINT(Debug,15,(&Debug,
				 "hashtype_br_change_imap: %s: Have remote account\n",
				 item_r->hostname));
		
		/* Is this correct? */
		ret = join_connection_hm(give_imap_connection(dir),rdata->ra,CON_greeting,
					 passhm);
		
		if (!ret) {
		    DPRINT(Debug,15,(&Debug,
				     "hashtype_br_change_imap: %s: join_connection_hm failed\n",
				     item_r->hostname));
		}
	    }

	    if (ret && hmcert_needcheck == rdata->need_cert_check) {
		int r;
		enum hmcert r2;
		struct connection_cache *c3 = give_imap_connection(dir);

		if ((r = prepare_connection_verify(c3)) < 1) {
		    DPRINT(Debug,15,(&Debug,
				     "hashtype_br_change_imap: %s: prepare_connection_verify=%d failed\n",
				     item_r->hostname,r));
		    ret = 0;
		    goto failure;
		}

		if ((r2 = hmremote_verify_cert(item_r,rdata,&(c3->C),
					       1 /*  verify failure, if not encypted */)) < hmcert_ok) {

		    DPRINT(Debug,15,(&Debug,
				     "hashtype_br_change_imap: %s: hmremote_verify_cert=%d failed\n",
				     item_r->hostname,r2));
		    ret = 0;
		    goto failure;	       	
		}
	    }

	    if (rdata->need_cert_check < hmcert_ok) {
		DPRINT(Debug,15,(&Debug,
				 "hashtype_br_change_imap: %s: need_cert_check=%d\n",
				 item_r->hostname,rdata->need_cert_check));
		ret = 0;
		goto failure;
	    }

	    /* increments refcount of passhm */
	    imap_browser_set_passhm(dir,passhm);

	} else {
	    DPRINT(Debug,15,(&Debug,
			     "hashtype_br_change_imap: %s: type not changed?\n",
			     item_r->hostname));
	}
	break;
		    
    default:
	DPRINT(Debug,15,(&Debug,
			 "hashtype_br_change_imap: %s: No imap connection\n",
			 item_r->hostname));
	break;
    }

    if (!ret) {
    failure:

	/* because connection is not on cache, must be free'ed */
	if (rdata->CX) {
	    int r = free_connection(& (rdata->CX),NULL);

	    if (!r) {
		DPRINT(Debug,15,(&Debug,
				 "hashtype_br_change_imap: free_connection failed\n"));
	    }
	}

	if (rdata->ra)
	    free_remote_account(& (rdata->ra));

	DPRINT(Debug,15,(&Debug,
			 "hashtype_br_change_imap: Failure, canceling type change\n"));
	/* CANCEL type change  */
	
	change_folder_browser_type(dir,&hashmark_browser);
    }

    if (passhm)
	free_browser_passhm(&passhm);

    if (rdata)
	hashtype_free_rdata(&rdata);

    DPRINT(Debug,15,(&Debug,
		     "hashtype_br_change_imap=%d%s\n",
		     ret, ret ? "" : " (failed)"));

    return ret;
}

static int hashtype_brchdir_remote(item,dir,hashmark_data,path_tail,dispname)
     struct hashmark_item       * item;
     struct folder_browser      * dir;
     union hashmark_data        * hashmark_data;
     const struct string        * path_tail; /* NULL if to root dir of hashmark */
     struct string             ** dispname;
{

    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_brchdir_remote",
              "Bad magic number",0);

    switch (htrem_have_connection(item,hashmark_data,1)) {

    case hmcon_none:
	DPRINT(Debug,15,(&Debug,"hashtype_brchdir_remote: no connection\n"));
	break;

    case hmcon_pop:

	DPRINT(Debug,15,(&Debug,"hashtype_brchdir_remote: pop connection (not supported)\n"));

	/* If user requires directory content, only IMAP can provide it,
	   so this is not supported */

	break;

    case hmcon_imap:
	
	DPRINT(Debug,15,(&Debug,"hashtype_brchdir_remote: imap connection\n"));

	/*   NOTE:  union hashmark_data   is not valid after change_folder_browser_type
	            is called !!!!
	*/
	
	if (hashtype_br_change_imap(item,hashmark_data,dir)) {
	    
	    
	    ret = dir->type->browser_change_it(dir,path_tail,dispname);
	    
	    if (!ret) {
		DPRINT(Debug,15,(&Debug,"hashtype_brchdir_remote: real browser_change_it FAILED\n"));	
	    }
	    
	} else {
	    
	    DPRINT(Debug,16,(&Debug,
			     "hashtype_brchdir_remote: failed to change browser type to real type\n"));
	    
	}

	break;
    }

    DPRINT(Debug,15,(&Debug,"hashtype_brchdir_remote=%d%s\n",
		     ret,
		     ret ? "" : " (failed)"));

    return ret;
}

static int hashtype_verify_racon_remote(item,passhm,C)
     const struct hashmark_item * item;
     struct browser_passhm      * passhm;
     struct remote_account *C;
{
    struct hashmark_remote * item_r ;
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_verify_racon_remote",
              "Bad magic number",0);

    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_verify_racon_remote",
	      "Bad magic (hashmark_remote, item)",0);
    item_r = item->u.remote;

    if (BROWSER_PASSHM_magic != passhm->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_verify_racon_remote",
              "Bad magic number (browser_passhm)",0);

    
    if (hmcert_needcheck == passhm->need_cert_check)
	passhm->need_cert_check = hmremote_verify_cert_1(item_r,
							 C,
							 1,
							 passhm->contype);

    if (hmcert_fail  == passhm->need_cert_check) {
	    DPRINT(Debug,15,(&Debug,
			     "hashtype_verify_racon_remote: %s: certificate verify failure\n",
			     item_r->hostname));
	    ret = 0;
    } else
	ret = passhm->need_cert_check >= hmcert_ok;
    
    DPRINT(Debug,15,(&Debug,
		     "hashtype_verify_racon_remote=%d\n",
		     ret));
    
    return ret;
}

static enum hmcert hashtype_hmcert_needcheck P_((const char *host,
						 struct hashmark_remote * item_r));
static enum hmcert hashtype_hmcert_needcheck(host,item_r)
     const char             * host;
     struct hashmark_remote * item_r;
{

    if (item_r->verify_tls_certificate ||
	item_r->require_tls_peer_name ||
	item_r->req_tls_peername_count > 0) {
	
	DPRINT(Debug,15,(&Debug,
			 "hashtype_hmcert_needcheck: %s: need certificate checks\n",
			 host));
	
	return hmcert_needcheck;

    } 

    DPRINT(Debug,15,(&Debug,
		     "hashtype_hmcert_needcheck: %s: no hashmark specific certificate checks\n",
		     host));	 
    
    return hmcert_none;	
}


static int hashtype_passhm_open_ra2_remote(item,passhm,ra,give_service_entry_flag,est)
     const struct hashmark_item * item;
     struct browser_passhm      * passhm;
     struct remote_account      * ra;
     int                          give_service_entry_flag;
					     
     /* For enumerate_service_type */
     struct enum_service_type   * est;
{
    struct hashmark_remote    * item_r;
    struct service_entry      * se = NULL;
    struct service_entry      * result_se = NULL;

    int                         ret = 0;
    int                         got_port = 0;
    const struct service_type * got_type = NULL;

    struct cancel_data *main_cancel = NULL /* Used if DNS lookup was
					      cancelable */;
    
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_passhm_open_ra2_remote",
              "Bad magic number",0);

    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_passhm_open_ra2_remote",
	      "Bad magic (hashmark_remote, item)",0);
    item_r = item->u.remote;

    if (BROWSER_PASSHM_magic != passhm->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_passhm_open_ra2_remote",
              "Bad magic number (browser_passhm)",0);


    if (REMOTE_ACCOUNT_magic != ra->magic) 
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_passhm_open_ra2_remote",
              "Bad magic number (remote_account)",0);

        /* increments refcount, caller must call free_service_entry() */    
    se = htrem_give_service_entry(item_r,
				  give_service_entry_flag,&main_cancel);
    
    if (!se) {
	if (main_cancel && is_canceled(main_cancel)) {
	    DPRINT(Debug,15,(&Debug,
			     "hashtype_passhm_open_ra2_remote: DNS lookup canceled\n"));
	} else {
	    DPRINT(Debug,15,(&Debug,
			     "hashtype_passhm_open_ra2_remote: %s not found, flag %d\n",
			     item_r->hostname,
			     give_service_entry_flag));
	}	
	goto fail;
    }

    ra->host = strmcpy(ra->host,se->official_name);
    if (connect_remote_account_2(ra,se,main_cancel,
				 &result_se,
				 &got_type,
				 &got_port,
				 est)) {

	DPRINT(Debug,15,(&Debug,
			 "hashtype_passhm_open_ra2_remote: %s: connected, got port %d",
			 ra->host,got_port));
	if (got_type) {
	    DPRINT(Debug,15,(&Debug,", type=%p %s defport %d",
			     got_type,
			     service_type_name(got_type),
			     service_type_defport(got_type)));
	}
	DPRINT(Debug,15,(&Debug,"\n"));

	passhm->need_cert_check = hashtype_hmcert_needcheck(ra->host,item_r);
	
	ret = 1;
    }
    
 fail:
    /* 'se' can be NULL */
    free_service_entry(&se);
    free_service_entry(&result_se);

    if (main_cancel)
	free_cancel(& (main_cancel));

    DPRINT(Debug,15,(&Debug,
		     "hashtype_passhm_open_ra2_remote=%d: %s\n",
		     ret,ra->host ? ra->host : "???"));

    return ret; 
}


static int hashtype_passhm_open_ra_remote(item,passhm,ra,default_portlist,
					  give_service_entry_flag,force_port)
     const struct hashmark_item * item;
     struct browser_passhm      * passhm;
     struct remote_account      * ra;
     const PORTS                  default_portlist[];
     int                          give_service_entry_flag;
     PORTS                        force_port;
{
    struct hashmark_remote * item_r;
    struct service_entry * se = NULL;

    int ret = 0;
    int got = 0;
    
    struct cancel_data *main_cancel = NULL /* Used if DNS lookup was
					       cancelable */;
	        
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_passhm_open_ra_remote",
              "Bad magic number",0);

    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_passhm_open_ra_remote",
	      "Bad magic (hashmark_remote, item)",0);
    item_r = item->u.remote;

    if (BROWSER_PASSHM_magic != passhm->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_passhm_open_ra_remote",
              "Bad magic number (browser_passhm)",0);


    if (REMOTE_ACCOUNT_magic != ra->magic) 
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_passhm_open_ra_remote",
              "Bad magic number (remote_account)",0);
    
    /* increments refcount, caller must call free_service_entry() */    
    se = htrem_give_service_entry(item_r,
				  give_service_entry_flag,&main_cancel);
    
    if (!se) {
	if (main_cancel && is_canceled(main_cancel)) {
	    DPRINT(Debug,15,(&Debug,
			     "hashtype_passhm_open_ra_remote: DNS lookup canceled\n"));
	} else {
	    DPRINT(Debug,15,(&Debug,
			     "hashtype_passhm_open_ra_remote: %s not found, flag %d\n",
			     item_r->hostname,
			     give_service_entry_flag));
	}	
	goto fail;
    }

    ra->host = strmcpy(ra->host,se->official_name);
    
    if (connect_remote_account(ra,&got,se,
			       default_portlist,
			       force_port,main_cancel)) {

	DPRINT(Debug,15,(&Debug,
			 "hashtype_passhm_open_ra_remote: %s: connected, got port %d\n",
			 ra->host,got));

	passhm->need_cert_check = hashtype_hmcert_needcheck(ra->host,item_r);
	
	ret = 1;
    }
    
 fail:
    /* 'se' can be NULL */
    free_service_entry(&se);
    
    if (main_cancel)
	free_cancel(& (main_cancel));

    DPRINT(Debug,15,(&Debug,
		     "hashtype_passhm_open_ra_remote=%d: %s\n",
		     ret,ra->host ? ra->host : "???"));

    return ret; 
}

static int hashtype_selectbr_item_remote(item,dir,hashmark_data,path_tail,dispname,newpos)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     const struct string   * path_tail;    /* NULL if hashmark only is selected */
     struct string        ** dispname;
     int                   * newpos;
{
    int ret = 0;
    int have_path        = path_tail != NULL;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_selectbr_item_remote",
              "Bad magic number",0);

    switch (htrem_have_connection(item,hashmark_data,have_path)) {

    case hmcon_none:
	if (have_path) {
	    DPRINT(Debug,15,(&Debug,"hashtype_selectbr_item_remote: no connection\n"));
	} else {
	    DPRINT(Debug,15,(&Debug,"hashtype_selectbr_item_remote: no connection (ignored)\n"));
	    ret = 1;

	    /* Null selection when remote mode ... */
	    set_hashmark(set_dir_selection(dir,NULL,NULL,
					   BROWSER_EXIST       /* Assume existance */
					   |BROWSER_MAILFILE   /* Assume folder    */),
			 item);

	}
	break;

    case hmcon_pop:

	if (have_path) {
	    DPRINT(Debug,15,(&Debug,"hashtype_selectbr_item_remote: pop connection (not supported)\n"));
	} else {
	    DPRINT(Debug,15,(&Debug,"hashtype_selectbr_item_remote: pop connection (ok)\n"));

	    ret = 1;

	    /* Null selection when remote mode ... */
	    set_hashmark(set_dir_selection(dir,NULL,NULL,
					   BROWSER_EXIST       /* Assume existance */
					   |BROWSER_MAILFILE   /* Assume folder    */),
			 item);
	}

	break;

    case hmcon_imap:

	if (have_path) {	
	    DPRINT(Debug,15,(&Debug,"hashtype_selectbr_item_remote: imap connection (need change type)\n"));
	
	    /*   NOTE:  union hashmark_data   is not valid after change_folder_browser_type
		 is called !!!!
	    */
		
	    if (hashtype_br_change_imap(item,hashmark_data,dir)) {
		
		ret = dir->type->browser_select_it(dir,path_tail,
						   dispname,newpos);
	    
		if (!ret) {
		    DPRINT(Debug,15,(&Debug,
				     "hashtype_selectbr_item_remote: real browser_select_it FAILED\n"));
		}
		
	    } else {
		
		DPRINT(Debug,16,(&Debug,
				 "hashtype_selectbr_item_remote: failed to change browser type to real type\n"));
		
	    }

	} else {
	    DPRINT(Debug,15,(&Debug,"hashtype_selectbr_item_remote: imap connection (ok)\n"));
	    ret = 1;

	    /* Null selection when remote mode ... */
	    set_hashmark(set_dir_selection(dir,NULL,NULL,
					   BROWSER_EXIST       /* Assume existance */
					   |BROWSER_MAILFILE   /* Assume folder    */),
			 item);
	    
	}

	break;
    }
    
    DPRINT(Debug,15,(&Debug,"hashtype_selectbr_item_remote=%d%s\n",
		     ret,
		     ret ? "" : " (failed)"));

    return ret;
}

static void htrem_set_folder_name P_((struct folder_info    * res, 
				      struct hashmark_item  * item,
				      struct hashmark_rdata  * rdata));
static void htrem_set_folder_name(res,item,rdata)
     struct folder_info * res;
     struct hashmark_item  * item;
     struct hashmark_rdata  * rdata;
{
    struct hashmark_remote * item_r ;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "htrem_set_folder_name",
              "Bad magic number",0);

    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "htrem_set_folder_name",
	      "Bad magic (hashmark_remote, item)",0);
    item_r = item->u.remote;

    if (HASHMARK_RDATA_magic !=  rdata->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "htrem_set_folder_name",
              "Bad magic number (hashmark_rdata)",0);

    if (rdata->ra) {
	const char * username =  rdata->ra->username;
	const char * host     =  rdata->ra->host;
	
	if (!username)
	    username = htrem_get_username(item,NULL);
	if (!username)
	    username = "<unknown>";

	if (!host)
	    host = item_r->hostname;
	if (!host)
	    host = "<unknown>";

	res -> cur_folder_sys  = 
	    elm_message(FRM("%s@%s"),username,host);

	if (rdata->ra->username && rdata->ra->host)
	    res -> cur_folder_disp = 
		format_string(FRM("%s@%s"),
			      rdata->ra->username,rdata->ra->host);
	else if (item->hashmark_name)
	    res -> cur_folder_disp = 
		 format_string(FRM("#%S"),
			       item->hashmark_name);
	else
	    res -> cur_folder_disp = 
		format_string(FRM("%s@%s"),
			      username,host);

    } else {
	const char * username = htrem_get_username(item,NULL);

	res -> cur_folder_sys  = 
	    elm_message(FRM("%s@%s"),
			username ? username : "<unknown>",
			item_r->hostname      ? item_r->hostname      : "<unknown>");
	
	if (item->hashmark_name)
	    res -> cur_folder_disp = 
		format_string(FRM("#%S"),
			      item->hashmark_name);
	else
	    res -> cur_folder_disp = 
		format_string(FRM("%s@%s"),
			      username ? username : "<unknown>",
			      item_r->hostname      ? item_r->hostname      : "<unknown>");
    }
}

static struct folder_info * hashtype_folder_from_remote(item,dir,
							hashmark_data,
							treat_as_spooled)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     int                     treat_as_spooled;
{
    struct folder_info * res = NULL;

    struct hashmark_rdata  * rdata ;

    struct remote_account * status = NULL;
    struct browser_passhm  * passhm = NULL;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_folder_from_remote",
              "Bad magic number",0);

    if (HASHMARK_RDATA_magic !=  hashmark_data->remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_br_change_imap",
              "Bad magic number (hashmark_rdata)",0);
    rdata = hashmark_data->remote;
    inc_hashtype_rdata_refcount(rdata);  /* So that can be free'ed on end */

    res = mbx_new_folder();
 
    switch (htrem_have_connection(item,hashmark_data,0)) {

    case hmcon_pop:
	DPRINT(Debug,15,(&Debug,"hashtype_folder_from_remote: pop connection\n"));

	if (rdata->ra) {
	
	    htrem_set_folder_name(res,item,rdata);

	    DPRINT(Debug,15,(&Debug,"hashtype_folder_from_remote: POP mailbox %S\n",
			     res -> cur_folder_disp));

	    res->folder_type = POP_MBX;
	    res->folder_type->init_it(res);

	    passhm = new_browser_passhm(item,rdata->from_address,
					rdata->need_cert_check,
					rdata->have_con);
	    
	    /* Clears rdata->ra */
	    /* increments refcount of passhm */
	    status = set_pop_connection(res,rdata->ra,passhm);
	    free_remote_account(& (rdata->ra));

	} else {
	    DPRINT(Debug,15,(&Debug,"hashtype_folder_from_remote: no remote account\n"));
	    goto no_account;
	}

	break;

    case hmcon_imap:
	DPRINT(Debug,15,(&Debug,"hashtype_brchdir_remote: imap connection\n"));
	
	if (rdata->CX) {
	    int r ;
	    
	    htrem_set_folder_name(res,item,rdata);
	    
	    DPRINT(Debug,15,(&Debug,"hashtype_folder_from_remote: IMAP mailbox %S\n",
			     res -> cur_folder_disp));

	    res->folder_type = IMAP_MBX;
	    res->folder_type->init_it(res);

	    passhm = new_browser_passhm(item,rdata->from_address,
					rdata->need_cert_check,
					rdata->have_con);

	    /* Clears rdata->CX->C */
	    /* increments refcount of passhm */
	    status = set_imap_connection_C(res,rdata->CX,passhm);
	    
	    r = free_connection(& (rdata->CX), NULL);
	    if (!r) {
		DPRINT(Debug,15,(&Debug,"hashtype_brchdir_remote: free_connection failed\n"));
	    }

	    res->p->a.imap_mbx.folder = safe_strdup("INBOX");


	} else if (rdata->ra) {
	    
	    htrem_set_folder_name(res,item,rdata);

	    DPRINT(Debug,15,(&Debug,"hashtype_folder_from_remote: IMAP mailbox %S\n",
			     res -> cur_folder_disp));

	    res->folder_type = IMAP_MBX;
	    res->folder_type->init_it(res);

	    passhm = new_browser_passhm(item,rdata->from_address,
					rdata->need_cert_check,
					rdata->have_con);

	    /* Clears rdata->ra */
	    /* increments refcount of passhm */
	    status = set_imap_connection(res,rdata->ra,
					 passhm);
	    free_remote_account(& (rdata->ra));
	    
	    res->p->a.imap_mbx.folder = safe_strdup("INBOX");

	} else {
	    DPRINT(Debug,15,(&Debug,"hashtype_folder_from_remote: no remote account\n"));
	    goto no_account;
	}

	break;


    case hmcon_none:
	DPRINT(Debug,15,(&Debug,"hashtype_folder_from_remote: no connection\n"));

    no_account:

	htrem_set_folder_name(res,item,rdata);
	    
	res->folder_type = NO_NAME;
	res->folder_type->init_it(res);

	break;

    }

    if (passhm)
	free_browser_passhm(&passhm);

    if (rdata)
	hashtype_free_rdata(&rdata);	
    if (status && res)
	set_remote_tempmbox(res,status);

    return res;
}

static int hashtype_remote_lookup_remote(item,se,lookup_flags,username,hostname,addr,
					 useraddr_flags,cancel_p)
     const struct hashmark_item * item;
     struct service_entry      ** se;
     int                          lookup_flags;     
     char                      ** username;
     char                      ** hostname;
     struct address            ** addr;
     int                        * useraddr_flags;
     struct cancel_data         ** cancel_p /* May be NULL,
					       Used if dns lookup was
					       cancelable */;
{
    struct hashmark_remote * item_r ;
    struct service_entry * Entry = NULL;
    int ret = 1;
    
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_remote_lookup_remote",
              "Bad magic number",0);

    if (HASHMARK_REMOTE_magic != item->u.remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_remote_lookup_remote",
	      "Bad magic (hashmark_remote, item)",0);
    item_r = item->u.remote;

    if (se || hostname || lookup_flags) {
	Entry = htrem_give_service_entry(item_r,lookup_flags,
					 cancel_p);
	ret = Entry != NULL;
    }

    if (username || addr) {
	const char * u;

	if (addr && *addr)
	    free_address(addr);

	u = htrem_get_username(item,addr);
	if (!u)
	    ret = 0;
	if (username) {
	    *username = strmcpy(*username,u);
	}

    }

    if (hostname) {
	if (Entry)
	    *hostname = strmcpy(*hostname,Entry->official_name);
	else if (item_r->hostname)
	    *hostname = strmcpy(*hostname,item_r->hostname);
    }

    if (se) {
	if (*se)
	    free_service_entry(se);
	*se = Entry;
    } else /* Entry can be NULL */
	free_service_entry(&Entry);

    if (useraddr_flags) {
	*useraddr_flags = 0;

	if (item_r->self_cc)
	    *useraddr_flags |= USERADDR_SELF_CC;

	if (item_r->self_bcc)
	    *useraddr_flags |= USERADDR_SELF_BCC;

    }
    
    DPRINT(Debug,15,(&Debug,"hashtype_remote_lookup_remote=%d%s\n",
		     ret,
		     ret ? "" : " (failed)"));
    
    return ret;
}



static int hashtype_selection_is_remote(item,dir,hashmark_data,folder)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     struct folder_info    * folder;
{
    int ret = 0;
    struct hashmark_rdata  * rdata ;


    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_selection_is_remote",
              "Bad magic number",0);

    if (HASHMARK_RDATA_magic !=  hashmark_data->remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_selection_is_remote",
              "Bad magic number (hashmark_rdata)",0);
    rdata = hashmark_data->remote;

    if (rdata->ra) {
	switch (rdata->have_con) {
	case hmcon_none:

	    DPRINT(Debug,15,(&Debug,"hashtype_selection_is_remote: No connection\n"));
	    
	    ret = 0;
	    break;
	    
	    /* Probably not correct ??? */
	case hmcon_pop:
	    
	    if (folder->folder_type == POP_MBX) {
		ret = same_remote_account_host_user(give_pop_remote_account(folder),
						    rdata->ra);
	    } else {
		DPRINT(Debug,15,(&Debug,"hashtype_selection_is_remote: POP connection, no POP mailbox\n"));
	    }
	    break;
		
	case hmcon_imap: 
	    
	    if (folder->folder_type == IMAP_MBX) {

		ret =  imap_same_host_user_nopath(folder,rdata->ra);

	    } else {
		DPRINT(Debug,15,(&Debug,"hashtype_selection_is_remote: IMAP connection, no IMAP mailbox\n"));
	    }

	    break;
	}
	    
    } else {
	DPRINT(Debug,15,(&Debug,"hashtype_selection_is_remote: No remote account\n"));
    }

    DPRINT(Debug,15,(&Debug,"hashtype_selection_is_remote=%d%s -- not supported\n",
		     ret,
		     ret ? "" : " (failed)"));

    return ret; 
}


static enum haskmark_valid_v hashtype_valid_on_user_remote(item)
    const struct hashmark_item  * item;
{
    enum haskmark_valid_v ret = hashmark_unusable;
    const char * u;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_valid_on_user_remote",
              "Bad magic number",0);

    u = htrem_get_username(item,NULL);
    if (u) {
	DPRINT(Debug,15,(&Debug,"hashtype_valid_on_user_remote: Remote user %s\n",
			 u));
	ret = hashmark_valid;

    }

    DPRINT(Debug,15,(&Debug,"hashtype_valid_on_user_remote=%d",
		     ret));

    switch(ret) {
    case hashmark_unusable: DPRINT(Debug,15,(&Debug," hashmark_unusable")); break;
    case hashmark_valid:    DPRINT(Debug,15,(&Debug," hashmark_valid")); break;
    case hashmark_default_menu: DPRINT(Debug,15,(&Debug," hashmark_default_menu")); break;
    case hashmark_hidden:   DPRINT(Debug,15,(&Debug," hashmark_hidden")); break;
    }

    DPRINT(Debug,15,(&Debug,"\n"));


    return ret;

}

/* Return 1 if succeed, changes browser type */
static int hashtype_prepare_write_remote(item,dir,hashmark_data,ptr)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     WRITE_STATE             ptr;
{
    int ret = 0;
    
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_prepare_write_remote",
              "Bad magic number",0);
    
    switch (htrem_have_connection(item,hashmark_data,
				  1 /* Need browser */)) {
	
    case hmcon_none:
	
	DPRINT(Debug,15,(&Debug,
			 "hashtype_prepare_write_remote: No connection\n"));
	
	ret = 0;
	break;
	
    case hmcon_pop:
	DPRINT(Debug,15,(&Debug,
			 "hashtype_prepare_write_remote: POP connection (not supported)\n"));	    
	ret = 0;
	break;
	
    case hmcon_imap:
	DPRINT(Debug,15,(&Debug,
			 "hashtype_prepare_write_remote: IMAP connection\n"));
	
	/*   NOTE:  union hashmark_data   is not valid after change_folder_browser_type
	     is called !!!!
	*/
	
	if (hashtype_br_change_imap(item,hashmark_data,dir)) {

	    ret = dir->type->browser_prepare_write_it(dir,ptr);

	    if (!ret) {
		DPRINT(Debug,15,(&Debug,"hashtype_prepare_write_remote: browser_prepare_write_it FAILED\n"));	
	    }
	    
	} else {
	    
	    DPRINT(Debug,16,(&Debug,
			     "hashtype_prepare_write_remote: failed to change browser type to real type\n"));
	    
	}
		
	break;
    }
        
    DPRINT(Debug,15,(&Debug,
		     "hashtype_prepare_write_remote=%d\n",ret));
    return ret;
}



/* Return directory prefix and l_item,
   changes browser type
*/
static  struct string * hashtype_cat_hashmark_remote(h_item,dir,hashmark_data,l_item)
     struct hashmark_item  * h_item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     const struct string   * l_item;
{
    struct string * ret = NULL; 
    struct hashmark_rdata  * rdata ;
    
    if (HASHMARK_ITEM_magic != h_item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_cat_hashmark_remote",
              "Bad magic number",0);

    if (HASHMARK_RDATA_magic !=  hashmark_data->remote->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_cat_hashmark_remote",
              "Bad magic number (hashmark_rdata)",0);
    rdata = hashmark_data->remote;

    if (rdata->ra) {
	switch (rdata->have_con) {
	case hmcon_none:
	    DPRINT(Debug,15,(&Debug,"hashtype_cat_hashmark_remote: No connection\n"));

	    goto fallback;

	case hmcon_pop:
	    DPRINT(Debug,15,(&Debug,"hashtype_cat_hashmark_remote: POP connection\n"));

	    goto fallback;

	case hmcon_imap:
	    DPRINT(Debug,15,(&Debug,"hashtype_cat_hashmark_remote: IMAP connection\n"));

	    /*   NOTE:  union hashmark_data   is not valid after change_folder_browser_type
		 is called !!!!
	    */
	
	    if (hashtype_br_change_imap(h_item,hashmark_data,dir)) {

		ret = dir->type->browser_cat_it(dir,l_item);

		if (!ret) {
		    DPRINT(Debug,15,(&Debug,
				     "hashtype_cat_hashmark_remote: real browser_cat_it FAILED\n"));
		    goto fallback;
		}	
	    } else {
		DPRINT(Debug,15,(&Debug,
				 "hashtype_cat_hashmark_remote: Not changed to IMAP\n"));
		goto fallback;
	    }
	    break;
	}

    } else {
	DPRINT(Debug,15,(&Debug,
			 "hashtype_cat_hashmark_remote: No remote account\n"));

    fallback:
	if (h_item->hashmark_name) {
	    DPRINT(Debug,15,(&Debug,
			     "hashtype_cat_hashmark_remote: Using hashmark name %S\n",
			     h_item->hashmark_name));
	    
	    if (l_item)
		ret = format_string(FRM("#%S:%S"),
				    h_item->hashmark_name,
				    l_item);
	    else
		ret = format_string(FRM("#%S"),
				    h_item->hashmark_name);
	}
    }
	    
    if (ret) {
	DPRINT(Debug,15,(&Debug,
			 "hashtype_cat_hashmark_remote=%S\n",
			 ret));
    } else {
	DPRINT(Debug,15,(&Debug,
			 "hashtype_cat_hashmark_remote=NULL\n",
			 ret));

    }
    
    return ret;
}


void free_usermap_files()
{
    if (usermap_files) {
	int i;

	for (i = 0; i < usermap_files_count; i++) {
	    
	    free_hashmark_usermap_file(& (usermap_files[i]));

	}

	free(usermap_files);
	usermap_files = NULL;
    }
    usermap_files_count = 0;

}

#else

void free_usermap_files()
{

}

#endif




/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
