static char rcsid[] = "@(#)$Id: hashmark_browser.c,v 2.5 2022/02/20 17:24:01 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.5 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mbox.h"
#include "s_me.h"
#include "hashmark.h"

DEBUG_VAR(Debug,__FILE__,"mbox");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}



#define HASHMARK_BROWSER_magic	0xF520

struct HASHMARK_BROWSER {
    unsigned short            magic;  /* HASHMARK_BROWSER_magic */

    struct hashmark_item   * selected_hashmark;
    union hashmark_data      hashmark_data;    
};

S_(browser_zero_dir browser_zero_hashmark)
static  void browser_zero_hashmark P_((struct folder_browser *dir));
static  void browser_zero_hashmark(dir) 
     struct folder_browser *dir;
{
    DPRINT(Debug,11,(&Debug,"browser_zero_hashmark: dir=%p\n", dir));
    
    dir->a.hashmark_browser = safe_zero_alloc(sizeof (* (dir->a.hashmark_browser)));

    dir->a.hashmark_browser->magic = HASHMARK_BROWSER_magic;
    dir->a.hashmark_browser->selected_hashmark = NULL;
    dir->a.hashmark_browser->hashmark_data.dummy = NULL;

}


S_(browser_free_dir browser_free_hashmark)
static void browser_free_hashmark P_((struct folder_browser *dir));
static void browser_free_hashmark(dir)
     struct folder_browser *dir;
{
    DPRINT(Debug,11,(&Debug,"browser_free_hashmark: dir=%p\n", dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_free_hashmark",
	      "Bad magic type",0);

    if (dir->a.hashmark_browser->selected_hashmark) {
	DPRINT(Debug,11,(&Debug, "browser_free_hashmark: removing hashmark\n"));
	
	if (dir->a.hashmark_browser->hashmark_data.dummy)
	    hashmark_free_data(dir->a.hashmark_browser->selected_hashmark,
			       & (dir->a.hashmark_browser->hashmark_data));

	free_hashmark_item(& (dir->a.hashmark_browser->selected_hashmark));

    } else if (dir->a.hashmark_browser->hashmark_data.dummy) {
	
	DPRINT(Debug,1,(&Debug, 
			"browser_free_hashmark: hasmark_data without selected_hashmark\n"));
	free (dir->a.hashmark_browser->hashmark_data.dummy);
	dir->a.hashmark_browser->hashmark_data.dummy = NULL;
       
    }

    dir->a.hashmark_browser->magic = 0; /* invalidate */
    free(dir->a.hashmark_browser);
    dir->a.hashmark_browser = NULL;
}

static int b_hashmark_update_selected P_((struct folder_browser *dir,
					  const struct string *rel_dirname,
					  struct string **tail));
static int b_hashmark_update_selected(dir,rel_dirname,tail)
     struct folder_browser *dir;
     const struct string *rel_dirname;
     struct string **tail;
{
    int ret = 0;

    if (tail && *tail)
	free_string(tail);

    if (dir->a.hashmark_browser->selected_hashmark) {
	const struct string * N = 
	    hashmark_item_name(dir->a.hashmark_browser->selected_hashmark);
	
	if (N) {
	    int L = string_have_prefix(rel_dirname,N);

	    DPRINT(Debug,11,(&Debug,"b_hashmark_update_selected: selected hashmark %S\n",
			     N));
	    
	    if (L > 0) {
		int L1 = string_len(rel_dirname);	    

		if (L < L1) {   /* Look seperator */
		    uint16 code = give_unicode_from_string(rel_dirname,L);

		    switch (code) {
			int POS;

		    case 0x003A /* : */:
			ret = 1;
			POS = L+1;
			if (tail && L1 > L)
			    *tail = clip_from_string(rel_dirname,&POS,L1);
			break;
		    case 0x002F /* / */:
			ret = 1;
			POS = L;  /* / is included to tail */
			if (tail)
			    *tail = clip_from_string(rel_dirname,&POS,L1);
			break;
		    default:	  /* This error is likely not reached */	       
			lib_error(CATGETS(elm_msg_cat, MeSet, MeHMbadSeparatorAfterHashmark,
					  "Bad separator after hashmark %S."),
				  N);
		    }

		} else {
		    DPRINT(Debug,11,(&Debug, "b_hashmark_update_selected: No separator\n"));
		    ret = 1;
		}
		
	    } else {
		DPRINT(Debug,11,(&Debug, "b_hashmark_update_selected: hashmark no match or empty hashmark name\n"));
		goto fail;
	    }

	} else {
	    DPRINT(Debug,11,(&Debug, "b_hashmark_update_selected: selected hashmark do not have name!\n"));
	    goto fail;
	}
	
    } else {
	int idx;

    fail:
	if (dir->a.hashmark_browser->selected_hashmark) {
	    DPRINT(Debug,11,(&Debug, "b_hashmark_update_selected: changing hashmark\n"));

	    if (dir->a.hashmark_browser->hashmark_data.dummy)
		hashmark_free_data(dir->a.hashmark_browser->selected_hashmark,
				   & (dir->a.hashmark_browser->hashmark_data));

	    free_hashmark_item(& (dir->a.hashmark_browser->selected_hashmark));
	}


	idx = string_have_ascii_sep(rel_dirname,
				    hashmark_RESERVED_SEPARATOR);
	if (idx < 0) {
	    DPRINT(Debug,11,(&Debug, "b_hashmark_update_selected: No separator\n"));

	    dir->a.hashmark_browser->selected_hashmark = 
		search_hashmark_name(rel_dirname,dir->sel_type);

	    if (dir->a.hashmark_browser->selected_hashmark) {
		DPRINT(Debug,11,(&Debug, "b_hashmark_update_selected: Found hashmark %S\n",
				 rel_dirname));
		ret = 1;

		hashmark_init_data(dir->a.hashmark_browser->selected_hashmark,
				   & (dir->a.hashmark_browser->hashmark_data));
	    }

	} else if (idx > 0) {    /* Look seperator */
	    int POS = 0;
	    struct string * name =  clip_from_string(rel_dirname,&POS,idx);
	    uint16 code = give_unicode_from_string(rel_dirname,idx);
	    int L1 = string_len(rel_dirname);	    

	    switch (code) {

	    case 0x003A /* : */:
		ret = 1;
		POS = idx+1;
		if (tail && L1 > idx)
		    *tail = clip_from_string(rel_dirname,&POS,L1);
		break;
	    case 0x002F /* / */:
		ret = 1;
		POS = idx;  /* / is included to tail */
		if (tail)
		    *tail = clip_from_string(rel_dirname,&POS,L1);
		break;
	    default:	  /* This error is likely not reached */	       
		lib_error(CATGETS(elm_msg_cat, MeSet, MeHMbadSeparatorAfterHashmark,
				  "Bad separator after hashmark %S."),
			  name);
	    }

	    if (ret) {

		dir->a.hashmark_browser->selected_hashmark = 
		    search_hashmark_name(name,dir->sel_type);

		if (dir->a.hashmark_browser->selected_hashmark) {
		    DPRINT(Debug,11,(&Debug, "b_hashmark_update_selected: Found hashmark %S\n",
				     name));

		    hashmark_init_data(dir->a.hashmark_browser->selected_hashmark,
				       & (dir->a.hashmark_browser->hashmark_data));

		} else
		    ret = 0;		
	    }

	    free_string(&name);

	} else {
	    DPRINT(Debug,11,(&Debug, "b_hashmark_update_selected: Empty hashmark\n"));
	}	    
    }

    DPRINT(Debug,11,(&Debug, "b_hashmark_update_selected=%d",ret));

    if (tail && *tail) {
	DPRINT(Debug,11,(&Debug, "; tail %S",
			 *tail));
    }

    DPRINT(Debug,11,(&Debug, "\n"));

    return ret;
}

void set_hashmark(entry,item)
     struct name_vector    * entry;
     struct hashmark_item  * item;
{
    if (entry->a.hashmark_item)
	free_hashmark_item(& (entry->a.hashmark_item));

    entry->a.hashmark_item = item;
    inc_hashmark_item_refcount(entry->a.hashmark_item);
}

static void b_hashmark_default_listing P_((struct folder_browser *dir));
static void b_hashmark_default_listing(dir)
     struct folder_browser *dir;
{
    int len = 0;
    struct hashmark_item **vector;

    DPRINT(Debug,11,(&Debug,"b_hashmark_default_listing: dir=%p\n", dir));
    
    if (dir->dirname)
	free_string(&dir->dirname);
    dir->dirname =  format_string(FRM("#"));

    if (dir->sys_dir) {
	free(dir->sys_dir);
	dir->sys_dir = NULL;
    }


    vector =  hashmark_dir(&len,dir->sel_type);

    /* Make empty selection folder */
    clear_dir_vector(dir);
 
    if (vector) {
	int i;
	
	for (i = 0; i < len; i++) {
	    if (hashmark_visible(vector[i])) {
		const struct string * N = 
		    hashmark_item_name(vector[i]);
		
		/* WARNING: add_dir_vector does not allocate strings -- 
		 *          it just assign pointers!
		 */
		
		set_hashmark(add_dir_vector(dir,
					    /* Probably 'wrong' charset */
					    us2s(stream_from_string(N,1,NULL)),
					    dup_string(N),
					    hashmark_browser_flags(vector[i])),
			     vector[i]);
		
		free_hashmark_item(& (vector[i]));
	    }
	}

	free(vector);
	vector = NULL;
    }
}

S_(browser_change_dir browser_change_hashmark)
static int browser_change_hashmark P_((struct folder_browser *dir,
				       const struct string *rel_dirname,
				       struct string **dispname));
static int browser_change_hashmark(dir,rel_dirname,dispname)
     struct folder_browser *dir;
     const struct string *rel_dirname;
     struct string **dispname;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_change_hashmark: dir=%p\n", dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_change_hashmark",
	      "Bad magic type",0);

    /* rel_dirname is name after # -character */
    
    if (rel_dirname) {
	struct string * tail = NULL;

	ret = b_hashmark_update_selected(dir,rel_dirname,&tail);
	
	if (ret) {

	    if (dir->a.hashmark_browser->selected_hashmark) {

		ret = hashmark_changebr_dir(dir->a.hashmark_browser->selected_hashmark,
					    dir,
					    & (dir->a.hashmark_browser->hashmark_data),
					    tail,dispname);

	    } else {
		DPRINT(Debug,11,(&Debug, 
				 "browser_change_hashmark: Hashmark not selected?\n"));
		ret = 0;
	    }
	}

	if (tail)
	    free_string(&tail);
    } else {
	DPRINT(Debug,11,(&Debug, 
			 "browser_change_hashmark:  To default directory\n"));  
	
	if (dir->a.hashmark_browser->selected_hashmark) {
	    DPRINT(Debug,11,(&Debug, "browser_change_hashmark: removing hashmark\n"));

	    if (dir->a.hashmark_browser->hashmark_data.dummy)
		hashmark_free_data(dir->a.hashmark_browser->selected_hashmark,
				   & (dir->a.hashmark_browser->hashmark_data));
	    
	    
	    free_hashmark_item(& (dir->a.hashmark_browser->selected_hashmark));
	} else if (dir->a.hashmark_browser->hashmark_data.dummy) {
	    
	    DPRINT(Debug,1,(&Debug, 
			    "browser_change_hashmark: hasmark_data without selected_hashmark\n"));
	    free (dir->a.hashmark_browser->hashmark_data.dummy);
	    dir->a.hashmark_browser->hashmark_data.dummy = NULL;	    
	}
	
	b_hashmark_default_listing(dir);
	ret = 1;
    }

    DPRINT(Debug,11,(&Debug, "browser_change_hashmark=%d\n", ret));
    return ret;
}


S_(browser_select_dir browser_select_hashmark)
static int browser_select_hashmark P_((struct folder_browser *dir,
				       const struct string *rel_itemname,
				       struct string **dispname,
				       int *newpos));
static int browser_select_hashmark(dir,rel_itemname,dispname,newpos)
     struct folder_browser *dir;
     const struct string *rel_itemname;
     struct string **dispname;
     int *newpos;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_select_hashmark: dir=%p\n", dir));


    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_select_hashmark",
	      "Bad magic type",0);

    /* rel_itemname is name after # -character */

    if (rel_itemname) {
	struct string * tail = NULL;

	ret = b_hashmark_update_selected(dir,rel_itemname,&tail);
	
	if (ret) {

	    if (dir->a.hashmark_browser->selected_hashmark) {

		ret = hashmark_selectbr_item(dir->a.hashmark_browser->selected_hashmark,
					     dir,
					     & (dir->a.hashmark_browser->hashmark_data),
					     tail,dispname,newpos);

	    } else {
		DPRINT(Debug,11,(&Debug, 
				 "browser_select_hashmark: Hashmark not selected?\n"));
		ret = 0;
	    }
	}

	if (tail)
	    free_string(&tail);
	
    } else {
	DPRINT(Debug,11,(&Debug, 
			 "browser_select_hashmark:  Default directory not selectable\n"));  
    }

    DPRINT(Debug,11,(&Debug, "browser_select_hashmark=%d\n", ret));
    return ret;
}


int hashmark_browser_passthru_change(dir,item,dispname)
     struct folder_browser *dir;
     struct hashmark_item *item;
     struct string **dispname;
{
    int ret = 0;

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"hashmark_browser_passthru_change",
	      "Bad magic type",0);

    if (item != dir->a.hashmark_browser->selected_hashmark) {
	
	if (dir->a.hashmark_browser->selected_hashmark) {
	    DPRINT(Debug,11,(&Debug,"hashmark_browser_passthru_change: changing selected hashmark\n"));
	    
	    if (dir->a.hashmark_browser->hashmark_data.dummy)
		hashmark_free_data(dir->a.hashmark_browser->selected_hashmark,
				   & (dir->a.hashmark_browser->hashmark_data));
	    
	    free_hashmark_item(& (dir->a.hashmark_browser->selected_hashmark));
	    
	} else {
	    DPRINT(Debug,11,(&Debug,"hashmark_browser_passthru_change: setting selected hashmark\n"));
	}
	
	
	dir->a.hashmark_browser->selected_hashmark = item;
	inc_hashmark_item_refcount(dir->a.hashmark_browser->selected_hashmark);
	
	hashmark_init_data(dir->a.hashmark_browser->selected_hashmark,
			   & (dir->a.hashmark_browser->hashmark_data));
    }

    if (dir->a.hashmark_browser->selected_hashmark) {

	ret = hashmark_changebr_dir(dir->a.hashmark_browser->selected_hashmark,
				    dir,
				    & (dir->a.hashmark_browser->hashmark_data),
				    NULL,dispname);
    } else {
	DPRINT(Debug,11,(&Debug, 
			 "hashmark_browser_passthru_change: Hashmark not selected?\n"));
	ret = 0;
    }
    
    DPRINT(Debug,11,(&Debug, "hashmark_browser_passthru_change=%d\n",
		     ret));
    
    return ret;	
}


S_(browser_change_v_dir browser_change_v_hashmark)
static int browser_change_v_hashmark P_((struct folder_browser *dir,
				      struct name_vector *X,
				      struct string **dispname));
static int browser_change_v_hashmark(dir,X,dispname)
     struct folder_browser *dir;
     struct name_vector *X;
     struct string **dispname;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_change_v_hashmark: dir=%p\n", dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_change_v_hashmark",
	      "Bad magic type",0);

    if (X->a.hashmark_item) {
	
	ret = hashmark_browser_passthru_change(dir,X->a.hashmark_item,dispname);


    } else {
	DPRINT(Debug,11,(&Debug, "browser_change_v_hashmark: No hashmark_item\n"));
    }
    
    DPRINT(Debug,11,(&Debug, "browser_change_v_hashmark=%d\n",
		     ret));

    return ret;
}

/* return

   > 0  :    call change_dir with new value of *dispname
             *disp_tail should be current tail part after *dispname

*/

S_(browser_change_up_dir browser_change_up_hashmark)
static int browser_change_up_hashmark P_((struct folder_browser *dir,
				       struct string **dispname,
				       struct string ** disp_tail));
static int browser_change_up_hashmark(dir,dispname,disp_tail)
     struct folder_browser *dir;
     struct string **dispname;
     struct string ** disp_tail;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_change_up_hashmark: dir=%p\n", dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_change_up_hashmark",
	      "Bad magic type",0);

    if (dir->a.hashmark_browser->selected_hashmark) {

	DPRINT(Debug,11,(&Debug,"browser_change_up_hashmark:  To default directory\n"));  

	if (disp_tail) {
	    const struct string * N = 
		hashmark_item_name(dir->a.hashmark_browser->selected_hashmark);
	    if (N)
		*disp_tail = dup_string(N);
	}
	
	free_hashmark_item(& (dir->a.hashmark_browser->selected_hashmark));

	if (*dispname)
		free_string(dispname);	    
	*dispname = format_string(FRM("#"));

	/* browser_change_hashmark will update dir->dirname */
	ret = 1;

    } else {
	/* Just return default menu */
	ret = -1;
    }

    DPRINT(Debug,11,(&Debug,"browser_change_up_hashmark=%d", ret));
    if (ret && *dispname) {
	DPRINT(Debug,11,(&Debug,"; *dispname=%S",*dispname));
    }
    if (disp_tail && *disp_tail) {
	DPRINT(Debug,11,(&Debug,"; *disp_tail=%S",*disp_tail));
    }
    DPRINT(Debug,11,(&Debug,"\n"));

    return ret;   
}


S_(browser_give_title_dir browser_give_title_hashmark)
static struct string * browser_give_title_hashmark P_((struct folder_browser *dir));
static struct string * browser_give_title_hashmark(dir)
     struct folder_browser *dir;
{
    static struct string *ret = NULL;

    DPRINT(Debug,11,(&Debug,"browser_give_title_hashmark: dir=%p\n", dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_give_title_hashmark",
	      "Bad magic type",0);

    if (dir->filter)
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeHashmarkDirFilter,
				    "Directory listing for hashmarks with filter %S"),
			    dir->filter);
    
    else
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeHashmarkDir,
				    "Directory listing for hashmarks"));
			    

    DPRINT(Debug,11,(&Debug,"browser_give_title_hashmark=%S\n", ret));
    
    return ret;
}

S_(browser_separator_dir browser_separator_hashmark)
static char browser_separator_hashmark P_((struct folder_browser *dir));
static char browser_separator_hashmark(dir)
     struct folder_browser *dir;
{

    DPRINT(Debug,11,(&Debug,"browser_seperator_hashmark: dir=%p\n", dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_separator_hashmark",
	      "Bad magic type",0);


    /* XXXX is this correct? */

    return ':';
}

S_(browser_name_dir browser_name_hashmark)
static struct string * browser_name_hashmark P_((struct folder_browser *dir));
static struct string * browser_name_hashmark(dir)
     struct folder_browser *dir;
{
    static struct string *ret = NULL;

    DPRINT(Debug,11,(&Debug,"browser_name_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_name_hashmark",
	      "Bad magic type",0);

    if (dir->filter)
	ret = format_string(FRM("Directory listing for hashmarks with filter %S"),
			    dir->filter);
    else
	ret = format_string(FRM("Directory listing for hashmarks"));


    DPRINT(Debug,11,(&Debug,"browser_name_hashmark=%S\n", ret));

    return ret;
}


S_(browser_cat_dir browser_cat_hashmark)
static struct string * browser_cat_hashmark P_((struct folder_browser *dir,
						const struct string * item));
static struct string * browser_cat_hashmark(dir,item)
     struct folder_browser *dir;
     const struct string * item;
{
    struct string * ret = NULL;

    DPRINT(Debug,11,(&Debug,"browser_cat_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_cat_hashmark",
	      "Bad magic type",0);

    /* IS THIS CORRECT? */

     if (dir->a.hashmark_browser->selected_hashmark) {

	 const struct string * N = 
	     hashmark_item_name(dir->a.hashmark_browser->selected_hashmark);

	 if (N) {
	     DPRINT(Debug,11,(&Debug,"browser_cat_hashmark: selected hashmark %S\n",
			      N));
	 }
	 
	 /* Return directory prefix and l_item,
	    changes browser type
	 */
	 ret = hashmark_cat_item(dir->a.hashmark_browser->selected_hashmark,
				 dir,
				 & (dir->a.hashmark_browser->hashmark_data),
				 item);

	 if (ret) {
	     DPRINT(Debug,11,(&Debug,"browser_cat_hashmark: hashmark_cat_item succeed %S\n",
			      ret));
	 } else {
	     				 
	     DPRINT(Debug,11,(&Debug,"browser_cat_hashmark: hashmark_cat_item failed! \n"));
	 }
	 
	 
     } else {
	 ret = format_string(FRM("#"));   

	 if (ret) {
	     DPRINT(Debug,11,(&Debug,"browser_cat_hashmark: directory prefix: %S\n",ret));
	     
	     if (item)  {
		 DPRINT(Debug,11,(&Debug,"browser_cat_hashmark: item %S\n",
				  item));
		 
		 append_string(&ret,item,0);
	     }    
	 }
     }


     if (ret) {
	 DPRINT(Debug,11,(&Debug,"browser_cat_hashmark=%S\n",ret));
     } else {
	 DPRINT(Debug,11,(&Debug,"browser_cat_hashmark=NULL\n"));
     }

     return ret;
}

S_(browser_folder_from_dir browser_folder_from_hashmark)
static struct folder_info * 
browser_folder_from_hashmark P_((struct folder_browser *dir,
				 int treat_as_spooled));

static struct folder_info * browser_folder_from_hashmark(dir, treat_as_spooled)
     struct folder_browser *dir;
     int treat_as_spooled;
{
    struct folder_info   * res     = NULL;

    DPRINT(Debug,11,(&Debug,"browser_folder_from_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_folder_from_hashmark",
	      "Bad magic type",0);

    if (dir->a.hashmark_browser->selected_hashmark) {

	 const struct string * N = 
	     hashmark_item_name(dir->a.hashmark_browser->selected_hashmark);

	 if (N) {
	     DPRINT(Debug,11,(&Debug,"browser_folder_from_hashmark: selected hashmark %S\n",
			      N));
	 }
	 
	res = hashmark_folder_from_browser(dir->a.hashmark_browser->selected_hashmark,
					   dir,
					   & (dir->a.hashmark_browser->hashmark_data),
					   treat_as_spooled);

    } else {
	DPRINT(Debug,11,(&Debug, 
			 "browser_folder_from_hashmark: Hashmark not selected?\n"));
    }

    if (res) {
	DPRINT(Debug,11,(&Debug,"browser_folder_from_hashmark=%p\n",res));
    } else {
	DPRINT(Debug,11,(&Debug,"browser_folder_from_hashmark=NULL\n"));
    }
    return res;
}

S_(browser_create_selection_dir browser_create_selection_hashmark)
static int browser_create_selection_hashmark P_((struct folder_browser *dir));
static int browser_create_selection_hashmark(dir)
     struct folder_browser *dir;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,
		     "browser_create_selection_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_create_selection_hashmark",
	      "Bad magic type",0);

    if (dir->a.hashmark_browser->selected_hashmark) {
	
	const struct string * N = 
	    hashmark_item_name(dir->a.hashmark_browser->selected_hashmark);

	if (N) 	{
	    switch(dir->sel_type) {
	    case selection_folder:
		lib_error(CATGETS(elm_msg_cat, MeSet, MeCreatFolderHashmarkNotSupp,
				  "Creating of folder for hashmark %S not supported"),
			  N);
		break;
	    case selection_file:
		lib_error(CATGETS(elm_msg_cat, MeSet, MeCreatFileFHashmarkNotSupp,
				  "Creating of file for hashmark %S not supported"),
			  N);
		break;
	    case selection_url:
	    case NUM_selection_type:
		/* Not used */
		break;
	    }
	}

    } else {
	DPRINT(Debug,11,(&Debug, 
			 "browser_create_selection_hashmark: Hashmark not selected?\n"));
    }

    DPRINT(Debug,11,(&Debug,"browser_create_selection_hashmark=%d\n", ret));
    
    return ret;
}

S_(zero_ws_fields_browser zero_ws_fields_hashmark)
static void zero_ws_fields_hashmark P_((WRITE_STATE ptr));
static void zero_ws_fields_hashmark(ptr)
     WRITE_STATE ptr;
{

}


S_(free_ws_fields_browser free_ws_fields_hashmark)
static void free_ws_fields_hashmark P_((WRITE_STATE ptr));
static void free_ws_fields_hashmark(ptr)
     WRITE_STATE ptr;
{

}

S_(browser_prepare_write_dir browser_prepare_write_hashmark)
static int browser_prepare_write_hashmark P_((struct folder_browser *dir,
					   WRITE_STATE ptr));
static int browser_prepare_write_hashmark(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_prepare_write_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_prepare_write_hashmark",
	      "Bad magic type",0);

    if (dir->a.hashmark_browser->selected_hashmark) {
	
	/* Return 1 if succeed, changes browser type */
	ret = hashmark_prepare_write(dir->a.hashmark_browser->selected_hashmark,
				     dir,
				     & (dir->a.hashmark_browser->hashmark_data),
				     ptr);	
    } else {
	DPRINT(Debug,11,(&Debug,"browser_prepare_write_hashmark: Hashmark not selected?\n"));
    }
    

    
    DPRINT(Debug,11,(&Debug,"browser_prepare_write_hashmark=%d\n", ret));

    return ret;
}

S_(browser_end_write_dir browser_end_write_hashmark)
static int browser_end_write_hashmark P_((struct folder_browser *dir,
			      WRITE_STATE ptr));
static int browser_end_write_hashmark(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_end_write_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_end_write_hashmark",
	      "Bad magic type",0);

    DPRINT(Debug,11,(&Debug,"browser_end_write_hashmark=%d\n", ret));
    
    return ret;
}

S_(browser_sync_write_dir browser_sync_write_hashmark)
static int browser_sync_write_hashmark P_((struct folder_browser *dir,
					WRITE_STATE ptr));
static int browser_sync_write_hashmark(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_sync_write_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_sync_write_hashmark",
	      "Bad magic type",0);

    DPRINT(Debug,11,(&Debug,"browser_sync_write_hashmark=%d (unsupported)\n", 
		     ret));
    
    return ret;
}

/* Returns -1L on failure */
S_(browser_tell_dir_ws browser_tell_hashmark_ws)
static long browser_tell_hashmark_ws P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr));
static long browser_tell_hashmark_ws(dir,write_state_ptr)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
{
    long ret = -1L;

    DPRINT(Debug,11,(&Debug,"browser_tell_hashmark_ws: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_tell_hashmark_ws",
	      "Bad magic type",0);

    DPRINT(Debug,11,(&Debug,"browser_tell_hashmark_ws=%ld\n",ret));
    return ret;
}

S_(browser_seek_dir_ws browser_seek_hashmark_ws)
/* Returns 0 on failure */
static int browser_seek_hashmark_ws P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr,
				      long pos));
static int browser_seek_hashmark_ws(dir,write_state_ptr,pos)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     long pos;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_seek_hashmark_ws: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_seek_hashmark_ws",
	      "Bad magic type",0);

    DPRINT(Debug,11,(&Debug,"browser_seek_hashmark_ws=%d\n",ret));
    return ret;
}

/* Return 0 on failure */
S_(browser_write_dir_ws browser_write_hashmark_ws)
static int browser_write_hashmark_ws P_((struct folder_browser *dir,
				      WRITE_STATE ptr,
				      int l, const char *buffer));
static int browser_write_hashmark_ws(dir,ptr,l,buffer)
     struct folder_browser *dir;
     WRITE_STATE ptr;
     int l; 
     const char *buffer;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_write_hashmark_ws: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_write_hashmark_ws",
	      "Bad magic type",0);

    DPRINT(Debug,11,(&Debug,"browser_write_hashmark_ws=%d\n",ret));
    return ret;
}

S_(browser_start_we_dir browser_start_we_hashmark)
static int browser_start_we_hashmark P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr,
				      int write_envelope,
				      struct header_rec *current_header,
				      int *env_flags));
static int browser_start_we_hashmark(dir,write_state_ptr,
				   write_envelope,current_header,
				   env_flags)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
     int *env_flags;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_start_we_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_start_we_hashmark",
	      "Bad magic type",0);

    DPRINT(Debug,11,(&Debug,"browser_start_we_hashmark=%d\n",ret));
    return ret;
}

S_(browser_end_we_dir browser_end_we_hashmark)
static int browser_end_we_hashmark P_((struct folder_browser *dir,
				  WRITE_STATE write_state_ptr,
				  int write_envelope,
				  struct header_rec *current_header));
static int browser_end_we_hashmark(dir,write_state_ptr,
			      write_envelope,current_header)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_end_we_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_end_we_hashmark",
	      "Bad magic type",0);

    DPRINT(Debug,11,(&Debug,"browser_end_we_hashmark=%d\n",ret));
    return ret;
}

S_(browser_selection_is_folder browser_selection_is_hashmark)
static int browser_selection_is_hashmark P_((struct folder_browser *dir,
					     struct folder_info *folder));

static int browser_selection_is_hashmark(dir,folder)
     struct folder_browser *dir;
     struct folder_info *folder;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_selection_is_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_selection_is_hashmark",
	      "Bad magic type",0);

    if (dir->a.hashmark_browser->selected_hashmark) {

	ret = hashmark_selection_is_folder(dir->a.hashmark_browser->selected_hashmark,
					   dir,
					   & (dir->a.hashmark_browser->hashmark_data),
					   folder);

    } else {
	DPRINT(Debug,11,(&Debug, 
			 "browser_selection_is_hashmark: Hashmark not selected?\n"));
    }


     DPRINT(Debug,11,(&Debug,
		     "browser_selection_is_hashmark=%d\n",ret));
    return ret;
}

S_(browser_make_ref_folder browser_make_ref_hashmark)
static int browser_make_ref_hashmark P_((struct folder_browser *dir,
				      char **refname, int *iscopy,
				      int is_text));
static int browser_make_ref_hashmark(dir,refname,iscopy,is_text)
     struct folder_browser *dir;
     char **refname; 
     int *iscopy;
     int is_text;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_make_ref_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_make_ref_hashmark",
	      "Bad magic type",0);

    /* Unsupported */

    DPRINT(Debug,11,(&Debug,"browser_make_ref_hashmark=%d\n",ret));
    
    return ret;
}

S_(browser_update_dir browser_update_hashmark)
static void browser_update_hashmark P_((struct folder_browser *dir));
static void browser_update_hashmark(dir)
     struct folder_browser *dir;
{

    DPRINT(Debug,11,(&Debug,"browser_update_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_update_hashmark",
	      "Bad magic type",0);


    clear_dir_vector(dir);
    b_hashmark_default_listing(dir);

}

/* Call only if BROWSER_NEEDSTAT is set */
S_(browser_do_stat browser_hashmark_do_stat)
static void browser_hashmark_do_stat P_((struct folder_browser *dir,
				      int idx));
static void browser_hashmark_do_stat(dir,idx)
     struct folder_browser *dir;
     int idx;
{
    char * entryname UNUSED_VAROK;
    int flags        UNUSED_VAROK;

    DPRINT(Debug,13,(&Debug,
		     "browser_hashmark_do_stat: dir=%p, idx=%d\n", 
		     dir,idx));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_hashmark_do_stat",
	      "Bad magic type",0);

    if (idx < 0 || idx >= dir->vector_len)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_hashmark_do_stat",
	      "Bad index",0);

    flags     = dir->vector[idx].flags;
    entryname = dir->vector[idx].sys_name;

    flags &= ~BROWSER_NEEDSTAT;

    
    /* Not used */
    

    DPRINT(Debug,13,(&Debug,
		     "browser_hashmark_do_stat: changing flags %X -> %X\n",
		     dir->vector[idx].flags,flags));

    dir->vector[idx].flags = flags;

    /* Not supported */
}


static int sort_by_hashmark P_((const void *A, const void *B));
static int sort_by_hashmark(A,B)
     const void *A; 
     const void *B;
{
    const struct name_vector * A1 = A;
    const struct name_vector * B1 = B;

    int r = string_cmp(A1->disp_name,
		       B1->disp_name,
		       0  /* compare failure */);

    if (0 == r)
	r = strcmp(A1->sys_name,B1->sys_name);
   
    if (0 == r) {
	if (A1->a.hashmark_item && B1->a.hashmark_item)
	    r = hashmark_type(B1->a.hashmark_item)
		- hashmark_type(A1->a.hashmark_item);
	else if (B1->a.hashmark_item)
	    r = 1;
	else if (A1->a.hashmark_item)
	    r = -1;
    }

    return r;
}

S_(browser_folder_sort_dir browser_folder_sort_hashmark)
static void browser_folder_sort_hashmark P_((struct folder_browser *dir,
					     print_sort_message * print));
static void browser_folder_sort_hashmark(dir,print)
     struct folder_browser *dir;
     print_sort_message * print;
{

    DPRINT(Debug,11,(&Debug,"browser_folder_sort_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_folder_sort_hashmark",
	      "Bad magic type",0);

    if (dir->vector_len < 2)
	return;
    
    /* No other sorting orders */

    qsort(dir->vector,dir->vector_len,sizeof (dir->vector[0]),
	  sort_by_hashmark);
}

S_(browser_free_name_vector_dir browser_free_name_vector_hashmark)
static void browser_free_name_vector_hashmark P_((struct folder_browser *dir,
					       struct name_vector *entry));
static void browser_free_name_vector_hashmark(dir,entry)
     struct folder_browser *dir;
     struct name_vector *entry;
{
    DPRINT(Debug,11,(&Debug,"browser_free_name_vector_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "browser_free_name_vector_hashmark",
	      "Bad magic type",0);

    if (entry->a.hashmark_item)
	free_hashmark_item(& (entry->a.hashmark_item));

}

/* Increments refcount */
S_(browser_gethm_name_vector_dir browser_gethm_name_vector_hashmark)
static struct hashmark_item *
    browser_gethm_name_vector_hashmark P_((struct folder_browser * dir,
					  struct name_vector    * entry));
static struct hashmark_item *browser_gethm_name_vector_hashmark(dir,entry)
     struct folder_browser * dir;
     struct name_vector    * entry;
{
    DPRINT(Debug,11,(&Debug,"browser_gethm_name_vector_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "browser_gethm_name_vector_hashmark",
	      "Bad magic type",0);
     
    if (entry->a.hashmark_item) {
	inc_hashmark_item_refcount(entry->a.hashmark_item);

	
	DPRINT(Debug,11,(&Debug,"browser_gethm_name_vector_hashmark=%p\n", 
			 entry->a.hashmark_item));

	return entry->a.hashmark_item;
    }

    DPRINT(Debug,11,(&Debug,"browser_gethm_name_vector_hashmark=NULL\n"));

    return NULL;
}
    
/* Returns 1 if hashmark saved
   Increments refcount, if hashmark saved */
S_(browser_sethm_name_vector_dir browser_sethm_name_vector_hashmark)
static int browser_sethm_name_vector_hashmark P_((struct folder_browser * dir,
						 struct name_vector    * entry,
						 struct hashmark_item  * hm
						 ));
static int browser_sethm_name_vector_hashmark(dir,entry,hm)
     struct folder_browser * dir;
     struct name_vector    * entry;
     struct hashmark_item  * hm;
{
    DPRINT(Debug,11,(&Debug,"browser_sethm_name_vector_hashmark: dir=%p\n", 
		     dir));
       
    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "browser_sethm_name_vector_hashmark",
	      "Bad magic type",0);
    
    set_hashmark(entry,hm);

    DPRINT(Debug,11,(&Debug,"browser_sethm_name_vector_hashmark=1\n"));
    
    return 1;
}

static const struct string * browser_line_idx_hashmark 
                      P_((struct folder_browser *dir,
			  struct name_vector *entry,
			  const struct string  **comment_p));
static const struct string * browser_line_idx_hashmark(dir,entry,comment_p)
     struct folder_browser *dir;
     struct name_vector *entry;
     const struct string  **comment_p;
{
    DPRINT(Debug,11,(&Debug,"browser_line_idx_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "browser_line_idx_hashmark",
	      "Bad magic type",0);

    if (entry->a.hashmark_item) {
	const struct string * S = 
	    hashmark_item_name(entry->a.hashmark_item);

	if (comment_p) {
	    *comment_p = 
		hashmark_item_description(entry->a.hashmark_item);

	    DPRINT(Debug,11,(&Debug,
			     "browser_line_idx_hashmark: comment=%S\n",
			     *comment_p));
	}

	if (S) {
	    DPRINT(Debug,11,(&Debug,
			     "browser_line_idx_hashmark=%S\n",
			     S));
	    return S;
	}
    }

    DPRINT(Debug,11,(&Debug,"browser_line_idx_hashmark=NULL\n"));

    return NULL;
}

S_(browser_reload_dir browser_reload_hashmark)
static int browser_reload_hashmark P_((struct folder_browser *dir));
static int browser_reload_hashmark(dir)
     struct folder_browser *dir;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_reload_hashmark: dir=%p\n", 
		     dir));

    if (HASHMARK_BROWSER_magic != dir->a.hashmark_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "browser_reload_hashmark",
	      "Bad magic type",0);

    b_hashmark_default_listing(dir);
    ret = 1;

    DPRINT(Debug,11,(&Debug, "browser_reload_hashmark=%d\n", ret));

    return ret;    
}

struct browser_type hashmark_browser = {
    BROWSER_TYPE_magic,
    browser_zero_hashmark,
    browser_free_hashmark,
    browser_change_hashmark,
    browser_give_title_hashmark,
    browser_separator_hashmark,
    browser_name_hashmark,
    browser_cat_hashmark,
    browser_select_hashmark,    
    browser_folder_from_hashmark,
    browser_change_v_hashmark,
    browser_change_up_hashmark,
    browser_create_selection_hashmark,
    zero_ws_fields_hashmark,
    free_ws_fields_hashmark,
    browser_prepare_write_hashmark,
    browser_end_write_hashmark,
    browser_tell_hashmark_ws,
    browser_seek_hashmark_ws,
    browser_write_hashmark_ws,
    browser_start_we_hashmark,
    browser_end_we_hashmark,
    browser_selection_is_hashmark,
    browser_make_ref_hashmark,
    browser_update_hashmark,      
    browser_hashmark_do_stat,     
    browser_sync_write_hashmark,
    browser_folder_sort_hashmark,
    browser_free_name_vector_hashmark,
    browser_reset_filter_generic,
/* Not called, select_dir_item_by_idx() is used
   only on fbrowser (Elm 2.5 stype file browser) */
    browser_select_idx_generic,
    browser_line_idx_hashmark,
    browser_reload_hashmark,
    browser_gethm_name_vector_hashmark,
    browser_sethm_name_vector_hashmark
};

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

