static char rcsid[] = "@(#)$Id: common_local.c,v 2.26 2024/06/10 18:05:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.26 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ****************************************************************************/

#include "def_mbox.h"
#include "mbxlocal_imp.h"

#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"file");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

static const unsigned char * cs2us P_((const char *str));
static const unsigned char * cs2us(str)
     const char *str;
{
    return (const unsigned char *)str;
}

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}



/* rel_itemname is relative to type -- not include user@hostname 

   rel_itemname is NULL is idicating that selection is on
   dir->a.dummy_browser->remote
*/

int real_select_local(dir,rel_itemname,relative,newpos)
     struct folder_browser *dir;
     const struct string *rel_itemname;
     struct string * relative;
     int *newpos;
{
    struct string * Lstr =  NULL;
    char          * str  = NULL;
    int F = 0;
    int idx;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"real_select_local",
	      "Bad magic type (folder_browser)",0);
    
    idx = browser_select_generic(dir,dir->dirname,
				 rel_itemname,relative,&Lstr,&str,'/',
				 local_fs_charset);

    /* dir can be dummy_browser */
    

    F = local_selection_stat(str,
			     browser_local_last_read(dir));

    if (idx >= 0) {
 	DPRINT(Debug,11,(&Debug,
			 "real_select_local: Using index %d: %s\n",
			 idx,dir->vector[idx].sys_name));

	if (dir->vector[idx].flags & BROWSER_NEEDSTAT) {
	    DPRINT(Debug,11,(&Debug,
			     "real_select_local: Updating flags -- doing stat\n"));
	    dir->type->browser_stat_routine(dir,idx);
	}  

	/* WARNING: set_dir_selection does not allocate strings -- 
	 *          it just assign pointers!
	 */
	set_dir_selection(dir,
			  safe_strdup(dir->vector[idx].sys_name),
			  dup_string(dir->vector[idx].disp_name),
			  dir->vector[idx].flags|F);

	if (newpos)
	    *newpos = idx;

    } else {
	struct string * ConvRelative = convert_string(local_fs_charset,
						      relative,0);
	char * rel_str = us2s(stream_from_string(ConvRelative,0,NULL));

	DPRINT(Debug,11,(&Debug,
			 "real_select_local: Using given name: %s\n",
			 rel_str));

	if (-2 == idx) {
	    DPRINT(Debug,11,(&Debug,
			     "real_select_local: Directory prefix was given.\n"));
	    F |= BROWSER_DIRPREFIX;
	}


	/* WARNING: set_dir_selection does not allocate strings -- 
	 *          it just assign pointers!
	 */
	set_dir_selection(dir,rel_str,dup_string(relative),0|F);	    
	free_string(&ConvRelative);
    }

    if (Lstr)
	free_string(&Lstr);
    if (str)
	free(str);

    DPRINT(Debug,12,(&Debug,
		     "real_select_local=1 dir=%p\n",
		     dir));
    return 1;
}

struct folder_info * real_folder_from_local(dir, treat_as_spooled,
					    assume_incoming)
     struct folder_browser *dir;
     int treat_as_spooled;
     int assume_incoming;
{
    
    /* NOTE: This is called also for  dir type remote_browser
     *
     */
    
    struct folder_info *res = NULL;

    DPRINT(Debug,12,(&Debug,"real_folder_from_local: dir=%p\n", dir));

    if (assume_incoming && 
	( !dir->sys_dir || !dir->sys_dir[0]) &&
	( !dir->selection->sys_name ||
	  0 == strcmp(dir->selection->sys_name,"INBOX"))) {

	unsigned int defaultfile_keyword = 0;
	unsigned int defaultfile_cmask   = 0;

	const char * default_val = give_dt_estr_as_str(&defaultfile_e,
						       "incoming-mailbox",
						       &defaultfile_keyword,
						       &defaultfile_cmask);
	
	if (default_val) {
	    /* Incoming mailbox (may also be IMAP mailbox) */
	    
	    /* Explicite disallow $MAIL to be interpreted as remote mailbox */
	    if (ison(defaultfile_cmask,FOLDER_is_local_file)) {
		
		enum folder_place place;
		
		DPRINT(Debug,10,(&Debug,
				 "real_folder_from_local: FOLDER_is_local_file is set for value\n"));
		
		res = mbx_new_folder();
		
		res -> cur_folder_sys = safe_strdup(default_val);
		res -> cur_folder_disp = new_string2(system_charset,
						     cs2us(default_val));
		
		res-> folder_type = get_folder_type(res -> cur_folder_sys,&place,
						    treat_as_spooled);
		
	    } else if ('/' == default_val[0]) {
		
		enum folder_place place;
		
		DPRINT(Debug,12,(&Debug,
				 "real_folder_from_local: Assuming local incoming mailbox\n"));
		
		res = mbx_new_folder();
		
		res -> cur_folder_sys = safe_strdup(default_val);
		res -> cur_folder_disp = new_string2(system_charset,
						     cs2us(default_val));
		
		res-> folder_type = get_folder_type(res -> cur_folder_sys,&place,
						    treat_as_spooled);
	    } else {
		DPRINT(Debug,12,(&Debug,
				 "real_folder_from_local: No local incoming mailbox\n"));
	    }  
	}

    } else if (dir->selection->sys_name) {  /* Must have local file */
	enum folder_place place;

	res = mbx_new_folder();

	if (dir->sys_dir && dir->sys_dir[0]) {
	    res -> cur_folder_sys = elm_message(FRM("%s/%s"),
						dir->sys_dir,
						dir->selection->sys_name);
	    res -> cur_folder_disp = format_string(FRM("%S/%S"),
						   dir->dirname,
						   dir->selection->disp_name);
	} else {
	    res -> cur_folder_sys  = safe_strdup(dir->selection->sys_name);
	    res -> cur_folder_disp = dup_string(dir->selection->disp_name);
	}
	res-> folder_type = get_folder_type(res -> cur_folder_sys,&place,
					    treat_as_spooled);

	switch (place) {

	case in_home:

	    if (!dir->sys_dir || ! home[0]) {
		
		DPRINT(Debug,12,(&Debug,"real_folder_from_local: in_place=home ignored\n"));
	    } else if (same_file(dir->sys_dir,home)) {
		free_string(&(res -> cur_folder_disp));
		res -> cur_folder_disp = format_string(FRM("~/%S"),
						       dir->selection->disp_name);
	    } else {
		DPRINT(Debug,12,(&Debug,"real_folder_from_local: in_place=home, but %s is not %s\n", 
				 dir->sys_dir, home));
	    }

	    break;
	case in_folders: {
	    /* give_dt_estr_as_str adds / to end */
	    const char * str = give_dt_estr_as_str(&folders_e,"maildir",NULL,NULL);

	    if (!dir->sys_dir || ! str) {
		
		DPRINT(Debug,12,(&Debug,"real_folder_from_local: in_place=in_folders ignored\n"));
	    } else if (same_file(dir->sys_dir,str)) {

		free_string(&(res -> cur_folder_disp));
		res -> cur_folder_disp = format_string(FRM("=%S"),
						       dir->selection->disp_name);

	    } else {
		DPRINT(Debug,12,(&Debug,"real_folder_from_local: in_place=in_folders, but %s is not %s\n", 
				 dir->sys_dir, str));
	    } 
	}
	    break;

	case in_extra_dir: {
	    /* give_dt_estr_as_str adds / to end */

	    const char * mbx_dir = give_dt_estr_as_str(&extra_mailbox_dir_e,
						       "extra-mailbox-dir",
						       NULL,NULL);
	    if (!dir->sys_dir || ! mbx_dir) {
		
		DPRINT(Debug,12,(&Debug,"real_folder_from_local: in_place=in_extra_dir ignored\n"));

	    } else if (same_file(dir->sys_dir,mbx_dir)) {
		int i;

		

		if (home[0] && ( i = strlen(home) ) &&
		    i < strlen(mbx_dir) &&
		    0 == strncmp(mbx_dir,home,i) &&
		    '/' == mbx_dir[i]) {

		    /* mbx_dir and therefore X ends with / */

		    struct string * X = new_string2(local_fs_charset,
						    cs2us(mbx_dir+i));

		    free_string(&(res -> cur_folder_disp));
		    res -> cur_folder_disp = format_string(FRM("~%S%S"),
							   X,
							   dir->selection->disp_name);
		    free_string(&X);

		} else {
		    DPRINT(Debug,12,(&Debug,
				     "real_folder_from_local: in_place=in_extra_dir, but extra-mailbox-dir %s is not under %s\n", 
				     mbx_dir,
				     home[0] ? home : "<NO HOME>"));			  
		}
	    } else {
		DPRINT(Debug,12,(&Debug,"real_folder_from_local: in_place=in_extra_dir, but %s is not %s\n", 
				 dir->sys_dir, mbx_dir));
	    }

	}
	    break;


	default:
	    break;
	}
	
    } else {
	DPRINT(Debug,12,(&Debug,"real_folder_from_local: No selection name\n"));
    }

    if (res) {

	res-> folder_type->init_it(res);

	DPRINT(Debug,12,(&Debug,"real_folder_from_local=%p\n",res));
    } else {
	DPRINT(Debug,12,(&Debug,"real_folder_from_local=NULL\n"));
    }

    return res;
}

char * resolve_selection_local(dir,assume_incoming)
     struct folder_browser *dir;
     int assume_incoming;   /* Assume incoming folder */
{
    char * name = NULL;

    /* NOTE: This is called also for  dir type remote_browser
     *
     */


    if (dir->sys_dir && dir->sys_dir[0]) {
	if (dir->selection->sys_name) 
	    name = elm_message(FRM("%s/%s"),
			       dir->sys_dir,
			       dir->selection->sys_name);
	else {
	    DPRINT(Debug,15,(&Debug,
			     "resolve_selection_local=NULL: no sys_name, sys_dir=%s\n",
			     dir->sys_dir));
	    return NULL;
	}

    } else {
	if (assume_incoming && 
	    ( ! dir->selection->sys_name ||
	      0 == strcmp(dir->selection->sys_name,"INBOX"))) {

	    unsigned int defaultfile_keyword = 0;
	    unsigned int defaultfile_cmask   = 0;
	    
	    const char * default_val = give_dt_estr_as_str(&defaultfile_e,
							   "incoming-mailbox",
							   &defaultfile_keyword,
							   &defaultfile_cmask);

	    if (default_val) {
		/* Incoming mailbox (may also be IMAP mailbox) */
		
		/* Explicite disallow $MAIL to be interpreted as remote mailbox */
		if (0 != (defaultfile_cmask & FOLDER_is_local_file)) {
		    
		    DPRINT(Debug,15,(&Debug,
				     "resolve_selection_local: FOLDER_is_local_file is set for value\n"));
		    
		    name = safe_strdup(default_val);
		    
		} else if ('/' == default_val[0]) {
		    
		    DPRINT(Debug,15,(&Debug,
				     "resolve_selection_local: Assuming local incoming mailbox\n"));
		    
		    name = safe_strdup(default_val);
		    
		} else {
		    
		    DPRINT(Debug,15,(&Debug,
				 "resolve_selection_local=NULL: no local incoming mailbox\n"));
		    return NULL;
		}
	    } else {

		DPRINT(Debug,15,(&Debug,
				 "resolve_selection_local=NULL: Failed to expand incoming-mailbox\n"));

		return NULL;
	    }

	} else if (dir->selection->sys_name) 
	    name = safe_strdup(dir->selection->sys_name);
	else {
	    DPRINT(Debug,15,(&Debug,
			     "resolve_selection_local=NULL: no sys_name\n"));
	    return NULL;
	}
    }


    DPRINT(Debug,15,(&Debug,
		     "resolve_selection_local=%s\n",
		     name));
    
    return name;
}

static int name_selection_is_local P_((const char * name,
				       const struct folder_info *folder));

static int name_selection_is_local(name,folder)
     const char * name;
     const struct folder_info *folder;
{
    int ret = 0;

    if (0 == strcmp(name,folder->cur_folder_sys)) {
	DPRINT(Debug,11,(&Debug,
			 "name_selection_is_local: Same filaname\n"));
	ret = 1;
    } else {
	struct stat XX, A;
	
	if (-1 == stat(name,&A)) {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,11,(&Debug,
			     "name_selection_is_local: Failed to stat %s: %s\n",
			     name,
			     strerror(err)));
	    ret = 0;
	    
	} else if ((NULL != folder ->p->fh_folder &&
		    0 == fstat(fileno(folder ->p->fh_folder),&XX)) 
		   ||
		   0 == stat(folder->cur_folder_sys,&XX)) {

	    if (XX.st_ino == A.st_ino && XX.st_dev == A.st_dev) {
		DPRINT(Debug,11,(&Debug,
				 "name_selection_is_local: Same ino and dev\n"));
		ret = 1;
	    } else
		ret = 0;
	} else {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,11,(&Debug,
			     "name_selection_is_local: Both fstat and stat failed: %s\n",
			     strerror(err)));
	    ret = 0;
	}
    }
    
    return ret;
}


int real_selection_is_local(dir,folder,assume_incoming)
     struct folder_browser *dir;
     struct folder_info *folder;
     int assume_incoming;   /* Assume incoming folder */
{
    int ret = 0;

    /* NOTE: This is called also for  dir type remote_browser
     *
     */

    DPRINT(Debug,11,(&Debug,
		     "real_selection_is_local: dir=%p, folder=%p, assume_incoming=%d\n", 
		     dir,folder,assume_incoming));

    if (folder->folder_type != &read_only &&
	folder->folder_type != &non_spool &&
	folder->folder_type != &spool) {
	DPRINT(Debug,11,(&Debug,
			 "real_selection_is_local: Not a local folder\n"));
	ret = 0;
    } else {
	char * name = resolve_selection_local(dir,assume_incoming);
	
	if (name) {
	    ret = name_selection_is_local(name,folder);
	    free(name);
	
	} else {
	    DPRINT(Debug,11,(&Debug,
			     "real_selection_is_local: No selection name\n"));
	}
    }

    DPRINT(Debug,11,(&Debug,"real_selection_is_local=%d\n",ret));
    return ret;
}

int real_create_selection_local(dir,assume_incoming)
     struct folder_browser *dir;
     int assume_incoming;   /* Assume incoming folder */
{
    char * name = NULL;
    int ret = 0;

    /* NOTE: This is called also for  dir type remote_browser
     *
     */



    DPRINT(Debug,11,(&Debug,
		     "real_create_selection_local: dir=%p, assume_incoming=%d\n", 
		     dir,assume_incoming));

    name = resolve_selection_local(dir,assume_incoming);

    if (name) {
    
	ret = create_as_user(name);
	
	free(name);

    } else {
	DPRINT(Debug,11,(&Debug,
			 "browser_create_selection_local: No selection name\n"));
    }

    DPRINT(Debug,11,(&Debug,
		     "real_create_selection_local=%d\n",ret));
    return ret;
}

int real_prepare_write_local(dir,ptr,filename)
     struct folder_browser *dir;
     WRITE_STATE ptr;
     const char *filename;
{
    struct stat stat_buf, *XBuf = NULL;

    int ret = 0;
    int err = 0;
    
    switch(dir->sel_type) {
    case selection_file:
	if (0 != (err = can_open(filename, "w"))) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeCouldnWriteFile,
			      "Couldn't write to file %s: %s"), 
		      filename,strerror(err));
	    goto fail;
	}

	DPRINT(Debug,4,(&Debug,"Writing to file '%s'...\n", filename));

	ptr->a.local.folder_lock = NULL;

	if ((ptr->a.local.save_fd = open(filename,O_WRONLY,0600)) == -1 ||
	    (ptr->a.local.save_file = fdopen(ptr->a.local.save_fd,"w")) == NULL) {
	    int err = errno;
	    
	    DPRINT(Debug,2,(&Debug,
			    "Error: couldn't write to specified file %s\n", 
			    filename));

	    lib_error(CATGETS(elm_msg_cat, MeSet, MeCouldnWriteFile,
			      "Couldn't write to file %s: %s"), 
		      filename,strerror(err));
	    goto fail;
	}	
	ret = 1;
	
	break;
    case selection_folder:
	if (0 != (err = can_open(filename, "a"))) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeCouldntAppendFolder,
			      "Couldn't append to folder %s: %s"), 
		      filename,strerror(err));
	    goto fail;
	}
		
	if (save_file_stats(filename,&stat_buf) > 0)
	    XBuf = &stat_buf;

	DPRINT(Debug,4,(&Debug, "Saving mail to folder '%s'...\n", filename));

	{
	    const char * mailhome_val = give_dt_estr_as_str(&mailhome_dir_e,
							    "mailhome-dir",
							    NULL,NULL);

	    if (mailhome_val &&
		in_directory(XBuf,filename,mailhome_val)) {
		DPRINT(Debug,4,(&Debug, " in mailhome\n"));
		ptr->a.local.folder_lock = &mailhome_locking;
	    } else
		ptr->a.local.folder_lock = &folder_locking;
	    
	}
   
	if ((ptr->a.local.save_fd = open(filename,O_RDWR,0600)) == -1 ||
	    (ptr->a.local.save_file = fdopen(ptr->a.local.save_fd,"r+")) == NULL) {
	    int err = errno;
	    
	    DPRINT(Debug,2,(&Debug,
			    "Error: couldn't append to specified folder %s\n", 
			    filename));
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeCouldntAppendFolder,
			      "Couldn't append to folder %s: %s"), 
		      filename,strerror(err));
	    goto fail;
	}
    
	if (lock_in_copy) {
	    /*  lock the file to be modified to serialize access with other */
	    /* programs like procmail. */
	
	    if (Grab_the_file(ptr->a.local.save_fd,
			      ptr->a.local.folder_lock) != FLOCKING_OK){
		int err = errno;
		fclose(ptr->a.local.save_file);
		
		ptr->a.local.save_file = NULL;
		ptr->a.local.save_fd   = -1;
		restore_file_stats(filename);
		
		DPRINT(Debug,2,(&Debug,
				"Error: couldn't lock specified folder %s (save)\n", 
				filename));
		lib_error(CATGETS(elm_msg_cat, MeSet, MeCouldntLockFolder,
				  "Couldn't lock folder %s: %s"), 		      
			  filename,
			  strerror(err));
		goto fail;
	    }	
	}
    
	/* copy_message want now seek backwards ... */
	if (0 != fseek(ptr->a.local.save_file,0,SEEK_END)) {
	    int err = errno;
	    if (lock_in_copy)
		Release_the_file(ptr->a.local.save_fd,
				 ptr->a.local.folder_lock);
	    
	    ptr->a.local.save_file = NULL;
	    ptr->a.local.save_fd   = -1;
	    restore_file_stats(filename);
	    
	    DPRINT(Debug,2,(&Debug,
			    "Error: couldn't seek to end of specified folder %s (save)\n", 
			    filename));
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeCouldntAppendFolder,
			      "Couldn't append to folder %s: %s"), 
		      filename,strerror(err));
	    
	    goto fail;
	}    
	ret = 1;
	break;
    case selection_url:
    case NUM_selection_type:
	/* Not used */
	break;
    }

 fail:
    
    DPRINT(Debug,12,(&Debug,
		     "real_prepare_write_local=%d dir=%p filename=%s\n", 
		     ret,dir,filename));
    return ret;
}

int real_end_write_local(dir,ptr,filename)
     struct folder_browser *dir;
     WRITE_STATE ptr;
     const char *filename;
{
    int ret = 0;

    switch(dir->sel_type) {
    case selection_file:
	break;
    case selection_folder:
	
	if (lock_in_copy)
	    Release_the_file(ptr->a.local.save_fd,
			     ptr->a.local.folder_lock); /* XXX LOCKING */
	break;
    case selection_url:
    case NUM_selection_type:
	/* Not used */
	break;
    }

    /** ptr->a.local.save_file and ptr->a.local.save_fd shares same
	file descriptor!
    */

    ret = 0 == fclose(ptr->a.local.save_file);
    ptr->a.local.save_file = NULL;
    ptr->a.local.save_fd   = -1;
    ptr->a.local.folder_lock = NULL;

    switch(dir->sel_type) {
    case selection_file:
	break;
    case selection_folder:
	restore_file_stats(filename);
	break;
    case selection_url:
    case NUM_selection_type:
	/* Not used */
	break;
    }

    DPRINT(Debug,12,(&Debug,
		     "real_end_write_local=%d dir=%p filename=%s\n", 
		     ret,dir,filename));
    return ret;
}

int real_sync_write_local(dir,ptr,filename)
     struct folder_browser *dir;
     WRITE_STATE ptr;
     const char *filename;
{
    int ret = 0;

    /** ptr->a.local.save_file and ptr->a.local.save_fd shares same
	file descriptor!
    */

    ret = 0 == fflush(ptr->a.local.save_file);
    if (!ret) {
	DPRINT(Debug,12,(&Debug,
			 "real_sync_write_local: fflush failed\n"));
    }

    if (-1 == fsync(ptr->a.local.save_fd)) {
	DPRINT(Debug,12,(&Debug,
			 "real_sync_write_local: fsync failed\n"));
	ret = 0;
    }

    switch(dir->sel_type) {
    case selection_file:
	break;
    case selection_folder:
	break;
    case selection_url:
    case NUM_selection_type:
	break;
    }

    DPRINT(Debug,12,(&Debug,
		     "real_sync_write_local=%d dir=%p filename=%s\n", 
		     ret,dir,filename));
    return ret;
}

long real_browser_tell_ws(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    /* Both ftell and real_browser_tell_ws returns -1 on failure */
    long ret = ftell(ptr->a.local.save_file);

    if (-1 == ret) {
	int err UNUSED_VAROK = errno;
	DPRINT(Debug,12,(&Debug,
			 "real_browser_tell_ws: ERROR (errno=%d): %s\n",
			 err,strerror(err)));
    }

    DPRINT(Debug,12,(&Debug,
		     "real_browser_tell_ws=%ld dir=%p\n", 
		     ret,dir));
    return ret;
}

/* Returns 0 on failure! */
int real_browser_seek_ws(dir,ptr,pos)
     struct folder_browser *dir;
     WRITE_STATE ptr;
     long pos;
{
    int ret = 0;

    if (0 == fseek(ptr->a.local.save_file,pos,SEEK_SET))
	ret = 1;
    else {
	int err UNUSED_VAROK = errno;
	DPRINT(Debug,12,(&Debug,
			 "real_browser_seek_ws: ERROR (errno=%d): %s\n",
			 err,strerror(err)));
    }

    DPRINT(Debug,12,(&Debug,
		     "real_browser_seek_ws=%d dir=%p\n", 
		     ret,dir));
    return ret;
}

int real_browser_write_ws(dir,ptr,l,buffer)
     struct folder_browser *dir;
     WRITE_STATE ptr;
     int l; 
     const char *buffer;
{
    int ret = 1;    
    int count = 0;
    
    while (ret && count < l) {
	int x = fwrite(buffer+count,1,l-count,ptr->a.local.save_file);
	
	if (x > 0) 
	    count += x;
	else if (ferror(ptr->a.local.save_file)) {
	    int err = errno;

	    if (err != EINTR) {
		DPRINT(Debug,12,(&Debug,
				 "real_browser_write_ws: ERROR (errno=%d): %s\n",
				 err,strerror(err)));
		ret = 0;
	    } else {
		clearerr(ptr->a.local.save_file);
	    }
	}

	if (0 == x && ret) {
	    DPRINT(Debug,12,(&Debug,
			     "real_browser_write_ws: Looping...\n"));
	}
    }

    DPRINT(Debug,12,(&Debug,
		     "real_browser_write_ws=%d dir=%p (%d written of %d)\n", 
		     ret,dir,count,l));
    return ret;
}

int real_start_we_local(dir,ptr,current_header,env_flags)
     struct folder_browser *dir;
     WRITE_STATE ptr;
     struct header_rec *current_header;
     int *env_flags;
{
    int ret = 0;
    char * buffer1 = ctime(& current_header->received_time);

    *env_flags = 0;

    if (!buffer1) {
	DPRINT(Debug,12,(&Debug,
			 "real_start_we_local: ctime() failed\n"));
	goto fail;
    }
    
    if (have_MMDF) {
	DPRINT(Debug,5,(&Debug,
			"real_start_we_local: Write MMDF message separator\n"));
	if (fprintf(ptr->a.local.save_file, "%s", MSG_SEPARATOR) == EOF) {
	    DPRINT(Debug,1,(&Debug, "real_start_we_local fails\n"));
	    goto fail;
	}
	*env_flags = WE_ADD_RETURN_PATH;
    }
    
    if (fprintf(ptr->a.local.save_file,
	       "From %s %.24s\n", 
	       current_header->env_from,
	       buffer1) == EOF) {
	DPRINT(Debug,1,(&Debug,"real_start_we_local fails\n"));
	goto fail;
    }
    ret = 1;

 fail:
    DPRINT(Debug,12,(&Debug,
		     "real_start_we_local=%d dir=%p\n", 
		     ret,dir));

    return ret;
}

int real_end_we_local(dir,ptr,current_header)
     struct folder_browser *dir;
     WRITE_STATE ptr;
     struct header_rec *current_header;
{
    int ret = 0;

    long END_pos = ftell(ptr->a.local.save_file);
    char buffer[2];
    int i;
    long NEW_pos = END_pos - 2L;

    /* parse_body_routine() excludes ending empty line from
       content_length, so add empty line always to mail 

       see also mbx_mark_keep_normal() on localmbx.c 
    */

    buffer[0] = '\0';
    buffer[1] = '\0';
    if (NEW_pos < 0L)
	NEW_pos = 0L;
	  
    if (0 != fseek(ptr->a.local.save_file,NEW_pos,SEEK_SET)) {
	DPRINT(Debug,5,(&Debug,
			"real_end_we_local: Failed to seek to %ld (output)\n",
			NEW_pos)); 
    }
    
    i = fread(buffer,sizeof (char),2,ptr->a.local.save_file);
    if (i < 0) {
	DPRINT(Debug,5,(&Debug,
			"real_end_we_local: Read error! (output)\n"));
	i = 0;
    } else {
	DPRINT(Debug,5,(&Debug,
			"real_end_we_local: read %d bytes (output). \n",i));
    }
    
    for (; i < 2; i++) {
	buffer[i] = '\0';
    }

    DPRINT(Debug,5,(&Debug,
		    "real_end_we_local: last two bytes %d %d (output)\n",
		    buffer[0], buffer[1])); 
    
    if (0 != fseek(ptr->a.local.save_file,0L,SEEK_CUR)) {
	DPRINT(Debug,5,(&Debug,
			"real_end_we_local: Failed to seek 0 bytes (output)\n"));
    }

    if (buffer[1] != '\n') {
	/* No \n in end ? */ 	
	DPRINT(Debug,5,(&Debug,
			"real_end_we_local: NL missing from end of mail\n")); 
	if (fprintf(ptr->a.local.save_file, "\n") == EOF) {
	    DPRINT(Debug,5,(&Debug,
			    "real_end_we_local: Failed to add NL to end of mail\n"));
	    DPRINT(Debug,1,(&Debug,"real_end_we_local fails\n"));
	    goto fail;
	}
    }

    /* NOTE:
     * Ending empty line or MSG_SEPARATOR is not part of 
     * mail, but instead part of folder format, so it
     * should NOT be included on calculation of content-length
     */
    if (!have_MMDF) {

	/* blank line to keep mailx happy *sigh* */      
	if (fprintf(ptr->a.local.save_file, "\n") == EOF) {
	    DPRINT(Debug,5,(&Debug,
			    "real_end_we_local: Failed to add second NL to end of mail\n"));
	    DPRINT(Debug,1,(&Debug, "real_end_we_local fails\n"));
	    goto fail;
	}

    } else {
	DPRINT(Debug,5,(&Debug,
			"real_end_we_local: Write MMDF message separator (end)\n"));
	if (fprintf(ptr->a.local.save_file, "%s", MSG_SEPARATOR) == EOF) {
	    DPRINT(Debug,1,(&Debug, "real_end_we_local fails\n"));
	    goto fail;
	}   
    }
    ret = 1;

 fail:
    DPRINT(Debug,12,(&Debug,
		     "real_end_we_local=%d dir=%p\n", 
		     ret,dir));
      return ret;
}

int real_make_ref_local(dir,refname,iscopy,is_text)
     struct folder_browser *dir;
     char **refname; 
     int *iscopy;
     int is_text;
{
    char * name = NULL;
    int ret = 0;

    *refname = NULL;
    *iscopy  = 0;

    if (dir->selection->sys_name) {

	if (dir->sys_dir && dir->sys_dir[0])
	    name = elm_message(FRM("%s/%s"),
			       dir->sys_dir,
			       dir->selection->sys_name);
	else
	    name = safe_strdup(dir->selection->sys_name);
	
	if (access(name,READ_ACCESS) < 0) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeErrorRead,
			      "Can't read %S: %s"),
		      dir->selection->disp_name,strerror(err));
	    ret = 0;
	    free(name);
	    name = NULL;
	} else {
	    *iscopy = 0;
	    *refname = name;
	    name = NULL;
	    ret = 1;
	    
	    DPRINT(Debug,12,(&Debug,"real_make_ref_local: *refname=%s\n", 
			     *refname));
	}

    } else {
	DPRINT(Debug,12,(&Debug,"real_make_ref_local: No selection name\n"));
    }

    DPRINT(Debug,12,(&Debug,"real_make_ref_local=%d", ret));
    
    return ret;
}


int local_stat_flags(pathname,buf,this_last_read) 
     const char * pathname;
     struct stat * buf;
     /* upgrade_last_read() is assumed to be called */
     struct last_read_cache * this_last_read;
{

    int flags = 0;

    if (
#ifdef S_ISDIR
	!S_ISDIR(buf->st_mode)
#else
	S_IFDIR != (buf->st_mode & S_IFMT)
#endif
	) {
	    flags |= BROWSER_NODIR;

	    DPRINT(Debug,13,(&Debug,
			     "local_stat_flags : %s: not on directory\n",pathname));
    }

    if (
#ifdef S_ISREG
	!S_ISREG(buf->st_mode)
#else
	S_IFREG != (buf->st_mode & S_IFMT)
#endif
	) {
	flags |= BROWSER_NOFOLDER;

	DPRINT(Debug,13,(&Debug,
			 "local_stat_flags : %s: not on folder (file)\n",pathname));
	

    } else {
	time_t atime          = buf->st_atime;
	time_t last_read_time = buf->st_atime;
	int found_last_read = 0;
	char * X UNUSED_VAROK = "";
	
	if (this_last_read) {
	    found_last_read = have_last_read(this_last_read,
					     pathname,buf,
					     &last_read_time);
	    if (found_last_read &&
		last_read_time > atime) {

		DPRINT(Debug,13,(&Debug,
				 "local_stat_flags : %s: updating atime %d -> last_read %d\n",
				 pathname,atime,last_read_time));

		atime = last_read_time;
	    } else {
		X = found_last_read ? "old last_read entry" : "No last_read entry";
	    }
	} else {
	    X = "last_read not enabled or updated";
	}
	    
	if (buf->st_mtime > atime) {
	    DPRINT(Debug,13,(&Debug,
			     "local_stat_flags : %s: mtime=%d > atime=%d -- new mail? %s\n",
			     pathname,buf->st_mtime,atime,X));
	    setit(flags,BROWSER_MARKED);
	}
    }
    
    if ((BROWSER_NOFOLDER|BROWSER_NODIR) == flags) {
	DPRINT(Debug,13,(&Debug,
			 "local_stat_flags: %s: Not a directory or folder (file)\n",
			 pathname));
    }
    
    return flags;
}

int local_selection_stat(str,dir_last_read)
     const char * str;
     struct last_read_cache * dir_last_read;
{
    int F = 0;
    struct stat    bufX;	 /* stat command  */

    enum syscall_status r_stat;
    
    /* give_dt_estr_as_str adds / to end */
    const char * folders_val = give_dt_estr_as_str(&folders_e,"maildir",NULL,NULL);
    const char * mbx_dir = give_dt_estr_as_str(&extra_mailbox_dir_e,
					       "extra-mailbox-dir",NULL,NULL);
    
    const char * mailhome_val = give_dt_estr_as_str(&mailhome_dir_e,
						    "mailhome-dir",
						    NULL,NULL);

    r_stat = stat(str,&bufX);
    switch (r_stat) {
    case syscall_success /* 0 */: {
	
	struct last_read_cache * this_last_read =
	    upgrade_last_read(dir_last_read,
			      str,&bufX);
	
	DPRINT(Debug,11,(&Debug,
		    "local_selection_stat: %s exists\n",
		    str));
	F = BROWSER_EXIST;

	if (
#ifdef S_ISDIR
            S_ISDIR(bufX.st_mode)
#else
            S_IFDIR == (bufX.st_mode & S_IFMT)
#endif
	    ) {

	    DPRINT(Debug,11,(&Debug,
			     "local_selection_stat: %s is directory\n",
			     str));	     

	    F |= BROWSER_DIRPREFIX;
	}

	F |= local_stat_flags(str,&bufX,this_last_read);

	if (0 == (F & BROWSER_NOFOLDER)) {

	    if (mailhome_val &&
		in_directory(&bufX,str,mailhome_val))
		F |= BROWSER_MAILFILE;
	    else if (mbx_dir &&
		     in_directory(&bufX,str,mbx_dir))
		F |= BROWSER_MAILFILE;
	    else if (folders_val &&
		     in_directory(&bufX,str,folders_val))
		F |= BROWSER_MAILFILE;
	}

	/* Decrements refcount */
	if (this_last_read)
	    free_last_read_cache(&this_last_read);
    }
	break;
    case syscall_error /* -1 */: {
	int err UNUSED_VAROK = errno;
	
	DPRINT(Debug,11,(&Debug,
			 "local_selection_stat: %s : stat failed: %s\n",
			 str,strerror(err)));

	F |= BROWSER_NOFOLDER|BROWSER_NODIR;
    }
	break;
    }

    return F;

}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

