static char rcsid[] = "@(#)$Id: outheaders.c,v 2.17 2020/11/15 16:02:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.17 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mailer.h"
#include "hdr_imp.h"
#include "s_me.h"

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif         

DEBUG_VAR(Debug,__FILE__,"header");


static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static char * from_addr_literal2 P_((const char *domain));
static char * from_addr_literal2(domain)
     const char *domain;
{
    char * Q = NULL;

    /* If on username is strange characters (specially if username
       is from some other user database backend than from /etc/passwd)
       then quote it...
    */
    
    if (NULL != strpbrk(username,
			":\"\\!#%&()=?',;.:<>")) {
	Q = elm_message(FRM("%Q"),username);

	DPRINT(Debug,6,
	       (&Debug, "from_addr_literal2: Quoting username: %s\n",Q));

    } else 
	Q = safe_strdup(username);

    if (domain) {
	
	DPRINT(Debug,10,(&Debug,"from_addr_literal2: domain=%s\n",
			 domain));
	
	Q = strmcat(Q,"@");
	Q = strmcat(Q, domain);
    }

    DPRINT(Debug,10,(&Debug,
		    "from_addr_literal2=%s\n",Q));
    
    return Q;
}


char * from_addr_literal1(add_dom,used_domain)
     int add_dom;
     char **used_domain  /* result malloced */;
{
    const char * maildomain = NULL;
    char * Q = NULL;
    
    if (add_dom) {
	const char * have_mailname = useful_mailname();		   
       	
	if (have_mailname)
	    maildomain = have_mailname;
	else {
	    maildomain = hostfullname;
	    
	    DPRINT(Debug,6,(&Debug,
			    "from_addr_literal1: mailname not set or not useful, using hostfullname=%s\n",
			    maildomain));
	}
		    
	if (used_domain) {
	    *used_domain = strmcpy(*used_domain,
				   maildomain);
	}
	
    } else if (used_domain && *used_domain) {
	free(*used_domain);
	*used_domain = NULL;
    }

    Q = from_addr_literal2(maildomain);
    
    DPRINT(Debug,6,(&Debug,
		    "from_addr_literal1=%s\n",Q));
    
    return Q;
}

char * from_addr_literal(mailer_info,used_domain)
     struct mailer_info *mailer_info;
     char **used_domain  /* result malloced */;
{

    int add_dom = (mailer_info &&
		   query_mailer_info(mailer_info,MI_USE_DOMAIN));
	
		   
    return from_addr_literal1(add_dom,used_domain);   
}

struct string * username_to_surface_addr()
{
    char *s = safe_strdup(username);     /* Use username as surface addr */

    struct string *s1 = NULL;
    char *x;

    /* No fancy quotation stuff on textual form for editing */
    for (x = s; *x; x++) {
	switch(*x) {
	case '"': case '\\': case '(': case ')': case ':':
	case '<': case '>':
	    *x = '_';
	    break;
	}
    }
    
    s1 = new_string2(display_charset,s2us(s));

    free(s);

    return s1;
}

void force_from_addr(expanded,force_domain)
     struct expanded_address * expanded;
     const char              * force_domain;
{
    struct string *s1 = username_to_surface_addr();
    char * Q = from_addr_literal2(force_domain);

    if (Q) {
	int pos;

	free_expanded_address(expanded);
	
	DPRINT(Debug,14, (&Debug, 
			  "force_from_addr: address=%s",
			  Q));
	if (full_username_s) {
	    DPRINT(Debug,14, (&Debug, " full username=%S",
			      full_username_s));
	}
	DPRINT(Debug,14, (&Debug, "\n"));
	
	pos = add_expanded_addr_(expanded,Q,full_username_s,NULL);
	
	add_textual_addr_(expanded,s1,pos,1);
	free(Q);
	Q = NULL;
    } else {
	DPRINT(Debug,14, (&Debug, 
			  "force_from_addr: No address\n"));
    }   
	
    free_string(&s1);
}

/* Returns >= 1 if from address was added */
 enum make_from_status make_from_addr(expanded,mailer_info,used_domain,
				      used_address)
     struct expanded_address * expanded;
     struct mailer_info      * mailer_info;
     char                   ** used_domain  /* result malloced */;
     char                   ** used_address /* result malloced */;
{
    enum make_from_status ret = make_from_normal;
    
    char            * domain = NULL;

    struct string *s1 = username_to_surface_addr();

    char * Q = from_addr_literal(mailer_info,
				 &domain);

    if (used_domain && *used_domain) {
	free(*used_domain);
	*used_domain = NULL;
    }

    if (used_address && *used_address) {
	free(*used_address);
	*used_address = NULL;
    }
    
    if (Q) {
    
	if (domain) {
	    ret = make_from_normal;
	    if (! domain[0]) {
		DPRINT(Debug,14, (&Debug, 
				  "make_from_addr: addr %s, empty domain\n",
				  Q));
		
		ret = make_from_fail;
	    } else if ('[' == domain[0]) {

		int len = rfc822_toklen(domain);
		
		if (len < 2 || domain[len] || domain[len-1] != ']') {
		    DPRINT(Debug,14,
			   (&Debug, 
			    "make_from_addr: addr %s, domain %s failed to parse as literal\n",
			    Q,domain));

		    lib_error(CATGETS(elm_msg_cat, MeSet,
				      MeFromBadAddressLiteral,
				      "Invalid literal %s for From address %s"),
			      domain,Q);
		    
		    ret = make_from_fail;
		} else {
		    DPRINT(Debug,14,
			   (&Debug, 
			    "make_from_addr: addr %s, domain %s is literal\n",
			    Q,domain));
		    ret = make_from_literal;
		}

	    } else {
		const char * whitelisted_name =
		    is_whitelisted_valid_domain(domain);
		
		if (whitelisted_name) {
		     DPRINT(Debug,14,
			    (&Debug, 
			     "make_from_addr: addr %s: domain %s is whitelisted on %s\n",
			     Q, whitelisted_name,domain));
		     ret = make_from_whitelisted;
		     
		} else {
		    const char * reserved_name =
			is_special_use_domain(domain);
		    
		    if (reserved_name) {
			DPRINT(Debug,14,
			       (&Debug, 
				"make_from_addr: addr %s: domain %s is reserved on %s\n",
				Q, reserved_name,domain));

			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeFromAddressReservedDomain,
					  "Reserved domain %s is unsupported for From address %s"),
				  reserved_name,Q);
			
			ret =  make_from_fail;
		    }
		}
	    }
	} else {
	    DPRINT(Debug,14, (&Debug, 
			      "make_from_addr: addr %s, no domain\n",
		  Q));
	    ret = make_from_localaddr;
	}
		
	if (ret) {
	    int pos;

	    free_expanded_address(expanded);
	    
	    
	    DPRINT(Debug,14, (&Debug, 
			      "make_from_addr: address=%s",
			      Q));
	    if (full_username_s) {
		DPRINT(Debug,14, (&Debug, " full username=%S",
				  full_username_s));
	    }
	    DPRINT(Debug,14, (&Debug, "\n"));
	    
	    pos = add_expanded_addr_(expanded,Q,full_username_s,NULL);
	    
	    add_textual_addr_(expanded,s1,pos,1);

	    
	    free_string(&s1);
	}

	if (used_address) {
	    *used_address = Q;
	} else {
	    free(Q);
	}
	Q = NULL;
	
    } else {
	DPRINT(Debug,14, (&Debug, "make_from_addr: No address\n"));
	ret = make_from_fail;
    }
    
    if (domain) {
	if (used_domain) 
	    *used_domain = domain;
	else
	    free(domain);
	domain = NULL;
    }
    
    DPRINT(Debug,14, (&Debug, 
		      "make_from_addr=%d",ret));

    switch (ret) {
    case make_from_fail:        DPRINT(Debug,14, (&Debug," make_from_fail"));        break;
    case make_from_normal:      DPRINT(Debug,14, (&Debug," make_from_normal"));      break;
    case make_from_localaddr:   DPRINT(Debug,14, (&Debug," make_from_localaddr"));   break;
    case make_from_literal:     DPRINT(Debug,14, (&Debug," make_from_literal"));     break;
    case make_from_whitelisted: DPRINT(Debug,14, (&Debug," make_from_whitelisted")); break;
    }
    
    if (used_domain && *used_domain) {
	DPRINT(Debug,14, (&Debug, ", used domain=%s",
			  *used_domain));
    }

    if (used_address && *used_address) {
	DPRINT(Debug,14, (&Debug, ", used address=%s",
		  *used_address));
    }
    
    DPRINT(Debug,14, (&Debug, "\n"));
    
    return ret;
}

void zero_mailing_headers (hdrs)
     struct mailing_headers *hdrs;
{
    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)hdrs,sizeof (*hdrs));
 
    hdrs->magic       = MAIL_HDR_magic;
    
    hdrs->subject     = NULL;
    zero_expanded_address(&(hdrs->from));

    hdrs->in_reply_to = NULL;
    hdrs->references  = NULL;
    hdrs->date        = NULL;
    hdrs->expires     = NULL;
    hdrs->message_id  = NULL;
    hdrs->message_digest = NULL;
    hdrs->action      = NULL;
    hdrs->priority    = NULL;
    hdrs->sender    = NULL;
    zero_expanded_address(&(hdrs->reply_to));
    zero_expanded_address(&(hdrs->to));
    zero_expanded_address(&(hdrs->cc));
    hdrs->xmailer     = NULL;
    hdrs->encoded_user_agent = NULL;
    hdrs->user_header       = NULL;
    hdrs->user_header_count = 0;
    zero_expanded_address(&(hdrs->bcc));
    hdrs->precedence   = NULL;
    hdrs->env_from     = NULL;
}

void free_mailing_headers (hdrs)
     struct mailing_headers *hdrs;
{
    if (hdrs->magic   != MAIL_HDR_magic) 
    	panic("HEADERS PANIC",__FILE__,__LINE__,"free_mailing_headers",
	      "Bad magic number",0);

    if (hdrs->subject)
	free_string(&(hdrs->subject));
    free_expanded_address(&(hdrs->from));
    if (hdrs->in_reply_to)
	free_references(& hdrs->in_reply_to);
    if (hdrs->references)
	free_references(& hdrs->references);
    if (hdrs->date)
	free_expanded_date(& (hdrs->date));
    if (hdrs->expires)
	free_expanded_expires(& (hdrs->expires));
    if (hdrs->message_id)
	free_message_id(& (hdrs->message_id));
    if (hdrs->message_digest)
	free_digest_proc(&  (hdrs->message_digest));
    if (hdrs->action)
	free(hdrs->action);
    hdrs->action      = NULL;
    if (hdrs->priority)
	free(hdrs->priority);
    hdrs->priority    = NULL;
    if (hdrs->sender)
	free(hdrs->sender);
    hdrs->sender     = NULL;
    free_expanded_address(&(hdrs->reply_to));
    free_expanded_address(&(hdrs->to));
    free_expanded_address(&(hdrs->cc));

    if (hdrs->xmailer)
	free_string(&(hdrs->xmailer));

    if (hdrs->encoded_user_agent)
	free(hdrs->encoded_user_agent);
    hdrs->encoded_user_agent = NULL;
    
    if (hdrs->user_header) {
	int i;

	for (i = 0; i < hdrs->user_header_count; i++) {
	    if (hdrs->user_header[i].value)
		free_string(&(hdrs->user_header[i].value));
	    hdrs->user_header[i].name = NULL;
	}
	free(hdrs->user_header);
	hdrs->user_header       = NULL;
    }
    hdrs->user_header_count = 0;

    free_expanded_address(&(hdrs->bcc));
    if (hdrs->precedence)
	free(hdrs->precedence);
    hdrs->precedence   = NULL;

    if (hdrs->env_from)
	mailer_free_env_from( &(hdrs->env_from) );

    /* Make sure that dangling pointters are catched */
    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)hdrs,sizeof (*hdrs));
    
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
