static char rcsid[] = "@(#)$Id: mail_gets.c,v 2.7 2024/06/10 18:05:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.7 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 lib/mail_gets.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1992 USENET Community Trust
 *****************************************************************************/

/** get a line from the mail file, but be tolerant of nulls

  The length of the line is returned

**/

#include "elm_defs.h"

/* For INT_MAX */
#include <limits.h>


DEBUG_VAR(Debug,__FILE__,"core");

int mail_gets(buffer, size, mailfile)
     char *buffer;
     size_t size;
     FILE *mailfile;
{
    int line_bytes = 0, ch;
    char *c = buffer;

    size--; /* allow room for zero terminator on end, just in case */


    /* getc should return EOF on feof or ferror so  we
       do not test them on loop condition:
       
       !feof(mailfile) && !ferror(mailfile)

    */

    while ((size_t)line_bytes < size) {
	ch = getc(mailfile); /* Macro, faster than  fgetc() ! */

	if (ch == EOF) {

	    if (line_bytes > 0 && *(c-1) != '\n') {
	        ++line_bytes;
	    	*c++ = '\n';
	    }
	    break;
	}

	*c++ = ch;
	++line_bytes;

	if (ch == '\n')
	    break;

	if (line_bytes == INT_MAX) {    /* Should be impossible? */
	    DPRINT(Debug,6,(&Debug,
			    "mail_gets: INT_MAX (%d) bytes read!\n",
			    line_bytes));
	    break;
	}
	    
    }
    *c = 0;	/* mail_gets is also used on places where zero
		 * terminated strings are assumed 
		 */
    return line_bytes;
}

/* malloc_gets added by Kari Hurtta 
   Do not return \n

   return -1 if limit exceeded, buffer is still alloced
   return -2 if error or if feof() is true before reading anything
*/

int malloc_gets(buffer, limit, mailfile)
     char **buffer;
     int limit;    /* -1 if no limit; actually limit is INT_MAX */
     FILE *mailfile;
{
    char * buf     = *buffer;
    int line_bytes = 0;
    size_t alloced    = 0;

    if (feof(mailfile))
	return -2;   /* Is called after EOF */

    while (!feof(mailfile) && !ferror(mailfile) && 
	   (line_bytes < limit || limit < 0)) {

	int ch = getc(mailfile); /* Macro, faster than  fgetc() ! */

	if (EOF == ch ||
	    '\n' == ch)
	    break;

	if (line_bytes+1 > alloced) {
	    size_t n = alloced + 100;
	    if (limit > 0 && n > (size_t)limit && limit > line_bytes)
		n = limit;

	    buf = safe_realloc(buf,n);
	    alloced = n;
	}
	buf[line_bytes++] = ch;

	if (line_bytes == INT_MAX) {       /* Should be impossible? */
	    DPRINT(Debug,6,(&Debug,
			    "malloc_gets: INT_MAX (%d) bytes read!\n",
			    line_bytes));
	    break;
	}

    }

    if (line_bytes > 0 ||
	( !feof(mailfile) && !ferror(mailfile))) {

	buf = safe_realloc(buf,line_bytes+1);
	buf[line_bytes] = '\0';
    }
    
    *buffer = buf;

    if ((limit > 0 && line_bytes >= limit) ||
	line_bytes == INT_MAX)
	return -1;
    
    if (ferror(mailfile))
	return -2;

    return line_bytes;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
