static char rcsid[] = "@(#)$Id: digest_wrapper.c,v 1.3 2022/06/26 16:55:20 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.3 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"

DEBUG_VAR(Debug,__FILE__,"digest");

#include "digest_imp.h"

struct digest_proc *malloc_digest_proc(t)
     const struct digest_proc_type *t;
{
    struct digest_proc *ret = safe_zero_alloc(sizeof (*ret));

    if (DIGEST_PROC_TYPE_magic != t->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "malloc_digest_proc",
	      "Bad magic number (digest_proc_type)",0);
       
    ret->magic       =  DIGEST_PROC_magic;
    ret->digest_type = t;
    ret->refcount    = 1;

    ret->mode = digest_proc_initial;

    ret->context.dummy = NULL;
    ret->digest_result = NULL;
           
    return ret;
}

static void digest_free_data P_((struct digest_proc *ptr));
static void digest_free_data(ptr)
     struct digest_proc *ptr;
{
    if (DIGEST_PROC_TYPE_magic != ptr->digest_type->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "digest_free_data",
	      "Bad magic number (digest_proc_type)",0);

    if (ptr->context.dummy) 
	ptr->digest_type->free_ctx(ptr);
    
    if (ptr->context.dummy) {
	DPRINT(Debug,1,(&Debug,"digest_free_data: %p: context not free'ed (%p)\n",
			ptr,ptr->context.dummy));

	free(ptr->context.dummy);
	ptr->context.dummy = NULL;
    }

    if (ptr->digest_result) {
	free(ptr->digest_result);
	ptr->digest_result = NULL;
    }

}


void free_digest_proc(ptr)
     struct digest_proc **ptr;
{

    if (DIGEST_PROC_magic != (*ptr)->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "free_digest_proc",
	      "Bad magic number (digest_proc)",0);

    if ((*ptr)->refcount < 1)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "free_digest_proc",
	      "Bad refcount (digest_proc)",0);
	
    (*ptr)->refcount--;

    if ((*ptr)->refcount > 0) { /* Do not free */

	(*ptr) = NULL;           /* Refefence count for this pointer is     */
        return;                  /* decrement, so this must have be reset   */                                         
    }

    digest_free_data(*ptr);
    
    (*ptr)->digest_type = NULL;
    
    (*ptr)->magic       = 0;     /* Invalidate */
    free(*ptr);
    *ptr = NULL;

}

void inc_digest_proc_refcount(ptr)
     struct digest_proc *ptr;
{
    if (DIGEST_PROC_magic != ptr->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "inc_digest_proc_refcount",
	      "Bad magic number (digest_proc)",0);

    ptr->refcount++;
}

const struct digest_proc_type * digest_type(ptr)
     struct digest_proc *ptr;
{
    if (DIGEST_PROC_magic != ptr->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "digest_type",
	      "Bad magic number (digest_proc)",0);
    
    if (DIGEST_PROC_TYPE_magic != ptr->digest_type->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "digest_type",
	      "Bad magic number (digest_proc_type)",0);


    return ptr->digest_type;
}


/* Resets digest_proc mode */
void  start_digest_proc(ptr)
     struct digest_proc *ptr;
{
    if (DIGEST_PROC_magic != ptr->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "start_digest_proc",
	      "Bad magic number (digest_proc)",0);
    
    if (DIGEST_PROC_TYPE_magic != ptr->digest_type->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "start_digest_proc",
	      "Bad magic number (digest_proc_type)",0);

    switch (ptr->mode) {
    
    case digest_proc_ended:
	digest_free_data(ptr);
    
	ptr->mode = digest_proc_initial;
	/* FALLTHRU */
    case digest_proc_initial:
	if (! ptr->context.dummy)
	    ptr->digest_type->alloc_ctx(ptr);

	ptr->mode = digest_proc_started;
	/* FALLTHRU */
    case digest_proc_started:
	if (! ptr->context.dummy)
	    panic("DIGEST PANIC",__FILE__,__LINE__,
		  "start_digest_proc",
		  "Context not allocated",0);
	
	/* Restart it */
	ptr->digest_type->init_ctx(ptr);
	return;
    }

    panic("DIGEST PANIC",__FILE__,__LINE__,
	  "start_digest_proc",
	  "Bad state",0);
}

void feed_digest_proc(ptr,data,data_len)
     struct digest_proc *ptr;
     const unsigned char data[];
     const size_t        data_len;
{
    if (DIGEST_PROC_magic != ptr->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "feed_digest_proc",
	      "Bad magic number (digest_proc)",0);
    
    if (DIGEST_PROC_TYPE_magic != ptr->digest_type->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "feed_digest_proc",
	      "Bad magic number (digest_proc_type)",0);

    switch (ptr->mode) {
    case digest_proc_ended:
	
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "feed_digest_proc",
	      "Bad state",0);

	break;

    case digest_proc_initial:
	if (! ptr->context.dummy)
	    ptr->digest_type->alloc_ctx(ptr);
	
	if (! ptr->context.dummy)
	    panic("DIGEST PANIC",__FILE__,__LINE__,
		  "feed_digest_proc",
		  "Context not allocated",0);
	
	/* Start it */
	ptr->digest_type->init_ctx(ptr);

	ptr->mode = digest_proc_started;
	/* FALLTHRU */
    case digest_proc_started:
	if (! ptr->context.dummy)
	    panic("DIGEST PANIC",__FILE__,__LINE__,
		  "feed_digest_proc",
		  "Context not allocated",0);
	
	ptr->digest_type->update_ctx(ptr,data,data_len);
	return;
    }
    
    panic("DIGEST PANIC",__FILE__,__LINE__,
	  "feed_digest_proc",
	  "Bad state",0);
}

/* Return digest length -- constant */
size_t len_digest_proc(ptr)
     struct digest_proc *ptr;
{
    if (DIGEST_PROC_magic != ptr->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "len_digest_proc",
	      "Bad magic number (digest_proc)",0);
    
    if (DIGEST_PROC_TYPE_magic != ptr->digest_type->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "len_digest_proc",
	      "Bad magic number (digest_proc_type)",0);
    
    return ptr->digest_type->digest_len;
}

enum digest_proc_mode  mode_digest_proc(ptr)
     struct digest_proc *ptr;
{
    if (DIGEST_PROC_magic != ptr->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "mode_digest_proc",
	      "Bad magic number (digest_proc)",0);
    
    return ptr->mode;
}

size_t end_digest_proc(ptr,digest_buffer,digest_buffer_len)
     struct digest_proc *ptr;
     unsigned char digest_buffer[];
     size_t        digest_buffer_len;
{
    if (DIGEST_PROC_magic != ptr->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "end_digest_proc",
	      "Bad magic number (digest_proc)",0);
    
    if (DIGEST_PROC_TYPE_magic != ptr->digest_type->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "end_digest_proc",
	      "Bad magic number (digest_proc_type)",0);

    if (digest_buffer_len < 1)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "end_digest_proc",
	      "Bad digest_buffer_len",0);

    /* bzero is defined on hdrs/elm_defs.h */

    bzero(digest_buffer,digest_buffer_len);
        
    switch (ptr->mode) {
	size_t len;
	
    case digest_proc_initial:
	if (! ptr->context.dummy)
	    ptr->digest_type->alloc_ctx(ptr);
	
	if (! ptr->context.dummy)
	    panic("DIGEST PANIC",__FILE__,__LINE__,
		  "feed_digest_proc",
		  "Context not allocated",0);
	
	/* Start it */
	ptr->digest_type->init_ctx(ptr);

	ptr->mode = digest_proc_started;
	/* FALLTHRU */

    case digest_proc_started:
	if (! ptr->context.dummy)
	    panic("DIGEST PANIC",__FILE__,__LINE__,
		  "end_digest_proc",
		  "Context not allocated",0);
	
	
	ptr->digest_result =
	    safe_realloc(ptr->digest_result,
			 ptr->digest_type->digest_len);

	ptr->digest_type->final_ctx(ptr,
				    ptr->digest_result,
				    ptr->digest_type->digest_len);

	ptr->mode = digest_proc_ended;
	/* FALLTHRU */
    case digest_proc_ended:
	if (!ptr->digest_result)
	    panic("DIGEST PANIC",__FILE__,__LINE__,
		  "end_digest_proc",
		  "digest not stored",0);
	    	
	len = ptr->digest_type->digest_len;
	if (len > digest_buffer_len) {
	    DPRINT(Debug,10,(&Debug,
			    "end_digest_proc: %p: truncated, digest length %zu, buffer length %zu\n",
			     ptr,ptr->digest_type->digest_len,digest_buffer_len));
	    len = digest_buffer_len;
	}

	memcpy(digest_buffer,ptr->digest_result,len);

	return len;
    }

    panic("DIGEST PANIC",__FILE__,__LINE__,
	  "end_digest_proc",
	  "Bad state",0);

    return 0;
}


size_t result_digest_proc(ptr,digest_buffer,digest_buffer_len)
     struct digest_proc *ptr;
     unsigned char digest_buffer[];
     size_t        digest_buffer_len;
{
    if (DIGEST_PROC_magic != ptr->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "result_digest_proc",
	      "Bad magic number (digest_proc)",0);
    
    if (DIGEST_PROC_TYPE_magic != ptr->digest_type->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "result_digest_proc",
	      "Bad magic number (digest_proc_type)",0);

    if (digest_buffer_len < 1)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "result_digest_proc",
	      "Bad digest_buffer_len",0);

    /* bzero is defined on hdrs/elm_defs.h */

    bzero(digest_buffer,digest_buffer_len);
        
    switch (ptr->mode) {
	size_t len;

    case digest_proc_initial:
	 panic("DIGEST PANIC",__FILE__,__LINE__,
	       "result_digest_proc",
	       "Digest not collected",0);
	 break;
    case digest_proc_started:
	 panic("DIGEST PANIC",__FILE__,__LINE__,
	       "result_digest_proc",
	       "end_digest_proc not called",0);
	 break;

    case digest_proc_ended:
	if (!ptr->digest_result)
	    panic("DIGEST PANIC",__FILE__,__LINE__,
		  "result_digest_proc",
		  "digest not stored",0);
	    	
	len = ptr->digest_type->digest_len;
	if (len > digest_buffer_len) {
	    DPRINT(Debug,10,(&Debug,
			     "result_digest_proc: %p: truncated, digest length %zu, buffer length %zu\n",
			     ptr,ptr->digest_type->digest_len,digest_buffer_len));
	    len = digest_buffer_len;
	}
	
	memcpy(digest_buffer,ptr->digest_result,len);
	
	return len;
    }
    
    panic("DIGEST PANIC",__FILE__,__LINE__,
	  "result_digest_proc",
	  "Bad state",0);

    return 0;
}

enum digest_proc_comp digest_proc_equal(ptr1,ptr2)
     const struct digest_proc *ptr1;
     const struct digest_proc *ptr2;
{
    enum digest_proc_comp ret = digest_proc_differ;
    
    if (DIGEST_PROC_magic != ptr1->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "digest_proc_equal",
	      "Bad magic number (digest_proc, ptr1)",0);
    
    if (DIGEST_PROC_TYPE_magic != ptr1->digest_type->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "digest_proc_equal",
	      "Bad magic number (digest_proc_type, ptr1)",0);

    if (DIGEST_PROC_magic != ptr2->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "digest_proc_equal",
	      "Bad magic number (digest_proc, ptr2)",0);
    
    if (DIGEST_PROC_TYPE_magic != ptr1->digest_type->magic)
	panic("DIGEST PANIC",__FILE__,__LINE__,
	      "digest_proc_equal",
	      "Bad magic number (digest_proc_type, ptr2)",0);


    DPRINT(Debug,14,(&Debug,"digest_proc_equal: ptr1=%p (digest_type %p len %zu)",
		     ptr1, ptr1->digest_type,ptr1->digest_type->digest_len));

    if (digest_proc_ended == ptr1->mode && ptr1->digest_result) {
	size_t i;

	DPRINT(Debug,14,(&Debug," ["));

	for (i = 0; i < ptr1->digest_type->digest_len; i++)
	    DPRINT(Debug,14,(&Debug,"%02x",ptr1->digest_result[i]));

	DPRINT(Debug,14,(&Debug,"]"));
    }

    DPRINT(Debug,14,(&Debug," ptr2=%p (digest_type %p len %zu)",
		     ptr2, ptr2->digest_type,ptr2->digest_type->digest_len));

    if (digest_proc_ended == ptr2->mode && ptr2->digest_result) {
	size_t i;

	DPRINT(Debug,14,(&Debug," ["));

	for (i = 0; i < ptr2->digest_type->digest_len; i++)
	    DPRINT(Debug,14,(&Debug,"%02x",ptr2->digest_result[i]));

	DPRINT(Debug,14,(&Debug,"]"));
    }
		     
    DPRINT(Debug,14,(&Debug,"\n"));

    if (digest_proc_ended != ptr1->mode || ! ptr1->digest_result  ||
	digest_proc_ended != ptr1->mode || ! ptr1->digest_result) {
	ret = digest_proc_incomplete;
	goto out;
    }

    if (ptr1->digest_type == ptr2->digest_type) {
	size_t x;

	ret = digest_proc_same;
	DPRINT(Debug,14,(&Debug,"digest_proc_equal: Digest types match, len %zu [",
			 ptr1->digest_type->digest_len));
	
	
	for (x = 0; x < ptr1->digest_type->digest_len; x++) {
	    if (ptr1->digest_result[x]  != ptr2->digest_result[x]) {
		ret = digest_proc_differ;
		
		DPRINT(Debug,14,(&Debug,"XX"));
		
	    } else {
		DPRINT(Debug,14,(&Debug,"%02x",ptr1->digest_result[x]));
	    }
	}

	DPRINT(Debug,14,(&Debug,"]\n"));
	
    } else {
	DPRINT(Debug,14,(&Debug,"digest_proc_equal: Digest types differs\n"));
	
	ret = digest_proc_differ;
    }


 out:

    DPRINT(Debug,14,(&Debug,"digest_proc_equal=%d",ret));
    switch (ret) {
    case digest_proc_incomplete:  DPRINT(Debug,14,(&Debug, " digest_proc_incomplete")); break;
    case digest_proc_differ:      DPRINT(Debug,14,(&Debug, " digest_proc_differ"));     break;
    case digest_proc_same:	  DPRINT(Debug,14,(&Debug, " digest_proc_same"));       break;
    }
    DPRINT(Debug,14,(&Debug,"\n"));

    return ret;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
