static char rcsid[] = "@(#)$Id: delay_unlink.c,v 2.4 2024/06/10 18:05:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.4 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/


#include "elm_defs.h"
#include "schedule_time.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"file");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

#define DELAY_UNLINK_magic	0xF103


static void timed_task_free_du P_((struct timed_task **task));
static void inc_timed_task_du_refcount P_((struct timed_task *task));
static enum timed_task_r timed_task_action_du
    P_((struct timed_task *task /* NULL if task on alter_timed_task()
				   was not given */,
	const struct schedule_timelimit *now));


struct delay_unlink {
    unsigned short            magic;      /* DELAY_UNLINK_magic */

    int refcount;

    dev_t     st_dev;
    ino_t     st_ino; 
    
    char  ** name;
    size_t   name_count;

    struct timed_task_handle  * timer;

    struct schedule_timelimit deadline;

    unsigned      may_unlink      :1;
    unsigned      time_expired    :1;
    unsigned      have_st_dev_ino :1;
    unsigned      unlinked        :1;
    
} * alloc_delay_unlink P_((const char *name,
			   int fd, int timer_secs));


static int delay_unlink_add_name0 P_((struct delay_unlink * rec,
				     struct stat         * stat_buffer,
				     const char          * name));
static int delay_unlink_add_name0(rec,stat_buffer,name)
     struct delay_unlink * rec;
     struct stat         * stat_buffer;
     const char          * name;
{
    int ret = 0;
    
    if (DELAY_UNLINK_magic != rec->magic)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "delay_unlink_add_name0",
	      "Bad magic number (delay_unlink)",0);

    DPRINT(Debug,11,(&Debug,"delay_unlink_add_name0: rec %p: file %s: dev/ino = %ld/%ld",
		     rec,name,
		     (long)(stat_buffer->st_dev),
		     (long)(stat_buffer->st_ino)));
    
    if (rec->have_st_dev_ino) {
    	      
	if ( rec->st_dev == stat_buffer->st_dev &&
	     rec->st_ino == stat_buffer->st_ino) {
	    
	    size_t n = rec->name_count +1;
	    
	    DPRINT(Debug,11,(&Debug," OK --> %zu names\n",n));
	    
	    rec->name = safe_array_realloc(rec->name,n,
					   sizeof (rec->name[0]));
	    
	    rec->name[rec->name_count] = safe_strdup(name);

	    DPRINT(Debug,11,(&Debug,"delay_unlink_add_name0: rec %p: #%zu = %s\n",
			     rec,rec->name_count,rec->name[rec->name_count]));
	    
	    rec->name_count =  n;
	    ret = 1;

	    rec->unlinked = 0;
	    
	} else {
	    DPRINT(Debug,11,(&Debug," != rec dev/ino = %ld/%ld\n",
			     (long)(rec->st_dev),
			     (long)(rec->st_ino)));	
	}
	    
    } else {	
	DPRINT(Debug,11,(&Debug,": no rec dev/ino information!\n",
			 rec,name));
    }
	
    DPRINT(Debug,11,(&Debug,"delay_unlink_add_name0=%d; rec=%p\n",ret,rec));
    return ret;
}

int delay_unlink_add_name(rec,name)
     struct delay_unlink * rec;
     const char          * name;
{
    int ret = 0;
    struct stat stat_buffer;
    int r;
    
    if (DELAY_UNLINK_magic != rec->magic)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "delay_unlink_add_name",
	      "Bad magic number (delay_unlink)",0);

    DPRINT(Debug,11,(&Debug,"delay_unlink_add_name: rec %p: file %s",
		     rec,name));

    
    if (rec->name) {
	size_t i;

	for (i = 0; i < rec->name_count; i++) {

	    if (0 == strcmp(rec->name[i],name)) {
		DPRINT(Debug,11,(&Debug,": Name already added #%zu\n",i));
		ret = 1;
		goto out;
	    }
	}
    }

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)&stat_buffer,sizeof stat_buffer);

    r = stat(name,&stat_buffer);

    if (-1 == r) {
	int err = errno;

	DPRINT(Debug,11,(&Debug,": stat failed:  %s (errno %d)\n",
			 strerror(err),err));

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDUStatFail,
			  "File %s stat failed (errno=%d): %s"),
		  name,err,strerror(err));
	
    } else if (0 == r) {
	DPRINT(Debug,11,(&Debug,": dev/ino = %ld/%ld\n",
			 (long)stat_buffer.st_dev,
			 (long)stat_buffer.st_ino));

	ret = delay_unlink_add_name0(rec,&stat_buffer,name);
    } else {
	DPRINT(Debug,11,(&Debug,"\n"));	    
    }

 out:
    DPRINT(Debug,11,(&Debug,"delay_unlink_add_name=%d (%s)\n",
		     ret,
		     ret ? "succeed" : "failure"));
    
    return ret;
}

static void delay_unlink_check_unlink P_((struct delay_unlink * rec,
					  int force));
static void delay_unlink_check_unlink(rec,force)
     struct delay_unlink * rec;
     int force;
{
    if (DELAY_UNLINK_magic != rec->magic)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "delay_unlink_check_unlink",
	      "Bad magic number (delay_unlink)",0);

    DPRINT(Debug,11,(&Debug,"delay_unlink_check_unlink: rec %p%s",
		     rec,
		     force ? ", force" : ""));

    if (rec->have_st_dev_ino) {
	DPRINT(Debug,11,(&Debug,", dev/ino = %ld/%ld",
			 (long)(rec->st_dev),
			 (long)(rec->st_ino)));	
	
	
    } else {	
	DPRINT(Debug,11,(&Debug,", no dev/ino information!"));
    }

    if (rec->may_unlink) {
	DPRINT(Debug,11,(&Debug,", may unlink"));
    }
    if (rec->time_expired) {
	DPRINT(Debug,11,(&Debug,", time expired"));
    }
    if (rec->unlinked) {
	DPRINT(Debug,11,(&Debug,", unlinked\n"));
	return;
    }

    if (! rec->may_unlink) {
	if (force) {
	    DPRINT(Debug,11,(&Debug,", keeping files on free"));
	}

	DPRINT(Debug,11,(&Debug,", will not unlink\n"));
	return;	
    }
    if (force) {
	DPRINT(Debug,11,(&Debug,", final unlink"));
    } else if (! rec->time_expired && rec->timer) {
	DPRINT(Debug,11,(&Debug,", will not unlink\n"));
	return;
    }
    DPRINT(Debug,11,(&Debug,", checking unlinkable files\n"));
    
    if (rec->name) {
	size_t i;

	for (i = 0; i < rec->name_count; i++) {
	    struct stat stat_buffer;
	    int r;

	    /* bzero is defined hdrs/elm_defs.h */
	    bzero((void *)&stat_buffer,sizeof stat_buffer);
	    
	    r = stat(rec->name[i],&stat_buffer);
	    if (-1 == r) {
		int err = errno;
		
		DPRINT(Debug,11,(&Debug,"delay_unlink_check_unlink: rec %p: #%zu: stat %s: %s (errno %d)\n",
				 rec,i, rec->name[i],strerror(err),err));
		
		if (ENOENT == err)
		    continue;

		lib_error(CATGETS(elm_msg_cat, MeSet, MeDUStatFail,
				  "File %s stat failed (errno=%d): %s"),
			  rec->name[i],err,strerror(err));

	    } else if (0 == r) {

		DPRINT(Debug,11,(&Debug,"delay_unlink_check_unlink: rec %p: #%zu: file %s: dev/ino = %ld/%ld",
				 rec,i,
				 rec->name[i],
				 (long)stat_buffer.st_dev,
				 (long)stat_buffer.st_ino));

		
		
		if (rec->have_st_dev_ino) {
		    if ( rec->st_dev != stat_buffer.st_dev ||
			 rec->st_ino != stat_buffer.st_ino) {
			
			DPRINT(Debug,11,(&Debug," != rec dev/ino = %ld/%ld\n",
					 (long)(rec->st_dev),
					 (long)(rec->st_ino)));	
			continue;
		    } else {
			DPRINT(Debug,11,(&Debug, " OK\n"));
		    }
		} else {
		    DPRINT(Debug,11,(&Debug, "\n"));
		}

		r = unlink(rec->name[i]);

		if (-1 == r) {

		    int err = errno;
		    
		    DPRINT(Debug,11,(&Debug,"delay_unlink_check_unlink: rec %p: #%zu: unlink %s: %s (errno %d)\n",
				     rec,i,rec->name[i],strerror(err),err));
		    
		    if (ENOENT == err)
			continue;
		    
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeDUUnlinkFail,
				      "File %s unlink failed (errno=%d): %s"),
			  rec->name[i],err,strerror(err));


		} else if (0 == r) {
		    DPRINT(Debug,11,(&Debug,"delay_unlink_check_unlink: rec %p: #%zu: file %s: unlinked\n",
				     rec,i,rec->name[i]));
		    
		}		
	    }
	}
    }

    /* DO not reprocess */
    rec->unlinked = 1;
}

     
void delay_unlink_mark_done(rec)
     struct delay_unlink * rec;
{
    if (DELAY_UNLINK_magic != rec->magic)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "delay_unlink_mark_done",
	      "Bad magic number (delay_unlink)",0);

    DPRINT(Debug,11,(&Debug,"delay_unlink_mark_done: rec %p\n",
		     rec));

    rec->may_unlink = 1;

    delay_unlink_check_unlink(rec,0);
}

static struct timed_task * alloc_timed_task P_((struct delay_unlink  * delayed_file));

struct delay_unlink * alloc_delay_unlink(name,fd,timer_secs)
     const char *name;
     int fd;
     int timer_secs;
{
    struct stat stat_buffer;
    int r;
            
    struct delay_unlink *ret =
	safe_zero_alloc(sizeof (*ret));

    int ok = 1;
    
    ret->refcount = 1;
    
    ret->st_dev   = 0;
    ret->st_ino   = 0;

    ret->name       = NULL;
    ret->name_count = 0;

    ret->timer      = NULL;
    ret->deadline   = NO_schedule_timelimit;
    
    ret->may_unlink      = 0;
    ret->time_expired    = 0;
    ret->have_st_dev_ino = 0;
    ret->unlinked        = 0;

    ret->magic = DELAY_UNLINK_magic;
    
    if (fd >= 0) {

	/* bzero is defined hdrs/elm_defs.h */
	bzero((void *)&stat_buffer,sizeof stat_buffer);
	
	r  = fstat(fd,&stat_buffer);
	if (-1 == r) {
	    int err = errno;

	    DPRINT(Debug,11,(&Debug,"alloc_delay_unlink: ret=%p: fstat %d: file %s: %s (errno %d)\n",
			     ret,fd,name,strerror(err),err));
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDUFStatFail,
			      "File %s fstat (fd=%d) failed (errno=%d): %s"),
		      name,fd,err,strerror(err));

	    ok = 0;
	    
	} else if (0 == r) {

	    ret->st_dev = stat_buffer.st_dev;
	    ret->st_ino = stat_buffer.st_ino;

	    ret->have_st_dev_ino = 1;

	    DPRINT(Debug,11,(&Debug,"alloc_delay_unlink: ret=%p: fstat %d: file %s: dev/ino = %lu/%lu\n",
			     ret,fd,name,
			     (unsigned long)stat_buffer.st_dev,
			     (unsigned long)stat_buffer.st_ino));
	}	
    }
    
    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)&stat_buffer,sizeof stat_buffer);
    
    r = stat(name,&stat_buffer);

    if (-1 == r) {
	int err = errno;

	DPRINT(Debug,11,(&Debug,"alloc_delay_unlink: ret=%p: stat %s: %s (errno %d)\n",
			 ret,name,strerror(err),err));

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDUStatFail,
			  "File %s stat failed (errno=%d): %s"),
		  name,err,strerror(err));

	ok = 0;
	
    } else if (0 == r) {

	DPRINT(Debug,11,(&Debug,"alloc_delay_unlink: ret=%p: stat %s: dev/ino = %lu/%lu\n",
			 ret,name,
			 (unsigned long)stat_buffer.st_dev,
			 (unsigned long)stat_buffer.st_ino));
	
	if (0 == ret->have_st_dev_ino) {

	    DPRINT(Debug,11,(&Debug,"alloc_delay_unlink: ret=%p: using stat for dev/ino\n",
			     ret));
	    
	    ret->st_dev = stat_buffer.st_dev;
	    ret->st_ino = stat_buffer.st_ino;
	    
	    ret->have_st_dev_ino = 1;
	}
	
	if (delay_unlink_add_name0(ret,&stat_buffer,name)) {
	    DPRINT(Debug,11,(&Debug,"alloc_delay_unlink: added name %s\n",name));
	}
    }

    
    if (timer_secs >= 0) {

	if (schedule_set_next_timeout(&(ret->deadline),
				      timer_secs)) {

	    struct timed_task *task = alloc_timed_task(ret);
	    
	    if (alter_timed_task(&(ret->timer),
				 task,
				 timed_task_free_du,
				 inc_timed_task_du_refcount,
				 timed_task_action_du,
				 & (ret->deadline))) {
		    DPRINT(Debug,11,(&Debug,
				     "alloc_delay_unlink: ret=%p: Timer installed\n",
				     ret));
	    } else {
		DPRINT(Debug,11,(&Debug,
				 "alloc_delay_unlink: ret=%p: failed to set timer, POLL_method=%d\n",
				 ret,POLL_method));
		ok = 0;
		
	    }
	    
	    timed_task_free_du(&task);
	    
	} else {
	    DPRINT(Debug,11,(&Debug,"alloc_delay_unlink: ret=%p: failed to set deadline, timer_secs=%d\n",
			     ret,timer_secs));

	    ok = 0;
	}
	
    }
    
    if (!ok) {
	DPRINT(Debug,11,(&Debug,"alloc_delay_unlink: ret=%p: resetting\n"));
	free_delay_unlink(&ret);
    }

    DPRINT(Debug,11,(&Debug,"alloc_delay_unlink=%p (%s)\n",
		     ret,
		     ok ? "succeed" : "failure"));
    
    return ret;
}

void inc_delay_unlink_recount(delay_unlink_file)
     struct delay_unlink *delay_unlink_file;
{
    if (DELAY_UNLINK_magic != delay_unlink_file->magic)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "inc_delay_unlink_recount",
	      "Bad magic number (delay_unlink)",0);

    delay_unlink_file->refcount++;
}

enum free_delay_mode { free_normal = 0,
		       from_timed_task_free_du,
		       from_timed_task_action_du
};

static void free_delay_unlink0 P_((struct delay_unlink **delay_unlink_file,
				   enum free_delay_mode mode));
static void free_delay_unlink0(delay_unlink_file,mode)
     struct delay_unlink **delay_unlink_file;
     enum free_delay_mode mode;
{
    if (DELAY_UNLINK_magic != (*delay_unlink_file)->magic)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "free_delay_unlink0",
	      "Bad magic number (delay_unlink)",0);

    DPRINT(Debug,12,(&Debug,"free_delay_unlink0: rec %p: refcount=%d mode=%d",
		     (*delay_unlink_file),
		     (*delay_unlink_file)->refcount,mode));
    switch(mode) {
    case free_normal:               DPRINT(Debug,12,(&Debug," free_normal"));               break;
    case from_timed_task_free_du:   DPRINT(Debug,12,(&Debug," from_timed_task_free_du"));   break;
    case from_timed_task_action_du: DPRINT(Debug,12,(&Debug," from_timed_task_action_du")); break;
    }
    DPRINT(Debug,12,(&Debug,"\n"));

    if ((*delay_unlink_file)->refcount < 1)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "free_delay_unlink0",
	      "Bad refcount",0);

    (*delay_unlink_file)->refcount--;
    if ((*delay_unlink_file)->refcount > 0) {
	
	
	switch(mode) {
	case free_normal: 
	    if ((*delay_unlink_file)->time_expired &&
		(*delay_unlink_file)->timer) {
		
		DPRINT(Debug,11,(&Debug,
				 "free_delay_unlink0: rec %p: time expired, have %d other references, resetting timer %p\n",
				 (*delay_unlink_file),(*delay_unlink_file)->refcount,
				 (*delay_unlink_file)->timer
				 ));
		
		alter_timed_task(& ((*delay_unlink_file)->timer),
				 NULL,
				 NO_timed_task_free,
				 NO_inc_timed_task_refcount,
				 NO_timed_task_action,
				 &((*delay_unlink_file)->deadline));
	    } else if ((*delay_unlink_file)->timer) {
		DPRINT(Debug,11,(&Debug,"free_delay_unlink0: rec %p: have %d other references, marking timer %p for free_schedule() cleanup\n",
				 (*delay_unlink_file),(*delay_unlink_file)->refcount,
				 (*delay_unlink_file)->timer));
		timed_task_alter_cleanup(& ((*delay_unlink_file)->timer),
					 tt_allow_free_schedule_cleanup);
				 
	    }
	    break;
	case from_timed_task_action_du:
	    if ((*delay_unlink_file)->timer) {
		DPRINT(Debug,11,(&Debug,
				 "free_delay_unlink0: rec %p: have %d other references, resetting timer %p\n",
				 (*delay_unlink_file),(*delay_unlink_file)->refcount,
				 (*delay_unlink_file)->timer
				 ));
		
		alter_timed_task(& ((*delay_unlink_file)->timer),
				 NULL,
				 NO_timed_task_free,
				 NO_inc_timed_task_refcount,
				 NO_timed_task_action,
				 &((*delay_unlink_file)->deadline));		
	    }
	    
	    break;
	case from_timed_task_free_du:
	    DPRINT(Debug,11,(&Debug,
			     "free_delay_unlink0: rec %p: from timed_task_free_du, have %d other references\n",
			     (*delay_unlink_file),(*delay_unlink_file)->refcount));
	    break;
	}
	/* Just reset this reference */

	DPRINT(Debug,12,(&Debug,"free_delay_unlink0: rec %p: resetting this reference\n",
			 *delay_unlink_file));
	
	*delay_unlink_file = NULL;
	return;
    }

    DPRINT(Debug,11,(&Debug,"free_delay_unlink0: rec %p: this was last reference\n",
		     (*delay_unlink_file)));

    if ((*delay_unlink_file)->timer) {	    
	timed_task_alter_cleanup(& ((*delay_unlink_file)->timer),
				 tt_discard_free_schedule_cleanup);
    }
    
    switch(mode) {
    case free_normal:
    case from_timed_task_action_du:
	if ((*delay_unlink_file)->timer) {	    
	    DPRINT(Debug,11,(&Debug,
			     "free_delay_unlink0: rec %p: resetting timer handle %p\n",
			     (*delay_unlink_file),(*delay_unlink_file)->timer));
		   
	    alter_timed_task(& ((*delay_unlink_file)->timer),
			     NULL,
			     NO_timed_task_free,
			     NO_inc_timed_task_refcount,
			     NO_timed_task_action,
			     &((*delay_unlink_file)->deadline));
	}
	break;
    case from_timed_task_free_du: 
	if ((*delay_unlink_file)->timer) {
	    DPRINT(Debug,11,(&Debug,
			     "free_delay_unlink0: rec %p: from timed_task_free_du, losing timer handle %p\n",
			     (*delay_unlink_file),(*delay_unlink_file)->timer));
	    (*delay_unlink_file)->timer = NULL;
	}
	break;
    }
    
    delay_unlink_check_unlink(*delay_unlink_file,1);

    if ((*delay_unlink_file)->name) {
	size_t i;

	for (i = 0; i < (*delay_unlink_file)->name_count; i++) {
	    if ((*delay_unlink_file)->name[i]) {
		free((*delay_unlink_file)->name[i]);
		(*delay_unlink_file)->name[i] = NULL;
	    }
	}

	free((*delay_unlink_file)->name);
	(*delay_unlink_file)->name = NULL;
    }
    (*delay_unlink_file)->name_count = 0;
	
    (*delay_unlink_file)->magic = 0;  /* Invalidate */
    free(*delay_unlink_file);
    *delay_unlink_file = 0;    
}


void free_delay_unlink(delay_unlink_file)
     struct delay_unlink **delay_unlink_file;
{
    if (DELAY_UNLINK_magic != (*delay_unlink_file)->magic)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "free_delay_unlink",
	      "Bad magic number (delay_unlink)",0);

    DPRINT(Debug,11,(&Debug,"free_delay_unlink: rec %p: refcount=%d\n",
		     (*delay_unlink_file),
		     (*delay_unlink_file)->refcount));
    
    if ((*delay_unlink_file)->refcount < 1)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "free_delay_unlink",
	      "Bad refcount",0);

    free_delay_unlink0(delay_unlink_file,free_normal);    
}


#define TIMED_TASK_delay_unlink_magic	0xF104

struct timed_task {
    unsigned short            magic;         /* TIMED_TASK_delay_unlink_magic */

    int refcount;
    
    struct delay_unlink  * delayed_file;     
    
};

static struct timed_task * alloc_timed_task(delayed_file)
     struct delay_unlink  * delayed_file;
{
    struct timed_task  * ret =
	safe_zero_alloc(sizeof (*ret));

    ret->magic = TIMED_TASK_delay_unlink_magic;
    ret->refcount = 1;

    ret->delayed_file = delayed_file;
    if (ret->delayed_file)
	inc_delay_unlink_recount(ret->delayed_file);
    
    return ret;
}

S_(timed_task_free_f timed_task_free_du)
static void timed_task_free_du(task)
     struct timed_task **task;
{
    if (TIMED_TASK_delay_unlink_magic != (*task)->magic)
	  panic("SCHEDULE PANIC",__FILE__,__LINE__,
		"timed_task_free_du",
		"Bad magic number (timed_task)",0);

    DPRINT(Debug,11,(&Debug,"timed_task_free_du: task %p: refcount=%d\n",
		     (*task),
		     (*task)->refcount));
    
    if ((*task)->refcount < 1)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "timed_task_free_du",
	      "Bad refcount",0);
	
    (*task)->refcount--;
    if ((*task)->refcount > 0) {
	/* Just reset this reference */
	DPRINT(Debug,11,(&Debug,"timed_task_free_du: task %p: reset this reference\n",(*task)));

	*task = NULL;
	return;
    }

    DPRINT(Debug,11,(&Debug,"timed_task_free_du: task %p: this is last reference\n",(*task)));
    
    if ((*task)->delayed_file) {

	DPRINT(Debug,11,(&Debug,"timed_task_free_du: task %p: freeing delayed file rec %p\n",
			 (*task),
			 (*task)->delayed_file));


	free_delay_unlink0(& ((*task)->delayed_file),from_timed_task_free_du);
    }
	
    (*task)->magic = 0; /* Invalidate */
    
    free(*task);
    *task = NULL;
}

S_(inc_timed_task_refcount_f inc_timed_task_du_refcount)
static void inc_timed_task_du_refcount(task)
     struct timed_task *task;
{
    if (TIMED_TASK_delay_unlink_magic != task->magic)
	panic("SCHEDULE PANIC",__FILE__,__LINE__,
	      "inc_timed_task_du_refcount",
	      "Bad magic number (timed_task)",0);

    task->refcount++;
}

S_(timed_task_action_f timed_task_action_du)
static  enum timed_task_r timed_task_action_du(task,now)
     struct timed_task *task /* NULL if task on alter_timed_task()
				was not given */;
     const struct schedule_timelimit *now;
{
    enum timed_task_r r = timed_task_default;
    
    if (task) {
	if (TIMED_TASK_delay_unlink_magic != task->magic)
	    panic("SCHEDULE PANIC",__FILE__,__LINE__,
		  "timed_task_action_du",
		  "Bad magic number (timed_task)",0);

	DPRINT(Debug,11,(&Debug,"timed_task_action_du: task %p, processing timer action",
			 task));
	
	if (task->delayed_file) {
	    DPRINT(Debug,11,(&Debug,"\n"));
	    
	    if (DELAY_UNLINK_magic != task->delayed_file->magic)
		panic("SCHEDULE PANIC",__FILE__,__LINE__,
		      "timed_task_action_du",
		      "Bad magic number (delay_unlink)",0);

	    if (task->delayed_file->time_expired) {
		DPRINT(Debug,11,(&Debug,
				 "timed_task_action_du: task %p, delayed_file rec %p marked already time expired\n",
				 task,task->delayed_file));
	    } else {
		DPRINT(Debug,11,(&Debug,
				 "timed_task_action_du: task %p, delayed_file rec %p marking time expired\n",
				 task,task->delayed_file));
		/* Mark timer launched */
		task->delayed_file->time_expired = 1;
	    }
		
	    delay_unlink_check_unlink(task->delayed_file,0);

	    if (1 == task->delayed_file->refcount) {
		DPRINT(Debug,11,(&Debug,
				 "timed_task_action_du: task %p, this is last reference to delayed_file rec %p -- freeing it\n",
				 task,task->delayed_file));
		free_delay_unlink0(& (task->delayed_file),from_timed_task_action_du);
	    }
	    
	} else {
	    DPRINT(Debug,11,(&Debug,", no delayed_file rec\n"));
	}
	
	r =  timed_task_disable;
    } else {
	DPRINT(Debug,11,(&Debug,"timed_task_action_du: no task\n"));
    }

    DPRINT(Debug,11,(&Debug,"timed_task_action_du=%d",r));
    switch (r) {
    case timed_task_default:
	DPRINT(Debug,11,(&Debug," timed_task_default"));
	break;
    case timed_task_retry:      /* mark task as not runned */
	DPRINT(Debug,11,(&Debug," timed_task_retry"));
	break;
    case timed_task_disable:     /* mark task as already runned */
	DPRINT(Debug,11,(&Debug," timed_task_disable"));
	break;
    }    
    if (task) {
	DPRINT(Debug,11,(&Debug," task=%p",task));
    }
    DPRINT(Debug,11,(&Debug,"\n"));
    
    return r;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
