static char rcsid[] = "@(#)$Id: debug.c,v 2.18 2023/12/13 16:55:32 hurtta Exp $";

/************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.18 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ***********************************************************************/

#include "elm_defs.h"
#include "s_elm.h"

#include "patchlevel.h"

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}


struct debug_target {    
    int                   debug_fd;
    const char          * file;

    struct class_level_pair {
	const char          * class;
	int                   level;    
    }  *                   match;
    int                    match_count;   

    struct debug_target * next;

    /* In startup quite huge number of debug output can be
       needed to be buffered */
    char                buffer[4000];
    volatile int        buffer_ptr;
    size_t              lostcount;

    unsigned int        use_temp :1;
    
} * debug_list = NULL;

static const char * DEBUG_base = "ELMLIB";

/* Given arhument must be statically alloced */
void init_debugfile(progname) 
     const char *progname;
{
    DEBUG_base = progname;
}

static int flush_debug_buffer P_((struct debug_target *a,
				  struct debug_struct *prefix));

static void  init_debug P_((struct debug_target * A));
static void  init_debug(A)
     struct debug_target * A;
{
    DEBUG_VAR(temp,__FILE__,"");
    char * filename = elm_message(FRM("%s/%s"),
				  A-> use_temp ?
				  "/tmp" : home,
				  A->file);
    char * pp;

    int fd = -1;
    int r;

    temp.target = A;

    /* For compability of old method 
       "ELM:debug.info" is renamed to "ELM:debug.last"
     */
    
    if (! A-> use_temp &&
	NULL != (pp = strrchr(A->file,':')) &&
	0 == strcmp(pp,":debug.info") &&

	access(filename, ACCESS_EXISTS) == 0) {     /* already one! */
	char *newfname = elm_message(FRM("%s/%s:debug.last"), 
				     home,DEBUG_base);
	if (0 == rename(filename, newfname)) {
	    debug_action_call(&temp,
			      "RENAMEd old debug file %s to %s\n",
			      filename,newfname);
	}
	free(newfname);	
    }
    
    if (0 == can_open(filename,"a"))
	fd = open(filename,O_CREAT|O_WRONLY|O_APPEND,00600);

    if (-1 == fd) {	
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldNotOpenDebugFile,
			  "Could not open file %s for debug output!\n"),
		  filename);
	
	goto out;
    }
    
    if (-1 != A->debug_fd)
	close(A->debug_fd);
    A->debug_fd = fd;

#ifdef FD_CLOEXEC
    r = fcntl(fd, F_SETFD, FD_CLOEXEC);
#else
    r = fcntl(fd, F_SETFD, 1);
#endif    
    
    if (-1 == r) {
	int err UNUSED_VAROK = errno;
        
	debug_action_call(&temp,
			  "init_debug: Failed to set close-on-exec flag for %s: %s\n",
			  filename,
			  strerror(err));
    } else if (0 == r) {
	debug_action_call(&temp,
			  "init_debug: Set close-on-exec flag for %s\n",
			  filename);
    }
    
    elm_chown(filename, userid, groupid,NULL); /* file owned by user */

    if (0 != A->buffer_ptr)    
	debug_action_call(&temp,
			  "\n======================================================\n");

    debug_action_call(&temp,
		      "Debug output of the ELM library. Version %s PL%s\n\n",
		      VERSION,PATCHLEVEL);
    
 out:
    free(filename);
}

static void close_all P_((void));
static void close_all() 
{
    DEBUG_VAR(temp,__FILE__,"");
    struct debug_target  * A;

    for (A = debug_list; A; A = A->next) {
	temp.target = A;
	
	debug_action_call(&temp,"\nClosing debug file...\n");
	
	flush_debug_buffer(A,NULL);

	if (-1 != A->debug_fd)
	    close(A->debug_fd);

	A->debug_fd = -1;	
    }
}

int set_debugging(arg)
     const char *arg;
{
    struct debug_target      * X = NULL;
    char * match_filename   = NULL;
    struct class_level_pair    C;

    struct class_level_pair   *tmp1;
    int use_temp = 0;

    DEBUG_VAR(temp,__FILE__,"");

    char * ptr;
    long l;

    C.class = "";
    C.level = 0;

    l = strtol(arg,&ptr,10);

    if ('\0' == *ptr)
	C.level = l;
    else if (NULL == (ptr = strpbrk(arg,":=#"))) {
	C.class = arg;
	C.level = 1;
    } else {
	char *ptr1;
	char * tmp__1 = safe_strdup(arg);

	tmp__1[ptr-arg] = '\0';
	C.class = tmp__1;

	if ('=' == *ptr) {
	    match_filename = safe_strdup(ptr+1);
				  
	    ptr = strchr(match_filename,':');
	    if (ptr) {
		*ptr = '\0';
		ptr++;
	    }
	} else if ('#' == *ptr) {

	    match_filename = safe_strdup(ptr+1);
	    use_temp = 1;
	    
	    ptr = strchr(match_filename,':');
	    if (ptr) {
		*ptr = '\0';
		ptr++;
	    }

	    
	    
	} else
	    ptr++;
	
	
	if (!ptr) 
	    C.level = 1;
	else {
	    l = strtol(ptr,&ptr1,10);
	    if ('\0' != *ptr1) {
		if (match_filename)
		    free(match_filename);
		return 0;
	    }
	    C.level = l;
	}
    }

    if (!match_filename)
	match_filename = elm_message(FRM("%s:debug.info"),
				     DEBUG_base);

    for (X = debug_list; X; X = X -> next)
	if (X->use_temp == use_temp &&
	    0 == strcmp(X->file,match_filename))
	    break;

    if (X) {
	free(match_filename);
	match_filename = NULL;
    } else {
	X = safe_malloc(sizeof (*X));

	X->debug_fd = -1;
	X->file     = match_filename;       
	X->next     = debug_list;
	X->match       = NULL;
	X->match_count = 0;
	X->use_temp    = use_temp;

	X->buffer[0]   = '\0';
	X->buffer_ptr  = 0;
	X->lostcount   = 0;
	
	match_filename = NULL;

	if (home[0])
	    init_debug(X);

	debug_list = X;  
    }


    /* safe_realloc() calls DPRINT and therefore may refere X->match
       before it is assigned .. so we do not use safe_realloc() 
       on here ... */

    tmp1 = realloc(X->match, 
		   (1 + X->match_count) * sizeof (X->match[0]));

    if (tmp1) {
	X->match = tmp1;
	X->match[X->match_count++] = C;
    } else {
	debug_action_call(&temp,
			  "Can't alloc memory for class %s at debug level %d (size %d)\n",
			  C.class,C.level,
			  (1 + X->match_count) * sizeof (X->match[0]));
    }
			        
    if (!debug_list)
	atexit(close_all);

    temp.target = X;

    debug_action_call(&temp,
		      "Debug output for class %s at debug level %d\n",
		      C.class,C.level);

    return 1;
}

void debug_user_init()
{
    struct debug_target  * A;

    /* frm calls user_init() several times ... */
    
    static int init_done = 0;

    if (init_done)
	return;
    init_done = 1;

    for (A = debug_list; A; A = A->next) {
	init_debug(A);
    }
    
}

void debug_level_check(a)
     struct debug_struct *a;
{
    struct debug_target  * A;
    
    for (A = debug_list; A; A = A->next) {
	int i;

	int found = 0;
	int level = 0;
	
	for (i = 0; i < A->match_count; i++) {
	    int final = 0;
	    
	    if ('\0' == A->match[i].class[0] ||				
		(final = (0 == strcmp(a->class,A->match[i].class)))) {
		
		if (final) {
		    found = 1;
		    level = A->match[i].level;
		    break;
		} else {
		    found = 1;
		    if (level < A->match[i].level)
			level = A->match[i].level;
		}		
	    }
	}

	if (found) {
	    a->target = A;	    
	    a->active  = level;
	    
	    return;
	}		
    }

    /* If user initialized, mark that debug_level_check() need
       not recalled */
    if ('\0' != home[0])
	a->active = 0;
}

void debug_file_header(a)
     struct debug_struct *a;
{
    if (a->active > 0 &&
	-1 != a->target->debug_fd &&
	!a->header_printed) {
	/* Not really called from signal handler, but 
	   better safe and anyway this for this 
	   debug_action_sigcall() is enough 
	*/
	debug_action_sigcall(a,"\n");
	debug_action_sigcall(a,
			     "\n======================================================\n");
	debug_action_sigcall(a,"Class: %.10s            File: %s\n",
			     a->class,a->file);
	debug_action_sigcall(a,"     : %s\n",a->ID);
	debug_action_sigcall(a,
			     "======================================================\n");
	a->header_printed = 1;
    }
}

/* can be called from signal handler!! 
   RACE condition: May lose characters
*/

static void print_prefix P_((struct debug_struct *a));

static int flush_debug_buffer(a,prefix)
     struct debug_target *a;
     struct debug_struct *prefix;
{    
    if (a->debug_fd != -1) {
	int X = write(a->debug_fd,a->buffer,a->buffer_ptr);

	if (X > 0) {
	    int len = a->buffer_ptr;
	    len -= X;

	    if (len > 0)
		memmove(a->buffer,a->buffer+X,len);
	    if (len >= 0)
		a->buffer_ptr = len;
	    else 
		a->buffer_ptr = 0;

	    if (0 == a->buffer_ptr && a->lostcount > 0) {
		static char TRUNCATED[] = "\n---- Debug output truncated ----\nLost ";
		static char CHARACTERS[] = " characters.\n";
		
		X = write(a->debug_fd,TRUNCATED,sizeof TRUNCATED -1);
		
		if (X > 0) {
		    const char seq[] = "0123456789";
		    const int base = 10;
		    char lostbuffer[1 + 3 * sizeof (a->lostcount)];
		    size_t val = a->lostcount;
		    size_t r;
		    size_t x = sizeof lostbuffer;
		    
		    do {
			unsigned tmp;
			
			tmp = val % base;
			r   = val / 10;
			val = r;
			x--;
			lostbuffer[x] = seq[tmp];
			
		    } while(r && x > 0);
		    
		    X = write(a->debug_fd,lostbuffer+x,sizeof lostbuffer -x);

		    if (X > 0)
			X = write(a->debug_fd,CHARACTERS,sizeof CHARACTERS -1);
		}

		a->lostcount = 0;

		
		if (prefix) {
		    /* prefix must be smaller than sizeof a->buffer */
		    print_prefix(prefix);
		}			       
	    }

	    
	    return 1;
	}
    }
    
    return 0;
}

#define INC(p) { (p)++; if (!*(p)) break; }

/* RACE condition:
   May lose some characters when called from signal handler
*/

static void PUTC P_((struct debug_target *a, int c, struct debug_struct *prefix));

#if __GNUC__ 
__inline__
#endif
static void PUTC(a,c,prefix)
     struct debug_target *a; 
     int c;
     struct debug_struct *prefix;
{ 
    int x;

    if (a->buffer_ptr >= sizeof a->buffer)
	if (!flush_debug_buffer(a,prefix)) {
	    a->lostcount++;
	    
	    return;       /* Buffer full -- can't flush */
	}

    x = a->buffer_ptr++; 

    if (x < sizeof a->buffer) { 
	a->buffer[x] = c; 
	if ('\n' == c) flush_debug_buffer(a,prefix); 
    } 
}



static void print_prefix(a)
     struct debug_struct *a;
{
    char pidbuffer[8];
    const char seq[] = "0123456789";
    const int base = 10;
    
    /* Prefix with file name */
    if (0 == a->target->buffer_ptr ||
	( a->target->buffer_ptr > 0 &&
	  a->target->buffer_ptr < sizeof (a->target->buffer) &&
	  '\n' == a->target->buffer[a->target->buffer_ptr-1] )
	) {
	int i;
	int skip = 0;
	unsigned val;
	size_t x;
	unsigned r;
	
	/* Skip pathnames from prefix */
	for (i = 0; i < 100 && a->file[i]; i++) {
	    if ('/' == a->file[i])
		skip = i+1;
	}

	for (i = 0; i < 10 && a->file[i+skip]; i++) {
	    PUTC(a->target,a->file[i+skip],
		 NULL  /* do not call recursively print_prefix(a) */);
	}
	for (;i < 10; i++) {
	    PUTC(a->target,' ',
		 NULL  /* do not call recursively print_prefix(a) */);
	}
	PUTC(a->target,':',
	     NULL  /* do not call recursively print_prefix(a) */);

	val = getpid();
	x = sizeof pidbuffer;

	do {
	    unsigned tmp;

	    tmp = val % base;
	    r   = val / 10;
	    val = r;
	    x--;
	    pidbuffer[x] = seq[tmp];
	    	    
	} while(r && x > 0);

	while (x > 0) {
	    x--;
	    pidbuffer[x] = ' ';
	}
	for (i = 0; i < sizeof pidbuffer; i++) {
	    PUTC(a->target,pidbuffer[i],
		 NULL  /* do not call recursively print_prefix(a) */);
	}
	PUTC(a->target,':',
	     NULL  /* do not call recursively print_prefix(a) */);	
	PUTC(a->target,' ',
	     NULL  /* do not call recursively print_prefix(a) */);
	
    }
}

#define SKIP_TO_ARG(a,s) \
    if (*s != '%') { PUTC(a->target,*s,a);				\
		    if ('\n' == *s && '\0' != *(s+1)) { print_prefix(a); } \
                    continue; } \
   INC(s); \
   if ('%' == *s) { PUTC(a->target,*s,a); continue; } 

#define SKIP_WIDTH(a,s,prefix)      \
   if ('0' == *s) { INC(s); } \
   else if ('-' == *s) { INC(s); } \
   else if ('+' == *s) { INC(s); } \
   if (*s >= '1' && *s <= '9') { \
     while (*s >= '0' && *s <= '9') {  INC(s); } \
     if ('$' == *s) {  PUTC(a,'?',prefix); break; }	 \
   } else if ('*' == *s) { INC(s); } \
   if ('.' == *s) { \
      INC(s); \
      if (*s >= '1' && *s <= '9') { \
	  while (*s >= '0' && *s <= '9') { INC(s); } \
      } else if ('*' == *s) { INC(s); } \
   } \
   if ('l' == *s) { INC(s); } \
   else if ('z' == *s) { INC(s); }

#define MAX_ARGS        5

#define MAX_WIDTH     128


/* This may called from signal handler, so do not malloc
   memory on here!
*/

static int handle_common P_((struct debug_target *target,
			     struct  format_elem *elems, int pos,
			     struct debug_struct *prefix));

static int handle_common(target,elems,pos,prefix)
     struct debug_target *target;
     struct  format_elem *elems;
     int pos;
     struct debug_struct *prefix;
{
    char buffer[MAX_WIDTH];
    int len;

    switch (elems[pos].format_chr) {
	
	int l;
	char *str, *a;
	int quote;

    case 'c':
	if (V_chr_val != elems[pos].type) {
	    PUTC(target,'?',prefix); 
	    PUTC(target,'t',prefix); 
	    PUTC(target,'?',prefix); 
	    return 0;
	}
	l = 1;
	while (l < elems[pos].val1 && !elems[pos].left) { 
	    PUTC(target,elems[pos].fill,prefix); l++; 
	}
	PUTC(target,elems[pos].value.chr_val,prefix);
	while (l < elems[pos].val1 && elems[pos].left) { 
	    PUTC(target,' ',prefix); l++; 
	}
	return 1;

    case 'p':
    case 'd': case'i': case 'x': case 'X': case 'o': case 'u':
    case 'f':

	len = convert_number(buffer, sizeof buffer, &(elems[pos]));
	if (0 == len) {
	    PUTC(target,'?',prefix); 
	    PUTC(target,'l',prefix); 
	    PUTC(target,'?',prefix); 
	    return 0;
	}
	for (l = 0; l < len; l++) {
	    PUTC(target,buffer[l],prefix);  
	}
	return 1;

    case 's': case 'Q': 
	if (V_str_val != elems[pos].type) {
	    PUTC(target,'?',prefix); 
	    return 0;
	}
		    
	str = elems[pos].value.str_val;
	quote = ('Q' == elems[pos].format_chr);

	if (!str) {
	    PUTC(target,'?',prefix); 
	    PUTC(target,'t',prefix); 
	    PUTC(target,'?',prefix); 
	    return 0;
	}
		

	if (elems[pos].val2 >= 0) {
	    
	    for (a=str; a - str < elems[pos].val2 && *a; a++) 
		/* Empty */ ;
	    
	    l = a - str;


	} else
	    l = strlen(str);

	if (quote) {
	    l += 2;
	    if (elems[pos].val2 >= 0) {
		for (a=str;  a - str < elems[pos].val2 && *a; a++) {
		    if (*a == '\\' || *a == '"')
			l++;
		}
	    } else {
		for (a=str; *a; a++) {
		    if (*a == '\\' || *a == '"')
			l++;
		}
	    }
	}

	if (elems[pos].val2 < 0) 
	    elems[pos].val2 = l;
	if (quote)
	    elems[pos].val2 -= 2;

	while (l < elems[pos].val1 && !elems[pos].left) { 
	    PUTC(target,elems[pos].fill,prefix); l++; 
	}
		
	if (quote) {
	    PUTC(target,'"',prefix); 
	}
	for (a = str; a - str < elems[pos].val2 && *a; a++) { 
	    if (quote && (*a == '\\' || *a == '"')) {
		PUTC(target,'\\',prefix); 
	    }
	    PUTC(target,*a,prefix); 
	}
	if (quote) {
	    PUTC(target,'"',prefix); 
	}
	while (l < elems[pos].val1 && elems[pos].left) { 
	    PUTC(target,' ',prefix); l++; 
	}
	return 1;
    }
    return 0;

}

/* Must NOT be called from signal handler ... */
void debug_action_call(
#if ANSI_C
		       struct debug_struct *a, const char * format, ...
#else
		       a, format, va_alist
#endif
		       )
#if !ANSI_C
     struct debug_struct *a; 
     const char * format; 
     va_dcl
#endif
{
    int count, x, max_elems;
    char * format_error = NULL;
    int pos;
    
    struct  format_elem A[MAX_ARGS], *elems;
    const char * s;

    va_list vl;

    if (!a->target)
	return;

    print_prefix(a);

    x = 0;
    for (s = format; *s; s++) 
	if ('%' == *s && '%' != *(s+1))
	    x++;
    
    if (x > MAX_ARGS && 
	/* Allow failure of allocation! */
	NULL != (elems = malloc(x * sizeof (*elems))))
	max_elems = x;
    else {
	elems = A;
	max_elems = MAX_ARGS;
    }

    Va_start(vl, format);           /*  defined in hdrs/elm_defs.h */
    
    count = parse_format_args (elems,max_elems, format,vl, &format_error);

    va_end(vl);
    
    if (format_error) {
	PUTC(a->target,'?',a);  
	PUTC(a->target,'\n',a);  
	for (s = format_error; *s; s++) {
	    PUTC(a->target,*s,a);  
	}
	PUTC(a->target,'\n',a);  
    }

    pos = -1;
    for (s = format; *s; s++) {
	SKIP_TO_ARG(a,s);

	pos++;

	SKIP_WIDTH(a->target,s,a);

	if (pos < 0 || pos >= count) {
	    PUTC(a->target,'?',a); 
	    PUTC(a->target,'O',a); 
	    PUTC(a->target,'?',a); 
	    break; 
	}

	if (elems[pos].format_chr != *s) {
	    PUTC(a->target,'?',a); 
	    PUTC(a->target,'f',a); 
	    PUTC(a->target,'?',a); 
	    break;
	}

	if (!handle_common(a->target,elems,pos,a))
	    switch(*s) {
		struct string * S, *S1;
		struct charset_state *ch;
		char *temp;
		int l,X;
		char *p;

	    case 'C':
		if (V_cs_val != elems[pos].type) {
		    PUTC(a->target,'?',a); 
		    break;
		}
		ch = elems[pos].value.cs_val;
	    
		S = new_string(display_charset);
		add_state_to_string(S,ch);
		
		temp = us2s(stream_from_string(S,1,NULL));

		l = 1;
		while (l < elems[pos].val1 && !elems[pos].left) { 
		    PUTC(a->target,elems[pos].fill,a); l++; 
		}
		for (p=temp; *p; p++) {
		    PUTC(a->target,*p,a); 
		}
		while (l < elems[pos].val1 && elems[pos].left) { 
		    PUTC(a->target,' ',a); l++; 
		}
		free(temp);
		free_string(&S);
		break;

	    case 'S':
		if (V_string_val != elems[pos].type) {
		    PUTC(a->target,'?',a); 
		    break;
		}
		
		S = elems[pos].value.string_val;
		if (!S) {
		    PUTC(a->target,'?',a);
                    break;
		}

		if (!verify_string(S)) {
		    PUTC(a->target,'?',a); 
		    break;
	        }

		S1 = convert_string(display_charset,S,1);

		if (!verify_string(S1)) {
		    PUTC(a->target,'?',a); 
		    break;
	        }
		
		l = string_len(S1);
		if (elems[pos].val2 < 0) 
		    elems[pos].val2 = l;
		
		X = 0;
		temp = us2s(streamclip_from_string(S1,&X,elems[pos].val2,
						   NULL,NULL));    /* TODO: Should use printable_len ? */
		
		while (l < elems[pos].val1 && !elems[pos].left) { 
		    PUTC(a->target,elems[pos].fill,a); l++; 
		}

		for (p=temp; *p; p++) {
		    PUTC(a->target,*p,a); 
		}
		
		free(temp);
		free_string(&S1);
		
		while (l < elems[pos].val1 && elems[pos].left) { 
		    PUTC(a->target,' ',a); l++; 
		}
		
		break;

	    default:
		PUTC(a->target,'?',a); 
		PUTC(a->target,'u',a); 
		PUTC(a->target,'?',a); 
		break;
	    }
		
    }
	
    if (elems != A) {
	free(elems);
	elems = NULL;
    }

}

static int ok_byte P_((int byte));

#if __GNUC__ 
__inline__
#endif
static int ok_byte(byte)
     int byte;
{
    if (!isascii(byte))
	return 0;
    if (iscntrl(byte))
	return 0;
    if (!isprint(byte))
	return 0;
    if ('"' == byte)
	return 0;
    return 1;
}

void debug_puts(a,len,str)
     struct debug_struct *a; 
     int len;
     const char *str;
{
    print_prefix(a);

    if (str) {
	int i;

	for (i = 0; i < len; i++) {
	    PUTC(a->target,str[i],a);
	    if ('\n' == str[i]) {
		if (i < len-1)
		    print_prefix(a);
	    }
	}
    }
}

void debug_print_buffer(a,len,str)
     struct debug_struct *a; 
     int len;
     unsigned char *str;
{
    int i;
    int q = 0;

    print_prefix(a);

    if (!str) {
	PUTC(a->target,'N',a);
	PUTC(a->target,'U',a);
	PUTC(a->target,'L',a);
	PUTC(a->target,'L',a);
    } else for (i = 0; i < len; i++) {
	
	if (!q && i < len-1 && 
	    ok_byte(str[i]) && ok_byte(str[i+1])) {
	    PUTC(a->target,' ',a);
	    PUTC(a->target,'\"',a);
	    q = 1;
	} 

	if (!q)
	    PUTC(a->target,' ',a);
	else if (!ok_byte(str[i])) {
	    PUTC(a->target,'\"',a);
	    q = 0;
	    PUTC(a->target,' ',a);
	}

	if (q)
	    PUTC(a->target,str[i],a);
	else if ('\r' == str[i]) {
	    PUTC(a->target,'C',a);
	    PUTC(a->target,'R',a);
	} else if ('\n' == str[i]) {
	    PUTC(a->target,'L',a);
	    PUTC(a->target,'F',a);

	    if (i < len-1)
		PUTC(a->target,'\\',a);
	    PUTC(a->target,'\n',a);
	    if (i < len-1)
		print_prefix(a);
	} else {
	    int a1 = str[i]/16;
	    int a2 = str[i]%16;

	    PUTC(a->target,'x',a);
	    if (a1 >= 0 && a1 < sizeof hexchars) PUTC(a->target,hexchars[a1],a); 
	    else PUTC(a->target,'?',a);

	    if (a2 >= 0 && a2 < sizeof hexchars) PUTC(a->target,hexchars[a2],a); 
	    else PUTC(a->target,'?',a);		
	}	    
    }

    if (q) {
	PUTC(a->target,'\"',a);
	q = 0;
	PUTC(a->target,' ',a);
    } else if (len > 0 && '\n' != str[len-1])
	PUTC(a->target,' ',a);    
}

/* resulting FILE * can be fclosed after use */
FILE * debug_to_FILE(a)
     struct debug_struct *a;
{
    if (!a->target)
	return NULL;
    flush_debug_buffer(a->target,NULL);
    
    if (a->target->debug_fd != -1) {
	int fd = dup(a->target->debug_fd);

	return fdopen(fd,"w");
    }
    return NULL;
}

/* This IS called from signal handler */
void debug_action_sigcall(
#if ANSI_C
		       struct debug_struct *a, const char * format, ...
#else
		       a, format, va_alist
#endif
		       )
#if !ANSI_C
     struct debug_struct *a; 
     const char * format; 
     va_dcl
#endif
{
    int count, pos;
    char * format_error = NULL;
    const char *s;

    struct  format_elem A[MAX_ARGS];

    va_list vl;

    if (!a->target)
	return;
    print_prefix(a);

    Va_start(vl, format);           /* defined in hdrs/elm_defs.h  */

    count = parse_format_args (A,MAX_ARGS, format,vl, &format_error);

    va_end(vl);
    
    if (format_error) {
	PUTC(a->target,'?',a);  
	PUTC(a->target,'\n',a);  
	for (s = format_error; *s; s++) {
	    PUTC(a->target,*s,a);  
	}
	PUTC(a->target,'\n',a);  
    }

    pos = -1;
    for (s = format; *s; s++) {
	SKIP_TO_ARG(a,s);

	pos++;

	SKIP_WIDTH(a->target,s,a);

	if (pos < 0 || pos >= count) {
	    PUTC(a->target,'?',a); 
	    PUTC(a->target,'O',a); 
	    PUTC(a->target,'?',a); 
	    break; 
	}

	if (A[pos].format_chr != *s) {
	    PUTC(a->target,'?',a); 
	    PUTC(a->target,'f',a); 
	    PUTC(a->target,'?',a); 
	    break;
	}

	if (!handle_common(a->target,A,pos,a)) {
	    PUTC(a->target,'?',a);
	    PUTC(a->target,'u',a);
	    PUTC(a->target,'?',a);

	}
    }
    
}

#ifdef BACKTRACE
#include <execinfo.h>   /* Needed for backtrace() function */

/* This IS called from signal handler -- used on panic() */
void panic_print_backtrace(buffer,size)
     void *const *buffer; 
     int size;
{
    struct debug_target *ptr;

    for (ptr = debug_list; ptr; ptr= ptr-> next) {
	flush_debug_buffer(ptr,NULL);

	if (-1 != ptr->debug_fd)
	    backtrace_symbols_fd(buffer,size,ptr->debug_fd);
    }
}

/* Can not print prefix */

void debug_print_backtrace(a,buffer,size)
     struct debug_struct *a; 
     void *const *buffer; 
     int size;
{
    struct debug_target *ptr;
    
    if (!a->target)
	return;

    ptr = a->target;
    flush_debug_buffer(ptr,NULL);

    if (-1 != ptr->debug_fd)
	backtrace_symbols_fd(buffer,size,ptr->debug_fd);
}

#endif

/* This IS called from signal handler -- used on panic(),
   returns largest debug_level
 */
int panic_dprint(
#if ANSI_C
		  const char * format, ...
#else
		  format, va_alist
#endif
		  )
#if !ANSI_C
     const char * format; 
     va_dcl
#endif
{
    int level = 0;
    int count;
    char * format_error = NULL;
       
    struct  format_elem A[MAX_ARGS];
    struct debug_target *ptr;

    va_list vl;

    Va_start(vl, format);           /* defined in hdrs/elm_defs.h  */
    
    count = parse_format_args (A,MAX_ARGS, format,vl, &format_error);

    va_end(vl);
    
    for (ptr = debug_list; ptr; ptr= ptr-> next) {
	int i;
	const char *s;
	int pos;

	DEBUG_VAR(temp,__FILE__,"");

	temp.target = ptr;
	
	for (i = 0; i < ptr->match_count; i++) {
	    if (ptr->match[i].level > level)
		level = ptr->match[i].level;
	}

	print_prefix(&temp);
	
	if (format_error) {
	    PUTC(ptr,'?',&temp);  
	    PUTC(ptr,'\n',&temp);  
	    for (s = format_error; *s; s++) {
		PUTC(ptr,*s,&temp);  
	    }
	    PUTC(ptr,'\n',&temp);  
	}
	
	pos = -1;
	for (s = format; *s; s++) {
	    SKIP_TO_ARG((&temp),s);
	    
	    pos++;
	    
	    SKIP_WIDTH(ptr,s,(&temp));
	    
	    if (pos < 0 || pos >= count) {
		PUTC(ptr,'?',&temp); 
		PUTC(ptr,'o',&temp); 
		PUTC(ptr,'?',&temp); 
		break; 
	    }
	    
	    if (A[pos].format_chr != *s) {
		PUTC(ptr,'?',&temp); 
		PUTC(ptr,'f',&temp); 
		PUTC(ptr,'?',&temp); 
		break;
	    }
	    
	    if (!handle_common(ptr,A,pos,&temp)) {
		PUTC(ptr,'?',&temp);
		PUTC(ptr,'u',&temp);
		PUTC(ptr,'?',&temp);
	    }
	}	
    }

    return level;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
