static char rcsid[] = "@(#)$Id: parsestring.c,v 2.13 2021/07/13 07:58:36 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.13 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

/* This does parsing for struct string *

   File getaddr.c includes correspond address parsing routine for
   char *
*/

#include "def_addr.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"addr");


static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str)
     const char *str;
{
    return (const unsigned char *)str;
}


static char *us2s P_((unsigned char *str));
static char *us2s(str)
     unsigned char *str;
{
    return (char *)str;
}

static struct string * unquote_comment P_((struct string * value,
					   unq_can_print_error_f can_print_error,
					   const char *filename /* MAY be NULL */,
					   int lineno,
					   int *haserror));
static struct string * unquote_comment(value,can_print_error,filename,lineno,haserror)
     struct string * value;
     unq_can_print_error_f can_print_error;
     const char *filename /* MAY be NULL */;
     int lineno;
     int *haserror;
{
    charset_t cs = get_string_type(value);
    int len = string_len(value);
    int i;

    struct string * ret = new_string(cs);

    for (i = 0; i < len;) {
	int dummy;
	uint16 code = give_unicode_from_string(value,i);

	if (0 == i      && 0x0028  /* ( */  == code) {
	    i++;
	    continue;

	}
	if (len-1 == i  && 0x0029  /* ) */  == code) {
	    i++;
	    continue;
	}

	if (0x005C  /*  \  */ == code) {
	    i++;

	    if (i < len) {
		/* NO-OP */
		code = give_unicode_from_string(value,i);
	    } else {
		if (can_print_error(filename,lineno,value)) {

		    if (filename)
			lib_error(CATGETS(elm_msg_cat, MeSet,
					  MeUnqTrailingBackslash,
					  "%s: %d: Trailing Backslash (\\) on %S"),
				  filename,lineno,value);
			else
			    lib_error(CATGETS(elm_msg_cat, MeSet,
					      MeTrailingBackslashOnString,
					      "Trailing Backslash (\\) on %S"),
				      value);
		}

		if (haserror)
		    *haserror = 1;
		break;
	    }
	}
	
	string_copy_character(&ret,value,&i,1,&dummy); 
    }
	
    return ret;
}

/* This should do same for 'struct string_token * tokenized'

   than 

   look_special_tokens() does for 'char **tokenized'

   However, this assumes that MIME decoding is not needed
*/

static void string_look_tokens P_((const struct string_token * tokenized,
				   uint16              * token_chars,
				   int  start,
				   int *ended,
				   struct string       **comments,
				   struct string_token ** scanned,
				   unq_can_print_error_f can_print_error,
				   const char *filename /* MAY be NULL */,
				   int lineno,
				   int *haserror));


static void string_look_tokens(tokenized,token_chars,start,ended,
			       comments,scanned,
			       can_print_error,
			       filename,lineno,haserror) 
     const struct string_token * tokenized;
     uint16              * token_chars;
     int  start;
     int *ended;
     struct string       **comments;
     struct string_token ** scanned;
     unq_can_print_error_f can_print_error;
     const char *filename /* MAY be NULL */;
     int lineno;
     int *haserror;
{
    int end;

    DPRINT(Debug,25,(&Debug, 
		     "string_look_tokens: start=%d\n",
		     start));

    *scanned = NULL;

    for (end = start; tokenized[end].token; end++) {

	int j;
	int found = 0;

	for (j = 0; token_chars[j]; j++) {
	    if (token_chars[j] == tokenized[end].special)
		found = 1;
	}

	if (found)
	    break;

	DPRINT(Debug,25,(&Debug," [%d]=%S",end,tokenized[end].token));
    }
    DPRINT(Debug,25,(&Debug,"\n"));

    if (!tokenized[end].token) {
	DPRINT(Debug,25,(&Debug,"string_look_tokens: end=%d NO MATCH\n",end));
    } else {
	DPRINT(Debug,25,(&Debug,"string_look_tokens: end=%d MATCH=%S\n",
			 end,tokenized[end].token));
    }
    

   if (end > start) {
	int count = end-start;
	int i, in_ptr,out_ptr;

	struct string_token *res;

	DPRINT(Debug,26,(&Debug,"string_look_tokens: count=%d\n",
			 count));
	
	res = safe_calloc((count+1), sizeof (res[0]));
	for (i = 0; i <= count; i++) {
	    res[i].token   = NULL;
	    res[i].special = 0;
	}

	for (in_ptr=start, out_ptr=0;
	     in_ptr < end && out_ptr < count;
	     in_ptr++) {

	    /* String_tokenize() marks all whitespca with
	       special = 0x0020
	    */

	    if (0x0020 == tokenized[in_ptr].special) {

		if (*comments && string_len(*comments) && in_ptr < end-1)
		    append_string(comments,tokenized[in_ptr].token,1);

	    } else if (0x0028    /* '(' */ == tokenized[in_ptr].special) {
		
		/* ( ) characters are stripped */

		struct string * S = unquote_comment(tokenized[in_ptr].token,
						    can_print_error,filename,
						    lineno,haserror);

		append_string(comments,S,1);

		free_string(&S);

	    } else {

		res[out_ptr].token   = dup_string(tokenized[in_ptr].token);
		res[out_ptr].special = tokenized[in_ptr].special;
		out_ptr++;
	    }

	}

	DPRINT(Debug,26,(&Debug,"string_look_tokens: real count=%d, scanned:",
			 out_ptr));
	
	for (i = 0; i < out_ptr; i++) {
	    DPRINT(Debug,26,(&Debug," [%d]=%S",i,res[i].token));
	}
	*scanned=res;

	DPRINT(Debug,26,(&Debug,
			 "\nstring_look_tokens: *comments=%S\n",
			 *comments));
   }

   if (ended)
       *ended = end;
   
}

/* Prints error message if result is not ascii ... */
static char * ascify_address P_((const struct string  * address));
static char * ascify_address(address)
     const struct string  * address;
{
    charset_t ascii_ptr = MIME_name_to_charset("US-ASCII",0);

    char * ascaddr = NULL;
    struct string *Y = NULL;

    if (! ascii_ptr)
	panic("ADDR PANIC",__FILE__,__LINE__,"ascify_address",
	      "US-ASCII not found",0);
    
    if (!can_ascii_string(address))
	lib_error(CATGETS(elm_msg_cat, MeSet,
			  MeAsciiRequiredAddress,
			  "Ascii required for address %S"),
		  address);
    
    Y = convert_string(ascii_ptr,address,0);
    ascaddr = us2s(stream_from_string(Y,0,NULL));
    free_string(&Y);

    return ascaddr;
}

struct address * new_address_string(address,phrase,comment)
     const struct string  * address;
     const struct string  * phrase;
     const struct string  * comment;
{

    char           * ascaddr = NULL;
    struct address * ret = NULL; 
      
    if (address) {
	ascaddr = ascify_address(address);

    }

    ret = new_address(ascaddr,phrase,comment);

    if (ascaddr)
	free(ascaddr);

    return ret;
}

/* Caller must free string */
struct string *string_quote_phrase(phrase) 
     const struct string *phrase;

{
    charset_t cs       = get_string_type(phrase);
    struct string *ret = new_string(cs);
    int i;
    int len = string_len(phrase);

    add_ascii_to_string(ret,s2us("\""));

    for (i = 0; i < len;) {
	int dummy;
	const uint16 code = give_unicode_from_string(phrase,i);
    
	if (0x005C    /* '\\'*/  == code ||
	    0x0022    /* '"' */  == code)
	    add_ascii_to_string(ret,s2us("\""));
	
	string_copy_character(&ret,phrase,&i,1,&dummy); 

    } 

    add_ascii_to_string(ret,s2us("\""));

    return ret;
}

void append_quoted_to_string(ret,value)
     struct string ** ret;
     const struct string * value;
{
    struct string * tmp = string_quote_phrase(value);
    
    append_string(ret,tmp,0);
    
    free_string(&tmp);
}


/* Caller must free string */
struct string *string_quote_comment(phrase) 
     const struct string *phrase;
{
    struct string *ret = new_string(display_charset);
    int i;
    int len = string_len(phrase);

    for (i = 0; i < len;) {
	int dummy;
	const uint16 code = give_unicode_from_string(phrase,i);
    
	if (0x005C    /* '\\'*/  == code ||
	    0x0028    /* ( */ == code || 
	    0x0029    /* ) */ == code)
	    add_ascii_to_string(ret,s2us("\""));
	
	string_copy_character(&ret,phrase,&i,1,&dummy); 

    } 

    return ret;
}

/* Actually MIME special characters do not need quotation
   but we quote them anyway
*/
static int string_phrase_need_quote P_((const struct string *phrase));
static int string_phrase_need_quote(phrase)
     const struct string *phrase;
{
    int i;
    int len = string_len(phrase);

    for (i = 0; i < len; i++) {
	const uint16 code = give_unicode_from_string(phrase,i);

	if (code == UNICODE_BAD_CHAR ||
	    unicode_is_special(code,TOK_mail|TOK_mime))
	    return 1;	
    }

    return 0;
}

int string_nonnull(S)
     const struct string *S;
{
    if (!S)
	return 0;
    return string_len(S);
}

/* Caller must free string */
struct string * address_to_string(addr)
     const struct address *addr;
{
    struct string *ret = NULL;
    

    const struct string *phrase  = address_get_phrase(addr);
    const struct string *comment = address_get_comment(addr);
    const char *         ascaddr = address_get_ascii_addr(addr);

    if (string_nonnull(phrase)) {
       
	if (string_phrase_need_quote(phrase)) 
	    append_quoted_to_string(&ret,phrase);
	else
	    append_string(&ret,phrase,0);
    }

    if (!ret)
	ret = new_string(display_charset);

    if (string_nonnull(phrase) || !ascaddr || !ascaddr[0] || '@' == ascaddr[0]) {
	add_ascii_to_string(ret,s2us(" <"));
	if (ascaddr)
	    add_ascii_to_string(ret,cs2us(ascaddr));	 
	add_ascii_to_string(ret,s2us(">"));

    } else {
	add_ascii_to_string(ret,cs2us(ascaddr));	
    }

    if (string_nonnull(comment)) {
	struct string * tmp = string_quote_comment(comment);

	add_ascii_to_string(ret,s2us(" ("));
	append_string(&ret,tmp,1);
	add_ascii_to_string(ret,s2us(")"));

	free_string(&tmp);
    }
     
    return ret;
}

/* Caller must frees string */
struct string * addr_list_to_string(list)
     const  struct addr_list *list;
{
    struct string * ret = NULL;

    int idx;
    int addr_item_count = addr_list_item_count(list);
    int group_item_count = addr_list_group_count(list);
    int current_group = -1;
    int last_group = -1;

    for (idx = 0; idx < addr_item_count; idx++) {

	int group = -1;
	const struct address * address = 
	    addr_list_get_item(list,idx,&group);

	struct string * tmp2 = address_to_string(address); 

    
	if (current_group != group &&
	    current_group != -1) {
	    add_ascii_to_string(ret,s2us(";"));
	}

	if (idx > 0) {
	    add_ascii_to_string(ret,s2us(", "));
	}
	
	if (group >= 0 &&
	    current_group == -1) {
	    
	    const struct string * groupname1 =
		addr_list_get_group(list,group);
	   
	    int g;
		
	    /* Print empty groups */
	    
	    for (g = last_group+1; g < group; g++) {
		const struct string * groupname =
		    addr_list_get_group(list,g);

		append_quoted_to_string(&ret,groupname);

		add_ascii_to_string(ret,s2us(":;, "));
	    }

	    if (last_group < group)
		last_group = group;

	    append_quoted_to_string(&ret,groupname1);

	    add_ascii_to_string(ret,s2us(": "));
	    current_group = group;	
	}

	append_string(&ret,tmp2,1);
	free_string(&tmp2);	
    }

    if (current_group != -1) {
	add_ascii_to_string(ret,s2us(";"));
    }

    /* Print empty groups */
    
    if (last_group+1 < group_item_count) {
	int g;

	if (idx > 0) {
	    add_ascii_to_string(ret,s2us(", "));
	}

	for (g = last_group+1; g < group_item_count; g++) {

	    const struct string * groupname =
		addr_list_get_group(list,g);
		
	    append_quoted_to_string(&ret,groupname);
	    
	    add_ascii_to_string(ret,s2us(":;"));
	    
	    if (g < group_item_count-1) {
		add_ascii_to_string(ret,s2us(", "));
	    }
	}
    }

    return ret;

}

int unq_can_print_error(filename,lineno,value)
     const char          * filename;
     int                   lineno;
     const struct string * value;
{
    /* DUMMY */
    
    return 1;
}

struct string * unquote_string(value,can_print_error,filename,lineno,
			       haserror)
     const struct string * value;
     unq_can_print_error_f can_print_error;
     const char *filename /* MAY be NULL */;
     int lineno;
     int *haserror;
{
    charset_t cs = get_string_type(value);
    int len = string_len(value);
    int i;
    int q = 0;

    struct string * ret = new_string(cs);

    for (i = 0; i < len;) {
	int dummy;
	uint16 code = give_unicode_from_string(value,i);

	if (0x0022  /*  "  */   == code) {
	    q = !q;
	    i++;
	    continue;
	}

	/* Recognize \ escape only in quoted strings */
	if (0x005C  /*  \  */ == code && q) {
	    i++;

	    if (i < len) {
		/* NO-OP */
		code = give_unicode_from_string(value,i);
	    } else {

		if (can_print_error(filename,lineno,value)) {

		    if (filename)
			lib_error(CATGETS(elm_msg_cat, MeSet,
					  MeUnqTrailingBackslash,
					  "%s: %d: Trailing Backslash (\\) on %S"),
				  filename,lineno,value);
			else
			    lib_error(CATGETS(elm_msg_cat, MeSet,
					      MeTrailingBackslashOnString,
					      "Trailing Backslash (\\) on %S"),
				      value);

		}


		if (haserror)
		    *haserror = 1;
		break;

	    }
	}
	
	string_copy_character(&ret,value,&i,1,&dummy); 
    }
	
    if (q) {
	if (can_print_error(filename,lineno,value)) {
	    
	    if (filename)
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeUnqNoEnQuote,
				  "%s: %d: Missing ending quote (\") on %S"),
			  filename,lineno,value);
	    else
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeMissingEndQuote,
				  "Missing ending quote (\") on %S"),
			  value);
	}
	
	if (haserror)
	    *haserror = 1;
    }

    return ret;
}

struct string * string_scanned_to_phrase 
     P_((const struct string_token * scanned,
	 unq_can_print_error_f can_print_error,
	 const char *filename /* MAY be NULL */,
	 int lineno,int *haserror));

struct string * string_scanned_to_phrase(scanned,can_print_error,
					 filename,lineno,haserror)
     const struct string_token * scanned;
     unq_can_print_error_f can_print_error;
     const char *filename /* MAY be NULL */;
     int lineno;
     int *haserror;
{
    struct string * phrase = NULL;
    int i;

    DPRINT(Debug,25,(&Debug, 
		     "string_scanned_to_phrase:"));


    /* It is assumed that MIME decoding is not required
       because this is not ASCII version
    */
    
    for (i = 0; scanned[i].token; i++) {
	struct string * S = unquote_string(scanned[i].token,
					   can_print_error,filename,
					   lineno,haserror);

	DPRINT(Debug,25,(&Debug, " [%d]=%S",i,scanned[i].token));
	if (phrase)
	    add_ascii_to_string(phrase,s2us(" "));

	append_string(&phrase,S,1);
	free_string(&S);
    }
    
    DPRINT(Debug,25,(&Debug, "\nstring_scanned_to_phrase=%p=%S\n",
		     phrase,phrase));

    return phrase;
}

struct string * string_scanned_to_address P_((const struct string_token * scanned));
struct string * string_scanned_to_address(scanned)
     const struct string_token * scanned;
{
    struct string * address       = NULL;

    int i;

    DPRINT(Debug,25,(&Debug, 
		     "string_scanned_to_address:"));


    for (i = 0; scanned[i].token; i++) {
	DPRINT(Debug,25,(&Debug, " [%d]=%S",i,scanned[i].token));
	append_string(&address,scanned[i].token,0);
    }

    DPRINT(Debug,25,(&Debug, "\nstring_scanned_to_address=%p=%S\n",
		     address,address));
    return address;
}


static uint16 token_chars[] = {
    0x003A    /* ':' */,
    0x003C    /* '<' */,
    0x003E    /* '>' */,
    0x002C    /* ',' */,
    0x003B    /* ';' */,
    0
};

struct address * parse_one_tokenized_address(tokenized,tok_start,tok_end,
					     parse_error,can_print_error,
					     filename,lineno)
     const struct string_token * tokenized;
     int tok_start; 
     int tok_end;
     int * parse_error;
     unq_can_print_error_f can_print_error;
     const char *filename /* MAY be NULL */;
     int lineno;
{
    int walk = tok_start, next_point;

    struct string_token * scanned = NULL;
    struct string       * comments = NULL;

    struct string * phrase        = NULL;
    struct string * address       = NULL;

    struct address * ret = NULL;

    if (parse_error)
	*parse_error = 0;

    string_look_tokens(tokenized,token_chars,walk,&next_point,
		       &comments,&scanned,can_print_error,
		       filename,lineno,parse_error);

    /* Get phrase */

    if (next_point < tok_end &&
	0x003C /* < */ == tokenized[next_point].special) {


	if (scanned) {
	    struct string * XX = 
		string_scanned_to_phrase(scanned,can_print_error,
					 filename,lineno,parse_error);
	    if (XX) {
		append_string(&phrase,XX,1);
		free_string(&XX);
	    }

	    free_string_tokenized(&scanned);
	}

	if (next_point+1 < tok_end &&
	    0x003E /*  > */ == tokenized[next_point+1].special) {

	    DPRINT(Debug,10,(&Debug, 
			     "parse_one_tokenized_address: Bounce address <> seen\n"));

	    address = new_string(ASCII_SET);
	    
	    walk = next_point+2;
	    
	    goto was_bounce_address;
	}
	
    skip_route:
	walk = next_point+1;

	string_look_tokens(tokenized,token_chars,walk,&next_point,
			   &comments,&scanned,can_print_error,
			   filename,lineno,parse_error);

	/* Skip route addr */

	if (tokenized[next_point].token &&
	    (0x002C /* , */ == tokenized[next_point].special ||
	     0x003A /* : */ == tokenized[next_point].special)) {

	    free_string_tokenized(&scanned);
	    goto skip_route;
	}

	if (next_point >= tok_end ||
	    0x003E /* > */ != tokenized[next_point].special)
	    goto syntax_error;
	
	if (scanned) {
	    struct string * XX = string_scanned_to_address(scanned);
	    if (XX) {
		append_string(&address,XX,0);
		free_string(&XX);
	    }

	    free_string_tokenized(&scanned);
	}

	walk = next_point+1;

    was_bounce_address:
	
	string_look_tokens(tokenized,token_chars,walk,&next_point,
			   &comments,&scanned,can_print_error,
			   filename,lineno,parse_error);

	if (scanned && scanned[0].token)
	    goto syntax_error; 

    } else if (next_point >= tok_end) {
	/* store address */

	if (scanned) {
	    int i;

	    for (i = 0; scanned[i].token; i++) 
		append_string(&address,scanned[i].token,0);

	    free_string_tokenized(&scanned);
	}
	
    }  else {
    syntax_error:

	if (parse_error)
	    *parse_error = 1;

	if (tokenized[next_point].token) {
	    DPRINT(Debug,1,(&Debug, 
			    "parse_one_tokenized_address: next token: %S\n",
			    tokenized[next_point].token));

	    if (tokenized[next_point].special) {
		DPRINT(Debug,1,(&Debug, "       special: %04x\n",
				tokenized[next_point].special));
	    }	   
	}
    }

    if (scanned)
	free_string_tokenized(&scanned);

    ret = new_address_string(address,phrase,comments);

    if (address)
	free_string(&address);
    if (phrase)
	free_string(&phrase);
    if (comments)
	free_string(&comments);
    
    return ret;
}

struct address   * parse_one_string_address(str,can_print_error,
					    filename,lineno,
					    haserror) 
     const struct string *str;
     unq_can_print_error_f can_print_error;
     const char *filename /* MAY be NULL */;
     int lineno;
     int * haserror;
{
    struct address * ret = NULL;
    int parse_error = 0;
    int tok_end = 0;

    struct string_token * tokenized = string_tokenize(str,TOK_mail);


    if (!tokenized) {
	DPRINT(Debug,9,(&Debug, 
			"parse_one_string_address=NULL\n"));
	return NULL;
    }
    
    for (tok_end = 0; tokenized[tok_end].token; tok_end++)
	;
    
    DPRINT(Debug,11,(&Debug, 
		     "parse_one_string_address: tok_end=%d\n",
		     tok_end));
    
    ret = parse_one_tokenized_address(tokenized,0,tok_end,& parse_error,
				      can_print_error,filename,lineno);
     
    if (parse_error) {
	
	DPRINT(Debug,1,(&Debug, 
			"parse_one_string_address: parse error, address=%S\n",
			str));
       
	if (can_print_error(filename,lineno,str)) {
	    
	    if (filename)
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeParseOneError,
				  "%s: %d: Parse error on address: %S"),
			  filename,lineno,str);
		else
		    lib_error(CATGETS(elm_msg_cat, MeSet,
				      MeParseErrorAddressS,
				      "Parse error on address: %S"),
			      str);	
	}

	if (haserror)
	    *haserror = 1;
    }

    free_string_tokenized(&tokenized);

    return ret;
}

struct addr_list * parse_tokenized_address(tokenized,tok_start,tok_end,
					   defcharset,can_print_error,
					   filename,lineno,haserror)
     const struct string_token * tokenized;
     int tok_start; 
     int tok_end;
     charset_t defcharset;
     unq_can_print_error_f can_print_error;
     const char *filename /* MAY be NULL */;
     int lineno;
     int *haserror;
{
    int count=1;

    struct {
	int in_group;
	int in_bracket;
    } state;

    struct addr_list *result = NULL;
    
    struct tmpstore res;
    int group_num = -1;

    int outer,inner;

    zero_tmpstore(&res);

    DPRINT(Debug,9,(&Debug, 
		    "parse_tokenized_address: tok_start=%d, tok_end=%d\n",
		    tok_start,tok_end));
		    
    for (outer = 0; outer < tok_end; outer++) {
	DPRINT(Debug,10,(&Debug, 
			 "                       : [%d] = \"%S\"",
			 outer,tokenized[outer].token));
	if (tokenized[outer].special) {
	    DPRINT(Debug,10,(&Debug, ", special = %04x",
			     tokenized[outer].special));
	}
	DPRINT(Debug,10,(&Debug,"\n"));

	if (0x002C  /* , */ == tokenized[outer].special)
	    count++;
    }

    DPRINT(Debug,11,(&Debug, 
		     "parse_tokenized_address: count=%d\n",
		     count));
		    
    result = new_empty_addr_list(count);

    state.in_group   = 0;
    state.in_bracket = 0;  
    
    for (outer = tok_start; outer < tok_end; outer = inner) {
	struct string_token * scanned = NULL;
	struct string *comments = NULL;
	int  tok = 0x0000;

	DPRINT(Debug,25,(&Debug, 
			 "parse_tokenized_address: [%d]=%.10S... state: in_group=%d in_bracket=%d\n",
			 outer,tokenized[outer].token,state.in_group,state.in_bracket));

	
	string_look_tokens(tokenized,token_chars,outer,&inner,
			   &comments,&scanned,can_print_error,
			   filename,lineno,haserror);
	
	if (inner < tok_end) {
	    tok = tokenized[inner].special;
	    DPRINT(Debug,25,(&Debug, 
			     "parse_tokenized_address: [%d] token=%04x (%S)\n",
			     inner,tok,tokenized[inner].token));
	    inner++;
	} else {
	    tok = 0x0000;
	    DPRINT(Debug,25,(&Debug, 
			     "parse_tokenized_address: [%d] token=EOS\n",inner));
	}

	/* state engine */
	if (!state.in_group) {
	    if (!state.in_bracket) 
		
		switch(tok) {
		    
		case 0x003A    /* ':' */:
		    group_num = -1;

		    if (scanned && scanned[0].token) {
			struct string * XX = 
			    string_scanned_to_phrase(scanned,
						     can_print_error,
						     filename,lineno,
						     haserror);

			DPRINT(Debug,25,(&Debug, 
					 "... storing group phrase: %S\n",
					 XX));

			group_num = add_group_to_list(result,XX);
			free_string(&XX);
		    }


		    state.in_group = 1;
		    break;

		case 0x003C    /* '<' */:
		    if (scanned && scanned[0].token) {
			struct string * XX = 
			    string_scanned_to_phrase(scanned,
						     can_print_error,
						     filename,lineno,
						     haserror);

			DPRINT(Debug,25,(&Debug, 
					 "... storing address phrase: %S\n",XX));
			
			append_string(&res.fullname,XX,1);
			free_string(&XX);
		    }

		    state.in_bracket = 1;
		    break;

		case  0x003E    /* '>' */:
		    DPRINT(Debug,25,(&Debug, 
				     "... Parse error, next token '>'\n"));
		    if (haserror)
			*haserror = 1;
				    
		    break;

		case 0x003B    /* ';' */:
		    DPRINT(Debug,25,(&Debug, 
				     "... Parse error, next token ';'\n"));
		    
		    if (haserror)
			*haserror = 1;

		    break;

		case 0x002C    /* ',' */:
		default: /* 0x0000 */

		    if (scanned && scanned[0].token) { 
			char * ascaddr = NULL;
			struct string * XX = string_scanned_to_address(scanned);
		       
			if (res.addr) {
			    DPRINT(Debug,25,(&Debug,
					     ".... parse error, address after '>': %S\n",
					     XX));

			    if (haserror)
				*haserror = 1;
			}

			DPRINT(Debug,25,(&Debug, 
					 "... storing address: %S\n",XX));
			
			ascaddr = ascify_address(XX);
			free_string(&XX);

			res.addr = strmcat(res.addr,ascaddr);
			free(ascaddr);

			/* This do not convert comments to phrase/fullname
			   convert_comment  is ignored

			   its is for situation where header's are parsed
			*/

		    }   

		    if (comments) {
			DPRINT(Debug,25,(&Debug,  
					 "... storing comments: %S\n", comments));
			append_string(&res.comment,comments,1);
			free_string(&comments);
		    } 

		    if (res.addr || res.fullname || res.comment) {
			DPRINT(Debug,25,(&Debug, "... parsed address, group %d\n",
					 group_num));
			tmpstore_to_result(result,&res,group_num,
					   defcharset);
		    }
			
		    if (!state.in_group &&
			group_num != -1) {
			DPRINT(Debug,25,(&Debug, "... Resetting group number\n"));
			group_num = -1;
		    }
		    
		    break;

		} else switch(tok) {    /* state.in_bracket */

		case  0x003A    /* ':' */:

		    DPRINT(Debug,25,(&Debug, 
				     "... skipping route\n"));
		    break;

		case 0x003C    /* '<' */:
		    DPRINT(Debug,25,(&Debug, 
				     "... Parse error, next token '<'\n"));

		    if (haserror)
			*haserror = 1;

		    break;

		case 0x003E    /* '>' */:
		store_addr1:

		    if (scanned && scanned[0].token) { 
			char * ascaddr = NULL;
			struct string * XX = string_scanned_to_address(scanned);

			DPRINT(Debug,25,(&Debug, 
					 "... storing address: %S\n",XX));
			
			ascaddr = ascify_address(XX);
			free_string(&XX);

			res.addr = strmcat(res.addr,ascaddr);
			free(ascaddr);
		    }

		    if (res.addr || res.fullname || res.comment) {

			/* NOTE: There may be (comment) after <addr>
			         so wait for it if not end of string
			*/

			if (inner >= tok_end) {
			    DPRINT(Debug,25,(&Debug, "... parsed address (EOS), group %d\n",
					     group_num));
			    tmpstore_to_result(result,&res,group_num,
					       defcharset);
			}
		    }
		    
		    state.in_bracket = 0;

		    break;

		case 0x002C    /* ',' */:
		    
		    DPRINT(Debug,25,(&Debug, 
				     "... skipping route\n"));
		    break;
		    
		case 0x003B    /* ';' */:
		    
		    DPRINT(Debug,25,(&Debug, 
				     "... Parse error, next token ';'\n"));

		    if (haserror)
			*haserror = 1;

		    break;
		    
		default: /* 0x0000 */

		    DPRINT(Debug,25,(&Debug, 
				     "... Parse error, missing '>'\n"));

		    if (haserror)
			*haserror = 1;
		    
		    goto store_addr1;	  
		}

	} else { /* state.in_group */

	    if (!state.in_bracket)
		switch(tok) {
		case  0x003A    /* ':' */:

		    DPRINT(Debug,25,(&Debug, 
				     "... Parse error, next token ':'\n"));

		    if (haserror)
			*haserror = 1;

		    break;

		case 0x003C    /* '<' */:
		    
		    if (scanned && scanned[0].token) { 
			struct string * XX = 
			    string_scanned_to_phrase(scanned,
						     can_print_error,
						     filename,lineno,
						     haserror);

			DPRINT(Debug,25,(&Debug, 
					"... storing address phrase: %S\n",
					XX));
			
			append_string(& res.fullname,XX,1);
			free_string(&XX);
		    }

		    state.in_bracket = 1;
		    break;

		case 0x003E    /* '>' */:

		    DPRINT(Debug,25,(&Debug, 
				     "... Parse error, next token '>'\n"));

		    if (haserror)
			*haserror = 1;

		    break;

		case 0x003B    /* ';' */:
		store_addr2:
		    
		    DPRINT(Debug,25,(&Debug, 
				     "... end of group\n"));

		    state.in_group = 0;
		    /* Resetting group_num is delayed ... */

		    /* FALLTHRU */
		case 0x002C    /* ',' */:
		    
		    if (scanned && scanned[0].token) {
			char * ascaddr = NULL;
			struct string * XX = string_scanned_to_address(scanned);
			
			if (res.addr) {
			    DPRINT(Debug,25,(&Debug,
					     ".... parse error, address after '>': %S\n",
					     XX));

			    if (haserror)
				*haserror = 1;
			}


			DPRINT(Debug,25,(&Debug, 
					 "... storing address: %S\n",XX));
			
			ascaddr = ascify_address(XX);
			free_string(&XX);
			
			res.addr = strmcat(res.addr,ascaddr);
			free(ascaddr);
		    }

		    /* This do not convert comments to phrase/fullname
		       convert_comment  is ignored
		       
		       its is for situation where header's are parsed
		    */
		    
		    if (comments) {
			DPRINT(Debug,25,(&Debug,  
					 "... storing comments: %S\n", comments));
			append_string(&res.comment,comments,1);
			free_string(&comments);
		    } 
		    
		    if (res.addr || res.fullname || res.comment) {
			DPRINT(Debug,25,(&Debug, "... parsed address, group %d\n",
					 group_num));
			tmpstore_to_result(result,&res,group_num,
					   defcharset);
		    }
		    
		    if (!state.in_group &&
			group_num != -1) {
			DPRINT(Debug,25,(&Debug, "... Resetting group number\n"));
			group_num = -1;
		    }

		    break;

		default: /* 0x0000 */

		    DPRINT(Debug,25,(&Debug, 
				     "... Parse error, missing ';'\n"));

		    if (haserror)
			*haserror = 1;
		   
		    goto store_addr2;

		} else switch(tok) { /* state.in_bracket */

		case 0x003A    /* ':' */:

		    DPRINT(Debug,25,(&Debug, 
				     "... skipping route\n"));
		    break;

		case 0x003C    /* '<' */:
		    DPRINT(Debug,25,(&Debug, 
				     "... Parse error, next token '<'\n"));

		    if (haserror)
			*haserror = 1;

		    break;

		case 0x003B    /* ';' */:
		    DPRINT(Debug,25,(&Debug, 
				     "... Parse error, next token ';', missing '>' ... end of group\n"));

		    if (haserror)
			*haserror = 1;

		    state.in_group = 0;
		    /* Resetting group_num is delayed ... */

		    /* FALLTHRU */
		case  0x003E    /* '>' */:
		store_addr3:
		    
		    if (scanned && scanned[0].token) {
			char * ascaddr = NULL;
			struct string * XX = string_scanned_to_address(scanned);
			
			DPRINT(Debug,25,(&Debug, 
					 "... storing address: %S\n",XX));
			
			ascaddr = ascify_address(XX);
			free_string(&XX);
			
			res.addr = strmcat(res.addr,ascaddr);
			free(ascaddr);
		    }   

		    if (res.addr || res.fullname || res.comment)  {

			/* NOTE: There may be (comment) after <addr>
			   so wait for it if not end of string
			*/
			
			if (inner >= tok_end) {
			    DPRINT(Debug,25,(&Debug, "... parsed address (EOS), group %d\n",
					     group_num));
			    tmpstore_to_result(result,&res,group_num,
					       defcharset);
			}
		    }
		    break;

		case 0x002C    /* ',' */:
		    DPRINT(Debug,25,(&Debug, 
				     "... skipping route\n"));
		    break;

		default: /* 0x0000 */

		    DPRINT(Debug,25,(&Debug, 
				     "... Parse error, missing '>', missing ';' ... end of group\n"));

		    if (haserror)
			*haserror = 1;

		    state.in_group = 0;
		    /* Resetting group_num is delayed ... */

		    goto store_addr3;
		}
	}

	if (comments) {
	    DPRINT(Debug,25,(&Debug, " ... ignoring comments: %S\n", 
			     comments));


	    free_string(&comments);
	}

	if (scanned)
	    free_string_tokenized(&scanned);
    }

    if (res.addr) {
	DPRINT(Debug,3,(&Debug, 
			"parse_tokenized_address: res.addr NOT NULL: %s\n", 
			res.addr));
	free(res.addr); res.addr = NULL;
    }
    if (res.fullname) {
	DPRINT(Debug,3,(&Debug, 
			"parse_tokenized_address: res.fullname NOT NULL: %S\n",
			res.fullname));
	free_string(&res.fullname);
    }
    if (res.comment) {
	DPRINT(Debug,3,(&Debug, 
			"parse_tokenized_address: res.comment NOT NULL: %S\n",
			res.comment));
	free_string(&res.comment);
    }
    res.magic = 0;    /* Invalidate */

    
    return result;
}

struct addr_list * parse_string_address(str,can_print_error,
					filename,lineno,haserror) 
     const struct string *str;
     unq_can_print_error_f can_print_error;
     const char *filename /* MAY be NULL */;
     int lineno;
     int * haserror; 
{

    int tok_end;
           
    struct string_token * tokenized = string_tokenize(str,TOK_mail);
    struct addr_list *result = NULL;

    DPRINT(Debug,9,(&Debug, 
		    "parse_string_address: str=%S\n",
		    str));
    if (!tokenized) {
	DPRINT(Debug,9,(&Debug, 
			"parse_string_address=NULL\n"));

	return NULL;
    }

    for (tok_end = 0; tokenized[tok_end].token; tok_end++)
	;

    DPRINT(Debug,11,(&Debug, 
		     "parse_string_address: tok_end=%d\n",
		     tok_end));
    
    result = parse_tokenized_address(tokenized,0,tok_end,
				     get_string_type(str),
				     can_print_error,
				     filename,lineno,haserror);
   
    free_string_tokenized(&tokenized);

    DPRINT(Debug,11,(&Debug, 
		     "parse_string_address=%p\n",
		     result));

    return result;
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

