static char rcsid[] = "@(#)$Id: parsarpdat.c,v 2.2 2020/05/12 18:18:19 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.2 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 lib/parsarpdat.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1993 USENET Community Trust
 *****************************************************************************/

#include "def_addr.h"

DEBUG_VAR(Debug,__FILE__,"header");

/*

Quoting from RFC 822:
     5.  DATE AND TIME SPECIFICATION

     5.1.  SYNTAX

     date-time   =  [ day "," ] date time        ; dd mm yy
						 ;  hh:mm:ss zzz

     day         =  "Mon"  / "Tue" /  "Wed"  / "Thu"
		 /  "Fri"  / "Sat" /  "Sun"

     date        =  1*2DIGIT month 2DIGIT        ; day month year
						 ;  e.g. 20 Jun 82

     month       =  "Jan"  /  "Feb" /  "Mar"  /  "Apr"
		 /  "May"  /  "Jun" /  "Jul"  /  "Aug"
		 /  "Sep"  /  "Oct" /  "Nov"  /  "Dec"

     time        =  hour zone                    ; ANSI and Military

     hour        =  2DIGIT ":" 2DIGIT [":" 2DIGIT]
						 ; 00:00:00 - 23:59:59

     zone        =  "UT"  / "GMT"                ; Universal Time
						 ; North American : UT
		 /  "EST" / "EDT"                ;  Eastern:  - 5/ - 4
		 /  "CST" / "CDT"                ;  Central:  - 6/ - 5
		 /  "MST" / "MDT"                ;  Mountain: - 7/ - 6
		 /  "PST" / "PDT"                ;  Pacific:  - 8/ - 7
		 /  1ALPHA                       ; Military: Z = UT;
						 ;  A:-1; (J not used)
						 ;  M:-12; N:+1; Y:+12
		 / ( ("+" / "-") 4DIGIT )        ; Local differential
						 ;  hours+min. (HHMM)
*/

enum parse_date_time_result parse_parts_date_time(str,time_zone,size_time_zone,
						  month_p,day_p,year_p,
						  hours_p,mins_p,secs_p,
						  tz_p,wday_p)
     const char *str;
     char *time_zone;
     int size_time_zone;
     int *month_p;
     int *day_p;
     int *year_p;
     int *hours_p;
     int *mins_p;
     int *secs_p;
     int *tz_p;
     int *wday_p /* 0 - 6 */;
{
    enum parse_date_time_result ret = parse_dt_fail;

    /*
     * Parse a date field in either RFC-822 or Unix date(1) format.
     */

    char field[STRING], save_tz[STRING];	
    int month, day, year=0, hours, mins, secs, tz, len, i;

    static int report_time_MAX = 1;
    
    if (report_time_MAX) {
	DPRINT(Debug,1,(&Debug,
			"Using time_MAX = %ld\n",
			(long)time_MAX));
	report_time_MAX = 0;
    }
    
    /*
     * Since this is an RFC-822 field, there might be parenthetical
     * comments.  Yank them out.  Note that strip_parens() returns
     * a pointer to static data.
     */
    str = strip_parens(str);

      /*
       * The first field is an optional day of the week.  If it exists
       * it is supposed to have a trailing comma by RFC-822, but we won't
       * complain if it doesn't.  If the date string was generated by
       * the Unix date(1) command then it won't have the comma.  We don't
       * do anything with this information, just skip over it if it exists.
       */
    if ((len = get_word(str, 0, field, sizeof(field))) < 0)
	goto failed;
    if (cvt_dayname_to_daynum(field, &i)) {
	/* cvt_dayname_to_daynum   :  Convert a day name to number (Sun = 1) */
	
       if (wday_p)
	   *wday_p = i-1;
	str += len;
    } else if (wday_p)
       *wday_p = -1;
	
    /*
     * Peek at the next character to determine what format to
     * parse the rest of the line as.
     */
    while (isspace(*str))
	++str;

    if (!isdigit(*str)) {
	
	/*
	 * Parse the line in Unix date(1) format.  The syntax is:
	 *
	 *	month day hh:mm:ss [tz] year
	 *
	 * e.g. "Jun 21 06:45:44 CDT 1989".  The timezone is optional.
	 */

	DPRINT(Debug,7,(&Debug,
			"parse_parts_date_time parsing \"%s\" in time(1) format\n", 
			str));

	/* <month> */
	if ((len = get_word(str, 0, field, sizeof(field))) < 0 ||
	    !cvt_monthname_to_monthnum(field, &month))
	    goto failed;
	str += len;

	/* <day> */
	if ((len = get_word(str, 0, field, sizeof(field))) < 0 ||
	    (day = atonum(field)) < 0)
	    goto failed;
	str += len;

	/* <hh:mm:ss> */
	if ((len = get_word(str, 0, field, sizeof(field))) < 0 ||
	    !cvt_timestr_to_hhmmss(field, &hours, &mins, &secs))
	    goto failed;
	str += len;

	/* optional <tz> */
	save_tz[0] = save_tz[1] = '\0';
	tz = 0;
	while ((len = get_word(str, 0, field, sizeof(field))) > 0 &&
	       cvt_timezone_to_offset(field, &i, sizeof field)) {
	    (void) strfcat(save_tz, " ", sizeof save_tz);
	    (void) strfcat(save_tz, field, sizeof save_tz);
	    tz += i;
	    str += len;
	}

	/* <year> */
	if ((len = get_word(str, 0, field, sizeof(field))) < 0 ||
	    (year = atonum(field)) < 0)
	    goto failed;
	str += len;

	ret =  parse_dt_unix;
	/* there might be more...but we ignore it */

    } else {
	
	/*
	 * Parse the line in RFC-822 format.  The syntax is:
	 *
	 *	day month year hh:mm:ss zone
	 *
	 * e.g. "17 Nov 92 23:34:25 CST".
	 *
	 * Really "9 Feb 2018 19:31:20 +0200"
	 *
	 *     
	 */

	DPRINT(Debug,7,(&Debug,
			"parse_parts_date_time parsing \"%s\" in RFC-822 format\n", 
			str));
	
	/* <day> */
	if ((len = get_word(str, 0, field, sizeof(field))) < 0 ||
	    (day = atonum(field)) < 0)
	    goto failed;
	str += len;

	/* <month> */
	if ((len = get_word(str, 0, field, sizeof(field))) < 0 ||
	    !cvt_monthname_to_monthnum(field, &month))
	    goto failed;
	str += len;

	/* <year> */
	if ((len = get_word(str, 0, field, sizeof(field))) < 0 ||
	    !cvt_yearstr_to_yearnum(field, &year))
	    goto failed;
	str += len;


	/* <hh:mm:ss> */
	if ((len = get_word(str, 0, field, sizeof(field))) < 0) {
	    
	    /* on expires header there is no necessary time -part */
	    
	    save_tz[0] = save_tz[1] = '\0';
	    tz = 0;
	    hours = 0;
	    mins = 0;
	    secs = 0;
	    DPRINT(Debug,7,(&Debug,
			    "  only date given (no time and timezone)\n"));

	    ret = parse_dt_dateonly;
	    goto notime;
	}
	
        if (!cvt_timestr_to_hhmmss(field, &hours, &mins, &secs))
	    goto failed;
	str += len;
	
	/* <tz> - silently ignore bogus or missing timezones */
	save_tz[0] = save_tz[1] = '\0';
	tz = 0;
	while ((len = get_word(str, 0, field, sizeof(field))) > 0 &&
	       cvt_timezone_to_offset(field, &i, sizeof field)) {
	    (void) strfcat(save_tz, " ", sizeof save_tz);
	    (void) strfcat(save_tz, field, sizeof save_tz);
	    tz += i;
	    str += len;
	}

	ret = parse_dt_mail;
	/* there might be more...but we ignore it */

    }

 notime:

    DPRINT(Debug,7,(&Debug, "  year=%d month=%d day=%d\n", year, month, day));
    DPRINT(Debug,7,(&Debug, "  hours=%d mins=%d secs=%d tz=%d\n",
		    hours, mins, secs, tz));
        
    if (time_zone) {
	strfcpy(time_zone, save_tz+1, size_time_zone);
	DPRINT(Debug,7,(&Debug, "  time_zone=%s\n",
			time_zone));
    }
    if (wday_p) {
         DPRINT(Debug,7,(&Debug, "  wday=%d\n",
			 *wday_p));
    }
    
    *month_p = month;
    *day_p   = day;
    *year_p  = year;
    *hours_p = hours;
    *mins_p  = mins;
    *secs_p  = secs;
    *tz_p    = tz;

    DPRINT(Debug,4,(&Debug, "parse_parts_date_time=%d\n",ret));
    
    return ret;

 failed:

    DPRINT(Debug,4,(&Debug, "parse_parts_date_time failed at \"%s\"\n",
		    (len <= 0 ? "<premature eol>" : field)));
    
    
    return     parse_dt_fail;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

