static char rcsid[] = "@(#)$Id: message-id.c,v 2.8 2019/03/11 18:23:55 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.8 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_addr.h"

DEBUG_VAR(Debug,__FILE__,"addr");

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

#define MSGID_magic          0xE802

struct message_id {

    unsigned short magic;        /* MSGID_magic */
    
    char          * left;
    char          * domain;
    struct string *comment;
  
};


struct message_id  * new_message_id(left,domain,comment)
     const char * left;
     const char * domain;
     const struct string *comment;
{
    struct message_id *ret = safe_zero_alloc(sizeof(*ret));
      
    ret->magic = MSGID_magic;

    ret->left    = left    ? safe_strdup(left)   : NULL;
    ret->domain  = domain  ? safe_strdup(domain) : NULL;
    ret->comment = comment ? dup_string(comment) : NULL;

    return ret;
}

void free_message_id(id)
     struct message_id  ** id;
{
    if (MSGID_magic != (*id)->magic)
	panic("ID PANIC",__FILE__,__LINE__,"free_message_id",
	      "Bad magic number",0);

    if ((*id)->left)
      free((*id)->left);
    (*id)->left = NULL;

    if ((*id)->domain)
      free((*id)->domain);
    (*id)->domain = NULL;

    if ((*id)->comment)
      free_string(& ((*id)->comment));

    (*id)->magic   = 0; /* Invalidate */
    free(*id);
    *id = NULL;
}

struct message_id * dup_message_id(id)
     const struct message_id *id;
{
    if (MSGID_magic != id->magic)
	panic("ID PANIC",__FILE__,__LINE__,"dup_message_id",
	      "Bad magic number",0);

    return new_message_id(id->left,id->domain,id->comment);
}

int same_message_id(id1,id2,ignore_comment)
     const struct message_id *id1;
     const struct message_id *id2;
     int ignore_comment;
{
    if (MSGID_magic != id1->magic)
	panic("ID PANIC",__FILE__,__LINE__,"same_message_id",
	      "Bad magic number (id1)",0);

    if (MSGID_magic != id2->magic)
	panic("ID PANIC",__FILE__,__LINE__,"same_message_id",
	      "Bad magic number (id2)",0);

    if (id1->left && id2->left) {
	if (0 != strcmp(id1->left,id2->left))
	    return 0;
    } else if (id1->left || id2->left) 
	return 0;

    if (id1->domain && id2->domain) {
	if (0 != strcmp(id1->domain,id2->domain))
	    return 0;
    } else if (id1->domain || id2->domain) 
	return 0;


    if (ignore_comment)
	return 1;

    if (id1->comment && id2->comment) {
	if (0 != string_cmp(id1->comment,id2->comment,99))
	    return 0;
    } else if (id1->comment || id2->comment) 
	return 0;
	       
    return 1;
}

static int cmpstrnull P_((const char *a, const char *b));
static int cmpstrnull(a,b) 
     const char *a;
     const char *b;
{
    int r = strcmp(a ? a : "", b ? b : "");

    if (0 == r) {
	if (!a && !b)
	    return 0;
	if (!a && b)
	    return 1;
	if (a && !b)
	    return -1;
    }

    return r;
}


int cmp_message_id(id1,id2)
     const struct message_id *id1;
     const struct message_id *id2;
{
    int r;

    if (MSGID_magic != id1->magic)
	panic("ID PANIC",__FILE__,__LINE__,"cmp_message_id",
	      "Bad magic number (id1)",0);

    if (MSGID_magic != id2->magic)
	panic("ID PANIC",__FILE__,__LINE__,"cmp_message_id",
	      "Bad magic number (id2)",0);

    r = cmpstrnull(id1->domain,id2->domain);
    if (r)
	return r;

    r = cmpstrnull(id1->left,id2->left);
    if (r)
	return r;


    if (id1->comment && id2->comment)
	/* XXX uncomparable are equal ??? */
	r = string_cmp(id1->comment,id2->comment,0);
    else if (id1->comment)
	return 1;
    else if (id2->comment)
	return -1;

    return r;
}




/* Result is malloced,  < > are included to result, no comments */
char * get_message_id_ascii_brackets(id)
     const struct message_id *id;
{
    char * ret = NULL;

    if (MSGID_magic != id->magic)
	panic("ID PANIC",__FILE__,__LINE__,"get_message_id_ascii_brackets",
	      "Bad magic number",0);


    if (! id->left && !id->domain)
	return NULL;

    ret = safe_strdup("<");

    if (id->left)
	ret = strmcat(ret,id->left);

    if (id->domain) {
	ret = strmcat(ret,"@");
	ret = strmcat(ret,id->domain);
    }
    
    ret = strmcat(ret,">");

    return ret;
}

struct string * message_id_to_string(id)
      const struct message_id *id;
{
    struct string *ret = NULL;
    
    if (MSGID_magic != id->magic)
	panic("ID PANIC",__FILE__,__LINE__,"message_id_to_string",
	      "Bad magic number",0);

    if (! id->left && !id->domain && !id->comment)
	return NULL;

    ret = new_string(id->comment ?
		     get_string_type(id->comment) :
		     ASCII_SET);

    add_ascii_to_string(ret,s2us("<"));
    if (id->left)
	add_ascii_to_string(ret,s2us(id->left));
    if (id->domain) {
	add_ascii_to_string(ret,s2us("@"));
	add_ascii_to_string(ret,s2us(id->domain));
    }
    add_ascii_to_string(ret,s2us(">"));

    if (string_nonnull(id->comment)) {
	struct string * tmp = string_quote_comment(id->comment);

	add_ascii_to_string(ret,s2us(" ("));
	append_string(&ret,tmp,1);
	add_ascii_to_string(ret,s2us(")"));

	free_string(&tmp);	
    }
    
    return ret;
}

const struct string * message_id_comment(id)
     const struct message_id *id;
{
    if (MSGID_magic != id->magic)
	panic("ID PANIC",__FILE__,__LINE__,"message_id_comment",
	      "Bad magic number",0);

    return id->comment;
}

const char * message_id_left(id)
     const struct message_id  *id;
{
    if (MSGID_magic != id->magic)
	panic("ID PANIC",__FILE__,__LINE__,"message_id_left",
	      "Bad magic number",0);
    
    return id->left;
}

const char * message_id_domain(id) 
     const struct message_id  *id;
{
   if (MSGID_magic != id->magic)
       panic("ID PANIC",__FILE__,__LINE__,"message_id_domain",
	      "Bad magic number",0);
   
    return id->domain;

}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
